/*
 * drivers/net/gt642x0_mpsc.c
 *
 * UART driver for Galileo/Marvell GT-642x0 MPSC
 *
 * Original Author: Chris Zankel <chris@mvista.com>
 * Maintainer: Troy Benjegerdes <tbenjegerdes@mvista.com> and
 * 		Mark Greer <mgreer@mvista.com>
 *
 * Copyright 2001 MontaVista Software Inc.
 *
 * Based on arch/ppc/8xx_io/uart.c by Dan Malek
 *
 * In an ideal world, this driver could be used by both MIPS and PowerPC
 * that use the gt64240 and gt64260 bridge chips. I need to either get a
 * 64240 mips board or find someone to test this.. -- Troy
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 */ 

/* Note:
 *  It is hard or almost impossible to get a continuous flow going with the
 *  SDMA engine. So we have to syncronize with the transmitter manually for
 *  each transmision, which means we have to wait for the transmitter to
 *  finish with the old transmision before we can start a new one.
 */

#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif
#include <linux/module.h>

#include <linux/types.h>
#ifdef LOCAL_HEADERS
#include "serial_local.h"
#else
#include <linux/serial.h>
#include <linux/serialP.h>
#include <linux/serial_reg.h>
#include <asm/serial.h>
#define LOCAL_VERSTRING ""
#endif

#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/interrupt.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/major.h>
#include <linux/string.h>
#include <linux/fcntl.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/mm.h>
#include <linux/malloc.h>
#include <linux/version.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <linux/delay.h>
#ifdef CONFIG_SERIAL_CONSOLE
#include <linux/console.h>
#ifndef CONFIG_SERIAL_CONSOLE_PORT
#define CONFIG_SERIAL_CONSOLE_PORT	0
#endif
#endif
#ifdef CONFIG_MAGIC_SYSRQ
#include <linux/sysrq.h>
#endif

#include <asm/semaphore.h>
#include <linux/config.h>
#include <linux/version.h>
#include <linux/pci.h>
#include <asm/pci.h>
#include <asm/uaccess.h>

#include "gt64260_mpsc.h"
#ifdef __powerpc__
#include <asm/gt64260.h>
#else
#error gt64260 not yet supported on this arch
#endif

#ifndef CONFIG_NOT_CACHE_COHERENT
#warning "hackery..., defining CONFIG_NOT_CACHE_COHERENT "
#define CONFIG_NOT_CACHE_COHERENT
#endif

#ifdef CONFIG_NOT_CACHE_COHERENT
#include <asm/cache.h>
#endif

/* ============================ configuration ============================== */

/* basic configuration of the mpsc's */

#define GT_CONFIG_SERIAL_PORT_MULTIPLEX 0x1102	/* connect MPSC0 + 3 ports of RMII */
#define GT_CONFIG_CLOCK_ROUTING_REGISTER 0x00000100 /* usr BRG0 for MPSC0, BRG1 for MPSC1 */
#if 0
#define GT_CONFIG_DEFAULT_MPSC_MRR   0x007ffe38
#define GT_CONFIG_DEFAULT_MPSC_MRR   0x807ffe38	/* MPSC0/1 to Serial Port 0/1*/
#else
#define GT_CONFIG_DEFAULT_MPSC_MRR   0x3ffffe38	/* MPSC0/1 to Serial Port 0/1*/
#endif
#define GT_CONFIG_DEFAULT_UART_UBSR  0x00000000
#define GT_CONFIG_DEFAULT_UART_CHR   0x00000000
#define GT_CONFIG_DEFAULT_UART_MIR   4		/* 4 idle chars before rx closes descr. */
#define GT_CONFIG_DEFAULT_UART_AR    0x00000000
#define GT_CONFIG_DEFAULT_UART_ESR   0x00000000
#define GT_CONFIG_DEFAULT_UART_UBSR  0x00000000
#define GT_CONFIG_DEFAULT_UART_MMCRL 0x000004c4	/* UART, Enable Trans/Rec, NullM */
#if 0
#define GT_CONFIG_DEFAULT_UART_MMCRH 0x044005f8	/* rcdv=3, rdw=1, tcdv=3 */
#else
#if 1
#define GT_CONFIG_DEFAULT_UART_MMCRH 0x04400400	/* rcdv=3, rdw=1, tcdv=3 */
#else
#define GT_CONFIG_DEFAULT_UART_MMCRH 0x064007ec	/* rcdv=3, rdw=1, tcdv=3 */
#endif
#endif
#define GT_CONFIG_DEFAULT_BRG_BCR    0x00210000	/* Enable, TClk */
#define GT_CONFIG_DEFAULT_MPSC_MPCR  (3<<GT_MPSC_MPCR_CL_SHIFT) /* 8N1 */

#warning /* this needs to be NOT hardcoded */
#define GT_CONFIG_BUSFREQ 133*1024*1024
#define BRG_CDV(baud)	((GT_CONFIG_BUSFREQ/(32*baud)) - 1)

/* the number of buffer descriptors and their sizes. */
/* this is a hack for some cache coherency issues */
#define RX_NUM_DESCRIPTORS 32
#define RX_BUF_SIZE 128
#define TX_NUM_DESCRIPTORS 32
#define TX_BUF_SIZE 128

/* misc. */

#undef SERIAL_PARANOIA_CHECK
#undef CONFIG_SERIAL_NOPAUSE_IO
#undef SERIAL_DO_RESTART

#define WAIT_FOR_TXD_TIME	10000	/* [ms] */

/* set of debugging defines */

#undef SERIAL_DEBUG_INTR
#undef SERIAL_DEBUG_OPEN
#undef SERIAL_DEBUG_FLOW
#undef SERIAL_DEBUG_RS_WAIT_UNTIL_SENT
#undef SERIAL_RS_WRITE_DEBUG

/* ========================== global variables ============================= */

#ifdef CONFIG_KGDB
extern void breakpoint(void);
extern void set_debug_traps(void);
extern int  kgdb_output_string (const char* s, unsigned int count);
int kgdb_debug_port;
#endif

static char *serial_name = "GT64260 MPSC driver";
static char *serial_version = "1.00";

static DECLARE_TASK_QUEUE(tq_serial);

static struct tty_driver serial_driver, callout_driver;
static int serial_refcount;
static int serial_console_setup(struct console*, char*);

#define _INLINE_ inline

#define TX_WAKEUP	ASYNC_SHARE_IRQ
#define RS_EVENT_WRITE_WAKEUP       0

#ifndef MIN
#define MIN(a,b)	((a) < (b) ? (a) : (b))
#endif

/* ========================== local declarations =========================== */

/*
 *The async_struct in serial.h does not really give us what we need,
 * so we define our own here. 
 */

typedef struct serial_info {
	int			magic;
	int			flags;
	struct serial_state	*state;
	struct tty_struct 	*tty;

	int			read_status_mas1k;
	int			ignore_status_mask;
	int			timeout;
	int			line;
	int			x_char;	/* xon/xoff character */
	int			close_delay;
	unsigned short		closing_wait;
	unsigned short		closing_wait2;
	unsigned long		event;
	unsigned long		last_active;
	int			blocked_open; /* # of blocked opens */
	long			session; /* Session of opening process */
	long			pgrp; /* pgrp of opening process */
	struct tq_struct	tqueue;
	struct tq_struct	tqueue_hangup;
	wait_queue_head_t	open_wait;
	wait_queue_head_t	close_wait;

	/* rx/tx descriptor pointer */

	gt_rx_descriptor_t 	*rd_cur;
	gt_rx_descriptor_t 	*rd_first;
	gt_tx_descriptor_t 	*td_tail;
	gt_tx_descriptor_t 	*td_first;

	/* descriptor/buffer memory */

	int size;
	char *mem;
	dma_addr_t dmah;

} ser_info_t;

u32	ubsr_cache = 0;
u32	cr_cache = 0;


/* ========================== forward declaration ========================== */

static void change_speed(ser_info_t *info);
static void rs_wait_until_sent(struct tty_struct *tty, int timeout);
static void serial_console_write(struct console*, const char*, unsigned);
static kdev_t serial_console_device(struct console*);
static int serial_console_wait_key(struct console*);

/* =========================== global variables ============================ */

int gt64260_mpsc_baud_rate = 115200;	/* default baud rate */

/* ============================ local variables ============================ */

/* this driver is not using uart, clk, port, irq */
static struct serial_state rs_table[] = {
	/* UART, CLK,  PORT, 			IRQ, FLAGS, NUM */
	{     0,   0,   0,   0,     0,   0 },
	{     0,   0,   0,   0,     0,   1 }
};

/* number of serial ports available on this board */

#define NUM_PORTS (sizeof(rs_table)/sizeof(struct serial_state))

#undef CONFIGURED_SERIAL_PORT(info)
#define CONFIGURED_SERIAL_PORT(info) ((info)->state)

static struct tty_struct *serial_table[NUM_PORTS];
static struct termios *serial_termios[NUM_PORTS];
static struct termios *serial_termios_locked[NUM_PORTS];

/* baud Table */

static int baud_table[] = {
	0, 50, 75, 110, 134, 150, 200, 300, 600, 1200, 1800, 2400, 4800,
	9600, 19200, 38400, 57600, 115200, 230400, 460800, 0 };

static int baud_idx;

#ifdef DECLARE_MUTEX
static DECLARE_MUTEX(td_desc_sem);
#else
static struct semaphore td_desc_sem = MUTEX;
#endif

#ifdef CONFIG_SERIAL_CONSOLE

/* console */

static struct console sercons = {
	name:		"ttyS",
	write:		serial_console_write,
	device:		serial_console_device,
	wait_key:	serial_console_wait_key,
	setup:		serial_console_setup,
	flags:		CON_PRINTBUFFER,
#if 0	/* XXXX */
	index:		CONFIG_SERIAL_CONSOLE_PORT,
#else
	index:		-1,
#endif
};

char sercon_txbuffer[2][PAGE_SIZE] __attribute__ ((aligned (PAGE_SIZE)));
char sercon_rxbuffer[2][PAGE_SIZE] __attribute__ ((aligned (PAGE_SIZE)));

gt_tx_descriptor_t *sercon_td[2];	/* two ports */
gt_rx_descriptor_t *sercon_rd[2];

/*
 * For the initial console we are using only one buffer for each the receiver
 * and transmitter.
 */

#endif /* CONFIG_SERIAL_CONSOLE */

#ifdef CONFIG_KGDB

static char kgdb_buf[RX_BUF_SIZE], *kgdp;
static int kgdb_chars;
unsigned int kgdb_baud_rate = 9600;

#endif /* CONFIG_KGDB */



/* ============================ local functions ============================ */


/* serial_paranoia_check()
 *
 * This functions only verifies the info and device parametes.
 *
 * Input Variable(s):
 *   ser_info_t*	info structure,
 *   kdev_t		device,
 *   const char*	routine (to be executed if magic does not match).
 *
 * Output Variable(s):
 *   None.
 *
 * Returns:
 *  0 (no error) or 1 (error)
 */

static inline int serial_paranoia_check (ser_info_t *info, kdev_t device,
					 const char *routine)
{
#ifdef SERIAL_PARANOIA_CHECK
	static const char *badmagic =
		"Warning: bad magic number for serial struct (%s) in %s\n";
	static const char *badinfo =
		"Warning: null async_struct for (%s) in %s\n";

	if (!info) {
		printk(badinfo, kdevname(device), routine);
		return 1;
	}
	if (info->magic != SERIAL_MAGIC) {
		printk(badmagic, kdevname(device), routine);
		return 1;
	}
#endif
	return 0;
}


/* rs_transmitter_sync()
 *
 * This functions only syncronizes with transmitter.
 *
 * Input Variable(s):
 *   unsigned int	sdma base address.
 *
 * Output Variable(s):
 *   None.
 *
 * Returns:
 *  -1 (no error) or 0 (timeout)
 */

static inline int rs_transmitter_sync(unsigned int sdma)
{
	int i;

	
	for (i=0; 
	     i < WAIT_FOR_TXD_TIME 
		     && (gt_read(sdma+GT_SDMA_SDCM) & GT_SDMA_SDCM_TXD);
	     i++)
		udelay(1000);
	
	if (i==WAIT_FOR_TXD_TIME)
		printk("transmitter sync error!!\n");
	
	return i<WAIT_FOR_TXD_TIME;
}


/* rs_stop()
 * 
 * This routine is called before setting tty->stopped.
 * It disables transmitter interrupts, as necessary.
 *
 * Input Variable(s):
 *   struct tty_struct*		Pointer to the tty.
 *
 * Outpu Variable(s):
 *   None.
 *
 * Returns:
 *   Void.
 */

static void rs_stop (struct tty_struct *tty)
{

	ser_info_t *info = (ser_info_t*)tty->driver_data;
	unsigned long flags;
	unsigned int sdma;


	if (serial_paranoia_check(info, tty->device, "rs_stop"))
		return;
	
	save_flags(flags); cli();

	sdma = GT_BASE_SDMA0 + info->state->line * GT_BASE_SDMA_DIST;

	/* wait for transmitter to finish */

	rs_transmitter_sync(sdma);

	/* abort transmission */

	gt_write(sdma+GT_SDMA_SDCM, 
		     gt_read(sdma+GT_SDMA_SDCM) | GT_SDMA_SDCM_STD);

	/* disable and clear interrupts */

#if 0 /* XXXX */
	gt_write(GT_SDMA_MASK, 
		     gt_read(GT_SDMA_MASK) 
		     & ~(info->line == 0? 0x000f : 0x0f00));
	gt_write(GT_SDMA_CAUSE,
		     gt_read(GT_SDMA_CAUSE) 
		     & (info->line == 0? 0xfffffff0 : 0xfffff0ff));
#else
	gt_write(GT_SDMA_MASK,  0x00000000);
	gt_write(GT_SDMA_CAUSE, 0x00000000);
#endif

	restore_flags(flags);
}


/* rs_start()
 *
 * This routine is called before resetting tty->stopped.
 * It enables transmitter and receiver interrupts, as necessary.
 *
 * Input Variable(s):
 *   struct tty_struct*		Pointer to the tty.
 *
 * Outpu Variable(s):
 *   None.
 *
 * Returns:
 *   Void.
 */

static void rs_start(struct tty_struct *tty)
{
	ser_info_t *info = (ser_info_t *)tty->driver_data;
	unsigned long flags;
	unsigned int sdma;


	//	printk("rs_start\n");

	if (serial_paranoia_check(info, tty->device, "rs_stop"))
		return;
	
	sdma = GT_BASE_SDMA0 + info->state->line * GT_BASE_SDMA_DIST;
	
	save_flags(flags); cli();
	
	/* set sdma pointer and start transmission */
	
	if (gt_read(sdma+GT_SDMA_SDCM) & GT_SDMA_SDCM_TXD)
		goto done;	/* already running */
		
	/* clear and enable interrupts */

#if 0 /* XXXX */
	gt_write(GT_SDMA_CAUSE,
		     gt_read(GT_SDMA_CAUSE) 
		     & (info->line == 0? 0xfffffff0 : 0xfffff0ff));
	gt_write(GT_SDMA_MASK, 
		     gt_read(GT_SDMA_MASK) 
		     | (info->line == 0? 0x000f : 0x0f00));
#else
	gt_write(GT_SDMA_CAUSE, 0x00000000);
	gt_write(GT_SDMA_MASK,  0x00000003);
#endif

	gt_write(sdma+GT_SDMA_SCTDP, __pa(info->td_tail));
	gt_write(sdma+GT_SDMA_SFTDP, __pa(info->td_tail));
	
	gt_write(sdma+GT_SDMA_SDCM, 
		     gt_read(sdma+GT_SDMA_SDCM) | GT_SDMA_SDCM_TXD);
	
done:
	restore_flags(flags);
}


/* -------------------------- interrupt handling --------------------------- */

/* Here starts the interrupt handling routines.  All of the following
 * subroutines are declared as inline and are folded into
 * rs_interrupt().  They were separated out for readability's sake.
 *
 * Note: rs_interrupt() is a "fast" interrupt, which means that it
 * runs with interrupts turned off.  People who may want to modify
 * rs_interrupt() should try to keep the interrupt handler as fast as
 * possible.  After you are done making modifications, it is not a bad
 * idea to do:
 * 
 * gcc -S -DKERNEL -Wall -Wstrict-prototypes -O6 -fomit-frame-pointer serial.c
 *
 * and look at the resulting assemble code in serial.s.
 *
 * 				- Ted Ts'o (tytso@mit.edu), 7-Mar-93
 */

/* rs_sched_event() 
 * 
 * This routine is used by the interrupt handler to schedule
 * processing in the software interrupt portion of the driver.
 *
 * Input Variable(s):
 *   ser_info_t*	Serial device,
 *   int		Type of event.
 *
 * Output Variable(s):
 *  None
 * 
 * Returns:
 *  void
 */

static inline void rs_sched_event(ser_info_t *info, int event)
{
	info->event |= 1 << event;
	queue_task(&info->tqueue, &tq_serial);
	mark_bh(SERIAL_BH);
}


/* reveive_chars() 
 * 
 * This routine copies as many characters as were received by
 * the MPSC to the tty buffer or until the tty buffer is full.
 *
 * Input Variable(s):
 *   ser_info_t*	Serial device,
 *   int		1: clean descriptors even if there is
 *                         no space for the characters
 * Output Variable(s):
 *   None
 *
 * Returns
 *   void
 */

static inline void receive_chars (ser_info_t *info, int force, struct pt_regs *regs)
{
	struct tty_struct *tty = info->tty;
	struct	async_icount *icount;
	gt_rx_descriptor_t* rd;
	unsigned char *cp;
	unsigned int mpsc;
	int i;

	
	rd = info->rd_cur;
	mpsc = GT_BASE_MPSC0 + info->line * GT_BASE_MPSC_DIST;

	/* do all buffers */

#ifdef CONFIG_NOT_CACHE_COHERENT
	invalidate_dcache_range((unsigned long)rd,
				(unsigned long)(rd+1)-1);
#endif
	while((rd->cmd_stat & GT_SDMA_DESC_CMDSTAT_OWN) == 0) {
		
		/* get buffer and number of received chars */

		cp = __va(rd->buffer);
		i = rd->bytecnt;

#ifdef CONFIG_NOT_CACHE_COHERENT
		invalidate_dcache_range((unsigned long)cp,
				(unsigned long)cp + RX_BUF_SIZE - 1);
#endif

#ifdef SERIAL_DEBUG_INTR
		printk("DR%02x:%08x.%d..", info->line, rd->cmd_stat, i);
#endif

		/* Check to see if there is room in the tty buffer for
		 * the characters. If not, we exit now, leaving the
		 * descriptor with the character.  We'll pick
		 * them up again on the next receive interrupt (which could
		 * be a timeout).
		 * If we are forced to, we clean one descriptor 
		 * and then return (even if characters are lost).
		 */

#ifdef CONFIG_KGDB
		if ((info->flags & ASYNC_INITIALIZED) == 0)
			tty = NULL;
#endif

		if (tty && (tty->flip.count + i) >= TTY_FLIPBUF_SIZE) {
			if (force) {
				printk(KERN_WARNING "Out of receiver buffer, dropping characters!\n");
				rd->cmd_stat &= ~GT_SDMA_DESC_CMDSTAT_STATUS_MASK;
				rd->cmd_stat |= GT_SDMA_DESC_CMDSTAT_OWN;
#ifdef CONFIG_NOT_CACHE_COHERENT
				flush_dcache_range((unsigned long)rd,
						(unsigned long)(rd + 1) - 1);
#endif
				rd = __va(rd->next_desc_ptr);
#ifdef CONFIG_NOT_CACHE_COHERENT
				invalidate_dcache_range((unsigned long)rd,
						(unsigned long)(rd + 1) - 1);
#endif
			}
			break;
		}
		
		if (tty && rd->cmd_stat & GT_SDMA_DESC_CMDSTAT_BR) {
			icount->brk++;
#ifdef SERIAL_DEBUG_INTR
			printk("handling break....");
#endif
			*tty->flip.flag_buf_ptr = TTY_BREAK;
			if (info->flags & ASYNC_SAK)
				do_SAK(tty);
		}
		
		while (i-- > 0) {
			
			/* copy character to buffer */

#ifdef CONFIG_KGDB
			if (info->line == kgdb_debug_port) {
				
				if (*(char*)((unsigned long)cp ^ 7) == 0x03) {
					
					/* flush input buffer */
					while (!(rd->cmd_stat & GT_SDMA_DESC_CMDSTAT_OWN)) {
						rd->cmd_stat &= ~GT_SDMA_DESC_CMDSTAT_STATUS_MASK;
						rd->cmd_stat |=  GT_SDMA_DESC_CMDSTAT_OWN;
#ifdef CONFIG_NOT_CACHE_COHERENT
						flush_dcache_range((unsigned long)rd, (unsigned long)(rd + 1) - 1);
#endif
						rd = __va(rd->next_desc_ptr);
#ifdef CONFIG_NOT_CACHE_COHERENT
						invalidate_dcache_range((unsigned long)rd, (unsigned long)(rd + 1) - 1);
#endif
					}
					regs->msr |= MSR_SE;
					info->rd_cur = rd;
					return;
				} else if (*(char*)((unsigned long)cp ^ 7) == '$') {
					regs->msr |= MSR_SE;
					return;
				} else if (!tty) {
					/* tty not open, process next character */
					cp++;
					continue;
				}
				
			}
#endif
			
			*tty->flip.char_buf_ptr++ = 
				*(char*)((unsigned long)cp++ ^ 7);
			
			
			if (rd->cmd_stat & GT_SDMA_DESC_CMDSTAT_ES) {
				
				if (rd->cmd_stat & GT_SDMA_DESC_CMDSTAT_PE) {
					icount->parity++;
					*tty->flip.flag_buf_ptr = TTY_PARITY;
				} else if (rd->cmd_stat & GT_SDMA_DESC_CMDSTAT_FR) {
					icount->frame++;
					*tty->flip.flag_buf_ptr = TTY_FRAME;
				}
				
				if (rd->cmd_stat & GT_SDMA_DESC_CMDSTAT_OR) {
					icount->overrun++;
					if (tty->flip.count < TTY_FLIPBUF_SIZE) {
						tty->flip.count++;
						tty->flip.flag_buf_ptr++;
						tty->flip.char_buf_ptr++;
						*tty->flip.flag_buf_ptr = TTY_OVERRUN;
					}
				}
			}
			if (tty->flip.count >= TTY_FLIPBUF_SIZE)
				break;
			
			tty->flip.flag_buf_ptr++;
			tty->flip.char_buf_ptr++;
			tty->flip.count++;
		}
		
		/* clean receive descriptor and go to the next one */
		
		rd->cmd_stat &= ~GT_SDMA_DESC_CMDSTAT_STATUS_MASK;
		rd->cmd_stat |= GT_SDMA_DESC_CMDSTAT_OWN;
#ifdef CONFIG_NOT_CACHE_COHERENT
		flush_dcache_range((unsigned long)rd,
				(unsigned long)(rd + 1) - 1);
#endif
		rd = __va(rd->next_desc_ptr);
#ifdef CONFIG_NOT_CACHE_COHERENT
		invalidate_dcache_range((unsigned long)rd,
				(unsigned long)(rd+1)-1);
#endif

	}
	info->rd_cur = rd;
#ifdef CONFIG_KGDB
	if (tty)
#endif
		queue_task(&tty->flip.tqueue, &tq_timer);

}

/* receive_break() 
 * 
 * This routine is part of the interrupt handler and is calles
 * whenever a break was received. 
 *
 * Input Variable(s):
 *   ser_info_t*	serial info structure.
 *
 * Output Variable(s):
 *   None
 *
 * Returns
 *   void
 */

static inline void receive_break(ser_info_t *info)
{
	struct tty_struct *tty = info->tty;

	printk("break!!\n");
	
	info->state->icount.brk++;

	/* Check to see if there is room in the tty buffer for
	 * the break.  If not, we exit now, losing the break.  FIXME
	 */

	if ((tty->flip.count + 1) >= TTY_FLIPBUF_SIZE)
		return;
	*(tty->flip.flag_buf_ptr++) = TTY_BREAK;
	*(tty->flip.char_buf_ptr++) = 0;
	tty->flip.count++;
	
	queue_task(&tty->flip.tqueue, &tq_timer);
}


#ifdef MODEM_CONTROL

/* check_modem_status() - NOT IMPLEMENTED YET
 * 
 *  Checks and handles changes of the modem status lines.
 *  (like RTS/CTS flow control, hangup, etc.)
 *
 * Input Variable(s):
 *   ser_info_t*	Serial device,
 *
 * Output Variable(s):
 *   None.
 *
 * Returns:
 *   void
 */

static inline void check_modem_status(struct async_struct *info)
{
	int status;
	struct async_icount *icount;
	

	status = serial_in(info, GT_MPSC_MMCRL);

	/* CTS flow */
	
	/* hangup */
	
	
}

#endif /* MODEM_CONTROL */
	
/* rs_mpsc_interrupt() - NOT IMPLEMENTED YET 
 *
 *  This is the main interrupt handler for the MPSC controller.
 *  It mainly handles modem status lines.
 *
 */


/* rs_sdma_interrupt()
 *
 *  This is the main interrupt handler for both SDMA engines.
 *  It only handles the following two receiver interrupts:
 *   - rxbuf: copy all received characters to the tty buffer.
 *   - rxerr: force to clean at least one descriptor and start the
 *            receiver again.
 *  The transmitter interrupts are ignored.
 *
 * Input Variable(s):
 *  int			interrupt number (ignored),
 *  void*		private data (ignored),
 *  struct pt_regs*	CPU register (ignored).
 *
 * Output Variable(s):
 *  None.
 *
 * Returns:
 *  void
 *
 */

static void rs_sdma_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned int events;
	ser_info_t* info;

	/* get and acknowledge */
#if 0 /* XXXX */
	events = gt_read(GT_SDMA_CAUSE);
#else
	{ /* Search rd's for intr reason since can't read CAUSE reg */
		gt_rx_descriptor_t* rd;

		events = 0;

		if ((info = (struct serial_info *)rs_table[0].info) != NULL) {
			rd = info->rd_cur;
#ifdef CONFIG_NOT_CACHE_COHERENT
			invalidate_dcache_range((unsigned long)rd,
						(unsigned long)(rd+1)-1);
#endif

			/* can go infinite XXXX */
			while((rd->cmd_stat & GT_SDMA_DESC_CMDSTAT_OWN) == 0) {
				if (rd->cmd_stat & GT_SDMA_DESC_CMDSTAT_ES) {
					events |= GT_SDMA_SDMA0_RXERR;
				}
				else {
					events |= GT_SDMA_SDMA0_RXBUF;
				}

				rd = __va(rd->next_desc_ptr);
#ifdef CONFIG_NOT_CACHE_COHERENT
				invalidate_dcache_range((unsigned long)rd,
							(unsigned long)(rd+1)-1);
#endif
			}
		}

		if ((info = (struct serial_info *)rs_table[1].info) != NULL) {
			rd = info->rd_cur;
#ifdef CONFIG_NOT_CACHE_COHERENT
			invalidate_dcache_range((unsigned long)rd,
						(unsigned long)(rd+1)-1);
#endif

			/* can go infinite XXXX */
			while((rd->cmd_stat & GT_SDMA_DESC_CMDSTAT_OWN) == 0) {
				if (rd->cmd_stat & GT_SDMA_DESC_CMDSTAT_ES) {
					events |= GT_SDMA_SDMA1_RXERR;
				}
				else {
					events |= GT_SDMA_SDMA1_RXBUF;
				}

				rd = __va(rd->next_desc_ptr);
#ifdef CONFIG_NOT_CACHE_COHERENT
				invalidate_dcache_range((unsigned long)rd,
							(unsigned long)(rd+1)-1);
#endif
			}
		}
	}
#endif

	gt_write(GT_SDMA_CAUSE, 0);

#ifdef SERIAL_DEBUG_INTR
	printk("rs_interrupt_single()\n");
#endif
	
	/* process received characters */

	if ((events & GT_SDMA_SDMA0_RXBUF)
	    && (info = (struct serial_info *)rs_table[0].info) != NULL) {
		info->last_active = jiffies;
		receive_chars(info,0, regs);
	}

	/* receiver error: clean at least one descriptor and restart receiver */

	if ((events & GT_SDMA_SDMA0_RXERR)
	    && (info = (struct serial_info *)rs_table[0].info) != NULL) {
		gt_rx_descriptor_t* rd_restart = info->rd_cur;
		receive_chars(info,1,regs);	/* force to clean at least one desc */
		
		gt_write(GT_BASE_SDMA0 + GT_SDMA_SCRDP, (unsigned int)rd_restart); 
		gt_write(GT_BASE_SDMA0 + GT_SDMA_SDCM, 
			     gt_read(GT_BASE_SDMA0+GT_SDMA_SDCM) 
			     | GT_SDMA_SDCM_ERD);
	}

	/* same for UART1 */

	if ((events & GT_SDMA_SDMA1_RXBUF)
	    && (info = (struct serial_info *)rs_table[1].info) != NULL) {
		info->last_active = jiffies;
		receive_chars(info,0,regs); 
	}

	if ((events & GT_SDMA_SDMA1_RXERR)
	    && (info = (struct serial_info *)rs_table[1].info) != NULL) {
		gt_rx_descriptor_t* rd_restart = info->rd_cur;
		receive_chars(info,1,regs);	/* force to clean at least one desc */
		
		gt_write(GT_BASE_SDMA1 + GT_SDMA_SCRDP, (unsigned int)rd_restart); 
		gt_write(GT_BASE_SDMA1 + GT_SDMA_SDCM, 
			     gt_read(GT_BASE_SDMA0+GT_SDMA_SDCM) 
			     | GT_SDMA_SDCM_ERD);
	}
 
#ifdef MODEM_CONTROL
	check_modem_status(info);
#endif /* MODEM_CONTROL */
}

/* ---------------- Here ends the serial interrupt routines ------------------*/

/* do_reial_bh()
 *
 * This routine is used to handle the "bottom half" processing for the
 * serial driver, known also the "software interrupt" processing.
 * This processing is done at the kernel interrupt level, after the
 * rs_interrupt() has returned, BUT WITH INTERRUPTS TURNED ON.  This
 * is where time-consuming activities which can not be done in the
 * interrupt driver proper are done; the interrupt driver schedules
 * them using rs_sched_event(), and they get done here.
 */

static void do_serial_bh(void)
{
	run_task_queue(&tq_serial);
}

static void do_softint(void *private_)
{
	ser_info_t	*info = (ser_info_t *) private_;
	struct tty_struct	*tty;
	

	tty = info->tty;
	if (!tty)
		return;

	if (test_and_clear_bit(RS_EVENT_WRITE_WAKEUP, &info->event)) {
		if ((tty->flags & (1 << TTY_DO_WRITE_WAKEUP)) &&
		    tty->ldisc.write_wakeup)
			(tty->ldisc.write_wakeup)(tty);
		wake_up_interruptible(&tty->write_wait);
	}
}

#ifdef MODEM_CONTROL

/* do_serial_hangup()
 *
 * This routine is called from the scheduler tqueue when the interrupt
 * routine has signalled that a hangup has occurred.  The path of
 * hangup processing is:
 *
 * 	serial interrupt routine -> (scheduler tqueue) ->
 * 	do_serial_hangup() -> tty->hangup() -> rs_hangup()
 * 
 */

static void do_serial_hangup(void *private)
{
	struct serial_info *info = (struct serial_info *) private;
	struct tty_struct *tty;

	printk("hangup\n");
	
	tty = info->tty;
	if (tty)
		tty_hangup(tty);
	MOD_DEC_USE_COUNT;
}

#endif MODEM_CONTROL

/* rs_start()
 *
 * This routine is called before resetting tty->stopped.
 * It enables transmitter and receiver interrupts, as necessary.
 *
 * Input Variable(s):
 *   struct tty_struct*		Pointer to the tty.
 *
 * Output Variable(s):
 *   None.
 *
 * Returns:
 *   void
 */

static int startup(ser_info_t *info)
{
	unsigned long flags;
	unsigned int sdma, mpsc;
	
#if 0
	gt_rx_descriptor_t *rd;
	gt_tx_descriptor_t *td;
#endif

	int i;


	/* return if already initialized */

	if (info->flags & ASYNC_INITIALIZED)
		return 0;

	save_flags(flags); cli();

#ifdef SERIAL_DEBUG_OPEN
	printk("starting up ttys%d ...", info->line);
#endif

#ifdef MODEM_CONTROL
	/* FIXME: clear info-field for the modem control lines */
	info->MCR = 0;
	if (info->tty->termios->c_cflag & CBAUD)
		info->MCR = UART_MCR_DTR | UART_MCR_RTS;
#endif /* MODEM_CONTROL */

	if (info->tty)
		clear_bit(TTY_IO_ERROR, &info->tty->flags);

	sdma = GT_BASE_SDMA0 + info->line * GT_BASE_SDMA_DIST;
	mpsc = GT_BASE_MPSC0 + info->line * GT_BASE_MPSC_DIST;

	/* Stop MPSC */

	//	down(&td_desc_sem);

	rs_transmitter_sync(sdma);
	gt_write(sdma+GT_SDMA_SDCM, GT_SDMA_SDCM_AR | GT_SDMA_SDCM_STD);
	
	for (i=0; i < 1000 && (gt_read(sdma+GT_SDMA_SDCM) & (GT_SDMA_SDCM_AR | GT_SDMA_SDCM_AT)); i++)
		udelay(1000);
			
	change_speed(info); 

	/* clear and enable interrupt */

#if 0 /* XXXX */
	gt_write(GT_SDMA_CAUSE,
		     gt_read(GT_SDMA_CAUSE) 
		     & (info->line == 0? 0xfffffff0 : 0xfffff0ff));
	gt_write(GT_SDMA_MASK, 
		     gt_read(GT_SDMA_MASK) 
		     | (info->line == 0? 0x000f : 0x0f00));
#else
	gt_write(GT_SDMA_CAUSE, 0x00000000);
	gt_write(GT_SDMA_MASK,  0x00000003);
#endif

	/* set flag that the uart is initialized */

	info->flags |= ASYNC_INITIALIZED;

#if 0
	/* clean the receiver and transmitter descriptors */

	rd = info->rd_cur = info->rd_first;

	for (i=0; i<RX_NUM_DESCRIPTORS; i++) {
		rd[i].bufsize = RX_BUF_SIZE;
		rd[i].bytecnt = 0;
		rd[i].cmd_stat = GT_SDMA_DESC_CMDSTAT_EI
			| GT_SDMA_DESC_CMDSTAT_F 
			| GT_SDMA_DESC_CMDSTAT_L 
			| GT_SDMA_DESC_CMDSTAT_OWN;
	}
			
	td = info->td_tail = info->td_first;

	for (i=0; i<TX_NUM_DESCRIPTORS; i++) {
		td[i].bytecnt = 0;
		td[i].shadow = 0;
		td[i].cmd_stat = GT_SDMA_DESC_CMDSTAT_F 
			| GT_SDMA_DESC_CMDSTAT_L;
	}

	/* set receiver and transmitter descriptors */

	gt_write(sdma+GT_SDMA_SCRDP, __pa(rd));
	gt_write(sdma+GT_SDMA_SCTDP, __pa(td));
	gt_write(sdma+GT_SDMA_SFTDP, __pa(td));
#endif
	/* start receiver and go to 'hunt mode' */

	gt_write(sdma+GT_SDMA_SDCM, 
		     gt_read(sdma+GT_SDMA_SDCM) | GT_SDMA_SDCM_ERD | GT_SDMA_SDCM_TXD);

#if 0	 /* XXXX */
	gt_write(mpsc+GT_MPSC_UART_CR, 
XXX		     gt_read(mpsc+GT_MPSC_UART_CR) | GT_MPSC_UART_CR_EH);

#else
	gt_write(mpsc+GT_MPSC_UART_CR, cr_cache | GT_MPSC_UART_CR_EH);
	udelay(100);
#endif
	restore_flags(flags);

	//	up(&td_desc_sem);

	return 0;
}


/* shutdown()
 *
 *  This routine will shutdown a serial port; interrupts are disabled, and
 *  DTR is dropped if the hangup on close termio flag is on.
 *
 * Input Variable(s):
 *  ser_info_t*		tty-info
 *
 * Output Variable(s):
 *  None.
 *
 * Returns:
 *  void
 */

static void shutdown(ser_info_t * info)
{
	unsigned long flags;


	/* return if it has already been shut down */

	if (!(info->flags & ASYNC_INITIALIZED))
		return;

#ifdef SERIAL_DEBUG_OPEN
	printk("Shutting down serial port %d ...", info->line);
#endif

	//	down(&td_desc_sem);

	rs_transmitter_sync(GT_BASE_SDMA0 + info->state->line * GT_BASE_SDMA_DIST);

	save_flags(flags); cli();  

	/* disable and clear interrupts */

#ifdef CONFIG_KGDB
	if (kgdb_debug_port != info->line)
#endif
	{
#if 0 /* XXXX */
		gt_write(GT_SDMA_MASK, 
			     gt_read(GT_SDMA_MASK) 
			     & ~(info->line == 0? 0x000f : 0x0f00));
		gt_write(GT_SDMA_CAUSE,
			     gt_read(GT_SDMA_CAUSE) 
			     & (info->line == 0? 0xfffffff0 : 0xfffff0ff));
#else
	gt_write(GT_SDMA_MASK,  0x00000000);
	gt_write(GT_SDMA_CAUSE, 0x00000000);
#endif
	}
	info->flags &= ~ASYNC_INITIALIZED;

	restore_flags(flags);

	//	up(&td_desc_sem);


	return;
}


/* change_speed()
 *
 *  This routine is called to set the UART divisor registers to match
 *  the specified baud rate for a serial port.
 *
 * Input Variable(s):
 *  ser_info_t*	tty info.
 *
 * Output Variable(s):
 *  None.
 *
 * Returns:
 *  void
 *
 */

static void change_speed(ser_info_t *info)
{
	unsigned int cflag;
	unsigned long flags;
	int i;
	unsigned int mpcr = 0, cr = 0, brg = 0;
	int baud_rate;
	int bits;


	if (!info->tty || !info->tty->termios)
		return;
	cflag = info->tty->termios->c_cflag;
	if (!CONFIGURED_SERIAL_PORT(info))
		return;

	/* byte size and parity */

	switch (cflag & CSIZE) {
	case CS5: mpcr = ((bits=5)-5)<<GT_MPSC_MPCR_CL_SHIFT; break;
	case CS6: mpcr = ((bits=6)-5)<<GT_MPSC_MPCR_CL_SHIFT; break;
	case CS7: mpcr = ((bits=7)-5)<<GT_MPSC_MPCR_CL_SHIFT; break;
	case CS8:
	default:  mpcr = ((bits=8)-5)<<GT_MPSC_MPCR_CL_SHIFT; break;
	}

	if (cflag & CSTOPB) {
		bits++;
		mpcr |= GT_MPSC_MPCR_SBL_2;	/* two stop bits */
	}

	if (cflag & PARENB) {
		bits++;
		cr = GT_MPSC_UART_CR_TEV 
			| GT_MPSC_UART_CR_REV;	/* odd parity */
	}
	
	if (!(cflag & PARODD))
		cr |= GT_MPSC_UART_CR_TPM_EVEN
			| GT_MPSC_UART_CR_RPM_EVEN;	/* change to even */

	/* Determine divisor based on baud rate */
	
	i = cflag & CBAUD;
	if (i >= (sizeof(baud_table)/sizeof(int)))
		baud_rate = 9600;
	else
		baud_rate = baud_table[i];

	/* FIXME: use Bclkin (from MPP) ?? */
       	brg = GT_CONFIG_DEFAULT_BRG_BCR | BRG_CDV(baud_rate);

	info->timeout = (TX_BUF_SIZE*HZ*(bits+1));
	info->timeout += HZ/50;		/* Add .02 seconds of slop */

#ifdef MODEM_CONTROL

	/* FIXME: */
	/* CTS flow control flag and modem status interrupts */
	info->IER &= ~UART_IER_MSI;
	if (info->flags & ASYNC_HARDPPS_CD)
		info->IER |= UART_IER_MSI;
	if (cflag & CRTSCTS) {
		info->flags |= ASYNC_CTS_FLOW;
		info->IER |= UART_IER_MSI;
	} else
		info->flags &= ~ASYNC_CTS_FLOW;
	if (cflag & CLOCAL)
		info->flags &= ~ASYNC_CHECK_CD;
	else {
		info->flags |= ASYNC_CHECK_CD;
		info->IER |= UART_IER_MSI;
	}
	serial_out(info, UART_IER, info->IER);

#endif /* MODEM_CONTROL */

	/*
	 * Set up parity check flag
	 */

#define RELEVANT_IFLAG(iflag) (iflag & (IGNBRK|BRKINT|IGNPAR|PARMRK|INPCK))

	/* FIXME: 	info->read_status_mask = GT_MPSC_MASK_MPSC_MPSCOVR;*/
	if (I_INPCK(info->tty)) /* FIXME: cannot mask this kind of interrupt ?? */
		; /*info->read_status_mask |= xx_SC_FR | xx_SC_PR; */
	if (I_BRKINT(info->tty) || I_PARMRK(info->tty))
		; /* FIXME: cannot mask this kind of interrupt ?? */
	/* info->read_status_mask |= BD_SC_BR; */
	
	/*
	 * Characters to ignore
	 */

	info->ignore_status_mask = 0;
#if 0
	if (I_IGNPAR(info->tty))
		info->ignore_status_mask |= BD_SC_PR | BD_SC_FR;
	if (I_IGNBRK(info->tty)) {
		info->ignore_status_mask |= BD_SC_BR;
		/*
		 * If we're ignore parity and break indicators, ignore 
		 * overruns too.  (For real raw support).
		 */
		if (I_IGNPAR(info->tty))
			info->ignore_status_mask |= BD_SC_OV;
	}
#endif
	/*
	 * !!! ignore all characters if CREAD is not set
	 */
#if 0
	if ((cflag & CREAD) == 0)
		info->read_status_mask &= ~BD_SC_EMPTY;
#endif
	save_flags(flags); cli();

	//gt_write(info->state->brgbase + GT_BRG_BCR, brg );
	//gt_write(info->state->line * mpscbase + GT_MPSC_MPCR, mpcr);
	//gt_write(info->state->mpscbase + GT_MPSC_UART_CR, cr);
	/*	gt_write(GT_BASE_BRG0 + info->state->line * GT_BASE_BRG_DIST 
		+ GT_BRG_BCR, brg ); */
	gt_write(GT_BASE_MPSC0 + info->state->line * GT_BASE_MPSC_DIST 
		     + GT_MPSC_MPCR, mpcr);
	gt_write(GT_BASE_MPSC0 + info->state->line * GT_BASE_MPSC_DIST 
		     + GT_MPSC_UART_CR, cr);
	cr_cache = cr;

	restore_flags(flags);
}


/* rs_put_char()
 *
 *  This routine is called to put one single character over the UART.
 *
 * Input Variable(s):
 *  struct tty_struct*	TTY,
 *  unsigned char	single character.
 *
 * Output Variable(s):
 *  None.
 *
 * Returns:
 *  void
 */

static void rs_put_char(struct tty_struct *tty, unsigned char ch)
{
	ser_info_t *info = (ser_info_t *)tty->driver_data;
	gt_tx_descriptor_t *td;
	unsigned int sdma;


	sdma = GT_BASE_SDMA0 + info->state->line * GT_BASE_SDMA_DIST; 

#ifdef CONFIG_KGDB
	if (info->state->line == kgdb_debug_port)
		/* Try to let stub handle output. Returns true if it did. */ 
		if (kgdb_output_string(&ch, 1))
			return;
#endif

	if (serial_paranoia_check(info, tty->device, "rs_put_char"))
		return;

	if (!tty)
		return;

	/* wait for transmitter to finish his last job */
	/* needs to be done here! */

	down(&td_desc_sem);

	rs_transmitter_sync(sdma);

	td = info->td_tail;

	/* no free buffer available */
#ifdef CONFIG_NOT_CACHE_COHERENT
	invalidate_dcache_range((unsigned long)td,
			(unsigned long)(td+1)-1);
#endif
		
	if (td->cmd_stat & GT_SDMA_DESC_CMDSTAT_OWN) {
		info->flags |= TX_WAKEUP;
	       	up(&td_desc_sem);
		return;
	}

	info->td_tail = __va(td->next_desc_ptr);

	/* put the character to the buffer */

	*(unsigned char*)((unsigned long)__va(td->buffer) ^ 7) = ch;

	td->bytecnt = 1;
	td->shadow = 1;

	td->cmd_stat = 
		GT_SDMA_DESC_CMDSTAT_F
		| GT_SDMA_DESC_CMDSTAT_L
		| GT_SDMA_DESC_CMDSTAT_OWN;


	/* start transmitter */

#ifdef CONFIG_NOT_CACHE_COHERENT
			flush_dcache_range((unsigned long)__va(td->buffer),
				(unsigned long)__va(td->buffer) + TX_BUF_SIZE);
			flush_dcache_range((unsigned long)td,
					(unsigned long)(td + 1) - 1);
#endif
	mb();

	gt_write(sdma+GT_SDMA_SCTDP, __pa(td));
	gt_write(sdma+GT_SDMA_SFTDP, __pa(td)); 
	gt_write(sdma+GT_SDMA_SDCM,  
		     gt_read(sdma+GT_SDMA_SDCM) | GT_SDMA_SDCM_TXD);

	up(&td_desc_sem);

	return;
}


/*
 * This routine is called to flush the buffer(s).
 */

static void rs_flush_chars(struct tty_struct *tty)
{
	ser_info_t *info = (ser_info_t *)tty->driver_data;
				

	if (serial_paranoia_check(info, tty->device, "rs_flush_buffer"))
		return;

	/* There is nothing to "flush", whatever we gave the MPSC
	 * is on its way out. */
	
	wake_up_interruptible(&tty->write_wait);
	if ((tty->flags & (1 << TTY_DO_WRITE_WAKEUP)) &&
	    tty->ldisc.write_wakeup)
		(tty->ldisc.write_wakeup)(tty);
	info->flags &= ~TX_WAKEUP;
}

/* rs_write()
 *
 *  This routine is called to write a string to the UART.
 *
 * Input Variable(s):
 *  struct tty_struct*	TTY,
 *  int			1: user space, 0: kernel space,
 *  const uchar*	pointer to the buffer that holds the string,
 *  int			lenght.
 *
 * Output Variable(s):
 *  None.
 *
 * Returns:
 *  number of characters send, or -<error> if an error occured.
 */

static int rs_write(struct tty_struct * tty, int from_user,
				 const unsigned char *buf, int count)
{
	gt_tx_descriptor_t *td_cur, *td_head;
	int ch, i, c;

	ser_info_t *info = (ser_info_t *)tty->driver_data;
	unsigned int sdma = GT_BASE_SDMA0 + info->state->line * GT_BASE_SDMA_DIST;
	int ret = 0;


#ifdef SERIAL_RS_WRITE_DEBUG
	printk("rs_write (tty 0x%x from_user %d buf 0x%x count 0x%x)\n",
			tty, from_user, buf, count);
#endif

#ifdef CONFIG_KGDB
	if (info->state->line == kgdb_debug_port)
		/* Try to let stub handle output. Returns true if it did. */ 
		if (kgdb_output_string(buf, count))
			return ret;
#endif

	if (serial_paranoia_check(info, tty->device, "rs_write"))
		return 0;

	if (!tty) 
		return 0;

	/* we have to wait for the last transmision to be finished
	 * before we fill a new descriptor */

	down(&td_desc_sem);

	rs_transmitter_sync(sdma);

	td_head = td_cur = info->td_tail;

	/* fill up as many buffers as we can */

	while (1) {

		c = MIN(count, TX_BUF_SIZE);
		
		/* are we done? */

		if (c <= 0)
			break;
		
		/* there must always be one unused descriptor 
		 * between the last and the first one */
#ifdef CONFIG_NOT_CACHE_COHERENT
		invalidate_dcache_range((unsigned long)td_cur,
					(unsigned long)(td_cur+1)-1);
#endif
		
		if ((td_cur->cmd_stat & GT_SDMA_DESC_CMDSTAT_OWN)
		    || (__va(td_cur->next_desc_ptr) == td_head)) {
			info->flags |= TX_WAKEUP;
			break;
		}

		/* copy characters to buffer */

		if (from_user) {
			for (i=0; i<c; i++) {
				if (get_user(ch, buf+i) != 0) {
					ret = -EFAULT;
					break;
				}
				*(char*)((unsigned long)__va(td_cur->buffer+i) ^ 7) = 
					(char) ch;
			}
		} else {
			for (i=0; i<c; i++)
				*(char*)((unsigned long)__va(td_cur->buffer+i) ^ 7) = 
					*(char*)(buf+i);
		}

		/* setup descriptor */

		td_cur->bytecnt = c;
		td_cur->shadow = c;
		if (td_cur != td_head)
			td_cur->cmd_stat = GT_SDMA_DESC_CMDSTAT_OWN
				| GT_SDMA_DESC_CMDSTAT_L
				| GT_SDMA_DESC_CMDSTAT_F;
		else
			td_cur->cmd_stat = 
				GT_SDMA_DESC_CMDSTAT_F | GT_SDMA_DESC_CMDSTAT_L;
		
#ifdef CONFIG_NOT_CACHE_COHERENT
		flush_dcache_range((unsigned long)td_cur,
			(unsigned long)(td_cur+1)-1);
		flush_dcache_range((unsigned long)__va(td_cur->buffer),
			(unsigned long)__va(td_cur->buffer) + TX_BUF_SIZE - 1);
#endif
		
		buf += c;
		count -= c;
		ret += c;
		
		td_cur = __va(td_cur->next_desc_ptr);
	}
	

	if (td_cur != td_head) {

		/* set new td_tail in info-structure */
		td_head->cmd_stat |= 
			(GT_SDMA_DESC_CMDSTAT_F | GT_SDMA_DESC_CMDSTAT_OWN);
		info->td_tail = td_cur;

		/* restart transmitter */

#ifdef CONFIG_NOT_CACHE_COHERENT
		flush_dcache_range((unsigned long)td_head,
				(unsigned long)(td_head+1)-1);
#endif
		mb();	
		gt_write(sdma+GT_SDMA_SCTDP, __pa(td_head));
		gt_write(sdma+GT_SDMA_SFTDP, __pa(td_head)); 
		gt_write(sdma+GT_SDMA_SDCM, 
			     gt_read(sdma+GT_SDMA_SDCM) | GT_SDMA_SDCM_TXD);
		
	}

	up(&td_desc_sem);

	return ret;
}

/* rs_write_room()
 *
 *  Returns how much space is available. We only make a quick check of one buffer.
 *
 * Input Variable(s):
 *  struct tty_struct*	TTY,
 *
 * Output Variable(s):
 *  None.
 *
 * Returns:
 *  Amount of free space in the buffer.
 *
 */

static int rs_write_room(struct tty_struct *tty)
{
	ser_info_t *info = (ser_info_t *)tty->driver_data;
	gt_tx_descriptor_t *td, *tdh;
	int ret = 0;


	td = tdh = info->td_tail;

	if (serial_paranoia_check(info, tty->device, "rs_write_room"))
		return 0;

	info->flags |= TX_WAKEUP;

	while((td->cmd_stat & GT_SDMA_DESC_CMDSTAT_OWN) == 0) {
		info->flags &= ~TX_WAKEUP;
		ret += TX_BUF_SIZE;
		if ((td = __va(td->next_desc_ptr)) == tdh)
			break;
	}

	return ret;
}

/* rs_chars_in_buffer()
 *
 *  This functions returns the number of characters that are in the buffer.
 *
 * Input Variable(s):
 *  struct tty_struct*	TTY
 *
 * Output Variable(s):
 *  None.
 *
 * Returns:
 *  Number of characters in the receiver buffer(s).
 *
 */

static int rs_chars_in_buffer(struct tty_struct *tty)
{
	ser_info_t *info = (ser_info_t *)tty->driver_data;
	gt_rx_descriptor_t *rd = info->rd_cur;
	int ret = 0;


	if (serial_paranoia_check(info, tty->device, "rs_chars_in_buffer"))
		return 0;
	
#ifdef CONFIG_NOT_CACHE_COHERENT
	invalidate_dcache_range((unsigned long)rd,
			(unsigned long)(rd + 1) - 1);
#endif
	while((rd->cmd_stat & GT_SDMA_DESC_CMDSTAT_OWN) == 0) {
		ret += rd->bytecnt;
		rd = __va(rd->next_desc_ptr);
#ifdef CONFIG_NOT_CACHE_COHERENT
		invalidate_dcache_range((unsigned long)rd,
				(unsigned long)(rd + 1) - 1);
#endif
	}

	return 0;
}

/* rs_flush_buffer()
 *
 */
/* FIXME: what does it do?? how do we implement this, flush all buffer(s) ?? */

static void rs_flush_buffer(struct tty_struct *tty)
{
	ser_info_t *info = (ser_info_t *)tty->driver_data;
	

	if (serial_paranoia_check(info, tty->device, "rs_flush_buffer"))
		return;
	
	/* There is nothing to "flush", whatever we gave the SDMA/MPSC
	 * is on its way out. */
	
	wake_up_interruptible(&tty->write_wait);
	if ((tty->flags & (1 << TTY_DO_WRITE_WAKEUP)) &&
	    tty->ldisc.write_wakeup)
		(tty->ldisc.write_wakeup)(tty);
	info->flags &= ~TX_WAKEUP;
}


/* rs_send_xchar()
 *
 *  This function is used to send a high-priority XON/XOFF character to
 *  the device
 *
 *
 */

static void rs_send_xchar(struct tty_struct *tty, char ch)
{
	ser_info_t *info = (ser_info_t *)tty->driver_data;
	unsigned int mpscbase;

	printk("send xchar\n");
	return;

	mpscbase = GT_BASE_MPSC0 + info->state->line * GT_BASE_MPSC_DIST;

	gt_write(mpscbase + GT_MPSC_UART_UBSR,
		     (ubsr_cache & GT_MPSC_UART_UBSR_TCS_MASK)
		     | (unsigned int) ch);

	gt_write(mpscbase + GT_MPSC_UART_CR,
		     cr_cache | GT_MPSC_UART_CR_TCS);
}

/* rs_throttle()
 * 
 * This routine is called by the upper-layer tty layer to signal that
 * incoming characters should be throttled.
 */

static void rs_throttle(struct tty_struct * tty)
{
	ser_info_t *info = (ser_info_t *)tty->driver_data;

	printk("throttle\n");

	return;

#ifdef SERIAL_DEBUG_THROTTLE
	char	buf[64];
	
	printk("throttle %s: %d....\n", _tty_name(tty, buf),
	       tty->ldisc.chars_in_buffer(tty));
#endif
	if (serial_paranoia_check(info, tty->device, "rs_throttle"))
		return;
	
	if (I_IXOFF(tty))
		rs_send_xchar(tty, STOP_CHAR(tty));

#ifdef modem_control
	if (tty->termios->c_cflag & CRTSCTS)
		info->MCR &= ~UART_MCR_RTS;

	cli();
	serial_out(info, UART_MCR, info->MCR);
	sti();
#endif
}

/* rs_unthrottle()
 * 
 * This routine is called by the upper-layer tty layer to signal FIXME: 
 */

static void rs_unthrottle(struct tty_struct * tty)
{
	ser_info_t *info = (ser_info_t *)tty->driver_data;

#ifdef SERIAL_DEBUG_THROTTLE
	char	buf[64];
	
	printk("unthrottle %s: %d....\n", _tty_name(tty, buf),
	       tty->ldisc.chars_in_buffer(tty));
#endif

	printk("unthrottle\n");
	return;

	if (serial_paranoia_check(info, tty->device, "rs_unthrottle"))
		return;
	
	if (I_IXOFF(tty)) {
		if (info->x_char)
			info->x_char = 0;
		else
			rs_send_xchar(tty, START_CHAR(tty));
	}
#ifdef modem_control
	if (tty->termios->c_cflag & CRTSCTS)
		info->MCR |= UART_MCR_RTS;
	cli();
	serial_out(info, UART_MCR, info->MCR);
	sti();
#endif
}

/* ------------------------ rs_ioctl() and friends ------------------------- */

static int set_serial_info(struct serial_info * info,
			   struct serial_struct * new_info)
{

	printk("set_ser_inf\n");

	/* FIXME: */

	return 0;
}


/*
 * get_lsr_info - get line status register info
 *
 * Purpose: Let user call ioctl() to get info when the UART physically
 * 	    is emptied.  On bus types like RS485, the transmitter must
 * 	    release the bus after transmitting. This must be done when
 * 	    the transmit shift register is empty, not be done when the
 * 	    transmit holding register is empty.  This functionality
 * 	    allows an RS485 driver to be written in user space. 
 */
static int get_lsr_info(struct serial_info * info, unsigned int *value)
{
	/* FIXME: */
	printk("get_lsr_in\n");
	return 0;
}


static int get_modem_info(struct serial_info * info, unsigned int *value)
{
	/* FIXME: */
	printk("get_mod\n");
	return 0;
}


static int set_modem_info(struct serial_info * info, unsigned int cmd,
			  unsigned int *value)
{
	/* FIXME : */
	printk("set_mopd_inf\n");
	return 0;
}

/* rs_break()
 *
 */

static void send_break(ser_info_t *info, int duration)
{
	printk("send_break\n");
	/* FIXME: */
}


/* rs_ioctl()
 *
 */

static int rs_ioctl(struct tty_struct *tty, struct file * file,
		    unsigned int cmd, unsigned long arg)
{
	int error;
	ser_info_t *info = (ser_info_t *)tty->driver_data;
	int retval;
	struct async_icount cnow;	/* kernel counter temps */
	struct serial_icounter_struct *p_cuser;	/* user space */


	if (serial_paranoia_check(info, tty->device, "rs_ioctl"))
		return -ENODEV;

	if ((cmd != TIOCMIWAIT) && (cmd != TIOCGICOUNT)) {
		if (tty->flags & (1 << TTY_IO_ERROR))
			{
				printk("ioctl error: return -EIO\n");
		    return -EIO;
			}
	}
	
	switch (cmd) {
		case TCSBRK:	/* SVID version: non-zero arg --> no break */
			printk("TCSBRK\n");
			retval = tty_check_change(tty);
			if (retval)
				return retval;
			tty_wait_until_sent(tty, 0);
			if (signal_pending(current))
				return -EINTR;
			if (!arg) {
				send_break(info, HZ/4);	/* 1/4 second */
				if (signal_pending(current))
					return -EINTR;
			}
			return 0;
		case TCSBRKP:	/* support for POSIX tcsendbreak() */
			printk("TCSBRKP\n");
			retval = tty_check_change(tty);
			if (retval)
				return retval;
			tty_wait_until_sent(tty, 0);
			if (signal_pending(current))
				return -EINTR;
			send_break(info, arg ? arg*(HZ/10) : HZ/4);
			if (signal_pending(current))
				return -EINTR;
			return 0;
		case TIOCSBRK:
			printk("TIOCSBRK\n");
			retval = tty_check_change(tty);
			if (retval)
				return retval;
			tty_wait_until_sent(tty, 0);
			//			begin_break(info);
			return 0;
		case TIOCCBRK:
			printk("TIOCCBRKK\n");
			retval = tty_check_change(tty);
			if (retval)
				return retval;
			//			end_break(info);
			return 0;
		case TIOCGSOFTCAR:
			printk("TIOCSOFTCAR\n");

			return put_user(C_CLOCAL(tty) ? 1 : 0, (int *) arg);
		case TIOCSSOFTCAR:
			printk("TIOCSSOFTCAR\n");

			error = get_user(arg, (unsigned int *) arg);
			if (error)
				return error;
			tty->termios->c_cflag =
				((tty->termios->c_cflag & ~CLOCAL) |
				 (arg ? CLOCAL : 0));
			return 0;
		case TIOCMGET:
			printk("TIOCNGET\n");

			return get_modem_info(info, (unsigned int *) arg);
		case TIOCMBIS:
		case TIOCMBIC:
		case TIOCMSET:
			printk("TIOCMBIS/BIC/SET\n");
			return set_modem_info(info, cmd, (unsigned int *) arg);
#ifdef maybe
		case TIOCSERGETLSR: /* Get line status register */

			return get_lsr_info(info, (unsigned int *) arg);
#endif
		/*
		 * Wait for any of the 4 modem inputs (DCD,RI,DSR,CTS) to change
		 * - mask passed in arg for lines of interest
 		 *   (use |'ed TIOCM_RNG/DSR/CD/CTS for masking)
		 * Caller should use TIOCGICOUNT to see which one it was
		 */
		 case TIOCMIWAIT:
			printk("TIOCMIWAIT\n");
#ifdef modem_control
			cli();
			/* note the counters on entry */
			cprev = info->state->icount;
			sti();
			while (1) {
				interruptible_sleep_on(&info->delta_msr_wait);
				/* see if a signal did it */
				if (signal_pending(current))
					return -ERESTARTSYS;
				cli();
				cnow = info->state->icount; /* atomic copy */
				sti();
				if (cnow.rng == cprev.rng && cnow.dsr == cprev.dsr && 
				    cnow.dcd == cprev.dcd && cnow.cts == cprev.cts)
					return -EIO; /* no change => error */
				if ( ((arg & TIOCM_RNG) && (cnow.rng != cprev.rng)) ||
				     ((arg & TIOCM_DSR) && (cnow.dsr != cprev.dsr)) ||
				     ((arg & TIOCM_CD)  && (cnow.dcd != cprev.dcd)) ||
				     ((arg & TIOCM_CTS) && (cnow.cts != cprev.cts)) ) {
					return 0;
				}
				cprev = cnow;
			}
			/* NOTREACHED */
#else
			return 0;
#endif

		/* 
		 * Get counter of input serial line interrupts (DCD,RI,DSR,CTS)
		 * Return: write counters to the user passed counter struct
		 * NB: both 1->0 and 0->1 transitions are counted except for
		 *     RI where only 0->1 is counted.
		 */
		case TIOCGICOUNT:
			printk("TIOCGICOUNT\n");

			cli();
			cnow = info->state->icount;
			sti();
			p_cuser = (struct serial_icounter_struct *) arg;
			error = put_user(cnow.cts, &p_cuser->cts);
			if (error) return error;
			error = put_user(cnow.dsr, &p_cuser->dsr);
			if (error) return error;
			error = put_user(cnow.rng, &p_cuser->rng);
			if (error) return error;
			error = put_user(cnow.dcd, &p_cuser->dcd);
			if (error) return error;
			return 0;

		default:
			return -ENOIOCTLCMD;
		}
	return 0;
}


/* rs_set_termioa()
 *
 */

static void rs_set_termios(struct tty_struct *tty, struct termios *old_termios)
{
	ser_info_t *info = (ser_info_t *)tty->driver_data;


	if (   (tty->termios->c_cflag == old_termios->c_cflag)
	    && (   RELEVANT_IFLAG(tty->termios->c_iflag) 
		== RELEVANT_IFLAG(old_termios->c_iflag)))
	  return;

	change_speed(info);

#ifdef modem_control
	/* Handle transition to B0 status */
	if ((old_termios->c_cflag & CBAUD) &&
	    !(tty->termios->c_cflag & CBAUD)) {
		info->MCR &= ~(UART_MCR_DTR|UART_MCR_RTS);
		cli();
		serial_out(info, UART_MCR, info->MCR);
		sti();
	}
	
	/* Handle transition away from B0 status */
	if (!(old_termios->c_cflag & CBAUD) &&
	    (tty->termios->c_cflag & CBAUD)) {
		info->MCR |= UART_MCR_DTR;
		if (!tty->hw_stopped ||
		    !(tty->termios->c_cflag & CRTSCTS)) {
			info->MCR |= UART_MCR_RTS;
		}
		cli();
		serial_out(info, UART_MCR, info->MCR);
		sti();
	}
	
	/* Handle turning off CRTSCTS */
	if ((old_termios->c_cflag & CRTSCTS) &&
	    !(tty->termios->c_cflag & CRTSCTS)) {
		tty->hw_stopped = 0;
		rs_8xx_start(tty);
	}
#endif

#if 0
	/*
	 * No need to wake up processes in open wait, since they
	 * sample the CLOCAL flag once, and don't recheck it.
	 * XXX  It's not clear whether the current behavior is correct
	 * or not.  Hence, this may change.....
	 */
	if (!(old_termios->c_cflag & CLOCAL) &&
	    (tty->termios->c_cflag & CLOCAL))
		wake_up_interruptible(&info->open_wait);
#endif
}


/* rs_close()
 * 
 * This routine is called when the serial port gets closed.  First, we
 * wait for the last remaining data to be sent.  Then, we unlink its
 * async structure from the interrupt chain if necessary, and we free
 * that IRQ if nothing is left in the chain.
 */
static void rs_close(struct tty_struct *tty, struct file * filp)
{
	ser_info_t *info = (ser_info_t *)tty->driver_data;
	struct serial_state *state;
	unsigned long	flags;


	if (!info || serial_paranoia_check(info, tty->device, "rs_close"))
		return;

	state = info->state;
	
	save_flags(flags); cli();
	
	if (tty_hung_up_p(filp)) {
		/*		DBG_CNT("before DEC-hung"); */
		MOD_DEC_USE_COUNT;
		restore_flags(flags);
		return;
	}
	
#ifdef SERIAL_DEBUG_OPEN
	printk("rs_close ttys%d, count = %d\n", info->line, state->count);
#endif
	if ((tty->count == 1) && (state->count != 1)) {
		/*
		 * Uh, oh.  tty->count is 1, which means that the tty
		 * structure will be freed.  state->count should always
		 * be one in these conditions.  If it's greater than
		 * one, we've got real problems, since it means the
		 * serial port won't be shutdown.
		 */
		printk("rs_close: bad serial port count; tty->count is 1, "
		       "state->count is %d\n", state->count);
		state->count = 1;
	}
	if (--state->count < 0) {
		printk("rs_close: bad serial port count for ttys%d: %d\n",
		       info->line, state->count);
		state->count = 0;
	}
	if (state->count) {
		/*		DBG_CNT("before DEC-2"); */
		MOD_DEC_USE_COUNT;
		restore_flags(flags);
		return;
	}
	info->flags |= ASYNC_CLOSING;
	/*
	 * Save the termios structure, since this port may have
	 * separate termios for callout and dialin.
	 */
	if (info->flags & ASYNC_NORMAL_ACTIVE)
		info->state->normal_termios = *tty->termios;
	if (info->flags & ASYNC_CALLOUT_ACTIVE)
		info->state->callout_termios = *tty->termios;
	/*
	 * Now we wait for the transmit buffer to clear; and we notify 
	 * the line discipline to only process XON/XOFF characters.
	 */
	tty->closing = 1;
	if (info->closing_wait != ASYNC_CLOSING_WAIT_NONE)
		tty_wait_until_sent(tty, info->closing_wait);
	/*
	 * At this point we stop accepting input.  To do this, we
	 * disable the receive line status interrupts, and tell the
	 * interrupt driver to stop checking the data ready bit in the
	 * line status register.
	 */
#if 0
	info->read_status_mask &= ~BD_SC_EMPTY;
	if (info->flags & ASYNC_INITIALIZED) {
		idx = PORT_NUM(info->state->smc_scc_num);
		if (info->state->smc_scc_num & NUM_IS_SCC) {
			sccp = &cpmp->cp_scc[idx];
			sccp->scc_sccm &= ~UART_SCCM_RX;
			sccp->scc_gsmrl &= ~SCC_GSMRL_ENR;
		}
		else {
			smcp = &cpmp->cp_smc[idx];
			smcp->smc_smcm &= ~SMCM_RX;
			smcp->smc_smcmr &= ~SMCMR_REN;
		}
		/*
		 * Before we drop DTR, make sure the UART transmitter
		 * has completely drained; this is especially
		 * important if there is a transmit FIFO!
		 */
		rs_wait_until_sent(tty, info->timeout);
	}
#endif
	shutdown(info);
	if (tty->driver.flush_buffer)
		tty->driver.flush_buffer(tty);
	if (tty->ldisc.flush_buffer)
		tty->ldisc.flush_buffer(tty);
	tty->closing = 0;
	info->event = 0;
	info->tty = 0;
	if (info->blocked_open) {
		if (info->close_delay) {
			current->state = TASK_INTERRUPTIBLE;
			schedule_timeout(info->close_delay);
		}
		wake_up_interruptible(&info->open_wait);
	}
	info->flags &= ~(ASYNC_NORMAL_ACTIVE|ASYNC_CALLOUT_ACTIVE|
			 ASYNC_CLOSING);
	wake_up_interruptible(&info->close_wait);
	MOD_DEC_USE_COUNT;
	restore_flags(flags);
}


/*
 * rs_wait_until_sent() --- wait until the transmitter is empty
 * FIXME!! 
 */
static void rs_wait_until_sent(struct tty_struct *tty, int timeout)
{
	ser_info_t *info = (ser_info_t *)tty->driver_data;
	unsigned long orig_jiffies, char_time;


	if (serial_paranoia_check(info, tty->device, "rs_wait_until_sent"))
		return;

#ifdef maybe
	if (info->state->type == PORT_UNKNOWN)
		return;
#endif

	orig_jiffies = jiffies;

	/*
	 * Set the check interval to be 1/5 of the estimated time to
	 * send a single character, and make it at least 1.  The check
	 * interval should also be less than the timeout.
	 * 
	 * Note: we have to use pretty tight timings here to satisfy
	 * the NIST-PCTS.
	 */
	char_time = 1;
	if (timeout)
		char_time = MIN(char_time, timeout);
#ifdef SERIAL_DEBUG_RS_WAIT_UNTIL_SENT
	printk("In rs_wait_until_sent(%d) check=%lu...", timeout, char_time);
	printk("jiff=%lu...", jiffies);
#endif

	/* We go through the loop at least once because we can't tell
	 * exactly when the last character exits the shifter.  There can
	 * be at least two characters waiting to be sent after the buffers
	 * are empty.
	 */
	do {
#ifdef SERIAL_DEBUG_RS_WAIT_UNTIL_SENT
//		printk("lsr = %d (jiff=%lu)...", lsr, jiffies);
#endif
		current->state = TASK_INTERRUPTIBLE;
/*		current->counter = 0;	 make us low-priority */
		schedule_timeout(char_time);
		if (signal_pending(current))
			break;
		if (timeout && ((orig_jiffies + timeout) < jiffies))
			break;

#ifdef CONFIG_NOT_CACHE_COHERENT
		flush_dcache_range((unsigned long)(info->td_tail),
				(unsigned long)(info->td_tail + 1) - 1);
#endif
	} while (info->td_tail->cmd_stat & GT_SDMA_DESC_CMDSTAT_OWN);

	current->state = TASK_RUNNING;
#ifdef SERIAL_DEBUG_RS_WAIT_UNTIL_SENT
//	printk("lsr = %d (jiff=%lu)...done\n", lsr, jiffies);
#endif
}


/* rs_hangup()
 * 
 * Called by tty_hangup() when a hangup is signaled.
 */

static void rs_hangup(struct tty_struct *tty)
{
	ser_info_t *info = (ser_info_t *)tty->driver_data;
	struct serial_state *state = info->state;

	printk("hangup\n");
	
	if (serial_paranoia_check(info, tty->device, "rs_hangup"))
		return;

	state = info->state;
	
	rs_flush_buffer(tty);
	shutdown(info);
	info->event = 0;
	state->count = 0;
	info->flags &= ~(ASYNC_NORMAL_ACTIVE|ASYNC_CALLOUT_ACTIVE);
	info->tty = 0;
	wake_up_interruptible(&info->open_wait);
}

/* ------------------------- rs_open() and friends ------------------------- */

static int block_til_ready(struct tty_struct *tty, struct file * filp,
			   ser_info_t *info)
{

	printk("bloc_til_re\n");
	return 0;
}


static int get_ser_info_struct(int line, ser_info_t **ret_info)
{
	struct serial_state *sstate;


	sstate = rs_table + line;
	if (sstate->info) {
		sstate->count++;
		*ret_info = (ser_info_t*)sstate->info;
		return 0;
	} else
		return -ENOMEM;

	return 0;
}

/*
 * This routine is called whenever a serial port is opened.  It
 * enables interrupts for a serial port, linking in its async structure into
 * the IRQ chain.   It also performs the serial-specific
 * initialization for the tty structure.
 */

static int rs_open(struct tty_struct *tty, struct file * filp)
{
	ser_info_t	*info;
	int 		retval, line;


	MOD_INC_USE_COUNT;
	line = MINOR(tty->device) - tty->driver.minor_start;
	if ((line < 0) || (line >= NUM_PORTS)) {
		MOD_DEC_USE_COUNT;
		return -ENODEV;
	}

	retval = get_ser_info_struct(line, &info);
	if (retval) {
		printk("get_ser_info < 0\n");
		MOD_DEC_USE_COUNT;
		return retval;
	}

	tty->driver_data = info;
	info->tty = tty;
 
	if (serial_paranoia_check(info, tty->device, "rs_open")) {
		MOD_DEC_USE_COUNT;
		return -ENODEV;
	}

#ifdef SERIAL_DEBUG_OPEN
	printk("rs_open %s%d, count = %d\n", tty->driver.name, info->line,
	       info->state->count);
#endif

#if (LINUX_VERSION_CODE > 0x20100)
	info->tty->low_latency = (info->flags & ASYNC_LOW_LATENCY) ? 1 : 0;
#endif

	/*
	 * Start up serial port
	 */

	retval = startup(info);
	if (retval) {
		printk("statup <0\n");
		MOD_DEC_USE_COUNT;
		return retval;
	}

#if 0
	retval = block_til_ready(tty, filp, info);
	if (retval) {
#ifdef SERIAL_DEBUG_OPEN
		printk("rs_open returning after block_til_ready with %d\n",
		       retval);
#endif
		MOD_DEC_USE_COUNT;
		return retval;
	}
#endif
	if ((info->state->count == 1) &&
	    (info->flags & ASYNC_SPLIT_TERMIOS)) {
		if (tty->driver.subtype == SERIAL_TYPE_NORMAL)
			*tty->termios = info->state->normal_termios;
		else 
			*tty->termios = info->state->callout_termios;
		change_speed(info); 
	}

#ifdef CONFIG_SERIAL_CONSOLE
	if (sercons.cflag && sercons.index == line) {
		tty->termios->c_cflag = sercons.cflag;
		sercons.cflag = 0;
		change_speed(info); 
	}
#endif

	info->session = current->session;
	info->pgrp = current->pgrp;

#ifdef SERIAL_DEBUG_OPEN
	printk("rs_open ttys%d successful...", info->line);
#endif
	return 0;
}


/* -------------------------- /proc fs routines ---------------------------- */

static int inline line_info(char *buf, struct serial_state *state)
{
	int	ret;


	ret = sprintf(buf, "%d: uart\n", state->line);
	return ret;

#ifdef notdef
	/*
	 * Figure out the current RS-232 lines
	 */
	if (!info) {
		info = &scr_info;	/* This is just for serial_{in,out} */

		info->magic = SERIAL_MAGIC;
		info->port = state->port;
		info->flags = state->flags;
		info->quot = 0;
		info->tty = 0;
	}
	cli();
	status = serial_in(info, UART_MSR);
	control = info ? info->MCR : serial_in(info, UART_MCR);
	sti();
	
	stat_buf[0] = 0;
	stat_buf[1] = 0;
	if (control & UART_MCR_RTS)
		strcat(stat_buf, "|RTS");
	if (status & UART_MSR_CTS)
		strcat(stat_buf, "|CTS");
	if (control & UART_MCR_DTR)
		strcat(stat_buf, "|DTR");
	if (status & UART_MSR_DSR)
		strcat(stat_buf, "|DSR");
	if (status & UART_MSR_DCD)
		strcat(stat_buf, "|CD");
	if (status & UART_MSR_RI)
		strcat(stat_buf, "|RI");

	if (info->quot) {
		ret += sprintf(buf+ret, " baud:%d",
			       state->baud_base / info->quot);
	}

	ret += sprintf(buf+ret, " tx:%d rx:%d",
		      state->icount.tx, state->icount.rx);

	if (state->icount.frame)
		ret += sprintf(buf+ret, " fe:%d", state->icount.frame);
	
	if (state->icount.parity)
		ret += sprintf(buf+ret, " pe:%d", state->icount.parity);
	
	if (state->icount.brk)
		ret += sprintf(buf+ret, " brk:%d", state->icount.brk);	

	if (state->icount.overrun)
		ret += sprintf(buf+ret, " oe:%d", state->icount.overrun);

	/*
	 * Last thing is the RS-232 status lines
	 */
	ret += sprintf(buf+ret, " %s\n", stat_buf+1);
#endif
	return ret;
}

static int rs_read_proc(char *page, char **start, off_t off, int count,
		      int *eof, void *data)
{
	int i, len = 0;
	off_t	begin = 0;


	len += sprintf(page, "serinfo:1.0 driver:%s\n", serial_version);
	for (i = 0; i < NUM_PORTS && len < 4000; i++) {
		len += line_info(page + len, &rs_table[i]);
		if (len+begin > off+count)
			goto done;
		if (len+begin < off) {
			begin += len;
			len = 0;
		}
	}
	*eof = 1;
done:
	if (off >= len+begin)
		return 0;
	*start = page + (begin-off);
	return ((count < begin+len-off) ? count : begin+len-off);
}

/* ------------------------ rs_init() and friends -------------------------- */

/* rs_init() is called at boot-time to initialize the serial driver. */

/* show_serial_version()
 *
 * This routine prints out the appropriate serial driver version
 * number, and identifies which options were configured into this
 * driver.
 */

static _INLINE_ void show_serial_version(void)
{
 	printk(KERN_INFO "%s version %s\n", serial_name, serial_version);
}


/* =========================== global functions ============================ */

/*
 * The serial driver boot-time initialization code!
 */

int __init rs_gt64260_mpsc_init(void)
{
	struct serial_state *state;
	ser_info_t	*info;

	dma_addr_t dmah;
	char *mem;
	int size;
	unsigned char *buffer;

	unsigned int sdma;
	unsigned int mpsc;

	gt_rx_descriptor_t *rd;
	gt_tx_descriptor_t *td;

	int retval = 0;	
	int i,l;

	init_bh(SERIAL_BH, do_serial_bh);

	show_serial_version();

	/* Initialize the tty_driver structure */
	
	__clear_user(&serial_driver,sizeof(struct tty_driver));
	serial_driver.magic = TTY_DRIVER_MAGIC;
#if (LINUX_VERSION_CODE > 0x20100)
	serial_driver.driver_name = "serial";
#endif
#if (LINUX_VERSION_CODE > 0x2032D && defined(CONFIG_DEVFS_FS))
	serial_driver.name = "tts/%d"; 
#else
	serial_driver.name = "ttyS"; 
#endif
	serial_driver.major = TTY_MAJOR;
	serial_driver.minor_start = 64;
	serial_driver.name_base = 0;
	serial_driver.num = NUM_PORTS;
	serial_driver.type = TTY_DRIVER_TYPE_SERIAL;
	serial_driver.subtype = SERIAL_TYPE_NORMAL;
	serial_driver.init_termios = tty_std_termios;
	serial_driver.init_termios.c_cflag =
		baud_idx | CS8 | CREAD | HUPCL | CLOCAL;
	serial_driver.flags = TTY_DRIVER_REAL_RAW | TTY_DRIVER_NO_DEVFS;
	serial_driver.refcount = &serial_refcount;
	serial_driver.table = serial_table;
	serial_driver.termios = serial_termios;
	serial_driver.termios_locked = serial_termios_locked;

	serial_driver.open = rs_open;
	serial_driver.close = rs_close;
	serial_driver.write = rs_write;
	serial_driver.put_char = rs_put_char;
	/*	serial_driver.flush_chars = rs_flush_chars; */
	serial_driver.write_room = rs_write_room;
	serial_driver.chars_in_buffer = rs_chars_in_buffer;
	serial_driver.flush_buffer = rs_flush_buffer;
	serial_driver.ioctl = rs_ioctl;
	serial_driver.throttle = rs_throttle;
	serial_driver.unthrottle = rs_unthrottle;
	serial_driver.set_termios = rs_set_termios;
	serial_driver.stop = rs_stop;
	serial_driver.start = rs_start;
	serial_driver.hangup = rs_hangup;
#if (LINUX_VERSION_CODE >= 131394) /* Linux 2.1.66 */
	/*	serial_driver.break_ctl = rs_break; */
#endif
#if (LINUX_VERSION_CODE >= 131343)
	serial_driver.send_xchar = rs_send_xchar;
	serial_driver.wait_until_sent = rs_wait_until_sent;
	serial_driver.read_proc = rs_read_proc;
#endif
	
	/*
	 * The callout device is just like normal device except for
	 * major number and the subtype code.
	 */

	callout_driver = serial_driver;
#if (LINUX_VERSION_CODE > 0x2032D && defined(CONFIG_DEVFS_FS))
	callout_driver.name = "cua/%d"; 
#else
	callout_driver.name = "cua"; 
#endif
	callout_driver.major = TTYAUX_MAJOR;
	callout_driver.subtype = SERIAL_TYPE_CALLOUT;
#if (LINUX_VERSION_CODE >= 131343)
	callout_driver.read_proc = 0;
	callout_driver.proc_entry = 0;
#endif

	if (tty_register_driver(&serial_driver))
		panic("Couldn't register serial driver\n");
	if (tty_register_driver(&callout_driver))
		panic("Couldn't register callout driver\n");

#if 0 /* FIXME: */
	tty_register_devfs(&serial_driver, 0,
			   serial_driver.minor_start + state->line);
	tty_register_devfs(&callout_driver, 0,
			   callout_driver.minor_start + state->line);
#endif

#ifdef CONFIG_SERIAL_CONSOLE
	sdma = GT_BASE_SDMA0 + GT_BASE_SDMA_DIST * CONFIG_SERIAL_CONSOLE_PORT;
	rs_transmitter_sync(sdma);
#endif

	/* initialize the main routing register and port multiplex */
			
	gt_write(GT_MPSC_MRR, GT_CONFIG_DEFAULT_MPSC_MRR);
	gt_write(GT_SERIAL_PORT_MULTIPLEX, 
		     GT_CONFIG_SERIAL_PORT_MULTIPLEX);
	
	/* config the Clock Routing Register for MPSC0 and MPSC1 */
	
	gt_write(GT_MPSC_TCRR, GT_CONFIG_CLOCK_ROUTING_REGISTER);
	gt_write(GT_MPSC_RCRR, GT_CONFIG_CLOCK_ROUTING_REGISTER);

	/* disable interrupts for SDMA0/1 */
	
	gt_write(GT_SDMA_MASK, 0);

	/* initialize all ports */

	for (l = 0, state = rs_table; l < NUM_PORTS; l++,state++) {
		state->magic = SSTATE_MAGIC;
		state->line = l;
		state->type = PORT_UNKNOWN;
		state->custom_divisor = 0;
		state->close_delay = 5*HZ/10;
		state->closing_wait = 30*HZ;
		state->callout_termios = callout_driver.init_termios;
		state->normal_termios = serial_driver.init_termios;
		state->icount.cts = state->icount.dsr = 
			state->icount.rng = state->icount.dcd = 0;
		state->icount.rx = state->icount.tx = 0;
		state->icount.frame = state->icount.parity = 0;
		state->icount.overrun = state->icount.brk = 0;
		printk(KERN_INFO "ttyS%02d\n", l);

		sdma = GT_BASE_SDMA0 + GT_BASE_SDMA_DIST * l;
		mpsc = GT_BASE_MPSC0 + GT_BASE_MPSC_DIST * l;

#ifdef CONFIG_SERIAL_CONSOLE
		/* If we just printed the message on the console port,
		 * and we are bout to initialize it for general use,
		 * we have to wait for the transmitter to be finished 
		 */
		if (i == CONFIG_SERIAL_CONSOLE_PORT)
			rs_transmitter_sync(sdma);
#endif

		info = kmalloc(sizeof(ser_info_t), GFP_KERNEL);
		if (info) {
			memset(info, 0, sizeof(ser_info_t));
			init_waitqueue_head(&info->open_wait);
			init_waitqueue_head(&info->close_wait);
			/*	init_waitqueue_head(&info->delta_msr_wait); */
			info->magic = SERIAL_MAGIC;
			info->flags = state->flags;
			info->line = l;
			info->tqueue.routine = do_softint;
			info->tqueue.data = info;
			info->state = state;
			state->info = (struct async_struct *)info;

			/* allocate memory for the tx/rx descriptors and set them up */
			
			size = (sizeof(gt_rx_descriptor_t)+((RX_BUF_SIZE+15)&~15))*RX_NUM_DESCRIPTORS+
				(sizeof(gt_tx_descriptor_t)+((TX_BUF_SIZE+15)&~15))*TX_NUM_DESCRIPTORS;
			
			if ((mem = pci_alloc_consistent(NULL, size, &dmah)) == NULL)
				break; 	/* Outch! */

			info->dmah = dmah;
			info->mem = mem;
			info->size = size;

			/* initialize all descriptors */

			rd = (gt_rx_descriptor_t*)mem;
			td = (gt_tx_descriptor_t*)(mem + sizeof(gt_rx_descriptor_t) * RX_NUM_DESCRIPTORS);

			info->rd_first = info->rd_cur = rd;
			info->td_first = info->td_tail = td;
			
			buffer = mem
				+ sizeof(gt_rx_descriptor_t) * RX_NUM_DESCRIPTORS
				+ sizeof(gt_tx_descriptor_t) * TX_NUM_DESCRIPTORS;
			
			for (i=0; i<RX_NUM_DESCRIPTORS; i++) {
				rd[i].bufsize = RX_BUF_SIZE;
				rd[i].bytecnt = 0;
				rd[i].buffer = __pa(buffer);
				rd[i].cmd_stat = GT_SDMA_DESC_CMDSTAT_EI
					| GT_SDMA_DESC_CMDSTAT_F 
					| GT_SDMA_DESC_CMDSTAT_L 
					| GT_SDMA_DESC_CMDSTAT_OWN;
				rd[i].next_desc_ptr = __pa(&rd[i+1]);
				buffer += (RX_BUF_SIZE+15)&~15;
			}
			
			for (i=0; i<TX_NUM_DESCRIPTORS; i++) {
				td[i].bytecnt = 0;
				td[i].shadow = 0;
				td[i].buffer = __pa(buffer);
				td[i].cmd_stat = GT_SDMA_DESC_CMDSTAT_F 
					| GT_SDMA_DESC_CMDSTAT_L;
				/* GT_SDMA_DESC_CMDSTAT_EI */
				td[i].next_desc_ptr = __pa(&td[i+1]);
				buffer += (TX_BUF_SIZE+15)&~15;
			}
			
			/* make it a cyclic ring */
			
			rd[RX_NUM_DESCRIPTORS-1].next_desc_ptr = __pa(rd);
			td[TX_NUM_DESCRIPTORS-1].next_desc_ptr = __pa(td);

#ifdef CONFIG_NOT_CACHE_COHERENT
			flush_dcache_range((unsigned long)rd,
					(unsigned long)(rd + 1) - 1);
			flush_dcache_range((unsigned long)td,
					(unsigned long)(td + 1) - 1);
#endif
			mb();	
			/* Stop MPSC */

			gt_write(sdma+GT_SDMA_SDCM, GT_SDMA_SDCM_AR | GT_SDMA_SDCM_STD);

			for (i=0; i < 1000 && (gt_read(sdma+GT_SDMA_SDCM) & (GT_SDMA_SDCM_AR | GT_SDMA_SDCM_AT)); i++)
				udelay(1000);
			
			/* initialize the MPSC */

			gt_write(mpsc + GT_MPSC_MMCRL, GT_CONFIG_DEFAULT_UART_MMCRL);
			gt_write(mpsc + GT_MPSC_MMCRH, GT_CONFIG_DEFAULT_UART_MMCRH);
			gt_write(mpsc + GT_MPSC_UART_UBSR, GT_CONFIG_DEFAULT_UART_UBSR);
			ubsr_cache = GT_CONFIG_DEFAULT_UART_UBSR;
			gt_write(mpsc + GT_MPSC_UART_AR, GT_CONFIG_DEFAULT_UART_AR);
			gt_write(mpsc + GT_MPSC_UART_ESR, GT_CONFIG_DEFAULT_UART_ESR);
			gt_write(mpsc + GT_MPSC_UART_MIR, GT_CONFIG_DEFAULT_UART_MIR);
			gt_write(mpsc + GT_MPSC_CHR4, GT_CONFIG_DEFAULT_UART_CHR);
			gt_write(mpsc + GT_MPSC_CHR5, GT_CONFIG_DEFAULT_UART_CHR);
			gt_write(mpsc + GT_MPSC_CHR6, GT_CONFIG_DEFAULT_UART_CHR);
			gt_write(mpsc + GT_MPSC_CHR7, GT_CONFIG_DEFAULT_UART_CHR);
			gt_write(mpsc + GT_MPSC_CHR8, GT_CONFIG_DEFAULT_UART_CHR);
			
			gt_write(sdma + GT_SDMA_SDC, GT_SDMA_SDC_RFT | GT_SDMA_SDC_SFM
				     | GT_SDMA_SDC_BLMR | GT_SDMA_SDC_BLMT);

			/* write the pointer to CDP */
			
			gt_write(sdma+GT_SDMA_SCRDP, __pa(info->rd_cur));
			gt_write(sdma+GT_SDMA_SCTDP, __pa(info->td_tail));
			gt_write(sdma+GT_SDMA_SFTDP, __pa(info->td_tail));

			gt_write(sdma+GT_SDMA_SDCM, GT_SDMA_SDCM_ERD);

#if 0	 /* XXXX */
			gt_write(mpsc+GT_MPSC_UART_CR, 
XXXX				     gt_read(mpsc+GT_MPSC_UART_CR) | GT_MPSC_UART_CR_EH);
#else
			gt_write(mpsc+GT_MPSC_UART_CR,
					  cr_cache | GT_MPSC_UART_CR_EH);
			udelay(100);
#endif
		}
	}
	retval = request_irq(GT64260_IRQ_SDMA, rs_sdma_interrupt,SA_SHIRQ, "MPSC UART", NULL);

#ifdef CONFIG_KGDB

	/* clear and enable interrupt for the kgdb port though the tty has not been initialized */

#if 0 /* XXXX */
	gt_write(GT_SDMA_CAUSE,
		     gt_read(GT_SDMA_CAUSE) 
		     & (kgdb_debug_port == 0? 0xfffffff0 : 0xfffff0ff));
	gt_write(GT_SDMA_MASK, 
		     gt_read(GT_SDMA_MASK) 
		     | (kgdb_debug_port == 0? 0x000f : 0x0f00));
#else
	gt_write(GT_SDMA_CAUSE, 0x00000000);
	gt_write(GT_SDMA_MASK,  0x00000003);
#endif

#endif /* CONFIG_KGDB */

	return retval;
}


#ifdef RS_FINI_IMPLEMENTED


static void __exit rs_gt64260_mpsc_fini()
{
	unsigned long flags;


	save_flags(flags); cli();
	remove_bh(SERIAL_BH);

	if ((ret = tty_unregister_driver(&serial_driver)))
		printk("serial: failed to unregister serial driver (%d)\n",
		       ret);
	if ((ret = tty_unregister_driver(&callout_driver)))
		printk("serial: failed to unregister callout driver (%d)\n", 
		       ret);
	restore_flags(flags);
	

	for (i=0; i < NR_PORTS; i++) {

		/* free memory */
	
		pci_free_consistent(NULL, info->size, info->mem, info->dmah); 
		info->mem = NULL; 

		/* free info structure */

		if ((info = rs_table[i].info)) {
			rs_table[i].info = NULL;
			kfree(info);
		}
	}
}


module_init(rs_gt64260_mpsc_init);
module_exit(rs_gt64260_mpsc_fini);
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("GT64260 MPSC serial driver");
MODULE_AUTHOR("Chris Zankel <chris@mvista.com>");

#endif


/* ============================ serial console ============================ */

#ifdef CONFIG_SERIAL_CONSOLE

/*
 * The serial console driver used during boot.  Note that these names
 * clash with those found in "serial.c", so we currently can't support
 * the 16xxx uarts and these at the same time.  I will fix this to become
 * an indirect function call from tty_io.c (or something).
 */


static void serial_console_setup_descriptors(int port)
{
	gt_rx_descriptor_t *rd;
	gt_tx_descriptor_t *td;

	sercon_rd[port] = rd = (gt_rx_descriptor_t*)sercon_rxbuffer[port];
	sercon_td[port] = td = (gt_tx_descriptor_t*)sercon_txbuffer[port];
	
	/* setup receiver and transmit descriptors and buffers */

	rd->bufsize = RX_BUF_SIZE;
	rd->bytecnt = 0;
	rd->buffer = __pa((char*)&rd[1]);
	rd->cmd_stat = GT_SDMA_DESC_CMDSTAT_F 
		| GT_SDMA_DESC_CMDSTAT_L 
		| GT_SDMA_DESC_CMDSTAT_OWN;
	rd->next_desc_ptr = __pa(rd);

#ifdef CONFIG_NOT_CACHE_COHERENT
	flush_dcache_range((unsigned long)rd,
			   (unsigned long)(rd + 1) - 1);
#endif

	td->bytecnt = 0;
	td->shadow = 0;
	td->buffer = __pa((char*)&td[1]);
	td->cmd_stat = GT_SDMA_DESC_CMDSTAT_F 
		| GT_SDMA_DESC_CMDSTAT_L;
	td->next_desc_ptr = __pa(td);

#ifdef CONFIG_NOT_CACHE_COHERENT
	flush_dcache_range((unsigned long)td,
			   (unsigned long)(td + 1) - 1);
#endif
}

/* serial_console_write()
 *
 *  Print a string to the serial port trying not to disturb any possible
 *  real use of the port.
 *
 * Input Variable(s):
 *  struct console*	Console,
 *  const char*		String,
 *  unsigned		Length.
 *
 * Output Variable(s):
 *  None.
 *
 * Returns:
 *  void
 *
 */

static void __serial_console_write(int port, const char *s, unsigned count)
{
	/*	struct serial_state *ser = rs_table + port; */
	gt_tx_descriptor_t *td;
	unsigned int i;
	unsigned int sdma, mpsc;
	volatile u_char	ch;

#if 0
	/*
	if ((info = (ser_info_t *)ser->info)) 
	td = info->td_tail;
	else
	*/
#endif

	td = sercon_td[port];
	
	sdma = GT_BASE_SDMA0 + port * GT_BASE_SDMA_DIST;
	mpsc = GT_BASE_MPSC0 + port * GT_BASE_MPSC_DIST;
	
	ch = *s++;

	while (count >0) {

		/* wait until transmitter has stopped */
	      
		if (rs_transmitter_sync(sdma) == 0)
			return;

#ifdef CONFIG_NOT_CACHE_COHERENT
		invalidate_dcache_range((unsigned long)td,
					(unsigned long)(td+1)-1);
#endif
				
		/* We assume we have a free buffer now */
		
		/* fill buffer */
		
		for (i=0; i < TX_BUF_SIZE && count > 0; i++) {
			*((unsigned char *)__va((td->buffer+i) ^ 7)) = ch; 
			if (ch != 10) {
				/* if a LF, also do CR... */
				ch = *s++; count--;
			} else {
				ch = 13;
			}
		}
		
#ifdef CONFIG_NOT_CACHE_COHERENT
		flush_dcache_range((unsigned long)__va(td->buffer),
			(unsigned long)__va(td->buffer) + TX_BUF_SIZE - 1);
#endif
		
		td->bytecnt = i;
		td->shadow = i;
		td->cmd_stat = GT_SDMA_DESC_CMDSTAT_L
			| GT_SDMA_DESC_CMDSTAT_F
			| GT_SDMA_DESC_CMDSTAT_OWN;
		
#ifdef CONFIG_NOT_CACHE_COHERENT
		flush_dcache_range((unsigned long)td,
				   (unsigned long)(td+1) - 1);
#endif
		mb();
		/* start transmitter */
		
		gt_write(sdma+GT_SDMA_SCTDP, __pa(td));
		gt_write(sdma+GT_SDMA_SFTDP, __pa(td));
		
		gt_write(sdma+GT_SDMA_SDCM,
			     gt_read(sdma+GT_SDMA_SDCM) | GT_SDMA_SDCM_TXD);

		td = __va(td->next_desc_ptr);
	}

	sercon_td[port] = td;

	rs_transmitter_sync(sdma);
}

static void serial_console_write(struct console *c, const char *s,
				 unsigned count)
{
#ifdef CONFIG_KGDB
	if (c->index == kgdb_debug_port)
		/* Try to let stub handle output. Returns true if it did. */ 
		if (kgdb_output_string(s, count))
			return;
#endif
	__serial_console_write(c->index, s, count);
}

/*
 * Receive character from the serial port.  This only works well
 * before the port is initialized for real use.
 */

static int __serial_console_wait_key(int port, int xmon, char *obuf)
{
	/* XXX Errata on 64260A--can't read MPSC regs */
#if 0
	struct serial_state *ser = rs_table + port;
	ser_info_t *info;
	gt_rx_descriptor_t* rd;
	unsigned char *cp;
	int i, cnt;


	if ((info = (ser_info_t *)ser->info))
		rd = info->rd_cur;
	else
		rd = sercon_rd[port];

	/* XMON just wants a poll.  If no character, return -1, else
	 * return the character.
	 */

	while (1) {

#ifdef CONFIG_NOT_CACHE_COHERENT
		if (!info)
			invalidate_dcache_range((unsigned long)rd,
						(unsigned long)(rd+1) - 1);
#endif
		if ((rd->cmd_stat & GT_SDMA_DESC_CMDSTAT_OWN) == 0)
			break;

		/* make sure that the receiver is running */

		if (!(gt_read(GT_BASE_SDMA0 + port * GT_BASE_SDMA_DIST + GT_SDMA_SDCM)
		      & GT_SDMA_SDCM_ERD)) {
			gt_write(GT_BASE_SDMA0 + port * GT_BASE_SDMA_DIST + GT_SDMA_SDCM, 
				     gt_read(GT_BASE_SDMA0 + port * GT_BASE_SDMA_DIST + GT_SDMA_SDCM) | GT_SDMA_SDCM_ERD);
#if 0	 /* XXXX */
			gt_write(GT_BASE_MPSC0 + port * GT_BASE_MPSC_DIST + GT_MPSC_UART_CR, 
XXX				     gt_read(GT_BASE_MPSC0 + port * GT_BASE_MPSC_DIST + GT_MPSC_UART_CR) | GT_MPSC_UART_CR_EH);
#else
			udelay(100);
#endif
		}	
		if (xmon)
			return -1;
	}

	cnt = rd->bytecnt;
	cp = __va(rd->buffer);

#ifdef CONFIG_NOT_CACHE_COHERENT
	if (!info)
		invalidate_dcache_range((unsigned long)cp,
					(unsigned long)cp + RX_BUF_SIZE - 1);
#endif

	for (i=cnt; i>0; i--)
		*obuf++ = *(char*)((unsigned long)cp++ ^ 7);
	
	rd->cmd_stat &= ~GT_SDMA_DESC_CMDSTAT_STATUS_MASK;
	rd->cmd_stat |= GT_SDMA_DESC_CMDSTAT_OWN;
	rd->bytecnt = 0;

#ifdef CONFIG_NOT_CACHE_COHERENT
	if (!info)
		flush_dcache_range((unsigned long)rd,
				   (unsigned long)(rd + 1) - 1);
#endif
	mb();
	if (info)
		info->rd_cur = __va(rd->next_desc_ptr);
	else
		sercon_rd[port] = __va(rd->next_desc_ptr);

	if (!(gt_read(GT_BASE_SDMA0 + port * GT_BASE_SDMA_DIST + GT_SDMA_SDCM)
	      & GT_SDMA_SDCM_ERD)) {
		gt_write(GT_BASE_SDMA0 + port * GT_BASE_SDMA_DIST + GT_SDMA_SDCM, 
			     gt_read(GT_BASE_SDMA0 + port * GT_BASE_SDMA_DIST + GT_SDMA_SDCM) | GT_SDMA_SDCM_ERD);
#if 0	 /* XXXX */
		gt_write(GT_BASE_MPSC0 + port * GT_BASE_MPSC_DIST + GT_MPSC_UART_CR, 
XXX			     gt_read(GT_BASE_MPSC0 + port * GT_BASE_MPSC_DIST + GT_MPSC_UART_CR) | GT_MPSC_UART_CR_EH);
#else
		udelay(100);
#endif
	}	
	return((int)cnt);
#else
	return -1;
#endif
}

static int serial_console_wait_key(struct console *c)
{
	return __serial_console_wait_key(c->index, 0, NULL);
}


static kdev_t serial_console_device(struct console *c)
{
	return MKDEV(TTY_MAJOR, 64 + c->index);
}


#ifdef CONFIG_XMON
/* FIXME: Not supported yet */
#endif

#ifdef CONFIG_KGDB

/* initialize the port for kgdb. At this stage the target cannot be interrupted */

void kgdb_map_scc(int port)
{
	unsigned int sdma = GT_BASE_SDMA0 + GT_BASE_SDMA_DIST * port;
	unsigned int mpsc = GT_BASE_MPSC0 + GT_BASE_MPSC_DIST * port;


	kgdb_debug_port = port;
	
	/* initialize the main routing register and port multiplex */
			
	gt_write(GT_MPSC_MRR, GT_CONFIG_DEFAULT_MPSC_MRR);
	gt_write(GT_SERIAL_PORT_MULTIPLEX, 
		     GT_CONFIG_SERIAL_PORT_MULTIPLEX);
	
	/* config the Clock Routing Register for MPSC0 and MPSC1 */
	
	gt_write(GT_MPSC_TCRR, GT_CONFIG_CLOCK_ROUTING_REGISTER);
	gt_write(GT_MPSC_RCRR, GT_CONFIG_CLOCK_ROUTING_REGISTER);

	/* disable interrupts for SDMA0/1 */
	
	gt_write(GT_SDMA_MASK, 0);

	/* setup rx/tx descriptor(s) */

	serial_console_setup_descriptors(port);

	/* write the pointer to CDP */
	
	gt_write(sdma+GT_SDMA_SCRDP, __pa(sercon_rd[port]));
	gt_write(sdma+GT_SDMA_SCTDP, __pa(sercon_td[port]));
	gt_write(sdma+GT_SDMA_SFTDP, __pa(sercon_td[port]));
	
	/* change MPSC to single frame mode */

	/* we use the little endian mode because of a bug (see errata) */
	
	gt_write(sdma + GT_SDMA_SDC, GT_SDMA_SDC_RFT | GT_SDMA_SDC_SFM
		     | GT_SDMA_SDC_BLMR | GT_SDMA_SDC_BLMT);
	
	/* set baud rate (using BCR input) */
	
	/* TODO: fix this mess!! */
#if 0	/* XXXX */
	gt_write(GT_BASE_BRG0 + port * GT_BASE_BRG_DIST + GT_BRG_BCR, 
		     GT_CONFIG_DEFAULT_BRG_BCR 
		     | BRG_CDV(kgdb_baud_rate));
#else
	/* 36 for 133 MHz TCLK
	gt_write(GT64260_BRG_0_BCR, ((8 << 18) | (1 << 16) | 36));
	*/
	/* 27 for 100 MHz TCLK */
	gt_write(GT64260_BRG_0_BCR, ((8 << 18) | (1 << 16) | 27));
#endif
	
	/* initialize the MPSC */
	
	gt_write(mpsc + GT_MPSC_MMCRL, GT_CONFIG_DEFAULT_UART_MMCRL);
	gt_write(mpsc + GT_MPSC_MMCRH, GT_CONFIG_DEFAULT_UART_MMCRH);
	gt_write(mpsc + GT_MPSC_UART_UBSR,GT_CONFIG_DEFAULT_UART_UBSR);
	ubsr_cache = GT_CONFIG_DEFAULT_UART_UBSR;
	gt_write(mpsc + GT_MPSC_UART_AR, GT_CONFIG_DEFAULT_UART_AR);
	gt_write(mpsc + GT_MPSC_UART_ESR, GT_CONFIG_DEFAULT_UART_ESR);
	gt_write(mpsc + GT_MPSC_UART_MIR, GT_CONFIG_DEFAULT_UART_MIR);
	gt_write(mpsc + GT_MPSC_CHR4, GT_CONFIG_DEFAULT_UART_CHR);
	gt_write(mpsc + GT_MPSC_CHR5, GT_CONFIG_DEFAULT_UART_CHR);
	gt_write(mpsc + GT_MPSC_CHR6, GT_CONFIG_DEFAULT_UART_CHR);
	gt_write(mpsc + GT_MPSC_CHR7, GT_CONFIG_DEFAULT_UART_CHR);
	gt_write(mpsc + GT_MPSC_CHR8, GT_CONFIG_DEFAULT_UART_CHR);
	gt_write(mpsc + GT_MPSC_MPCR, GT_CONFIG_DEFAULT_MPSC_MPCR);
	
	/* start receiver */
	
	gt_write(sdma + GT_SDMA_SDCM, GT_SDMA_SDCM_ERD);
#if 0	 /* XXXX */
	gt_write(mpsc+GT_MPSC_UART_CR, 
XXX		     gt_read(mpsc+GT_MPSC_UART_CR) | GT_MPSC_UART_CR_EH);
#else
	gt_write(mpsc+GT_MPSC_UART_CR, cr_cache | GT_MPSC_UART_CR_EH);
	udelay(100);
#endif
	
}

/* putDebugChar()/ getDebugChar()
 *
 *  The putDebugChar() function prints one character at a time. 
 *  getDebugChar() waits for one character and returns it.
 */

void putDebugChar(char ch)
{
	__serial_console_write(kgdb_debug_port, &ch, 1);
}

unsigned char getDebugChar(void)
{
	if (kgdb_chars <= 0) {
		kgdb_chars = __serial_console_wait_key(kgdb_debug_port, 0, kgdb_buf);
		kgdp = kgdb_buf;
	}
	kgdb_chars--;

	return (*kgdp++);
}

void kgdb_interruptible(int yes)
{
	if (yes) {
#if 0 /* XXXX */
		gt_write(GT_SDMA_CAUSE,
			     gt_read(GT_SDMA_CAUSE) 
			     & (kgdb_debug_port == 0? 0xfffffff0 : 0xfffff0ff));
		gt_write(GT_SDMA_MASK, 
			     gt_read(GT_SDMA_MASK) 
			     | (kgdb_debug_port == 0? 0x000f : 0x0f00));
#else
		gt_write(GT_SDMA_CAUSE, 0x00000000);
		gt_write(GT_SDMA_MASK,  0x00000003);
#endif
	} else {
#if 0 /* XXXX */
		gt_write(GT_SDMA_MASK, 
			     gt_read(GT_SDMA_MASK) 
			     & ~(kgdb_debug_port == 0? 0x000f : 0x0f00));
		gt_write(GT_SDMA_CAUSE,
			     gt_read(GT_SDMA_CAUSE) 
			     & (kgdb_debug_port == 0? 0xfffffff0 : 0xfffff0ff));
#else
		gt_write(GT_SDMA_MASK,  0x00000000);
		gt_write(GT_SDMA_CAUSE, 0x00000000);
#endif
	}
}

#endif



/* console_mpsc_init()
 *
 * Register console.
 */

void __init gt64260mpsc_console_init(long kmem_start, long kmem_end)
{
	register_console(&sercons);
#if 0
	return kmem_start;
#endif
}




static int 
__init serial_console_setup(struct console *con, char *options)
{
	struct		serial_state *ser;
	unsigned int	sdma, brg, mpsc;
	int bidx, baud_idx, baud_rate;
	int port = con->index;
	int i;
	
	/* set baud rate */
       
	baud_idx = 13;	/* default baud rate: 9600 */
	for (bidx = 0; bidx < (sizeof(baud_table) / sizeof(int)); bidx++) {
		if (gt64260_mpsc_baud_rate  == baud_table[bidx]) {
			baud_idx = bidx;
			baud_rate = gt64260_mpsc_baud_rate;
			break;
		}
	}
	
	/* default: 8N1 */
	
	con->cflag = CREAD|CLOCAL|bidx|CS8;
	ser = rs_table + port;
	sdma = GT_BASE_SDMA0 + ser->line * GT_BASE_SDMA_DIST;
	brg = GT_BASE_BRG0 + ser->line * GT_BASE_BRG_DIST;
	mpsc = GT_BASE_MPSC0 + ser->line * GT_BASE_MPSC_DIST;
	
	/* Stop MPSC */

	rs_transmitter_sync(sdma);

	gt_write(sdma+GT_SDMA_SDCM, GT_SDMA_SDCM_AR | GT_SDMA_SDCM_STD);
	
	/* XXX do this nicer... */
#if 0
	for (i=0; i < 1000 && (gt_read(sdma+GT_SDMA_SDCM) & (GT_SDMA_SDCM_AR | GT_SDMA_SDCM_AT)); i++)
		udelay(1000);
#else
	mdelay(100); /* just wait a while .. can read the registers (ugh) */
#endif	

	/* initialize the mpsc (main routing register, port multiplex) */
	
	gt_write(GT_MPSC_MRR, GT_CONFIG_DEFAULT_MPSC_MRR);
	gt_write(GT_SERIAL_PORT_MULTIPLEX, 
		     GT_CONFIG_SERIAL_PORT_MULTIPLEX);
	
	/* config the Clock Routing Register for MPSC0 and MPSC1 */
	
	gt_write(GT_MPSC_TCRR, GT_CONFIG_CLOCK_ROUTING_REGISTER);
	gt_write(GT_MPSC_RCRR, GT_CONFIG_CLOCK_ROUTING_REGISTER);

	/* setup descriptors */

	serial_console_setup_descriptors(port);

	/* write the pointer to CDP */
	
	gt_write(sdma+GT_SDMA_SCRDP, __pa(sercon_rd[port]));
	gt_write(sdma+GT_SDMA_SCTDP, __pa(sercon_td[port]));
	gt_write(sdma+GT_SDMA_SFTDP, __pa(sercon_td[port]));

	/* change MPSC to single frame mode */
	/* we use the little endian mode because of a bug (see errata) */
	gt_write(sdma + GT_SDMA_SDC, GT_SDMA_SDC_RFT | GT_SDMA_SDC_SFM
		  | GT_SDMA_SDC_BLMR | GT_SDMA_SDC_BLMT | (3<<12));
	/* set baud rate to 115200 from B_in */

	/* FIXME: hard coding... */
#warning stupid hardcoding
#if 0	/* XXXX */
	gt_write(brg + GT_BRG_BCR, GT_CONFIG_DEFAULT_BRG_BCR 
		     | BRG_CDV(115200) | (8 << 18));
#else
	/* 36 for 133 MHz TCLK
	gt_write(GT64260_BRG_0_BCR, ((8 << 18) | (1 << 16) | 36));
	*/
	/* 27 for 100 MHz TCLK */
	gt_write(GT64260_BRG_0_BCR, ((8 << 18) | (1 << 16) | 27));
#endif

	/* initialize the MPSC machine */

	gt_write(mpsc + GT_MPSC_MMCRL, GT_CONFIG_DEFAULT_UART_MMCRL);
	gt_write(mpsc + GT_MPSC_MMCRH, GT_CONFIG_DEFAULT_UART_MMCRH);
	gt_write(mpsc + GT_MPSC_UART_UBSR,GT_CONFIG_DEFAULT_UART_UBSR);
	ubsr_cache = GT_CONFIG_DEFAULT_UART_UBSR;
	gt_write(mpsc + GT_MPSC_UART_AR, GT_CONFIG_DEFAULT_UART_AR);
	gt_write(mpsc + GT_MPSC_UART_ESR, GT_CONFIG_DEFAULT_UART_ESR);
	gt_write(mpsc + GT_MPSC_UART_MIR, GT_CONFIG_DEFAULT_UART_MIR);
	gt_write(mpsc + GT_MPSC_CHR4, GT_CONFIG_DEFAULT_UART_CHR);
	gt_write(mpsc + GT_MPSC_CHR5, GT_CONFIG_DEFAULT_UART_CHR);
	gt_write(mpsc + GT_MPSC_CHR6, GT_CONFIG_DEFAULT_UART_CHR);
	gt_write(mpsc + GT_MPSC_CHR7, GT_CONFIG_DEFAULT_UART_CHR);
	gt_write(mpsc + GT_MPSC_CHR8, GT_CONFIG_DEFAULT_UART_CHR);

	/* start receiver: go to hunt mode */

	gt_write(sdma + GT_SDMA_SDCM, GT_SDMA_SDCM_ERD);
#if 0	 /* XXXX */
	gt_write(mpsc+GT_MPSC_UART_CR, 
XXX		     gt_read(mpsc+GT_MPSC_UART_CR) | GT_MPSC_UART_CR_EH);
#else
	gt_write(mpsc+GT_MPSC_UART_CR, cr_cache | GT_MPSC_UART_CR_EH);
	udelay(100);
#endif

	return 0;
}

#endif /* CONFIG_SERIAL_CONSOLE */
