/*
 * basil_flash.c 0.1 2002/04/03
 *
 * flash rom mappings on NRS Basil-S1(tx4925)
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/cfi.h>
#include <linux/mtd/flashchip.h>
#include <linux/config.h>
#include <linux/delay.h>

#include <asm/tx4925/tx4925.h>
#include <asm/tx4925/basil_s1/basil_s1.h>

#undef BASIL_FLASH_DEBUG

#if defined(BASIL_FLASH_DEBUG)
#define BASIL_DBG(str...)	{ char tmp[256]; sprintf(tmp, str); printk("%s:%s", __FUNCTION__, tmp); }
#else
#define BASIL_DBG(str...)
#endif	/* defined(BASIL_FLASH_DEBUG) */

#define BASIL_MTD_VERSION		"Ver 1.0"
#define BASIL_DIPSW				TX4925_RD(TX4925_PIO_PIODI)
#define BASIL_ROM_BOOT			(BASIL_DIPSW & (1 << 12))
#define BASIL_CARD_BOOT			(!BASIL_ROM_BOOT)


#define BASIL_FLASHROM_SIZE       0x800000    /* 4MB x 2 */
#define BASIL_FLASHROM_BUSWIDTH   2

#if 0 /* del start: Change mtd1 3MB -> 3.25MB : 2003/04/04T09:30 */
#define BASIL_FLASHROM_ADDR0      0x1FC00000	/* 0x1fc0_0000-0x1fef_ffff */
#define BASIL_FLASHROM_SIZE0      0x00300000    /* 3MB */

#define BASIL_FLASHROM_ADDR1      0x1FF00000	/* 0x1ff0_0000-0x1ff0_ffff */
#define BASIL_FLASHROM_SIZE1      0x00010000    /* 64KB */

/* change start: mtd2 64kb -> 128kb : 2003/01/24T09:32 RSK Hirata */
#define BASIL_FLASHROM_ADDR2      0x1FF10000	/* 0x1ff1_0000-0x1ff2_ffff */
#define BASIL_FLASHROM_SIZE2      0x00020000    /* 128KB */

#if 0 /* del start: Add mtd4 64kB : 2003/02/03T13:13 RSK Hirata */
#define BASIL_FLASHROM_ADDR3      0x1FF30000	/* 0x1ff3_0000-0x1fff_ffff */
#define BASIL_FLASHROM_SIZE3      0x000D0000    /* 832KB */

/* #define BASIL_FLASHROM_ADDR2      0x1FF10000	*//* 0x1ff1_0000-0x1ff1_ffff */
/* #define BASIL_FLASHROM_SIZE2      0x00010000 *//* 64KB */

/* #define BASIL_FLASHROM_ADDR3      0x1FF20000	*//* 0x1ff2_0000-0x1fff_ffff */
/* #define BASIL_FLASHROM_SIZE3      0x000E0000 *//* 896KB */
/* change end : mtd2 64kb -> 128kb : 2003/01/24T09:32 RSK Hirata */

#define BASIL_FLASH_NUM		4
#endif /* del end: Add mtd4 64kB : 2003/02/03T13:13 RSK Hirata */

/* add start: Add mtd4 64kB : 2003/02/03T13:13 RSK Hirata */
#define BASIL_FLASHROM_ADDR3      0x1FF30000    /* 0x1ff3_0000-0x1ffe_ffff */
#define BASIL_FLASHROM_SIZE3      0x000C0000    /* 768KB */
#define BASIL_FLASHROM_ADDR4      0x1FFF0000    /* 0x1fff_0000-0x1fff_ffff */
#define BASIL_FLASHROM_SIZE4      0x00010000    /* 64KB */
#define BASIL_FLASH_NUM         5
/* add end: Add mtd4 64kB : 2003/02/03T13:13 RSK Hirata */

#else

#define BASIL_FLASHROM_ADDR0      0x1FC00000	/* 0x1fc0_0000-0x1ff3_ffff */
#define BASIL_FLASHROM_SIZE0      0x00340000    /* 3.25MB */

#define BASIL_FLASHROM_ADDR1      0x1FF40000	/* 0x1ff4_0000-0x1ff4_ffff */
#define BASIL_FLASHROM_SIZE1      0x00010000    /* 64KB */

#define BASIL_FLASHROM_ADDR2      0x1FF50000	/* 0x1ff5_0000-0x1ff6_ffff */
#define BASIL_FLASHROM_SIZE2      0x00020000    /* 128KB */

#define BASIL_FLASHROM_ADDR3      0x1FF70000    /* 0x1ff7_0000-0x1ffe_ffff */
#define BASIL_FLASHROM_SIZE3      0x00080000    /* 512KB */

#define BASIL_FLASHROM_ADDR4      0x1FFF0000    /* 0x1fff_0000-0x1fff_ffff */
#define BASIL_FLASHROM_SIZE4      0x00010000    /* 64KB */
#define BASIL_FLASH_NUM         5

#endif /* del start: Change mtd1 3MB -> 3.25MB : 2003/04/04T09:30 */

extern struct mtd_info* cfi_probe( struct map_info* );
static struct mtd_info *basil_flash[BASIL_FLASH_NUM];

__u8 flbasil_read8( struct map_info *map, unsigned long ofs )
{
	return *( (volatile __u8*)( map->map_priv_1 + ofs ) );
}

__u16 flbasil_read16( struct map_info *map, unsigned long ofs )
{
BASIL_DBG(" start [%lx]+[%lx] -> d[%lx]\n", (u32)(map->map_priv_1), ofs, *( (volatile __u16*)( map->map_priv_1 + ofs ) ));
	return *( (volatile __u16*)( map->map_priv_1 + ofs ) );
}

__u32 flbasil_read32( struct map_info *map, unsigned long ofs )
{
	return *( (volatile __u32*)( map->map_priv_1 + ofs ) );
}

void flbasil_copy_from(	struct map_info	*map, void *to, unsigned long from, ssize_t len )
{
	memcpy_fromio( to, ( map->map_priv_1 + from ), len );
}

void flbasil_write8( struct map_info *map, __u8 d, unsigned long ofs )
{
	*( (volatile __u8*)( map->map_priv_1 + ofs ) ) = d;
}

void flbasil_write16( struct map_info *map, __u16 d, unsigned long ofs )
{
/* debug by ishi TOSHIBA-IT */
BASIL_DBG(" start [%lx]+[%lx] <- d[%lx]\n", (u32)(map->map_priv_1), ofs, (u32)d);
	*( (volatile __u16*)( map->map_priv_1 + ofs ) ) = d;
}

void flbasil_write32( struct map_info *map, __u32 d, unsigned long ofs )
{
	*( (volatile __u32*)( map->map_priv_1 + ofs ) ) = d;
}

void flbasil_copy_to( struct map_info *map, unsigned long to, const void *from, ssize_t len )
{
	switch ( len ) {
	case 1 :
		flbasil_write8( map, *(__u8*)from, to );
		break;
	case 2 :
		flbasil_write16( map, *(__u16*)from, to );
		break;
	case 4 :
		flbasil_write32( map, *(__u32*)from, to );
		break;
	default :
		memcpy_toio( ( map->map_priv_1 + to ), from, len );
		break;
	}
}


static struct map_info flbasil_map[BASIL_FLASH_NUM] = {
	{
		"Boot Area",              /* name		*/
		BASIL_FLASHROM_SIZE0,     /* size		*/
		BASIL_FLASHROM_BUSWIDTH,  /* buswidth	*/
		flbasil_read8,            /* read8		*/
		flbasil_read16,           /* read16		*/
		flbasil_read32,           /* read32		*/
		flbasil_copy_from,        /* copy_from	*/
		flbasil_write8,           /* write8		*/
		flbasil_write16,          /* write16	*/
		flbasil_write32,          /* write32	*/
		flbasil_copy_to           /* copy_to	*/
	},
	{
		"Kernel Flag",            /* name		*/
		BASIL_FLASHROM_SIZE1,     /* size		*/
		BASIL_FLASHROM_BUSWIDTH,  /* buswidth	*/
		flbasil_read8,            /* read8		*/
		flbasil_read16,           /* read16		*/
		flbasil_read32,           /* read32		*/
		flbasil_copy_from,        /* copy_from	*/
		flbasil_write8,           /* write8		*/
		flbasil_write16,          /* write16	*/
		flbasil_write32,          /* write32	*/
		flbasil_copy_to           /* copy_to	*/
	},
	{
		"Serial Number",          /* name		*/
		BASIL_FLASHROM_SIZE2,     /* size		*/
		BASIL_FLASHROM_BUSWIDTH,  /* buswidth	*/
		flbasil_read8,            /* read8		*/
		flbasil_read16,           /* read16		*/
		flbasil_read32,           /* read32		*/
		flbasil_copy_from,        /* copy_from	*/
		flbasil_write8,           /* write8		*/
		flbasil_write16,          /* write16	*/
		flbasil_write32,          /* write32	*/
		flbasil_copy_to           /* copy_to	*/
	},
	{
		"User Area",              /* name		*/
		BASIL_FLASHROM_SIZE3,     /* size		*/
		BASIL_FLASHROM_BUSWIDTH,  /* buswidth	*/
		flbasil_read8,            /* read8		*/
		flbasil_read16,           /* read16		*/
		flbasil_read32,           /* read32		*/
		flbasil_copy_from,        /* copy_from	*/
		flbasil_write8,           /* write8		*/
		flbasil_write16,          /* write16	*/
		flbasil_write32,          /* write32	*/
		flbasil_copy_to           /* copy_to	*/
	},
/* add start: Add mtd4 64kB : 2003/02/03T13:13 RSK Hirata */
	{
		"Selfcheck Flag",         /* name	*/
		BASIL_FLASHROM_SIZE4,     /* size	*/
		BASIL_FLASHROM_BUSWIDTH,  /* buswidth	*/
		flbasil_read8,            /* read8	*/
		flbasil_read16,           /* read16	*/
		flbasil_read32,           /* read32	*/
		flbasil_copy_from,        /* copy_from	*/
		flbasil_write8,           /* write8	*/
		flbasil_write16,          /* write16	*/
		flbasil_write32,          /* write32	*/
		flbasil_copy_to           /* copy_to	*/
	},
/* add end: Add mtd4 64kB : 2003/02/03T13:13 RSK Hirata */
};


static int flbasil_init( int dev_num )
{
    unsigned long   flash_addr, flash_size, iomap;

	switch ( dev_num ) {
	case 0:
		/* check BOOT device */
		if (IS_CARD_BOOT) {
			/* debug by ishi TOSHIBA-IT */
			BASIL_DBG(" detect card boot\nskip boot rom mapping\n");
			return 0;
		}
	    flash_addr = BASIL_FLASHROM_ADDR0 ;
		break;
	case 1:
	    flash_addr = BASIL_FLASHROM_ADDR1 ;
		break;
	case 2:
	    flash_addr = BASIL_FLASHROM_ADDR2 ;
		break;
	case 3:
	    flash_addr = BASIL_FLASHROM_ADDR3 ;
		break;
/* add start: Add mtd4 64kB : 2003/04/18T10:30 RSK Hirata */
	case 4:
	    flash_addr = BASIL_FLASHROM_ADDR4 ;
		break;
/* add stop : Add mtd4 64kB : 2003/04/18T10:30 RSK Hirata */
	default:
		return -ENXIO;
		break;
	}

	flash_size = flbasil_map[dev_num].size;

	printk( KERN_NOTICE "flbasil flash device%d: %lx at %lx\n",
			dev_num, flash_size, flash_addr );

	iomap = (unsigned long)ioremap( flash_addr, flash_size );
	DEBUG( MTD_DEBUG_LEVEL3, 
		"flash%d: I/O remap  %#8.8x to %#8.8x\n", 
		dev_num, flash_addr, iomap );

	if( !iomap ){
		printk( "flash%d: Failed to ioremap\n", dev_num );
		return -EIO;
	}

	if( iomap ){
		BASIL_DBG("map[%d]", dev_num);
		flbasil_map[dev_num].map_priv_1 = iomap;
#if 0 /* debug by ishi */
		basil_flash[dev_num]
			= cfi_probe( &flbasil_map[dev_num] );
#else
		BASIL_DBG("start do_map_probe(%lx)", (u32)(&flbasil_map[dev_num]))
		basil_flash[dev_num]
			= (struct mtd_info*)do_map_probe("cfi_probe", &flbasil_map[dev_num] );
		basil_flash[dev_num]->size = flash_size;
#endif

		BASIL_DBG("basil_flash[%d] = %lx", dev_num, (u32)(basil_flash[dev_num]))
		if( basil_flash[dev_num] ){
#ifdef MODULE
			basil_flash[dev_num]->module = &__this_module;
#endif
			add_mtd_device( basil_flash[dev_num] );
			BASIL_DBG("flash%d: device add success.\n", dev_num );
		} else {
			iounmap( (void*)flbasil_map[dev_num].map_priv_1 );
			flbasil_map[dev_num].map_priv_1 = 0;
			printk( "flash%d: device add failed...\n", dev_num );

			return -ENXIO;
		}
	}

	return 0;
}

static void flbasil_clean( int dev_num )
{
	if( basil_flash[dev_num] ){
		del_mtd_device( basil_flash[dev_num] );
	        map_destroy( basil_flash[dev_num] );
	}
}


int __init init_flbasil( void )
{
	int cnt, ret;
	int rt = -EIO;

	for (cnt = 0; cnt < BASIL_FLASH_NUM; cnt++) {
		basil_flash[cnt] = NULL;
		ret = flbasil_init( cnt );
		if (ret == 0) {
			rt = 0;
		} else if ((ret != -EIO) && (rt != 0)) {
			rt = -ENXIO;
		}
	}

	if (rt == 0) {
		printk("Basil-S1 MTD Flash driver loaded successfully.\n");
		ROM_WP_OFF;	
	} else {
		printk("Basil-S1 MTD Flash driver loading failed .\n");
	}
	return rt;
}

static void __exit cleanup_flbasil(void)
{
	int cnt;

	ROM_WP_ON;
	for( cnt = 0; cnt < BASIL_FLASH_NUM; cnt++ ){
		flbasil_clean( cnt );
	}
}

module_init(init_flbasil);
module_exit(cleanup_flbasil);


