/*
 * linux/drivers/char/rtc_rx5c348.c
 *
 * driver for RICOH Rx5c348 RTC (use Tx4925 SPI Module)
 *
 */

#include <linux/rtc_rx5c348.h>
#include <asm/tx4925/tx4925.h>
#ifdef CONFIG_RICOH_BASIL_S1
#include <asm/tx4925/basil_s1/basil_s1.h>
#endif

#undef	RTC_RX5C348_DEBUG


static void
rtc_rx5c348_tx4925_spi_init( void )
{
    TX4925_WR(TX4925_SPI_SPMCR, TX4925_SPI_SPMCR_OPMODE_CONFIG);
    TX4925_WR(TX4925_SPI_SPCR0, 0x0000000F);
    TX4925_WR(TX4925_SPI_SPCR1, 0x00001308);
}

static void
rtc_rx5c348_tx4925_spi_active( void )
{
    TX4925_WR(TX4925_SPI_SPMCR, TX4925_SPI_SPMCR_OPMODE_ACTIVE);
}

static unsigned char
rtc_rx5c348_tx4925_spi_read( unsigned char adr )
{
    volatile unsigned long temp;
 
    /* Wait for Idle */
    while ((TX4925_RD(TX4925_SPI_SPSR) & TX4925_SPI_SPSR_SIDLE) == 0);

    /* Clear Receive FIFO */
    while((TX4925_RD(TX4925_SPI_SPSR) & TX4925_SPI_SPSR_SRRDY) != 0) 
        temp = TX4925_RD(TX4925_SPI_SPDR);

    /* Address Pointer & Transfer Format Write */
    temp = (((adr << 4) & 0xf0) | RTC_RX5C348_READ_SINGLE);
    TX4925_WR(TX4925_SPI_SPDR, temp);

    /* Wait for Transmit Ready */
    while ((TX4925_RD(TX4925_SPI_SPSR) & TX4925_SPI_SPSR_STRDY) == 0);

    /* Dummy Write for Data Read */
    TX4925_WR(TX4925_SPI_SPDR, 0);

    /* Wait for Idle */
    while ((TX4925_RD(TX4925_SPI_SPSR) & TX4925_SPI_SPSR_SIDLE) == 0);
    temp = TX4925_RD(TX4925_SPI_SPDR);
    temp = TX4925_RD(TX4925_SPI_SPDR);

    return((unsigned char)(temp & 0xff));
}

static void
rtc_rx5c348_tx4925_spi_write( unsigned char adr, unsigned char data )
{
    volatile unsigned char temp;

    /* Wait for Idle */
    while ((TX4925_RD(TX4925_SPI_SPSR) & TX4925_SPI_SPSR_SIDLE) == 0);

    /* Address Pointer & Transfer Format Write */
    temp = (((adr << 4) & 0xf0) | RTC_RX5C348_WRITE_SINGLE);
    TX4925_WR(TX4925_SPI_SPDR, temp);

    /* Wait for Transmit Ready */
    while ((TX4925_RD(TX4925_SPI_SPSR) & TX4925_SPI_SPSR_STRDY) == 0);

    /* Data Write */
    TX4925_WR(TX4925_SPI_SPDR, data);

    /* Wait for Idle */
    while ((TX4925_RD(TX4925_SPI_SPSR) & TX4925_SPI_SPSR_SIDLE) == 0);
}

static void
rtc_rx5c348_udelay(unsigned long count)
{
    int  i, j;
    volatile unsigned long temp;

    for (i = 0; i < count; i++) {
        for(j = 0; j < 10; j++)
            temp = *(volatile unsigned long *)(0xbfc00000);
    }
}

unsigned long
rtc_rx5c348_get_time( void )
{
    u32 century, year, month, day, hour, minute, second;
    unsigned long t;

    /* Tx4925 SPI Module Init */
    rtc_rx5c348_tx4925_spi_init();

    /* Assert RTC CE */
    RTC_RX5C348_CE_ASSERT;

    /* Few delay */
    rtc_rx5c348_udelay(1);

    rtc_rx5c348_tx4925_spi_active();

    /* read time data */ 
    second  = (u32)rtc_rx5c348_tx4925_spi_read(RTC_RX5C348_SEC_REG);
    minute  = (u32)rtc_rx5c348_tx4925_spi_read(RTC_RX5C348_MIN_REG);
    hour    = (u32)rtc_rx5c348_tx4925_spi_read(RTC_RX5C348_HOUR_REG);
    day     = (u32)rtc_rx5c348_tx4925_spi_read(RTC_RX5C348_MDAY_REG);
    month   = (u32)rtc_rx5c348_tx4925_spi_read(RTC_RX5C348_MONTH_REG);
    year    = (u32)rtc_rx5c348_tx4925_spi_read(RTC_RX5C348_YEAR_REG);
    century = (month & RTC_RX5C348_MONTH_20) ? 20 : 19; 

#ifdef RTC_RX5C348_DEBUG
    printk( "rtc_rx5c348_get_time() -- year  %02X\n", year);
    printk( "rtc_rx5c348_get_time() -- month %02X\n", month);
    printk( "rtc_rx5c348_get_time() -- day   %02X\n", day);
    printk( "rtc_rx5c348_get_time() -- hour  %02X\n", hour);
    printk( "rtc_rx5c348_get_time() -- min   %02X\n", minute);
    printk( "rtc_rx5c348_get_time() -- sec   %02X\n", second);
#endif

    year    = BCD_TO_BIN( (year   & RTC_RX5C348_MASK_YEAR)  );
    month   = BCD_TO_BIN( (month  & RTC_RX5C348_MASK_MONTH) ); 
    day     = BCD_TO_BIN( (day    & RTC_RX5C348_MASK_MDAY)  );
    hour    = BCD_TO_BIN( (hour   & RTC_RX5C348_MASK_HOUR)  );
    minute  = BCD_TO_BIN( (minute & RTC_RX5C348_MASK_MIN)   );
    second  = BCD_TO_BIN( (second & RTC_RX5C348_MASK_SEC)   );

    /* Deassert RTC CE */
    RTC_RX5C348_CE_DEASSERT;

    /* mktime( YYYY, MM=1-12, DAY=1-31, HOUR=0-23, MIN=0-59, SEC=0-59 ); */
    t = mktime( (century*100)+year, month, day, hour, minute, second );
    return( t );
}


int 
rtc_rx5c348_set_time( unsigned long t ) 
{
    struct rtc_time tm;
    unsigned long the_time; 
    unsigned char year, month, mday, wday, hour, min, sec;

    /* to_tm( YYYY, MM=0-11, DAY=1-31, HOUR=0-23, MIN=0-59, SEC=0-59 ); */
    to_tm(t, &tm);

    /* Tx4925 SPI Module Init */
    rtc_rx5c348_tx4925_spi_init();

    /* Assert RTC CE */
    RTC_RX5C348_CE_ASSERT;

    /* Few delay */
    rtc_rx5c348_udelay(1);

    rtc_rx5c348_tx4925_spi_active();

    year  = ( BIN_TO_BCD( (tm.tm_year%100) ) & RTC_RX5C348_MASK_YEAR  );
    month = ( BIN_TO_BCD( (tm.tm_mon+1)    ) & RTC_RX5C348_MASK_MONTH );
    if ((tm.tm_year/100) == 20)
        month |= RTC_RX5C348_MONTH_20;
    mday  = ( BIN_TO_BCD( tm.tm_mday       ) & RTC_RX5C348_MASK_MDAY  );
    wday  = ( BIN_TO_BCD( tm.tm_wday       ) & RTC_RX5C348_MASK_WDAY  );
    hour  = ( BIN_TO_BCD( tm.tm_hour       ) & RTC_RX5C348_MASK_HOUR  );
    min   = ( BIN_TO_BCD( tm.tm_min        ) & RTC_RX5C348_MASK_MIN   );
    sec   = ( BIN_TO_BCD( tm.tm_sec        ) & RTC_RX5C348_MASK_SEC   );

    /* write time data */ 
    rtc_rx5c348_tx4925_spi_write(RTC_RX5C348_SEC_REG,  sec);
    rtc_rx5c348_tx4925_spi_write(RTC_RX5C348_MIN_REG,  min);
    rtc_rx5c348_tx4925_spi_write(RTC_RX5C348_HOUR_REG, hour);
    rtc_rx5c348_tx4925_spi_write(RTC_RX5C348_WDAY_REG, wday);
    rtc_rx5c348_tx4925_spi_write(RTC_RX5C348_MDAY_REG, mday);
    rtc_rx5c348_tx4925_spi_write(RTC_RX5C348_MONTH_REG,month);
    rtc_rx5c348_tx4925_spi_write(RTC_RX5C348_YEAR_REG, year);

    the_time = rtc_rx5c348_get_time();

    /* Deassert RTC CE */
    RTC_RX5C348_CE_DEASSERT;

    return( the_time );
}


void __init
rtc_rx5c348_init( void )
{
    unsigned char ctrl, ctrl2;

    rtc_get_time = rtc_rx5c348_get_time;
    rtc_set_time = rtc_rx5c348_set_time;

    /* Tx4925 SPI Module Init */
    rtc_rx5c348_tx4925_spi_init();

    /* Assert RTC CE */
    RTC_RX5C348_CE_ASSERT;

    /* Few delay */
    rtc_rx5c348_udelay(1);

    rtc_rx5c348_tx4925_spi_active();

#ifdef RTC_RX5C348_DEBUG
    ctrl = rtc_rx5c348_tx4925_spi_read(RTC_RX5C348_CTRL1_REG);
    printk( "rtc_rx5c348_init() -- CTRL1 REG 0x%02X\n", ctrl);
#endif

    ctrl2 = rtc_rx5c348_tx4925_spi_read(RTC_RX5C348_CTRL2_REG);

#ifdef RTC_RX5C348_DEBUG
    printk( "rtc_rx5c348_init() -- CTRL2 REG 0x%02X\n", ctrl2);
#endif

    ctrl = (RTC_RX5C348_CTRL1_24 | RTC_RX5C348_CTRL1_CLEN2);
    rtc_rx5c348_tx4925_spi_write(RTC_RX5C348_CTRL1_REG, ctrl);
    ctrl = RTC_RX5C348_CTRL2_CLEN1;
    rtc_rx5c348_tx4925_spi_write(RTC_RX5C348_CTRL2_REG, ctrl);

    printk( "Rx5C348(CTRL2:0x%02x) ", ctrl2);
    if (ctrl2 & RTC_RX5C348_CTRL2_XSTP) {
        printk( "-- Set Default Data\n" );
        rtc_rx5c348_set_time(0);
    } else {
        if (ctrl2 & RTC_RX5C348_CTRL2_VDET) {
            printk( "-- battery flag indicates low voltage\n" );
	} else {
            printk( "-- battery flag indicates normal\n" );
        }
    }

    /* Deassert RTC CE */
    RTC_RX5C348_CE_DEASSERT;

    return;
}
