/*
    commands.c: Commands sent to the card
    Copyright (C) 2003-2004   Ludovic Rousseau

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

/*
 * $Id: commands.c 1747 2005-11-24 15:49:00Z rousseau $
 */

#include <stdio.h>	// CCIDPCSC:0035 sprintf()gp邽

#include <string.h>
#include <stdlib.h>
#include <PCSC/pcsclite.h>
#include <PCSC/ifdhandler.h>

#include "commands.h"
#include "openct/proto-t1.h"
#include "ccid.h"
#include "defs.h"
#include "ccid_ifdhandler.h"
#include "config.h"
#include "debug.h"


// CCIDPCSC:0035 LCN[h{ɕKvmF
//#ifdef HAVE_PTHREAD
//#include <pthread.h>
//#endif


/*
 * Possible values :
 * 3 -> 1.8V, 3V, 5V
 * 2 -> 3V, 5V
 * 1 -> 5V only
 */
/*
 * To be safe we only use 5V
 * otherwise we would have to parse the ATR and get the value of TAi (i>2) when
 * in T=15
 */
#define DEFAULT_VOLTAGE 1 /* start with 5 volts */

#define max( a, b )   ( ( ( a ) > ( b ) ) ? ( a ) : ( b ) )

// CCIDPCSC:0033
unsigned char arqb_data[20];
extern CcidDesc CcidSlots[PCSCLITE_MAX_READERS];
extern unsigned char ATR_buf[];
int bitcnt( unsigned char prm );
int membitcnt( void *data, int size );
// CCIDPCSC:0033

/* internal functions */
static RESPONSECODE CmdXfrBlockTPDU_T0(unsigned int reader_index,
	unsigned int tx_length, unsigned char tx_buffer[], unsigned int *rx_length,
	unsigned char rx_buffer[]);

static RESPONSECODE CmdXfrBlockCHAR_T0(unsigned int reader_index, unsigned int
	tx_length, unsigned char tx_buffer[], unsigned int *rx_length, unsigned
	char rx_buffer[]);

static void i2dw(int value, unsigned char *buffer);


/*****************************************************************************
 *
 *					CmdPowerOn
 *
 ****************************************************************************/
#if (1)	// CCIDPCSC:0032
/* --- START --- DENSO SERIALΉŁ@CmdPowerOn()  --- START --- */
RESPONSECODE CmdPowerOn(unsigned int reader_index, unsigned int * nlength,
	unsigned char buffer[])
{
	RESPONSECODE return_value = IFD_SUCCESS;

#ifdef	CCIDPCSC_DEBUG
	printf("CmdPowerOn (CARRIER_PCD ON)-------------------------------\n");
#endif	// CCIDPCSC_DEBUG

	return_value = RequestCard(reader_index, buffer);

	if(return_value == IFD_SUCCESS){
	        *nlength = IFD_IDM_DATA_SIZE;
	}

	return return_value;
} /* CmdPowerOn */
/* --- END --- DENSO SERIALΉŁ@CmdPowerOn()  --- END --- */
#else	// CCIDPCSC:0032
RESPONSECODE CmdPowerOn(unsigned int reader_index, unsigned int * nlength,
	unsigned char buffer[])
{
	unsigned char cmd[10];
	status_t res;
	int length, count = 1;
	unsigned int atr_len;
	RESPONSECODE return_value = IFD_SUCCESS;
	_ccid_descriptor *ccid_descriptor = get_ccid_descriptor(reader_index);
	char voltage;

	/* store length of buffer[] */
	length = *nlength;

	if (ccid_descriptor->dwFeatures & CCID_CLASS_AUTO_VOLTAGE)
		voltage = 0;	/* automatic voltage selection */
	else
		voltage = DEFAULT_VOLTAGE;

again:
	cmd[0] = 0x62; /* IccPowerOn */
	cmd[1] = cmd[2] = cmd[3] = cmd[4] = 0;	/* dwLength */
	cmd[5] = ccid_descriptor->bCurrentSlotIndex;	/* slot number */
	cmd[6] = (*ccid_descriptor->pbSeq)++;
	cmd[7] = voltage;
	cmd[8] = cmd[9] = 0; /* RFU */

	res = WritePort(reader_index, sizeof(cmd), cmd);
	if (res != STATUS_SUCCESS)
		return IFD_COMMUNICATION_ERROR;

	/* reset available buffer size */
	/* needed if we go back after a switch to ISO mode */
	*nlength = length;

	res = ReadPort(reader_index, nlength, buffer);
	if (res != STATUS_SUCCESS)
		return IFD_COMMUNICATION_ERROR;

	if (buffer[STATUS_OFFSET] & CCID_COMMAND_FAILED)
	{
		ccid_error(buffer[ERROR_OFFSET], __FILE__, __LINE__, __FUNCTION__);    /* bError */

		if (0xBB == buffer[ERROR_OFFSET] &&	/* Protocol error in EMV mode */
			((GEMPC433 == ccid_descriptor->readerID)
			|| (CHERRYXX33 == ccid_descriptor->readerID)))
		{
			unsigned char cmd[] = {0x1F, 0x01};
			unsigned char res[1];
			unsigned int res_length = sizeof(res);

			if ((return_value = CmdEscape(reader_index, cmd, sizeof(cmd), res,
				&res_length)) != IFD_SUCCESS)
				return return_value;

			/* avoid looping if we can't switch mode */
			if (count--)
				goto again;
			else
				DEBUG_CRITICAL("Can't set reader in ISO mode");
		}

		/* continue with 3 volts and 5 volts */
		if (voltage > 1)
		{
			voltage--;
			goto again;
		}

		return IFD_COMMUNICATION_ERROR;
	}

	/* extract the ATR */
	atr_len = dw2i(buffer, 1);	/* ATR length */
	if (atr_len > *nlength)
		atr_len = *nlength;
	else
		*nlength = atr_len;

	memmove(buffer, buffer+10, atr_len);

	return return_value;
} /* CmdPowerOn */
#endif	// CCIDPCSC:0032


/*****************************************************************************
 *
 *					SecurePINVerify
 *
 ****************************************************************************/
RESPONSECODE SecurePINVerify(unsigned int reader_index,
	const unsigned char TxBuffer[], unsigned int TxLength,
	unsigned char RxBuffer[], unsigned int *RxLength)
{
	unsigned char cmd[11+14+CMD_BUF_SIZE];
	unsigned int a, b;
	_ccid_descriptor *ccid_descriptor = get_ccid_descriptor(reader_index);
	int old_read_timeout;
	RESPONSECODE ret;

	cmd[0] = 0x69;	/* Secure */
	cmd[5] = ccid_descriptor->bCurrentSlotIndex;	/* slot number */
	cmd[6] = (*ccid_descriptor->pbSeq)++;
	cmd[7] = 0;		/* bBWI */
	cmd[8] = 0;		/* wLevelParameter */
	cmd[9] = 0;
	cmd[10] = 0;	/* bPINOperation: PIN Verification */

	/* 19 is the size of the PCSCv2 PIN verify structure
	 * The equivalent CCID structure is only 14-bytes long */
	if (TxLength > 19+CMD_BUF_SIZE) /* command too large? */
	{
		DEBUG_INFO3("Command too long: %d > %d", TxLength, 16+CMD_BUF_SIZE);
		*RxLength = 0;
		return IFD_NOT_SUPPORTED;
	}

	if (TxLength < 19+4 /* 4 = APDU size */)	/* command too short? */
	{
		DEBUG_INFO3("Command too short: %d < %d", TxLength, 19+4);
		*RxLength = 0;
		return IFD_NOT_SUPPORTED;
	}

	if (dw2i(TxBuffer, 15) + 19 != TxLength) /* ulDataLength field coherency */
	{
		DEBUG_INFO3("Wrong lengths: %d %d", TxBuffer[15] + 19, TxLength);
		*RxLength = 0;
		return IFD_NOT_SUPPORTED;
	}

	/* Build a CCID block from a PC/SC V2.1.2 Part 10 block */
	for (a = 11, b = 0; b < TxLength; b++)
	{
		if (1 == b) /* bTimeOut2 field */
			/* Ignore the second timeout as there's nothing we can do with
			 * it currently */
			continue;

		if ((b >= 15) && (b <= 18)) /* ulDataLength field (4 bytes) */
			/* the ulDataLength field is not present in the CCID frame
			 * so do not copy */
			continue;

		/* copy the CCID block 'verbatim' */
		cmd[a] = TxBuffer[b];
		a++;
	}

	/* SPR532 and Case 1 APDU */
	if ((SPR532 == ccid_descriptor->readerID) && (TxBuffer[15] == 4))
	{
		RESPONSECODE return_value;
		unsigned char cmd[] = { 0x80, 0x02, 0x00 };
		unsigned char res[1];
		unsigned int res_length = sizeof(res);

		/* the SPR532 will append the PIN code without any padding */
		return_value = CmdEscape(reader_index, cmd, sizeof(cmd), res,
			&res_length);
		if (return_value != IFD_SUCCESS)
		{
			ccid_error(res[ERROR_OFFSET], __FILE__, __LINE__, __FUNCTION__);
			return return_value;
		}
	}

	i2dw(a - 10, cmd + 1);  /* CCID message length */

	old_read_timeout = ccid_descriptor -> readTimeout;
	ccid_descriptor -> readTimeout = max(30, TxBuffer[0]);	/* at least 30 seconds */

	if (WritePort(reader_index, a, cmd) != STATUS_SUCCESS)
	{
		*RxLength = 0;
		return IFD_COMMUNICATION_ERROR;
	}

	ret = CCID_Receive(reader_index, RxLength, RxBuffer);

	ccid_descriptor -> readTimeout = old_read_timeout;
	return ret;
} /* SecurePINVerify */


/*****************************************************************************
 *
 *					SecurePINModify
 *
 ****************************************************************************/
RESPONSECODE SecurePINModify(unsigned int reader_index,
	const unsigned char TxBuffer[], unsigned int TxLength,
	unsigned char RxBuffer[], unsigned int *RxLength)
{
	unsigned char cmd[11+19+CMD_BUF_SIZE];
	unsigned int a, b;
	_ccid_descriptor *ccid_descriptor = get_ccid_descriptor(reader_index);
	int old_read_timeout;
	RESPONSECODE ret;

	cmd[0] = 0x69;	/* Secure */
	cmd[5] = ccid_descriptor->bCurrentSlotIndex;	/* slot number */
	cmd[6] = (*ccid_descriptor->pbSeq)++;
	cmd[7] = 0;		/* bBWI */
	cmd[8] = 0;		/* wLevelParameter */
	cmd[9] = 0;
	cmd[10] = 1;	/* bPINOperation: PIN Modification */

	/* 24 is the size of the PCSC PIN modify structure
	 * The equivalent CCID structure is only 18 or 19-bytes long */
	if ((TxLength > 19+CMD_BUF_SIZE) /* command too large? */
		|| (TxLength < 18+4 /* 4 = APDU size */) /* command too short? */
		|| (TxBuffer[20] + 24 != TxLength)) /* ulDataLength field coherency */
	{
		*RxLength = 0;
		return IFD_NOT_SUPPORTED;
	}

	/* Make sure in the beginning if bNumberMessage is valid or not */
	if (TxBuffer[11] > 3)
	{
		*RxLength = 0;
		return IFD_NOT_SUPPORTED;
	}

	/* Build a CCID block from a PC/SC V2.1.2 Part 10 block */

	/* Do adjustments as needed - CCID spec is not exact with some
	 * details in the format of the structure, per-reader adaptions
	 * might be needed.
	 */
	for (a = 11, b = 0; b < TxLength; b++)
	{
		if (1 == b) /* bTimeOut2 */
			/* Ignore the second timeout as there's nothing we can do with it
			 * currently */
			continue;

		if (15 == b) /* bMsgIndex2 */
		{
			/* in CCID the bMsgIndex2 is present only if bNumberMessage != 0 */
			if (0 == TxBuffer[11])
				continue;
		}

		if (16 == b) /* bMsgIndex3 */
		{
			/*
			 * SPR 532 and Cherry ST 2000C has no display but requires _all_
			 * bMsgIndex fields with bNumberMessage set to 0.
			 */
			if ((SPR532 == ccid_descriptor->readerID)
				|| (CHERRYST2000 == ccid_descriptor->readerID))
			{
				cmd[21] = 0x00; /* set bNumberMessages to 0 */
				cmd[a] = cmd[a-1] = cmd[a+1] = 0;	/* bMsgIndex123 */
				a += 2;
				continue;
			}

			/* in CCID the bMsgIndex3 is present only if bNumberMessage == 3 */
			if (TxBuffer[11] < 3)
				continue;
		}

		if ((b >= 20) && (b <= 23)) /* ulDataLength field (4 bytes) */
			/* the ulDataLength field is not present in the CCID frame
			 * so do not copy */
			continue;

		/* copy to the CCID block 'verbatim' */
		cmd[a] = TxBuffer[b];
		a++;
 	}

	/* We know the size of the CCID message now */
	i2dw(a - 10, cmd + 1);	/* command length (includes bPINOperation) */

	old_read_timeout = ccid_descriptor -> readTimeout;
	ccid_descriptor -> readTimeout = max(30, TxBuffer[0]);	/* at least 30 seconds */

	if (WritePort(reader_index, a, cmd) != STATUS_SUCCESS)
	{
		*RxLength = 0;
 		return IFD_COMMUNICATION_ERROR;
	}

 	ret = CCID_Receive(reader_index, RxLength, RxBuffer);

	ccid_descriptor -> readTimeout = old_read_timeout;
	return ret;
} /* SecurePINModify */


/*****************************************************************************
 *
 *					Escape
 *
 ****************************************************************************/
RESPONSECODE CmdEscape(unsigned int reader_index,
	const unsigned char TxBuffer[], unsigned int TxLength,
	unsigned char RxBuffer[], unsigned int *RxLength)
{
	unsigned char *cmd_in, *cmd_out;
	status_t res;
	unsigned int length_in, length_out;
	RESPONSECODE return_value = IFD_SUCCESS;
	_ccid_descriptor *ccid_descriptor = get_ccid_descriptor(reader_index);

again:
	/* allocate buffers */
	length_in = 10 + TxLength;
	if (NULL == (cmd_in = malloc(length_in)))
		return IFD_COMMUNICATION_ERROR;

	length_out = 10 + *RxLength;
	if (NULL == (cmd_out = malloc(length_out)))
	{
		free(cmd_in);
		return IFD_COMMUNICATION_ERROR;
	}

	cmd_in[0] = 0x6B; /* PC_to_RDR_Escape */
	i2dw(length_in - 10, cmd_in+1);	/* dwLength */
	cmd_in[5] = ccid_descriptor->bCurrentSlotIndex;	/* slot number */
	cmd_in[6] = (*ccid_descriptor->pbSeq)++;
	cmd_in[7] = cmd_in[8] = cmd_in[9] = 0; /* RFU */

	/* copy the command */
	memcpy(&cmd_in[10], TxBuffer, TxLength);

	res = WritePort(reader_index, length_in, cmd_in);
	free(cmd_in);
	if (res != STATUS_SUCCESS)
	{
		free(cmd_out);
		return IFD_COMMUNICATION_ERROR;
	}

	res = ReadPort(reader_index, &length_out, cmd_out);

	/* replay the command if NAK
	 * This (generally) happens only for the first command sent to the reader
	 * with the serial protocol so it is not really needed for all the other
	 * ReadPort() calls */
	if (STATUS_COMM_NAK == res)
	{
		free(cmd_out);
		goto again;
	}

	if (res != STATUS_SUCCESS)
	{
		free(cmd_out);
		return IFD_COMMUNICATION_ERROR;
	}

	if (cmd_out[STATUS_OFFSET] & CCID_COMMAND_FAILED)
	{
		ccid_error(cmd_out[ERROR_OFFSET], __FILE__, __LINE__, __FUNCTION__);    /* bError */
		return_value = IFD_COMMUNICATION_ERROR;
	}

	/* copy the response */
	length_out = dw2i(cmd_out, 1);
	if (length_out > *RxLength)
		length_out = *RxLength;
	*RxLength = length_out;
	memcpy(RxBuffer, &cmd_out[10], length_out);

	free(cmd_out);

	return return_value;
} /* Escape */


/*****************************************************************************
 *
 *					CmdPowerOff
 *
 ****************************************************************************/
#if (1)	// CCIDPCSC:0032
/* --- START --- DENSO SERIALΉŁ@CmdPowerOff()  --- START --- */
RESPONSECODE CmdPowerOff(unsigned int reader_index)
{
	RESPONSECODE return_value = IFD_SUCCESS;

#ifdef	CCIDPCSC_DEBUG
	printf("CmdPowerOff (CARRIER_PCD OFF)-------------------------------\n");
#endif	// CCIDPCSC_DEBUG

	return_value = CarrierOFF(reader_index);

	return return_value;
} /* CmdPowerOff */
/* --- END --- DENSO SERIALΉŁ@CmdPowerOff()  --- END --- */
#else	// CCIDPCSC:0032
RESPONSECODE CmdPowerOff(unsigned int reader_index)
{
	unsigned char cmd[10];
	status_t res;
	unsigned int length;
	RESPONSECODE return_value = IFD_SUCCESS;
	_ccid_descriptor *ccid_descriptor = get_ccid_descriptor(reader_index);

	cmd[0] = 0x63; /* IccPowerOff */
	cmd[1] = cmd[2] = cmd[3] = cmd[4] = 0;	/* dwLength */
	cmd[5] = ccid_descriptor->bCurrentSlotIndex;	/* slot number */
	cmd[6] = (*ccid_descriptor->pbSeq)++;
	cmd[7] = cmd[8] = cmd[9] = 0; /* RFU */

	res = WritePort(reader_index, sizeof(cmd), cmd);
	if (res != STATUS_SUCCESS)
		return IFD_COMMUNICATION_ERROR;

	length = sizeof(cmd);
	res = ReadPort(reader_index, &length, cmd);
	if (res != STATUS_SUCCESS)
		return IFD_COMMUNICATION_ERROR;

	if (cmd[STATUS_OFFSET] & CCID_COMMAND_FAILED)
	{
		ccid_error(cmd[ERROR_OFFSET], __FILE__, __LINE__, __FUNCTION__);    /* bError */
		return_value = IFD_COMMUNICATION_ERROR;
	}

	return return_value;
} /* CmdPowerOff */
#endif	// CCIDPCSC:0032


/*****************************************************************************
 *
 *					CmdGetSlotStatus
 *
 ****************************************************************************/
#if (0)	// CCIDPCSC:0032
/* --- START --- DENSO SERIALΉŁ@CmdGetSlotStatus()  --- START --- */
RESPONSECODE CmdGetSlotStatus(unsigned int reader_index, unsigned char buffer[])
{
	int length;
	unsigned char	rcv_buff[IFD_DEV_CMD_SIZE];
	RESPONSECODE return_value = IFD_SUCCESS;
	RESPONSECODE ret = IFD_SUCCESS;

	/* IDm擾ĂȂꍇ́ACmdPowerOn()	*/
	/* foCX̃J[h̗LĎ		*/
	if(0 == membitcnt(ATR_buf,IFD_IDM_DATA_SIZE)){
		/* J[h݂̑OFFɂ */
		buffer[IFD_PRESENSE_OFFSET]=IFD_CARD_OFF;

		ret = CmdPowerOn(reader_index,&length,rcv_buff);

		if(ret == IFD_NO_CARD){
#ifdef	CCIDPCSC_DEBUG
			printf("CmdGetSlotStatus : Command Success : IFD_NO_CARD \n");
#endif	// CCIDPCSC_DEBUG
/* 2005.06.06 */
{
	int i;
	char dbg_buff[40];
	for(i = 0 ; i < 16 ; i++){
		sprintf(&dbg_buff[i*2],"%02x",rcv_buff[i]);
	}
#ifdef	CCIDPCSC_DEBUG
	printf("CmdGetSlotStatus : CmdPowerOn data = %s (IFD_NO_CARD)\n",dbg_buff);
#endif	// CCIDPCSC_DEBUG
}
			/* R}h͐킾J[h݂͑Ȃ	*/
			memset(ATR_buf,0,IFD_IDM_DATA_SIZE);
			/* J[h݂̑OFFɂ */
			buffer[IFD_PRESENSE_OFFSET]=IFD_CARD_OFF;
			/* J[hfoCXɖŁÂ͐ */
			ret = IFD_SUCCESS;
			return	ret;
		}
		else if(ret != IFD_SUCCESS){
			/* R}hsNG */
#ifdef	CCIDPCSC_DEBUG
			printf("CmdPowerOn NG \n");
#endif	// CCIDPCSC_DEBUG
			/* IDm i[GANA	*/
			memset(ATR_buf,0,IFD_IDM_DATA_SIZE);
			/* J[h݂̑OFFɂ */
			buffer[IFD_PRESENSE_OFFSET]=IFD_CARD_OFF;
			return	ret;
	    }
		}else {
			/* IDm擾Ă鎞̓foCXɃJ[h	*/
			/* ꍇȂ̂CardPolling()ŃJ[h	*/
			/* foCXɌpĒuĂ邩Ď	*/

			ret = CardPolling(reader_index, rcv_buff);
			if(ret != IFD_SUCCESS){
/* 2005.06.06 */
{
	int i;
	char dbg_buff[40];
	for(i = 0 ; i < 16 ; i++){
		sprintf(&dbg_buff[i*2],"%02x",rcv_buff[i]);
	}
#ifdef	CCIDPCSC_DEBUG
	printf("CmdGetSlotStatus : CardPolling data = %s\n",dbg_buff);
#endif	// CCIDPCSC_DEBUG
}
#ifdef	CCIDPCSC_DEBUG
			printf("Polling Error 2 \n");
#endif	// CCIDPCSC_DEBUG
			/* IDm i[GANA	*/
			memset(ATR_buf,0,IFD_IDM_DATA_SIZE);
			/* J[h݂̑OFFɂ */
			buffer[IFD_PRESENSE_OFFSET]=IFD_CARD_OFF;
			return	ret;
	    }
	}
	/* IDm compare */
	if(memcmp(ATR_buf,rcv_buff,IFD_IDM_DATA_SIZE) == 0){
		/* VKŃfoCXɃJ[h݂邱ƂmFA܂	 */
		/* Õ|[OŎ擾J[hfoCXɑ݂	 */
		buffer[IFD_PRESENSE_OFFSET]=IFD_CARD_ON;
	} else {
		/* ܂ŃfoCXɑ݂ĂJ[hOꂽ */
		/* IDm i[GANA	*/
		memset(ATR_buf,0,IFD_IDM_DATA_SIZE);
		/* J[h݂̑OFFɂ */
		buffer[IFD_PRESENSE_OFFSET]=IFD_CARD_OFF;
	}

	return return_value;
} /* CmdGetSlotStatus */
/* --- EMD --- DENSO SERIALΉŁ@CmdGetSlotStatus()  --- END --- */
#else	// CCIDPCSC:0032
RESPONSECODE CmdGetSlotStatus(unsigned int reader_index, unsigned char buffer[])
{
	unsigned char cmd[10];
	status_t res;
	unsigned int length;
	RESPONSECODE return_value = IFD_SUCCESS;
	_ccid_descriptor *ccid_descriptor = get_ccid_descriptor(reader_index);

	cmd[0] = 0x65; /* GetSlotStatus */
	cmd[1] = cmd[2] = cmd[3] = cmd[4] = 0;	/* dwLength */
	cmd[5] = ccid_descriptor->bCurrentSlotIndex;	/* slot number */
	cmd[6] = (*ccid_descriptor->pbSeq)++;
	cmd[7] = cmd[8] = cmd[9] = 0; /* RFU */

	res = WritePort(reader_index, sizeof(cmd), cmd);
	if (res != STATUS_SUCCESS)
		return IFD_COMMUNICATION_ERROR;

	length = SIZE_GET_SLOT_STATUS;
	res = ReadPort(reader_index, &length, buffer);
	if (res != STATUS_SUCCESS)
		return IFD_COMMUNICATION_ERROR;

	if (buffer[STATUS_OFFSET] & CCID_COMMAND_FAILED)
	{
		ccid_error(buffer[ERROR_OFFSET], __FILE__, __LINE__, __FUNCTION__);    /* bError */

		/* card absent or mute is not an communication error */
		if (buffer[ERROR_OFFSET] != 0xFE)
			return_value = IFD_COMMUNICATION_ERROR;
	}

	return return_value;
} /* CmdGetSlotStatus */
#endif	// CCIDPCSC:0032


/*****************************************************************************
 *
 *					CmdXfrBlock
 *
 ****************************************************************************/
RESPONSECODE CmdXfrBlock(unsigned int reader_index, unsigned int tx_length,
	unsigned char tx_buffer[], unsigned int *rx_length,
	unsigned char rx_buffer[], int protocol)
{
	RESPONSECODE return_value = IFD_SUCCESS;
	_ccid_descriptor *ccid_descriptor = get_ccid_descriptor(reader_index);

	/* command length too big for CCID reader? */
	if (tx_length > ccid_descriptor->dwMaxCCIDMessageLength)
	{
		DEBUG_CRITICAL3("Command too long (%d bytes) for max: %d bytes",
			tx_length, ccid_descriptor->dwMaxCCIDMessageLength);
		return_value = IFD_COMMUNICATION_ERROR;
		goto clean_up_and_return;
	}

	/* command length too big for CCID driver? */
	if (tx_length > CMD_BUF_SIZE)
	{
		DEBUG_CRITICAL3("Command too long (%d bytes) for max: %d bytes",
			tx_length, CMD_BUF_SIZE);
		return_value = IFD_COMMUNICATION_ERROR;
		goto clean_up_and_return;
	}

	/* APDU or TPDU? */
	switch (ccid_descriptor->dwFeatures & CCID_CLASS_EXCHANGE_MASK)
	{
		case CCID_CLASS_TPDU:
			if (protocol == T_0)
				return_value = CmdXfrBlockTPDU_T0(reader_index,
					tx_length, tx_buffer, rx_length, rx_buffer);
			else
				if (protocol == T_1)
					return_value = CmdXfrBlockTPDU_T1(reader_index, tx_length,
						tx_buffer, rx_length, rx_buffer);
				else
					return_value = IFD_PROTOCOL_NOT_SUPPORTED;
			break;

		case CCID_CLASS_SHORT_APDU:
		case CCID_CLASS_EXTENDED_APDU:
			/* We only support extended APDU if the reader can support the
			 * command length. See test above */
			return_value = CmdXfrBlockTPDU_T0(reader_index,
				tx_length, tx_buffer, rx_length, rx_buffer);
			break;

		case CCID_CLASS_CHARACTER:
			if (protocol == T_0)
				return_value = CmdXfrBlockCHAR_T0(reader_index, tx_length,
					tx_buffer, rx_length, rx_buffer);
			else
				if (protocol == T_1)
					return_value = CmdXfrBlockTPDU_T1(reader_index, tx_length,
						tx_buffer, rx_length, rx_buffer);
 				else
					return_value = IFD_PROTOCOL_NOT_SUPPORTED;
			break;

		default:
			*rx_length = 0;
			return_value = IFD_COMMUNICATION_ERROR;
	}

clean_up_and_return:
	return return_value;
} /* CmdXfrBlock */


/*****************************************************************************
 *
 *					CCID_Transmit
 *
 ****************************************************************************/
RESPONSECODE CCID_Transmit(unsigned int reader_index, unsigned int tx_length,
	const unsigned char tx_buffer[], unsigned short rx_length, unsigned char bBWI)
{
	unsigned char cmd[10+CMD_BUF_SIZE];	/* CCID + APDU buffer */
	_ccid_descriptor *ccid_descriptor = get_ccid_descriptor(reader_index);

	cmd[0] = 0x6F; /* XfrBlock */
	i2dw(tx_length, cmd+1);	/* APDU length */
	cmd[5] = ccid_descriptor->bCurrentSlotIndex;	/* slot number */
	cmd[6] = (*ccid_descriptor->pbSeq)++;
	cmd[7] = bBWI;	/* extend block waiting timeout */
	cmd[8] = rx_length & 0xFF;	/* Expected length, in character mode only */
	cmd[9] = (rx_length >> 8) & 0xFF;

	/* check that the command is not too large */
	if (tx_length > CMD_BUF_SIZE)
		return IFD_NOT_SUPPORTED;

	memcpy(cmd+10, tx_buffer, tx_length);

	if (WritePort(reader_index, 10+tx_length, cmd) != STATUS_SUCCESS)
		return IFD_COMMUNICATION_ERROR;

	return IFD_SUCCESS;
} /* CCID_Transmit */


/*****************************************************************************
 *
 *					CCID_Receive
 *
 ****************************************************************************/
RESPONSECODE CCID_Receive(unsigned int reader_index, unsigned int *rx_length,
	unsigned char rx_buffer[])
{
	unsigned char cmd[10+CMD_BUF_SIZE];	/* CCID + APDU buffer */
	unsigned int length;

time_request:
	length = sizeof(cmd);
	if (ReadPort(reader_index, &length, cmd) != STATUS_SUCCESS)
	{
		*rx_length = 0;
		return IFD_COMMUNICATION_ERROR;
	}

	if (cmd[STATUS_OFFSET] & CCID_COMMAND_FAILED)
	{
		ccid_error(cmd[ERROR_OFFSET], __FILE__, __LINE__, __FUNCTION__);    /* bError */
		switch (cmd[ERROR_OFFSET])
		{
			case 0xEF:	/* cancel */
				if (*rx_length < 2)
					return IFD_COMMUNICATION_ERROR;
				rx_buffer[0]= 0x64;
				rx_buffer[1]= 0x01;
				*rx_length = 2;
				return IFD_SUCCESS;

			case 0xF0:	/* timeout */
				if (*rx_length < 2)
					return IFD_COMMUNICATION_ERROR;
				rx_buffer[0]= 0x64;
				rx_buffer[1]= 0x00;
				*rx_length = 2;
				return IFD_SUCCESS;

			case 0xFD:	/* Parity error during exchange */
				*rx_length = 0; /* nothing received */
				return IFD_PARITY_ERROR;

			default:
				*rx_length = 0; /* nothing received */
				return IFD_COMMUNICATION_ERROR;
		}
	}

	if (cmd[STATUS_OFFSET] & CCID_TIME_EXTENSION)
	{
		DEBUG_COMM2("Time extension requested: 0x%02X", cmd[ERROR_OFFSET]);
		goto time_request;
	}

	length = dw2i(cmd, 1);
	if (length < *rx_length)
		*rx_length = length;
	else
		length = *rx_length;
	memcpy(rx_buffer, cmd+10, length);

	return IFD_SUCCESS;
} /* CCID_Receive */


/*****************************************************************************
 *
 *					CmdXfrBlockTPDU_T0
 *
 ****************************************************************************/
static RESPONSECODE CmdXfrBlockTPDU_T0(unsigned int reader_index,
	unsigned int tx_length, unsigned char tx_buffer[], unsigned int *rx_length,
	unsigned char rx_buffer[])
{
	RESPONSECODE return_value = IFD_SUCCESS;

	DEBUG_COMM2("T=0: %d bytes", tx_length);

	return_value = CCID_Transmit(reader_index, tx_length, tx_buffer, 0, 0);
	if (return_value != IFD_SUCCESS)
		return return_value;

	return CCID_Receive(reader_index, rx_length, rx_buffer);
} /* CmdXfrBlockTPDU_T0 */


/*****************************************************************************
 *
 *					T0CmdParsing
 *
 ****************************************************************************/
static RESPONSECODE T0CmdParsing(unsigned char *cmd, unsigned int cmd_len,
	unsigned int *exp_len)
{
	*exp_len = 0;

	/* Ref: 7816-4 Annex A */
	switch (cmd_len)
	{
		case 4:	/* Case 1 */
			*exp_len = 2; /* SW1 and SW2 only */
			break;

		case 5: /* Case 2 */
			if (cmd[4] != 0)
				*exp_len = cmd[4] + 2;
			else
				*exp_len = 256 + 2;
			break;

		default: /* Case 3 */
			if (cmd_len > 5 && cmd_len == (unsigned int)(cmd[4] + 5))
				*exp_len = 2; /* SW1 and SW2 only */
			else
				return IFD_COMMUNICATION_ERROR;	/* situation not supported */
			break;
	}

	return IFD_SUCCESS;
} /* T0CmdParsing */


/*****************************************************************************
 *
 *					T0ProcACK
 *
 ****************************************************************************/
static RESPONSECODE T0ProcACK(unsigned int reader_index,
	unsigned char **snd_buf, unsigned int *snd_len,
	unsigned char **rcv_buf, unsigned int *rcv_len,
	unsigned char **in_buf, unsigned int *in_len,
	unsigned int proc_len, int is_rcv)
{
	RESPONSECODE return_value;
	unsigned int remain_len;
	unsigned char tmp_buf[512];
	unsigned int ret_len;

	DEBUG_COMM2("Enter, is_rcv = %d", is_rcv);

	if (is_rcv == 1)
	{	/* Receiving mode */
		if (*in_len > 0)
		{	/* There are still available data in our buffer */
			if (*in_len >= proc_len)
			{
				/* We only need to get the data from our buffer */
				memcpy(*rcv_buf, *in_buf, proc_len);
				*rcv_buf += proc_len;
				*in_buf += proc_len;
				*rcv_len += proc_len;
				*in_len -= proc_len;

				return IFD_SUCCESS;
			}
			else
			{
				/* Move all data in the input buffer to the reply buffer */
				remain_len = proc_len - *in_len;
				memcpy(*rcv_buf, *in_buf, *in_len);
				*rcv_buf += *in_len;
				*in_buf += *in_len;
				*rcv_len += *in_len;
				*in_len = 0;
			}
		}
		else
			/* There is no data in our tmp_buf,
			 * we have to read all data we needed */
			remain_len = proc_len;

		/* Read the expected data from the smartcard */
		if (*in_len != 0)
		{
			DEBUG_CRITICAL("*in_len != 0");
			return IFD_COMMUNICATION_ERROR;
		}

		memset(tmp_buf, 0, sizeof(tmp_buf));

		ret_len = remain_len;
		return_value = CCID_Transmit(reader_index, 0, *snd_buf, ret_len, 0);
		if (return_value != IFD_SUCCESS)
			return return_value;

		return_value = CCID_Receive(reader_index, &ret_len, tmp_buf);
		if (return_value != IFD_SUCCESS)
			return return_value;

		memcpy(*rcv_buf, tmp_buf, remain_len);
		*rcv_buf += remain_len, *rcv_len += remain_len;

		/* If ret_len != remain_len, our logic is erroneous */
		if (ret_len != remain_len)
		{
			DEBUG_CRITICAL("ret_len != remain_len");
			return IFD_COMMUNICATION_ERROR;
		}
	}
	else
	{	/* Sending mode */

		return_value = CCID_Transmit(reader_index, proc_len, *snd_buf, 1, 0);
		if (return_value != IFD_SUCCESS)
			return return_value;

		*snd_len -= proc_len;
		*snd_buf += proc_len;
	}

	DEBUG_COMM("Exit");

	return IFD_SUCCESS;
} /* T0ProcACK */


/*****************************************************************************
 *
 *					T0ProcSW1
 *
 ****************************************************************************/
static RESPONSECODE T0ProcSW1(unsigned int reader_index,
	unsigned char *rcv_buf, unsigned int *rcv_len,
	unsigned char *in_buf, unsigned int in_len)
{
	RESPONSECODE return_value = IFD_SUCCESS;
	UCHAR tmp_buf[512];
	unsigned char *rcv_buf_tmp = rcv_buf;
	const unsigned int rcv_len_tmp = *rcv_len;
	unsigned char sw1, sw2;

	/* store the SW1 */
	sw1 = *rcv_buf = *in_buf;
	rcv_buf++;
	in_buf++;
	in_len--;
	(*rcv_len)++;

	/* store the SW2 */
	if (0 == in_len)
	{
		return_value = CCID_Transmit(reader_index, 0, rcv_buf, 1, 0);
		if (return_value != IFD_SUCCESS)
			return return_value;

		in_len = 1;

		return_value = CCID_Receive(reader_index, &in_len, tmp_buf);
		if (return_value != IFD_SUCCESS)
			return return_value;

		in_buf = tmp_buf;
	}
	sw2 = *rcv_buf = *in_buf;
	rcv_buf++;
	in_buf++;
	in_len--;
	(*rcv_len)++;

	if (return_value != IFD_SUCCESS)
	{
		rcv_buf_tmp[0] = rcv_buf_tmp[1] = 0;
		*rcv_len = rcv_len_tmp;
	}

	DEBUG_COMM3("Exit: SW=%02X %02X", sw1, sw2);

	return return_value;
} /* T0ProcSW1 */


/*****************************************************************************
 *
 *					CmdXfrBlockCHAR_T0
 *
 ****************************************************************************/
static RESPONSECODE CmdXfrBlockCHAR_T0(unsigned int reader_index,
	unsigned int snd_len, unsigned char snd_buf[], unsigned int *rcv_len,
	unsigned char rcv_buf[])
{
	int is_rcv;
	unsigned char cmd[5];
	unsigned char tmp_buf[512];
	unsigned int exp_len, in_len;
	unsigned char ins, *in_buf;
	RESPONSECODE return_value = IFD_SUCCESS;

	DEBUG_COMM2("T=0: %d bytes", snd_len);

	in_buf = tmp_buf;
	in_len = 0;
	*rcv_len = 0;

	return_value = T0CmdParsing(snd_buf, snd_len, &exp_len);
	if (return_value != IFD_SUCCESS)
	{
		DEBUG_CRITICAL("T0CmdParsing failed");
		return IFD_COMMUNICATION_ERROR;
	}

	if (snd_len == 5 || snd_len == 4)
		is_rcv = 1;
	else
		is_rcv = 0;

	/* Command to send to the smart card (must be 5 bytes, from 7816 p.15) */
	memset(cmd, 0, sizeof(cmd));
	if (snd_len == 4)
	{
		memcpy(cmd, snd_buf, 4);
		snd_buf += 4;
		snd_len -= 4;
	}
	else
	{
		memcpy(cmd, snd_buf, 5);
		snd_buf += 5;
		snd_len -= 5;
	}

	/* Make sure this is a valid command by checking the INS field */
	ins = cmd[1];
	if ((ins & 0xF0) == 0x60 ||	/* 7816-3 8.3.2 */
		(ins & 0xF0) == 0x90)
	{
		DEBUG_CRITICAL2("fatal: INS (0x%02X) = 0x6X or 0x9X", ins);
		return IFD_COMMUNICATION_ERROR;
	}

	return_value = CCID_Transmit(reader_index, 5, cmd, 1, 0);
	if (return_value != IFD_SUCCESS)
		return return_value;

	while (1)
	{
		if (in_len == 0)
		{
			in_len = 1;
			return_value = CCID_Receive(reader_index, &in_len, tmp_buf);
			if (return_value != IFD_SUCCESS)
			{
				DEBUG_CRITICAL("CCID_Receive failed");
				return return_value;
			}
			in_buf = tmp_buf;
		}
		if (in_len == 0)
		{
			/* Suppose we should be able to get data.
			 * If not, error. Set the time-out error */
			DEBUG_CRITICAL("error: in_len = 0");
			return IFD_RESPONSE_TIMEOUT;
		}

		/* Start to process the procedure bytes */
		if (*in_buf == 0x60)
		{
			in_len = 0;
			return_value = CCID_Transmit(reader_index, 0, cmd, 1, 0);

			if (return_value != IFD_SUCCESS)
				return return_value;

			continue;
		}
		else if (*in_buf == ins || *in_buf == (ins ^ 0x01))
		{
			/* ACK => To transfer all remaining data bytes */
			in_buf++, in_len--;
			if (is_rcv)
				return_value = T0ProcACK(reader_index, &snd_buf, &snd_len,
					&rcv_buf, rcv_len, &in_buf, &in_len, exp_len - *rcv_len, 1);
			else
				return_value = T0ProcACK(reader_index, &snd_buf, &snd_len,
					&rcv_buf, rcv_len, &in_buf, &in_len, snd_len, 0);

			if (*rcv_len == exp_len)
				return return_value;

			continue;
		}
		else if (*in_buf == (ins ^ 0xFF) || *in_buf == (ins ^ 0xFE))
		{
			/* ACK => To transfer 1 remaining bytes */
			in_buf++, in_len--;
			return_value = T0ProcACK(reader_index, &snd_buf, &snd_len,
				&rcv_buf, rcv_len, &in_buf, &in_len, 1, is_rcv);

			if (return_value != IFD_SUCCESS)
				return return_value;

			continue;
		}
		else if ((*in_buf & 0xF0) == 0x60 || (*in_buf & 0xF0) == 0x90)
			/* SW1 */
			return T0ProcSW1(reader_index, rcv_buf, rcv_len, in_buf, in_len);

		/* Error, unrecognized situation found */
		DEBUG_CRITICAL2("Unrecognized Procedure byte (0x%02X) found!", *in_buf);
		return return_value;
	}

	return return_value;
} /* CmdXfrBlockCHAR_T0 */


/*****************************************************************************
 *
 *					CmdXfrBlockTPDU_T1
 *
 ****************************************************************************/
#if (1)	// CCIDPCSC:0032
/* --- START --- DENSO SERIALΉŁ@CmdXfrBlockTPDU_T1()  --- START --- */
RESPONSECODE CmdXfrBlockTPDU_T1(unsigned int reader_index,
	unsigned int tx_length, unsigned char tx_buffer[], unsigned int *rx_length,
	unsigned char rx_buffer[])
{
	RESPONSECODE return_value = IFD_SUCCESS;
	short	rdata_len;

#ifdef	CCIDPCSC_DEBUG
	printf("CmdXfrBlockTPDU_T1 -------------------------------\n");
#endif	// CCIDPCSC_DEBUG

	return_value = Exec_Command(reader_index, tx_buffer, tx_length,
					rx_buffer, rx_length);
	if(return_value != IFD_SUCCESS) {
#ifdef	CCIDPCSC_DEBUG
		printf("CmdXfrBlockTPDU_T1 Retry\n");
#endif	// CCIDPCSC_DEBUG
		return_value = Exec_Command(reader_index, tx_buffer, tx_length,
						rx_buffer, rx_length);
	}
	    
	if(return_value == IFD_SUCCESS) {
		rdata_len = s2i(rx_buffer,IFD_RCV_CMD_LENGTH_OFFSET);
#ifdef	CCIDPCSC_DEBUG
		printf("rdata_len == %d\n",rdata_len);
#endif	// CCIDPCSC_DEBUG
	}

	return return_value;
} /* CmdXfrBlockTPDU_T1 */
/* --- END --- DENSO SERIALΉŁ@CmdXfrBlockTPDU_T1()  --- END --- */
#else	// CCIDPCSC:0032
static RESPONSECODE CmdXfrBlockTPDU_T1(unsigned int reader_index,
	unsigned int tx_length, unsigned char tx_buffer[], unsigned int *rx_length,
	unsigned char rx_buffer[])
{
	RESPONSECODE return_value = IFD_SUCCESS;
	int ret;

	DEBUG_COMM2("T=1: %d bytes", tx_length);

	ret = t1_transceive(&((get_ccid_slot(reader_index)) -> t1), 0,
		tx_buffer, tx_length, rx_buffer, *rx_length);

	if (ret < 0)
	{
		*rx_length = 0;
		return_value = IFD_COMMUNICATION_ERROR;
	}
	else
		*rx_length = ret;

	return return_value;
} /* CmdXfrBlockTPDU_T1 */
#endif	// CCIDPCSC:0032


/*****************************************************************************
 *
 *					SetParameters
 *
 ****************************************************************************/
RESPONSECODE SetParameters(unsigned int reader_index, char protocol,
	unsigned int length, unsigned char buffer[])
{
	unsigned char cmd[10+CMD_BUF_SIZE];	/* CCID + APDU buffer */
	_ccid_descriptor *ccid_descriptor = get_ccid_descriptor(reader_index);

	DEBUG_COMM2("length: %d bytes", length);

	cmd[0] = 0x61; /* SetParameters */
	i2dw(length, cmd+1);	/* APDU length */
	cmd[5] = ccid_descriptor->bCurrentSlotIndex;	/* slot number */
	cmd[6] = (*ccid_descriptor->pbSeq)++;
	cmd[7] = protocol;	/* bProtocolNum */
	cmd[8] = cmd[9] = 0; /* RFU */

	/* check that the command is not too large */
	if (length > CMD_BUF_SIZE)
		return IFD_NOT_SUPPORTED;

	memcpy(cmd+10, buffer, length);

	if (WritePort(reader_index, 10+length, cmd) != STATUS_SUCCESS)
		return IFD_COMMUNICATION_ERROR;

	length = sizeof(cmd);
	if (ReadPort(reader_index, &length, cmd) != STATUS_SUCCESS)
		return IFD_COMMUNICATION_ERROR;

	if (cmd[STATUS_OFFSET] & CCID_COMMAND_FAILED)
	{
		ccid_error(cmd[ERROR_OFFSET], __FILE__, __LINE__, __FUNCTION__);    /* bError */
		if (0x00 == cmd[ERROR_OFFSET])	/* command not supported */
			return IFD_NOT_SUPPORTED;
		else
			return IFD_COMMUNICATION_ERROR;
	}

	return IFD_SUCCESS;
} /* SetParameters */


/*****************************************************************************
 *
 *					isCharLevel
 *
 ****************************************************************************/
int isCharLevel(int reader_index)
{
	return CCID_CLASS_CHARACTER == (get_ccid_descriptor(reader_index)->dwFeatures & CCID_CLASS_EXCHANGE_MASK);
} /* isCharLevel */


/*****************************************************************************
 *
 *					i2dw
 *
 ****************************************************************************/
static void i2dw(int value, unsigned char buffer[])
{
	buffer[0] = value & 0xFF;
	buffer[1] = (value >> 8) & 0xFF;
	buffer[2] = (value >> 16) & 0xFF;
	buffer[3] = (value >> 24) & 0xFF;
} /* i2dw */



/************************************************************************/
/*									*/
/*	Make XXXXX Command						*/
/*									*/
/************************************************************************/
int Make_InitReset_Cmd(unsigned char *cmd,unsigned int *cmdlen)
{
	cmd[CMD_RCB_OFFSET] 		= CMD_RCB_IFD;
	cmd[CMD_LEN_HIGH_OFFSET] 	= 0x00;
	cmd[CMD_LEN_LOW_OFFSET] 	= READER_INIT_DATA_SIZE;
	cmd[CMD_CLA_OFFSET] 		= 0x00; // CLA
	cmd[CMD_INS_OFFSET] 		= CARD_CTL_RESET; // INS
	cmd[CMD_PARAM1_OFFSET] 		= 0x00; // P1
	cmd[CMD_PARAM2_OFFSET] 		= 0x00; // P2
	cmd[CMD_LE_OFFSET] 		= 0x01; // Le
	cmd[(CMD_DATA_AREA_OFFSET + READER_INIT_DATA_SIZE)] 
		= getBcc(cmd, (CMD_HEADER_SIZE + READER_INIT_DATA_SIZE));

	*cmdlen = (CMD_HEADER_SIZE + READER_INIT_DATA_SIZE + CMD_BCC_SIZE);
	return(0);
}

int Make_Check_Device_Cmd(unsigned char *cmd, unsigned int *cmdlen)
{
	/* OPEN ICC */
	cmd[CMD_RCB_OFFSET] 		= CMD_RCB_IFD;
	cmd[CMD_LEN_HIGH_OFFSET] 	= 0x00;
	cmd[CMD_LEN_LOW_OFFSET] 	= CHK_DEVICE_DATA_SIZE;
	cmd[CMD_CLA_OFFSET] 		= 0x00; // CLA
	cmd[CMD_INS_OFFSET] 		= CHECK_STATUS; // INS
	cmd[CMD_PARAM1_OFFSET] 		= (CHK_DEVICE_DCTL | CHK_DEVICE_CCTL); // P1
	cmd[CMD_PARAM2_OFFSET] 		= 0x00; // P2
	cmd[CMD_LE_OFFSET] 		= 0x01; // P2
	cmd[(CMD_DATA_AREA_OFFSET + CHK_DEVICE_DATA_SIZE)] 
		= getBcc(cmd, (CMD_HEADER_SIZE + READER_INIT_DATA_SIZE));

	*cmdlen = (CMD_HEADER_SIZE + CHK_DEVICE_DATA_SIZE + CMD_BCC_SIZE);
	return(0);
}

int Make_Get_Device_Status_Cmd(unsigned char *cmd, unsigned int *cmdlen)
{
	/* OPEN ICC */
	cmd[CMD_RCB_OFFSET] 		= CMD_RCB_IFD;
	cmd[CMD_LEN_HIGH_OFFSET] 	= 0x00;
	cmd[CMD_LEN_LOW_OFFSET] 	= GET_DEVICE_DATA_SIZE;
	cmd[CMD_CLA_OFFSET] 		= 0x00; // CLA
	cmd[CMD_INS_OFFSET] 		= GET_IFD_STATUS; // INS
	cmd[CMD_PARAM1_OFFSET] 		= 0x00; // P1
	cmd[CMD_PARAM2_OFFSET] 		= 0x00; // P2
	cmd[(CMD_DATA_AREA_OFFSET + GET_DEVICE_DATA_SIZE)] 
		= getBcc(cmd, (CMD_HEADER_SIZE + GET_DEVICE_DATA_SIZE));

	*cmdlen = (CMD_HEADER_SIZE + GET_DEVICE_DATA_SIZE + CMD_BCC_SIZE);
    return(0);
}

int Make_Set_Device_Speed_Cmd(unsigned char *cmd, unsigned int *cmdlen, unsigned char speed)
{
	/* OPEN ICC */
	cmd[CMD_RCB_OFFSET] 		= CMD_RCB_IFD;
	cmd[CMD_LEN_HIGH_OFFSET] 	= 0x00;
	cmd[CMD_LEN_LOW_OFFSET] 	= SET_DEV_SPEED_DATA_SIZE;
	cmd[CMD_CLA_OFFSET] 		= 0x00; // CLA
	cmd[CMD_INS_OFFSET] 		= SET_IFD_SPEED; // INS
	cmd[CMD_PARAM1_OFFSET] 		= speed; // P1
	cmd[CMD_PARAM2_OFFSET] 		= 0x00; // P2
	cmd[(CMD_DATA_AREA_OFFSET + SET_DEV_SPEED_DATA_SIZE)] 
		= getBcc(cmd, (CMD_HEADER_SIZE + SET_DEV_SPEED_DATA_SIZE));

	*cmdlen = (CMD_HEADER_SIZE + SET_DEV_SPEED_DATA_SIZE + CMD_BCC_SIZE);
	return(0);
}

int Make_Reset_CCTL_Cmd(unsigned char *cmd, unsigned int *cmdlen)
{
	/* OPEN ICC */
	cmd[CMD_RCB_OFFSET] 		= CMD_RCB_CCTL;
	cmd[CMD_LEN_HIGH_OFFSET] 	= 0x00;
	cmd[CMD_LEN_LOW_OFFSET] 	= RESET_CCTL_DATA_SIZE;
	cmd[CMD_CLA_OFFSET] 		= 0x00; // CLA
	cmd[CMD_INS_OFFSET] 		= CARD_CTL_RESET; // INS
	cmd[CMD_PARAM1_OFFSET] 		= 0x00; // P1
	cmd[CMD_PARAM2_OFFSET] 		= 0x00; // P2
	cmd[(CMD_DATA_AREA_OFFSET + RESET_CCTL_DATA_SIZE)] 
		= getBcc(cmd, (CMD_HEADER_SIZE + RESET_CCTL_DATA_SIZE));

	*cmdlen = (CMD_HEADER_SIZE + RESET_CCTL_DATA_SIZE + CMD_BCC_SIZE);
	return(0);
}

int Make_Get_CCTL_Info_Cmd(unsigned char *cmd, unsigned int *cmdlen)
{
	/* Get Card Control Infomation */
	cmd[CMD_RCB_OFFSET] 		= CMD_RCB_CCTL;
	cmd[CMD_LEN_HIGH_OFFSET] 	= 0x00;
	cmd[CMD_LEN_LOW_OFFSET] 	= GET_CCTL_INFO_DATA_SIZE;
	cmd[CMD_CLA_OFFSET] 		= 0x00; // CLA
	cmd[CMD_INS_OFFSET] 		= GET_CARD_CTL_STATUS; // INS
	cmd[CMD_PARAM1_OFFSET] 		= 0x00; // P1
	cmd[CMD_PARAM2_OFFSET] 		= 0x00; // P2
	cmd[(CMD_DATA_AREA_OFFSET + GET_CCTL_INFO_DATA_SIZE)] 
		= getBcc(cmd, (CMD_HEADER_SIZE + GET_CCTL_INFO_DATA_SIZE));

	*cmdlen = (CMD_HEADER_SIZE + GET_CCTL_INFO_DATA_SIZE + CMD_BCC_SIZE);
	return(0);
}

int Make_Get_CCTL_Status_Cmd(unsigned char *cmd, unsigned int *cmdlen)
{
	/* Get Card Control Infomation */
	cmd[CMD_RCB_OFFSET] 		= CMD_RCB_CCTL;
	cmd[CMD_LEN_HIGH_OFFSET] 	= 0x00;
	cmd[CMD_LEN_LOW_OFFSET] 	= GET_CCTL_STATUS_DATA_SIZE;
	cmd[CMD_CLA_OFFSET] 		= 0x00; // CLA
	cmd[CMD_INS_OFFSET] 		= GET_CARD_STATUS; // INS
	cmd[CMD_PARAM1_OFFSET] 		= 0x00; // P1
	cmd[CMD_PARAM2_OFFSET] 		= 0x00; // P2
	cmd[(CMD_DATA_AREA_OFFSET + GET_CCTL_STATUS_DATA_SIZE)] 
		= getBcc(cmd, (CMD_HEADER_SIZE + GET_CCTL_STATUS_DATA_SIZE));

	*cmdlen = (CMD_HEADER_SIZE + GET_CCTL_STATUS_DATA_SIZE + CMD_BCC_SIZE);
	return(0);
}

int Make_Carrier_CCTL_ON_Cmd(unsigned char *cmd, unsigned int *cmdlen)
{
	/* CARRIER PCD ON */
	cmd[CMD_RCB_OFFSET] 		= CMD_RCB_CCTL;
	cmd[CMD_LEN_HIGH_OFFSET] 	= 0x00;
	cmd[CMD_LEN_LOW_OFFSET] 	= CMD_CARRIOR_DATA_SIZE;
	cmd[CMD_CLA_OFFSET] 		= 0x00; // CLA
	cmd[CMD_INS_OFFSET] 		= CARRIOR; // INS
	cmd[CMD_PARAM1_OFFSET] 		= CARRIOR_ON; // P1
	cmd[CMD_PARAM2_OFFSET] 		= 0x00; // P2
	cmd[(CMD_DATA_AREA_OFFSET + CMD_CARRIOR_DATA_SIZE)] 
		= getBcc(cmd, (CMD_HEADER_SIZE + CMD_CARRIOR_DATA_SIZE));

	*cmdlen = (CMD_HEADER_SIZE + CMD_CARRIOR_DATA_SIZE + CMD_BCC_SIZE);
	return(0);
}

int Make_Carrier_CCTL_OFF_Cmd(unsigned char *cmd, unsigned int *cmdlen)
{
	/* CARRIER PCD ON */
	cmd[CMD_RCB_OFFSET] 		= CMD_RCB_CCTL;
	cmd[CMD_LEN_HIGH_OFFSET] 	= 0x00;
	cmd[CMD_LEN_LOW_OFFSET] 	= 0x04;
	cmd[CMD_CLA_OFFSET] 		= 0x00; // CLA
	cmd[CMD_INS_OFFSET] 		= CARRIOR; // INS
	cmd[CMD_PARAM1_OFFSET] 		= CARRIOR_OFF; // P1
	cmd[CMD_PARAM2_OFFSET] 		= 0x00; // P2
	cmd[(CMD_DATA_AREA_OFFSET + CMD_CARRIOR_DATA_SIZE)] 
		= getBcc(cmd, (CMD_HEADER_SIZE + CMD_CARRIOR_DATA_SIZE));

	*cmdlen = (CMD_HEADER_SIZE + CMD_CARRIOR_DATA_SIZE + CMD_BCC_SIZE);
	return(0);
}

int Make_Set_CCTL_RcvMode_Cmd(unsigned char *cmd, unsigned int *cmdlen)
{
	/* Get Card Control Infomation */
	cmd[CMD_RCB_OFFSET] 		= CMD_RCB_CCTL;
	cmd[CMD_LEN_HIGH_OFFSET] 	= 0x00;
	cmd[CMD_LEN_LOW_OFFSET] 	= SET_RCV_MODE_DATA_SIZE;
	cmd[CMD_CLA_OFFSET] 		= 0x00; // CLA
	cmd[CMD_INS_OFFSET] 		= SET_RECEIVE_MODE; // INS
	cmd[CMD_PARAM1_OFFSET] 		= RCV_RETRY; // P1
	cmd[CMD_PARAM2_OFFSET] 		= 0x00; // P2
	cmd[(CMD_DATA_AREA_OFFSET + SET_RCV_MODE_DATA_SIZE)] 
		= getBcc(cmd, (CMD_HEADER_SIZE + SET_RCV_MODE_DATA_SIZE));

	*cmdlen = (CMD_HEADER_SIZE + SET_RCV_MODE_DATA_SIZE + CMD_BCC_SIZE);
	return(0);
}

/********************************************************/
/* Make LED Control Command				*/
/********************************************************/
void Make_LED_CTL_Cmd(unsigned char *cmd, unsigned int *cmdlen ,unsigned char mode,
	unsigned short	on_time, unsigned short	off_time)
{
	/* CARRIER PCD ON */
	cmd[CMD_RCB_OFFSET] 		= CMD_RCB_IFD;
	cmd[CMD_LEN_HIGH_OFFSET] 	= 0x00;
	cmd[CMD_LEN_LOW_OFFSET] 	= LED_CTL_DATA_SIZE;
	cmd[CMD_CLA_OFFSET] 		= 0x00; // CLA
	cmd[CMD_INS_OFFSET] 		= LED_CTL; // INS
	cmd[CMD_PARAM1_OFFSET] 		= mode; // P1
	cmd[CMD_PARAM2_OFFSET] 		= 0x00; // P2
	cmd[CMD_LC_OFFSET] 		= LED_BLINK_DATA_SIZE;
	/* ȉ̃f[^LED _ł鎞̂ݗL */
	memcpy(&cmd[LED_CTL_BLINK_ON_OFFSET],&on_time,sizeof(unsigned short));
	memcpy(&cmd[LED_CTL_BLINK_OFF_OFFSET],&off_time,sizeof(unsigned short));
	cmd[(CMD_DATA_AREA_OFFSET + LED_CTL_DATA_SIZE)] 
		= getBcc(cmd, (CMD_HEADER_SIZE + LED_CTL_DATA_SIZE));

	*cmdlen = (CMD_HEADER_SIZE + LED_CTL_DATA_SIZE + CMD_BCC_SIZE);
	return;
}

/********************************************************/
/* Felica pR}h                                    */
/********************************************************/
int Make_Set_Card_Status_Cmd(unsigned char *cmd, unsigned int *cmdlen)
{
	/* Request_All_B */
	cmd[CMD_RCB_OFFSET] 		= CMD_RCB_CCTL;
	cmd[CMD_LEN_HIGH_OFFSET] 	= 0x00;
	cmd[CMD_LEN_LOW_OFFSET] 	= SET_CARD_STATUS_DATA_SIZE;
	cmd[CMD_CLA_OFFSET] 		= 0x00; // CLA
	cmd[CMD_INS_OFFSET] 		= SET_CARD_STATUS; // INS
	cmd[CMD_PARAM1_OFFSET] 		= CARD_TYPE_C; // P1
	cmd[CMD_PARAM2_OFFSET] 		= (CARD_RCV_SPD_B212K | CARD_SND_SPD_B212K); // P2
	cmd[(CMD_DATA_AREA_OFFSET + SET_CARD_STATUS_DATA_SIZE)] 
		= getBcc(cmd, (CMD_HEADER_SIZE + SET_CARD_STATUS_DATA_SIZE));

	*cmdlen = (CMD_HEADER_SIZE + SET_CARD_STATUS_DATA_SIZE + CMD_BCC_SIZE);
	return(0);
}

int Make_Request_All_C_Cmd(unsigned char *cmd, unsigned int *cmdlen)
{
	/* Request_All_C */
	cmd[CMD_RCB_OFFSET] 		= CMD_RCB_CCTL;
	cmd[CMD_LEN_HIGH_OFFSET] 	= 0x00;
	cmd[CMD_LEN_LOW_OFFSET] 	= REQ_ALL_C_CMD_DATA_SIZE;
	cmd[CMD_CLA_OFFSET] 		= 0x00; // CLA
	cmd[CMD_INS_OFFSET] 		= REQUEST_ALL_C; // INS
	cmd[CMD_PARAM1_OFFSET] 		= RFU_VALUE; // P1
	cmd[CMD_PARAM2_OFFSET] 		= REQ_TIMESLOT_F; // P2 (time slot)
	cmd[(CMD_DATA_AREA_OFFSET + REQ_ALL_C_CMD_DATA_SIZE)] 
		= getBcc(cmd, (CMD_HEADER_SIZE + REQ_ALL_C_CMD_DATA_SIZE));

	*cmdlen = (CMD_HEADER_SIZE + REQ_ALL_C_CMD_DATA_SIZE + CMD_BCC_SIZE);
	return(0);
}


/****************************************************************************/
/* 1. Reader Init (INIT PCD) */
/****************************************************************************/
RESPONSECODE ReaderInit(unsigned int reader_index)
{
	unsigned char cmd[IFD_DEV_CMD_SIZE];
	unsigned int length;
	int	datalength;

#ifdef	CCIDPCSC_DEBUG
	printf("ReaderInit (INIT_Reset) -------------------------------\n");
#endif	// CCIDPCSC_DEBUG
	Make_InitReset_Cmd(cmd,&length);
	if (WritePort(reader_index, length, cmd) != STATUS_SUCCESS){
		return IFD_COMMUNICATION_ERROR;
	}
#ifdef	CCIDPCSC_DEBUG
	{
		int i;
		char dbg_buff[300];
		if ( (0 < length) && (length < IFD_CMD_LENGTH_MAX) ) {
			for(i = 0 ; i < length ; i++){
				sprintf(&dbg_buff[i*2],"%02x",cmd[i]);
			}
			printf("Device_Reset Cmd Send. cmd = %s\n",dbg_buff);
		}
	}
#endif	// CCIDPCSC_DEBUG

	if (ReadPort(reader_index, &length, cmd) != STATUS_SUCCESS){
#ifdef	CCIDPCSC_DEBUG
		printf("ReaderInit: ReadPort Error. \n");
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}

	/* Length Check */
	datalength = (int)s2i(cmd,IFD_RCV_CMD_LENGTH_OFFSET);
	if(datalength != (length - (CMD_HEADER_SIZE+CMD_BCC_SIZE))){   /* (RCB + LENGTH + BCC) */
#ifdef	CCIDPCSC_DEBUG
		printf("ReaderInit Receive Length error.\n");
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}

#ifdef	CCIDPCSC_DEBUG
	{
		int i;
		char dbg_buff[300];
		if ( (0 < length) && (length < IFD_CMD_LENGTH_MAX) ) {
			for(i = 0 ; i < length ; i++){
				sprintf(&dbg_buff[i*2],"%02x",cmd[i]);
			}
			printf("Device_Reset Cmd Ans. cmd = %s\n",dbg_buff);
		}
	}
#endif	// CCIDPCSC_DEBUG

	return IFD_SUCCESS;
}
/****************************************************************************/
/* 2. Check Device Ctl */
/****************************************************************************/
RESPONSECODE ChkDeviceStatus(unsigned int reader_index,unsigned char * buffer)
{
	unsigned char cmd[IFD_DEV_CMD_SIZE];
	int length;
	int datalength;

#ifdef	CCIDPCSC_DEBUG
	printf("CHECK DEVICE STATUS -------------------------------\n");
#endif	// CCIDPCSC_DEBUG
	Make_Check_Device_Cmd(cmd,&length);

	if (WritePort(reader_index, length, cmd) != STATUS_SUCCESS){
		return IFD_COMMUNICATION_ERROR;
	}

	if (ReadPort(reader_index, &length, buffer) != STATUS_SUCCESS){
#ifdef	CCIDPCSC_DEBUG
		printf("ChkDeviceStatus ReadPort error.\n");
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}
        /* Length Check */
	datalength = (int)s2i(buffer,IFD_RCV_CMD_LENGTH_OFFSET);
	if(datalength != (length - (CMD_HEADER_SIZE+CMD_BCC_SIZE))){   /* (RCB + LENGTH + BCC) */
#ifdef	CCIDPCSC_DEBUG
		printf("ChkDeviceStatus Receive Length error.\n");
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}


	memcpy(buffer,cmd,length) ;

	return IFD_SUCCESS;
}
/****************************************************************************/
/* 3. Get Device Status */
/****************************************************************************/
RESPONSECODE GetDeviceStatus(unsigned int reader_index,unsigned char * buffer)
{
	unsigned char cmd[IFD_DEV_CMD_SIZE];
	int length;
	int datalength;

#ifdef	CCIDPCSC_DEBUG
	printf("GET DEVICE STATUS -------------------------------\n");
#endif	// CCIDPCSC_DEBUG
	Make_Get_Device_Status_Cmd(cmd,&length);

	if (WritePort(reader_index, length, cmd) != STATUS_SUCCESS){
		return IFD_COMMUNICATION_ERROR;
	}

	if (ReadPort(reader_index, &length, buffer) != STATUS_SUCCESS){
#ifdef	CCIDPCSC_DEBUG
		printf("GetDeviceStatus ReadPort error.\n");
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}
	/* Length Check */
	datalength = (int)s2i(buffer,IFD_RCV_CMD_LENGTH_OFFSET);
	if(datalength != (length - (CMD_HEADER_SIZE+CMD_BCC_SIZE))){   /* (RCB + LENGTH + BCC) */
#ifdef	CCIDPCSC_DEBUG
		printf("GetDeviceStatus Receive Length error.\n");
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}

	return IFD_SUCCESS;
}

/****************************************************************************/
/* 3.5  Set Device Speed */
/****************************************************************************/
RESPONSECODE SetDeviceSpeed(unsigned int reader_index ,unsigned char speed)
{
	unsigned char cmd[IFD_DEV_CMD_SIZE];
	unsigned char buffer[IFD_DEV_CMD_SIZE];
	int length;
	int datalength;

#ifdef	CCIDPCSC_DEBUG
	printf("GET DEVICE STATUS -------------------------------\n");
#endif	// CCIDPCSC_DEBUG
	Make_Set_Device_Speed_Cmd(cmd,&length,speed);

	if (WritePort(reader_index, length, cmd) != STATUS_SUCCESS){
		return IFD_COMMUNICATION_ERROR;
	}

#ifdef	CCIDPCSC_DEBUG
	{
		char dbg_buff[IFD_DEV_CMD_SIZE];
		int i;
		if ( (0 < length) && ( length < IFD_CMD_LENGTH_MAX)){
			for(i = 0 ; i < length ; i++){
				sprintf(&dbg_buff[i*2],"%02x",cmd[i]);
			}
			printf("SetDeviceSpeed cmd == %s\n",dbg_buff);
		}
	}
#endif	// CCIDPCSC_DEBUG

	if (ReadPort(reader_index, &length, buffer) != STATUS_SUCCESS){
#ifdef	CCIDPCSC_DEBUG
		printf("SetDeviceSpeed ReadPort error.\n");
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}
	/* Length Check */
	datalength = (int)s2i(buffer,IFD_RCV_CMD_LENGTH_OFFSET);
	if(datalength != (length - (CMD_HEADER_SIZE+CMD_BCC_SIZE))){   /* (RCB + LENGTH + BCC) */
#ifdef	CCIDPCSC_DEBUG
		printf("SetDeviceSpeed Receive Length error. length == %d\n",length);
		{
			int i;
			char dbg_buff[300];
			for(i = 0 ; i < 6 ; i++){
				sprintf(&dbg_buff[i*2],"%02x",buffer[i]);
			}
			printf("Device_LED_Control Cmd Ans. cmd = %s\n",dbg_buff);
		}
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}

#ifdef	CCIDPCSC_DEBUG
	{
		char dbg_buff[IFD_DEV_CMD_SIZE];
		int i;
		if ( (0 < length) && ( length < IFD_CMD_LENGTH_MAX)){
			for(i = 0 ; i < length ; i++){
				sprintf(&dbg_buff[i*2],"%02x",buffer[i]);
			}
			printf("SetDeviceSpeed buffer == %s\n",dbg_buff);
		}
	}
#endif	// CCIDPCSC_DEBUG
	if(buffer[3] != 0x90){
		return IFD_COMMUNICATION_ERROR;
	}
	
	return IFD_SUCCESS;
}

/****************************************************************************/
/* 4. Reset Cart Control */
/****************************************************************************/
RESPONSECODE ResetCardCtl(unsigned int reader_index,unsigned char *buffer)
{
	unsigned char cmd[IFD_DEV_CMD_SIZE];
	int length;
	int datalength;

#ifdef	CCIDPCSC_DEBUG
	printf("Reset Card Control -------------------------------\n");
#endif	// CCIDPCSC_DEBUG
	Make_Reset_CCTL_Cmd(cmd,&length);

	if (WritePort(reader_index, length, cmd) != STATUS_SUCCESS){
		return IFD_COMMUNICATION_ERROR; 
	}
#ifdef	CCIDPCSC_DEBUG
	{
		int i;
		char dbg_buff[300];
		if( (length < 0) && ( length < IFD_CMD_LENGTH_MAX)) {
			for(i = 0 ; i < length ; i++){
				sprintf(&dbg_buff[i*2],"%02x",cmd[i]);
			}
			printf("Device_Reset_CCTL Cmd Send. cmd = %s\n",dbg_buff);
		}
	}
#endif	// CCIDPCSC_DEBUG

	if (ReadPort(reader_index, &length, cmd) != STATUS_SUCCESS){
#ifdef	CCIDPCSC_DEBUG
		printf("ResetCardCtl ReadPort error.\n");
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}
	/* Length Check */
	datalength = (int)s2i(cmd,IFD_RCV_CMD_LENGTH_OFFSET);
	if(datalength != (length - (CMD_HEADER_SIZE+CMD_BCC_SIZE))){   /* (RCB + LENGTH + BCC) */
#ifdef	CCIDPCSC_DEBUG
		printf("ResetCardCtl Receive Length error.\n");
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}


#ifdef	IFD_MDEBUG
	{
		int i;
		char dbg_buff[300];
		if( (length < 0) && ( length < IFD_CMD_LENGTH_MAX)) {
			for(i = 0 ; i < length ; i++){
				sprintf(&dbg_buff[i*2],"%02x",cmd[i]);
			}
#ifdef	CCIDPCSC_DEBUG
			printf("Device_Reset_CCTL Cmd Ans. cmd = %s\n",dbg_buff);
#endif	// CCIDPCSC_DEBUG
		}
	}
#endif
	memcpy(buffer,cmd,length) ;

	return IFD_SUCCESS;
}
/****************************************************************************/
/* 7. Smart Card Carrier ON */
/****************************************************************************/
RESPONSECODE CarrierON(unsigned int reader_index)
{
	unsigned char cmd[IFD_DEV_CMD_SIZE];
	int length;
	int datalength;

#ifdef	CCIDPCSC_DEBUG
	printf("Smart Card Carrier ON -------------------------------\n");
#endif	// CCIDPCSC_DEBUG
	Make_Carrier_CCTL_ON_Cmd(cmd,&length);

	if (WritePort(reader_index, length, cmd) != STATUS_SUCCESS) {
		return IFD_COMMUNICATION_ERROR;
	}
#ifdef	CCIDPCSC_DEBUG
	{
		int i;
		char dbg_buff[300];
		if( (length < 0) && ( length < IFD_CMD_LENGTH_MAX)) {
			for(i = 0 ; i < length ; i++){
				sprintf(&dbg_buff[i*2],"%02x",cmd[i]);
			}
			printf("Device_Carrier_ON Cmd Send. cmd = %s\n",dbg_buff);
		}
	}
#endif	// CCIDPCSC_DEBUG

	if (ReadPort(reader_index, &length, cmd) != STATUS_SUCCESS){
#ifdef	CCIDPCSC_DEBUG
		printf("CarrierON ReadPort error.\n");
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}

	/* Length Check */
	datalength = (int)s2i(cmd,IFD_RCV_CMD_LENGTH_OFFSET);
	if(datalength != (length - (CMD_HEADER_SIZE+CMD_BCC_SIZE))){   /* (RCB + LENGTH + BCC) */
#ifdef	CCIDPCSC_DEBUG
		printf("CarrierON Receive Length error.\n");
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}

#ifdef	CCIDPCSC_DEBUG
	{
		int i;
		char dbg_buff[300];
		if( (length < 0) && ( length < IFD_CMD_LENGTH_MAX)) {
			for(i = 0 ; i < length ; i++){
				sprintf(&dbg_buff[i*2],"%02x",cmd[i]);
			}
			printf("Device_Carrier_ON Cmd Ans. cmd = %s\n",dbg_buff);
		}
	}
#endif	// CCIDPCSC_DEBUG

	return IFD_SUCCESS;
}
/****************************************************************************/
/* 8. Smart Card Carrier OFF */
/****************************************************************************/
RESPONSECODE CarrierOFF(unsigned int reader_index)
{
	unsigned char cmd[IFD_DEV_CMD_SIZE];
	int length;
	int datalength;

#ifdef	CCIDPCSC_DEBUG
	printf("Smart Card Carrier OFF -------------------------------\n");
#endif	// CCIDPCSC_DEBUG
	Make_Carrier_CCTL_OFF_Cmd(cmd,&length);

	if (WritePort(reader_index, length, cmd) != STATUS_SUCCESS){
		return IFD_COMMUNICATION_ERROR;
	}
	if (ReadPort(reader_index, &length, cmd) != STATUS_SUCCESS){
#ifdef	CCIDPCSC_DEBUG
		printf("CarrierOFF ReadPort error.\n");
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}

	/* Length Check */
	datalength = (int)s2i(cmd,IFD_RCV_CMD_LENGTH_OFFSET);
	if(datalength != (length - (CMD_HEADER_SIZE+CMD_BCC_SIZE))){   /* (RCB + LENGTH + BCC) */
#ifdef	CCIDPCSC_DEBUG
		printf("CarrierOFF Receive Length error.\n");
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}

	return IFD_SUCCESS;
}
/****************************************************************************/
/* 9. Set Card Status */
/****************************************************************************/
RESPONSECODE SetCardStatus(unsigned int reader_index)
{
	unsigned char cmd[IFD_DEV_CMD_SIZE];
	unsigned char buffer[256];
	int length;
	int datalength;

	Make_Set_Card_Status_Cmd(cmd,&length);

	if (WritePort(reader_index, length, cmd) != STATUS_SUCCESS){
		return IFD_COMMUNICATION_ERROR;
	}

	if (ReadPort(reader_index, &length, buffer) != STATUS_SUCCESS){
#ifdef	CCIDPCSC_DEBUG
		printf("SetCardStatus ReadPort error.\n");
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}
	/* Length Check */
	datalength = (int)s2i(buffer,IFD_RCV_CMD_LENGTH_OFFSET);
	if(datalength != (length - (CMD_HEADER_SIZE+CMD_BCC_SIZE))){   /* (RCB + LENGTH + BCC) */
#ifdef	CCIDPCSC_DEBUG
		printf("SetCardStatus Receive Length error.\n");
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}

	if(buffer[3] != 0x90){
		/* Not type-C Mode */
		return IFD_COMMUNICATION_ERROR;
	}
	return IFD_SUCCESS;
}
/****************************************************************************/
/* 8. Set Receive Mode  */
/****************************************************************************/
RESPONSECODE SetReceiveMode(unsigned int reader_index)
{
	unsigned char cmd[IFD_DEV_CMD_SIZE];
	int length;
	int datalength;

#ifdef	CCIDPCSC_DEBUG
	printf("Smart Card Carrier OFF -------------------------------\n");
#endif	// CCIDPCSC_DEBUG
	Make_Set_CCTL_RcvMode_Cmd(cmd,&length);

	if (WritePort(reader_index, length, cmd) != STATUS_SUCCESS){
		return IFD_COMMUNICATION_ERROR;
	}

	if (ReadPort(reader_index, &length, cmd) != STATUS_SUCCESS){
#ifdef	CCIDPCSC_DEBUG
		printf("SetReceiveMode ReadPort error.\n");
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}
	/* Length Check */
	datalength = (int)s2i(cmd,IFD_RCV_CMD_LENGTH_OFFSET);
	if(datalength != (length - (CMD_HEADER_SIZE+CMD_BCC_SIZE))){   /* (RCB + LENGTH + BCC) */
#ifdef	CCIDPCSC_DEBUG
		printf("SetReceiveMode Receive Length error.\n");
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}


	return IFD_SUCCESS;
}
/****************************************************************************/
/* 10. Request All C */
/****************************************************************************/
RESPONSECODE RequestCard(unsigned int reader_index,unsigned char *buffer)
{
	unsigned char cmd[IFD_DEV_CMD_SIZE];
	unsigned char rcv_buff[IFD_DEV_CMD_SIZE];
	int length;
	int rcv_length;
	int datalength;
	int ret;
	int retry_flag;
	int retry_count=0;
	short	rdata_len;

#ifdef	CCIDPCSC_DEBUG
	printf("Reqest All C -------------------------------\n");
#endif	// CCIDPCSC_DEBUG
	Make_Request_All_C_Cmd(cmd,&length);
#if 1
	do {
		retry_flag = IFD_OFF;
		ret = Exec_Command(reader_index,cmd,length,rcv_buff,&rcv_length);
		if(ret != IFD_SUCCESS){
			retry_flag = IFD_ON;
		}
	    else{
			rdata_len = s2i(rcv_buff,IFD_RCV_CMD_LENGTH_OFFSET);
			/* J[h̉Ȃ */
			if(rdata_len == 1) {
				switch(rcv_buff[REQ_ALL_C_ERR_DATA_OFFSET]){
				case 0xef:
				case 0xed:
					retry_flag = IFD_ON;
#ifdef	CCIDPCSC_DEBUG
					printf("No Card Response \n");
#endif	// CCIDPCSC_DEBUG
					break;
				default:
					retry_flag = IFD_OFF;
					break;
				}
			}
			/* J[h̉͂AJ[h񂪎擾oĂȂ */
			if(rcv_buff[REQ_ALL_C_CARD_COUNT] == CARD_NONE){
				retry_flag = IFD_ON;
				CarrierOFF(reader_index);
				CarrierON(reader_index);
				usleep(IFD_CARRIER_WAIT);
#ifdef	CCIDPCSC_DEBUG
				printf("RequestCard : No Card Response \n");
#endif	// CCIDPCSC_DEBUG
			}
#ifdef	CCIDPCSC_DEBUG
			printf("RequestCard : retry_flag OFF \n");
#endif	// CCIDPCSC_DEBUG
			if(retry_flag == IFD_OFF){
				break;
			}	
		}
		retry_count++;
	} while(retry_count < 2);
#ifdef	CCIDPCSC_DEBUG
	printf("RequestCard : retry_Count == %d \n",retry_count);
#endif	// CCIDPCSC_DEBUG
	if(retry_count >= 2) {
#ifdef	CCIDPCSC_DEBUG
		printf("RequestCard : retry OUT \n" );
#endif	// CCIDPCSC_DEBUG
	}
	if(ret != IFD_SUCCESS){
		return IFD_COMMUNICATION_ERROR;
	}
#else
	ret = Exec_Command(reader_index,cmd,length,rcv_buff,&rcv_length);
	if(ret != IFD_SUCCESS){
		return IFD_COMMUNICATION_ERROR;
	}
#endif
	/* Length Check */
	datalength = (int)s2i(rcv_buff,IFD_RCV_CMD_LENGTH_OFFSET);
	if(datalength != (rcv_length - (CMD_HEADER_SIZE + CMD_BCC_SIZE))){	/* (RCB + LENGTH + BCC) */
#ifdef	CCIDPCSC_DEBUG
		printf("Device Polling Receive Length error. %d RcvLength = %x\n",(length - (1+2+1)),datalength);
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}
#if 1
	/* Read Buffer Check */
	if((rcv_buff[CMD_RESULT_SW1_OFFSET(rcv_length)] == (unsigned char)CMD_RESULT_SW1_SUCCESS) &&
		(rcv_buff[CMD_RESULT_SW2_OFFSET(rcv_length)] == (unsigned char)CMD_RESULT_SW2_NOMAL)) {

#ifdef	CCIDPCSC_DEBUG
		printf("Command Result,%02x%02x\n",rcv_buff[CMD_RESULT_SW1_OFFSET(rcv_length)],
				rcv_buff[CMD_RESULT_SW2_OFFSET(rcv_length)]);
#endif	// CCIDPCSC_DEBUG
		/* Cards Collision ? */
		if(rcv_buff[REQ_ALL_C_CARD_COLLISION] & (char)CARD_COLLISION){

#ifdef	CCIDPCSC_DEBUG
			printf("Cards Collision.\n");
#endif	// CCIDPCSC_DEBUG
			return	IFD_COMMUNICATION_ERROR ;

		}else if(rcv_buff[REQ_ALL_C_CARD_COLLISION] == (char)CARD_NO_COLLISION) {

			if(rcv_buff[REQ_ALL_C_CARD_COUNT] == (char)CARD_NONE) {
#ifdef	CCIDPCSC_DEBUG
				printf("Nothing Card on the Device.\n");
#endif	// CCIDPCSC_DEBUG
				return IFD_NO_CARD ;
			}
		}
	} else {
#ifdef	CCIDPCSC_DEBUG
		printf("Card ERROR \n");
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR ;

	}    
#endif

#ifdef	CCIDPCSC_DEBUG
	{
		int i;
		char dbg_buff[300];
		if( (length < 0) && ( length < IFD_CMD_LENGTH_MAX)) {
			for(i = 0 ; i < length ; i++){
				sprintf(&dbg_buff[i*2],"%02x",rcv_buff[i]);
			}
			printf("Device_Polling Cmd Ans. cmd = 0x%s\n",dbg_buff);
		}
	}
#endif	// CCIDPCSC_DEBUG
	memcpy(ATR_buf, &rcv_buff[IFD_RCV_CMD_DATA_OFFSET], IFD_IDM_DATA_SIZE);
	memcpy(buffer, &rcv_buff[IFD_RCV_CMD_DATA_OFFSET], IFD_IDM_DATA_SIZE);

	return IFD_SUCCESS;
}


/****************************************************************************/
/* 12. LED Control */
/****************************************************************************/
RESPONSECODE Device_LED_Control(unsigned int reader_index, unsigned char mode, unsigned short on_time,
	unsigned short off_time)
{

	unsigned char cmd[IFD_DEV_CMD_SIZE];
	unsigned int cmdlen;
	unsigned int length;
	int datalength;

#ifdef	CCIDPCSC_DEBUG
	printf("Device_LED_Control -------------------------------\n");
#endif	// CCIDPCSC_DEBUG
	/* R}hҏW */
	Make_LED_CTL_Cmd(cmd, &cmdlen , mode, on_time, off_time);

	/* R}hfoCX֑M */
	if (WritePort(reader_index, cmdlen, cmd) != STATUS_SUCCESS){
		return IFD_COMMUNICATION_ERROR;
	}
#ifdef	CCIDPCSC_DEBUG
	{
		int i;
		char dbg_buff[300];
		if( (cmdlen < 0) && ( cmdlen < IFD_CMD_LENGTH_MAX)) {
			for(i = 0 ; i < cmdlen ; i++){
				sprintf(&dbg_buff[i*2],"%02x",cmd[i]);
			}
			printf("Device_LED_Control Cmd Send. cmd = %s\n",dbg_buff);
		}
	}
#endif	// CCIDPCSC_DEBUG

	/* foCX̃R}hM */
	if (ReadPort(reader_index, &length, cmd) != STATUS_SUCCESS){
#ifdef	CCIDPCSC_DEBUG
		printf("Device_LED_Control ReadPort error.\n");
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}

	/* Length Check */
	datalength = (int)s2i(cmd,IFD_RCV_CMD_LENGTH_OFFSET);
	if(datalength != (length - (CMD_HEADER_SIZE+CMD_BCC_SIZE))){   /* (RCB + LENGTH + BCC) */
#ifdef	CCIDPCSC_DEBUG
		printf("Device_LED_Control Receive Length error.\n");
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}

#ifdef	CCIDPCSC_DEBUG
	{
		int i;
		char dbg_buff[300];
		if( (length < 0) && ( length < IFD_CMD_LENGTH_MAX)) {
			for(i = 0 ; i < length ; i++){
				sprintf(&dbg_buff[i*2],"%02x",cmd[i]);
			}
			printf("Device_LED_Control Cmd Ans. cmd = %s\n",dbg_buff);
		}
	}
#endif	// CCIDPCSC_DEBUG
	return IFD_SUCCESS;
}


/****************************************************************************/
/* 12. Reader Start (OPEN ICC) */
/****************************************************************************/
RESPONSECODE ReaderStart(unsigned int reader_index)
{
	unsigned char res[IFD_DEV_CMD_SIZE];
	RESPONSECODE return_value = IFD_SUCCESS;

	// J[h䕔Zbg
	// return_value = ResetCardCtl(lun,res);
	return_value = ResetCardCtl(reader_index,res);
	if(return_value != IFD_SUCCESS){
#ifdef	CCIDPCSC_DEBUG
		printf("CCID_S:CMD: Reset Cart Ctl Error.\n");
#endif	// CCIDPCSC_DEBUG
		return return_value;
	}

	// foCX擾
	return_value = ChkDeviceStatus(reader_index,res);
	if(return_value != IFD_SUCCESS){
#ifdef	CCIDPCSC_DEBUG
		printf("CCID_S:CMD: Check Device Sts Error.\n");
#endif	// CCIDPCSC_DEBUG
		return return_value;
	}

	// foCX擾
	return_value = GetDeviceStatus(reader_index,res);
	if(return_value != IFD_SUCCESS){
#ifdef	CCIDPCSC_DEBUG
		printf("CCID_S:CMD: Get Dev Sts Error.\n");
#endif	// CCIDPCSC_DEBUG
		return return_value;
	}
	
	// LAON
	return_value = CarrierON(reader_index);
	if(return_value != IFD_SUCCESS){
#ifdef	CCIDPCSC_DEBUG
		printf("CCID_S:CMD: Carrier ON Error.\n");
#endif	// CCIDPCSC_DEBUG
		return return_value;
	}
	// Card Control Set
	return_value = SetCardStatus(reader_index);
	if(return_value != IFD_SUCCESS){
#ifdef	CCIDPCSC_DEBUG
		printf("CCID_S:CMD:Set Card CTL Status Error.\n");
#endif	// CCIDPCSC_DEBUG
		return return_value;
	}
	// Card Control Set
	return_value = SetReceiveMode(reader_index);
	if(return_value != IFD_SUCCESS){
#ifdef	CCIDPCSC_DEBUG
		printf("CCID_S:CMD:Set Card CTL Status Error.\n");
#endif	// CCIDPCSC_DEBUG
		return return_value;
	}

	return IFD_SUCCESS;
}

/* calculate BCC */
char getBcc(char* str, int len)
{
	int i;
	char ret = 0;

	for (i = 0; i < len; i++)
		ret ^= *(str + i);

	return ret;
}

int bitcnt( unsigned char prm )
{
	int count;

	for(count = 0 ; 0 != prm ; prm = (unsigned char)(prm >> 1)){
		if(prm & 0x01){
			count++;
		}
	}
	return count;
}

int membitcnt( void *data, int size )
{
	int count;
	unsigned char *buff;

	buff = (unsigned char *)data;
	for(count = 0;size > 0;size--,buff++){
		count = count + bitcnt( *buff );
	}
	return count;
}


/***************************************************************************/

RESPONSECODE Exec_Command(unsigned int reader_index, unsigned char *sendbuf, unsigned int sendlen, unsigned char *resbuf, unsigned int *reslen)
{
	int	data_length;

	if ( WriteSerial(reader_index, sendlen, sendbuf) != STATUS_SUCCESS  ){
#ifdef	CCIDPCSC_DEBUG
		printf("ICCIFD:WriteSerial Error ! \n");
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}
    
	if (ReadPort(reader_index, reslen, resbuf) == STATUS_SUCCESS){

		/* J[h̉̏ꍇ */
		/* (R}hʂ͐툵j	  */
		data_length = (int)s2i(resbuf,IFD_RCV_CMD_LENGTH_OFFSET);
		if((data_length == 0) || (data_length > IFD_CMD_BUFFER_MAX)){
#ifdef	CCIDPCSC_DEBUG
			printf("ICCIFD:ExecCommand : ReadPort Length Error ! data_length == %d\n",data_length);
#endif	// CCIDPCSC_DEBUG
			return IFD_COMMUNICATION_ERROR;
		}
#ifdef	CCIDPCSC_DEBUG
{
	int i;
	char dbg_buff[512];
	if( (*reslen > 0) && ( *reslen < IFD_CMD_LENGTH_MAX)) {
		for(i = 0 ; i < *reslen ; i++){
			sprintf(&dbg_buff[i*2],"%02x",resbuf[i]);
		}
		printf("Exec_Command Auth Cmd Ans. cmd = %s reslen = %d\n",dbg_buff,*reslen);
	}
}
#endif	// CCIDPCSC_DEBUG
	} else {
#ifdef	CCIDPCSC_DEBUG
		printf("Exec_Command : Receive NG!!! ret = %x\n",ret);
#endif	// CCIDPCSC_DEBUG
		return IFD_COMMUNICATION_ERROR;
	}

	return IFD_SUCCESS;
}

