/* $USAGI: ipsec4_input.c,v 1.21 2003/05/22 18:18:37 mk Exp $ */
/*
 * Copyright (C)2001 USAGI/WIDE Project
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Authors:
 *   Kazunori MIYAZAWA <miyazawa@linux-ipv6.org> / USAGI
 *   Mitsuru KANDA <mk@linux-ipv6.org> / USAGI
 *
 */
#include <linux/config.h>
#include <linux/module.h>
#include <linux/init.h>
#include <asm/byteorder.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/sysctl.h>
#include <linux/inet.h>
#include <linux/ipv6.h>
#include <linux/tcp.h>
#include <linux/udp.h>
#include <linux/smp.h>
#include <linux/list.h>
#include <asm/uaccess.h>
#include <net/checksum.h>
#include <net/ip.h>
#include <net/sadb.h>
#include <net/spd.h>
#include <net/addrconf.h>
#include <net/snmp.h>  
#include <linux/in.h>
#include <linux/ipsec.h>
#include <linux/pfkeyv2.h> /* sa proto type */
#include <linux/pfkey.h>

/* XXX must insert window size  sa_replay_window member */
static int check_replay_window(struct sa_replay_window *rw, __u32 hdrseq)
{	
	__u32 diff;
	__u32 seq = ntohl(hdrseq);

	if (!sysctl_ipsec_replay_window) {
		IPSEC4_DEBUG("disable replay window check, skip!\n");
		return 1;
	}

	IPSEC4_DEBUG("overflow: %u\n" 
		     "    size: %u\n" 
		     " seq_num: %x\n" 
		     "last_seq: %x\n" 
		     "  bitmap: %08x\n" 
		     " curr seq: %x\n",
			rw->overflow, rw->size, rw->seq_num, rw->last_seq, rw->bitmap, seq);
	if (seq == 0) {
		return 0; /* first == 0 or wrapped */
	}

	if (seq > rw->last_seq) return 1; /* larger is good */

	diff = rw->last_seq - seq;

	if (diff >= rw->size) {
		return 0; /* too old or wrapped */
	}

	if ( rw->bitmap & ((u_long)1 << diff) ) {
		return 0; /* already seen */
	}

	return 1; /* out of order but good */
}

static void update_replay_window(struct sa_replay_window *rw, __u32 hdrseq)
{
	__u32 diff;
	__u32 seq = ntohl(hdrseq);

	if (!sysctl_ipsec_replay_window) {
		IPSEC4_DEBUG("disable replay window check, skip!\n");
		return;
	}

	if (seq == 0) return;

	if (seq > rw->last_seq) { /* new larger sequence number */
		diff = seq - rw->last_seq;
		if (diff < rw->size) {  /* In window */
			rw->bitmap <<= diff;
			rw->bitmap |= 1; /* set bit for this packet */
		} else {
			rw->bitmap = 1; /* This packet has a "way larger" */
		}

		rw->last_seq = seq;
	}

	diff = rw->last_seq - seq;
	rw->bitmap |= ((u_long)1 << diff); /* mark as seen */
}

static int ipsec4_input_check_ah(struct sk_buff **skb, struct ip_auth_hdr *authhdr, struct sa_index *sa_idx, u8 *nexthdr)
{
	int rtn = 0;
	__u8* authdata;
	size_t authsize;
	char *packet;
	int offset;
	struct ip_auth_hdr *pseudo_authhdr = NULL;

	IPSEC4_DEBUG("start auth header processing\n");

	if (!((*skb)&&authhdr)) {
		IPSEC4_DEBUG("parameters is invalid\n");
		rtn = -EINVAL;
		goto finish;
	}

	/* Check SPI */
	IPSEC4_DEBUG("authhdr->spi is 0x%x\n", ntohl(authhdr->spi));

	sa_index_init(sa_idx);
	((struct sockaddr_in *)&sa_idx->dst)->sin_addr.s_addr = (*skb)->nh.iph->daddr;
	((struct sockaddr_in *)&sa_idx->dst)->sin_family = AF_INET;
	sa_idx->prefixlen_d = 32;
	sa_idx->ipsec_proto = SADB_SATYPE_AH;
	sa_idx->spi = authhdr->spi;

	sa_idx->sa = sadb_find_by_sa_index(sa_idx);

	if (!sa_idx->sa) {
		if (net_ratelimit()) {
			printk(KERN_ERR "IPSEC: SA(%d.%d.%d.%d AH %d): Can't find SA\n",
			       NIPQUAD(((struct sockaddr_in *)&sa_idx->dst)->sin_addr),
			       ntohs(sa_idx->spi));
		}
		rtn = -EINVAL;
		goto finish;
	}

	write_lock_bh(&sa_idx->sa->lock);

	if (sa_idx->sa->auth_algo.algo == SADB_AALG_NONE ) {
		if (net_ratelimit())
			IPSEC4_DEBUG("not found auth algo.\n");
		rtn = -EINVAL;
		goto unlock_finish;
	}

	if (!check_replay_window(&sa_idx->sa->replay_window, authhdr->seq_no)) {
		if (net_ratelimit()) {
			printk(KERN_ERR "IPSEC: SA(%d.%d.%d.%d AH %d): Replay check error\n",
				NIPQUAD(((struct sockaddr_in *)&sa_idx->dst)->sin_addr), 
				ntohs(sa_idx->spi));
		}
		rtn = -EINVAL;
		goto unlock_finish;
	}

	authsize = ntohs((*skb)->nh.iph->tot_len);

	if (authsize > (*skb)->len + ((char*)(*skb)->data - (char*)(*skb)->nh.iph)) {
		if (net_ratelimit())
			IPSEC4_DEBUG("the packet length is wrong\n");
		rtn = -EINVAL;
		goto unlock_finish;
	}

	packet = kmalloc(((authsize + 3) & ~3) + 
			sa_idx->sa->auth_algo.dx->di->blocksize, GFP_ATOMIC);
	if (!packet) {
		if (net_ratelimit())
			IPSEC4_DEBUG("can't get memory for pakcet\n");
		rtn = -ENOMEM;
		goto unlock_finish;
	}
	authdata = packet + ((authsize + 3) & ~3);

	offset = (char *)((*skb)->nh.iph) - (char *)((*skb)->data);

	if (skb_copy_bits(*skb, offset, packet, authsize)) {
		if (net_ratelimit())
			IPSEC4_DEBUG("packet copy failed\n");
		rtn = -EINVAL;
		goto unlock_finish;
	}

	((struct iphdr*)packet)->tos      = 0;
	((struct iphdr*)packet)->frag_off = 0;
	((struct iphdr*)packet)->ttl      = 0;
	((struct iphdr*)packet)->check    = 0;

	pseudo_authhdr = (struct ip_auth_hdr*)(packet + (((char*)authhdr) - ((char*)(*skb)->nh.iph)));
	memset(&pseudo_authhdr->auth_data[0], 0, ((authhdr->hdrlen - 1) << 2));

	sa_idx->sa->auth_algo.dx->di->hmac_atomic(sa_idx->sa->auth_algo.dx,
			sa_idx->sa->auth_algo.key,
			sa_idx->sa->auth_algo.key_len,
			packet, authsize, authdata);	 

	/* Originally, IABG uses "for" loop for matching authentication data.	*/
	/* I change it into memcmp routine.					*/
	if (memcmp(authdata, authhdr->auth_data, sa_idx->sa->auth_algo.digest_len)) {
		if (net_ratelimit()) {
			printk(KERN_ERR "IPSEC: SA(%d.%d.%d.%d AH %d): Invalid checksum\n",
			       NIPQUAD(((struct sockaddr_in *)&sa_idx->dst)->sin_addr), 
			       ntohs(sa_idx->spi));
		}
		kfree(packet);
		rtn = IPSEC_ACTION_DROP;
		goto unlock_finish;
	}
	kfree(packet);

	rtn = IPSEC_ACTION_AUTH;
	*nexthdr = authhdr->nexthdr;
	update_replay_window(&sa_idx->sa->replay_window, authhdr->seq_no);

	(*skb)->security |= RCV_AUTH; /* ? we must rewrite linux/ipsec.h */
	{
		int authhdrlen = (authhdr->hdrlen + 2) << 2;
		int payloadlen = (*skb)->len - (*skb)->data_len - authhdrlen;
		IPSEC4_DEBUG("authhdrlen=%d, payloadlen=%d\n", authhdrlen, payloadlen);
		if (payloadlen > 0) {
			memmove(authhdr, ((char*)authhdr) + authhdrlen, payloadlen);
			(*skb)->len -= authhdrlen;
			(*skb)->nh.iph->protocol = *nexthdr;
			(*skb)->nh.iph->tot_len = htons(ntohs((*skb)->nh.iph->tot_len) - authhdrlen);			
			(*skb)->nh.iph->check = ip_fast_csum((unsigned char*)((*skb)->nh.iph), (*skb)->nh.iph->ihl);
		} else {
			rtn = -EINVAL;
			goto unlock_finish;
		}
	}


	if (!sa_idx->sa->fuse_time) {
		sa_idx->sa->fuse_time = jiffies;
		sa_idx->sa->lifetime_c.usetime = (sa_idx->sa->fuse_time) / HZ;
		ipsec_sa_mod_timer(sa_idx->sa);
		IPSEC4_DEBUG("set fuse_time = %lu\n", sa_idx->sa->fuse_time);
	}
	sa_idx->sa->lifetime_c.bytes += (*skb)->tail - (*skb)->head;
	IPSEC4_DEBUG("sa->lifetime_c.bytes=%-9u %-9u\n",	/* XXX: %-18Lu */
			(__u32)((sa_idx->sa->lifetime_c.bytes) >> 32), (__u32)(sa_idx->sa->lifetime_c.bytes));
	if (sa_idx->sa->lifetime_c.bytes >= sa_idx->sa->lifetime_s.bytes && sa_idx->sa->lifetime_s.bytes) {
		sa_idx->sa->state = SADB_SASTATE_DYING;
		IPSEC4_DEBUG("change sa state DYING\n");
	}
	if (sa_idx->sa->lifetime_c.bytes >= sa_idx->sa->lifetime_h.bytes && sa_idx->sa->lifetime_h.bytes) {
		sa_idx->sa->state = SADB_SASTATE_DEAD;
		IPSEC4_DEBUG("change sa state DEAD\n");
	}

unlock_finish:
	write_unlock_bh(&sa_idx->sa->lock);  /* unlock SA */
	ipsec_sa_put(sa_idx->sa);
finish:
	return rtn;
}

static int ipsec4_input_check_esp(struct sk_buff **skb, struct ip_esp_hdr* esphdr, struct sa_index *sa_idx, u8 *nexthdr)
{
	int len = 0;
	int rtn = IPSEC_ACTION_DROP;
	u8 *authdata = NULL;
	u8 *srcdata = NULL;
	int srcsize = 0, totalsize = 0, hashsize = 0, encsize = 0;

	IPSEC4_DEBUG("start esp processing\n");
	if (!(*skb&&esphdr)) {
		printk(KERN_ERR "ipsec4_input_check_esp: parameters are invalid\n");
		goto finish;
	}	

	if (skb_is_nonlinear(*skb)) {
		u16 offset = ((char*)esphdr) - (char*)((*skb)->nh.raw);
		if (!skb_linearize(*skb, GFP_ATOMIC)) {
			esphdr = (struct ip_esp_hdr*)((*skb)->nh.raw + offset);
		} else {
			printk(KERN_ERR "ipsec4_input_check_esp: counld not linearize skb\n");
			rtn = -EINVAL;
			goto finish;
		} 
	}

	/* Check SPI */
	IPSEC4_DEBUG("esphdr->spi is 0x%x\n", ntohl(esphdr->spi));

	sa_index_init(sa_idx);
	((struct sockaddr_in *)&sa_idx->dst)->sin_addr.s_addr = (*skb)->nh.iph->daddr;
	((struct sockaddr_in *)&sa_idx->dst)->sin_family = AF_INET;
	sa_idx->prefixlen_d = 32;
	sa_idx->ipsec_proto = SADB_SATYPE_ESP;
	sa_idx->spi = esphdr->spi;

	sa_idx->sa = sadb_find_by_sa_index(sa_idx);

	if (!sa_idx->sa) {
		if (net_ratelimit()) {
			printk(KERN_ERR "IPSEC: (%d.%d.%d.%d ESP %d) Can't find SA\n",
			       NIPQUAD(((struct sockaddr_in *)&sa_idx->dst)->sin_addr), 
			       sa_idx->spi);
		}
		rtn = -EINVAL;
		goto finish;
	}

	write_lock_bh(&sa_idx->sa->lock);

	if ( sa_idx->sa->esp_algo.algo == SADB_EALG_NONE ) {
		if (net_ratelimit())
			printk(KERN_ERR "ipsec4_input_check_esp: not found encryption algorithm in SA!\n");
		rtn = -EINVAL;
		goto unlock_finish;
	}

	len = ntohs((*skb)->nh.iph->tot_len);

	if (len > (*skb)->len + ((char*)(*skb)->data - (char*)(*skb)->nh.iph)) {
		if (net_ratelimit())
			printk(KERN_ERR "ipsec4_input_check_esp: received packet length is wrong\n");
		rtn = -EINVAL;
		goto unlock_finish;
	}

	totalsize = len - (((char*)esphdr) - ((char*)(*skb)->nh.iph));

	if (!(sa_idx->sa->esp_algo.cx->ci)) {
		if (net_ratelimit())
			printk(KERN_ERR "ipsec4_input_check_esp: not found esp algo\n");
		rtn = -EINVAL;
		goto unlock_finish;
	}

	if ( !check_replay_window(&sa_idx->sa->replay_window, esphdr->seq_no) ) {
		if (net_ratelimit()) {
			printk(KERN_ERR "IPSEC: SA(%d.%d.%d.%d ESP %d): Replay check error\n",
			       NIPQUAD(((struct sockaddr_in *)&sa_idx->dst)->sin_addr),
			       sa_idx->spi);
		}
		kfree(srcdata);
		rtn = -EINVAL;
		goto unlock_finish;
	}

	encsize = totalsize - sa_idx->sa->esp_algo.cx->ci->ivsize - 8;
							/* 8 = SPI + Sequence Number */	

	if ( sa_idx->sa->auth_algo.algo != SADB_AALG_NONE ) {
		/* Calculate size */
		/* The tail of payload does not have to be aligned		*/
		/* with a multiple number of 64 bit.				*/
		/* 64 bit alignment is adapted to the position of top of header.*/
		hashsize = sa_idx->sa->auth_algo.digest_len;
		encsize -= hashsize;
		authdata=kmalloc(sa_idx->sa->auth_algo.dx->di->blocksize, GFP_ATOMIC);
		sa_idx->sa->auth_algo.dx->di->hmac_atomic(sa_idx->sa->auth_algo.dx,
			sa_idx->sa->auth_algo.key,
			sa_idx->sa->auth_algo.key_len,
			(char*)esphdr, totalsize - hashsize, authdata);	 
		/* Originally, IABG uses "for" loop for matching authentication data. */
		/* I change it into memcmp routine. */

		if (memcmp(authdata, &((char*)esphdr)[totalsize - hashsize],
				sa_idx->sa->auth_algo.digest_len )) {
			if (net_ratelimit()) {
				printk(KERN_ERR "IPSEC: SA(%d.%d.%d.%d ESP %d): Invalid checksum\n",
				       NIPQUAD(((struct sockaddr_in*)&sa_idx->dst)->sin_addr), sa_idx->spi);
			}
			kfree(authdata);
			rtn = IPSEC_ACTION_DROP;
			goto unlock_finish;
		}
		kfree(authdata);
		authdata = NULL;
	}

	/* Decrypt data */
	srcdata = kmalloc(encsize, GFP_ATOMIC);
	if (!srcdata) {
		if (net_ratelimit())
			printk(KERN_ERR "ipsec4_input_check_esp: can't allocate memory for decrypt\n");
		rtn = -ENOMEM;
		goto unlock_finish;
	}

	IPSEC4_DEBUG("len=%d, totalsize=%d, encsize=%d\n",
			len, totalsize, encsize);

	if (!(sa_idx->sa->esp_algo.iv)) { /* first packet */
		sa_idx->sa->esp_algo.iv = kmalloc(sa_idx->sa->esp_algo.cx->ci->ivsize, GFP_ATOMIC);
	}

	memcpy(sa_idx->sa->esp_algo.iv, esphdr->enc_data, sa_idx->sa->esp_algo.cx->ci->ivsize);
	sa_idx->sa->esp_algo.cx->ci->decrypt_atomic_iv(sa_idx->sa->esp_algo.cx,
			((u8 *)(esphdr->enc_data)) + sa_idx->sa->esp_algo.cx->ci->ivsize,
			srcdata, encsize, sa_idx->sa->esp_algo.iv);

	/* encsize - (pad_len + next_hdr) - pad_len */
	srcsize = encsize - 2 - srcdata[encsize-2];
	IPSEC4_DEBUG("Original data is srcsize=%d, padlength=%d\n", srcsize, srcdata[encsize-2]);
	if (srcsize <= 0) {
		if (net_ratelimit()) {
			printk(KERN_ERR "IPSEC: SA(%d.%d.%d.%d ESP %d):Decrypted packet contains garbage\n",
			       NIPQUAD(((struct sockaddr_in *)&sa_idx->dst)->sin_addr), sa_idx->spi);
		}
		kfree(srcdata);
		rtn = IPSEC_ACTION_DROP;
		goto unlock_finish;
	}

	update_replay_window(&sa_idx->sa->replay_window, esphdr->seq_no);

	memcpy(esphdr, srcdata, srcsize);

	(*skb)->nh.iph->protocol = *nexthdr = srcdata[encsize-1];
	(*skb)->nh.iph->tot_len = htons(ntohs((*skb)->nh.iph->tot_len) - 10 - srcdata[encsize]);			
							/* 10 = sizeof(struct ip_esp_hdr) - 8 + 2 */
	(*skb)->nh.iph->check = ip_fast_csum((unsigned char*)((*skb)->nh.iph), (*skb)->nh.iph->ihl);

	skb_trim(*skb, (*skb)->len +  srcsize - totalsize);
	(*skb)->nh.iph->tot_len = htons(((char *)esphdr - (char *)((*skb)->nh.iph)) + srcsize);

	kfree(srcdata);
	srcdata = NULL;

	rtn = IPSEC_ACTION_ESP;

	/* Otherwise checksum of fragmented udp packets fails (udp.c, csum_fold) */
	(*skb)->ip_summed = CHECKSUM_UNNECESSARY; 
	(*skb)->security |= RCV_CRYPT;

	if (!sa_idx->sa->fuse_time) {
		sa_idx->sa->fuse_time = jiffies;
		sa_idx->sa->lifetime_c.usetime = (sa_idx->sa->fuse_time) / HZ;
		ipsec_sa_mod_timer(sa_idx->sa);
		IPSEC4_DEBUG("set fuse_time = %lu\n", (sa_idx->sa->fuse_time));
	}
	sa_idx->sa->lifetime_c.bytes += totalsize;
	IPSEC4_DEBUG("sa->bytes=%-9u %-9u\n",			/* XXX: %-18Lu */
			(__u32)((sa_idx->sa->lifetime_c.bytes) >> 32), (__u32)(sa_idx->sa->lifetime_c.bytes));
	if (sa_idx->sa->lifetime_c.bytes >= sa_idx->sa->lifetime_s.bytes && sa_idx->sa->lifetime_s.bytes) {
		sa_idx->sa->state = SADB_SASTATE_DYING;
		IPSEC4_DEBUG("change sa state DYING\n");
	}
	if (sa_idx->sa->lifetime_c.bytes >= sa_idx->sa->lifetime_h.bytes && sa_idx->sa->lifetime_h.bytes) {
		sa_idx->sa->state = SADB_SASTATE_DEAD;
		IPSEC4_DEBUG("change sa state DEAD\n");
	}


unlock_finish:
	write_unlock_bh(&sa_idx->sa->lock); /* unlock SA */
	ipsec_sa_put(sa_idx->sa);
		
finish:
	return rtn;
}

int ipsec4_input_check(struct sk_buff **skb)
{
	int rtn = 0;
	int result = IPSEC_ACTION_BYPASS;
	u8 nexthdr = (*skb)->nh.iph->protocol;
	struct sa_index auth_sa_idx;
	struct sa_index esp_sa_idx;
	struct selector selector;
	struct ipsec_sp *policy = NULL;
	struct iphdr *hdr = (*skb)->nh.iph;

	IPSEC4_DEBUG("called\n");
#ifdef CONFIG_IPSEC_DEBUG
	IPSEC4_DEBUG("src addr: %d.%d.%d.%d\n", NIPQUAD(hdr->saddr));
	IPSEC4_DEBUG("dst addr: %d.%d.%d.%d\n", NIPQUAD(hdr->daddr));
	IPSEC4_DEBUG("hdr->tot_len is %d\n", ntohs(hdr->tot_len)); 
#endif /* CONFIG_IPSEC_DEBUG */
	
	if ( hdr->protocol == IPPROTO_UDP ) { /* IKE */
		if (pskb_may_pull(*skb, (*skb)->h.raw - (*skb)->data + sizeof(struct udphdr))) {
			if ((*skb)->h.uh->source == htons(500)
				 && (*skb)->h.uh->dest == htons(500)) {
					IPSEC4_DEBUG("received IKE packet. skip!\n");
					goto finish;
			}
		}
	}


	if (nexthdr == IPPROTO_AH) {
		result |= ipsec4_input_check_ah(skb, (struct ip_auth_hdr*)((*skb)->h.raw), &auth_sa_idx, &nexthdr);
	}

	if (nexthdr == IPPROTO_ESP) {
		result |= ipsec4_input_check_esp(skb, (struct ip_esp_hdr*)((*skb)->h.raw), &esp_sa_idx, &nexthdr);
	}

	if (result&IPSEC_ACTION_DROP) {
		IPSEC4_DEBUG("result is drop.\n");
		rtn = -EINVAL;
		goto finish;
	}

	/* copy selector XXX port */
	memset(&selector, 0, sizeof(struct selector));
	
#ifdef CONFIG_IPV4_IPSEC_TUNNEL
	if (nexthdr == IPPROTO_IPIP)
		selector.mode = IPSEC_MODE_TUNNEL;
#endif
	selector.proto = nexthdr;

	IPSEC4_DEBUG("nexthdr = %u\n", nexthdr);

	if (nexthdr == IPPROTO_TCP) {
		struct tcphdr *th = (*skb)->h.th;
		((struct sockaddr_in *)&selector.src)->sin_port = th->source;
		((struct sockaddr_in *)&selector.dst)->sin_port = th->dest;
	}

	if (nexthdr == IPPROTO_UDP) { 
		struct udphdr *uh = (*skb)->h.uh; 
		((struct sockaddr_in *)&selector.src)->sin_port = uh->source;
		((struct sockaddr_in *)&selector.dst)->sin_port = uh->dest;
	}

#ifdef CONFIG_IPV4_IPSEC_TUNNEL
	if (selector.mode == IPSEC_MODE_TUNNEL) {
		struct iphdr *h = NULL;
		if (pskb_may_pull(*skb, (*skb)->h.raw - (*skb)->data + sizeof(struct iphdr))) {
			h = (struct iphdr *)(*skb)->h.raw;

			((struct sockaddr_in *)&selector.src)->sin_family = AF_INET;
			((struct sockaddr_in *)&selector.src)->sin_addr.s_addr = h->saddr;
			((struct sockaddr_in *)&selector.dst)->sin_family = AF_INET;
			((struct sockaddr_in *)&selector.dst)->sin_addr.s_addr = h->daddr;
		} else {
			rtn = -EINVAL;
			goto finish;
		}
	} else {
#endif
		((struct sockaddr_in *)&selector.src)->sin_family = AF_INET;
		((struct sockaddr_in *)&selector.src)->sin_addr.s_addr = (*skb)->nh.iph->saddr;
		((struct sockaddr_in *)&selector.dst)->sin_family = AF_INET;
		((struct sockaddr_in *)&selector.dst)->sin_addr.s_addr = (*skb)->nh.iph->daddr;
#ifdef CONFIG_IPV4_IPSEC_TUNNEL
	}
#endif
	selector.prefixlen_d = 32;
	selector.prefixlen_s = 32;

	/* beggining of matching check selector and policy */
	IPSEC4_DEBUG("start match check SA and policy.\n");

#ifdef CONFIG_IPSEC_DEBUG
	IPSEC4_DEBUG("selector dst addr: %d.%d.%d.%d\n", 
		     NIPQUAD(((struct sockaddr_in *)&selector.dst)->sin_addr));
	IPSEC4_DEBUG("selector src addr: %d.%d.%d.%d\n", 
		     NIPQUAD(((struct sockaddr_in *)&selector.src)->sin_addr));
#endif /* CONFIG_IPSEC_DEBUG */
	policy = ipsec_sp_get(&selector);
		
	if (policy) {

		read_lock_bh(&policy->lock);

		/* non-ipsec packet processing: If this packet doesn't
		 * have any IPSEC headers, then consult policy to see
		 * what to do with packet. If policy says to apply IPSEC,
		 * and there is an SA, then pass packet to netxt layer,
		 * if ther isn't an SA, then drop the packet.
		 */
		if (policy->policy_action == IPSEC_POLICY_DROP) {
			rtn = -EINVAL;
			read_unlock_bh(&policy->lock);
			IPSEC4_DEBUG("a policy is drop, drop packet!\n"); 
			goto finish;
		}

		if (policy->policy_action == IPSEC_POLICY_BYPASS) {
			rtn = 0;
			read_unlock_bh(&policy->lock);
			IPSEC4_DEBUG("a policy is bypass, through!\n"); 
			goto finish;
		}

		if (policy->policy_action == IPSEC_POLICY_APPLY) {
			if (result&IPSEC_ACTION_AUTH) {
				if (policy->auth_sa_idx) {
					if (sa_index_compare(&auth_sa_idx, policy->auth_sa_idx)) {
						rtn = -EINVAL;
					}
				} else {
					rtn = -EINVAL;
				}
			} else {
				if (policy->auth_sa_idx) {
					rtn = -EINVAL;
				}
			}

			if (result&IPSEC_ACTION_ESP) {
				if (policy->esp_sa_idx) {
					if (sa_index_compare(&esp_sa_idx, policy->esp_sa_idx)) {
						rtn = -EINVAL;
					}
				} else {
					rtn = -EINVAL;
				}
			} else {
				if (policy->esp_sa_idx) {
					rtn = -EINVAL;
				}
			}
			IPSEC4_DEBUG("matching pair of SA and policy, through=%d\n", rtn); 
		}

		read_unlock_bh(&policy->lock);
		ipsec_sp_put(policy);
	} else {
		if (!result) {
			rtn = 0;
		} else {
			IPSEC4_DEBUG("matching pair of SA and policy not found, through!\n"); 
			rtn = -EINVAL;
			goto finish;		
		}
	}

	IPSEC4_DEBUG("end match check SA and policy.\n");
	/* end of matching check selector and policy */


finish:

	return rtn;
}

