/*
 * SPI support for Philips SWIFT SPI IP on PNX0105,
 *
 * Author: Andrei Ivolgin <aivolgin@ru.mvista.com>
 * Copyright (c) 2004 MontaVista Software Inc.
 * 
 * $Id: spi-pnx0105.h,v 1.1.2.1 2004/09/17 12:11:42 wool Exp $
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef _SPI_PNX0105_DRIVER
#define _SPI_PNX0105_DRIVER

#include <asm/arch/platform.h>

#define ADAPTER_NAME	"SWIFT SPI"
#define ALGORITHM_NAME	"PNX0105 SPI"

#define SPI_CLOCK		0x10
#define FIFO_CHUNK_SIZE		56

#define SPI_BUFFER_SIZE 0xFFFF 

/*  GPIO related definitions  */
#define PADMUX1_PINS		0x80
#define PADMUX1_MODE0		0x90
#define PADMUX1_MODE0SET	0x94
#define PADMUX1_MODE0RESET	0x98
#define PADMUX1_MODE1		0xA0
#define PADMUX1_MODE1SET	0xA4
#define PADMUX1_MODE1RESET	0xA8

#define GPIO_PIN_SPI_CE			(1<<15)
#define PADMUX1_BASE_ADDR		IO_ADDRESS(IOCONF_BASE)

#define gpio_write_reg(val,reg)		writel (val, PADMUX1_BASE_ADDR + reg)
#define gpio_read_reg(reg)		readl (PADMUX1_BASE_ADDR + reg)



/*  SPI related defines  */

typedef struct {		/*  Offset  */
	u32	global;		/*  0x00  */
	u32	con;		/*  0x04  */
	u32	frm;		/*  0x08  */
	u32	ier;		/*  0x0C  */
	u32	stat;		/*  0x10  */
	u32	dat;		/*  0x14  */
	u32	dat_msk;	/*  0x18  */
	u32	mask;		/*  0x1C  */
	u32	addr;		/*  0x20  */
} spi_regbank_t;


typedef struct {		/*  Offset  */
	u32	ctrl;		/*  0x00  */
	u32	count;		/*  0x04  */
	u32	status;		/*  0x08  */
} spi_timer_t;


#define SPI_BASE_ADDR		IO_ADDRESS(SPI_BASE)


#define SPI_TIMER_OFFSET	0x400
#define SPI_DAT_REG_OFFSET	0x14
#define SPI_TIMER_BASE_ADDR	(SPI_BASE_ADDR + SPI_TIMER_OFFSET)


#define spi_regs		(*((volatile spi_regbank_t *)SPI_BASE_ADDR))
#define spi_timer		(*((volatile spi_timer_t *)SPI_TIMER_BASE_ADDR))

#define SPI_INTERRUPT		IRQ_SPI


/*******************************************************************/
/* ----- Types and defines:  -----     */

#define SPI_RECEIVE                        0
#define SPI_TRANSMIT                       1

#define SPI_ENDIAN_SWAP_NO                 0
#define SPI_ENDIAN_SWAP_YES                1


/* Bit definitions for SPI_GLOBAL register */
#define SPI_GLOBAL_BLRES_SPI               0x00000002  /* R/W - Software reset, active high */
#define SPI_GLOBAL_SPI_ON                  0x00000001  /* R/W - SPI interface on */

/* Bit definitions for SPI_CON register */
#define SPI_CON_SPI_BIDIR                  0x00800000  /* R/W - SPI data in bidir. mux */
#define SPI_CON_SPI_BHALT                  0x00400000  /* R/W - Halt control */
#define SPI_CON_SPI_BPOL                   0x00200000  /* R/W - Busy signal active polarity */
#define SPI_CON_SPI_SHFT                   0x00100000  /* R/W - Data shifting enable in mask mode */
#define SPI_CON_SPI_MSB                    0x00080000  /* R/W - Transfer LSB or MSB first */
#define SPI_CON_SPI_EN                     0x00040000  /* R/W - SPULSE signal enable */
#define SPI_CON_SPI_MODE                   0x00030000  /* R/W - SCK polarity and phase modes */
#define SPI_CON_SPI_MODE3                  0x00030000  /* R/W - SCK polarity and phase mode 3 */
#define SPI_CON_SPI_MODE2                  0x00020000  /* R/W - SCK polarity and phase mode 2 */
#define SPI_CON_SPI_MODE1                  0x00010000  /* R/W - SCK polarity and phase mode 1 */
#define SPI_CON_SPI_MODE0                  0x00000000  /* R/W - SCK polarity and phase mode 0 */
#define SPI_CON_RXTX                       0x00008000  /* R/W - Transfer direction */
#define SPI_CON_THR                        0x00004000  /* R/W - Threshold selection */
#define SPI_CON_SHIFT_OFF                  0x00002000  /* R/W - Inhibits generation of clock pulses on sck_spi pin */
#define SPI_CON_BITNUM                     0x00001E00  /* R/W - No of bits to tx or rx in one block transfer */
#define SPI_CON_CS_EN                      0x00000100  /* R/W - Disable use of CS in slave mode */
#define SPI_CON_MS                         0x00000080  /* R/W - Selection of master or slave mode */
#define SPI_CON_RATE                       0x0000007F  /* R/W - Transmit/receive rate */

/* Bit definitions for SPI_FRM register */
#define SPI_FRM_SPIF                       0x0000FFFF  /* R/W - Number of frames to be transfered */

/* Bit definitions for SPI_IER register */
#define SPI_IER_SPI_INTCS                  0x00000004  /* R/W - SPI CS level change interrupt enable */
#define SPI_IER_SPI_INTEOT                 0x00000002  /* R/W - End of transfer interrupt enable */
#define SPI_IER_SPI_INTTHR                 0x00000001  /* R/W - FIFO threshold interrupt enable */

/* Bit definitions for SPI_STAT register */
#define SPI_STAT_SPI_INTCLR                0x00000100  /* R/WC - Clear interrupt */
#define SPI_STAT_SPI_EOT                   0x00000080  /* R/W - End of transfer*/ 
#define SPI_STAT_SPI_BSY_SL                0x00000040  /* R/W - Status of the input pin spi_busy */
#define SPI_STAT_SPI_CSL                   0x00000020  /* R/W - Indication of the edge on SPI_CS that caused an int. */
#define SPI_STAT_SPI_CSS                   0x00000010  /* R/W - Level of SPI_CS has changed in slave mode */
#define SPI_STAT_SPI_BUSY                  0x00000008  /* R/W - A data transfer is ongoing */
#define SPI_STAT_SPI_BF                    0x00000004  /* R/W - FIFO is full */
#define SPI_STAT_SPI_THR                   0x00000002  /* R/W - No of entries in Tx/Rx FIFO */
#define SPI_STAT_SPI_BE                    0x00000001  /* R/W - FIFO is empty */

/* Bit definitions for SPI_DAT register */
#define SPI_DAT_SPID                       0x0000FFFF  /* R/W - SPI data bits */

/* Bit definitions for SPI_DAT_MSK register */
#define SPI_DAT_MSK_SPIDM                  0x0000FFFF  /*  W  - SPI data bits to send using the masking mode */

/* Bit definitions for SPI_MASK register */
#define SPI_MASK_SPIMSK                    0x0000FFFF  /* R/W - Masking bits used for validating data bits */

/* Bit definitions for SPI_ADDR register */
#define SPI_ADDR_SPIAD                     0x0000FFFF  /* R/W - Address bits to add to the data bits */


/* Bit definitions for SPI_TIMER_CTRL_REG register */
#define SPI_TIMER_CTRL_REG_TIRQE           0x00000004  /* R/W - Timer interrupt enable */
#define SPI_TIMER_CTRL_REG_PIRQE           0x00000002  /* R/W - Peripheral interrupt enable */
#define SPI_TIMER_CTRL_REG_MODE            0x00000001  /* R/W - Mode */

/* Bit definitions for SPI_TIMER_COUNT_REG register */
#define SPI_TIMER_COUNT_REG                0x0000FFFF  /* R/W - Timed interrupt period */

/* Bit definitions for SPI_TIMER_STATUS_REG register */
#define SPI_TIMER_STATUS_REG_INT_STAT      0x00008000  /* R/C - Interrupt status */
#define SPI_TIMER_STATUS_REG_FIFO_DEPTH    0x0000007F  /*  R  - FIFO depth value (for debug purpose) */

#endif
