/*
 * include/linux/i2c-pnx0105.h
 *
 * Header file for i2c support for PNX0105.
 *
 * Author: Dennis Kovalev <dkovalev@ru.mvista.com>
 *
 * 2004 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

#ifndef _I2C_PNX0105_H
#define _I2C_PNX0105_H

typedef enum {
	mstatus_tdi	= 0x00000001,
	mstatus_afi	= 0x00000002,
	mstatus_nai	= 0x00000004,
	mstatus_drmi	= 0x00000008,
	mstatus_active	= 0x00000020,
	mstatus_scl	= 0x00000040,
	mstatus_sda	= 0x00000080,
	mstatus_rff	= 0x00000100,
	mstatus_rfe	= 0x00000200,
	mstatus_tff	= 0x00000400,
	mstatus_tfe	= 0x00000800,
} mstatus_bits_t;

typedef enum {
	mcntrl_tdie	= 0x00000001,
	mcntrl_afie	= 0x00000002,
	mcntrl_naie	= 0x00000004,
	mcntrl_drmie	= 0x00000008,
	/* drsie is not used here */
	mcntrl_rffie	= 0x00000020,
	mcntrl_daie	= 0x00000040,
	mcntrl_tffie	= 0x00000080,
	mcntrl_reset	= 0x00000100,
	mcntrl_cdbmode	= 0x00000400,   
} mcntrl_bits_t;

typedef enum {
	sstatus_afi	= 0x00000002,
	sstatus_nai	= 0x00000004,
	sstatus_drsi	= 0x00000010,
	sstatus_active	= 0x00000020,
	sstatus_scl	= 0x00000040,
	sstatus_sda	= 0x00000080,
	sstatus_rff	= 0x00000100,
	sstatus_rfe	= 0x00000200,
	sstatus_tffs	= 0x00001000,
	sstatus_tfes	= 0x00002000,
} sstatus_bits_t;

typedef enum {
	scntrl_afie	= 0x00000002,
	scntrl_naie	= 0x00000004,
	scntrl_drsie	= 0x00000010,
	scntrl_rffie	= 0x00000020,
	scntrl_daie	= 0x00000040,
	scntrl_reset	= 0x00000100,
	scntrl_tffsie	= 0x00000400,
} scntrl_bits_t;

typedef struct {   		/* All registers: */
	u32	fifo;		/* Offset 0x00 */
	u32	status;		/* Offset 0x04 */
	u32	cntrl;		/* Offset 0x08 */
	u32	clocklo;	/* Offset 0x0C */
	u32	clockhi;	/* Offset 0x10 */
	u32	address;	/* Offset 0x14 */
	u32	rx_level;	/* Offset 0x18 */
	u32	tx_level;	/* Offset 0x1C */
	u32	rx_byte;	/* Offset 0x20 */
	u32	tx_byte;	/* Offset 0x24 */
	u32	tx_slave_fifo;	/* OffSet 0x28 */
	u32	tx_slave_level;	/* Offset 0x2C */
} i2c_pnx0105_master_t;

typedef struct {		/* Only slave registers: */
	u32	rx_fifo;	/* Offset 0x00 */
	u32	status;		/* Offset 0x04 */
	u32	cntrl;		/* Offset 0x08 */
	u32	not_used1;	/* Offset 0x0C */
	u32	not_used2;	/* Offset 0x10 */
	u32	address;	/* Offset 0x14 */
	u32	rx_level;	/* Offset 0x18 */
	u32	not_used4;	/* Offset 0x1C */
	u32	not_used5;	/* Offset 0x20 */
	u32	not_used6;	/* Offset 0x24 */
	u32	tx_fifo;	/* OffSet 0x28 */
	u32	not_used7;	/* Offset 0x2C */   
} i2c_pnx0105_slave_t;

typedef enum {
	rw_bit		= 0x00000001,
	start_bit	= 0x00000100,
	stop_bit	= 0x00000200,
} i2c_pnx0105_prot_bits_t;

typedef enum {
	xmit,
	rcv,
	nodata,
} i2c_pnx0105_mode_t;

typedef enum {
	master,
	slave,
} i2c_pnx0105_adap_mode_t;

#define I2C_BUFFER_SIZE   0x20

typedef struct {
	int			ret;		/* Return value */
	i2c_pnx0105_mode_t	mode;		/* Interface mode */
	struct semaphore	sem;		/* Mutex for this interface */
	struct completion 	complete;	/* I/O completion */
	struct timer_list	timer;		/* Timeout */
	char *			buf;		/* Data buffer */
	int			len;		/* Length of data buffer */
} i2c_pnx0105_mif_t;

typedef struct {
	u32	base;
	int	irq;
	i2c_pnx0105_adap_mode_t	mode;
	int	slave_addr;
	void	(*slave_recv_cb)(void *);
	void	(*slave_send_cb)(void *);
	volatile i2c_pnx0105_master_t *	master;
	volatile i2c_pnx0105_slave_t *	slave;
	char	buffer [I2C_BUFFER_SIZE];	/* contains data received from I2C bus */
	int	buf_index;			/* index within I2C buffer (see above) */
	i2c_pnx0105_mif_t	mif;
} i2c_pnx0105_algo_data_t;

#include <asm/arch/platform.h>

#define SSA_IICMASTER0_BASE	I2C0_BASE
#define SSA_IICMASTER1_BASE	I2C1_BASE
#define INT_IIC_CONTROLLER0	IRQ_I2C0
#define INT_IIC_CONTROLLER1	IRQ_I2C1

#define HCLK_MHZ		132 		/* TODO: Make dynamic. */
#define TIMEOUT			(2*(HZ))
#define I2C_PNX0105_SPEED_KHZ	100

#define I2C_BLOCK_SIZE		0x100

#define CHIP_NAME		"PNX0105 I2C"
#define ADAPTER0_NAME		"PNX0105 I2C-0"
#define ADAPTER1_NAME		"PNX0105 I2C-1"

extern int i2c_slave_set_recv_cb (void(*i2c_slave_recv_cb)(void *), unsigned int adap);
extern int i2c_slave_set_send_cb (void(*i2c_slave_send_cb)(void *), unsigned int adap);
extern int i2c_slave_send_data(char *buf, int len, void *data);
extern int i2c_slave_recv_data(char *buf, int len, void *data);

#endif
