/*
 * ocp.h
 *
 *
 * 	Current Maintainer
 *      Armin Kuster akuster@pacbell.net
 *      Jan, 2002
 *
 *
 *
 * This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR   IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT,  INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *	Version 1.0 (02/01/26) - A. Kuster
 *	Initial version	 -
 *
 *	Version 1.1 (04/25/02)
 *		added active to ocp_dev for use when the device 
 *		needs to be unregistered temporarily and the data lives
 *		on in an structure such as net_device.
 *
 *	Version  1.2 (05/15/02) - Armin
 *	added DMA_TO* defines from pci.h
 *
 *	Version 1.3 (05/24/02) - Armin
 *	name change for API's to <class>_<action>_<object>
 *	added type_info strust
 *
 */

#ifdef __KERNEL__
#ifndef __OCP_H__
#define __OCP_H__

#include <linux/list.h>
#include <linux/config.h>
#include <linux/devfs_fs_kernel.h>
#include <asm/ocp_ids.h>

#include <asm/mmu.h>		/* For phys_addr_t */
#include <asm/rwsem.h>
#include <asm/semaphore.h>

#if defined(CONFIG_IBM_OCP)
#include <platforms/ibm_ocp.h>
#endif

#if defined(CONFIG_MPC_OCP)
#include <asm/mpc_ocp.h>
#endif

#define OCP_MAX_IRQS	7
#define MAX_EMACS	4
#define OCP_IRQ_NA	-1	/* used when ocp device does not have an irq */
#define OCP_IRQ_MUL	-2	/* used for ocp devices with multiply irqs */
#define OCP_NULL_TYPE	-1	/* used to mark end of list */
#define OCP_CPM_NA	0	/* No Clock or Power Management avaliable */
#define OCP_PADDR_NA    0       /* No MMIO registers */
#define EMACS_PER_ZMII	4

#define OCP_ANY_ID      (~0)
#define OCP_ANY_INDEX   -1

extern struct list_head ocp_list;
extern struct list_head         ocp_devices;
extern struct list_head         ocp_drivers;
extern struct rw_semaphore      ocp_devices_sem;
extern struct semaphore         ocp_drivers_sem;

struct ocp_device_id {
        unsigned int    vendor, function;       /* Vendor and function ID or OCP_ANY_ID */
        unsigned long   driver_data;            /* Data private to the driver */
};

enum ocp_type {
	PCI = 0,
	GPT,			/* General purpose timers */
	UART,
	OPB,
	IIC,
	GPIO,
	EMAC,
	ZMII,
	RGMIIB,
	IDE,
	USB,
	SCI,			/* Smart card */
	AUDIO,
	SSP,			/* sync serial port */
	SCP,			/* serial controller port */
	SCC,			/* serial contoller */
	VIDEO,
	DMA,
	UIC,
	RTC,
	LED
};

struct type_info {
	char name[16];
	char desc[50];
};

#ifdef CONFIG_85xx
struct ocp_def {
        unsigned int    vendor;
        unsigned int    function;
        int             index;
        phys_addr_t     paddr;
        int             irq;
        unsigned long   pm;
        void            *additions;
	enum ocp_type type; /* added for compile errors */
	unsigned long cpm;  /* ditto */
};
#else
struct ocp_def {
	enum ocp_type type;
	phys_addr_t paddr;
	int irq;
	unsigned long cpm;
};
#endif

/* Struct for single ocp device managment */
struct ocp_dev {
	struct list_head ocp_list;
	char name[16];
	u16 num;
	enum ocp_type type;	/* OCP device type */
	phys_addr_t paddr;
	void *vaddr;
	u32 flags;
	int irq;
	void *ocpdev;		/* ocp device struct  pointer */
#if defined(CONFIG_PM)
	u32 current_state;	/* Current operating state. In ACPI-speak,
				   this is D0-D3, D0 being fully functional,
				   and D3 being off. */

	int (*save_state) (u32 state);	/* Save Device Context */
	int (*suspend) (u32 state);	/* Device suspended */
	int (*resume) (u32 state);	/* Device woken up */
	int (*enable_wake) (u32 state, int enable);	/* Enable wake event */
#endif
#if defined(CONFIG_OCP_PROC)
	struct proc_dir_entry *procent;	/* device entry in /proc/bus/ocp */
#endif
};
#define ocp_dev_g(n) list_entry(n, struct ocp_dev, ocp_list)

/* Struct for a given device instance */
struct ocp_device {
	struct list_head	link;
	char			name[80];	/* device name */
	struct ocp_def		*def;		/* device definition */
	void			*drvdata;	/* driver data for this device */
	struct ocp_driver	*driver;
	u32			current_state;	/* Current operating state. In ACPI-speak,
						   this is D0-D3, D0 being fully functional,
						   and D3 being off. */
};

/* Structure for a device driver */
struct ocp_driver {
	const char		   	*name;
	const struct ocp_device_id	*id_table;	/* NULL if wants all devices */

	int	 (*probe)  (struct ocp_device *dev);	/* New device inserted */
	void	 (*remove) (struct ocp_device *dev);	/* Device removed (NULL if not a
							   hot-plug capable driver) */
	int 	 (*save_state) (struct ocp_device *dev, u32 state);  /* Save Device Context */
	int 	 (*suspend) (struct ocp_device *dev, u32 state);     /* Device suspended */
	int 	 (*resume) (struct ocp_device *dev);	             /* Device woken up */
	int 	 (*enable_wake) (struct ocp_device *dev, u32 state, int enable);
	                                                             /* Enable wake event */
	struct list_head		link;
};

#if defined (CONFIG_85xx)
/* Similar to the helpers above, these manipulate per-ocp_dev
 * driver-specific data.  Currently stored as ocp_dev::ocpdev,
 * a void pointer, but it is not present on older kernels.
 */
static inline void *
ocp_get_drvdata(struct ocp_device *pdev)
{
	return pdev->drvdata;
}

static inline void
ocp_set_drvdata(struct ocp_device *pdev, void *data)
{
	pdev->drvdata = data;
}

#else
/* Similar to the helpers above, these manipulate per-ocp_dev
 * driver-specific data.  Currently stored as ocp_dev::ocpdev,
 * a void pointer, but it is not present on older kernels.
 */
static inline void *
ocp_get_drvdata(struct ocp_dev *pdev)
{
	return pdev->ocpdev;
}

static inline void
ocp_set_drvdata(struct ocp_dev *pdev, void *data)
{
	pdev->ocpdev = data;
}
#endif

extern int ocp_register(struct ocp_dev *drv);
extern void ocp_unregister(struct ocp_dev *drv);
extern struct ocp_dev *ocp_alloc_dev(int size);
extern void ocp_free_dev(void *dev);
extern struct ocp_dev *ocp_get_dev(int type, int index);
extern int ocp_proc_attach_device(struct ocp_dev *dev);
extern int ocp_proc_detach_device(struct ocp_dev *dev);
extern unsigned long ocp_get_paddr(int type, int dev_num);
extern int ocp_get_max(int type);
extern int ocp_get_irq(int type, int dev_num);
extern int ocp_get_pm(int type, int dev_num);

#if defined (CONFIG_85xx)
#if defined (CONFIG_PM)
/*
 * This is right for the IBM 405 and 440 but will need to be
 * generalized if the OCP stuff gets used on other processors.
 */
static inline void
ocp_force_power_off(struct ocp_device *odev)
{
	mtdcr(DCRN_CPMFR, mfdcr(DCRN_CPMFR) | odev->def->pm);
}

static inline void
ocp_force_power_on(struct ocp_device *odev)
{
	mtdcr(DCRN_CPMFR, mfdcr(DCRN_CPMFR) & ~odev->def->pm);
}
#else
#define ocp_force_power_off(x)	(void)(x)
#define ocp_force_power_on(x)	(void)(x)
#endif
#endif

/* Register/Unregister an OCP driver */
extern int ocp_register_driver(struct ocp_driver *drv);
extern void ocp_unregister_driver(struct ocp_driver *drv);

/* Build list of devices */
extern int ocp_early_init(void);

/* Initialize the driver portion of OCP management layer */
extern int ocp_driver_init(void);

/* Find a device by index */
extern struct ocp_device *ocp_find_device(unsigned int vendor, unsigned int function, int index);

/* Get a def by index */
extern struct ocp_def *ocp_get_one_device(unsigned int vendor, unsigned int function, int index);

/* Add a device by index */
extern int ocp_add_one_device(struct ocp_def *def);

/* Remove a device by index */
extern int ocp_remove_one_device(unsigned int vendor, unsigned int function, int index);

#endif				/* __OCP_H__ */
#endif				/* __KERNEL__ */
