/*
  * include/asm-mips/vr7701.h
  *
  * NEC VR7701 definitions.
  *
  * Author: MontaVista Software, Inc. <source@mvista.com>
  *
  * Based on "Preliminary User's Manual: uPD30671 VR7701(tm) 64/32 bit
  * microprocessor
  * Document No. U16334EJ1V1UM00 (1st edition) 
  * Date Published October 2002 J CP(K)
  *
  * 2003 (c) MontaVista Software, Inc. This file is licensed under
  * the terms of the GNU General Public License version 2. This program
  * is licensed "as is" without any warranty of any kind, whether express
  * or implied.
  */
#ifndef _ASM_VR7701_H
#define _ASM_VR7701_H

#include "linux/serial_reg.h"

/*** VR7701 on-chip peripheral registers base address ***/
#define VR7701_BASE      0x18000000
#define VR7701_BASE_VIRT 0xB8000000
#define VR7701_IO_BASE   0xB8850000

/*** Register access primitives ***/
#ifndef _LANGUAGE_ASSEMBLY

#include <linux/types.h>

#define	vr7701_inb(x)          *(volatile u8*)(VR7701_BASE_VIRT + (x))
#define	vr7701_outb(x, y)      *(volatile u8*)(VR7701_BASE_VIRT + (x)) = (y)
#define	vr7701_inw(x)          *(volatile u16*)(VR7701_BASE_VIRT + (x))
#define	vr7701_outw(x, y)      *(volatile u16*)(VR7701_BASE_VIRT + (x)) = (y)
#define	vr7701_inl(x)          *(volatile u32*)(VR7701_BASE_VIRT + (x))
#define	vr7701_outl(x, y)      *(volatile u32*)(VR7701_BASE_VIRT + (x)) = (y)

#endif

/*** VR7701 Interrupt definitions ***/
/* 
 * On rockhopper board M1535+ bridge installed with integrated i8259-like
 * interrupt controller for PC-like peripheral. From driver compatibility
 * reasons, we map 8259 interrupt numbers starting from 0.
 */
#define I8259_IRQ_BASE      0
#define NUM_I8259_IRQ       16

#define VR7701_CPU_IRQ_BASE 16
#define VR7701_NUM_CPU_IRQ  8

#define VR7701_SYS_IRQ_BASE 24
#define VR7701_NUM_SYS_IRQ  16

#define VR7701_NMI_IRQ_BASE 40
#define VR7701_NUM_NMI_IRQ  10

/* CPU interrupts - according to CP0 Cause register */
#define VR7701_CPU_SW1      (VR7701_CPU_IRQ_BASE + 0) /* IP0 - software int */
#define VR7701_CPU_SW2      (VR7701_CPU_IRQ_BASE + 1) /* IP1 - software int */
#define VR7701_CPU_INT      (VR7701_CPU_IRQ_BASE + 2) /* IP2 - external 
                                                          ordinary interrupt */
#define VR7701_CPU_RES1     (VR7701_CPU_IRQ_BASE + 3) /* IP3 - reserved */
#define VR7701_CPU_RES2     (VR7701_CPU_IRQ_BASE + 4) /* IP4 - reserved */
#define VR7701_CPU_RES3     (VR7701_CPU_IRQ_BASE + 5) /* IP5 - reserved */
#define VR7701_CPU_RES4     (VR7701_CPU_IRQ_BASE + 6) /* IP6 - reserved */
#define VR7701_CPU_TIMER    (VR7701_CPU_IRQ_BASE + 7) /* IP7 - Timer int */

/* Interrupts cascaded to CPU_INT by ICU block */
#define VR7701_IRQ_NMI      (VR7701_SYS_IRQ_BASE + 0) /* NMI interrupt */
#define VR7701_IRQ_PCIX     (VR7701_SYS_IRQ_BASE + 1) /* PCI-X interrupt */
#define VR7701_IRQ_ETH      (VR7701_SYS_IRQ_BASE + 2) /* Ethernet 1/2 int */
#define VR7701_IRQ_LOCALBUS (VR7701_SYS_IRQ_BASE + 3) /* LocalBus */
#define VR7701_IRQ_UART     (VR7701_SYS_IRQ_BASE + 4) /* UART unit */
#define VR7701_IRQ_CSI      (VR7701_SYS_IRQ_BASE + 4) /* clocked serial 
                                                         interface unit 
                                                         (== UART) */
#define VR7701_IRQ_TIMER1   (VR7701_SYS_IRQ_BASE + 5) /* BIU timer, timer 0-3 */
#define VR7701_IRQ_TIMER2   (VR7701_SYS_IRQ_BASE + 6) /* Timer 4-7 */
#define VR7701_IRQ_PERF     (VR7701_SYS_IRQ_BASE + 7) /* Timer 8, performance 
                                                         monitor/event counter*/
#define VR7701_IRQ_INTP0    (VR7701_SYS_IRQ_BASE + 8) /* External int INTP0 */
#define VR7701_IRQ_INTP1    (VR7701_SYS_IRQ_BASE + 9) /* External int INTP1 */
#define VR7701_IRQ_INTP2    (VR7701_SYS_IRQ_BASE + 10)/* External int INTP2 */
#define VR7701_IRQ_INTP3    (VR7701_SYS_IRQ_BASE + 11)/* External int INTP3 */
#define VR7701_IRQ_INTP4    (VR7701_SYS_IRQ_BASE + 12)/* External int INTP4 */
#define VR7701_IRQ_INTP5    (VR7701_SYS_IRQ_BASE + 13)/* External int INTP5 */
#define VR7701_IRQ_INTP6    (VR7701_SYS_IRQ_BASE + 14)/* External int INTP6 */
#define VR7701_IRQ_INTP7    (VR7701_SYS_IRQ_BASE + 15)/* External int INTP7 */

/* Interrupts cascaded to IRQ_NMI by NMI block */
#define VR7701_NMI_EXT         (VR7701_NMI_IRQ_BASE + 0) /* External NMI sig */
#define VR7701_NMI_BIU_ERR     (VR7701_NMI_IRQ_BASE + 1) /* Bus Interface Unit
                                                            bus error */
#define VR7701_NMI_DRAM_CORR   (VR7701_NMI_IRQ_BASE + 2) /* DRAM ECC correctable
                                                            error */
#define VR7701_NMI_DRAM_ERR    (VR7701_NMI_IRQ_BASE + 3) /* DRAM ECC error */
#define VR7701_NMI_CACHE_PAR   (VR7701_NMI_IRQ_BASE + 4) /* Secondary cache 
                                                            parity error */
#define VR7701_NMI_CACHE_CORR  (VR7701_NMI_IRQ_BASE + 5) /* Secondary cache ECC
                                                            correctable error */
#define VR7701_NMI_CACHE_ERR   (VR7701_NMI_IRQ_BASE + 6) /* Secondary cache ECC
                                                            error */
#define VR7701_NMI_BCU_PAR     (VR7701_NMI_IRQ_BASE + 7) /* Local bus parity 
                                                            error between 
                                                            LocalBus and  
                                                            OnChipBus */
#define VR7701_NMI_ETH1_PAR    (VR7701_NMI_IRQ_BASE + 8) /* Ethernet #1 parity*/
#define VR7701_NMI_ETH2_PAR    (VR7701_NMI_IRQ_BASE + 9) /* Ethernet #2 parity*/

/*** INTC (interrupt controller) block definition ***/
#define VR7701_INTC_BASE  0x800B00

/* INTC Interrupt channels */
#define VR7701_INTCH_NMI      1  /* Interrupt from NMI block */
#define VR7701_INTCH_PCIX     2  /* PCI-X interrupt */
#define VR7701_INTCH_ETH      3  /* Ethernet 1/2 interrupt */
#define VR7701_INTCH_LOCALBUS 4  /* LocalBus */
#define VR7701_INTCH_UART     5  /* UART unit */
#define VR7701_INTCH_CSI      5  /* clocked serial interface unit (== UART) */
#define VR7701_INTCH_TIMER1   6  /* BIU timer, timer 0-3 */
#define VR7701_INTCH_TIMER2   7  /* Timer 4-7 */
#define VR7701_INTCH_PERF     8  /* Timer 8, performance monitor/event counter*/
#define VR7701_INTCH_INTP0    9  /* External interrupt INTP0 */
#define VR7701_INTCH_INTP1    10 /* External interrupt INTP1 */
#define VR7701_INTCH_INTP2    11 /* External interrupt INTP2 */
#define VR7701_INTCH_INTP3    12 /* External interrupt INTP3 */
#define VR7701_INTCH_INTP4    13 /* External interrupt INTP4 */
#define VR7701_INTCH_INTP5    14 /* External interrupt INTP5 */
#define VR7701_INTCH_INTP6    15 /* External interrupt INTP6 */
#define VR7701_INTCH_INTP7    16 /* External interrupt INTP7 */

/* NMI interrupt channels */
#define VR7701_NMICH_EXT         1  /* External NMI signal */
#define VR7701_NMICH_BIU_ERR     2  /* Bus Interface Unit bus error */
#define VR7701_NMICH_DRAM_CORR   3  /* DRAM ECC correctable error */
#define VR7701_NMICH_DRAM_ERR    4  /* DRAM ECC error */
#define VR7701_NMICH_CACHE_PAR   5  /* Secondary cache parity error */
#define VR7701_NMICH_CACHE_CORR  6  /* Secondary cache ECC correctable error */
#define VR7701_NMICH_CACHE_ERR   7  /* Secondary cache ECC error */
#define VR7701_NMICH_BCU_PAR     8  /* Local bus parity error between LocalBus
                                       and OnChipBus */
#define VR7701_NMICH_ETH1_PAR    9  /* Ethernet #1 parity */
#define VR7701_NMICH_ETH2_PAR    10 /* Ethernet #2 parity */

#define VR7701_INTC_CHN(num) (1 << ((num) - 1))

/* Interrupt level registers */
#define VR7701_INTC_CH_LEVEL(n) (VR7701_INTC_BASE + (((n) - 1) * 2))
#define VR7701_INTC_LEVEL_MIN    0
#define VR7701_INTC_LEVEL_MAX    15

/* Clear interrupt request (in edge detection mode) */
#define VR7701_INTC_INT_CLR         (VR7701_INTC_BASE + 0x20)

/* Set interrupt request (in edge detection mode) */
#define VR7701_INTC_INT_SET         (VR7701_INTC_BASE + 0x22)

/* Input status of interrupt requests */
#define VR7701_INTC_INT_PENDING     (VR7701_INTC_BASE + 0x24)

/* Level/Edge register - type of interrupt input detection */
#define VR7701_INTC_LVLEDGE         (VR7701_INTC_BASE + 0x26)

/* Interrupt input mask register */
#define VR7701_INTC_INT_MASK        (VR7701_INTC_BASE + 0x28)

/* Interrupt level mask register */
#define VR7701_INTC_LVL_MASK        (VR7701_INTC_BASE + 0x2A)

/* Interrupt acknowledge register */
#define VR7701_INTC_INT_ACK         (VR7701_INTC_BASE + 0x2C)

/* Interrupt vector register */
#define VR7701_INTC_INT_VECTOR      (VR7701_INTC_BASE + 0x2E)
#define VR7701_INTC_INT_VECTOR_SHIFT 3

/* Interrupt status register (status of interrupt inputs before masking) */
#define VR7701_INTC_INT_STATUS      (VR7701_INTC_BASE + 0x30)

/* Active level for external interrupt inputs */
#define VR7701_INTC_ACT_LV          (VR7701_INTC_BASE + 0x32)
#define VR7701_INTC_ACT_LV_INTP0_LOW    0x00
#define VR7701_INTC_ACT_LV_INTP0_HIGH   0x01
#define VR7701_INTC_ACT_LV_INTP1_LOW    0x00
#define VR7701_INTC_ACT_LV_INTP1_HIGH   0x02
#define VR7701_INTC_ACT_LV_INTP2_LOW    0x00
#define VR7701_INTC_ACT_LV_INTP2_HIGH   0x04
#define VR7701_INTC_ACT_LV_INTP3_LOW    0x00
#define VR7701_INTC_ACT_LV_INTP3_HIGH   0x08
#define VR7701_INTC_ACT_LV_INTP4_LOW    0x00
#define VR7701_INTC_ACT_LV_INTP4_HIGH   0x10
#define VR7701_INTC_ACT_LV_INTP5_LOW    0x00
#define VR7701_INTC_ACT_LV_INTP5_HIGH   0x20
#define VR7701_INTC_ACT_LV_INTP6_LOW    0x00
#define VR7701_INTC_ACT_LV_INTP6_HIGH   0x40
#define VR7701_INTC_ACT_LV_INTP7_LOW    0x00
#define VR7701_INTC_ACT_LV_INTP7_HIGH   0x80


/* NMI interrupt request clear register */
#define VR7701_INTC_NMI_CLR         (VR7701_INTC_BASE + 0x40)

/* NMI interrupt request set register */
#define VR7701_INTC_NMI_SET         (VR7701_INTC_BASE + 0x42)

/* Input status of the NMI request register */
#define VR7701_INTC_NMI_PENDING     (VR7701_INTC_BASE + 0x44)

/* NMI mask register */
#define VR7701_INTC_NMI_MASK        (VR7701_INTC_BASE + 0x46)

/* NMI output destination (NMI of CPU core or ICU channel 1) */
#define VR7701_INTC_NMI_OUT_SEL     (VR7701_INTC_BASE + 0x48)

/* Input status of NMI interupt requests */
#define VR7701_INTC_NMI_STATUS      (VR7701_INTC_BASE + 0x4A)

/*** GPIO interface definitions ***/
/* Pin function selection */
#define VR7701_GPIO_SEL   0x800900
#define VR7701_GPIO_SEL_U1         0xC000  /* UART channel 1 pin mode */
#define VR7701_GPIO_SEL_U1_UART    0x0000  /*   UART mode */
#define VR7701_GPIO_SEL_U1_GP      0x4000  /*   GP mode */
#define VR7701_GPIO_SEL_U1_GPUART  0x8000  /*   GP/UART mode */
#define VR7701_GPIO_SEL_U1_CSIUART 0xC000  /*   CSI/UART mode */
#define VR7701_GPIO_SEL_U2         0x3000  /* UART channel 2 pin mode */
#define VR7701_GPIO_SEL_U2_UART    0x0000  /*   UART mode */
#define VR7701_GPIO_SEL_U2_GP      0x1000  /*   GP mode */
#define VR7701_GPIO_SEL_U2_GPUART  0x2000  /*   GP/UART mode */
#define VR7701_GPIO_SEL_MI1        0x0C00  /* MII channel 1 pin mode */
#define VR7701_GPIO_SEL_MI1_MII    0x0000  /*   MII mode */
#define VR7701_GPIO_SEL_MI1_GP     0x0400  /*   GP mode */
#define VR7701_GPIO_SEL_MI2        0x0300  /* MII channel 2 pin mode */
#define VR7701_GPIO_SEL_MI2_MII    0x0000  /*   MII mode */
#define VR7701_GPIO_SEL_MI2_GP     0x0100  /*   GP mode */
#define VR7701_GPIO_SEL_INTP7      0x0080  /* INTP7 pin mode: */
#define VR7701_GPIO_SEL_INTP7_INT  0x0000  /*   interrupt mode */
#define VR7701_GPIO_SEL_INTP7_GP   0x0080  /*   GP mode */
#define VR7701_GPIO_SEL_INTP6      0x0040  /* INTP6 pin mode: */
#define VR7701_GPIO_SEL_INTP6_INT  0x0000  /*   interrupt mode */
#define VR7701_GPIO_SEL_INTP6_GP   0x0040  /*   GP mode */
#define VR7701_GPIO_SEL_INTP5      0x0020  /* INTP5 pin mode: */
#define VR7701_GPIO_SEL_INTP5_INT  0x0000  /*   interrupt mode */
#define VR7701_GPIO_SEL_INTP5_GP   0x0020  /*   GP mode */
#define VR7701_GPIO_SEL_INTP4      0x0010  /* INTP4 pin mode: */
#define VR7701_GPIO_SEL_INTP4_INT  0x0000  /*   interrupt mode */
#define VR7701_GPIO_SEL_INTP4_GP   0x0010  /*   GP mode */
#define VR7701_GPIO_SEL_INTP3      0x0008  /* INTP3 pin mode: */
#define VR7701_GPIO_SEL_INTP3_INT  0x0000  /*   interrupt mode */
#define VR7701_GPIO_SEL_INTP3_GP   0x0008  /*   GP mode */
#define VR7701_GPIO_SEL_INTP2      0x0004  /* INTP2 pin mode: */
#define VR7701_GPIO_SEL_INTP2_INT  0x0000  /*   interrupt mode */
#define VR7701_GPIO_SEL_INTP2_GP   0x0004  /*   GP mode */
#define VR7701_GPIO_SEL_INTP1      0x0002  /* INTP1 pin mode: */
#define VR7701_GPIO_SEL_INTP1_INT  0x0000  /*   interrupt mode */
#define VR7701_GPIO_SEL_INTP1_GP   0x0002  /*   GP mode */
#define VR7701_GPIO_SEL_INTP0      0x0001  /* INTP0 pin mode: */
#define VR7701_GPIO_SEL_INTP0_INT  0x0000  /*   interrupt mode */
#define VR7701_GPIO_SEL_INTP0_GP   0x0001  /*   GP mode */

#define VR7701_GPIO_IOSEL1  0x800910 /* GP15-GP0 pin direction */
#define VR7701_GPIO_IOSEL2  0x800912 /* GP31-GP16 pin direction */
#define VR7701_GPIO_IOSEL3  0x800914 /* GP47-GP32 pin direction */
#define VR7701_GPIO_IOSEL4  0x800916 /* GP57-GP48 pin direction */
#define VR7701_GPIO_DATA1   0x800920 /* GP15-GP0 data register */
#define VR7701_GPIO_DATA2   0x800922 /* GP31-GP16 data register */
#define VR7701_GPIO_DATA3   0x800924 /* GP47-GP32 data register */
#define VR7701_GPIO_DATA4   0x800926 /* GP57-GP48 data register */

/*** UART block definitions ***/
#define VR7701_UART1_BASE 0x800C00
#define VR7701_UART2_BASE 0x800C40
#define VR7701_UART_STEP 2
#define VR7701_UART_REG_SHIFT 1

#define VR7701_UART_BAUD_BASE 1152000

#define VR7701_UART1_RBR (VR7701_UART1_BASE + (UART_RX * VR7701_UART_STEP))
#define VR7701_UART1_THR (VR7701_UART1_BASE + (UART_TX * VR7701_UART_STEP))
#define VR7701_UART1_DLL (VR7701_UART1_BASE + (UART_DLL * VR7701_UART_STEP))
#define VR7701_UART1_IER (VR7701_UART1_BASE + (UART_IER * VR7701_UART_STEP))
#define VR7701_UART1_DLM (VR7701_UART1_BASE + (UART_DLM * VR7701_UART_STEP))
#define VR7701_UART1_IIR (VR7701_UART1_BASE + (UART_IIR * VR7701_UART_STEP))
#define VR7701_UART1_FCR (VR7701_UART1_BASE + (UART_FCR * VR7701_UART_STEP))
#define VR7701_UART1_LCR (VR7701_UART1_BASE + (UART_LCR * VR7701_UART_STEP))
#define VR7701_UART1_MCR (VR7701_UART1_BASE + (UART_MCR * VR7701_UART_STEP))
#define VR7701_UART1_LSR (VR7701_UART1_BASE + (UART_LSR * VR7701_UART_STEP))
#define VR7701_UART1_MSR (VR7701_UART1_BASE + (UART_MSR * VR7701_UART_STEP))
#define VR7701_UART1_SCR (VR7701_UART1_BASE + (UART_SCR * VR7701_UART_STEP))
#define VR7701_UART1_RESET  (VR7701_UART1_BASE + 0x10)

#define VR7701_UART2_RBR (VR7701_UART2_BASE + (UART_RX * VR7701_UART_STEP))
#define VR7701_UART2_THR (VR7701_UART2_BASE + (UART_TX * VR7701_UART_STEP))
#define VR7701_UART2_DLL (VR7701_UART2_BASE + (UART_DLL * VR7701_UART_STEP))
#define VR7701_UART2_IER (VR7701_UART2_BASE + (UART_IER * VR7701_UART_STEP))
#define VR7701_UART2_DLM (VR7701_UART2_BASE + (UART_DLM * VR7701_UART_STEP))
#define VR7701_UART2_IIR (VR7701_UART2_BASE + (UART_IIR * VR7701_UART_STEP))
#define VR7701_UART2_FCR (VR7701_UART2_BASE + (UART_FCR * VR7701_UART_STEP))
#define VR7701_UART2_LCR (VR7701_UART2_BASE + (UART_LCR * VR7701_UART_STEP))
#define VR7701_UART2_MCR (VR7701_UART2_BASE + (UART_MCR * VR7701_UART_STEP))
#define VR7701_UART2_LSR (VR7701_UART2_BASE + (UART_LSR * VR7701_UART_STEP))
#define VR7701_UART2_MSR (VR7701_UART2_BASE + (UART_MSR * VR7701_UART_STEP))
#define VR7701_UART2_SCR (VR7701_UART2_BASE + (UART_SCR * VR7701_UART_STEP))
#define VR7701_UART2_RESET  (VR7701_UART2_BASE + 0x10)

/*** Clocked Serial Interface Unit ***/
/* Operation mode register */
#define VR7701_CSI_MODE  0x800C80 

#define VR7701_CSI_MODE_CSIE     0x80    /* CSI Unit Enable */
#define VR7701_CSI_MODE_TRMD     0x40    /* Set transmission/reception mode */
#define VR7701_CSI_MODE_CCL8     0x00    /* Data length is 8 bit */
#define VR7701_CSI_MODE_CCL16    0x20    /* Data lenght is 16 bit */
#define VR7701_CSI_MODE_DIR_MSB  0x00    /* Comm direction MSB first */
#define VR7701_CSI_MODE_DIR_LSB  0x10    /* Comm direction LSB first */
#define VR7701_CSI_MODE_AUTO     0x04    /* Consecuentive transfer mode */
#define VR7701_CSI_MODE_CSOT     0x01    /* Communication status (1-progress)*/

/* Clock selection register */
#define VR7701_CSI_CLKSEL  0x800C81
#define VR7701_CSI_CLKSEL_CKP    0x10    /* Clock phase */
#define VR7701_CSI_CLKSEL_DAP    0x08    /* Data phase */
#define VR7701_CSI_CLKSEL_CKS    0x07    /* Clock selection: */
#define VR7701_CSI_CLKSEL_DIV1   0x01    /* 8.33333 MHz */
#define VR7701_CSI_CLKSEL_DIV2   0x02    /* 4.16667 MHz */
#define VR7701_CSI_CLLSEL_DIV4   0x03    /* 2.08333 MHz */
#define VR7701_CSI_CLKSEL_DIV8   0x04    /* 1.04167 MHz */
#define VR7701_CSI_CLKSEL_DIV16  0x05    /* 0.52083 MHz */
#define VR7701_CSI_CLKSEL_DIV32  0x06    /* 0.26042 MHz */

#define VR7701_CSI_SIRB    0x800c82  /* Receive next register */
#define VR7701_CSI_SOTB    0x800c84  /* Transmit next register */
#define VR7701_CSI_SIRBE   0x800c86  /* Receive data register */
#define VR7701_CSI_SOTBF   0x800c88  /* Transmit data register */
#define VR7701_CSI_SIO     0x800c8a  /* Parallel-to-serial converter status*/

/* CSI Control register */
#define VR7701_CSI_CNT     0x800C90
#define VR7701_CSI_CNT_CSIRST    0x8000 /* Reset CSI Unit */
#define VR7701_CSI_CNT_TTRG_EN   0x1000 /* Transmit stop trigger enable */
#define VR7701_CSI_CNT_TFIFOF    0x0800 /* Transmit FIFO full */
#define VR7701_CSI_CNT_TFIFOE    0x0400 /* Tranmist FIFO enable */
#define VR7701_CSI_CNT_SIO_V     0x0080 /* Valid data in CSI_SIO register */
#define VR7701_CSI_CNT_SIRB_V    0x0040 /* Valid data in CSI_SIRB register*/
#define VR7701_CSI_CNT_RFULLS    0x0020 /* Enable automatic stop of Rx */
#define VR7701_CSI_CNT_RTRGEN    0x0010 /* Receive stop trigger enable */
#define VR7701_CSI_CNT_RFIFOF    0x0008 /* Receive FIFO full */
#define VR7701_CSI_CNT_RFIFOE    0x0004 /* Receive FIFO enable */

/* CSI Interrupt Register */
#define VR7701_CSI_INT     0x800C92
#define VR7701_CSI_INT_END       0x8000 /* Transmit/receive end interrupt */
#define VR7701_CSI_INT_EMP       0x0100 /* Transmit FIFO empty interrupt */
#define VR7701_CSI_INT_OVER      0x0001 /* Receive FIFO overflow interrupt*/

#define VR7701_CSI_IFIFOV  0x800C94  /* Receive FIFO validity */
#define VR7701_CSI_OFIFOV  0x800C96  /* Transmit FIFO validity */
#define VR7701_CSI_IFIFO   0x800C98  /* Read data from the receive FIFO */
#define VR7701_CSI_OFIFO   0x800C9A  /* Write data to transmit FIFO */
#define VR7701_CSI_FIFOTRG 0x800C9B  /* number of times data to be Tx/Rx'd */

/*** Ethernet Controller ***/
#define VR7701_ETHERNET1_BASE 0x802000
#define VR7701_ETHERNET2_BASE 0x802400

/* Ethernet controller registers are defined in nec_candy.c driver */

/*** PCI-X Interface Unit definitions ***/
/* Layout of some registers corresponds to the PCI configuration space;
   appropriate definitions can be taken from pci.h */
#define VR7701_PCI_VENDOR_ID  0x840000 /* PCI Vendor ID, NEC = 0x1033 */
#define VR7701_PCI_DEVICE_ID  0x840002 /* PCI Device ID, VR7701 = 0x010C */
#define VR7701_PCI_COMMAND    0x840004 /* PCI Command register */
#define VR7701_PCI_STATUS     0x840006 /* Status register */
#define VR7701_PCI_REVID      0x840008 /* Revision ID */
#define VR7701_PCI_PROGIF     0x840009 /* Programming interface */
#define VR7701_PCI_SUBCLASS   0x84000a /* PCI sub-class */
#define VR7701_PCI_CLASS      0x84000b /* PCI base class */
#define VR7701_PCI_CLSIZE     0x84000c /* cache line size */
#define VR7701_PCI_LAT        0x84000d /* latency timer */
#define VR7701_PCI_HDRTYPE    0x84000e /* Header type */
#define VR7701_PCI_BIST       0x84000f /* Built-in self test register */
#define VR7701_PCI_BAR0       0x840010 /* PCI target base address 0, LSW */
#define VR7701_PCI_BAR0_MSW   0x840014 /* PCI target base address 0, MSW */
#define VR7701_PCI_BAR1       0x840018 /* PCI target base address 1, LSW */
#define VR7701_PCI_BAR1_MSW   0x84001c /* PCI target base address 1, MSW */
#define VR7701_PCI_SUBSYS_VID 0x84002c /* Subsystem vendor ID */
#define VR7701_PCI_SUBSYS_ID  0x84002e /* Subsystem ID */
#define VR7701_PCI_CAPP       0x840034 /* Capabilities Pointer */
#define VR7701_PCI_INTLINE    0x84003c /* Interrupt line */
#define VR7701_PCI_INTPIN     0x84003d /* Interrupt pin */
#define VR7701_PCI_MIN_GNT    0x84003e /* Minimum grant time */
#define VR7701_PCI_MAX_LAT    0x84003f /* Maximum latency */

/* Configuration space access */
#define VR7701_PCI_CFGADDR    0x840040 /* Configuration cycle address */
#define VR7701_PCI_CFGDATA    0x840044 /* Configuration cycle data */
#define VR7701_PCI_CFGATTR    0x840048 /* Configuration attribute */
#define VR7701_PCI_CFGATTR_READ  0x06000000 /* Config. space read */
#define VR7701_PCI_CFGATTR_WRITE 0x06000000 /* Config. space write */

/* PCI Interrupt control/status */
#define VR7701_PCI_INT_CTL    0x840050 /* Interrupt control register */
#define VR7701_PCI_INT_CTL_INTA_ASSERT    1 /* Assert INTA# */
#define VR7701_PCI_INT_CTL_INTA_DEASSERT  0 /* De-assert INTA# */

#define VR7701_PCI_INT_STATUS 0x840054 /* Status of interrupt request signals*/
#define VR7701_PCI_INT_STATUS_INTA 0x01   /* INTA# status */
#define VR7701_PCI_INT_STATUS_INTB 0x02   /* INTB# status */
#define VR7701_PCI_INT_STATUS_INTC 0x04   /* INTC# status */
#define VR7701_PCI_INT_STATUS_INTD 0x08   /* INTD# status */

#define VR7701_PCI_INT_MASK   0x840058 /* PCI Interrupt mask */
#define VR7701_PCI_INT_MASK_INTA   0x01   /* INTA# mask */
#define VR7701_PCI_INT_MASK_INTB   0x02   /* INTB# mask */
#define VR7701_PCI_INT_MASK_INTC   0x04   /* INTC# mask */
#define VR7701_PCI_INT_MASK_INTD   0x08   /* INTD# mask */

/* PCI error control */
#define VR7701_PCI_SERR_STATUS 0x84005C /* SERR Status register */
#define VR7701_PCI_SERR_STATUS_SS  0x01    /* PCI SERR# signal asserted */
#define VR7701_PCI_SERR_STATUS_SSM 0x02    /* Mask of SERR# signal report */

#define VR7701_PCI_ERR_ADDR     0x840060 /* PCI error address, LSW */
#define VR7701_PCI_ERR_ADDR_MSW 0x840064 /* PCI error address, MSW */
#define VR7701_PCI_ERR_STATUS   0x840068 /* PCI error status register */
#define VR7701_PCI_ERR_MASK     0x84006C /* PCI error mask register */
#define VR7701_PCI_ERR_STATUS_RCO  0x20000 /* Retry count out */
#define VR7701_PCI_ERR_STATUS_SCCO 0x10000 /* Split completion count out */
#define VR7701_PCI_ERR_STATUS_USC  0x00400 /* Unexpected split completion */
#define VR7701_PCI_ERR_STATUS_SCD  0x00200 /* Split completion discard */
#define VR7701_PCI_ERR_STATUS_RSCE 0x00100 /* Split completion error recvd*/
#define VR7701_PCI_ERR_STATUS_DCD  0x00080 /* Delayed completion discard */
#define VR7701_PCI_ERR_STATUS_DAP  0x00040 /* Addr parity error detection */
#define VR7701_PCI_ERR_STATUS_DDP  0x00020 /* Data parity error detection */
#define VR7701_PCI_ERR_STATUS_STA  0x00010 /* Target abort report */
#define VR7701_PCI_ERR_STATUS_DMDP 0x00004 /* Master data parity error */
#define VR7701_PCI_ERR_STATUS_RTA  0x00002 /* Target abort reception */
#define VR7701_PCI_ERR_STATUS_RMA  0x00001 /* Master abort reception */

#define VR7701_PCI_OCBE_STATUS  0x840078 /* On-chip bus error status */
#define VR7701_PCI_OCBE_MASK    0x84007C /* On-chip bus error mask */
#define VR7701_PCI_OCBE_STATUS_OPE   0x10000 /* OCB parity error */
#define VR7701_PCI_OCBE_STATUS_BL64E 0x01000 /* Burst len 64-byte error */
#define VR7701_PCI_OCBE_STATUS_BL32E 0x00800 /* Burst len 32-byte error */
#define VR7701_PCI_OCBE_STATUS_BL16E 0x00400 /* Burst len 16-byte error */
#define VR7701_PCI_OCBE_STATUS_BL8E  0x00200 /* Burst len 8-byte error */
#define VR7701_PCI_OCBE_STATUS_BL4E  0x00100 /* Burst len 4-byte error */
#define VR7701_PCI_OCBE_STATUS_RCOE  0x00010 /* Retry count out error */
#define VR7701_PCI_OCBE_STATUS_DNDE  0x00008 /* Device non-detection err */
#define VR7701_PCI_OCBE_STATUS_BEE   0x00004 /* Byte enable error */

/* Message-signalled interrupts capability */
#define VR7701_PCI_MSI_CTL      0x840082 /* Message control register */
#define VR7701_PCI_MSI_CTL_64AC     0x0080 /* 64-bit address messages enabled*/ 
#define VR7701_PCI_MSI_CTL_MME      0x0070 /* Number of messages assigned */
#define VR7701_PCI_MSI_CTL_MME1     0x0000 /*   1 message */
#define VR7701_PCI_MSI_CTL_MME2     0x0010 /*   2 messages or less */
#define VR7701_PCI_MSI_CTL_MME4     0x0020 /*   4 messages or less */
#define VR7701_PCI_MSI_CTL_MME8     0x0030 /*   8 messages or less */
#define VR7701_PCI_MSI_CTL_MME16    0x0040 /*   16 messages or less */
#define VR7701_PCI_MSI_CTL_MME32    0x0050 /*   32 messages or less */
#define VR7701_PCI_MSI_CTL_MMC      0x000e /* Number of messages requested
                                              by a function */
#define VR7701_PCI_MSI_CTL_MMC1     0x0000 /*   1 message */
#define VR7701_PCI_MSI_CTL_MMC2     0x0002 /*   2 messages or less */
#define VR7701_PCI_MSI_CTL_MMC4     0x0004 /*   4 messages or less */
#define VR7701_PCI_MSI_CTL_MMC8     0x0006 /*   8 messages or less */
#define VR7701_PCI_MSI_CTL_MMC16    0x0008 /*   16 messages or less */
#define VR7701_PCI_MSI_CTL_MMC32    0x000a /*   32 messages or less */
#define VR7701_PCI_MSI_CTL_MSIE     0x0001 /* MSI interface enable */

#define VR7701_PCI_MSI_ADDR     0x84084 /* Message address register */
#define VR7701_PCI_MSI_ADDR_UP  0x84088 /* Message upper address */
#define VR7701_PCI_MSI_DATA     0x8408C /* Message data register */

/* PCI-X capability */
#define VR7701_PCIX_CMD            0x840092 /* PCI-X command register */
#define VR7701_PCIX_STATUS         0x840094 /* PCI-X status register */
#define VR7701_PCIX_SPLCOMPERR_MSG 0x840098 /* Split completion error msg */

/* Target memory mapping */
#define VR7701_PCI_TGT1_SEG     0x8400a0 /* Target memory 1 segment */
#define VR7701_PCI_TGT2_SEG     0x8400b0 /* Target memory 2 segment */
#define VR7701_PCI_TGT1_MASK    0x8400a4 /* Target memory 1 mask */
#define VR7701_PCI_TGT2_MASK    0x8400b4 /* Target memory 2 mask */
#define VR7701_PCI_TGT_MASK_EN  0x00000001 /* target window enable */
#define VR7701_PCI_TGT_MASK_MIN 0x10000  /* minimal size of the window */
#define VR7701_PCI_TGT1_SWAP    0x8400a8 /* Target memory 1 swap */
#define VR7701_PCI_TGT2_SWAP    0x8400b8 /* Target memory 1 swap */
#define VR7701_PCI_TGT_SWAP_BYTE  0x00000000 /* Byte swap */
#define VR7701_PCI_TGT_SWAP_HALF  0x00000001 /* Halfword (2 bytes) swap */
#define VR7701_PCI_TGT_SWAP_WORD  0x00000002 /* Word (4 bytes) swap */
#define VR7701_PCI_TGT_SWAP_NO    0x00000003 /* No swap */

/* PCI Master segment */
#define VR7701_PCI_MST_SEG      0x8400c0 /* PCI master segment LSW reg */
#define VR7701_PCI_MST_SEG_MSW  0x8400c4
#define VR7701_PCI_MST_SEG_SWAP   0x00000003 /* Endian swap mode */
#define VR7701_PCI_MST_SWAP_BYTE  0x00000000 /* Byte swap */
#define VR7701_PCI_MST_SWAP_HALF  0x00000001 /* Halfword (2 bytes) swap */
#define VR7701_PCI_MST_SWAP_WORD  0x00000002 /* Word (4 bytes) swap */
#define VR7701_PCI_MST_SWAP_NO    0x00000003 /* No swap */
#define VR7701_PCI_MST_PREFETCH   0x00000004 /* Prefetching allowed */

/* PCI-X unit control */
#define VR7701_PCI_RESET        0x8400f0 /* PCI Reset register */
#define VR7701_PCI_RESET_ASSERTED    0x00000000 /* Reset asserted */
#define VR7701_PCI_RESET_DEASSERTED  0x00000001 /* Reset de-asserted */

#define VR7701_PCI_UCTL         0x8400f4 /* Unit control register */
#define VR7701_PCI_UCTL_INITEND   0x80000000 /* PCI unit initialisation
                                                completed */
#define VR7701_PCI_UCTL_PER       0x00001000 /* Generates parity error */
#define VR7701_PCI_UCTL_MATTR_CCH 0x00000100 /* Type of access to cache
                                                attribute signale of
                                                OnChipBus */
#define VR7701_PCI_UCTL_PBA       0x00000020 /* Functions as arbiter */
#define VR7701_PCI_UCTL_HOST      0x00000010 /* Host device on PCI bus */
#define VR7701_PCI_UCTL_CLK       0x00000006 /* Frequency of PCI clock */
#define VR7701_PCI_UCTL_CLK_33    0x00000000 /*   33 MHz */
#define VR7701_PCI_UCTL_CLK_66    0x00000002 /*   66 MHz */
#define VR7701_PCI_UCTL_CLK_100   0x00000004 /*   100 MHz */
#define VR7701_PCI_UCTL_CLK_132   0x00000006 /*   132 MHz */
#define VR7701_PCI_UCTL_PCIMODE   0x00000001 /* PCI mode of operation */

#define VR7701_PCI_CNTOUT       0x8400f8 /* Count out timer register */
#define VR7701_PCI_CNTOUT_EN      0x10000000 /* Enable count-out function */
#define VR7701_PCI_SLAVE_FLUSH  0x8400fc /* Slave data flush register */
#define VR7701_PCI_SLAVE_FLUSH_DATA 0x00000001 /* Data not flushed */

/* PCI-X OnChipBus mapping */
#define VR7701_PCI_OCB_MAP      0x100000000ULL
#define VR7701_PCI_OCB_SIZE     0x800000000ULL

/*** Timer Unit ***/
/* Prescaler registers */
#define VR7701_TIMER_PRE_A      0x800A00 /* Prescaler CH-A register */
#define VR7701_TIMER_PRE_B      0x800A02 /* Prescaler CH-B register */
#define VR7701_TIMER_PRE_C      0x800A04 /* Prescaler CH-C register */
#define VR7701_TIMER_PRE_D      0x800A06 /* Prescaler CH-D register */
#define VR7701_TIMER_PRE_E      0x800A08 /* Prescaler CH-E register */
#define VR7701_TIMER_PRE_F      0x800A0A /* Prescaler CH-F register */
#define VR7701_TIMER_PRE_G      0x800A0C /* Prescaler CH-G register */
#define VR7701_TIMER_PRE_RATE_SHIFT 2
#define VR7701_TIMER_PRE_EN     0x2
#define VR7701_TIMER_PRE_INTERN 0x1

/* Clock selectors */
#define VR7701_TIMER_CLK_SEL_A  0x800A10 /* Counters 0-3 clock select */
#define VR7701_TIMER_CLK_SEL_B  0x800A12 /* Counters 4-7 clock select */
#define VR7701_TIMER_CLK_SEL_C  0x800A14 /* Counter 8 clock select */

#define VR7701_TIMER_TC_STS(_n) (0x800A20 + VR7701_TIMER_TC_STEP * (_n))
                                           /* Mode registers */
#define VR7701_TIMER_TC_EN(_n)  (0x800A22 + VR7701_TIMER_TC_STEP * (_n))
                                           /* Enable registers */
#define VR7701_TIMER_TC_DATA0(_n) (0x800A28 + VR7701_TIMER_TC_STEP * (_n))
                                           /* Data registers (0:15) */
#define VR7701_TIMER_TC_DATA1(_n) (0x800A2A + VR7701_TIMER_TC_STEP * (_n))
                                           /* Data registers (16:31) */
#define VR7701_TIMER_TC_DATA2(_n) (0x800A2C + VR7701_TIMER_TC_STEP * (_n))
                                           /* Data registers (32:47) */
#define VR7701_TIMER_TC_DATA3(_n) (0x800A2E + VR7701_TIMER_TC_STEP * (_n))
                                           /* Data registers (48:63) */

/* Secondary Cache */
#define VR7701_SC_ALLOC       0x000048 /* Allocate control register */
#define VR7701_SC_NOALLOC_OCB    0x10 /* OnChipBus */
#define VR7701_SC_NOALLOC_3      0x08 /* 0xC000000-0xFFFFFFF */
#define VR7701_SC_NOALLOC_2      0x04 /* 0x8000000-0xBFFFFFF */
#define VR7701_SC__OALLOC_1      0x02 /* 0x4000000-0x7FFFFFF */
#define VR7701_SC_NOALLOC_0      0x01 /* 0x0000000-0x3FFFFFF */

#define VR7701_SC_STORE       0x000050 /* Store the tag */
#define VR7701_SC_LOAD        0x000058 /* Load the tag */
#define VR7701_SC_WBINV       0x000060 /* Write back invalidate register */
#define VR7701_SC_HIT_WBINV   0x000068 /* Hit write back invalidate reg */
#define VR7701_SC_TAG         0x000100 /* Tag register */
#define VR7701_SC_TAG_SC         0xffff0000 /* SC tag */
#define VR7701_SC_TAG_NAB        0xffffffc0 /* no-allocate buffer tag */
#define VR7701_SC_TAG_DIRTY      0x00000004 /* Dirty bit */
#define VR7701_SC_TAG_VALID      0x00000002 /* Valid bit */
#define VR7701_SC_TAG_FREEZE     0x00000001 /* Freeze bit */

#define VR7701_SC_ECC         0x000108 /* Error Correction Code bits */
#define VR7701_SC_LRU         0x000110 /* LRU bits for cache line */
#define VR7701_SC_WC          0x000118 /* Write Control register */
#define VR7701_SC_WC_ECC         0x00000001 /* Generate ECC code on write */

#define VR7701_SC_WAY0    0  /* Way 0 */
#define VR7701_SC_WAY1    1  /* Way 1 */
#define VR7701_SC_WAY2    2  /* Way 2 */
#define VR7701_SC_WAY3    3  /* Way 3 */
#define VR7701_SC_NAB     4  /* No-allocate buffer */

/* Bus Interface Unit (BIU) */
#define VR7701_BIU_CONFIG_1   0x800000 /* BIU Config 1 register */
#define VR7701_BIU_CONFIG_1_SPD  0x00000002 /* System parity check disable */
#define VR7701_BIU_CONFIG_1_PAM  0x00000001 /* Physical address mode */

#define VR7701_BIU_CONFIG_2   0x800000 /* BIU Config 2 register */
#define VR7701_BIU_CONFIG_2_BE   0x00000001 /* Big endian mode */
#define VR7701_BIU_CONFIG_2_DIV  0x0000000E /* DEVCPU Division mode */
#define VR7701_BIU_CONFIG_2_DIV_2_0 0x0 /* Div 2.0 */
#define VR7701_BIU_CONFIG_2_DIV_2_5 0x2 /* Div 2.5 */
#define VR7701_BIU_CONFIG_2_DIV_3_0 0x4 /* Div 3.0 */
#define VR7701_BIU_CONFIG_2_DIV_3_5 0x6 /* Div 3.5 */
#define VR7701_BIU_CONFIG_2_DIV_4_0 0x8 /* Div 4.0 */
#define VR7701_BIU_CONFIG_2_DIV_4_5 0xA /* Div 4.5 */
#define VR7701_BIU_CONFIG_2_DIV_5_0 0xC /* Div 5.0 */
#define VR7701_BIU_CONFIG_2_DIV_5_5 0xE /* Div 5.5 */

#define VR7701_BIU_REVID      0x800040 /* Revision ID */
#define VR7701_BIU_REVID_MAJ     0x000000F0 /* Major revision id */
#define VR7701_BIU_REVID_MIN     0x0000000F /* Minor revision id */

#define VR7701_BIU_ERR_1      0x800080 /* OnChipBus error info 1 - err addr */
#define VR7701_BIU_ERR_2      0x800088 /* OnChipBus error info 2 */
#define VR7701_BIU_ERR_2_ADDRHI  0x0000000F /* Address 35:32 */
#define VR7701_BIU_ERR_2_WR      0x00000010 /* Write access */
#define VR7701_BIU_ERR_2_MEM     0x00000020 /* Memory access */
#define VR7701_BIU_ERR_2_LEN     0x000007C0 /* Transfer data length: */
#define VR7701_BIU_ERR_2_LEN_4   0x00000000 /*    4-byte transfer */
#define VR7701_BIU_ERR_2_LEN_8   0x00000040 /*    8-byte transfer */
#define VR7701_BIU_ERR_2_LEN_16  0x00000080 /*    16-byte transfer */
#define VR7701_BIU_ERR_2_LEN_32  0x000000C0 /*    32-byte transfer */
#define VR7701_BIU_ERR_2_LEN_64  0x00000100 /*    64-byte transfer */
#define VR7701_BIU_ERR_2_LEN_128 0x00000140 /*    128-byte transfer */
#define VR7701_BIU_ERR_2_INFO    0x0000F000 /* Error info */
#define VR7701_BIU_ERR_2_INFO_CMD         0x0000   /* Command error */
#define VR7701_BIU_ERR_2_UNALIGNED_BURST  0x1000   /* Unaligned burst access*/
#define VR7701_BIU_ERR_2_CONN_BE          0x2000   /* Connection byte enable*/
#define VR7701_BIU_ERR_2_NODEV            0x3000   /* Device not detected */
#define VR7701_BIU_ERR_2_RETRY            0x4000   /* Retry counter out */
#define VR7701_BIU_ERR_2_UNALIGNED        0x6000   /* Unaligned access not
                                                      supported */
#define VR7701_BIU_ERR_2_OVERSIZE_4       0x8000   /* Oversize err (4 bytes)*/
#define VR7701_BIU_ERR_2_OVERSIZE_8       0x9000   /* Oversize err (8 bytes)*/
#define VR7701_BIU_ERR_2_OVERSIZE_16      0xa000   /* Oversize err (16 bytes)*/
#define VR7701_BIU_ERR_2_OVERSIZE_32      0xb000   /* Oversize err (32 bytes)*/
#define VR7701_BIU_ERR_2_OVERSIZE_64      0xc000   /* Oversize err (64 bytes)*/
#define VR7701_BIU_ERR_2_OVERSIZE_128     0xd000   /* Oversize err (128bytes)*/

#endif /* _ASM_VR7701_H */
