/*
 *
 * BRIEF MODULE DESCRIPTION
 *	Definitions for IDT RC32300 CPU Core.
 *
 * Author: Steve Longerbeam <stevel@mvista.com, or source@mvista.com>
 *
 * 2000,2001 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

#ifndef _RC32300_H_
#define _RC32300_H_

#include <linux/delay.h>
#include <asm/io.h>

#ifdef CONFIG_IDT_79S334
#include <asm/rc32300/79s334.h>
#elif defined(CONFIG_IDT_79EB355)
#include <asm/rc32300/79eb355.h>
#elif defined(CONFIG_IDT_79RP355)
#include <asm/rc32300/79rp355.h>
#endif

extern int idtprintf(const char *fmt, ...);

/* cpu pipeline flush */
static inline void rc32300_sync(void)
{
        __asm__ volatile ("sync");
}

static inline void rc32300_sync_udelay(int us)
{
        __asm__ volatile ("sync");
        udelay(us);
}

static inline void rc32300_sync_delay(int ms)
{
        __asm__ volatile ("sync");
        mdelay(ms);
}

/*
 * Macros to access internal RC32300 registers. No byte
 * swapping should be done when accessing the internal
 * registers.
 */
static inline u8 rc32300_readb(unsigned long pa)
{
	return *((volatile u8 *)KSEG1ADDR(pa));
}
static inline u16 rc32300_readw(unsigned long pa)
{
	return *((volatile u16 *)KSEG1ADDR(pa));
}
static inline u32 rc32300_readl(unsigned long pa)
{
	return *((volatile u32 *)KSEG1ADDR(pa));
}
static inline void rc32300_writeb(u8 val, unsigned long pa)
{
	*((volatile u8 *)KSEG1ADDR(pa)) = val;
}
static inline void rc32300_writew(u16 val, unsigned long pa)
{
	*((volatile u16 *)KSEG1ADDR(pa)) = val;
}
static inline void rc32300_writel(u32 val, unsigned long pa)
{
	*((volatile u32 *)KSEG1ADDR(pa)) = val;
}

/*
 * C access to CLZ and CLO instructions
 * (count leading zeroes/ones).
 */
static inline int rc32300_clz(unsigned long val)
{
	int ret;
        __asm__ volatile (
		".set\tnoreorder\n\t"
		".set\tnoat\n\t"
		".set\tmips32\n\t"
		"clz\t%0,%1\n\t"
                ".set\tmips0\n\t"
                ".set\tat\n\t"
                ".set\treorder"
                : "=r" (ret)
		: "r" (val));

	return ret;
}
static inline int rc32300_clo(unsigned long val)
{
	int ret;
        __asm__ volatile (
		".set\tnoreorder\n\t"
		".set\tnoat\n\t"
		".set\tmips32\n\t"
		"clo\t%0,%1\n\t"
                ".set\tmips0\n\t"
                ".set\tat\n\t"
                ".set\treorder"
                : "=r" (ret)
		: "r" (val));

	return ret;
}

#endif  /* _RC32300_H_ */
