/*
 * include/asm-mips/intrinsity/fpga.h
 *
 * Include file for Intrinsity FPGA.
 *
 * Author: MontaVista Software, Inc. <source@mvista.com>
 *
 * 2004 (c) MontaVista, Software, Inc.  This file is licensed under the terms
 * of the GNU General Public License version 2.  This program is licensed
 * "as is" without any warranty of any kind, whether express or implied.
 */
#ifndef	_INTRINSITY_FPGA_H
#define	_INTRINSITY_FPGA_H

#include <asm/intrinsity/memmap.h>

/* FPGA registers */

/* Clock frequencies */
#define	FM_FPGA_XTAL_FREQ	33000000	/* determines uart & counter speed */

/* Register offsets from FM_FGPA_BASE */
#define	FM_FPGA_STATUS		0x00	/* overall FPGA status */
#define	FM_FPGA_CONFIG		0x04	/* configuration and interrupt acknowledge */
#define	FM_FPGA_TEST_REG0	0x08	/* Test register 0 */
#define	FM_FPGA_TEST_REG1	0x0C	/* Test register 1 */
#define	FM_FPGA_COUNT		0x10	/* periodic counter value (32-bits) */
#define	FM_FPGA_ONESHOT		0x14	/* one-shot counter value (16-bits) */
#define	FM_FPGA_CRC_IN		0x18	/* checksum accumulator in  (32-bits) */
#define	FM_FPGA_CRC_OUT		0x1C	/* checksum accumulator out (32-bits) */
#define	FM_FPGA_LED		0x20	/* LED display (8-bits, word accessed) */

/* Bits in FM_FPGA_STATUS */
#define	FM_FPGA_COUNT_INT	0x00001	/* Periodic counter interrupt is active */
#define	FM_FPGA_NMI_INT		0x00002	/* NMI interrupt is active */
#define	FM_FPGA_CRC_BUSY	0x00004	/* CRC generator is busy */
#define	FM_FPGA_FLASH_READY	0x00008	/* Flash is ready (not erasing or programming) */
#define	FM_FPGA_FLASH_ENABLED	0x00010	/* Flash is enabled */
#define	FM_FPGA_I2C_INT		0x00020	/* I2C interrupt is active */
#define	FM_FPGA_UART_INT	0x00040	/* UART interrupt is active */
#define	FM_FPGA_LM86_ALERT_INT	0x00080	/* LM86 Alert interrupt is active */
#define	FM_FPGA_LM86_CRIT_INT	0x00100	/* LM86 Critical interrupt is active */
#define	FM_FPGA_WDT_INT		0x00200	/* Watchdog Timeout interrupt is active */
#define	FM_FPGA_PCI_A_INT	0x00400	/* PCI-A interrupt is active */

/* Bits in FM_FPGA_CONFIG */
#define	FM_FPGA_COUNT_RESET	0x00001	/* Clear FM_FPGA_COUNT_INT */
#define	FM_FPGA_NMI_RESET	0x00002	/* Clear FM_FPGA_NMI_INT */
#define	FM_FPGA_NMISET		0x00004	/* Set FM_FPGA_NMI_INT and cause NMIL */
#define	FM_FPGA_FLASHACC	0x00008	/* Set to allow FLASH programming acceleration */
#define	FM_FPGA_COUNT_START	0x00010	/* Start/stop periodic counter */
#define	FM_FPGA_ONESHOT_ENB	0x00020	/* GWETL one-shot counter is enabled */
#define	FM_FPGA_FLASH_OVERRIDE	0x00040	/* Enable FLASH (override PromICE) */
#define	FM_FPGA_CRC_ENABLE	0x00080	/* Enable CRC */
#define	FM_FPGA_FORCERESET	0x00100	/* Force a board reset */
#define	FM_FPGA_UART_ALT	0x00200	/* Select alternate UART I/O pins */
#define	FM_FPGA_LM86_ALERT_ENB	0x00400	/* Enable LM86 Alert to cause interrupt */
#define	FM_FPGA_LM86_CRIT_ENB	0x00800	/* Enable LM86 Critical to cause NMI */
#define	FM_FPGA_WDT_ENB		0x01000	/* Enable Wachdog Timeout */
#define	FM_FPGA_WDT_RESET	0x02000	/* Clear FM_FPGA_WDT_INT, start/reset counter */
#define	FM_FPGA_WDT_5SEC	0x00000	/* Set Watchdog Timeout period to 5 seconds */
#define	FM_FPGA_WDT_1SEC	0x04000	/* Set Watchdog Timeout period to 1 seconds */
#define	FM_FPGA_WDT_2SEC	0x08000	/* Set Watchdog Timeout period to 2 seconds */
#define	FM_FPGA_WDT_3SEC	0x0C000	/* Set Watchdog Timeout period to 3 seconds */
#define	FM_FPGA_PCI_A_ENB	0x10000	/* Enable PCI-A to cause interrupt */
#define	FM_FPGA_UART_ENB	0x20000	/* Enable UART to cause interrupt */
#define	FM_FPGA_I2C_ENB		0x40000	/* Enable I2C to cause interrupt */
#define	FM_FPGA_COUNT_ENB	0x80000	/* Enable COUNT to cause interrupt */

/* Register offsets from FM_I2C_BASE */
#define	FM_I2C_PRELO		0	/* Clock Prescale low-byte */
#define	FM_I2C_PREHI		1	/* Clock Prescale high-byte */
#define	FM_I2C_CTRL		2	/* Control */
#define	FM_I2C_TX		3	/* Transmit (write) */
#define	FM_I2C_RX		3	/* Receive (read) */
#define	FM_I2C_CMD		4	/* Command (write) */
#define	FM_I2C_STATUS		4	/* Status (read) */

#define	FM_FPGA_I2C_FREQ	400000	/* I2C bus frequency */
#define	FM_I2C_PRESCALE		((FM_FPGA_XTAL_FREQ/(5*FM_FPGA_I2C_FREQ))-1)	/* prescale value */

/* Bits in FM_I2C_CTRL */
#define	FM_I2C_CTRL_EN		0x80	/* Enable */
#define	FM_I2C_CTRL_IEN		0x40	/* Interrupt enable */

/* Bits in FM_I2C_CMD (write only???) */
#define	FM_I2C_CMD_STA		0x80	/* Generate (repeated) start */
#define	FM_I2C_CMD_STO		0x40	/* Generate stop */
#define	FM_I2C_CMD_RD		0x20	/* Read from slave */
#define	FM_I2C_CMD_WR		0x10	/* Write to slave */
#define	FM_I2C_CMD_ACK		0x08	/* when receiver, sender sent ACK(0) or NACK(1) */
#define	FM_I2C_CMD_IACK		0x01	/* set to clear interrupt condition */

/* Bits in FM_I2C_STATUS (read only) */
#define	FM_I2C_STATUS_RXACK	0x80	/* received acknowledge from slave (1=NACK, 0=ACK) */
#define	FM_I2C_STATUS_BUSY	0x40	/* I2C busy (1=start, 0=stop) */
#define	FM_I2C_STATUS_TIP	0x02	/* transfer in progess */
#define	FM_I2C_STATUS_IF	0x01	/* interrupt flag, set when interrupt is pending */

#define	FM_LM86_SLAVE_ADDR	0x4C	/* unique I2C address for this device */

/* LM86 registers */
#define	FM_LM86_LT		0x00	/* Local temperature (read only) */
#define	FM_LM86_RTHB		0x01	/* Remote temp High byte (read only) */
#define	FM_LM86_SR		0x02	/* Status (read only) */
#define	FM_LM86_C_R		0x03	/* Configuration (read) */
#define	FM_LM86_CR_R		0x04	/* Conversion rate (read) */
#define	FM_LM86_LHS_R		0x05	/* Local High Setpoint (read) */
#define	FM_LM86_LLS_R		0x06	/* Local Low Setpoint (read) */
#define	FM_LM86_RHSHB_R		0x07	/* Remote High Setpoint, high byte (read) */
#define	FM_LM86_RLSHB_R		0x08	/* Remote Low Setpoint, high byte (read) */
#define	FM_LM86_C_W		0x09	/* Configuration (write) */
#define	FM_LM86_CR_W		0x0A	/* Conversion rate (write) */
#define	FM_LM86_LHS_W		0x0B	/* Local High Setpoint (write) */
#define	FM_LM86_LLS_W		0x0C	/* Local Low Setpoint (write) */
#define	FM_LM86_RHSHB_W		0x0D	/* Remote High Setpoint, high byte (write) */
#define	FM_LM86_RLSHB_W		0x0E	/* Remote Low Setpoint, high byte (write) */
#define	FM_LM86_ONESHOT		0x0F	/* One shot conversion (write only) */
#define	FM_LM86_RTLB		0x10	/* Remote temperature (read only) */
#define	FM_LM86_RTOHB		0x11	/* Remote Temperature Offset, high byte (r/w) */
#define	FM_LM86_RTOLB		0x12	/* Remote Temperature Offset, low byte (r/w) */
#define	FM_LM86_RHSLB		0x13	/* Remote High Setpoint, low byte (r/w) */
#define	FM_LM86_RLSLB		0x14	/* Remote Low Setpoint, low byte (r/w) */
#define	FM_LM86_RCS		0x19	/* Remote T_CRIT Setpoint (r/w) */
#define	FM_LM86_LCS		0x20	/* Local T_CRIT Setpoint (r/w) */
#define	FM_LM86_TH		0x21	/* T_CRIT Hysteresis (r/w) */
#define	FM_LM86_RDTF		0xBF	/* Remote Diode Temperature Filter (r/w) */
#define	FM_LM86_RMID		0xFE	/* Manufacturer's ID (read only) */
#define	FM_LM86_RDR		0xFF	/* Stepping/Die Revision Code (read only) */

/* Bits in FM_LM86_C */
#define	FM_LM86_C_ALERT_MASK	0x80	/* mask ALERT's */
#define	FM_LM86_C_SHUTDOWN	0x40	/* disable when set */
#define	FM_LM86_C_RTCRIT_MASK	0x10	/* set to mask T_CRIT on remote temp exceeding limit */
#define	FM_LM86_C_LTCRIT_MASK	0x04	/* set to mask T_CRIT on local temp exceeding limit */
#define	FM_LM86_C_FAULT_QUEUE	0x01	/* 3 consecutive HIGH/LOW/T_CRIT measurements generate 'outside limit' */
#define	FM_LM86_C_DISABLE	0xD4	/* shutdown and disable everything */

/* Register offsets from FM_UART_BASE */
#define FM_UART_RX		0x00	/* In:  Receive buffer (DLAB=0) */
#define FM_UART_TX		0x00	/* Out: Transmit buffer (DLAB=0) */
#define FM_UART_DLL		0x00	/* Out: Divisor Latch Low (DLAB=1) */
#define FM_UART_DLM		0x01	/* Out: Divisor Latch High (DLAB=1) */
#define FM_UART_IER		0x01	/* Out: Interrupt Enable Register */
#define FM_UART_IIR		0x02	/* In:  Interrupt ID Register */
#define FM_UART_FCR		0x02	/* Out: FIFO Control Register */
#define FM_UART_LCR		0x03	/* Out: Line Control Register */
#define FM_UART_MCR		0x04	/* Out: Modem Control Register */
#define FM_UART_LSR		0x05	/* In:  Line Status Register */
#define FM_UART_MSR		0x06	/* In:  Modem Status Register */
#define FM_UART_SCR		0x07	/* I/O: Scratch Register */

/* Bits in FM_UART_LCR */
#define FM_UART_LCR_DLAB	0x80	/* Divisor latch access bit */
#define FM_UART_LCR_WLEN8	0x03	/* Wordlength: 8 bits */

/* Bits in FM_UART_LSR */
#define FM_UART_LSR_TEMT	0x40	/* Transmitter empty */
#define FM_UART_LSR_THRE	0x20	/* Transmit-hold-register empty */
#define FM_UART_LSR_DR		0x01	/* Receiver data ready */

#ifdef	__ASSEMBLY__
	.macro	fm_led	val
	la	k0,FM_FPGA_BASE
	li	k1,\val
	sw	k1,FM_FPGA_LED(k0)
	.endm

	.macro	fm_putc ch
	la	k0,FM_UART_BASE
1:	lb	k1,FM_UART_LSR(k0)
	andi	k1,k1,FM_UART_LSR_THRE
	beqz	k1,1b
	nop
	li	k1,\ch
	sb	k1,FM_UART_TX(k0)
	sync
	.endm

	# wait for a character to arrive, put it in regno
	.macro	fm_getc regno
	la	k0,FM_UART_BASE
1:	lb	k1,FM_UART_LSR(k0)
	andi	k1,k1,FM_UART_LSR_DR
	beqz	k1,1b
	nop
	lb	\regno,FM_UART_TX(k0)
	sync
	.endm

	# issue 'cmd', wait for it to finish (k0=i2cbase, k1,t0=scratch)
	.macro	fm_i2c_cmd	cmd
	li	k1,\cmd
	sb	k1,FM_I2C_CMD(k0)
	sync
	li	t0,10000
1:
	sub	t0,t0,1
	beqz	t0,1f
	lb	k1,FM_I2C_STATUS(k0)
	and	k1,k1,FM_I2C_STATUS_TIP
	bnez	k1,1b
	nop
1:
	.endm

	# place a byte in the I2C tx register (k0=i2cbase, k1=scratch)
	.macro	fm_i2c_tx	val
	li	k1,\val
	sb	k1,FM_I2C_TX(k0)
	sync
	.endm

	# write a byte to a particular I2C slave/address
	.macro	fm_i2c_write	slave,address,val
	la		k0,FM_I2C_BASE
	fm_i2c_tx	\slave*2
	fm_i2c_cmd	FM_I2C_CMD_STA+FM_I2C_CMD_WR
	fm_i2c_tx	\address
	fm_i2c_cmd	FM_I2C_CMD_WR
	fm_i2c_tx	\val
	fm_i2c_cmd	FM_I2C_CMD_STO+FM_I2C_CMD_WR
	.endm

	# read a byte from a particular I2C slave/address, return in regno
	.macro	fm_i2c_read	slave,address,regno
	la		k0,FM_I2C_BASE
	fm_i2c_tx	\slave*2
	fm_i2c_cmd	FM_I2C_CMD_STA+FM_I2C_CMD_WR
	fm_i2c_tx	\address
	fm_i2c_cmd	FM_I2C_CMD_WR
	fm_i2c_tx	(\slave*2)+1
	fm_i2c_cmd	FM_I2C_CMD_STA+FM_I2C_CMD_WR
	fm_i2c_cmd	FM_I2C_CMD_STO+FM_I2C_CMD_RD+FM_I2C_CMD_ACK
	lb		\regno,FM_I2C_RX(k0)
	.endm
#else

#define	FPGA(off)	((volatile unsigned int *)FM_FPGA_BASE)[(FM_FPGA_##off)/4]
#define	I2C(off)	((volatile unsigned char *)FM_I2C_BASE)[FM_I2C_##off]
#define	UART(off)	((volatile unsigned char *)FM_UART_BASE)[FM_UART_##off]

static __inline__ void fm_i2c_cmd(unsigned char cmd)
{
	int timeout = 100000;

	I2C(CMD) = cmd;
	__asm__("sync");
	while ((I2C(STATUS) & FM_I2C_STATUS_TIP) && --timeout) ;
}

static __inline__ void fm_i2c_tx(unsigned char v)
{
	I2C(TX) = v;
	__asm__("sync");
}

static __inline__ unsigned char fm_i2c_rx(void)
{
	return I2C(RX);
}

static __inline__ void fm_i2c_write(int slave, int address, int val)
{
	fm_i2c_tx(slave << 1);
	fm_i2c_cmd(FM_I2C_CMD_STA | FM_I2C_CMD_WR);
	fm_i2c_tx(address);
	fm_i2c_cmd(FM_I2C_CMD_WR);
	fm_i2c_tx(val);
	fm_i2c_cmd(FM_I2C_CMD_STO | FM_I2C_CMD_WR);
}

static __inline__ int fm_i2c_read(int slave, int address)
{
	fm_i2c_tx(slave << 1);
	fm_i2c_cmd(FM_I2C_CMD_STA | FM_I2C_CMD_WR);
	fm_i2c_tx(address);
	fm_i2c_cmd(FM_I2C_CMD_WR);
	fm_i2c_tx((slave << 1) + 1);
	fm_i2c_cmd(FM_I2C_CMD_STA | FM_I2C_CMD_WR);
	fm_i2c_cmd(FM_I2C_CMD_STO | FM_I2C_CMD_RD | FM_I2C_CMD_ACK);
	return fm_i2c_rx();
}

static __inline__ void fm_led(int val)
{
	FPGA(LED) = val;	/* only the low byte of 'val' is displayed */
}

static __inline__ int fm_getc(void)
{
	__asm__("sync");
	while ((UART(LSR) & FM_UART_LSR_DR) == 0) ;
	return UART(RX);
}

static __inline__ void fm_putc(int val)
{
	__asm__("sync");
	while ((UART(LSR) & FM_UART_LSR_THRE) == 0) ;
	UART(TX) = val;
	__asm__("sync");
}

static __inline__ void fm_puts(char *s)
{
	while (*s)
		fm_putc(*s++);
}

/* reset watchdog timer to keep the system alive */
static __inline__ void fm_keepalive(void)
{
	FPGA(CONFIG) |= FM_FPGA_WDT_RESET;
}

/* start watchdog timer, 5 second fuse */
static __inline__ void fm_watchdog_start(void)
{
	FPGA(CONFIG) |= FM_FPGA_WDT_ENB | FM_FPGA_WDT_5SEC;	/* enable it */
	fm_keepalive();
}

/* Enable interrupts from fpga */
static __inline__ void fm_fpga_interrupt_setup(void)
{
	FPGA(CONFIG) |= FM_FPGA_PCI_A_ENB | FM_FPGA_UART_ENB | FM_FPGA_I2C_ENB;
}

#endif				/* __ASSEMBLY__ */

#endif				/* _INTRINSITY_FPGA_H */
