/*
 * include/asm-mips/intrinsity/dma.h
 *
 * Include file for Intrinsity DMA
 *
 * Author: MontaVista Software, Inc. <source@mvista.com>
 *
 * 2004 (c) MontaVista, Software, Inc.  This file is licensed under the terms
 * of the GNU General Public License version 2.  This program is licensed
 * "as is" without any warranty of any kind, whether express or implied.
 */
#ifndef	_INTRINSITY_DMA_H
#define	_INTRINSITY_DMA_H

#include <asm/intrinsity/memmap.h>

/* DMA controller */

/* Register offsets from FM_DMA?_BASE */
#define	FM_DMA_CONTROL					0x000
#define	FM_DMA_STATUS					0x040
#define	FM_DMA_CURRENT					0x080
#define	FM_DMA_TAIL					0x0c0
#define	FM_DMA_UPDATE_HEAD				0x100
#define	FM_DMA_UPDATE_TAIL				0x140
#define	FM_DMA_UPDATE_RELEASE				0x180

/* Bits in FM_DMA_CONTROL */
#define	FM_DMA_CONTROL_WAIT_CNT				0x00ff0000
#define	FM_DMA_CONTROL_CLR_SIZ_EX			0x00000080
#define	FM_DMA_CONTROL_CLR_MSM_EX			0x00000040
#define	FM_DMA_CONTROL_CLR_OVF_EX			0x00000020
#define	FM_DMA_CONTROL_CLR_DAT_EX			0x00000010
#define	FM_DMA_CONTROL_CLR_EOLA_EX			0x00000008
#define	FM_DMA_CONTROL_EOLA				0x00000002
#define	FM_DMA_CONTROL_ENABLE				0x00000001

/* Bits in FM_DMA_STATUS */
#define	FM_DMA_STATUS_WAIT_CNT				0x00ff0000
#define	FM_DMA_STATUS_SIZ_EX				0x00000080
#define	FM_DMA_STATUS_MSM_EX				0x00000040
#define	FM_DMA_STATUS_OVF_EX				0x00000020
#define	FM_DMA_STATUS_DAT_EX				0x00000010
#define	FM_DMA_STATUS_EOLA_EX				0x00000008
#define	FM_DMA_STATUS_ACTIVE				0x00000004
#define	FM_DMA_STATUS_EOLA				0x00000002
#define	FM_DMA_STATUS_ENABLE				0x00000001

/* fm_dma_descriptor_t attributes */
#define	FM_DMA_ATTRIBUTES_CACHE_W_CACHEABLE		0x00000008
#define	FM_DMA_ATTRIBUTES_CACHE_R_CACHEABLE		0x00000004

#define	FM_DMA_ATTRIBUTES_TYPE_COPY			0x00000000
#define	FM_DMA_ATTRIBUTES_TYPE_CONSTANT_COPY		0x00000001
#define	FM_DMA_ATTRIBUTES_TYPE_CONSTANT_COMPARE		0x00000002
#define	FM_DMA_ATTRIBUTES_TYPE_TEST_AND_SET		0x00000003

/* fm_dma_descriptor_t {src|dst}TransferSize */	
#define FM_DMA_TRANSFERSIZE_4HI				0x08	/* high 4 bytes of 8 */
#define FM_DMA_TRANSFERSIZE_4LO				0x18	/* low  4 bytes of 8 */
#define FM_DMA_TRANSFERSIZE_5HI				0x07	/* high 4 bytes of 8 */
#define FM_DMA_TRANSFERSIZE_5LO				0x17	/* low  4 bytes of 8 */
#define FM_DMA_TRANSFERSIZE_6HI				0x09	/* high 6 bytes of 8 */
#define FM_DMA_TRANSFERSIZE_6LO				0x19	/* low  6 bytes of 8 */
#define FM_DMA_TRANSFERSIZE_7HI				0x0a	/* high 7 bytes of 8 */
#define FM_DMA_TRANSFERSIZE_7LO				0x1a	/* low  7 bytes of 8 */
#define FM_DMA_TRANSFERSIZE_8				0x0b
#define FM_DMA_TRANSFERSIZE_16				0x1b
#define FM_DMA_TRANSFERSIZE_32				0x0c
#define FM_DMA_TRANSFERSIZE_64				0x1c

#ifndef	__ASSEMBLY__

#define	DMA(base,off)	base[(FM_DMA_##off)/4]

struct fm_dma_descriptor_t {
	unsigned int	srcAddress;		/* physical source address, 8-byte aligned */
	unsigned int	srcCount;		/* number of 8-byte chunks from source (0=disable) */
	unsigned int	srcTransferSize;
	unsigned int	srcStrideSize;		/* stride in units of 8-bytes */
	unsigned int	srcFirstByteEnable;
	unsigned int	srcMidByteEnable;
	unsigned int	srcLastByteEnable;
	unsigned int	dstAddress;		/* physical destination address, 8-byte aligned */
	unsigned int	dstCount;		/* number of 8-byte chunks to destination (0=disable) */
	unsigned int	dstTransferSize;
	unsigned int	dstStrideSize;		/* stride in units of 8-bytes */
	unsigned int	dstFirstByteEnable;
	unsigned int	dstMidByteEnable;
	unsigned int	dstLastByteEnable;
	unsigned int	attributes;
	unsigned int	next;			/* physical address of next descriptor (0=stop) */
	};

/* NOTE:  these functions expect physical addresses, not kernel virtual */

/* initialize a dma descriptor as a memory to memory copy operation */
/* NOTE: This routine currently assumes a count of 2048 bytes */
static __inline__ void
fm_dma_init_descriptor_copy(struct fm_dma_descriptor_t *d, unsigned int src, 
                            unsigned int dst, int count)
{
	d->srcAddress         = src;
	d->srcCount           = 256;
	d->srcTransferSize    = FM_DMA_TRANSFERSIZE_64;
	d->srcStrideSize      = 1;
	d->srcFirstByteEnable = 0xff;
	d->srcMidByteEnable   = 0xff;
	d->srcLastByteEnable  = 0xff;
	d->dstAddress         = dst;
	d->dstCount           = 256;
	d->dstTransferSize    = FM_DMA_TRANSFERSIZE_64;
	d->dstStrideSize      = 1;
	d->dstFirstByteEnable = 0xff;
	d->dstMidByteEnable   = 0xff;
	d->dstLastByteEnable  = 0xff;
	d->attributes         = FM_DMA_ATTRIBUTES_TYPE_COPY;
	d->next               = 0;
}

/* initialize a dma descriptor as a constant compare of memory operation */
static __inline__ void
fm_dma_init_descriptor_constant_compare(struct fm_dma_descriptor_t *d, 
                                        volatile unsigned int m, unsigned int v)
{
	int isOdd = (int)m & 0x4; /* alignment of m. x0 or x8 are even, x4 or xC are odd */

	d->srcAddress         = m;
	d->srcCount           = 1;
	d->srcTransferSize    = v;
	d->srcStrideSize      = 0;
	d->srcFirstByteEnable = isOdd ? 0x0f : 0xf0;
	d->srcMidByteEnable   = 0;
	d->srcLastByteEnable  = 0;
	d->dstAddress         = 0;
	d->dstCount           = 0;
	d->dstTransferSize    = 0;
	d->dstStrideSize      = 1;	/* DMA bug workaround */
	d->dstFirstByteEnable = 0;
	d->dstMidByteEnable   = 0;
	d->dstLastByteEnable  = 0;
	d->attributes         = FM_DMA_ATTRIBUTES_TYPE_CONSTANT_COMPARE;
	d->next               = 0;
}

/* initialize a dma descriptor as a constant copy to memory operation */
static __inline__ void
fm_dma_init_descriptor_constant_copy(struct fm_dma_descriptor_t *d, 
                                     volatile unsigned int m, unsigned int v)
{
	int isOdd = (int)m & 0x4; /* alignment of m. x0 or x8 are even, x4 or xC are odd */

	d->srcAddress         = 0;
	d->srcCount           = 0;
	d->srcTransferSize    = v;
	d->srcStrideSize      = 0;
	d->srcFirstByteEnable = 0;
	d->srcMidByteEnable   = 0;
	d->srcLastByteEnable  = 0;
	d->dstAddress         = m;
	d->dstCount           = 1;
	d->dstTransferSize    = isOdd ? 0x18 : 0x08;
	d->dstStrideSize      = 1;
	d->dstFirstByteEnable = isOdd ? 0x0f : 0xf0;
	d->dstMidByteEnable   = 0;
	d->dstLastByteEnable  = 0;
	d->attributes         = FM_DMA_ATTRIBUTES_TYPE_CONSTANT_COPY;
	d->next               = 0;
}

static __inline__ void
fm_dma_print_descriptor(struct fm_dma_descriptor_t *d)
{
	printk(    "%08x: src(addr %08x, count %x, tsize %x, stride %02x, enable %02x:%02x:%02x)\n",
		(unsigned int)d,
		d->srcAddress,
		d->srcCount,
		d->srcTransferSize,
		d->srcStrideSize,
		d->srcFirstByteEnable,
		d->srcMidByteEnable,
		d->srcLastByteEnable);
	printk("          dst(addr %08x, count %x, tsize %x, stride %02x, enable %02x:%02x:%02x) attr %x next %x\n",
		d->dstAddress,
		d->dstCount,
		d->dstTransferSize,
		d->dstStrideSize,
		d->dstFirstByteEnable,
		d->dstMidByteEnable,
		d->dstLastByteEnable,
		d->attributes,
		d->next);
}

static __inline__ void
fm_dma_print(volatile unsigned int *d)
{
	printk("DMA Controller %d State:\n",(((unsigned int)d)>>3)&0x7);
	printk("      Control: %08x\n",DMA(d,CONTROL));
	printk("       Status: %08x\n",DMA(d,STATUS));
	printk("      Current: %08x\n",DMA(d,CURRENT));
	printk("         Tail: %08x\n",DMA(d,TAIL));
	printk("   UpdateHead: %08x\n",DMA(d,UPDATE_HEAD));
	printk("   UpdateTail: %08x\n",DMA(d,UPDATE_TAIL));
	printk("UpdateRelease: %08x\n",DMA(d,UPDATE_RELEASE));
}

static __inline__ void
fm_dma_start(volatile unsigned int *d, unsigned int head, unsigned int tail, unsigned int release_semaphore)
{
	DMA(d,CONTROL)        = DMA(d,STATUS) | FM_DMA_CONTROL_ENABLE;	/* enable channel */
	__asm__("sync");
	DMA(d,CURRENT)        = 0;
	DMA(d,TAIL)           = 0;
	DMA(d,UPDATE_TAIL)    = tail;
	DMA(d,UPDATE_RELEASE) = release_semaphore;
	__asm__("sync");
	DMA(d,UPDATE_HEAD)    = head;	/* start the DMA */ 
	__asm__("sync");
}

static __inline__ void
fm_dma_stop(volatile unsigned int *d)
{
	DMA(d,CONTROL)        = DMA(d,STATUS) & ~FM_DMA_STATUS_ACTIVE;
	__asm__("sync");
}

static __inline__ void
fm_dma_interrupt_setup(void)
{
	/* TBD */
}

static __inline__ void
fm_dma_interrupt_ack(void)
{
	/* TBD */
	__asm__("sync");
}

#endif	/* __ASSEMBLY__ */

#endif	/* _INTRINSITY_DMA_H */
