/*
 * linux/include/asm-arm/arch-s3c2800/uncompress.h
 *
 * Author: MontaVista Software, Inc. <source@mvista.com>
 *
 * (c) 2003-2004 MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 *
 *
 * HISTORY:
 *
 * 20031022: Brad Parker <brad@heeltoe.com>
 *	Initially based on:
 * 		linux-2.4.18-rmk6-swl6/include/asm-arm/arch-s3c2410/uncompress.h
 *		Copyright (C) 2002 SAMSUNG ELECTORNICS
 *			SW.LEE (hitchcar@sec.samsung.com)
 * 20040301: George G. Davis <gdavis@mvista.com>
 *	Complete rewrite to resolve numerous compile warnings, remove cruft,
 *	added support for autodetection of serial console (1st enabled) or
 *	no forward progress messages are displayed if no serial port enabled;
 *	and lastly check XMIT ready status based on FIFO mode as per S3C2800
 *	UM.
 *
 *
 * TODO:
 *
 * 1. Determine cause of XMIT overruns and eliminate char delay work around
 *    for same when overrun problem is resolved.
 *
 */

#include <linux/types.h>
#include <asm/hardware/serial_s3c2800.h>


/*
 * Access macros for the S3C2800 UARTs:
 *
 * Shamelessly borrowed from linux/drivers/serial/s3c2800.c
 */

#define __raw_readb(p)		*(volatile u8 *)(p)
#define __raw_readl(p)		*(volatile u32 *)(p)
#define __raw_writeb(v,p)	*(volatile u8 *)(p) = (v)
#define __raw_writel(v,p)	*(volatile u32 *)(p) = (v)

#define UART_GET_CHAR(p)	__raw_readb((p) + S3C2800_UARTRXH0_OFF)
#define UART_PUT_CHAR(p,c)	__raw_writeb((c), (p) + S3C2800_UARTTXH0_OFF)

#define UART_GET_ULCON(p)      	__raw_readl((p) + S3C2800_UARTLCON_OFF)
#define UART_GET_UCON(p)       	__raw_readl((p) + S3C2800_UARTCON_OFF)
#define UART_GET_UFCON(p)      	__raw_readl((p) + S3C2800_UARTFCON_OFF)
#define UART_GET_UMCON(p)      	__raw_readl((p) + S3C2800_UARTMCON_OFF)
#define UART_GET_UBRDIV(p)     	__raw_readl((p) + S3C2800_UARTBRDIV_OFF)

#define UART_GET_UTRSTAT(p)   	__raw_readl((p) + S3C2800_UARTTRSTAT_OFF)
#define UART_GET_UERSTAT(p)    	__raw_readl((p) + S3C2800_UARTERSTAT_OFF)
#define UART_GET_UFSTAT(p)     	__raw_readl((p) + S3C2800_UARTFSTAT_OFF)
#define UART_GET_UMSTAT(p)     	__raw_readl((p) + S3C2800_UARTMSTAT_OFF)

#define UART_PUT_ULCON(p,c)    	__raw_writel(c, (p) + S3C2800_UARTLCON_OFF)
#define UART_PUT_UCON(p,c)     	__raw_writel(c, (p) + S3C2800_UARTCON_OFF)
#define UART_PUT_UFCON(p,c)    	__raw_writel(c, (p) + S3C2800_UARTFCON_OFF)
#define UART_PUT_UMCON(p,c)    	__raw_writel(c, (p) + S3C2800_UARTMCON_OFF)
#define UART_PUT_UBRDIV(p,c)   	__raw_writel(c, (p) + S3C2800_UARTBRDIV_OFF)


#define UART_RX_DATA(s)	\
	(((s) & S3C2800_UTRSTAT_RXDR) == S3C2800_UTRSTAT_RXDR)
#define UART_TX_READY(s)	\
	(((s) & S3C2800_UTRSTAT_TXFE) == S3C2800_UTRSTAT_TXFE)
#define TX_FIFOCOUNT(port)	\
	(((UART_GET_UFSTAT(port))>>4)&0xf)


/********************************************/

static inline void
delay(volatile int t)
{
	volatile int i;

	for (i = 0; i < t; i += 1)
		barrier();
}

static inline void
xrdy_wait(volatile void * port)
{
	u32 status;

	if (UART_GET_UFCON(port) & S3C2800_UFCON_FIFOMODE) {
		/* FIFO mode enabled */
		do {
			/* Wait until Xmit FIFO not full */
			status = TX_FIFOCOUNT(port);
		} while (status > 14);
	} else {
                /* FIFO mode disabled */
		do {
			/* Wait until Transmit buffer register is empty */
			status = UART_GET_UTRSTAT(port);
		} while (!(status & S3C2800_UTRSTAT_TXFE));
        }

	/* REVISIT: This is lame but still required to avoid xmit overruns. */
	delay(50000);
}

static void
puts(const char *s)
{
	volatile void * port;
	u32 status;

	/* Determine which serial port to use */
	do {
		port = (void *)(PA_UART_BASE);
		if ((UART_GET_UCON(port) & (3 << 2)) == (1 << 2)) break;
		port = (void *)(PA_UART_BASE+0x10000);
		if ((UART_GET_UCON(port) & (3 << 2)) == (1 << 2)) break;
		/* Quiet boot if no serial ports are enabled... */
		return;
	} while (0);

	/*
	 *      Now, xmit each character
	 */
	while (*s) {
		xrdy_wait(port);
		UART_PUT_CHAR(port, *s);
		if (*s++ == '\n') {
			xrdy_wait(port);
			UART_PUT_CHAR(port, '\r');
		}
	}
}

/********************************************/

#define arch_decomp_wdog()
#define arch_decomp_setup()
