/*
 * include/asm-arm/arch-s3c2800/preem_latency.h
 *
 * Support for preemptible kernel latency measurement times
 *
 * The readclock() macro must return the contents of a monotonic incremental
 * counter.
 *
 * NOTES:
 *
 * 1. This implementation uses the S3C2800 TIMER0 and current jiffies
 *    for interrupt latency measurements. This same timer is also used
 *    for both system timer and HRT sub-HZ time offset calculations.
 *    The setup of this timer is performed during system timer interrupt
 *    setup. Operation of this timer should not be modified. It is of
 *    course safe to read this timer at any time.
 *
 * 2. Since this 16-bit timer lacks sufficient range for measurement of
 *    interrupt latencies in excess of 2^16/CLOCK_RATE seconds, we add
 *    current jiffies to the sub-Hz offset time to avoid erroneous
 *    measurement results due to timer overflow.
 *
 * 3. N.B., if for some reason servicing of the timer interrupt is delayed
 *    in excess of 2^16/CLOCK_RATE seconds, then timer overflow will in
 *    fact occur and we can no longer guarantee time keeping accuracy.
 *
 * Author: George G. Davis <gdavis@mvista.com>
 *
 * 2003-2004 (c) MontaVista Software, Inc.
 *
 * This file is licensed under the terms of the GNU General Public
 * License version 2. This program is licensed "as is" without any
 * warranty of any kind, whether express or implied.
 */

#ifndef __ASM_ARCH_PREEM_LATENCY_H_
#define __ASM_ARCH_PREEM_LATENCY_H_

#include <linux/config.h>
#include <linux/types.h>
#include <linux/timex.h>
#include <asm/hardware.h>

#define readclock_init()

#define readclock(x)							\
	do {								\
		extern unsigned short systimer_mark;			\
		unsigned short delta;					\
		unsigned int ret;					\
		unsigned int temp_jiffies;				\
									\
		do {							\
			/* snapshot jiffies */				\
			temp_jiffies = jiffies;				\
			barrier();					\
									\
			/* calculate cycles since the current jiffy */	\
			delta = ~((unsigned short)(rTMCNT0)) -		\
				systimer_mark;				\
			ret = (unsigned int)(delta) +			\
			      temp_jiffies * (CLOCK_TICK_RATE / HZ); 	\
									\
			barrier();					\
			/* repeat if jiffies has changed */		\
		} while(unlikely(temp_jiffies != jiffies));		\
									\
		x = ret;						\
	} while (0)

static inline unsigned int
clock_to_usecs(unsigned int x)
{
	return (unsigned long long)(x * tick) / (CLOCK_TICK_RATE / HZ);
}

#define TICKS_PER_USEC		1	/* Er, um, 0.1 exactly */

#endif
