/*
 *  linux/include/asm-arm/arch-pnx0105/sdma.h
 *
 *  Simple DMA driver header for PNX0105
 *
 *  Author:	Vitaly Wool
 *  Created:	Aug 17, 2004
 *  Copyright:	MontaVista Software Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 */

#ifndef __PNX0105_SDMA_H__
#define __PNX0105_SDMA_H__

/*
 * Necessary #define's 
 */
#define SDMA_SLAVE_NR_MEMORY        0
#define SDMA_SLAVE_NR_UART_RX       1
#define SDMA_SLAVE_NR_UART_TX       2
#define SDMA_SLAVE_NR_IIC_0         3
#define SDMA_SLAVE_NR_IIC_1         4
#define SDMA_SLAVE_NR_SPI           5
#define SDMA_SLAVE_NR_SAO_1_LEFT    6
#define SDMA_SLAVE_NR_SAO_1_RIGHT   7
#define SDMA_SLAVE_NR_SAO_2_LEFT    8
#define SDMA_SLAVE_NR_SAO_2_RIGHT   9
#define SDMA_SLAVE_NR_SAI_1_LEFT    10
#define SDMA_SLAVE_NR_SAI_1_RIGHT   11
#define SDMA_SLAVE_NR_SAI_2_LEFT    12
#define SDMA_SLAVE_NR_SAI_2_RIGHT   13
#define SDMA_SLAVE_NR_SAI_3_LEFT    14
#define SDMA_SLAVE_NR_SAI_3_RIGHT   15
#define SDMA_SLAVE_NR_LCD           16


#define SDMA_TRANSFER_WORD          0
#define SDMA_TRANSFER_HALF_WORD     1
#define SDMA_TRANSFER_BYTE          2

#define SDMA_INVERT_ENDIAN_NO       0
#define SDMA_INVERT_ENDIAN_YES      1

#define SDMA_COMPANION_DISABLE      0
#define SDMA_COMPANION_ENABLE       1

#define SDMA_CIRC_BUF_DISABLE       0
#define SDMA_CIRC_BUF_ENABLE        1

/*
 * Type definitions
 */ 

/*
 * SDMA configuration structure 
 */
typedef struct sdma_config 
{
	unsigned long write_slave_nr;
	unsigned long read_slave_nr;
	unsigned long transfer_size;
	unsigned long invert_endian;
	unsigned long companion_channel;
	unsigned long companion_enable;
	unsigned long circular_buffer;
} sdma_config_t;

/*
 * SDMA setup structure
 */
typedef struct sdma_setup
{
	unsigned long src_address;
	unsigned long dest_address;
	unsigned long trans_length;
	unsigned long packed_config;
} sdma_setup_t;

/*
 * SDMA scatter-gather list structure 
 */
typedef struct sdma_sg_ll
{
	sdma_setup_t setup;
	struct sdma_sg_ll *next_entry;
} sdma_sg_ll_t;

/*
 * Type of interrupt
 */
typedef enum 
{
	SDMA_IRQ_FINISHED = 0,
	SDMA_IRQ_HALFWAY,
	SDMA_IRQ_SOFTINT,
	SDMA_IRQ_DMAABORT
} sdma_irq_type_t;

/*
 * API definition
 */

/*
 * Pack SDMA configuration structure into a single word
 *
 * Function parameters:
 * 1st parameter - SDMA configuration structure
 * 2nd parameter - pointer to the value ready to be filled into the
 *                 configuration register
 *
 * Returns: 0 on success, otherwise failure
 */
int sdma_pack_config(const sdma_config_t *, unsigned long *);

/*
 * Program SDMA channel
 *
 * Function parameters:
 * 1st parameter - channel number, obtained from sdma_request_channel()
 * 2nd parameter - ptr to the structure containing setup info for the channel
 *
 * Returns: 0 on success, otherwise failure 
 */
int sdma_prog_channel (unsigned int, sdma_setup_t   *);

/*
 * Request SDMA channel
 *
 * Function parameters:
 * 1st parameter - free-form string identifier of channel. Must be non-NULL
 * 2nd parameter - callback function to be invoked when an interrupt
 *                 occurs for this channel
 * 3rd parameter - additional data (callback-specific context) to be passed
 *                 to the callback function when it's invoked
 *
 * Callback parameters:
 * 1st parameter - channel number for which an IRQ occured
 * 2nd parameter - what type of interrupt has happened for a channel
 * 3rd parameter - additional data (callback-specific context)
 * 4th parameter - registers structure as passed to interrupt handler
 *
 * Returns: channel number on success, otherwise (negative) failure 
 */
int sdma_request_channel (char *, void (*)(int, sdma_irq_type_t, void *, struct pt_regs *), void *);

/*
 * Mask/unmask interrupts for the specified channel
 *
 * Function parameters:
 * 1st parameter - SDMA channel number
 * 2nd parameter - 0 to unmask halfway-done interrupt, 1 to mask it
 * 3rd parameter - 0 to unmask on-finish interrupt, 1 to mask it
 *
 * Returns: 0 on success, otherwise failure
 */
int sdma_set_irq_mask(unsigned int, int, int);

/*
 * Start SDMA channel
 *
 * Function parameters:
 * 1st parameter - SDMA channel number
 *
 * Returns: 0 on success, otherwise failure 
 */
int sdma_start_channel (unsigned int);

/*
 * Stop SDMA channel
 *
 * Function parameters:
 * 1st parameter - SDMA channel number
 *
 * Returns: 0 on success, otherwise failure 
 */
int sdma_stop_channel (unsigned int);

/*
 * Release SDMA channel
 *
 * Function parameters:
 * 1st parameter - SDMA channel number
 *
 * Returns: 0 on success, otherwise failure 
 */
int sdma_release_channel (unsigned int);

/*
 * Read channel counter
 *
 * Function parameters:
 * 1st parameter - SDMA channel number
 * 2nd parameter - ptr to the counter variable to be filled
 *
 * Returns: 0 on success, otherwise failure 
 */
int sdma_read_dsp_counter (unsigned int, unsigned int *);

/*
 * Write channel counter
 *
 * Function parameters:
 * 1st parameter - SDMA channel number
 * 2nd parameter - value to be written
 *
 * Returns: 0 on success, otherwise failure 
 */
int sdma_write_dsp_counter (unsigned int, unsigned long);

/*
 * Read current channel state
 *
 * Function parameters:
 * 1st parameter - SDMA channel number
 * 2nd parameter - ptr to the source address variable to be filled
 * 3rd parameter - ptr to the destination address variable to be filled
 * 4th parameter - ptr to the transfer length variable to be filled
 * 5th parameter - ptr to the configuration variable to be filled
 * 6th parameter - ptr to the enable flag variable to be filled
 * 7th parameter - ptr to the address counter variable to be filled
 *
 * Returns: 0 on success, otherwise failure 
 */
int sdma_current_state    (unsigned int, unsigned int *, unsigned int *, unsigned int *, unsigned int  *, unsigned int  *, unsigned int  *);

/*
 * Request SDMA SG channel
 * Reserves two consequent channels with the biggest numbers (actually,
 * 6th and 7th)
 *
 * Function parameters:
 * 1st parameter - free-form string identifier of channel.
 * 2nd parameter - callback function to be invoked when an interrupt
 *                 occurs for this channel
 * 3rd parameter - additional data (callback-specific context) to be passed
 *                 to the callback function when it's invoked
 *
 * Callback parameters:
 * 1st parameter - channel number for which an IRQ occured
 * 2nd parameter - what type of interrupt has happened for a channel
 * 3rd parameter - additional data (callback-specific context)
 * 4th parameter - registers structure as passed to interrupt handler
 *
 * Returns: bigger channel number (actually 7) on success, otherwise 
 * (negative) failure 
 */
int sdma_request_sg_channel (char *, void (*)(int, sdma_irq_type_t, void *, struct pt_regs *), void *);

/*
 * Prepare SG list for programming into the SDMA controller
 * This function is intended to set right companion channel for each
 * entry in the list except the last one and to set the last entry of 
 * the scatter-gather list according to one of the methods to define 
 * the last entry described.
 *
 * Function parameters:
 * 1st parameter - channel number returned by sdma_request_sg_channel()
 *                 (actually 7)
 * 2nd parameter - ptr to the first scatter gather list entry
 *
 * Returns: 0 on success, otherwise failure 
 */
int sdma_prepare_sg_list(int, sdma_sg_ll_t *);

/*
 * Program SDMA SG channel
 * This function is very similar to sdma_prog_channel, but in this case
 * the user has to supply only the linked-list address
 *
 * Function parameters:
 * 1st parameter - channel number returned by sdma_request_sg_channel() 
 *                 (actually 7)
 * 2nd parameter - ptr to the first entry in the linked list
 *
 * Returns: 0 on success, otherwise failure 
 */
int sdma_prog_sg_channel(int, unsigned long *);

/*
 * Release SDMA SG channel
 *
 * Function parameters:
 * 1st parameter - channel number returned by sdma_request_sg_channel()
 * (actually 7)
 *
 * Returns: 0 on success, otherwise failure 
 */
int sdma_release_sg_channel (unsigned int);

#endif /* __PNX0105_SDMA_H__ */

