/*
 * include/asm-arm/arch-pnx0105/preem_latency.h
 *
 * Support for preemptible kernel latency measurement times
 *
 * The readclock() macro must return the contents of a monotonic incremental
 * counter.
 *
 * NOTES:
 *
 * 1. This implementation uses the ARM Integrator TIMER2 exclusively for
 *    preemptible latency measurements. It must not be used for any other
 *    purpose during preemptible kernel latency measurement.
 *
 * 2. This implementation assumes that the TIMER2 clk source is 24MHz.
 *
 * 3. Due to limited range of the timers, it is necessary to make tradeoffs
 *    between min/max latency periods. As currently configured, the min/max
 *    latency which can be measured is approximately 667nS/43.7mS respectively.
 *    If latency times exceed approx 43.7 milliseconds, overflow will occur.
 *
 */

#include <linux/config.h>
#include <linux/types.h>
#include <asm/hardware.h>

typedef struct itimer {
	volatile u32 load;
	volatile u32 value;
	volatile u32 ctrl;
	volatile u32 clr;
} itimer_t;

#define readclock_init()	do {					\
	volatile itimer_t *timer =					\
			(itimer_t *)TIMER0_VA_BASE;			\
	timer->ctrl = 0;						\
	timer->load = 0;						\
	timer->clr = 0;							\
	timer->ctrl = 0x84;	/* Enable, Fin / 16 */			\
	} while (0)

#define readclock(x)	do {						\
		/* 32-bit left justify so the math just works. */	\
		x = ~((itimer_t *)TIMER0_VA_BASE)->value << 16;		\
	} while (0)

#define clock_to_usecs(x)	((x >> 12) / 12)	/* Assumes 12Mhz clk */

/*
 * How long is the timer interval?
 */
#define TICKS_PER_USEC          1	/* approx 1/precisely 0.8  : ( */

