/*
 * linux/include/asm-arm/arch-pnx0105/ide.h
 *
 * Copyright (C) 2003 Philips Semiconductors
 *
 * Rewritten to support PCCard for PNX0105 by Vitaly Wool 
 * 	<vwool@ru.mvista.com>
 * Copyright (C) 2004 MontaVista Software, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */
#include <linux/delay.h>
#include <asm/irq.h>

#define IDE_BM_TIMEOUT		80000000

#define HAVE_ARCH_IN_BYTE
#define HAVE_ARCH_OUT_BYTE

#ifdef PNX0105_SPECIFIC_PCCARD_IO
#define IN_BYTE(p)     (unsigned char)__pnx0105_atareg_read(p)
#define IN_WORD(p)     (unsigned short)__pnx0105_atareg_read(p)
#define OUT_BYTE(b,p)  __pnx0105_atareg_write(b,p)
#define OUT_WORD(w,p)  __pnx0105_atareg_write(w,p)
#else
#define IN_BYTE(p)     inb(p)
#define IN_WORD(p)     inw(p)
#define OUT_BYTE(b,p)  outb(b,p)
#define OUT_WORD(w,p)  outw(w,p)
#endif

#define PNX0105_PCCARD_IO_PORT		IO_ADDRESS(PCCARD_REGS_BASE)

#define PNX0105_PCCARD_IO_BASE		IO_ADDRESS(PCCARD_REGS_BASE)

#define PNX0105_PCCARD_AUTOMATION_OFFSET	0x18
#define PNX0105_PCCARD_VALIDATION_OFFSET	0x1c
#define PNX0105_PCCARD_AUTOCTRL_OFFSET		0xc8
#define PNX0105_PCCARD_SRESET_OFFSET		0xc00

#define PNX0105_PCCARD_SLOTA_CTRL0_OFFSET	0x80
#define PNX0105_PCCARD_SLOTA_CTRL1_OFFSET	0x84
#define PNX0105_PCCARD_SLOTA_HOTPLUG_OFFSET	0x88
#define PNX0105_PCCARD_SLOTB_CTRL0_OFFSET	0xa0
#define PNX0105_PCCARD_SLOTB_CTRL1_OFFSET	0xa4
#define PNX0105_PCCARD_SLOTB_HOTPLUG_OFFSET	0xa8

#define PNX0105_PCCARD_IDE_BASE		(PNX0105_PCCARD_IO_BASE + 0x40)

#define PNX0105_PCCARD_IDE_DATA_OFFSET		0x00
#define PNX0105_PCCARD_IDE_ERROR_OFFSET		0x04
#define PNX0105_PCCARD_IDE_NSECTOR_OFFSET	0x08
#define PNX0105_PCCARD_IDE_SECTOR_OFFSET	0x0c
#define PNX0105_PCCARD_IDE_LCYL_OFFSET		0x10
#define PNX0105_PCCARD_IDE_HCYL_OFFSET		0x14
#define PNX0105_PCCARD_IDE_SELECT_OFFSET	0x18
#define PNX0105_PCCARD_IDE_STATUS_OFFSET	0x1c
#define PNX0105_PCCARD_IDE_CONTROL_OFFSET	0x20

#define PNX0105_PCCARD_BM_BASE		(PNX0105_PCCARD_IO_BASE + 0xc04)

#define PNX0105_PCCARD_BM_CMD_REG_OFSET		0x00
#define PNX0105_PCCARD_BM_STAT_REG_OFFSET	0x04
#define PNX0105_PCCARD_BM_INT_REG_OFFSET	0x08
#define PNX0105_PCCARD_BM_INT_ENA_REG_OFFSET	0x0c
#define PNX0105_PCCARD_BM_PRD_REG_OFFSET	0x10

static inline unsigned int __pnx0105_atareg_read(volatile unsigned long reg)
{
	unsigned int val;
	volatile unsigned long autom = inl(PNX0105_PCCARD_IO_PORT + PNX0105_PCCARD_AUTOMATION_OFFSET);
	if (autom & 1)
		outl(0, PNX0105_PCCARD_IO_PORT + PNX0105_PCCARD_AUTOMATION_OFFSET);

	readl (reg); 
	while ((readl (PNX0105_PCCARD_IO_PORT + PNX0105_PCCARD_VALIDATION_OFFSET) & 0x01) == 0);
	val = readl(reg);
	if (autom & 1)
		outl(1, PNX0105_PCCARD_IO_PORT + PNX0105_PCCARD_AUTOMATION_OFFSET);

	return val;
}

static inline unsigned int __pnx0105_atareg_write(unsigned int value, volatile unsigned long reg)
{
	volatile unsigned long autom = inl(PNX0105_PCCARD_IO_PORT + PNX0105_PCCARD_AUTOMATION_OFFSET);
	if (autom & 1)
		outl(0, PNX0105_PCCARD_IO_PORT + PNX0105_PCCARD_AUTOMATION_OFFSET);
	writel (value, reg); 

	while ((readl (PNX0105_PCCARD_IO_PORT + PNX0105_PCCARD_VALIDATION_OFFSET) & 0x02) != 0);
	if (autom & 1)
		outl(1, PNX0105_PCCARD_IO_PORT + PNX0105_PCCARD_AUTOMATION_OFFSET);
	return 0;
}


static inline void __pnx0105_ata_sreset (unsigned long base, int targetslot, int control0_data, int control1_data, int hotplug_data)
{
    writel (1, base + PNX0105_PCCARD_SRESET_OFFSET); 

    control1_data &= 0xFF7FFFFF;

    if (targetslot) {
        /*  [Slot B = regular IDE/ATA port.] */
        writel (control0_data, base + PNX0105_PCCARD_SLOTB_CTRL0_OFFSET);
        writel (control1_data, base + PNX0105_PCCARD_SLOTB_CTRL1_OFFSET);
        writel (hotplug_data,  base + PNX0105_PCCARD_SLOTB_HOTPLUG_OFFSET);
        udelay (1000);
        writel (control1_data | 0x00800000, base + PNX0105_PCCARD_SLOTB_CTRL1_OFFSET); 
        targetslot = 1;
    } 
    else 
    {
        /*  [Slot A = PC Card slot.]  Runs slot in ATA mode. */
        writel (control0_data, base + PNX0105_PCCARD_SLOTA_CTRL0_OFFSET);
        writel (control1_data | 0x00800000, base + PNX0105_PCCARD_SLOTA_CTRL1_OFFSET);
        writel (hotplug_data, base + PNX0105_PCCARD_SLOTA_HOTPLUG_OFFSET);
        writel (control1_data, base + PNX0105_PCCARD_SLOTA_CTRL1_OFFSET);
        udelay (1000);
        writel (control1_data | 0x00800000, base + PNX0105_PCCARD_SLOTA_CTRL1_OFFSET); 
        targetslot = 0;
    }

    writel (targetslot, base + PNX0105_PCCARD_AUTOCTRL_OFFSET);

}

/*
 * Set up a hw structure for a specified data port, control port and IRQ.
 * This should follow whatever the default interface uses.
 */
static __inline__ void
ide_init_hwif_ports(hw_regs_t *hw, int data_port, int ctrl_port, int *irq)
{
	ide_ioreg_t reg = (ide_ioreg_t) data_port;

        hw->io_ports[IDE_DATA_OFFSET] = reg + PNX0105_PCCARD_IDE_DATA_OFFSET;
        hw->io_ports[IDE_ERROR_OFFSET] = reg + PNX0105_PCCARD_IDE_ERROR_OFFSET;
        hw->io_ports[IDE_NSECTOR_OFFSET] = reg + PNX0105_PCCARD_IDE_NSECTOR_OFFSET;
        hw->io_ports[IDE_SECTOR_OFFSET] = reg + PNX0105_PCCARD_IDE_SECTOR_OFFSET;
        hw->io_ports[IDE_LCYL_OFFSET] = reg + PNX0105_PCCARD_IDE_LCYL_OFFSET;
        hw->io_ports[IDE_HCYL_OFFSET] = reg + PNX0105_PCCARD_IDE_HCYL_OFFSET;
        hw->io_ports[IDE_SELECT_OFFSET] = reg + PNX0105_PCCARD_IDE_SELECT_OFFSET;
        hw->io_ports[IDE_STATUS_OFFSET] = reg + PNX0105_PCCARD_IDE_STATUS_OFFSET;

	hw->io_ports[IDE_CONTROL_OFFSET] = (ide_ioreg_t) ctrl_port;

	if (irq)
		*irq = 0;
}

/*
 * This registers the standard ports for this architecture with the IDE
 * driver.
 */
static __inline__ void ide_init_default_hwifs(void)
{

	hw_regs_t hw;
	unsigned long delay;
	int timeout = 31000000;

	__pnx0105_ata_sreset (PNX0105_PCCARD_IO_PORT, 0, 0x08c808c8, 0x00180A78, 0x1f);
  
  
	udelay(10) ;

	writel (0x00, PNX0105_PCCARD_IO_PORT + PNX0105_PCCARD_AUTOMATION_OFFSET); /* disable automation */

	for (delay = 1000000; delay; delay -= 100)
	{
		udelay(100);
		if (!(__pnx0105_atareg_read(PNX0105_PCCARD_IDE_BASE + PNX0105_PCCARD_IDE_STATUS_OFFSET) & 0x80))
			break;
	}
	
	if (!delay)		
		printk ("Interface did not go non-busy before initial SRST\n");

	__pnx0105_atareg_write (0x06, PNX0105_PCCARD_IDE_BASE + PNX0105_PCCARD_IDE_CONTROL_OFFSET);    /* assert SRST and disable interrupts */
	udelay (10);                                                             /* documentation gives minimum SRST assertion time of 5 uSec */
	__pnx0105_atareg_write (0x02, PNX0105_PCCARD_IDE_BASE + PNX0105_PCCARD_IDE_CONTROL_OFFSET);    /* de-assert SRST (with ints still disabled) */
	udelay (2000); 

	do {
		int delay = 10;
		udelay (delay);
		timeout -= delay;

		if (timeout < 0) {
			printk ("%s: Timeout waiting for master SRST\n", __FILE__);
		}
	} while ( (__pnx0105_atareg_read (PNX0105_PCCARD_IDE_BASE + PNX0105_PCCARD_IDE_STATUS_OFFSET) & 0x80) != 0);

	__pnx0105_atareg_write (0x10, PNX0105_PCCARD_IDE_BASE + PNX0105_PCCARD_IDE_SELECT_OFFSET);
	
	do {
		int delay = 100;
		udelay (delay);
		timeout -= delay;
         
		if (timeout < 0) {
			printk ("%s: Timeout waiting for slave SRST\n", __FILE__);
		}
	} while ((__pnx0105_atareg_read (PNX0105_PCCARD_IDE_BASE + PNX0105_PCCARD_IDE_STATUS_OFFSET) & 0x80) != 0);

	writel (0xff, PNX0105_PCCARD_BM_BASE + PNX0105_PCCARD_BM_INT_REG_OFFSET);	
	writel ((1 << 0) |	/* short tables   BM int */
		(1 << 1) |	/* short transfer BM int */
		(1 << 2) |	/* normal end     BM int */
		(1 << 4) |	/* bus error      BM int */
		(1 << 5),	/* timeout        BM int */
		PNX0105_PCCARD_BM_BASE + PNX0105_PCCARD_BM_INT_ENA_REG_OFFSET); 

	memset(&hw, 0, sizeof(hw));
	ide_init_hwif_ports(&hw, PNX0105_PCCARD_IDE_BASE, PNX0105_PCCARD_IDE_BASE + 0x20, NULL);
	hw.irq =  INT_PCCARD;

	ide_register_hw(&hw, NULL);
}

