/*
 * include/asm-arm/arch-mx1ads/time.h
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Copyright (C) 2002 Shane Nay (shane@minirl.com)
 * 2003-2004 (c) MontaVista Software, Inc. <source@mvista.com>
 */

#include <linux/delay.h>
#include <linux/sc_math.h>
#include <asm/system.h>
#include <asm/leds.h>
#include <asm/timex.h>
#include <asm/hrtime.h>
#include <asm/hardware.h>
#include <asm/mach-types.h>

#define SC_ARCH2USEC        31
#define SC_USEC2ARCH        29

#define arch_cycle_to_usec(cycles)    \
    mpy_sc_n(SC_ARCH2USEC, (cycles), scaled_usec_per_arch_cycle)

#define usec_to_arch_cycle(nsec)    \
    mpy_sc_n(SC_USEC2ARCH, (nsec), scaled_arch_cycles_per_usec)

unsigned long mx1_systimer_mark;
int arch_cycles_per_jiffy;
int scaled_usec_per_arch_cycle;
int scaled_arch_cycles_per_usec;

unsigned long
rdtsc(void)
{
	return ((volatile mx1_gptimer_t *) (TIMER2_VA_BASE))->counter;
}

unsigned long
clock_to_usecs(unsigned long x)
{
	return arch_cycle_to_usec(x);
}

/*
 * Return the number of microseconds since the last timer interrupt.
 * Note that interrupts have been disabled by do_gettimeoffset().
 */
static unsigned long
mx1ads_gettimeoffset(void)
{
	return clock_to_usecs(rdtsc() - mx1_systimer_mark);
}

/*
 * timer interrupt handler
 */
static void
mx1ads_timer_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	volatile mx1_gptimer_t *hz_timer;
	unsigned long now;

	now = rdtsc();

	hz_timer = (volatile mx1_gptimer_t *) TIMER1_VA_BASE;

	/* clear previous timer interrupt */
	if (hz_timer->status) {
		hz_timer->status = 0x0;
	}

	do {
		barrier();
		mx1_systimer_mark += arch_cycles_per_jiffy;
		do_timer(regs);
		do_leds();
		do_set_rtc();
		barrier();
	} while (unlikely((now - mx1_systimer_mark) >= arch_cycles_per_jiffy));

	do_profile(regs);
}

/*
 * Set up the timer interrupt.
 */
static inline void
setup_timer(void)
{
	extern unsigned long (*gettimeoffset) (void);
	volatile mx1_gptimer_t *hz_timer;
	volatile mx1_gptimer_t *tsc_timer;

	hz_timer = (volatile mx1_gptimer_t *) TIMER1_VA_BASE;
	tsc_timer = (volatile mx1_gptimer_t *) TIMER2_VA_BASE;

	mx1_systimer_mark = 0;

	gettimeoffset = mx1ads_gettimeoffset;

	if (hz_timer->control & TIM_TEN) {
		hz_timer->control = 0;
		tsc_timer->control = 0;
		udelay(1);
		hz_timer->control = TIM_SWR;
		tsc_timer->control = TIM_SWR;
		udelay(1);
		hz_timer->control = 0;
		tsc_timer->control = 0;
		udelay(1);
	}

	timer_irq.handler = mx1ads_timer_interrupt;
	timer_irq.flags = SA_INTERRUPT;

	setup_arm_irq(IRQ_TIMER1, &timer_irq);

	if (machine_is_mx1ads()) {
		switch (__raw_readl(IO_ADDRESS(MX1_SIC_ID))) {
		case 0x0005901d: /* MX1ADS Ver 1.1 MC9328MX1 1L44N Rev 1.1 */
		case 0x04d4c01d: /* MX1ADS Ver 1.0 MC9328MXL 0L45N Rev 1.0 */
			/* Pre silicon rev 2.2 uses 32.000KHz XTAL */
			arch_cycles_per_jiffy = CLOCK_TICK_RATE / HZ;
			scaled_usec_per_arch_cycle =
			    SC_n(SC_ARCH2USEC, USEC_PER_SEC) / CLOCK_TICK_RATE;
			scaled_arch_cycles_per_usec =
			    SC_n(SC_USEC2ARCH, CLOCK_TICK_RATE) / USEC_PER_SEC;
#ifdef	CONFIG_HIGH_RES_TIMERS
			scaled_nsec_per_arch_cycle =
			    SC_n(SC_ARCH2NSEC, NSEC_PER_SEC) / CLOCK_TICK_RATE;
			scaled_arch_cycles_per_nsec =
			    SC_n(SC_NSEC2ARCH, CLOCK_TICK_RATE) / NSEC_PER_SEC;
#endif
			break;
		case 0x0001901d: /* MX1ADS Ver 1.1 MC9328MX1 2L44N Rev 2.2 */
		case 0x00d4c01d: /* MX1ADS Ver 1.1 MC9328MXL 2L45N Rev 2.2 */
		default:
			/* Silicon rev 2.2 boards have 32.768KHz XTAL */
			arch_cycles_per_jiffy = 16384000 / 8 / HZ;
			scaled_usec_per_arch_cycle =
			    SC_n(SC_ARCH2USEC, USEC_PER_SEC) / (16384000 / 8);
			scaled_arch_cycles_per_usec =
			    SC_n(SC_USEC2ARCH, (16384000 / 8)) / USEC_PER_SEC;
#ifdef	CONFIG_HIGH_RES_TIMERS
			scaled_nsec_per_arch_cycle =
			    SC_n(SC_ARCH2NSEC, NSEC_PER_SEC) / (16384000 / 8);
			scaled_arch_cycles_per_nsec =
			    SC_n(SC_NSEC2ARCH, (16384000 / 8)) / NSEC_PER_SEC;
#endif
			break;
		}
	} else {
		arch_cycles_per_jiffy = CLOCK_TICK_RATE / HZ;
		scaled_usec_per_arch_cycle =
		    SC_n(SC_ARCH2USEC, USEC_PER_SEC) / (16384000 / 8);
		scaled_arch_cycles_per_usec =
		    SC_n(SC_USEC2ARCH, (16384000 / 8)) / USEC_PER_SEC;
#ifdef	CONFIG_HIGH_RES_TIMERS
		scaled_nsec_per_arch_cycle =
		    SC_n(SC_ARCH2NSEC, NSEC_PER_SEC) / CLOCK_TICK_RATE;
		scaled_arch_cycles_per_nsec =
		    SC_n(SC_NSEC2ARCH, CLOCK_TICK_RATE) / NSEC_PER_SEC;
#endif
	}

	hz_timer->control = TIM_IRQEN | TIM_CLKSOURCE_PERCLK1;
	tsc_timer->control = TIM_FRR | TIM_CLKSOURCE_PERCLK1;
	hz_timer->prescaler = 7;
	tsc_timer->prescaler = 7;
	hz_timer->compare = arch_cycles_per_jiffy - 1;
	hz_timer->control |= TIM_TEN;
	tsc_timer->control |= TIM_TEN;
}
