/*
 * include/asm-arm/arch-ixp425/ixp4xx.h
 *
 * Register definitions for IXP4xx
 *
 * Copyright (C) 2004 Intel Corporation.
 *
 * Maintainer: Deepak Saxena <dsaxena@mvista.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

#ifndef __ASM_ARCH_HARDWARE_H__
#error "Do not include this directly, please #include <asm/hardware.h>"
#endif

#ifndef _ASM_ARM_IXP4XX_H_
#define _ASM_ARM_IXP4XX_H_

#ifndef BIT
#define BIT(bit)	(1 << (bit))
#endif


/*
 *
 * IXP4XX Memory map:
 *
 * Start Address	Stop Address	Size				Description
 * =========================================================================
 *
 * 0x00000000		0x0FFFFFFF		0x10000000	(256M)	Expansion Bus
 *
 * 0x00000000		0x3FFFFFFF		0x40000000	  (1G)	SDRAM
 *
 * 0x40000000		0x47FFFFFF		0x08000000	(128M)	Reserved
 *
 * 0x48000000		0x4FFFFFFF		0x08000000	(128M)	PCI Memory
 *
 * 0x50000000		0x5FFFFFFF		0x10000000	(256M)	Expansion Bus
 *
 * 0x60000000		0x63FFFFFF		0x04000000	 (64M)	QMgr
 *
 * 0x64000000		0xBFFFFFFF							Reserved
 *
 * 0xC0000000		0xC3FFFFFF		0x04000000	 (64M)	PCI Config
 *
 * 0xC4000000		0xC7FFFFFF		0x04000000	 (64M)	Expansion Bus Config
 *
 * 0xC8000000		0xCBFFFFFF		0x04000000	 (64M)	Peripherals
 *
 * 0xCC000000		0xCCFFFFFF		0x01000000	 (16M)	SDRAM Config
 *
 * 0xCD000000		0xCDFFFFFF		0x01000000	 (16M)	USB Host Controller (46x only)
 *
 * 0xCE000000		0xFFFFFFFF							Reserved
 *
 */


/**************************************************
 * Expansion Bus
 *************************************************/
#define IXP4XX_EXP_BUS_BASE1_PHYS		(0x00000000)
#define IXP4XX_EXP_BUS_BASE2_PHYS		(0x50000000)
#define IXP4XX_EXP_BUS_BASE2_VIRT		(0xF0000000)

#define IXP4XX_EXP_BUS_BASE_PHYS		IXP4XX_EXP_BUS_BASE2_PHYS
#define IXP4XX_EXP_BUS_BASE_VIRT		IXP4XX_EXP_BUS_BASE2_VIRT

#define IXP4XX_EXP_BUS_CSX_REGION_SIZE	(0x01000000)

/*
 * Expansion Bus 'lives' at either base1 or base 2 depending on the value of
 * Exp Bus config registers:
 *
 * Setting bit 31 of IXP4XX_EXP_CFG0 puts SDRAM at zero,
 * and the expansion bus to IXP4XX_EXP_BUS_BASE2
 *
 */

/* define to support 32M devices */
#undef	IXP4XX_EXP_BUS_32M_SUPPORT

#ifdef CONFIG_ARCH_IXDP465
#define	IXP4XX_EXP_BUS_32M_SUPPORT
#endif


#ifdef IXP4XX_EXP_BUS_32M_SUPPORT

#define IXP4XX_EXP_BUS_REGION_SIZE		(0x0F000000)

#define IXP4XX_EXP_BUS_CS0_OFFSET		0x00000000
#define IXP4XX_EXP_BUS_CS1_OFFSET		0x02000000
#define IXP4XX_EXP_BUS_CS2_OFFSET		0x04000000
#define IXP4XX_EXP_BUS_CS3_OFFSET		0x06000000
#define IXP4XX_EXP_BUS_CS4_OFFSET		0x08000000
#define IXP4XX_EXP_BUS_CS5_OFFSET		0x0a000000
#define IXP4XX_EXP_BUS_CS6_OFFSET		0x0c000000
#define IXP4XX_EXP_BUS_CS7_OFFSET		0x0e000000

#else

#define IXP4XX_EXP_BUS_REGION_SIZE		(0x08000000)

#define IXP4XX_EXP_BUS_CS0_OFFSET		0x00000000
#define IXP4XX_EXP_BUS_CS1_OFFSET		0x01000000
#define IXP4XX_EXP_BUS_CS2_OFFSET		0x02000000
#define IXP4XX_EXP_BUS_CS3_OFFSET		0x03000000
#define IXP4XX_EXP_BUS_CS4_OFFSET		0x04000000
#define IXP4XX_EXP_BUS_CS5_OFFSET		0x05000000
#define IXP4XX_EXP_BUS_CS6_OFFSET		0x06000000
#define IXP4XX_EXP_BUS_CS7_OFFSET		0x07000000

#endif

#define IXP4XX_EXP_BUS_CS0_BASE_PHYS	(IXP4XX_EXP_BUS_BASE2_PHYS + IXP4XX_EXP_BUS_CS0_OFFSET)
#define IXP4XX_EXP_BUS_CS1_BASE_PHYS	(IXP4XX_EXP_BUS_BASE2_PHYS + IXP4XX_EXP_BUS_CS1_OFFSET)
#define IXP4XX_EXP_BUS_CS2_BASE_PHYS	(IXP4XX_EXP_BUS_BASE2_PHYS + IXP4XX_EXP_BUS_CS2_OFFSET)
#define IXP4XX_EXP_BUS_CS3_BASE_PHYS	(IXP4XX_EXP_BUS_BASE2_PHYS + IXP4XX_EXP_BUS_CS3_OFFSET)
#define IXP4XX_EXP_BUS_CS4_BASE_PHYS	(IXP4XX_EXP_BUS_BASE2_PHYS + IXP4XX_EXP_BUS_CS4_OFFSET)
#define IXP4XX_EXP_BUS_CS5_BASE_PHYS	(IXP4XX_EXP_BUS_BASE2_PHYS + IXP4XX_EXP_BUS_CS5_OFFSET)
#define IXP4XX_EXP_BUS_CS6_BASE_PHYS	(IXP4XX_EXP_BUS_BASE2_PHYS + IXP4XX_EXP_BUS_CS6_OFFSET)
#define IXP4XX_EXP_BUS_CS7_BASE_PHYS	(IXP4XX_EXP_BUS_BASE2_PHYS + IXP4XX_EXP_BUS_CS7_OFFSET)

#define IXP4XX_EXP_BUS_CS0_BASE_VIRT	(IXP4XX_EXP_BUS_BASE2_VIRT + IXP4XX_EXP_BUS_CS0_OFFSET)
#define IXP4XX_EXP_BUS_CS1_BASE_VIRT	(IXP4XX_EXP_BUS_BASE2_VIRT + IXP4XX_EXP_BUS_CS1_OFFSET)
#define IXP4XX_EXP_BUS_CS2_BASE_VIRT	(IXP4XX_EXP_BUS_BASE2_VIRT + IXP4XX_EXP_BUS_CS2_OFFSET)
#define IXP4XX_EXP_BUS_CS3_BASE_VIRT	(IXP4XX_EXP_BUS_BASE2_VIRT + IXP4XX_EXP_BUS_CS3_OFFSET)
#define IXP4XX_EXP_BUS_CS4_BASE_VIRT	(IXP4XX_EXP_BUS_BASE2_VIRT + IXP4XX_EXP_BUS_CS4_OFFSET)
#define IXP4XX_EXP_BUS_CS5_BASE_VIRT	(IXP4XX_EXP_BUS_BASE2_VIRT + IXP4XX_EXP_BUS_CS5_OFFSET)
#define IXP4XX_EXP_BUS_CS6_BASE_VIRT	(IXP4XX_EXP_BUS_BASE2_VIRT + IXP4XX_EXP_BUS_CS6_OFFSET)
#define IXP4XX_EXP_BUS_CS7_BASE_VIRT	(IXP4XX_EXP_BUS_BASE2_VIRT + IXP4XX_EXP_BUS_CS7_OFFSET)

/**************************************************
 * Q Manager
 *************************************************/
#define IXP4XX_QMGR_BASE_PHYS			(0x60000000)
#define IXP4XX_QMGR_BASE_VIRT			(0xFF100000)
#define IXP4XX_QMGR_REGION_SIZE			(0x00004000)

/**************************************************
 * PCI Configuration space
 *************************************************/
#define IXP4XX_PCI_CFG_BASE_PHYS		(0xC0000000)
#define IXP4XX_PCI_CFG_BASE_VIRT		(0xFF01C000)
#define IXP4XX_PCI_CFG_REGION_SIZE		(0x00001000)

#define PCI_NP_AD_OFFSET				0x00
#define PCI_NP_CBE_OFFSET				0x04
#define PCI_NP_WDATA_OFFSET				0x08
#define PCI_NP_RDATA_OFFSET				0x0c
#define PCI_CRP_AD_CBE_OFFSET			0x10
#define PCI_CRP_WDATA_OFFSET			0x14
#define PCI_CRP_RDATA_OFFSET			0x18
#define PCI_CSR_OFFSET					0x1c
#define PCI_ISR_OFFSET					0x20
#define PCI_INTEN_OFFSET				0x24
#define PCI_DMACTRL_OFFSET				0x28
#define PCI_AHBMEMBASE_OFFSET			0x2c
#define PCI_AHBIOBASE_OFFSET			0x30
#define PCI_PCIMEMBASE_OFFSET			0x34
#define PCI_AHBDOORBELL_OFFSET			0x38
#define PCI_PCIDOORBELL_OFFSET			0x3C
#define PCI_ATPDMA0_AHBADDR_OFFSET		0x40
#define PCI_ATPDMA0_PCIADDR_OFFSET		0x44
#define PCI_ATPDMA0_LENADDR_OFFSET		0x48
#define PCI_ATPDMA1_AHBADDR_OFFSET		0x4C
#define PCI_ATPDMA1_PCIADDR_OFFSET		0x50
#define PCI_ATPDMA1_LENADDR_OFFSET		0x54

#define IXP4XX_PCI_CSR(x) ((volatile u32 *)(IXP4XX_PCI_CFG_BASE_VIRT+(x)))

#define PCI_NP_AD						IXP4XX_PCI_CSR(PCI_NP_AD_OFFSET)
#define PCI_NP_CBE						IXP4XX_PCI_CSR(PCI_NP_CBE_OFFSET)
#define PCI_NP_WDATA					IXP4XX_PCI_CSR(PCI_NP_WDATA_OFFSET)
#define PCI_NP_RDATA					IXP4XX_PCI_CSR(PCI_NP_RDATA_OFFSET)
#define PCI_CRP_AD_CBE					IXP4XX_PCI_CSR(PCI_CRP_AD_CBE_OFFSET)
#define PCI_CRP_WDATA					IXP4XX_PCI_CSR(PCI_CRP_WDATA_OFFSET)
#define PCI_CRP_RDATA					IXP4XX_PCI_CSR(PCI_CRP_RDATA_OFFSET)
#define PCI_CSR							IXP4XX_PCI_CSR(PCI_CSR_OFFSET)
#define PCI_ISR							IXP4XX_PCI_CSR(PCI_ISR_OFFSET)
#define PCI_INTEN						IXP4XX_PCI_CSR(PCI_INTEN_OFFSET)
#define PCI_DMACTRL						IXP4XX_PCI_CSR(PCI_DMACTRL_OFFSET)
#define PCI_AHBMEMBASE					IXP4XX_PCI_CSR(PCI_AHBMEMBASE_OFFSET)
#define PCI_AHBIOBASE					IXP4XX_PCI_CSR(PCI_AHBIOBASE_OFFSET)
#define PCI_PCIMEMBASE					IXP4XX_PCI_CSR(PCI_PCIMEMBASE_OFFSET)
#define PCI_AHBDOORBELL					IXP4XX_PCI_CSR(PCI_AHBDOORBELL_OFFSET)
#define PCI_PCIDOORBELL					IXP4XX_PCI_CSR(PCI_PCIDOORBELL_OFFSET)
#define PCI_ATPDMA0_AHBADDR				IXP4XX_PCI_CSR(PCI_ATPDMA0_AHBADDR_OFFSET)
#define PCI_ATPDMA0_PCIADDR				IXP4XX_PCI_CSR(PCI_ATPDMA0_PCIADDR_OFFSET)
#define PCI_ATPDMA0_LENADDR				IXP4XX_PCI_CSR(PCI_ATPDMA0_LENADDR_OFFSET)
#define PCI_ATPDMA1_AHBADDR				IXP4XX_PCI_CSR(PCI_ATPDMA1_AHBADDR_OFFSET)
#define PCI_ATPDMA1_PCIADDR				IXP4XX_PCI_CSR(PCI_ATPDMA1_PCIADDR_OFFSET)
#define PCI_ATPDMA1_LENADDR				IXP4XX_PCI_CSR(PCI_ATPDMA1_LENADDR_OFFSET)

/* CSR bit definitions */
#define PCI_CSR_HOST					BIT(0)
#define PCI_CSR_ARBEN					BIT(1)
#define PCI_CSR_ADS						BIT(2)
#define PCI_CSR_PDS						BIT(3)
#define PCI_CSR_ABE						BIT(4)
#define PCI_CSR_DBT						BIT(5)
#define PCI_CSR_ASE						BIT(8)
#define PCI_CSR_IC						BIT(15)

/* ISR (Interrupt status) Register bit definitions */
#define PCI_ISR_PSE						BIT(0)
#define PCI_ISR_PFE						BIT(1)
#define PCI_ISR_PPE						BIT(2)
#define PCI_ISR_AHBE					BIT(3)
#define PCI_ISR_APDC					BIT(4)
#define PCI_ISR_PADC					BIT(5)
#define PCI_ISR_ADB						BIT(6)
#define PCI_ISR_PDB						BIT(7)

/* INTEN (Interrupt Enable) Register bit definitions */
#define PCI_INTEN_PSE					BIT(0)
#define PCI_INTEN_PFE					BIT(1)
#define PCI_INTEN_PPE					BIT(2)
#define PCI_INTEN_AHBE					BIT(3)
#define PCI_INTEN_APDC					BIT(4)
#define PCI_INTEN_PADC					BIT(5)
#define PCI_INTEN_ADB					BIT(6)
#define PCI_INTEN_PDB					BIT(7)

/* Shift value for byte enable on NP cmd/byte enable register */
#define IXP4XX_PCI_NP_CBE_BESL			4

/* PCI commands supported by NP access unit */
#define NP_CMD_IOREAD					0x2
#define NP_CMD_IOWRITE					0x3
#define NP_CMD_CONFIGREAD				0xa
#define NP_CMD_CONFIGWRITE				0xb
#define NP_CMD_MEMREAD					0x6
#define NP_CMD_MEMWRITE					0x7

#ifndef __ASSEMBLY__
extern int (*ixp425_pci_read)(u32 addr, u32 cmd, u32* data);
extern int	ixp425_pci_write(u32 addr, u32 cmd, u32 data);
extern void ixp425_pci_init(void *);
#endif

/* Constants for CRP access into local config space */
#define CRP_AD_CBE_BESL					20
#define CRP_AD_CBE_WRITE				BIT(16)

/**************************************************
 * Expansion Bus Configuration space
 *************************************************/
#define IXP4XX_EXP_CFG_BASE_PHYS		(0xC4000000)
#define IXP4XX_EXP_CFG_BASE_VIRT		(0xFF01D000)
#define IXP4XX_EXP_CFG_REGION_SIZE		(0x00001000)

#define IXP4XX_EXP_CS0_OFFSET			0x00	/* Timing and Control Chip Select 0 */
#define IXP4XX_EXP_CS1_OFFSET			0x04	/* Timing and Control Chip Select 1 */
#define IXP4XX_EXP_CS2_OFFSET			0x08	/* Timing and Control Chip Select 2 */
#define IXP4XX_EXP_CS3_OFFSET			0x0C	/* Timing and Control Chip Select 3 */
#define IXP4XX_EXP_CS4_OFFSET			0x10	/* Timing and Control Chip Select 4 */
#define IXP4XX_EXP_CS5_OFFSET			0x14	/* Timing and Control Chip Select 5 */
#define IXP4XX_EXP_CS6_OFFSET			0x18	/* Timing and Control Chip Select 6 */
#define IXP4XX_EXP_CS7_OFFSET			0x1C	/* Timing and Control Chip Select 7 */

#define IXP4XX_EXP_CFG0_OFFSET			0x20	/* General Purpose Config 0 */
#define IXP4XX_EXP_CFG1_OFFSET			0x24	/* General Purpose Config 1 */

/* 46x only */
#define IXP4XX_EXP_FUSE_OFFSET			0x028	/* Specifies fuse value		*/
#define IXP4XX_EXP_SMII_OFFSET			0x02c	/* DLL bits for SMII		*/
#define IXP4XX_EXP_MST_OFFSET			0x090	/* Bus arbitration prioriy	*/
#define IXP4XX_EXP_INADDR_OFFSET		0x100	/* Upper AHB address		*/
#define IXP4XX_EXP_LOCK0_OFFSET			0x104	/* Locking multiple masters */
#define IXP4XX_EXP_LOCK1_OFFSET			0x108	/* Locking multiple masters */
#define IXP4XX_EXP_PAR_OFFSET 			0x10C	/* Parity Error Status		*/
#define IXP4XX_EXP_SYNC_OFFSET			0x120	/* Read Latency Count		*/
#define IXP4XX_EXP_RCOMP_OFFSET			0x124	/* SMII RComp Control		*/

#define IXP4XX_EXP_REG(x) ((volatile u32 *)(IXP4XX_EXP_CFG_BASE_VIRT+(x)))

#define IXP4XX_EXP_CS0					IXP4XX_EXP_REG(IXP4XX_EXP_CS0_OFFSET)
#define IXP4XX_EXP_CS1					IXP4XX_EXP_REG(IXP4XX_EXP_CS1_OFFSET)
#define IXP4XX_EXP_CS2					IXP4XX_EXP_REG(IXP4XX_EXP_CS2_OFFSET)
#define IXP4XX_EXP_CS3					IXP4XX_EXP_REG(IXP4XX_EXP_CS3_OFFSET)
#define IXP4XX_EXP_CS4					IXP4XX_EXP_REG(IXP4XX_EXP_CS4_OFFSET)
#define IXP4XX_EXP_CS5					IXP4XX_EXP_REG(IXP4XX_EXP_CS5_OFFSET)
#define IXP4XX_EXP_CS6					IXP4XX_EXP_REG(IXP4XX_EXP_CS6_OFFSET)
#define IXP4XX_EXP_CS7					IXP4XX_EXP_REG(IXP4XX_EXP_CS7_OFFSET)

#define IXP4XX_EXP_CFG0					IXP4XX_EXP_REG(IXP4XX_EXP_CFG0_OFFSET)
#define IXP4XX_EXP_CFG1					IXP4XX_EXP_REG(IXP4XX_EXP_CFG1_OFFSET)

#define IXP4XX_EXP_FUSE					IXP4XX_EXP_REG(IXP4XX_EXP_FUSE_OFFSET)


/* EXP CSx */
#define IXP4XX_EXP_CS_BYTE_EN			BIT(0)
#define IXP4XX_EXP_CS_WR_EN				BIT(1)
#define IXP4XX_EXP_CS_WORD_EN			BIT(2)
#define IXP4XX_EXP_CS_SPLIT_EN			BIT(3)
#define IXP4XX_EXP_CS_MUX_EN			BIT(4)
#define IXP4XX_EXP_CS_HRDY_POL			BIT(5)
#define IXP4XX_EXP_CS_BYTE_RD16			BIT(6)

/* 46x only */
#define IXP4XX_EXP_CS_CHIP				BIT(7)
#define IXP4XX_EXP_CS_SYNC				BIT(8)
#define IXP4XX_EXP_CS_PAR_EN			BIT(30)

#define IXP4XX_EXP_CS_EN				BIT(31)

/* EXP CFG 0 */
#define IXP4XX_EXP_CFG0_8_FLASH			BIT(0)
#define IXP4XX_EXP_CFG0_PCI_HOST		BIT(1)
#define IXP4XX_EXP_CFG0_PCI_ARB			BIT(2)
#define IXP4XX_EXP_CFG0_PCI_CLK			BIT(4)

/* 46x only */
#define IXP4XX_EXP_CFG0_DRIVE0			BIT(5)
#define IXP4XX_EXP_CFG0_ARB				BIT(6)
#define IXP4XX_EXP_CFG0_32_FLASH		BIT(7)
#define IXP4XX_EXP_CFG0_USB_CLK			BIT(8)
#define IXP4XX_EXP_CFG0_DRIVE1			BIT(9)
#define IXP4XX_EXP_CFG0_IOWAIT			BIT(10)
#define IXP4XX_EXP_CFG0_CUST			(0x001e0000)

#define IXP4XX_EXP_CFG0_MEM_MAP			BIT(31)
#define IXP4XX_EXP_CFG0_CORE_CLK		(0x00e00000)

/* EXP CFG 1 */
#define IXP4XX_EXP_CFG1_SW_INT0			BIT(0)
#define IXP4XX_EXP_CFG1_SW_INT1			BIT(1)
#define IXP4XX_EXP_CFG1_SWAP_EN			BIT(8)

/* 46x only */
#define IXP4XX_EXP_CFG1_FORCE_SWAP 		BIT(9)
#define IXP4XX_EXP_CFG1_EXP_SWAP_EN		BIT(10)
#define IXP4XX_EXP_CFG1_NPEA_EEN		BIT(12)
#define IXP4XX_EXP_CFG1_NPEB_EEN		BIT(13)
#define IXP4XX_EXP_CFG1_NPEC_EEN		BIT(14)
#define IXP4XX_EXP_CFG1_NPEA_SMII		BIT(16)
#define IXP4XX_EXP_CFG1_NPEB_SMII		BIT(17)
#define IXP4XX_EXP_CFG1_NPEC_SMII		BIT(18)
#define IXP4XX_EXP_CFG1_SMII_SYNC		BIT(19)
#define IXP4XX_EXP_CFG1_MPI_EN			BIT(31)
/* Flash Defines */
#define IXP4XX_FLASH_WRITABLE			IXP4XX_EXP_CS_WR_EN

#ifdef IXP4XX_EXP_BUS_32M_SUPPORT
/* Enabled, T1=3, T2=3, T3=3, T4=1, T5=2, 32M, Byte Access */
#define IXP4XX_FLASH_DEFAULT			(0xbcd20240)
#else
/* Enabled, T1=3, T2=3, T3=3, T4=1, T5=2, 16M, Byte Access */
#define IXP4XX_FLASH_DEFAULT			(0xbcd23c40)
#endif

#define IXP4XX_FLASH_WRITE				(IXP4XX_FLASH_WRITABLE|IXP4XX_FLASH_DEFAULT)

/**************************************************
 * Peripherals
 *************************************************/
#define IXP4XX_PERIPHERAL_BASE_PHYS		(0xC8000000)
#define IXP4XX_PERIPHERAL_BASE_VIRT		(0xFF000000)
#define IXP4XX_PERIPHERAL_REGION_SIZE	(0x00013000)

#define IXP4XX_UART1_BASE_PHYS			(IXP4XX_PERIPHERAL_BASE_PHYS + 0x00000)
#define IXP4XX_UART2_BASE_PHYS			(IXP4XX_PERIPHERAL_BASE_PHYS + 0x01000)
#define IXP4XX_PMU_BASE_PHYS			(IXP4XX_PERIPHERAL_BASE_PHYS + 0x02000)
#define IXP4XX_INTC_BASE_PHYS			(IXP4XX_PERIPHERAL_BASE_PHYS + 0x03000)
#define IXP4XX_GPIO_BASE_PHYS			(IXP4XX_PERIPHERAL_BASE_PHYS + 0x04000)
#define IXP4XX_TIMER_BASE_PHYS			(IXP4XX_PERIPHERAL_BASE_PHYS + 0x05000)
#define IXP4XX_NPEA_BASE_PHYS			(IXP4XX_PERIPHERAL_BASE_PHYS + 0x06000)
#define IXP4XX_NPEB_BASE_PHYS			(IXP4XX_PERIPHERAL_BASE_PHYS + 0x07000)
#define IXP4XX_NPEC_BASE_PHYS			(IXP4XX_PERIPHERAL_BASE_PHYS + 0x08000)
#define IXP4XX_EthB0_BASE_PHYS			(IXP4XX_PERIPHERAL_BASE_PHYS + 0x09000)
#define IXP4XX_EthC_BASE_PHYS			(IXP4XX_PERIPHERAL_BASE_PHYS + 0x0A000)
#define IXP4XX_USB_BASE_PHYS			(IXP4XX_PERIPHERAL_BASE_PHYS + 0x0B000)

/* 46x only */
#define IXP4XX_EthA_BASE_PHYS			(IXP4XX_PERIPHERAL_BASE_PHYS + 0x0C000)
#define IXP4XX_EthB1_BASE_PHYS			(IXP4XX_PERIPHERAL_BASE_PHYS + 0x0D000)
#define IXP4XX_EthB2_BASE_PHYS			(IXP4XX_PERIPHERAL_BASE_PHYS + 0x0E000)
#define IXP4XX_EthB3_BASE_PHYS			(IXP4XX_PERIPHERAL_BASE_PHYS + 0x0F000)
#define IXP4XX_TIMESYNC_BASE_PHYS		(IXP4XX_PERIPHERAL_BASE_PHYS + 0x10000)
#define IXP4XX_I2C_BASE_PHYS			(IXP4XX_PERIPHERAL_BASE_PHYS + 0x11000)
#define IXP4XX_SSP_BASE_PHYS			(IXP4XX_PERIPHERAL_BASE_PHYS + 0x12000)

#define IXP4XX_UART1_BASE_VIRT			(IXP4XX_PERIPHERAL_BASE_VIRT + 0x00000)
#define IXP4XX_UART2_BASE_VIRT			(IXP4XX_PERIPHERAL_BASE_VIRT + 0x01000)
#define IXP4XX_PMU_BASE_VIRT			(IXP4XX_PERIPHERAL_BASE_VIRT + 0x02000)
#define IXP4XX_INTC_BASE_VIRT			(IXP4XX_PERIPHERAL_BASE_VIRT + 0x03000)
#define IXP4XX_GPIO_BASE_VIRT			(IXP4XX_PERIPHERAL_BASE_VIRT + 0x04000)
#define IXP4XX_TIMER_BASE_VIRT			(IXP4XX_PERIPHERAL_BASE_VIRT + 0x05000)
#define IXP4XX_NPEA_BASE_VIRT			(IXP4XX_PERIPHERAL_BASE_VIRT + 0x06000)
#define IXP4XX_NPEB_BASE_VIRT			(IXP4XX_PERIPHERAL_BASE_VIRT + 0x07000)
#define IXP4XX_NPEC_BASE_VIRT			(IXP4XX_PERIPHERAL_BASE_VIRT + 0x08000)
#define IXP4XX_EthB0_BASE_VIRT			(IXP4XX_PERIPHERAL_BASE_VIRT + 0x09000)
#define IXP4XX_EthC_BASE_VIRT			(IXP4XX_PERIPHERAL_BASE_VIRT + 0x0A000)
#define IXP4XX_USB_BASE_VIRT			(IXP4XX_PERIPHERAL_BASE_VIRT + 0x0B000)

/* 46x only */
#define IXP4XX_EthA_BASE_VIRT			(IXP4XX_PERIPHERAL_BASE_VIRT + 0x0C000)
#define IXP4XX_EthB1_BASE_VIRT			(IXP4XX_PERIPHERAL_BASE_VIRT + 0x0D000)
#define IXP4XX_EthB2_BASE_VIRT			(IXP4XX_PERIPHERAL_BASE_VIRT + 0x0E000)
#define IXP4XX_EthB3_BASE_VIRT			(IXP4XX_PERIPHERAL_BASE_VIRT + 0x0F000)
#define IXP4XX_TIMESYNC_BASE_VIRT		(IXP4XX_PERIPHERAL_BASE_VIRT + 0x10000)
#define IXP4XX_I2C_BASE_VIRT			(IXP4XX_PERIPHERAL_BASE_VIRT + 0x11000)
#define IXP4XX_SSP_BASE_VIRT			(IXP4XX_PERIPHERAL_BASE_VIRT + 0x12000)

/* Clock Speed Definitions */
#define IXP4XX_PERIPHERAL_BUS_CLOCK		(66)	/* 66Mhz APB BUS   */

/**************************************************
 * UARTs
 *	 IXP4XX_UART1_BASE , IXP4XX_UART2_BASE.
 *************************************************/
#define IXP4XX_CONSOLE_UART_BASE_VIRT	IXP4XX_UART1_BASE_VIRT
#define IXP4XX_CONSOLE_UART_BASE_PHYS	IXP4XX_UART1_BASE_PHYS

#undef	UART_NO_RX_INTERRUPT

#define IXP4XX_UART_XTAL				14745600

/**************************************************
 * Interrupt Controller
 *************************************************/
#define IXP4XX_INTC_ST_OFFSET			0x00 /* Interrupt Status */
#define IXP4XX_INTC_EN_OFFSET			0x04 /* Interrupt Enable */
#define IXP4XX_INTC_SEL_OFFSET			0x08 /* Interrupt IRQ/FIQ Select */
#define IXP4XX_INTC_IST_OFFSET			0x0C /* IRQ Status */
#define IXP4XX_INTC_FST_OFFSET			0x10 /* FIQ Status */
#define IXP4XX_INTC_PRI_OFFSET			0x14 /* Interrupt Priority */
#define IXP4XX_INTC_IPRI_OFFSET			0x18 /* IRQ Highest Pri Int */
#define IXP4XX_INTC_FPRI_OFFSET			0x1C /* FIQ Highest Pri Int */

/* 46x only (upper 32 IRQs) */ 
#define IXP4XX_INTC_ST2_OFFSET			0x20 /* Interrupt Status 2 */
#define IXP4XX_INTC_EN2_OFFSET			0x24 /* Interrupt Enable 2 */
#define IXP4XX_INTC_SEL2_OFFSET			0x28 /* Interrupt IRQ/FIQ Select 2 */
#define IXP4XX_INTC_IST2_OFFSET			0x2C /* IRQ Status 2 */
#define IXP4XX_INTC_FST2_OFFSET			0x30 /* FIQ Status 2 */
#define IXP4XX_INTC_EEN_OFFSET			0x34 /* Error High Pri Enable */

#define IXP4XX_INTC_REG(x) ((volatile u32 *)(IXP4XX_INTC_BASE_VIRT+(x)))

#define IXP4XX_INTC_ST					IXP4XX_INTC_REG(IXP4XX_INTC_ST_OFFSET)
#define IXP4XX_INTC_EN					IXP4XX_INTC_REG(IXP4XX_INTC_EN_OFFSET)
#define IXP4XX_INTC_SEL					IXP4XX_INTC_REG(IXP4XX_INTC_SEL_OFFSET)
#define IXP4XX_INTC_IST					IXP4XX_INTC_REG(IXP4XX_INTC_IST_OFFSET)
#define IXP4XX_INTC_FST					IXP4XX_INTC_REG(IXP4XX_INTC_FST_OFFSET)
#define IXP4XX_INTC_PRI					IXP4XX_INTC_REG(IXP4XX_INTC_PRI_OFFSET)
#define IXP4XX_INTC_IPRI				IXP4XX_INTC_REG(IXP4XX_INTC_IPRI_OFFSET)
#define IXP4XX_INTC_FPRI				IXP4XX_INTC_REG(IXP4XX_INTC_FPRI_OFFSET)

/* 46x only (upper 32 IRQs) */
#define IXP4XX_INTC_ST2					IXP4XX_INTC_REG(IXP4XX_INTC_ST2_OFFSET)
#define IXP4XX_INTC_EN2					IXP4XX_INTC_REG(IXP4XX_INTC_EN2_OFFSET)
#define IXP4XX_INTC_SEL2				IXP4XX_INTC_REG(IXP4XX_INTC_SEL2_OFFSET)
#define IXP4XX_INTC_IST2				IXP4XX_INTC_REG(IXP4XX_INTC_IST2_OFFSET)
#define IXP4XX_INTC_FIST2				IXP4XX_INTC_REG(IXP4XX_INTC_FST2_OFFSET)
#define IXP4XX_INTC_EEN					IXP4XX_INTC_REG(IXP4XX_INTC_EEN_OFFSET)

/**************************************************
 * GPIO
 *************************************************/
#define IXP4XX_GPIO_GPOUTR_OFFSET		0x00
#define IXP4XX_GPIO_GPOER_OFFSET		0x04
#define IXP4XX_GPIO_GPINR_OFFSET		0x08
#define IXP4XX_GPIO_GPISR_OFFSET		0x0C
#define IXP4XX_GPIO_GPIT1R_OFFSET		0x10
#define IXP4XX_GPIO_GPIT2R_OFFSET		0x14
#define IXP4XX_GPIO_GPCLKR_OFFSET		0x18
#define IXP4XX_GPIO_GPDBSELR_OFFSET		0x1C

#define IXP4XX_GPIO_REG(x) ((volatile u32 *)(IXP4XX_GPIO_BASE_VIRT+(x)))

#define IXP4XX_GPIO_GPOUTR				IXP4XX_GPIO_REG(IXP4XX_GPIO_GPOUTR_OFFSET)
#define IXP4XX_GPIO_GPOER				IXP4XX_GPIO_REG(IXP4XX_GPIO_GPOER_OFFSET)
#define IXP4XX_GPIO_GPINR				IXP4XX_GPIO_REG(IXP4XX_GPIO_GPINR_OFFSET)
#define IXP4XX_GPIO_GPISR				IXP4XX_GPIO_REG(IXP4XX_GPIO_GPISR_OFFSET)
#define IXP4XX_GPIO_GPIT1R				IXP4XX_GPIO_REG(IXP4XX_GPIO_GPIT1R_OFFSET)
#define IXP4XX_GPIO_GPIT2R				IXP4XX_GPIO_REG(IXP4XX_GPIO_GPIT2R_OFFSET)
#define IXP4XX_GPIO_GPCLKR				IXP4XX_GPIO_REG(IXP4XX_GPIO_GPCLKR_OFFSET)
#define IXP4XX_GPIO_GPDBSELR			IXP4XX_GPIO_REG(IXP4XX_GPIO_GPDBSELR_OFFSET)

/**************************************************
 * Timer
 **************************************************/
#define IXP4XX_OSTS_OFFSET				0x00  /* Continious TimeStamp */
#define IXP4XX_OST1_OFFSET				0x04  /* Timer 1 Timestamp */
#define IXP4XX_OSRT1_OFFSET				0x08  /* Timer 1 Reload */
#define IXP4XX_OST2_OFFSET				0x0C  /* Timer 2 Timestamp */
#define IXP4XX_OSRT2_OFFSET				0x10  /* Timer 2 Reload */
#define IXP4XX_OSWT_OFFSET				0x14  /* Watchdog Timer */
#define IXP4XX_OSWE_OFFSET				0x18  /* Watchdog Enable */
#define IXP4XX_OSWK_OFFSET				0x1C  /* Watchdog Key */
#define IXP4XX_OSST_OFFSET				0x20  /* Timer Status */

/* 46x only */ 
#define IXP4XX_OSTSC_OFFSET				0x24  /* Timestamp Compare */
#define IXP4XX_OSTSCFG_OFFSET			0x28  /* Timestamp Configuration */
#define IXP4XX_OSTSPRE_OFFSET			0x2C  /* Timestamp Prescale */
#define IXP4XX_OSCFG1_OFFSET			0x30  /* General Purpose Timer 1 Configuration */
#define IXP4XX_OSPRE1_OFFSET			0x34  /* General Purpose Timer 1 Prescale */
#define IXP4XX_OSCFG2_OFFSET			0x38  /* General Purpose Timer 2 Configuration */
#define IXP4XX_OSPRE2_OFFSET			0x3C  /* General Purpose Timer 2 Prescale */
#define IXP4XX_OSTSPRES_OFFSET			0x40  /* Timestamp Prescale Status */
#define IXP4XX_OSPRE1S_OFFSET			0x44  /* Timer 1 Prescale Status */
#define IXP4XX_OSPRE2S_OFFSET			0x48  /* Timer 2 Prescale Status */

#define IXP4XX_TIMER_REG(x) ((volatile u32 *)(IXP4XX_TIMER_BASE_VIRT+(x)))

#define IXP4XX_OSTS						IXP4XX_TIMER_REG(IXP4XX_OSTS_OFFSET)
#define IXP4XX_OST1						IXP4XX_TIMER_REG(IXP4XX_OST1_OFFSET)
#define IXP4XX_OSRT1					IXP4XX_TIMER_REG(IXP4XX_OSRT1_OFFSET)
#define IXP4XX_OST2						IXP4XX_TIMER_REG(IXP4XX_OST2_OFFSET)
#define IXP4XX_OSRT2					IXP4XX_TIMER_REG(IXP4XX_OSRT2_OFFSET)
#define IXP4XX_OSWT						IXP4XX_TIMER_REG(IXP4XX_OSWT_OFFSET)
#define IXP4XX_OSWE						IXP4XX_TIMER_REG(IXP4XX_OSWE_OFFSET)
#define IXP4XX_OSWK						IXP4XX_TIMER_REG(IXP4XX_OSWK_OFFSET)
#define IXP4XX_OSST						IXP4XX_TIMER_REG(IXP4XX_OSST_OFFSET)

/* 46x only */
#define IXP4XX_OSTSC					IXP4XX_TIMER_REG(IXP4XX_OSTSC_OFFSET)
#define IXP4XX_OSTSCFG					IXP4XX_TIMER_REG(IXP4XX_OSTSCFG_OFFSET)
#define IXP4XX_OSTSPRE					IXP4XX_TIMER_REG(IXP4XX_OSTSPRE_OFFSET)
#define IXP4XX_OSCFG1					IXP4XX_TIMER_REG(IXP4XX_OSCFG1_OFFSET)
#define IXP4XX_OSPRE1					IXP4XX_TIMER_REG(IXP4XX_OSPRE1_OFFSET)
#define IXP4XX_OSCFG2					IXP4XX_TIMER_REG(IXP4XX_OSCFG2_OFFSET)
#define IXP4XX_OSPRE2					IXP4XX_TIMER_REG(IXP4XX_OSPRE2_OFFSET)
#define IXP4XX_OSTSPRES					IXP4XX_TIMER_REG(IXP4XX_OSTSPRES_OFFSET)
#define IXP4XX_OSPRE1S					IXP4XX_TIMER_REG(IXP4XX_OSPRE1S_OFFSET)
#define IXP4XX_OSPRE2S					IXP4XX_TIMER_REG(IXP4XX_OSPRE2S_OFFSET)

/* General Purpose Timer */
#define IXP4XX_OST_ENABLE			   BIT(0)
#define IXP4XX_OST_ONE_SHOT			   BIT(1)
#define IXP4XX_OST_RELOAD_MASK		   (0x3)
#define IXP4XX_OST_DISABLED			   (0x0)
/* WDT - Watchdog */
#define IXP4XX_WDT_RESET_ENABLE		   BIT(0)
#define IXP4XX_WDT_IRQ_ENABLE		   BIT(1)
#define IXP4XX_WDT_COUNT_ENABLE		   BIT(2)
#define IXP4XX_WDT_KEY				   (0x0000482E)
/* OSST - Timer Status */
#define IXP4XX_OSST_TIMER_1_PEND	   BIT(0)
#define IXP4XX_OSST_TIMER_2_PEND	   BIT(1)
#define IXP4XX_OSST_TIMER_TS_PEND	   BIT(2)
#define IXP4XX_OSST_TIMER_WDOG_PEND	   BIT(3)
#define IXP4XX_OSST_TIMER_WARM_RESET   BIT(4)
/* OSTSCFG - Timestamp configuration */
#define IXP4XX_OSTSCFG_ENABLE		   BIT(0)
#define IXP4XX_OSTSCFG_COUNT_ENABLE	   BIT(1)
/* OSCGF - General Purpose Timer Configuration */
#define IXP4XX_OSCFG_RELOAD_BIT0	   BIT(0)
#define IXP4XX_OSCFG_RELOAD_BIT1	   BIT(1)
#define IXP4XX_OSCFG_SCALE_ENABLE	   BIT(2)
#define IXP4XX_OSCFG_COUNT_ENABLE	   BIT(3)

/**************************************************
 * I2C - 46x only
 **************************************************/
#define IXP4XX_I2C_ICR_OFFSET			0x00  /* Control */
#define IXP4XX_I2C_ISR_OFFSET			0x04  /* Status */
#define IXP4XX_I2C_ISAR_OFFSET			0x08  /* Slave Address */
#define IXP4XX_I2C_IDBR_OFFSET			0x0C  /* Data Buffer */
#define IXP4XX_I2C_ICCR_OFFSET			0x10  /* Clock Count */
#define IXP4XX_I2C_IBMR_OFFSET			0x14  /* Bus Monitor */

#define IXP4XX_I2C_REG(x) ((volatile u32 *)(IXP4XX_I2C_BASE_VIRT+(x)))

#define IXP4XX_I2C_ICR					IXP4XX_I2C_REG(IXP4XX_I2C_ICR_OFFSET)
#define IXP4XX_I2C_ISR					IXP4XX_I2C_REG(IXP4XX_I2C_ISR_OFFSET)
#define IXP4XX_I2C_ISAR					IXP4XX_I2C_REG(IXP4XX_I2C_ISAR_OFFSET)
#define IXP4XX_I2C_IDBR					IXP4XX_I2C_REG(IXP4XX_I2C_IDBR_OFFSET)
#define IXP4XX_I2C_ICCR					IXP4XX_I2C_REG(IXP4XX_I2C_ICCR_OFFSET)
#define IXP4XX_I2C_IBMR					IXP4XX_I2C_REG(IXP4XX_I2C_IBMR_OFFSET)

#endif

