/*
 * linux/drivers/video/versatile-clcd/versatile-clcd.c
 *
 * Philippe Robin - David A Rusling
 *
 * Copyright (C) 2002,2003 ARM Limited
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file COPYING in the main directory of this archive
 * for more details.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/tty.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/fb.h>
#include <linux/init.h>
#include <linux/bootmem.h>

/* frame buffer includes */
#include <video/fbcon.h>
#include <video/fbcon-cfb8.h>
#include <video/fbcon-cfb16.h>
#include <video/fbcon-cfb24.h>
#include <video/fbcon-cfb32.h>

// CLCD definitions
#include "clcd.h"

#define DEBUG 0

/* We need 20msec delays during power-up/down */
#define  AFTER(msec) mdelay(msec)

// sizes of things
typedef unsigned char BYTE;	/*  8 bits */
typedef unsigned short WORD;	/* 16 bits */
typedef unsigned long DWORD;	/* 32 bits */

struct clcd_reg_area 
{
    unsigned int pmem ;
    unsigned int mem ;
    unsigned int len ;
    unsigned int io ;
} ;

/*
 *  The hardware specific data in the following structure 
 *  uniquely defines a video mode.
 */
struct clcdfb_par
{
    int width;
    int height;
    int depth;
    int pixclock;

    /* device specific information */
    struct clcd_reg_area clcd_reg_area[3] ;
};

#define NR_PALETTE 256
typedef struct
{
    unsigned char T;
    unsigned char R;
    unsigned char G;
    unsigned char B;
} colour_t;

/*
 * Data structure describing each frame buffer device found.
 */
struct clcd_fb
{
    struct clcd_fb *next;

    //  Use the generic frame buffer operations (fbgen_*).
    struct fb_info_gen fb_info_gen;

    // Device specific information - displayed screen
    struct clcdfb_par par;

    // Real CLCD device
    struct clcd_reg_area clcd_reg_area[2];

    // Colour maps
    union
    {
#ifdef FBCON_HAS_CFB16
	u16 cmap16[16];
#endif
#ifdef FBCON_HAS_CFB24
	u32 cmap24[16];
#endif
#ifdef FBCON_HAS_CFB32
	u32 cmap32[16];
#endif
    }
    cmap;

    // Palette
    colour_t palette[NR_PALETTE];

    // Which display
    struct display display;
};

// list of found CLCD vx1 devices
static struct clcd_fb *clcd_fbs = NULL;
static unsigned int clcd_framebuffer_base;

// The current 'new' frame buffer 
static struct clcd_fb new_fb;

static int PPL;
static int LPP;

/* CLCD - HW specific helper routines */
static int hw_init(struct clcdfb_par *par);
static int hw_set_par(struct clcdfb_par *par);
static void hw_set_colour(struct clcdfb_par *par, unsigned char regno,
			  unsigned char r, unsigned char g, unsigned char b);
static int hw_blank(struct clcdfb_par *par, int blank_mode) ;

/* This is limited to 16 characters when displayed by X startup */
static char clcd_name[16] = "CLCD FB";

/* --------------------------------------------------------------------------------- */
/* CLCD ENVIRONMENT SUPPORT ROUTINES                                                 */
/* --------------------------------------------------------------------------------- */

/* ------------------- chipset specific functions -------------------------- */


static void clcd_detect(void)
{
    /*
     *  This function should detect the current video mode settings and store
     *  it as the default video mode
     */
}

static int clcd_encode_fix(struct fb_fix_screeninfo *fix, const void *par,
			   struct fb_info_gen *info)
{
    struct clcd_fb *fb = (struct clcd_fb *)info->info.par;
    struct clcdfb_par *p = (struct clcdfb_par *)par;

#if 0
    printk(KERN_INFO __FILE__ ": clcd_encode_fix(fix=0x%x, par=0x%x) called\n",
	   (unsigned int)fix, (unsigned int)par);
#endif

    /*
     *  This function should fill in the 'fix' structure based on the values
     *  in the `par' structure.
     */

    // .....fixed screen area
    strcpy(fix->id, clcd_name);
    fix->smem_start = (unsigned long)fb->clcd_reg_area[0].pmem;
    fix->smem_len = fb->clcd_reg_area[0].len;
    fix->type = FB_TYPE_PACKED_PIXELS;

    fix->type_aux = 0;
    fix->visual = p->depth <= 8 ? FB_VISUAL_PSEUDOCOLOR : FB_VISUAL_TRUECOLOR;
    fix->xpanstep = 0;
    fix->ypanstep = 0;
    fix->ywrapstep = 0;
    fix->line_length = (fb->par.width * fb->par.depth) / 8;
    fix->mmio_start = (unsigned long)fb->clcd_reg_area[1].pmem;
    fix->mmio_len = fb->clcd_reg_area[1].len;
    fix->accel = FB_ACCEL_NONE;

    return 0;
}

static int clcd_decode_var(const struct fb_var_screeninfo *var, void *par,
			   struct fb_info_gen *info)
{
    struct clcd_fb *fb = (struct clcd_fb *)info->info.par;
    struct clcdfb_par *p = (struct clcdfb_par *)par;

#if 0
    printk(KERN_INFO __FILE__ ": clcd_decode_var(var=0x%x, par=0x%x) called\n",
	   (unsigned int)var, (unsigned int)par);
    printk("depth = %d  width = %d height = %d pixclock = %d\n", 
	   BITS_PER_PIXEL, (PPL+1)*16, LPP+1, PIXEL_CLOCK_MHZ);
#endif

    /*
     *  Get the video params out of 'var'. If a value doesn't fit, round it up,
     *  if it's too big, return -EINVAL.
     *
     *  Suggestion: Round up in the following order: bits_per_pixel, xres,
     *  yres, xres_virtual, yres_virtual, xoffset, yoffset, grayscale,
     *  bitfields, horizontal timing, vertical timing.
     */

    p->depth = (var->bits_per_pixel + 7) & ~7 ;
    p->width = (var->xres + 7) & ~7 ;
    p->height = var->yres ;
    p->pixclock = var->pixclock ;

    /* Fail if these do not match our configured mode */
    if (p->depth    != BITS_PER_PIXEL ||
	p->width    != (PPL+1)*16 ||
	p->height   != LPP+1 ||
	p->pixclock != PIXEL_CLOCK_MHZ)
      {
	  p->depth = BITS_PER_PIXEL;
	  p->width = (PPL+1)*16;
	  p->height =  LPP+1;
	  p->pixclock =  PIXEL_CLOCK_MHZ;
	  return -EINVAL;
      }

    p->clcd_reg_area[0] = fb->clcd_reg_area[0] ;
    p->clcd_reg_area[1] = fb->clcd_reg_area[1] ;
    return 0 ;
}

static int clcd_encode_var(struct fb_var_screeninfo *var, const void *par,
			   struct fb_info_gen *info)
{
    struct clcdfb_par *p = (struct clcdfb_par *)par;

#if 0
    printk(KERN_INFO __FILE__ ": clcd_encode_var(var=0x%x, par=0x%x) called\n",
	   (unsigned int)var, (unsigned int)par);
    printk("clcd_encode: depth = %d  width = %d height = %d pixclock = %d\n", 
	   p->depth, p->width, p->height, p->pixclock);
#endif

    /*
     *  Fill the 'var' structure based on the values in 'par' and maybe other
     *  values read out of the hardware.
     */

    // variable screen area
    memset(var, 0, sizeof(struct fb_var_screeninfo));

    var->xres = var->xres_virtual = p->width;
    var->yres = var->yres_virtual = p->height;
    var->bits_per_pixel = p->depth;
    var->nonstd = 0;
    var->activate = FB_ACTIVATE_NOW;
    var->height = -1;
    var->width = -1;
    var->accel_flags = 0;
    var->pixclock = p->pixclock;
    var->sync = FB_SYNC_COMP_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT;

    // dunno about these...
    var->right_margin = 0;
    var->hsync_len = 0;
    var->left_margin = 0;
    var->lower_margin = 0;
    var->vsync_len = 0;
    var->upper_margin = 0;

    var->red.msb_right = 0;
    var->green.msb_right = 0;
    var->blue.msb_right = 0;

    switch (p->depth)
    {
	case 8:
	    var->red.length = var->green.length = var->blue.length = 8;
	    var->red.offset = var->green.offset = var->blue.offset = 0;
	    break;
	case 16:
	    var->red.length = var->green.length = var->blue.length = 5;
	    var->red.offset = 0;
	    var->green.offset = 5;
	    var->blue.offset = 10;
	    break;
	case 32:
	    var->transp.offset = 24;
	    var->red.offset = 16;
	    var->green.offset = 8;
	    var->red.length = var->green.length = var->blue.length =
		var->transp.length = 8;
	    break;
	case 24:
	    var->blue.offset = 16;
	    var->green.offset = 8;
	    var->red.length = var->green.length = var->blue.length = 8;
	    break;
    }

    return 0;
}

static void clcd_get_par(void *par, struct fb_info_gen *info)
{
    struct clcd_fb *fb = (struct clcd_fb *)info->info.par;
    struct clcdfb_par *p = (struct clcdfb_par *)par;

#if 0
    printk(KERN_INFO __FILE__ ": clcd_get_par(par=0x%x, info=0x%x) called\n",
	   (unsigned int)par, (unsigned int)info);
#endif

    // copy it over from the current one
    *p = fb->par;
}

static void clcd_set_par(const void *par, struct fb_info_gen *info)
{
    struct clcd_fb *fb = (struct clcd_fb *)info->info.par;
    struct clcdfb_par *p = (struct clcdfb_par *)par;

#if 0
    printk(KERN_INFO __FILE__ ": clcd_set_par(par=0x%x, info=0x%x) called\n",
	   (unsigned int)par, (unsigned int)info);
#endif

    /*
     *  Have things changed?
     */
    if (
	(fb->par.width != p->width) ||
	(fb->par.height != p->height) ||
	(fb->par.depth != p->depth) ||
	(fb->par.pixclock != p->pixclock)
	) {
      hw_set_par(p) ;
    }

    // save the current hardware set up
    fb->par = *p;
}

static int clcd_getcolreg(unsigned regno, unsigned *red, unsigned *green,
			  unsigned *blue, unsigned *transp,
			  struct fb_info *info)
{
    struct clcd_fb *fb = (struct clcd_fb *)info->par;

#if 0
    printk(KERN_INFO __FILE__ ": clcd_getcolreg(%d) called\n", regno);
#endif

    /*
     *  Read a single color register and split it into colors/transparent.
     *  The return values must have a 16 bit magnitude.
     *  Return != 0 for invalid regno.
     */

    if (regno < 256)
    {
	*red = fb->palette[regno].R << 8 | fb->palette[regno].R;
	*green = fb->palette[regno].G << 8 | fb->palette[regno].G;
	*blue = fb->palette[regno].B << 8 | fb->palette[regno].B;
	*transp = fb->palette[regno].T << 8 | fb->palette[regno].T;
    }
    return regno > 255;
}

static int clcd_setcolreg(unsigned regno, unsigned red, unsigned green,
			  unsigned blue, unsigned transp, struct fb_info *info)
{
    struct clcd_fb *fb = (struct clcd_fb *)info->par;

#if 0
    printk(KERN_INFO __FILE__ ": clcd_setcolreg(%d) called\n", regno);
#endif
    /*
     *  Set a single color register. The values supplied have a 16 bit
     *  magnitude.
     *  Return != 0 for invalid regno.
     */

    if (regno < 16)
    {
	switch (fb->par.depth)
	{
#ifdef FBCON_HAS_CFB8
	    case 8:
		break;
#endif
#ifdef FBCON_HAS_CFB16
	    case 16:
		  /* Want 5 bits of each */
		fb->cmap.cmap16[regno] =
		    (((u32) red   & 0xf800) >> 11)|
		    (((u32) green & 0xf800) >> 6) |
		    (((u32) blue  & 0xf800) >> 1);
		break;
#endif
#ifdef FBCON_HAS_CFB24
	    case 24:
		fb->cmap.cmap24[regno] =
		    (((u32) blue & 0xff00) << 8) |
		    ((u32) green & 0xff00) | (((u32) red & 0xff00) >> 8);
		break;
#endif
#ifdef FBCON_HAS_CFB32
	    case 32:
		fb->cmap.cmap32[regno] =
		    (((u32) transp & 0xff00) << 16) |
		    (((u32) red & 0xff00) << 8) |
		    (((u32) green & 0xff00)) | (((u32) blue & 0xff00) >> 8);
		break;
#endif
	    default:
		printk(KERN_INFO __FILE__ "bad depth %u\n", fb->par.depth);
		break;
	}
    }
    if (regno < 256)
    {
	fb->palette[regno].R = red >> 8;
	fb->palette[regno].G = green >> 8;
	fb->palette[regno].B = blue >> 8;
	fb->palette[regno].T = transp >> 8;
	if (fb->par.depth == 8)
	{
	    hw_set_colour(&fb->par, regno, red, green, blue);
	}
    }
    return regno > 255;
}

static int clcd_pan_display(const struct fb_var_screeninfo *var,
			    struct fb_info_gen *info)
{
  /* No hardware panning */
  return -EINVAL;
}

static int clcd_blank(int blank_mode, struct fb_info_gen *info)
{
    struct clcd_fb *fb = (struct clcd_fb *)info->info.par ;

    hw_blank(&fb->par, blank_mode) ;
    return 0 ;
}

static void clcd_set_disp(const void *par, struct display *disp,
			  struct fb_info_gen *info)
{
    struct clcd_fb *fb = (struct clcd_fb *)info->info.par;
    struct clcdfb_par *p = (struct clcdfb_par *)par;

#if DEBUG
    printk(KERN_INFO __FILE__ ": clcd_set_disp(par=0x%x) called\n",
	   (unsigned int)par);
#endif

    /*
     *  Fill in a pointer with the virtual address of the frame buffer.
     *  Fill in a pointer to appropriate low level text console operations (and
     *  optionally a pointer to help data) for the video mode `par' of your
     *  video hardware. These can be generic software routines, or hardware
     *  accelerated routines specifically tailored for your hardware.
     *  If you don't have any appropriate operations, you must fill in a
     *  pointer to dummy operations, and there will be no text output.
     */
    disp->screen_base = (void *)fb->clcd_reg_area[0].mem;
    switch (p->depth)
    {
#ifdef FBCON_HAS_CFB8
	case 8:
	    disp->dispsw = &fbcon_cfb8;
	    disp->dispsw_data = NULL;
	    break;
#endif
#ifdef FBCON_HAS_CFB16
	case 16:
	    disp->dispsw = &fbcon_cfb16;
	    disp->dispsw_data = fb->cmap.cmap16;	/* console palette */
	    break;
#endif
#ifdef FBCON_HAS_CFB24
	case 24:
	    disp->dispsw = &fbcon_cfb24;
	    disp->dispsw_data = fb->cmap.cmap24;	/* console palette */
	    break;
#endif
#ifdef FBCON_HAS_CFB32
	case 32:
	    disp->dispsw = &fbcon_cfb32;
	    disp->dispsw_data = fb->cmap.cmap32;	/* console palette */
	    break;
#endif
	default:
	    disp->dispsw = &fbcon_dummy;
	    break;
    }
}


/*
 *  Get the Fixed Part of the Display
 */

static int clcdfb_get_fix(struct fb_fix_screeninfo *fix, int con, struct fb_info *fb)
{
    struct fb_info_gen *info = (struct fb_info_gen*)fb;
    char par[info->parsize];

#if DEBUG
    printk(KERN_INFO __FILE__ ": clcdfb_get_fix(con = %d).\n", con);
#endif
    if (con == -1)
        clcd_get_par((void *)par, info);
    else
	clcd_decode_var(&fb_display[con].var, (void *)par, info);

    memset(fix, 0, sizeof(struct fb_fix_screeninfo));
    clcd_encode_fix(fix, (void *)par, info);
    return 0;
}

/*
 *  Get the User Defined Part of the Display
 */

static int clcd_get_var(struct fb_var_screeninfo *var, int con, struct fb_info *fb)
{
    struct fb_info_gen *info = (struct fb_info_gen *)fb;
    char par[info->parsize];

#if DEBUG
    printk(KERN_INFO __FILE__ ": clcdfb_get_var(con = %d).\n", con);
#endif
    if (con == -1) {
        clcd_get_par((void *)par, info);
	clcd_encode_var(var, (void *)par, info);
    } else
	*var = fb_display[con].var;

    return 0;
}

/*
 *  Set the User Defined Part of the Display
 */

static int clcd_set_var(struct fb_var_screeninfo *var, int con, struct fb_info *fb)
{
    struct fb_info_gen *info = (struct fb_info_gen *)fb;
    char par[info->parsize];
    struct display *display;
    int err;
    int oldxres, oldyres, oldbpp, oldxres_virtual, oldyres_virtual, oldyoffset;
    struct fb_bitfield oldred, oldgreen, oldblue;
    int activate = var->activate;

#if DEBUG
    printk(KERN_INFO __FILE__ ": clcdfb_set_var(con = %d).\n", con);
#endif
    if (con >= 0)
	display = &fb_display[con];
    else
	display = info->info.disp;	/* used during initialization */

    if ((err = clcd_decode_var(var, &par, info)))
	return err;

    clcd_encode_var(var, &par, info);

    if ((activate & FB_ACTIVATE_MASK) == FB_ACTIVATE_NOW) {
	oldxres = display->var.xres;
	oldyres = display->var.yres;
	oldxres_virtual = display->var.xres_virtual;
	oldyres_virtual = display->var.yres_virtual;
	oldbpp = display->var.bits_per_pixel;
	oldyoffset = display->var.yoffset;

	display->var = *var;

	if (oldxres != var->xres || oldyres != var->yres ||
	    oldxres_virtual != var->xres_virtual ||
	    oldyres_virtual != var->yres_virtual ||
	    oldbpp != var->bits_per_pixel ||
	    (!(memcmp(&oldred, &(var->red), sizeof(struct fb_bitfield)))) || 
	    (!(memcmp(&oldgreen, &(var->green), sizeof(struct fb_bitfield)))) ||
	    (!(memcmp(&oldblue, &(var->blue), sizeof(struct fb_bitfield)))) ||
	    oldyoffset != var->yoffset) {

	    display->screen_base = (char *)clcd_framebuffer_base;
	    fbgen_set_disp(con, info);
	    if (fb->changevar)
		(*fb->changevar)(con);
	    if ((err = fb_alloc_cmap(&display->cmap, 0, 0)))
		return err;
	    fbgen_install_cmap(con, info);
	}
    }
    var->activate = 0;

    return 0;
}

/**
 * fbgen_get_cmap - get the colormap
 *
 * Returns negative errno on error, or zero for success.
 */

int clcd_get_cmap(struct fb_cmap *cmap, int kspc, int con,
		   struct fb_info *info)
{
#if DEBUG
    printk(KERN_INFO __FILE__ ": clcdfb_get_cmap(con = %d).\n", con);
#endif
    if (!info->display_fg || con == info->display_fg->vc_num) /* current console? */
	return fb_get_cmap(cmap, kspc, clcd_getcolreg, info);
    else
	if (fb_display[con].cmap.len)	/* non default colormap ? */
	    fb_copy_cmap(&fb_display[con].cmap, cmap, kspc ? 0 : 2);
	else {
	    int size = fb_display[con].var.bits_per_pixel == 16 ? 64 : 256;
	    fb_copy_cmap(fb_default_cmap(size), cmap, kspc ? 0 : 2);
	}
    return 0;
}

/*
 *  Set the Colormap
 */
static int clcd_set_cmap(struct fb_cmap *cmap, int kspc, int con,
			  struct fb_info *info)
{
    int err;
    struct display *disp;

#if DEBUG
    printk(KERN_INFO __FILE__ ": clcdfb_set_cmap(con = %d).\n", con);
#endif
    if (con >= 0)
    	disp = &fb_display[con];
    else
        disp = info->disp;
    if (!disp->cmap.len) {	/* no colormap allocated? */
	int size = disp->var.bits_per_pixel == 16 ? 32 : 256;
	if ((err = fb_alloc_cmap(&disp->cmap, size, 0)))
	    return err;
    }
    if (!info->display_fg || con == info->display_fg->vc_num)	/* current console? */
	return fb_set_cmap(cmap, kspc, clcd_setcolreg, info);
    else
	fb_copy_cmap(cmap, &disp->cmap, kspc ? 0 : 1);

    return 0;
}


struct fbgen_hwswitch clcd_switch = {
    clcd_detect,
    clcd_encode_fix,
    clcd_decode_var,
    clcd_encode_var,
    clcd_get_par,
    clcd_set_par,
    clcd_getcolreg,
    clcd_setcolreg,
    clcd_pan_display,
    clcd_blank,
    clcd_set_disp
};

/*
 *  In most cases the `generic' routines (fbgen_*) should be satisfactory.
 *  However, you're free to fill in your own replacements.
 */

static struct fb_ops clcdfb_ops = {
    .owner		= THIS_MODULE,
    .fb_get_fix		= clcdfb_get_fix,
    .fb_get_var		= clcd_get_var,
    .fb_set_var		= clcd_set_var,
    .fb_get_cmap	= clcd_get_cmap,
    .fb_set_cmap	= clcd_set_cmap,
    .fb_pan_display	= fbgen_pan_display,
};


    static int __devinit 
clcd_found_device(int irq,
		  unsigned int mem_base_0, unsigned long mem_len_0, unsigned int pmem_base_0, 
		  unsigned int mem_base_1, unsigned long mem_len_1, unsigned int pmem_base_1)
{
    int clcdid;

    /*
     * tell the world what we have found 
     */
    printk(KERN_INFO
	   "Versatile Colour LCD panel found at membase (0x%X/%ldK), IRQ %d\n",
	   (unsigned int)pmem_base_1, mem_len_1 >> 10, irq) ;

    /* Say what sort of panel we are assuming */
    printk(KERN_INFO "Using settings for " PANEL_NAME " display\n");

    memset(&new_fb, 0, sizeof(struct clcd_fb));

#if 1
    printk(KERN_INFO __FILE__ ": fb @ 0x%x\n", (unsigned int)&new_fb);
    printk(KERN_INFO __FILE__ ": par @ 0x%x\n", (unsigned int)&new_fb.par);
#endif

    /*
     * fill it out with.device information
     */
    new_fb.clcd_reg_area[0].pmem = (unsigned int)pmem_base_0;
    new_fb.clcd_reg_area[0].mem = (unsigned int)mem_base_0;
    new_fb.clcd_reg_area[0].len = (unsigned int)mem_len_0;
    new_fb.clcd_reg_area[0].io = 0 ;
    new_fb.clcd_reg_area[1].pmem = (unsigned int)pmem_base_1;
    new_fb.clcd_reg_area[1].mem = (unsigned int)mem_base_1;
    new_fb.clcd_reg_area[1].len = (unsigned int)mem_len_1;
    new_fb.clcd_reg_area[1].io = 0 ;

    /*
     * ...fb_info_gen
     */
    new_fb.fb_info_gen.info.par = (void *)&new_fb;
    strcpy(new_fb.fb_info_gen.info.modename, clcd_name);
    new_fb.fb_info_gen.parsize = sizeof(struct clcdfb_par);

    new_fb.fb_info_gen.fbhw = &clcd_switch;
    new_fb.fb_info_gen.info.changevar = NULL;
    new_fb.fb_info_gen.info.node = -1;
    new_fb.fb_info_gen.info.fbops = &clcdfb_ops;
    /*
     * new_fb.display.scrollmode = SCROLL_YREDRAW;
     */
    new_fb.fb_info_gen.info.disp = &new_fb.display;
    new_fb.fb_info_gen.info.switch_con = &fbgen_switch;
    new_fb.fb_info_gen.info.updatevar = &fbgen_update_var;
    new_fb.fb_info_gen.info.blank = &fbgen_blank;
    new_fb.fb_info_gen.info.flags = FBINFO_FLAG_DEFAULT;

    /*
     * check what resolution we have 
     */
    clcdid = *(unsigned int *)(IO_ADDRESS(VERSATILE_SYS_BASE) + VERSATILE_SYS_CLCD_OFFSET);
    clcdid = (clcdid >> 8) & 0xF;
    if (clcdid == VERSATILE_3x8_PANEL) {
	PPL = (320/16) - 1;
	LPP = 240 - 1;
    } else {
	PPL = (640/16) - 1;
	LPP = 480 - 1;
    }
    /*
     *...par
     */
    new_fb.par.width = (PPL+1)*16 ;
    new_fb.par.height = LPP+1 ;
    new_fb.par.depth = BITS_PER_PIXEL ;
    new_fb.par.pixclock = PIXEL_CLOCK_MHZ ;
    new_fb.par.clcd_reg_area[0] = new_fb.clcd_reg_area[0] ;
    new_fb.par.clcd_reg_area[1] = new_fb.clcd_reg_area[1] ;

    /*
     * reset the device (ready for use)
     */
    if (hw_init(&new_fb.par) < 0)
    {
	printk(KERN_INFO __FILE__ ": failed to initialise hardware\n");
	return -EFAULT;
    }

    /* This should give a reasonable default video mode */
    fbgen_get_var(&new_fb.display.var, -1, &new_fb.fb_info_gen.info);
    fbgen_do_set_var(&new_fb.display.var, 1, &new_fb.fb_info_gen);
    fbgen_set_disp(-1, &new_fb.fb_info_gen);
    fbgen_install_cmap(0, &new_fb.fb_info_gen);

    if (register_framebuffer(&new_fb.fb_info_gen.info) < 0)
    {
	printk(KERN_INFO __FILE__ ": ERROR - Cannot register framebuffer\n");
	return -EINVAL;
    }

    printk(KERN_INFO "fb%d: %s frame buffer device\n",
	   GET_FB_IDX(new_fb.fb_info_gen.info.node),
	   new_fb.fb_info_gen.info.modename);

    // insert it into the list of Clcd frame buffers
    new_fb.next = clcd_fbs;
    clcd_fbs = &new_fb;

    // success
    return 0;
}

/* --------------------------------------------------------------------------------- */
/* MODULE STUFF                                                                      */
/* --------------------------------------------------------------------------------- */

int __init clcd_init(void)
{
    unsigned int *regs ;

#if DEBUG
    printk(KERN_INFO __FILE__ ": clcd_init() called\n");
#endif

    /*
     * Map the registers into kernel memory
     */
    regs = (unsigned int *)IO_ADDRESS(VERSATILE_CLCD_BASE);

    /*
     * display info about frame buffer (which was allocated during driver setup)
     */
    printk(KERN_INFO __FILE__ ": framebuffer = 0x%p, regs = 0x%p\n", 
	   (void *)clcd_framebuffer_base, regs);
    printk(KERN_INFO __FILE__ ": framebuffer phys = 0x%p, regs = 0x%p\n", 
	   (void *)virt_to_phys(clcd_framebuffer_base), regs);

    if ( (void *)clcd_framebuffer_base == NULL )
	return -ENOMEM;

#if DEBUG
    printk(KERN_INFO __FILE__ ": Registers mapped to 0x%p, frame buffer to 0x%p\n",
	   regs, (void *)clcd_framebuffer_base);
#endif

    return clcd_found_device(0, (u_int)clcd_framebuffer_base, SZ_4M, (u_int)clcd_framebuffer_base,
			        (u_int) regs, SZ_4K, (u_int) regs);
}

#ifdef MODULE
int init_module(void)
{

    printk("The CLCD FB driver has not been set up to be used as a module.\n");
    printk("Please see the source code for some helpful comments on this matter.");
    return -EINVAL;
}

void cleanup_module(void)
{
}
#else
static int setup = 0;
static int initialized = 0;

int __init clcdfb_setup(char *options)
{
#if DEBUG
    printk(KERN_INFO __FILE__ ": clcdfb_setup() called\n");
#endif
    if (!setup)
    {
      printk(KERN_INFO __FILE__ ": Allocate framebuffer of %i\n", SZ_4M);
      clcd_framebuffer_base = (unsigned int)alloc_bootmem_pages(SZ_4M);
      printk(KERN_INFO __FILE__ ": Framebuffer allocated at 0x%x\n",clcd_framebuffer_base);
      if (clcd_framebuffer_base != 0) setup=1;
    }
    return 0;
}

int __init clcdfb_init(void)
{
#if DEBUG
    printk(KERN_INFO __FILE__ ": clcdfb_init() called\n");
#endif

    if (!initialized)
    {
      if (setup)
      {
        initialized = 1;
        clcd_init();
      }
      else
      {
        printk(KERN_INFO __FILE__ ": driver was not initialised because clcd_framebuffer_base has not been set up correctly\n");
        return -ENOMEM;
      }
    }
    /* never return failure, user can hotplug later... */
    return 0;
}
#endif /* MODULE */

/* --------------------------------------------------------------------------------- */
/* VERSATILE COLOR LCD SPECIFIC ROUTINES                                                  */
/* --------------------------------------------------------------------------------- */

#define WRITE_REG(b, o, v) *(volatile unsigned int *)((unsigned int)(b)+(o)) = v 

static void hw_set_colour(struct clcdfb_par *par, unsigned char regno,
			  unsigned char r, unsigned char g, unsigned char b)
{
    volatile unsigned int *palette = (volatile unsigned int *)(par->clcd_reg_area[1].mem + CLCD_PALETTE);
    unsigned int val ;

	/* Input values are 8-bit; we have 5-bit resolution */
    val  = ((r>>3) & 0x1f);
    val |= ((g>>3) & 0x1f) << 5 ;
    val |= ((b>>3) & 0x1f) << 10 ;

    /*
     * Must write the palette registers as 32-bit words, not 16-bit
     */
    if (regno & 1)
	palette[regno/2] = (palette[regno/2] & 0xFFFF) | (val <<16) ;
    else
	palette[regno/2] = (palette[regno/2] & 0xFFFF0000) | val ;
    
    // printk("index %d r = 0x%x, g = 0x%x, b = 0x%x, v = 0x%x\n", regno, r, g, b, val) ;
}


static int hw_init(struct clcdfb_par *par)
{
    unsigned char *framebuffer;
    volatile unsigned int *regs;
    volatile unsigned int *palette;
    unsigned int clcdid = 0;
    int j ;

#if DEBUG
    printk(KERN_INFO __FILE__ ": hw_init(0x%x) called\n", (unsigned int)par);
#endif

    framebuffer = (unsigned char *) par->clcd_reg_area[0].mem ;
    regs        = (volatile unsigned int *) par->clcd_reg_area[1].mem ;
    palette     = (volatile unsigned int *)(par->clcd_reg_area[1].mem + CLCD_PALETTE) ;

#if DEBUG
    printk(KERN_INFO __FILE__ "framebuffer = 0x%p regs = 0x%p palette = 0x%p\n",
	   framebuffer, regs, palette);
    printk(KERN_INFO __FILE__ "select display\n");
#endif

    /*
     * Set the screen to 5-5-5 mux 8bit or 24bit mode, enable the CLCD and VGA
     */
    *(volatile u_int *)(IO_ADDRESS(VERSATILE_SYS_BASE) + VERSATILE_SYS_CLCD_OFFSET) &= ~VERSATILE_SYS_CLCD_MODE;

#if BITS_PER_PIXEL==24
        *(volatile u_int *)(IO_ADDRESS(VERSATILE_SYS_BASE) + VERSATILE_SYS_CLCD_OFFSET) |= VERSATILE_SYS_CLCD_24;
#else
        *(volatile u_int *)(IO_ADDRESS(VERSATILE_SYS_BASE) + VERSATILE_SYS_CLCD_OFFSET) |= VERSATILE_SYS_CLCD_555;
#endif

    *(volatile u_int *)(IO_ADDRESS(VERSATILE_SYS_BASE) + VERSATILE_SYS_CLCD_OFFSET) |= VERSATILE_SYS_CLCD_EN;

#if DEBUG
    printk(KERN_INFO __FILE__ ": clearing frame buffer @ 0x%x\n", (unsigned int)framebuffer) ;
#endif

    /*
     * Clear frame buffer 
     */
    memset(framebuffer, 0, ((PPL+1)*16 * (LPP+1))*(1<<LCDBPP)/8);

#if DEBUG
    printk(KERN_INFO __FILE__ ": clearing palette @ 0x%x\n", (unsigned int)palette) ;
#endif

    /*
     * and the palette to all white
     */
    for (j=0; j < PALETTE_WORDS; ++j)
        palette[j] = -1 ;

#if DEBUG
    printk(KERN_INFO __FILE__ ": Initialising device (regs @ 0x%x)\n", (unsigned int)regs) ;
#endif
    /*
     * Detect what type of LCD panel is connected and set the frequency.
     */
    clcdid = *(unsigned int *)(IO_ADDRESS(VERSATILE_SYS_BASE) + VERSATILE_SYS_CLCD_OFFSET);
    clcdid = (clcdid >> 8) & 0xF;

    if (clcdid == VERSATILE_3x8_PANEL) {
        WRITE_REG(regs, CLCD_TIM0, 0x0505054c);
        WRITE_REG(regs, CLCD_TIM1, 0x050514ef);
        WRITE_REG(regs, CLCD_TIM2, 0x053f1800);
	__raw_writel(0xa05f, IO_ADDRESS(VERSATILE_SYS_BASE) + VERSATILE_SYS_LOCK_OFFSET);
	__raw_writel(0x2c2a, IO_ADDRESS(VERSATILE_SYS_BASE) + VERSATILE_SYS_OSC4_OFFSET);
	__raw_writel(0, IO_ADDRESS(VERSATILE_SYS_BASE) + VERSATILE_SYS_LOCK_OFFSET);
    } else if (clcdid == VERSATILE_8x4_PANEL) {
        /*
	 * Program the CLCD controller registers and start the CLCD
	 */
        WRITE_REG(regs, CLCD_TIM0, ((PPL<<2) | (HSW<<8) | (HFP<<16) | (HBP<<24))) ;
	WRITE_REG(regs, CLCD_TIM1, ((LPP) | (VSW<<10) | (VFP<<16) | (VBP<<24)));
	WRITE_REG(regs, CLCD_TIM2, (PCD) | (CLKSEL<<5) | (ACB<<6) | (IVS<<11) | 
		  (IHS<<12) | (IPC<<13) | (IOE<<14) | (CPL<<16) | (BCD<<26)) ;
	WRITE_REG(regs, CLCD_TIM3, (CLCD_LED) | (CLCD_LEE<<16)) ;
    } else if (clcdid == VERSATILE_2x2_PANEL) {
        /*
	 * not tested.
	 */
    } else {
        printk("Warning: set default timing values\n");


        WRITE_REG(regs, CLCD_TIM0, 0x3f1f3f9c);
        WRITE_REG(regs, CLCD_TIM1, 0x090b61df);
        WRITE_REG(regs, CLCD_TIM2, 0x067f1800);
    }

    /*
     * Set the framebuffer address
     */
    WRITE_REG(regs, CLCD_UBAS, (unsigned int)__virt_to_phys(framebuffer)) ;
    WRITE_REG(regs, CLCD_LBAS, 0) ;
    WRITE_REG(regs, CLCD_IENB, 0<<1) ;

    /*
     * set the control register, power OFF. Set interrupt on back porch
     */
    WRITE_REG(regs, CLCD_CNTL,
			  (LCDEN) | (LCDBPP<<1) | (LCDBW<<4) | (LCDTFT<<5) | (LCDMONO8<<6) | 
			  (LCDDUAL<<7) | (BGR<<8) | (BEBO<<9) | (BEPO<<10) | (LCDVCOMP<<12) |
			  (LDMAFIFOTME<<15) | (WATERMARK<<16)) ;	

    /* then wait 20msec ... */
    AFTER(20);

    /* and now apply power */
    WRITE_REG(regs, CLCD_CNTL,
			  (LCDEN) | (LCDBPP<<1) | (LCDBW<<4) | (LCDTFT<<5) | (LCDMONO8<<6) | 
			  (LCDDUAL<<7) | (BGR<<8) | (BEBO<<9) | (BEPO<<10) | (LCDPWR<<11) |
			  (LCDVCOMP<<12) | (LDMAFIFOTME<<15) | (WATERMARK<<16)) ;	

#if DEBUG
	printk(KERN_INFO __FILE__ ": Registers set to\n%08x %08x %08x %08x\n%08x %08x %08x %08x\n",
		   regs[CLCD_TIM0/sizeof(*regs)],
		   regs[CLCD_TIM1/sizeof(*regs)],
		   regs[CLCD_TIM2/sizeof(*regs)],
		   regs[CLCD_TIM3/sizeof(*regs)],
		   regs[CLCD_UBAS/sizeof(*regs)],
		   regs[CLCD_LBAS/sizeof(*regs)],
		   regs[CLCD_IENB/sizeof(*regs)],
		   regs[CLCD_CNTL/sizeof(*regs)]);
#endif

    return 0;    /* success */
}

static int hw_set_par(struct clcdfb_par *par)
{
#if DEBUG
    printk(KERN_INFO __FILE__ ": hw_set_par () - %d x %d, %d bpp\n",
	   par->width, par->height, par->depth);
#endif

    return 0;
}

static int hw_blank(struct clcdfb_par *par, int blank_mode)
{
    volatile unsigned int *regs =
	  (volatile unsigned int *) par->clcd_reg_area[1].mem ;

#if DEBUG
    printk(KERN_INFO __FILE__ ": hw_blank(blank_mode=%d) called\n",
	   blank_mode);
#endif

    /*
     *  Blank the screen if blank_mode != 0, else unblank. If blank == NULL
     *  then the caller blanks by setting the CLUT (Color Look Up Table) to all
     *  black. Return 0 if blanking succeeded, != 0 if un-/blanking failed due
     *  to e.g. a video mode which doesn't support it. Implements VESA suspend
     *  and powerdown modes on hardware that supports disabling hsync/vsync:
     *    blank_mode == 2: suspend vsync
     *    blank_mode == 3: suspend hsync
     *    blank_mode == 4: powerdown
     */

    return 1;	/* does not work at the moment */
}

