/*
 * drivers/video/omap/omap1610.h
 *
 * BRIEF MODULE DESCRIPTION
 *  This file is the header file for the omap1610 framebuffer.
 * (based on drivers/video/omap/omap1510.h)
 *
 * Author: MontaVista Software, Inc.
 *	   source@mvista.com
 *
 * 2004 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

#ifndef _fb_omap1610_h_
#define _fb_omap1610_h_

#include <asm/hardware.h>
#include <asm/arch/ck.h>
#include <linux/kernel.h>


#define NUM_XPIXELS 240
#define NUM_YPIXELS 320

#define COLOR_16BIT

#ifdef COLOR_16BIT
#define BPP 16
#define PALETTE_SIZE 32
#define BPP_MASK 0x4000
#else
#define BPP 8
#define PALETTE_SIZE 512
#define BPP_MASK 0x3000
#endif

#define MAX_FRAMEBUFFER_SIZE (((NUM_XPIXELS) * (NUM_YPIXELS) * (BPP)/8) + \
                              (PALETTE_SIZE) + PAGE_SIZE)
// Put the framebuffer at top of SRAM
#undef SRAM_FRAMEBUFFER_MEMORY			      
#define SRAM_FRAMEBUFFER_MEMORY (OMAP1610_SRAM_BASE + OMAP1610_SRAM_SIZE - MAX_FRAMEBUFFER_SIZE)
/* we use this to check for valid displays in set_var */
struct _displays {
	int xres;
	int yres;
	int bpp;
};
static struct _displays displays_supported[] =
    { {NUM_XPIXELS, NUM_YPIXELS, 16} };

static struct fb_var_screeninfo tifb_default8 = {
      xres:NUM_XPIXELS,
      yres:NUM_YPIXELS,
      xres_virtual:NUM_XPIXELS,
      yres_virtual:NUM_YPIXELS,
      xoffset:0,
      yoffset:0,
      bits_per_pixel:8,
      grayscale:0,
      red:{0, 8, 0},
      green:{0, 8, 0},
      blue:{0, 8, 0},
      transp:{0, 0, 0},
      nonstd:0,
      activate:0,
      height:-1,
      width:-1,
      accel_flags:0,
      pixclock:171521,
      left_margin:64,
      right_margin:64,
      upper_margin:32,
      lower_margin:32,
      hsync_len:64,
      vsync_len:2,
      sync:0,
      vmode:FB_VMODE_NONINTERLACED
};

static struct fb_var_screeninfo tifb_default16 = {
      xres:NUM_XPIXELS,
      yres:NUM_YPIXELS,
      xres_virtual:NUM_XPIXELS,
      yres_virtual:NUM_YPIXELS,
      xoffset:0,
      yoffset:0,
      bits_per_pixel:16,
      grayscale:0,
      red:{11, 5, 0},
      green:{5, 6, 0},
      blue:{0, 5, 0},
      transp:{0, 0, 0},
      nonstd:0,
      activate:0,
      height:-1,
      width:-1,
      accel_flags:0,
      pixclock:171521,
      left_margin:64,
      right_margin:64,
      upper_margin:32,
      lower_margin:32,
      hsync_len:64,
      vsync_len:2,
      sync:0,
      vmode:FB_VMODE_NONINTERLACED
};

static inline void
hardware_enable(void)
{
	/* initialize LCD */
	outw(inw(GPIO_DIRECTION_REG) & (~0xe000), GPIO_DIRECTION_REG);
	outw(0xe000, GPIO_SET_DATAOUT_REG);

	ck_set_rate(lcd_ck, (ck_get_rate(tc_ck) / 2));
	ck_enable(lcd_ck);

}

static inline void
hardware_disable(void)
{
	/* stop LCD */
	outw(inw(GPIO_DIRECTION_REG) | 0xe000, GPIO_DIRECTION_REG);
	ck_disable(lcd_ck);
}

static inline int
lcd_active(void)
{
	/* Return true if LCD controller is running. */
	return (LCD_READ(TI925_LCD_CONTROL) & 1);
}

static inline void
lcd_enable(void)
{
	unsigned long value;

	/* enable the LCD */
	value = LCD_READ(TI925_LCD_CONTROL);
	value |= 1;
	LCD_WRITE(value, TI925_LCD_CONTROL);
}

static inline void
lcd_disable(void)
{
	unsigned long value;

	/* enable the LCD */
	value = LCD_READ(TI925_LCD_CONTROL);
	value &= ~1;
	LCD_WRITE(value, TI925_LCD_CONTROL);

	/* Wait until current frame finished. */
	/*  Poll on LCDStatus[Done] bit. */
	for (value = 1000; value && (!(LCD_READ(TI925_LCD_STATUS) & 0x1));
	     value--)
		udelay(100);
}

static inline void
orl(unsigned long val, unsigned long reg)
{
	outl((inl(reg) | val), reg);
}

static inline unsigned long
omap1610_dma_reg_addr(int global, int chan, int reg)
{
	return (OMAP1610_DMA_BASE + (global <<10) +(chan << 6) + (reg << 1));
}

static inline void
lcd_write_top_address(unsigned long addr)
{
	unsigned long reg;

	reg = omap1610_dma_reg_addr(NO_GLOBAL_DMA_ACCESS, LCD_CHANNEL,
				    DMA_LCD_TOP_B1_U);

	pr_debug("%s\n%lx %lx\n", __FUNCTION__, reg, addr >> 16);

	outw(addr >> 16, reg);

	pr_debug("wrote %lx %x\n", reg, inw(reg));

	reg = omap1610_dma_reg_addr(NO_GLOBAL_DMA_ACCESS, LCD_CHANNEL,
				    DMA_LCD_TOP_B1_L);

	pr_debug("%lx %lx\n", reg, addr & 0xffff);

	outw(addr & 0xffff, reg);

	pr_debug("wrote %lx %x\n", reg, inw(reg));
}

static inline void
lcd_write_bottom_address(unsigned long addr)
{
	unsigned long reg = omap1610_dma_reg_addr(NO_GLOBAL_DMA_ACCESS,
						  LCD_CHANNEL,
						  DMA_LCD_BOT_B1_U);

	pr_debug("%s\n%lx %lx\n", __FUNCTION__, reg, addr >> 16);

	outw(addr >> 16, reg);

	pr_debug("wrote %lx %x\n", reg, inw(reg));

	reg = omap1610_dma_reg_addr(NO_GLOBAL_DMA_ACCESS,
				    LCD_CHANNEL, DMA_LCD_BOT_B1_L);

	pr_debug("%lx %lx\n", reg, addr & 0xffff);

	outw(addr & 0xffff, reg);

	pr_debug("wrote %lx %x\n", reg, inw(reg));
}

static inline void
lcd_dma_enable(void)
{
	unsigned long reg = omap1610_dma_reg_addr(NO_GLOBAL_DMA_ACCESS,
						  LCD_CHANNEL,
						  DMA_LCD_CTRL);
	pr_debug("%lx\n", reg);

	reg = omap1610_dma_reg_addr(NO_GLOBAL_DMA_ACCESS, LCD_CHANNEL,
				    DMA_LCD_CCR);
	outw(0x5700, reg);
	pr_debug("wrote %lx %x\n", reg, (int) inw(reg));

	reg = omap1610_dma_reg_addr(NO_GLOBAL_DMA_ACCESS, LCD_CHANNEL,
				    DMA_LCD_CSDP);
	outw(0x142, reg);
	pr_debug("wrote %lx %x\n", reg, (int) inw(reg));

	reg = omap1610_dma_reg_addr(NO_GLOBAL_DMA_ACCESS, LCD_CHANNEL,
				    DMA_LCD_CTRL);
#ifdef CONFIG_FB_SDRAM
	outw((LCD_BUS_ERROR_IT_IE), reg);
#else
	outw((LCD_BUS_ERROR_IT_IE | LCD_SOURCE_IMIF), reg);
#endif

	pr_debug("wrote %lx %x\n", reg, (int) inw(reg));

	reg = omap1610_dma_reg_addr(NO_GLOBAL_DMA_ACCESS, LCD_CHANNEL,
				    DMA_LCD_SRC_EN_B1);
	outw(38408, reg);
	pr_debug("wrote %lx %lx\n", reg, (int) inw(reg));

	reg = omap1610_dma_reg_addr(NO_GLOBAL_DMA_ACCESS, LCD_CHANNEL,
				    DMA_LCD_SRC_FN_B1);
	outw(1, reg);
	pr_debug("wrote %lx %lx\n", reg, (int) inw(reg));

	reg = omap1610_dma_reg_addr(NO_GLOBAL_DMA_ACCESS, LCD_CHANNEL,
				    DMA_LCD_LCH_CTRL);
	outw(0x4, reg);
	pr_debug("wrote %lx %lx\n", reg, (int) inw(reg));
}

#ifdef FB_DEBUG
static inline void
print_lcd_regs(void)
{
	pr_debug("%s\n", __FUNCTION__);

	pr_debug("control: %8.8x status %8.8x timing0 %8.8x"
	    " timing1 %8.8x timing2 %8.8x line %4d\n",
	    (int) LCD_READ(TI925_LCD_CONTROL),
	    (int) LCD_READ(TI925_LCD_STATUS),
	    (int) LCD_READ(TI925_LCD_TIMING0),
	    (int) LCD_READ(TI925_LCD_TIMING1),
	    (int) LCD_READ(TI925_LCD_TIMING2),
	    (int) LCD_READ(LCD_DISPLAYSTATUS) & 0x3FF);

}
#else
static inline void
print_lcd_regs(void)
{
}
#endif

static void
tifb_init_dma(void)
{
	int size = current_par.palette_size + current_par.screen_size;

	lcd_dma_enable();
	lcd_write_top_address((unsigned long) current_par.p_palette_base);
	lcd_write_bottom_address((unsigned long) current_par.p_palette_base +
				 size - 2);

	pr_debug("%s size=%d\n", __FUNCTION__, size);
	pr_debug("%s base=%lx\n", __FUNCTION__, current_par.p_palette_base);
	pr_debug("5s end=%lx\n", __FUNCTION__, current_par.p_palette_base + size - 2);
	print_lcd_regs();
}

static void __init
tifb_init_registers(void)
{
	unsigned long value;
	pr_debug("%s\n", __FUNCTION__);

	hardware_enable();

	value = 0x88;
	LCD_WRITE(value, TI925_LCD_CONTROL);

	/* setup the Timing0 */
	value = NUM_XPIXELS - 1;	/* current_par.xres -1; */
	value |= ((4 - 1) << 10);	/* HSW width for LCD is 4-1 */
	value |= ((4 - 1) << 16);	/* H Front Porch is 4-1 */
	value |= ((40 - 1) << 24);	/* H Back Porch is 40-1 */

	/* value = 0x27030cef;  This is what the above should equal */
	LCD_WRITE(value, TI925_LCD_TIMING0);

	/* setup the Timing1 */
	value = NUM_YPIXELS - 1;	/* current_par.yres - 1; */
	value |= (2 << 10);	/* VSW width for the LCD is 2 */
	value |= (2 << 16);	/* Front Porch is 2 */
	value |= (8 << 24);	/* Back Porch is 8 */

	//value = 0x0802093f;  This is what the above should equal
	LCD_WRITE(value, TI925_LCD_TIMING1);

	value = 0x00300008;
	LCD_WRITE(value, TI925_LCD_TIMING2);

	/* setup dma */
	tifb_init_dma();
	print_lcd_regs();

	/* REVISIT: This is req'd to get LCD DMA to free run. */
	ck_disable(dma_ck);

}

static void
lcd_dma_irq_handler(int irq, void *dummy, struct pt_regs *fp)
{
	unsigned long status = inw(omap1610_dma_reg_addr(NO_GLOBAL_DMA_ACCESS,
							 LCD_CHANNEL,
							 DMA_LCD_CTRL));
	if (status & (1 << 5))
		printk("LCD DMA bus error\n");

	if (status & (1 << 4))
		printk("LCD DMA end of block 2\n");

	if (status & (1 << 3))
		printk("LCD DMA end of block 1\n");
}

static void
lcd_irq_handler(int irq, void *dummy, struct pt_regs *fp)
{
	unsigned long status;
	int reset = 0;
	static int abc = 0;
	static int fuf = 0;
	static int sync = 0;
	static int done = 0;

	pr_debug("%s\n", __FUNCTION__);

	status = LCD_READ(TI925_LCD_STATUS);

	if (status & (1 << 3)) {
		pr_debug("ABC interrupt\n");

		status &= (1 << 3);
		LCD_WRITE(status, TI925_LCD_STATUS);
		abc++;
	}

	if (status & (1 << 5)) {
		pr_debug("FUF interrupt\n");

		reset = 1;
		fuf++;
	}

	if (status & (1 << 2)) {
		pr_debug("synch interrupt\n");

		reset = 1;
		sync++;
		print_lcd_regs();
	}

	if (status & (1 << 0)) {
		pr_debug("DONE interrupt\n");

		reset = 1;
		done++;
	}

	if ((sync % 1024) == 0) {
		pr_debug(" abc %d fuf %d sync %d done %d\n", abc, fuf, sync, done);
		print_lcd_regs();
	}

	if (reset) {
		/* This will clear the error condition and restart the LCD */
		unsigned long control;

		control = inl(IO_ADDRESS(TI925_LCD_CONTROL));
		control &= ~(1 << 0);
		outl(control, IO_ADDRESS(TI925_LCD_CONTROL));
		control = inl(IO_ADDRESS(TI925_LCD_CONTROL));
		control |= (1 << 0);
		outl(control, IO_ADDRESS(TI925_LCD_CONTROL));
	}
}

#endif				/* _fb_omap1610_h_ */

