/* 
 * drivers/video/dbmx21fb.c
 *
 * MX21ADS LCD Controller framebuffer driver
 *
 * Author: MontaVista Software, Inc. <source@mvista.com>.
 *
 * This file is based on dbmx21fb.c from Motorola Dragonball MX2 ADS BSP
 * Copyright 2002, 2003 Motorola, Inc. All Rights Reserved.
 *
 * 2004 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */
#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/ctype.h>
#include <linux/mm.h>
#include <linux/tty.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/fb.h>
#include <linux/delay.h>
#include <linux/wrapper.h>
#include <linux/selection.h>
#include <linux/console.h>
#include <linux/kd.h>
#include <linux/vt_kern.h>

#include <asm/hardware.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/mach-types.h>
#include <asm/uaccess.h>
#include <asm/proc/pgtable.h>

#include <video/fbcon.h>
#include <video/fbcon-mfb.h>
#include <video/fbcon-cfb4.h>
#include <video/fbcon-cfb8.h>
#include <video/fbcon-cfb16.h>

#include <asm/arch/memory.h>
#include <asm/arch/gpio.h>

#include "dbmx21fb.h"

#ifdef CONFIG_PM
#include <linux/pm.h>
struct pm_dev *pm;
#endif

#define HARDWARE_CURSOR

struct fbcon_font_desc *font;

/* LCD controller parameters */
struct dbmx21fb_par {
	unsigned char *screen_start_address;	/* Screen Start Address */
	unsigned char *v_screen_start_address;	/* Virtual Screen Start Address */
	unsigned long screen_memory_size;	/* screen memory size */
	unsigned int palette_size;
	unsigned int max_xres;
	unsigned int max_yres;
	unsigned int xres;
	unsigned int yres;
	unsigned int xres_virtual;
	unsigned int yres_virtual;
	unsigned int max_bpp;
	unsigned int bits_per_pixel;
	unsigned int currcon;
	unsigned int visual;
	unsigned int TFT:1;
	unsigned int color:1;
	unsigned int sharp:1;

	unsigned short cfb16[16];
};

#ifdef HARDWARE_CURSOR
/* hardware cursor parameters */
struct dbmx21fb_cursor {
	int startx;
	int starty;
	int blinkenable;
	int blink_rate;
	int width;
	int height;
	int color[3];
	int state;
};

/* Frame buffer of LCD information */
struct dbmx21fb_info {
	struct display_switch dispsw;
	struct dbmx21fb_cursor cursor;
};
#endif				/*  HARDWARE_CURSOR */

static unsigned char *p_framebuffer_memory_address;
static unsigned char *v_framebuffer_memory_address;

/* Fake monspecs to fill in fbinfo structure */
static struct fb_monspecs monspecs __initdata = {
	30000, 70000, 50, 65, 0	/* Generic */
};

/* color map initial */
static unsigned short __attribute__ ((unused)) color4map[16] = {
0x0000, 0x000f, 0x00f0, 0x0f2a, 0x0f00, 0x0f0f, 0x0f88, 0x0ccc,
	    0x0888, 0x00ff, 0x00f8, 0x0f44, 0x0fa6, 0x0f22, 0x0ff0, 0x0fff};

static struct display global_disp;	/* Initial (default) Display Settings */
static struct fb_info fb_info;
static struct fb_var_screeninfo init_var = { };
static struct dbmx21fb_par current_par = { };

/* Frame buffer device API */
static int dbmx21fb_get_fix(struct fb_fix_screeninfo *fix, int con,
			    struct fb_info *info);
static int dbmx21fb_get_var(struct fb_var_screeninfo *var, int con,
			    struct fb_info *info);
static int dbmx21fb_set_var(struct fb_var_screeninfo *var, int con,
			    struct fb_info *info);
static int dbmx21fb_get_cmap(struct fb_cmap *cmap, int kspc, int con,
			     struct fb_info *info);
static int dbmx21fb_set_cmap(struct fb_cmap *cmap, int kspc, int con,
			     struct fb_info *info);

/* Interface to the low level driver  */
static int dbmx21fb_switch(int con, struct fb_info *info);
static void dbmx21fb_blank(int blank, struct fb_info *info);
static int dbmx21fb_updatevar(int con, struct fb_info *info);

/*  Internal routines  */
static int reserve_fb_memory(void);
static void enable_lcd_controller(void);
static void disable_lcd_controller(void);
static int encode_var(struct fb_var_screeninfo *var, struct dbmx21fb_par *par);
static int decode_var(struct fb_var_screeninfo *var, struct dbmx21fb_par *par);

/* initialization routines */
static void __init init_lcd_system(void);
static int __init init_lcd(void);
static void __init init_fbinfo(void);
static int __init reserve_fb_memory(void);

/* frame buffer ops */
static struct fb_ops dbmx21fb_ops = {
	.owner		= THIS_MODULE,
	.fb_get_fix	= dbmx21fb_get_fix,
	.fb_get_var	= dbmx21fb_get_var,
	.fb_set_var	= dbmx21fb_set_var,
	.fb_get_cmap	= dbmx21fb_get_cmap,
	.fb_set_cmap	= dbmx21fb_set_cmap,
};

/* Hardware Cursor */
#ifdef HARDWARE_CURSOR
static void dbmx21fb_cursor(struct display *p, int mode, int x, int y);
static int dbmx21fb_set_font(struct display *d, int width, int height);
static unsigned char cursor_color_map[] = { 0xf8 };
static void dbmx21fb_set_cursor_state(struct dbmx21fb_info *fb,
				      unsigned int state);
static void dbmx21fb_set_cursor(struct dbmx21fb_info *fb);
static void dbmx21fb_set_cursor_blink(struct dbmx21fb_info *fb, int blink);

#ifdef FBCON_HAS_CFB4
struct display_switch dbmx21fb_cfb4 = {
	.setup		= fbcon_cfb4_setup,
	.bmove		= fbcon_cfb4_bmove,
	.clear		= fbcon_cfb4_clear,
	.putc 		= fbcon_cfb4_putc,
	.putcs 		= fbcon_cfb4_putcs,
	.revc		= fbcon_cfb4_revc,
	.cursor		= dbmx21fb_cursor,
	.set_font 	= dbmx21fb_set_font,
	.fontwidthmask	= FONTWIDTH(4) | FONTWIDTH(8) | FONTWIDTH(16)
};
#endif

#ifdef FBCON_HAS_CFB8
struct display_switch dbmx21fb_cfb8 = {
	.setup		= fbcon_cfb8_setup,
	.bmove		= fbcon_cfb8_bmove,
	.clear		= fbcon_cfb8_clear,
	.putc 		= fbcon_cfb8_putc,
	.putcs 		= fbcon_cfb8_putcs,
	.revc		= fbcon_cfb8_revc,
	.cursor		= dbmx21fb_cursor,
	.set_font 	= dbmx21fb_set_font,
	.fontwidthmask	= FONTWIDTH(4) | FONTWIDTH(8) | FONTWIDTH(16)
};
#endif

#ifdef FBCON_HAS_CFB16
struct display_switch dbmx21fb_cfb16 = {
	.setup		= fbcon_cfb16_setup,
	.bmove		= fbcon_cfb16_bmove,
	.clear		= fbcon_cfb16_clear,
	.putc 		= fbcon_cfb16_putc,
	.putcs 		= fbcon_cfb16_putcs,
	.revc		= fbcon_cfb16_revc,
	.cursor		= dbmx21fb_cursor,
	.set_font 	= dbmx21fb_set_font,
	.fontwidthmask	= FONTWIDTH(4) | FONTWIDTH(8) | FONTWIDTH(16)
};
#endif

#endif /* HARDWARE_CURSOR */

/*****************************************************************************
 * Function Name: dbmx21fb_getcolreg()
 *
 * Input: 	regno	: Color register ID
 *		red	: Color map red[]
 *		green	: Color map green[]
 *		blue	: Color map blue[]
 *	transparent	: Flag
 *		info	: Fb_info database
 *
 * Value Returned: int	: Return status.If no error, return 0.
 *	
 * Description: Transfer to fb_xxx_cmap handlers as parameters to
 *		control color registers
 *
******************************************************************************/
#define RED	0xf00
#define GREEN	0xf0
#define BLUE	0x0f
static int
dbmx21fb_getcolreg(unsigned int regno, unsigned int *red, unsigned int *green, unsigned int *blue,
		   unsigned int *trans, struct fb_info *info)
{
	unsigned int val;

	if (regno >= current_par.palette_size)
		return 1;

	val = LCDC_BPLUT_BASE(regno);

	if ((current_par.bits_per_pixel == 4) && (!current_par.color)) {
		*red = *green = *blue = (val & BLUE) << 4;	/* TODO */
		*trans = 0;
	} else {
		*red = (val & RED) << 4;
		*green = (val & GREEN) << 8;
		*blue = (val & BLUE) << 12;
		*trans = 0;
	}

	return 0;
}

/*****************************************************************************
 * Function Name: dbmx21fb_setcolreg()
 *
 * Input: 	regno	: Color register ID
 *		red	: Color map red[]
 *		green	: Color map green[]
 *		blue	: Color map blue[]
 *	transparent	: Flag
 *		info	: Fb_info database
 *
 * Value Returned: int 	: Return status.If no error, return 0.
 *
 * Description: Transfer to fb_xxx_cmap handlers as parameters to
 *		control color registers
 *
 **********************************************************F*******************/
static int
dbmx21fb_setcolreg(unsigned int regno, unsigned int red, unsigned int green, unsigned int blue,
		   unsigned int trans, struct fb_info *info)
{
	unsigned int val = 0;

	if (regno >= current_par.palette_size) {
		return 1;
	}

	if ((current_par.bits_per_pixel == 4) && (!current_par.color))
		val = (blue & 0x00f) << 12;	/* TODO */
	else {
		val = (blue >> 12) & BLUE;
		val |= (green >> 8) & GREEN;
		val |= (red >> 4) & RED;
	}

	if (regno < 16) {
		current_par.cfb16[regno] = regno | regno << 5 | regno << 10;
	}

	LCDC_BPLUT_BASE(regno) = val;

	return 0;
}

/*****************************************************************************
 * Function Name: dbmx21fb_get_cmap()
 *
 * Input: 	cmap	: Ouput data pointer
 *		kspc   	: Kernel space flag
 *		con    	: Console ID
 *		info	: Frame buffer information
 *
 * Value Returned: int	: Return status.If no error, return 0.
 *	
 * Description: Data is copied from hardware or local or system DISPAY,
 *		and copied to cmap.
 *
******************************************************************************/
static int
dbmx21fb_get_cmap(struct fb_cmap *cmap, int kspc, int con, struct fb_info *info)
{
	int err = 0;

	pr_debug("dbmx21fb_get_cmap: current_par.visual=%d\n", 
		current_par.visual);
		
	if (con == current_par.currcon)
		err = fb_get_cmap(cmap, kspc, dbmx21fb_getcolreg, info);
	else if (fb_display[con].cmap.len)
		fb_copy_cmap(&fb_display[con].cmap, cmap, kspc ? 0 : 2);
	else
		fb_copy_cmap(fb_default_cmap(current_par.palette_size),
			     cmap, kspc ? 0 : 2);

	return err;
}

/*****************************************************************************
 * Function Name: dbmx21fb_set_cmap()
 *
 * Input: 	cmap	: Ouput data pointer
 *		kspc   	: Kernel space flag
 *		con    	: Console ID
 *		info	: Frame buffer information
 *
 * Value Returned: int	: Return status.If no error, return 0.
 *	
 * Description: Copy data from cmap and copy to DISPLAY. If DISPLAy has no cmap,
 * 		allocate memory for it. If DISPLAY is current console and visible,
 * 		then hardware color map shall be set.
 *
******************************************************************************/
static int
dbmx21fb_set_cmap(struct fb_cmap *cmap, int kspc, int con, struct fb_info *info)
{
	int err = 0;

	pr_debug("dbmx21fb_set_cmap: current_par.visual=%d\n", 
		current_par.visual);

	if (!fb_display[con].cmap.len)
		err = fb_alloc_cmap(&fb_display[con].cmap,
					 current_par.palette_size, 0);

	if (!err) {
		if (con == current_par.currcon) {
			err = fb_set_cmap(cmap, kspc, dbmx21fb_setcolreg, info);
		}
		fb_copy_cmap(cmap, &fb_display[con].cmap, kspc ? 0 : 1);
	}

	return err;
}

/*****************************************************************************
 * Function Name: dbmx21fb_get_var()
 *
 * Input: 	var	: Iuput data pointer
 *		con	: Console ID
 *		info	: Frame buffer information
 *
 * Value Returned: int	: Return status.If no error, return 0.
 *
 * Functions Called: 	encode_var()
 *	
 * Description: Get color map from current, or global display[console]
 * 		used by ioctl
 *
******************************************************************************/
static int
dbmx21fb_get_var(struct fb_var_screeninfo *var, int con, struct fb_info *info)
{
	if (con == -1) 
		encode_var(var, &current_par);
	else
		*var = fb_display[con].var;

	return 0;
}

/*****************************************************************************
 * Function Name: dbmx21fb_updatevar()
 *
 * Value Returned: VOID
 *
 * Functions Called: VOID
 *	
 * Description: Fill in display switch with LCD information,
 *
******************************************************************************/
static int
dbmx21fb_updatevar(int con, struct fb_info *info)
{
	return 0;
}

/*****************************************************************************
 * Function Name: dbmx21fb_set_dispsw()
 *
 * Input: 	display		: Iuput data pointer
 *		dbmx21fb_info   	: Frame buffer of LCD information
 *
 * Value Returned: VOID
 *
 * Functions Called: VOID
 *	
 * Description: Fill in display switch with LCD information,
 *
******************************************************************************/
static void
dbmx21fb_set_dispsw(struct display *disp
#ifdef HARDWARE_CURSOR
		    , struct dbmx21fb_info *info
#endif
    )
{
	switch (disp->var.bits_per_pixel) {
#ifdef HARDWARE_CURSOR
#ifdef FBCON_HAS_CFB4
	case 4:
		fb_info.fix.visual = FB_VISUAL_PSEUDOCOLOR;
		info->dispsw = dbmx21fb_cfb4;
		disp->dispsw = &info->dispsw;
		disp->dispsw_data = NULL;
		break;
#endif
#ifdef FBCON_HAS_CFB8
	case 8:
		fb_info.fix.visual = FB_VISUAL_PSEUDOCOLOR;
		info->dispsw = dbmx21fb_cfb8;
		disp->dispsw = &info->dispsw;
		disp->dispsw_data = NULL;
		break;
#endif
#ifdef FBCON_HAS_CFB16
	case 12:
	case 16:
		fb_info.fix.visual = FB_VISUAL_DIRECTCOLOR;
		info->dispsw = dbmx21fb_cfb16;
		disp->dispsw = &info->dispsw;
		disp->dispsw_data = current_par.cfb16;
		break;
#endif

#else /* HARDWARE_CURSOR */
	/* first step disable the hardware cursor */
#ifdef FBCON_HAS_CFB4
	case 4:
		fb_info.fix.visual = FB_VISUAL_PSEUDOCOLOR;
		disp->dispsw = &fbcon_cfb4;
		disp->dispsw_data = NULL;
		break;
#endif

#ifdef FBCON_HAS_CFB8
	case 8:
		fb_info.fix.visual = FB_VISUAL_PSEUDOCOLOR;
		disp->dispsw = &fbcon_cfb8;
		disp->dispsw_data = NULL;
		break;
#endif

#ifdef FBCON_HAS_CFB16
	case 12:
	case 16:
		fb_info.fix.visual = FB_VISUAL_DIRECTCOLOR;
		disp->dispsw = &fbcon_cfb16;
		disp->dispsw_data = current_par.cfb16;
		break;
#endif

#endif	/* HARDWARE_CURSOR */
	default:
		disp->dispsw = &fbcon_dummy;
		disp->dispsw_data = NULL;
		break;
	}

#ifdef HARDWARE_CURSOR
	if (&info->cursor) {
		info->dispsw.cursor = dbmx21fb_cursor;
		info->dispsw.set_font = dbmx21fb_set_font;
	}
#endif	/* HARDWARE_CURSOR */
}

/*****************************************************************************
 * Function Name: dbmx21fb_set_var()
 *
 * Input: 	var	: Iuput data pointer
 *		con	: Console ID
 *		info	: Frame buffer information
 *
 * Value Returned: int	: Return status.If no error, return 0.
 *
 * Functions Called: 	dbmx21fb_decode_var()
 * 			dbmx21fb_encode_var()
 *  			dbmx21fb_set_dispsw()
 *
 * Description: set current_par by var, also set display data, specially the console
 * 		related fileops, then enable the lcd controller, and set cmap to
 * 		hardware.
 *
******************************************************************************/
static int
dbmx21fb_set_var(struct fb_var_screeninfo *var, int con, struct fb_info *info)
{
	struct display *display;
	int err, chgvar = 0;
	struct dbmx21fb_par par;

	if (con >= 0)
		display = &fb_display[con]; /* Display settings for console */
	else
		display = &global_disp;	/* Default display settings */

	/* 
	 * Decode var contents into a par structure, adjusting any 
	 * out of range values. 
	 */
	if ((err = decode_var(var, &par))) {
		return err;
	}

	/* Store adjusted par values into var structure */
	encode_var(var, &par);

	if ((var->activate & FB_ACTIVATE_MASK) == FB_ACTIVATE_TEST)
		return 0;
	else if (((var->activate & FB_ACTIVATE_MASK) != FB_ACTIVATE_NOW) &&
		 ((var->activate & FB_ACTIVATE_MASK) != FB_ACTIVATE_NXTOPEN))
		return -EINVAL;

	if (con >= 0) {
		if ((display->var.xres != var->xres) ||
		    (display->var.yres != var->yres) ||
		    (display->var.xres_virtual != var->xres_virtual) ||
		    (display->var.yres_virtual != var->yres_virtual) ||
		    (display->var.sync != var->sync) ||
		    (display->var.bits_per_pixel != var->bits_per_pixel) ||
		    (memcmp(&display->var.red, &var->red, sizeof(var->red))) ||
		    (memcmp(&display->var.green, &var->green, 
		    		sizeof(var->green))) ||
		    (memcmp(&display->var.blue, &var->blue, sizeof(var->blue))))
			chgvar = 1;
	}

	display->var = *var;
	display->screen_base = par.v_screen_start_address;
	display->visual = par.visual;
	display->type = FB_TYPE_PACKED_PIXELS;
	display->type_aux = 0;
	display->ypanstep = 0;
	display->ywrapstep = 0;
	display->line_length = display->next_line = (var->xres * 16) / 8;

	display->can_soft_blank = 1;
	display->inverse = 0;

	dbmx21fb_set_dispsw(display
#ifdef HARDWARE_CURSOR
			    , (struct dbmx21fb_info *) info
#endif
	    );

	/* 
	 * If the console has changed and the console has defined
	 * a changevar function, call that function. 
	 */
	if (chgvar && info && info->changevar)
		info->changevar(con);	/* TODO */

	/* 
	 * If the current console is selected and it's not truecolor,
	 * update the palette
	 */
	if ((con == current_par.currcon)) {
		struct fb_cmap *cmap;

		current_par = par;	/* TODO */
		if (display->cmap.len)
			cmap = &display->cmap;
		else
			cmap = fb_default_cmap(current_par.palette_size);

		fb_set_cmap(cmap, 1, dbmx21fb_setcolreg, info);
	}

	/* If the current console is selected, activate the new var. */
	if (con == current_par.currcon) {
		init_var = *var;
		enable_lcd_controller();
	}

	return 0;
}

/*****************************************************************************
 * Function Name: dbmx21fb_get_fix()
 *
 * Input: 	fix	: Ouput data pointer
 *		con	: Console ID
 *		info	: Frame buffer information
 *
 * Value Returned: int	: Return status.If no error, return 0.
 *
 * Functions Called: VOID
 *	
 * Description: get fix from display data, current_par data
 * 		used by ioctl
 * 				
******************************************************************************/
static int
dbmx21fb_get_fix(struct fb_fix_screeninfo *fix, int con, struct fb_info *info)
{
	struct display *display;

	memset(fix, 0, sizeof (struct fb_fix_screeninfo));
	strcpy(fix->id, DBMX21_NAME);

	if (con >= 0) {
		pr_debug("dbmx21fb_get_fix: Using console specific display "
			"for con=%d\n", con);
		display = &fb_display[con]; /* Display settings for console */
	} else
		display = &global_disp;	/* Default display settings */

	fix->smem_start = (unsigned long) current_par.screen_start_address;
	fix->smem_len = current_par.screen_memory_size;
	fix->type = display->type;
	fix->type_aux = display->type_aux;
	fix->xpanstep = 0;
	fix->ypanstep = display->ypanstep;
	fix->ywrapstep = display->ywrapstep;
	fix->visual = display->visual;
	fix->line_length = display->line_length;
	fix->accel = FB_ACCEL_NONE;

	return 0;
}

/*****************************************************************************
 * Function Name: dbmx21fb_blank()
 *
 * Input: 	blank	: Blank flag
 *		info	: Frame buffer database
 *
 * Value Returned: VOID
 *
 * Functions Called: 	enable_lcd_controller()
 * 			disable_lcd_controller()
 *	
 * Description: blank the screen, if blank, disable lcd controller, while if no blank
 * 		set cmap and enable lcd controller
 *
******************************************************************************/
static void
dbmx21fb_blank(int blank, struct fb_info *info)
{
}

/*****************************************************************************
 * Function Name: dbmx21fb_switch()
 *
 * Input:  	info	: Frame buffer database
 *
 * Value Returned: VOID
 *
 * Functions Called:
 *	
 * Description: Switch to another console
 *
******************************************************************************/
static int
dbmx21fb_switch(int con, struct fb_info *info)
{
	struct fb_cmap *cmap;
	
	if (current_par.currcon >= 0) {
		/* Get the colormap for the selected console */
		cmap = &fb_display[current_par.currcon].cmap;

		if (cmap->len)
			fb_get_cmap(cmap, 1, dbmx21fb_getcolreg, info);
	}

	current_par.currcon = con;
	fb_display[con].var.activate = FB_ACTIVATE_NOW;
	dbmx21fb_set_var(&fb_display[con].var, con, info);

	return 0;
}

/*****************************************************************************
 * Function Name: _encode_par()
 *
 * Input:  	var	: Input var data
 *		par	: LCD controller parameters
 *
 * Value Returned: VOID
 *
 * Functions Called:
 *
 * Description: use current_par to set a var structure
 *
******************************************************************************/
static int
encode_var(struct fb_var_screeninfo *var, struct dbmx21fb_par *par)
{
	/*
	 * Don't know if really want to zero var on entry.
	 * Look at set_var to see. If so, may need to add extra params to par
	 */
	var->xres = par->xres;
	var->yres = par->yres;
	var->xres_virtual = par->xres_virtual;
	var->yres_virtual = par->yres_virtual;
	var->bits_per_pixel = par->bits_per_pixel;

	pr_debug("encode_var:var->xress=%d\n", var->xres);
	pr_debug("encode_var:var->yres=%d\n", var->yres);
	pr_debug("encode_var:var->xres_virtual=%d\n", var->xres_virtual);
	pr_debug("encode_var:var->yres_virtual=%d\n", var->yres_virtual);
	pr_debug("encode_var:var->bits_per_pixel=%d\n", var->bits_per_pixel);

	switch (var->bits_per_pixel) {
	case 2:
	case 4:
	case 8:
		var->red.length = 4;
		var->green = var->red;
		var->blue = var->red;
		var->transp.length = 0;
		break;
	case 12:		/* This case should differ for Active/Passive mode */
	case 16:
		pr_debug("encode_var:16->a\n");
		var->red.length = 5;
		var->green.length = 6;
		var->blue.length = 5;
		var->transp.length = 0;
		var->red.offset = 11;
		var->green.offset = 5;
		var->blue.offset = 0;
		var->transp.offset = 0;
		break;
	}

	return 0;
}

/*****************************************************************************
 * Function Name: decode_var
 *
 * Input:  	var	: Input var data
 *		par	: LCD controller parameters
 *
 * Value Returned: VOID
 *
 * Functions Called: VOID
 *	
 * Description: Get the video params out of 'var'. If a value doesn't fit,
 * 		round it up,if it's too big, return -EINVAL.
 *
 * Cautions: Round up in the following order: bits_per_pixel, xres,
 * 	yres, xres_virtual, yres_virtual, xoffset, yoffset, grayscale,
 * 	bitfields, horizontal timing, vertical timing. 			
 *
******************************************************************************/
static int
decode_var(struct fb_var_screeninfo *var, struct dbmx21fb_par *par)
{
	*par = current_par;
	
	if ((par->xres = var->xres) < MIN_XRES)
		par->xres = MIN_XRES;
	if ((par->yres = var->yres) < MIN_YRES)
		par->yres = MIN_YRES;
	if (par->xres > current_par.max_xres)
		par->xres = current_par.max_xres;
	if (par->yres > current_par.max_yres)
		par->yres = current_par.max_yres;
	par->xres_virtual =
	    var->xres_virtual < par->xres ? par->xres : var->xres_virtual;
	par->yres_virtual =
	    var->yres_virtual < par->yres ? par->yres : var->yres_virtual;
	par->bits_per_pixel = var->bits_per_pixel;

	switch (par->bits_per_pixel) {
#ifdef FBCON_HAS_CFB4
	case 4:
		par->visual = FB_VISUAL_PSEUDOCOLOR;
		par->palette_size = 16;
		break;
#endif
#ifdef FBCON_HAS_CFB8
	case 8:
		par->visual = FB_VISUAL_PSEUDOCOLOR;
		par->palette_size = 256;
		break;
#endif
#ifdef FBCON_HAS_CFB16
	case 12:		/* RGB 444 */
	case 16:		/* RGB 565 */
		par->visual = FB_VISUAL_TRUECOLOR;
		par->palette_size = 256;
		break;
#endif
	default:
		return -EINVAL;
	}

	par->screen_start_address =
	    (unsigned char *) ((unsigned long) p_framebuffer_memory_address + 
	    			PAGE_SIZE);
	par->v_screen_start_address =
	    (unsigned char *) ((unsigned long) v_framebuffer_memory_address + 
	    			PAGE_SIZE);

	return 0;
}

/*****************************************************************************
 * Function Name: reserve_fb_memory()
 *
 * Input: VOID
 *
 * Value Returned: VOID
 *
 * Functions Called: 	
 *
 * Description: get data out of var structure and set related LCD controller registers
 *
******************************************************************************/
static int __init
reserve_fb_memory(void)
{
	unsigned int 	required_pages;
	unsigned int 	extra_pages;
	unsigned int 	order;
	struct page 	*page;
	char 		*allocated_region;

	pr_debug("reserve_fb_memory: frame buffer memory size = %x\n",
		    (unsigned int) ALLOCATED_FB_MEM_SIZE);

	if (v_framebuffer_memory_address != NULL)
		return -EINVAL;

	/* Find order required to allocate enough memory for framebuffer */
	required_pages = ALLOCATED_FB_MEM_SIZE >> PAGE_SHIFT;

	for (order = 0; required_pages >> order; order++)
		;

	extra_pages = (1 << order) - required_pages;

	if ((allocated_region =
	     (char *) __get_free_pages(GFP_KERNEL | GFP_DMA, order)) == NULL) {
		return -ENOMEM;
	}

	v_framebuffer_memory_address =
	    (unsigned char *) allocated_region + (extra_pages << PAGE_SHIFT);

	p_framebuffer_memory_address =
	    (unsigned char *) __virt_to_phys((unsigned long) v_framebuffer_memory_address);

	/* 
	 * Free all pages that we don't need but were given to us because 
	 * __get_free_pages() works on powers of 2. 
	 */
	for (; extra_pages; extra_pages--)
		free_page((unsigned int) allocated_region +
			  ((extra_pages - 1) << PAGE_SHIFT));
	
	/* 
	 * Set reserved flag for fb memory to allow it to be remapped into
	 * user space by the common fbmem driver using remap_page_range(). 
	 */
	
	for (page = virt_to_page(v_framebuffer_memory_address);
	     page <
	     virt_to_page(v_framebuffer_memory_address + ALLOCATED_FB_MEM_SIZE);
	     page++) {
		mem_map_reserve(page);
	}

	current_par.screen_start_address = (unsigned char *) 
		((unsigned long) p_framebuffer_memory_address + PAGE_SIZE);
	current_par.v_screen_start_address = (unsigned char *) 
		((unsigned long) v_framebuffer_memory_address + PAGE_SIZE);

	pr_debug("reserve_fb_memory:physical screen start addres: %x\n",
		(unsigned long) p_framebuffer_memory_address + PAGE_SIZE);

	return (v_framebuffer_memory_address == NULL ? -EINVAL : 0);
}

/*****************************************************************************
 * Function Name: enable_lcd_controller()
 *
 * Input: VOID
 *
 * Value Returned: VOID
 *
 * Functions Called: 	
 *	
 * Description: enable Lcd controller, setup registers,
 *		base on current_par value
 *
******************************************************************************/
static void
enable_lcd_controller(void)
{
	CRM_PCCR0 |= (PCCR0_SLCD_PIXCLK_EN | PCCR0_HCLK_LCDC_EN);

	*((volatile uint16_t *) (MX2ADS_PER_IOBASE + 0x800000)) |= 0x0200;
}

/*****************************************************************************
 * Function Name: disable_lcd_controller()
 *
 * Input: VOID
 *
 * Value Returned: VOID
 *
 * Functions Called: VOID
 *
 * Description: just disable the LCD controller
 * 		disable lcd interrupt. others, i have no ideas
 *
******************************************************************************/
static void
disable_lcd_controller(void)
{
	CRM_PCCR0 &= ~(PCCR0_SLCD_PIXCLK_EN | PCCR0_HCLK_LCDC_EN);

	*((volatile uint16_t *) (MX2ADS_PER_IOBASE + 0x800000)) &= ~0x0200;
}

/*****************************************************************************
 * Function Name: init_lcd_system
 *
 * Input: VOID
 *
 * Value Returned: VOID
 *
 * Functions Called:
 *
 * Description: initialise GPIO
 *
******************************************************************************/
static void __init
init_lcd_system(void)
{
	mx2_register_gpios(PORT_A, 0xFFFFFFFF, PRIMARY);
}

/*****************************************************************************
 * Function Name: init_lcd
 *
 * Input: VOID
 *
 * Value Returned: VOID
 *
 * Functions Called: decode_var()
 *
 * Description: initialize the LCD controller, use current_par for 12bpp
 *
******************************************************************************/
static int __init
init_lcd(void)
{
	unsigned int val;

	decode_var(&init_var, &current_par);

	pr_debug("init_lcd:Writing LSSAR with %x\n",
		(unsigned int) current_par.screen_start_address);
	LCDC_LSSAR = (unsigned int) current_par.screen_start_address;

	val = 0;
	val = current_par.xres / 16;
	val = val << 20;
	val += current_par.yres;

	pr_debug("init_lcd:par.x=%x, y=%x\n", current_par.xres,
		current_par.yres);

	LCDC_LSR = val;

	pr_debug("init_lcd:LSR=%x\n", LCDC_LSR);

	val = 0;
	val = current_par.xres_virtual / 2;
	LCDC_LVPWR = val;
	pr_debug("init_lcd:LVPWR=%x\n", LCDC_LVPWR);

	LCDC_LPCR = 0xFB108BC7;
	pr_debug("init_lcd:LPCR=%x\n", LCDC_LPCR);

	/* Sharp Configuration Register */
	LCDC_LSCR = 0x00120300;
	pr_debug("init_lcd:LSCR=%x\n", LCDC_LSCR);

	LCDC_LHCR = 0x04000F06;
	pr_debug("init_lcd:LHCR=%x\n", LCDC_LHCR);

	LCDC_LVCR = 0x04000907;
	pr_debug("init_lcd:LVCR=%x\n", LCDC_LVCR);

	LCDC_LPCCR = 0x00A903FF;
	pr_debug("init_lcd:LPCCR=%x\n", LCDC_LPCCR);

	LCDC_LRMCR = 0x00000000;
	pr_debug("init_lcd:LRMCR=%x\n", LCDC_LRMCR);

	LCDC_LDCR = 0x0004000F;
	pr_debug("init_lcd:LDCR=%x\n", LCDC_LDCR);

	return 0;
}

#ifdef CONFIG_PM
static void
lcd_pm_resume(void)
{
	enable_lcd_controller();

	/* enable normal LD output for the Sharp TFT panel */
	LCDC_LPCCR &= ~0x8000;
}

static void
lcd_pm_suspend(void)
{
	/* disable LD output to zero the Sharp TFT panel */
	LCDC_LPCCR |= 0x8000;
	mdelay(10);
	disable_lcd_controller();
}

static int
lcd_pm_handler(struct pm_dev *dev, pm_request_t rqst, void *data)
{
	switch (rqst) {
	case PM_RESUME:
		lcd_pm_resume();
		break;
	case PM_SUSPEND:
		lcd_pm_suspend();
		break;
	default:
		return -EINVAL;
	}
	return 0;
}
#endif

/*****************************************************************************
 * Function Name: dbmx21fb_init()
 *
 * Input: VOID
 *
 * Value Returned: int 		: Return status.If no error, return 0.
 *
 * Functions Called: 	init_fbinfo()
 *			disable_irq()
 *	 		enable_irq()
 *			init_lcd()
 *			dbmx21fb_init_cursor()
 *
 * Description: initialization module, all of init routine's entry point
 * 		initialize fb_info, init_var, current_par
 * 		and setup interrupt, memory, lcd controller
 *
******************************************************************************/
int __init
dbmx21fb_init(void)
{
	int ret;
	
	init_lcd_system();

	init_fbinfo();

	if ((ret = reserve_fb_memory()) != 0) {
		printk(KERN_ERR
		       "failed for reserved DBMX frame buffer memory\n");
		return ret;
	}

	if (dbmx21fb_set_var(&init_var, -1, &fb_info)) 
		;

	init_lcd();
	enable_lcd_controller();

	register_framebuffer(&fb_info);

#ifdef CONFIG_PM
	pm = pm_register(PM_SYS_DEV, PM_SYS_VGA, lcd_pm_handler);
#endif
	/* This driver cannot be unloaded at the moment */
	MOD_INC_USE_COUNT;

	return 0;
}

/*****************************************************************************
 * Function Name: dbmx21fb_setup()
 *
 * Input: info	: VOID
 *
 * Value Returned: int	: Return status.If no error, return 0.
 *
 * Functions Called: VOID
 *	
 * Description: basically, this routine used to parse command line parameters, which
 * 		is initialization parameters for lcd controller, such as freq, xres,
 * 		yres, and so on
 *
******************************************************************************/
int __init
dbmx21fb_setup(char *options)
{
	return 0;
}

/*****************************************************************************
 * Function Name: init_fbinfo()
 *
 * Input: VOID
 *
 * Value Returned: VOID
 *
 * Functions Called: VOID
 *
 * Description: while 16bpp is used to store a 12 bits pixels packet, but
 * 		it is not a really 16bpp system. maybe in-compatiable with
 * 		other system or GUI.There are some field in var which specify
 *		the red/green/blue offset in a 16bit word, just little endian is
 * 		concerned
 *
******************************************************************************/
static void __init
init_fbinfo(void)
{
	strcpy(fb_info.modename, DBMX21_NAME);
	strcpy(fb_info.fontname, "Acorn8x8");

	fb_info.node = -1;
	fb_info.flags = 0;
	fb_info.fbops = &dbmx21fb_ops;
	fb_info.monspecs = monspecs;
	fb_info.disp = &global_disp;
	fb_info.changevar = NULL;
	fb_info.switch_con = dbmx21fb_switch;
	fb_info.updatevar = dbmx21fb_updatevar;
	fb_info.blank = dbmx21fb_blank;

	/* setup initial parameters */
	memset(&init_var, 0, sizeof (init_var));

	init_var.transp.length = 0;
	init_var.nonstd = 0;
	init_var.activate = FB_ACTIVATE_NOW;
	init_var.xoffset = 0;
	init_var.yoffset = 0;
	init_var.height = -1;
	init_var.width = -1;
	init_var.vmode = FB_VMODE_NONINTERLACED;

	current_par.max_xres = LCD_MAXX;
	current_par.max_yres = LCD_MAXY;
	current_par.max_bpp = LCD_MAX_BPP;
	init_var.red.length = 5;
	init_var.green.length = 6;
	init_var.blue.length = 5;
	init_var.red.offset = 11;
	init_var.green.offset = 5;
	init_var.blue.offset = 0;

	init_var.grayscale = 16;
	init_var.sync = 0;
	init_var.pixclock = 171521;

	current_par.screen_start_address = NULL;
	current_par.v_screen_start_address = NULL;
	current_par.screen_memory_size = MAX_PIXEL_MEM_SIZE;
	current_par.currcon = -1;

	init_var.xres = current_par.max_xres;
	init_var.yres = current_par.max_yres;
	init_var.xres_virtual = init_var.xres;
	init_var.yres_virtual = init_var.yres;
	init_var.bits_per_pixel = current_par.max_bpp;
}

/* Hardware cursor support */
#ifdef HARDWARE_CURSOR
 /*****************************************************************************
 * Function Name: dbmx21fb_set_cursor_color()
 *
 * Input:   fb	: frame buffer database
 *	    red	: red component level in the cursor
 *	  green	: green component level in the cursor
 *	   blue	: blue component level in the cursor	
 *
 * Value Returned: VOID
 *
 * Functions Called: VOID	
 *	
 * Description: Set color of hardware cursor
 *
 * Modification History:
 *		10 DEC,2001, Zhang Juan
******************************************************************************/
static void
dbmx21fb_set_cursor_color(struct dbmx21fb_info *fb, unsigned char *red,
			  unsigned char *green, unsigned char *blue)
{
	struct dbmx21fb_cursor *c = &fb->cursor;
	unsigned int color;

	c->color[0] = *red;
	c->color[1] = *green;
	c->color[2] = *blue;
	color = (unsigned int) *red;
	color |= (unsigned int) (*green >> 5);
	color |= (unsigned int) (*blue >> 11);

	LCDC_LCCMR = color;
}

 /*****************************************************************************
 * Function Name: dbmx21fb_set_cursor()
 *
 * Input:   fb	: frame buffer database
 *
 * Value Returned: VOID
 *
 * Functions Called: VOID	
 *	
 * Description: Load information of hardware cursor
 *
 * Modification History:
 *		10 DEC,2001, Zhang Juan
******************************************************************************/
static void
dbmx21fb_set_cursor(struct dbmx21fb_info *fb)
{
	struct dbmx21fb_cursor *c = &fb->cursor;
	unsigned int temp, tempReg, x, y;

	pr_debug("dbmx21fb_set_cursor:BLINK_RATE=%x\n", c->blink_rate);
	pr_debug("dbmx21fb_set_cursor:width=%x\n", c->width);
	pr_debug("dbmx21fb_set_cursor:height=%x\n", c->height);

	x = c->startx << 16;
	if (c->state == LCD_CURSOR_ON)
		x |= CURSOR_ON_MASK;

#ifdef FBCON_HAS_CFB4
	else if (c->state == LCD_CURSOR_REVERSED)
		x |= CURSOR_REVERSED_MASK;
	else if (c->state == LCD_CURSOR_ON_WHITE)
		x |= CURSOR_WHITE_MASK;
#elif defined(FBCON_HAS_CFB8)||defined(FBCON_HAS_CFB16)
	else if (c->state == LCD_CURSOR_INVERT_BGD)
		x |= CURSOR_INVERT_MASK;
	else if (c->state == LCD_CURSOR_AND_BGD)
		x |= CURSOR_AND_BGD_MASK;
	else if (c->state == LCD_CURSOR_OR_BGD)
		x |= CURSOR_OR_BGD_MASK;
	else if (c->state == LCD_CURSOR_XOR_BGD)
		x |= CURSOR_XOR_BGD_MASK;
#endif
	else
		x = c->startx;

	y = c->starty;

	temp = (unsigned int) x | (unsigned int) y;
	LCDC_LCPR = temp;

	temp = (unsigned int) ((c->width << 8) | (c->height));
	tempReg = (unsigned int) ((temp << 16) | c->blink_rate);

	LCDC_LCWHBR = tempReg;

	if (c->blinkenable)
		dbmx21fb_set_cursor_blink(fb, c->blink_rate);
	pr_debug("dbmx21fb_set_cursor: LCDC_LCWHBR=%x\n", LCDC_LCWHBR);
}

 /*****************************************************************************
 * Function Name: dbmx21fb_set_cursor_blink()
 *
 * Input:   fb  : frame buffer database
 *	 blink	: input blink frequency of cursor	
 *
 * Value Returned: VOID
 *
 * Functions Called: VOID	
 *	
 * Description: Set blink frequency of hardware cursor
 *
 * Modification History:
 *		10 DEC,2001, Zhang Juan
******************************************************************************/
static void
dbmx21fb_set_cursor_blink(struct dbmx21fb_info *fb, int blink)
{
	struct dbmx21fb_cursor *c = &fb->cursor;
	unsigned long temp, tempReg;
	unsigned long PCD, XMAX, YMAX, PCLKDIV2;
	unsigned long tempMicroPeriod;

	if (!c) {
		printk(KERN_ERR "dbmx21fb_set_cursor: dbmx21fb_cursor null\n");
		return;
	}

	c->blink_rate = blink;

	tempReg = LCDC_LSR;
	XMAX = (tempReg & XMAX_MASK) >> 20;
	YMAX = tempReg & YMAX_MASK;
	tempReg = CRM_PCDR;
	PCLKDIV2 = (tempReg & PCLKDIV2_MASK) >> 22;
	tempReg = LCDC_LPCR;
	PCD = tempReg & PCD_MASK;

	temp = (PCLKDIV2 + 1);

	if (!blink) {
		/* disable the blinking cursor function when frequency is 0 */
		tempReg = LCDC_LCWHBR;
		tempReg &= CURSORBLINK_DIS_MASK;
		LCDC_LCWHBR = tempReg;
	} else {

		tempMicroPeriod = temp * XMAX * YMAX * (PCD + 1);
		temp = 96 * 10000000 / (blink * tempMicroPeriod);
		tempReg = LCDC_LCWHBR;
		tempReg |= CURSORBLINK_EN_MASK;
		tempReg |= temp;
		LCDC_LCWHBR = tempReg;
		pr_debug("dbmx21fb_set_cursor_blink: LCDC_LCWHBR=%x\n", 
			LCDC_LCWHBR);
	}
}

 /*****************************************************************************
 * Function Name: dbmx21fb_set_cursor_state()
 *
 * Input:   fb  : frame buffer database
 *	 state	: The status of the cursor to be set. e.g.
 *            		LCD_CURSOR_OFF
 *                      LCD_CURSOR_ON
 *                      LCD_CURSOR_REVERSED
 *                      LCD_CURSOR_ON_WHITE
 *                      LCD_CURSOR_OR_BGD
 *                      LCD_CURSOR_XOR_BGD
 *                      LCD_CURSOR_AND_BGD
 *
 * Value Returned: VOID
 *
 * Functions Called: VOID	
 *	
 * Description: Set state of cursor
 *
 * Modification History:
 *		10 DEC,2001, Zhang Juan
******************************************************************************/
static void
dbmx21fb_set_cursor_state(struct dbmx21fb_info *fb, unsigned int state)
{
	struct dbmx21fb_cursor *c = &fb->cursor;
	unsigned int temp;

	c->state = state;
	temp = LCDC_LCPR;
	temp &= CURSOR_OFF_MASK;

	if (state == LCD_CURSOR_OFF)
		temp = temp;
	else if (state == LCD_CURSOR_ON)
		temp |= CURSOR_ON_MASK;
#ifdef FBCON_HAS_CFB4
	else if (state == LCD_CURSOR_REVERSED)
		temp |= CURSOR_REVERSED_MASK;
	else if (state == LCD_CURSOR_ON_WHITE)
		temp |= CURSOR_WHITE_MASK;
#elif defined(FBCON_HAS_CFB8) || defined(FBCON_HAS_CFB16)
	else if (state == LCD_CURSOR_INVERT_BGD)
		temp |= CURSOR_INVERT_MASK;
	else if (state == LCD_CURSOR_OR_BGD)
		temp |= CURSOR_OR_BGD_MASK;
	else if (state == LCD_CURSOR_XOR_BGD)
		temp |= CURSOR_XOR_BGD_MASK;
	else if (state == LCD_CURSOR_AND_BGD)
		temp |= CURSOR_AND_BGD_MASK;
#endif
	LCDC_LCPR = temp;
	pr_debug("dbmx21fb_set_cursor_state: LCDC_LCPR=%x\n", LCDC_LCPR);
}

/*****************************************************************************
 * Function Name: dbmx21fb_cursor()
 *
 * Input:   fb     		: frame buffer database
 *
 * Value Returned: 	cursor : The structure of hardware cursor
 *
 * Functions Called: 	dbmx21fb_set_cursor()
 *			dbmx21fb_set_cursor_state()
 *	
 * Description: The entry for display switch to operate hardware cursor
 *
 * Modification History:
 *		10 DEC,2001, Zhang Juan
******************************************************************************/
static void
dbmx21fb_cursor(struct display *p, int mode, int x, int y)
{
	struct dbmx21fb_info *fb = (struct dbmx21fb_info *) p->fb_info;
	struct dbmx21fb_cursor *c = &fb->cursor;

	if (c == 0)
		return;

	x *= fontwidth(p);
	y *= fontheight(p);

	c->startx = x;
	c->starty = y;

	switch (mode) {
	case CM_ERASE:
		dbmx21fb_set_cursor_state(fb, LCD_CURSOR_OFF);
		break;
	case CM_DRAW:
	case CM_MOVE:
		c->state = LCD_CURSOR_ON;
		dbmx21fb_set_cursor(fb);
		dbmx21fb_set_cursor_state(fb, c->state);
		break;
	}
}

/*****************************************************************************
 * Function Name: dbmx21fb_set_font()
 *
 * Input:   display	: console datebase
 * 	    width	: The new width of cursor to be set.
 * 	    height	: The new height of cursor position to be set
 *	
 * Value Returned: int	: Return status.If no error, return 0.
 *
 * Functions Called: dbmx21fb_set_cursor()
 *		     dbmx21fb_set_cursor_color()
 *	
 * Description: Set  font for cursor
 *
 * Modification History:
 *		10 DEC,2001, Zhang Juan
******************************************************************************/
static int
dbmx21fb_set_font(struct display *d, int width, int height)
{
	struct dbmx21fb_info *fb = (struct dbmx21fb_info *) d->fb_info;
	struct dbmx21fb_cursor *c = &fb->cursor;

	if (!d) {
		printk(KERN_ERR "dbmx21fb_set_font display is null\n");
		return -1;
	}
	
	if (c) {
		if (!width || !height) {
			width = 16;
			height = 16;
		}

		c->width = width;
		c->height = height;

		dbmx21fb_set_cursor(fb);
		dbmx21fb_set_cursor_color(fb, cursor_color_map,
					  cursor_color_map, cursor_color_map);
	} else {
		pr_debug("set cursor failed, cursor == null\n");
	}

	return 1;
}
#endif	/* HARDWARE_CURSOR */