/*
 * linux/drivers/usb/usb-ohci-omap1510.c
 *
 * USB OHCI Support for OMAP1510/161x
 *
 * Author: MontaVista Software, Inc. <source@mvista.com>
 *	Copyright (c) 2004 MontaVista Software, Inc.
 *	Copyright (c) Russell King <rmk@arm.linux.org.uk>
 *	Copyright (c) Brad Parkers <brad@heeltoe.com>
 *
 * The outline of this code was taken from Brad Parkers <brad@heeltoe.com>
 * original OHCI driver modifications, and reworked into a cleaner form
 * by Russell King <rmk@arm.linux.org.uk>.
 */
/*
 * Changes:
 *  MontaVista Software Inc. <source@mvista.com>
 *  - Added Support for OMAP OSK USB Host with integrated USB transceiver.
 */
#include <linux/module.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/ioport.h>
#include <linux/interrupt.h>
#include <linux/slab.h>
#include <linux/pci.h>		/* for pci_pool_* prototypes */
#include <linux/usb.h>

#ifdef CONFIG_OMAP_OSK
#include <linux/i2c.h>
#include <linux/tps65010.h>
#endif

#include <asm/hardware.h>
#include <asm/irq.h>
#include <asm/io.h>

#include "usb-ohci.h"

#define OMAP1510_LB_OFFSET (0x30000000UL)

#ifdef CONFIG_ARCH_OMAP1610
#define OMAP1610_SET_FUNC_MUX_CTRL(mode,reg,bit) outl((inl(reg)&~(0x7<<bit))|(mode<<bit),reg)
#define OMAP1610_CONFIRM_MUX_SETUP() outl(0xeaef,COMP_MODE_CTRL_0)
#define open_drain(bit,reg) outl(inl(reg)|(1<<bit),reg)
#endif

extern int
__devinit hc_add_ohci(struct pci_dev *dev, int irq, void *membase,
		      unsigned long flags, ohci_t ** ohci,
		      const char *name, const char *slot_name);
extern void hc_remove_ohci(ohci_t * ohci);

static ohci_t *omap1510_ohci;

/* bogus pci_dev structure */
static struct pci_dev bogus_pcidev;

static int __devinit omap1510_ohci_configure(void);
static void omap1510_ohci_release(void);

#ifdef CONFIG_OMAP_OSK
static int initstate_region = 0;

static void
tps65010_enable_gpio1(void)
{
	u8 reg;

	reg = tps65010_read(TPS65010_I2C_DEFGPIO);
	tps65010_write(reg | DEFGPIO_GPIO1_VAL | DEFGPIO_IO1,
		       TPS65010_I2C_DEFGPIO);
}

static void
tps65010_disable_gpio1(void)
{
	u8 reg;

	reg = tps65010_read(TPS65010_I2C_DEFGPIO);
	tps65010_write(reg & ~DEFGPIO_IO1, TPS65010_I2C_DEFGPIO);
}
#endif

static int __devinit
omap1510_ohci_configure(void)
{
#ifndef CONFIG_ARCH_OMAP1610
	/* TO DO:  make a proper header file for all of these registers. */
#ifdef CONFIG_OMAP_INNOVATOR
	volatile unsigned char *fpga_usb_host_ctrl =
	    (unsigned char *) 0xe800020c;
#endif
	volatile unsigned short *apll_ctrl_reg = (unsigned short *) 0xfffe084c;
	volatile unsigned short *dpll_ctrl_reg = (unsigned short *) 0xfffe083c;
	volatile unsigned short *soft_req_reg = (unsigned short *) 0xfffe0834;
	volatile unsigned short *clock_ctrl_reg = (unsigned short *) 0xfffe0830;
	volatile unsigned long *mod_conf_ctrl_0 = (unsigned long *) 0xfffe1080;

	volatile unsigned long *lb_clock_div = (unsigned long *) 0xfffec10c;
	volatile unsigned short *lb_mmu_cntl_reg =
	    (unsigned short *) 0xfffec208;
	volatile unsigned short *lb_mmu_lock_reg =
	    (unsigned short *) 0xfffec224;
	volatile unsigned short *lb_mmu_ld_tlb_reg =
	    (unsigned short *) 0xfffec228;
	volatile unsigned short *lb_mmu_cam_h_reg =
	    (unsigned short *) 0xfffec22c;
	volatile unsigned short *lb_mmu_cam_l_reg =
	    (unsigned short *) 0xfffec230;
	volatile unsigned short *lb_mmu_ram_h_reg =
	    (unsigned short *) 0xfffec234;
	volatile unsigned short *lb_mmu_ram_l_reg =
	    (unsigned short *) 0xfffec238;

	int tlb;
	unsigned long lbaddr, physaddr;
#endif
	int ret;

#ifndef CONFIG_ARCH_OMAP1610

#define APLL_CTRL_REG_APLL_NDPLL_SWITCH		0x0001
#define DPLL_CTRL_REG_PLL_ENABLE		0x0010
#define DPLL_CTRL_REG_LOCK			0x0001
#define SOFT_REQ_REG_DPLL_REQ			0x0001
#define CLOCK_CTRL_REG_USB_MCLK_EN		0x0010
#define MOD_CONF_CTRL_0_USB_HOST_HHC_UHOST_EN	0x00000200

	*apll_ctrl_reg &= ~APLL_CTRL_REG_APLL_NDPLL_SWITCH;
	*dpll_ctrl_reg |= DPLL_CTRL_REG_PLL_ENABLE;
	*soft_req_reg |= SOFT_REQ_REG_DPLL_REQ;
	while (!(*dpll_ctrl_reg & DPLL_CTRL_REG_LOCK)) ;
	*clock_ctrl_reg |= CLOCK_CTRL_REG_USB_MCLK_EN;
	*ARM_IDLECT2 |= (1 << EN_LBFREECK) | (1 << EN_LBCK);
	*mod_conf_ctrl_0 |= MOD_CONF_CTRL_0_USB_HOST_HHC_UHOST_EN;

#ifdef CONFIG_OMAP_INNOVATOR
	*fpga_usb_host_ctrl |= 0x20;
#endif

	*lb_clock_div = (*lb_clock_div & 0xfffffff8) | 0x4;
	*lb_mmu_cntl_reg = 0x3;
	udelay(200);
	for (tlb = 0; tlb < 32; tlb++) {
		lbaddr = tlb * 0x00100000 + OMAP1510_LB_OFFSET;
		physaddr = tlb * 0x00100000 + PHYS_OFFSET;
		*lb_mmu_cam_h_reg = (lbaddr & 0x0fffffff) >> 22;
		*lb_mmu_cam_l_reg = ((lbaddr & 0x003ffc00) >> 6) | 0xc;
		*lb_mmu_ram_h_reg = physaddr >> 16;
		*lb_mmu_ram_l_reg = (physaddr & 0x0000fc00) | 0x300;
		*lb_mmu_lock_reg = tlb << 4;
		*lb_mmu_ld_tlb_reg = 0x1;
	}
	*lb_mmu_cntl_reg = 0x7;
	udelay(200);
#else				/* CONFIG_ARCH_OMAP1610 */
#define MOD_CONF_CTRL_0_USB_HOST_HHC_UHOST_EN	0x00000200

	/* set up pinout */

#ifdef CONFIG_OMAP_OSK
	/* Configure USB Host connection with integrated USB transceiver */

	/* USB pin group 0 configuration */
	OMAP1610_SET_FUNC_MUX_CTRL(4,FUNC_MUX_CTRL_D,3);  /*USB.PUEN set to High-Z*/
	open_drain(5,PULL_DWN_CTRL_3);

	/* activate USB.DM and USB.DP and internal transceiver, disable internal pull-downs */
	outl(0x186,USB_TRANSCEIVER_CTRL);
#endif

	OMAP1610_CONFIRM_MUX_SETUP();


	/* 
	 * Set up USB Host without OTG function.
	 * OTG registers still must be configured 
	 */

	/* 
	 * RESET_CONTROL.0 = 1 (CONF_OCP_RESET_R) 
	 * RESET_CONTROL.2 = 1 (CONF_ARMIO_RESET_R)
	 */
	outl(inl(RESET_CONTROL) | 5, RESET_CONTROL);

	/* ARM_RSTCT2.0 = 1(PER_EN) */
	*ARM_RSTCT2 |= 1;

	/* 
	 * ARM_IDLECT3.0 = 1 (EN_OCPI_CK) 
	 * ARM_IDLECT3.1 = 0 (IDLOCPI_ARM) make OCPI idle when MPU is idle
	 */
	*ARM_IDLECT3 = ((*ARM_IDLECT3) & ~(1 << 1)) | 1;

	/* Enable everything in OCPI. Seems like everything must be enabled */
	*OCPI_PROT &= ~0xff;
	*OCPI_SEC &= ~0x7f;

	/* 
	 * CLOCK_CTRL_REG.5 = 0 (DIS_USB_PVCI_CLK)
	 * CLOCK_CTRL_REG.4 = 1 (USB_MCLK_EN)
	 */
	outl((inl(CLOCK_CTRL_REG) & ~(1 << 5)), CLOCK_CTRL_REG);
	outl((inl(CLOCK_CTRL_REG) | (1 << 4)), CLOCK_CTRL_REG);

	/*
	 * enable 48 and 12 MHz clocks
	 * MOD_CONF_CTRL_0.9 = 1 (CONF_MOD_USB_HOST_HHC_UHOST_EN_R) 
	 * enable uart on USB port 1
	 * MOD_CONF_CTRL_0.11 = 0 (CONF_MOD_USB_HOST_UART_SELECT_R) 
	 */
	outl((inl(MOD_CONF_CTRL_0) | (1 << 9)), MOD_CONF_CTRL_0);
	outl((inl(MOD_CONF_CTRL_0) & ~(1 << 11)), MOD_CONF_CTRL_0);

	/*
	 * SOFT_REQ_REG.8 = 1 (SOFT_USB_OTG_DPLL_REQ)
	 * SOFT_REQ_REG.4 = x (USB_REQ_EN) USB Client HW DPLL req
	 * SOFT_REQ_REG.3 = 1 (SOFT_USB_REQ) 
	 * SOFT_REQ_REG.0 = 1 (SOFT_DPLL_REQ) ULPD_PLL clock req
	 */
	outl((inl(SOFT_REQ_REG) | (1 << 0) | (1 << 3) | (1 << 8)),
	     SOFT_REQ_REG);

	/* 
	 * USB Host clock HW
	 * SOFT_DISABLE_REQ_REG.6 = 0 (DIS_USB_HOST_DPLL_REQ) 
	 * SOFT_DISABLE_REQ_REG.3 = 0 (DIS_PERIPH_REQ) 
	 */
	outl((inl(SOFT_DISABLE_REQ_REG) & ~((1 << 6) | (1 << 3))),
	     SOFT_DISABLE_REQ_REG);

#ifdef CONFIG_OMAP_OSK
	outl((3 << 16), OTG_SYSCON_1);	/* set USB0_TRX_MODE to 3 */
#endif

#ifdef CONFIG_OMAP_OSK
	outl((1 << 30) | (4 << 16) | (1 << 8) | 16, OTG_SYSCON_2);
#endif
	/* 
	 * OTG_CTRL.18 = x (BSESSVLD) 
	 * host attached (VBUS present)
	 * other bits must be 0
	 */
#ifdef CONFIG_OMAP_OSK
	outl(0, OTG_CTRL);
#else
	outl((inl(OTG_CTRL) & (1 << 18)), OTG_CTRL);
#endif

	outl(0, OTG_IRQ_EN);

#endif				/* CONFIG_ARCH_OMAP1610 */
	/*
	 * Fill in some fields of the bogus pci_dev.
	 */
	memset(&bogus_pcidev, 0, sizeof (struct pci_dev));
	strcpy(bogus_pcidev.name, "OMAP1510 OHCI");
	strcpy(bogus_pcidev.slot_name, "builtin");
	bogus_pcidev.resource[0].name = "OHCI Operational Registers";
	bogus_pcidev.resource[0].start = 0xfffba000;
	bogus_pcidev.resource[0].end = 0xfffba000 + 4096;	/* REVISIT */
	bogus_pcidev.resource[0].flags = 0;
	bogus_pcidev.irq = IH2_BASE + 6;

	/*
	 * Initialise the generic OHCI driver.
	 */
	ret = hc_add_ohci(&bogus_pcidev, bogus_pcidev.irq,
			  (void *) bogus_pcidev.resource[0].start, 0,
			  &omap1510_ohci, "usb-ohci", "omap1510");

	return ret;
}

static void
omap1510_ohci_release(void)
{
	/* REVISIT: Need to properly shut off clocks here. */
	volatile unsigned long *mod_conf_ctrl_0 = (unsigned long *) 0xfffe1080;

	hc_remove_ohci(omap1510_ohci);

	*mod_conf_ctrl_0 &= ~MOD_CONF_CTRL_0_USB_HOST_HHC_UHOST_EN;

#ifdef CONFIG_ARCH_OMAP1610
	outl(inl(OTG_SYSCON_2) & ~(1 << 8), OTG_SYSCON_2);
#endif
}

#ifdef CONFIG_OMAP_OSK
/* Release IO region */
static void
omap_release_io(void)
{
	if (initstate_region) {
		release_region(OTG_BASE, OTG_IOSIZE);
		initstate_region = 0;
	}
}
#endif

static int __init
omap1510_ohci_init(void)
{
	int ret = 0;

#ifdef CONFIG_OMAP_OSK
	if (!request_region(OTG_BASE, OTG_IOSIZE, "OMAP OSK USB OHCI")) {
		err("OTG is already in use");
		omap_release_io();
		return -ENODEV;
	}
	initstate_region = 1;

	tps65010_enable_gpio1();
#endif

	ret = omap1510_ohci_configure();

	return 0;
}

static void __exit
omap1510_ohci_exit(void)
{
	omap1510_ohci_release();

#ifdef CONFIG_OMAP_OSK
	omap_release_io();
	tps65010_disable_gpio1();
#endif
}

module_init(omap1510_ohci_init);
module_exit(omap1510_ohci_exit);

MODULE_LICENSE("GPL");
