/* **************************************************************************
 * linux/drivers/usb/device/bi/musbhsfc-opb.c --
 * Inventra Musbhsfc-opb USB bus interface driver.
 *
 * This was derived from the gen.c file.
 *
 * Copyright (c) 2000, 2001, 2002 Lineo
 *
 * By:
 *      Stuart Lynne <sl@lineo.com>,
 *      Tom Rushworth <tbr@lineo.com>,
 *      Bruce Balden <balden@lineo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 *
 * (C) Copyright IBM Corp. 2004 Christophe Lombard <christophe_lombard@fr.ibm.com>
 *
 *
 * **************************************************************************/

#include <linux/config.h>
#include <linux/module.h>

#include "../usbd-export.h"
#include "../usbd-build.h"
#include "../usbd-module.h"

MODULE_AUTHOR("source@mvista.com");
MODULE_DESCRIPTION("Musbhsfc-opb Inventra USB Device Bus Interface");
MODULE_LICENSE("GPL");
USBD_MODULE_INFO("musbhsfc-opb.c 0.0.1");

#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/init.h>

#include <asm/atomic.h>
#include <asm/io.h>

#include <linux/netdevice.h>

#include <asm/irq.h>
#include <asm/system.h>

#include <asm/types.h>

#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/pgtable.h>
#include <linux/delay.h>

#include "../usbd.h"
#include "../usbd-func.h"
#include "../usbd-bus.h"
#include "../usbd-inline.h"
#include "../usbd-debug.h"
#include "usbd-bi.h"

#include "musbhsfc-opb.h"
#include <platforms/ibm440ep.h>

/* **************************************************************************
 *
 * **************************************************************************/
#define MIN(a,b) ((a) < (b) ? (a) : (b))
#define MAX(a,b) ((a) > (b) ? (a) : (b))

// ioremapped ohci_base
static void *mem_base;

#define musbhsfc_readb(addr) in_8((volatile u8 *)(mem_base + addr))
#define musbhsfc_writeb(addr,b) out_8((volatile u8 *)(mem_base + addr), (b))
#define musbhsfc_readw(addr) in_be16((volatile u16 *)(mem_base + addr))
#define musbhsfc_readl(addr) in_be32((volatile u32 *)(mem_base + addr))
#define musbhsfc_writew(addr, b) out_be16((volatile u16 *)(mem_base + addr),(b))
#define musbhsfc_writel(addr, b) out_be32((volatile u32 *)(mem_base + addr),(b))

/* Some kind of debugging output... */
//#define DEBUGLEVEL 1
#undef DEBUGLEVEL

#if DEBUGLEVEL
#define UDC_DBG(lvl,args,...) \
     if (lvl <= DEBUGLEVEL) { \
        printk(KERN_INFO "musbhsfc-opb.c %d: " args "\n", \
                __LINE__, ##__VA_ARGS__); \
     }
#else                           /* The bugs still exists... */
#define UDC_DBG(lvl,args,...)
#endif

// DMA Mode
#define USEDMA 1
//#undef USEDMA

static struct usb_device_instance *udc_device;  // required for the interrupt handler

/*
   PHYSICAL and LOGICAL Endpoints:

   Logical Endpoint 0 corresponds to Physical Endpoint 0
   Logical Endpoints 1 and 2 correspond to Physical Endpoint 1
   Logical Endpoints 3 and 4 correspond to Physical Endpoint 2
   Logical Endpoints 5 and 6 correspond to Physical Endpoint 3

   In this device driver ep means logical endpoint
**/

/*
 * ep_endpoints - map physical endpoints to logical endpoints
 * Inlude endpoint 0 + all endpoints
 */
static struct usb_endpoint_instance *ep_endpoints[USBBI_UDC_MAX_LOGICAL_ENDPOINTS];

/* bits set for config mode byte for the PHYSICAL endpoint id */
static struct {
  char    configByteIN;         // Config IN use by musbhsfc_setConfig function
  char    configByteOUT;        // Config OUT use by musbhsfc_setConfig function
  char    epPhysical;           // endpoint physical index
  char    spare1;               // spare to align all struct on full word.
  short   spare2;               // spare to align all struct on full word.
  short   endpointIntMask;      // Interrupt mask
} endpoints_configByte[USBBI_UDC_MAX_LOGICAL_ENDPOINTS];

/* data structure used for the USB device instance */
static struct {
  short   inFlags;              /* bits set by in interrupts */
  short   outFlags;             /* bits set by out interrupts */
  char    usbFlags;             /* bits set by USB interrupts */
  char    spare1;               // spare to align all struct on full word.
  short   spare2;               // spare to align all struct on full word.
#ifdef USEDMA
  int     dma2plb4;             // bits set by DMA to plb4 status */
#endif
} musbhsfc_data;

static struct urb *ep0_urb;
static unsigned char usb_address = 0;
static char first_connection = 0;

#ifdef USEDMA
static char *ps_alloc;
#endif

/*

  USB 2.0 and DMA

  Only for the USB 2.0 Device, use the dma access
  Configuration:

       Channel                       DMA Internal Source
       DMA to plb4 Channel 0         USB 2.0 Device Endpoint 1 IN
       DMA to plb4 Channel 1         USB 2.0 Device Endpoint 2 IN
       DMA to plb4 Channel 2         USB 2.0 Device Endpoint 1 OUT
       DMA to plb4 Channel 3         USB 2.0 Device Endpoint 2 OUT

       udc_setup_ep(...)
         Register DMA interrupt in the kernel handler (only for
         the endpoints 1 and 2 in the bulk mode. It'not necessary for the others
         mode: interrupt and isochronous)

       musbhsfc_configureDMA_OUT(...)
         For USB 2.0 device, we use DMA transfert to get data from the FIFO
         The size of the data is unknown, the the reception of a packet of less than
         the maximum packet can be used to indicate the end of the transfert. This can be
         done by using he DMA mode 1.
         DMA mode 1 will be used to transfert data.
         Enable AUtoClear and DMAEnab and DMA mode bits in the OutCSR register
         (High Byte, bits D0, D2 and D4) and the interrupt for the endpoint should be
         enabled.

         The MUSBHSFC-OPB will then set the DMA request line high whenever it receives a
         packet of the maximum packet size from the host. During the last burst read by the
         DMA controller of any packet of the maximum packet size, the DMA Request line will
         de-assert (see the DMA Read timing diagram in Section 17.4 of the MUSBHSFC-OPB
         Product Specification). The DMA controller should respond to the DMA Request
         line being de-asserted by terminating the transfer when the current programmed burst
         count completes. The OutPktRdy bit (OutCSR(Low byte).D7) will be automatically
         cleared when the burst completes. The DMA Request line will then go high again
         when another packet of the maximum packet size is available to be read, and the
         above process will then be repeated. At the end of each MaxP packet of data, the
         DMA controller will terminate the current burst in response to the de-assertion of
         the DMA Request line and will then remain in the inactive state.

         When a packet less than the maximum packet size is received, no DMA request will be
         generated but the processor will be interrupted instead. The DMA controller will remain
         inactive since no DMA request has been generated. The processor should then read the
         OutCount register for the endpoint to determine the size of this short packet and then
         either read this packet manually or reprogram the DMA controller to read this packet.
         If the final transfer is to be performed by the DMA controller then the processor should
         program the DMA controller terminal count with the transfer size read from the OutCount
         register and manually start a non-burst DMA transfer. The DMA controller should
         then terminate the transfer when the terminal count is reached and interrupt the processor.
         The OutPktRdy bit will then need to be cleared by the processor.

       musbhsfc_readInterruptRegister(...)
         Read DMA to PLB4 Status Register to provide informations about DMA status

       musbhsfc_processOutEndpoint(...)
         For USB 2.0 device, we use DMA transfert to get data from the FIFO
            1. The DMA has finished previous transfert data
               We must send urb to the function device.
               The MUSBHSFC-OPB device indicates the last packet of the DMA transfert
               The size of the last packat is null or less than the maximum
               packet size.
               We must, in the DMA mode 1, manually clear the OutPktRdy bit and read
               the last packet in the FIFO.
            2. A new urb arrives from the host. Only for the endpoint 1 or 2 and in
               the bulk mode and > 16 bytes transfert

       udc_release_udc_irq(...)
         For the endpoint 1 or 2 in the bulk mode release the DMA to plb4 irq

*/
static char usb2_device     = 0;    // USB Device 2.0 or 1.1
static char speedIndication = 0;    // High or Full speed
#ifdef USEDMA
static char dmaConfigured   = 0;    // DMA has been configured
#endif

extern unsigned int udc_interrupts;

/* the three states endpoint 0 can be in */
#define EP0_IDLE 0
#define EP0_RX 1
#define EP0_TX 2

#define ENDPOINT_DATA0 1
#define ENDPOINT_STALL 2
#define ENDPOINT_UNSTALL 4


/* **************************************************************************
 * IO
 * **************************************************************************/

/* **************************************************************************
 * musbhsfc_setAddress - Set Address/Enable command to USB Device
 *
 * @address: address number
 * Return nothing.
 * **************************************************************************/
void musbhsfc_setAddress (char address)
{
   char byte;
   UDC_DBG(2, "musbhsfc_setAddress %i", address);

   byte = address & USBBI_FADDR_ADDR_MASK;
   byte |= USBBI_FADDR_UPDATE;

   musbhsfc_writeb(USBBI_FADDR_ADDR, byte);
}


/* **************************************************************************
 * musbhsfc_selectEndpoint - Select Endpoint (through index register)
 *
 * @ep: logical endpoint
 * Return the physical endpoint.
 *
 * **************************************************************************/
unsigned char musbhsfc_selectEndpoint (unsigned int ep)
{
   UDC_DBG(2, "musbhsfc_selectEndpoint %i (logical endpoint)", ep);

   musbhsfc_writeb(USBBI_INDEX_ADDR, endpoints_configByte[ep].epPhysical);
   return endpoints_configByte[ep].epPhysical;
}

#ifdef USEDMA
/* **************************************************************************
 * musbhsfc_enableDMA_OUT - Configure and enable DMA
 *
 * @ep: logical endpoint
 * Return nothing.
 *
 * DMA Mode:
 * For USB 2.0 device, we use DMA transfert to get data from the FIFO
 * The size of the data is unknown, the the reception of a packet of less than
 * the maximum packet can be used to indicate the end of the transfert. This can be
 * done by using he DMA mode 1.
 *
 * **************************************************************************/
void musbhsfc_enableDMA_OUT (unsigned int ep)
{
  unsigned long physicalAddrDest;
  unsigned char epPhysical;

  physicalAddrDest = iopa((unsigned long)ps_alloc);

  // physical endpoint
  epPhysical = endpoints_configByte[ep].epPhysical;

  // Configure DMA To plb4 core
  if (epPhysical == 1) // Endpoint 1
    enable_dma2pl4_peripheral_to_memory(2,
                                        (unsigned long)(USBBI_COMMON_REGISTERS + (USBBI_FIFO_EP0_ADDR + (epPhysical << 2))),
                                        physicalAddrDest,
                                        USBBI_MAX_BUFFER_SIZE);
  else // Endpoint 2
    enable_dma2pl4_peripheral_to_memory(3,
                                        (unsigned long)(USBBI_COMMON_REGISTERS + (USBBI_FIFO_EP0_ADDR + (epPhysical << 2))),
                                        physicalAddrDest,
                                        USBBI_MAX_BUFFER_SIZE);
}

/* **************************************************************************
 * musbhsfc_configureDMA_OUT - Configure musbhsfc core for DMA access
 *
 * @ep: logical endpoint
 * Return nothing.
 *
 * DMA Mode:
 * For USB 2.0 device, we use DMA transfert to get data from the FIFO
 * The size of the data is unknown, the the reception of a packet of less than
 * the maximum packet can be used to indicate the end of the transfert. This can be
 * done by using he DMA mode 1.
 *
 * **************************************************************************/
void musbhsfc_configureDMA_OUT (unsigned int ep)
{
  dmaConfigured = 1;

  if (ps_alloc == NULL)
     ps_alloc = (unsigned char *) kmalloc(USBBI_MAX_BUFFER_SIZE, GFP_KERNEL);

  if(ps_alloc == NULL)
  {
    printk(KERN_ERR "DMA: FATAL ERROR! kmalloc failed on ps_alloc.\n");
    dmaConfigured = 0;
    return;
  }

  // Configure DMA on the USB Device: Autoclear + DMA Enable + DMA Mode 1
  musbhsfc_writeb(USBBI_OUTCSRH_ADDR,
                  musbhsfc_readb(USBBI_OUTCSRH_ADDR) |
                  USBBI_OUTCSRH_AUTOCLR |
                  USBBI_OUTCSRH_DMAENA  |
                  USBBI_OUTCSRH_DMAMODE);
}
#endif

/* **************************************************************************
 * musbhsfc_setConfig - Configure the USB device with Mode
 *                      (ISO/BULK/INT and IN/OUT/IN_OUT)
 *                    + MAX Packet size
 *                    - for Out EP, clear the data toggle flag.
 * @ep: logical endpoint
 * Return nothing.
 *
 * **************************************************************************/
void musbhsfc_setConfig (unsigned int ep)
{
  if ( (ep_endpoints[ep]) && (ep != 0))
  {
    UDC_DBG(2, "musbhsfc_setConfig %i", ep);

    // Select Endpoint (through index register)
    musbhsfc_selectEndpoint(ep);

    // Configure Control status Register for IN/OUT endpoint
    // (InCSR/OutCSR registers)
    // Maximum packet size for IN/OUT endpoint  (InMasxP/OutMaxP registers)
    switch (ep_endpoints[ep]->endpoint_address & USB_ENDPOINT_DIR_MASK)
    {
      case USB_DIR_OUT:
         musbhsfc_writeb(USBBI_OUTCSRH_ADDR,
                         musbhsfc_readb(USBBI_OUTCSRH_ADDR) |
                         endpoints_configByte[ep].configByteOUT);

         musbhsfc_writew(USBBI_OUTMAXP_ADDR,
                         ep_endpoints[ep]->rcv_packetSize);

      break;

      case USB_DIR_IN:
         musbhsfc_writeb(USBBI_INCSRH_ADDR,
                         musbhsfc_readb(USBBI_INCSRH_ADDR) |
                         endpoints_configByte[ep].configByteIN);

         musbhsfc_writew(USBBI_INMAXP_ADDR,
                         ep_endpoints[ep]->tx_packetSize);
      break;
    }
  }
}


/* **************************************************************************
 * musbhsfc_enableInterrupt - Enable interrupts for the endpoint
 *
 * @ep: logical endpoint
 * Return nothing.
 * **************************************************************************/
void musbhsfc_enableInterrupt (unsigned int ep)
{
  UDC_DBG(2, "musbhsfc_enableInterrupt %i", ep);
  unsigned short intrine  = 0x0000;
  unsigned short introute = 0x0000;

  // Read previous values
  intrine  = musbhsfc_readw(USBBI_INTRINE_ADDR);
  introute = musbhsfc_readw(USBBI_INTROUTE_ADDR);

  // The Endpoint 0 uses a single interrupt
  if (ep ==0)
  {
     intrine |= 0x0001;
  }
  else
  { // all others endpoints
     if (ep_endpoints[ep])
     {
       switch (ep_endpoints[ep]->endpoint_address & USB_ENDPOINT_DIR_MASK)
       {
         case USB_DIR_OUT:
           introute |= endpoints_configByte[ep].endpointIntMask;
           break;

         case USB_DIR_IN:
           intrine |= endpoints_configByte[ep].endpointIntMask;
           break;
        }
     }
  }

  /* Enable endpoint interrupts by writing in registers */
  musbhsfc_writew(USBBI_INTRINE_ADDR, intrine);
  musbhsfc_writew(USBBI_INTROUTE_ADDR, introute);
}


/* **************************************************************************
 * musbhsfc_setEndpointStatus - Set the status of the endpoint
 *
 * @ep: logical endpoint
 * Return nothing.
 * **************************************************************************/
void musbhsfc_setEndpointStatus (unsigned int ep, unsigned char state)
{
  // Select Endpoint (through index register)
  musbhsfc_selectEndpoint(ep);

  // Endpoint 0 - Control endpoint
  if (ep == 0)
  {
    /* Control endpoint */
    if ((state & ENDPOINT_STALL) != 0)
    {
      /* Set SendStall and SERVICED OUT PKT RDY bits */
      musbhsfc_writeb(USBBI_CSR0_ADDR,
                      musbhsfc_readb(USBBI_CSR0_ADDR) |
                                     USBBI_CSR0_SENDSTALL |
                                     USBBI_CSR0_SVDOUTPKTRDY);
    }
    else if ((state & ENDPOINT_UNSTALL) != 0)
    {
      /* Clear SEND STALL bit */
      musbhsfc_writeb(USBBI_CSR0_ADDR,
                      musbhsfc_readb(USBBI_CSR0_ADDR) &
                                    ~USBBI_CSR0_SENDSTALL);
    }
  }
  else  // all others Endpoints
  {
    if ((state & ENDPOINT_STALL) != 0)
    {
      /* Set SendStall and SERVICED OUT PKT RDY bits */
      switch (ep_endpoints[ep]->endpoint_address & USB_ENDPOINT_DIR_MASK)
      {
      case USB_DIR_OUT:
      musbhsfc_writeb(USBBI_OUTCSR_ADDR,
                      musbhsfc_readb(USBBI_OUTCSR_ADDR) |
                                     USBBI_OUTCSR_SENDSTALL);

      break;

      case USB_DIR_IN:
      musbhsfc_writeb(USBBI_INCSR_ADDR,
                      musbhsfc_readb(USBBI_INCSR_ADDR) |
                                     USBBI_INCSR_SENDSTALL);

      break;
      }
    }
    else if ((state & ENDPOINT_UNSTALL) != 0)
    {
      /* Clear SEND STALL bit  */
      switch (ep_endpoints[ep]->endpoint_address & USB_ENDPOINT_DIR_MASK)
      {
      case USB_DIR_OUT:
      musbhsfc_writeb(USBBI_OUTCSR_ADDR,
                      musbhsfc_readb(USBBI_OUTCSR_ADDR) &
                                    ~USBBI_OUTCSR_SENDSTALL);

      break;

      case USB_DIR_IN:
      musbhsfc_writeb(USBBI_INCSR_ADDR,
                      musbhsfc_readb(USBBI_INCSR_ADDR) &
                                    ~USBBI_INCSR_SENDSTALL);

      break;
      }
    }
    else
    {
      /* Reset to DATA0 */
      switch (ep_endpoints[ep]->endpoint_address & USB_ENDPOINT_DIR_MASK)
      {
      case USB_DIR_OUT:
          musbhsfc_writeb(USBBI_OUTCSR_ADDR,
                          musbhsfc_readb(USBBI_OUTCSR_ADDR) |
                                         USBBI_OUTCSR_CLRDATATOG);

      break;

      case USB_DIR_IN:
        musbhsfc_writeb(USBBI_INCSR_ADDR,
                        musbhsfc_readb(USBBI_INCSR_ADDR) |
                                       USBBI_INCSR_CLRDATATOG);

      break;
      }
    }
  }
}

/* **************************************************************************
 * musbhsfc_readFIFO - read the FIFO
 *
 * @ep: logical endpoint
 * @nBytes: number of bytes to read from the OUT FIFO
 * @buffer: data received from the OUT FIFO
 *
 * Return nothing.
 * **************************************************************************/
void musbhsfc_readFIFO(unsigned int ep, unsigned int nBytes, void * buffer)
{
  unsigned char    epPhysical;
  unsigned char    *B_ptr;
  unsigned short   *S_ptr;
  unsigned int     nShorts = 0;

  if (nBytes)
  {
     // physical endpoint
     epPhysical = endpoints_configByte[ep].epPhysical;

     nShorts =  nBytes>>1;
     S_ptr = (unsigned short *) buffer;

     while (nShorts)
     {
        *S_ptr = musbhsfc_readw(USBBI_FIFO_EP0_ADDR + (epPhysical << 2));
         S_ptr++;
         --nShorts;
     }

     if (nBytes & 0x01)
     {
       B_ptr = (unsigned char *) S_ptr;
       *B_ptr = musbhsfc_readb(USBBI_FIFO_EP0_ADDR + (epPhysical << 2));
     }
  }
}

/* **************************************************************************
 * musbhsfc_writeFIFO - write the FIFO
 *
 * @ep: logical endpoint
 * @nBytes: number of bytes to write in the FIFO
 * @buffer: data to transmit to the IN FIFO
 *
 * Return nothing.
 * **************************************************************************/
void musbhsfc_writeFIFO(unsigned int ep, unsigned int nBytes, void * buffer)
{
  unsigned char    epPhysical;
  unsigned char    *B_ptr;
  unsigned short   *S_ptr;
  unsigned int     nShorts = 0;


  if (nBytes)
  {
     // physical endpoint
     epPhysical = endpoints_configByte[ep].epPhysical;

     nShorts = nBytes>>1;
     S_ptr = (unsigned short *) buffer;

     // write FIFO one byte at a time.
     while (nShorts)
     {
       musbhsfc_writew(USBBI_FIFO_EP0_ADDR + (epPhysical << 2), *S_ptr);
       S_ptr++;
       --nShorts;
     }
     if (nBytes & 0x01)
     {
        B_ptr = (unsigned char *) S_ptr;
        musbhsfc_writeb(USBBI_FIFO_EP0_ADDR + (epPhysical << 2), *B_ptr);
     }
  }
}

#define USB_DEVICE_INTERFACE_MASK  0x00000002
#define USB_DEVICE_MODE(reg)       ((reg & USB_DEVICE_INTERFACE_MASK))

/* **************************************************************************
 * musbhsfc_usbDevice - Check USB1.1 or USB2.0
 *
 * Check the usb device the user has selected
 * ************************************************************************/
void musbhsfc_usbDevice(void)
{
   u32 usbDeviceInterface = SDR_READ(SDR0_USB);

   /* Check the USB Device interface enabled: USB2.0 or USB 1.1 */
   if (!USB_DEVICE_MODE(usbDeviceInterface)) {
     usb2_device = 1;
   } else {
     usb2_device = 0;
   }
}


/* **************************************************************************
 * musbhsfc_speedMode - Check HighSpeed / Fullspeeed Mode
 *
 * Turn on the USB 1.1 or 2.0 Phy interface.
 * ************************************************************************/
void musbhsfc_speedMode(void)
{
  unsigned char byteHighspeed = musbhsfc_readb(USBBI_POWER_ADDR) & USBBI_POWER_HS_MODE;

  /* Check the USB Device interface enabled: USB2.0 or USB 1.1 */
  if (usb2_device) {

    if (byteHighspeed) {
       // High-Speed Configuration
       printk (KERN_INFO "USB Device 2.0 : High Speed Mode\n");
    } else {
       // Full-Speed Configuration
       printk (KERN_INFO "USB Device 2.0 : Full Speed Mode\n");
    }

   } else  {
     /* USB 1.1 device selected on 440EP core */
     /* USB 1.1 Phy interface */
     musbhsfc_writeb(USBBI_POWER_ADDR, byteHighspeed | USBBI_POWER_FS_PHY_ENAB);
     printk (KERN_INFO "USB Device 1.1 : Full Speed Mode\n");
  }
}

/* **************************************************************************
 * Control (endpoint zero)
 * **************************************************************************/

/* **************************************************************************
 * musbhsfc_processEp0IdleMode -
 * Process an interrupt for endpoint 0 when it is in "IDLE" mode.
 *
 *  Refer to Chapter 6.5.1 Idle Mode - Setup Phase of Control transfer
 *
 *  If Endpoint 0 is in IDLE state, the only valid reason an interrupt can be
 *  generated is as a result of the core receiving data from the USB bus.
 *  The service routine must check for this by testing the OutPktRdy bit.
 *  If this bit is set, then the core has received a SETUP packet.
 *  This must be unloaded from the FIFO and decoded to determine the action the
 *  core must take. Depending on the command contained within the SETUP packet,
 *  Endpoint 0 will enter one of three states:
 *
 *   If the command is a single packet transaction
 *    ( SET_ADDRESS, SET_INTERFACE etc ) without any data phase,
 *    the endpoint will remain in IDLE state.
 *   If the command has an OUT data phase ( SET_DESCRIPTOR etc ) the
 *    endpoint will enter RX state.
 *   If the command has an IN data phase ( GET_DESCRIPTOR etc ) the endpoint
 *    will enter TX state.
 *
 *
 * Return nothing.
 * **************************************************************************/
void musbhsfc_processEp0IdleMode(unsigned char csr0)
{
   unsigned short outCount;
   struct   usb_device_request *req;

   UDC_DBG(2, "musbhsfc_processEp0IdleMode. csr0: 0x%x", csr0);

   // Packet received ?
   if( csr0 & USBBI_CSR0_OUTPKTRDY)
   {
      /* This interrupt is generated
         When the core sets the OutPktRdy bit (CSR0.D7)
         after a valid token has been received and
         data has been written to the FIFO.
      **/

      /* OutCount is a 16-bit read-only register that holds the number
         of received data bytes in the packet in the OUT FIFO.
      **/
      outCount = musbhsfc_readw(USBBI_OUTCOUNT_ADDR);
      UDC_DBG(3, "musbhsfc_processEp0IdleMode. outCount: %i", outCount);

      if (outCount != 0)
      {
         req = &ep0_urb->device_request;

         // Unload FIFO - Read Only the 8 bytes (USB Device Request)
         musbhsfc_readFIFO(0, 8, (unsigned char *)req);

         UDC_DBG (2, "bmRequestType:%02x bRequest:%02x wValue:%04x wIndex:%04x wLength:%04x",
                  req->bmRequestType, req->bRequest, le16_to_cpu (req->wValue),
                  le16_to_cpu (req->wIndex), le16_to_cpu (req->wLength));

         // Command has Data Phase ?

         // No
         // Check the Number of bytes to transfert if there is a Data stage
         if (ep0_urb->device_request.wLength == 0)
         {
           /* Process Command
              The command is a single packet transaction
              ( SET_ADDRESS, SET_INTERFACE etc )
              without any data phase, the endpoint will remain in IDLE state.
            */

           /* set DataEnd and ServicedOutPktRdy
              to indicate last data packet and status phase */
           musbhsfc_writeb(USBBI_CSR0_ADDR, csr0 |
                                            USBBI_CSR0_DATAEND | 
                                            USBBI_CSR0_SVDOUTPKTRDY);

           /* usbd_recv_setup: ep0_recv_setup - called to indicate URB has been received
            Check if this is a setup packet, process the device request, put results
            back into the urb and return zero or non-zero to indicate success (DATA)
            or failure (STALL).
            **/
           if (usbd_recv_setup (ep0_urb))
           {
             UDC_DBG(1, "musbhsfc_processEp0IdleMode: usb_recv_setup failed, stalling");
             udc_stall_ep (0);
             return;
           }

           // State -> IDLE for Eendpoint 0
           ep_endpoints[0]->state = EP0_IDLE;

           return;
         }

         // Yes
         // Set ServicedOutPktRdy
         musbhsfc_writeb(USBBI_CSR0_ADDR, csr0 | USBBI_CSR0_SVDOUTPKTRDY);

         /* usbd_recv_setup: ep0_recv_setup - called to indicate URB has been received
            Check if this is a setup packet, process the device request, put results
            back into the urb and return zero or non-zero to indicate success (DATA)
            or failure (STALL).
         **/
         if (usbd_recv_setup (ep0_urb))
         {
            UDC_DBG(1, "musbhsfc_processEp0IdleMode: usb_recv_setup failed, stalling");
            udc_stall_ep (0);
            return;
         }

         if ((ep0_urb->device_request.bmRequestType & USB_REQ_DIRECTION_MASK)
             == USB_REQ_DEVICE2HOST )
         {
             // Data Phase = IN -> State =TX
             ep_endpoints[0]->state = EP0_TX;
         }

         if ((ep0_urb->device_request.bmRequestType & USB_REQ_DIRECTION_MASK)
             == USB_REQ_HOST2DEVICE )
         {
            // Data Phase = OUT -> State =RX
            ep_endpoints[0]->state = EP0_RX;
         }
      }
   }
}


/* **************************************************************************
 * musbhsfc_processEp0TxMode -
 * Process an interrupt for endpoint 0 when it is in "TX" mode.
 *
 *  Refer to Chapter 6.5.2 TX Mode - IN Data Phase for Control transfer
 *
 *  The interrupt (csr0) indicates that the core has received an IN token and
 *  data from the FIFO has been sent.
 *  The firmware must respond to this either by placing more data in the FIFO
 *  if the host is still expecting more data1 or by setting the DataEnd bit
 *  to indicate that the data phase is complete.
 *  Once the data phase of the transaction has been completed, Endpoint 0
 *  should be returned to IDLE state to await the next control transaction.
 *
 * Return nothing.
 * **************************************************************************/
void musbhsfc_processEp0TxMode(void)
{
  UDC_DBG(2, "musbhsfc_processEp0TxMode. ep0_urb->actual_length: %i",
          ep0_urb->actual_length);

  unsigned int nBytes;
  unsigned int ep0_packetsize_max = udc_ep0_packetsize();
  unsigned char byte;
  unsigned char lastPacket = 0;

  // Determine number of bytes to send
  if (ep0_urb->actual_length <= ep0_packetsize_max)
  {
    // all bytes in the ep0_urb to send will be insert in the FIFO IN
    nBytes = ep0_urb->actual_length;
    ep0_urb->actual_length = 0;
    lastPacket = 1;
  }
  else
  {
    // not enough space in the FIFO
    nBytes = ep0_packetsize_max;
    ep0_urb->actual_length -= ep0_packetsize_max;
    lastPacket = 0;
  }

  // Write <= MAxP bytes to FIFO
  musbhsfc_writeFIFO(0, nBytes, ep0_urb->buffer);

  // Last Packet ?
  if (lastPacket)
  {
    // Yes
    UDC_DBG(2, "musbhsfc_processEp0TxMode. Last packet, ep0_urb->actual_length :%i",
             ep0_urb->actual_length);
    ep_endpoints[0]->state = EP0_IDLE;  // State -> IDLE for Eendpoint 0
    byte = USBBI_CSR0_INPKTRDY |
           USBBI_CSR0_DATAEND;     // Set InPktRdy & DataEnd
  }
  else
  {
    // NO
    UDC_DBG(2, "musbhsfc_processEp0TxMode. Not last packet, ep0_urb->actual_length :%i",
             ep0_urb->actual_length);
    ep0_urb->buffer = ep0_urb->buffer + nBytes;
    byte = USBBI_CSR0_INPKTRDY;    // Set InPktRdy
  }

  musbhsfc_writeb(USBBI_CSR0_ADDR, byte);
}


/* **************************************************************************
 * musbhsfc_processEp0RxMode -
 * Process an interrupt for endpoint 0 when it is in "RX" mode.
 *
 *  Refer to Chapter 6.5.3 RX Mode - OUT Data Phase for Control transfer
 *
 *  The interrupt (csr0) indicates that a data packet has been received. The
 *  firmware must respond by unloading the received data from the FIFO.
 *  The firmware must then determine whether it has received all of the
 *  expected data1. If it has, the firmware should set the DataEnd bit
 *  and return Endpoint 0 to IDLE state.
 *  If more data is expected, the firmware should set the ServicedOutPktRdy
 *  bit (CSR0.D1) to indicate that it has read the data in the FIFO
 *  and leave the endpoint in RX state.
 *
 * Return nothing.
 * **************************************************************************/
void musbhsfc_processEp0RxMode(void)
{
  UDC_DBG(1, "musbhsfc_processEp0RxMode. ep0_urb->actual_length: %i",
          ep0_urb->actual_length);
}

/* **************************************************************************
 * musbhsfc_processEndpoint0 -
 * Endpoint 0 interrupt routine for MUSBHSFC-OPB firmware
 *
 * Return nothing.
 * **************************************************************************/
void musbhsfc_processEndpoint0(void)
{
  unsigned char csr0;

  /* Select Endpoint 0 */
  musbhsfc_selectEndpoint(0);

  /* Read csr0 Reg */
  csr0 = musbhsfc_readb(USBBI_CSR0_ADDR);

  UDC_DBG(2, "musbhsfc_processEndpoint0. csr0: 0x%x", csr0);

  /* Whenever the Endpoint 0 service routine is entered, the firmware must
     first check to see if the current control transfer has been ended due to
     either a STALL condition or a premature end of control transfer.
     If the control transfer ends due to a STALL condition, the SentStall bit
     would be set. If the control transfer ends due to a premature
     end of control transfer, the SetupEnd bit would be set.
     In either case, the firmware should abort processing the current control
     transfer and set the state to IDLE.
     (See Inventra: chapter Programmer'guide - Endpoint 0 Service Routine
  **/

  // Check for SentStall
  if (csr0 & USBBI_CSR0_SENTSTALL)
  {
     /* This interrupt is generated
        When the core sets the SentStall bit (CSR0.D5)
        after a control transaction is ended due to a
        protocol violation.
     **/

     // Clear SentStall bit
     musbhsfc_writeb(USBBI_CSR0_ADDR,
                     csr0 & ~USBBI_CSR0_SENTSTALL);

     // State -> IDLE for Eendpoint 0
     ep_endpoints[0]->state = EP0_IDLE;
  }

  // Check for SetupEnd
  if (csr0 & USBBI_CSR0_SETUPEND)
  {
     /* This interrupt is generated
        When the core sets the SetupEnd bit (CSR0.D3)
        because a control transfer has ended before DataEnd (CSR0.D4) is set.
     **/

     // Set ServiceSetupEnd
     musbhsfc_writeb(USBBI_CSR0_ADDR,
                     csr0 & USBBI_CSR0_SVDSETUPEND);

     // State -> IDLE for Eendpoint 0
     ep_endpoints[0]->state = EP0_IDLE;
  }

  // State = IDLE ?
  if (ep_endpoints[0]->state == EP0_IDLE)
     musbhsfc_processEp0IdleMode(csr0);

  // State = TX ?
  if (ep_endpoints[0]->state == EP0_TX)
     musbhsfc_processEp0TxMode();

  // State = RX ?
  if (ep_endpoints[0]->state == EP0_RX)
     musbhsfc_processEp0RxMode();
}

/* **************************************************************************
 * Any other Endpoints (not Ep0)
 * **************************************************************************/

/* **************************************************************************
 * send_data - send packet via endpoint
 * @ep: logical endpoint number
 * @bp: pointer to data
 * @size: bytes to write
 * **************************************************************************/
static void __inline__ send_data (unsigned char ep,
                                  unsigned char *bp,
                                  unsigned char size)
{
//if (bp && size)
  if (bp)
  {
    // copy data from buffer to chip
    musbhsfc_writeFIFO(ep, size, bp);

    /* Select Endpoint ep */
    musbhsfc_selectEndpoint(ep);

    // in any case set the packet ready bit, even if empty FIFO.
    musbhsfc_writeb(USBBI_INCSR_ADDR,
                    musbhsfc_readb(USBBI_INCSR_ADDR) | USBBI_INCSR_INPKTRDY);

    UDC_DBG(3, "send_data,  ep:%i size:%i", ep, size);
  }
}

/* **************************************************************************
 * start_in - start transmit
 * @ep: logical endpoint
 *
 * **************************************************************************/
static void __inline__ start_in (unsigned int ep,
                                 struct usb_endpoint_instance *endpoint,
                                 int restart)
{
  if (endpoint->tx_urb)
  {
    struct urb *urb = endpoint->tx_urb;

    if ((urb->actual_length - endpoint->sent) > 0)
    {
      endpoint->last =
        MIN (urb->actual_length - endpoint->sent, endpoint->tx_packetSize);
      send_data (ep, urb->buffer + endpoint->sent, endpoint->last);
      //endpoint->sent += endpoint->last;
      // the endpoint->sent field will only be updated after
      // reception of the next interrupt (data really sent)
    }
    else
    { // only inform that last packet, by sending an empty packet.
      endpoint->last = 0;
      send_data (ep, urb->buffer + endpoint->sent, 0);
    }
  }
}

/* **************************************************************************
 * start_out - start receive data
 * @ep: logical endpoint number
 * @size: size of data to receive
 * @endpoint:
 *
 * return nothing
 *
 * Extract data from fifo without dma access
 * **************************************************************************/
static void __inline__ start_out (unsigned int ep,
                                  unsigned short size,
                                  struct usb_endpoint_instance *endpoint)
{
  struct urb *urb;

  if ((urb = endpoint->rcv_urb))
  {
    // read the Out FIFO in rcv_urb buffer
    // only if something to read and if enough space in buffer.
    if (size > 0)
    {
      // The usbd_rcv_complete_irq function is described in the usbdinline.h module :
      // if the current received data is short (less than full packetsize) which
      // indicates the end of the bulk transfer, we have received the maximum
      // transfersize, or if we do not have enough room to receive another packet
      // then pass this data up to the function driver
      if ((urb->buffer_length - urb->actual_length) >= size)
      {
        musbhsfc_readFIFO(ep, size, urb->buffer + urb->actual_length);
        usbd_rcv_complete_irq (endpoint, size, 0);
      }
      else
      { // should not occur, because previous call to USBBI_rcv_complete_irq()
        // already checked that enough remaining buffer space.
        usbd_rcv_complete_irq (endpoint, size, 1);
      }
    }
    else
    { // in case out stream is just a multiple of packetsize
      usbd_rcv_complete_irq (endpoint, 0, 0);
    }
  }
  else
  { // should not occur condition because
    // The rcv_urb is supposed to be always active and set
    // either     by call to bi_config()
    // or a previous call to usbd_rcv_complete_irq()
    UDC_DBG(2, "start_out. %i (logical endpoint) empty rcv_urb.", ep);
  }
}

/* **************************************************************************
 * musbhsfc_processOutEndpoint -
 * Endpoint interrupt routine for MUSBHSFC firmware
 *
 * @ep: logical endpoint
 * Return nothing.
 *
 * Assumption: at this level there is no need to determine mode (ISO/BULK/INT)
 *
 * For USB 2.0 device, we use DMA transfert to get data from the FIFO
 * only for the endpoint 1 or 2 and in the bulk mode and > 16 bytes transfert
 * **************************************************************************/
void musbhsfc_processOutEndpoint(unsigned char ep)
{
  unsigned char  OutCSRlow = 0;
  unsigned short outCount  = 0;
#ifdef USEDMA
  struct usb_endpoint_instance *endpoint;
  unsigned long  physicalAddrDest = 0;
  unsigned char  epPhysical = 0;
  unsigned int   sizeDMABytes = 0;
  unsigned int   i = 0;
  unsigned char  *buffer_aux;
#endif

  if (ep_endpoints[ep])
  {
    // Only Endpoint OUT
    if ( (ep_endpoints[ep]->endpoint_address & USB_ENDPOINT_DIR_MASK) == USB_DIR_OUT)
    {
      /* Select correct Endpoint */
      musbhsfc_selectEndpoint(ep);

      // Read OutCSR Reg
      OutCSRlow = musbhsfc_readb(USBBI_OUTCSR_ADDR);

      // Check for SentStall
      if (OutCSRlow & USBBI_OUTCSR_SENTSTALL)
      {
        // nothing really specific to be done as FIFO already flushed.
        // Clear SentStall bit (but keep all other bits)
         musbhsfc_writeb(USBBI_OUTCSR_ADDR,
                         OutCSRlow & ~USBBI_OUTCSR_SENTSTALL);
      }
      else if (OutCSRlow & (USBBI_OUTCSR_DATA_ERROR |
                            USBBI_OUTCSR_OVERRUN ))
      {
        printk (KERN_ERR "musbhsfc_processOutEndpoint: Error received, OutCSRlow:0x%x\n", OutCSRlow);
      }
      else if (OutCSRlow & USBBI_OUTCSR_OUTPKTRDY)
      {
         outCount = musbhsfc_readw(USBBI_OUTCOUNT_ADDR);

#ifdef USEDMA
         // For USB 2.0 device, we use DMA transfert to get data from the FIFO
         // dmaConfigured indicates the usb device and dma core have been configured
         // (usb device 2.0 + mode bulk and physical endpoint equals 1 or 2)
         if (dmaConfigured)
         {
            // physical endpoint
            epPhysical = endpoints_configByte[ep].epPhysical;

            // Get destination address from dma2plb4
            if (epPhysical == 1) // Endpoint 1
               physicalAddrDest = get_dma2pl4_dst_addr(2);

            if (epPhysical == 2) // Endpoint 2
               physicalAddrDest = get_dma2pl4_dst_addr(3);

            // Calculate transfered bytes
            sizeDMABytes = physicalAddrDest - iopa((unsigned long)ps_alloc);
            if (sizeDMABytes)  // The DMA core has been solicited. Some bytes have been read from the fifo
            {
               for (i = 0; i < sizeDMABytes; i+= ep_endpoints[ep]->rcv_packetSize)
               {
                  endpoint = ep_endpoints[ep];
                  buffer_aux = ep_endpoints[ep]->rcv_urb->buffer;
                  ep_endpoints[ep]->rcv_urb->buffer = ps_alloc + i;
                  usbd_rcv_complete_irq (ep_endpoints[ep], ep_endpoints[ep]->rcv_packetSize, 0);
                  endpoint->rcv_urb->buffer = buffer_aux;
               }
            }
         }
#endif
         // read data and call upper layer
         // note that done also if outCount is null to inform upper layer.
         start_out(ep, outCount, ep_endpoints[ep]);

         // in anycase clean the OutPktRdy bit
         musbhsfc_writeb(USBBI_OUTCSR_ADDR,
                         OutCSRlow & ~USBBI_OUTCSR_OUTPKTRDY);

#ifdef USEDMA
         // Configure for the next bursts of data
         if (dmaConfigured)
            musbhsfc_enableDMA_OUT(ep);
#endif
      }
    }
  }
}

/* **************************************************************************
 * musbhsfc_processInEndpoint -
 * Endpoint interrupt routine for MUSBHSFC firmware
 *
 * Return nothing.
 *
 * Assumption: at this level there is no need to determine mode (ISO/BULK/INT)
 * **************************************************************************/
void musbhsfc_processInEndpoint(unsigned char ep)
{
  unsigned char inCSRlow;
//unsigned char inCSRhigh;

  UDC_DBG(3, "musbhsfc_processInEndpoint, endpoint: %i", ep);

  if (ep_endpoints[ep])
  {
    // Only Endpoint IN
    if ( (ep_endpoints[ep]->endpoint_address & USB_ENDPOINT_DIR_MASK) == USB_DIR_IN)
    {
      /* Select correct Endpoint */
      musbhsfc_selectEndpoint(ep);

      // Read inCSR Reg
      inCSRlow = musbhsfc_readb(USBBI_INCSR_ADDR);

      // Check for SentStall
      if (inCSRlow & USBBI_INCSR_SENTSTALL)
      {
        // nothing really specific to be done as FIFO already flushed.
        // Clear SentStall bit (but keep all other bits)
        musbhsfc_writeb(USBBI_INCSR_ADDR,
                        inCSRlow & ~USBBI_INCSR_SENTSTALL);

      }
      else
      {
         if (ep_endpoints[ep]->tx_urb)
         {
           // What is required to be done:
           //
           // * find a tx_urb if none active at this time.
           // * update the current tx_usb with size of just sent data.
           // * if all data of tx_urb has been sent then inform above code layers.
           //
           //   The three above points are done by the usbd_tx_complete_irq() call.
           //
           // * enqueue in the FIFO all or part of the data of current tx_urb.
           //   this is done by the start_in() call.
           usbd_tx_complete_irq (ep_endpoints[ep], 0);
           start_in (ep, ep_endpoints[ep], 0);
         }
      }
    }
  }
}


/* **************************************************************************
 * Interrupt Handler
 * **************************************************************************/

/* **************************************************************************
 * musbhsfc_readInterruptRegister -
 * Read/clear interrupt Registers of the USB Device
 *
 * Return nothing.
 * **************************************************************************/
void musbhsfc_readInterruptRegister(void)
{
   /* read/clear and save interrupt regs */
   musbhsfc_data.inFlags  = musbhsfc_readw(USBBI_INTRIN_ADDR);
   musbhsfc_data.outFlags = musbhsfc_readw(USBBI_INTROUT_ADDR);
   musbhsfc_data.usbFlags = musbhsfc_readb(USBBI_INTRUSB_ADDR);

#ifdef USEDMA
   if (dmaConfigured) /* Only for the USB2.0 Device */
   {
      musbhsfc_data.dma2plb4 = get_dma2pl4_status();
   }
#endif
}


/* **************************************************************************
 * musbhsfc_processInterrupt - read/process the interrupt status register
 *
 * Return nothing.
 * **************************************************************************/
void musbhsfc_processInterrupt(void)
{
  /* When the MCU is interrupted with a USB interrupt, it
     needs to read the interrupt status register to determine
     which endpoint(s) have caused the interrupt and jump to
     the appropriate routine. If multiple endpoints have caused
     the interrupt, Endpoint 0 should be serviced first, followed
     by the other endpoints. The Suspend interrupt should be
     serviced last. */

  unsigned char ep;

  /* Reads/clears interrupt Registers of the USB Device*/
  musbhsfc_readInterruptRegister();

  // Interrupt service routine

  // RESUME Interrupt
  if (musbhsfc_data.usbFlags & USBBI_INTRUSB_RESUME)
  {
      usbd_device_event (udc_device, DEVICE_BUS_ACTIVITY, 0);
  }

  // RESET Interrupt
  if (musbhsfc_data.usbFlags & USBBI_INTRUSB_RESET)
  {
    ep_endpoints[0]->state = EP0_IDLE;
    usbd_device_event_irq (udc_device, DEVICE_RESET, 0);
    speedIndication = 0;
  }

  // Endpoint0 Interrupt
  if (musbhsfc_data.inFlags & USBBI_INTRIN_EP0)
  {
    // Turn on the USB 1.1 or 2.0 Phy interface.
    if (usb2_device && (speedIndication == 0)) { /* Only for USB 2.0 device */
      musbhsfc_speedMode();
      speedIndication = 1;
    }
    musbhsfc_processEndpoint0();
  }

  // IN Interrupt
  if (musbhsfc_data.inFlags & ~USBBI_INTRIN_EP0)
  {
    for (ep = 1; ep < USBBI_UDC_MAX_LOGICAL_ENDPOINTS; ep++)
    {
      // handle successively all bits that are set in IntrIn register
      if (musbhsfc_data.inFlags & endpoints_configByte[ep].endpointIntMask)
        musbhsfc_processInEndpoint(ep);
    }
  }

  // OUT Interrupt
  if (musbhsfc_data.outFlags)
  {
    for (ep = 1; ep < USBBI_UDC_MAX_LOGICAL_ENDPOINTS; ep++)
    {
      // handle successively all bits that are set in IntrOut register
      if (musbhsfc_data.outFlags & endpoints_configByte[ep].endpointIntMask)
        musbhsfc_processOutEndpoint(ep);
    }
  }

#ifdef USEDMA
  // For USB 2.0 device, we use DMA transfert to get data from the FIFO
  // dmaConfigured indicates the usb device and dma core have been configured
  // (usb device 2.0 + mode bulk and physical endpoint equals 1 or 2)
  if ((dmaConfigured) && (musbhsfc_data.dma2plb4))
  {
     printk(KERN_INFO "musbhsfc_processInterrupt: dma2plb4 status 0x%x", musbhsfc_data.dma2plb4);
  }
#endif

  // SUSPEND Interrupt
  if (musbhsfc_data.usbFlags & USBBI_INTRUSB_SUSPEND)
       usbd_device_event (udc_device, DEVICE_BUS_INACTIVE, 0);
}


/* **************************************************************************
 * musbhsfc_int_hndlr - interrupt handler
 *
 * This is the function called by kernel
 * when Interrupt from USB core raised to UIC core.
 * (after registration done below by request_irq() call)
 * **************************************************************************/
static void musbhsfc_int_hndlr (int irq, void *dev_id, struct pt_regs *regs)
{
  udc_interrupts++;

  // read-process the interrupt status register
  musbhsfc_processInterrupt();
}

/* **************************************************************************/
/* **************************************************************************
 * Start of public functions.
 * **************************************************************************/

/* **************************************************************************
 * udc_start_in_irq - start transmit
 * @eendpoint: endpoint instance
 *
 * Called by bus interface driver
 * to see if we need to start a data transmission.
 *
 * Assumption: we are called in an interrupt mode, so no need to save irq.
 * refer to udc_start_in() if need this function outside of interrupt mode.
 * **************************************************************************/
void udc_start_in_irq (struct usb_endpoint_instance *endpoint)
{
  UDC_DBG(2, "udc_start_in_irq *******************************");

  if (endpoint)
  {
    // We are triggering a new buffer transmission only
    // if none are currently active.
    // Note that this function is only called by above layers
    // but not by the bus interface layer.
    start_in (endpoint->endpoint_address & 0x7f, endpoint, 0);
  }
}

/* **************************************************************************
 * udc_init - initialize
 *
 * Return non-zero if we cannot see device.
 * **************************************************************************/
int udc_init (void)
{
  UDC_DBG(2, "udc_init ***************************************");

  mem_base = ioremap64(USBBI_COMMON_REGISTERS,
                       USBBI_PCIL0_SIZE);
  if (!mem_base) {
    printk (KERN_ERR "udc_init: Error mapping Musbhsfc-opb memory\n");
    return -EFAULT;
  }

  /* Initialize Configuration */

  /* Initialize Endpoint information with default values*/
  // Endpoint 0
  endpoints_configByte[0].endpointIntMask = 0x0001;
  endpoints_configByte[0].epPhysical      = 0x00;

  // Logical Endpoints 1 and 2 correspond to Physical Endpoint 1
  endpoints_configByte[1].configByteIN    = USBBI_EP_CONF_IN_DEFAULT;
  endpoints_configByte[1].configByteOUT   = USBBI_EP_CONF_OUT_DEFAULT;
  endpoints_configByte[1].epPhysical      = 0x01;
  endpoints_configByte[1].endpointIntMask = 0x0002;
  endpoints_configByte[2].configByteIN    = USBBI_EP_CONF_IN_DEFAULT;
  endpoints_configByte[2].configByteOUT   = USBBI_EP_CONF_OUT_DEFAULT;
  endpoints_configByte[2].epPhysical      = 0x01;
  endpoints_configByte[2].endpointIntMask = 0x0002;

  // Logical Endpoints 3 and 4 correspond to Physical Endpoint 2
  endpoints_configByte[3].configByteIN    = USBBI_EP_CONF_IN_DEFAULT;
  endpoints_configByte[3].configByteOUT   = USBBI_EP_CONF_OUT_DEFAULT;
  endpoints_configByte[3].epPhysical      = 0x02;
  endpoints_configByte[3].endpointIntMask = 0x0004;
  endpoints_configByte[4].configByteIN    = USBBI_EP_CONF_IN_DEFAULT;
  endpoints_configByte[4].configByteOUT   = USBBI_EP_CONF_OUT_DEFAULT;
  endpoints_configByte[4].epPhysical      = 0x02;
  endpoints_configByte[4].endpointIntMask = 0x0004;

  // Logical Endpoints 5 and 6 correspond to Physical Endpoint 3
  endpoints_configByte[5].configByteIN    = USBBI_EP_CONF_IN_DEFAULT;
  endpoints_configByte[5].configByteOUT   = USBBI_EP_CONF_OUT_DEFAULT;
  endpoints_configByte[5].epPhysical      = 0x03;
  endpoints_configByte[5].endpointIntMask = 0x0008;
  endpoints_configByte[6].configByteIN    = USBBI_EP_CONF_IN_DEFAULT;
  endpoints_configByte[6].configByteOUT   = USBBI_EP_CONF_OUT_DEFAULT;
  endpoints_configByte[6].epPhysical      = 0x03;
  endpoints_configByte[6].endpointIntMask = 0x0008;

  /* Initialize hardware */

  /* disable address decoding. Set Address to 0 */
  musbhsfc_setAddress(0);    /* disable device */

  /* Clear spurious interrupt
  * all actived interrupts are cleared when these registers are read.
  */
  musbhsfc_readw(USBBI_INTRIN_ADDR);     // clear all, not only the used one
  musbhsfc_readb(USBBI_INTRUSB_ADDR);
  musbhsfc_readw(USBBI_INTROUT_ADDR);    // clear all, not only the used one

  /* disable interrupts */
  musbhsfc_writew(USBBI_INTRINE_ADDR , 0x0000);
  musbhsfc_writew(USBBI_INTROUTE_ADDR, 0x0000);

  /* Enable Suspend,Resume and USB Reset interrupts in USB interrupt register */
  musbhsfc_writeb(USBBI_INTRUSBE_ADDR,
                 (USBBI_INTRUSB_SUSPEND |
                  USBBI_INTRUSB_RESUME |
                  USBBI_INTRUSB_RESET));
  return 0;
}

/* **************************************************************************
 * udc_start_in - start transmit       
 * @eendpoint: endpoint instance
 *
 * Called by bus interface driver
 * to see if we need to start a data transmission.
 *
 * **************************************************************************/
void udc_start_in (struct usb_endpoint_instance *endpoint)
{
  printk (KERN_INFO "udc_start_in (NOTHING TO DO ?)\n");
}



/* **************************************************************************
 * udc_stall_ep - stall endpoint
 * @ep: logical endpoint
 *
 * Stall the endpoint.
 *
 * **************************************************************************/
void udc_stall_ep (unsigned int ep)
{
  UDC_DBG(2, "udc_stall_ep: %i", ep);
  if (ep < USBBI_UDC_MAX_LOGICAL_ENDPOINTS)
  {
    // stall
    // Set the status of the endpoint
    musbhsfc_setEndpointStatus(ep, ENDPOINT_STALL);
  }
}


/* **************************************************************************
 * udc_reset_ep - reset endpoint
 * @ep: logical endpoint
 *
 * reset the endpoint.
 *
 * **************************************************************************/
void udc_reset_ep (unsigned int ep)
{
  unsigned char CSRlow;


  UDC_DBG(2, "udc_reset_ep: %i", ep);
  if (ep < USBBI_UDC_MAX_LOGICAL_ENDPOINTS) {
    // reset
    if (ep != 0) {
       // Select Endpoint (through index register)
       musbhsfc_selectEndpoint(ep);
       switch (ep_endpoints[ep]->endpoint_address & USB_ENDPOINT_DIR_MASK)
       {
       case USB_DIR_OUT:
         // Read CSR Reg
         CSRlow = musbhsfc_readb(USBBI_OUTCSR_ADDR);

         if (CSRlow & USBBI_OUTCSR_FIFOFULL)
            CSRlow = CSRlow | USBBI_OUTCSR_CLRDATATOG | USBBI_OUTCSR_FLUSHFIFO;
         else
            CSRlow = CSRlow | USBBI_OUTCSR_CLRDATATOG;

         // TODO: do I have to reset the OverRun bit ?
         musbhsfc_writeb(USBBI_OUTCSR_ADDR, CSRlow);
         eieio();       // need to sync with reg access

         // read again to check if need to flush two packets.
         CSRlow = musbhsfc_readb(USBBI_OUTCSR_ADDR);
         if (CSRlow & USBBI_OUTCSR_FIFOFULL)
            musbhsfc_writeb(USBBI_OUTCSR_ADDR, CSRlow | USBBI_OUTCSR_FLUSHFIFO);

       break;

       case USB_DIR_IN:
         // Read CSR Reg
         CSRlow = musbhsfc_readb(USBBI_INCSR_ADDR);

         if (CSRlow & USBBI_INCSR_FIFONEMPTY)
            CSRlow = CSRlow | USBBI_INCSR_CLRDATATOG | USBBI_INCSR_FLUSHFIFO;
         else
            CSRlow = CSRlow | USBBI_INCSR_CLRDATATOG;

         // TODO: do I have to reset the UnderRun bit ?
         musbhsfc_writeb(USBBI_INCSR_ADDR, CSRlow);
         eieio();       // need to sync with reg access

         // read again to check if need to flush two packets.
         CSRlow = musbhsfc_readb(USBBI_INCSR_ADDR);
         if (CSRlow & USBBI_INCSR_FIFONEMPTY)
            musbhsfc_writeb(USBBI_INCSR_ADDR, CSRlow | USBBI_INCSR_FLUSHFIFO);

       break;
       }
    }
    else {
      // State -> IDLE for Eendpoint 0
      ep_endpoints[0]->state = EP0_IDLE;
    }
  }
}


/* **************************************************************************
 * udc_endpoint_halted - is endpoint halted
 * @ep: logical endpoint
 *
 * Return non-zero if endpoint is halted
 * **************************************************************************/
int udc_endpoint_halted (unsigned int ep)
{
  UDC_DBG(1, "udc_endpoint_halted. (Nothing to do)");
  return 0;
}


/* **************************************************************************
 * udc_set_address - set the USB address for this device
 * @address:
 *
 * Called from control endpoint function
 * after it decodes a set address setup packet.
 * **************************************************************************/
void udc_set_address (unsigned char address)
{
  UDC_DBG(2, "udc_set_address %d", address);

  // address cannot be setup until ack received
  usb_address = address;

  musbhsfc_setAddress(usb_address);
}

/* **************************************************************************
 * udc_serial_init - set a serial number if available
 * **************************************************************************/
int __init udc_serial_init (struct usb_bus_instance *bus)
{
  UDC_DBG(2, "udc_serial_init. (Nothing to do)");
  return -EINVAL;
}

/* **************************************************************************/

/* **************************************************************************
 * udc_max_endpoints - max logical endpoints
 *
 * Return number of physical endpoints.
 * **************************************************************************/
int udc_max_endpoints (void)
{
  return USBBI_UDC_MAX_LOGICAL_ENDPOINTS;
}


/* **************************************************************************
 * udc_check_ep - check logical endpoint
 * @logical_endpoint:  logical endpoint
 *
 * Return physical endpoint number
 * to use for this logical endpoint or zero if not valid.
 * **************************************************************************/
int udc_check_ep (int logical_endpoint, int packetsize)
{
  int ep = logical_endpoint & 0xf;

  UDC_DBG(2, "udc_check_ep, logical_endpoint :%02x, packetsize: %02x",
          logical_endpoint, packetsize);

  if ((ep >= USBBI_UDC_MAX_LOGICAL_ENDPOINTS)      ||
      ((ep == 0) && (packetsize > USBBI_MAX_PKT_CONTROL )) ||
      ((ep != 0) && (packetsize > USBBI_MAX_PKT_ENDPOINT)))
     return 0;

  /* Warning: For this Hardware Usb Device, in this function the physical enpoint
     number is equal to the logical enpoint number (the bEndpointAddress of the
     usb_endpoint_description structure described in the "function" file*/
  return ep;
}


/* **************************************************************************
 * udc_setup_ep - setup endpoint
 * @ep: logical endpoint
 * @endpoint:
 *
 * Associate a physical endpoint with endpoint_instance
 * For device USB 2.0 Enable DMA accesses
 * Register DMA interrupt in the kernel handler (only for the
 * endpoints 1 and 2 in the BULK mode.It'not necessary for the others
 * mode: interrupt and isochronous)
 *
 * **************************************************************************/
void udc_setup_ep (struct usb_device_instance *device, unsigned int ep,
                   struct usb_endpoint_instance *endpoint)
{
  /* This routine gets called by bi_modinit for endpoint 0 and from
   * bi_config for all of the other endpoints.  bi_config gets called
   * during the DEVICE_CREATE, DEVICE_CONFIGURED, and
   * DEVICE_SET_INTERFACE events.
  */
#ifdef USEDMA
  unsigned char epPhysical;
#endif

  if (ep < USBBI_UDC_MAX_LOGICAL_ENDPOINTS)
  {
    UDC_DBG(2, "udc_setup_ep %i", ep);

    // ep0
    if (ep == 0)
    {
       ep_endpoints[ep] = endpoint;
       ep_endpoints[ep]->state = EP0_IDLE;

       // Set the status of the endpoint
       musbhsfc_setEndpointStatus(ep, ENDPOINT_DATA0);

       // Enable interrupts for the endpoint
       musbhsfc_enableInterrupt(ep);
    }
    else {

      if (endpoint->endpoint_address == 0)
      {
         UDC_DBG(2, "udc_setup_ep %i not configured", ep);
         return;
      }

      ep_endpoints[ep] = endpoint;

      // Set the status of the endpoint
      musbhsfc_setEndpointStatus(ep, ENDPOINT_DATA0);

      // Enable interrupts for the endpoint
      musbhsfc_enableInterrupt(ep);

      /* Set corresponding config bits */
      switch (ep_endpoints[ep]->endpoint_address & USB_ENDPOINT_DIR_MASK)
      {
        case USB_DIR_OUT: // OUT (rx) endpoint

          if (ep != 0)
          {
            usbd_fill_rcv (device, endpoint, 5);
            endpoint->rcv_urb = first_urb_detached (&endpoint->rdy);
          }

          switch (ep_endpoints[ep]->rcv_attributes)
          {
            case BULK:
#ifdef USEDMA
              // Only for USB 2.0 Device and Endpoint 1 or 2)
              // physical endpoint
              epPhysical = endpoints_configByte[ep].epPhysical;

              if ((usb2_device) && (epPhysical == 1))       /* Endpoint 1 */
              {
                 // Configure DMA
                 musbhsfc_configureDMA_OUT(ep);
                 musbhsfc_enableDMA_OUT(ep);
              }
              if ((usb2_device) && (epPhysical == 2))       /* Endpoint 2 */
              {
                 // Configure DMA
                 musbhsfc_configureDMA_OUT(ep);
                 musbhsfc_enableDMA_OUT(ep);
              }
#endif
            break;
            case INTERRUPT:
              endpoints_configByte[ep].configByteOUT |= USBBI_OUTCSRH_DISNYET;
            break;
            case ISOCHRONOUS:
              endpoints_configByte[ep].configByteOUT |= USBBI_OUTCSRH_ISO;
            break;
          }
        break;

        case USB_DIR_IN: // IN (tx) endpoints

          switch (ep_endpoints[ep]->tx_attributes)
          {
            case BULK:
              // Set mode bit high (only strictly necessary if sharing a FIFO)
              endpoints_configByte[ep].configByteIN |= USBBI_INCSRH_MODE;
            break;
            case INTERRUPT:
              // Set mode bit high (only strictly necessary if sharing a FIFO)
              endpoints_configByte[ep].configByteIN |= USBBI_INCSRH_MODE;
            break;
            case ISOCHRONOUS:
              // Set mode bit high (only strictly necessary if sharing a FIFO)
              // Set Isochronous transfert
              endpoints_configByte[ep].configByteIN |= (USBBI_INCSRH_MODE | USBBI_INCSRH_ISO);
            break;
            }
        break;
      }

      // Configure the USB device with Mode (ISO/BULK/INT and IN/OUT/IN_OUT)
      // + MAX Packet size
      musbhsfc_setConfig(ep);
    }
  }

  // udc_regs();
}


/* **************************************************************************
 * udc_disable_ep - disable endpoint
 * @ep: logical endpoint
 *
 * Disable specified endpoint
 * For the USB 2.0 device and the endpoint 1 or 2 in the bulk mode
 * release the DMA to plb4 irq
 * **************************************************************************/
void udc_disable_ep (unsigned int ep)
{
  UDC_DBG(2, "udc_disable_ep %i", ep);

  if (ep < USBBI_UDC_MAX_LOGICAL_ENDPOINTS)
  {
    struct usb_endpoint_instance *endpoint;

    if ((endpoint = ep_endpoints[ep]))
    {
       ep_endpoints[ep] = NULL;
       usbd_flush_ep (endpoint);
       UDC_DBG(2, "udc_disable_ep - USBBI_flush_ep %i", ep);
    }
  }
}

/* **************************************************************************/

/* **************************************************************************
 * udc_connected - is the USB cable connected
 *
 * Return non-zeron if cable is connected.
 * **************************************************************************/
int udc_connected ()
{
  UDC_DBG(2, "udc_connected ");
  if (!first_connection)
     return 0;

  if(first_connection) {
     first_connection=0;
     return 1;
  }
  return 1;
}


/* **************************************************************************
 * udc_connect - enable pullup resistor
 *
 * Turn on the USB connection by enabling the pullup resistor.
 * **************************************************************************/
void udc_connect (void)
{
  UDC_DBG(2, "udc_connect (Nothing to do)");
}


/* **************************************************************************
 * udc_disconnect - disable pullup resistor
 *
 * Turn off the USB connection by disabling the pullup resistor.
 * **************************************************************************/
void udc_disconnect (void)
{
   UDC_DBG(2, "udc_disconnect (Nothing to do)");
   
   usb2_device     = 0;
   speedIndication = 0;
#ifdef USEDMA
   dmaConfigured   = 0;
#endif
}

/* **************************************************************************
 * udc_int_hndlr_cable - interrupt handler for cable
 * **************************************************************************/
static void udc_int_hndlr_cable (int irq, void *dev_id, struct pt_regs *regs)
{
   UDC_DBG(2, "udc_int_hndlr_cable (Nothing to do)");
}

/* **************************************************************************/

/* **************************************************************************
 * udc_enable_interrupts - enable interrupts
 *
 * Switch on UDC interrupts.
 * **************************************************************************/
void udc_all_interrupts (struct usb_device_instance *device)
{
  UDC_DBG(2, "udc_all_interrupts (Nothing to do)");
  /* Nothing to do */
}


/* **************************************************************************
 * udc_suspended_interrupts - enable suspended interrupts
 *
 * Switch on only UDC resume interrupt.
 * **************************************************************************/
void udc_suspended_interrupts (struct usb_device_instance *device)
{
  UDC_DBG(2, "udc_suspended_interrupts (Nothing to do)");
  /* Nothing to do */
}


/* **************************************************************************
 * udc_disable_interrupts - disable interrupts.
 *
 * switch off interrupts
 * **************************************************************************/
void udc_disable_interrupts (struct usb_device_instance *device)
{
  UDC_DBG(2, "udc_disable_interrupts. Switch off USBBI_INTRINE & USBBI_INTROUTE ");

  // switch off interrupts
  musbhsfc_writew(USBBI_INTRINE_ADDR, 0x00);
  musbhsfc_writew(USBBI_INTROUTE_ADDR, 0x00);

  musbhsfc_writeb(USBBI_INTRUSBE_ADDR, 0x00);
}

/* **************************************************************************/

/* **************************************************************************
 * udc_ep0_packetsize - return ep0 packetsize
 * **************************************************************************/
int udc_ep0_packetsize (void)
{
  return USBBI_MAX_PKT_CONTROL;
}

/* **************************************************************************
 * udc_enable - enable the UDC
 *
 * Switch on the UDC
 * **************************************************************************/
void udc_enable (struct usb_device_instance *device)
{
  UDC_DBG(2, "enable device %p, status %d", device, device->status);

  usb2_device     = 0;
  speedIndication = 0;
#ifdef USEDMA
  dmaConfigured   = 0;
#endif
   
  // Check the USB Device: USB 1.1 or 2.0 Phy interface.
  musbhsfc_usbDevice();
  if (usb2_device == 0) { /* Only for USB 1.1 device */
    musbhsfc_speedMode();
  }

  // save the device structure pointer
  udc_device = device;

  // ep0 urb
  if (!ep0_urb) {
    ep0_urb = usbd_alloc_urb (device, device->function_instance_array, 0, 512);
    if (!ep0_urb) {
       printk (KERN_ERR "ep0_enable: usbd_alloc_urb failed\n");
    }
  } else {
     printk (KERN_ERR "udc_enable: ep0_urb already allocated\n");
  }

  // enable UDC
  musbhsfc_setAddress(usb_address);
}


/* **************************************************************************
 * udc_disable - disable the UDC
 *
 * Switch off the UDC
 * **************************************************************************/
void udc_disable (void)
{
  UDC_DBG(2, "udc_disable *************************");

  // disable UDC
  musbhsfc_setAddress(0);    /* disable device */

  // reset device pointer
  udc_device = NULL;

  // ep0 urb
  if (ep0_urb)
  {
    usbd_dealloc_urb (ep0_urb);
    ep0_urb = NULL;
  }
  usb2_device     = 0;
  speedIndication = 0;
#ifdef USEDMA
  dmaConfigured   = 0;
#endif
}

/* **************************************************************************
 * udc_startup - allow udc code to do any additional startup
 * **************************************************************************/
void udc_startup_events (struct usb_device_instance *device)
{
  UDC_DBG(2, "udc_startup_events. (DEVICE_INIT - DEVICE_CREATE - DEVICE_HUB_CONFIGURED)");
  usbd_device_event (device, DEVICE_INIT, 0);
  usbd_device_event (device, DEVICE_CREATE, 0);
  usbd_device_event (device, DEVICE_HUB_CONFIGURED, 0);
  // usbd_device_event (device, DEVICE_RESET, 0); // XXX should be done from device event

  if (!first_connection)
       first_connection = 1;
}


/* **************************************************************************/

/* **************************************************************************
 * udc_name - return name of USB Device Controller
 * **************************************************************************/
char *udc_name (void)
{
  return USBBI_UDC_NAME;
}

/* **************************************************************************
 * udc_request_udc_irq - request UDC interrupt
 *
 * Return non-zero if not successful.
 * **************************************************************************/
int udc_request_udc_irq ()
{
  UDC_DBG(2, "udc_request_udc_irq. Installing interrupt handlers");
  UDC_DBG(2, "UDC_NAME = %s, UDC_IRQ = %i ",
          USBBI_UDC_NAME, USBBI_UDC_IRQ);

  // request IRQ  and IO region
  if (request_irq
      (USBBI_UDC_IRQ, musbhsfc_int_hndlr, SA_INTERRUPT | SA_SAMPLE_RANDOM,
       USBBI_UDC_NAME " USBD Bus Interface", NULL) != 0)
  {
     printk (KERN_ERR "udc_request_udc_irq: Couldn't request USB irq: %i\n",
             USBBI_UDC_IRQ);
     return -EINVAL;
  }

#ifdef USEDMA  
  // request IRQ  and IO region DMA
  if (request_irq
      (USBBI_UDC_DMA_ENDPOINT1_OUT_IRQ, musbhsfc_int_hndlr, SA_INTERRUPT | SA_SAMPLE_RANDOM,
       USBBI_UDC_DMA_NAME " DMA to PLB4 Endpoint1 OUT", NULL) != 0)
  {
    printk (KERN_ERR "Couldn't request DMA to PLB4 (Endpoint1 OUT) irq: %i\n",
    USBBI_UDC_DMA_ENDPOINT1_OUT_IRQ);
  }
  if (request_irq
      (USBBI_UDC_DMA_ENDPOINT2_OUT_IRQ, musbhsfc_int_hndlr, SA_INTERRUPT | SA_SAMPLE_RANDOM,
       USBBI_UDC_DMA_NAME " DMA to PLB4 Endpoint2 OUT", NULL) != 0)
  {
    printk (KERN_ERR "Couldn't request DMA to PLB4 (Endpoint2 OUT) irq: %i\n",
    USBBI_UDC_DMA_ENDPOINT2_OUT_IRQ);
  }
#endif
  
  return 0;
}

/* **************************************************************************
 * udc_request_cable_irq - request Cable interrupt
 *
 * Return non-zero if not successful.
 * **************************************************************************/
int udc_request_cable_irq ()
{
  UDC_DBG(2, "udc_request_cable_irq. (Nothing to do)");
  return 0;
}

/* **************************************************************************
 * udc_request_udc_io - request UDC io region
 *
 * Return non-zero if not successful.
 * **************************************************************************/
int udc_request_io ()
{
  UDC_DBG(2, "udc_request_io. (Nothing to do)");
  return 0;
}

/* **************************************************************************
 * udc_release_udc_irq - release UDC irq
 * **************************************************************************/
void udc_release_udc_irq ()
{
  UDC_DBG(1, "udc_release_udc_irq. Release UDC irq : %i", USBBI_UDC_IRQ);
  free_irq(USBBI_UDC_IRQ, NULL);

#ifdef USEDMA
  // Release irq about DMA 
  free_irq(USBBI_UDC_DMA_ENDPOINT1_OUT_IRQ, NULL);
  free_irq(USBBI_UDC_DMA_ENDPOINT2_OUT_IRQ, NULL);
#endif
}

/* **************************************************************************
 * udc_release_cable_irq - release Cable irq
 * **************************************************************************/
void udc_release_cable_irq ()
{
  UDC_DBG(2, "udc_release_cable_irq. (Nothing to do)");
}

/* **************************************************************************
 * udc_release_release_io - release UDC io region
 * **************************************************************************/
void udc_release_io ()
{
   UDC_DBG(2, "udc_release_io (Nothing to do)");
   /* Nothing to do */
}


/* **************************************************************************
 * udc_regs - dump registers
 *
 * Dump registers with printk
 * **************************************************************************/
void udc_regs (void)
{
  printk (KERN_INFO "Musbhsfc-opb USB device Driver - Dump Registers \n");
  printk (KERN_INFO "USBBI_INTRIN   : 0x%x \n", musbhsfc_readw(USBBI_INTRIN_ADDR));
  printk (KERN_INFO "USBBI_POWER    : 0x%x \n", musbhsfc_readb(USBBI_POWER_ADDR));
  printk (KERN_INFO "USBBI_FADDR    : 0x%x \n", musbhsfc_readb(USBBI_FADDR_ADDR));
  printk (KERN_INFO "USBBI_INTRINE  : 0x%x \n", musbhsfc_readw(USBBI_INTRINE_ADDR));
  printk (KERN_INFO "USBBI_INTROUT  : 0x%x \n", musbhsfc_readw(USBBI_INTROUT_ADDR));
  printk (KERN_INFO "USBBI_INTRUSBE : 0x%x \n", musbhsfc_readb(USBBI_INTRUSBE_ADDR));
  printk (KERN_INFO "USBBI_INTRUSB  : 0x%x \n", musbhsfc_readb(USBBI_INTRUSB_ADDR));
  printk (KERN_INFO "USBBI_INTROUTE : 0x%x \n", musbhsfc_readw(USBBI_INTROUTE_ADDR));
  printk (KERN_INFO "USBBI_TESTMODE : 0x%x \n", musbhsfc_readb(USBBI_TESTMODE_ADDR));
  printk (KERN_INFO "USBBI_INDEX    : 0x%x \n", musbhsfc_readb(USBBI_INDEX_ADDR));
  printk (KERN_INFO "USBBI_FRAME    : 0x%x \n", musbhsfc_readw(USBBI_FRAME_ADDR));
}

