/*
 * SPI support for PNX0105, based on the code for SAA7752 I2C.
 *
 * Copyright (c) 2004 MontaVista Software, Inc.
 * 
 * Author:  Andrey Ivolgin <aivolgin@ru.mvista.com>
 * 
 * $Id: spi-pnx0105.c,v 1.1.2.3 2004/10/12 15:04:39 wool Exp $
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/module.h>
#include <linux/config.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/pci.h>
#include <linux/types.h>
#include <linux/delay.h>
#include <linux/spi/spi.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/mm.h>
#include <linux/miscdevice.h>
#include <linux/timer.h>
#include <linux/spinlock.h>
#include <linux/slab.h>
#include <linux/completion.h>
#include <asm/hardware.h>
#include <asm/irq.h>
#include <asm/uaccess.h>
#include <asm/arch/sdma.h>


/*  Uncomment the following line for debugging info  */
#define SPIPNX_DEBUG

#ifdef SPIPNX_DEBUG
#define DEBUG 1
#endif

#define DBG(args...)	pr_debug(args)

#include <linux/spi/spi-pnx0105.h>


static int		 sdma_mode = 0;
static int		 spipnx_initialized;
static struct completion threshold;
static struct completion op_complete;
static void		*pPhysSpiDataReg = 0;
static dma_addr_t	 dma_buffer;
static void		*spi_buffer = 0;
static unsigned int	 spi_sdma_channel = 0;
static char		 sdma_ch_name[] = "SDMA channel";
static sdma_setup_t	 sdma_setup;
static sdma_config_t	 sdma_config;


static int spipnx_xfer (struct spi_adapter *, struct spi_msg*, int);

static struct spi_algorithm spipnx_algorithm = {
	name:		ALGORITHM_NAME,
	xfer:		spipnx_xfer,
};

static struct spi_adapter spipnx_adapter = {
	name:		ADAPTER_NAME,
	algo:		&spipnx_algorithm,
	owner:		THIS_MODULE,
};


static void 
spipnx_interrupt (int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned int i_stat = spi_regs.stat;
	
	if ( (i_stat & SPI_STAT_SPI_EOT) && (spi_regs.ier & SPI_IER_SPI_INTEOT) ) {
		spi_regs.ier &= ~SPI_IER_SPI_INTEOT;
		complete (&op_complete);
	}
	if ( (i_stat & SPI_STAT_SPI_THR) && (spi_regs.ier & SPI_IER_SPI_INTTHR) ) {
		spi_regs.ier &= ~SPI_IER_SPI_INTTHR;
		complete (&threshold);
	}
	spi_regs.stat |= SPI_STAT_SPI_INTCLR;    /*     clear interrupt   */
	spi_timer.status = 0L;
}


static void spipnx_sdma_cb (int channel_nr, sdma_irq_type_t irq_type, void *ptr,  struct pt_regs *regs)
{
	if (channel_nr == spi_sdma_channel)
		complete (&op_complete);
}


static void spipnx_select_chip (void)
{
    unsigned reg = gpio_read_reg (PADMUX1_MODE0);
    gpio_write_reg ((reg & ~GPIO_PIN_SPI_CE), PADMUX1_MODE0);
}


static void spipnx_unselect_chip (void)
{
    unsigned reg = gpio_read_reg (PADMUX1_MODE0);
    gpio_write_reg (reg | GPIO_PIN_SPI_CE, PADMUX1_MODE0);
}


static int
spipnx_xfer_nonsdma (struct spi_adapter *adap, struct spi_msg msgs[], int num)
{
	struct spi_msg *pmsg;
	int i, j, bptr;
	unsigned long flags;
	spinlock_t lock;
	
	spipnx_select_chip ();
	spi_regs.con |= SPI_CON_THR;
	
	for (i=0; i<num; i++) {
		pmsg = &msgs[i];

		if ( pmsg->flags & SPI_M_RD ) {                      /*  here we have to read data         */
			bptr = 0;
			init_completion (&op_complete);
			spi_regs.frm = 0x0000FFFF & (pmsg->len);     /*  tell the SPI how much we need     */
			spi_regs.con &= ~SPI_CON_RXTX;               /*  set mode to RX                    */
			spi_regs.con |= SPI_CON_SHIFT_OFF;           /*  this could affect, but how?       */
			spi_regs.ier = SPI_IER_SPI_INTEOT;           /*  enable end of transfer interrupt  */
			spi_regs.dat = 0x00;                         /*  dummy write to start transfer     */
			
			while (bptr < pmsg->len) {
				if ((pmsg->len)-bptr > FIFO_CHUNK_SIZE) {     /*  if there's data left for another  */
					init_completion (&threshold);                /*  init wait queue for another chunk */
					spin_lock_irq(&lock);
					spi_regs.ier |= SPI_IER_SPI_INTTHR;  /*  chunk, then enable THR interrupt  */
					spin_unlock_irq(&lock);
					wait_for_completion (&threshold);
				} else
					wait_for_completion (&op_complete);
				for (j=bptr;j<(((pmsg->len)-bptr<FIFO_CHUNK_SIZE)?(pmsg->len):(bptr+FIFO_CHUNK_SIZE));j++)
					pmsg->buf[j] = spi_regs.dat;
				bptr += ((pmsg->len)-bptr <FIFO_CHUNK_SIZE)?(pmsg->len - bptr):FIFO_CHUNK_SIZE;
			}
		} else {                                           /*  now we have to transmit data        */
			spi_regs.con |= SPI_CON_RXTX;
			spi_regs.frm = 0x0000FFFF & (pmsg->len);
			bptr = 0;
			init_completion (&op_complete);
			spi_regs.ier = SPI_IER_SPI_INTEOT;

			while (bptr < pmsg->len) {
				for (j=bptr;j<(((pmsg->len)-bptr<FIFO_CHUNK_SIZE)?(pmsg->len):(bptr+FIFO_CHUNK_SIZE));j++) {
					spi_regs.dat = pmsg->buf[j];
				}
				if ((pmsg->len)-bptr > FIFO_CHUNK_SIZE) {
					init_completion (&threshold);
					spin_lock_irq(&lock);
					spi_regs.ier |= SPI_IER_SPI_INTTHR;
					spin_unlock_irq(&lock);
					wait_for_completion (&threshold);
				}
				bptr += ((pmsg->len)-bptr <FIFO_CHUNK_SIZE)?(pmsg->len - bptr):FIFO_CHUNK_SIZE;
			}
			wait_for_completion (&op_complete);
		}
	}
	spipnx_unselect_chip ();
	spi_regs.ier &= ~SPI_IER_SPI_INTTHR;
	
	return num;
}


static int
spipnx_xfer_sdma (struct spi_adapter *adap, struct spi_msg msgs[], int num)
{
	struct spi_msg *pmsg;
	int i;
	
	spipnx_select_chip ();
	
	for (i=0; i<num; i++) {
		pmsg = &msgs[i];
		
		if ( pmsg->flags & SPI_M_RD ) {
			spi_regs.con &= ~SPI_CON_RXTX;
			spi_regs.con |= SPI_CON_SHIFT_OFF;    /*  disable generating clock while reading DAT reg  */
			spi_regs.frm = 0x0000FFFF & (pmsg->len);

			spi_regs.dat = 0;
			udelay(2*SPI_CLOCK);

			sdma_setup.src_address      = (unsigned int)pPhysSpiDataReg;
			sdma_setup.dest_address     = (unsigned int)dma_buffer;
			sdma_setup.trans_length     = pmsg->len;
			sdma_config.write_slave_nr  = SDMA_SLAVE_NR_MEMORY;
			sdma_config.read_slave_nr   = SDMA_SLAVE_NR_SPI;
			sdma_pack_config(&sdma_config, &sdma_setup.packed_config);
			sdma_prog_channel (spi_sdma_channel, &sdma_setup);

			init_completion (&op_complete);
			sdma_start_channel (spi_sdma_channel);
			wait_for_completion (&op_complete);
			sdma_stop_channel(spi_sdma_channel);
			memcpy ((void*)pmsg->buf, spi_buffer, pmsg->len);     /*     check buffer shifting    */
		} else {
			memcpy (spi_buffer, (void*)pmsg->buf, pmsg->len);
			spi_regs.con |= SPI_CON_RXTX;
			spi_regs.frm = 0x0000FFFF & (pmsg->len);
			spi_regs.con &= ~SPI_CON_SHIFT_OFF;
			
			sdma_setup.src_address      = (unsigned int)dma_buffer;
			sdma_setup.dest_address     = (unsigned int)pPhysSpiDataReg;
			sdma_setup.trans_length     = pmsg->len;
			sdma_config.write_slave_nr  = SDMA_SLAVE_NR_SPI;
			sdma_config.read_slave_nr   = SDMA_SLAVE_NR_MEMORY;
			sdma_pack_config(&sdma_config, &sdma_setup.packed_config);
			sdma_prog_channel (spi_sdma_channel, &sdma_setup);

			init_completion (&op_complete);
			sdma_start_channel (spi_sdma_channel);
			wait_for_completion (&op_complete);
			sdma_stop_channel(spi_sdma_channel);
			udelay(50*SPI_CLOCK);
		}
	}
	spipnx_unselect_chip ();
	
	return num;
}



static int
spipnx_xfer (struct spi_adapter *adap, struct spi_msg msgs[], int num)
{
	int status, retval, i;
	struct spi_msg *pmsg;
	
	DBG ("processing %d messages ...\n", num);
	for (i=0; i<num; i++) {
		pmsg = &msgs[i];
		if ((pmsg->len >= 0xFFFF) || (!pmsg->buf))
			return -EINVAL;
		DBG ("%d) %c - %d bytes\n", i, pmsg->flags?'R':'W', pmsg->len);
	}
	
	if (sdma_mode) {
		status = sdma_request_channel (sdma_ch_name, spipnx_sdma_cb, NULL);
		if (status < 0) {
			DBG ("No SDMA channel available\n");
			return -EBUSY;
		}
		spi_sdma_channel = (unsigned int)status;
		DBG ("acquired SDMA channel #%d\n", spi_sdma_channel);
		sdma_config.transfer_size     = SDMA_TRANSFER_BYTE;
		sdma_config.invert_endian     = SDMA_INVERT_ENDIAN_NO;
		sdma_config.companion_channel = 0;
		sdma_config.companion_enable  = SDMA_COMPANION_DISABLE;
		sdma_config.circular_buffer   = SDMA_CIRC_BUF_DISABLE;
	}

	if (sdma_mode)
		retval = spipnx_xfer_sdma (adap, msgs, num);
	else
		retval = spipnx_xfer_nonsdma (adap, msgs, num);

	if (sdma_mode) {
		status = sdma_release_channel (spi_sdma_channel);
		if (status < 0) {
			DBG ("Unable to release SDMA channel #%d\n", spi_sdma_channel);
			return -EIO;
		}
	}
	
	return retval;
}


static void spipnx_spi_init (void)
{
	spipnx_select_chip ();

	spi_regs.global = SPI_GLOBAL_BLRES_SPI | SPI_GLOBAL_SPI_ON;

	mdelay (5);
    
	spi_regs.global = SPI_GLOBAL_SPI_ON;                               /*  enable device once more             */
	spi_regs.con =  SPI_CON_SPI_MODE0 | SPI_CON_MS | SPI_CLOCK;
	spi_regs.con |= SPI_CON_SPI_BIDIR | (7 << 9);          /*  internal MUX setup(!) and set 8 bit transfers   */
	spipnx_unselect_chip();
}


static void __exit spipnx_cleanup(void)
{
	spi_regs.global = 0L;     /* disable SPI periph   */
	
	if (spipnx_initialized >= 3) {
		if ( spi_del_adapter (&spipnx_adapter) )
			DBG ("Deleting adapter '%s' failed\n", ADAPTER_NAME);
		spipnx_initialized--;
	}

	if (spipnx_initialized >= 2) {
		if (sdma_mode)
			consistent_free (spi_buffer, SPI_BUFFER_SIZE, dma_buffer);
		spipnx_initialized--;
	}
		
	if (spipnx_initialized >= 1) {
		free_irq(IRQ_SPI, NULL);
		spipnx_initialized--;
	}

	DBG ("module removed\n");
}


static int __init spipnx_init(void)
{
	unsigned int reg;
	spipnx_initialized = 0;
	DBG ("Loading spi-pnx0105.o in %s mode, SPI clock = PCLK/%d\n",\
			sdma_mode?"SMDA":"non-SDMA", (SPI_CLOCK+1)*2);
	
	reg = gpio_read_reg (PADMUX1_MODE1);                    /*  Select mode 1 on SPI_CE pin  */
	gpio_write_reg (reg | GPIO_PIN_SPI_CE, PADMUX1_MODE1);

	if (request_irq(IRQ_SPI, spipnx_interrupt, SA_INTERRUPT, "SPI", NULL) ) {
		DBG ("Unable to register IRQ\n");
		return -EBUSY;
	}
	spipnx_initialized++;
	
	if (sdma_mode) {
		spi_buffer = consistent_alloc (GFP_KERNEL, SPI_BUFFER_SIZE, &dma_buffer);
		if ( !spi_buffer ) {
			DBG ("Couldn't allocate buffer\n");
			free_irq(IRQ_SPI, NULL);
			return -ENOMEM;
		}
	}
	spipnx_initialized++;
	
	pPhysSpiDataReg = (void *)(SPI_BASE + SPI_DAT_REG_OFFSET);
	spipnx_spi_init ();
	if ( spi_add_adapter (&spipnx_adapter) ) {
		DBG ("adding adapter '%s' failed\n", ADAPTER_NAME);
		free_irq(IRQ_SPI, NULL);
		if (sdma_mode)
			consistent_free (spi_buffer, SPI_BUFFER_SIZE, dma_buffer);
		return -EFAULT;
	}
	spipnx_initialized++;
	
	memset(&sdma_config, 0, sizeof(sdma_config));
	memset(&sdma_setup, 0, sizeof(sdma_setup));

	printk ("SPI module for PNX0105 loaded.\n");
	return 0;
}


EXPORT_NO_SYMBOLS;

MODULE_AUTHOR("Andrey Ivolgin <aivolgin@ru.mvista.com>");
MODULE_DESCRIPTION("SPI driver for PNX0105.");
MODULE_LICENSE("GPL");

MODULE_PARM(sdma_mode, "i");
module_init(spipnx_init);
module_exit(spipnx_cleanup);
