/*
    spi-dev.c - spi-bus driver, char device interface  

    Copyright (C) 1995-97 Simon G. Vogl
    Copyright (C) 1998-99 Frodo Looijaard <frodol@dds.nl>
    Copyright (C) 2002 Compaq Computer Corporation

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* Adapted from i2c-dev module by Jamey Hicks <jamey.hicks@compaq.com> */

/* Note that this is a complete rewrite of Simon Vogl's i2c-dev module.
   But I have used so much of his original code and ideas that it seems
   only fair to recognize him as co-author -- Frodo */

/* The devfs code is contributed by Philipp Matthias Hahn 
   <pmhahn@titan.lahn.de> */

/* Modifications to allow work with current spi-core by 
   Andrey Ivolgin <aivolgin@ru.mvista.com>, Sep 2004
 */

/* devfs code corrected to support automatic device addition/deletion
   by Vitaly Wool <vwool@ru.mvista.com> (C) 2004 MontaVista Software, Inc. 
 */

/* $Id: spi-dev.c,v 1.1 2002/10/03 05:21:14 gdavis Exp $ */

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/slab.h>
#include <linux/version.h>
#if LINUX_KERNEL_VERSION >= KERNEL_VERSION(2,4,0)
#include <linux/smp_lock.h>
#endif /* LINUX_KERNEL_VERSION >= KERNEL_VERSION(2,4,0) */
#ifdef CONFIG_DEVFS_FS
#include <linux/devfs_fs_kernel.h>
#endif


/*  Define SPIDEV_DEBUG for debugging info  */
#undef SPIDEV_DEBUG

#ifdef SPIDEV_DEBUG
#define DBG(args...)	printk(KERN_INFO"spi-dev.o: " args)
#else
#define DBG(args...)
#endif


#include <linux/init.h>
#include <asm/uaccess.h>
#include <linux/spi/spi.h>


#define SPI_TRANSFER_MAX	65535
#define SPI_ADAP_MAX		32

/* struct file_operations changed too often in the 2.1 series for nice code */

#if LINUX_KERNEL_VERSION < KERNEL_VERSION(2,4,9)
static loff_t spidev_lseek (struct file *file, loff_t offset, int origin);
#endif
static ssize_t spidev_read (struct file *file, char *buf, size_t count, 
                            loff_t *offset);
static ssize_t spidev_write (struct file *file, const char *buf, size_t count, 
                             loff_t *offset);

static int spidev_open (struct inode *inode, struct file *file);
static int spidev_release (struct inode *inode, struct file *file);
static int spidev_ioctl (struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg);
static int spidev_attach_adapter (struct spi_adapter *);
static int spidev_detach_adapter (struct spi_adapter *);
static int __init spi_dev_init(void);
static void spidev_cleanup(void);

static struct file_operations spidev_fops = {
#if LINUX_KERNEL_VERSION >= KERNEL_VERSION(2,4,0)
	owner:		THIS_MODULE,
#endif /* LINUX_KERNEL_VERSION >= KERNEL_VERSION(2,4,0) */
#if LINUX_KERNEL_VERSION < KERNEL_VERSION(2,4,9)
	llseek:		spidev_lseek,
#else
	llseek:		no_llseek,
#endif
	read:		spidev_read,
	write:		spidev_write,
	open:		spidev_open,
	release:	spidev_release,
	ioctl:		spidev_ioctl,
};


static struct spi_adapter *spidev_adaps[SPI_ADAP_MAX];

#ifdef CONFIG_DEVFS_FS
static devfs_handle_t devfs_spi[SPI_ADAP_MAX];
static devfs_handle_t devfs_handle = NULL;
#endif

static struct spi_driver spidev_driver = {
	name:		"spi",
	attach_adapter: spidev_attach_adapter,
	detach_adapter: spidev_detach_adapter,
	owner:		THIS_MODULE,
};

static struct spi_client spidev_client_template = {
	driver:		&spidev_driver
};


static int spidev_initialized;


#if LINUX_KERNEL_VERSION < KERNEL_VERSION(2,4,9)
/* Note that the lseek function is called llseek in 2.1 kernels. But things
   are complicated enough as is. */
loff_t spidev_lseek (struct file *file, loff_t offset, int origin)
{
#ifdef SPIDEV_DEBUG	
	struct inode *inode = file->f_dentry->d_inode;
#endif
	DBG("spi-%d lseek to %ld bytes relative to %d.\n",
	       MINOR(inode->i_rdev),(long) offset,origin);

	return -ESPIPE;
}
#endif


static int spidev_ioctl (struct inode *inode, struct file *file, unsigned int cmd,
				unsigned long arg)
{
	struct spi_client *client = (struct spi_client *)file->private_data;
	struct spi_rdwr_ioctl_data rdwr_arg;
	struct spi_msg *rdwr_pa;
	int res, i;

	DBG ("spi-%d ioctl, cmd: 0x%x, arg: %lx.\n",
		MINOR (inode->i_rdev), cmd, arg);

	switch (cmd)  {
	case SPI_RDWR:
		if (copy_from_user(&rdwr_arg, 
				   (struct spi_rdwr_ioctl_data *)arg, 
				   sizeof(rdwr_arg)))
			return -EFAULT;

		rdwr_pa = (struct spi_msg *)
			kmalloc(rdwr_arg.nmsgs * sizeof(struct spi_msg), 
			GFP_KERNEL);

		if (rdwr_pa == NULL) return -ENOMEM;

		res = 0;
		for( i=0; i<rdwr_arg.nmsgs; i++ )  {
		    	if(copy_from_user(&(rdwr_pa[i]),
					&(rdwr_arg.msgs[i]),
					sizeof(rdwr_pa[i])))  {
			        res = -EFAULT;
				break;
			}
			rdwr_pa[i].buf = kmalloc(rdwr_pa[i].len, GFP_KERNEL);
			if(rdwr_pa[i].buf == NULL)  {
				res = -ENOMEM;
				break;
			}
			if(copy_from_user(rdwr_pa[i].buf,
				rdwr_arg.msgs[i].buf,
				rdwr_pa[i].len))  {
			    	kfree(rdwr_pa[i].buf);
			    	res = -EFAULT;
				break;
			}
		}
		if (!res)  {
			res = spi_transfer(client->adapter,
				rdwr_pa,
				rdwr_arg.nmsgs);
		}
		while(i-- > 0)
		{
			if( res>=0 && (rdwr_pa[i].flags & SPI_M_RD)) {
				if(copy_to_user(
					rdwr_arg.msgs[i].buf,
					rdwr_pa[i].buf,
					rdwr_pa[i].len)) {
					res = -EFAULT;
				}
			}
			kfree(rdwr_pa[i].buf);
		}
		kfree(rdwr_pa);
		return res;
		
	case SPI_CLK_RATE:
	case SPI_SDMA:
	default:
		break;
	}
		
	return 0;
}


static int spidev_attach_adapter (struct spi_adapter *adap)
{
	int i;
	char name[8];

	DBG ("attaching adapter '%s'\n", adap->name);
	for (i=0; i<SPI_ADAP_MAX; i++)
		if (!spidev_adaps[i]) {
			spidev_adaps[i] = adap;
#ifdef CONFIG_DEVFS_FS
			sprintf(name, "%d", i);
			devfs_spi[i] = devfs_register (devfs_handle, name,
				DEVFS_FL_DEFAULT, SPI_MAJOR, i,
				S_IFCHR | S_IRUSR | S_IWUSR,
				&spidev_fops, NULL);
#endif

			MOD_INC_USE_COUNT;
			return 0;
		}
	return -ENOMEM;
}


static int spidev_detach_adapter (struct spi_adapter *adap)
{
	int i;
	
	DBG ("detaching adapter '%s'\n", adap->name);
	for (i=0; i<SPI_ADAP_MAX; i++)
		if (spidev_adaps[i] == adap) {
#ifdef CONFIG_DEVFS_FS
			devfs_unregister(devfs_spi[i]);
#endif
			spidev_adaps[i] = NULL;
			MOD_DEC_USE_COUNT;
			return 0;
		}

	DBG ("trying to detach unknown adapter '%s'!\n", adap->name);
	return -EINVAL;
}


static ssize_t spidev_read (struct file *file, char *buf, size_t count,
                            loff_t *offset)
{
	char *tmp;
	int ret;
#ifdef SPIDEV_DEBUG
	struct inode *inode = file->f_dentry->d_inode;
#endif

	struct spi_client *client = (struct spi_client *)file->private_data;
	if(count > SPI_TRANSFER_MAX)
		count = SPI_TRANSFER_MAX;
	
	/* copy user space data to kernel space. */
	tmp = kmalloc(count,GFP_KERNEL);
	if (tmp == NULL)
		return -ENOMEM;

	DBG ("spi-%d reading %d bytes.\n",MINOR(inode->i_rdev),
	       count);

	ret = spi_read (client,0,tmp,count);
	if (ret >= 0)
		ret = copy_to_user(buf,tmp,count)?-EFAULT:ret;
	kfree(tmp);
	return ret;
}


static ssize_t spidev_write (struct file *file, const char *buf, size_t count,
                             loff_t *offset)
{
	int ret;
	char *tmp;
	struct spi_client *client = (struct spi_client *)file->private_data;
#ifdef SPIDEV_DEBUG
	struct inode *inode = file->f_dentry->d_inode;
#endif
	
	if(count > SPI_TRANSFER_MAX)
		count = SPI_TRANSFER_MAX;

	/* copy user space data to kernel space. */
	tmp = kmalloc(count,GFP_KERNEL);
	if (tmp == NULL)
		return -ENOMEM;
	if (copy_from_user(tmp,buf,count)) {
		kfree(tmp);
		return -EFAULT;
	}

	DBG ("spi-%d writing %d bytes.\n",MINOR(inode->i_rdev), count);
	ret = spi_write(client,0,tmp,count);
	kfree(tmp);
	return ret;
}


int spidev_open (struct inode *inode, struct file *file)
{
	unsigned int minor = MINOR(inode->i_rdev);
	struct spi_client *client;

	if ((minor >= SPI_ADAP_MAX) || (!(spidev_adaps[minor]))) {
		DBG ("trying to open unattached adapter spi-%d\n", minor);
		return -ENODEV;
	}
	 
	/* Note that we here allocate a client for later use, but we will *not*
	   register this client! Yes, this is safe. No, it is not very clean. */
	if(! (client = kmalloc(sizeof(struct spi_client),GFP_KERNEL)))
		return -ENOMEM;
	memcpy(client,&spidev_client_template,sizeof(struct spi_client));
	client->adapter = spidev_adaps[minor];
	file->private_data = client;
	DBG ("opened spi-%d, adapter\n",minor);
	
	return 0;
}


static int spidev_release (struct inode *inode, struct file *file)
{
#ifdef SPIDEV_DEBUG
	unsigned int minor = MINOR(inode->i_rdev);
#endif

	kfree(file->private_data);
	file->private_data=NULL;
	DBG ("Closed: spi-%d\n", minor);

	return 0;
}


static int __init spi_dev_init(void)
{
	int res;

	memset (spidev_adaps, 0, sizeof (struct spi_adapter*) * SPI_ADAP_MAX);
	DBG ("spi /dev entries driver module\n");

	spidev_initialized = 0;
#ifdef CONFIG_DEVFS_FS
	if (devfs_register_chrdev(SPI_MAJOR, "spi", &spidev_fops)) {
#else
	if (register_chrdev(SPI_MAJOR,"spi",&spidev_fops)) {
#endif
		DBG ("unable to get major %d for spi bus\n",
		       SPI_MAJOR);
		return -EIO;
	}
#ifdef CONFIG_DEVFS_FS
	devfs_handle = devfs_mk_dir(NULL, "spi", NULL);
#endif
	spidev_initialized ++;

	if ((res = spi_add_driver(&spidev_driver))) {
		DBG ("driver registration failed, module not inserted.\n");
		spidev_cleanup();
		return res;
	}
	spidev_initialized ++;
	
	DBG ("module loaded.\n");
	return 0;
}


static void spidev_cleanup(void)
{
	int res;
	
	if (spidev_initialized >= 2) {
		if ((res = spi_del_driver(&spidev_driver))) {
			DBG ("driver deregistration failed, "
			       "module not removed.\n");
			return;
		}
		spidev_initialized --;
	}

	if (spidev_initialized >= 1) {
#ifdef CONFIG_DEVFS_FS
		devfs_unregister(devfs_handle);
		if ((res = devfs_unregister_chrdev(SPI_MAJOR, "spi"))) {
#else
		if ((res = unregister_chrdev(SPI_MAJOR,"spi"))) {
#endif
			DBG ("unable to release major %d for spi bus\n",
			       SPI_MAJOR);
			return;
		}
		spidev_initialized --;
	}
}


EXPORT_NO_SYMBOLS;

MODULE_AUTHOR("Jamey Hicks <jamey.hicks@compaq.com> Frodo Looijaard <frodol@dds.nl> and Simon G. Vogl <simon@tk.uni-linz.ac.at>");
MODULE_DESCRIPTION("SPI /dev entries driver");
MODULE_LICENSE("GPL");

module_init(spi_dev_init);
module_exit(spidev_cleanup);

