/*
  * drivers/sound/omap1610-aic23.c
  *
  * Glue audio driver for the TI OMAP1610/11 & TI AIC23 CODEC
  *
  * Author: MontaVista Software, Inc. <source@mvista.com>
  *
  * Based on omap1510-aic23.c:
  * Copyright (c) 2000 Nicolas Pitre <nico@cam.org>
  * Copyright (C) 2001, Steve Johnson <stevej@ridgerun.com>
  *
  * 2004 (c) MontaVista Software, Inc. This file is licensed under
  * the terms of the GNU General Public License version 2. This program
  * is licensed "as is" without any warranty of any kind, whether express
  * or implied.
  */
#include <linux/module.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/slab.h>
#include <linux/errno.h>
#include <linux/sound.h>
#include <linux/soundcard.h>
#include <linux/i2c.h>
#include <linux/bitops.h>
#ifdef CONFIG_OMAP_OSK
#include <linux/tps65010.h>
#endif

#include <asm/semaphore.h>
#include <asm/uaccess.h>
#include <asm/hardware.h>
#include <asm/dma.h>
#include <asm/io.h>
#include <asm/hardware.h>
#include <asm/arch/ck.h>
#include <linux/kernel.h>

#include "omap-audio.h"
#include <asm/arch/mcbsp.h>
#include "aic23.h"


#define AUDIO_NAME		"OMAP1610_AIC23"

#define AUDIO_RATE_DEFAULT	44100

/*  Local prototypes */
static void     omap1610_audio_init(void *dummy);
static void     omap1610_audio_shutdown(void *dummy);
static int      omap1610_audio_ioctl(struct inode *inode, struct file *file,
				     uint cmd, ulong arg);
/* I2C stuff */
struct i2c_driver aic23_driver;
struct i2c_client *aic23_i2c_client;

static unsigned short ignore[] = { I2C_CLIENT_END };
static unsigned short normal_addr[] = { DEV_AIC23_ID, I2C_CLIENT_END };
static unsigned short normal_addr_range[] = { 0x1a, 0x1b, I2C_CLIENT_END };

static struct i2c_client_address_data addr_data = {
	.normal_i2c		= normal_addr,
	.normal_i2c_range	= normal_addr_range,
	.probe			= ignore,
	.probe_range		= ignore,
	.ignore			= ignore,
	.ignore_range		= ignore,
	.force			= ignore,
};

static int aic23_probe(struct i2c_adapter *adap);
static int aic23_detach(struct i2c_client *client);
static int aic23_command(struct i2c_client *client, unsigned int cmd,
			   void *arg);

struct i2c_driver aic23_driver = {
	.name		= "OMAP161x+TLV320AIC23",
	.id		= I2C_DRIVERID_EXP0,	/* Fake Id */
	.flags		= I2C_DF_NOTIFY,
	.attach_adapter	= aic23_probe,
	.detach_client	= aic23_detach,
	.command	= aic23_command
};

static int
aic23_attach(struct i2c_adapter *adap, int addr, unsigned short flags,
		int kind)
{
	struct i2c_client *c;

	c = (struct i2c_client *) kmalloc(sizeof (*c), GFP_KERNEL);

	if (!c)
		return -ENOMEM;

	strcpy(c->name, "OMAP161x+TLV320AIC23");
	c->id = aic23_driver.id;
	c->flags = 0;
	c->addr = addr;
	c->adapter = adap;
	c->driver = &aic23_driver;
	c->data = NULL;

	aic23_i2c_client = c;

	return i2c_attach_client(c);
}

static int
aic23_probe(struct i2c_adapter *adap)
{
	return i2c_probe(adap, &addr_data, aic23_attach);
}

static int
aic23_detach(struct i2c_client *client)
{
	i2c_detach_client(client);
	kfree(aic23_i2c_client);
	return 0;
}

/* No commands defined */
static int 
aic23_command(struct i2c_client *client, unsigned int cmd, void *arg)
{
	return 0;
}

/*
 * Simplified write routine.  I know the i2c message
 * is just two bytes and won't be more.
*/
static void
omap1610_write_i2c(u8 address, u16 data)
{
	u8              buf[2];
	int             ret = 0;
	int             retries = 5;

	pr_debug("omap1610_write_i2c: addr 0x%02x, data 0x%04x\n", address, data);
	buf[0] = ((address << 1) | ((data & 0x100) >> 8));
	buf[1] = (data & 0xff);

        while ((ret != sizeof(buf)) && (retries--)) {
                if (sizeof(buf) != (ret = i2c_master_send(aic23_i2c_client, 
					(const char *)buf, sizeof(buf)))) {
		    pr_debug("omap1610_write_i2c:"
			    " expected %d, received %d, address 0x%x\n",
			    sizeof (buf), ret, address);
		    pr_debug("retries - %d\n", retries);
                }
        }

}

static int input_or_output = 0;

static audio_stream_t output_stream = {
	id:		"AIC23 out",
	dma_dev:	eAudioTx,
};

static audio_stream_t input_stream = {
	id:		"AIC23 in",
	dma_dev:	eAudioRx,
};

static audio_state_t audio_state = {
	output_stream:	&output_stream,
	input_stream:	&input_stream,
	need_tx_for_rx:	0,
	hw_init:	omap1610_audio_init,
	hw_shutdown:	omap1610_audio_shutdown,
	client_ioctl:	omap1610_audio_ioctl,
	sem:		__MUTEX_INITIALIZER(audio_state.sem),
};

#define REC_MASK (SOUND_MASK_LINE | SOUND_MASK_MIC)
#define DEV_MASK (REC_MASK | SOUND_MASK_VOLUME)

#define SET_VOLUME 1
#define SET_LINEIN_VOLUME 2
#define SET_RECSRC 3

#define DEFAULT_VOLUME          93
#define DEFAULT_LININ_VOLUME    93	/* 0 ==> mute line in */
					/* use input vol of 75 for 0dB gain */
#define DEFAULT_MICIN_VOLUME    100

#define OUTPUT_VOLUME_MIN 	0x30
#define OUTPUT_VOLUME_MAX 	LHV_MAX
#define OUTPUT_VOLUME_RANGE 	(OUTPUT_VOLUME_MAX - OUTPUT_VOLUME_MIN)
#define OUTPUT_VOLUME_MASK 	OUTPUT_VOLUME_MAX

#define LININ_VOLUME_MIN 	0x0
#define LININ_VOLUME_MAX 	LIV_MAX
#define LININ_VOLUME_RANGE 	(LININ_VOLUME_MAX - LININ_VOLUME_MIN)
#define LININ_VOLUME_MASK 	LININ_VOLUME_MAX

typedef struct {
	u8	volume;
	u16	volume_reg;
	u8	line;
	u8	mic;
	u16	input_volume_reg;
	int	recsrc;	   /* currently enabled recording device */
	int	mod_cnt;
} aic23_local_t;

static aic23_local_t aic23_local = {
	volume:			DEFAULT_VOLUME,
	volume_reg:		LZC_ON,
	line:			DEFAULT_LININ_VOLUME,
	mic:			DEFAULT_MICIN_VOLUME,
	input_volume_reg:	0, 
	recsrc:			SOUND_MASK_LINE,
	mod_cnt:		0 
};

static void
aic23_update(int flag, unsigned int val)
{
	u8 volume;

	switch (flag) {
		
		case SET_VOLUME:
			aic23_local.volume_reg &= ~OUTPUT_VOLUME_MASK;
			/* Convert 0->100 volume to  0x30->0x7f volume range */
			volume = ((val * OUTPUT_VOLUME_RANGE) / 100) + 
				OUTPUT_VOLUME_MIN;
			aic23_local.volume_reg |= volume;
			omap1610_write_i2c(LEFT_CHANNEL_VOLUME_ADDR,
						aic23_local.volume_reg);
			omap1610_write_i2c(RIGHT_CHANNEL_VOLUME_ADDR,
						aic23_local.volume_reg);
			break;

		case SET_LINEIN_VOLUME:
			/* [4:0] 12dB to -34.5dB in 1.5 steps 10111=0 */
			aic23_local.input_volume_reg &= ~LININ_VOLUME_MASK;
			/* Convert 0->100 volume to 0x0->0x1f volume range */
			volume = ((val * LININ_VOLUME_RANGE) / 100) + 
				LININ_VOLUME_MIN;
			aic23_local.input_volume_reg |= volume;
			omap1610_write_i2c(LEFT_LINE_VOLUME_ADDR,
					aic23_local.input_volume_reg);
			omap1610_write_i2c(RIGHT_LINE_VOLUME_ADDR,
					aic23_local.input_volume_reg);
			break;

		case SET_RECSRC: {
				u16 reg = DAC_SELECTED;

				/*
				 * If more than one recording device selected,
				 * disable the device that is currently in use.
				 */
				if (hweight32(val) > 1)
					val &= ~aic23_local.recsrc;

				if (val == SOUND_MASK_MIC) {
					reg |= (INSEL_MIC | MICB_20DB);
					omap1610_write_i2c(ANALOG_AUDIO_CONTROL_ADDR, 
		 					reg);
				}
				else if (val == SOUND_MASK_LINE) {
					omap1610_write_i2c(ANALOG_AUDIO_CONTROL_ADDR, 
					reg);
				}
				else {
					printk(KERN_WARNING
					"omap1610-aic23: Wrong RECSRC value"
					" specified\n");
					return;
				}
				aic23_local.recsrc = val;
				break;
			}
		default:
			printk(KERN_WARNING "omap1610-aic23: Wrong aic23_update "
				"flag specified\n");
			break;
	  }
}

static int
mixer_ioctl(struct inode *inode, struct file *file, uint cmd, ulong arg)
{
	int	val;
	int	gain;
	int	ret = 0;
	int	nr = _IOC_NR(cmd);

	/*
	 * We only accept mixer (type 'M') ioctls.
	 */
	if (_IOC_TYPE(cmd) != 'M')
		return -EINVAL;

	pr_debug("mixer_ioctl: 0x%08x\n", cmd);

	if (cmd == SOUND_MIXER_INFO) {
		struct mixer_info mi;

		strncpy(mi.id, "AIC23", sizeof (mi.id));
		strncpy(mi.name, "TI AIC23", sizeof (mi.name));
		mi.modify_counter = aic23_local.mod_cnt;
		return copy_to_user((void *) arg, &mi, sizeof (mi));
	}

	if (_IOC_DIR(cmd) & _IOC_WRITE) {
		ret = get_user(val, (int *) arg);
		if (ret)
			goto out;

		/* Ignore separate left/right channel for now */
		gain = val & 0xff;

		switch (nr) {
			case SOUND_MIXER_VOLUME:
				pr_debug("mixer_ioctl: write "
					"SOUND_MIXER_VOLUME val=0x%x\n", val);

				aic23_local.volume = val;
				aic23_local.mod_cnt++;
				aic23_update(SET_VOLUME, gain);
				break;

			case SOUND_MIXER_LINE:
				pr_debug("mixer_ioctl: write "
					"SOUND_MIXER_LINE val=0x%x\n", val);

				aic23_local.line = val;
				aic23_local.mod_cnt++;
				aic23_update(SET_LINEIN_VOLUME, gain);
				break;

			case SOUND_MIXER_MIC:
				pr_debug("mixer_ioctl: write "
					"SOUND_MIXER_MIC val=0x%x\n", val);
				break;

			case SOUND_MIXER_RECSRC:
				pr_debug("mixer_ioctl: write "
					"SOUND_MIXER_RECSRC val=0x%x\n", val);

				if ((val & SOUND_MASK_LINE) || 
				    (val & SOUND_MASK_MIC)) {
					if (aic23_local.recsrc != val) {
						aic23_local.mod_cnt++;
						aic23_update(SET_RECSRC, val);
					}
				}
				else {
					ret = -EINVAL;
				}
				break;
			default:
				printk(KERN_WARNING "omap1610-aic23: unknown mixer write " 
					"ioctl flag specified\n");
				ret = -EINVAL;
				break;
		}
	}

	if (ret == 0 && _IOC_DIR(cmd) & _IOC_READ) {
		ret = 0;

		switch (nr) {
			case SOUND_MIXER_VOLUME:
				val = (aic23_local.volume << 8) | 
					aic23_local.volume;
				pr_debug("mixer_ioctl: read "
					"SOUND_MIXER_VOLUME val=0x%x\n", val);
				break;
			case SOUND_MIXER_LINE:
				val = (aic23_local.line << 8) | 
					aic23_local.line;
				pr_debug("mixer_ioctl: read "
					"SOUND_MIXER_LINE val=0x%x\n", val);
				break;
			case SOUND_MIXER_MIC:
				val = (aic23_local.mic << 8) | 
					aic23_local.mic;
				pr_debug("mixer_ioctl: read "
					"SOUND_MIXER_MIC val=0x%x\n", val);
				break;
			case SOUND_MIXER_RECSRC:
				val = aic23_local.recsrc;
				pr_debug("mixer_ioctl: read "
					"SOUND_MIXER_RECSRC val=0x%x\n", val);
				break;
			case SOUND_MIXER_RECMASK:
				val = REC_MASK;
				pr_debug("mixer_ioctl: read "
					"SOUND_MIXER_RECMASK val=0x%x\n", val);
				break;
			case SOUND_MIXER_DEVMASK:
				val = DEV_MASK;
				pr_debug("mixer_ioctl: read "
					"SOUND_MIXER_DEVMASK val=0x%x\n", val);
				break;
			case SOUND_MIXER_CAPS:
				val = SOUND_CAP_EXCL_INPUT;
				pr_debug("mixer_ioctl: read "
					"SOUND_MIXER_CAPS val=0x%x\n", val);
				break;
			case SOUND_MIXER_STEREODEVS:
				val = (SOUND_MASK_VOLUME | SOUND_MASK_LINE);
				pr_debug("mixer_ioctl: read "
					"SOUND_MIXER_STEREODEVS val=0x%x\n", val);
				break;
			default:
				val = 0;
				printk(KERN_WARNING "omap1610-aic23: unknown mixer read " 
					"ioctl flag specified\n");
				ret = -EINVAL;
				break;
		}

		if (ret == 0)
			ret = put_user(val, (int *) arg);
	}
out:
	return ret;
}

static struct file_operations omap1610_mixer_fops = {
	ioctl:	mixer_ioctl,
	owner:	THIS_MODULE
};

/*
 * Audio interface
 */

static long audio_samplerate = AUDIO_RATE_DEFAULT;

static void
omap1610_set_samplerate(long val)
{
	int	clock_divider = 0;
	int	sampling_rate = 0;
	int	bosr = 0;
	u16	tmp;
	
	/* We don't want to mess with clocks when frames are in flight */
	/* wait for any frame to complete */
	udelay(125);

	pr_debug("omap1610_set_samplerate: %ld\n", val);

#define CODEC_CLOCK 12000000
#if (CODEC_CLOCK == 12000000)

	/*
	 * We have the following clock sources:
	 * 12.000 MHz
	 *
	 *  Available sampling rates:
	 *  96kHz, 88.2KHz, 48kHz, 44.1kHz, 32kHz, 8kHz,
	 *  and half of those 24kHz, 16kHz, (4kHz)
	 *  Won't bother supporting those in ().
	 */
	if (val >= 96000)
		val = 96000;
	else if (val >= 88200)
		val = 88200;
	else if (val >= 48000)
		val = 48000;
	else if (val >= 44100)
		val = 44100;
	else if (val >= 32000)
		val = 32000;
	else if (val >= 24000)
		val = 24000;
	else if (val >= 16000)
		val = 16000;
	else
		val = 8000;

	/* Set the clock divider */
	switch (val) {
		case 24000:
		case 16000:
			clock_divider = CLKIN_HALF;
			break;
		default:
		break;
	}

	/* Set the sampling rate bits */
	bosr = 0;
	switch (val) {
		case 96000:
			sampling_rate = 0x07;
			break;
		case 88200:
			bosr = 1;
			sampling_rate = 0x0f;
			break;
		case 48000:
		case 24000:
			sampling_rate = 0;
			break;
		case 44100:
			bosr = 1;
			sampling_rate = 0x08;
			break;
		case 32000:
		case 16000:
			sampling_rate = 6;
			break;
		case 8000:
			sampling_rate = 3;
			break;
	}

	/* Set sample rate at CODEC */
	tmp = (clock_divider | (sampling_rate << SR_SHIFT) |
		(bosr << BOSR_SHIFT) | 
                0x01); /* Last bit uses USB 12MHz clk mode, supported on OSK5912 */
	omap1610_write_i2c(SAMPLE_RATE_CONTROL_ADDR, tmp);
#elif (CODEC_CLOCK == 16934400)

	/*
	 * We have the following clock sources:
	 * 16.9344 MHz
	 *
	 *  Available sampling rates:
	 *  88.2KHz, 44.1kHz, 22.05kHz
	 */
	if (val >= 88200)
		val = 88200;
	else if (val >= 44100)
		val = 44100;
	else if (val >= 22050)
		val = 22050;
	else
		val = 8018;

	/* Set the sampling rate bits */
	bosr = 1;
	switch (val) {
		case 88200:
			sampling_rate = 0x0f;
			break;
		case 44100:
			sampling_rate = 0x08;
			break;
		case 22050:
			clock_divider = CLKIN_HALF;
			sampling_rate = 0x08;
			break;
		case 8018:
			sampling_rate = 0x0b;
			break;
	}

	/* Set sample rate at CODEC */
	tmp = (clock_divider | (sampling_rate << SR_SHIFT) |
	     (bosr << BOSR_SHIFT));
	omap1610_write_i2c(SAMPLE_RATE_CONTROL_ADDR, tmp);
#else
#error This CODEC_CLOCK setting is not yet supported by driver.
#endif

	audio_samplerate = val;
}

static void
omap1610_audio_init(void *dummy)
{
	dma_regs_t     *dma_regs;
	u16             tmp;
	
	/* Reset codec */
	omap1610_write_i2c(RESET_CONTROL_ADDR, 0);

	/*
	 *  Now here's an ugly hack.  To use McBSP1, you need to enable
	 *  a clock on the DSP.  So enable the MPUI, set the clock,
	 *  and start the DSP.  
	 *
	 *  An even uglier, evil hack.  If this is going to take the DSP
	 *  out of reset, shove an idle loop at the reset vector
	 *  and make it loop instead of crash.  You will still see
	 *  a DSP watchdog timer go off.
	 *
	 *  I would prefer having a DSP program (probably the MP3 decoder) set
	 *  up the DSP, but this allows an ARM-only MP3 decoder to work.  With
	 *  this code, the DSP is never put into idle, so the OMAP chip cannot 
	 *  go into any decent low-power mode.  Also, all of the DSP interface
	 *  dependencies (such as MPUI_STROBE settings) would be taken care
	 *  of.  Plus, we wouldn't have to worry about different boot vector
	 *  addresses depending on the chip version.
	 */
	{
		u32 boot_vector;
		u8 c55_start[] = { 0x7A, 0x00, 0x00, 0x0C, 0x4A, 0x7A, 0x20, 
				0x20, 0x20, 0x20 };

		tmp = inw(0xfffece10);	/* read ARM_RSTCT1 register */
		if (0 == (tmp & 0x6)) {
		 	/* check if DSP is up */
			pr_debug("omap1610_audio_init: "
				"Bringing DSP out of reset.\n");
			if (0 == (tmp & 0x4)) {	
				/* MPUI in reset */
				tmp |= 0x4;
				outw(tmp, 0xfffece10);
				ck_enable(api_ck);
			}
			tmp = inw(0xe1008008);	/* read DSP_IDLECT2 register */
			if (6 != (tmp & 0x6)) {	/* DSP CLKM enable */
				tmp |= 0x6;
				outw(tmp, 0xe1008008);
			}
			tmp = inw(0xe1008014);	/* read DSP_RSTCT2 register */
			if (3 != (tmp & 0x3)) {	/* DSP PER_EN bit */
				tmp |= 0x3;
				outw(tmp, 0xe1008014);
			}
			tmp = inw(0xfffece00);	/* read ARM_CKCTL register */
			tmp |= 0x2000;
			outw(tmp, 0xfffece00);
			/* Write C55 code at reset vector */
			pr_debug("omap1610_audio_init: "
				"Bringing DSP out of reset - in progress.\n");
			boot_vector = 0x10000;

			memcpy((void *) (OMAP_DSP_BASE + boot_vector),
				&c55_start, sizeof (c55_start));
			outw(0x5, 0xfffec918);	/* Set DSP boot mode */
			tmp = inw(0xfffece10);	/* take DSP out of reset */
			tmp |= 0x6;
			outw(tmp, 0xfffece10);
		} else { /*  DSP's up, just check the clock/per bits */
			tmp = inw(0xe1008008);
	                /* Set EN_PERCK bit to enable external peripheral clock (OSK5912) */
	                tmp |= 0x04;
			outw(tmp, 0xe1008008);
			if (0 == (tmp & 0x2)) {	/* DSP CLKM enable */
				tmp |= 0x2;
				outw(tmp, 0xe1008008);
			}
			tmp = inw(0xe1008014);
	                /* Set DSP PER_EN bit to reset external peripherals connected to DSP (OSK5912)	*/	
			tmp |= 0x02;
			outw(tmp, 0xe1008014);
			if (0 == (tmp & 0x1)) {	/* DSP PER_EN bit */
				tmp |= 0x1;
				outw(tmp, 0xe1008014);
			}
		}
	}

	/* Configure the DMA channel and MCBSP */

	if (input_or_output & FMODE_WRITE) {
		dma_regs = output_stream.dma_regs;
		pr_debug("omap1610_audio_init: Setup DMA channel to McBSP1 "
			 "audio Tx dma_regs: %p\n", dma_regs);
		dma_regs->csdp = 0x0a01;
		/* source auto increment, don't enable yet */
		dma_regs->ccr = 0x1000 | audio_state.output_stream->dma_dev;	
		dma_regs->cicr = 0x23;
		dma_regs->cdsa_l = (AUDIO_DXR1 & 0xffff); 
		dma_regs->cdsa_u = (AUDIO_DXR1 >> 16);
		dma_regs->cfn = 0x1;
		omap_dma_setup(audio_state.output_stream->dma_dev, eDmaOut);
	}

	if (input_or_output & FMODE_READ) {
		dma_regs = input_stream.dma_regs;
		pr_debug("omap1610_audio_init: Setup DMA channel to McBSP1 "
			 "audio Rx dma_regs: %p\n", dma_regs);
		dma_regs->csdp = 0x0015;
		/* destn auto increment, don't enable yet */
		dma_regs->ccr = 0x4000 | audio_state.input_stream->dma_dev;	
		dma_regs->cicr = 0x23;
		dma_regs->cssa_l = (AUDIO_DRR1 & 0xffff);
		dma_regs->cssa_u = (AUDIO_DRR1 >> 16);
		dma_regs->cfn = 0x1;
		omap_dma_setup(audio_state.input_stream->dma_dev, eDmaIn);
	}

	/* Initialize McBSP Digital Audio channel in DSP mode */

	outw(0x0000, AUDIO_SPCR1);	/* disable */
	outw(0x0000, AUDIO_SPCR2);	/* disable */

	/* configure MCBSP to be the slave */

	outw((CLKXM | CLKRM | FSXP | FSRP | CLKXP | CLKRP | SCLKME),
	     AUDIO_PCR0);

	/* 16 bits per Receive/Transmit word, 2 word per frame */
	outw((XFRLEN1(1) | XWDLEN1(2)), AUDIO_XCR1);
	outw(XFRLEN2(0) | XWDLEN2(2) | XDATDLY(0) | XFIG, AUDIO_XCR2);
	outw((RFRLEN1(1) | RWDLEN1(2)), AUDIO_RCR1);
	outw((RFRLEN2(0) | RWDLEN2(2) | RDATDLY(0) | RFIG), AUDIO_RCR2);
	
	/* enable input */
	outw((RINTM(3) | RRST), AUDIO_SPCR1);	
	/* enable output */
	outw((FREE | FRST | GRST | XRST | XINTM(3)), AUDIO_SPCR2); 

	outw((FWID(15)), AUDIO_SRGR1);
	outw((GSYNC | CLKSP | CLKSM | FSGM | FPER(31)), AUDIO_SRGR2);

	pr_debug("omap1610_audio_init: McBSP enabled\n");
	pr_debug
	    ("McBSP1:  DRR2 =0x%04x, DRR1 =0x%04x, DXR2 =0x%04x, DXR1 =0x%04x\n",
	     inw(AUDIO_DRR2), inw(AUDIO_DRR1), inw(AUDIO_DXR2),
	     inw(AUDIO_DXR1));
	pr_debug
	    ("McBSP1:  SPCR2=0x%04x, SPCR1=0x%04x, RCR2 =0x%04x, RCR1 = 0x%04x\n",
	     inw(AUDIO_SPCR2), inw(AUDIO_SPCR1), inw(AUDIO_RCR2),
	     inw(AUDIO_RCR1));
	pr_debug
	    ("McBSP1:  XCR2 =0x%04x, XCR1 =0x%04x, SRGR2=0x%04x, SRGR1=0x%04x\n",
	     inw(AUDIO_XCR2), inw(AUDIO_XCR1), inw(AUDIO_SRGR2),
	     inw(AUDIO_SRGR1));
	pr_debug
	    ("McBSP1:  MCR2 =0x%04x, MCR1 =0x%04x, RCERA=0x%04x, RCERB=0x%04x\n",
	     inw(AUDIO_MCR2), inw(AUDIO_MCR1), inw(AUDIO_RCERA),
	     inw(AUDIO_RCERB));
	pr_debug("McBSP1:  XCERA=0x%04x, XCERB=0x%04x, PCR0 =0x%04x\n",
		inw(AUDIO_XCERA), inw(AUDIO_XCERB), inw(AUDIO_PCR0));

	/* Initialize the AIC23 internal state */

	/*  
	 * The AIC23 uses 9 bits for register control.  The
	 * extra bit gets placed in the LSB of the subregister
	 * address, and the address is shifted by one.
	 */

	/* Left/Right line input volume control */
	aic23_update(SET_LINEIN_VOLUME, aic23_local.line);

	/* Left/Right headphone channel volume control */
	aic23_update(SET_VOLUME, aic23_local.volume);

	/* 
	 * Analog audio path control turn on DAC, select input 
	 * recording source 
	 */
	aic23_update(SET_RECSRC, aic23_local.recsrc);

	/* Digital audio path control */
	tmp = DEEMP_44K;
	omap1610_write_i2c(DIGITAL_AUDIO_CONTROL_ADDR, tmp);

	/* Power control, everything is on */
	tmp = 0x0;
	omap1610_write_i2c(POWER_DOWN_CONTROL_ADDR, tmp);

	/* Digital audio interface, master, DSP MODE (LRP=1), 16 bit */
	tmp = LRP_ON | FOR_DSP | MS_MASTER;

	omap1610_write_i2c(DIGITAL_AUDIO_FORMAT_ADDR, tmp);

	/* Digital interface */
	tmp = ACT_ON;
	omap1610_write_i2c(DIGITAL_INTERFACE_ACT_ADDR, tmp);

	/* clock configuration */
	omap1610_set_samplerate(audio_samplerate);
}

static void
omap1610_audio_shutdown(void *dummy)
{
	/* 
	 * Turn off codec after it is done.  
	 * Can't do it immediately, since it may still have
	 * buffered data.
	 *
	 *  Wait 20ms (arbitrary value) and then turn it off.
	 */
	
	set_current_state(TASK_INTERRUPTIBLE);
	schedule_timeout(2);

	input_or_output = 0;
	
	/* Disable the McBSP channel */
	outw(0x0000, AUDIO_DXR1);	/* flush data */
	outw(0x0000, AUDIO_DXR1);	/* flush data */
	outw(0x0000, AUDIO_SPCR1);	/* disable SPCR1 */
	outw(0x0000, AUDIO_SPCR2);	/* disable SPCR2 */     

	omap1610_write_i2c(RESET_CONTROL_ADDR, 0);
	omap1610_write_i2c(POWER_DOWN_CONTROL_ADDR, 0xff);
}

static int
omap1610_audio_ioctl(struct inode *inode, struct file *file,
		     uint cmd, ulong arg)
{
	long            val;
	int             ret = 0;

	pr_debug("omap1610_audio_ioctl: 0x%08x\n", cmd);

	/*
	 * These are platform dependent ioctls which are not handled by the
	 * generic omap-audio module.
	 */
	switch (cmd) {
		case SNDCTL_DSP_STEREO:
			ret = get_user(val, (int *) arg);
			if (ret)
				return ret;
			/* the AIC23 is stereo only */
			ret = (val == 0) ? -EINVAL : 1;
			return put_user(ret, (int *) arg);

		case SNDCTL_DSP_CHANNELS:
		case SOUND_PCM_READ_CHANNELS:
			/* the AIC23 is stereo only */
			return put_user(2, (long *) arg);

		case SNDCTL_DSP_SPEED:
			ret = get_user(val, (long *) arg);
			if (ret)
				break;
		  omap1610_set_samplerate(val);

		  /* fall through */

		case SOUND_PCM_READ_RATE:
			return put_user(audio_samplerate, (long *) arg);

		case SNDCTL_DSP_SETFMT:
		case SNDCTL_DSP_GETFMTS:
			/* we can do 16-bit only */
			return put_user(AFMT_S16_LE, (long *) arg);

		default:
			/* Maybe this is meant for the mixer (As per OSS Docs) */
			return mixer_ioctl(inode, file, cmd, arg);
	}

	return ret;
}

static int
omap1610_audio_open(struct inode *inode, struct file *file)
{
	/* Set a flag so we know how to initialize hardware */

	if (file->f_mode & FMODE_WRITE)
		input_or_output |= FMODE_WRITE;
	
	if (file->f_mode & FMODE_READ) 
		input_or_output |= FMODE_READ;

	return omap_audio_attach(inode, file, &audio_state);
}

/*
 * Missing fields of this structure will be patched with the call
 * to omap_audio_attach().
 */
static struct file_operations omap1610_audio_fops = {
	open:	omap1610_audio_open,
	owner:	THIS_MODULE
};

static int      audio_dev_id, mixer_dev_id;

static int __init
omap1610_aic23_init(void)
{
	int	ret;
	u8	reg;

	ret = i2c_add_driver(&aic23_driver);

	if (ret < 0) {
		printk(KERN_INFO
			"OMAP1610 audio support failed to find AIC23.\n");
		goto out;
	}

#ifdef CONFIG_OMAP_OSK
	/* 
	 * OSK OMAP5912 TM Hardware Specification (5912TM_HDS) states:
	 * The AIC23 voltage comes from the LDO2 output 
	 * of the TPS65010 power controller. The LDO2 
	 * output should be raised to 3.0V for cleaner operation
	 * when using the CODEC.
	 */
	reg = tps65010_read(TPS65010_I2C_VREGS1);
	pr_debug("tps65010_read: reg 0x%02x, data 0x%04x\n", TPS65010_I2C_VREGS1, 
		reg);
	tps65010_write(reg | VREGS1_LDO21 | VREGS1_LDO20, TPS65010_I2C_VREGS1);
#endif
	/* register devices */

	audio_dev_id = register_sound_dsp(&omap1610_audio_fops, -1);

	mixer_dev_id = register_sound_mixer(&omap1610_mixer_fops, -1);

	printk(KERN_INFO "OMAP1610 audio support initialized\n");
	return 0;

out:
	return ret;
}

static void __exit
omap1610_aic23_exit(void)
{
	unregister_sound_dsp(audio_dev_id);
	unregister_sound_mixer(mixer_dev_id);
        i2c_del_driver(&aic23_driver);
}

module_init(omap1610_aic23_init);
module_exit(omap1610_aic23_exit);

MODULE_AUTHOR("MontaVista Software, Inc.");
MODULE_DESCRIPTION("Glue audio driver for the TI OMAP1610 & AIC23 codec.");
MODULE_LICENSE("GPL");

EXPORT_NO_SYMBOLS;
