/******************************************************************************

	mx1ads-dac3550a.c
	driver for Motorola MX1ADS on-board sound chip DAC3550a
	features:
	  - volume control
	  - OSS Sound Support
	  - volume is turned up after initialization to remove clicks
	  - supports only 16-bit stereo mode due to HW limitations.

	Author: MontaVista Software, Inc. <source@mvista.com>
	Copyright (c) 2003 MontaVista Software, Inc.

	Original code: ssiDAC3550a.c from Motorola MX1ADS BSP 0.3.4
	Author: Li Qin
	Copyright 2002, 2003 Motorola, Inc. All Rights Reserved

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
	
	Modifications:
	Nov 2003 - ver 1.1, MontaVista Software, Inc: added i2c search for dev-fs
	Oct 2004 - ver 2.0, MontaVista Software, Inc:
		reimplemented buffering. Now using a circular DMA buffer,
		adding new chunks of data roughly every 200ms

********************************************************************************/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/config.h>
#include <linux/init.h>

#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/i2c.h>
#ifdef CONFIG_PM
#include <linux/pm.h>
#endif

#include <asm/uaccess.h>
#include <asm/arch/platform.h>
#include <asm/irq.h>
#include <asm/arch/mx1ads-gpio.h>
#include <asm/io.h>
#include <asm/dma.h>
#include <asm/arch/dma.h>
#include <asm/mach/dma.h>

#include "sound_config.h"

/*register access macros*/
#define mx1_ssi_reg_out(r,x) outl(x,IO_ADDRESS((SSI_BASE + r)))
#define mx1_ssi_reg_in(r) inl(IO_ADDRESS((SSI_BASE + r)))
#define mx1_pll_reg_out(r,x) outl(x,IO_ADDRESS((PLL_BASE + r)))
#define mx1_pll_reg_in(r) inl(IO_ADDRESS((PLL_BASE + r)))
#define mx1_dma_chan_reg_out(r,c,x) outl(x,IO_ADDRESS((DMA_CH0_BASE + r + c * DMA_REG_SET_OFS * sizeof(VU32))))

/*DAC3550A I2C address */
#define DAC3550A_I2C_ADDR (0x9A>>1)

/*DAC3550A I2C registers */
#define DAC3550A_REG_SR 1
#define DAC3550A_REG_AVOL 2
#define DAC3550A_REG_GCFG 3

/*I2C access structures*/
static struct i2c_client *dac3550_client;
static struct file *dac3550_i2c;

/*ioctl data*/
static unsigned int dac3550a_bits = AFMT_S16_LE;
static signed short dac3550a_channels = 2;
static int dac3550a_speed = 44100;
static int dac3550a_dma_buf_delay;

/*mixer data*/
static int dac3550a_vol_right = 70;
static int dac3550a_vol_left = 70;

extern void mx1ads_request_dma_intr(dmach_t channel,
				    callback_t dma_callback,
				    err_callback_t dma_err_callback);
extern void mx1ads_free_dma_intr(dmach_t channel);

static volatile int dac3550a_busy;
static int dac3550a_dev = -1;
static dmach_t dac3550a_dma_chan = -1;

#ifdef CONFIG_PM
static struct pm_dev *dac3550a_pmdev;
#endif
static int dac3550a_mixerdev = -1;
static volatile char dac3550a_triggered;

static void *dac3550a_dma_buf;
static u32 dac3550a_dma_bufsize;
static char dac3550a_dma_buf_num;
static volatile char dac3550a_dma_buf_state;

static volatile int dac3550a_initstate_gpio;
static volatile int dac3550a_initstate_ssi;
static volatile int dac3550a_initstate_thread;
static volatile int dac3550a_initstate_i2c;

static struct timer_list dac3550a_dma_timer;

/*the DAC3550a registers are write-only,
so here global variables are defined to store values written to the DAC*/
static char dac3550a_avol_r, dac3550a_avol_l, dac3550a_sr, dac3550a_gcfg;

extern void DMAbuf_outputintr(int dev, int notify_only);
extern int DMAbuf_start_dma(int dev, unsigned long physaddr,
			    int count, int dma_mode);

static void dac3550a_start_input(int dev, unsigned long buf, int count,
				 int intrflag);
static int dac3550a_prepare_for_input(int dev, int bufsize, int nbufs);
static int dac3550a_prepare_for_output(int dev, int bufsize, int nbufs);
static void dac3550a_trigger(int dev, int state);
static void dac3550a_output_block(int dev, unsigned long buf,
				  int count, int intrflag);
static void dac3550a_close(int dev);
static int dac3550a_open(int dev, int mode);
static void dac3550a_halt_io(int dev);
static int dac3550a_ioctl(int dev, unsigned int cmd, caddr_t arg);
static int dac3550a_set_speed(int dev, int speed);
static signed short dac3550a_set_channels(int dev, signed short channels);
static unsigned int dac3550a_set_bits(int dev, unsigned int bits);
static int dac3550a_restore_all_regs(void);

#ifdef CONFIG_PM
static int dac3550a_pm_callback(struct pm_dev *pmdev,
				pm_request_t rqst, void *data);
#endif

static void dac3550a_dma_int_handler(void);
static void dac3550a_dma_err_handler(int errno);

static int dac3550a_mixer_ioctl(int dev, unsigned int cmd, caddr_t arg);
static int dac3550a_reg_write(char dac_reg);
static void dac3550a_clear_all_regs(void);
static int dac3550a_sound_thread(void *data);

int __init mx1ads_dac3550a_init(void);
void __init mx1ads_dac3550a_exit(void);

static int dac3550a_i2c_init(void);
static void dac3550a_i2c_exit(void);

DECLARE_MUTEX_LOCKED(dac3550a_thread_go);
DECLARE_COMPLETION(dac3550a_thread_exit);
DECLARE_COMPLETION(dac3550a_thread_stop);
wait_queue_head_t dac3550a_dma_int;
wait_queue_head_t dac3550a_dma_timer_int;

static volatile char dac3550a_thread_innerloop, dac3550a_thread_outerloop;

static struct audio_driver dac3550a_driver = {
	.owner = THIS_MODULE,
	.open = dac3550a_open,
	.close = dac3550a_close,
	.output_block = dac3550a_output_block,
	.start_input = dac3550a_start_input,
	.ioctl = dac3550a_ioctl,
	.prepare_for_input = dac3550a_prepare_for_input,
	.prepare_for_output = dac3550a_prepare_for_output,
	.halt_io = dac3550a_halt_io,
	.local_qlen = 0,	/*optional */
	.copy_user = 0,		/*optional */
	.halt_input = 0,	/*optional */
	.halt_output = 0,	/*optional */
	.trigger = dac3550a_trigger,
	.set_speed = dac3550a_set_speed,
	.set_bits = dac3550a_set_bits,
	.set_channels = dac3550a_set_channels,
	.postprocess_write = 0,	/*optional */
	.preprocess_read = 0,	/*optional */
	.mmap = 0,		/*optional */
};

static struct mixer_operations dac3550a_mixer_operations = {
	.owner = THIS_MODULE,
	.id = "DAC3550A",
	.name = "DAC3550a Volume Ctl",
	.ioctl = dac3550a_mixer_ioctl,
};

static inline void
dac3550a_output_silent_block(void)
{
	void *addr =
	    (void *) ((u32) dac3550a_dma_buf +
		      dac3550a_dma_bufsize * dac3550a_dma_buf_state);
	memset(addr, 0, dac3550a_dma_bufsize);
	consistent_sync(addr, dac3550a_dma_bufsize, PCI_DMA_TODEVICE);
	dac3550a_dma_buf_state++;
	if (dac3550a_dma_buf_state >= dac3550a_dma_buf_num)
		dac3550a_dma_buf_state = 0;
}

static int
dac3550a_sound_thread(void *data)
{
	daemonize();
	reparent_to_init();
	strcpy(current->comm, "dac3550a");
	int tmp_silent_blocks = 0;
	int half_buffer;

	while (dac3550a_thread_outerloop) {
		down(&dac3550a_thread_go);
		if (!dac3550a_thread_outerloop)
			break;

		dac3550a_restore_all_regs();

		/*wait some time so that OSS takes care of its buffers,
		   otherwise its pointers (heard in start of the play) will be jumbled */
		sleep_on_timeout(&dac3550a_dma_int, HZ / 10);

		do {
			/*fill out the entire i2s buffer */
			if (audio_devs[dac3550a_dev]->dmap_out->qlen > 0) {
				DMAbuf_outputintr(dac3550a_dev, 1);
				tmp_silent_blocks = 0;
			} else {
				dac3550a_output_silent_block();
				tmp_silent_blocks++;
			}
		} while (dac3550a_dma_buf_state != 0);

		half_buffer = (dac3550a_dma_buf_num >> 1);

		dac3550a_avol_r = dac3550a_vol_right * 56 / 100;
		dac3550a_avol_l = dac3550a_vol_left * 56 / 100;
		dac3550a_reg_write(DAC3550A_REG_AVOL);	/*turn the volume up */
		/*enable transfer and DMA */
		enable_dma(dac3550a_dma_chan);	/*start i2s data stream */
		mx1_ssi_reg_out(SSI_STCR, 0x280);	/*enable FIFO and DMA */
		mx1_ssi_reg_out(SSI_SCSR, 0xA300);	/*enable transmitter */
		dac3550a_dma_timer.expires = jiffies + dac3550a_dma_buf_delay;
		add_timer(&dac3550a_dma_timer);

		for (;;) {

			if (dac3550a_dma_buf_state == 0) {
				sleep_on(&dac3550a_dma_timer_int);	/*wait until I2S DMA reaches 1/2 of the buffer */
				if (tmp_silent_blocks > half_buffer)
					dac3550a_thread_innerloop = 0;
				if (!dac3550a_thread_innerloop)
					break;

			} else if (dac3550a_dma_buf_state == half_buffer) {

				sleep_on(&dac3550a_dma_int);	/*wait until I2S DMA reaches end of the buffer */
				if (tmp_silent_blocks > half_buffer)
					dac3550a_thread_innerloop = 0;
				if (!dac3550a_thread_innerloop)
					break;

				mod_timer(&dac3550a_dma_timer,
					  jiffies + dac3550a_dma_buf_delay);

			}

			/*fill out one half of the i2s buffer */
			if (audio_devs[dac3550a_dev]->dmap_out->qlen > 0) {
				DMAbuf_outputintr(dac3550a_dev, 1);
				tmp_silent_blocks = 0;
			} else {
				dac3550a_output_silent_block();
				tmp_silent_blocks++;
			}

		}

		dac3550a_clear_all_regs();
		del_timer(&dac3550a_dma_timer);
		dac3550a_dma_buf_state = 0;
		dac3550a_gcfg = 0x20;	/*bit 5 - low power mode */
		dac3550a_reg_write(DAC3550A_REG_GCFG);
		complete(&dac3550a_thread_stop);
		dac3550a_triggered = 0;
	}

	complete_and_exit(&dac3550a_thread_exit, 0);
	return 0;
}

static int
dac3550a_mixer_ioctl(int dev, unsigned int cmd, caddr_t arg)
{
	int val, left, right;

	if (_IOC_TYPE(cmd) != 'M') {
		return -EINVAL;
	}

	switch (cmd) {
	case SOUND_MIXER_READ_DEVMASK:
	case SOUND_MIXER_READ_STEREODEVS:
		val = SOUND_MASK_VOLUME;
		break;
	case SOUND_MIXER_READ_RECMASK:
		val = 0;
		break;
	case SOUND_MIXER_WRITE_RECSRC:
	case SOUND_MIXER_READ_RECSRC:
		val = SOUND_MIXER_NONE;
		break;

	case SOUND_MIXER_WRITE_VOLUME:
		if (get_user(val, (int *) arg))
			return -EFAULT;

		left = val & 0x00ff;
		right = (val >> 8) & 0x00ff;

		if (left > 100)
			left = 100;
		if (right > 100)
			right = 100;

		dac3550a_vol_right = right;
		dac3550a_vol_left = left;
		if (dac3550a_busy == 1) {
			dac3550a_avol_r = right * 56 / 100;
			dac3550a_avol_l = left * 56 / 100;
			dac3550a_reg_write(DAC3550A_REG_AVOL);
		}
	case SOUND_MIXER_READ_VOLUME:
		val = dac3550a_vol_left + ((dac3550a_vol_right << 8) & 0xFF00);
		break;

	default:
		return -EINVAL;
	}
	return put_user(val, (int *) arg);
}

/*write to the DAC control registers via I2C, uses dedicated global variables*/
static int
dac3550a_reg_write(char dac_reg)
{
	char msgbuf[3];
	int msglen;
	int tmp;

	if (dac3550a_i2c_init() < 0)
		return -EPERM;

	/*I2C slave address for the DAC is set in init in this module */
	switch (dac_reg) {
	case DAC3550A_REG_AVOL:
		msgbuf[1] = dac3550a_avol_r;
		msgbuf[2] = dac3550a_avol_l;
		msglen = 3;
		break;
	case DAC3550A_REG_SR:
		msgbuf[1] = dac3550a_sr;
		msglen = 2;
		break;
	case DAC3550A_REG_GCFG:
		msgbuf[1] = dac3550a_gcfg;
		msglen = 2;
		break;
	default:
		printk(KERN_ERR
		       "DAC3550A ERROR: writing to a wrong DAC register %d\n",
		       dac_reg);
		return -EINVAL;
	}

	msgbuf[0] = dac_reg;

	if ((tmp =
	     i2c_master_send(dac3550_client, &msgbuf[0], msglen)) != msglen) {
		printk(KERN_ERR
		       "DAC3550A ERROR: cannot write to I2C: sent %d, confirmed %d\n",
		       msglen, tmp);
		return -1;
	}

	return 0;
}

/*set prescale modulus*/
static void
dac3550a_set_pm(int number)
{
	unsigned long tmp;
	unsigned long num;

	num = (unsigned long) number;
	tmp = mx1_ssi_reg_in(SSI_STCCR);
	tmp &= 0xff00;
	tmp |= num;
	mx1_ssi_reg_out(SSI_STCCR, tmp);
}

/*set peripheral clock divider register*/
static void
dac3550a_set_pclk3(void)
{
	unsigned int div = 1;
	unsigned int tmp;

	tmp = mx1_pll_reg_in(PLL_PCDR);
	tmp &= ~PCDR_PCLKDIV3_MASK;	/* clear it */
	mx1_pll_reg_out(PLL_PCDR, tmp);
	tmp |= (((div - 1) << PCDR_PCLKDIV3_BIT) & PCDR_PCLKDIV3_MASK);
	mx1_pll_reg_out(PLL_PCDR, tmp);
}

/*restore necessary DAC I2C register values*/
static int
dac3550a_restore_all_regs(void)
{
	dac3550a_clear_all_regs();

	dac3550a_set_pclk3();

	mx1_ssi_reg_out(SSI_SCSR, (1 << 8));	/*enable SSI */
	mx1_ssi_reg_out(SSI_SOR, 0);

	/* SCSR
	   bit 0-7 - read only status bits
	   bit 8 - SSI enable
	   bit 9 -  transmit enable
	   bit 10 - receive enable
	   bits 11, 12 ignored in i2s mode
	   bits 14:13:
	   I2S master - 01
	   I2S slave - 10
	   SSI mode - 00 or 11
	   bit 15 - display perclk3 at SSI_RXCLK pin

	   1010 0001 0000 0000 0xA100 - init
	   1010 0011 0000 0000 0xA300 - trigger output */

	mx1_ssi_reg_out(SSI_SCSR, 0xa100);

	/* STCR
	   bits 0-6 ignored in i2s mode
	   bit 7 - enable FIFO
	   bit 8 - enable Int (don't enable if using DMA)
	   bit 9 - enable DMA
	   0000 0000 0000 0000 - init
	   0000 0010 1000 0000 0x0280 - trigger output */

	/*mx1_ssi_reg_out(SSI_STCR, 0); */

	/* SFCSR
	   bits 0-3 - tx FIFO empty watermark
	   min 1 - if 1 empty slot ...
	   max 8 - if 8 empty slots
	   bits 4-7 - rcv FIFO full watermark
	   min 1 - if 1 word rcvd ...
	   max 8 - if 8 words rcvd */

	mx1_ssi_reg_out(SSI_SFCSR, 0x82);

	dac3550a_set_channels(0, dac3550a_channels);
	dac3550a_set_bits(0, dac3550a_bits);
	dac3550a_set_speed(0, dac3550a_speed);

	return 0;
}

/*clear DAC I2C registers*/
/*disable SSI to prevent output to DAC*/
static void
dac3550a_clear_all_regs(void)
{
	disable_dma(dac3550a_dma_chan);	/*stop i2s data stream */
	dac3550a_avol_r = 0;
	dac3550a_avol_l = 0;
	dac3550a_reg_write(DAC3550A_REG_AVOL);
	dac3550a_sr = 0;
	dac3550a_gcfg = 0;

	mx1_ssi_reg_out(SSI_SCSR, (1 << 8));	/*keep SSI enbaled */
	mx1_ssi_reg_out(SSI_STCR, 0);
	mx1_ssi_reg_out(SSI_STCCR, 0);
	mx1_ssi_reg_out(SSI_SFCSR, 0);
	mx1_ssi_reg_out(SSI_SOR, (1 << 4));	/*flush tx fifo */
	mx1_ssi_reg_out(SSI_SOR, 0);
	mx1_ssi_reg_out(SSI_SOR, (1 << 6));	/*turn off clocks when SSI is disabled */
	mx1_ssi_reg_out(SSI_SCSR, 0);	/*disable SSI */
	dac3550a_reg_write(DAC3550A_REG_SR);
	dac3550a_reg_write(DAC3550A_REG_GCFG);
}

/* set 8 or 16 bits*/
static unsigned int
dac3550a_set_bits(int dev, unsigned int bits)
{
	unsigned long tmp;

	tmp = mx1_ssi_reg_in(SSI_STCCR);
	switch (bits) {
	case 0:
		return dac3550a_bits;
	case AFMT_S16_LE:
		dac3550a_bits = 16;
		tmp &= 0x00ff;
		tmp |= 0x6100;
		mx1_ssi_reg_out(SSI_STCCR, tmp);
		break;
	default:
		return -EINVAL;
	}
	return dac3550a_bits;
}

/* set MONO or STEREO*/
static signed short
dac3550a_set_channels(int dev, signed short channels)
{
	switch (channels) {
	case 0:
		return dac3550a_channels;
	case 2:
		dac3550a_channels = 2;
		dac3550a_gcfg = 0x04;
		if (dac3550a_reg_write(DAC3550A_REG_GCFG) < 0)
			return -1;
		break;
	default:
		return -EINVAL;
	}
	return dac3550a_channels;
}

/* set DAC sampling rate*/
/*
pclk3=1...pm=15...samplerate=46875.000000
pclk3=1...pm=16...samplerate=44117.000000
pclk3=1...pm=22...samplerate=32608.000000
pclk3=1...pm=33...samplerate=22058.000000
pclk3=1...pm=46...samplerate=15957.000000
pclk3=1...pm=67...samplerate=11029.000000
pclk3=1...pm=93...samplerate=7978.000000
*/
static int
dac3550a_set_speed(int dev, int speed)
{
	switch (speed) {
	case 0:
		return dac3550a_speed;
	case 8000:
		dac3550a_speed = 8000;
		dac3550a_dma_buf_num = 2;
		dac3550a_set_pm(93);
		dac3550a_sr = 0x0D;
		break;
	case 11025:
		dac3550a_speed = 11025;
		dac3550a_dma_buf_num = 2;
		dac3550a_set_pm(67);
		dac3550a_sr = 0x0C;
		break;
	case 16000:
		dac3550a_speed = 16000;
		dac3550a_dma_buf_num = 4;
		dac3550a_set_pm(46);
		dac3550a_sr = 0x0B;
		break;
	case 22050:
		dac3550a_speed = 22050;
		dac3550a_dma_buf_num = 4;
		dac3550a_set_pm(33);
		dac3550a_sr = 0x0A;
		break;
	case 32000:
		dac3550a_speed = 32000;
		dac3550a_dma_buf_num = 6;
		dac3550a_set_pm(22);
		dac3550a_sr = 0x09;
		break;
	case 44100:
		dac3550a_speed = 44100;
		dac3550a_dma_buf_num = 8;
		dac3550a_set_pm(16);
		dac3550a_sr = 0x08;
		break;
	case 48000:
		dac3550a_speed = 48000;
		dac3550a_dma_buf_num = 10;
		dac3550a_set_pm(15);
		dac3550a_sr = 0x08;
		break;

	default:
		return -EINVAL;
	}

	if (dac3550a_reg_write(DAC3550A_REG_SR) < 0)
		return -1;

	return dac3550a_speed;
}

static int
dac3550a_ioctl(int dev, unsigned int cmd, caddr_t arg)
{
	int val, ret = 0;

	switch (cmd) {
	case SOUND_PCM_WRITE_RATE:
		if (get_user(val, (int *) arg))
			return -EFAULT;
		ret = dac3550a_set_speed(dev, val);
		break;
	case SOUND_PCM_READ_RATE:
		ret = dac3550a_speed;
		break;
	case SNDCTL_DSP_STEREO:
		if (get_user(val, (int *) arg))
			return -EFAULT;
		ret = dac3550a_set_channels(dev, (val + 1)) - 1;
		break;
	case SOUND_PCM_WRITE_CHANNELS:
		if (get_user(val, (int *) arg))
			return -EFAULT;
		ret = dac3550a_set_channels(dev, val);
		break;
	case SOUND_PCM_READ_CHANNELS:
		ret = dac3550a_channels;
		break;
	case SNDCTL_DSP_SETFMT:
		if (get_user(val, (int *) arg))
			return -EFAULT;
		ret = dac3550a_set_bits(dev, val);
		break;
	case SOUND_PCM_READ_BITS:
		ret = dac3550a_bits;
		break;
	default:
		return -EINVAL;
	}
	return put_user(ret, (int *) arg);
}

#ifdef CONFIG_PM
/*power management event handling*/
static int
dac3550a_pm_callback(struct pm_dev *pmdev, pm_request_t rqst, void *data)
{
	switch (rqst) {
	case PM_SUSPEND:	/*put the DAC to stand by mode */
		if (dac3550a_triggered) {
			dac3550a_thread_innerloop = 0;
			wake_up(&dac3550a_dma_int);
			wake_up(&dac3550a_dma_timer_int);
			wait_for_completion(&dac3550a_thread_stop);
		}

		break;
	case PM_RESUME:	/*wake the DAC up */
		/*restore settings */
		if (dac3550a_busy) {
			dac3550a_trigger(0, 1);
		}
		break;
	}
	return 0;
}
#endif

static void
dac3550a_dma_timer_handler(unsigned long data)
{
	wake_up(&dac3550a_dma_timer_int);
}

static void
dac3550a_dma_int_handler(void)
{
	wake_up(&dac3550a_dma_int);
}

static void
dac3550a_dma_err_handler(int errno)
{
	printk(KERN_ERR "DAC3550A ERROR: dma_i2s_err_handler %d called!\n",
	       errno);
}

static int
dac3550a_i2c_init(void)
{
	char filename[20];
	int tmp;

	if (dac3550a_initstate_i2c)
		return 0;
	/*find the I2C driver we need */
	for (tmp = 0; tmp < I2C_ADAP_MAX; tmp++) {
#ifdef CONFIG_DEVFS_FS
		sprintf(filename, "/dev/i2c/%d", tmp);
#else
		sprintf(filename, "/dev/i2c-%d", tmp);
#endif

		if (!IS_ERR(dac3550_i2c = filp_open(filename, O_RDWR, 0))) {
			/*found some driver */
			dac3550_client =
			    (struct i2c_client *) dac3550_i2c->private_data;
			if (strlen(dac3550_client->adapter->name) >= 9) {
				if (!memcmp
				    (dac3550_client->adapter->name,
				     "DBMX1 I2C", 9))
					break;	/*we found our driver! */
			}
			filp_close(dac3550_i2c, NULL);
		}
	}

	if (tmp == I2C_ADAP_MAX) {	/*no matching I2C driver found */
		printk(KERN_ERR
		       "DAC3550A ERROR: cannot find DBMX1 I2C driver\n");
		return -EPERM;
	}

	dac3550_client->addr = DAC3550A_I2C_ADDR;
	dac3550a_initstate_i2c = 1;
	return 0;
}

static void
dac3550a_i2c_exit(void)
{
	if (dac3550a_initstate_i2c)
		filp_close(dac3550_i2c, NULL);
	dac3550a_initstate_i2c = 0;
}

/*start the device*/
static int
dac3550a_open(int dev, int mode)
{
	if (dac3550a_busy)
		return -EBUSY;
	if (mode != (int) OPEN_WRITE)
		return -1;	/*no input from this device */
	if (dac3550a_i2c_init() < 0)
		return -EPERM;

	dac3550a_busy = 1;
	dac3550a_triggered = 0;

	return (0);
}

/*restart the device*/
static void
dac3550a_halt_io(int dev)
{
	if (audio_devs[dev]->open_mode & OPEN_WRITE)
		audio_devs[dev]->enable_bits &= ~PCM_ENABLE_OUTPUT;
}

/*stop the device*/
static void
dac3550a_close(int dev)
{
	if (dac3550a_triggered) {
		wait_for_completion(&dac3550a_thread_stop);
	}

	if (dac3550a_dma_buf) {
		kfree(dac3550a_dma_buf);
		dac3550a_dma_buf = 0;
	}

	dac3550a_busy = 0;
}

/*this function is a part of dma buffer processing*/
static int
dac3550a_prepare_for_output(int dev, int bufsize, int nbufs)
{
	audio_devs[dac3550a_dev]->dmap_out->flags |= DMA_NODMA;

	if (!dac3550a_dma_buf) {

		dac3550a_dma_buf_state = 0;

		dac3550a_dma_bufsize = bufsize;
		dac3550a_dma_buf =
		    kmalloc(dac3550a_dma_bufsize * dac3550a_dma_buf_num,
			    GFP_KERNEL | GFP_DMA);

		if (!dac3550a_dma_buf) {
			printk(KERN_ERR
			       "DAC3550A ERROR: cannot allocate output DMA buffer\n");
			return -ENOMEM;
		}

		/*set start address of I2S DMA buffer */
		mx1_dma_chan_reg_out(DMA_SAR, dac3550a_dma_chan,
				     virt_to_phys(dac3550a_dma_buf));
		mx1_dma_chan_reg_out(DMA_CNTR, dac3550a_dma_chan,
				     dac3550a_dma_bufsize *
				     dac3550a_dma_buf_num);

		/*get number of jiffies that takes I2S DMA to get through 1/2 of the i2s buffer */
		dac3550a_dma_buf_delay =
		    ((dac3550a_dma_bufsize * (dac3550a_dma_buf_num >> 1) * HZ) /
		     (dac3550a_speed * 4)) + 2;

	}
	return 0;
}

/*this function is a part of dma buffer processing*/
static void
dac3550a_output_block(int dev, unsigned long buf, int count, int intrflag)
{
	/*copy data from OSS buffer to our I2S DMA buffer.
	   "count" is always the same */

	void *addr =
	    (void *) ((u32) dac3550a_dma_buf +
		      dac3550a_dma_bufsize * dac3550a_dma_buf_state);
	memcpy(addr, phys_to_virt(buf), count);
	consistent_sync(addr, count, PCI_DMA_TODEVICE);
	dac3550a_dma_buf_state++;
	if (dac3550a_dma_buf_state >= dac3550a_dma_buf_num)
		dac3550a_dma_buf_state = 0;

}

/*this function is a part of dma buffer processing*/
static void
dac3550a_trigger(int dev, int state)
{
	if (!state) {
		return;
	}

	if (!dac3550a_triggered) {
		init_completion(&dac3550a_thread_stop);
		dac3550a_thread_innerloop = 1;
		dac3550a_triggered = 1;
		up(&dac3550a_thread_go);	/*start thread */
	}
}

/*this function must not be called - there is no input from the device*/
static void
dac3550a_start_input(int dev, unsigned long buf, int count, int intrflag)
{
	printk(KERN_ERR "DAC3550A ERROR: start_input called!\n");
}

/*this function must not be called - there is no input from the device*/
static int
dac3550a_prepare_for_input(int dev, int bufsize, int nbufs)
{
	printk(KERN_ERR "DAC3550A ERROR: prepare_for_input called!\n");
	return 0;
}

/*initialize the device*/
int __init
mx1ads_dac3550a_init(void)
{
	int tmp;

	printk(KERN_INFO "MX1ADS DAC3550A Sound Driver Ver. 2.0\n");
	/*printk("%s %s\n", __TIME__, __DATE__); */

	tmp =
	    (int) request_region(IO_ADDRESS(SSI_BASE), 0x30, "mx1ads_dac3550a");
	if (!tmp) {
		printk(KERN_ERR "DAC3550A ERROR: SSI is already in use\n");

		mx1ads_dac3550a_exit();
		return -1;

	}
	dac3550a_initstate_ssi = 1;

	tmp = mx1_register_gpios(PORT_C, 0x1C0, PRIMARY | TRISTATE | OUTPUT);
	if (tmp < 0) {
		printk(KERN_ERR
		       "DAC3550A ERROR: PORTC mask 0x1C is already in use\n");

		mx1ads_dac3550a_exit();
		return tmp;
	}
	dac3550a_initstate_gpio = 1;

/*configure DMA support*/

	/* request a DMA channel for I2S FIFO data output */
	for (dac3550a_dma_chan = 0; dac3550a_dma_chan < MAX_DMA_CHANNELS;
	     dac3550a_dma_chan++) {
		if (!sound_alloc_dma(dac3550a_dma_chan, "I2S FIFO"))
			break;
	}

	if (dac3550a_dma_chan == MAX_DMA_CHANNELS) {
		printk(KERN_ERR
		       "DAC3550A ERROR: can't allocate a DMA cahnnel for I2S FIFO\n");
		dac3550a_dma_chan = -1;
		mx1ads_dac3550a_exit();
		return -1;
	}

	mx1_dma_chan_reg_out(DMA_DAR, dac3550a_dma_chan, 0x218000);	/* I2S TxFIFO register */
	/*target--fifo; source--linermem; des--16bit;sour--32bit;ren-- 1 */
	mx1_dma_chan_reg_out(DMA_CCR, dac3550a_dma_chan, 0x2088 | 0x4);	/*repeat added */
	mx1_dma_chan_reg_out(DMA_RSSR, dac3550a_dma_chan, 16);	/*NO.16 request */
	mx1_dma_chan_reg_out(DMA_BLR, dac3550a_dma_chan, 4);	/*4 bytes per burst */
	mx1_dma_chan_reg_out(DMA_RTOR, dac3550a_dma_chan, 0);	/* request timeout disabled */
	mx1_dma_chan_reg_out(DMA_BUCR, dac3550a_dma_chan, 0);	/* burst delay disabled */

	mx1ads_request_dma_intr(dac3550a_dma_chan,
				(callback_t) dac3550a_dma_int_handler,
				(err_callback_t) dac3550a_dma_err_handler);

	if ((dac3550a_dev = sound_install_audiodrv(AUDIO_DRIVER_VERSION,
						   "mx1ads dac3550a",
						   &dac3550a_driver,
						   sizeof (struct
							   audio_driver), 0,
						   AFMT_S16_LE, NULL,
						   -1, -1)) < 0) {
		printk(KERN_ERR
		       "DAC3550A ERROR: too many audio devices in the system!\n");
		mx1ads_dac3550a_exit();
		return dac3550a_dev;
	}

	if ((dac3550a_mixerdev = sound_alloc_mixerdev()) >= 0) {
		mixer_devs[dac3550a_mixerdev] = &dac3550a_mixer_operations;
	} else
		printk(KERN_WARNING
		       "DAC3550A WARNING: failed to load mixer device\n");

	dac3550a_thread_outerloop = 1;
	init_completion(&dac3550a_thread_exit);
	init_timer(&dac3550a_dma_timer);
	init_waitqueue_head(&dac3550a_dma_int);
	init_waitqueue_head(&dac3550a_dma_timer_int);
	dac3550a_dma_timer.function = dac3550a_dma_timer_handler;
	tmp =
	    kernel_thread(&dac3550a_sound_thread, NULL,
			  CLONE_FS | CLONE_FILES | CLONE_SIGHAND);
	if (tmp < 0) {

		printk(KERN_ERR "DAC3550A ERROR: could not start thread\n");
		return tmp;
	}

	dac3550a_initstate_thread = 1;

#ifdef CONFIG_PM
	dac3550a_pmdev =
	    pm_register(PM_UNKNOWN_DEV, PM_SYS_UNKNOWN, dac3550a_pm_callback);

	if (!dac3550a_pmdev)
		printk(KERN_WARNING
		       "DAC3550A WARNING: failed to init power management\n");
#endif

	return (0);
}

/*deinitialize the device*/
void __init
mx1ads_dac3550a_exit(void)
{

#ifdef CONFIG_PM
	if (dac3550a_pmdev)
		pm_unregister(dac3550a_pmdev);
#endif
	if (dac3550a_initstate_thread) {
		dac3550a_thread_outerloop = 0;
		up(&dac3550a_thread_go);
		wait_for_completion(&dac3550a_thread_exit);
	}

	if (dac3550a_dev >= 0) {
		dac3550a_i2c_exit();
		if (dac3550a_mixerdev >= 0)
			sound_unload_mixerdev(dac3550a_mixerdev);
		sound_unload_audiodev(dac3550a_dev);
	}

	if (dac3550a_dma_chan != -1) {
		mx1ads_free_dma_intr(dac3550a_dma_chan);
		sound_free_dma(dac3550a_dma_chan);
	}

	if (dac3550a_initstate_gpio)
		mx1_unregister_gpios(PORT_C, 0x1C0);
	if (dac3550a_initstate_ssi)
		release_region(IO_ADDRESS(SSI_BASE), 0x30);
}

MODULE_AUTHOR("MontaVista Software, Inc. <source@mvista.com>");
MODULE_DESCRIPTION("MX1ADS DAC3550A SOUND Driver");
MODULE_LICENSE("GPL");

module_init(mx1ads_dac3550a_init);
module_exit(mx1ads_dac3550a_exit);
