/*
 * <LIC_AMD_STD>
 * Copyright (c) 2004 Advanced Micro Devices, Inc.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * The full GNU General Public License is included in this distribution in the
 * file called COPYING
 * </LIC_AMD_STD>
 * <CTL_AMD_STD>
 * </CTL_AMD_STD>
 * <DOC_AMD_STD>
 * </DOC_AMD_STD>
*/
#include "gxlnxaud.h"
#include "linux/list.h"
#include "linux/kernel.h"

PDURAUDIO pGeode;
static struct amd_card *devs = NULL;
static struct pci_driver geode_pci_driver;
static int powermgmt = 1;
static int bIoAlloc = FALSE;

MODULE_AUTHOR("Jens Altmann");
MODULE_DESCRIPTION("AMD Geode  OSS Audio Driver");

MODULE_PARM(powermgmt, "i");
MODULE_PARM_DESC(powermgmt,
		 "Set to 0 to disable power management. Default is 1");
MODULE_LICENSE("GPL");

static char version[] __devinitdata =
    KERN_INFO "geode OSS: version " DRIVER_VERSION " time " __TIME__ " "
    __DATE__ "\n";

enum card_types_t {
	TYPE_GX1,
	TYPE_SCXX00,
	TYPE_GX2
};

static const char *card_names[] = {
	[TYPE_GX1] = "GX1 OSS",
	[TYPE_SCXX00] = "SCXX00 OSS",
	[TYPE_GX2] = "GX2 OSS"
};

/*------- function declarations ----------------------------------------------------*/
static int audio_shutdown_notify_callback(struct notifier_block *pNotify,
					  unsigned long event, void *buf);
static void dma_reload_thread(dma_thread_t * dma_thread);
static buff_list_t *get_buffer_list_head(struct list_head *pList);
static void release_buffers(struct amd_state *s);
static void amd_interrupt(int irq, void *dev_id, struct pt_regs *regs);
static void set_volume_level(struct amd_card *card, unsigned int mixer,
			     unsigned int val);
static int pwrmgmt_callback(struct pm_dev *dev, pm_request_t request, void *d);
static void start_dma_thread(void (*func) (dma_thread_t *),
			     struct amd_state *pAmd);
static void stop_dma_thread(dma_thread_t * dma_thread);
static void init_dma_thread(dma_thread_t * dma_thread, char *name);
static void exit_dma_thread(dma_thread_t * dma_thread);
static int get_bytes_in_playback_buffer(struct amd_state *pAmd);
static unsigned long get_current_dma_pos(unsigned char channel);
/*--------------------------------------------------------------------------------*/

static struct notifier_block audio_notify =
    { audio_shutdown_notify_callback, NULL, 0 };

struct amd_state {

	struct amd_card *card;
	/*wave stuff */
	unsigned long rateadc, ratedac;
	unsigned char fmt, enable;

	unsigned long intID;	/*stores the last interrupt mask */

	/*Only let 1 be opening at a time */
	struct semaphore open_sem;
	wait_queue_head_t open_wait;
	wait_queue_head_t close_wait;
	mode_t open_mode;
	/*soundcore stuff */
	int dev_audio;

	struct dmabuf {
		void *rawbuf;
		unsigned total_bytes;
		int count;
		unsigned long fragcnt;
		unsigned long lastpos;

		wait_queue_head_t wait;

		dma_thread_t dma_thread;
		struct list_head buffer_list;
		unsigned long bufferedSize;

		unsigned long fragsize;
		unsigned long dmasize;

		unsigned char BitsPerSample;
		unsigned char nChannels;
		volatile long BytesReadyInDMABuffer;
		volatile long BytesInDMABuffer;

		/*Flags */
		char zero_flags;	/*flags to control underrun zeroing */
		unsigned char DMAmapped;
		unsigned char ready;

		unsigned long ossfragsize;
		unsigned long ossfragshift;
		int ossmaxfrags;
		unsigned long subdivision;
		struct vm_area_struct vma;	/*used to save the mapping structure (For DirectDMA) */
	} dma_dac, dma_adc;
};

/*sleep states*/

#define SLEEP       2
#define AWAKE       1
#define OPERATIONAL 0

struct amd_card {
	unsigned int magic;
	/*We keep geode cards in a linked list */
	struct amd_card *next;
	int dev_mixer;
	int card_type;

	/*As most of this is static, */
	/*perhaps it should be a pointer to a global struct */
	struct mixer_goo {
		int modcnt;
		int supported_mixers;
		int stereo_mixers;
		int record_sources;
		void (*write_mixer) (struct amd_card * card, int mixer,
				     unsigned int left, unsigned int right);
		int (*recmask_io) (struct amd_card * card, int rw, int mask);
		unsigned int mixer_state[SOUND_MIXER_NRDEVICES];
	} mix;
	int power_regs;
	int in_suspend;
	wait_queue_head_t suspend_queue;
	struct amd_state channels[MAX_DSPS];

	/*this locks around the physical registers on the card */
	spinlock_t lock;

	/*hardware resources */
	struct pci_dev *pcidev;
	unsigned long iobase;
	unsigned long irq;
	int dock_mute_vol;
};

/**
 * \ingroup linux GX layer
 * \brief
 *  reads a value from the AC97 codec
 *
 *
 * \param   struct amd_card* card   pointer to the hw descriptor
 *          unsigned char               cmd    command to program into the AC97 codec
 *
 * \return  unsigned short              value read from codec register
 */
static unsigned short
read_ac97_val(struct amd_card *card, unsigned char cmd)
{
	return DURAUDIO_CodecRead(pGeode, cmd);
}

/*The mixer default values are in 0-100 OSS units*/
static unsigned int mixer_defaults[SOUND_MIXER_NRDEVICES] = {
	[SOUND_MIXER_VOLUME] = VOL_DEFAULT,
	[SOUND_MIXER_BASS] = VOL_DEFAULT,
	[SOUND_MIXER_TREBLE] = VOL_DEFAULT,
	[SOUND_MIXER_SPEAKER] = VOL_DEFAULT,
	[SOUND_MIXER_MIC] = MUTE_MASK,
	[SOUND_MIXER_LINE] = VOL_DEFAULT,
	[SOUND_MIXER_CD] = VOL_DEFAULT,
	[SOUND_MIXER_VIDEO] = VOL_DEFAULT,
	[SOUND_MIXER_LINE1] = VOL_DEFAULT,
	[SOUND_MIXER_PCM] = VOL_DEFAULT,
	[SOUND_MIXER_IGAIN] = VOL_DEFAULT
};

/*The AC97 values are in AC97 control register units*/
static struct ac97_mixer_hw {
	unsigned char offset;
	int scale;
} ac97_hw[SOUND_MIXER_NRDEVICES] = {
	[SOUND_MIXER_VOLUME] = {
	MASTER_VOLUME, MASTER_VOLUME_MAX},[SOUND_MIXER_BASS] = {
	MASTER_TONE_RL, 15},[SOUND_MIXER_TREBLE] = {
	MASTER_TONE_RL, 15},[SOUND_MIXER_SPEAKER] = {
	PC_BEEP_VOLUME, 15},[SOUND_MIXER_MIC] = {
	MIC_VOLUME, 31},[SOUND_MIXER_LINE] = {
	LINE_IN_VOLUME, 31},[SOUND_MIXER_CD] = {
	CD_VOLUME, 31},[SOUND_MIXER_VIDEO] = {
	VIDEO_VOLUME, 31},[SOUND_MIXER_LINE1] = {
	TV_VOLUME, 31},[SOUND_MIXER_PCM] = {
	PCM_OUT_VOL, 31},[SOUND_MIXER_IGAIN] = {
	RECORD_GAIN, 15}
};

/*write the OSS encoded volume to the given OSS encoded mixer,
  again caller's job to make sure all is well in arg land,
  call with spinlock held */

/*scale linear -> log */
static unsigned char conv2log[] = {
	0, 0, 15, 23, 30, 34, 38, 42, 45, 47,
	50, 52, 53, 55, 57, 58, 60, 61, 62,
	63, 65, 66, 67, 68, 69, 69, 70, 71,
	72, 73, 73, 74, 75, 75, 76, 77, 77,
	78, 78, 79, 80, 80, 81, 81, 82, 82,
	83, 83, 84, 84, 84, 85, 85, 86, 86,
	87, 87, 87, 88, 88, 88, 89, 89, 89,
	90, 90, 90, 91, 91, 91, 92, 92, 92,
	93, 93, 93, 94, 94, 94, 94, 95, 95,
	95, 95, 96, 96, 96, 96, 97, 97, 97,
	97, 98, 98, 98, 98, 99, 99, 99, 99, 99
};

/**
 * \ingroup linux GX layer
 * \brief
 *
 *  programs the choosen mixer slider with the values in left and right
 *
 * \param  struct amd_card* card   pointer to the hw descriptor
 *         int              mixer  mixer id(slider)
 *         unsigned int     left   value for the left channel
 *         unsigned int     right  value for the left channel
 */

static void
set_ac97_mixer_volume(struct amd_card *card, int mixer,
		      unsigned int lvol, unsigned int rvol)
{
	unsigned short vol;
	struct ac97_mixer_hw *pMixHwh;

	OS_DbgMsg("-->set_ac97_mixer_volume mixer=%d,r=%d,l=%d\n", mixer, lvol,
		  rvol);
	vol = 0;
	pMixHwh = &ac97_hw[mixer];
	if (AC97_STEREO_MASK & (1 << mixer)) {
		if (mixer == SOUND_MIXER_IGAIN) {
			rvol = (rvol * pMixHwh->scale) / 100;
			lvol = (lvol * pMixHwh->scale) / 100;
			if ((lvol == 0) && (rvol == 0)) {
				vol |= MUTE_MASK;
			}
		} else {
			if ((lvol == 0) && (rvol == 0)) {
				vol = MUTE_MASK;
			}
			rvol = ((100 - conv2log[rvol]) * pMixHwh->scale) / 100;
			lvol = ((100 - conv2log[lvol]) * pMixHwh->scale) / 100;
		}
		vol |= (lvol << 8) | rvol;
	}

	if (mixer == SOUND_MIXER_SPEAKER) {
		vol = (((100 - lvol) * pMixHwh->scale) / 100) << 1;
	} else if (mixer == SOUND_MIXER_MIC) {
		vol = read_ac97_val(card, pMixHwh->offset);
		OS_DbgMsg("read_ac97_val: vol=0x%X\n", vol);
		vol &= (~0x801F);
		vol |= (((100 - lvol) * pMixHwh->scale) / 100);
	} else if (mixer == SOUND_MIXER_BASS) {
		vol = read_ac97_val(card, pMixHwh->offset);
		vol &= (~0x0F00);
		vol |= ((((100 - lvol) * pMixHwh->scale) / 100) << 8) & 0x0E00;
	} else if (mixer == SOUND_MIXER_TREBLE) {
		vol = read_ac97_val(card, pMixHwh->offset);
		vol &= (~0x000F);
		vol |= (((100 - lvol) * pMixHwh->scale) / 100) & 0x000E;
	}
	OS_DbgMsg("CodecWrite: offset=0x%X ,vol=0x%X\n", pMixHwh->offset, vol);
	DURAUDIO_CodecWrite(pGeode, pMixHwh->offset, vol);
	if (MASTER_VOLUME == pMixHwh->offset) {
		DURAUDIO_CodecWrite(pGeode, LINE_LEV_OUT_VOL, vol);
	}
}

/* the tables assigs from  OSS <-> ac97 */

enum ac97_recsettings {
	AC97_REC_MIC = 0,
	AC97_REC_CD,
	AC97_REC_VIDEO,
	AC97_REC_AUX,
	AC97_REC_LINE,
	AC97_REC_STEREO,
	AC97_REC_MONO,
	AC97_REC_PHONE
};

static unsigned int ac97_oss_mask[] = {
	[AC97_REC_MIC] = SOUND_MASK_MIC,
	[AC97_REC_CD] = SOUND_MASK_CD,
	[AC97_REC_VIDEO] = SOUND_MASK_VIDEO,
	[AC97_REC_AUX] = SOUND_MASK_LINE1,
	[AC97_REC_LINE] = SOUND_MASK_LINE,
	[AC97_REC_PHONE] = SOUND_MASK_PHONEIN
};

/*indexed by bit position */

static unsigned int ac97_oss_rm[] = {
	[SOUND_MIXER_MIC] = AC97_REC_MIC,
	[SOUND_MIXER_CD] = AC97_REC_CD,
	[SOUND_MIXER_VIDEO] = AC97_REC_VIDEO,
	[SOUND_MIXER_LINE1] = AC97_REC_AUX,
	[SOUND_MIXER_LINE] = AC97_REC_LINE,
	[SOUND_MIXER_PHONEIN] = AC97_REC_PHONE
};

/**
 * \ingroup linux GX layer
 * \brief
 *  Indirectly called from IOCTL to read or write the record
 *  channel mask
 *
 * \param struct amd_card *pCard pointer to the hw descriptor
 *        int              bRead
 *        int              mask
 * \return
 */
static int
rw_ac97_record_mask(struct amd_card *pCard, int bRead, int mask)
{
	unsigned int recval, idx;
	int ret;

	idx = read_ac97_val(pCard, RECORD_SELECT) & 0x07;
	recval = ac97_oss_mask[idx];
	if (!bRead) {
		/*set a new value */
		if (mask != (int) recval) {
			mask &= (~recval);
		}
		recval = ffs(mask) - 1;
		recval = ac97_oss_rm[recval];
		recval |= recval << 8;	/*set both channels */
		DURAUDIO_CodecWrite(pGeode, RECORD_SELECT, recval);
		ret = 0;
	} else {		/*just read it */
		ret = (int) recval;
	}
	return ret;
};

/**
 * \ingroup linux GX layer
 * \brief
 * These routines handle accessing the second level
 * indirections to the wave ram.
 * Set the rate for this DAC channel
 *
 * \param  struct amd_state *s     pointer to the state staucture for the DMA channel
 *         unsigned int     rate   rate value to set
 */
static void
set_dac_rate(struct amd_state *s, unsigned int rate)
{
	OS_DbgMsg("Setting DAC rate to: %d\n", rate);
	s->ratedac = rate;
	DURAUDIO_SetCodecRate(pGeode, CHANNEL0_PLAYBACK, (unsigned long) rate);
}

/**
 * \ingroup linux GX layer
 * \brief
 * These routines handle accessing the second level
 * indirections to the wave ram.
 * Set the rate for this ADC channel
 *
 * \param  struct amd_state *s     pointer to the state staucture for the DMA channel
 *         unsigned int     rate   rate value to set
 */
static void
set_adc_rate(struct amd_state *s, unsigned int rate)
{
	OS_DbgMsg("Setting ADC rate to: %d\n", rate);

	s->rateadc = rate;
	DURAUDIO_SetCodecRate(pGeode, CHANNEL1_RECORD, (unsigned long) rate);
}

/**
 * \ingroup linux GX layer
 * \brief
 * Stops the DMA transfer for the choosen recording channel
 *
 * \param  struct amd_state *s     pointer to the state staucture for the DMA channel
 */
static inline void
stop_record(struct amd_state *s)
{
	if (!(s->enable & ADC_RUNNING))
		return;
	s->enable &= ~ADC_RUNNING;

	DURAUDIO_StopDMA(pGeode, CHANNEL1_RECORD);
}

/**
 * \ingroup linux GX layer
 * \brief
 * Stops the DMA transfer for the choosen play channel
 *
 * \param  struct amd_state *s     pointer to the state staucture for the DMA channel
 */
static void
stop_play(struct amd_state *pAmd)
{
	unsigned long flags = 0;
	spinlock_t lock;

	OS_DbgMsg("stop_play\n");
	if (pAmd->dma_dac.bufferedSize) {	/*play the rest of buffered data or, wait max 2s */
		OS_DbgMsg
		    ("wait until the buffer has been emptied before closing!\n");
		interruptible_sleep_on_timeout(&pAmd->close_wait, 200);
	}
	if (!(pAmd->enable & DAC_RUNNING)) {
		OS_DbgMsg("DAC Not running! - Returning...\n");
		return;
	}
	spin_lock_irqsave(&lock, flags);
	pAmd->dma_dac.BytesInDMABuffer = 0;
	pAmd->enable &= ~DAC_RUNNING;
	DURAUDIO_StopDMA(pGeode, CHANNEL0_PLAYBACK);
	spin_unlock_irqrestore(&lock, flags);
}

/**
 * \ingroup linux GX layer
 * \brief
 * Zero the unused bytes in the playback dma buffer
 *
 * \param  struct amd_state *s     pointer to the state staucture for the DMA channel
 */
void
zero_free_playback_buffer(struct amd_state *s)
{
	OS_DbgMsg("zero_free_playback_buffer\n");
	clear_bit(ZF_TRANSFER_SINCE_ZERO, &s->dma_dac.zero_flags);
	DURAUDIO_ZeroFillFreeBuffer(pGeode);
}

/**
 * \ingroup linux GX layer
 * \brief
 * Starts transferring data from DMA buffer to the codec
 *
 * \param  struct amd_state *s     pointer to the state staucture for the DMA channel
 *         int              flags  always used =START_DAC_START_ANY
 */
static void
start_play(struct amd_state *pAmd, int flags)
{
	long ready;
	long thresh;

	ready = pAmd->dma_dac.BytesReadyInDMABuffer;
	thresh = pAmd->dma_dac.fragsize;
	if (pAmd->card->in_suspend != OPERATIONAL) {
		OS_DbgMsg("start_play: DAC still not ready=%d . Returning...\n",
			  pAmd->card->in_suspend);
		return;
	}
	if (pAmd->enable & DAC_RUNNING) {
		OS_DbgMsg("start_play: DAC Already running. Returning...\n");
		return;
	}

	OS_DbgMsg("start_play: bytes ready: %X  start threshhold: %X \n",
		  ready, thresh);

	/*We only start if we have at least a fragment
	   or we have ANY bytes and are told to ignore the threshhold */

	if ((ready >= thresh) || (ready && (flags & START_DAC_START_ANY))) {
		OS_DbgMsg("start_play: Starting DMA\n");

		/*Before we start playback, make sure the rest of the
		   buffer zero'd for underrun */
		zero_free_playback_buffer(pAmd);
		pAmd->enable |= DAC_RUNNING;
		DURAUDIO_StartDMA(pGeode, CHANNEL0_PLAYBACK);
	} else if (pAmd->dma_dac.DMAmapped == TRUE) {
		pAmd->enable |= DAC_RUNNING;
		DURAUDIO_StartDMA(pGeode, CHANNEL0_PLAYBACK);
	}
}

/**
 * \ingroup linux GX layer
 * \brief
 * Utility function to get bytes in use safely and update cached value
 *
 * \param  struct amd_state *s     pointer to the state structure for the DMA channel
 *
 * \return unsigned long           number of bytes currently in the DMA buffer
 */
unsigned long
get_playback_bytes_used(struct amd_state *s)
{
	unsigned long used;
	if (s->enable & DAC_RUNNING) {	/*If the DAC is running, the ISR updates the value */
		used = s->dma_dac.BytesReadyInDMABuffer;
	} else {
		/*if not, we get it from Durango, and update the local copy */
		used = DURAUDIO_UpdateByteCounter(pGeode, CHANNEL0_PLAYBACK);
		s->dma_dac.BytesReadyInDMABuffer = used;
	}
	return used;
}

/**
 * \ingroup linux GX layer
 * \brief
 * Starts recording  into DMA buffer from the codec
 *
 * \param  struct amd_state *s     pointer to the state staucture for the DMA channel
 */
static void
start_record(struct amd_state *s)
{
	s->enable |= ADC_RUNNING;

	DURAUDIO_StartDMA(pGeode, CHANNEL1_RECORD);
}

/**
 * \ingroup linux GX layer
 * \brief
 * Finishes playback, closes the stream, disables the DMA controller
 * flushes the DMA buffer
 *
 * \param  struct amd_state *s     pointer to the state staucture for the DMA channel
 */
static void
amd_close_playback(struct amd_state *pAmd)
{
	stop_play(pAmd);
	OS_DbgMsg("Closing Stream. \n");
	DURAUDIO_WaveClose(pGeode, CHANNEL0_PLAYBACK);
	pAmd->dma_dac.ready = FALSE;
	release_buffers(pAmd);
}

/**
 * \ingroup linux GX layer
 * \brief
 * Finishes recording,empties the DMA buffer and,disables the DMA controller
 * the DMA buffer
 *
 * \param  struct amd_state *s     pointer to the state staucture for the DMA channel
 */
static void
amd_close_record(struct amd_state *s)
{
	stop_record(s);
	OS_DbgMsg("Closing Stream. \n");
	DURAUDIO_WaveClose(pGeode, CHANNEL1_RECORD);
	s->dma_adc.ready = FALSE;
}

/**
 * \ingroup linux GX layer
 * \brief
 * Programs the DMA for playback(BM0)  or recording (BM1)
 * A running playback will be cancelled by this function
 *
 * \param  struct amd_state *s     pointer to the state staucture for the DMA channel
 *         unsigned int     rec    flag indicating record or playback
 */
static int
prog_dmabuf(struct amd_state *s, unsigned rec)
{
	OS_DbgMsg("prog_dmabuf(%d)\n", rec);
	if (rec) {
		if (DURAUDIO_WaveOpen(pGeode,
				      CHANNEL1_RECORD,
				      s->rateadc,
				      s->dma_adc.nChannels,
				      s->dma_adc.BitsPerSample,
				      s->dma_adc.fragsize,
				      INTERVAL_BYTES,
				      s->dma_adc.dmasize,
				      s->dma_adc.DMAmapped)) {
			set_adc_rate(s, s->rateadc);

			s->dma_adc.ready = TRUE;
		} else {
			return FALSE;
		}
	} else {
		amd_close_playback(s);
		if (DURAUDIO_WaveOpen(pGeode,
				      CHANNEL0_PLAYBACK,
				      s->ratedac,
				      s->dma_dac.nChannels,
				      s->dma_dac.BitsPerSample,
				      s->dma_dac.fragsize,
				      INTERVAL_BYTES,
				      s->dma_dac.dmasize,
				      s->dma_dac.DMAmapped)) {
			OS_DbgMsg("Opening stream.\n");
			set_dac_rate(s, s->ratedac);
			s->dma_dac.ready = TRUE;
		} else {
			printk("Failed Opening stream (Allocation?)\n");
			return FALSE;
		}
	}
	return TRUE;
}

/**
 * \ingroup linux GX layer
 * \brief
 * Updates the current read and write pointer in the channel structure
 * Called from the ISR. If the IRQ flag for busmaster 0 is set
 * the (possibly blocked amd_write) function will be unblocked
 * amd_read does not block!
 *
 * \param  struct amd_state *s     pointer to the state structure for the DMA channel
 */
static void
amd_update_ptr(struct amd_state *s)
{
	long br = 0;
	unsigned int size = 0;
	unsigned long curpos = 0;
	unsigned long diff = 0;

	if (s->enable & DAC_RUNNING && (s->intID & BM0_IRQ)) {
		size = s->dma_dac.fragsize;
		br = get_bytes_in_playback_buffer(s);
		curpos = get_current_dma_pos(CHANNEL0_PLAYBACK);
		if (curpos > s->dma_dac.lastpos) {
			diff = curpos - s->dma_dac.lastpos;
		} else {
			diff = s->dma_dac.lastpos - curpos;
		}
		s->dma_dac.lastpos = curpos;
		s->dma_dac.total_bytes += diff;
		s->dma_dac.fragcnt += diff;
		s->dma_dac.BytesInDMABuffer -= diff;

		if (br == 0 && s->dma_dac.DMAmapped == FALSE) {
			OS_DbgMsg("stopping DMA=%X\n", br);
			s->enable &= ~DAC_RUNNING;
			pGeode->AudioChannel[CHANNEL0_PLAYBACK].IndirectDMA.
			    CurrentTransferPointer = 0;
			DURAUDIO_StopDMA(pGeode, CHANNEL0_PLAYBACK);
		}
		s->dma_dac.BytesReadyInDMABuffer = br;
		OS_DbgMsg(">ISR-DAC bytes remaining=%X\n", br);

		/*Output buffer underun protection Logic */

		/*The test and set are atomic, making this thread-safe */
		/*To avoid starvation for nearly empty buffers */
		if (test_bit(ZF_TRANSFER_SINCE_ZERO, &s->dma_dac.zero_flags)
		    && (br < (long) (size + Z_1MS_OF_BYTES))) {
			/*If amd_write is in the midst of a transfer, don't zero
			   as this will step on the user data */
			if (!test_bit
			    (ZF_TRANSFER_PENDING, &s->dma_dac.zero_flags)) {
				/*Near underflow  We zero the free part of the buffer. */
				zero_free_playback_buffer(s);
				/*and update the flags */
				if (br) {
					set_bit(ZF_PARTIAL_ZERO,
						&s->dma_dac.zero_flags);
				} else {
					clear_bit(ZF_PARTIAL_ZERO,
						  &s->dma_dac.zero_flags);
				}
				clear_bit(ZF_ZERO_SKIPPED,
					  &s->dma_dac.zero_flags);
			} else {
				/*Tell amd_write to zero after the transfer (as one was
				   needed) */
				set_bit(ZF_ZERO_SKIPPED,
					&s->dma_dac.zero_flags);
			}

		} else if (!br
			   && test_bit(ZF_PARTIAL_ZERO,
				       &s->dma_dac.zero_flags)) {
			/*This logic assures that:
			   (a) we do zero EVERYTHING
			   (b) don't zero after we have */

			if (!test_bit
			    (ZF_TRANSFER_PENDING, &s->dma_dac.zero_flags)) {
				/*If amd_write is in the midst of a transfer, don't zero
				   as this will step on the user data */

				/*In complete underflow  We zero the buffer (again, probably). */
				zero_free_playback_buffer(s);
				clear_bit(ZF_PARTIAL_ZERO,
					  &s->dma_dac.zero_flags);
				clear_bit(ZF_ZERO_SKIPPED,
					  &s->dma_dac.zero_flags);
			} else {
				/*Tell amd_write to zero after the transfer (as one was
				   needed) */
				set_bit(ZF_ZERO_SKIPPED,
					&s->dma_dac.zero_flags);
			}

		} else {
			clear_bit(ZF_ZERO_SKIPPED, &s->dma_dac.zero_flags);
		}
		if (s->intID & BM0_IRQ || br == 0) {
			OS_DbgMsg
			    (">amd_update_ptr:  DAC call wake_up_interruptible\n");
			wake_up_interruptible(&s->dma_dac.wait);
		}
	}

	if ((s->enable & ADC_RUNNING) && (s->intID & BM1_IRQ)) {
		s->dma_adc.BytesReadyInDMABuffer =
		    DURAUDIO_UpdateByteCounter(pGeode, CHANNEL1_RECORD);
		s->dma_adc.total_bytes += s->dma_adc.BytesReadyInDMABuffer;
		s->dma_adc.fragcnt += s->dma_adc.BytesReadyInDMABuffer;
		OS_DbgMsg(">ISR-ADC Bytes In DMA Buffer=0x%X\n",
			  s->dma_adc.BytesReadyInDMABuffer);
		if (s->intID & BM1_IRQ) {
			OS_DbgMsg
			    (">amd_update_ptr:  ADC call wake_up_interruptible\n");
			wake_up_interruptible(&s->dma_adc.wait);
		}
	}
}

/**
 * \ingroup linux GX layer
 * \brief
 * Called at AC97 controller interrupt. Reads the current interrupt mask
 * and actualize the read and write pointers
 * for recording and playback.
 *
 * \param  int               irq     unused
 *         void*             dev_id  the pointer to the hw descriptor
 *         struct pt_regs*   regs    unused
 */
static void
amd_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct amd_state *pAmd;
	struct amd_card *pCard;
	unsigned char IntID;

	pCard = (struct amd_card *) dev_id;
	IntID = DURAUDIO_InterruptID(pGeode);
	DURAUDIO_ClearIRQ(pGeode);
	if (IntID & (BM0_IRQ | BM1_IRQ)) {	/*only the busmaster irqs */
		pAmd = &pCard->channels[0];
		pAmd->intID = IntID;
		amd_update_ptr(pAmd);
	}
	return;
}

/**
 * \ingroup linux GX layer
 * \brief
 * Writes the mixer value for both channels
 *
 *
 * \param  struct amd_card *card  pointer to the hw descriptor
 *         unsigned int mixer     mixer control
 *         unsigned int val       MSB = right channel, LSB = left channel
 */
static void
set_volume_level(struct amd_card *card, unsigned int mixer, unsigned int val)
{
	unsigned int leftVal, rightVal;
	OS_DbgMsg("-->set_volume_level mixer=%d, val=%.8X\n", mixer, val);
	/*limit the values */
	rightVal = ((val >> 8) & 0xFF);
	rightVal = (rightVal > 100) ? 100 : rightVal;
	leftVal = ((val & 0xFF) > 100) ? 100 : (val & 0xFF);
	if (mixer == SOUND_MIXER_MIC) {
		rightVal = leftVal;
	}
	card->mix.mixer_state[mixer] = (rightVal << 8) | leftVal;
	card->mix.write_mixer(card, mixer, leftVal, rightVal);
}

/**
 * \ingroup linux GX layer
 * \brief
 * Writes the mixer value for all controls (sliders) with
 * the values stored in software
 *
 *
 * \param  struct amd_card *card  pointer to the hw descriptor
 */
static void
update_all_volume_levels(struct amd_card *card)
{
	int i;

	i = 0;
	while (i < SOUND_MIXER_NRDEVICES) {
		if (supported_mixer(card, i) != 0) {
			set_volume_level(card, i, card->mix.mixer_state[i]);
		}
		i++;
	}
}

/**
 * \ingroup linux GX layer
 * \brief
 * IOCTL handler function for the mixer device read commands
 *
 *
 * \param  struct amd_card *card  pointer to the hw descriptor
 *         unsigned int     cmd   ioctl command (defined in soundcard.h)
 * \return  int             read value
 *

 */
int
handle_mixer_read_ioctl(struct amd_card *pCard, unsigned int cmd)
{
	int val, i;

	OS_DbgMsg("-->handle_mixer_read_ioctl cmd=");
	switch (_IOC_NR(cmd)) {
	case SOUND_MIXER_STEREODEVS:	/*Mixer channels supporting stereo */
		{
			OS_DbgMsg("SOUND_MIXER_STEREODEVS\n");
			val = pCard->mix.stereo_mixers;
			break;
		}
	case SOUND_MIXER_DEVMASK:	/*give them the supported mixers */
		{
			OS_DbgMsg("SOUND_MIXER_DEVMASK\n");
			val = pCard->mix.supported_mixers;
			break;
		}
	case SOUND_MIXER_RECMASK:	/*arg contains a bit for each supported recording source */
		{
			OS_DbgMsg("SOUND_MIXER_RECMASK\n");
			val = pCard->mix.record_sources;
			break;
		}
	case SOUND_MIXER_CAPS:
		{
			OS_DbgMsg("SOUND_MIXER_CAPS\n");
			val = SOUND_CAP_EXCL_INPUT;
			break;
		}
	case SOUND_MIXER_RECSRC:	/*give them the current record source */
		{
			OS_DbgMsg("SOUND_MIXER_RECSRC\n");
			if (!pCard->mix.recmask_io) {
				val = 0;
			} else {
				val = pCard->mix.recmask_io(pCard, 1, 0);
			}
			break;
		}
	default:
		{
			i = _IOC_NR(cmd);

			if (!supported_mixer(pCard, i)) {
				OS_DbgMsg
				    ("<--handle_mixer_read_ioctl,failed! \n");
				return -EINVAL;
			}
			val = pCard->mix.mixer_state[i];
			OS_DbgMsg("%.X\n", cmd);
			break;
		}
	}
	OS_DbgMsg("<--handle_mixer_read_ioctl=%X\n", val);
	return val;
}

/**
 * \ingroup linux GX layer
 * \brief
 * IOCTL handler function for the mixer device
 *
 *
 * \param  struct amd_card *card  pointer to the hw descriptor
 *         unsigned int     cmd   ioctl command (defined in soundcard.h)
 *         unsigned long    arg   ioct argument
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
handle_ioctl_mixer(struct inode *inode, struct file *file,
		   unsigned int cmdcode, unsigned long arg)
{
	int i, val, ret;
	mixer_info info;
	_old_mixer_info oinfo;
	struct amd_card *pCard;
	unsigned long flags = 0;
	spinlock_t lock;

	OS_DbgMsg("-->handle_ioctl_mixer, cmd=0x%X\n", cmdcode);
	pCard = (struct amd_card *) file->private_data;
	val = 0;
	spin_lock_irqsave(&lock, flags);
	switch (cmdcode) {
	case OSS_GETVERSION:
		OS_DbgMsg("<--handle_ioctl_mixer, OSS_GETVERSION=%.8x\n",
			  SOUND_VERSION);
		spin_unlock_irqrestore(&lock, flags);
		return put_user((unsigned long) SOUND_VERSION, (int *) arg);

	case SOUND_MIXER_INFO:
		strncpy(info.id, card_names[pCard->card_type],
			sizeof (info.id));
		strncpy(info.name, card_names[pCard->card_type],
			sizeof (info.name));
		info.modify_counter = pCard->mix.modcnt;
		if (copy_to_user((void *) arg, &info, sizeof (info))) {
			spin_unlock_irqrestore(&lock, flags);
			OS_DbgMsg
			    ("<--handle_ioctl_mixer, SOUND_MIXER_INFO failed!\n");
			return -EFAULT;
		}
		spin_unlock_irqrestore(&lock, flags);
		OS_DbgMsg("<--handle_ioctl_mixer, SOUND_MIXER_INFO\n");
		return 0;

	case SOUND_OLD_MIXER_INFO:
		strncpy(oinfo.id, card_names[pCard->card_type],
			sizeof (oinfo.id));
		strncpy(oinfo.name, card_names[pCard->card_type],
			sizeof (oinfo.name));

		if (copy_to_user((void *) arg, &oinfo, sizeof (oinfo))) {
			spin_unlock_irqrestore(&lock, flags);
			OS_DbgMsg
			    ("<--handle_ioctl_mixer, SOUND_OLD_MIXER_INFO failed!\n");
			return -EFAULT;
		}
		spin_unlock_irqrestore(&lock, flags);
		OS_DbgMsg("<--handle_ioctl_mixer, SOUND_OLD_MIXER_INFO\n");
		return 0;
	default:
		break;
	}

	if (_IOC_TYPE(cmdcode) != 'M' || _IOC_SIZE(cmdcode) != sizeof (int)) {
		spin_unlock_irqrestore(&lock, flags);
		OS_DbgMsg("<--handle_ioctl_mixer,failed! \n");
		return -EINVAL;
	}
	if (_IOC_DIR(cmdcode) == _IOC_READ) {
		val = handle_mixer_read_ioctl(pCard, cmdcode);
		spin_unlock_irqrestore(&lock, flags);
		OS_DbgMsg("<--handle_ioctl_mixer \n");
		return put_user(val, (int *) arg);
	}

	if (_IOC_DIR(cmdcode) != (_IOC_WRITE | _IOC_READ)) {
		spin_unlock_irqrestore(&lock, flags);
		OS_DbgMsg("<--handle_ioctl_mixer,failed! \n");
		return -EINVAL;
	}

	pCard->mix.modcnt++;
	if (get_user(val, (int *) arg)) {
		spin_unlock_irqrestore(&lock, flags);
		OS_DbgMsg("<--handle_ioctl_mixer,failed! \n");
		return -EFAULT;
	}

	ret = 0;
	if (_IOC_NR(cmdcode) == SOUND_MIXER_RECSRC) {
		if (!pCard->mix.recmask_io) {
			ret = (-EINVAL);
		} else if (!val) {
			ret = 0;
		} else if (!(val &= pCard->mix.record_sources)) {
			ret = (-EINVAL);
		} else {
			pCard->mix.recmask_io(pCard, 0, val);
		}
	} else {
		i = _IOC_NR(cmdcode);
		if (!supported_mixer(pCard, i)) {
			ret = (-EINVAL);
		} else {
			set_volume_level(pCard, i, val);
		}
	}
	spin_unlock_irqrestore(&lock, flags);
	OS_DbgMsg("<--handle_ioctl_mixer =%X\n", ret);
	return ret;
}

/**
 * \ingroup linux GX layer
 * \brief
 * Open entry for the mixer device
 *
 *
 * \param  struct inode *inode
 *         struct file *file
 *
 * \return 0 if ok, -ENODEV if the device does not exist
 */
static int
mixdevice_open(struct inode *inode, struct file *file)
{
	struct amd_card *pCard;
	struct pci_dev *pdev;
	struct pci_driver *drvr;
	int minornum;
	unsigned long flags = 0;
	spinlock_t lock;

	OS_DbgMsg("-->mixdevice_open\n");
	pCard = NULL;
	minornum = MINOR(inode->i_rdev);
	spin_lock_irqsave(&lock, flags);
	pdev = pci_dev_g(pci_devices.next);
	while (pdev != pci_dev_g(&pci_devices)) {
		drvr = pci_dev_driver(pdev);
		if (drvr == &geode_pci_driver) {
			pCard = (struct amd_card *) pci_get_drvdata(pdev);
			if (pCard == NULL) {
				pdev = pci_dev_g(pdev->global_list.next);
				continue;
			}
			if (pCard->dev_mixer == minornum) {
				break;
			}
		}
		pdev = pci_dev_g(pdev->global_list.next);
	}
	if (pCard == NULL) {
		OS_DbgMsg("<--mixdevice_open failed: ENODEV\n");
		spin_unlock_irqrestore(&lock, flags);
		return (-ENODEV);
	}
	file->private_data = pCard;
	spin_unlock_irqrestore(&lock, flags);
	OS_DbgMsg("<--mixdevice_open\n");
	return 0;
}

/**
 * \ingroup linux GX layer
 * \brief
 * Close entry for the mixer device
 * Actually this function does nothing!
 *
 * \param  struct inode *inode
 *         struct file *file
 *
 * \return 0
 */
static int
mixdevice_release(struct inode *inode, struct file *file)
{
	return 0;
}

static struct file_operations amd_mixer_fops = {
      owner:THIS_MODULE,
      llseek:no_llseek,
      ioctl:handle_ioctl_mixer,
      open:mixdevice_open,
      release:mixdevice_release,
};

/**
 * \ingroup linux GX layer
 * \brief
 *
 *  File read entry. Recorded data in the DMA buffer will be
 *  copied into the provided buffer.
 *  If the recording is not already started the first call
 *  to this function starts the ADC and DMA transfer
 *
 * \param  struct file*     file    file pointer
 *         char*            buffer  receive buffer
 *         size_t           count   buffer size
 *         loff_t*          ppos    file position, not used
 *
 * \return ssize_t number of received bytes in buffer
 *                 -Error code else
 */

static ssize_t
amd_read(struct file *file, char *buffer, size_t count, loff_t * ppos)
{
	ssize_t ret;
	int retval;
	unsigned long BytesTransferred;
	unsigned long swPointer;
	unsigned long BytesReadyToRead;
	struct amd_state *pAmd;
	unsigned long flags = 0;
	spinlock_t lock;

	DECLARE_WAITQUEUE(wait, current);
	OS_DbgMsg("-->amd_read: want 0x%X bytes\n", count);
	retval = 0;
	pAmd = (struct amd_state *) file->private_data;
	if (ppos != &file->f_pos) {
		retval = (-ESPIPE);
	}
	if (pAmd->dma_adc.DMAmapped) {
		retval = (-ENXIO);
	}
	if (!access_ok(VERIFY_WRITE, buffer, count)) {
		retval = (-EFAULT);
	}
	if (pAmd->card->in_suspend == SLEEP) {
		interruptible_sleep_on(&pAmd->card->suspend_queue);
	}

	spin_lock_irqsave(&lock, flags);
	if (!pAmd->dma_adc.ready) {
		if (!prog_dmabuf(pAmd, 1)) {
			OS_DbgMsg(" prog_dmabuf failed!\n");
			retval = (-ENOMEM);
		}
	}
	if (retval != 0) {
		spin_unlock_irqrestore(&lock, flags);
		return retval;
	}

	/*Start ADC */
	if (!(pAmd->enable & ADC_RUNNING)) {
		start_record(pAmd);
	}
	if (count == 0) {
		spin_unlock_irqrestore(&lock, flags);
		return 0;
	}

	add_wait_queue(&pAmd->dma_adc.wait, &wait);
	swPointer = 0;
	ret = 0;
	while (count) {
		BytesReadyToRead =
		    DURAUDIO_UpdateByteCounter(pGeode, CHANNEL1_RECORD);
		pAmd->dma_adc.BytesReadyInDMABuffer = BytesReadyToRead;
		if (file->f_flags & O_NONBLOCK) {
			if (BytesReadyToRead == 0) {
				if (ret == 0) {
					ret = -EAGAIN;
				}
				OS_DbgMsg(" Un-Blocked!\n");
				goto out1;
			}
		} else {
			spin_unlock_irqrestore(&lock, flags);
			__set_current_state(TASK_INTERRUPTIBLE);
			schedule();
			if (signal_pending(current)) {
				OS_DbgMsg(" Signal Pending!\n");
				ret = ret ? ret : (-EINTR);
				remove_wait_queue(&pAmd->dma_adc.wait, &wait);
				set_current_state(TASK_RUNNING);
				OS_DbgMsg("<--amd_read= 0x%X bytes\n", ret);
				return ret;
			}
			spin_lock_irqsave(&lock, flags);
		}

		if (BytesReadyToRead > count) {
			BytesReadyToRead = count;
		}
		OS_DbgMsg("BytesReadyToRead: %X,  count: %X\n",
			  BytesReadyToRead, count);
		BytesTransferred =
		    DURAUDIO_TransferAudioData(pGeode, CHANNEL1_RECORD,
					       (unsigned char *) (buffer +
								  swPointer),
					       BytesReadyToRead);

		pAmd->dma_adc.BytesReadyInDMABuffer -= BytesTransferred;

		ret += BytesTransferred;
		count -= BytesTransferred;
		swPointer += BytesTransferred;
	}
      out1:
	remove_wait_queue(&pAmd->dma_adc.wait, &wait);
	set_current_state(TASK_RUNNING);
	spin_unlock_irqrestore(&lock, flags);
	OS_DbgMsg("<--amd_read= 0x%X bytes\n", ret);
	return ret;
}

/**
 * \ingroup linux GX layer
 * \brief
 *
 *  File write entry. The data in the buffer will be copied either
 *  directly into the DMA buffer, or if no  space lleft into a new
 *  allocated buffer. If a configurable buffer size has been reached
 *  the fucntion blocks until necessary buffer space is available.
 *  If the device has been opened unblocking, the function returns always,
 *  even if no bytes could be written into a buffer.
 *  If the playback is not already started the first call
 *  to this function starts the DAC and DMA transfer
 *
 * \param  struct file*     file    file pointer
 *         char*            buffer  receive buffer
 *         size_t           count   buffer size
 *         loff_t*          ppos    file position, not used
 *
 * \return ssize_t number of transmitted (buffered) bytes
 *                 -Error code else
 */

static ssize_t
amd_write(struct file *file, const char *buffer, size_t count, loff_t * ppos)
{
	struct amd_state *pAmd;
	ssize_t ret;
	size_t cnt;
	unsigned long BytesTransferred = 0;
	unsigned long BytesAvailableToFill = 0;
	unsigned long BytesInDMABuffer = 0;
	unsigned long swPointer;
	buff_list_t *pBuff;
	unsigned long flags = 0;
	spinlock_t lock;

	OS_DbgMsg("amd_write: count=0x%X\n", count);

	ret = 0;
	if (ppos != &file->f_pos) {
		ret = (-ESPIPE);
	}
	if (!access_ok(VERIFY_READ, buffer, count)) {
		ret = (-EFAULT);
	}
	pAmd = (struct amd_state *) file->private_data;
	if (pAmd->dma_dac.DMAmapped) {
		ret = (-ENXIO);
	}
	if (pAmd->card->in_suspend == SLEEP) {
		if (file->f_flags & O_NONBLOCK) {
			return (-EAGAIN);
		} else {
			OS_DbgMsg("amd_write: blocked by PM\n");
			interruptible_sleep_on(&pAmd->card->suspend_queue);
		}
	}

	spin_lock_irqsave(&lock, flags);
	if (!pAmd->dma_dac.ready) {
		if (!prog_dmabuf(pAmd, 0)) {
			OS_DbgMsg("prog_dmabuf failed!\n");
			spin_unlock_irqrestore(&lock, flags);
			ret = (-ENOMEM);
		}
	}
	if (ret != 0) {
		spin_unlock_irqrestore(&lock, flags);
		return ret;
	}
	if (count == 0) {	/*there is nothing to submit to the dma buffer */
		spin_unlock_irqrestore(&lock, flags);
		return 0;
	}
	ret = count;
	swPointer = 0;
	cnt = count;

	BytesInDMABuffer = get_bytes_in_playback_buffer(pAmd);
	BytesAvailableToFill = pAmd->dma_dac.dmasize - BytesInDMABuffer;
	while ((BytesAvailableToFill != 0) && (pAmd->dma_dac.bufferedSize == 0)
	       && (count != 0)) {
		OS_DbgMsg("BytesInDMABuffer: %X  BytesAvailableToFill: %X\n",
			  BytesInDMABuffer, BytesAvailableToFill);

		if (BytesAvailableToFill > count) {
			BytesAvailableToFill = count;
		}
		BytesTransferred = 0;

		/*The bit prevents the ISR from doing a underrun "zero the buffer" */
		set_bit(ZF_TRANSFER_PENDING, &pAmd->dma_dac.zero_flags);
		BytesTransferred = DURAUDIO_TransferAudioData(pGeode,
							      CHANNEL0_PLAYBACK,
							      (unsigned char
							       *) (buffer +
								   swPointer),
							      BytesAvailableToFill);
		pAmd->dma_dac.BytesInDMABuffer += BytesTransferred;
		OS_DbgMsg("BytesTransferred=0x%.X\n", BytesTransferred);

		/*Transfer since tells the ISR to zero on next underrun */
		/*set_bit(ZF_TRANSFER_SINCE_ZERO,&pAmd->dma_dac.zero_flags); */

		/*If an underrun event happened during the Transfer
		   we need to handle the buffer zeroing here.
		   -- if the buffer is full, then nothing will happen, but
		   this is safe in any case */

		if (test_bit(ZF_ZERO_SKIPPED, &pAmd->dma_dac.zero_flags)) {
			zero_free_playback_buffer(pAmd);
			clear_bit(ZF_ZERO_SKIPPED, &pAmd->dma_dac.zero_flags);
		}
		clear_bit(ZF_TRANSFER_PENDING, &pAmd->dma_dac.zero_flags);

		/*Update the counters and pointers based on the data
		   actually transferred (the returned value) */
		count -= BytesTransferred;
		swPointer += BytesTransferred;

		/* even though the ISR updates this value we still
		   have to add the recent bytes.  The symptom of the race
		   is the counters may get out of sync.  Causing an incomplete
		   "TransferAudio" above. */
		pAmd->dma_dac.BytesReadyInDMABuffer += BytesTransferred;

		/*Is the DAC running yet */
		if (!(pAmd->enable & DAC_RUNNING)) {
			OS_DbgMsg("Trying to start DAC...\n");
			/*if necessary restart the DMA thread */
			wake_up_interruptible(&pAmd->dma_dac.wait);
			start_play(pAmd, START_DAC_DEFAULT);
		}
		BytesInDMABuffer = get_bytes_in_playback_buffer(pAmd);
		BytesAvailableToFill = pAmd->dma_dac.dmasize - BytesInDMABuffer;
		OS_DbgMsg
		    ("BytesInDMABuffer=%X, BytesAvailableToFill=%X, count=%X\n",
		     BytesInDMABuffer, BytesAvailableToFill, count);
		pAmd->dma_dac.count = BytesInDMABuffer;
	}

	if (count) {		/*there are still data which does not fit into the dma buffer -> store them in the buffer list */
		OS_DbgMsg("bufferedSize=%X\n", pAmd->dma_dac.bufferedSize);
		if (file->f_flags & O_NONBLOCK) {
			ret = ret ? ret : (-EAGAIN);
			OS_DbgMsg("Un-Blocked!\n");
			goto out1;
		} else {
			while (pAmd->dma_dac.bufferedSize >= MAX_BUFFERED_BYTES) {
				spin_unlock_irqrestore(&lock, flags);
				OS_DbgMsg
				    ("go sleep and wait until buffersize is lower\n");
				interruptible_sleep_on(&pAmd->dma_dac.wait);
				OS_DbgMsg
				    ("amd_write woke up: bufferedSize=%X\n",
				     pAmd->dma_dac.bufferedSize);
				spin_lock_irqsave(&lock, flags);
			}
		}
		pBuff =
		    (buff_list_t *) kmalloc(count + sizeof (buff_list_t),
					    GFP_ATOMIC);
		if (pBuff == NULL) {
			spin_unlock_irqrestore(&lock, flags);
			OS_DbgMsg("cannot allocate buffer for pending data");
			OS_DbgMsg("<--return to application =ENOMEM\n");
			return (-ENOMEM);
		} else {
			copy_from_user(&pBuff->c, (buffer + swPointer), count);
			pBuff->size = count;
			pAmd->dma_dac.bufferedSize += count;
			list_add_tail((struct list_head *) pBuff,
				      &pAmd->dma_dac.buffer_list);
			OS_DbgMsg("queue remaining data in 0x%.X size=%X\n",
				  pBuff, count);
		}
	}
      out1:;
	spin_unlock_irqrestore(&lock, flags);
	OS_DbgMsg("<--return to application ret=%X\n", cnt);
	return ret;
}

/**
 * \ingroup linux GX layer
 * \brief
 *
 *  Helper function to free all allocated buffers.
 *  Called when the playback will be cancelled.
 *
 * \param  struct amd_state* s   pointer to the channel associated structure
 *
 */
static void
release_buffers(struct amd_state *s)
{
	buff_list_t *pBuff = NULL;

	pBuff = get_buffer_list_head(&s->dma_dac.buffer_list);
	while (pBuff != NULL) {
		OS_DbgMsg
		    ("remove buffer=0x%.8X, size=%X from list and return it\n",
		     pBuff, pBuff->size);
		list_del((struct list_head *) pBuff);
		kfree((void *) pBuff);
		pBuff = get_buffer_list_head(&s->dma_dac.buffer_list);
	}
	s->dma_dac.bufferedSize = 0;
}

/**
 * \ingroup linux GX layer
 * \brief
 *
 *  This is the thread function which does the dma buffer reloading
 *  The function blocks and will be activated periodically by the
 *  BM0 page interrupt. This interrupt occurs every time  if a
 *  PRD-page has been transmitted. If available, the tread fills
 *  the new data from the buffers filled in the amd_write function.
 *  The thread will be terminated if the playback terminates or no
 *  buffered data are available.
 *
 * \param  dma_thread_t *dma_thread   pointer to the channel associated structure
 *
 */
static void
dma_reload_thread(dma_thread_t * dma_thread)
{
	unsigned long BytesInDMABuffer;
	unsigned long BytesAvailableToFill;
	unsigned long BytesTransferred;
	unsigned long swPointer = 0;
	unsigned long count = 0;
	struct amd_state *pAmd = NULL;
	buff_list_t *pBuff = NULL;
	unsigned char *pB = NULL;
	unsigned long flags = 0;
	spinlock_t lock;

	pAmd = dma_thread->pAmdState;
	/* setup the thread environment */
	init_dma_thread(dma_thread, "geodeoss dma");
	OS_DbgMsg("dma_reload_thread\n");
	spin_lock_irqsave(&lock, flags);
	/* an endless loop in which we are doing our work */
	while (1) {
		/* fall asleep for one second */
		OS_DbgMsg("go sleep and wait until data are available\n");
		spin_unlock_irqrestore(&lock, flags);
		interruptible_sleep_on(&pAmd->dma_dac.wait);
		spin_lock_irqsave(&lock, flags);
		OS_DbgMsg("dma_reload_thread: thread woke up\n");

		/* here we are back from sleep because we caught a signal. */
		if (dma_thread->terminate || signal_pending(current)) {
			OS_DbgMsg
			    ("thread received a request to terminate itself\n");
			release_buffers(pAmd);
			break;
		}
		pBuff = get_buffer_list_head(&pAmd->dma_dac.buffer_list);
		BytesInDMABuffer = get_bytes_in_playback_buffer(pAmd);
		if (pBuff != NULL) {
			BytesAvailableToFill =
			    pAmd->dma_dac.dmasize - BytesInDMABuffer;
			pB = (unsigned char *) &pBuff->c;
			count = pBuff->size;
			OS_DbgMsg("drt: Transferring buffer 0x%.8X, size=%X\n",
				  pBuff, count);
			while (BytesAvailableToFill) {
				OS_DbgMsg
				    ("drt: BytesInDMABuffer: %X  BytesAvailableToFill: %X\n",
				     BytesInDMABuffer, BytesAvailableToFill);
				BytesTransferred = 0;
				/*The bit prevents the ISR from doing a underrun "zero the buffer" */
				set_bit(ZF_TRANSFER_PENDING,
					&pAmd->dma_dac.zero_flags);

				if (BytesAvailableToFill > count) {
					BytesAvailableToFill = count;
				}

				BytesTransferred =
				    DURAUDIO_TransferAudioData(pGeode,
							       CHANNEL0_PLAYBACK,
							       (unsigned char
								*) (pB +
								    swPointer),
							       BytesAvailableToFill);
				pAmd->dma_dac.BytesInDMABuffer +=
				    BytesTransferred;
				OS_DbgMsg("drt: BytesTransferred=0x%.X\n",
					  BytesTransferred);

				/*Transfer since tells the ISR to zero on next underrun */
				set_bit(ZF_TRANSFER_SINCE_ZERO,
					&pAmd->dma_dac.zero_flags);

				/*If an underrun event happened during the Transfer
				   we need to handle the buffer zeroing here.
				   -- if the buffer is full, then nothing will happen, but
				   this is safe in any case */

				if (test_bit
				    (ZF_ZERO_SKIPPED,
				     &pAmd->dma_dac.zero_flags)) {
					zero_free_playback_buffer(pAmd);
					clear_bit(ZF_ZERO_SKIPPED,
						  &pAmd->dma_dac.zero_flags);
				}
				clear_bit(ZF_TRANSFER_PENDING,
					  &pAmd->dma_dac.zero_flags);

				/*Update the counters and pointers based on the data
				   actually transferred (the returned value) */
				count -= BytesTransferred;

				if (pAmd->dma_dac.bufferedSize <
				    BytesTransferred) {
					OS_DbgMsg
					    ("drt: ERROR s->dma_dac.bufferedSize < BytesTransferred!!!\n");
					pAmd->dma_dac.bufferedSize = 0;
				} else {
					pAmd->dma_dac.bufferedSize -=
					    BytesTransferred;
				}
				swPointer += BytesTransferred;
				/*even though the ISR updates this value we still
				   have to add the recent bytes.  The symptom of the race
				   is the counters may get out of sync.  Causing an incomplete
				   "TransferAudio" above. */
				pAmd->dma_dac.BytesReadyInDMABuffer +=
				    BytesTransferred;

				/*Is the DAC running yet? */
				if (!(pAmd->enable & DAC_RUNNING)) {
					OS_DbgMsg
					    ("drt: Trying to start DAC...\n");
					start_play(pAmd, START_DAC_DEFAULT);
				}
				if (count) {	/*there are still bytes to send */
					OS_DbgMsg
					    ("drt: still %X bytes to transfer in this block ->>go sleep\n",
					     count);
					/*sleep until we have place in the buffer */
					spin_unlock_irqrestore(&lock, flags);
					interruptible_sleep_on(&pAmd->dma_dac.
							       wait);
					spin_lock_irqsave(&lock, flags);
					if (dma_thread->terminate
					    || signal_pending(current)) {
						/* we received a request to terminate ourself */
						OS_DbgMsg
						    ("thread received a request to terminate itself\n");
						release_buffers(pAmd);
						break;
					}
					OS_DbgMsg
					    ("drt: dma_reload_thread woke up ->continue transferring data \n");
					BytesInDMABuffer =
					    get_bytes_in_playback_buffer(pAmd);
					BytesAvailableToFill =
					    pAmd->dma_dac.dmasize -
					    BytesInDMABuffer;
				} else {
					/*remove buffer from list and return it */
					OS_DbgMsg
					    ("drt: remove buffer=0x%.8X from list and return it\n",
					     pBuff);
					list_del((struct list_head *) pBuff);
					kfree((void *) pBuff);
					swPointer = 0;
					BytesInDMABuffer =
					    get_bytes_in_playback_buffer(pAmd);
					BytesAvailableToFill =
					    pAmd->dma_dac.dmasize -
					    BytesInDMABuffer;
					if (BytesAvailableToFill) {
						pBuff =
						    get_buffer_list_head(&pAmd->
									 dma_dac.
									 buffer_list);
						if (pBuff == NULL) {
							break;	/*return into the endless wait loop */
						}
						pB = (unsigned char *) &pBuff->
						    c;
						count = pBuff->size;
						OS_DbgMsg
						    ("drt: Transferring next buffer 0x%.8X, size=%X\n",
						     pBuff, count);
					}
				}
				pAmd->dma_dac.count = BytesInDMABuffer;
			}

		} else {
			OS_DbgMsg("drt: no data available to transfer\n");
			if (get_bytes_in_playback_buffer(pAmd) == 0) {	/*the DMA buffer runs out */
				OS_DbgMsg
				    ("drt: go sleep because DMA runs empty\n");
				wake_up(&pAmd->close_wait);
				if (pAmd->dma_dac.DMAmapped == FALSE) {
					pAmd->enable &= ~DAC_RUNNING;
					pGeode->AudioChannel[CHANNEL0_PLAYBACK].
					    IndirectDMA.CurrentTransferPointer =
					    0;
					DURAUDIO_StopDMA(pGeode,
							 CHANNEL0_PLAYBACK);
				}
				spin_unlock_irqrestore(&lock, flags);
				interruptible_sleep_on(&pAmd->dma_dac.wait);
				spin_lock_irqsave(&lock, flags);
			}
		}
	}
	/* here we go only in case of termination of the thread */
	/* cleanup the thread, leave */
	spin_unlock_irqrestore(&lock, flags);
	exit_dma_thread(dma_thread);
	complete_and_exit(&dma_thread->thr_exited, 0);
	OS_DbgMsg("drt: dma_reload_thread terminated\n");
}

/**
 * \ingroup linux GX layer
 * \brief
 * The fuction is called from the audio application to
 * poll if data can be delivered to the DMA buffer.
 * Will be called only if the device has been opened unblocking.
 *
 *
 * \param    struct file *file
 *           struct poll_table_struct *wait
 * \return   unsigned int  =0 if no data might be transferred
 *           POLLIN | POLLRDNORM if data can be deliverd to amd_write
 */
static unsigned int
amd_poll(struct file *file, struct poll_table_struct *wait)
{
	struct amd_state *ps;
	unsigned int mask, fmask;
	unsigned long bytesInDmaBuffer;
	unsigned long flags = 0;
	spinlock_t lock;

	OS_DbgMsg("-->>amd_poll:\n");
	ps = (struct amd_state *) file->private_data;
	if (ps->card->in_suspend == SLEEP) {
		return (0);
	}

	mask = 0;
	fmask = file->f_mode & (FMODE_WRITE | FMODE_READ);
	if (fmask & FMODE_WRITE) {
		if (ps->dma_dac.ready || (prog_dmabuf(ps, 0))) {
			poll_wait(file, &ps->dma_dac.wait, wait);
			spin_lock_irqsave(&lock, flags);
			bytesInDmaBuffer = get_bytes_in_playback_buffer(ps);
			OS_DbgMsg("BytesIn DAC-DMABuffer=%X\n",
				  bytesInDmaBuffer);
			if (((unsigned long) ps->dma_dac.dmasize -
			     bytesInDmaBuffer) >= ps->dma_dac.fragsize) {
				mask |= POLLOUT | POLLWRNORM;
			}
			spin_unlock_irqrestore(&lock, flags);
			OS_DbgMsg("amd_poll write mask=%X\n", mask);
		}
	}
	if (fmask & FMODE_READ) {
		if (ps->dma_adc.ready || (prog_dmabuf(ps, 1))) {
			poll_wait(file, &ps->dma_adc.wait, wait);
			spin_lock_irqsave(&lock, flags);
			if (ps->dma_adc.BytesReadyInDMABuffer >=
			    (long) ps->dma_adc.fragsize) {
				OS_DbgMsg
				    ("BytesReadyIn ADC-DMABuffer=%X, dma_adc.fragsize=%X\n",
				     ps->dma_adc.BytesReadyInDMABuffer,
				     ps->dma_adc.fragsize);
				mask |= POLLIN | POLLRDNORM;
			}
			spin_unlock_irqrestore(&lock, flags);
			OS_DbgMsg("amd_poll read mask=%X\n", mask);
		}
	}
	return mask;
}

/**
 * \ingroup linux GX layer
 * \brief
 *
 *
 *
 * \param    struct file            *file
 *           struct vm_area_struct  *vma
 *
 * \return   int
 */

static int
dma_mem_map(struct file *file, struct vm_area_struct *vma)
{
	int ret;
	unsigned long size;
	unsigned long Channel = 0;
	struct amd_state *pAmd;
	struct dmabuf *pDmaBuff = NULL;

	OS_DbgMsg("amd_mmap:\n");
	ret = 0;
	pAmd = (struct amd_state *) file->private_data;
	lock_kernel();

	if (vma->vm_flags & VM_WRITE) {
		Channel = CHANNEL0_PLAYBACK;
		pAmd->dma_dac.DMAmapped = TRUE;
		if (prog_dmabuf(pAmd, 0)) {
			pDmaBuff = &pAmd->dma_dac;
		} else {
			pAmd->dma_dac.DMAmapped = FALSE;
			ret = -EAGAIN;
		}
	} else if (vma->vm_flags & VM_READ) {
		Channel = CHANNEL1_RECORD;
		pAmd->dma_adc.DMAmapped = TRUE;
		if (prog_dmabuf(pAmd, 1)) {
			pDmaBuff = &pAmd->dma_adc;
		} else {
			pAmd->dma_adc.DMAmapped = FALSE;
			ret = -EAGAIN;
		}
	} else {
		ret = -EINVAL;
	}

	if (ret == 0) {
		if (vma->vm_pgoff != 0) {
			unlock_kernel();
			return (-EINVAL);
		}
		size = vma->vm_end - vma->vm_start;
		OS_DbgMsg("size: %d   Allocated DMA size: %d\n", size,
			  pGeode->AudioChannel[Channel].DMA_AllocInfo.Size);

		if (size > pGeode->AudioChannel[Channel].DMA_AllocInfo.Size) {
			pDmaBuff->DMAmapped = FALSE;
			unlock_kernel();
			return (-EINVAL);
		}
		memcpy(&pDmaBuff->vma, vma, sizeof (struct vm_area_struct));
		pDmaBuff->rawbuf =
		    (void *) pGeode->AudioChannel[Channel].DMA_AllocInfo.
		    VirtualAddress;
		if (remap_page_range
		    (vma->vm_start, virt_to_phys(pDmaBuff->rawbuf), size,
		     vma->vm_page_prot)) {
			pDmaBuff->DMAmapped = FALSE;
			unlock_kernel();
			return (-EAGAIN);
		}

		pDmaBuff->DMAmapped = TRUE;
		pDmaBuff->ready = TRUE;
	}
	unlock_kernel();
	return ret;
}

/**
 * \ingroup linux GX layer
 * \brief
 *
 *  Empties the DMA buffer by playing bufferd bytes.
 *  The function blocks until the buffer is empty.
 *
 * \param   struct amd_state *s  channel state structure
 *
 * \return  int =0 if OK
 *          -EFAULT if the DAC is not ready
 */
int
empty_dac(struct amd_state *pAmd)
{
	unsigned long userBytes;

	userBytes = get_bytes_in_playback_buffer(pAmd);

	if (userBytes == 0) {	/*no more bytes available */
		return 0;
	}
	start_play(pAmd, START_DAC_START_ANY);
	if (!(pAmd->enable & DAC_RUNNING)) {
		OS_DbgMsg("geodeoss: empty_dac: couldn't start DAC\n");
		return (-EFAULT);
	}
	/*Play the remaining buffered bytes */
	do {
		interruptible_sleep_on(&pAmd->dma_dac.wait);
		if (signal_pending(current)) {
			OS_DbgMsg("empty_dac: Signal Pending!\n");
			userBytes = 0;
		} else {
			userBytes = get_bytes_in_playback_buffer(pAmd);
		}
	}
	while (userBytes);

	return 0;
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_SYNC
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static void
sndctl_dsp_sync(struct file *file, struct amd_state *pAmd)
{
	OS_DbgMsg("SNDCTL_DSP_SYNC\n");
	/*The app wants us to flush all the data we have in our buffers (If we are playing) */
	if ((file->f_mode & FMODE_WRITE) && (pAmd->enable & DAC_RUNNING)) {
		empty_dac(pAmd);
	}
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_SETDUPLEX
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sndctl_dsp_setduplex(struct file *file, struct amd_state *pAmd)
{
	OS_DbgMsg("SNDCTL_DSP_SETDUPLEX\n");
	return 0;
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_GETCAPS
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *         int*              pArg   ioctl argument
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sndctl_dsp_getcaps(struct file *file, struct amd_state *pAmd, int *pArg)
{
	OS_DbgMsg("SNDCTL_DSP_GETCAPS\n");
	return put_user(DSP_CAP_DUPLEX | DSP_CAP_REALTIME | DSP_CAP_TRIGGER |
			DSP_CAP_MMAP, pArg);
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_RESET
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sndctl_dsp_reset(struct file *file, struct amd_state *pAmd)
{
	OS_DbgMsg("SNDCTL_DSP_RESET\n");

	if (file->f_mode & FMODE_WRITE) {
		stop_play(pAmd);
		synchronize_irq();
		pAmd->dma_dac.ready = FALSE;
	}
	if (file->f_mode & FMODE_READ) {
		stop_record(pAmd);
		synchronize_irq();
		pAmd->dma_dac.ready = FALSE;
	}
	return 0;
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_SPEED
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *         int*              pArg   ioctl argument
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sndctl_dsp_speed(struct file *file, struct amd_state *pAmd, int *pArg)
{
	int val, ret;

	OS_DbgMsg("SNDCTL_DSP_SPEED\n");
	ret = 0;
	if (get_user(val, pArg)) {
		return -EFAULT;
	}
	if (val >= 0) {
		if (file->f_mode & FMODE_READ) {
			stop_record(pAmd);
			set_adc_rate(pAmd, val);
		}

		if (file->f_mode & FMODE_WRITE) {
			stop_play(pAmd);
			set_dac_rate(pAmd, val);
		}
	}
	OS_DbgMsg("Rate: %d\n",
		  (file->f_mode & FMODE_READ) ? pAmd->rateadc : pAmd->ratedac);
	if (file->f_mode & FMODE_READ) {
		ret = put_user(pAmd->rateadc, pArg);
	} else {
		ret = put_user(pAmd->ratedac, pArg);
	}
	return ret;
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_STEREO
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *         int*              pArg   ioctl argument
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sndctl_dsp_stereo(struct file *file, struct amd_state *pAmd, int *pArg)
{
	int val;
	OS_DbgMsg("SNDCTL_DSP_STEREO\n");

	if (get_user(val, pArg)) {
		return (-EFAULT);
	}
	if (file->f_mode & FMODE_READ) {
		if (val) {
			pAmd->dma_adc.nChannels = 2;
		} else {
			pAmd->dma_adc.nChannels = 1;
		}
	}
	if (file->f_mode & FMODE_WRITE) {
		if (val) {
			pAmd->dma_dac.nChannels = 2;
		} else {
			pAmd->dma_dac.nChannels = 1;
		}
	}
	return put_user(val, pArg);
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_CHANNELS
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *         int*              pArg   ioctl argument
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sndctl_dsp_channels(struct file *file, struct amd_state *pAmd, int *pArg)
{
	int val;
	OS_DbgMsg("SNDCTL_DSP_CHANNELS\n");

	if (get_user(val, pArg)) {
		return (-EFAULT);
	}
	if ((val != 1) && (val != 2)) {
		return (-EINVAL);
	}
	if (file->f_mode & FMODE_READ) {
		pAmd->dma_adc.nChannels = val;
	}

	if (file->f_mode & FMODE_WRITE) {
		pAmd->dma_dac.nChannels = val;
	}
	return put_user(val, pArg);
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_GETFMTS
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *         int*              pArg   ioctl argument
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sndctl_dsp_getfmts(struct file *file, struct amd_state *pAmd, int *pArg)
{
	OS_DbgMsg("SNDCTL_DSP_GETFMTS\n");
	return put_user(AFMT_U8 | AFMT_S16_LE, pArg);
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_SETFMT
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *         int*              pArg   ioctl argument
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sndctl_dsp_setfmt(struct file *file, struct amd_state *pAmd, int *pArg)
{
	int val;

	OS_DbgMsg("SNDCTL_DSP_SETFMT\n");
	if (get_user(val, pArg)) {
		return (-EFAULT);
	}
	if (val != AFMT_QUERY) {
		if (file->f_mode & FMODE_READ) {
			if (val == AFMT_S16_LE) {
				pAmd->dma_adc.BitsPerSample = 16;
			} else {
				/*Our hardware only does 16bit Stereo on DirectDMA. */
				if (pAmd->dma_adc.DMAmapped) {
					printk
					    ("We cannot do DirectDMA in anything different than 16bit Signed Little Endian format.\n");
					return (-EFAULT);
				}

				pAmd->dma_adc.BitsPerSample = 8;

				if (val != AFMT_U8) {
					/*This format is not supported */
					OS_DbgMsg
					    ("We don't support this format. Only Unsigned 8bit or 16bit Signed Little Endian formats.\n");
					return (-EINVAL);
				}
			}
		}

		if (file->f_mode & FMODE_WRITE) {
			OS_DbgMsg("Format: %08X\n", val);
			if (val == AFMT_S16_LE) {
				pAmd->dma_dac.BitsPerSample = 16;
			} else {
				/*Our hardware only does 16bit Stereo on DirectDMA. */
				if (pAmd->dma_dac.DMAmapped) {
					OS_DbgMsg
					    ("We cannot do DirectDMA in anything different than 16bit Signed Little Endian format.\n");
					return (-EFAULT);
				}

				pAmd->dma_dac.BitsPerSample = 8;

				if (val != AFMT_U8) {
					/*This format is not supported */
					OS_DbgMsg
					    ("We don't support this format. Only Unsigned 8bit or 16bit Signed Little Endian formats.\n");
					return (-EINVAL);
				}
			}
		}
	}
	OS_DbgMsg("BitsPerSample: %d  val:%d\n", pAmd->dma_dac.BitsPerSample,
		  val);
	return put_user(val, pArg);
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_POST
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sndctl_dsp_post(struct file *file, struct amd_state *pAmd)
{
	OS_DbgMsg("SNDCTL_DSP_POST\n");
	return 0;
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_GETTRIGGER
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *         int*              pArg   ioctl argument
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sndctl_dsp_gettrigger(struct file *file, struct amd_state *pAmd, int *pArg)
{
	int val;
	OS_DbgMsg("SNDCTL_DSP_GETTRIGGER\n");
	val = 0;
	if ((file->f_mode & FMODE_READ) && (pAmd->enable & ADC_RUNNING))
		val |= PCM_ENABLE_INPUT;

	if ((file->f_mode & FMODE_WRITE) && (pAmd->enable & DAC_RUNNING))
		val |= PCM_ENABLE_OUTPUT;
	return put_user(val, pArg);
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_SETTRIGGER
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *         int*              pArg   ioctl argument
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sndctl_dsp_settrigger(struct file *file, struct amd_state *pAmd, int *pArg)
{
	int val;

	OS_DbgMsg("SNDCTL_DSP_SETTRIGGER\n");
	if (get_user(val, pArg)) {
		return -EFAULT;
	}
	if (file->f_mode & FMODE_READ) {
		if (val & PCM_ENABLE_INPUT) {
			OS_DbgMsg
			    ("SNDCTL_DSP_SETTRIGGER: FMODE_READ - START\n");
			OS_DbgMsg("pAmd->enable: %d\n", pAmd->enable);

			if (pAmd->enable & ADC_RUNNING) {
				/*the ADC is already running */
				return 0;
			}
			if (pAmd->dma_adc.ready) {
				start_record(pAmd);
			}
		} else {
			stop_record(pAmd);
		}
	}

	if (file->f_mode & FMODE_WRITE) {
		if (val & PCM_ENABLE_OUTPUT) {
			OS_DbgMsg
			    ("SNDCTL_DSP_SETTRIGGER: FMODE_WRITE - START\n");
			/*OS_DbgMsg("pAmd->enable: %d\n",        pAmd->enable);
			   OS_DbgMsg("pAmd->dma_dac.ready: %s\n", pAmd->dma_dac.ready?"TRUE":"FALSE"); */

			if (pAmd->enable & DAC_RUNNING) {
				/*the DAC is already running */
				return 0;
			}
			if (pAmd->dma_dac.ready) {
				start_play(pAmd, START_DAC_START_ANY);
			}
		} else {
			stop_play(pAmd);
		}
	}
	return 0;
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_GETOSPACE
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *         int*              pArg   ioctl argument
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sndctl_dsp_getospace(struct file *file, struct amd_state *pAmd, int *pArg)
{
	long BytesInBuffer;
	audio_buf_info abinfo;

	OS_DbgMsg("SNDCTL_DSP_GETOSPACE: \n");

	if (!(file->f_mode & FMODE_WRITE)) {
		return -EINVAL;
	}

	abinfo.fragsize = pAmd->dma_dac.fragsize;
	abinfo.fragstotal = pAmd->dma_dac.dmasize / pAmd->dma_dac.fragsize;

	if (!pAmd->dma_dac.ready) {
		abinfo.bytes = (pAmd->dma_dac.dmasize);
		abinfo.fragments = abinfo.bytes / pAmd->dma_dac.fragsize;

		OS_DbgMsg
		    ("fragsize: %d fragstotal: %d Bytes: 0x%d  fragments: %d\n",
		     abinfo.fragsize, abinfo.fragstotal, abinfo.bytes,
		     abinfo.fragments);

		return copy_to_user((void *) pArg, &abinfo,
				    sizeof (abinfo)) ? -EFAULT : 0;
	}

	if (pAmd->enable & DAC_RUNNING) {
		BytesInBuffer = get_bytes_in_playback_buffer(pAmd);
	} else {
		BytesInBuffer = 0;
	}
	if (file->f_flags & O_NONBLOCK) {	/*if we get polled return the correct available size */
		/*align it to fragment size */
		abinfo.bytes = pAmd->dma_dac.dmasize - BytesInBuffer;
		abinfo.bytes =
		    (abinfo.bytes / pAmd->dma_dac.fragsize) *
		    pAmd->dma_dac.fragsize;
		abinfo.fragments = abinfo.bytes / pAmd->dma_dac.fragsize;
	} else {		/*if we are not polled, claim always maximal size. */

		abinfo.bytes = MAX_BUFFERED_BYTES - pAmd->dma_dac.bufferedSize;
		abinfo.fragments = abinfo.bytes / pAmd->dma_dac.fragsize;

	}
	OS_DbgMsg
	    ("Space: fragsize: %d bytes: %d fragstotal:%d fragments: %d\n",
	     pAmd->dma_dac.fragsize, abinfo.bytes, abinfo.fragstotal,
	     abinfo.fragments);
	return copy_to_user((void *) pArg, &abinfo,
			    sizeof (abinfo)) ? -EFAULT : 0;
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_GETISPACE
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *         int*              pArg   ioctl argument
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sndctl_dsp_getispace(struct file *file, struct amd_state *pAmd, int *pArg)
{
	audio_buf_info abinfo;

	OS_DbgMsg("SNDCTL_DSP_GETISPACE\n");

	if (!(file->f_mode & FMODE_READ)) {
		return -EINVAL;
	}
	if (!pAmd->dma_adc.ready) {
		return -EINVAL;
	}
	abinfo.fragsize = pAmd->dma_adc.fragsize;
	abinfo.bytes = pAmd->dma_dac.BytesReadyInDMABuffer;
	abinfo.fragstotal = pAmd->dma_adc.dmasize / pAmd->dma_adc.fragsize;
	abinfo.fragments = abinfo.bytes / pAmd->dma_adc.fragsize;
	OS_DbgMsg("fragsize: %d bytes: %d fragstotal:%d fragments: %d\n",
		  pAmd->dma_adc.fragsize, abinfo.bytes, abinfo.fragstotal,
		  abinfo.fragments);

	return copy_to_user((void *) pArg, &abinfo,
			    sizeof (abinfo)) ? -EFAULT : 0;
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_NONBLOCK
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sndctl_dsp_nonblock(struct file *file, struct amd_state *pAmd)
{
	OS_DbgMsg("SNDCTL_DSP_NONBLOCK\n");
	file->f_flags |= O_NONBLOCK;
	return 0;
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_GETODELAY
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *         int*              pArg   ioctl argument
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sndctl_dsp_getodelay(struct file *file, struct amd_state *pAmd, int *pArg)
{
	int val;

	OS_DbgMsg("SNDCTL_DSP_GETODELAY\n");

	if (!(file->f_mode & FMODE_WRITE)) {
		return (-EINVAL);
	}
	if (!pAmd->dma_dac.ready) {
		/*transfer has not been started */
		return (-EPIPE);
	}
	val =
	    (int) get_bytes_in_playback_buffer(pAmd) +
	    pAmd->dma_dac.bufferedSize;
	if ((pAmd->enable & DAC_RUNNING) && (val <= 0)) {
		amd_close_playback(pAmd);
		return put_user(0, pArg);
	}
	if (!pAmd->dma_dac.bufferedSize) {
		val = 0;	/*submit data asap */
	}
	OS_DbgMsg("Bytes Ready in DMA buffer: %X\n", val);
	return put_user(val, pArg);
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_GETIPTR
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *         int*              pArg   ioctl argument
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sndctl_dsp_getiptr(struct file *file, struct amd_state *pAmd, int *pArg)
{
	count_info cinfo;

	OS_DbgMsg("SNDCTL_DSP_GETIPTR\n");
	if (!(file->f_mode & FMODE_READ)) {
		return -EINVAL;
	}
	if (!pAmd->dma_adc.ready) {
		return -EINVAL;
	}
	cinfo.bytes = pAmd->dma_adc.total_bytes;
	cinfo.blocks = pAmd->dma_adc.fragcnt / pAmd->dma_adc.fragsize;
	cinfo.ptr = get_current_dma_pos(CHANNEL1_RECORD);
	cinfo.ptr =
	    cinfo.ptr / ((16 / pAmd->dma_adc.BitsPerSample) *
			 (2 / pAmd->dma_adc.nChannels));
	pAmd->dma_adc.fragcnt = 0;
	return copy_to_user((void *) pArg, &cinfo, sizeof (cinfo));

}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_GETOPTR
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *         int*              pArg   ioctl argument
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sndctl_dsp_getoptr(struct file *file, struct amd_state *pAmd, int *pArg)
{
	count_info cinfo;

	OS_DbgMsg("SNDCTL_DSP_GETOPTR\n");

	if (!(file->f_mode & FMODE_WRITE)) {
		return -EINVAL;
	}
	if (!pAmd->dma_dac.ready) {
		return -EINVAL;
	}
	cinfo.bytes = pAmd->dma_dac.total_bytes;
	cinfo.blocks = pAmd->dma_dac.fragcnt / pAmd->dma_dac.fragsize;
	cinfo.ptr = get_current_dma_pos(CHANNEL0_PLAYBACK);

	cinfo.ptr =
	    cinfo.ptr / ((16 / pAmd->dma_dac.BitsPerSample) *
			 (2 / pAmd->dma_dac.nChannels));
	OS_DbgMsg("  cinfo.bytes  =%u, ", cinfo.bytes);
	OS_DbgMsg("  cinfo.blocks =%u, ", cinfo.blocks);
	OS_DbgMsg("  cinfo.ptr    =0x%X\n", cinfo.ptr);
	pAmd->dma_dac.fragcnt = 0;
	return copy_to_user((void *) pArg, &cinfo, sizeof (cinfo));
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_GETBLKSIZE
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *         int*              pArg   ioctl argument
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sndctl_dsp_getblksize(struct file *file, struct amd_state *pAmd, int *pArg)
{
	OS_DbgMsg("SNDCTL_DSP_GETBLKSIZE\n");

	if (file->f_mode & FMODE_WRITE) {
		OS_DbgMsg("DAC fragsize: %X  dmasize: %X\n",
			  pAmd->dma_dac.fragsize, pAmd->dma_dac.dmasize);
		return put_user(pAmd->dma_dac.fragsize, pArg);
	} else {
		OS_DbgMsg("ADC fragsize: %X  dmasize: %X\n",
			  pAmd->dma_adc.fragsize, pAmd->dma_adc.dmasize);
		return put_user(pAmd->dma_adc.fragsize, pArg);
	}
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_SETFRAGMENT
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *         int*              pArg   ioctl argument
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sndctl_dsp_setfragment(struct file *file, struct amd_state *pAmd, int *pArg)
{
	int val;
	if (get_user(val, pArg)) {
		return (-EFAULT);
	}
	OS_DbgMsg("SNDCTL_DSP_SETFRAGMENT\n");

	if (file->f_mode & FMODE_READ) {
		pAmd->dma_adc.ossfragshift = val & 0xffff;
		pAmd->dma_adc.ossmaxfrags = (val >> 16) & 0xffff;

		pAmd->dma_adc.fragsize = 1;
		pAmd->dma_adc.fragsize <<= (val & 0x0000FFFF);
		pAmd->dma_adc.ossfragsize = pAmd->dma_adc.fragsize;

		/*0x7FFF means no limits for number of frags.  Set it to DEFAULT_NUMBER_OF_FRAGMENTS. */
		if (pAmd->dma_adc.ossmaxfrags == 0x7FFF) {
			pAmd->dma_adc.ossmaxfrags = DEFAULT_NUMBER_OF_FRAGMENTS;
		}
		if (pAmd->dma_adc.dmasize > (DEFAULT_DMA_BUFFER_SIZE)) {
			pAmd->dma_adc.dmasize = DEFAULT_DMA_BUFFER_SIZE;
			pAmd->dma_adc.fragsize = DEFAULT_FRAGMENT_SIZE;
			OS_DbgMsg
			    ("Adjusted to:  Fragsize: %X  MaxFrags: %X DMAsize: %X\n",
			     pAmd->dma_adc.ossfragsize,
			     (DEFAULT_DMA_BUFFER_SIZE /
			      pAmd->dma_adc.ossfragsize),
			     pAmd->dma_adc.dmasize);
		}

		pAmd->dma_adc.ready = FALSE;
		OS_DbgMsg("ADC: fragsize=0x%.4X maxfrags=0x%.4X\n",
			  pAmd->dma_adc.fragsize, pAmd->dma_adc.ossmaxfrags);
	}

	if (file->f_mode & FMODE_WRITE) {
		pAmd->dma_dac.ossfragshift = val & 0xffff;
		pAmd->dma_dac.ossmaxfrags = (val >> 16) & 0xffff;

		pAmd->dma_dac.ossfragsize = 1;
		pAmd->dma_dac.ossfragsize <<= (val & 0x0000FFFF);

		/*0x7FFF means no limits for number of frags.  Set it to DEFAULT_NUMBER_OF_FRAGMENTS. */
		if (pAmd->dma_dac.ossmaxfrags == 0x7FFF) {
			pAmd->dma_dac.ossmaxfrags = DEFAULT_NUMBER_OF_FRAGMENTS;
		}
		if (pAmd->dma_dac.dmasize > (DEFAULT_DMA_BUFFER_SIZE)) {
			pAmd->dma_dac.dmasize = DEFAULT_DMA_BUFFER_SIZE;
			pAmd->dma_dac.fragsize = DEFAULT_FRAGMENT_SIZE;
			OS_DbgMsg
			    ("Adjusted to:  Fragsize: %X  MaxFrags: %X DMAsize: %X\n",
			     pAmd->dma_dac.ossfragsize,
			     (DEFAULT_DMA_BUFFER_SIZE /
			      pAmd->dma_dac.ossfragsize),
			     pAmd->dma_dac.dmasize);
		}

		pAmd->dma_dac.ossmaxfrags =
		    pAmd->dma_dac.dmasize / pAmd->dma_dac.fragsize;

		pAmd->dma_dac.fragsize = pAmd->dma_dac.ossfragsize;
		pAmd->dma_dac.ready = FALSE;
		OS_DbgMsg("DAC: fragsize=0x%.4X maxfrags=0x%.4X\n",
			  pAmd->dma_dac.fragsize, pAmd->dma_dac.ossmaxfrags);
	}
	return 0;

}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_SUBDIVIDE
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *
 * \return  -ENOSYS  if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sndctl_dsp_subdivide(struct file *file, struct amd_state *pAmd, int *pArg)
{
	int val;
	OS_DbgMsg("geodeoss: SNDCTL_DSP_SUBDIVIDE\n");

	if ((file->f_mode & FMODE_READ && pAmd->dma_adc.subdivision)
	    || (file->f_mode & FMODE_WRITE && pAmd->dma_dac.subdivision)) {
		return (-EINVAL);
	}
	if (get_user(val, pArg)) {
		return (-EFAULT);
	}
	if (val != 1 && val != 2 && val != 4) {
		return (-EINVAL);
	}
	if (file->f_mode & FMODE_READ) {
		pAmd->dma_adc.subdivision = val;
	}
	if (file->f_mode & FMODE_WRITE) {
		pAmd->dma_dac.subdivision = val;
	}
	return 0;
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SOUND_PCM_READ_RATE
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *         int*              pArg   ioctl argument
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sound_pcm_read_rate(struct file *file, struct amd_state *pAmd, int *pArg)
{
	OS_DbgMsg("geodeoss: SOUND_PCM_READ_RATE\n");
	return put_user((file->f_mode & FMODE_READ) ? pAmd->rateadc : pAmd->
			ratedac, pArg);

}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SOUND_PCM_READ_CHANNELS
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *         int*              pArg   ioctl argument
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sound_pcm_read_channels(struct file *file, struct amd_state *pAmd, int *pArg)
{
	OS_DbgMsg("geodeoss: SOUND_PCM_READ_CHANNELS\n");
	return put_user((pAmd->fmt & ((file->f_mode & FMODE_READ)
				      ? (NSC_FMT_STEREO << NSC_ADC_SHIFT)
				      : (NSC_FMT_STEREO << NSC_DAC_SHIFT))) ? 2
			: 1, pArg);

}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SOUND_PCM_READ_BITS
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *         int*              pArg   ioctl argument
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sound_pcm_read_bits(struct file *file, struct amd_state *pAmd, int *pArg)
{
	OS_DbgMsg("geodeoss: SOUND_PCM_READ_BITS\n");
	return put_user((pAmd->fmt & ((file->f_mode & FMODE_READ)
				      ? (NSC_FMT_16BIT << NSC_ADC_SHIFT)
				      : (NSC_FMT_16BIT << NSC_DAC_SHIFT))) ? 16
			: 8, pArg);
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SOUND_PCM_WRITE_FILTER
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sound_pcm_write_filter(struct file *file, struct amd_state *pAmd)
{
	OS_DbgMsg("geodeoss: SOUND_PCM_WRITE_FILTER\n");
	return (-ENOSYS);
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SNDCTL_DSP_SETSYNCRO
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sndctl_dsp_setsyncro(struct file *file, struct amd_state *pAmd)
{
	OS_DbgMsg("geodeoss: SNDCTL_DSP_SETSYNCRO\n");
	return (-ENOSYS);
}

/**
 * \ingroup ioctl calls
 * \brief
 *
 * IOCTL handler function for SOUND_PCM_READ_FILTER
 *
 *
 * \param  struct file*      file
 *         struct amd_state *pAmd
 *
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
sound_pcm_read_filter(struct file *file, struct amd_state *pAmd)
{
	OS_DbgMsg("geodeoss: SOUND_PCM_READ_FILTER\n");
	return (-ENOSYS);
}

/**
 * \ingroup linux GX layer
 * \brief
 *
 * IOCTL handler function for the wave out device
 *
 *
 * \param  struct amd_card *card  pointer to the hw descriptor
 *         unsigned int     cmd   ioctl command (defined in soundcard.h)
 *         unsigned long    arg   ioct argument
 * \return  0        if successful
 *          -EINVAL  invalid value
 *          -EFAULT  else
 */
static int
amd_ioctl(struct inode *inode, struct file *file,
	  unsigned int cmd, unsigned long arg)
{
	struct amd_state *pAmd;
	int val, ret;
	unsigned long flags = 0;
	spinlock_t lock;

	OS_DbgMsg("geodeoss: amd_ioctl: cmd ");
	val = 0;
	ret = 0;
	pAmd = (struct amd_state *) file->private_data;
	if (pAmd->card->in_suspend == SLEEP) {
		interruptible_sleep_on(&pAmd->card->suspend_queue);
	}

	spin_lock_irqsave(&lock, flags);
	switch (cmd) {
	case OSS_GETVERSION:
		ret = put_user(SOUND_VERSION, (int *) arg);
		break;
	case SNDCTL_DSP_SYNC:
		sndctl_dsp_sync(file, pAmd);
		break;
	case SNDCTL_DSP_SETDUPLEX:
		ret = sndctl_dsp_setduplex(file, pAmd);
		break;
	case SNDCTL_DSP_GETCAPS:
		ret = sndctl_dsp_getcaps(file, pAmd, (int *) arg);
		break;
	case SNDCTL_DSP_RESET:
		ret = sndctl_dsp_reset(file, pAmd);
		break;
	case SNDCTL_DSP_SPEED:
		ret = sndctl_dsp_speed(file, pAmd, (int *) arg);
		break;
	case SNDCTL_DSP_STEREO:
		ret = sndctl_dsp_stereo(file, pAmd, (int *) arg);
		break;
	case SNDCTL_DSP_CHANNELS:
		ret = sndctl_dsp_channels(file, pAmd, (int *) arg);
		break;
	case SNDCTL_DSP_GETFMTS:
		ret = sndctl_dsp_getfmts(file, pAmd, (int *) arg);
		break;
	case SNDCTL_DSP_SETFMT:
		ret = sndctl_dsp_setfmt(file, pAmd, (int *) arg);
		break;
	case SNDCTL_DSP_POST:
		ret = sndctl_dsp_post(file, pAmd);
		break;
	case SNDCTL_DSP_GETTRIGGER:
		ret = sndctl_dsp_gettrigger(file, pAmd, (int *) arg);
		break;
	case SNDCTL_DSP_SETTRIGGER:
		ret = sndctl_dsp_settrigger(file, pAmd, (int *) arg);
		break;
	case SNDCTL_DSP_GETOSPACE:
		ret = sndctl_dsp_getospace(file, pAmd, (int *) arg);
		break;
	case SNDCTL_DSP_GETISPACE:
		ret = sndctl_dsp_getispace(file, pAmd, (int *) arg);
		break;
	case SNDCTL_DSP_NONBLOCK:
		ret = sndctl_dsp_nonblock(file, pAmd);
		break;
	case SNDCTL_DSP_GETODELAY:
		ret = sndctl_dsp_getodelay(file, pAmd, (int *) arg);
		break;
	case SNDCTL_DSP_GETIPTR:
		ret = sndctl_dsp_getiptr(file, pAmd, (int *) arg);
		break;
	case SNDCTL_DSP_GETOPTR:
		ret = sndctl_dsp_getoptr(file, pAmd, (int *) arg);
		break;
	case SNDCTL_DSP_GETBLKSIZE:
		ret = sndctl_dsp_getblksize(file, pAmd, (int *) arg);
		break;
	case SNDCTL_DSP_SETFRAGMENT:
		ret = sndctl_dsp_setfragment(file, pAmd, (int *) arg);
		break;
	case SNDCTL_DSP_SUBDIVIDE:
		ret = sndctl_dsp_subdivide(file, pAmd, (int *) arg);
		break;
	case SOUND_PCM_READ_RATE:
		ret = sound_pcm_read_rate(file, pAmd, (int *) arg);
		break;
	case SOUND_PCM_READ_CHANNELS:
		ret = sound_pcm_read_channels(file, pAmd, (int *) arg);
		break;
	case SOUND_PCM_READ_BITS:
		ret = sound_pcm_read_bits(file, pAmd, (int *) arg);
		break;
	case SOUND_PCM_WRITE_FILTER:
		ret = sound_pcm_write_filter(file, pAmd);
		break;
	case SNDCTL_DSP_SETSYNCRO:
		ret = sndctl_dsp_setsyncro(file, pAmd);
		break;
	case SOUND_PCM_READ_FILTER:
		ret = sound_pcm_read_filter(file, pAmd);
		break;
	default:
		ret = (-EINVAL);
		OS_DbgMsg("unknown\n");
		break;
	}
	spin_unlock_irqrestore(&lock, flags);
	return ret;
}

/**
 * \ingroup linux GX layer
 * \brief
 * Function entry to open the device (called by fileopen)
 * Initializes the channel structures with default values.
 * Blocks if the device is not free for opening.
 *
 * \param   struct inode *inode
 *          struct file  *file
 * \return  0 if ok
 *          -ENODEV if the device is not available
 */
static int
amd_open(struct inode *inode, struct file *file)
{
	struct amd_card *c = devs;
	struct amd_state *s = NULL;

	OS_DbgMsg("-->amd_open\n");

	if (c == NULL || c->in_suspend == SLEEP) {
		return (-ENODEV);
	}

	c->in_suspend = OPERATIONAL;
	s = &c->channels[0];
	file->private_data = s;

	/*wait for device to become free */
	down(&s->open_sem);

	while (s->open_mode & file->f_mode) {
		if (file->f_flags & O_NONBLOCK) {
			up(&s->open_sem);
			OS_DbgMsg("  amd_open failed: EWOULDBLOCK \n");
			return (-EWOULDBLOCK);
		}

		up(&s->open_sem);
		interruptible_sleep_on(&s->open_wait);

		if (signal_pending(current)) {
			return (-EINTR);
		}
		down(&s->open_sem);
	}

	s->intID = 0;

	/*Initializing everything... */

	if (file->f_mode & FMODE_READ) {
		s->dma_adc.ossfragsize = s->dma_adc.fragsize =
		    DEFAULT_FRAGMENT_SIZE;
		s->dma_adc.dmasize = DEFAULT_DMA_BUFFER_SIZE;

		s->dma_adc.DMAmapped = FALSE;
		s->dma_adc.total_bytes = 0;
		s->dma_adc.rawbuf = NULL;
		s->dma_adc.ready = FALSE;
		s->dma_adc.BytesReadyInDMABuffer = 0;
		s->dma_adc.BytesInDMABuffer = 0;
		s->dma_adc.fragcnt = 0;
		s->dma_adc.lastpos = 0;
		/*Our hardware only does 16bit Stereo on DirectDMA.
		   Assume it as default. */

		s->dma_adc.BitsPerSample = 16;
		s->dma_adc.nChannels = 2;
		s->dma_adc.dma_thread.function = NULL;
		s->dma_adc.dma_thread.thread = NULL;
		s->dma_adc.dma_thread.pAmdState = s;
		s->dma_adc.bufferedSize = 0;
		INIT_LIST_HEAD(&s->dma_adc.buffer_list);
		set_adc_rate(s, 8000);
	}

	if (file->f_mode & FMODE_WRITE) {
		s->dma_dac.ossfragsize = s->dma_dac.fragsize =
		    DEFAULT_FRAGMENT_SIZE;
		s->dma_dac.dmasize = DEFAULT_DMA_BUFFER_SIZE;

		s->dma_dac.DMAmapped = FALSE;
		s->dma_dac.total_bytes = 0;
		s->dma_dac.rawbuf = NULL;
		s->dma_dac.ready = FALSE;
		s->dma_dac.BytesReadyInDMABuffer = 0;
		s->dma_dac.BytesInDMABuffer = 0;
		s->dma_dac.fragcnt = 0;
		s->dma_dac.lastpos = 0;

		/*Our hardware only does 16bit Stereo on DirectDMA.
		   Assume it as default. */

		s->dma_dac.BitsPerSample = 16;
		s->dma_dac.nChannels = 2;
		s->dma_dac.bufferedSize = 0;
		INIT_LIST_HEAD(&s->dma_dac.buffer_list);
		set_dac_rate(s, 44100);
	}

	s->open_mode |= file->f_mode & (FMODE_READ | FMODE_WRITE);
	up(&s->open_sem);
	return 0;
}

/**
 * \ingroup linux GX layer
 * \brief
 * Function entry to close the device (called by fileclose)
 * Terminates recoprding and playing if active
 * Blocks if the device is not free for opening.
 *
 * \param   struct inode *inode
 *          struct file  *file
 * \return  0 if ok
 *          -ENODEV if the device is not available
 */
static int
amd_release(struct inode *inode, struct file *file)
{
	struct amd_state *pAmd = (struct amd_state *) file->private_data;

	OS_DbgMsg("amd_release\n");

	if ((file->f_mode & FMODE_WRITE)) {
		if (pAmd->dma_dac.ready) {
			/*Flushing DAC */
			OS_DbgMsg("Flushing DAC\n");
			empty_dac(pAmd);
		}
		amd_close_playback(pAmd);
	}
	if (file->f_mode & FMODE_READ) {
		amd_close_record(pAmd);
	}
	lock_kernel();
	down(&pAmd->open_sem);
	pAmd->open_mode &= (~file->f_mode) & (FMODE_READ | FMODE_WRITE);
	pAmd->dma_dac.rawbuf = pAmd->dma_adc.rawbuf = NULL;
	pAmd->dma_dac.DMAmapped = pAmd->dma_adc.DMAmapped = FALSE;
	pAmd->dma_dac.ready = pAmd->dma_adc.ready = FALSE;
	up(&pAmd->open_sem);
	wake_up(&pAmd->open_wait);
	unlock_kernel();
	return 0;
}

static struct file_operations amd_audio_fops = {
      owner:THIS_MODULE,
      llseek:no_llseek,
      read:amd_read,
      write:amd_write,
      poll:amd_poll,
      ioctl:amd_ioctl,
      mmap:dma_mem_map,
      open:amd_open,
      release:amd_release,
};

/**
 * \ingroup linux GX layer
 * \brief
 * Checks and initialize the power management features
 *
 *
 *
 * \param    struct pci_dev             *pcidev
 *           struct amd_card            *pCard
 */
static void
pwrmgmt_init(struct pci_dev *pcidev, struct amd_card *pCard)
{
	unsigned int n;
	struct pm_dev *pmdev;

	if (powermgmt != 1) {
		powermgmt = 0;
	}
	if (powermgmt) {
		pci_read_config_dword(pcidev, pCard->power_regs, &n);
		OS_DbgMsg(KERN_INFO
			  "geodeoss: PCI power management capability: 0x%x\n",
			  n >> 16);
		pmdev =
		    pm_register(PM_PCI_DEV, PM_PCI_ID(pcidev),
				pwrmgmt_callback);
		if (pmdev) {
			pmdev->data = pCard;
		} else {
			printk(KERN_WARNING "%s%s%s", "geodeoss: ",
			       "disabling PM: ", "PM registration failed\n");
			powermgmt = 0;
		}
	}
}

/**
 * \ingroup linux GX layer
 * \brief
 * Allocates all necessary ressoures for the driver
 *
 *
 *
 * \param    struct pci_dev             *pcidev
 *           struct amd_card            *pCard
 *           int                        cardType
 *
 * \return   int=0 if OK
 *              -ENODEV in case of error
 */
static int
alloc_ressources(struct pci_dev *pcidev, struct amd_card **pCard, int cardType)
{
	int ret;
	unsigned long baseaddr;

	/*enable the device */
	ret = pci_enable_device(pcidev);
	if (ret != 0) {
		OS_DbgMsg(KERN_WARNING
			  "geodeoss: FAILED - pci_enable_device!\n");
		return ret;
	}
	/*get the base address */
	baseaddr = (unsigned long) pci_resource_start(pcidev, 0);
	if (baseaddr == 0) {
		OS_DbgMsg(KERN_WARNING "geodeoss: FAILED - No IO Base!\n");
		return -ENODEV;
	}
	if (pcidev->irq == 0) {
		OS_DbgMsg(KERN_WARNING "geodeoss: FAILED - No IRQ!\n",
			  pcidev->irq);
		return -ENODEV;
	}
	pci_set_master(pcidev);

	pGeode = DURAUDIO_Initialize((unsigned int) pcidev->irq);
	if (pGeode == NULL) {
		return -ENODEV;
	}
	if (cardType == TYPE_GX2) {

		OS_DbgMsg("Allocate IO space.\n");
		if (request_region(baseaddr, 128, card_names[cardType]) == NULL) {
			OS_DbgMsg(KERN_WARNING
				  "geodeoss: can't allocate 128 bytes I/O at 0x%4.4x\n",
				  baseaddr);
			return -EBUSY;
		} else {
			bIoAlloc = TRUE;
		}
	}
	/*allocate memory for the hw description structure */
	*pCard = kmalloc(sizeof (struct amd_card), GFP_KERNEL);
	if (*pCard == NULL) {
		OS_DbgMsg(KERN_WARNING "geodeoss: cannot allocate memory\n");
		if ((*pCard)->card_type == TYPE_GX2) {
			release_region(baseaddr, 128);
			return -ENOMEM;
		}
	}
	memset((*pCard), 0, sizeof (struct amd_card));
	(*pCard)->iobase = baseaddr;
	(*pCard)->card_type = cardType;
	return 0;
}

/**
 * \ingroup linux GX layer
 * \brief
 * Initialize the mixer related data
 *
 *
 * \param    struct pci_dev             *pcidev
 *           struct amd_card            *pCard
 */
static void
mixer_init(struct pci_dev *pcidev, struct amd_card *pCard)
{
	OS_DbgMsg("-->mixer_init\n");
	pCard->mix.supported_mixers = AC97_SUPPORTED_MASK;
	pCard->mix.stereo_mixers = AC97_STEREO_MASK;
	pCard->mix.record_sources = AC97_RECORD_MASK;
	pCard->mix.write_mixer = set_ac97_mixer_volume;
	pCard->mix.recmask_io = rw_ac97_record_mask;
	pCard->mix.supported_mixers &= ~(SOUND_MASK_BASS | SOUND_MASK_TREBLE);

	if ((pCard->dev_mixer = register_sound_mixer(&amd_mixer_fops, -1)) < 0) {
		OS_DbgMsg("geodeoss: couldn't register mixer!\n");
	} else {
		memcpy(pCard->mix.mixer_state, mixer_defaults,
		       sizeof (pCard->mix.mixer_state));
		update_all_volume_levels(pCard);
	}
	OS_DbgMsg("<--mixer_init\n");
}

/**
 * \ingroup linux GX layer
 * \brief
 * Allocates the irq for the device
 *
 *
 * \param    struct pci_dev             *pcidev
 *           struct amd_card            *pCard
 *
 * \return   int  0 if successful
 */
static int
alloc_irq(struct pci_dev *pcidev, struct amd_card *pCard)
{
	struct amd_state *ps;
	int ret;

	ret =
	    request_irq(pCard->irq, amd_interrupt, SA_SHIRQ,
			card_names[pCard->card_type], pCard);
	if (ret != 0) {
		OS_DbgMsg(KERN_ERR "geodeoss: unable to allocate irq %d,\n",
			  pCard->irq);
		unregister_sound_mixer(pCard->dev_mixer);
		ps = &pCard->channels[0];
		if (ps->dev_audio != -1) {
			unregister_sound_dsp(ps->dev_audio);
		}
		if (pCard->card_type == TYPE_GX2) {
			release_region(pCard->iobase, 128);
		}
		unregister_reboot_notifier(&audio_notify);
		kfree(pCard);
		return ret;
	}
	return 0;
}

/**
 * \ingroup linux GX layer
 * \brief
 * Enables and initialize the HW device and
 * allocate memory for administrative structures.
 * At least the device will be registered in the system.
 *
 * \param    struct pci_dev             *pcidev
 *           const struct pci_device_id *pdid
 *
 * \return   int=0 if OK
 *              -ENODEV in case of error
 */
static int __init
probe_cs5535(struct pci_dev *pcidev, const struct pci_device_id *pdid)
{
	int card_type, ret, n;
	struct amd_card *pCard;
	struct amd_state *pAmd;
#ifndef MODULE
	static int modcnt = 0;
#endif

	OS_DbgMsg(KERN_WARNING "geodeoss: Probing hardware\n");
	card_type = pdid->driver_data;
#ifndef MODULE
	if (modcnt == 0) {
		modcnt++;
		printk(version);
	}
#endif

	pCard = NULL;
	OS_DbgMsg("Found Card: %s\n", card_names[card_type]);

	ret = alloc_ressources(pcidev, &pCard, card_type);
	if (ret != 0) {
		OS_DbgMsg("alloc_ressources: failed\n");
		return ret;
	}

	pCard->pcidev = pcidev;
	pCard->irq = pcidev->irq;
	spin_lock_init(&pCard->lock);
	pCard->dock_mute_vol = 50;
	devs = pCard;
	init_waitqueue_head(&pCard->suspend_queue);

	/*init the state structure */
	pAmd = &pCard->channels[0];
	pAmd->card = pCard;
	init_waitqueue_head(&pAmd->dma_adc.wait);
	init_waitqueue_head(&pAmd->dma_dac.wait);
	init_waitqueue_head(&pAmd->open_wait);
	init_waitqueue_head(&pAmd->close_wait);
	init_MUTEX(&pAmd->open_sem);

	pAmd->enable = 0;
	pAmd->dev_audio = register_sound_dsp(&amd_audio_fops, -1);

	OS_DbgMsg(KERN_INFO
		  "geodeoss: Configuring %s found at IO 0x%04X IRQ %d\n",
		  card_names[pCard->card_type], pCard->iobase, pCard->irq);
	pci_read_config_dword(pcidev, PCI_SUBSYSTEM_VENDOR_ID, &n);
	OS_DbgMsg(KERN_INFO "geodeoss:  subvendor id: 0x%08x\n", n);

	pwrmgmt_init(pcidev, pCard);
	mixer_init(pcidev, pCard);
	ret = alloc_irq(pcidev, pCard);
	if (ret != 0) {
		return ret;
	}
	pci_set_drvdata(pcidev, pCard);
	start_dma_thread(dma_reload_thread, pAmd);
	return 0;
}

/**
 * \ingroup linux GX layer
 * \brief
 * Function is called by the OS to remove the
 * device from system.Releases all allocated
 * sysytem ressources.
 *
 *
 * \param struct pci_dev *pcidev
 */

static void
remove_cs5535(struct pci_dev *pcidev)
{
	struct amd_card *pCard;
	struct amd_state *pAmd;

	OS_DbgMsg("geodeoss: remove\n");
	pCard = pci_get_drvdata(pcidev);
	pAmd = &pCard->channels[0];

	OS_DbgMsg("terminate dma_thread: pid=%d\n",
		  pAmd->dma_dac.dma_thread.thr_pid);
	stop_dma_thread(&pAmd->dma_dac.dma_thread);
	wait_for_completion(&pAmd->dma_dac.dma_thread.thr_exited);

	free_irq(pCard->irq, pCard);
	unregister_sound_mixer(pCard->dev_mixer);
	if (pAmd->dev_audio != -1) {
		unregister_sound_dsp(pAmd->dev_audio);
	}
	if (bIoAlloc == TRUE) {
		release_region(pCard->iobase, 128);
	}
	kfree(pCard);
	pci_set_drvdata(pcidev, NULL);
}

static struct pci_device_id geode_pci_tbl[] __devinitdata = {
	{PCI_VENDOR_CYRIX, PCI_DEVICE_ID_NSC_GX1, PCI_ANY_ID, PCI_ANY_ID, 0, 0,
	 TYPE_GX1},
	{PCI_VENDOR_NSC, PCI_DEVICE_ID_NSC_SCXX00, PCI_ANY_ID, PCI_ANY_ID, 0,
	 0, TYPE_SCXX00},
	{PCI_VENDOR_NSC, PCI_DEVICE_ID_NSC_GX2, PCI_ANY_ID, PCI_ANY_ID, 0, 0,
	 TYPE_GX2},
	{0,}
};

MODULE_DEVICE_TABLE(pci, geode_pci_tbl);

static struct pci_driver geode_pci_driver = {
      name:"geodeoss",
      id_table:geode_pci_tbl,
      probe:probe_cs5535,
      remove:remove_cs5535,
};

/**
 * \ingroup linux GX layer
 * \brief
 * Called by the OS when the driver will be loaded
 *
 *
 * \param
 * \return  int=returned value from pci_module_init
 */
int __init
audiodrv_init(void)
{
	int ret;

	OS_DbgMsg("geodeoss: audiodrv_init\n");
	ret = pci_module_init(&geode_pci_driver);
	if (ret >= 0) {
		OS_DbgMsg("geodeoss: pci_module_init Succeded\n");

		if (register_reboot_notifier(&audio_notify) != 0) {
			OS_DbgMsg(KERN_WARNING
				  "geodeoss: register_reboot_notifier failed\n");
		}
#ifdef MODULE
		printk(version);
#endif
	}
	return ret;
}

/**
 * \ingroup linux GX layer
 * \brief
 * This notifier is called when the kernel is really shut down.
 * It will remove all card instances too.
 *
 * \param struct notifier_block* nb
 *        unsigned long          event
 *        void *                 buf
 *
 * \return int
 */

static int
audio_shutdown_notify_callback(struct notifier_block *pNotify,
			       unsigned long event, void *buf)
{
	/*this notifier is called when the kernel is really shut down. */
	OS_DbgMsg("geodeoss: shutting down\n");
	pci_unregister_driver(&geode_pci_driver);
	return NOTIFY_OK;
}

/**
 * \ingroup linux GX layer
 * \brief
 * Called whe the drive will be unloaded
 *
 *
 */
void
audiodrv_exit(void)
{
	printk("geodeoss: unloading\n");
	pci_unregister_driver(&geode_pci_driver);
	if (powermgmt) {
		pm_unregister_all(pwrmgmt_callback);
	}
	unregister_reboot_notifier(&audio_notify);
}

/**
 * \ingroup linux GX layer
 * \brief
 * Called on PM request by the OS
 * But power management is actually not supported. so we
 * always  to be busy!
 *
 * \param  struct pm_dev* dev
 *         pm_request_t   rqst  PM request
 *         void*          data  (unused)
 *
 * \return int
 */
int
pwrmgmt_callback(struct pm_dev *dev, pm_request_t request, void *data)
{
	struct amd_card *pCard;
	struct amd_state *pAmd;
	int retCode;
	unsigned long flags = 0;
	spinlock_t lock;

	pCard = (struct amd_card *) dev->data;
	pAmd = &pCard->channels[0];
	retCode = 0;
	if (pCard != NULL) {
		if (request == PM_SUSPEND) {
			OS_DbgMsg("geodeoss: PM_SUSPEND received\n");
			pCard->in_suspend = SLEEP;

			if (pAmd->enable & DAC_RUNNING) {
				interruptible_sleep_on_timeout(&pAmd->
							       close_wait,
							       2000);
				OS_DbgMsg("geodeoss:  DURAUDIO_PauseDMA\n");
				stop_play(pAmd);
				zero_free_playback_buffer(pAmd);
			}
			spin_lock_irqsave(&lock, flags);
			DURAUDIO_SetPowerState(pGeode, DURAUDIO_D4);
			spin_unlock_irqrestore(&lock, flags);
			retCode = 0;
		} else if (request == PM_RESUME) {
			OS_DbgMsg("geodeoss: PM_RESUME received\n");
			pCard->in_suspend = AWAKE;
			wake_up_interruptible(&pCard->suspend_queue);

			spin_lock_irqsave(&lock, flags);
			DURAUDIO_SetPowerState(pGeode, DURAUDIO_D0);
			pCard->in_suspend = OPERATIONAL;
			if (pAmd->enable & DAC_RUNNING
			    || (pAmd->dma_dac.bufferedSize)
			    || get_bytes_in_playback_buffer(pAmd)) {
				OS_DbgMsg
				    ("geodeoss: restart play after PM_RESUME\n");
				start_play(pAmd, START_DAC_START_ANY);
				if (pAmd->dma_dac.bufferedSize) {
					wake_up_interruptible(&pAmd->dma_dac.
							      wait);
				}
			}
			spin_unlock_irqrestore(&lock, flags);
			retCode = 0;
		}
	}
	return retCode;
}

/**
 * \ingroup linux GX layer
 * \brief
 *
 * Helper function to retrieve the head of the
 * buffer list or NULL if the list is empty.
 *
 * \param   struct list_head* pList
 * \return  buff_list_t*
 */
static buff_list_t *
get_buffer_list_head(struct list_head *pList)
{
	if (list_empty(pList))
		return NULL;	/*the list is empty */
	return (buff_list_t *) pList->next;
}

/**
 * \ingroup linux GX layer
 * \brief
 * Launches the given thread.
 *
 *
 * \param  void *data  casted dma_thread_t pointer
 */
static void
dma_thread_launcher(void *data)
{
	dma_thread_t *dma_thread = data;
	dma_thread->thr_pid =
	    kernel_thread((int (*)(void *)) dma_thread->function,
			  (void *) dma_thread, 0);
}

/**
 * \ingroup linux GX layer
 * \brief
 * Create a new dma thread. Called by the creator.
 *
 *
 * \param   void (*func)(dma_thread_t *)  pointer to the thread function
 *          dma_thread_t *dma_thread      thread structure
 */
static void
start_dma_thread(void (*func) (dma_thread_t *), struct amd_state *pAmd)
{
	dma_thread_t *dma_thread;

	dma_thread = &pAmd->dma_dac.dma_thread;
	if (dma_thread->bRunning == FALSE) {
		OS_DbgMsg("dma_thread started\n");
		init_completion(&dma_thread->thr_exited);
		dma_thread->thread = NULL;
		dma_thread->pAmdState = pAmd;
		init_MUTEX_LOCKED(&dma_thread->startstop_sem);
		dma_thread->function = func;
		dma_thread->tq.sync = 0;
		INIT_LIST_HEAD(&dma_thread->tq.list);
		dma_thread->tq.routine = dma_thread_launcher;
		dma_thread->tq.data = dma_thread;
		schedule_task(&dma_thread->tq);
		down(&dma_thread->startstop_sem);
		dma_thread->bRunning = TRUE;
	}
}

/**
 * \ingroup linux GX layer
 * \brief
 *
 *  Stop the dma thread. Called by the removing instance
 *
 * \param dma_thread_t *dma_thread pointer to the thread stucture
 */
static void
stop_dma_thread(dma_thread_t * dma_thread)
{
	if (dma_thread->thread == NULL) {
		return;
	}
	lock_kernel();
	init_MUTEX_LOCKED(&dma_thread->startstop_sem);
	dma_thread->terminate = 1;
	kill_proc(dma_thread->thread->pid, SIGKILL, 1);
	down(&dma_thread->startstop_sem);
	unlock_kernel();
	kill_proc(2, SIGCHLD, 1);
}

/**
 * \ingroup linux GX layer
 * \brief
 *
 * Initialize new created thread. Called by the new thread
 *
 * \param dma_thread_t* dma_thread  pointer to the thread structure
 *        char*         name        name of the thread (displayed with "ps -aux"
 */
static void
init_dma_thread(dma_thread_t * dma_thread, char *name)
{
	lock_kernel();
	dma_thread->bRunning = FALSE;
	dma_thread->thread = current;
	siginitsetinv(&current->blocked,
		      sigmask(SIGKILL) | sigmask(SIGINT) | sigmask(SIGTERM));
	dma_thread->terminate = 0;
	sprintf(current->comm, name);
	unlock_kernel();
	up(&dma_thread->startstop_sem);
}

/**
 * \ingroup linux GX layer
 * \brief
 * Cleanup of thread. Called by the exiting thread.
 *
 *
 * \param dma_thread_t *dma_thread  pointer to the thread structure
 */
static void
exit_dma_thread(dma_thread_t * dma_thread)
{
	lock_kernel();
	dma_thread->thread = NULL;
	dma_thread->bRunning = FALSE;
	up(&dma_thread->startstop_sem);
}

/**
 * \ingroup linux GX layer
 * \brief
 * Returne the number of unplayed bytes in the DMA buffer
 * 
 *
 * \param struct amd_state *pAmd 
 *
 * \return int number of unplayed bytes in the DMA buffer
 */
static int
get_bytes_in_playback_buffer(struct amd_state *pAmd)
{
	return DURAUDIO_UpdateByteCounter(pGeode, CHANNEL0_PLAYBACK);
}

/**
 * \ingroup linux GX layer
 * \brief
 * Returne the current position in the DMA buffer
 * 
 * \param unsigned char channel
 *
 * \return unsigned long current pointer position
 */
static unsigned long
get_current_dma_pos(unsigned char channel)
{
	unsigned long DMACurrentPointer;

	DMACurrentPointer =
	    OS_ReadPortULong((unsigned short) pGeode->AudioChannel[channel].
			     AudioBusMaster.DMAPointer);
	return (DMACurrentPointer -
		pGeode->AudioChannel[channel].PRDTable[0].ulPhysAddr);
}

module_init(audiodrv_init);
module_exit(audiodrv_exit);
 /**/
