/*
 * <LIC_AMD_STD>
 * Copyright (c) 2004 Advanced Micro Devices, Inc.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * The full GNU General Public License is included in this distribution in the
 * file called COPYING
 * </LIC_AMD_STD>
 * <CTL_AMD_STD>
 * </CTL_AMD_STD>
 * <DOC_AMD_STD>
 * </DOC_AMD_STD>
 */

#include "duraudio.h"
#include "os_inc.h"

#define DURAUDIO_VERSION  "DURAUDIO_1.01.12"

const unsigned short VolumeLUT[] = {
	0x1f, 0x1e, 0x1d, 0x1c, 0x1b, 0x1a, 0x19, 0x18,
	0x17, 0x16, 0x15, 0x14, 0x13, 0x12, 0x11, 0x10,
	0x0f, 0x0e, 0x0d, 0x0c, 0x0b, 0x0a, 0x09, 0x08,
	0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00
};

const struct {
	char *Name;
	unsigned long ID;
	unsigned char fCS5535;
} Device[] = {
	{
	"Centaurus", 0x01001078, FALSE}, {
	"Scorpius", 0x04001078, FALSE}, {
	"Carmel", 0x0500100b, FALSE}, {
	"Tavor", 0x0510100b, FALSE}, {
	"Capstrano", 0x002B100b, TRUE}, {
"", 0x00000000, FALSE}};

const struct {
	char *SysName;
	unsigned long VendorID;
} CapSystem[] = {
	{
	"Hawk", 0x0028100B}, {
	"Windsor", 0x20801022}, {
"", 0x00000000}};

/* the following arrays contains 2 entries, One for input, one for output */

durLPWAVEFORMATEX g_pwfx[2];

/*----------------------------------------------------------------------------

 PDURAUDIO DURAUDIO_Initialize (unsigned int Irq)

 Initializes the 5530 hardware.
 It has to be called with the IRQ assigned by the OS

 This function:

 - Reset and initialize the CODEC 

 ------------------------------------------------------------------------------*/
PDURAUDIO
DURAUDIO_Initialize(unsigned int Irq)
{
	PDURAUDIO pGeode;

	OS_DbgMsg("DURAUDIO_Initialize\n");

	pGeode = (PDURAUDIO) OS_AllocateMemory(sizeof (DURAUDIO));

	if (pGeode == NULL)
		return FALSE;

	memset(pGeode, 0, sizeof (DURAUDIO));

	if (!DURAUDIO_InitAudioRegs(pGeode))
		return FALSE;

	/*------------------------------------------------------------------------
	             Interrupt-Related code
	--------------------------------------------------------------------------
	  As a default, we assume we are using DirectDMA
	*/
	pGeode->AudioChannel[CHANNEL0_PLAYBACK].fDirectDMA = TRUE;
	pGeode->AudioChannel[CHANNEL1_RECORD].fDirectDMA = TRUE;

	DURAUDIO_SetupIRQ(pGeode, (unsigned long) Irq);

	/*   Save the selected IRQ */

	pGeode->SelectedIRQ = Irq;

	/*  Prepare the bit mask (To clear Interrupts) */

	pGeode->uIRQMask = 1;
	pGeode->uIRQMask <<= pGeode->SelectedIRQ;
	pGeode->uIRQMask = ~pGeode->uIRQMask;

	OS_InitSpinLock();

	/*------------------------------------------------------------------------
	             End of Interrupt-Related code
	 -----------------------------------------------------------------------*/

	/*
	   Setup Flags for input and output buffers

	   v_nNextPage:  Points to the next buffer to be filled with data.
	   v_fMoreData:  Flag that signals if there is more data coming.
	 */
	pGeode->AudioChannel[CHANNEL0_PLAYBACK].v_fMoreData = FALSE;
	pGeode->AudioChannel[CHANNEL1_RECORD].v_fMoreData = FALSE;

	pGeode->AudioChannel[CHANNEL0_PLAYBACK].fAllocated = FALSE;
	pGeode->AudioChannel[CHANNEL1_RECORD].fAllocated = FALSE;

	return pGeode;
}

unsigned long
DURAUDIO_InitAudioRegs(PDURAUDIO pGeode)
{
	unsigned long F3BAR0_Phys, config_data, pci_config_addr;
	unsigned short i, DevIndex, pci_cmd, CapSysIndex;

	/*
	   PCI Audio/DMA  init

	   Go probe for PCI device reg spaces...
	   NOTE: This loop checks that the 5530 Bridge Header is found.

	 */
	for (i = 0; i < 32; i++) {

		/* Iterate over all dev numbers in PCI Config addr reg. */

		pci_config_addr = 0x80000000 | (i << 11);

		/* Get Vendor_ID+dev_id reg (1st word in PCI Header).  */

		OS_WritePortULong(PCI_CADDR, pci_config_addr);
		config_data = OS_ReadPortULong(PCI_CDATA);

		/* Check if the device is one of ours... */

		for (DevIndex = 0; Device[DevIndex].ID; ++DevIndex) {
			if (Device[DevIndex].ID == config_data)
				goto Found_Device;
		}
	}

	/*  Didn't find the Device ID.  Return. */

	OS_DbgMsg("Cant find PCI Audio dev!\n");
	return (FALSE);

	/*  Found the device ID.  Now initialize the device. */

      Found_Device:;

	OS_DbgMsg("Found Device: %s\n", Device[DevIndex].Name);

	pGeode->fCS5535 = Device[DevIndex].fCS5535;

	/*  Add GX3/5535 support */
	if (pGeode->fCS5535) {

		/*      Get Host Bridge Vendor_ID  */

		OS_WritePortULong(PCI_CADDR, 0x80000800);
		config_data = OS_ReadPortULong(PCI_CDATA);

		/* Check system name because of the same device ID on 5535 */

		for (CapSysIndex = 0; CapSystem[CapSysIndex].VendorID;
		     ++CapSysIndex) {
			if (CapSystem[CapSysIndex].VendorID == config_data)
				goto Found_5535_System;
		}

		OS_DbgMsg("Can't find 5535 System!\n");
		return (FALSE);

	      Found_5535_System:;

		OS_DbgMsg("Found 5535 Platform: %s\n",
			  CapSystem[CapSysIndex].SysName);

	}

	/* 10-13h. BASE REGISTER --> VSA Audio Regs are mem mapped at this loc. */

	pGeode->PCI_Header_Registers = pci_config_addr | PCI_FUNC3_AUDIO;
	F3BAR0_Phys = DURAUDIO_IO_Read32(pGeode->PCI_Header_Registers, 0x10);

	/*      Read the PCI Command register */

	pci_cmd =
	    DURAUDIO_IO_Read16(pGeode->PCI_Header_Registers, PCI_AUDIO_CMD_REG);

	if (F3BAR0_Phys & 0x00000001) {
		pGeode->fIOAccess = TRUE;

		/*   Set it to IO mapping */

		pci_cmd |= 0x05;
		DURAUDIO_IO_Write16(pGeode->PCI_Header_Registers,
				    PCI_AUDIO_CMD_REG, pci_cmd);

		pGeode->F3BAR0 = F3BAR0_Phys & 0xFFFFFFFE;
	} else {
		pGeode->fIOAccess = FALSE;

		/*  Set it to memory mapping */

		pci_cmd |= 0x06;
		DURAUDIO_IO_Write16(pGeode->PCI_Header_Registers,
				    PCI_AUDIO_CMD_REG, pci_cmd);

		pGeode->F3BAR0 = OS_Get_F3BAR0_Virt(F3BAR0_Phys);
	}

	if (!pGeode->F3BAR0) {
		OS_DbgMsg("Cant Map F3BAR0.\n");
		return (FALSE);
	}

	/*    Initialize the Audio BusMaster register pointers */

	pGeode->AudioChannel[CHANNEL0_PLAYBACK].AudioBusMaster.
	    CommandRegister = pGeode->F3BAR0 + 0x20;
	pGeode->AudioChannel[CHANNEL0_PLAYBACK].AudioBusMaster.
	    PRDTableAddress = pGeode->F3BAR0 + 0x24;
	pGeode->AudioChannel[CHANNEL0_PLAYBACK].AudioBusMaster.DMAPointer =
	    pGeode->F3BAR0 + 0x60;
	pGeode->AudioChannel[CHANNEL0_PLAYBACK].AudioBusMaster.DirectionBit =
	    PCI_READS;
	pGeode->AudioChannel[CHANNEL0_PLAYBACK].Running = FALSE;

	pGeode->AudioChannel[CHANNEL1_RECORD].AudioBusMaster.CommandRegister =
	    pGeode->F3BAR0 + 0x28;
	pGeode->AudioChannel[CHANNEL1_RECORD].AudioBusMaster.PRDTableAddress =
	    pGeode->F3BAR0 + 0x2C;
	pGeode->AudioChannel[CHANNEL1_RECORD].AudioBusMaster.DMAPointer =
	    pGeode->F3BAR0 + 0x64;
	pGeode->AudioChannel[CHANNEL1_RECORD].AudioBusMaster.DirectionBit =
	    PCI_WRITES;
	pGeode->AudioChannel[CHANNEL1_RECORD].Running = FALSE;

	/*------------------------------------------------------------------------
	                             Interrupt-Related code
	------------------------------------------------------------------------*/

	pGeode->IRQControlRegister = pGeode->F3BAR0 + 0x1C;
	pGeode->InternalIRQEnableRegister = pGeode->F3BAR0 + 0x1A;
	pGeode->AudioChannel[CHANNEL0_PLAYBACK].AudioBusMaster.
	    SMI_StatusRegister = pGeode->F3BAR0 + 0x21;
	pGeode->AudioChannel[CHANNEL1_RECORD].AudioBusMaster.
	    SMI_StatusRegister = pGeode->F3BAR0 + 0x29;

	/*------------------------------------------------------------------------
	                          End of Interrupt-Related code
	-------------------------------------------------------------------------*/

	/*  CODEC - RESET and volumes initalization. */

	if (pGeode->fCS5535) {

		/*  Set the Warm RESET and CODEC_COMMAND_NEW bits. */

		OS_WritePortULong((unsigned short) (pGeode->F3BAR0 +
						    CODEC_CONTROL_REG_5535),
				  0x00030000);

		if (!DURAUDIO_WaitForBit
		    (pGeode, CODEC_STATUS_REG_5535, BIT_CODEC_READY, SET, 400,
		     NULL)) {
			OS_DbgMsg("Primary Codec NOT Ready...Aborting\n");
			return FALSE;
		}
	} else {

		/*  If we have a SCXX00 device we do a AC97 RESET */

		if ((Device[DevIndex].ID == 0x0500100B)
		    || (Device[DevIndex].ID == 0x0510100B)) {
			unsigned char ResetControlRegister;

			/*  Read the Reset Control register */

			OS_WritePortULong(PCI_CADDR, 0x80009044);
			ResetControlRegister = OS_ReadPortUChar(PCI_CDATA);

			/*  Force bit 0 to 0 to avoid a reset of the whole system . */

			ResetControlRegister &= 0xFE;

			/* Reset the AC-Link (Set the AC97 Soft Reset bit to 1 - HIGH) */

			ResetControlRegister |= 0x80;
			OS_WritePortULong(PCI_CADDR, 0x80009044);
			OS_WritePortUChar(PCI_CDATA, ResetControlRegister);

			/*  Hold it HIGH for 100ms */

			OS_Sleep(100);

			ResetControlRegister &= 0x7F;
			OS_WritePortULong(PCI_CADDR, 0x80009044);
			OS_WritePortUChar(PCI_CDATA, ResetControlRegister);
		}

		/* RESET Codec */

		*((unsigned long *) (pGeode->F3BAR0 + CODEC_CMD_REG)) = 0L;
	}

	/*  Check which codec is being used */

	if (DURAUDIO_CodecRead(pGeode, AD1819A_VENDORID1) == 0x4144 &&
	    DURAUDIO_CodecRead(pGeode, AD1819A_VENDORID2) == 0x5303) {
		pGeode->fAD1819A = TRUE;
		/* Enable non-48kHz sample rates. */
		DURAUDIO_CodecWrite(pGeode, AD1819A_SER_CONF,
				    (unsigned
				     short) (DURAUDIO_CodecRead(pGeode,
								AD1819A_SER_CONF
								>> 8) |
					     AD1819A_SER_CONF_DRQEN));
	} else {
		pGeode->fAD1819A = FALSE;
		/* set the VRA bit to ON */
		DURAUDIO_CodecWrite(pGeode, EXT_AUDIO_CTRL_STAT,
				    (unsigned
				     short) (DURAUDIO_CodecRead(pGeode,
								EXT_AUDIO_CTRL_STAT)
					     | 0x0001));
	}

	DURAUDIO_CodecWrite(pGeode, MASTER_VOLUME, 0x0a0a);
	DURAUDIO_CodecWrite(pGeode, PCM_OUT_VOL, 0x0000);
	DURAUDIO_CodecWrite(pGeode, PC_BEEP_VOLUME, 0x0000);
	DURAUDIO_CodecWrite(pGeode, PHONE_VOLUME, 0x8000);
	DURAUDIO_CodecWrite(pGeode, MIC_VOLUME, 0x8048);
	DURAUDIO_CodecWrite(pGeode, LINE_IN_VOLUME, 0x0808);
	DURAUDIO_CodecWrite(pGeode, CD_VOLUME, 0x8000);
	DURAUDIO_CodecWrite(pGeode, VIDEO_VOLUME, 0x8000);
	DURAUDIO_CodecWrite(pGeode, TV_VOLUME, 0x8000);
	DURAUDIO_CodecWrite(pGeode, RECORD_SELECT, 0x0000);
	DURAUDIO_CodecWrite(pGeode, RECORD_GAIN, 0x0a0a);
	DURAUDIO_CodecWrite(pGeode, GENERAL_PURPOSE, 0x0200);
	DURAUDIO_CodecWrite(pGeode, MASTER_VOLUME_MONO, 0x0000);

	/*  Set all the power state bits to 0 (Reg 26h) */

	DURAUDIO_CodecWrite(pGeode, POWERDOWN_CTRL_STAT, 0x0000);

	pGeode->CurrentPowerState = DURAUDIO_D0;

	return TRUE;
}

void
DURAUDIO_SetPRDAddress(PDURAUDIO pGeode, unsigned long Channel,
		       unsigned long PRDPhys, unsigned long PRDVirt)
{
	pGeode->AudioChannel[Channel].PRD_AllocInfo.PhysicalAddress = PRDPhys;
	pGeode->AudioChannel[Channel].PRDTable = (PPRD_ENTRY) PRDVirt;

	/*  Sets the PRD physical address */

	DURAUDIO_SetDMARegs(pGeode, Channel);
}

void
DURAUDIO_ResetPRDReg(PDURAUDIO pGeode, unsigned long Channel,
		     unsigned long Location)
{
	OS_WritePortULong((unsigned short) (pGeode->AudioChannel[Channel].
					    AudioBusMaster.PRDTableAddress),
			  (unsigned long) Location);
}

unsigned char
DURAUDIO_IsEnabled(PDURAUDIO pGeode, unsigned long Channel)
{
	return (OS_ReadPortUChar
		((unsigned short) (pGeode->AudioChannel[Channel].
				   AudioBusMaster.
				   CommandRegister)) & ENABLE_BUSMASTER);
}

unsigned char
DURAUDIO_CheckBusMaster(PDURAUDIO pGeode, unsigned long Channel)
{
	unsigned char CommandRegStatus;

	if (pGeode->fIOAccess) {
		CommandRegStatus =
		    OS_ReadPortUChar((unsigned short) pGeode->
				     AudioChannel[Channel].AudioBusMaster.
				     CommandRegister);
	} else {
		CommandRegStatus =
		    *((unsigned char *) pGeode->AudioChannel[Channel].
		      AudioBusMaster.CommandRegister);
	}

	return CommandRegStatus;
}

void
DURAUDIO_SetDMARegs(PDURAUDIO pGeode, unsigned long Channel)
{
	if (pGeode->fIOAccess) {

		/* Points to PRD array's physical address. */

		OS_WritePortULong((unsigned short) pGeode->
				  AudioChannel[Channel].AudioBusMaster.
				  PRDTableAddress,
				  pGeode->AudioChannel[Channel].PRD_AllocInfo.
				  PhysicalAddress);
	} else {

		/*  Points to PRD array's physical address. */

		*((unsigned long *) pGeode->AudioChannel[Channel].
		  AudioBusMaster.PRDTableAddress) =
pGeode->AudioChannel[Channel].PRD_AllocInfo.PhysicalAddress;
	}
}

/*----------------------------------------------------------------------------
 void DURAUDIO_StartDMA (PDURAUDIO pGeode, unsigned long Channel)

 Starts the DMA for the channel specified by Channel.

 ---------------------------------------------------------------------------*/
void
DURAUDIO_StartDMA(PDURAUDIO pGeode, unsigned long Channel)
{
	if (pGeode->AudioChannel[Channel].Running)
		return;

	DURAUDIO_SetDMARegs(pGeode, Channel);

	DURAUDIO_ResumeDMA(pGeode, Channel);
	pGeode->AudioChannel[Channel].Running = TRUE;
}

/*----------------------------------------------------------------------------

 void DURAUDIO_StopDMA (PDURAUDIO pGeode, unsigned long Channel)

 Stops the DMA channel controlled by the specified PRD table.

 -----------------------------------------------------------------------------*/
void
DURAUDIO_StopDMA(PDURAUDIO pGeode, unsigned long Channel)
{
	unsigned int i;

	/*  If DMA is not running, don't do anything and return. */

	if (!pGeode->AudioChannel[Channel].Running)
		return;

	if (pGeode->fCS5535) {
		if (pGeode->fIOAccess) {

			/*  Turn OFF DMA */

			OS_WritePortUChar((unsigned short) pGeode->
					  AudioChannel[Channel].AudioBusMaster.
					  CommandRegister, STOP_BUSMASTER);
		} else {

			/*  Turn OFF DMA  */

			*((unsigned char *) pGeode->AudioChannel[Channel].
			  AudioBusMaster.CommandRegister) = STOP_BUSMASTER;
		}
	} else {

		/* We cannot simply set the Start/Stop bit in the Command register to 0
		   So, we need to set all the PRD the flags with EOTs and wait for the 
		   BusMaster to stop
		 */

		if (pGeode->AudioChannel[Channel].PRDTable) {
			for (i = 0;
			     i <
			     (pGeode->AudioChannel[Channel].
			      PRDEntriesAllocated - 1); ++i) {
				pGeode->AudioChannel[Channel].PRDTable[i].
				    SizeFlags |= PRD_EOT_BIT;
			}

			/*Wait for 6ms to ensure DMA is stopped before setting the 
			   Start/Stop bit 6ms should be enough to wait for 3 PRD entries 
			   at 8kHz (The lowest standard sample rate)
			 */
			OS_Sleep(6);

			/*  Turn OFF DMA */

			*((unsigned char *) pGeode->AudioChannel[Channel].
			  AudioBusMaster.CommandRegister) = STOP_BUSMASTER;
		}
	}

	pGeode->AudioChannel[Channel].Running = FALSE;
}

/*----------------------------------------------------------------------------

void DURAUDIO_PauseDMA (PDURAUDIO pGeode, unsigned long Channel)

	Pauses the DMA channel controlled by the specified PRD table.
(Only available on 5535).

----------------------------------------------------------------------------*/
void
DURAUDIO_PauseDMA(PDURAUDIO pGeode, unsigned long Channel)
{
	unsigned char BMStatus;

	BMStatus = DURAUDIO_CheckBusMaster(pGeode, Channel);

	if ((BMStatus & BUSMASTER_MASK) == ENABLE_BUSMASTER) {
		if (pGeode->fIOAccess) {
			OS_WritePortUChar((unsigned short) pGeode->
					  AudioChannel[Channel].AudioBusMaster.
					  CommandRegister,
					  (unsigned char) (pGeode->
							   AudioChannel
							   [Channel].
							   AudioBusMaster.
							   DirectionBit |
							   PAUSE_BUSMASTER));
		} else {
			*((unsigned char *) pGeode->AudioChannel[Channel].
			  AudioBusMaster.CommandRegister) =
pGeode->AudioChannel[Channel].AudioBusMaster.DirectionBit | PAUSE_BUSMASTER;
		}
	}
}

/*----------------------------------------------------------------------------

 void DURAUDIO_ResumeDMA (PDURAUDIO pGeode, unsigned long Channel)

 Resumes the DMA channel controlled by the specified PRD table
 (Only available on 5535).

----------------------------------------------------------------------------*/
void
DURAUDIO_ResumeDMA(PDURAUDIO pGeode, unsigned long Channel)
{
	if (pGeode->fIOAccess) {
		OS_WritePortUChar((unsigned short) pGeode->
				  AudioChannel[Channel].AudioBusMaster.
				  CommandRegister,
				  (unsigned
				   char) (((pGeode->AudioChannel[Channel].
					    AudioBusMaster.
					    DirectionBit) & 0xFC) |
					  ENABLE_BUSMASTER));
	} else {
		*((unsigned char *) pGeode->AudioChannel[Channel].
		  AudioBusMaster.CommandRegister) =
pGeode->AudioChannel[Channel].AudioBusMaster.DirectionBit | ENABLE_BUSMASTER;
	}
}

/*----------------------------------------------------------------------------

 void DURAUDIO_CodecWrite( PDURAUDIO pGeode, unsigned short CodecRegister, unsigned short CodecData  )

 Writes data to the CODEC

----------------------------------------------------------------------------*/
void
DURAUDIO_CodecWrite(PDURAUDIO pGeode, unsigned char CodecRegister,
		    unsigned short CodecData)
{
	int i;
	unsigned long CodecRegister_data = 0;
	unsigned long val = 0, Temp = 0, timeout;

	CodecRegister_data = ((unsigned long) CodecRegister) << 24;
	CodecRegister_data |= (unsigned long) CodecData;
	CodecRegister_data &= CODEC_COMMAND_MASK;

	if (pGeode->fCS5535) {

		/* Set the bit.  We are going to access the CODEC... */

		CodecRegister_data |= BIT_5535_CODEC_COMMAND_NEW;

		/* Write the data */

		OS_WritePortULong((unsigned short) (pGeode->F3BAR0 +
						    CODEC_CONTROL_REG_5535),
				  CodecRegister_data);

		/*  We need to wait for bit16 of the Codec control register to clear */

		Temp =
		    OS_ReadPortULong((unsigned short) (pGeode->F3BAR0 +
						       CODEC_CONTROL_REG_5535));
		timeout = 50;

		while ((Temp & BIT_5535_CODEC_COMMAND_NEW) && timeout--) {
			OS_Sleep(10);
			Temp =
			    OS_ReadPortULong((unsigned short) (pGeode->F3BAR0 +
							       CODEC_CONTROL_REG_5535));
		}

		if (!timeout) {
			OS_DbgMsg
			    ("Could not Write the CODEC!! BIT_5535_CODEC_COMMAND_NEW did not clear!\n");
		}
	} else {
		DURAUDIO_WaitFrameAndWrite(pGeode, CodecRegister_data);

		/*
		   Wait for Status Tag and Status Valid bits in the CODEC Status 
		   Register 
		 */

		for (i = 0; i <= 30000; i++) {
			if (pGeode->fIOAccess) {
				val =
				    OS_ReadPortULong((unsigned short) (pGeode->
								       F3BAR0 +
								       CODEC_STATUS_REG));
			} else {
				val =
				    *((unsigned long *) (pGeode->F3BAR0 +
							 CODEC_STATUS_REG));
			}

			if ((val & CODEC_STATUS_VALID)
			    || (val & CODEC_STATUS_NEW))
				break;
		}

		/*  This prevents multiple volume writes to a codec register  */
		DURAUDIO_WaitFrameAndWrite(pGeode,
					   0x80000000 | CodecRegister_data);
	}
}

unsigned short
DURAUDIO_CodecRead(PDURAUDIO pGeode, unsigned char CodecRegister)
{
	int i;
	unsigned long CodecRegister_data = 0;
	unsigned long timeout = 10;
	volatile unsigned long val = 0;

	CodecRegister_data = ((unsigned long) CodecRegister) << 24;
	CodecRegister_data |= 0x80000000;	/* High-bit set (p.106) is a CODEC reg READ */

	if (pGeode->fCS5535) {

		/*  Set the bit.  We are going to access the CODEC... */

		CodecRegister_data |= BIT_5535_CODEC_COMMAND_NEW;

		/*  Request the data */

		OS_WritePortULong((unsigned short) (pGeode->F3BAR0 +
						    CODEC_CONTROL_REG_5535),
				  CodecRegister_data);

		/* Now we need to wait for BIT_5535_CODEC_COMMAND_NEW of the Codec 
		   control register to clear  (For subsequent Reads/Writes)
		 */
		if (!DURAUDIO_WaitForBit
		    (pGeode, CODEC_CONTROL_REG_5535, BIT_5535_CODEC_COMMAND_NEW,
		     CLEAR, 500, NULL)) {
			OS_DbgMsg
			    ("BIT_5535_CODEC_COMMAND_NEW did not clear!!\n");
		}

		/* Wait for CODEC_STATUS_NEW and confirm the read of the requested
		   register */

		timeout = 50;
		do {
			if (pGeode->fIOAccess)
				val =
				    OS_ReadPortULong((unsigned short) (pGeode->
								       F3BAR0 +
								       CODEC_STATUS_REG_5535));
			else
				val =
				    *((unsigned long *) (pGeode->F3BAR0 +
							 CODEC_STATUS_REG_5535));

			if ((val & BIT_5535_CODEC_STATUS_NEW)
			    && ((unsigned long) CodecRegister ==
				((0xFF000000 & val) >> 24))) {
				break;
			} else {

				/* Wait for 10 miliseconds and try again */

				OS_Sleep(10);
			}
		} while (--timeout);

		if (!timeout) {
			OS_DbgMsg
			    ("Could not read the CODEC!!  Returning what we got.\n");
		}

	} else {
		do {
			DURAUDIO_WaitFrameAndWrite(pGeode, CodecRegister_data);

			/*  Wait for Status Tag and Status Valid bits in the CODEC 
			   Status Register */

			for (i = 0; i <= 3000; i++) {
				if (pGeode->fIOAccess) {
					val =
					    OS_ReadPortULong((unsigned
							      short) (pGeode->
								      F3BAR0 +
								      CODEC_STATUS_REG));
				} else {
					val =
					    *((unsigned long *) (pGeode->
								 F3BAR0 +
								 CODEC_STATUS_REG));
				}
			}
		} while ((((unsigned long) (0xFF & CodecRegister)) !=
			  ((0xFF000000 & val) >> 24)) && (--timeout));
		/* Check if the register read is the one we want */
	}

	return ((unsigned short) val);
}

unsigned char
DURAUDIO_WaitForBit(PDURAUDIO pGeode,
		    unsigned long Offset,
		    unsigned long Bit,
		    unsigned char Operation,
		    unsigned long timeout, unsigned long *pReturnValue)
{
	volatile unsigned long Temp;

	if (pGeode->fIOAccess)
		Temp =
		    OS_ReadPortULong((unsigned short) (pGeode->F3BAR0 +
						       Offset));
	else
		Temp = *((unsigned long *) (pGeode->F3BAR0 + Offset));

	while (timeout) {
		if (Operation == CLEAR) {
			if (!(Temp & Bit))
				break;
		} else {
			if (Temp & Bit)
				break;
		}

		/*  If the Bit is not clear yet, we wait for 1 milisecond and try 
		   again */

		OS_Sleep(1);

		if (pGeode->fIOAccess)
			Temp =
			    OS_ReadPortULong((unsigned short) (pGeode->F3BAR0 +
							       Offset));
		else
			Temp = *((unsigned long *) (pGeode->F3BAR0 + Offset));

		timeout--;
	};

	if (pReturnValue)
		*pReturnValue = Temp;

	if (!timeout) {
		return FALSE;
	}

	return TRUE;
}

/*----------------------------------------------------------------------------

	void DURAUDIO_WaitFrameAndWrite(unsigned long val)

	Waits for next avail pos in the CODEC's serial frame and sends the data.

	While the CODEC_CMD_VALID bit is set, it should not be overwritten.
	i.e., the codec cycles around to ship the command out at the next 
	avail pos in the serial frame.

----------------------------------------------------------------------------*/

void
DURAUDIO_WaitFrameAndWrite(PDURAUDIO pGeode, unsigned long val)
{
	unsigned long cmd_val, timeout;

	timeout = 30000;

	if (pGeode->fIOAccess) {
		if (pGeode->fCS5535) {
			while ((OS_ReadPortULong
				((unsigned short) (pGeode->F3BAR0 +
						   CODEC_CMD_REG)) &
				BIT_5535_CODEC_COMMAND_NEW) && (--timeout)) ;
		} else {
			while ((OS_ReadPortULong
				((unsigned short) (pGeode->F3BAR0 +
						   CODEC_CMD_REG)) &
				CODEC_CMD_VALID)
			       && (--timeout)) ;
		}

		cmd_val = val & CODEC_COMMAND_MASK;
		OS_WritePortULong((unsigned short) (pGeode->F3BAR0 +
						    CODEC_CMD_REG), cmd_val);
	} else {
		while ((*((unsigned long *) (pGeode->F3BAR0 + CODEC_CMD_REG)) &
			CODEC_CMD_VALID) && (--timeout)) ;

		cmd_val = val & CODEC_COMMAND_MASK;
		*((unsigned long *) (pGeode->F3BAR0 + CODEC_CMD_REG)) = cmd_val;
	}
}

/*----------------------------------------------------------------------------
void DURAUDIO_SetCodecRate(PDURAUDIO pGeode, unsigned long Channel, 
unsigned long Rate)

	Sets the CODEC Sample Rate
----------------------------------------------------------------------------*/
void
DURAUDIO_SetCodecRate(PDURAUDIO pGeode, unsigned long Channel,
		      unsigned long SampleRate)
{
	unsigned short val;

	OS_DbgMsg("Rate: %d\n", SampleRate);

	pGeode->AudioChannel[Channel].SampleRate = SampleRate;

	/*      If Double-Rate is supported (Bit 2 on register 28h)...  */

	val = DURAUDIO_CodecRead(pGeode, EXTENDED_AUDIO_ID);

	if (val & 0x02) {
		OS_DbgMsg("Codec supports Double rate.\n");
		val = DURAUDIO_CodecRead(pGeode, EXT_AUDIO_CTRL_STAT);

		if (SampleRate > 48000) {
			DURAUDIO_CodecWrite(pGeode, EXT_AUDIO_CTRL_STAT,
					    (unsigned short) (val | 0x0002));
			SampleRate /= 2;
		} else
			DURAUDIO_CodecWrite(pGeode, EXT_AUDIO_CTRL_STAT,
					    (unsigned short) (val & 0xFFFD));
	}

	if (pGeode->fAD1819A) {
		OS_DbgMsg("AD1819...\n");
		if (Channel)
			DURAUDIO_CodecWrite(pGeode, AD1819A_PCM_SR1,
					    (unsigned short) SampleRate);
		else
			DURAUDIO_CodecWrite(pGeode, AD1819A_PCM_SR0,
					    (unsigned short) SampleRate);
	} else {
		if (Channel)
			DURAUDIO_CodecWrite(pGeode, PCM_LR_ADC_RATE,
					    (unsigned short) SampleRate);
		else
			DURAUDIO_CodecWrite(pGeode, PCM_FRONT_DAC_RATE,
					    (unsigned short) SampleRate);
	}
}

/*----------------------------------------------------------------------------
unsigned long DURAUDIO_SetPowerState(PDURAUDIO pGeode, 
DURAUDIO_POWER_STATE NewPowerState)

	Sets the power state of the Codec

----------------------------------------------------------------------------*/
unsigned long
DURAUDIO_SetPowerState(PDURAUDIO pGeode, DURAUDIO_POWER_STATE NewPowerState)
{
	if (NewPowerState == pGeode->CurrentPowerState)
		return DURAUDIO_SUCCESS;

	switch (NewPowerState) {
	case DURAUDIO_D0:
		{
			switch (pGeode->CurrentPowerState) {
			case DURAUDIO_D0:
				break;

			case DURAUDIO_D1:
				{
					OS_DbgMsg("Coming back from D1.\n");

					if (pGeode->fCS5535) {

						/*      We are coming back from D1 on a 5535
						   so, we need to do a AC-Link Warm Reset */

						OS_WritePortULong((unsigned
								   short)
								  (pGeode->
								   F3BAR0 +
								   CODEC_CONTROL_REG_5535),
								  BIT_5535_ACLINK_WARM_RESET);
						if (!DURAUDIO_WaitForBit
						    (pGeode,
						     CODEC_STATUS_REG_5535,
						     BIT_CODEC_READY, SET, 400,
						     NULL)) {
							OS_DbgMsg
							    ("Primary Codec NOT Ready...Aborting\n");

							return DURAUDIO_ERROR;
						}

						/*      Reset the codec */

						DURAUDIO_ResetCodec(pGeode);

						OS_Sleep(50);
					}

					if (!DURAUDIO_CodecFullOn(pGeode)) {
						OS_DbgMsg
						    ("ERROR: CODEC did not power up!\n");

						return DURAUDIO_ERROR;
					}

					break;
				}

			case DURAUDIO_D2:
			case DURAUDIO_D3:
				{
					OS_DbgMsg("Coming back from D2/D3.\n");

					if (pGeode->fCS5535) {

						/*
						   We are coming back from D2 or D3 on a 5535
						   so, we need to do a AC-Link Warm Reset
						 */
						OS_WritePortULong((unsigned
								   short)
								  (pGeode->
								   F3BAR0 +
								   CODEC_CONTROL_REG_5535),
								  BIT_5535_ACLINK_WARM_RESET);
						if (!DURAUDIO_WaitForBit
						    (pGeode,
						     CODEC_STATUS_REG_5535,
						     BIT_CODEC_READY, SET, 400,
						     NULL)) {
							OS_DbgMsg
							    ("Primary Codec NOT Ready...Aborting\n");
							return DURAUDIO_ERROR;
						}

						/*      Reset the codec */

						DURAUDIO_ResetCodec(pGeode);

						OS_Sleep(50);
					}

					if (!DURAUDIO_CodecFullOn(pGeode)) {
						OS_DbgMsg
						    ("ERROR: CODEC did not power up!\n");
						return DURAUDIO_ERROR;
					}
					/*
					   If we are coming from D2, D3 or D4, we need to 
					   restore the controller and codec data.
					 */
					DURAUDIO_RestoreAudioContext(pGeode);

					break;
				}

			case DURAUDIO_D4:
				{
					OS_DbgMsg("Coming back from D4.\n");

					/*
					   If we are coming from D4 we have to initialize 
					   everything again because we would be coming back 
					   from hibernation or total shutdown
					 */

							/*-------------------------------------------------
										Interrupt-Related code
							--------------------------------------------------*/

					/*      Free virtual memory before allocation.  */

					OS_Free_VirtualAddress((unsigned long)
							       pGeode->
							       pInterruptID);

					OS_Free_F3BAR0_Virt(pGeode->F3BAR0);

					DURAUDIO_SetupIRQ(pGeode,
							  pGeode->SelectedIRQ);

					DURAUDIO_InitAudioRegs(pGeode);

							/*------------------------------------------------
										Interrupt-Related code
							-------------------------------------------------*/
					DURAUDIO_RestoreAudioContext(pGeode);

					break;
				}
			}

			pGeode->CurrentPowerState = DURAUDIO_D0;
			break;
		}

	case DURAUDIO_D1:
		{
			OS_DbgMsg("In D1.\n");

			if ((pGeode->AudioChannel[CHANNEL0_PLAYBACK].Running ==
			     FALSE)
			    && (pGeode->AudioChannel[CHANNEL1_RECORD].Running ==
				FALSE)) {
				if (pGeode->fCS5535) {
					/*      Powerdown ADC   */
					DURAUDIO_CodecWrite(pGeode,
							    POWERDOWN_CTRL_STAT,
							    DURAUDIO_PWR_PR0);
					if (!DURAUDIO_CheckCodecPowerBit
					    (pGeode, DURAUDIO_CODEC_POWER_ADC,
					     CLEAR)) {
						OS_DbgMsg
						    ("ERROR: CODEC ADC bit not cleared!\n");
						return DURAUDIO_ERROR;
					}
					/*      Powerdown DAC   */
					DURAUDIO_CodecWrite(pGeode,
							    POWERDOWN_CTRL_STAT,
							    DURAUDIO_PWR_DIGOFF);
					if (!DURAUDIO_CheckCodecPowerBit
					    (pGeode, DURAUDIO_CODEC_POWER_DAC,
					     CLEAR)) {
						OS_DbgMsg
						    ("ERROR: CODEC DAC bit not cleared!\n");
						return DURAUDIO_ERROR;
					}
					/*      Powerdown Analog        */
					DURAUDIO_CodecWrite(pGeode,
							    POWERDOWN_CTRL_STAT,
							    DURAUDIO_PWR_DIGOFF
							    |
							    DURAUDIO_PWR_ANLOFF);

					/*      5535 allows to powerdown AC-link        */

					/*
					   PLUS Powerdown AC_Link 
					   Powerdown Amp. causes unusual "pop" sound. This issue
					   will be further investigated in Castle. Currently we 
					   don't power down amplifier in D1. */
					DURAUDIO_CodecWrite(pGeode,
							    POWERDOWN_CTRL_STAT,
							    DURAUDIO_PWR_D1_HAWK);

					/*      Set the AC-Link ShutDown bit    */

					OS_WritePortULong((unsigned
							   short) (pGeode->
								   F3BAR0 +
								   CODEC_CONTROL_REG_5535),
							  BIT_5535_ACLINK_SHUTDOWN);
				} else {
					/*Power down ADC/DAC and ANL sections in GX1/SCxx00
					   Amp. powerdown causes "pop" sound
					   ADC/DAC powerdown only causes loud static sound, but no 
					   static sound is heard when ANL and ADC/DAC power down 
					   simultaneously */
					DURAUDIO_CodecWrite(pGeode,
							    POWERDOWN_CTRL_STAT,
							    DURAUDIO_PWR_DIGOFF
							    |
							    DURAUDIO_PWR_ANLOFF);
				}

				/*      We went to D1   */

				pGeode->CurrentPowerState = DURAUDIO_D1;
			} else {

				/*      If we are playing, we don't do anything and stay in D0 */

				pGeode->CurrentPowerState = DURAUDIO_D0;
			}

			break;
		}

	case DURAUDIO_D2:
	case DURAUDIO_D3:
	case DURAUDIO_D4:
		{
			OS_DbgMsg("In D2/D3/D4.\n");

			/*      
			   We are going into D2, D3 or D4.
			   In this mode the power could be removed or reduced to a point
			   where the AC97 controller looses information, 
			   so we must save the codec registers.
			 */
			DURAUDIO_SaveAudioContext(pGeode);

			/*      Powerdown CODEC */

			DURAUDIO_CodecWrite(pGeode, POWERDOWN_CTRL_STAT,
					    DURAUDIO_PWR_D4);

			if (pGeode->fCS5535) {

				/*      Set the AC-Link ShutDown bit    */

				OS_WritePortULong((unsigned short) (pGeode->
								    F3BAR0 +
								    CODEC_CONTROL_REG_5535),
						  BIT_5535_ACLINK_SHUTDOWN);
			}

			pGeode->CurrentPowerState = NewPowerState;
			break;
		}
	}

	return DURAUDIO_SUCCESS;
}

void
DURAUDIO_ResetCodec(PDURAUDIO pGeode)
{

	/*      Reset codec     */

	if (pGeode->fCS5535) {
		OS_WritePortULong((unsigned short) (pGeode->F3BAR0 +
						    CODEC_CONTROL_REG_5535),
				  0L);
	} else {
		*((unsigned long *) (pGeode->F3BAR0 + CODEC_CMD_REG)) = 0L;
	}
}

unsigned char
DURAUDIO_CheckCodecPowerBit(PDURAUDIO pGeode,
			    unsigned short Bit, unsigned char Operation)
{
	unsigned short CodecData, CodecTimeOut;

	CodecTimeOut = 10;
	do {

		/*      Read the power management register.     */

		CodecData = DURAUDIO_CodecRead(pGeode, POWERDOWN_CTRL_STAT);

		/*      Check the REF status. Should be ready.  */

		if (Operation == CLEAR) {
			if (!(CodecData & Bit))
				break;
			else
				CodecTimeOut--;
		} else {
			if (CodecData & Bit)
				break;
			else
				CodecTimeOut--;
		}

		/*      Let's wait a little, 10ms and then try again.   */

		OS_Sleep(10);
	} while (CodecTimeOut);

	/*      Check if we timed out.  */

	if (CodecTimeOut == 0) {
		return FALSE;
	}

	return TRUE;
}

unsigned char
DURAUDIO_CodecFullOn(PDURAUDIO pGeode)
{
	DURAUDIO_CodecRead(pGeode, POWERDOWN_CTRL_STAT);

	/*      Clear EAPD,PR6 and AC-link to power up external and HP amp and 
	   Digital interface    */
	DURAUDIO_CodecWrite(pGeode, POWERDOWN_CTRL_STAT, DURAUDIO_PWRUP_STEP1);

	/*      Clear PR3 to power up Analog (Vref off) */
	DURAUDIO_CodecWrite(pGeode, POWERDOWN_CTRL_STAT, DURAUDIO_PWRUP_STEP2);

	if (!DURAUDIO_CheckCodecPowerBit(pGeode, DURAUDIO_CODEC_POWER_REF, SET)) {
		OS_DbgMsg("REF timed out. CoDec not powered up.\n");
		return FALSE;
	}
	/*
	   A loud "pop" sound is heard without sufficient delay.
	   It means the REF ready bit being set doesn't reflect 
	   the real reference voltage status when this bit is being checked.
	   It is codec issue. Adding approximate 1 second delay is
	   the current workaround.
	 */
	OS_Sleep(1200);

	/*      Clear PR2 to power up Analog (Vref on)  */
	DURAUDIO_CodecWrite(pGeode, POWERDOWN_CTRL_STAT, DURAUDIO_PWRUP_STEP3);

	if (!DURAUDIO_CheckCodecPowerBit(pGeode, DURAUDIO_CODEC_POWER_ANL, SET)) {
		OS_DbgMsg("ANL timed out. CoDec not powered up.\n");
		return FALSE;
	}
	/*      Clear PR1 to power up DAC       */
	DURAUDIO_CodecWrite(pGeode, POWERDOWN_CTRL_STAT, DURAUDIO_PWRUP_STEP4);

	if (!DURAUDIO_CheckCodecPowerBit(pGeode, DURAUDIO_CODEC_POWER_DAC, SET)) {
		OS_DbgMsg("DAC timed out. CoDec not powered up.\n");
		return FALSE;
	}
	/*      Clear PR0 to power up ADC       */
	DURAUDIO_CodecWrite(pGeode, POWERDOWN_CTRL_STAT, DURAUDIO_PWRUP_STEP5);

	if (!DURAUDIO_CheckCodecPowerBit(pGeode, DURAUDIO_CODEC_POWER_ADC, SET)) {
		OS_DbgMsg("ADC timed out. CoDec not powered up.\n");
		return FALSE;
	}

	return TRUE;
}

void
DURAUDIO_SaveAudioContext(PDURAUDIO pGeode)
{
	unsigned char i, RegIndex = 0;
	unsigned long Channel;

	/*
	   Check if DMA is running for each channel.
	   If it is, save PRD table address and turn it OFF
	 */
	for (Channel = 0; Channel < MAX_CHANNELS; Channel++) {
		if (pGeode->AudioChannel[Channel].Running == TRUE) {

			/*      Save Current PRD address        */

			if (pGeode->fCS5535) {
				pGeode->
				    AudioBusMaster_PRDTableAddress[Channel] =
				    OS_ReadPortULong((unsigned short) (pGeode->
								       F3BAR0 +
								       0x24 +
								       (Channel
									*
									0x08)));

				/*      Stop DMA        */

				DURAUDIO_StopDMA(pGeode, Channel);
			} else {
				pGeode->
				    AudioBusMaster_PRDTableAddress[Channel] =
				    *((unsigned long *) (pGeode->F3BAR0 + 0x24 +
							 (Channel * 0x08)));

				/*      Stop DMA        */

				DURAUDIO_StopDMA(pGeode, Channel);
			}

			pGeode->AudioChannel[Channel].Running = TRUE;
		}
	}

	/*      Save Mixer volumes and settings */

	for (i = 0x02; i <= 0x1E; i += 2) {
		pGeode->CODECRegisters[RegIndex++] =
		    DURAUDIO_CodecRead(pGeode, i);
	}

	/*      Save Extended registers   (DAC/ADC rates etc...)        */

	for (i = 0x28; i <= 0x38; i += 2) {
		pGeode->CODECRegisters[RegIndex++] =
		    DURAUDIO_CodecRead(pGeode, i);
	}

	/*      Save F3BAR0 registers           */
	if (pGeode->fCS5535) {
		int i;

		for (i = 0x00; i < 0x7F; ++i) {
			unsigned long lTemp;
			unsigned short sTemp;

			if ((i < 0x0F) || ((i & 0x0F) == 0x04)
			    || ((i & 0x0F) == 0x0C)
			    || (i >= 0x5C)) {

				/*      Save 32-bit registers   */
				lTemp =
				    OS_ReadPortULong((unsigned short) (pGeode->
								       F3BAR0 +
								       i));

				pGeode->F3BARSave[i] =
				    (unsigned char) (lTemp & 0x000000FF);
				pGeode->F3BARSave[i + 1] =
				    (unsigned char) ((lTemp & 0x0000FF00) >> 8);
				pGeode->F3BARSave[i + 2] =
				    (unsigned char) ((lTemp & 0x00FF0000) >>
						     16);
				pGeode->F3BARSave[i + 3] =
				    (unsigned char) ((lTemp & 0xFF000000) >>
						     24);

				OS_DbgMsg("Saved: Offset [%02Xh]: %08X\n", i,
					  lTemp);
				i += 3;

			} else {
				if (i == 0x12) {
					/*      Save 16bit registers    */
					sTemp =
					    OS_ReadPortUShort((unsigned
							       short) (pGeode->
								       F3BAR0 +
								       i));

					pGeode->F3BARSave[i] =
					    (unsigned char) (sTemp & 0x00FF);
					pGeode->F3BARSave[i + 1] =
					    (unsigned char) ((sTemp & 0xFF00) >>
							     8);

					OS_DbgMsg
					    ("Saved: Offset [%02Xh]: %04X\n", i,
					     sTemp);
					i += 1;
				} else {

					/*      Save 8bit registers     */
					pGeode->F3BARSave[i] =
					    OS_ReadPortUChar((unsigned
							      short) (pGeode->
								      F3BAR0 +
								      i));
					/*
					   Make sure the bus master is disabled. We found the keyclick
					   activities were coming in the middle of this saving function.
					   So, the DMA was running again after checking the running flag
					   by disabling the bus master. The Bus Master register was saved
					   as it was (running). It caused the issue of keyclicks repeated
					   rapidly after resume due to restoring the bus master register
					   in the running state. Zero out the enable bits in the bus master
					   register here solved the keyclick issue in S2R. 
					 */
					if (((i & 0x0F) == 0x00)
					    || ((i & 0x0F) == 0x08)) {
						pGeode->F3BARSave[i] &= 0xFC;
					}
					OS_DbgMsg
					    ("Saved: Offset [%02Xh]: %02X\n", i,
					     pGeode->F3BARSave[i]);
				}
			}
		}
	} else {
		memcpy(pGeode->F3BARSave, (unsigned long *) pGeode->F3BAR0,
		       0x50);
		pGeode->F3BARSave[20] &= 0xFE;	/*      Disable Bus Master again        */
	}
}

void
DURAUDIO_RestoreAudioContext(PDURAUDIO pGeode)
{
	unsigned char i, RegIndex = 0;
	unsigned long Channel;

	OS_DbgMsg("Restoring Context.\n");

	/*      Restore F3BAR0 registers        */

	if (pGeode->fCS5535) {
		int i;

		for (i = 0x00; i < 0x7F; ++i) {
			unsigned long ulTemp, ulTemp1, ulTemp2, ulTemp3,
			    ulTemp4;
			unsigned short usTemp;

			if ((i < 0x0F) || ((i & 0x0F) == 0x04)
			    || ((i & 0x0F) == 0x0C)
			    || (i >= 0x5C)) {

				/*      Restore 32-bit registers        */

				ulTemp1 =
				    ((unsigned long) pGeode->F3BARSave[i]);
				ulTemp2 =
				    (((unsigned long) pGeode->
				      F3BARSave[i + 1]) << 8);
				ulTemp3 =
				    (((unsigned long) pGeode->
				      F3BARSave[i + 2]) << 16);
				ulTemp4 =
				    (((unsigned long) pGeode->
				      F3BARSave[i + 3]) << 24);

				ulTemp = ulTemp1 | ulTemp2 | ulTemp3 | ulTemp4;

				OS_DbgMsg("Restored: Offset [%02Xh]: %08X\n", i,
					  ulTemp);

				OS_WritePortULong((unsigned short) (pGeode->
								    F3BAR0 + i),
						  ulTemp);
				i += 3;
			} else {
				if (i == 0x12) {

					/*      Restore 16bit registers */

					usTemp =
					    ((unsigned short) (pGeode->
							       F3BARSave[i])
					     || (unsigned short) (pGeode->
								  F3BARSave[i +
									    1]
								  << 8));

					OS_DbgMsg
					    ("Restored: Offset [%02Xh]: %04X\n",
					     i, usTemp);

					OS_WritePortUShort((unsigned
							    short) (pGeode->
								    F3BAR0 + i),
							   usTemp);
					i += 1;
				} else {

					/*      Save 8bit registers     */

					OS_DbgMsg
					    ("Restored: Offset [%02Xh]: %02X\n",
					     i, pGeode->F3BARSave[i]);

					OS_WritePortUChar((unsigned
							   short) (pGeode->
								   F3BAR0 + i),
							  pGeode->F3BARSave[i]);
				}
			}
		}
	} else
		memcpy((unsigned long *) pGeode->F3BAR0, pGeode->F3BARSave,
		       0x50);

	/*      Restore Mixer volumes and settings      */

	for (i = 0x02; i <= 0x1E; i += 2) {
		DURAUDIO_CodecWrite(pGeode, i,
				    pGeode->CODECRegisters[RegIndex++]);
	}

	/*      Restore Extended registers   (DAC/ADC rates etc...)     */

	for (i = 0x28; i <= 0x38; i += 2) {
		DURAUDIO_CodecWrite(pGeode, i,
				    pGeode->CODECRegisters[RegIndex++]);
	}

	/*
	   Check, for each channel, if DMA was running before suspend.
	   If it was, turn it back ON.
	 */
	for (Channel = 0; Channel < MAX_CHANNELS; Channel++) {
		if (pGeode->AudioChannel[Channel].Running == TRUE) {

			/*      Put back the PRD pointer        */

			if (pGeode->fCS5535) {
				OS_WritePortULong((unsigned short) (pGeode->
								    F3BAR0 +
								    0x24 +
								    (Channel *
								     0x08)),
						  pGeode->
						  AudioBusMaster_PRDTableAddress
						  [Channel]);

			} else {
				*(unsigned long *) (pGeode->F3BAR0 + 0x24 +
						    (Channel * 0x08)) =
				    pGeode->
				    AudioBusMaster_PRDTableAddress[Channel];
			}

			/*      Start DMA       */

			DURAUDIO_ResumeDMA(pGeode, Channel);
		}
	}
}

/*----------------------------------------------------------------------------
unsigned short DURAUDIO_IO_Read16(unsigned long pdev, unsigned short port)

	Reads 16 bits to the PCI I/O space
----------------------------------------------------------------------------*/
unsigned short
DURAUDIO_IO_Read16(unsigned long pdev, unsigned short port)
{
	OS_WritePortULong(PCI_CADDR, pdev + port);

	return OS_ReadPortUShort(PCI_CDATA);
}

/*----------------------------------------------------------------------------
void DURAUDIO_IO_Write16(unsigned long pdev, unsigned short port, 
unsigned short data )

	Writes 16 bits to the PCI I/O space
----------------------------------------------------------------------------*/
void
DURAUDIO_IO_Write16(unsigned long pdev, unsigned short port,
		    unsigned short data)
{
	OS_WritePortULong(PCI_CADDR, pdev + port);

	OS_WritePortUShort(PCI_CDATA, data);
}

/*----------------------------------------------------------------------------
unsigned long DURAUDIO_IO_Read32(unsigned long pdev, unsigned short port)

	Reads 32 bits to the PCI I/O space
----------------------------------------------------------------------------*/
unsigned long
DURAUDIO_IO_Read32(unsigned long pdev, unsigned short port)
{
	OS_WritePortULong(PCI_CADDR, pdev + port);

	return (OS_ReadPortULong(PCI_CDATA));
}

/*----------------------------------------------------------------------------
void DURAUDIO_IO_Write32(unsigned long pdev, unsigned short port, 
unsigned long data)

	Writes 32 bits to the PCI I/O space
----------------------------------------------------------------------------*/
void
DURAUDIO_IO_Write32(unsigned long pdev, unsigned short port, unsigned long data)
{
	OS_WritePortULong(PCI_CADDR, pdev + port);

	OS_WritePortULong(PCI_CDATA, data);
}

/*----------------------------------------------------------------------------
unsigned long DURAUDIO_GetPositionInBuffer (PDURAUDIO pGeode, unsigned long 
Channel)

	Returns the offset in bytes of audio data played inside the block of the data pointed by 
	the current PRD entry (Only available on 5535).

----------------------------------------------------------------------------*/
unsigned long
DURAUDIO_GetPositionInBuffer(PDURAUDIO pGeode,
			     unsigned long Channel, unsigned long PRDIndex)
{
	unsigned long BeginningOfCurrentBuffer, DMACurrentPointer;
	unsigned char CommandRegStatus;

	if (!pGeode->AudioChannel[Channel].PRDTable)
		return 0;

	BeginningOfCurrentBuffer =
	    pGeode->AudioChannel[Channel].PRDTable[PRDIndex].ulPhysAddr;

	if (pGeode->fCS5535) {
		if (pGeode->fIOAccess) {
			CommandRegStatus =
			    OS_ReadPortUChar((unsigned short) pGeode->
					     AudioChannel[Channel].
					     AudioBusMaster.CommandRegister);
			DMACurrentPointer =
			    OS_ReadPortULong((unsigned short) pGeode->
					     AudioChannel[Channel].
					     AudioBusMaster.DMAPointer);
		} else {
			CommandRegStatus =
			    (unsigned char) pGeode->AudioChannel[Channel].
			    AudioBusMaster.CommandRegister;
			DMACurrentPointer =
			    (unsigned char) pGeode->AudioChannel[Channel].
			    AudioBusMaster.DMAPointer;
		}
	} else {
		OS_DbgMsg("Not available for GX1 and SCXX00 Platforms\n");
		return 0;
	}

	/*      If the BusMaster is not enabled, return 0 as position   */

	if (!(CommandRegStatus & ENABLE_BUSMASTER))
		return 0;

	return (DMACurrentPointer - BeginningOfCurrentBuffer);
}

/*----------------------------------------------------------------------------
unsigned long DURAUDIO_GetPosition (PDURAUDIO pGeode, unsigned long Channel)

	Returns the position in bytes of the current data being played. 
----------------------------------------------------------------------------*/
unsigned long
DURAUDIO_GetPosition(PDURAUDIO pGeode, unsigned long Channel)
{
	unsigned long DMACurrentPointer;
	unsigned long OffsetBytePointer, PRDIndex;

	if (!pGeode->AudioChannel[Channel].Running) {
		return 0;
	}

	if (pGeode->fCS5535) {
		DMACurrentPointer =
		    OS_ReadPortULong((unsigned short) pGeode->
				     AudioChannel[Channel].AudioBusMaster.
				     DMAPointer);
		pGeode->AudioChannel[Channel].CurrentDMAPointer =
		    DMACurrentPointer -
		    pGeode->AudioChannel[Channel].PRDTable[0].ulPhysAddr;
	} else {
		PRDIndex = DURAUDIO_GetCurrentPRDIndex(pGeode, Channel);

		OffsetBytePointer =
		    pGeode->AudioChannel[Channel].PRDTable[PRDIndex].ulPhysAddr;

		/*      If we are right at the JMP PRD entry, 
		   Return 0 (beginning of the buffer)   */

		if (OffsetBytePointer ==
		    pGeode->AudioChannel[Channel].PRD_AllocInfo.PhysicalAddress)
			return 0;

		/*      Now we are saving the Current DMA pointer 
		   into the structure   */

		pGeode->AudioChannel[Channel].CurrentDMAPointer =
		    OffsetBytePointer -
		    pGeode->AudioChannel[Channel].PRDTable[0].ulPhysAddr;
	}

	return pGeode->AudioChannel[Channel].CurrentDMAPointer;
}

/*----------------------------------------------------------------------------
unsigned long DURAUDIO_GetCurrentPRD (PDURAUDIO pGeode, unsigned long Channel)

	Returns the Current PRD reg contents, unmodified
----------------------------------------------------------------------------*/
unsigned long
DURAUDIO_GetCurrentPRD(PDURAUDIO pGeode, unsigned long Channel)
{
	unsigned long PRDPointer;

	if (pGeode->fIOAccess) {
		if (Channel == CHANNEL0_PLAYBACK)
			PRDPointer =
			    OS_ReadPortULong((unsigned short) (pGeode->F3BAR0 +
							       0x24));
		else
			PRDPointer =
			    OS_ReadPortULong((unsigned short) (pGeode->F3BAR0 +
							       0x2C));
	} else {
		if (Channel == CHANNEL0_PLAYBACK)
			PRDPointer =
			    *((unsigned long *) (pGeode->F3BAR0 + 0x24));
		else
			PRDPointer =
			    *((unsigned long *) (pGeode->F3BAR0 + 0x2C));
	}

	return PRDPointer;
}

/*----------------------------------------------------------------------------
unsigned long DURAUDIO_GetCurrentPRDIndex (PDURAUDIO pGeode, 
unsigned long Channel)

	Returns the Current PRD index being played
----------------------------------------------------------------------------*/
unsigned long
DURAUDIO_GetCurrentPRDIndex(PDURAUDIO pGeode, unsigned long Channel)
{
	unsigned char CommandRegStatus;
	unsigned long PRDPointer, PRDIndex;

	/*      If the BusMaster is not enabled, return 0 as position   */

	if (pGeode->fIOAccess) {
		CommandRegStatus =
		    OS_ReadPortUChar((unsigned short) pGeode->
				     AudioChannel[Channel].AudioBusMaster.
				     CommandRegister);
	} else {
		CommandRegStatus =
		    *((unsigned char *) pGeode->AudioChannel[Channel].
		      AudioBusMaster.CommandRegister);
	}

	if (!(CommandRegStatus & ENABLE_BUSMASTER))
		return 0;

	PRDPointer = DURAUDIO_GetCurrentPRD(pGeode, Channel);

	if (PRDPointer ==
	    pGeode->AudioChannel[Channel].PRD_AllocInfo.PhysicalAddress) {
		PRDIndex = 0;
	} else {
		PRDIndex =
		    (PRDPointer -
		     pGeode->AudioChannel[Channel].PRD_AllocInfo.
		     PhysicalAddress) / sizeof (PRD_ENTRY) - 1;
	}

	return PRDIndex;
}

/*----------------------------------------------------------------------------
unsigned long DURAUDIO_DisplayCODECValues (PDURAUDIO pGeode)

	Returns the position in bytes of the current data being played. 
----------------------------------------------------------------------------*/
void
DURAUDIO_DisplayCODECValues(PDURAUDIO pGeode)
{
	unsigned char i;
	unsigned short Value;

	OS_DbgMsg("REG	Value");
	OS_DbgMsg("------------");

	for (i = 0; i < 0x7F; i += 2) {
		Value = DURAUDIO_CodecRead(pGeode, i);

		OS_DbgMsg("%04X", Value);
	}
}

/*----------------------------------------------------------------------------
unsigned char DURAUDIO_AllocateDMA (PDURAUDIO pGeode, unsigned long Channel, 
unsigned long ulBufferSize)

	Allocates the DMA buffer of ulBufferSize bytes for the specified Channel 
----------------------------------------------------------------------------*/
unsigned char
DURAUDIO_AllocateDMA(PDURAUDIO pGeode, unsigned long Channel,
		     unsigned long ulBufferSize)
{
	OS_DbgMsg("Allocating DMA\n");

	if (!DURAUDIO_AllocateAlignedDMAMemory
	    (pGeode, ulBufferSize,
	     &pGeode->AudioChannel[Channel].DMA_AllocInfo)) {
		OS_DbgMsg("Could not allocate the DMA buffer.\n");
		return FALSE;
	}

	/*      Clear the Audio buffer  */

	memset(pGeode->AudioChannel[Channel].DMA_AllocInfo.VirtualAddress, 0,
	       ulBufferSize);

	/*
	   If we are using indirect DMA, we need another buffer to tranfer
	   the sample-converted data.
	 */
	if (!pGeode->AudioChannel[Channel].fDirectDMA) {

		/*      Allocate Buffer for sample conversion   */

		if (!DURAUDIO_AllocateAlignedDMAMemory(pGeode,
						       (ulBufferSize /
							pGeode->
							AudioChannel[Channel].
							IndirectDMA.
							SampleConversionFactor),
						       &pGeode->
						       AudioChannel[Channel].
						       Work_AllocInfo)) {
			OS_DbgMsg
			    ("DURAUDIO_AllocateDMA - Unable to allocate %d bytes of memory for Work buffer\n",
			     ulBufferSize);
			return FALSE;
		}

		OS_DbgMsg("Allocated %d bytes for the Indirect DMA buffer.\n",
			  (ulBufferSize /
			   pGeode->AudioChannel[Channel].IndirectDMA.
			   SampleConversionFactor));
		memset(pGeode->AudioChannel[Channel].Work_AllocInfo.
		       VirtualAddress, 0, ulBufferSize);
	}

	return TRUE;
}

/*----------------------------------------------------------------------------
void DURAUDIO_FreeDMA (PDURAUDIO pGeode, unsigned long Channel)

	Frees the DMA allocated by DURAUDIO_AllocateDMA
----------------------------------------------------------------------------*/
void
DURAUDIO_FreeDMA(PDURAUDIO pGeode, unsigned long Channel)
{
	OS_DbgMsg("Freeing DMA\n");

	DURAUDIO_FreeAlignedDMAMemory(pGeode,
				      &pGeode->AudioChannel[Channel].
				      DMA_AllocInfo);

	if (!pGeode->AudioChannel[Channel].fDirectDMA) {
		DURAUDIO_FreeAlignedDMAMemory(pGeode,
					      &pGeode->AudioChannel[Channel].
					      Work_AllocInfo);
	}
}

/*----------------------------------------------------------------------------
unsigned char DURAUDIO_AllocatePRD (PDURAUDIO pGeode, unsigned long Channel, 
unsigned long Size)

	Allocates the PRD table for a DMA buffer of DMABufferSize bytes for 
	the specified Channel 
----------------------------------------------------------------------------*/
unsigned char
DURAUDIO_AllocatePRD(PDURAUDIO pGeode,
		     unsigned long Channel, unsigned long DMABufferSize)
{
	unsigned long PRDsToAllocate;

	PRDsToAllocate = (DMABufferSize / SIZE_OF_EACH_PRD_BLOCK + 1);

	if (!DURAUDIO_AllocateAlignedDMAMemory(pGeode,
					       (PRDsToAllocate *
						sizeof (PRD_ENTRY)),
					       &pGeode->AudioChannel[Channel].
					       PRD_AllocInfo)) {
		OS_DbgMsg("Could not allocate PRD\n");
		return FALSE;
	}

	pGeode->AudioChannel[Channel].PRDTable =
	    (PPRD_ENTRY) pGeode->AudioChannel[Channel].PRD_AllocInfo.
	    VirtualAddress;

	pGeode->AudioChannel[Channel].DMA_AllocInfo.Size = DMABufferSize;
	pGeode->AudioChannel[Channel].PRDEntriesAllocated = PRDsToAllocate;

	return TRUE;
}

/*----------------------------------------------------------------------------
void DURAUDIO_FreePRD (PDURAUDIO pGeode, unsigned long Channel)

	Frees the PRD allocated by DURAUDIO_AllocatePRD
----------------------------------------------------------------------------*/
void
DURAUDIO_FreePRD(PDURAUDIO pGeode, unsigned long Channel)
{
	DURAUDIO_FreeAlignedDMAMemory(pGeode,
				      &pGeode->AudioChannel[Channel].
				      PRD_AllocInfo);

	pGeode->AudioChannel[Channel].PRDTable = NULL;
}

/*----------------------------------------------------------------------------

  unsigned char DURAUDIO_AllocateAlignedDMAMemory (PDURAUDIO pGeode, 
	unsigned long Channel, void *pAdapterObject, unsigned long Size)

  PDURAUDIO       pGeode          -   Pointer to the DURAUDIO structure.
  unsigned long   Channel         -   Channel to allocate the DMA space for.
  void            *pAdapterObject -   Pointer for the Adapter Object 
										(Only for Windows XP/98/2000)
  unsigned long   Size            -   Size to allocate.
  PALLOC_INFO     pAllocInfo      -   Pointer to a ALLOC_INFO strcuture 
	            					  that contains the information 
                                      for the space allocated (Physical, 
									  Virtual and Size).

  Return value: FALSE if failed to allocate, TRUE if succeeds.

  Allocates a 32-bytes aligned DMA memory block.  
  Our hardware needs to have DMA and PRD spaces 32-bytes aligned to work 
  correctly.

  1. We save the original Physical and virtual addresses in the AllocInfo 
	 structure to be able to release the memory afterwards.

  2. We allocate 256 extra bytes to allow the shift.

  3. If the physical adress is not aligned, keep adding to the physical 
     and virtual address until it aligns to 32-byte boundary.

----------------------------------------------------------------------------*/
unsigned char
DURAUDIO_AllocateAlignedDMAMemory(PDURAUDIO pGeode,
				  unsigned long Size, PALLOC_INFO pAllocInfo)
{
	pAllocInfo->Size = Size;
	pAllocInfo->OriginalSize = Size + 256;

	pAllocInfo->OriginalVirtualAddress =
	    (void *) OS_AllocateDMAMemory(pGeode->pAdapterObject,
					  pAllocInfo->OriginalSize,
					  &pAllocInfo->OriginalPhysicalAddress);

	if (!pAllocInfo->OriginalVirtualAddress) {
		OS_DbgMsg("Could not allocate DMA Memory.  Size:%d\n", Size);
		return FALSE;
	}

	pAllocInfo->VirtualAddress = pAllocInfo->OriginalVirtualAddress;
	pAllocInfo->PhysicalAddress = pAllocInfo->OriginalPhysicalAddress;

	if (pAllocInfo->PhysicalAddress & 0x0000001F) {
		OS_DbgMsg("DMA Memory is not 32-byte aligned! : 0x%08X\n",
			  pAllocInfo->PhysicalAddress);
		while (pAllocInfo->PhysicalAddress & 0x0000001F) {
			pAllocInfo->VirtualAddress++;
			pAllocInfo->PhysicalAddress++;

		}
		OS_DbgMsg("Fixed to: 0x%08X.\n", pAllocInfo->PhysicalAddress);
	}
	return TRUE;
}

/*----------------------------------------------------------------------------

  void DURAUDIO_FreeAlignedDMAMemory ( PDURAUDIO pGeode, 
	PALLOC_INFO  pAllocInfo )

  PDURAUDIO   pGeode      -   Pointer to the DURAUDIO Structure
  PALLOC_INFO pAllocInfo  -   Pointer to a ALLOC_INFO strcuture that contains 
							  the information for the space allocated 
							  (Physical, Virtual and Size).

  Frees the memory allocated by DURAUDIO_AllocateAlignedDMAMemory.

----------------------------------------------------------------------------*/
void
DURAUDIO_FreeAlignedDMAMemory(PDURAUDIO pGeode, PALLOC_INFO pAllocInfo)
{
	if (pAllocInfo->VirtualAddress) {
		OS_FreeDMAMemory(pGeode->pAdapterObject,
				 (void *) pAllocInfo->OriginalVirtualAddress,
				 pAllocInfo->OriginalPhysicalAddress,
				 pAllocInfo->OriginalSize);

		pAllocInfo->VirtualAddress = NULL;
	}
}

/*----------------------------------------------------------------------------
  void DURAUDIO_InitPRD  (PDURAUDIO     pGeode, 
                          unsigned long Channel, 
                          unsigned long DMAPhys)

  Initializes the PRD for the specified channel
----------------------------------------------------------------------------*/
void
DURAUDIO_InitPRD(PDURAUDIO pGeode, unsigned long Channel, unsigned long DMAPhys)
{

	/*      Fill the PRD Entries with the right sizes and flags=0   */

	unsigned int i;

	for (i = 0; i < pGeode->AudioChannel[Channel].PRDEntriesAllocated; ++i) {
		pGeode->AudioChannel[Channel].PRDTable[i].ulPhysAddr =
		    DMAPhys + (i * SIZE_OF_EACH_PRD_BLOCK);
		pGeode->AudioChannel[Channel].PRDTable[i].SizeFlags =
		    SIZE_OF_EACH_PRD_BLOCK;
	}
}

/*----------------------------------------------------------------------------
  void DURAUDIO_DisplayPRD(PDURAUDIO pGeode, unsigned long Channel)

  Display the contents of the PRD table for the specific channel
----------------------------------------------------------------------------*/
void
DURAUDIO_DisplayPRD(PDURAUDIO pGeode, unsigned long Channel)
{
	unsigned long i;

	for (i = 0; i <= (pGeode->AudioChannel[Channel].PRDEntriesAllocated);
	     ++i) {
		OS_DbgMsg("%08X %08X\n",
			  pGeode->AudioChannel[Channel].PRDTable[i].ulPhysAddr,
			  pGeode->AudioChannel[Channel].PRDTable[i].SizeFlags);
	}
}

/*----------------------------------------------------------------------------

  DURAUDIO_ReadStatus (PDURAUDIO pGeode, unsigned int BusMaster)

  Reads the Status bits of the specified Bus Master

----------------------------------------------------------------------------*/
unsigned char
DURAUDIO_ReadStatus(PDURAUDIO pGeode, unsigned int BusMaster)
{
	if (pGeode->fIOAccess) {
		return
		    OS_ReadPortUChar((unsigned short) (pGeode->F3BAR0 +
						       (0x21 +
							(8 * BusMaster))));
	} else {
		return
		    *((unsigned char *) (pGeode->F3BAR0 + 0x21 +
					 (8 * BusMaster)));
	}
}

/*----------------------------------------------------------------------------
   unsigned long DURAUDIO_GetVolume(PDURAUDIO pGeode)

   Returns the digital volume to the caller
----------------------------------------------------------------------------*/
unsigned long
DURAUDIO_GetVolume(PDURAUDIO pGeode)
{
	return pGeode->v_nVolume;
}

/*----------------------------------------------------------------------------
   void DURAUDIO_SetVolume(PDURAUDIO pGeode, unsigned long Vol)

   Sets the digital volume based on the value sent by the caller
----------------------------------------------------------------------------*/
void
DURAUDIO_SetVolume(PDURAUDIO pGeode, unsigned long Vol)
{
	unsigned short rVol, lVol, nVol;
	unsigned char rIdx, lIdx;

	pGeode->v_nVolume = Vol;

	rIdx = (unsigned char) ((Vol & 0xF8000000) >> (32 - 5));
	lIdx = (unsigned char) ((Vol & 0x0000F800) >> (16 - 5));
	rVol = VolumeLUT[rIdx];
	lVol = VolumeLUT[lIdx];
	nVol = rVol | (lVol << 8);
	if (nVol == 0x1f1f)
		nVol |= 0x8000;

	DURAUDIO_CodecWrite(pGeode, MASTER_VOLUME, nVol);
}

/*----------------------------------------------------------------------------

  unsigned char DURAUDIO_SetupDMA (PDURAUDIO pGeode, unsigned long Channel, 
	unsigned long DMABufferSize)

  - Allocates DMA Buffer and PRD
  - Divides the PRD in Blocks and Initializes it with the jump to the top

  Return value:  Returns FALSE if allocation fails

----------------------------------------------------------------------------*/
unsigned char
DURAUDIO_SetupDMA(PDURAUDIO pGeode, unsigned long Channel,
		  unsigned long DMABufferSize)
{
	if (!pGeode->AudioChannel[Channel].fAllocated) {

		/*      Allocate PRD and DMA Buffer     */

		if (!DURAUDIO_AllocatePRD(pGeode, Channel, DMABufferSize)) {
			return FALSE;
		}

		if (!DURAUDIO_AllocateDMA(pGeode, Channel, DMABufferSize)) {
			DURAUDIO_FreePRD(pGeode, Channel);
			return FALSE;
		}

		pGeode->AudioChannel[Channel].fAllocated = TRUE;
	}

	DURAUDIO_InitPRD(pGeode, Channel,
			 pGeode->AudioChannel[Channel].DMA_AllocInfo.
			 PhysicalAddress);

	return TRUE;
}

/*----------------------------------------------------------------------------
  void DURAUDIO_SetupPRD(PDURAUDIO      pGeode, 
                         unsigned  long Channel, 
                         unsigned  long SampleRate, 
                         unsigned  long InterruptFrequency,
                         unsigned  char FrequencyType)

  Setup the PRD entries of the specified channel with EOP to generate 
  interrupts based on the Sample rate and the InterruptFrequency 
  value expressed in Miliseconds.

----------------------------------------------------------------------------*/
void
DURAUDIO_SetupPRD(PDURAUDIO pGeode,
		  unsigned long Channel,
		  unsigned long SampleRate,
		  unsigned long InterruptFrequency, unsigned char FrequencyType)
{
	/*--------------------------------------------------------------------
	                              Interrupt-Related code
	---------------------------------------------------------------------*/

	unsigned long m_IntFreq, IntInc;
	unsigned int i;

	if (FrequencyType == INTERVAL_BYTES) {
		m_IntFreq = (InterruptFrequency / SIZE_OF_EACH_PRD_BLOCK);
	} else {
		m_IntFreq =
		    (((SampleRate * 4) * InterruptFrequency) / 1000) /
		    SIZE_OF_EACH_PRD_BLOCK;
	}

	IntInc = m_IntFreq - 1;

	for (i = 0;
	     i < (pGeode->AudioChannel[Channel].PRDEntriesAllocated - 1); ++i) {
		/* Zero the Flags */
		pGeode->AudioChannel[Channel].PRDTable[i].SizeFlags &=
		    0x0000FFFF;

		if (i == IntInc) {
			/* Set EOP bit */
			pGeode->AudioChannel[Channel].PRDTable[i].SizeFlags |=
			    PRD_EOP_BIT;
			IntInc += m_IntFreq;
		} else
			/* Clear Flags (keeps size) */
			pGeode->AudioChannel[Channel].PRDTable[i].SizeFlags &=
			    0x0FFFFFFF;
	}

	/*--------------------------------------------------------------------
	                          End of Interrupt-Related code
	----------------------------------------------------------------------*/

	/*      And, last, put the JMP to the top */

	pGeode->AudioChannel[Channel].PRDTable[pGeode->AudioChannel[Channel].
					       PRDEntriesAllocated -
					       1].ulPhysAddr =
	    pGeode->AudioChannel[Channel].PRD_AllocInfo.PhysicalAddress;
	pGeode->AudioChannel[Channel].PRDTable[pGeode->AudioChannel[Channel].
					       PRDEntriesAllocated -
					       1].SizeFlags = PRD_JMP_BIT;
}

/*-------------------------------------------------------------------------
                              Interrupt-Related code
--------------------------------------------------------------------------*/
/*	
	Indirect DMA Functions
*/

/*----------------------------------------------------------------------------

  void    DURAUDIO_WaveOpen ( PDURAUDIO       pGeode, 
                              unsigned long   Channel, 
                              unsigned long   SampleRate, 
                              unsigned char   nChannels, 
                              unsigned char   BitsPerSample)

  Sets up data pointers, saves data format into the structure and points to 
  the correct sample converter function.

----------------------------------------------------------------------------*/
unsigned char
DURAUDIO_WaveOpen(PDURAUDIO pGeode,
		  unsigned long Channel,
		  unsigned long SampleRate,
		  unsigned char nChannels,
		  unsigned char BitsPerSample,
		  unsigned long InterruptFrequency,
		  unsigned char FrequencyType,
		  unsigned long UserBufferSize, unsigned char DirectDMA)
{
	unsigned long DMABufferSize, InterruptFrequencyInDMABytes;

	/*      Save the format */

	pGeode->AudioChannel[Channel].IndirectDMA.SampleRate = SampleRate;
	pGeode->AudioChannel[Channel].IndirectDMA.nChannels = nChannels;
	pGeode->AudioChannel[Channel].IndirectDMA.BitsPerSample = BitsPerSample;

	/*      Set the data pointers to 0      */

	pGeode->AudioChannel[Channel].IndirectDMA.BytesRemainingInDMABuffer = 0;
	pGeode->AudioChannel[Channel].IndirectDMA.CurrentTransferPointer = 0;
	pGeode->AudioChannel[Channel].CurrentDMAPointer = 0;
	pGeode->AudioChannel[Channel].fDirectDMA = DirectDMA;

	if ((BitsPerSample == 0) || (nChannels == 0)) {
		OS_DbgMsg("DURAUDIO: Invalid BitsPerSample or nChannels\n");
		return FALSE;
	} else {
		pGeode->AudioChannel[Channel].IndirectDMA.
		    SampleConversionFactor =
		    ((16 / BitsPerSample) * (2 / nChannels));
	}

	OS_DbgMsg
	    ("SampleConversionFactor: %d, BitsPerSample: %d, nChannels: %d\n",
	     pGeode->AudioChannel[Channel].IndirectDMA.SampleConversionFactor,
	     BitsPerSample, nChannels);

	DMABufferSize =
	    UserBufferSize *
	    pGeode->AudioChannel[Channel].IndirectDMA.SampleConversionFactor;

	if (!DURAUDIO_SetupDMA(pGeode, Channel, DMABufferSize))
		return FALSE;

	if (FrequencyType == INTERVAL_BYTES) {
		InterruptFrequencyInDMABytes =
		    InterruptFrequency *
		    pGeode->AudioChannel[Channel].IndirectDMA.
		    SampleConversionFactor;

		DURAUDIO_SetupPRD(pGeode, Channel, SampleRate,
				  InterruptFrequencyInDMABytes, FrequencyType);
	} else {
		DURAUDIO_SetupPRD(pGeode, Channel, SampleRate,
				  InterruptFrequency, FrequencyType);
	}

	return TRUE;
}

/*----------------------------------------------------------------------------
   void DURAUDIO_WaveClose(PDURAUDIO pGeode, WAPI_INOUT Channel)

   Closes the Wave Stream.
   Sets the "In use" flag to FALSE;

----------------------------------------------------------------------------*/
void
DURAUDIO_WaveClose(PDURAUDIO pGeode, unsigned long Channel)
{
	OS_DbgMsg("DURAUDIO_WaveClose");

	if (pGeode->AudioChannel[Channel].fAllocated) {
		DURAUDIO_FreeDMA(pGeode, Channel);
		DURAUDIO_FreePRD(pGeode, Channel);

		pGeode->AudioChannel[Channel].fAllocated = FALSE;
	}
};

/*----------------------------------------------------------------------------

  unsigned char UpdateByteCounter(PDURAUDIO pGeode, unsigned long Channel)

  Returned value: The ACTUAL number of unplayed bytes in the buffer in terms 
	of user's  sample size.

----------------------------------------------------------------------------*/
long
DURAUDIO_UpdateByteCounter(PDURAUDIO pGeode, unsigned long Channel)
{
	long SavedDMAPointer, diff;
	unsigned long DMABufferSize;

	if (!pGeode->AudioChannel[Channel].PRDTable)
		return 0;

	OS_SpinLock();

	/*      Save the current DMA pointer    */

	SavedDMAPointer = pGeode->AudioChannel[Channel].CurrentDMAPointer;

	/*      Get the current DMA pointer     */

	DURAUDIO_GetPosition(pGeode, Channel);

	DMABufferSize = pGeode->AudioChannel[Channel].DMA_AllocInfo.Size;

	diff =
	    (DMABufferSize + pGeode->AudioChannel[Channel].CurrentDMAPointer -
	     SavedDMAPointer) % DMABufferSize;

	if (Channel == CHANNEL0_PLAYBACK) {
		pGeode->AudioChannel[Channel].IndirectDMA.
		    BytesRemainingInDMABuffer -= diff;
	} else {
		pGeode->AudioChannel[Channel].IndirectDMA.
		    BytesRemainingInDMABuffer += diff;
	}

	/*      Return the converted ammount of data in the buffer 
	   based on the Sample type     */

	if (pGeode->AudioChannel[Channel].IndirectDMA.
	    BytesRemainingInDMABuffer > 0) {
		OS_SpinUnlock();
		return (pGeode->AudioChannel[Channel].IndirectDMA.
			BytesRemainingInDMABuffer /
			pGeode->AudioChannel[Channel].IndirectDMA.
			SampleConversionFactor);
	} else {
		pGeode->AudioChannel[Channel].IndirectDMA.
		    BytesRemainingInDMABuffer = 0;
		OS_SpinUnlock();
		return 0;
	}
}

/*----------------------------------------------------------------------------

  unsigned long DURAUDIO_ConvertAndCopyData ( PDURAUDIO pGeode, 
	unsigned long Channel, unsigned char *pDMABuffer, unsigned long Size)

  Converts the data in the WorkBuffer from all formats to 16bit Stereo 
  and copies it to (or from, depending on Channel) the DMA buffer.  
  Formats already in 16bit Stereo don't need conversion.

----------------------------------------------------------------------------*/
void
DURAUDIO_ConvertAndCopyData(PDURAUDIO pGeode, unsigned long Channel,
			    unsigned char *pDMABuffer,
			    unsigned char *pUserBuffer,
			    unsigned long nDMABytesToTransfer)
{
	unsigned short wData;
	unsigned long ulOffset, BytesRecorded, SampleConversionFactor;
	unsigned char *pWorkBuffer, BitsPerSample, nChannels, Data;

	/*      Making things easier to read... */

	BitsPerSample = pGeode->AudioChannel[Channel].IndirectDMA.BitsPerSample;
	nChannels = pGeode->AudioChannel[Channel].IndirectDMA.nChannels;
	SampleConversionFactor =
	    pGeode->AudioChannel[Channel].IndirectDMA.SampleConversionFactor;
	pWorkBuffer =
	    pGeode->AudioChannel[Channel].Work_AllocInfo.VirtualAddress;

	ulOffset = 0;
	BytesRecorded = 0;

	if (Channel == CHANNEL0_PLAYBACK) {

		/*      If format is Stereo 16 and we are using a 5535, 
		   we move without conversion   */

		if ((nChannels == 2) && (BitsPerSample == 16)
		    && (pGeode->fCS5535)) {
			OS_CopyFromUser(pDMABuffer, pUserBuffer,
					nDMABytesToTransfer);
		} else {
			OS_CopyFromUser(pWorkBuffer, pUserBuffer,
					(nDMABytesToTransfer /
					 SampleConversionFactor));

			if (nChannels == 1) {

				/*      Mono 8bit       */

				if (BitsPerSample == 8) {
					while (ulOffset < nDMABytesToTransfer) {
						wData = (unsigned short)
						    *((pWorkBuffer +
						       BytesRecorded));
						wData =
						    (unsigned short)
						    BITS_8_TO_16(wData);

						*((unsigned short *) (pDMABuffer
								      +
								      ulOffset))
						    = wData;
						*((unsigned short *) (pDMABuffer
								      +
								      ulOffset +
								      2)) =
						    wData;

						ulOffset += 4;
						BytesRecorded += 1;
					}
				} else {

					/*      Mono 16bit      */

					while (ulOffset < nDMABytesToTransfer) {
						wData =
						    *((unsigned short
						       *) (pWorkBuffer +
							   BytesRecorded));

						*((unsigned short *) (pDMABuffer
								      +
								      ulOffset))
						    = wData;
						*((unsigned short *) (pDMABuffer
								      +
								      ulOffset +
								      2)) =
						    wData;

						ulOffset += 4;
						BytesRecorded += 2;
					}
				}
			} else {
				if (BitsPerSample == 8) {
					unsigned char ucData;

					/*      Stereo 8bit samples     */

					while (ulOffset < nDMABytesToTransfer) {
						ucData =
						    (unsigned char)
						    *(pWorkBuffer +
						      BytesRecorded);

						wData =
						    (unsigned short)
						    BITS_8_TO_16(ucData);
						*((unsigned short *) (pDMABuffer
								      +
								      ulOffset))
						    = wData;

						ucData =
						    *(pWorkBuffer +
						      BytesRecorded + 1);

						wData =
						    (unsigned short)
						    BITS_8_TO_16(ucData);
						*((unsigned short *) (pDMABuffer
								      +
								      ulOffset +
								      2)) =
						    wData;

						ulOffset += 4;
						BytesRecorded += 2;
					}
				} else {
					/*
					   Stereo 16bit samples

					   If we are here, it is because we have a old platform 
					   (Not 5535) 5530 has a bug that it has the channel order
					   inverted, so we need to manipulate the data and revert 
					   the channels to the correct position
					 */
					while (ulOffset < nDMABytesToTransfer) {
						wData =
						    *((unsigned short
						       *) (pWorkBuffer +
							   BytesRecorded));
						*((unsigned short *) (pDMABuffer
								      +
								      ulOffset +
								      2)) =
						    wData;

						wData =
						    *((unsigned short
						       *) (pWorkBuffer +
							   BytesRecorded + 2));
						*((unsigned short *) (pDMABuffer
								      +
								      ulOffset))
						    = wData;

						ulOffset += 4;
						BytesRecorded += 4;
					}
				}
			}
		}
	} else {

		/*      If format is Stereo 16bit and we are using a 5535, we move 
		   without conversion   */

		if ((nChannels == 2) && (BitsPerSample == 16)
		    && (pGeode->fCS5535)) {
			OS_DbgMsg("Transferring %d RECORDING bytes\n",
				  nDMABytesToTransfer);
			OS_CopyToUser(pUserBuffer, pDMABuffer,
				      nDMABytesToTransfer);
		} else {
			if (nChannels == 1) {
				if (BitsPerSample == 8) {

					/*      Mono 8bit samples       */

					while (ulOffset < nDMABytesToTransfer) {
						wData =
						    *((unsigned short
						       *) (pDMABuffer +
							   ulOffset));
						Data = BITS_16_TO_8(wData);
						*(pWorkBuffer + BytesRecorded) =
						    Data;

						ulOffset += 4;
						BytesRecorded += 1;
					}
				} else {

					/*      Mono 16bit samples      */

					while (ulOffset < nDMABytesToTransfer) {
						wData =
						    *((unsigned short
						       *) (pDMABuffer +
							   ulOffset));
						*((unsigned short
						   *) (pWorkBuffer +
						       BytesRecorded)) = wData;

						ulOffset += 4;
						BytesRecorded += 2;
					}
				}
			} else {
				if (BitsPerSample == 8) {

					/*      Stereo 8bit samples     */

					while (ulOffset < nDMABytesToTransfer) {
						wData =
						    *((unsigned short
						       *) (pDMABuffer +
							   ulOffset));
						Data = BITS_16_TO_8(wData);
						*(pWorkBuffer + BytesRecorded) =
						    Data;
						wData =
						    *((unsigned short
						       *) (pDMABuffer +
							   ulOffset + 2));
						Data = BITS_16_TO_8(wData);
						*(pWorkBuffer + BytesRecorded +
						  1) = Data;

						ulOffset += 4;
						BytesRecorded += 2;
					}

				} else {

					/*      Stereo 16bit samples    */

					/*
					   If we are here, it is because we have a old 
					   platform (Not 5535)  5530 has a bug that it has the
					   channel order inverted, so we need to manipulate the 
					   data and revert the channels to the correct position
					 */
					while (ulOffset < nDMABytesToTransfer) {
						wData =
						    *((unsigned short
						       *) (pDMABuffer +
							   ulOffset));
						*((unsigned short
						   *) (pWorkBuffer + ulOffset +
						       2)) = wData;

						wData =
						    *((unsigned short
						       *) (pDMABuffer +
							   ulOffset + 2));
						*((unsigned short
						   *) (pWorkBuffer +
						       ulOffset)) = wData;

						ulOffset += 4;
						BytesRecorded += 4;
					}
				}
			}

			OS_CopyToUser(pUserBuffer, pWorkBuffer, BytesRecorded);
		}
	}
}

/*----------------------------------------------------------------------------

  void DURAUDIO_ZeroFillFreeBuffer (PDURAUDIO pGeode)

  put in zero values to fill the free part of the playback buffer

----------------------------------------------------------------------------*/

void
DURAUDIO_ZeroFillFreeBuffer(PDURAUDIO pGeode)
{
	unsigned long dma_size, dma_used;
	unsigned long zero_count, zero_index;
	unsigned char *dma_buf;
	struct tagAudioChannel *chan = &pGeode->AudioChannel[CHANNEL0_PLAYBACK];
	struct tagIndirectDMA *iDMA = &chan->IndirectDMA;

	DURAUDIO_UpdateByteCounter(pGeode, CHANNEL0_PLAYBACK);

	/*      Making things easier to read... */

	dma_used = iDMA->BytesRemainingInDMABuffer;	/*       In DMA bytes   */
	dma_buf = chan->DMA_AllocInfo.VirtualAddress;
	dma_size = chan->DMA_AllocInfo.Size;

	zero_index = iDMA->CurrentTransferPointer;	/*       In DMA bytes   */
	zero_count = dma_size - dma_used;

	if ((zero_index + zero_count) > dma_size) {
		/*      Handle circular buffer wrap-around      */
		unsigned long zero_to_end = dma_size - zero_index;
		memset(dma_buf + zero_index, 0, zero_to_end);
		zero_index = 0;
		zero_count -= zero_to_end;
	}
	if (zero_count)
		memset(dma_buf + zero_index, 0, zero_count);

	return;
}

/*----------------------------------------------------------------------------

  unsigned long DURAUDIO_TransferAudioData (PDURAUDIO pGeode, unsigned long 
	Channel, unsigned char *pUserBuffer, unsigned long Size)

  Returned value: The ACTUAL number of bytes transferred in terms of user's
  sample size.

----------------------------------------------------------------------------*/

unsigned long
DURAUDIO_TransferAudioData(PDURAUDIO pGeode,
			   unsigned long Channel,
			   unsigned char *pUserBuffer, unsigned long Size)
{
	long SizeInConvertedBytes;
	unsigned long nDMABytesToTransfer, DMABufferSize,
	    CurrentTransferPointer;
	unsigned char *pDMABuffer, *pStartOfDMABuffer;
	unsigned char BitsPerSample, nChannels;

	/*      Making things easier to read... */

	BitsPerSample = pGeode->AudioChannel[Channel].IndirectDMA.BitsPerSample;
	nChannels = pGeode->AudioChannel[Channel].IndirectDMA.nChannels;
	DMABufferSize = pGeode->AudioChannel[Channel].DMA_AllocInfo.Size;
	CurrentTransferPointer =
	    pGeode->AudioChannel[Channel].IndirectDMA.CurrentTransferPointer;
	pStartOfDMABuffer =
	    pGeode->AudioChannel[Channel].DMA_AllocInfo.VirtualAddress;

	/*
	   All samples must be converted to Stereo 16Bit.
	   We calculate the actual size here
	 */
	SizeInConvertedBytes =
	    Size *
	    pGeode->AudioChannel[Channel].IndirectDMA.SampleConversionFactor;
	nDMABytesToTransfer = SizeInConvertedBytes;

	if (Channel == CHANNEL0_PLAYBACK) {
		/*
		   On Playback, we heck to see if the size of the block we are transferring is larger
		   than the space we have available in the buffer.  (We always try to keep the DMA buffer full).
		 */
		DURAUDIO_UpdateByteCounter(pGeode, CHANNEL0_PLAYBACK);

		if (SizeInConvertedBytes >
		    (long) (DMABufferSize -
			    pGeode->AudioChannel[Channel].IndirectDMA.
			    BytesRemainingInDMABuffer)) {
			OS_DbgMsg
			    ("**************************************************************************\n");
			OS_DbgMsg("DURAUDIO_TransferAudioData:\n");
			OS_DbgMsg
			    ("You cannot transfer this ammount (%d bytes) without overlapping the current data in the DMA buffer\n",
			     nDMABytesToTransfer);

			nDMABytesToTransfer =
			    (DMABufferSize -
			     pGeode->AudioChannel[Channel].IndirectDMA.
			     BytesRemainingInDMABuffer);

			OS_DbgMsg
			    ("Transferring only %d bytes - Check your code.\n",
			     nDMABytesToTransfer);
			OS_DbgMsg
			    ("******************************************************************************************\n");
		}
	} else {

		/*      On Recording, we transfer only what we already captured 
		   in the buffer        */

		DURAUDIO_UpdateByteCounter(pGeode, CHANNEL1_RECORD);

		if (SizeInConvertedBytes >
		    pGeode->AudioChannel[Channel].IndirectDMA.
		    BytesRemainingInDMABuffer) {
			nDMABytesToTransfer =
			    pGeode->AudioChannel[Channel].IndirectDMA.
			    BytesRemainingInDMABuffer;
		} else {

			/*      It is smaller or equal.  Use the whole thing.   */

			nDMABytesToTransfer = SizeInConvertedBytes;
		}
	}

	pDMABuffer = pStartOfDMABuffer + CurrentTransferPointer;

	if ((CurrentTransferPointer + nDMABytesToTransfer) > DMABufferSize) {
		unsigned long nPartialBytes;

		/*
		   It wrapped around.
		   We need to break the data in two pieces 
		 */
		nPartialBytes = DMABufferSize - CurrentTransferPointer;

		/*
		   Copy user Data to our WorkBuffer, convert and put the first piece 
		   from the current pointer to the end of the DMA buffer.
		 */
		DURAUDIO_ConvertAndCopyData(pGeode, Channel, pDMABuffer,
					    pUserBuffer, nPartialBytes);

		/*      We put the remaining data in the second piece from the beggining 
		   of the DMA buffer    */

		DURAUDIO_ConvertAndCopyData(pGeode, Channel, pStartOfDMABuffer,
					    (pUserBuffer +
					     (nPartialBytes /
					      pGeode->AudioChannel[Channel].
					      IndirectDMA.
					      SampleConversionFactor)),
					    (nDMABytesToTransfer -
					     nPartialBytes));

		/*      Update the byte pointer */

		CurrentTransferPointer = (nDMABytesToTransfer - nPartialBytes);
	} else {

		/*      Copy user data to our WorkBuffer, Convert and copy into
		   the DMA buffer       */

		DURAUDIO_ConvertAndCopyData(pGeode, Channel, pDMABuffer,
					    pUserBuffer, nDMABytesToTransfer);

		/*      Update the byte pointer */

		CurrentTransferPointer += nDMABytesToTransfer;

		if (CurrentTransferPointer == DMABufferSize) {

			/*      Pointer is equal to the DMA buffer size  Set it to 0    */

			CurrentTransferPointer = 0;
		}
	}

	if (Channel == CHANNEL0_PLAYBACK) {
		pGeode->AudioChannel[Channel].IndirectDMA.
		    BytesRemainingInDMABuffer += nDMABytesToTransfer;
	} else
		pGeode->AudioChannel[Channel].IndirectDMA.
		    BytesRemainingInDMABuffer -= nDMABytesToTransfer;

	/*      Update the real CurrentTransferPointer  */

	pGeode->AudioChannel[Channel].IndirectDMA.CurrentTransferPointer =
	    CurrentTransferPointer;

	return (nDMABytesToTransfer /
		pGeode->AudioChannel[Channel].IndirectDMA.
		SampleConversionFactor);
}

/*----------------------------------------------------------------------------

  void DURAUDIO_SetupIRQ (unsigned long Irq)

  - Programs the IRQ 
  - Sets VSA audio irq component to create the Virtual interrupt (CS5530)

----------------------------------------------------------------------------*/
void
DURAUDIO_SetupIRQ(PDURAUDIO pGeode, unsigned long Irq)
{
	if (!pGeode->fCS5535) {
		unsigned long SetIRQData;

		/*      If it is a CS5530, we have to configure VSA...  */

		/*      VSA2 IRQ config method  */

		OS_WritePortUShort(0xAC1C, 0xFC53);	/*Unlock virtual registers      */
		OS_WritePortUShort(0xAC1C, 0x108);	/*ID the audio config virtual regs      */
		OS_WritePortUShort(0xAC1E, (unsigned short) Irq);	/*Set the value */

		/*      VSA1 IRQ config method  */

		/*      Get the Irq and OR it with the command  */

		SetIRQData = ((Irq << 16) | 0xA00A);
		/*      Set the address */
		OS_WritePortULong((unsigned short) PCI_CADDR, 0x800090D0);
		/*      Send the command        */
		OS_WritePortULong((unsigned short) PCI_CDATA, SetIRQData);

		/*      Set the InterruptID address for the CS5530      */

		pGeode->pInterruptID =
		    (unsigned char *) OS_VirtualAddress(0x000004F0);
	}
}

/*----------------------------------------------------------------------------
  unsigned short DURAUDIO_InterruptID (PDURAUDIO pGeode)

  Returns the Interrupt Flags from the controller.
  This is the mirror of the controller's status register

  0x04    -   DMA0 interrupt (Wave Out)
  0x08    -   DMA1 interrupt (Wave In)

----------------------------------------------------------------------------*/
unsigned short
DURAUDIO_InterruptID(PDURAUDIO pGeode)
{
	volatile unsigned short *TempInterruptID, ID;

	if (pGeode->fCS5535) {
		if (pGeode->fIOAccess) {
			ID = (unsigned char)
			    OS_ReadPortUShort((unsigned short) (pGeode->F3BAR0 +
								0x12));
		} else {
			TempInterruptID =
			    (unsigned short *) (pGeode->F3BAR0 + 0x12);
			ID = *TempInterruptID;
		}
	} else {
		ID = *(pGeode->pInterruptID);

		/*      Clear the Interrupt ID bits     */

		*(pGeode->pInterruptID) = 0x00000000;

		/*      If there is no 0x4F0 mechanism, read it directly from the 
		   status register      */

		if (ID == 0) {
			if (pGeode->fIOAccess) {
				ID = (unsigned short)
				    OS_ReadPortUShort((unsigned short) (pGeode->
									F3BAR0 +
									0x12));
			} else {
				TempInterruptID =
				    (unsigned short *) (pGeode->F3BAR0 + 0x12);
				ID = *TempInterruptID;
			}
		}
	}

	return (ID);
}

/*----------------------------------------------------------------------------
  void DURAUDIO_ClearIRQ(PDURAUDIO pGeode)

  Clear the IRQ

----------------------------------------------------------------------------*/
void
DURAUDIO_ClearIRQ(PDURAUDIO pGeode)
{
	volatile unsigned long IRQStatus;
	volatile unsigned short wIRQStatus;

	if (pGeode->fCS5535) {
		DURAUDIO_ClearStat(pGeode, CHANNEL0_PLAYBACK);
		DURAUDIO_ClearStat(pGeode, CHANNEL1_RECORD);
	} else {

		/*      Read the IRQ Control register   */

		IRQStatus = *((unsigned long *) pGeode->IRQControlRegister);

		/*      Clear the respective IRQ bit    */

		IRQStatus = IRQStatus & (0xFFFF0000 | pGeode->uIRQMask);

		*((unsigned long *) pGeode->IRQControlRegister) = IRQStatus;

		/*      Read the Internal IRQ Enable register   */

		wIRQStatus =
		    *((unsigned short *) pGeode->InternalIRQEnableRegister);

		/*      Clear the respective IRQ bit - Set it back to external interrupt        */

		wIRQStatus = wIRQStatus & pGeode->uIRQMask;

		*((unsigned short *) pGeode->InternalIRQEnableRegister) =
		    wIRQStatus;
	}
}

/*----------------------------------------------------------------------------
 unsigned char DURAUDIO_ClearStat(PDURAUDIO pGeode, unsigned long Channel)

  Clear the status bit of the respective Channel

----------------------------------------------------------------------------*/
unsigned char
DURAUDIO_ClearStat(PDURAUDIO pGeode, unsigned long Channel)
{
	volatile unsigned char status;	/*       Volatile to force read-to-clear.       */

	/*      Read to clear   */

	if (pGeode->fIOAccess) {
		status =
		    OS_ReadPortUChar((unsigned short) pGeode->
				     AudioChannel[Channel].AudioBusMaster.
				     SMI_StatusRegister);
	} else {
		status =
		    *((unsigned char *) pGeode->AudioChannel[Channel].
		      AudioBusMaster.SMI_StatusRegister);
	}

	return status;
}

/*----------------------------------------------------------------------------
                              End of Interrupt-Related code
----------------------------------------------------------------------------*/
