/*
 *  linux/drivers/char/versatile.c
 *
 *  Driver for AMBA PL011 serial ports for Versatile board
 *
 *  Based on drivers/char/serial.c, by Linus Torvalds, Theodore Ts'o.
 *
 *  Copyright (c) 2002-2003 ARM Limited
 *  Copyright (C) 2000 Deep Blue Solutions Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * This is a generic driver for ARM AMBA-type serial ports (PL011).
 */
#include <linux/config.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/major.h>
#include <linux/string.h>
#include <linux/fcntl.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/circ_buf.h>
#include <linux/serial.h>
#include <linux/console.h>
#include <linux/sysrq.h>

#include <asm/system.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/uaccess.h>
#include <asm/bitops.h>

//#define VERSATILE_DEBUG

#if defined(CONFIG_SERIAL_VERSATILE_PB_CONSOLE) && defined(CONFIG_MAGIC_SYSRQ)
#define SUPPORT_SYSRQ
#endif

#include <linux/serial_core.h>

#include <asm/hardware/serial_cintegrator.h>

#define UART_NR			3

// FIXME: will add the 4th serial port later, interrupts on SIC for uart3

#define SERIAL_VERSATILE_MAJOR	204
#define SERIAL_VERSATILE_MINOR	64
#define SERIAL_VERSATILE_NR		UART_NR

#define CALLOUT_AMBA_NAME	"cuacm"
#define CALLOUT_AMBA_MAJOR	205
#define CALLOUT_AMBA_MINOR	64
#define CALLOUT_AMBA_NR		UART_NR

static struct tty_driver normal, callout;
static struct tty_struct *versatile_table[UART_NR];
static struct termios *versatile_termios[UART_NR], *versatile_termios_locked[UART_NR];
#ifdef SUPPORT_SYSRQ
static struct console versatile_console;
#endif

static void versatile_uart_write(const char *s, u_int count, int port_nb);

#define VERSATILE_ISR_PASS_LIMIT	256

/*
 * Access macros for the AMBA UARTs
 */
#define UART_GET_INT_STATUS(p)	__raw_readl((p)->membase + AMBA_UARTMIS)
#define UART_PUT_ICR(p, c)	__raw_writel((c), (p)->membase + AMBA_UARTICR)
#define UART_GET_FR(p)		__raw_readl((p)->membase + AMBA_UARTFR)
#define UART_PUT_IBRD(p, c)	__raw_writel((c), (p)->membase + AMBA_UARTIBRD)
#define UART_GET_IBRD(p)	(__raw_readl((p)->membase + AMBA_UARTIBRD) & 0xFFFF)
#define UART_PUT_FBRD(p, c)	__raw_writel((c), (p)->membase + AMBA_UARTFBRD)
#define UART_GET_FBRD(p)	(__raw_readl((p)->membase + AMBA_UARTFBRD) & 0xFFFF)
#define UART_GET_CHAR(p)	(unsigned char)__raw_readl((p)->membase + AMBA_UARTDR)
/* XXX */
#define UART_PUT_CHAR(p, c)	__raw_writel((c), (p)->membase + AMBA_UARTDR)
#define UART_GET_RSR(p)		(unsigned char)__raw_readl((p)->membase + AMBA_UARTRSR)
#define UART_GET_CR(p)		__raw_readl((p)->membase + AMBA_UARTCR)
#define UART_PUT_CR(p,c)	__raw_writel((c), (p)->membase + AMBA_UARTCR)
#define UART_GET_LCRH(p)	__raw_readl((p)->membase + AMBA_UARTLCR_H)
#define UART_PUT_LCRH(p,c)	__raw_writel((c), (p)->membase + AMBA_UARTLCR_H)
#define UART_RX_DATA(s)		(((s) & AMBA_UARTFR_RXFE) == 0)
#define UART_TX_READY(s)	(((s) & AMBA_UARTFR_TXFF) == 0)
#define UART_TX_EMPTY(p)	((UART_GET_FR(p) & AMBA_UARTFR_TMSK) == 0)
#define UART_GET_IMSC(p)	__raw_readl((p)->membase + AMBA_UARTIMSC)
#define UART_PUT_IMSC(p, c)	__raw_writel((c), (p)->membase + AMBA_UARTIMSC)
#define UART_GET_IFLS(p)	__raw_readl((p)->membase + AMBA_UARTIFLS)
#define UART_PUT_IFLS(p, c)	__raw_writel((c), (p)->membase + AMBA_UARTIFLS)

#define UART_DUMMY_RSR_RX	256
#define UART_PORT_SIZE		64

/*
 * Our private driver data mappings.
 */
#define drv_old_status	driver_priv

static void UART_PL011_SetSpeed(struct uart_port *port, int speed)
{
    unsigned int temp;
    unsigned int divider;
    unsigned int remainder;
    unsigned int fraction;

    temp      = 16 * speed;
    divider   = port->uartclk / temp;
    remainder = port->uartclk % temp;
    temp      = (128 * remainder) / temp;
    fraction  = temp / 2;

    if (temp & 1)
        fraction++;

    UART_PUT_IBRD(port, divider);
    UART_PUT_FBRD(port, fraction);

    /*
    ** Must write LCRH after LCRM and/or LCRL.
    */
    UART_PUT_LCRH(port, UART_GET_LCRH(port));
}

static void versatileuart_stop_tx(struct uart_port *port, u_int from_tty)
{
    unsigned int imsc;

    imsc = UART_GET_IMSC(port);
    imsc &= ~AMBA_UARTIMSC_TXIM;
    imsc = UART_PUT_IMSC(port, imsc);

}

static void versatileuart_start_tx(struct uart_port *port, u_int nonempty, u_int from_tty)
{
    if (nonempty) {
            unsigned int imsc;

#ifdef VERSATILE_DEBUG
        printk("versatile_startx()\n");
#endif
        imsc = UART_GET_IMSC(port);
        imsc |= AMBA_UARTIMSC_TXIM;
        imsc = UART_PUT_IMSC(port, imsc);
    }
}

static void versatileuart_stop_rx(struct uart_port *port)
{
    unsigned int imsc;

    imsc = UART_GET_IMSC(port);
    imsc &= ~(AMBA_UARTIMSC_RTIM | AMBA_UARTIMSC_RXIM);
    imsc = UART_PUT_IMSC(port, imsc);
}

static void versatileuart_enable_ms(struct uart_port *port)
{
	unsigned int imsc;

	imsc = UART_GET_IMSC(port);
	imsc |= (AMBA_UARTIMSC_DSRMIM | AMBA_UARTIMSC_DCDMIM | 
                AMBA_UARTIMSC_CTSMIM | AMBA_UARTIMSC_RIMIM);
	imsc = UART_PUT_IMSC(port, imsc);
}

static void
#ifdef SUPPORT_SYSRQ
versatileuart_rx_chars(struct uart_info *info, struct pt_regs *regs)
#else
versatileuart_rx_chars(struct uart_info *info)
#endif
{
	struct tty_struct *tty = info->tty;
	unsigned int status, ch, rsr, max_count = 256;
	struct uart_port *port = info->port;

	status = UART_GET_FR(port);
	while (UART_RX_DATA(status) && max_count--) {
		if (tty->flip.count >= TTY_FLIPBUF_SIZE) {
			tty->flip.tqueue.routine((void *)tty);
			if (tty->flip.count >= TTY_FLIPBUF_SIZE) {
				printk(KERN_WARNING "TTY_DONT_FLIP set\n");
				return;
			}
		}

		ch = UART_GET_CHAR(port);

		*tty->flip.char_buf_ptr = ch;
		*tty->flip.flag_buf_ptr = TTY_NORMAL;
		port->icount.rx++;

		/*
		 * Note that the error handling code is
		 * out of the main execution path
		 */
		rsr = UART_GET_RSR(port) | UART_DUMMY_RSR_RX;
		if (rsr & AMBA_UARTRSR_ANY) {
			if (rsr & AMBA_UARTRSR_BE) {
				rsr &= ~(AMBA_UARTRSR_FE | AMBA_UARTRSR_PE);
				port->icount.brk++;
				if (uart_handle_break(info, &versatile_console))
					goto ignore_char;
			} else if (rsr & AMBA_UARTRSR_PE)
				port->icount.parity++;
			else if (rsr & AMBA_UARTRSR_FE)
				port->icount.frame++;
			if (rsr & AMBA_UARTRSR_OE)
				port->icount.overrun++;

			rsr &= port->read_status_mask;

			if (rsr & AMBA_UARTRSR_BE)
				*tty->flip.flag_buf_ptr = TTY_BREAK;
			else if (rsr & AMBA_UARTRSR_PE)
				*tty->flip.flag_buf_ptr = TTY_PARITY;
			else if (rsr & AMBA_UARTRSR_FE)
				*tty->flip.flag_buf_ptr = TTY_FRAME;
		}

		if (uart_handle_sysrq_char(info, ch, regs))
			goto ignore_char;

		if ((rsr & port->ignore_status_mask) == 0) {
			tty->flip.flag_buf_ptr++;
			tty->flip.char_buf_ptr++;
			tty->flip.count++;
		}
		if ((rsr & AMBA_UARTRSR_OE) &&
		    tty->flip.count < TTY_FLIPBUF_SIZE) {
			/*
			 * Overrun is special, since it's reported
			 * immediately, and doesn't affect the current
			 * character
			 */
			*tty->flip.char_buf_ptr++ = 0;
			*tty->flip.flag_buf_ptr++ = TTY_OVERRUN;
			tty->flip.count++;
		}
	ignore_char:
		status = UART_GET_FR(port);
	}
	tty_flip_buffer_push(tty);
	return;
}

static void versatileuart_tx_chars(struct uart_info *info)
{
	struct uart_port *port = info->port;
	int count;
        u_int status;

	if (port->x_char) {
		UART_PUT_CHAR(port, port->x_char);
		port->icount.tx++;
		port->x_char = 0;
		return;
	}
	if (info->xmit.head == info->xmit.tail
	    || info->tty->stopped
	    || info->tty->hw_stopped) {
		versatileuart_stop_tx(port, 0);
		return;
	}

	count = port->fifosize >> 1;
	do {
		do {
			status = UART_GET_FR(port);
		} while (!UART_TX_READY(status));
		UART_PUT_CHAR(port, info->xmit.buf[info->xmit.tail]);
		info->xmit.tail = (info->xmit.tail + 1) & (UART_XMIT_SIZE - 1);
		port->icount.tx++;
		if (info->xmit.head == info->xmit.tail)
			break;
	} while (--count > 0);

	if (CIRC_CNT(info->xmit.head, info->xmit.tail, UART_XMIT_SIZE) <
			WAKEUP_CHARS)
		uart_event(info, EVT_WRITE_WAKEUP);

	if (info->xmit.head == info->xmit.tail)
		versatileuart_stop_tx(info->port, 0);
}

static void versatileuart_modem_status(struct uart_info *info)
{
	struct uart_port *port = info->port;
	unsigned int status, delta;

	UART_PUT_ICR(port, 0);

	status = UART_GET_FR(port) & AMBA_UARTFR_MODEM_ANY;

	delta = status ^ info->drv_old_status;
	info->drv_old_status = status;

	if (!delta)
		return;

	if (delta & AMBA_UARTFR_DCD)
		uart_handle_dcd_change(info, status & AMBA_UARTFR_DCD);

	if (delta & AMBA_UARTFR_DSR)
		port->icount.dsr++;

	if (delta & AMBA_UARTFR_CTS)
		uart_handle_cts_change(info, status & AMBA_UARTFR_CTS);

	wake_up_interruptible(&info->delta_msr_wait);
}

static void versatileuart_int(int irq, void *dev_id, struct pt_regs *regs)
{
	struct uart_info *info = dev_id;
	unsigned int status, pass_counter = VERSATILE_ISR_PASS_LIMIT;

	status = UART_GET_INT_STATUS(info->port);
	do {
		if (status & (AMBA_UARTMIS_RT | AMBA_UARTMIS_RX))
#ifdef SUPPORT_SYSRQ
			versatileuart_rx_chars(info, regs);
#else
			versatileuart_rx_chars(info);
#endif
		if (status & AMBA_UARTMIS_TX)
			versatileuart_tx_chars(info);

		if (status & (AMBA_UARTMIS_DSRM | AMBA_UARTMIS_DCDM | 
                              AMBA_UARTMIS_CTSM | AMBA_UARTMIS_RIM))
			versatileuart_modem_status(info);

		if (pass_counter-- == 0)
			break;

		status = UART_GET_INT_STATUS(info->port);
	} while (status & (AMBA_UARTMIS_RT | AMBA_UARTMIS_RX |
			   AMBA_UARTMIS_TX));
}

static u_int versatileuart_tx_empty(struct uart_port *port)
{
	return UART_GET_FR(port) & AMBA_UARTFR_BUSY ? 0 : TIOCSER_TEMT;
}

static u_int versatileuart_get_mctrl(struct uart_port *port)
{
	unsigned int result = 0;
	unsigned int status;

	status = UART_GET_FR(port);
	if (status & AMBA_UARTFR_DCD)
		result |= TIOCM_CAR;
	if (status & AMBA_UARTFR_DSR)
		result |= TIOCM_DSR;
	if (status & AMBA_UARTFR_CTS)
		result |= TIOCM_CTS;

	return result;
}

static void versatileuart_set_mctrl(struct uart_port *port, u_int mctrl)
{
	u_int ctrl;

#ifdef VERSATILE_DEBUG
        printk("versatile_set_mctrl(0x%x), cr = 0x%x\n", mctrl, UART_GET_CR(port));
#endif
        ctrl = UART_GET_CR(port);

	if (mctrl & TIOCM_RTS)
                ctrl |= AMBA_UARTCR_RTS;
	else
		ctrl &= ~AMBA_UARTCR_RTS;

	if (mctrl & TIOCM_DTR)
		ctrl |= AMBA_UARTCR_DTR;
	else
		ctrl &= ~AMBA_UARTCR_DTR;

        UART_PUT_CR(port, ctrl);
}

static void versatileuart_break_ctl(struct uart_port *port, int break_state)
{
	unsigned int lcr_h;

	lcr_h = UART_GET_LCRH(port);
	if (break_state == -1)
		lcr_h |= AMBA_UARTLCR_H_BRK;
	else
		lcr_h &= ~AMBA_UARTLCR_H_BRK;
	UART_PUT_LCRH(port, lcr_h);
}

static int versatileuart_startup(struct uart_port *port, struct uart_info *info)
{
	int retval;
        u_int ctrl = 0;

#ifdef VERSATILE_DEBUG
        printk("startup(irq = %d)\n", port->irq);
#endif
	/*
	 * Allocate the IRQ
	 */
	retval = request_irq(port->irq, versatileuart_int, 0, "amba_versatile", info);
	if (retval)
		return retval;

	/*
	 * initialise the old status of the modem signals
	 */
	info->drv_old_status = UART_GET_FR(port) & AMBA_UARTFR_MODEM_ANY;
        ctrl = UART_GET_LCRH(port);
        ctrl |= (3 << AMBA_UARTLCR_H_WLEN_IDX) | AMBA_UARTLCR_H_FEN;
        UART_PUT_LCRH(port, ctrl);

	UART_PUT_CR(port, AMBA_UARTCR_UARTEN | AMBA_UARTCR_RXE | AMBA_UARTCR_TXE);
	/*
	 * Finally, enable interrupts
	 */
        UART_PUT_ICR(port, 0xFFFF);
        UART_PUT_IMSC(port, (AMBA_UARTIMSC_RXIM | AMBA_UARTIMSC_RTIM));

        switch (port->irq) {
	case INT_UARTINT0:
            versatile_uart_write("  ", 2, 0);
	    break;
	case INT_UARTINT1:
            versatile_uart_write("  ", 2, 1);
	    break;
	case INT_UARTINT2:
            versatile_uart_write("  ", 2, 2);
	    break;
	}

	return 0;
}

static void versatileuart_shutdown(struct uart_port *port, struct uart_info *info)
{
#ifdef VERSATILE_DEBUG
        printk("shutdown()\n");
#endif
	/*
	 * Free the interrupt
	 */
	free_irq(port->irq, info);

	/*
	 * disable all interrupts, disable the port
	 */
	UART_PUT_IMSC(port, 0);

	/* disable break condition and fifos */
	UART_PUT_LCRH(port, UART_GET_LCRH(port) &
		~(AMBA_UARTLCR_H_BRK | AMBA_UARTLCR_H_FEN));
}

static void versatileuart_change_speed(struct uart_port *port, u_int cflag, u_int iflag, u_int quot)
{
	u_int lcr_h, old_imsc;
        u_int temp, divider, remainder, fraction;
	unsigned long flags;

#ifdef VERSATILE_DEBUG
	printk("speed(cflag = 0x%x, speed = %d)\n", cflag, quot);
#endif
	/* byte size and parity */
#ifdef FIXME
	switch (cflag & CSIZE) {
	case CS5: lcr_h = AMBA_UARTLCR_H_WLEN_5; break;
	case CS6: lcr_h = AMBA_UARTLCR_H_WLEN_6; break;
	case CS7: lcr_h = AMBA_UARTLCR_H_WLEN_7; break;
	default:  lcr_h = AMBA_UARTLCR_H_WLEN_8; break; // CS8
	}
#endif
        lcr_h = AMBA_UARTLCR_H_WLEN_8;  /* FIXME */

	if (cflag & CSTOPB)
		lcr_h |= AMBA_UARTLCR_H_STP2;
	if (cflag & PARENB) {
		lcr_h |= AMBA_UARTLCR_H_PEN;
		if (!(cflag & PARODD))
			lcr_h |= AMBA_UARTLCR_H_EPS;
	}
	if (port->fifosize > 1)
		lcr_h |= AMBA_UARTLCR_H_FEN;

	port->read_status_mask = AMBA_UARTRSR_OE;
	if (iflag & INPCK)
		port->read_status_mask |= AMBA_UARTRSR_FE | AMBA_UARTRSR_PE;
	if (iflag & (BRKINT | PARMRK))
		port->read_status_mask |= AMBA_UARTRSR_BE;

	/*
	 * Characters to ignore
	 */
	port->ignore_status_mask = 0;
	if (iflag & IGNPAR)
		port->ignore_status_mask |= AMBA_UARTRSR_FE | AMBA_UARTRSR_PE;
	if (iflag & IGNBRK) {
		port->ignore_status_mask |= AMBA_UARTRSR_BE;
		/*
		 * If we're ignoring parity and break indicators,
		 * ignore overruns too (for real raw support).
		 */
		if (iflag & IGNPAR)
			port->ignore_status_mask |= AMBA_UARTRSR_OE;
	}

	/*
	 * Ignore all characters if CREAD is not set.
	 */
	if ((cflag & CREAD) == 0)
		port->ignore_status_mask |= UART_DUMMY_RSR_RX;

	/* first, disable everything */
	
	save_flags(flags); cli();
	old_imsc = UART_GET_IMSC(port);

	if ((port->flags & ASYNC_HARDPPS_CD) ||
	    (cflag & CRTSCTS) || !(cflag & CLOCAL)) {
		old_imsc |= AMBA_UARTIMSC_CTSMIM;
        }

	UART_PUT_IMSC(port, 0);

	/* 
         * Set baud rate.
         *   quot = uartclk / (16 * baud) [cf. serial_core.c]
         */

        temp = port->uartclk / quot;
        divider = quot;
        remainder = port->uartclk % temp;
        temp = (128 * remainder) / temp;
        fraction = temp / 2;

        if ( temp & 1 ) 
            fraction++;

        UART_PUT_IBRD(port, divider);

        UART_PUT_FBRD(port, fraction);

	/*
	 * ----------v----------v----------v----------v-----
	 * NOTE: MUST BE WRITTEN AFTER UARTLCR_M & UARTLCR_L
	 * ----------^----------^----------^----------^-----
	 */

	UART_PUT_LCRH(port, lcr_h);
	UART_PUT_IMSC(port, old_imsc);

	restore_flags(flags);
}

static const char *versatileuart_type(struct uart_port *port)
{
	return port->type == PORT_CINTEGRATOR ? "VERSATILE" : NULL;
}

/*
 * Release the memory region(s) being used by 'port'
 */
static void versatileuart_release_port(struct uart_port *port)
{
	release_mem_region(port->mapbase, UART_PORT_SIZE);
}

/*
 * Request the memory region(s) being used by 'port'
 */
static int versatileuart_request_port(struct uart_port *port)
{
	return request_mem_region(port->mapbase, UART_PORT_SIZE, "versatile_amba")
			!= NULL ? 0 : -EBUSY;
}

/*
 * Configure/autoconfigure the port.
 */
static void versatileuart_config_port(struct uart_port *port, int flags)
{
	if (flags & UART_CONFIG_TYPE) {
		port->type = PORT_CINTEGRATOR;
		versatileuart_request_port(port);
	}
}

/*
 * verify the new serial_struct (for TIOCSSERIAL).
 */
static int versatileuart_verify_port(struct uart_port *port, struct serial_struct *ser)
{
	int ret = 0;
	if (ser->type != PORT_UNKNOWN && ser->type != PORT_CINTEGRATOR)
		ret = -EINVAL;
	if (ser->irq < 0 || ser->irq >= NR_IRQS)
		ret = -EINVAL;
	if (ser->baud_base < 9600)
		ret = -EINVAL;
	return ret;
}

static struct uart_ops amba_pops = {
	tx_empty:	versatileuart_tx_empty,
	set_mctrl:	versatileuart_set_mctrl,
	get_mctrl:	versatileuart_get_mctrl,
	stop_tx:	versatileuart_stop_tx,
	start_tx:	versatileuart_start_tx,
	stop_rx:	versatileuart_stop_rx,
	enable_ms:	versatileuart_enable_ms,
	break_ctl:	versatileuart_break_ctl,
	startup:	versatileuart_startup,
	shutdown:	versatileuart_shutdown,
	change_speed:	versatileuart_change_speed,
	type:		versatileuart_type,
	release_port:	versatileuart_release_port,
	request_port:	versatileuart_request_port,
	config_port:	versatileuart_config_port,
	verify_port:	versatileuart_verify_port,
};

static struct uart_port amba_ports[UART_NR] = {
	{
		membase:	(void *)IO_ADDRESS(VERSATILE_UART0_BASE),
		mapbase:	VERSATILE_UART0_BASE,
		iotype:		SERIAL_IO_MEM,
		irq:		INT_UARTINT0,
		uartclk:	24000000,
		fifosize:	16,
		unused:		{ 4, 5 }, /*driver_priv:	PORT_CTRLS(5, 4), */
		ops:		&amba_pops,
		flags:		ASYNC_BOOT_AUTOCONF,
	},
	{
		membase:	(void *)IO_ADDRESS(VERSATILE_UART1_BASE),
		mapbase:	VERSATILE_UART1_BASE,
		iotype:		SERIAL_IO_MEM,
		irq:		INT_UARTINT1,
		uartclk:	24000000,
		fifosize:	16,
		unused:		{ 6, 7 }, /*driver_priv:	PORT_CTRLS(7, 6), */
		ops:		&amba_pops,
		flags:		ASYNC_BOOT_AUTOCONF,
	},
	{
		membase:	(void *)IO_ADDRESS(VERSATILE_UART2_BASE),
		mapbase:	VERSATILE_UART2_BASE,
		iotype:		SERIAL_IO_MEM,
		irq:		INT_UARTINT2,
		uartclk:	24000000,
		fifosize:	16,
		unused:		{ 8, 9 }, /*driver_priv:	PORT_CTRLS(8, 9), */
		ops:		&amba_pops,
		flags:		ASYNC_BOOT_AUTOCONF,
	}
};

#ifdef CONFIG_SERIAL_VERSATILE_PB_CONSOLE
#ifdef used_and_not_const_char_pointer
static int versatileuart_console_read(struct uart_port *port, char *s, u_int count)
{
	unsigned int status;
	int c;
#ifdef VERSATILE_DEBUG
	printk("versatile_console_read() called\n");
#endif

	c = 0;
	while (c < count) {
		status = UART_GET_FR(port);
		if (UART_RX_DATA(status)) {
			*s++ = UART_GET_CHAR(port);
			c++;
		} else {
			// nothing more to get, return
			return c;
		}
	}
	// return the count
	return c;
}
#endif

static void versatileuart_console_write(struct console *co, const char *s, u_int count)
{
	struct uart_port *port = amba_ports + co->index;
        unsigned int status, old_cr;
	int i;

	/*
	 *	First save the CR then disable the interrupts
	 */
	UART_PL011_SetSpeed(port, 38400);
	UART_PUT_LCRH(port, (3 << AMBA_UARTLCR_H_WLEN_IDX)); // | AMBA_UARTLCR_H_FEN);

	old_cr = UART_GET_CR(port);
	UART_PUT_CR(port, AMBA_UARTCR_UARTEN | AMBA_UARTCR_TXE);

	/*
	 *	Now, do each character
	 */
	for (i = 0; i < count; i++) {
		do {
			status = UART_GET_FR(port);
		} while (!UART_TX_READY(status));
		UART_PUT_CHAR(port, s[i]);
		if (s[i] == '\n') {
			do {
				status = UART_GET_FR(port);
			} while (!UART_TX_READY(status));
			UART_PUT_CHAR(port, '\r');
		}
	}

	/*
	 *	Finally, wait for transmitter to become empty
	 *	and restore the TCR
	 */
	do {
		status = UART_GET_FR(port);
	} while (!UART_TX_READY(status));

	UART_PUT_CR(port, old_cr);
}

static void versatile_uart_write(const char *s, u_int count, int port_nb)
{
	struct uart_port *port = &amba_ports[port_nb];
        unsigned int status, old_cr;
	int i;

	/*
	 *	First save the CR then disable the interrupts
	 */
        UART_PL011_SetSpeed(port, 38400);
        UART_PUT_LCRH(port, (3 << AMBA_UARTLCR_H_WLEN_IDX));
	old_cr = UART_GET_CR(port);
	UART_PUT_CR(port, old_cr | (AMBA_UARTCR_UARTEN | AMBA_UARTCR_TXE));

	/*
	 *	Now, do each character
	 */
	for (i = 0; i < count; i++) {
		do {
			status = UART_GET_FR(port);
		} while (!UART_TX_READY(status));
		UART_PUT_CHAR(port, s[i]);
		if (s[i] == '\n') {
			do {
				status = UART_GET_FR(port);
			} while (!UART_TX_READY(status));
			UART_PUT_CHAR(port, '\r');
		}
	}

	/*
	 *	Finally, wait for transmitter to become empty
	 *	and restore the TCR
	 */
	do {
		status = UART_GET_FR(port);
	} while (!UART_TX_READY(status));
}

static kdev_t versatileuart_console_device(struct console *co)
{
	return MKDEV(SERIAL_VERSATILE_MAJOR, SERIAL_VERSATILE_MINOR + co->index);
}

static int versatileuart_console_wait_key(struct console *co)
{
	struct uart_port *port = amba_ports + co->index;
	unsigned int status;

	do {
		status = UART_GET_FR(port);
	} while (!UART_RX_DATA(status));
	return UART_GET_CHAR(port);
}

static void __init
versatileuart_console_get_options(struct uart_port *port, int *baud, int *parity, int *bits)
{
	if (UART_GET_CR(port) & AMBA_UARTCR_UARTEN) {
		u_int lcr_h, quot;
		lcr_h = UART_GET_LCRH(port);

		*parity = 'n';
		if (lcr_h & AMBA_UARTLCR_H_PEN) {
			if (lcr_h & AMBA_UARTLCR_H_EPS)
				*parity = 'e';
			else
				*parity = 'o';
		}

		if ((lcr_h & AMBA_UARTLCR_H_WLEN_MSK) == AMBA_UARTLCR_H_WLEN_7)
			*bits = 7;
		else
			*bits = 8;

                quot = UART_GET_IBRD(port);
		*baud = port->uartclk / (16 * quot);
                printk("console_ get_options: baud = %d\n", *baud);
	}
}

static int __init versatileuart_console_setup(struct console *co, char *options)
{
	struct uart_port *port;
	int baud = 38400;
	int bits = 8;
	int parity = 'n';
	int flow = 'n';

	/*
	 * Check whether an invalid uart number has been specified, and
	 * if so, search for the first available port that does have
	 * console support.
	 */
	port = uart_get_console(amba_ports, UART_NR, co);

	if (options)
		uart_parse_options(options, &baud, &parity, &bits, &flow);
	else
		versatileuart_console_get_options(port, &baud, &parity, &bits);

	return uart_set_options(port, co, baud, parity, bits, flow);
}

static struct console versatile_console = {
	name:		"ttyCM",
	write:		versatileuart_console_write,
#ifdef used_and_not_const_char_pointer
	read:		versatileuart_console_read,
#endif
	device:		versatileuart_console_device,
/*	wait_key:	versatileuart_console_wait_key, */
	setup:		versatileuart_console_setup,
	flags:		CON_PRINTBUFFER,
	index:		-1,
};

void __init versatileuart_console_init(void)
{
	register_console(&versatile_console);
}

#define VERSATILE_CONSOLE	&versatile_console
#else
#define VERSATILE_CONSOLE	NULL
#endif

static struct uart_driver versatile_reg = {
	owner:			THIS_MODULE,
	normal_major:		SERIAL_VERSATILE_MAJOR,
#ifdef CONFIG_DEVFS_FS
	normal_name:		"ttyCM%d",
	callout_name:		"cuacm%d",
#else
	normal_name:		"ttyCM",
	callout_name:		"cuacm",
#endif
	normal_driver:		&normal,
	callout_major:		CALLOUT_AMBA_MAJOR,
	callout_driver:		&callout,
	table:			versatile_table,
	termios:		versatile_termios,
	termios_locked:		versatile_termios_locked,
	minor:			SERIAL_VERSATILE_MINOR,
	nr:			UART_NR,
	port:			amba_ports,
	cons:			VERSATILE_CONSOLE,
};

static int __init versatileuart_init(void)
{
	return uart_register_driver(&versatile_reg);
}

static void __exit versatileuart_exit(void)
{
	uart_unregister_driver(&versatile_reg);
}

module_init(versatileuart_init);
module_exit(versatileuart_exit);

EXPORT_NO_SYMBOLS;

MODULE_AUTHOR("ARM Ltd");
MODULE_DESCRIPTION("ARM Compact Integrator serial port driver");
MODULE_LICENSE("GPL");
