/*
 * linux/drivers/char/serial_pnx0105.c
 *
 * Driver for Philips IP_3106 VPB UART
 *
 * Based on drivers/char/serial.c, by Linus Torvalds, Theodore Ts'o.
 *
 * Copyright (C) 2003 Philips Semiconductors
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

//-----------------------------------------------------------------------------
// Standard include files:
//-----------------------------------------------------------------------------

#include <linux/config.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/major.h>
#include <linux/string.h>
#include <linux/fcntl.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/circ_buf.h>
#include <linux/serial.h>
#include <linux/console.h>
#include <linux/sysrq.h>
#include <asm/system.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/uaccess.h>
#include <asm/bitops.h>

#include <linux/serial_core.h>

//-----------------------------------------------------------------------------
// Project include files:
//-----------------------------------------------------------------------------

#include <asm/hardware/serial_pnx0105.h>

//-----------------------------------------------------------------------------
// Types and defines:
//-----------------------------------------------------------------------------

//#define IP3106_THRE_IRQ_PROBLEM_EXISTS    1


#define VERSION_STR                  "1.0"


#ifdef IP3106_DEBUG
#define IP3106_MSG(string, args...)  printk(KERN_DEBUG "IP3106 - " string, ##args)
#else
#define IP3106_MSG(string, args...)
#endif // #ifdef IP3106_DEBUG


#define IP3106_UART_MEMMAP_SIZE      (SZ_4K)

#define IP3106_UART_MAJOR            204
#define IP3106_CALLOUT_MAJOR         205
#define IP3106_UART_MINOR            48
#define IP3106_NR_OF_UARTS           1            // Only one UART supported for PNX0105

#define IP3106_UART_CLOCK_FREQ       66000000     // Uart clk is set to 66MHz for PNX0105 board
#define IP3106_UART_FIFO_SIZE        64           // fifo size for UART type 16750

#define IP3106_RX_TRIG_LEV_ZERO      0
#define IP3106_RX_TRIG_LEV_LOW       1
#define IP3106_RX_TRIG_LEV_MEDIUM    2
#define IP3106_RX_TRIG_LEV_HIGH      3

#define IP3106_BAUDRATE_NORMAL       1
#define IP3106_BAUDRATE_ACCURATE     2

#define IP3106_UART_LSR_ERROR_ANY    (IP3106_UART_LSR_BI | IP3106_UART_LSR_FE | IP3106_UART_LSR_PE | IP3106_UART_LSR_OE)
#define IP3106_UART_MSR_MODEM_ANY    (IP3106_UART_MSR_DCD | IP3106_UART_MSR_DSR | IP3106_UART_MSR_CTS)
#define IP3106_UART_DUMMY_LSR_RX     0x100

// Access macros for the IP_3106 UART

#define IP3106_UART_GET_CHAR(p)      readb  (     (p)->membase + IP3106_UART_RBR_REG)
#define IP3106_UART_PUT_CHAR(p,c)    writel ((c), (p)->membase + IP3106_UART_THR_REG)
#define IP3106_UART_GET_DLL(p)       readb  (     (p)->membase + IP3106_UART_DLL_REG)
#define IP3106_UART_PUT_DLL(p,c)     writel ((c), (p)->membase + IP3106_UART_DLL_REG)
#define IP3106_UART_GET_IER(p)       readb  (     (p)->membase + IP3106_UART_IER_REG)
#define IP3106_UART_PUT_IER(p,c)     writel ((c), (p)->membase + IP3106_UART_IER_REG)
#define IP3106_UART_GET_DLM(p)       readb  (     (p)->membase + IP3106_UART_DLM_REG)
#define IP3106_UART_PUT_DLM(p,c)     writel ((c), (p)->membase + IP3106_UART_DLM_REG)
#define IP3106_UART_GET_IIR(p)       readb  (     (p)->membase + IP3106_UART_IIR_REG)
#define IP3106_UART_PUT_FCR(p,c)     writel ((c), (p)->membase + IP3106_UART_FCR_REG)
#define IP3106_UART_GET_LCR(p)       readb  (     (p)->membase + IP3106_UART_LCR_REG)
#define IP3106_UART_PUT_LCR(p,c)     writel ((c), (p)->membase + IP3106_UART_LCR_REG)
#define IP3106_UART_GET_MCR(p)       readb  (     (p)->membase + IP3106_UART_MCR_REG)
#define IP3106_UART_PUT_MCR(p,c)     writel ((c), (p)->membase + IP3106_UART_MCR_REG)
#define IP3106_UART_GET_LSR(p)       readb  (     (p)->membase + IP3106_UART_LSR_REG)
#define IP3106_UART_GET_MSR(p)       readb  (     (p)->membase + IP3106_UART_MSR_REG)
#define IP3106_UART_GET_FDR(p)       readb  (     (p)->membase + IP3106_UART_FDR_REG)
#define IP3106_UART_PUT_FDR(p,c)     writel ((c), (p)->membase + IP3106_UART_FDR_REG)
#define IP3106_UART_GET_CFG(p)       readb  (     (p)->membase + IP3106_UART_CFG_REG)
#define IP3106_UART_GET_MID(p)       readb  (     (p)->membase + IP3106_UART_MID_REG)

// Our private driver data mappings
#define drv_old_status	driver_priv


typedef struct _ip3106_baud_settings
{
    unsigned int baud_rate;
    unsigned int mulval;
    unsigned int divaddval;
    unsigned int divisor;
} ip3106_baud_settings_t;


//-----------------------------------------------------------------------------
// Internal function prototypes:
//-----------------------------------------------------------------------------

// irq functions

static void irq_handler_for_ip3106   (int irq, void *dev_id, struct pt_regs *regs);
static void irq_handler_rx_chars     (struct uart_info *info, struct pt_regs *regs);
static void irq_handler_tx_chars     (struct uart_info *info);
static void irq_handler_modem_status (struct uart_info *info);

// functions

static void         ip3106_stop_tx      (struct uart_port *port, unsigned int from_tty);
static void         ip3106_start_tx     (struct uart_port *port, unsigned int nonempty, unsigned int from_tty);
static void         ip3106_stop_rx      (struct uart_port *port);
static void         ip3106_enable_ms    (struct uart_port *port);
static unsigned int ip3106_tx_empty     (struct uart_port *port);
static unsigned int ip3106_get_mctrl    (struct uart_port *port);
static void         ip3106_set_mctrl    (struct uart_port *port, unsigned int mctrl);
static void         ip3106_break_ctl    (struct uart_port *port, int break_state);
static int          ip3106_startup      (struct uart_port *port, struct uart_info *info);
static void         ip3106_shutdown     (struct uart_port *port, struct uart_info *info);
static void         ip3106_change_speed (struct uart_port *port, unsigned int cflag, unsigned int iflag, unsigned int quot);
static void         get_baud_settings   (struct uart_port *port, unsigned int quot,
                                         unsigned int *pmulval, unsigned int *pdivaddval, unsigned int *pdivisor);
static const char  *ip3106_type         (struct uart_port *port);
static void         ip3106_release_port (struct uart_port *port);
static int          ip3106_request_port (struct uart_port *port);
static void         ip3106_config_port  (struct uart_port *port, int flags);
static int          ip3106_verify_port  (struct uart_port *port, struct serial_struct *ser);

// console functions

#ifdef CONFIG_SERIAL_PNX0105_CONSOLE
#ifdef used_and_not_const_char_pointer
static int         ip3106_console_read        (struct uart_port *port, char *s, unsigned int count);
#endif // #ifdef used_and_not_const_char_pointer
static void        ip3106_console_write       (struct console *co, const char *s, unsigned int count);
static kdev_t      ip3106_console_device      (struct console *co);
static int         ip3106_console_wait_key    (struct console *co);
static int __init  ip3106_console_setup       (struct console *co, char *options);
static void __init ip3106_console_get_options (struct uart_port *port, int *baud, int *parity, int *bits);
#endif // #ifdef CONFIG_SERIAL_PNX0105_CONSOLE

//-----------------------------------------------------------------------------
// Global data:
//-----------------------------------------------------------------------------

static struct uart_ops ip3106_pops =
{
    stop_tx:         ip3106_stop_tx,
    start_tx:        ip3106_start_tx,
    stop_rx:         ip3106_stop_rx,
    enable_ms:       ip3106_enable_ms,
    tx_empty:        ip3106_tx_empty,
    get_mctrl:       ip3106_get_mctrl,
    set_mctrl:       ip3106_set_mctrl,
    break_ctl:       ip3106_break_ctl,
    startup:         ip3106_startup,
    shutdown:        ip3106_shutdown,
    change_speed:    ip3106_change_speed,
    type:            ip3106_type,
    release_port:    ip3106_release_port,
    request_port:    ip3106_request_port,
    config_port:     ip3106_config_port,
    verify_port:     ip3106_verify_port,
};


static struct tty_driver  normal;
static struct tty_driver  callout;
static struct tty_struct  *ip3106_table;
static struct termios     *ip3106_termios;
static struct termios     *ip3106_termios_locked;


static struct uart_port ip3106_ports =
{
    membase:     (void *)IO_ADDRESS(UART_BASE),
    mapbase:     UART_BASE,
    iotype:      SERIAL_IO_MEM,
    irq:         IRQ_UART,
    uartclk:     IP3106_UART_CLOCK_FREQ,
    fifosize:    IP3106_UART_FIFO_SIZE,
    unused:      {0, IP3106_RX_TRIG_LEV_ZERO, IP3106_BAUDRATE_ACCURATE}, // private data
    ops:         &ip3106_pops,
    flags:       ASYNC_BOOT_AUTOCONF,
};


#ifdef CONFIG_SERIAL_PNX0105_CONSOLE
static struct console ip3106_console =
{
    name:            "ttyPU",
    write:           ip3106_console_write,
#ifdef used_and_not_const_char_pointer
    read:            ip3106_console_read,
#endif // #ifdef used_and_not_const_char_pointer
    device:          ip3106_console_device,
//    wait_key:        ip3106_console_wait_key,
    setup:           ip3106_console_setup,
    flags:           CON_PRINTBUFFER,
    index:           -1,
};
#define IP3106_CONSOLE    &ip3106_console
#else
#define IP3106_CONSOLE    NULL
#endif // #ifdef CONFIG_SERIAL_PNX0105_CONSOLE


static struct uart_driver ip3106_reg =
{
    owner:           THIS_MODULE,
    normal_major:    IP3106_UART_MAJOR,
    normal_driver:   &normal,
    callout_major:   IP3106_CALLOUT_MAJOR,
    callout_driver:  &callout,
#ifdef CONFIG_DEVFS_FS
    normal_name:     "ttyPU%d",
    callout_name:    "cuapu%d",
#else
    normal_name:     "ttyPU",
    callout_name:    "cuapu",
#endif // #ifdef CONFIG_DEVFS_FS
    minor:           IP3106_UART_MINOR,
    nr:              IP3106_NR_OF_UARTS,
    table:           &ip3106_table,
    termios:         &ip3106_termios,
    termios_locked:  &ip3106_termios_locked,
    port:            &ip3106_ports,
    cons:            IP3106_CONSOLE,
};


// Caution! Adding a baud-rate to this table is not 'fail safe'.
// Make sure that the integer outcome of the next calculation, the quotient,
// is unique in this table:    quotient = uart_clk_freq / (16 * baud_rate)
static ip3106_baud_settings_t baud_reg_table[] =
{
//   This table is for a UART clock frequency of 66MHz
//   baud_rate, mulval, divaddval, divisor
    {      50,      1,        1,    41250},
    {      75,      3,        1,    41250},
    {     110,     11,        9,    20625},
    {     134,      3,        4,    13144},
    {     150,      3,        1,    20625},
    {     300,      3,        1,    10313},
    {     600,      3,        2,     4125},
    {    1200,      3,        2,     2063},
    {    1800,      9,        1,     2063},
    {    2000,      1,        0,     2063},
    {    2400,     12,       13,      825},
    {    3600,      5,        2,      818},
    {    4800,     12,       13,      413},
    {    7200,      5,        2,      409},
    {    9600,      5,        2,      307},
    {   19200,     12,       13,      103},
    {   38400,      8,       12,       43},
    {   56000,      7,        5,       43},
    {   57600,      7,        1,       63},
    {  112000,      7,        6,       20},
    {  115200,      5,       13,       10},
    {  230400,     14,       11,       10},
    {  460800,     14,        4,        7},
    {  921600,      2,        1,        3},
    { 1048576,     10,        3,        3}
};


//-----------------------------------------------------------------------------
// Local functions:
//-----------------------------------------------------------------------------

static void irq_handler_for_ip3106 (int irq, void *dev_id, struct pt_regs *regs)
{
	struct uart_info *info = dev_id;
    unsigned int  uart_irq_status = 0;
    unsigned char uart_int_status_field = 0;

    uart_irq_status = IP3106_UART_GET_IIR(info->port);
    if (!(uart_irq_status & IP3106_UART_IIR_IS))
    {
        uart_int_status_field = uart_irq_status & IP3106_UART_IIR_IID_MASK;
        if (uart_int_status_field == IP3106_UART_IIR_IID_RLS)        // highest priority
        {
            // the interrupt IP3106_UART_IER_RLSIE should not be enabled
            // because the IP3106_UART_IIR_IID_RDA interrupt will handle it all !!!!
            printk(KERN_WARNING "IP3106 - RLS interrupt should be disabled\n");
        }
        else if (uart_int_status_field == IP3106_UART_IIR_IID_RDA)
        {
            irq_handler_rx_chars (info, regs);
        }
        else if (uart_int_status_field == IP3106_UART_IIR_IID_CTO)
        {
            // this timeout is only used with auto baud-rate detection.
            // since this is not yet supported in this driver
            //      ===> do nothing
        }
        else if (uart_int_status_field == IP3106_UART_IIR_IID_THRE)
        {
            irq_handler_tx_chars (info);
        }
        else if (uart_int_status_field == IP3106_UART_IIR_IID_MS)    // lowest priority
        {
            irq_handler_modem_status (info);
        }
        else
        {
            // this should never happen
            IP3106_MSG ("irq_handler_for_ip3106, unknown irq status 0x%x\n", uart_irq_status);
        }
    }
}

//-----------------------------------------------------------------------------

static void do_hid_tasklet(unsigned long);
DECLARE_TASKLET(hid_tasklet, do_hid_tasklet, 0);

void do_hid_tasklet(unsigned long unused)
{
    unsigned int ier_reg = 0;

    ier_reg = IP3106_UART_GET_IER(&ip3106_ports);
    IP3106_UART_PUT_IER(&ip3106_ports, ier_reg | IP3106_UART_IER_RDAIE);
}

static void irq_handler_rx_chars (struct uart_info *info, struct pt_regs *regs)
{
    struct uart_port *port    = info->port;
    struct tty_struct *tty    = info->tty;
    unsigned int max_count    = 256;
    volatile unsigned int lsr_reg      = 0;
    unsigned int rx_char      = 0;
    unsigned int error_status = 0;
    unsigned int ier_reg = 0;

    ier_reg = IP3106_UART_GET_IER(port);
    IP3106_UART_PUT_IER(port, ier_reg & ~IP3106_UART_IER_RDAIE);

    while (((lsr_reg = IP3106_UART_GET_LSR(port)) & IP3106_UART_LSR_DR) && max_count--)
    {
        if (tty->flip.count >= TTY_FLIPBUF_SIZE)
        {
            tty->flip.tqueue.routine((void *)tty);
            if (tty->flip.count >= TTY_FLIPBUF_SIZE)
            {
                tasklet_schedule(&hid_tasklet);
                return; //continue;
            }
        }

        rx_char = IP3106_UART_GET_CHAR(port);

        *tty->flip.char_buf_ptr = rx_char;
        *tty->flip.flag_buf_ptr = TTY_NORMAL;
        port->icount.rx++;

        // Note that the error handling code is
        // out of the main execution path
        error_status = (lsr_reg & IP3106_UART_LSR_ERROR_ANY) | IP3106_UART_DUMMY_LSR_RX;
        if (error_status & IP3106_UART_LSR_ERROR_ANY)
        {
            if (error_status & IP3106_UART_LSR_BI)
            {
                error_status &= ~(IP3106_UART_LSR_FE | IP3106_UART_LSR_PE);
                port->icount.brk++;
                if (uart_handle_break(info, &ip3106_console))
                    continue;
            }
            else if (error_status & IP3106_UART_LSR_PE)
                port->icount.parity++;
            else if (error_status & IP3106_UART_LSR_FE)
                port->icount.frame++;

            if (error_status & IP3106_UART_LSR_OE)
            {
                port->icount.overrun++;
            }

            error_status &= port->read_status_mask;

            if (error_status & IP3106_UART_LSR_BI)
                *tty->flip.flag_buf_ptr = TTY_BREAK;
            else if (error_status & IP3106_UART_LSR_PE)
                *tty->flip.flag_buf_ptr = TTY_PARITY;
            else if (error_status & IP3106_UART_LSR_FE)
                *tty->flip.flag_buf_ptr = TTY_FRAME;
        }

        if (uart_handle_sysrq_char(info, rx_char, regs))
        {
            continue;
        }

        if ((error_status & port->ignore_status_mask) == 0)
        {
            tty->flip.flag_buf_ptr++;
            tty->flip.char_buf_ptr++;
            tty->flip.count++;
        }

        if ((error_status & IP3106_UART_LSR_OE) &&
            tty->flip.count < TTY_FLIPBUF_SIZE)
        {
			// Overrun is special, since it's reported
			// immediately, and doesn't affect the current
			// character
            *tty->flip.char_buf_ptr++ = 0;
            *tty->flip.flag_buf_ptr++ = TTY_OVERRUN;
            tty->flip.count++;
        }

    }
    tty_flip_buffer_push(tty);
    IP3106_UART_PUT_IER(port, ier_reg);
}

//-----------------------------------------------------------------------------

static void irq_handler_tx_chars (struct uart_info *info)
{
    struct uart_port *port = info->port;
    int count = 0;

    if (port->x_char)
    {
        IP3106_UART_PUT_CHAR(port, port->x_char);
        port->icount.tx++;
        port->x_char = 0;
        return;
    }
    if (info->xmit.head == info->xmit.tail ||
        info->tty->stopped ||
        info->tty->hw_stopped)
    {
        ip3106_stop_tx(port, 0);
        return;
    }

    count = port->fifosize;
    do
    {
        IP3106_UART_PUT_CHAR(port, info->xmit.buf[info->xmit.tail]);
        info->xmit.tail = (info->xmit.tail + 1) & (UART_XMIT_SIZE - 1);
        port->icount.tx++;
        if (info->xmit.head == info->xmit.tail)
            break;
    } while (--count > 0);

    if (CIRC_CNT(info->xmit.head, info->xmit.tail, UART_XMIT_SIZE) < WAKEUP_CHARS)
        uart_event(info, EVT_WRITE_WAKEUP);

    if (info->xmit.head == info->xmit.tail)
        ip3106_stop_tx(info->port, 0);
}

//-----------------------------------------------------------------------------

static void irq_handler_modem_status (struct uart_info *info)
{
    struct uart_port *port = info->port;
    unsigned int status = 0;
    unsigned int delta = 0;

    status = IP3106_UART_GET_MSR(port) & IP3106_UART_MSR_MODEM_ANY;

    delta = status ^ info->drv_old_status;
    info->drv_old_status = status;

    if (!delta)
        return;

    if (delta & IP3106_UART_MSR_DCD)
        uart_handle_dcd_change (info, status & IP3106_UART_MSR_DCD);

    if (delta & IP3106_UART_MSR_DSR)
        port->icount.dsr++;

    if (delta & IP3106_UART_MSR_CTS)
        uart_handle_cts_change (info, status & IP3106_UART_MSR_CTS);

    wake_up_interruptible (&info->delta_msr_wait);
}

//-----------------------------------------------------------------------------

static void ip3106_stop_tx (struct uart_port *port, unsigned int from_tty)
{
    unsigned int ier_reg = 0;

    ier_reg = IP3106_UART_GET_IER(port);
    ier_reg &= ~IP3106_UART_IER_THREIE;
    IP3106_UART_PUT_IER(port, ier_reg);
}

//-----------------------------------------------------------------------------

static void ip3106_start_tx (struct uart_port *port, unsigned int nonempty, unsigned int from_tty)
{
    if (nonempty)
    {
        unsigned int ier_reg = 0;

        ier_reg = IP3106_UART_GET_IER(port);
        ier_reg |= IP3106_UART_IER_THREIE;
        IP3106_UART_PUT_IER(port, ier_reg);
    }
}

//-----------------------------------------------------------------------------

static void ip3106_stop_rx (struct uart_port *port)
{
    unsigned int ier_reg = 0;

    ier_reg = IP3106_UART_GET_IER(port);
    ier_reg &= ~IP3106_UART_IER_RDAIE;
    IP3106_UART_PUT_IER(port, ier_reg);
}

//-----------------------------------------------------------------------------

static void ip3106_enable_ms (struct uart_port *port)
{
    unsigned int ier_reg = 0;

    ier_reg = IP3106_UART_GET_IER(port);
// JVDG 03-06-2004    THIS INTERRUPT IS BLOKKING THE SYSTEM ON BOOTUP.
//                    IT HAS SOMETHING TO DO WITH THE BOARD DESIGN.
//                    WE DO NOT ENABLE THIS IRQ UNTIL WE KNOW WHAT IS CAUSING THIS.
//    ier_reg |= IP3106_UART_IER_MSIE;
    IP3106_UART_PUT_IER(port, ier_reg);
}

//-----------------------------------------------------------------------------

static unsigned int ip3106_tx_empty (struct uart_port *port)
{
    unsigned int lsr_reg = 0;

    lsr_reg = IP3106_UART_GET_LSR(port);
    if (lsr_reg & IP3106_UART_LSR_TEMT)
    {
        return TIOCSER_TEMT;
    }
    return 0;
}

//-----------------------------------------------------------------------------

static unsigned int ip3106_get_mctrl (struct uart_port *port)
{
    unsigned int result = 0;
    unsigned int status = 0;

    status = IP3106_UART_GET_MSR(port);
    if (status & IP3106_UART_MSR_DCD)
        result |= TIOCM_CAR;
    if (status & IP3106_UART_MSR_DSR)
        result |= TIOCM_DSR;
    if (status & IP3106_UART_MSR_CTS)
        result |= TIOCM_CTS;

    return result;
}

//-----------------------------------------------------------------------------

static void ip3106_set_mctrl (struct uart_port *port, unsigned int mctrl)
{
    unsigned int mcr_reg = 0;

    mcr_reg = IP3106_UART_GET_MCR(port);
    if (mctrl & TIOCM_RTS)
    {
        mcr_reg &= ~IP3106_UART_MCR_RTS;
    }
    else
    {
        mcr_reg |= IP3106_UART_MCR_RTS;
    }
    if (mctrl & TIOCM_DTR)
    {
        mcr_reg &= ~IP3106_UART_MCR_DTR;
    }
    else
    {
        mcr_reg |= IP3106_UART_MCR_DTR;
    }
    IP3106_UART_PUT_MCR(port, mcr_reg);
}

//-----------------------------------------------------------------------------

static void ip3106_break_ctl (struct uart_port *port, int break_state)
{
    unsigned int lcr_reg = 0;

    lcr_reg = IP3106_UART_GET_LCR(port);
    if (break_state == -1)
        lcr_reg |= IP3106_UART_LCR_BRK;
    else
        lcr_reg &= ~IP3106_UART_LCR_BRK;
    IP3106_UART_PUT_LCR(port, lcr_reg);
}

//-----------------------------------------------------------------------------

static int ip3106_startup (struct uart_port *port, struct uart_info *info)
{
    int retval = 0;

    retval = request_irq (port->irq, irq_handler_for_ip3106, 0, "uart", info);
    if (retval)
    {
        IP3106_MSG ("request_irq() returned error %d\n", retval);
        return retval;
    }

    // initialise the old status of the modem signals
    info->drv_old_status = IP3106_UART_GET_MSR(port) & IP3106_UART_MSR_MODEM_ANY;

    // Finally, enable interrupts
    IP3106_UART_PUT_IER(port, IP3106_UART_IER_RDAIE);

    return 0;
}

//-----------------------------------------------------------------------------

static void ip3106_shutdown (struct uart_port *port, struct uart_info *info)
{
    unsigned int  rxtriglevel = 0;
    unsigned int  lcr_reg = 0;
    unsigned int  fcr_reg = 0;

    free_irq (port->irq, info);
    // disable all interrupts, disable the port
    IP3106_UART_PUT_IER(port, 0);

    // disable break condition
    lcr_reg = IP3106_UART_GET_LCR(port);
    lcr_reg &= ~IP3106_UART_LCR_BRK;
    IP3106_UART_PUT_LCR(port, lcr_reg);

    // disable fifos
    switch (port->unused[1])
    {
    case IP3106_RX_TRIG_LEV_HIGH:
        rxtriglevel = 3;
        break;
    case IP3106_RX_TRIG_LEV_MEDIUM:
        rxtriglevel = 2;
        break;
    case IP3106_RX_TRIG_LEV_LOW:
        rxtriglevel = 1;
        break;
    default:
        rxtriglevel = 0;
        break;
    }
    fcr_reg |= (rxtriglevel << 6);
    fcr_reg |= IP3106_UART_FCR_TFRST;
    fcr_reg |= IP3106_UART_FCR_RFRST;
    fcr_reg &= ~IP3106_UART_FCR_FEN;
    IP3106_UART_PUT_FCR(port, fcr_reg);
}

//-----------------------------------------------------------------------------

static void ip3106_change_speed (struct uart_port *port, unsigned int cflag, unsigned int iflag, unsigned int quot)
{
    unsigned int  rxtriglevel = 0;
    unsigned int  lcr_reg   = 0;
    unsigned int  fcr_reg   = 0;
    unsigned int  ier_reg   = 0;
    unsigned int  divisor   = 0;
    unsigned int  mulval    = 0;
    unsigned int  divaddval = 0;
    unsigned long flags     = 0;

    // byte size and parity
    switch (cflag & CSIZE)
    {
    case CS5:
        lcr_reg |= IP3106_UART_LCR_WLEN_5;
        break;
    case CS6:
        lcr_reg |= IP3106_UART_LCR_WLEN_6;
        break;
    case CS7:
        lcr_reg |= IP3106_UART_LCR_WLEN_7;
        break;
    default: // CS8
        lcr_reg |= IP3106_UART_LCR_WLEN_8;
        break;
    }

    if (cflag & CSTOPB)
    {
        lcr_reg |= IP3106_UART_LCR_STOP;
    }
    if (cflag & PARENB)
    {
        lcr_reg |= IP3106_UART_LCR_PEN;
        if (!(cflag & PARODD))
        {
            lcr_reg |= IP3106_UART_LCR_PEVEN;
        }
    }
    if (port->fifosize > 1)
    {
        switch (port->unused[1])
        {
        case IP3106_RX_TRIG_LEV_HIGH:
            rxtriglevel = 3;
            break;
        case IP3106_RX_TRIG_LEV_MEDIUM:
            rxtriglevel = 2;
            break;
        case IP3106_RX_TRIG_LEV_LOW:
            rxtriglevel = 1;
            break;
        default:
            rxtriglevel = 0;
            break;
        }
        fcr_reg |= (rxtriglevel << 6);
        fcr_reg |= IP3106_UART_FCR_TFRST;
        fcr_reg |= IP3106_UART_FCR_RFRST;
        fcr_reg |= IP3106_UART_FCR_FEN;
    }

    port->read_status_mask = IP3106_UART_LSR_OE;
    if (iflag & INPCK)
    {
        port->read_status_mask |= IP3106_UART_LSR_FE | IP3106_UART_LSR_PE;
    }
    if (iflag & (BRKINT | PARMRK))
    {
        port->read_status_mask |= IP3106_UART_LSR_BI;
    }

    // Characters to ignore
    port->ignore_status_mask = 0;
    if (iflag & IGNPAR)
    {
        port->ignore_status_mask |= IP3106_UART_LSR_FE | IP3106_UART_LSR_PE;
    }
    if (iflag & IGNBRK)
    {
        port->ignore_status_mask |= IP3106_UART_LSR_BI;
        // If we're ignoring parity and break indicators,
        // ignore overruns too (for real raw support)
        if (iflag & IGNPAR)
        {
            port->ignore_status_mask |= IP3106_UART_LSR_OE;
        }
    }

    // Ignore all characters if CREAD is not set
    if ((cflag & CREAD) == 0)
    {
        port->ignore_status_mask |= IP3106_UART_DUMMY_LSR_RX;
    }

    // first, disable everything
    save_flags (flags); cli ();
    ier_reg = IP3106_UART_GET_IER(port) & ~IP3106_UART_IER_MSIE;

    if ((port->flags & ASYNC_HARDPPS_CD) ||
        (cflag & CRTSCTS) || !(cflag & CLOCAL))
    {
        ier_reg |= IP3106_UART_IER_MSIE;
    }

    IP3106_UART_PUT_IER(port, 0);

    // Set baud rate
    IP3106_UART_PUT_LCR(port, IP3106_UART_LCR_DLAB);
    if (port->unused[2] == IP3106_BAUDRATE_NORMAL)
    {
        mulval    = 1;
        divaddval = 0;
        divisor   = quot;
    }
    else
    {
        // accurate baud-rate generation required!!!
        get_baud_settings (port, quot, &mulval, &divaddval, &divisor);
    }

    IP3106_UART_PUT_FDR(port, ((mulval << 4) | divaddval));
    IP3106_UART_PUT_DLM(port, ((divisor & 0xFF00) >> 8));
    IP3106_UART_PUT_DLL(port,  (divisor & 0x00FF));

    // ----------v----------v----------v-----
    //  NOTE: MUST BE WRITTEN AFTER DLM & DLL
    // ----------^----------^----------^-----
    IP3106_UART_PUT_LCR(port, lcr_reg);  // implicitly DLAB is reset to zero
    IP3106_UART_PUT_FCR(port, fcr_reg);
    IP3106_UART_PUT_IER(port, ier_reg);

    restore_flags (flags);
}

//-----------------------------------------------------------------------------

static void get_baud_settings (struct uart_port *port, unsigned int quot,
                               unsigned int *pmulval, unsigned int *pdivaddval, unsigned int *pdivisor)
{
    unsigned int i       = 0;
    unsigned int my_quot = 0;

    *pmulval    = 1;
    *pdivaddval = 0;
    *pdivisor   = quot;

    i = sizeof(baud_reg_table) / sizeof(ip3106_baud_settings_t);
    while (i > 0)
    {
        i--;
        my_quot = port->uartclk / (16 * baud_reg_table[i].baud_rate);
        if (quot == my_quot)
        {
            IP3106_MSG ("port ttyPU%d, selected accurate baud_rate = %d\n", port->unused[0], baud_reg_table[i].baud_rate);
            *pmulval    = baud_reg_table[i].mulval;
            *pdivaddval = baud_reg_table[i].divaddval;
            *pdivisor   = baud_reg_table[i].divisor;
            return;
        }
    }
}

//-----------------------------------------------------------------------------

static const char *ip3106_type (struct uart_port *port)
{
    return port->type == PORT_16750 ? "IP_3106" : NULL;
}

//-----------------------------------------------------------------------------

static void ip3106_release_port (struct uart_port *port)
{
    // do nothing
}

//-----------------------------------------------------------------------------

static int ip3106_request_port (struct uart_port *port)
{
    return 0;
}

//-----------------------------------------------------------------------------

static void ip3106_config_port (struct uart_port *port, int flags)
{
    if (flags & UART_CONFIG_TYPE)
    {
        port->type = PORT_16750;
        ip3106_request_port(port);
    }
}

//-----------------------------------------------------------------------------

static int ip3106_verify_port (struct uart_port *port, struct serial_struct *ser)
{
    int ret = 0;
    if (ser->type != PORT_UNKNOWN && ser->type != PORT_16750)
        ret = -EINVAL;
    if (ser->irq < 0 || ser->irq >= NR_IRQS)
        ret = -EINVAL;
    if (ser->baud_base < 9600)
        ret = -EINVAL;
    return ret;
}

//-----------------------------------------------------------------------------

#ifdef CONFIG_SERIAL_PNX0105_CONSOLE
#ifdef used_and_not_const_char_pointer
static int ip3106_console_read (struct uart_port *port, char *s, unsigned int count)
{
    unsigned int status = 0;
    int my_cnt = 0;

    while (my_cnt < count)
    {
        status = IP3106_UART_GET_LSR(port);
        if (status & IP3106_UART_LSR_DR)
        {
            *s++ = IP3106_UART_GET_CHAR(port);
            my_cnt++;
        }
        else
        {
            // nothing more to get, return
            return my_cnt;
        }
    }
    return my_cnt;
}
#endif // #ifdef used_and_not_const_char_pointer

//-----------------------------------------------------------------------------

static void ip3106_console_write (struct console *co, const char *s, unsigned int count)
{
    unsigned int status  = 0;
    unsigned int ier_reg = 0;
    unsigned int i       = 0;

    // First save IER then disable the interrupts
    ier_reg = IP3106_UART_GET_IER(&ip3106_ports);
    IP3106_UART_PUT_IER(&ip3106_ports, 0);

    // Now, do write each character
    for (i = 0; i < count; i++)
    {
        do
        {
            status = IP3106_UART_GET_LSR(&ip3106_ports);
        } while ((status & IP3106_UART_LSR_THRE) == 0);
        IP3106_UART_PUT_CHAR(&ip3106_ports, s[i]);

        if (s[i] == '\n')
        {
            do
            {
                status = IP3106_UART_GET_LSR(&ip3106_ports);
            } while ((status & IP3106_UART_LSR_THRE) == 0);
            IP3106_UART_PUT_CHAR(&ip3106_ports, '\r');
        }
    }

    // Finally, wait for transmitter to become empty and restore the IER
    do
    {
        status = IP3106_UART_GET_LSR(&ip3106_ports);
    } while ((status & IP3106_UART_LSR_TEMT) == 0);

    IP3106_UART_PUT_IER(&ip3106_ports, ier_reg);
}

//-----------------------------------------------------------------------------

static kdev_t ip3106_console_device (struct console *co)
{
    return MKDEV(IP3106_UART_MAJOR, IP3106_UART_MINOR + co->index);
}

//-----------------------------------------------------------------------------

static int ip3106_console_wait_key (struct console *co)
{
    unsigned int status = 0;

    do
    {
        status = IP3106_UART_GET_LSR(&ip3106_ports);
    } while ((status & IP3106_UART_LSR_DR) == 0);
    return IP3106_UART_GET_CHAR(&ip3106_ports);
}

//-----------------------------------------------------------------------------

static int __init ip3106_console_setup (struct console *co, char *options)
{
    struct uart_port *port = 0;
    int baud   = 38400;
    int bits   = 8;
    int parity = 'n';
    int flow   = 'n';

    // Check whether an invalid uart number has been specified, and
    // if so, search for the first available port that does have
    // console support.
    port = uart_get_console(&ip3106_ports, IP3106_NR_OF_UARTS, co);

    if (options)
        uart_parse_options(options, &baud, &parity, &bits, &flow);
    else
        ip3106_console_get_options(port, &baud, &parity, &bits);

    return uart_set_options(port, co, baud, parity, bits, flow);
}

//-----------------------------------------------------------------------------

static void __init ip3106_console_get_options (struct uart_port *port, int *baud, int *parity, int *bits)
{
    unsigned int  i         = 0;
    unsigned int  lcr_reg   = 0;
    unsigned int  fdr_reg   = 0;
    unsigned int  mulval    = 0;
    unsigned int  divaddval = 0;
    unsigned int  divisor   = 0;
    unsigned long flags     = 0;

    lcr_reg = IP3106_UART_GET_LCR(port);

    *parity = 'n';
    if (lcr_reg & IP3106_UART_LCR_PEN)
    {
        if (lcr_reg & IP3106_UART_LCR_PEVEN)
            *parity = 'e';
        else
            *parity = 'o';
    }

    if ((lcr_reg & IP3106_UART_LCR_WLEN_MASK) == IP3106_UART_LCR_WLEN_7)
        *bits = 7;
    else
        *bits = 8;

    // first, disable everything
    save_flags (flags); cli ();
    IP3106_UART_PUT_LCR(port, lcr_reg | IP3106_UART_LCR_DLAB);

    fdr_reg   = IP3106_UART_GET_FDR(port);
    divaddval = fdr_reg & IP3106_UART_FDR_DIVVAL;
    mulval    = (fdr_reg & IP3106_UART_FDR_MULVAL) >> 4;
    divisor   = (IP3106_UART_GET_DLM(port) << 8) | IP3106_UART_GET_DLL(port);

    IP3106_UART_PUT_LCR(port, lcr_reg);
    restore_flags (flags);

    if (divaddval == 0 && mulval == 1)
    {
        // normal baud-rate precision
        *baud = port->uartclk / (16 * divisor);
    }
    else
    {
        // accurate baud-rate precision
        i = sizeof(baud_reg_table) / sizeof(ip3106_baud_settings_t);
        while (i > 0)
        {
            i--;
            if (baud_reg_table[i].mulval    == mulval &&
                baud_reg_table[i].divaddval == divaddval &&
                baud_reg_table[i].divisor   == divisor)
            {
                *baud = baud_reg_table[i].baud_rate;
                return;
            }
        }
    }
}

//-----------------------------------------------------------------------------

void __init ip3106_console_init (void)
{
    unsigned int quot = 0;

    quot = ip3106_ports.uartclk / (16 * 38400);
    ip3106_change_speed (&ip3106_ports, CS8, IGNPAR | IGNBRK, quot);
    register_console (&ip3106_console);
}

#endif // #ifdef CONFIG_SERIAL_PNX0105_CONSOLE


//-----------------------------------------------------------------------------
// Exported functions:
//-----------------------------------------------------------------------------

MODULE_AUTHOR("Johan van der Graaf");
MODULE_DESCRIPTION("IP_3106 VPB UART Driver");
MODULE_SUPPORTED_DEVICE("ttyPU");
MODULE_LICENSE("GPL");

//-----------------------------------------------------------------------------
// FUNCTION:    cleanup_ip3106
//
// DESCRIPTION: 
//
// RETURN:      
//
// NOTES:       
//-----------------------------------------------------------------------------

static void __exit cleanup_ip3106 (void)
{
    uart_unregister_driver (&ip3106_reg);
    IP3106_MSG ("module unloaded\n");
}

//-----------------------------------------------------------------------------
// FUNCTION:    init_ip3106
//
// DESCRIPTION: 
//
// RETURN:      
//
// NOTES:       
//-----------------------------------------------------------------------------

static int __init init_ip3106 (void)
{
    int           retval  = 0;
    unsigned int  i       = 0;
    char          dummy   = 0;

    // disable all interrupts, disable the port
    IP3106_UART_PUT_IER(&ip3106_ports, 0);
    // make sure that the fifo is empty
    for (i=0;i<IP3106_UART_FIFO_SIZE;i++) {
        dummy = IP3106_UART_GET_CHAR (&ip3106_ports);
    }

    retval = uart_register_driver (&ip3106_reg);
    if (retval)
    {
        IP3106_MSG ("uart_register_driver() returned error %d\n", retval);
        return retval;
    }

#ifdef IP3106_THRE_IRQ_PROBLEM_EXISTS
    // this is needed since the IP3106 UART doesn't generate the
    // 'transmitter holding register empty' interrupt after reset/power on.
    // transmitting some character initially will get this interrupt working.
    // when this problem is fixed this code can be removed.
    {
        unsigned int old_mcr_reg = 0;
        unsigned int lsr_reg = 0;

        old_mcr_reg = IP3106_UART_GET_MCR(&ip3106_ports);
        IP3106_UART_PUT_MCR(&ip3106_ports, IP3106_UART_MCR_LOOP);
        IP3106_UART_PUT_CHAR(&ip3106_ports, '0');
        do
        {
            lsr_reg = IP3106_UART_GET_LSR(&ip3106_ports);
        } while (!(lsr_reg & IP3106_UART_LSR_DR));
        IP3106_UART_GET_CHAR(&ip3106_ports);
        IP3106_UART_PUT_MCR(&ip3106_ports, old_mcr_reg);
    }
#endif // #ifdef IP3106_THRE_IRQ_PROBLEM_EXISTS

    IP3106_MSG ("module loaded (Version %s, build on %s, %s)\n", VERSION_STR, __DATE__, __TIME__);
    return 0;
}


module_init (init_ip3106);
module_exit (cleanup_ip3106);

EXPORT_NO_SYMBOLS;

//-----------------------------------------------------------------------------
// End of file
//-----------------------------------------------------------------------------
