/*
 * drivers/pcmcia/omap1610_ss.c
 * BRIEF MODULE DESCRIPTION
 *	PCMCIA Socket Driver for OMAP 1610 Compact Flash interface
 *
 * Author: MontaVista Software, Inc.
 *	   source@mvista.com
 *
 * 2004 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

/*
 * OMAP1610 compact flash controller implements simple fixed mapping
 * of PC Card resources to the memory.
 */

#include <linux/types.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/mm.h>
#include <linux/vmalloc.h>
#include <asm/errno.h>

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/arch/irq.h>

#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/ss.h>
#include <pcmcia/bulkmem.h>
#include <pcmcia/cistpl.h>
#include "cs_internal.h"

#define MODNAME "omap1610_ss"

/* #define OMAP1610_SS_DEBUG 1 */

#ifndef OMAP1610_SS_DEBUG
#define OMAP1610_SS_DEBUG 0
#endif

#if OMAP1610_SS_DEBUG
#define DPRINTK(args...)	printk(MODNAME ": " args)
#else
#define DPRINTK(args...)
#endif

#if 0
#define OMAP1610_CFLASH_INT	INT_CF
#else
#define OMAP1610_CFLASH_INT	INT_GPIO_62
#endif

#define OMAP1610_SS_POLL_PERIOD	(1 * HZ)

typedef struct omap1610_socket_t {
	void (*handler) (void *info, u_int events);
	void *handler_info;
	u_int status;
	socket_state_t state;
	pccard_io_map io_maps[MAX_IO_WIN];
	pccard_mem_map mem_maps[MAX_WIN];
} omap1610_socket_t;

static omap1610_socket_t omap1610_socket;

/* Event poll timer structure */
static struct timer_list poll_timer;

static socket_cap_t omap1610_socket_cap = {
	.features = SS_CAP_PCCARD | SS_CAP_STATIC_MAP,
				/* support 16 bit cards */
				/* mappings are fixed in host memory */
	.irq_mask = 0,		/* Disable "ISA" interrupts */
	.map_size = 2 * 1024,	/* 2Kb fixed window size */
	.pci_irq = OMAP1610_CFLASH_INT,	/* Interrupt number */
};

static int
omap1610_ss_init(unsigned int sock)
{
	omap1610_socket_t *sp = &omap1610_socket;

	DPRINTK("omap1610_ss_init(%d)\n", sock);

	sp->status = 0;
	sp->state.Vcc = 50;
	sp->state.Vpp = 50;

	return 0;
}

static int
omap1610_ss_suspend(unsigned int sock)
{
	DPRINTK("omap1610_ss_suspend(%d)\n", sock);
	return 0;
}

static int
omap1610_ss_register_callback(unsigned int sock,
			      void (*handler) (void *, unsigned int),
			      void *info)
{
	omap1610_socket_t *sp = &omap1610_socket;

	DPRINTK("omap1610_ss_register_callback(%d)\n", sock);
	sp->handler = handler;
	sp->handler_info = info;
	if (handler == 0) {
		MOD_DEC_USE_COUNT;
	} else {
		MOD_INC_USE_COUNT;
	}
	return 0;
}

static int
omap1610_ss_inquire_socket(unsigned int sock, socket_cap_t * cap)
{
	DPRINTK("omap1610_ss_inquire_socket(%d)\n", sock);

	*cap = omap1610_socket_cap;
	return 0;
}

static int
omap1610_ss_get_status(unsigned int sock, u_int * value)
{
	u_int status = 0;

	if (!(readw(OMAP1610_CF_STATUS) & OMAP1610_CF_STATUS_CD))
		status |= SS_DETECT | SS_READY | SS_POWERON;

	DPRINTK("omap1610_ss_get_status(%d) = %x\n", sock, status);

	*value = status;
	return 0;
}

static int
omap1610_ss_get_socket(unsigned int sock, socket_state_t * state)
{
	omap1610_socket_t *sp = &omap1610_socket;

	DPRINTK("omap1610_ss_get_socket(%d)\n", sock);

	*state = sp->state;
	return 0;
}

static int
omap1610_ss_set_socket(unsigned int sock, socket_state_t * state)
{
	omap1610_socket_t *sp = &omap1610_socket;

	DPRINTK("omap1610_ss_set_socket(sock=%d, flags=%x, csc_mask=%x, "
		"Vcc=%d, Vpp=%d, io_irq=%d)\n",
		sock, state->flags, state->csc_mask, state->Vcc,
		state->Vpp, state->io_irq);

	if (state->flags & SS_RESET)
		writew(OMAP1610_CF_CONTROL_RESET, OMAP1610_CF_CONTROL);
	else
		writew(0, OMAP1610_CF_CONTROL);

	sp->state = *state;
	return 0;
}

static int
omap1610_ss_get_io_map(unsigned int sock, struct pccard_io_map *io)
{
	omap1610_socket_t *sp = &omap1610_socket;
	int map = io->map;

	DPRINTK("omap1610_ss_get_io_map(%d, %d)\n", sock, map);
	if (map >= MAX_IO_WIN)
		return -EINVAL;

	*io = sp->io_maps[map];
	return 0;
}

static int
omap1610_ss_set_io_map(unsigned int sock, struct pccard_io_map *io)
{
	omap1610_socket_t *sp = &omap1610_socket;
	int map = io->map;
	struct pccard_io_map *sio;

	DPRINTK("omap1610_ss_set_io_map(sock=%d, map=%d, flags=0x%x, "
		"speed=%dns, start=0x%04x, stop=0x%04x)\n",
		sock, map, io->flags, io->speed, io->start, io->stop);
	if (map >= MAX_IO_WIN)
		return -EINVAL;
	sio = &sp->io_maps[map];

	if (io->flags & MAP_ATTRIB) {
		io->start = OMAP1610_CF_ATTRIB_START;
		io->stop = io->start + OMAP1610_CF_ATTRIB_SIZE - 1;
		io->flags = (io->flags & MAP_ACTIVE) | MAP_16BIT | MAP_ATTRIB;
	} else {
		io->start = OMAP1610_CF_IO_BASE;
		io->stop = io->start + OMAP1610_CF_IO_SIZE - 1;
		io->flags = (io->flags & MAP_ACTIVE) | MAP_16BIT;
	}

	*sio = *io;
	return 0;
}

static int
omap1610_ss_get_mem_map(unsigned int sock, struct pccard_mem_map *mem)
{
	omap1610_socket_t *sp = &omap1610_socket;
	int map = mem->map;

	DPRINTK("omap1610_ss_get_mem_map(%d, %d)\n", sock, map);
	if (map >= MAX_WIN)
		return -EINVAL;

	*mem = sp->mem_maps[map];
	return 0;
}

static int
omap1610_ss_set_mem_map(unsigned int sock, struct pccard_mem_map *mem)
{
	omap1610_socket_t *sp = &omap1610_socket;
	struct pccard_mem_map *smem;
	int map = mem->map;

	DPRINTK("omap1610_ss_set_mem_map(sock=%d, map=%d, flags=0x%x, "
		"sys_start=0x%08lx, sys_end=0x%08lx, card_start=0x%08x)\n",
		sock, map, mem->flags, mem->sys_start, mem->sys_stop,
		mem->card_start);

	if (map >= MAX_WIN)
		return -EINVAL;
	smem = &sp->mem_maps[map];

	if (mem->flags & MAP_ATTRIB) {
		mem->sys_start = OMAP1610_CF_ATTRIB_START;
		mem->sys_stop = mem->sys_start + OMAP1610_CF_ATTRIB_SIZE - 1;
		mem->flags = (mem->flags & MAP_ACTIVE) | MAP_16BIT | MAP_ATTRIB;
	} else {
		mem->sys_start = OMAP1610_CF_MEM_START;
		mem->sys_stop = mem->sys_start + OMAP1610_CF_MEM_SIZE - 1;
		mem->flags = (mem->flags & MAP_ACTIVE) | MAP_16BIT;
	}

	*smem = *mem;

	return 0;
}

static void
omap1610_ss_proc_setup(unsigned int sock, struct proc_dir_entry *base)
{
	DPRINTK("omap1610_ss_proc_setup(%d)\n", sock);
}

static struct pccard_operations omap1610_ss_operations = {
	.init			= omap1610_ss_init,
	.suspend		= omap1610_ss_suspend,
	.register_callback	= omap1610_ss_register_callback,
	.inquire_socket		= omap1610_ss_inquire_socket,
	.get_status		= omap1610_ss_get_status,
	.get_socket		= omap1610_ss_get_socket,
	.set_socket		= omap1610_ss_set_socket,
	.get_io_map		= omap1610_ss_get_io_map,
	.set_io_map		= omap1610_ss_set_io_map,
	.get_mem_map		= omap1610_ss_get_mem_map,
	.set_mem_map		= omap1610_ss_set_mem_map,
	.proc_setup		= omap1610_ss_proc_setup
};

static void
omap1610_pcmcia_task_handler(void *data)
{
	omap1610_socket_t *sp = &omap1610_socket;
	u_int status = 0;
	DPRINTK("status check: 0x%04x\n", readw(OMAP1610_CF_STATUS));
	if (!(readw(OMAP1610_CF_STATUS) & OMAP1610_CF_STATUS_CD))
		status |= SS_DETECT;
	if ((sp->status & SS_DETECT) != (status & SS_DETECT)) {
		DPRINTK("omap1610_pcmcia_task_handler: status changed: "
			"old 0x%04x new 0x%04x\n", sp->status, status);
		if (sp->handler)
			sp->handler(sp->handler_info, SS_DETECT);
	}
	sp->status = status;
}

static struct tq_struct omap1610_pcmcia_task = {
      .routine = omap1610_pcmcia_task_handler
};

static void
omap1610_poll_event(unsigned long dummy)
{
	poll_timer.function = omap1610_poll_event;
	poll_timer.expires = jiffies + OMAP1610_SS_POLL_PERIOD;
	add_timer(&poll_timer);
	schedule_task(&omap1610_pcmcia_task);
}

#if OMAP1610_SS_DEBUG
static void
cis_hex_dump(const unsigned char *x, int len)
{
	int i;

	for (i = 0; i < len; i++) {
		if (!(i & 0xf))
			printk("\n%08x", (unsigned) (x));
		printk(" %02x", *(volatile unsigned short *) x);
		x += 2;
	}
	printk("\n");
}
#endif

static int __init
init_omap1610_ss(void)
{
#if OMAP1610_SS_DEBUG
	cis_hex_dump((char *) OMAP1610_CF_ATTRIB_BASE,
		     OMAP1610_CF_ATTRIB_SIZE / 2);
#endif
	if (register_ss_entry(1, &omap1610_ss_operations) < 0) {
		printk(KERN_ERR "Unable to register omap1610 socket service\n");
		return -ENXIO;
	}
	omap1610_tune_irq(OMAP1610_CFLASH_INT, IFL_LOWLEVEL);
	omap1610_poll_event(0);
	DPRINTK("init_omap1610_ss: driver initialized\n");
	return 0;
}

static void __exit
exit_omap1610_ss(void)
{
	del_timer_sync(&poll_timer);
	unregister_ss_entry(&omap1610_ss_operations);
	flush_scheduled_tasks();
	DPRINTK("exit_omap1610_ss: driver finished\n");
}

module_init(init_omap1610_ss);
module_exit(exit_omap1610_ss);

MODULE_DESCRIPTION("OMAP1610 CF Socket driver");
MODULE_LICENSE("GPL");
