/* $Id: xgemac_options.c,v 1.1.4.1 2004/07/12 11:29:20 kurtsman Exp $ */
/******************************************************************************
*
*       XILINX IS PROVIDING THIS DESIGN, CODE, OR INFORMATION "AS IS"
*       AS A COURTESY TO YOU, SOLELY FOR USE IN DEVELOPING PROGRAMS AND
*       SOLUTIONS FOR XILINX DEVICES.  BY PROVIDING THIS DESIGN, CODE,
*       OR INFORMATION AS ONE POSSIBLE IMPLEMENTATION OF THIS FEATURE,
*       APPLICATION OR STANDARD, XILINX IS MAKING NO REPRESENTATION
*       THAT THIS IMPLEMENTATION IS FREE FROM ANY CLAIMS OF INFRINGEMENT,
*       AND YOU ARE RESPONSIBLE FOR OBTAINING ANY RIGHTS YOU MAY REQUIRE
*       FOR YOUR IMPLEMENTATION.  XILINX EXPRESSLY DISCLAIMS ANY
*       WARRANTY WHATSOEVER WITH RESPECT TO THE ADEQUACY OF THE
*       IMPLEMENTATION, INCLUDING BUT NOT LIMITED TO ANY WARRANTIES OR
*       REPRESENTATIONS THAT THIS IMPLEMENTATION IS FREE FROM CLAIMS OF
*       INFRINGEMENT, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
*       FOR A PARTICULAR PURPOSE.
*
*       (c) Copyright 2002 Xilinx Inc.
*       All rights reserved.
*
******************************************************************************/
/*****************************************************************************/
/**
*
* @file xgemac_options.c
*
* Functions in this file handle configuration of the XGemac driver.
*
* <pre>
* MODIFICATION HISTORY:
*
* Who  Date     Changes
* ---- -------- -----------------------------------------------
* ecm  01/13/03 First release
* ecm  03/25/03 Revision update
* rmm  05/28/03 Changed interframe gap API, process auto negotiate option
* rmm  06/04/03 Moved XGE_MAX_IFG definition to xgemac_l.h. Fixed assert
*               on MAX_IFG value in XGemac_SetInterframeGap(). Added
*               support for XGE_VLAN_OPTION and XGE_JUMBO_OPTION. Updated
*               description of XGemac_SetInterframeGap().
* rmm  11/14/03 Added XGE_NO_SGEND_INT_OPTION processing.
* </pre>
*
******************************************************************************/

/***************************** Include Files *********************************/

#include "xbasic_types.h"
#include "xgemac_i.h"
#include "xio.h"

/************************** Constant Definitions *****************************/

/**************************** Type Definitions *******************************/

/***************** Macros (Inline Functions) Definitions *********************/

/************************** Function Prototypes ******************************/

/************************** Variable Definitions *****************************/

/*
 * A table of options and masks. This table maps the user-visible options with
 * the control register masks. It is used in Set/GetOptions as an alternative
 * to a series of if/else pairs. Note that the polled options does not have a
 * corresponding entry in the control register, so it does not exist in the
 * table.
 */
typedef struct {
	u32 Option;
	u32 Mask;
} OptionMap;

static OptionMap OptionsTable[] = {
	{XGE_UNICAST_OPTION, XGE_ECR_UNICAST_ENABLE_MASK},
	{XGE_BROADCAST_OPTION, XGE_ECR_BROAD_ENABLE_MASK},
	{XGE_PROMISC_OPTION, XGE_ECR_PROMISC_ENABLE_MASK},
	{XGE_FDUPLEX_OPTION, XGE_ECR_FULL_DUPLEX_MASK},
	{XGE_LOOPBACK_OPTION, XGE_ECR_LOOPBACK_MASK},
	{XGE_MULTICAST_OPTION, XGE_ECR_MULTI_ENABLE_MASK},
	{XGE_FLOW_CONTROL_OPTION,
	 XGE_ECR_XMIT_PAUSE_ENABLE_MASK | XGE_ECR_RECV_PAUSE_ENABLE_MASK},
	{XGE_INSERT_PAD_OPTION, XGE_ECR_XMIT_PAD_ENABLE_MASK},
	{XGE_INSERT_FCS_OPTION, XGE_ECR_XMIT_FCS_ENABLE_MASK},
	{XGE_AUTO_NEGOTIATE_OPTION, XGE_ECR_AUTONEG_ENABLE_MASK},
	{XGE_STRIP_PAD_FCS_OPTION, XGE_ECR_RECV_STRIP_ENABLE_MASK},
	{XGE_VLAN_OPTION,
	 XGE_ECR_XMIT_VLAN_ENABLE_MASK | XGE_ECR_RECV_VLAN_ENABLE_MASK},
	{XGE_JUMBO_OPTION,
	 XGE_ECR_XMIT_JUMBO_ENABLE_MASK | XGE_ECR_RECV_JUMBO_ENABLE_MASK}
};

#define XGE_NUM_OPTIONS (sizeof(OptionsTable)/sizeof(OptionMap))

/*****************************************************************************/
/**
*
* Set Ethernet driver/device options. The options (XGE_*_OPTION) constants can
* be OR'd together to set/clear multiple options. A one (1) in the bit-mask turns
* an option on, and a zero (0) turns the option off.
*
* The device must be stopped before calling this function.
*
* @param InstancePtr is a pointer to the XGemac instance to be worked on.
* @param OptionsFlag is a bit-mask representing the Ethernet options to turn on
*        or off. See xgemac.h for a description of the available options.
*
* @return
*
* - XST_SUCCESS if the options were set successfully
* - XST_DEVICE_IS_STARTED if the device has not yet been stopped
*
* @note
*
* This function is not thread-safe and makes use of internal resources that are
* shared between the Start, Stop, and SetOptions functions, so if one task
* might be setting device options while another is trying to start the device,
* protection of this shared data (typically using a semaphore) is required.
*
******************************************************************************/
XStatus
XGemac_SetOptions(XGemac * InstancePtr, u32 OptionsFlag)
{
	u32 ControlReg;
	int Index;

	XASSERT_NONVOID(InstancePtr != NULL);
	XASSERT_NONVOID(InstancePtr->IsReady == XCOMPONENT_IS_READY);

	if (InstancePtr->IsStarted == XCOMPONENT_IS_STARTED) {
		return XST_DEVICE_IS_STARTED;
	}

	ControlReg = XIo_In32(InstancePtr->BaseAddress + XGE_ECR_OFFSET);

	/*
	 * Loop through the options table, turning the option on or off
	 * depending on whether the bit is set in the incoming options flag.
	 */
	for (Index = 0; Index < XGE_NUM_OPTIONS; Index++) {
		if (OptionsFlag & OptionsTable[Index].Option) {
			ControlReg |= OptionsTable[Index].Mask;		/* turn it on */
		} else {
			ControlReg &= ~OptionsTable[Index].Mask;	/* turn it off */
		}
	}

	/*
	 * Now write the control register. Leave it to the upper layers
	 * to restart the device.
	 */
	XIo_Out32(InstancePtr->BaseAddress + XGE_ECR_OFFSET, ControlReg);

	/*
	 * Check the polled option
	 */
	if (OptionsFlag & XGE_POLLED_OPTION) {
		InstancePtr->IsPolled = TRUE;
	} else {
		InstancePtr->IsPolled = FALSE;
	}

	/*
	 * Check the No SGEND option
	 */
	if (OptionsFlag & XGE_NO_SGEND_INT_OPTION) {
		InstancePtr->IsSgEndDisable = TRUE;
	} else {
		InstancePtr->IsSgEndDisable = FALSE;
	}
	return XST_SUCCESS;
}

/*****************************************************************************/
/**
*
* Get Ethernet driver/device options. The 32-bit value returned is a bit-mask
* representing the options (XGE_*_OPTION).  A one (1) in the bit-mask means 
* the option is on, and a zero (0) means the option is off.
*
* @param InstancePtr is a pointer to the XGemac instance to be worked on.
*
* @return
*
* The 32-bit value of the Ethernet options. The value is a bit-mask
* representing all options that are currently enabled. See xgemac.h for a
* description of the available options.
*
* @note
*
* None.
*
******************************************************************************/
u32
XGemac_GetOptions(XGemac * InstancePtr)
{
	u32 OptionsFlag = 0;
	u32 ControlReg;
	int Index;

	XASSERT_NONVOID(InstancePtr != NULL);
	XASSERT_NONVOID(InstancePtr->IsReady == XCOMPONENT_IS_READY);

	/*
	 * Get the control register to determine which options are currently set.
	 */
	ControlReg = XIo_In32(InstancePtr->BaseAddress + XGE_ECR_OFFSET);

	/*
	 * Loop through the options table to determine which options are set
	 */
	for (Index = 0; Index < XGE_NUM_OPTIONS; Index++) {
		if (ControlReg & OptionsTable[Index].Mask) {
			OptionsFlag |= OptionsTable[Index].Option;
		}
	}

	if (InstancePtr->IsPolled) {
		OptionsFlag |= XGE_POLLED_OPTION;
	}

	if (InstancePtr->IsSgEndDisable) {
		OptionsFlag |= XGE_NO_SGEND_INT_OPTION;
	}

	return OptionsFlag;
}
