/* $Id: xgemac_intr.c,v 1.1.4.1 2004/07/12 11:29:20 kurtsman Exp $ */
/******************************************************************************
*
*       XILINX IS PROVIDING THIS DESIGN, CODE, OR INFORMATION "AS IS"
*       AS A COURTESY TO YOU, SOLELY FOR USE IN DEVELOPING PROGRAMS AND
*       SOLUTIONS FOR XILINX DEVICES.  BY PROVIDING THIS DESIGN, CODE,
*       OR INFORMATION AS ONE POSSIBLE IMPLEMENTATION OF THIS FEATURE,
*       APPLICATION OR STANDARD, XILINX IS MAKING NO REPRESENTATION
*       THAT THIS IMPLEMENTATION IS FREE FROM ANY CLAIMS OF INFRINGEMENT,
*       AND YOU ARE RESPONSIBLE FOR OBTAINING ANY RIGHTS YOU MAY REQUIRE
*       FOR YOUR IMPLEMENTATION.  XILINX EXPRESSLY DISCLAIMS ANY
*       WARRANTY WHATSOEVER WITH RESPECT TO THE ADEQUACY OF THE
*       IMPLEMENTATION, INCLUDING BUT NOT LIMITED TO ANY WARRANTIES OR
*       REPRESENTATIONS THAT THIS IMPLEMENTATION IS FREE FROM CLAIMS OF
*       INFRINGEMENT, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
*       FOR A PARTICULAR PURPOSE.
*
*       (c) Copyright 2002 Xilinx Inc.
*       All rights reserved.
*
******************************************************************************/
/*****************************************************************************/
/**
*
* @file xgemac_intr.c
*
* This file contains general interrupt-related functions of the XGemac driver.
*
* <pre>
* MODIFICATION HISTORY:
*
* Who  Date     Changes
* ---- -------- -----------------------------------------------
* ecm  01/13/03 First release
* ecm  03/25/03 Revision update
* rmm  05/28/03 Revision update
* rmm  09/23/03 Removed use of XGE_EIR_RECV_DFIFO_OVER_MASK, XGE_EIR
*               _RECV_MISSED_FRAME_MASK, XGE_EIR_RECV_COLLISION_MASK,
*               XGE_EIR_RECV_SHORT_ERROR_MASK
* </pre>
*
******************************************************************************/

/***************************** Include Files *********************************/

#include "xbasic_types.h"
#include "xgemac_i.h"
#include "xio.h"
#include "xipif_v1_23_b.h"	/* Uses v1.23b of the IPIF */

/************************** Constant Definitions *****************************/

/**************************** Type Definitions *******************************/

/***************** Macros (Inline Functions) Definitions *********************/

/************************** Variable Definitions *****************************/

/************************** Function Prototypes ******************************/

/*****************************************************************************/
/**
*
* Set the callback function for handling asynchronous errors.  The upper layer
* software should call this function during initialization.
*
* The error callback is invoked by the driver within interrupt context, so it
* needs to do its job quickly. If there are potentially slow operations within
* the callback, these should be done at task-level.
*
* The Xilinx errors that must be handled by the callback are:
* - XST_DMA_ERROR indicates an unrecoverable DMA error occurred. This is
*   typically a bus error or bus timeout. The handler must reset and
*   re-configure the device.
* - XST_FIFO_ERROR indicates an unrecoverable FIFO error occurred. This is a
*   deadlock condition in the packet FIFO. The handler must reset and
*   re-configure the device.
* - XST_RESET_ERROR indicates an unrecoverable MAC error occurred, usually an
*   overrun or underrun. The handler must reset and re-configure the device.
* - XST_DMA_SG_NO_LIST indicates an attempt was made to access a scatter-gather
*   DMA list that has not yet been created.
* - XST_DMA_SG_LIST_EMPTY indicates the driver tried to get a descriptor from
*   the receive descriptor list, but the list was empty.
*
* @param InstancePtr is a pointer to the XGemac instance to be worked on.
* @param CallBackRef is reference data to be passed back to the callback
*        function. Its value is arbitrary and not used by the driver.
* @param FuncPtr is the pointer to the callback function.
*
* @return
*
* None.
*
* @note
*
* None.
*
******************************************************************************/
void
XGemac_SetErrorHandler(XGemac * InstancePtr, void * CallBackRef,
			XGemac_ErrorHandler FuncPtr)
{
	XASSERT_VOID(InstancePtr != NULL);
	XASSERT_VOID(FuncPtr != NULL);
	XASSERT_VOID(InstancePtr->IsReady == XCOMPONENT_IS_READY);

	InstancePtr->ErrorHandler = FuncPtr;
	InstancePtr->ErrorRef = CallBackRef;
}

/****************************************************************************/
/*
*
* Check the interrupt status bits of the Ethernet MAC for errors. Errors
* currently handled are:
* - Receive length FIFO overrun. Indicates data was lost due to the receive
*   length FIFO becoming full during the reception of a packet. Only a device
*   reset clears this condition.
* - Receive length FIFO underrun. An attempt to read an empty FIFO. Only a
*   device reset clears this condition.
* - Transmit status FIFO overrun. Indicates data was lost due to the transmit
*   status FIFO becoming full following the transmission of a packet. Only a
*   device reset clears this condition.
* - Transmit status FIFO underrun. An attempt to read an empty FIFO. Only a
*   device reset clears this condition.
* - Transmit length FIFO overrun. Indicates data was lost due to the transmit
*   length FIFO becoming full following the transmission of a packet. Only a
*   device reset clears this condition.
* - Transmit length FIFO underrun. An attempt to read an empty FIFO. Only a
*   device reset clears this condition.
* - Receive data errors:
*   - Receive missed frame error. Valid data was lost by the MAC.
*   - Receive collision error. Data was lost by the MAC due to a collision.
*   - Receive FCS error.  Data was dicarded by the MAC due to FCS error.
*   - Receive length field error. Data was dicarded by the MAC due to an invalid
*     length field in the packet.
*   - Receive short error. Data was dicarded by the MAC because a packet was
*     shorter than allowed.
*   - Receive long error. Data was dicarded by the MAC because a packet was
*     longer than allowed.
*   - Receive alignment error. Data was truncated by the MAC because its length
*     was not byte-aligned.
*
* @param InstancePtr is a pointer to the XGemac instance to be worked on.
* @param IntrStatus is the contents of the interrupt status register to be checked
*
* @return
*
* None.
*
* @note
*
* This function is intended for internal use only.
*
******************************************************************************/
void
XGemac_CheckEmacError(XGemac * InstancePtr, u32 IntrStatus)
{
	u32 ResetError = FALSE;

	/*
	 * First check for receive fifo overrun/underrun errors. Most require a
	 * reset by the user to clear, but the data FIFO overrun error does not.
	 */
	if (IntrStatus & XGE_EIR_RECV_LFIFO_OVER_MASK) {
		InstancePtr->Stats.RecvOverrunErrors++;
		InstancePtr->Stats.FifoErrors++;
		ResetError = TRUE;			/* requires a reset */
	}

	if (IntrStatus & XGE_EIR_RECV_LFIFO_UNDER_MASK) {
		InstancePtr->Stats.RecvUnderrunErrors++;
		InstancePtr->Stats.FifoErrors++;
		ResetError = TRUE;			/* requires a reset */
	}

	/*
	 * Now check for general receive errors. Get the latest count where
	 * available, otherwise just bump the statistic so we know the interrupt
	 * occurred.
	 */
	if (IntrStatus & XGE_EIR_RECV_ERROR_MASK) {
		if (IntrStatus & XGE_EIR_RECV_LEN_ERROR_MASK) {
			InstancePtr->Stats.RecvLengthFieldErrors++;
		}

		if (IntrStatus & XGE_EIR_RECV_LONG_ERROR_MASK) {
			InstancePtr->Stats.RecvLongErrors++;
		}

		if (IntrStatus & XGE_EIR_RECV_SLOT_LENGTH_ERROR_MASK) {
			InstancePtr->Stats.RecvSlotLengthErrors++;
		}

		if (IntrStatus & (XGE_EIR_RECV_FCS_ERROR_MASK |
				XGE_EIR_RECV_CRC_ENGINE_ERROR_MASK)) {
			InstancePtr->Stats.RecvFcsErrors++;
		}

		/*
		 * Bump recv interrupts stats only if not scatter-gather DMA (this
		 * stat gets bumped elsewhere in that case)
		 */
		if (!XGemac_mIsSgDma(InstancePtr)) {
			InstancePtr->Stats.RecvInterrupts++;	/* TODO: double bump? */
		}

	}

	/*
	 * Check for transmit errors. These apply to both DMA and non-DMA modes
	 * of operation. The entire device should be reset after overruns or
	 * underruns.
	 */
	if (IntrStatus & (XGE_EIR_XMIT_SFIFO_OVER_MASK |
			XGE_EIR_XMIT_LFIFO_OVER_MASK)) {
		InstancePtr->Stats.XmitOverrunErrors++;
		InstancePtr->Stats.FifoErrors++;
		ResetError = TRUE;
	}

	if (IntrStatus & (XGE_EIR_XMIT_SFIFO_UNDER_MASK |
			XGE_EIR_XMIT_LFIFO_UNDER_MASK)) {
		InstancePtr->Stats.XmitUnderrunErrors++;
		InstancePtr->Stats.FifoErrors++;
		ResetError = TRUE;
	}

	if (ResetError) {
		/*
		 * If a reset error occurred, disable the GEMAC interrupts since the
		 * reset-causing interrupt(s) is latched in the GEMAC - meaning it will
		 * keep occurring until the device is reset. In order to give the higher
		 * layer software time to reset the device, we have to disable the
		 * overrun/underrun interrupts until that happens. We trust that the
		 * higher layer resets the device. We are able to get away with disabling
		 * all GEMAC interrupts since the only interrupts it generates are for
		 * error conditions, and we don't care about any more errors right now.
		 */
		XIIF_V123B_WRITE_IIER(InstancePtr->BaseAddress, 0);

		/*
		 * Invoke the error handler callback, which should result in a reset
		 * of the device by the upper layer software.
		 */
		InstancePtr->ErrorHandler(InstancePtr->ErrorRef, XST_RESET_ERROR);
	}
}

/*****************************************************************************/
/*
*
* Check the receive packet FIFO for errors. FIFO error interrupts are:
* - Deadlock.  See the XPacketFifo component for a description of deadlock on a
*   FIFO.
*
* @param InstancePtr is a pointer to the XGemac instance to be worked on.
*
* @return
*
* Although the function returns void, it can return an asynchronous error to the
* application through the error handler.  It can return XST_FIFO_ERROR if a FIFO
* error occurred.
*
* @note
*
* This function is intended for internal use only.
*
******************************************************************************/
void
XGemac_CheckFifoRecvError(XGemac * InstancePtr)
{
	/*
	 * Although the deadlock is currently the only interrupt from a packet
	 * FIFO, make sure it is deadlocked before taking action. There is no
	 * need to clear this interrupt since it requires a reset of the device.
	 */
	 if (XPF_V200A_IS_DEADLOCKED(&InstancePtr->RecvFifo)) {
		u32 IntrEnable;

		InstancePtr->Stats.FifoErrors++;

		/*
		 * Invoke the error callback function, which should result in a reset
		 * of the device by the upper layer software. We first need to disable
		 * the FIFO interrupt, since otherwise the upper layer thread that
		 * handles the reset may never run because this interrupt condition
		 * doesn't go away until a reset occurs (there is no way to ack it).
		 */
		IntrEnable = XIIF_V123B_READ_DIER(InstancePtr->BaseAddress);
		XIIF_V123B_WRITE_DIER(InstancePtr->BaseAddress,
			IntrEnable & ~XGE_IPIF_RECV_FIFO_MASK);

		InstancePtr->ErrorHandler(InstancePtr->ErrorRef, XST_FIFO_ERROR);
	}
}

/*****************************************************************************/
/*
*
* Check the send packet FIFO for errors. FIFO error interrupts are:
* - Deadlock. See the XPacketFifo component for a description of deadlock on a
*   FIFO.
*
* @param InstancePtr is a pointer to the XGemac instance to be worked on.
*
* @return
*
* Although the function returns void, it can return an asynchronous error to the
* application through the error handler.  It can return XST_FIFO_ERROR if a FIFO
* error occurred.
*
* @note
*
* This function is intended for internal use only.
*
******************************************************************************/
void
XGemac_CheckFifoSendError(XGemac * InstancePtr)
{
	/*
	 * Although the deadlock is currently the only interrupt from a packet
	 * FIFO, make sure it is deadlocked before taking action. There is no
	 * need to clear this interrupt since it requires a reset of the device.
	 */
	if (XPF_V200A_IS_DEADLOCKED(&InstancePtr->SendFifo)) {
		u32 IntrEnable;

		InstancePtr->Stats.FifoErrors++;

		/*
		 * Invoke the error callback function, which should result in a reset
		 * of the device by the upper layer software. We first need to disable
		 * the FIFO interrupt, since otherwise the upper layer thread that
		 * handles the reset may never run because this interrupt condition
		 * doesn't go away until a reset occurs (there is no way to ack it).
		 */
		IntrEnable = XIIF_V123B_READ_DIER(InstancePtr->BaseAddress);
		XIIF_V123B_WRITE_DIER(InstancePtr->BaseAddress,
			IntrEnable & ~XGE_IPIF_SEND_FIFO_MASK);

		InstancePtr->ErrorHandler(InstancePtr->ErrorRef, XST_FIFO_ERROR);
	}
}

