/* $Id: xgemac_control.c,v 1.1.4.1 2004/07/12 11:29:20 kurtsman Exp $ */
/******************************************************************************
*
*       XILINX IS PROVIDING THIS DESIGN, CODE, OR INFORMATION "AS IS"
*       AS A COURTESY TO YOU, SOLELY FOR USE IN DEVELOPING PROGRAMS AND
*       SOLUTIONS FOR XILINX DEVICES.  BY PROVIDING THIS DESIGN, CODE,
*       OR INFORMATION AS ONE POSSIBLE IMPLEMENTATION OF THIS FEATURE,
*       APPLICATION OR STANDARD, XILINX IS MAKING NO REPRESENTATION
*       THAT THIS IMPLEMENTATION IS FREE FROM ANY CLAIMS OF INFRINGEMENT,
*       AND YOU ARE RESPONSIBLE FOR OBTAINING ANY RIGHTS YOU MAY REQUIRE
*       FOR YOUR IMPLEMENTATION.  XILINX EXPRESSLY DISCLAIMS ANY
*       WARRANTY WHATSOEVER WITH RESPECT TO THE ADEQUACY OF THE
*       IMPLEMENTATION, INCLUDING BUT NOT LIMITED TO ANY WARRANTIES OR
*       REPRESENTATIONS THAT THIS IMPLEMENTATION IS FREE FROM CLAIMS OF
*       INFRINGEMENT, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
*       FOR A PARTICULAR PURPOSE.
*
*       (c) Copyright 2002 Xilinx Inc.
*       All rights reserved.
*
******************************************************************************/
/*****************************************************************************/
/**
*
* @file xgemac_control.c
*
* Functions in this file handle various control functions of the XGemac driver.
*
* <pre>
* MODIFICATION HISTORY:
*
* Who  Date     Changes
* ---- -------- -----------------------------------------------
* rmm  06/04/03 First release
* </pre>
*
******************************************************************************/

/***************************** Include Files *********************************/
#include "xbasic_types.h"
#include "xgemac.h"
#include "xgemac_l.h"
#include "xio.h"

/************************** Constant Definitions *****************************/

/**************************** Type Definitions *******************************/

/***************** Macros (Inline Functions) Definitions *********************/

/************************** Function Prototypes ******************************/

/************************** Variable Definitions *****************************/

/*****************************************************************************/
/**
*
* Set the Interframe Gap (IFG), which is the time the MAC delays between
* transmitting frames.
*
* The device must be stopped before setting the interframe gap.
*
* @param InstancePtr is a pointer to the XGemac instance to be worked on.
* @param Ifg is the interframe gap to set, the LSB is 8 bit times.
*
* @return
*
* - XST_SUCCESS if the interframe gap was set successfully
* - XST_DEVICE_IS_STARTED if the device has not been stopped
*
* @note
*
* None.
*
******************************************************************************/
XStatus
XGemac_SetInterframeGap(XGemac * InstancePtr, u8 Ifg)
{
	XASSERT_NONVOID(InstancePtr != NULL);
	XASSERT_NONVOID(InstancePtr->IsReady == XCOMPONENT_IS_READY);

	/*
	 * Be sure device has been stopped
	 */
	if (InstancePtr->IsStarted == XCOMPONENT_IS_STARTED) {
		return XST_DEVICE_IS_STARTED;
	}

	XIo_Out32(InstancePtr->BaseAddress + XGE_IFGP_OFFSET,
		Ifg << XGE_IFGP_SHIFT);

	return XST_SUCCESS;
}

/*****************************************************************************/
/**
*
* Get the interframe gap. See the description of interframe gap above in
* XGemac_SetInterframeGap().
*
* @param InstancePtr is a pointer to the XGemac instance to be worked on.
* @param IfgPtr is a pointer to an 8-bit buffer into which the interframe gap
*        value will be copied. The LSB value is 8 bit times.
*
* @return
*
* None. The values of the interframe gap parts are copied into the
* output parameters.
*
******************************************************************************/
void
XGemac_GetInterframeGap(XGemac * InstancePtr, u8 * IfgPtr)
{
	u32 Ifg;

	XASSERT_VOID(InstancePtr != NULL);
	XASSERT_VOID(IfgPtr != NULL);
	XASSERT_VOID(InstancePtr->IsReady == XCOMPONENT_IS_READY);

	Ifg = XIo_In32(InstancePtr->BaseAddress + XGE_IFGP_OFFSET);
	*IfgPtr = (Ifg & XGE_IFGP_MASK) >> XGE_IFGP_SHIFT;
}

/*****************************************************************************/
/**
*
* Send a pause packet. When called, the GEMAC hardware will initiate 
* transmission of an automatically formed pause packet. This action will not
* disrupt any frame transmission in progress but will take priority over any
* pending frame transmission. The pause frame will be sent even if the transmitter
* is in the paused state.
*
* For this function to have any effect, the XGE_FLOW_CONTROL option must be
* set (see XGemac_SetOptions)).
*
* @param InstancePtr is a pointer to the XGemac instance to be worked on.
* @param PausePeriod is the amount of time to pause. The LSB is 512 bit times.
*
* @return
*
* - XST_SUCCESS if the pause frame transmission mechanism was successfully
*   started.
* - XST_DEVICE_IS_STARTED if the device has not been stopped
*
******************************************************************************/
XStatus
XGemac_SendPause(XGemac * InstancePtr, u16 PausePeriod)
{
	XASSERT_NONVOID(InstancePtr != NULL);
	XASSERT_NONVOID(InstancePtr->IsReady == XCOMPONENT_IS_READY);

	XIo_Out32(InstancePtr->BaseAddress + XGE_TPPR_OFFSET,
		(u32)PausePeriod);
	return XST_SUCCESS;
}


/*****************************************************************************/
/**
*
* Read a PHY register through the GMII Management Control mechanism.
*
* @param InstancePtr is a pointer to the XGemac instance to be worked on.
* @param PhyAddress is the address of the PHY to be accessed. Valid range
*        is 0 to 31.
* @param Register is the register in the PHY to be accessed. Valid range
*        is 0 to 31.
* @param DataPtr is an output parameter that will contain the contents of the
*        register.
*
* @return
*
* - XST_SUCCESS if the PHY register was successfully read and its contents were
*   placed in DataPtr.
* - XST_NO_FEATURE if GMII is not present with this GEMAC instance.
* - XST_DEVICE_BUSY if another GMII read/write operation is already in progresss.
* - XST_FAILURE if an GMII read error is detected
*
* @note
*
* This function blocks until the read operation has completed. If there is a
* HW problem then this function may not return.
*
******************************************************************************/
XStatus
XGemac_MgtRead(XGemac * InstancePtr, int PhyAddress, int Register,
		u16 * DataPtr)
{
	u32 RegCr;

	XASSERT_NONVOID(InstancePtr != NULL);
	XASSERT_NONVOID(InstancePtr->IsReady == XCOMPONENT_IS_READY);
	XASSERT_NONVOID(DataPtr != NULL);
	XASSERT_NONVOID(PhyAddress <=
			(XGE_MGTCR_PHY_ADDR_MASK >> XGE_MGTCR_PHY_ADDR_SHIFT));
	XASSERT_NONVOID(Register <=
			(XGE_MGTCR_REG_ADDR_MASK >> XGE_MGTCR_REG_ADDR_SHIFT));

	/* does this instance have GMII */
	if (!InstancePtr->HasGmii) {
		return XST_NO_FEATURE;
	}

	/* verify no GMII operation is in progress */
	RegCr = XIo_In32(InstancePtr->BaseAddress + XGE_MGTCR_OFFSET);
	if (RegCr & XGE_MGTCR_START_MASK) {
		return XST_DEVICE_BUSY;
	}

	/* format the mgtcr then write it */
	RegCr =
	    XGE_MGTCR_START_MASK |
	    XGE_MGTCR_RW_NOT_MASK |
	    XGE_MGTCR_MII_ENABLE_MASK |
	    (PhyAddress << XGE_MGTCR_PHY_ADDR_SHIFT) |
	    (Register << XGE_MGTCR_REG_ADDR_SHIFT);

	XIo_Out32(InstancePtr->BaseAddress + XGE_MGTCR_OFFSET, RegCr);

	/* wait for read to complete */
	do {
		RegCr = XIo_In32(InstancePtr->BaseAddress + XGE_MGTCR_OFFSET);
	} while (RegCr & XGE_MGTCR_START_MASK);

	/* an error reading */
	if (RegCr & XGE_MGTCR_RD_ERROR_MASK) {
		return XST_FAILURE;
	}

	/* read was successfull, get the data and exit */
	*DataPtr = (u16) XIo_In32(InstancePtr->BaseAddress + XGE_MGTDR_OFFSET);

	return XST_SUCCESS;
}

/*****************************************************************************/
/**
*
* Write to a PHY register through the GMII Management Control mechanism.
*
* @param InstancePtr is a pointer to the XGemac instance to be worked on.
* @param PhyAddress is the address of the PHY to be accessed.
* @param Register is the register in the PHY to be accessed.
* @param Data is the what will be written to the register.
*
* @return
*
* - XST_SUCCESS if the PHY register was successfully read and its contents are
*   placed in DataPtr.
* - XST_DEVICE_BUSY if another GMII read/write operation is already in progresss.
* - XST_NO_FEATURE if GMII is not present with this GEMAC instance.
*
* @note
*
* This function blocks until the write operation has completed. If there is a
* HW problem then this function may not return.
*
******************************************************************************/
XStatus
XGemac_MgtWrite(XGemac * InstancePtr, int PhyAddress, int Register, u16 Data)
{
	u32 RegCr;

	XASSERT_NONVOID(InstancePtr != NULL);
	XASSERT_NONVOID(InstancePtr->IsReady == XCOMPONENT_IS_READY);
	XASSERT_NONVOID(PhyAddress <=
			(XGE_MGTCR_PHY_ADDR_MASK >> XGE_MGTCR_PHY_ADDR_SHIFT));
	XASSERT_NONVOID(Register <=
			(XGE_MGTCR_REG_ADDR_MASK >> XGE_MGTCR_REG_ADDR_SHIFT));

	/* does this instance have GMII */
	if (!InstancePtr->HasGmii) {
		return XST_NO_FEATURE;
	}

	/* verify no GMII operation is in progress */
	RegCr = XIo_In32(InstancePtr->BaseAddress + XGE_MGTCR_OFFSET);
	if (RegCr & XGE_MGTCR_START_MASK) {
		return XST_DEVICE_BUSY;
	}

	/* format the mgtcr */
	RegCr =
	    XGE_MGTCR_START_MASK |
	    XGE_MGTCR_MII_ENABLE_MASK |
	    (PhyAddress << XGE_MGTCR_PHY_ADDR_SHIFT) |
	    (Register << XGE_MGTCR_REG_ADDR_SHIFT);

	/* write the mgtdr followed by the mgtcr */
	XIo_Out32(InstancePtr->BaseAddress + XGE_MGTDR_OFFSET, (u32) Data);
	XIo_Out32(InstancePtr->BaseAddress + XGE_MGTCR_OFFSET, RegCr);

	/* wait for write to complete */
	do {
		RegCr = XIo_In32(InstancePtr->BaseAddress + XGE_MGTCR_OFFSET);
	} while (RegCr & XGE_MGTCR_START_MASK);

	/* success */
	return XST_SUCCESS;
}

