/*
 * adapter.c
 *
 * Xilinx Ethernet Adapter component to interface XGemac component to Linux
 *
 * Author: MontaVista Software, Inc.
 *         source@mvista.com
 *
 * 2002-2004 (c) MontaVista, Software, Inc.  This file is licensed under the terms
 * of the GNU General Public License version 2.1.  This program is licensed
 * "as is" without any warranty of any kind, whether express or implied.
 */

/*
 * This driver is a bit unusual in that it is composed of two logical
 * parts where one part is the OS independent code and the other part is
 * the OS dependent code.  Xilinx provides their drivers split in this
 * fashion.  This file represents the Linux OS dependent part known as
 * the Linux adapter.  The other files in this directory are the OS
 * independent files as provided by Xilinx with no changes made to them.
 * The names exported by those files begin with XGemac_.  All functions
 * in this file that are called by Linux have names that begin with
 * xenet_.  The functions in this file that have Handler in their name
 * are registered as callbacks with the underlying Xilinx OS independent
 * layer.  Any other functions are static helper functions.
 */

#include <linux/module.h>
#include <asm/uaccess.h>
#include <linux/init.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/gmii.h>
#include <linux/pci.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/atomic.h>
#include <linux/ethtool.h>
#include <asm/time.h>
#include <asm/delay.h>

#include <xbasic_types.h>
#include "xgemac.h"
#include "xgemac_i.h"
#include "xipif_v1_23_b.h"

#undef XGE_DFT_SEND_DESC
#define XGE_DFT_SEND_DESC   64
#undef XGE_DFT_RECV_DESC
#define XGE_DFT_RECV_DESC   256

#define DRIVER_NAME "Xilinx Gig Eth MAC driver"
#define DRIVER_VERSION "1.0"

MODULE_AUTHOR("MontaVista Software, Inc. <source@mvista.com>");
MODULE_DESCRIPTION(DRIVER_NAME);
MODULE_LICENSE("GPL");

#define TX_TIMEOUT   (60*HZ)	/* Transmission timeout is 60 seconds. */

#define ALIGNMENT_SEND	1

/* On the OPB, the 10/100 EMAC requires data to be aligned to 4 bytes.
 * On the PLB, the 10/100 EMAC requires data to be aligned to 8 bytes.
 * For simplicity, we always align to 8 bytes.
 */
#define ALIGNMENT_RECV	8

/* BUFFER_ALIGN(adr) calculates the number of bytes to the next alignment. */
#define BUFFER_ALIGNSEND(adr) ((ALIGNMENT_SEND - ((u32) adr)) % ALIGNMENT_SEND)
#define BUFFER_ALIGNRECV(adr) ((ALIGNMENT_RECV - ((u32) adr)) % ALIGNMENT_RECV)

#define BD_IN_BRAM        0

/* physical to virtual pointer conversion in descriptors space */
#define P_TO_V(InstancePtr, p) \
	((p) ? \
	((InstancePtr)->VirtPtr + ((u32)(p) - (u32)(InstancePtr)->PhyPtr)) : \
	 0)

int bh_entry = 0;

/*
 * Our private per device data.  When a net_device is allocated we will
 * ask for enough extra space for this.
 */
struct net_local {
	struct list_head rcv;
	XBufDescriptor * rcvBdPtr;
	int rcvBds;
	struct list_head xmit;
	XBufDescriptor * xmitBdPtr;
	int xmitBds;

	struct net_device_stats stats;	/* Statistics for this device */
	struct net_device *next_dev;	/* The next device in dev_list */
	struct net_device *dev;		/* this device */
	struct timer_list phy_timer;	/* PHY monitoring timer */

	u32 index;		/* Which interface is this */
	XInterruptHandler Isr;	/* Pointer to the XGemac ISR routine */
	u8 gmii_addr;		/* The GMII address of the PHY */
	/*
	 * The underlying OS independent code needs space as well.  A
	 * pointer to the following XGemac structure will be passed to
	 * any XGemac_ function that requires it.  However, we treat the
	 * data as an opaque object in this file (meaning that we never
	 * reference any of the fields inside of the structure).
	 */
	XGemac Emac;
	unsigned int max_frame_size;

	void *desc_space;
	dma_addr_t desc_space_handle;
	int desc_space_size;

	struct sk_buff* deferred_skb;
};

/* List of devices we're handling and a lock to give us atomic access. */
static struct net_device *dev_list = NULL;
static spinlock_t dev_lock = SPIN_LOCK_UNLOCKED;

/* for exclusion of all program flows (processes, ISRs and BHs) possible to share data with current one */
static spinlock_t reset_lock = SPIN_LOCK_UNLOCKED;

/* Helper function to determine if a given XGemac error warrants a reset. */
extern inline int
status_requires_reset(XStatus s)
{
	return (s == XST_DMA_ERROR || s == XST_FIFO_ERROR ||
		s == XST_RESET_ERROR || s == XST_DMA_SG_NO_LIST ||
		s == XST_DMA_SG_LIST_EMPTY);
}

/* BH statics */
LIST_HEAD(receivedQueue);
static spinlock_t rcvSpin = SPIN_LOCK_UNLOCKED;
LIST_HEAD(sentQueue);
static spinlock_t xmitSpin = SPIN_LOCK_UNLOCKED;

/* SAATODO: This function will be moved into the Xilinx code. */
/*****************************************************************************/
/**
*
* Lookup the device configuration based on the emac instance.  The table
* EmacConfigTable contains the configuration info for each device in the system.
*
* @param Instance is the index of the emac being looked up.
*
* @return
*
* A pointer to the configuration table entry corresponding to the given
* device ID, or NULL if no match is found.
*
* @note
*
* None.
*
******************************************************************************/
XGemac_Config *
XGemac_GetConfig(int Instance)
{
	if (Instance < 0 || Instance >= XPAR_XGEMAC_NUM_INSTANCES) {
		return NULL;
	}
	printk("XGemac: Device instance %d found\n",Instance);
	return &XGemac_ConfigTable[Instance];
}

/*
 * The following are notes regarding the critical sections in this
 * driver and how they are protected.
 *
 * dev_list
 * There is a spinlock protecting the device list.  It isn't really
 * necessary yet because the list is only manipulated at init and
 * cleanup, but it's there because it is basically free and if we start
 * doing hot add and removal of ethernet devices when the FPGA is
 * reprogrammed while the system is up, we'll need to protect the list.
 *
 * XGemac_Start, XGemac_Stop and XGemac_SetOptions are not thread safe.
 * These functions are called from xenet_open(), xenet_close(), reset(),
 * and xenet_set_multicast_list().  xenet_open() and xenet_close()
 * should be safe because when they do start and stop, they don't have
 * interrupts or timers enabled.  The other side is that they won't be
 * called while a timer or interrupt is being handled.  Next covered is
 * the interaction between reset() and xenet_set_multicast_list().  This
 * is taken care of by disabling the ethernet IRQ and bottom half when
 * the multicast list is being set.  The inverse case is covered because
 * a call to xenet_set_multicast_list() will not happen in the middle of
 * a timer or interrupt being serviced.  Finally, the interaction
 * between reset() being called from various places needs to be
 * considered.  reset() is called from poll_gmii() and xenet_tx_timeout()
 * which are timer bottom halves as well as FifoRecvHandler() and
 * ErrorHandler() which are interrupt handlers.  The timer bottom halves
 * won't interrupt an interrupt handler (or each other) so that is
 * covered.  The interrupt handlers won't interrupt each other either.
 * That leaves the case of interrupt handlers interrupting timer bottom
 * halves.  This is handled simply by disabling the interrupt when the
 * bottom half calls reset().
 *
 * XGemac_MgtRead and XGemac_MgtWrite are not thread safe.
 * These functions are called from get_phy_status(), xenet_ioctl() and
 * probe().  probe() is only called from xenet_init() so it is not an
 * issue (nothing is really up and running yet).  get_phy_status() is
 * called from both poll_gmii() (a timer bottom half) and xenet_open().
 * These shouldn't interfere with each other because xenet_open() is
 * what starts the poll_gmii() timer.  xenet_open() and xenet_ioctl()
 * should be safe as well because they will be sequential.  That leaves
 * the interaction between poll_gmii() and xenet_ioctl().  While the
 * timer bottom half is executing, a new ioctl won't come in so that is
 * taken care of.  That leaves the one case of the poll_gmii timer
 * popping while handling an ioctl.  To take care of that case, the
 * timer is deleted when the ioctl comes in and then added back in after
 * the ioctl is finished.
 */

/*
 * Helper function to reset the underlying hardware.  This is called
 * when we get into such deep trouble that we don't know how to handle
 * otherwise.
 */
typedef enum DUPLEX { UNKNOWN_DUPLEX, HALF_DUPLEX, FULL_DUPLEX } DUPLEX;

static void
reset(struct net_device *dev)
{
	struct net_local *lp = (struct net_local *) dev->priv;
	u32 Options;
	u8 IfgPart1;
	u8 SendThreshold;
	u32 SendWaitBound;
	u8 RecvThreshold;
	u32 RecvWaitBound;

        printk("reset\n");
        
	/* Shouldn't really be necessary, but shouldn't hurt. */
	netif_stop_queue(dev);

	/*
	 * XGemac_Reset puts the device back to the default state.  We need
	 * to save all the settings we don't already know, reset, restore
	 * the settings, and then restart the emac.
	 */
	XGemac_GetInterframeGap(&lp->Emac, &IfgPart1);
	Options = XGemac_GetOptions(&lp->Emac);
	Options |= XGE_FDUPLEX_OPTION;
	if (XGemac_mIsSgDma(&lp->Emac)) {
		/*
		 * The following four functions will return an error if we are
		 * not doing scatter-gather DMA.  We just checked that so we
		 * can safely ignore the return values.  We cast them to void
		 * to make that explicit.
		 */
		(void) XGemac_GetPktThreshold(&lp->Emac, XGE_SEND,
					     &SendThreshold);
		(void) XGemac_GetPktWaitBound(&lp->Emac, XGE_SEND,
					     &SendWaitBound);
		(void) XGemac_GetPktThreshold(&lp->Emac, XGE_RECV,
					     &RecvThreshold);
		(void) XGemac_GetPktWaitBound(&lp->Emac, XGE_RECV,
					     &RecvWaitBound);
	}

	XGemac_Reset(&lp->Emac);

	/*
	 * The following three functions will return an error if the
	 * EMAC is already started.  We just stopped it by calling
	 * XGemac_Reset() so we can safely ignore the return values.
	 * We cast them to void to make that explicit.
	 */
	(void) XGemac_SetMacAddress(&lp->Emac, dev->dev_addr);
	(void) XGemac_SetInterframeGap(&lp->Emac, IfgPart1);
	(void) XGemac_SetOptions(&lp->Emac, Options);
	if (XGemac_mIsSgDma(&lp->Emac)) {
		/*
		 * The following four functions will return an error if
		 * we are not doing scatter-gather DMA or if the EMAC is
		 * already started.  We just checked that we are indeed
		 * doing scatter-gather and we just stopped the EMAC so
		 * we can safely ignore the return values.  We cast them
		 * to void to make that explicit.
		 */
		(void) XGemac_SetPktThreshold(&lp->Emac, XGE_SEND,
					     SendThreshold);
		(void) XGemac_SetPktWaitBound(&lp->Emac, XGE_SEND,
					     SendWaitBound);
		(void) XGemac_SetPktThreshold(&lp->Emac, XGE_RECV,
					     RecvThreshold);
		(void) XGemac_SetPktWaitBound(&lp->Emac, XGE_RECV,
					     RecvWaitBound);
		while (!(XDmaChannel_IsSgListEmpty(&(lp->Emac.SendChannel)))) {	/* list isn't empty, has to be cleared */
			XStatus ret;
			XBufDescriptor* BdPtr;

			if ((ret = XDmaChannel_GetDescriptor (&(lp->Emac.SendChannel), &BdPtr)) != XST_SUCCESS) {
				printk (KERN_ERR "SgDma ring structure ERROR %d\n", ret);
				break;
			}
			XBufDescriptor_Unlock(BdPtr);
			pci_unmap_single(NULL,
				(u32) XBufDescriptor_GetSrcAddress(BdPtr),
				 XBufDescriptor_GetLength(BdPtr), PCI_DMA_TODEVICE);
			lp->stats.tx_errors++;
		}
	} else {
		if (lp->deferred_skb) {
			dev_kfree_skb(lp->deferred_skb);
			lp->deferred_skb = NULL;
			lp->stats.tx_errors++;
		}
	}

	/*
	 * XGemac_Start returns an error when: it is already started, the send
	 * and receive handlers are not set, or a scatter-gather DMA list is
	 * missing.  None of these can happen at this point, so we cast the
	 * return to void to make that explicit.
	 */
	(void) XGemac_Start(&lp->Emac);

	/* We're all ready to go.  Start the queue in case it was stopped. */
	if (!bh_entry)
		netif_wake_queue(dev);
}

static int
get_phy_status(struct net_device *dev, DUPLEX * duplex, int *linkup)
{
	struct net_local *lp = (struct net_local *) dev->priv;
	u16 reg;
	XStatus xs;

	xs = XGemac_MgtRead(&lp->Emac, lp->gmii_addr, GMII_BMCR, &reg);
	if (xs != XST_SUCCESS) {
		printk(KERN_ERR
		       "%s: Could not read PHY control register; error %d\n",
		       dev->name, xs);
		return -1;
	}
	*duplex = (reg & BMCR_FULLDPLX) ? FULL_DUPLEX : HALF_DUPLEX;

	xs = XGemac_MgtRead(&lp->Emac, lp->gmii_addr, GMII_BMSR, &reg);
	if (xs != XST_SUCCESS) {
		printk(KERN_ERR
		       "%s: Could not read PHY status register; error %d\n",
		       dev->name, xs);
		return -1;
	}
	*linkup = (reg & BMSR_LSTATUS) != 0;

	return 0;
}

/*
 * This routine is used for two purposes.  The first is to keep the
 * EMAC's duplex setting in sync with the PHY's.  The second is to keep
 * the system apprised of the state of the link.  Note that this driver
 * does not configure the PHY.  Either the PHY should be configured for
 * auto-negotiation or it should be handled by something like mii-tool.
 */
static void
poll_gmii(unsigned long data)
{
	struct net_device *dev = (struct net_device *) data;
	struct net_local *lp = (struct net_local *) dev->priv;
	u32 Options;
	DUPLEX phy_duplex, mac_duplex;
	int phy_carrier, netif_carrier;
	unsigned long flags;
	/* First, find out what's going on with the PHY. */
	if (get_phy_status(dev, &phy_duplex, &phy_carrier)) {
		printk(KERN_ERR "%s: Terminating link monitoring.\n",
		       dev->name);
		return;
	}

	/* Second, figure out if we have the EMAC in half or full duplex. */
	Options = XGemac_GetOptions(&lp->Emac);
	mac_duplex = (Options & XGE_FDUPLEX_OPTION) ? FULL_DUPLEX : HALF_DUPLEX;

	/* Now see if there is a mismatch. */
	if ((mac_duplex != phy_duplex) || (mac_duplex != FULL_DUPLEX)) {	/* Normally Full duplex have to be always */
		/*
		 * Make sure that no interrupts come in that could cause
		 * reentrancy problems in reset.
		 */
		spin_lock_irqsave(reset_lock, flags);
		reset(dev);		/* the function sets Emac options to match the PHY */
		spin_unlock_irqrestore(reset_lock, flags);
		printk(KERN_INFO "%s: Duplex has been changed, isn't FULL DUPLEX\n",
			dev->name);
	}

	netif_carrier = netif_carrier_ok(dev) != 0;

	if (phy_carrier != netif_carrier) {
		if (phy_carrier) {
			printk(KERN_INFO "%s: Link carrier restored.\n",
			       dev->name);
			netif_carrier_on(dev);
		} else {
			printk(KERN_INFO "%s: Link carrier lost.\n", dev->name);
			netif_carrier_off(dev);
		}
	}

	/* Set up the timer so we'll get called again in 2 seconds. */
	lp->phy_timer.expires = jiffies + 2 * HZ;
	add_timer(&lp->phy_timer);
}

/*
 * This routine is registered with the OS as the function to call when
 * the EMAC interrupts.  It in turn, calls the Xilinx OS independent
 * interrupt function.  There are different interrupt functions for FIFO
 * and scatter-gather so we just set a pointer (Isr) into our private
 * data so we don't have to figure it out here.  The Xilinx OS
 * independent interrupt function will in turn call any callbacks that
 * we have registered for various conditions.
 */
static void
xenet_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct net_device *dev = dev_id;
	struct net_local *lp = (struct net_local *) dev->priv;

	/* Call it. */
	(*(lp->Isr)) (&lp->Emac);
}

static int
xenet_open(struct net_device *dev)
{
	struct net_local *lp = (struct net_local *) dev->priv;
	u32 Options;
	DUPLEX phy_duplex, mac_duplex;
	int phy_carrier;

	/*
	 * Just to be safe, stop the device first.  If the device is already
	 * stopped, an error will be returned.  In this case, we don't really
	 * care, so cast it to void to make it explicit.
	 */
	(void) XGemac_Stop(&lp->Emac);

	/* Set the MAC address each time opened. */
	if (XGemac_SetMacAddress(&lp->Emac, dev->dev_addr) != XST_SUCCESS) {
		printk(KERN_ERR "%s: Could not set MAC address.\n", dev->name);
		return -EIO;
	}

	/*
	 * If the device is not configured for polled mode, connect to the
	 * interrupt controller and enable interrupts.  Currently, there
	 * isn't any code to set polled mode, so this check is probably
	 * superfluous.
	 */
	Options = XGemac_GetOptions(&lp->Emac);
	printk("xenet_open; Options: %x\n", Options);
	Options |= XGE_FDUPLEX_OPTION | XGE_FLOW_CONTROL_OPTION;
	Options |= XGE_NO_SGEND_INT_OPTION;
#ifdef CONFIG_XILINX_GIGE_JUMBO
	Options |= XGE_JUMBO_OPTION;
#else
	Options &= ~XGE_JUMBO_OPTION;
#endif
	(void) XGemac_SetOptions(&lp->Emac, Options);
        Options = XGemac_GetOptions(&lp->Emac);
        printk("xenet_open; Options: %x\n", Options);
 
        if ((Options & XGE_POLLED_OPTION) == 0) {
		int retval;
		/* Grab the IRQ */
		retval =
		    request_irq(dev->irq, &xenet_interrupt, 0, dev->name, dev);
		if (retval) {
			printk(KERN_ERR
			       "%s: Could not allocate interrupt %d.\n",
			       dev->name, dev->irq);
			return retval;
		}
	}

	/* Set the EMAC's duplex setting based upon what the PHY says. */
	if (!get_phy_status(dev, &phy_duplex, &phy_carrier)) {
		/* We successfully got the PHY status. */
		mac_duplex = ((Options & XGE_FDUPLEX_OPTION)
			      ? FULL_DUPLEX : HALF_DUPLEX);
		if (mac_duplex != phy_duplex) {
			switch (phy_duplex) {
			case HALF_DUPLEX:
				Options &= ~XGE_FDUPLEX_OPTION;
				break;
			case FULL_DUPLEX:
				Options |= XGE_FDUPLEX_OPTION;
				break;
			case UNKNOWN_DUPLEX:
				break;
			}
			/*
			 * The following function will return an error
			 * if the EMAC is already started.  We know it
			 * isn't started so we can safely ignore the
			 * return value.  We cast it to void to make
			 * that explicit.
			 */
			(void) XGemac_SetOptions(&lp->Emac, Options);
		}
	}

	INIT_LIST_HEAD(&(lp->rcv));
	lp->rcvBds = 0;
	INIT_LIST_HEAD(&(lp->xmit));
	lp->xmitBds = 0;

	if (XGemac_Start(&lp->Emac) != XST_SUCCESS) {
		printk(KERN_ERR "%s: Could not start device.\n", dev->name);
		free_irq(dev->irq, dev);
		return -EBUSY;
	}
        
        {
          u8 vals, valr;
          (void) XGemac_GetPktThreshold(&lp->Emac, XGE_SEND,
                                       &vals);
          (void) XGemac_GetPktThreshold(&lp->Emac, XGE_RECV,
                                       &valr);
          printk("%s: Send Threshold: %d, Receive Threshold: %d\n",
                 dev->name, vals, valr);
        }
        
	/* We're ready to go. */
	MOD_INC_USE_COUNT;
	netif_start_queue(dev);

	/* Set up the PHY monitoring timer. */
	lp->phy_timer.expires = jiffies + 2 * HZ;
	lp->phy_timer.data = (unsigned long) dev;
	lp->phy_timer.function = &poll_gmii;
	add_timer(&lp->phy_timer);
	return 0;
}
static int
xenet_close(struct net_device *dev)
{
	struct net_local *lp = (struct net_local *) dev->priv;
	unsigned long flags;

	/* Shut down the PHY monitoring timer. */
	del_timer_sync(&lp->phy_timer);

	netif_stop_queue(dev);

	/*
	 * If not in polled mode, free the interrupt.  Currently, there
	 * isn't any code to set polled mode, so this check is probably
	 * superfluous.
	 */
	if ((XGemac_GetOptions(&lp->Emac) & XGE_POLLED_OPTION) == 0)
		free_irq(dev->irq, dev);

	spin_lock_irqsave(rcvSpin, flags);
	list_del(&(lp->rcv));
	spin_unlock_irqrestore(rcvSpin, flags);
	spin_lock_irqsave(xmitSpin, flags);
	list_del(&(lp->xmit));
	spin_unlock_irqrestore(xmitSpin, flags);

	if (XGemac_Stop(&lp->Emac) != XST_SUCCESS) {
		printk(KERN_ERR "%s: Could not stop device.\n", dev->name);
		return -EBUSY;
	}

	MOD_DEC_USE_COUNT;
	return 0;
}

static struct net_device_stats *
xenet_get_stats(struct net_device *dev)
{
	struct net_local *lp = (struct net_local *) dev->priv;
	return &lp->stats;
}

static int
xenet_change_mtu(struct net_device *dev, int new_mtu)
{
	struct net_local *lp = (struct net_local *) dev->priv;
	int max_frame = new_mtu + XGE_HDR_SIZE + XGE_TRL_SIZE;
	int min_frame = XGE_HDR_SIZE + XGE_TRL_SIZE +1;

	if((max_frame < min_frame) ||
			(max_frame > lp->max_frame_size))
		return -EINVAL;
	dev->mtu = new_mtu;
	return 0;
}

static int
xenet_FifoSend(struct sk_buff *orig_skb, struct net_device *dev)
{
	struct net_local *lp = (struct net_local *) dev->priv;
	struct sk_buff *new_skb;
	unsigned int len;
	XStatus result;
	unsigned long flags;

	len = orig_skb->len;

        new_skb = orig_skb;
        spin_lock_irqsave(reset_lock, flags);
	if ((result = XGemac_FifoSend(&lp->Emac, (u8 *) new_skb->data, len)) ==
			XST_PFIFO_NO_ROOM) {
		netif_stop_queue(dev);
		lp->deferred_skb = new_skb;
		spin_unlock_irqrestore(reset_lock, flags);
		return 0;
	}
	spin_unlock_irqrestore(reset_lock, flags);
        dev_kfree_skb(new_skb);
	if (result != XST_SUCCESS) {
		lp->stats.tx_errors++;
		return -EIO;
	} else {
		lp->stats.tx_bytes += len;
		dev->trans_start = jiffies;
	}
	return 0;
}

/* The callback function for completed frames sent in FIFO mode. */
static void
FifoSendHandler(void *CallbackRef)
{
	struct net_device *dev = (struct net_device *) CallbackRef;
	struct net_local *lp = (struct net_local *) dev->priv;
	XStatus result;

	if (lp->deferred_skb) {
		if ((result = XGemac_FifoSend(&lp->Emac, (u8 *) lp->deferred_skb->data, lp->deferred_skb->len)) ==
				XST_PFIFO_NO_ROOM) {
			return;
		} else {
			dev_kfree_skb(lp->deferred_skb);
			lp->deferred_skb = NULL;
			netif_wake_queue(dev);
		}
	}
	if (result == XST_SUCCESS)
		lp->stats.tx_packets++;
	else
		lp->stats.tx_errors++;
}

/* The send function for frames sent in DMA mode. */
static int
xenet_SgSend(struct sk_buff *orig_skb, struct net_device *dev)
{
	struct net_local *lp = (struct net_local *) dev->priv;
	unsigned int len;
	struct sk_buff *new_skb;
	XBufDescriptor bd;
	int result;
	dma_addr_t new_skb_vaddr;
	unsigned long flags;

	len = orig_skb->len;

	/* we must be long aligned for dma h/w */
	if (BUFFER_ALIGNSEND(orig_skb->data)) {
		printk("%s#%d: software ERROR: alignment here isn't necessary, but data unaligned by (%d) bytes\n",
			__FUNCTION__,__LINE__,
			BUFFER_ALIGNSEND(orig_skb->data));
		return -EIO;
	} else {
		new_skb = orig_skb;
	}
	new_skb_vaddr = (u32) pci_map_single(NULL, new_skb->data,
					     new_skb->len, PCI_DMA_TODEVICE);
	/*
	 * lock the buffer descriptor to prevent lower layers from reusing
	 * it before the adapter has a chance to deallocate the buffer
	 * attached to it. The adapter will unlock it in the callback function
	 * that handles confirmation of transmits
	 */
	XBufDescriptor_Initialize(&bd);
	XBufDescriptor_Lock(&bd);
	XBufDescriptor_SetControl(&bd, XDC_DMACR_DRE_MODE_MASK);
	XBufDescriptor_SetSrcAddress(&bd, new_skb_vaddr);
	XBufDescriptor_SetLength(&bd, new_skb->len);
	XBufDescriptor_SetId(&bd, new_skb);
	XBufDescriptor_SetLast(&bd);

	spin_lock_irqsave(reset_lock, flags);
	result = XGemac_SgSend(&lp->Emac, &bd, XGE_SGDMA_NODELAY);
	if (result != XST_SUCCESS) {
		dev_kfree_skb(new_skb);
		lp->stats.tx_dropped++;
		printk(KERN_ERR "%s: Could not send transmit buffer (%d).\n",
		       dev->name, result);
		spin_unlock_irqrestore(reset_lock, flags);
                /* for some reason the kernel doesn't like -EBUSY here,
                 * so just return 0 and let the stack handle dropped packets.
                 */
                /*		return -EBUSY;	*/
                return 0;
	}
	dev->trans_start = jiffies;
	spin_unlock_irqrestore(reset_lock, flags);
	return 0;
}

/* The callback function for completed frames sent in DMA mode. */
static void SgSendHandlerBH (unsigned long p);
static void SgRecvHandlerBH (unsigned long p);

DECLARE_TASKLET (SgSendBH, SgSendHandlerBH, 0);
DECLARE_TASKLET (SgRecvBH, SgRecvHandlerBH, 0);

static void
SgSendHandlerBH (unsigned long p)
{
	struct net_device *dev;
	struct net_local *lp;
	XBufDescriptor * BdPtr;
	u32 NumBds;
	u32 len;
	XBufDescriptor *curbd;
	unsigned long flags;
	struct sk_buff *skb;
	dma_addr_t skb_baddr;

	while (1) {
		spin_lock_irqsave(xmitSpin,flags);
		if (list_empty(&sentQueue)) {
			spin_unlock_irqrestore(xmitSpin,flags);
			break;
 		}

		lp = list_entry(sentQueue.next, struct net_local, xmit);
		list_del_init(&(lp->xmit));
		NumBds = lp->xmitBds;
		bh_entry--;
		BdPtr = lp->xmitBdPtr;
		dev = lp->dev;
		while(NumBds != 0) {
			NumBds--;

			len = XBufDescriptor_GetLength(BdPtr);
			skb_baddr = (dma_addr_t) XBufDescriptor_GetSrcAddress(BdPtr);
			pci_unmap_single(NULL,
				skb_baddr,
				len, PCI_DMA_TODEVICE);

			lp->stats.tx_bytes += len;
			lp->stats.tx_packets++;

			/* get ptr to skb */
			skb = (struct sk_buff *) XBufDescriptor_GetId(BdPtr);
			dev_kfree_skb_any(skb);
			curbd = BdPtr;
			BdPtr = P_TO_V(&lp->Emac.SendChannel,
				XBufDescriptor_GetNextPtr(BdPtr));
			XBufDescriptor_Unlock(curbd);
 		}
		netif_wake_queue(dev);
		spin_unlock_irqrestore(xmitSpin,flags);
	}
}
static void
SgSendHandler(void *CallBackRef, XBufDescriptor * BdPtr, u32 NumBds)
{
 	struct net_device *dev = (struct net_device *) CallBackRef;
 	struct net_local *lp = (struct net_local *) dev->priv;
	struct list_head* cur_lp = NULL;

	spin_lock(xmitSpin);
	list_for_each (cur_lp, &sentQueue) {
		if (cur_lp == &(lp->xmit)) {
			lp->xmitBds += NumBds;
			break;
		}
	}
	if (cur_lp != &(lp->xmit)) {
		lp->xmitBds = NumBds;
		lp->xmitBdPtr = BdPtr;
		list_add_tail(&lp->xmit,&sentQueue);
		bh_entry++;
		tasklet_schedule (&SgSendBH);
 	}
	spin_unlock(xmitSpin);
}

static void
xenet_tx_timeout(struct net_device *dev)
{
	struct net_local *lp = (struct net_local *) dev->priv;
	unsigned long flags;

	printk("%s: Exceeded transmit timeout of %lu ms.  Resetting emac.\n",
	       dev->name, TX_TIMEOUT * 1000UL / HZ);

	lp->stats.tx_errors++;

	/*
	 * Make sure that no interrupts come in that could cause reentrancy
	 * problems in reset.
	 */
	spin_lock_irqsave(reset_lock, flags);
	reset(dev);
	spin_unlock_irqrestore(reset_lock, flags);
}

/* The callback function for frames received when in FIFO mode. */
static void
FifoRecvHandler(void *CallbackRef)
{
	struct net_device *dev = (struct net_device *) CallbackRef;
	struct net_local *lp = (struct net_local *) dev->priv;
	struct sk_buff *skb;
	unsigned int align;
	u32 len;
	XStatus Result;

	/*
	 * The OS independent Xilinx EMAC code does not provide a
	 * function to get the length of an incoming packet and a
	 * separate call to actually get the packet data.  It does this
	 * because they didn't add any code to keep the hardware's
	 * receive length and data FIFOs in sync.  Instead, they require
	 * that you send a maximal length buffer so that they can read
	 * the length and data FIFOs in a single chunk of code so that
	 * they can't get out of sync.  So, we need to allocate an skb
	 * that can hold a maximal sized packet.  The OS independent
	 * code needs to see the data 32/64-bit aligned, so we tack on an
	 * extra four just in case we need to do an skb_reserve to get
	 * it that way.
	 */
	len = lp->max_frame_size;
	if (!(skb = /*dev_ */ alloc_skb(len + ALIGNMENT_RECV, GFP_ATOMIC))) {
		/* Couldn't get memory. */
		lp->stats.rx_dropped++;
		printk(KERN_ERR "%s: Could not allocate receive buffer.\n",
		       dev->name);
		return;
	}

	/*
	 * A new skb should have the data word aligned, but this code is
	 * here just in case that isn't true...  Calculate how many
	 * bytes we should reserve to get the data to start on a word
	 * boundary.  */
	align = BUFFER_ALIGNRECV(skb->data);
	if (align)
		skb_reserve(skb, align);
	Result = XGemac_FifoRecv(&lp->Emac, (u8 *) skb->data, &len);
	if (Result != XST_SUCCESS) {
		int need_reset = status_requires_reset(Result);

		lp->stats.rx_errors++;
		dev_kfree_skb(skb);
		printk(KERN_ERR "%s: Could not receive buffer, error=%d.\n",
		       		dev->name, Result);
		if (need_reset)
			reset(dev);

		return;
	}

	skb_put(skb, len);	/* Tell the skb how much data we got. */
	skb->dev = dev;		/* Fill out required meta-data. */
	skb->protocol = eth_type_trans(skb, dev);
	/* skb->ip_summed = CHECKSUM_NONE; */

	lp->stats.rx_packets++;
	lp->stats.rx_bytes += len;

	netif_rx(skb);		/* Send the packet upstream. */
}

static void
SgRecvHandlerBH (unsigned long p)
{
	struct net_device *dev;
	struct net_local *lp;
	struct sk_buff *skb, *new_skb;
	u32 len, skb_baddr, new_skb_baddr;
	u32 align;
	XStatus result;
	XBufDescriptor *curbd;
	unsigned long flags;
	u32 NumBds;
	XBufDescriptor * BdPtr;

	while (1) {
		spin_lock_irqsave(rcvSpin,flags);
		if (list_empty(&receivedQueue)) {
			spin_unlock_irqrestore(rcvSpin,flags);
			break;
		}
		lp = list_entry(receivedQueue.next, struct net_local, rcv);
		list_del_init(&(lp->rcv));
		NumBds = lp->rcvBds;
		BdPtr = lp->rcvBdPtr;
		dev = lp->dev;
		spin_unlock_irqrestore(rcvSpin,flags);
		while (NumBds != 0) {
			NumBds--;

			/* get ptr to skb */
			skb = (struct sk_buff *) XBufDescriptor_GetId(BdPtr);
			len = XBufDescriptor_GetLength(BdPtr);

			/* we have all the information we need - move on */
			curbd = BdPtr;
			BdPtr = P_TO_V(&lp->Emac.RecvChannel,
				XBufDescriptor_GetNextPtr(curbd));
			skb_baddr = (dma_addr_t)XBufDescriptor_GetDestAddress(curbd);
			pci_unmap_single(NULL, skb_baddr, len, PCI_DMA_FROMDEVICE);

			/* replace skb with a new one */
			new_skb = alloc_skb(lp->max_frame_size + ALIGNMENT_RECV, GFP_ATOMIC);
			if (new_skb == NULL) {
				printk("SgRecvHandler: no mem for new_skb\n");
				return;
			}
			/* make sure we're long-word aligned */
			align = BUFFER_ALIGNRECV(new_skb->data);
			if (align) {
				skb_reserve(new_skb, align);
			}
			new_skb_baddr = (u32) pci_map_single(NULL, new_skb->data,
						     lp->max_frame_size,
						     PCI_DMA_FROMDEVICE);

			XBufDescriptor_SetDestAddress(curbd, new_skb_baddr);
			XBufDescriptor_SetLength(curbd, lp->max_frame_size);
			XBufDescriptor_SetId(curbd, new_skb);
			XBufDescriptor_Unlock(curbd);

			/* give the descriptor back to the driver */
			result = XGemac_SgRecv(&lp->Emac, curbd);
			if (result != XST_SUCCESS) {
				printk(KERN_ERR "SgRecvHandler: SgRecv unsuccessful (%d)\n",result);
				return;
			}

			/* back to the original skb */
			skb->len = len;
			skb->dev = dev;
			skb->protocol = eth_type_trans(skb, dev);
			skb->ip_summed = CHECKSUM_NONE;
			lp->stats.rx_packets++;
			lp->stats.rx_bytes += len;
			netif_rx(skb);          /* Send the packet upstream. */
		}
	}
}

static void
SgRecvHandler(void *CallBackRef, XBufDescriptor * BdPtr, u32 NumBds)
{
	struct net_device *dev = (struct net_device *) CallBackRef;
	struct net_local *lp = (struct net_local *) dev->priv;
	struct list_head* cur_lp = NULL;

	spin_lock(rcvSpin);
	list_for_each (cur_lp, &receivedQueue) {
		if (cur_lp == &(lp->rcv)) {
			lp->rcvBds += NumBds;
			break;
		}
	}
	if (cur_lp != &(lp->rcv)) {
		lp->rcvBds = NumBds;
		lp->rcvBdPtr = BdPtr;
		list_add_tail(&lp->rcv, &receivedQueue);
		tasklet_schedule (&SgRecvBH);
	}
	spin_unlock(rcvSpin);
}

/* The callback function for errors. */
static void
ErrorHandler(void *CallbackRef, XStatus Code)
{
	struct net_device* dev = (struct net_device*) CallbackRef;
	int need_reset;
	unsigned long flags;

	need_reset = status_requires_reset(Code);
	printk(KERN_ERR "%s: device error %d%s\n",
	       dev->name, Code,need_reset ? ", resetting device." : "");
	if (need_reset) {
		spin_lock_irqsave(reset_lock, flags);
 		reset(dev);
		spin_unlock_irqrestore(reset_lock, flags);
	}
}

static int
descriptor_init(struct net_device *dev)
{
	struct net_local *lp = (struct net_local *) dev->priv;
	int i, recvsize, sendsize;
	int dftsize;
	u32 *recvpoolptr, *sendpoolptr;
	void *recvpoolphy, *sendpoolphy;
	XStatus result;

	/* calc size of descriptor space pool; alloc from non-cached memory */
	dftsize = (XGE_DFT_RECV_DESC + XGE_DFT_SEND_DESC) *
	    sizeof (XBufDescriptor);
        
#if BD_IN_BRAM == 0
	lp->desc_space = consistent_alloc(GFP_ATOMIC, dftsize,
					  &lp->desc_space_handle);
#else
        lp->desc_space_handle = 0xffff8000;
        lp->desc_space = ioremap(lp->desc_space_handle, dftsize);
#endif
	if (lp->desc_space == 0) {
		return -1;
	}

        printk("desc_init; phy: %x, virt: %x, size: %x\n",
               lp->desc_space_handle, (unsigned int)lp->desc_space, dftsize);
        
	lp->desc_space_size = dftsize;

	/* calc size of send and recv descriptor space */
	recvsize = XGE_DFT_RECV_DESC * sizeof (XBufDescriptor);
	sendsize = XGE_DFT_SEND_DESC * sizeof (XBufDescriptor);

	recvpoolptr = lp->desc_space;
	sendpoolptr = (void *) ((u32) lp->desc_space + recvsize);

	recvpoolphy = (void *) lp->desc_space_handle;
	sendpoolphy = (void *) ((u32) lp->desc_space_handle + recvsize);

	/* add ptr to descriptor space to the driver */
	if ((result = XGemac_SetSgRecvSpace(&lp->Emac, recvpoolptr, recvsize, recvpoolphy)) != XST_SUCCESS) {
		printk (KERN_ERR "XGemac_SetSgRecvSpace ERROR %d\n", result);
		return -EIO;
	}
	if ((result = XGemac_SetSgSendSpace(&lp->Emac, sendpoolptr, sendsize, sendpoolphy)) != XST_SUCCESS) {
		printk (KERN_ERR "XGemac_SetSgSendSpace ERROR %d\n", result);
		return -EIO;
	}
	/* allocate skb's and give them to the dma engine */
	for (i = 0; i < XGE_DFT_RECV_DESC; i++) {
		struct sk_buff *skb;
		XBufDescriptor bd;
		int result;
		u32 skb_vaddr, align;

		skb = alloc_skb(lp->max_frame_size + ALIGNMENT_RECV, GFP_ATOMIC);
		if (skb == 0) {
			return -1;
		}

		align = BUFFER_ALIGNRECV(skb->data);
		if (align)
			skb_reserve(skb, align);

		skb_vaddr = (u32) pci_map_single(NULL, skb->data,
						 lp->max_frame_size,
						 PCI_DMA_FROMDEVICE);

		/*
		 * initialize descriptors and set buffer address
		 * buffer length gets max frame size
		 */
		XBufDescriptor_Initialize(&bd);
		XBufDescriptor_Lock(&bd);
		XBufDescriptor_SetDestAddress(&bd, skb_vaddr);
		XBufDescriptor_SetLength(&bd, lp->max_frame_size);
		XBufDescriptor_SetId(&bd, skb);

		/*
		 * descriptor with attached buffer to the driver and
		 * let it make it ready for frame reception
		 */
		result = XGemac_SgRecv(&lp->Emac, &bd);
		if (result != XST_SUCCESS) {
			return -1;
		}
	}
	return 0;
}

void
free_descriptor_skb (struct net_device *dev)
{
	struct net_local *lp = (struct net_local *) dev->priv;
	int i;
	XBufDescriptor* BdPtr;
	struct sk_buff* skb;
	dma_addr_t mappedAddr;

	BdPtr = (XBufDescriptor*)lp->Emac.RecvChannel.VirtPtr;
	for (i=0; i<XGE_DFT_RECV_DESC; i++) {
		skb = (struct sk_buff*)XBufDescriptor_GetId(BdPtr);
		mappedAddr = (dma_addr_t)XBufDescriptor_GetDestAddress(BdPtr);
		pci_unmap_single(NULL, mappedAddr, XBufDescriptor_GetLength(BdPtr), PCI_DMA_FROMDEVICE);
		dev_kfree_skb(skb);
		BdPtr = P_TO_V(&lp->Emac.RecvChannel,XBufDescriptor_GetNextPtr(BdPtr));
	}
}

static int
xenet_ethtool_get_settings (struct net_device *dev, struct ethtool_cmd* ecmd)
{
	int ret;
	struct net_local *lp = (struct net_local *) dev->priv;
	u32 mac_options;
	u8 threshold;
	u16 gmii_cmd;
	u16 gmii_status;
	u16 gmii_advControl;
	XStatus xs;
 
	memset (ecmd, 0, sizeof(struct ethtool_cmd));
	mac_options = XGemac_GetOptions (&(lp->Emac));
	xs = XGemac_MgtRead(&lp->Emac, lp->gmii_addr, GMII_BMCR, &gmii_cmd);
	if (xs != XST_SUCCESS) {
		printk(KERN_ERR
		       "%s: Could not read gmii command register; error %d\n",
		       dev->name, xs);
		return -1;
	}
	xs = XGemac_MgtRead(&lp->Emac, lp->gmii_addr, GMII_BMSR, &gmii_status);
	if (xs != XST_SUCCESS) {
		printk(KERN_ERR
		       "%s: Could not read gmii status register; error %d\n",
		       dev->name, xs);
		return -1;
	}
	xs = XGemac_MgtRead(&lp->Emac, lp->gmii_addr, GMII_ADVERTISE, &gmii_advControl);
	if (xs != XST_SUCCESS) {
		printk(KERN_ERR
		       "%s: Could not read gmii advertisement control register; error %d\n",
		       dev->name, xs);
		return -1;
	}
 
	if (mac_options & XGE_FDUPLEX_OPTION)
		ecmd->duplex = DUPLEX_FULL;
	else
		ecmd->duplex = DUPLEX_HALF;
	if (lp->Emac.HasGmii) {
		ecmd->supported |= SUPPORTED_MII;
	} else
		ecmd->supported &= (~SUPPORTED_MII);
	ecmd->port = PORT_MII;
	if ((gmii_cmd & BMCR_SPEEDMSB) && !(gmii_cmd & BMCR_SPEEDLSB))
		ecmd->speed = SPEED_1000;
	if (gmii_status & BMSR_ANEGCAPABLE)
		ecmd->supported |= SUPPORTED_Autoneg;
	if (gmii_status & BMSR_ANEGCOMPLETE) {
		ecmd->autoneg = AUTONEG_ENABLE;
		ecmd->advertising |= ADVERTISED_Autoneg;
		if (gmii_advControl & ADVERTISE_FULL)
			ecmd->speed = SPEED_1000;
		else
			return -1;
	} else {
		ecmd->autoneg = AUTONEG_DISABLE;
		if (gmii_cmd & BMCR_SPEEDMSB)
			ecmd->speed = SPEED_1000;
		else
			return -1;
	}
	ecmd->phy_address = lp->Emac.PhysAddress;
	ecmd->transceiver = XCVR_INTERNAL;
	if (XGemac_mIsSgDma(&lp->Emac)) {
		if ((ret = XGemac_GetPktThreshold(&lp->Emac, XGE_SEND, &threshold)) == XST_SUCCESS) {
			ecmd->maxtxpkt = threshold;
		} else
			return -EIO;
		if ((ret = XGemac_GetPktThreshold(&lp->Emac, XGE_RECV, &threshold)) == XST_SUCCESS) {
			ecmd->maxrxpkt = threshold;
		} else
			return -EIO;
	}
	return 0;
}
 
static int
xenet_ethtool_get_coalesce (struct net_device *dev, struct ethtool_coalesce* ec)
{
	int ret;
	struct net_local *lp = (struct net_local *) dev->priv;
	u8 threshold;

	memset (ec, 0, sizeof(struct ethtool_coalesce));
	if ((ret = XGemac_GetPktThreshold(&lp->Emac, XGE_RECV, &threshold)) != XST_SUCCESS) {
		printk(KERN_INFO "XGemac_GetPktThreshold error %d\n", ret);
		return -EIO;
	}
	ec->rx_max_coalesced_frames = threshold;
	if ((ret = XGemac_GetPktWaitBound (&lp->Emac, XGE_RECV, &(ec->rx_coalesce_usecs))) != XST_SUCCESS) {
		printk (KERN_INFO "XGemac_GetPktWaitBound error %d\n", ret);
		return -EIO;
	}
	if ((ret = XGemac_GetPktThreshold(&lp->Emac, XGE_SEND, &threshold)) != XST_SUCCESS) {
		printk (KERN_INFO "XGemac_GetPktThreshold send error %d\n", ret);
		return -EIO;
	}
	ec->tx_max_coalesced_frames = threshold; 
	if ((ret = XGemac_GetPktWaitBound (&lp->Emac, XGE_SEND, &(ec->tx_coalesce_usecs))) != XST_SUCCESS) {
		printk (KERN_INFO "XGemac_GetPktWaitBound send error %d\n", ret);
		return -EIO;
	}
	return 0;
}

static int
xenet_ethtool_set_coalesce (struct net_device *dev, struct ethtool_coalesce* ec)
{
	int ret;
	struct net_local *lp = (struct net_local *) dev->priv;
	unsigned long flags;

	spin_lock_irqsave(reset_lock, flags);
	if ((ret = XGemac_Stop(&lp->Emac)) != XST_SUCCESS)
		return -EIO;
	if ((ret = XGemac_SetPktThreshold(&lp->Emac, XGE_RECV, ec->rx_max_coalesced_frames)) != XST_SUCCESS) {
		printk (KERN_INFO "XGemac_SetPktThreshold error %d\n", ret);
		return -EIO;
	} 
	if ((ret = XGemac_SetPktWaitBound (&lp->Emac, XGE_RECV, ec->rx_coalesce_usecs)) != XST_SUCCESS) {
		printk (KERN_INFO "XGemac_SetPktWaitBound error %d\n", ret);
		return -EIO;
	}
	if ((ret = XGemac_SetPktThreshold(&lp->Emac, XGE_SEND, ec->tx_max_coalesced_frames)) != XST_SUCCESS) {
		printk (KERN_INFO "XGemac_SetPktThreshold send error %d\n", ret);
		return -EIO;
	} 
	if ((ret = XGemac_SetPktWaitBound (&lp->Emac, XGE_SEND, ec->tx_coalesce_usecs)) != XST_SUCCESS) {
		printk (KERN_INFO "XGemac_SetPktWaitBound send error %d\n", ret);
		return -EIO;
	}
	if ((ret = XGemac_Start(&lp->Emac)) != XST_SUCCESS)
		return -EIO;
	spin_unlock_irqrestore(reset_lock, flags);
	return 0;
}

static int 
xenet_ethtool_get_drvinfo (struct net_device *dev, struct ethtool_drvinfo* ed)
{
	memset (ed, 0, sizeof(struct ethtool_drvinfo));
	strcpy (ed->driver, DRIVER_NAME);
	strcpy (ed->version, DRIVER_VERSION);
	return 0; 
}

static int
xenet_ethtool_get_ringparam (struct net_device *dev, struct ethtool_ringparam* erp)
{
	memset (erp, 0, sizeof(struct ethtool_ringparam));
	erp->rx_max_pending = XGE_DFT_RECV_DESC;
	erp->tx_max_pending = XGE_DFT_SEND_DESC;
	erp->rx_pending = XGE_DFT_RECV_DESC;
	erp->tx_pending = XGE_DFT_SEND_DESC;
	return 0;
}

#define EMAC_REGS_N	32
struct mac_regsDump {
	struct ethtool_regs hd;
	u16 data[EMAC_REGS_N];
};

static void
xenet_ethtool_get_regs (struct net_device *dev, struct ethtool_regs* regs, void* ret)
{
	struct net_local *lp = (struct net_local *) dev->priv;
	struct mac_regsDump* dump = (struct mac_regsDump*)regs;
	int i;
	XStatus r;

	dump->hd.version = 0;
	dump->hd.len = EMAC_REGS_N * sizeof(dump->data);
	for (i=0; i<EMAC_REGS_N; i++) {
		if ((r = XGemac_MgtRead (&(lp->Emac), lp->gmii_addr, i, &(dump->data[i]))) != XST_SUCCESS) {
			printk (KERN_INFO "PhyRead ERROR %d\n", r);
			*(int*)ret = -EIO;
			return;
		}
	}
	*(int*)ret = 0;
}

static int
xenet_do_ethtool_ioctl (struct net_device *dev, struct ifreq *rq)
{
	struct net_local *lp = (struct net_local *) dev->priv;
	struct ethtool_cmd ecmd;
	struct ethtool_coalesce eco;
	struct ethtool_drvinfo edrv;
	struct ethtool_ringparam erp;
	struct ethtool_pauseparam epp;
	struct mac_regsDump regs;
	int ret = -EOPNOTSUPP;
	XStatus result;
	u32 Options;
	u16 gmii_reg_sset;
	u16 gmii_reg_spause;
	u16 gmii_reg_autoneg;
	u32 flags;

	if (copy_from_user(&ecmd, rq->ifr_data, sizeof (ecmd.cmd)))
		return -EFAULT;
	switch (ecmd.cmd) {
	case ETHTOOL_GSET:
		ret = xenet_ethtool_get_settings(dev, &ecmd);
		if (ret >= 0) {
			if (copy_to_user(rq->ifr_data, &ecmd, sizeof (ecmd)))
				ret = -EFAULT;
		}
		break;
	case ETHTOOL_SSET:
		if (copy_from_user(&ecmd, rq->ifr_data, sizeof (struct ethtool_cmd)))
			return -EFAULT;
		gmii_reg_sset = 0;
		if (ecmd.autoneg == AUTONEG_ENABLE) {
			gmii_reg_sset |= (BMCR_ANENABLE | BMCR_ANRESTART);
			spin_lock_irqsave(reset_lock, flags);
			result = XGemac_MgtWrite(&lp->Emac, lp->gmii_addr,
						GMII_BMCR, gmii_reg_sset);
			if (result != XST_SUCCESS) {
				spin_unlock_irqrestore(reset_lock, flags);
				ret = -EIO;
				break;
			}
			result = XGemac_MgtRead(&lp->Emac, lp->gmii_addr, GMII_ADVERTISE, &gmii_reg_sset);
			if (result != XST_SUCCESS) {
				spin_unlock_irqrestore(reset_lock, flags);
				ret = -EIO;
				break;
			}
			result = XGemac_MgtWrite(&lp->Emac, lp->gmii_addr, GMII_ADVERTISE, gmii_reg_sset);
			spin_unlock_irqrestore(reset_lock, flags);
			if (result != XST_SUCCESS) {
				ret = -EIO;
				break;
			}
		} else {
			gmii_reg_sset &= ~(BMCR_ANENABLE | BMCR_ANRESTART);
			spin_lock_irqsave(reset_lock, flags);
			result = XGemac_MgtWrite(&lp->Emac, lp->gmii_addr,
						GMII_BMCR, gmii_reg_sset);
			spin_unlock_irqrestore(reset_lock, flags);
			if (result != XST_SUCCESS) {
				ret = -EIO;
				break;
			}
		}
		ret = 0;
		break;
	case ETHTOOL_GPAUSEPARAM:
		ret = xenet_ethtool_get_settings(dev, &ecmd);
		if (ret < 0) {
			break;
		}
		epp.cmd = ecmd.cmd;
		epp.autoneg = ecmd.autoneg;
		Options = XGemac_GetOptions(&lp->Emac);
		if (Options & XGE_INSERT_PAD_OPTION) {
			epp.rx_pause = 1;
			epp.tx_pause = 1;
		} else {
			epp.rx_pause = 0;
			epp.tx_pause = 0;
		}
		if (copy_to_user(rq->ifr_data, &epp, sizeof(struct ethtool_pauseparam)))
			ret = -EFAULT;
		else
			ret = 0;
		break;
	case ETHTOOL_SPAUSEPARAM:
		if (copy_from_user(&epp, rq->ifr_data, sizeof (struct ethtool_pauseparam)))
			return -EFAULT;
		ret = xenet_ethtool_get_settings(dev, &ecmd);
		if (ret < 0) {
			break;
		}
		epp.cmd = ecmd.cmd;
		gmii_reg_spause = 0;
		if (epp.autoneg == AUTONEG_ENABLE) {
			gmii_reg_spause |= (BMCR_ANENABLE | BMCR_ANRESTART);
		}
		spin_lock_irqsave(reset_lock, flags);
		result = XGemac_MgtWrite(&lp->Emac, lp->gmii_addr,
					GMII_BMCR, gmii_reg_spause);
		spin_unlock_irqrestore(reset_lock, flags);
		if (result != XST_SUCCESS) {
			ret = -EIO;
			break;
		}
		if (epp.rx_pause != epp.tx_pause) {
			ret = 0;
			break;
		} else {
			spin_lock_irqsave(reset_lock, flags);
			(void)XGemac_Stop(&(lp->Emac));
			Options = XGemac_GetOptions(&lp->Emac);
			if (epp.rx_pause)
				Options |= XGE_INSERT_PAD_OPTION;
			else
				Options &= ~XGE_INSERT_PAD_OPTION;
			(void)XGemac_SetOptions(&lp->Emac,Options);
			(void)XGemac_Start(&(lp->Emac));
			spin_unlock_irqrestore(reset_lock, flags);
		}
		ret = 0;
		break;
	case ETHTOOL_GCOALESCE:
		if (!(XGemac_mIsSgDma(&lp->Emac)))
			break;
		eco.cmd = ecmd.cmd;
		ret = xenet_ethtool_get_coalesce(dev, &eco);
		if (ret >= 0) {
			if (copy_to_user(rq->ifr_data, &eco, sizeof (struct ethtool_coalesce)))
				ret = -EFAULT;
		}
		break;
	case ETHTOOL_SCOALESCE:
		if (!(XGemac_mIsSgDma(&lp->Emac)))
			break;
		if (copy_from_user(&eco, rq->ifr_data, sizeof (struct ethtool_coalesce)))
			return -EFAULT;
		ret = xenet_ethtool_set_coalesce(dev, &eco);
		break;
	case ETHTOOL_GDRVINFO:
		edrv.cmd = edrv.cmd;
		ret = xenet_ethtool_get_drvinfo(dev, &edrv);
		if (ret >= 0) {
			if (copy_to_user(rq->ifr_data, &edrv, sizeof (struct ethtool_drvinfo)))
				ret = -EFAULT;
		}
		break;
	case ETHTOOL_GREGS:
		regs.hd.cmd = edrv.cmd;
		xenet_ethtool_get_regs (dev, &(regs.hd), &ret);
		if (ret >= 0) {
			if (copy_to_user(rq->ifr_data, &regs, sizeof (struct mac_regsDump)))
				ret = -EFAULT;
		}
		break;
	case ETHTOOL_GRINGPARAM:
		erp.cmd = edrv.cmd;
		ret = xenet_ethtool_get_ringparam (dev, &(erp));
		if (ret >= 0) {
			if (copy_to_user(rq->ifr_data, &erp, sizeof (struct ethtool_ringparam)))
				ret = -EFAULT;
		}
		break;
	case ETHTOOL_NWAY_RST:
		epp.cmd = ecmd.cmd;
		gmii_reg_autoneg = 0;
		gmii_reg_autoneg |= (BMCR_ANENABLE | BMCR_ANRESTART);
		spin_lock_irqsave(reset_lock, flags);
		result = XGemac_MgtWrite(&lp->Emac, lp->gmii_addr,
					GMII_BMCR, gmii_reg_autoneg);
		spin_unlock_irqrestore(reset_lock, flags);
		if (result != XST_SUCCESS) {
			ret = -EIO;
			break;
		}
		ret = 0;
		break;
	default:
		break;
	}
	return ret;
}

static int
xenet_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
	struct net_local *lp = (struct net_local *) dev->priv;
	/* gmii_ioctl_data has 4 u16 fields: phy_id, reg_num, val_in & val_out */
	struct gmii_ioctl_data *data = (struct gmii_ioctl_data *) &rq->ifr_data;
	struct {
		__u8 threshold;
		__u32 direction;
	} thr_arg;
	struct {
		__u32 waitbound;
		__u32 direction;
	} wbnd_arg ;
	XStatus ret;
	unsigned long flags;

	XStatus Result;

	switch (cmd) {
	case SIOCETHTOOL:
		return xenet_do_ethtool_ioctl(dev, rq);
	case SIOCGMIIPHY:	/* Get address of GMII PHY in use. */
	case SIOCDEVPRIVATE:	/* for binary compat, remove in 2.5 */
		data->phy_id = lp->gmii_addr;
		/* Fall Through */

	case SIOCGMIIREG:	/* Read GMII PHY register. */
	case SIOCDEVPRIVATE + 1:	/* for binary compat, remove in 2.5 */
		if (data->phy_id > 31 || data->reg_num > 31)
			return -ENXIO;

		/* Stop the PHY timer to prevent reentrancy. */
		del_timer_sync(&lp->phy_timer);
		spin_lock_irqsave(reset_lock, flags);
		Result = XGemac_MgtRead(&lp->Emac, data->phy_id,
				       data->reg_num, &data->val_out);
		spin_unlock_irqrestore(reset_lock, flags);
		/* Start the PHY timer up again. */
		lp->phy_timer.expires = jiffies + 2 * HZ;
		add_timer(&lp->phy_timer);

		if (Result != XST_SUCCESS) {
			printk(KERN_ERR
			       "%s: Could not read from PHY, error=%d.\n",
			       dev->name, Result);
			return (Result == XST_EMAC_MII_BUSY) ? -EBUSY : -EIO;
		}
		return 0;

	case SIOCSMIIREG:	/* Write GMII PHY register. */
	case SIOCDEVPRIVATE + 2:	/* for binary compat, remove in 2.5 */
		if (!capable(CAP_NET_ADMIN))
			return -EPERM;

		if (data->phy_id > 31 || data->reg_num > 31)
			return -ENXIO;

		/* Stop the PHY timer to prevent reentrancy. */
		del_timer_sync(&lp->phy_timer);
		spin_lock_irqsave(reset_lock, flags);
		Result = XGemac_MgtWrite(&lp->Emac, data->phy_id,
					data->reg_num, data->val_in);
		spin_unlock_irqrestore(reset_lock, flags);
		/* Start the PHY timer up again. */
		lp->phy_timer.expires = jiffies + 2 * HZ;
		add_timer(&lp->phy_timer);

		if (Result != XST_SUCCESS) {
			printk(KERN_ERR
			       "%s: Could not write to PHY, error=%d.\n",
			       dev->name, Result);
			return (Result == XST_EMAC_MII_BUSY) ? -EBUSY : -EIO;
		}
		return 0;

	case SIOCDEVPRIVATE + 3:		/* set THRESHOLD */
		if (copy_from_user(&thr_arg, rq->ifr_data, sizeof(thr_arg))) {
			return -EFAULT;
		}
		spin_lock_irqsave(reset_lock, flags);
		if ((ret = XGemac_Stop(&lp->Emac)) != XST_SUCCESS) {
			return -EIO;
		}
		if ((ret = XGemac_SetPktThreshold(&lp->Emac, thr_arg.direction, thr_arg.threshold)) != XST_SUCCESS) {
			return -EIO;
		}
		if ((ret = XGemac_Start(&lp->Emac)) != XST_SUCCESS) {
			return -EIO;
		}
		spin_unlock_irqrestore(reset_lock, flags);
		return 0;

	case SIOCDEVPRIVATE + 4:		/* set WAITBOUND */
		if (copy_from_user(&wbnd_arg, rq->ifr_data, sizeof(wbnd_arg))) {
			return -EFAULT;
		}
		spin_lock_irqsave(reset_lock, flags);
		if ((ret = XGemac_Stop(&lp->Emac)) != XST_SUCCESS) {
			return -EIO;
		}
		if ((ret = XGemac_SetPktWaitBound(&lp->Emac, wbnd_arg.direction, wbnd_arg.waitbound)) != XST_SUCCESS) {
			return -EIO;
		}
		if ((ret = XGemac_Start(&lp->Emac)) != XST_SUCCESS) {
			return -EIO;
		}
		spin_unlock_irqrestore(reset_lock, flags);
		return 0;

	case SIOCDEVPRIVATE + 5:		/* get THRESHOLD */
		if (copy_from_user(&thr_arg, rq->ifr_data, sizeof(thr_arg))) {
			return -EFAULT;
		}
		if ((ret = XGemac_GetPktThreshold(&lp->Emac, thr_arg.direction, &(thr_arg.threshold))) != XST_SUCCESS) {
			return -EIO;
		}
		if (copy_to_user(rq->ifr_data, &thr_arg, sizeof(thr_arg))) {
			return -EFAULT;
		}
		return 0;

	case SIOCDEVPRIVATE + 6:		/* get WAITBOUND */
		if (copy_from_user(&wbnd_arg, rq->ifr_data, sizeof(wbnd_arg))) {
			return -EFAULT;
		}
		if ((ret = XGemac_GetPktWaitBound(&lp->Emac, wbnd_arg.direction, &(wbnd_arg.waitbound))) != XST_SUCCESS) {
			return -EIO;
		}
		if (copy_to_user(rq->ifr_data, &wbnd_arg, sizeof(wbnd_arg))) {
			return -EFAULT;
		}
		return 0;

	default:
		return -EOPNOTSUPP;
	}
}

static void
remove_head_dev(void)
{
	struct net_local *lp;
	struct net_device *dev;
	XGemac_Config *cfg;

	/* Pull the head off of dev_list. */
	spin_lock(&dev_lock);
	dev = dev_list;
	lp = (struct net_local *) dev->priv;
	dev_list = lp->next_dev;
	spin_unlock(&dev_lock);

	/* Put the physical address back */
	cfg = XGemac_GetConfig(lp->index);
	iounmap((void *) cfg->BaseAddress);
	cfg->BaseAddress = cfg->PhysAddress;

	/* Free up the memory. */
	if (lp->desc_space)
        {
#if BD_IN_BRAM == 0
		free_descriptor_skb(dev);
		consistent_free(lp->desc_space);
#else
                iounmap(lp->desc_space);
#endif
        }

	kfree(lp);

	unregister_netdev(dev);
	kfree(dev);
}

static int __init
probe(int index)
{
	static const unsigned long remap_size
	    = XPAR_GEMAC_0_HIGHADDR - XPAR_GEMAC_0_BASEADDR + 1;
	struct net_device *dev;
	struct net_local *lp;
	XGemac_Config *cfg;
	unsigned int irq;
	u32 maddr;
	XStatus xs;

	switch (index) {
#if defined(XPAR_INTC_0_GEMAC_0_VEC_ID)
	case 0:
		irq = 31 - XPAR_INTC_0_GEMAC_0_VEC_ID;
		break;
#if defined(XPAR_INTC_0_GEMAC_1_VEC_ID)
	case 1:
		irq = 31 - XPAR_INTC_0_GEMAC_1_VEC_ID;
		break;
#if defined(XPAR_INTC_0_GEMAC_2_VEC_ID)
	case 2:
		irq = 31 - XPAR_INTC_0_GEMAC_2_VEC_ID;
		break;
#if defined(XPAR_INTC_0_GEMAC_3_VEC_ID)
#error Edit this file to add more devices.
#endif				/* 3 */
#endif				/* 2 */
#endif				/* 1 */
#endif				/* 0 */
	default:
		return -ENODEV;
	}

	/* Find the config for our device. */
	cfg = XGemac_GetConfig(index);
	if (!cfg)
		return -ENODEV;

	dev = init_etherdev(0, sizeof (struct net_local));
	if (!dev) {
		printk(KERN_ERR "Could not allocate Xilinx enet device %d.\n",
		       index);
		return -ENOMEM;
	}
	SET_MODULE_OWNER(dev);

	ether_setup(dev);
	dev->irq = irq;

	/* Initialize our private data. */
	lp = (struct net_local *) dev->priv;
	memset(lp, 0, sizeof (struct net_local));
	lp->index = index;
	lp->dev = dev;
	spin_lock_init(&lp->skb_lock);

	/* Make it the head of dev_list. */
	spin_lock(&dev_lock);
	lp->next_dev = dev_list;
	dev_list = dev;
	spin_unlock(&dev_lock);

	/* Change the addresses to be virtual */
	cfg->PhysAddress = cfg->BaseAddress;
	cfg->BaseAddress = (u32) ioremap(cfg->PhysAddress, remap_size);

	if (XGemac_Initialize(&lp->Emac, cfg->DeviceId) != XST_SUCCESS) {
		printk(KERN_ERR "%s: Could not initialize device.\n",
		       dev->name);
		remove_head_dev();
		return -ENODEV;
	}

	memcpy(dev->dev_addr, ((bd_t *) __res)->bi_enetaddr, 6);
	if (XGemac_SetMacAddress(&lp->Emac, dev->dev_addr) != XST_SUCCESS) {
		/* should not fail right after an initialize */
		printk(KERN_ERR "%s: Could not set MAC address.\n", dev->name);
		remove_head_dev();
		return -EIO;
	}
#ifdef CONFIG_XILINX_GIGE_JUMBO
	lp->max_frame_size = XGE_MAX_JUMBO_FRAME_SIZE;
#else
	lp->max_frame_size = XGE_MAX_FRAME_SIZE;
#endif

	if (XGemac_mIsSgDma(&lp->Emac)) {
		int result;

		printk(KERN_ERR "%s: using sgDMA mode.\n", dev->name);
		XGemac_SetSgRecvHandler(&lp->Emac, dev, SgRecvHandler);
		XGemac_SetSgSendHandler(&lp->Emac, dev, SgSendHandler);
		dev->hard_start_xmit = xenet_SgSend;
		lp->Isr = XGemac_IntrHandlerDma;

		result = descriptor_init(dev);
		if (result) {
			remove_head_dev();
			return -EIO;
		}

                /* set the packet threshold */
		if ((xs = XGemac_SetPktThreshold(&lp->Emac, XGE_SEND, 16)) != XST_SUCCESS) {
			printk(KERN_ERR "%s: Could not set send pkt threshold, ERROR %d",
			dev->name,xs);
		}
                if ((xs = XGemac_SetPktThreshold(&lp->Emac, XGE_RECV, 2)) != XST_SUCCESS) {
			printk(KERN_ERR "%s: Could not set receive  pkt threshold ERROR %d",
				dev->name,xs);
		}
                if ((xs = XGemac_SetPktWaitBound(&lp->Emac, XGE_SEND, 5)) != XST_SUCCESS) {
			printk(KERN_ERR "%s: Could not set send waitbound, ERROR %d",
			dev->name,xs);
		}
                if ((xs = XGemac_SetPktWaitBound(&lp->Emac, XGE_RECV, 2)) != XST_SUCCESS) {
			printk(KERN_ERR "%s: Could not set receive  waitbound, ERROR %d",
			dev->name,xs);
		}
	} else {
		printk(KERN_ERR "%s: using fifo mode.\n", dev->name);
		XGemac_SetFifoRecvHandler(&lp->Emac, dev, FifoRecvHandler);
		XGemac_SetFifoSendHandler(&lp->Emac, dev, FifoSendHandler);
		dev->hard_start_xmit = xenet_FifoSend;
		lp->Isr = XGemac_IntrHandlerFifo;
	}
	XGemac_SetErrorHandler(&lp->Emac, dev, ErrorHandler);

	/* Scan to find the PHY. */
	lp->gmii_addr = 0xFF;
	for (maddr = 0; maddr < 31; maddr++) {
		XStatus Result;
		u16 reg;

		Result = XGemac_MgtRead(&lp->Emac, maddr, GMII_BMCR, &reg);
		/*
		 * XGemac_MgtRead is currently returning XST_SUCCESS even
		 * when reading from non-existent addresses.  Work
		 * around this by doing a primitive validation on the
		 * control word we get back.
		 */
			if (Result == XST_SUCCESS && (reg & 0x21ff) == 0x0140) {
			lp->gmii_addr = maddr;
			break;
		}
	}
	if (lp->gmii_addr == 0xFF) {
		lp->gmii_addr = 0;
		printk(KERN_WARNING
		       "%s: No PHY detected.  Assuming a PHY at address %d.\n",
		       dev->name, lp->gmii_addr);
	}
	dev->open = xenet_open;
	dev->stop = xenet_close;
	dev->change_mtu = xenet_change_mtu;
	dev->get_stats = xenet_get_stats;
	dev->flags &= ~IFF_MULTICAST;
	dev->do_ioctl = xenet_ioctl;
	dev->tx_timeout = xenet_tx_timeout;
	dev->watchdog_timeo = TX_TIMEOUT;

	printk(KERN_INFO
	       "%s: Xilinx EMAC #%d at 0x%08X mapped to 0x%08X, irq=%d\n",
	       dev->name, index, cfg->PhysAddress, cfg->BaseAddress, dev->irq);

	/* print h/w id  */
	{
		u32 id = XIo_In32(cfg->BaseAddress + XIIF_V123B_RESETR_OFFSET);

		printk("%s: id %d.%d%c; block id %d, type %d\n",
		       dev->name, (id >> 28) & 0xf, (id >> 21) & 0x7f,
		       ((id >> 16) & 0x1f) + 'a',
		       (id >> 16) & 0xff, (id >> 0) & 0xff);
	}

	return 0;
}

static int __init
xenet_init(void)
{
	int index = 0;

	while (probe(index++) == 0) ;
	/* If we found at least one, report success. */
	return (index > 1) ? 0 : -ENODEV;
}

static void __exit
xenet_cleanup(void)
{
	while (dev_list)
		remove_head_dev();
}

EXPORT_NO_SYMBOLS;

module_init(xenet_init);
module_exit(xenet_cleanup);
