/*
  * driver/net/nec_candy.c
  *
  * NEC candy ethernet driver.
  *
  * Author: Jun Sun <jsun@mvista.com>
  *
  * 2001-2004 (c) MontaVista Software, Inc. This file is licensed under
  * the terms of the GNU General Public License version 2. This program
  * is licensed "as is" without any warranty of any kind, whether express
  * or implied.
  */
/*
 * Changes:
 *  MontaVista Software Inc. <source@mvista.com>
 *  - Added MII interface library support
 *  - Added Linux ethtool package support (based on 8139cp.c driver)
 *  - Added network interface message level support
 *  - Fixed VLTP register initialisation (set to zero)
 *
 *  MontaVista Software Inc. <source@mvista.com>
 *  - Added support for VR4133 (CCR register defines additional bits).
 *
 *  MontaVista Software Inc. <source@mvista.com>
 *  - Added support for NEC VR7701.
 *
 *  MontaVista Software Inc.<asapkov@ru.mvista.com> or <source@mvista.com>
 *  - Added support for NEC CMB-VR4133.
 */

#define DRV_NAME		"nec_candy"
#define DRV_VERSION		"0.3"
#define DRV_RELDATE		"Nov 13, 2003"

#include <linux/init.h>
#include <linux/types.h>
#include <linux/kernel.h>	/* get_options(), printk(), panic().. */
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/netdevice.h>
#include <linux/param.h>	/* for HZ */
#include <linux/etherdevice.h>	/* init_etherdev() */
#include <linux/if.h>
#include <linux/ethtool.h>
#include <linux/mii.h>
#include <linux/spinlock.h>

#include <asm/addrspace.h>	/* KSEGx() */
#include <asm/irq.h>		/* NR_IRQS */
#include <asm/io.h>		/* virt_to_phys(), dma_cache_xx() */
#include <asm/uaccess.h>
#include <linux/timer.h>
#include <asm/pgtable.h>

/***********************************************************************
 * debug
 ***********************************************************************
 */

#define CANDY_DEF_MSG_ENABLE	(NETIF_MSG_DRV		| \
				 NETIF_MSG_PROBE 	| \
				 NETIF_MSG_LINK)

static int debug = -1;
// #define              DEBUG_NEC_CANDY
// #define              DEBUG_VERBOSE_NEC_CANDY
// #define      SHOW_BUG                /* targeted bug reporting msgs */

#ifdef DEBUG_VERBOSE_NEC_CANDY
#define DEBUG_NEC_CANDY
#endif

#ifdef DEBUG_NEC_CANDY

#define ASSERT(x) if (!(x)) { panic("%s:%d - assert failed!\n", __FILE__, __LINE__); }
#define VERIFY(x, y) ASSERT(x y)
#define DEBUG(x)  do { x; } while (0)

#else

#define ASSERT(x)
#define VERIFY(x, y) x
#define DEBUG(x)

#endif

#ifdef DEBUG_VERBOSE_NEC_CANDY
#define DEBUG_VERBOSE(x)  do { x; } while (0)
#else
#define DEBUG_VERBOSE(x)
#endif

/***********************************************************************
 * Configure 
 ***********************************************************************
 */
/*
 * This driver supports multiple nec_candy chips.
 */
#define		MAX_NUM_DEVS		2

#define		TX_RING_SIZE		32
#define		RX_RING_SIZE		32

#define		RX_BUF_SIZE		1536
#define		ETH_FRAME_SIZE		1536

#define		TX_TIMEOUT		4*HZ

/* rx_copybreak:  for smaller packet we copy them to avoid emulated
 * unaligned access overhead.
 *
 * Set it to 1518 to always copy ( you should do that on fast machines)
 *
 * Set it to 0 to avoid any copy.
 *
 * On Korva, some value in the middle might be appropriate.
 */
static int rx_copybreak = 1518;

/***********************************************************************
 * hardware bug workaround
 ***********************************************************************
 */

#define		WORKAROUND_E7_AFCE
#define		WORKAROUND_E10_PRM_AMC
#define		WORKAROUND_E13_TXFC
#define		WORKAROUND_E8_TX_STALL

#if defined(CONFIG_NEC_CMB_VR7701)
#undef		WORKAROUND_E7_AFCE
#undef		WORKAROUND_E10_PRM_AMC
#undef		WORKAROUND_E13_TXFC
#undef		WORKAROUND_E8_TX_STALL
#endif

#ifdef CONFIG_NEC_CMBVR4133
#undef		WORKAROUND_E7_AFCE
#undef		WORKAROUND_E10_PRM_AMC
#undef		WORKAROUND_E13_TXFC
#define		WORKAROUND_E21_PAD
#define		WORKAROUND_E10_VR4133
#endif

/***********************************************************************
 * Candy.h macros
 ***********************************************************************
 */
/*---------------------------------------------------------------------------*/
/* PHY link status                                                           */
/*---------------------------------------------------------------------------*/
#define LINK_UP     1
#define LINK_DOWN   0

/*---------------------------------------------------------------------------*/
/* receive mode & related definitions                                        */
/*---------------------------------------------------------------------------*/
#define ACCEPT_ALL_PHYS         0x0001
#define ACCEPT_ALL_MCASTS       0x0002
#define ACCEPT_ALL_BCASTS       0x0004
#define ACCEPT_QUALIFIED_MCAST  0x0008
#define ACCEPT_STATION          0x0010
#define MAC_LOOPBACK            0x0020

/*---------------------------------------------------------------------------*/
/* MACC1 - MAC configuration register 1 (00H R/W)                            */
/*---------------------------------------------------------------------------*/
#define MACLB       0x00004000	/* MAC loopback                          */
#define TXFC        0x00000800	/* Transmit flow control enable          */
#define RXFC        0x00000400	/* Receive flow control enable           */
#define SRXEN       0x00000200	/* Receive enable                        */
#define PARF        0x00000100	/* Control packet pass                   */
#define PUREP       0x00000080	/* Pure preamble                         */
#define FLCHT       0x00000040	/* Length field check                    */
#define NOBO        0x00000020	/* No Back Off                           */
#define CRCEN       0x00000008	/* CRC append enable                     */
#define PADEN       0x00000004	/* PAD append enable                     */
#define FULLD       0x00000002	/* Full duplex enable                    */
#define HUGEN       0x00000001	/* Large packet enable                   */
#define MACC1_RESERVED  0x00004fef	/* reserved bit 0                        */

/*---------------------------------------------------------------------------*/
/* MACC2 - MAC configuration register 2 (04H R/W)                            */
/*---------------------------------------------------------------------------*/
#define MCRST       0x00000400	/* MAC Control Block software reset      */
#define RFRST       0x00000200	/* Receive Function Block software reset */
#define TFRST       0x00000100	/* Transmit Function Block software reset */
#define BPNB        0x00000040	/* Back Pressure No Back Off             */
#define APD         0x00000020	/* Auto VLAN PAD                         */
#define VPD         0x00000010	/* VLAN PAD mode                         */
#define MACC2_RESERVED  0x00000770	/* reserved bit 0                        */

/*---------------------------------------------------------------------------*/
/* IPGT - Back-to-Back IPG register (08H R/W)                                */
/*---------------------------------------------------------------------------*/
#define IPGT        0x00000013	/* Back-To-Back IPG default value        */
#define IPGT_RESERVED   0x0000007f	/* reserved bit 0                        */

/*---------------------------------------------------------------------------*/
/* IPGR - Non Back-to-Back IPG register (0CH R/W)                            */
/*---------------------------------------------------------------------------*/
#define IPGR1       0x00000e00	/* Back-To-Back IPG default value        */
#define IPGR2       0x00000013	/* Back-To-Back IPG default value        */
#define IPGR_RESERVED   0x00007f7f	/* reserved bit 0                        */

/*---------------------------------------------------------------------------*/
/* CLRT - Collision register (10H R/W)                                       */
/*---------------------------------------------------------------------------*/
#define LCOLW       0x00003800	/* Late collision window default value   */
#define RETRY       0x0000000f	/* Maximum number of retry default value */
#define CLRT_RESERVED   0x00003f0f	/* reserved bit 0                        */

/*---------------------------------------------------------------------------*/
/* LMAX - Maximum Packet Length register (14H R/W)                           */
/*---------------------------------------------------------------------------*/
// #define MAXF        0x000005f2      /* Maximum packet length value(1522byte) */
#define MAXF        0x000005ee	/* Maximum packet length value(1518byte) */
#define LMAX_RESERVED   0x0000ffff	/* reserved bit 0                        */

/*---------------------------------------------------------------------------*/
/* RETX - Retry count register (20H R/W)                                     */
/*---------------------------------------------------------------------------*/
#define RETX_MASK   0x0000000f	/* Retry counter                         */
#define RETX_RESERVED   0x0000000f	/* reserved bit 0                        */

/*---------------------------------------------------------------------------*/
/* LSA2 - Station address register 2 (54H R/W)                               */
/*---------------------------------------------------------------------------*/
#define LSA2_MASK   0x0000ffff	/* Station address SA (47:32)            */
#define LSA2_RESERVED   0x0000ffff	/* reserved bit 0                        */

/*---------------------------------------------------------------------------*/
/* LSA1 - Station address register 1 (58H R/W)                               */
/*---------------------------------------------------------------------------*/
#define LSA1_MASK   0xffffffff	/* Station address SA(31:0)              */

/*---------------------------------------------------------------------------*/
/* PTVR - Pause timer read register (5CH Read)                               */
/*---------------------------------------------------------------------------*/
#define PTCT_MASK   0x0000ffff	/* Pause timer counter                   */

/*---------------------------------------------------------------------------*/
/* VLTP - VLAN type register (64H R/W)                                       */
/*---------------------------------------------------------------------------*/
#define VLTP        0x00008100	/* VLAN type ( etpid:0x81 tci:0x00 )     */
#define VLTP_RESERVED   0x0000ffff	/* reserved bit 0                        */

/*---------------------------------------------------------------------------*/
/* MIIC - MII Configuration register (80H R/W)                               */
/*---------------------------------------------------------------------------*/
#define MISRT       0x00008000	/* MII Management Interface Block software reset */
#define CLKS25      0x00000000	/* HCLK is equal to 25 MHz               */
#define CLKS33      0x00000004	/* HCLK is less than or equal to 33 MHz  */
#define CLKS50      0x00000008	/* HCLK is less than or equal to 50 MHz  */
#define CLKS66      0x0000000c	/* HCLK is less than or equal to 66 MHz  */
#define MIIC_RESERVED   0x0000800c	/* reserved bit 0                        */

/*---------------------------------------------------------------------------*/
/* MCMD - MII command register (94H Write)                                   */
/*---------------------------------------------------------------------------*/
#define SCANC       0x00000002	/* SCAN command                          */
#define RSTAT       0x00000001	/* MII management read                   */

/*---------------------------------------------------------------------------*/
/* MADR - MII address register (98H R/W)                                     */
/*---------------------------------------------------------------------------*/
#define FIAD_MASK   0x00001f00	/* MII PHY address                       */
#define FIAD_SHIFT  8
#define RGAD_MASK   0x0000001f	/* MII register address                  */
#define MADR_RESERVED   0x00001f1f	/* reserved bit 0                        */

/*---------------------------------------------------------------------------*/
/* MWTD - MII write data register  (9CH R/W)                                 */
/*---------------------------------------------------------------------------*/
#define CTLD_MASK   0x0000ffff	/* MII write data                        */
#define MWTD_RESERVED   0x0000ffff	/* reserved bit 0                        */

/*---------------------------------------------------------------------------*/
/* MRDD - MII read data register (A0H Read)                                  */
/*---------------------------------------------------------------------------*/
#define PRSD_MASK   0x0000ffff	/* MII read data                         */

/*---------------------------------------------------------------------------*/
/* MIND - MII indicator register (A4H Read)                                  */
/*---------------------------------------------------------------------------*/
#define NVALID      0x00000004	/* SCAN command start status             */
#define SCANA       0x00000002	/* SCAN command active                   */
#define BUSY        0x00000001	/* BUSY                                  */

/*---------------------------------------------------------------------------*/
/* STLC - STL configuration register (C0H R/W)                               */
/*---------------------------------------------------------------------------*/
#define ATZ         0x00000004	/* Statistics counter read reset         */
#define STLC_RESERVED   0x00000004	/* reserved bit 0                        */

/*---------------------------------------------------------------------------*/
/* AFR - Address Filter register (C8H R/W)                                   */
/*---------------------------------------------------------------------------*/
#define PRO         0x00000008	/* Promiscuous mode                      */
#define PRM         0x00000004	/* Accept Multicast                      */
#define AMC         0x00000002	/* Accept Multicast ( qualified )        */
#define ABC         0x00000001	/* Accept Broadcast                      */
#define AFR_RESERVED    0x0000000f	/* reserved bit 0                        */

/*---------------------------------------------------------------------------*/
/* CAR1 - CARRY register1 (DCH R/W)                                          */
/*---------------------------------------------------------------------------*/
#define C1VT        0x00008000	/* RVBT counter carry bit                */
#define C1UT        0x00004000	/* TUCA counter carry bit                */
#define C1BT        0x00002000	/* TBCA counter carry bit                */
#define C1MT        0x00001000	/* TMCA counter carry bit                */
#define C1PT        0x00000800	/* TPCT counter carry bit                */
#define C1TB        0x00000400	/* TBYT counter carry bit                */
#define C1MX        0x00000200	/* RMAX counter carry bit                */
#define C11K        0x00000100	/* R1K counter carry bit                 */
#define C1FE        0x00000080	/* R511 counter carry bit                */
#define C1TF        0x00000040	/* R255 counter carry bit                */
#define C1OT        0x00000020	/* R127 counter carry bit                */
#define C1SF        0x00000010	/* R64 counter carry bit                 */
#define C1BR        0x00000008	/* RBCA counter carry bit                */
#define C1MR        0x00000004	/* RBCA counter carry bit                */
#define C1PR        0x00000002	/* RPKT counter carry bit                */
#define C1RB        0x00000001	/* RBYT counter carry bit                */
#define CAR1_RESERVED   0x0000ffff	/* reserved bit 0                        */

/*---------------------------------------------------------------------------*/
/* CAR2 - CARRY register2 (E0H R/W)                                          */
/*---------------------------------------------------------------------------*/
#define C2SV        0x80000000	/* Status vector overrun bit             */
#define C2IM        0x00400000	/* TIME counter carry bit                */
#define C2CS        0x00200000	/* TCSE counter carry bit                */
#define C2BC        0x00100000	/* TNCL counter carry bit                */
#define C2XC        0x00080000	/* TXCL counter carry bit                */
#define C2LC        0x00040000	/* TLCL counter carry bit                */
#define C2MC        0x00020000	/* TMCL counter carry bit                */
#define C2SC        0x00010000	/* TSCL counter carry bit                */
#define C2XD        0x00008000	/* TXDF counter carry bit                */
#define C2DF        0x00004000	/* TDFR counter carry bit                */
#define C2XF        0x00002000	/* TXPF counter carry bit                */
#define C2TE        0x00001000	/* TFCS counter carry bit                */
#define C2JB        0x00000800	/* RBJR counter carry bit                */
#define C2FG        0x00000400	/* RFRG counter carry bit                */
#define C2OV        0x00000200	/* ROVR counter carry bit                */
#define C2UN        0x00000100	/* RUND counter carry bit                */
#define C2FC        0x00000080	/* RFCR counter carry bit                */
#define C2CD        0x00000040	/* RCDE counter carry bit                */
#define C2FO        0x00000020	/* RFLR counter carry bit                */
#define C2AL        0x00000010	/* RALN counter carry bit                */
#define C2UO        0x00000008	/* RXUO counter carry bit                */
#define C2PF        0x00000004	/* RXPF counter carry bit                */
#define C2CF        0x00000002	/* RXCF counter carry bit                */
#define C2RE        0x00000001	/* RFCS counter carry bit                */
#define CAR2_RESERVED   0x807fffff	/* reserved bit 0                        */

/*---------------------------------------------------------------------------*/
/* TXCFG - Transmit Configuration (200H R/W)                                 */
/*---------------------------------------------------------------------------*/
#define AFCE        0x00000001	/* Automatic Flow Control Enable         */
#define DTBS1       0x00000000	/* DMA Transmit Burst Size 1 word        */
#define DTBS2       0x00010000	/* DMA Transmit Burst Size 2 word        */
#define DTBS4       0x00020000	/* DMA Transmit Burst Size 4 word        */
#define DTBS8       0x00030000	/* DMA Transmit Burst Size 8 word        */
#define DTBS16      0x00040000	/* DMA Transmit Burst Size 16 word       */
#define DTBS32      0x00050000	/* DMA Transmit Burst Size 32 word       */
#define DTBS64      0x00060000	/* DMA Transmit Burst Size 64 word       */
#define TXE         0x80000000	/* Transmit Enable                       */
#define TXCFG_RESERVED  0x80070001	/* reserved bit                          */

/*---------------------------------------------------------------------------*/
/* TXFC - Transmit FiFo Control (204H R/W)                                   */
/*---------------------------------------------------------------------------*/
#define TPTV_MASK       0xffff0000	/* Transmit Pause Timer Value mask       */
#define TPTV            0x10000000	/* default 0x1000 slot time (1slot:512bit) */
#define TX_DRTH_MASK    0x0000fc00	/* Transmit Fill Threshold Level mask    */
#define TX_DRTH         0x00004000	/* default 010000b (16word, 64byte)      */
#define TX_FLTH_MASK    0x000000fc	/* Transmit Drain Threshold Level mask   */
#ifdef CONFIG_NEC_CMBVR4133
/* We set this to 160 bytes for VR4133 because of the hardware restriction:
the sum of TX_FLTH and DTBS should be <= 192 bytes. DTBS is set to 32 bytes
for VR4133. */
#define TX_FLTH         0x000000a0	/* 101000b (160byte)     */
#else
#define TX_FLTH         0x000000c0	/* default 110000b (48word, 192byte)     */
#endif
#define TXFC_RESERVED   0xfffffcfc	/* reserved bit                          */

/*---------------------------------------------------------------------------*/
/* TXST - Transmit Status  (20CH R)                                          */
/*---------------------------------------------------------------------------*/
#define CSE_AB      0x80000000	/* carrier lost (abort)                      */
#define TBP         0x40000000	/* back pressure occurred                    */
#define TPP         0x20000000	/* packet requested during PAUSE             */
#define TPCF        0x10000000	/* transmit PAUSE control frame              */
#define TCFR        0x08000000	/* transmit control frame                    */
#define TUDR_AB     0x04000000	/* underrun (abort)                          */
#define TGNT_AB     0x02000000	/* greater than LMAX (abort)                 */
#define LCOL_AB     0x01000000	/* late collision  (abort)                   */
#define ECOL_AB     0x00800000	/* excessive collisions (abort)              */
#define TEDFR_AB    0x00400000	/* excessive defer (abort)                   */
#define TDFR        0x00200000	/* single defer                              */
#define TBRO        0x00100000	/* broadcast packet                          */
#define TMUL        0x00080000	/* multicast packet                          */
#define TDONE       0x00040000	/* transmit complete                         */
#define TFLOR       0x00020000	/* length field  was over 1518 bytes         */
#define TFLER       0x00010000	/* length field didn't match actual length   */
#define TCRCE       0x00008000	/* CRC error                                 */
#define TCBC_MASK   0x00007800	/* number of collisions                      */
#define TBYT_MASK   0x000007ff	/* number of the transmitted bytes           */

/*---------------------------------------------------------------------------*/
/* RXCFG - Receive Configuration (218H R/W)                                  */
/*---------------------------------------------------------------------------*/
#define DRBS1       0x00000000	/* DMA Receive Burst Size 1 word         */
#define DRBS2       0x00010000	/* DMA Receive Burst Size 2 word         */
#define DRBS4       0x00020000	/* DMA Receive Burst Size 4 word         */
#define DRBS8       0x00030000	/* DMA Receive Burst Size 8 word         */
#define DRBS16      0x00040000	/* DMA Receive Burst Size 16 word        */
#define DRBS32      0x00050000	/* DMA Receive Burst Size 32 word        */
#define DRBS64      0x00060000	/* DMA Receive Burst Size 64 word        */
#define RXE         0x80000000	/* Receive Enable                        */
#define RXCFG_RESERVED  0x80070000	/* reserved bit                          */

/*---------------------------------------------------------------------------*/
/* RXFC - Receive FiFo Control (21CH R/W)                                    */
/*---------------------------------------------------------------------------*/
#define UWM_MASK        0xfc000000	/* Upper Water Mark mask             */
#define UWM             0xc0000000	/* default 110000b ( 48word, 192byte ) */
#define LWM_MASK        0x00fc0000	/* Lower Water Mark mask             */
#define LWM             0x00400000	/* default 010000b (16word, 64byte)  */
#define RX_DRTH_MASK    0x000000fc	/* Receive Drain Threshold Level     */
#define RX_DRTH16W      0x00000040	/* default 010000b (16word, 64byte)  */
#define RX_DRTH28W      0x00000070	/* default 011100b (28word, 112byte) */
#define RXFC_RESERVED   0xfcfc00fc	/* reserved bit                      */

/*---------------------------------------------------------------------------*/
/* RXST - Receive Status (224H R)                                            */
/*---------------------------------------------------------------------------*/
#define RLENE       0x80000000	/* less than 64 or larger than 1518      */
#define VLAN        0x40000000	/*  match VLTP                           */
#define USOP        0x20000000	/* unknown OP code control frame         */
#define RPCF        0x10000000	/* receive PAUSE control frame           */
#define RCFR        0x08000000	/* receive control frame                 */
#define DBNB        0x04000000	/* alignment error                       */
#define RBRO        0x02000000	/* broadcast packet                      */
#define RMUL        0x01000000	/* multicast packet                      */
#define RX_OK       0x00800000	/* receive OK                            */
#define RLOR        0x00400000	/* length field was over 1518 bytes      */
#define RLER        0x00200000	/* length field didn't match actual length */
#define RCRCE       0x00100000	/* CRC error                             */
#define RCVE        0x00080000	/* RXER was detected (PHY error)         */
#define CEPS        0x00040000	/* false Carrier                         */
#define REPS        0x00020000	/* preamble+SFD or +one data nibble      */
#define PAIG        0x00010000	/* carrier length 3036 octets
				   or Short IPG or invalid preamble or invalid SFD */
#define RBYT_MASK   0x0000ffff	/* received byte count                   */

/*---------------------------------------------------------------------------*/
/* RXPD - Receive Pool Descriptor (230H R/W)                                 */
/*---------------------------------------------------------------------------*/
#define AL          0x70000000	/* Alert Level default value             */
#define AL_MASK     0x70000000
#define RNOD_MASK   0x0000ffff	/* Remaining Number of Descriptor        */
#define RXPD_RESERVED   0x7000ffff	/* reserved bit                          */

/*---------------------------------------------------------------------------*/
/* CCR - Candy Configuration Register (234H R / 240H W)                      */
/*---------------------------------------------------------------------------*/
#define SRT         0x00000001	/* Candy Software Reset                  */

#define	MII_PIN_SELECT	0x20000000	/* MII pin selection                     */
#define	RMII_MODE	0x40000000	/* RMII mode                             */
#define	SPD100		0x80000000	/* Operation speed in RMII mode          */

/*---------------------------------------------------------------------------*/
/* ISR - Interrupt Serves Register (238H R with clear)                       */
/* MSR - Mask Serves Register      (23cH R/W)                                */
/*---------------------------------------------------------------------------*/
#define BUSERR      0x80000000	/* IBUS Error                                */
#define XMTDN       0x00008000	/* Transmit Done                             */
#define TBDR        0x00004000	/* Transmit Buffer Descriptor Request at Null */
#define TFLE        0x00002000	/* Transmit Frame Length Exceed              */
#define UR          0x00001000	/* Underrun                                  */
#define TABR        0x00000800	/* Transmit Aborted                          */
#define TCF         0x00000400	/* Transmit Control Frame                    */
#define RCVDN       0x00000080	/* Receive Done                              */
#define RBDRS       0x00000040	/* Receive Buffer Descriptor Request at alert level */
#define RBDRU       0x00000020	/* Receive Buffer Descriptor Request at zero */
#define OF          0x00000010	/* Overflow                                  */
#define LFAL        0x00000008	/* Link Failed                               */
#define CARRY       0x00000001	/* statistics counters carry flag            */
#define ISR_RESERVED    0x8000fcf9	/* reserved bit                          */

#define INT_ISR_TX_MASK         0x0000FC00	/* ISR TX bits mask */
#define INT_ISR_RX_MASK         0x000000F0	/* ISR RX bits mask */

/*---------------------------------------------------------------------------*/
/* Transmit/Receive Status bit definition in Transmit/Receive Descriptor     */
/*---------------------------------------------------------------------------*/
#define LAST        0x8000	/* Last Descriptor                       */
#define DB_LP       0x4000	/* Data Buffer / Link Pointer            */
#define OWN         0x2000	/* Owner 1:used by candy, 0:host set     */
/*---------------------------------------------------------------------------*/
/* Transmit Status bit definition in Transmit Descriptor                     */
/*---------------------------------------------------------------------------*/
#define DBRE        0x1000	/* Data Buffer Read Error                */
#define TUDR        0x0800	/* Transmit Underrun Error               */
#define CSE         0x0400	/* Carrier Sense Lost Error              */
#define LCOL        0x0200	/* Late Collision                        */
#define ECOL        0x0100	/* Excessive Collision                   */
#define EDFR        0x0080	/* Excessive Deferral                    */
#define TGNT        0x0004	/* Transmit Giant Frame                  */
#define HBF         0x0002	/* Heart Beat Fail for ENDEC mode        */
#define TOK         0x0001	/* Transmit OK                           */
/*---------------------------------------------------------------------------*/
/* Receive Status bit definition in Receive Descriptor                       */
/*---------------------------------------------------------------------------*/
#define DBWE        0x1000	/* Data Buffer Write Error               */
#define FTYP_MASK   0x0e00	/* Frame Type                            */
#define    BCASTF      0x0000	/* Broadcast Frame                       */
#define    MCASTF      0x0200	/* Multicast Frame                       */
#define    UCASTF      0x0400	/* Unicast Frame                         */
#define    VLANF       0x0600	/* VLAN Frame                            */
#define    PAUSEF      0x0800	/* PAUSE control Frame                   */
#define    CTLF        0x0a00	/* Control Frame                         */
#define OVRN        0x0100	/* Overrun Error                         */
#define RUNT        0x0080	/* Runt packet                           */
#define FRGE        0x0040	/* Fragment Error                        */
#define RCV         0x0020	/* Detects RXER                          */
#define FC          0x0010	/* False Carrier                         */
#define CRCE        0x0008	/* CRC Error                             */
#define FAE         0x0004	/* Frame Alignment Error                 */
#define RFLE        0x0002	/* Receive Frame Length Error            */
#define RXOK        0x0001	/* Receive OK                            */

#if defined(CONFIG_NEC_CMBVR4133)
#define CANDY_REGS_VER		2	/* version 2 */
#else
#define CANDY_REGS_VER		1	/* version 1 */
#endif

#define CANDY_REGS_SIZE		(0x3ff + 1)

/***********************************************************************
 * data structure
 ***********************************************************************
 */
typedef volatile struct {
	ulong macc1;		/* 0x00     MAC configuration register 1             */
	ulong macc2;		/* 0x04     MAC configuration register 2             */
	ulong ipgt;		/* 0x08     Back-to-Back IPG register                */
	ulong ipgr;		/* 0x0c     Non Back-to-Back IPG register            */
	ulong clrt;		/* 0x10     Collision register */
	ulong lmax;		/* 0x14     Max packet length register               */
	ulong reserved0[2];
	ulong retx;		/* 0x20     Retry count register                     */
	ulong reserved1[12];
	ulong lsa2;		/* 0x54     Station Address register 2               */
	ulong lsa1;		/* 0x58     Station Address register 1               */
	ulong ptvr;		/* 0x5c     Pause timer value read register          */
	ulong reserved2[1];
	ulong vltp;		/* 0x64     VLAN type register                       */
	ulong reserved3[6];
	ulong miic;		/* 0x80     MII configuration register               */
	ulong reserved4[4];
	ulong mcmd;		/* 0x94     MII command register                     */
	ulong madr;		/* 0x98     MII address register                     */
	ulong mwtd;		/* 0x9c     MII write data register                  */
	ulong mrdd;		/* 0xa0     MII read data register                   */
	ulong mind;		/* 0xa4     MII indicator register                   */
	ulong reserved5[6];
	ulong stlc;		/* 0xc0     Statistics counter configuration register */
	ulong reserved6[1];
	ulong afr;		/* 0xc8     Address filter register                  */
	ulong ht1;		/* 0xcc     Hash table register 1                    */
	ulong ht2;		/* 0xd0     Hash table register 2                    */
	ulong reserved7[2];
	ulong car1;		/* 0xdc     Carry register 1                         */
	ulong car2;		/* 0xe0     Carry register 2                         */
	ulong reserved8[19];
	ulong cam1;		/* 0x130    Carry mask register 1                    */
	ulong cam2;		/* 0x134    Carry mask register 2                    */
	ulong reserved9[2];

	/* RX Statistics Counter */
	ulong rbyt;		/* 0x140    Receive Byte Counter                     */
	ulong rpkt;		/* 0x144    Receive Packet Counter                   */
	ulong rfcs;		/* 0x148    Receive FCS Error Counter                */
	ulong rmca;		/* 0x14c    Receive Multicast Packet Counter         */
	ulong rbca;		/* 0x150    Receive Broadcast Packet Counter         */
	ulong rxcf;		/* 0x154    Receive Control Frame Packet Counter     */
	ulong rxpf;		/* 0x158    Receive PAUSE Frame Packet Counter       */
	ulong rxuo;		/* 0x15c    Receive Unknown OP code Counter          */
	ulong raln;		/* 0x160    Receive Alignment Error Counter          */
	ulong rflr;		/* 0x164    Receive Frame Length Out of Range Counter */
	ulong rcde;		/* 0x168    Receive Code Error Counter               */
	ulong rfcr;		/* 0x16c    Receive False Carrier Counter            */
	ulong rund;		/* 0x170    Receive Undersize Packet Counter         */
	ulong rovr;		/* 0x174    Receive Oversize Packet Counter          */
	ulong rfrg;		/* 0x178    Receive Error Undersize Packet Counter   */
	ulong rjbr;		/* 0x17c    Receive Error Oversize Packet Counter    */
	ulong r64;		/* 0x180    Receive 64 Byte Frame Counter            */
	ulong r127;		/* 0x184    Receive 65 to 127 Byte Frame Counter     */
	ulong r255;		/* 0x188    Receive 128 to 255 Byte Frame Counter    */
	ulong r511;		/* 0x18c    Receive 256 to 511 Byte Frame Counter    */
	ulong r1k;		/* 0x190    Receive 512 to 1023 Byte Frame Counter   */
	ulong rmax;		/* 0x194    Receive Over 1023 Byte Frame Counter     */
	ulong rvbt;		/* 0x198    Receive Valid Byte Counter               */
	ulong reserved10[9];

	/* Tx Statistics Counter */
	ulong tbyt;		/* 0x1c0    Transmit Byte Counter                    */
	ulong tpct;		/* 0x1c4    Transmit Packet Counter                  */
	ulong tfcs;		/* 0x1c8    Transmit CRC Error Packet Counter        */
	ulong tmca;		/* 0x1cc    Transmit Multicast Packet Counter        */
	ulong tbca;		/* 0x1d0    Transmit Broadcast Packet Counter        */
	ulong tuca;		/* 0x1d4    Transmit Unicast Packet Counter          */
	ulong txpf;		/* 0x1d8    Transmit PAUSE control Frame Counter     */
	ulong tdfr;		/* 0x1dc    Transmit Single Deferral Packet Counter  */
	ulong txdf;		/* 0x1e0    Transmit Excessive Deferral Packet Counter  */
	ulong tscl;		/* 0x1e4    Transmit Single Collision Packet Counter    */
	ulong tmcl;		/* 0x1e8    Transmit Multiple collision Packet Counter  */
	ulong tlcl;		/* 0x1ec    Transmit Late Collision Packet Counter      */
	ulong txcl;		/* 0x1f0    Transmit Excessive Collision Packet Counter */
	ulong tncl;		/* 0x1f4    Transmit Total Collision Counter         */
	ulong tcse;		/* 0x1f8    Transmit Carrier Sense Error Counter     */
	ulong time;		/* 0x1fc    Transmit Internal MAC Error Counter      */

/*---------------------------------------------------------------------------*/
/* Candy DMA and FiFo Management registers                                   */
/*---------------------------------------------------------------------------*/
	ulong txcfg;		/* 0x200    Transmit Configuration                   */
	ulong txfc;		/* 0x204    Transmit FiFo Control                    */
	ulong txd;		/* 0x208    Transmit Data                            */
	ulong txst;		/* 0x20c    Transmit Status                          */
	ulong txfap;		/* 0x210    TX FiFo access pointer                   */
	ulong txdp;		/* 0x214    Transmit Descriptor Pointer              */
	ulong rxcfg;		/* 0x218    Receive Configuration                    */
	ulong rxfc;		/* 0x21c    Receive FiFo Control                     */
	ulong rxd;		/* 0x220    Receive Data                             */
	ulong rxst;		/* 0x224    Receive Status                           */
	ulong rxfap;		/* 0x228    RX FiFo access pointer                   */
	ulong rxdp;		/* 0x22c    Receive Descriptor Pointer               */
	ulong rxpd;		/* 0x230    Receive Pool Descriptor                  */

/*---------------------------------------------------------------------------*/
/* Candy Interrupt and Configuratioin registers                              */
/*---------------------------------------------------------------------------*/
	ulong ccr;		/* 0x234    CANDY Configuration Read/Write register  */
	ulong isr;		/* 0x238    Interrupt Serves register                */
	ulong msr;		/* 0x23c    Mask Serves register                     */
} candy_regs;

/*
 * descriptor structure
 */
struct candy_desc {
#if defined(__LITTLE_ENDIAN)
	ushort size;
	ushort status;
#elif defined(__BIG_ENDIAN)
	ushort status;
	ushort size;
#else
#error "No endian format defined!"
#endif
	ulong pointer;
};

/*
 * private data structure for candy driver.
 */
struct candy_private {
	struct candy_desc *tx_ring;
	struct sk_buff *tx_skb[TX_RING_SIZE];

	struct candy_desc *rx_ring;
	struct sk_buff *rx_skb[RX_RING_SIZE];
	int rx_disable;

	uint rx_head;
	uint tx_head, tx_stop, tx_tail;
	int tx_count;

	struct net_device_stats stats;

	u32 msg_enable;

	spinlock_t lock;

	struct mii_if_info mii_if;

	/* MII status */
	struct link_status {
		uint fullduplex:1;
		uint speed100:1;
		uint linkOK:1;
	} link_status;

	struct timer_list phy_timer;

	/* hardware related */
	candy_regs *regs;

	/* house keeping */
	struct net_device *dev;
	struct candy_private *next;
};

/***********************************************************************
 * global data
 ***********************************************************************
 */

/* These identify the driver base version and may not be removed. */
static char version[] __devinitdata =
    DRV_NAME ": 10/100 Ethernet driver v" DRV_VERSION " (" DRV_RELDATE ")";

static struct candy_private *candy_priv_head = NULL;

/***********************************************************************
 * helpers
 ***********************************************************************
 */
#define		candy_in(x)		(*(volatile ulong*)&(x))
#define		candy_out(x, y)		(*(volatile ulong*)&(x)) = (y)

#define		candy_set_bits(x, mask)  candy_out(x, candy_in(x) | mask)
#define		candy_clear_bits(x, mask)  candy_out(x, candy_in(x) & ~mask)

#define		candy_set_macc1_bits(x, mask) \
		candy_out(x, (candy_in(x) | mask) & MACC1_RESERVED)
#define		candy_clear_macc1_bits(x, mask) \
		candy_out(x, (candy_in(x) & ~mask) & MACC1_RESERVED)

#define		candy_set_macc2_bits(x, mask) \
		candy_out(x, (candy_in(x) | mask) & MACC2_RESERVED)
#define		candy_clear_macc2_bits(x, mask) \
		candy_out(x, (candy_in(x) & ~mask) & MACC2_RESERVED)

/***********************************************************************
 * low-level hardware functions
 ***********************************************************************
 */

/*
 * mdio_write: write a single MII register.
 *
 * INPUTS:  dev         = network device 
 *          phy_id      = MII PHY address to write
 *          location    = MII register offset
 *          val         = value to write
 * RETURNS: None
 *
 */
static void
mdio_write(struct net_device *dev, int phy_id, int location, int val)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;
	candy_regs *p = pp->regs;

	/* wait for busy */
	while (candy_in(p->mind) & BUSY) ;

	/* start the write */

	candy_out(p->madr,
		  ((phy_id << FIAD_SHIFT) & FIAD_MASK) |
		  (location & RGAD_MASK));
	candy_out(p->mwtd, (ulong) val);
}

/*
 * mdio_read: read a single MII register.
 *
 * INPUTS:  dev         = network device 
 *          phy_id      = MII PHY address to read
 *          location    = MII register offset
 * RETURNS: value
 */
static int
mdio_read(struct net_device *dev, int phy_id, int location)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;
	candy_regs *p = pp->regs;

	/* wait for busy */
	while (candy_in(p->mind) & BUSY) ;

	/*  mac_reg->mcmd = 0; */
	candy_out(p->madr,
		  ((phy_id << FIAD_SHIFT) & FIAD_MASK) |
		  (location & RGAD_MASK));
	candy_out(p->mcmd, RSTAT);

	/* wait for busy */
	while (candy_in(p->mind) & BUSY) ;

	return (ushort) candy_in(p->mrdd);
}

static void
candy_set_media_speed(struct candy_private *pp)
{
	int lpa, advertise, media; 
	ulong ccr = RMII_MODE | MII_PIN_SELECT;

	lpa = mdio_read(pp->mii_if.dev, pp->mii_if.phy_id, MII_LPA);
	advertise = mdio_read(pp->mii_if.dev, pp->mii_if.phy_id, MII_ADVERTISE);
	                                               
	if(lpa == 0 || advertise == 0) {
		candy_out(pp->regs->ccr, RMII_MODE | MII_PIN_SELECT);
		pp->link_status.speed100 = 1;
		pp->link_status.fullduplex = 1;
		return;
	}

	media = mii_nway_result(lpa & advertise);
	switch(media) {
		case LPA_10FULL:
		case LPA_10HALF:
			ccr |= SPD100;
			pp->link_status.speed100 = 0;
			break;
		default:
			pp->link_status.speed100 = 1;
	}

	switch(media) {
		case LPA_10HALF:
		case LPA_100HALF:
			pp->link_status.fullduplex = 0;
			break;
		default:
			pp->link_status.fullduplex = 1;
	}

	candy_out(pp->regs->ccr, ccr);
}

static int
candy_ethtool_ioctl(struct candy_private *pp, void *useraddr)
{
	u32 ethcmd;

	/* dev_ioctl() in ../../net/core/dev.c has already checked
	   capable(CAP_NET_ADMIN), so don't bother with that here.  */

	if (get_user(ethcmd, (u32 *) useraddr))
		return -EFAULT;

	switch (ethcmd) {

	case ETHTOOL_GDRVINFO:{
			struct ethtool_drvinfo info = { ETHTOOL_GDRVINFO };
			strcpy(info.driver, DRV_NAME);
			strcpy(info.version, DRV_VERSION);
			strcpy(info.bus_info, "");
			info.regdump_len = CANDY_REGS_SIZE;
			info.n_stats = 0;
			if (copy_to_user(useraddr, &info, sizeof (info)))
				return -EFAULT;
			return 0;
		}

		/* get settings */
	case ETHTOOL_GSET:{
			struct ethtool_cmd ecmd = { ETHTOOL_GSET };
			spin_lock_irq(&pp->lock);
			mii_ethtool_gset(&pp->mii_if, &ecmd);
			spin_unlock_irq(&pp->lock);
			if (copy_to_user(useraddr, &ecmd, sizeof (ecmd)))
				return -EFAULT;
			return 0;
		}
		/* set settings */
	case ETHTOOL_SSET:{
			int r;
			struct ethtool_cmd ecmd;
			if (copy_from_user(&ecmd, useraddr, sizeof (ecmd)))
				return -EFAULT;
			spin_lock_irq(&pp->lock);
			r = mii_ethtool_sset(&pp->mii_if, &ecmd);
			spin_unlock_irq(&pp->lock);
			return r;
		}
		/* restart autonegotiation */
	case ETHTOOL_NWAY_RST:{
			return mii_nway_restart(&pp->mii_if);
		}
		/* get link status */
	case ETHTOOL_GLINK:{
			struct ethtool_value edata = { ETHTOOL_GLINK };
			edata.data = mii_link_ok(&pp->mii_if);
			if (copy_to_user(useraddr, &edata, sizeof (edata)))
				return -EFAULT;
			return 0;
		}

		/* get message-level */
	case ETHTOOL_GMSGLVL:{
			struct ethtool_value edata = { ETHTOOL_GMSGLVL };
			edata.data = pp->msg_enable;
			if (copy_to_user(useraddr, &edata, sizeof (edata)))
				return -EFAULT;
			return 0;
		}
		/* set message-level */
	case ETHTOOL_SMSGLVL:{
			struct ethtool_value edata;
			if (copy_from_user(&edata, useraddr, sizeof (edata)))
				return -EFAULT;
			pp->msg_enable = edata.data;
			return 0;
		}

		/* register's dump */
	case ETHTOOL_GREGS:{
			struct ethtool_regs regs;
			u8 *regbuf = kmalloc(CANDY_REGS_SIZE, GFP_KERNEL);
			int rc;

			if (!regbuf)
				return -ENOMEM;
			memset(regbuf, 0, CANDY_REGS_SIZE);

			rc = copy_from_user(&regs, useraddr, sizeof (regs));
			if (rc) {
				rc = -EFAULT;
				goto err_out_gregs;
			}

			if (regs.len > CANDY_REGS_SIZE)
				regs.len = CANDY_REGS_SIZE;
			if (regs.len < CANDY_REGS_SIZE) {
				rc = -EINVAL;
				goto err_out_gregs;
			}

			regs.version = CANDY_REGS_VER;
			rc = copy_to_user(useraddr, &regs, sizeof (regs));
			if (rc) {
				rc = -EFAULT;
				goto err_out_gregs;
			}

			useraddr += offsetof(struct ethtool_regs, data);

			spin_lock_irq(&pp->lock);
			memcpy_fromio(regbuf, pp->regs, CANDY_REGS_SIZE);
			spin_unlock_irq(&pp->lock);

			if (copy_to_user(useraddr, regbuf, regs.len))
				rc = -EFAULT;

		      err_out_gregs:
			kfree(regbuf);
			return rc;
		}

		/* get/set TX checksumming */
	case ETHTOOL_GTXCSUM:{
			struct ethtool_value edata = { ETHTOOL_GTXCSUM };

			edata.data = (pp->dev->features & NETIF_F_IP_CSUM) != 0;
			if (copy_to_user(useraddr, &edata, sizeof (edata)))
				return -EFAULT;
			return 0;
		}
	case ETHTOOL_STXCSUM:{
			struct ethtool_value edata;

			if (copy_from_user(&edata, useraddr, sizeof (edata)))
				return -EFAULT;

			if (edata.data)
				pp->dev->features |= NETIF_F_IP_CSUM;
			else
				pp->dev->features &= ~NETIF_F_IP_CSUM;

			return 0;
		}

		/* get/set scatter-gather */
	case ETHTOOL_GSG:{
			struct ethtool_value edata = { ETHTOOL_GSG };

			edata.data = (pp->dev->features & NETIF_F_SG) != 0;
			if (copy_to_user(useraddr, &edata, sizeof (edata)))
				return -EFAULT;
			return 0;
		}
	case ETHTOOL_SSG:{
			struct ethtool_value edata;

			if (copy_from_user(&edata, useraddr, sizeof (edata)))
				return -EFAULT;

			if (edata.data)
				pp->dev->features |= NETIF_F_SG;
			else
				pp->dev->features &= ~NETIF_F_SG;

			return 0;
		}

	default:
		break;
	}

	return -EOPNOTSUPP;
}

static int
candy_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
	struct candy_private *pp = dev->priv;
	struct mii_ioctl_data *mii = (struct mii_ioctl_data *) &rq->ifr_data;
	int rc;

	if (!netif_running(dev))
		return -EINVAL;

	if (cmd == SIOCETHTOOL)
		return candy_ethtool_ioctl(pp, (void *) rq->ifr_data);

	spin_lock_irq(&pp->lock);
	rc = generic_mii_ioctl(&pp->mii_if, mii, cmd, NULL);
	spin_unlock_irq(&pp->lock);
	return rc;
}

/*
 * SetMacAddr: set MAC address
 *
 * INPUTS:  addr    = pointer to MAC address {0xaa,0xbb,0xcc,0xdd,0xee,0xff}
 * RETURNS: None
 *
 */
static void
SetMacAddr(struct net_device *dev, u_char * addr)
{
	candy_regs *p = ((struct candy_private *) dev->priv)->regs;

	candy_out(p->lsa2, (addr[0] << 8 | addr[1]) & LSA2_MASK);
	candy_out(p->lsa1,
		  addr[2] << 24 | addr[3] << 16 | addr[4] << 8 | addr[5]);
}

/*
 * SetRxMode: set receive mode.
 *
 * INPUTS:  mode    = ACCEPT_ALL_PHYS, ACCEPT_ALL_MCASTS, ACCEPT_ALL_BCASTS,
 *                    ACCEPT_QUALIFIED_MCAST, ACCEPT_STATION, MAC_LOOPBACK
 * RETURNS: None
 *
 */
#if 0
static void
SetRxMode(struct net_device *dev, ushort mode)
{
	candy_regs *p = ((struct candy_private *) dev->priv)->regs;
	ulong value = 0;

	if (mode & ACCEPT_ALL_PHYS)
		value |= PRO;
	if (mode & ACCEPT_ALL_MCASTS)
		value |= PRM;
	if (mode & ACCEPT_ALL_BCASTS)
		value |= ABC;
	if (mode & ACCEPT_QUALIFIED_MCAST)
		value |= AMC;

	/* if the mode is only ACCEPT_STATION, value is 0. */
	candy_out(p->afr, value);

	if (mode & MAC_LOOPBACK) {
		candy_set_macc1_bits(p->macc1, MACLB | FULLD);
		printk(KERN_INFO __FILE__ " : MAC loop back mode.\n");
	}

	return;
}
#endif

/*
 * This is called when system boots up and/or after ether chip is reset.
 */
static void
candy_hw_init(struct net_device *dev)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;
	candy_regs *p = pp->regs;
	unsigned int temp;

	/*
	 * software reset
	 */
	candy_out(p->ccr, SRT);
	candy_in(p->isr);

	/*
	 * MAC software reset
	 */
	candy_out(p->macc2, MCRST);
	candy_out(p->macc2, 0x0);

	/*
	 * MII software reset
	 */
	candy_out(p->miic, MISRT);
	candy_out(p->miic, 0);

	/*
	 * initialize MAC
	 */
	temp = TXFC | RXFC | CRCEN | PADEN;
#ifdef WORKAROUND_E13_TXFC
	temp &= ~TXFC;
#endif
#ifdef WORKAROUND_E21_PAD
	temp &= ~PADEN;
#endif
	candy_out(p->macc1, temp);
	candy_out(p->macc2, APD);
	candy_out(p->ipgt, IPGT);
	candy_out(p->ipgr, IPGR1 | IPGR2);
	candy_out(p->clrt, LCOLW | RETRY);
	candy_out(p->lmax, MAXF);
	SetMacAddr(dev, dev->dev_addr);
	candy_out(p->vltp, 0x0);
	candy_out(p->miic, CLKS66);

	/*
	 * initialize DMA / FIFO
	 */
#ifdef WORKAROUND_E7_AFCE
	candy_out(p->txcfg, DTBS8);
#else
	candy_out(p->txcfg, DTBS8 | AFCE);
#endif
	candy_out(p->txcfg, DTBS8);
	candy_out(p->txfc, TPTV | TX_DRTH | TX_FLTH);
	candy_out(p->rxcfg, DRBS4);
	candy_out(p->rxfc, UWM | LWM | RX_DRTH16W);
	candy_out(p->rxpd, AL);

	/* Set pins for RMII use */
	candy_set_media_speed(pp);

	/*
	 * disable all interrupts until dev is opened later
	 */
	candy_out(p->msr, 0);
}

static void
candy_down(struct net_device *dev)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;
	candy_regs *p = pp->regs;

	DEBUG_VERBOSE(printk("candy_down() invoked.\n"));

	candy_clear_macc1_bits(p->macc1, SRXEN);

	candy_clear_bits(p->txcfg, TXE);
	candy_clear_bits(p->rxcfg, RXE);
}

static void
candy_up(struct net_device *dev)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;
	candy_regs *p = pp->regs;

	DEBUG_VERBOSE(printk("candy_up() invoked.\n"));

	/* candy must be down right now */
	ASSERT((candy_in(p->txcfg) & TXE) == 0);
	ASSERT((candy_in(p->rxcfg) & RXE) == 0);

	// korva_set_bits(KORVA_S_GMR, 0x1);

	/* soft reset rx/tx */
	candy_set_macc2_bits(p->macc2, TFRST);
	candy_clear_macc2_bits(p->macc2, TFRST);
	candy_set_macc2_bits(p->macc2, RFRST);
	candy_clear_macc2_bits(p->macc2, RFRST);

	/* set default receive mode */
	// SetRxMode(dev, ACCEPT_STATION | ACCEPT_ALL_MCASTS | ACCEPT_ALL_BCASTS );
	// SetRxMode(dev, ACCEPT_STATION | ACCEPT_ALL_BCASTS );

#ifdef	WORKAROUND_E10_PRM_AMC
	candy_out(p->afr, ABC);
#else
	candy_out(p->afr, PRM | ABC | AMC);
#endif

	/* enable transmit and receive */
	candy_set_bits(p->txcfg, TXE);
	candy_out(p->rxcfg, RXE | DRBS4);

	/* set number of descriptors */
	candy_out(p->rxpd, AL | (RX_RING_SIZE & RNOD_MASK));

	/* set the receive descriptor pointer */
	candy_out(p->rxdp, PHYSADDR(&pp->rx_ring[pp->rx_head]));

	candy_set_macc1_bits(p->macc1, SRXEN);

	candy_set_media_speed(pp);

	/* turn on interrupts */
	candy_in(p->isr);
	candy_out(p->msr, 0xffffffff & ISR_RESERVED);
}

static unsigned int
hashit(char *addr)
{
	int i;
	int j;
	unsigned char nibblehigh;
	unsigned char nibblelow;
	unsigned long da_crc = 0L;
	unsigned long hash_p = 0L;
	unsigned long hash_byte = 0L;
	unsigned long hash_bit = 0L;
	unsigned long crc32 = 0xffffffffL;
	unsigned long crcpoly2 = 0xedb88320L;

	for (i = 0; i < 6; i++) {
		nibblelow = addr[i] & 0x0f;
		nibblehigh = addr[i] >> 4;
		crc32 ^= nibblelow;
		for (j = 0; j < 4; j++) {
			if (crc32 & 1) {
				crc32 = (crc32 >> 1) ^ crcpoly2;
			} else {
				crc32 >>= 1;
			}
		}
		crc32 ^= nibblehigh;
		for (j = 0; j < 4; j++) {
			if (crc32 & 1) {
				crc32 = (crc32 >> 1) ^ crcpoly2;
			} else {
				crc32 >>= 1;
			}
		}
	}
	da_crc = crc32 & 0x000001f8L;
	for (j = 31; j >= 23; j--) {
		hash_p = hash_p * 2 + da_crc % 2;
		da_crc = da_crc / 2;
	}
	hash_byte = (hash_p & 0x00000038) >> 3;
	hash_bit = (hash_p & 0x00000007);
	return ((hash_byte << 16) | (hash_bit & 0xFFFF));
}

static void
candy_set_filter(struct net_device *dev, int on)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;
	candy_regs *p = pp->regs;
	struct dev_mc_list *mclist = dev->mc_list;
	unsigned long ht1 = 0;
	unsigned long ht2 = 0;
	int i;
	union {
		unsigned long fusion;
		unsigned short byte;
		unsigned short bit;
	} val;

	DEBUG_VERBOSE(printk
		      ("candy_set_filter() invoked - %s\n", on ? "on" : "off"));

	for (i = 0, mclist = dev->mc_list;
	     mclist && i < dev->mc_count; i++, mclist = mclist->next) {
		val.fusion = hashit(mclist->dmi_addr);
		if (val.byte > 3)
			ht1 |= (1 << ((val.byte - 4) * 8 + val.bit));
		else
			ht2 |= (1 << (val.byte * 8 + val.bit));
	}

	/* lock ints */
	candy_out(p->ht1, ht1);
	candy_out(p->ht2, ht2);
}

/*
 * Apparently candy tx can stall due to various reasons.
 * This routine will attempt to recover from tx stall.
 */
static void
candy_error_recover(struct net_device *dev)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;
	candy_regs *p = pp->regs;

	spin_lock(&pp->lock);

#if 0				/* solution 1 */
	candy_close(dev);
	candy_hw_init(dev);
	candy_open(dev);

	/* this flag is not set by candy_open().  */
	dev->flags |= IFF_UP;
#endif

#if 1				/* solution 2 */
	netif_stop_queue(dev);

	candy_down(dev);
	candy_hw_init(dev);
	candy_up(dev);

	netif_wake_queue(dev);

	/* restart transmitting */
	pp->tx_stop = pp->tx_tail;
	candy_out(p->txdp, PHYSADDR(&pp->tx_ring[pp->tx_head]));
#endif

	spin_unlock(&pp->lock);
}

#if defined(WORKAROUND_E8_TX_STALL)
/*
 * This implements the workaround described for E-8
 */
static void
tx_stall_recover(struct net_device *dev)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;
	candy_regs *p = pp->regs;

	/* E-8 bug only happens when receiving is on */
	if ((candy_in(p->macc1) & SRXEN) && (candy_in(p->rxcfg) & RXE)) {

		candy_clear_macc1_bits(p->macc1, SRXEN);
		candy_clear_bits(p->rxcfg, RXE);

		udelay(20);

		candy_out(p->txdp, PHYSADDR(&pp->tx_ring[pp->tx_head]));

		candy_set_bits(p->rxcfg, RXE);
		candy_set_macc1_bits(p->macc1, SRXEN);

	} else {
		candy_out(p->txdp, PHYSADDR(&pp->tx_ring[pp->tx_head]));
	}
}
#endif

/***********************************************************************
 * hardware-independent helper routine
 ***********************************************************************
 */
static void
candy_init_rings(struct net_device *dev)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;
	int i;

	DEBUG_VERBOSE(printk("candy_init_rings() invoked.\n"));

	/* tx rings */
	for (i = 0; i < TX_RING_SIZE; i++) {
		pp->tx_ring[i].status = 0;
		pp->tx_ring[i].size = 0;
		pp->tx_ring[i].pointer = 0;
	}
	/* this may be not necessary, if we reset every txdpr in intr */
	pp->tx_ring[TX_RING_SIZE].status = 0;
	pp->tx_ring[TX_RING_SIZE].size = 0x0;
	pp->tx_ring[TX_RING_SIZE].pointer = PHYSADDR(pp->tx_ring);

	pp->tx_tail = pp->tx_stop = pp->tx_head = 0;
	pp->tx_count = 0;

	/* rx rings */
	for (i = 0; i < RX_RING_SIZE; i++) {
		pp->rx_ring[i].status = DB_LP;
		pp->rx_ring[i].size = RX_BUF_SIZE;
		pp->rx_ring[i].pointer = PHYSADDR(pp->rx_skb[i]->data);
	}
	pp->rx_ring[RX_RING_SIZE].status = 0;	/* link back to the beginning */
	pp->rx_ring[RX_RING_SIZE].size = 0xffff;
	pp->rx_ring[RX_RING_SIZE].pointer = PHYSADDR(pp->rx_ring);

	pp->rx_head = 0;
	pp->rx_disable = 0;
}

static void
candy_check_intr(ulong isr)
{
	static const char *intr_name[32] = {
		"carry flag",
		"reserved",
		"reserved",
		"link failed",
		"overflow",
		"receive buffer desc request at zero",
		"receive buffer desc request at alert level",
		"receive done",

		"reserved",
		"reserved",
		"control frame transmit",
		"transmit aborted",
		"underrun",
		"transmit frame length exceed",
		"transmit buffer descriptor request at NULL",
		"transmit done",

		"reserved",
		"reserved",
		"reserved",
		"reserved",	/* 4 */
		"reserved",
		"reserved",
		"reserved",
		"reserved",

		"reserved",
		"reserved",
		"reserved",
		"reserved",	/* 4 */
		"reserved",
		"reserved",
		"reserved",
		"IBUS error"
	};
	ulong i, j;

	for (i = 0, j = 1; i < 32; j <<= 1, i++) {
		if (j & isr)
			printk("\t%s\n", intr_name[i]);
	}
}

static /* inline */ void
reclaim_one_rx_desc(struct net_device *dev, char *buf)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;
	candy_regs *p = pp->regs;
	struct candy_desc *dp = &pp->rx_ring[pp->rx_head];

	ASSERT(KSEGX(dp) == KSEG1);

	if (buf != NULL) {
		dp->pointer = PHYSADDR(buf);
	}
	dp->status = DB_LP;	/* 1 stands for buffer vs link ptr */
	dp->size = RX_BUF_SIZE;

	/* we need to clean up cache here.  Otherwise we may have some
	 * dirty cache while ether controller is feeding fresh pkt data
	 * to the physical RAM.  Data corruption could happen.
	 */
	dma_cache_wback_inv(KSEG0ADDR(dp->pointer), RX_BUF_SIZE);

	if (++pp->rx_head == RX_RING_SIZE)
		pp->rx_head = 0;
	ASSERT(pp->rx_head < RX_RING_SIZE);

	/* tell hardware we have one descriptor to work with */
	candy_out(p->rxpd, AL | 1);

	/* check rx_disable */
	if (pp->rx_disable) {
		pp->rx_disable = 0;
		candy_out(p->rxdp, PHYSADDR(dp));
		candy_set_macc1_bits(p->macc1, SRXEN);
		DEBUG(printk(KERN_WARNING "%s : re-enable SRXEN\n", dev->name));
	}
}

static void
handle_rx_error(struct net_device *dev, struct candy_desc *dp, ulong isr)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;

	ASSERT(KSEGX(dp) == KSEG1);

	if (netif_msg_rx_err(pp))
		printk(KERN_DEBUG "%s: rx_err, rx_ring[%d] error,"
		       "status =%04x, size=%d, isr = 0x%08lx.\n",
		       dev->name, pp->rx_head, dp->status, dp->size, isr);

	/* log some errors that hardware don't log */
	pp->stats.rx_errors++;
	if (isr & OF)
		pp->stats.rx_fifo_errors++;
	if (isr & RBDRU)
		pp->stats.rx_missed_errors++;
	if (dp->status & OVRN)
		pp->stats.rx_over_errors++;
}

static void
candy_rx(struct net_device *dev, ulong isr)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;
	candy_regs *p = pp->regs;
	struct candy_desc *dp = &pp->rx_ring[pp->rx_head];
	int pkt_len;
	struct sk_buff *newskb;
	struct sk_buff *rxskb;
	int i;
	int skb_size;

	ASSERT(KSEGX(dp) == KSEG1);

	if (netif_msg_rx_status(pp)) {
		printk(KERN_DEBUG "%s: rx, stat=0x%08lx\n",
		       dev->name, candy_in(p->rxst));
		printk(KERN_DEBUG "\trx_head = %d, {0x%04x, %u, 0x%08lx}\n",
		       pp->rx_head, dp->status, dp->size, dp->pointer);
	}

	/* check if we still have any empty receive descriptor */
	if (isr & RBDRU) {
		printk(KERN_ERR
		       "%s : no more receive buffers.  Stop receiving.",
		       dev->name);
		candy_clear_bits(p->macc1, SRXEN);
		pp->rx_disable = 1;
	}

	/* FIXME : we are fetching packets.  How do we know where the
	 * end is?  When OWN bit is 0 (in previous linux driver)?
	 *
	 * Checking OWN bit alone is not reliable especially when kgdb
	 * is used.  Hack to work around.  We really need to understand
	 * what entries we can consume here.
	 */
	for (i = 0;; i++) {
		dp = &pp->rx_ring[pp->rx_head];

		if (((dp->status & OWN) == 0) &&
				((pp->rx_disable == 0) || (i != 0))) {
			/* no frame received in this descriptor yet */
			break;
		}

		/* handle the error case */
		if ((dp->status & RXOK) == 0) {
			handle_rx_error(dev, dp, isr);
			reclaim_one_rx_desc(dev, NULL);
			continue;
		}

		/* oversize? */
		if (dp->size > ETH_FRAME_LEN) {
			if (netif_msg_rx_status(pp))
				printk(KERN_DEBUG "%s: rx, oversized pkt\n",
				       dev->name);
			pkt_len = ETH_FRAME_LEN;
		} else {
			pkt_len = dp->size;
		}

		/* we got a good packet */

		/* STRATEGY: ether packet has 14 bytes.  So we will
		 * suffer from emulated unaligned access if we pass
		 * the skb straight to upper layer.  An alternative is
		 * to copy the buffer by offset of 2 and then pass it up.
		 * Then the overhead is copying.
		 *
		 * In general, it is more beneficial to copy if we have smaller
		 * packet.  Also, it is more beneficial to copy if we have
		 * faster machines.
		 *
		 * To keep it flexible, we will leave rx_copybreak flexible.
		 */

		if (pkt_len < rx_copybreak) {
			skb_size = pkt_len + 2;
		} else {
			skb_size = RX_BUF_SIZE;
		}

		/* allocate a new skb */
		newskb = dev_alloc_skb(skb_size);
		if (newskb == NULL) {
			printk(KERN_ERR
			       "%s: Memory squeeze, dropping packet.\n",
			       dev->name);
			reclaim_one_rx_desc(dev, NULL);
			pp->stats.rx_dropped++;
			continue;
		}

		if (pkt_len <= rx_copybreak) {
			/* we will copy */
			rxskb = pp->rx_skb[pp->rx_head];
			ASSERT(rxskb->data == (void *) KSEG0ADDR(dp->pointer));

			newskb->dev = dev;
			skb_reserve(newskb, 2);	/* align IP pkt to 16-byte */
			eth_copy_and_sum(newskb, rxskb->data, pkt_len, 0);

			rxskb = newskb;
			reclaim_one_rx_desc(dev, NULL);
		} else {
			/* use the new skb to replace the recived one */
			rxskb = pp->rx_skb[pp->rx_head];

			newskb->dev = dev;
			pp->rx_skb[pp->rx_head] = newskb;
			reclaim_one_rx_desc(dev, newskb->data);

		}

		skb_put(rxskb, pkt_len);
		rxskb->protocol = eth_type_trans(rxskb, dev);
		rxskb->ip_summed = CHECKSUM_UNNECESSARY;
		netif_rx(rxskb);

		dev->last_rx = jiffies;
	}

	/* when we are out here, should rxdp be the same as 
	 * &pp->rx_ring[pp->head]?
	 */
	// ASSERT(candy_in(p->rxdp) == PHYSADDR(&pp->rx_ring[pp->rx_head]));
	if (netif_msg_rx_status(pp))
		printk(KERN_DEBUG "%s: rx, processed %d frames.\n",
		       dev->name, i);
#ifdef SHOW_BUG
	if ((candy_in(p->rxdp) != PHYSADDR(&pp->rx_ring[pp->rx_head])) &&
	    (candy_in(p->rxdp) != PHYSADDR(&pp->rx_ring[RX_RING_SIZE]))) {
		int i;
		printk
		    ("%s : unexpected out of rx - rx_ring[rx_head] = (%04x, %d)\n",
		     dev->name, dp->status, dp->size);
		for (i = 0; i < RX_RING_SIZE + 1; i++) {
			if (p->rxdp == PHYSADDR(&pp->rx_ring[i]))
				break;
		}
		if (i == RX_RING_SIZE + 1)
			panic("cannot find current rx tail");
		printk("\trx_head = %d, rx_tail = %d\n", pp->rx_head, i);
	}
#endif
	return;
}

static void
append_one_tx_desc(struct net_device *dev,
		   ushort status,
		   ushort size, ulong pointer, struct sk_buff *skb)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;
	struct candy_desc *dp = &pp->tx_ring[pp->tx_tail];
#ifdef WORKAROUND_E10_VR4133
	int i, shift = pointer % 4;
#endif

	dp->status = status;
	dp->size = size;
	dp->pointer = pointer;
#ifdef WORKAROUND_E10_VR4133
	if(shift && (size == 30 || size == 31)) {
		if(skb_tailroom(skb) >= 4 - shift) {
			shift = 4 - shift;
			skb->data += shift;
			dp->pointer += shift;
			for(i = size + shift - 1; i >= 0; i--)
				skb->data[i] = skb->data[i - shift];
			dma_cache_wback_inv((ulong) (skb->data), skb->len);
		} else if(skb_headroom(skb) >=  shift) {
			skb->data -= shift;
			dp->pointer -= shift;
			for(i = 0; i < size; i++)
				skb->data[i] = skb->data[i + shift];
			dma_cache_wback_inv((ulong) (skb->data), skb->len);
		} else {
			panic("nec_candy.c: Unable to do a workaround for hardware bug (Restriction 10 for VR4133).\n");
		}
	}
#endif
	pp->tx_skb[pp->tx_tail] = skb;

	pp->tx_count++;
	if (++pp->tx_tail == TX_RING_SIZE)
		pp->tx_tail = 0;
}

static void
reclaim_one_tx_desc(struct net_device *dev)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;
	struct candy_desc *dp = &pp->tx_ring[pp->tx_head];

	dp->status = 0;
	dp->size = 0;
	dp->pointer = 0;

	/* free skb */
	if (pp->tx_skb[pp->tx_head]) {
		dev_kfree_skb_irq(pp->tx_skb[pp->tx_head]);
		pp->tx_skb[pp->tx_head] = NULL;
	}

	pp->tx_count--;
	if (++pp->tx_head == TX_RING_SIZE)
		pp->tx_head = 0;
}

static void
restart_tx_hw(struct net_device *dev)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;
	candy_regs *p = pp->regs;
#ifdef DEBUG_NEC_CANDY
	struct candy_desc *dp = &pp->tx_ring[pp->tx_tail];
#endif
	/* we need at least one desc for null descriptor */
	ASSERT(pp->tx_count < TX_RING_SIZE);

	/* the desc should already be set NULL */
	ASSERT(dp->status == 0);
	ASSERT(dp->size == 0);
	ASSERT(dp->pointer == 0);
	append_one_tx_desc(dev, 0, 0, 0, NULL);

	/* start */
	ASSERT(pp->tx_head == pp->tx_stop);
	pp->tx_stop = pp->tx_tail;
	candy_out(p->txdp, PHYSADDR(&pp->tx_ring[pp->tx_head]));
}

static void
handle_tx_error(struct net_device *dev, struct candy_desc *dp, ulong isr)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;

	ASSERT(KSEGX(dp) == KSEG1);

	if (netif_msg_tx_err(pp))
		printk(KERN_DEBUG "%s: tx err, tx_ring[%d] error, "
		       "status = %04x, isr = 0x%08lx.\n",
		       dev->name, pp->tx_head, dp->status, isr);

#if defined(WORKAROUND_E8_TX_STALL)
	tx_stall_recover(dev);
#endif

	pp->stats.tx_errors++;
	pp->stats.tx_aborted_errors++;
	if (dp->status & TUDR)
		pp->stats.tx_fifo_errors++;
	if (dp->status & HBF)
		pp->stats.tx_heartbeat_errors++;
}

static void
candy_tx_done(struct net_device *dev, ulong isr)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;
	struct candy_desc *dp;

	if (netif_msg_tx_done(pp))
		printk(KERN_DEBUG "%s: tx done, process %d frames from %d\n",
		       dev->name, pp->tx_tail > pp->tx_head ?
		       pp->tx_tail - pp->tx_head :
		       pp->tx_tail + TX_RING_SIZE - pp->tx_head, pp->tx_head);

	spin_lock(&pp->lock);	/* sync with xmit() */

	while (pp->tx_head != pp->tx_stop) {
		dp = &pp->tx_ring[pp->tx_head];
		ASSERT(KSEGX(dp) == KSEG1);

		/* deal with null descriptor, the "stop" packet */
		if (dp->status == 0) {
			ASSERT(dp->size == 0);
			ASSERT(dp->pointer == 0);
			reclaim_one_tx_desc(dev);
			continue;
		}

		/* how about checking OWN bit */
		if (!(dp->status & OWN)) {
			if (netif_msg_tx_done(pp))
				printk(KERN_DEBUG "%s: tx done, "
				       "found pkt being sent. Break the loop\n",
				       dev->name);
			break;
		}

		/* handle error */
		if (!(dp->status & TOK)) {
			handle_tx_error(dev, dp, isr);
			/* transmitter should have stopped */
			break;
		}

		/* reclaim the descriptor */
		if (!pp->tx_skb[pp->tx_head]) {
			printk(KERN_ERR "%s: tx_done but without skb!\n",
			       dev->name);
		}
		reclaim_one_tx_desc(dev);

		/* FIXME: The Japanese version has a tx restart under
		 * certain error conditions.  Don't understand it.
		 */
	}

	/* check if tx has stopped */
	if ((pp->tx_head == pp->tx_stop) && (pp->tx_stop != pp->tx_tail)) {
		restart_tx_hw(dev);
	}

	/* check if queue were stopped */
	if (netif_queue_stopped(dev) && (pp->tx_count < TX_RING_SIZE - 2)) {
		if (netif_msg_tx_done(pp))
			printk(KERN_DEBUG "%s: tx done, queue becomes free,"
			       "wake up net queue\n", dev->name);
		netif_wake_queue(dev);
	}

	spin_unlock(&pp->lock);
}

static void
candy_update_stats(struct net_device *dev)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;
	candy_regs *p = pp->regs;

	/* some stats we get from hardware, while the rest we do
	 * counting by ourselves
	 */
	pp->stats.rx_packets = candy_in(p->rpkt);
	pp->stats.tx_packets = candy_in(p->tpct);
	pp->stats.rx_bytes = candy_in(p->rbyt);
	pp->stats.tx_bytes = candy_in(p->tbyt);

	/* we count rx_errors, tx_errors, rx_dropped, tx_dropped */

	pp->stats.multicast = candy_in(p->rmca);
	pp->stats.multicast = candy_in(p->tncl);

	pp->stats.rx_length_errors = candy_in(p->rund) +
	    candy_in(p->rovr) + candy_in(p->rfrg) + candy_in(p->rjbr);

	/* we count rx_over_errors */

	pp->stats.rx_crc_errors = candy_in(p->rfcs);
	pp->stats.rx_frame_errors = candy_in(p->raln);

	/* we count rx_fifo_errors and rx_missed_errors */

	/* we count tx_aborted_errors */

	pp->stats.tx_carrier_errors = candy_in(p->tcse);

	/* we count tx_fifo_errors, heartbeat_errors */

	pp->stats.tx_window_errors = candy_in(p->tlcl);

	/* we don't have rx_compressed and tx_compressed */
}

/***********************************************************************
 * high-level linux-related functions
 ***********************************************************************
 */
static int candy_open(struct net_device *dev);
static int candy_close(struct net_device *dev);
static void
candy_interrupt(int irq, void *dev_instance, struct pt_regs *regs)
{
	struct net_device *dev = (struct net_device *) dev_instance;
	struct candy_private *pp = (struct candy_private *) dev->priv;
	candy_regs *p = pp->regs;
	ulong isr;

	isr = candy_in(p->isr);

	if (netif_msg_intr(pp)) {
		printk(KERN_DEBUG "%s: intr, isr = 0x%08lx\n", dev->name, isr);
		candy_check_intr(isr);
	}

	if (isr & BUSERR) {
		printk(KERN_ERR "%s: bus error ... resetting\n", dev->name);
		candy_error_recover(dev);
		return;

	}

	if (isr & INT_ISR_RX_MASK) {
		candy_rx(dev, isr);

	}
	if (isr & INT_ISR_TX_MASK) {
		candy_tx_done(dev, isr);
	}

	/* we may need to do something with other intrs too in the future */
}

static void
candy_get_phy_status(struct net_device *dev, int *duplex, int *linkup)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;
	u16 reg;

	reg = mdio_read(pp->mii_if.dev, pp->mii_if.phy_id, MII_BMCR);

	if (!(reg & BMCR_ANENABLE)) {
		/*
		 * Auto-negotiation is disabled so the full duplex bit in
		 * the control register tells us if the PHY is running
		 * half or full duplex.
		 */
		*duplex = (reg & BMCR_FULLDPLX) ? 1 : 0;
	} else {
		/*
		 * Auto-negotiation is enabled.  Figure out what was
		 * negotiated by looking for the best mode in the union
		 * of what we and our partner advertise.
		 */
		u16 advertise, partner, negotiated;

		advertise = mdio_read(pp->mii_if.dev, pp->mii_if.phy_id, MII_ADVERTISE);
		partner = mdio_read(pp->mii_if.dev, pp->mii_if.phy_id, MII_LPA);

		negotiated = advertise & partner & ADVERTISE_ALL;
		if (negotiated & ADVERTISE_100FULL)
			*duplex = 1;
		else if (negotiated & ADVERTISE_100HALF)
			*duplex = 0;
		else if (negotiated & ADVERTISE_10FULL)
			*duplex = 1;
		else
			*duplex = 0;
	}

	reg = mdio_read(pp->mii_if.dev, pp->mii_if.phy_id, MII_BMSR);

	*linkup = (reg & BMSR_LSTATUS) != 0;
}

static void
candy_poll_mii(unsigned long data)
{
	struct net_device *dev = (struct net_device *) data;
	struct candy_private *pp = (struct candy_private *) dev->priv;
	int phy_duplex, mac_duplex;
	int phy_carrier, netif_carrier;

	/* First, find out what's going on with the PHY. */
	candy_get_phy_status(dev, &phy_duplex, &phy_carrier);

	/* Second, figure out if we have the EMAC in half or full duplex. */
	mac_duplex = pp->link_status.fullduplex;

	/* Now see if there is a mismatch. */
	if (mac_duplex != phy_duplex) {
		/* reset */
		netif_stop_queue(dev);

		candy_down(dev);
		candy_hw_init(dev);
		candy_up(dev);

		netif_wake_queue(dev);
		if (mac_duplex == 1) {
			candy_clear_macc1_bits(pp->regs->macc1, FULLD);
			printk(KERN_INFO "%s: Duplex has been changed: now %s\n", dev->name, "HALF_DUPLEX");
		} else {
			candy_set_macc1_bits(pp->regs->macc1, FULLD);
			printk(KERN_INFO "%s: Duplex has been changed: now %s\n", dev->name, "FULL_DUPLEX");
		}
		candy_set_media_speed(pp);
		pp->link_status.fullduplex = phy_duplex;
		pp->mii_if.full_duplex = phy_duplex;
	}
	netif_carrier = netif_carrier_ok(dev) != 0;

	if (phy_carrier != netif_carrier) {
		if (phy_carrier) {
			candy_set_media_speed(pp);
			printk(KERN_INFO "%s: Link carrier restored.\n",
			       dev->name);
			netif_carrier_on(dev);
		} else {
			printk(KERN_INFO "%s: Link carrier lost.\n", dev->name);
			netif_carrier_off(dev);
		}
	}

	/* Set up the timer so we'll get called again in 2 seconds. */
	pp->phy_timer.expires = jiffies + 2 * HZ;
	add_timer(&pp->phy_timer);
}

static int
candy_open(struct net_device *dev)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;
	int retval;
	int count = 0;

	if (netif_msg_ifup(pp))
		printk(KERN_DEBUG "%s: enabling interface\n", dev->name);

	MOD_INC_USE_COUNT;

	candy_init_rings(dev);

	candy_hw_init(dev);
	candy_up(dev);

	dev->flags |= IFF_RUNNING;

	/* request IRQ */
	retval = request_irq(dev->irq, &candy_interrupt, SA_SHIRQ,
			     dev->name, dev);
	if (retval) {
		MOD_DEC_USE_COUNT;
		printk(KERN_ERR "%s: unable to get IRQ %d\n",
		       dev->name, dev->irq);
		return retval;
	}

	netif_carrier_off(dev);
	mii_check_media(&pp->mii_if, netif_msg_link(pp), 1);
	
	if (pp->mii_if.full_duplex) 
		candy_set_macc1_bits(pp->regs->macc1, FULLD);
	else
		candy_clear_macc1_bits(pp->regs->macc1, FULLD);

	candy_set_media_speed(pp);

	pp->phy_timer.data = (unsigned long) dev;
	pp->phy_timer.function = &candy_poll_mii;
	candy_poll_mii(dev);

	while(!netif_carrier_ok(dev) && ++count < 5) {
		udelay(10);
	}

	netif_start_queue(dev);

	return 0;
}

static int
candy_close(struct net_device *dev)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;

	if (netif_msg_ifdown(pp))
		printk(KERN_DEBUG "%s: disabling interface\n", dev->name);

	del_timer(&pp->phy_timer);

	dev->flags &= ~(IFF_UP | IFF_RUNNING);

	if (netif_device_present(dev)) {
		netif_stop_queue(dev);
		netif_carrier_off(dev);
		candy_down(dev);

		/* free tx skb */
		while (pp->tx_tail != pp->tx_head) {
			if (pp->tx_skb[pp->tx_head]) {
				dev_kfree_skb_irq(pp->tx_skb[pp->tx_head]);
				pp->tx_skb[pp->tx_head] = NULL;
			}

			pp->tx_count--;
			ASSERT(pp->tx_count >= 0);
			if (++pp->tx_head == TX_RING_SIZE)
				pp->tx_head = 0;
		}
	}

	free_irq(dev->irq, dev);

	MOD_DEC_USE_COUNT;

	return 0;
}

static int
candy_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;
	ulong flags;
#ifdef DEBUG_NEC_CANDY
	struct candy_desc *dp = &pp->tx_ring[pp->tx_tail];
#endif
	ASSERT(KSEGX(dp) == KSEG1);

	if (netif_msg_tx_queued(pp))
		printk(KERN_DEBUG "%s: tx queued, skblen %d\n",
		       dev->name, skb->len);

	/* check packet size */
	if (skb->len > ETH_FRAME_LEN) {
		printk(KERN_ERR "%s: packet size too big, %d\n", dev->name,
		       skb->len);
		pp->stats.tx_dropped++;
		return 1;
	}

	spin_lock_irqsave(&pp->lock, flags);

	/* check to see if tx_ring is full */
	if (pp->tx_count >= TX_RING_SIZE - 1) {
		printk(KERN_ERR "%s: TX ring full, packet dropped.\n",
		       dev->name);
		pp->stats.tx_dropped++;
		spin_unlock_irqrestore(&pp->lock, flags);
		/* why the queue was not stopped before we get here? */
		netif_stop_queue(dev);
		return 1;
	}

	/* add the descriptor */
	{
		ushort temp=skb->len;
#ifdef WORKAROUND_E21_PAD
		if (temp < 60)
			temp = 60;
#endif
		dma_cache_wback_inv((ulong) (skb->data), skb->len);
		append_one_tx_desc(dev, LAST | DB_LP, temp, PHYSADDR(skb->data),
			   	   skb);
	}

	/* logistics */
	dev->trans_start = jiffies;

	/* do we need to start sending or just append */
	if ((pp->tx_head == pp->tx_stop) && (pp->tx_stop != pp->tx_tail)) {
		restart_tx_hw(dev);
	}

	if (pp->tx_count >= TX_RING_SIZE - 2) {
		netif_stop_queue(dev);
	}

	spin_unlock_irqrestore(&pp->lock, flags);

	return 0;
}

static struct net_device_stats *
candy_get_stats(struct net_device *dev)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;
	unsigned long flags;

	if (netif_device_present(dev)) {
		spin_lock_irqsave(&pp->lock, flags);
		candy_update_stats(dev);
		spin_unlock_irqrestore(&pp->lock, flags);
	}
	return &pp->stats;
}

static void
candy_set_rx_mode(struct net_device *dev)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;
	candy_regs *p = pp->regs;
	ulong val;

	DEBUG_VERBOSE(printk("candy_set_rx_mode() invoked.\n"));

	/* TODO: need to acquire spinlock and stop receiving */

	val = candy_in(p->afr);
	val &= ~PRO;
	if (dev->flags & IFF_PROMISC) {
		val |= PRO;
	} else if ((dev->flags & IFF_ALLMULTI) || (dev->mc_count > 64)) {
		/* disable promiscuous mode, use normal mode */
		candy_set_filter(dev, 0);
	} else if (dev->mc_count) {
		/* walk the address list, and load the filter */
		candy_set_filter(dev, 1);
	}
#ifdef	WORKAROUND_E10_PRM_AMC
	candy_out(p->afr, val & ABC);
#else
	candy_out(p->afr, val);
#endif
}

static void
candy_tx_timeout(struct net_device *dev)
{
	struct candy_private *pp = (struct candy_private *) dev->priv;

	printk(KERN_ERR "%s : tx_timeout.\n", dev->name);

	pp->stats.tx_errors++;

	candy_error_recover(dev);
}

static void __init
candy_init_one(uint irq, ulong base_addr, ulong pmd_addr, u_char mac_addr[])
{
	struct net_device *dev;
	candy_regs *p;
	struct candy_private *pp;
	int i;

	printk (KERN_INFO DRV_NAME ": Probe candy chip at "
		"irq=%d, base_addr=%08lx, pmd_addr=%08lx, "
		"mac_addr=%02x:%02x:%02x:%02x:%02x:%02x\n",
		irq, base_addr, pmd_addr, mac_addr[0], mac_addr[1], 
		mac_addr[2], mac_addr[3], mac_addr[4], mac_addr[5]);

	/* hardware is already initialized.  We just need do some Linux
	 * related initialization.
	 */

	/* create net_device structure */
	dev = init_etherdev(NULL, sizeof (struct candy_private));
	if (!dev) {
		printk(KERN_ERR "ether device alloc failed. aborting\n");
		goto bailout;
	}

	/* init some device related data/func ptrs */
	dev->base_addr = base_addr;
	dev->irq = irq;

	for (i = 0; i < 6; i++)
		dev->dev_addr[i] = mac_addr[i];

	dev->open = candy_open;
	dev->stop = candy_close;
	dev->do_ioctl = candy_ioctl;
	dev->hard_start_xmit = candy_xmit;
	dev->get_stats = candy_get_stats;
	dev->set_multicast_list = candy_set_rx_mode;
	dev->tx_timeout = candy_tx_timeout;
	dev->watchdog_timeo = TX_TIMEOUT;

	/* init private data */
	ASSERT(dev->priv != NULL);
	pp = (struct candy_private *) dev->priv;
	p = pp->regs = (candy_regs *) base_addr;

	/* alloc tx/rx rings and rx buffers */
	pp->tx_ring = kmalloc(sizeof (struct candy_desc) * (TX_RING_SIZE + 1),
			      GFP_ATOMIC);
	pp->rx_ring = kmalloc(sizeof (struct candy_desc) * (RX_RING_SIZE + 1),
			      GFP_ATOMIC);
	if (!pp->rx_ring || !pp->tx_ring)
		goto bailout;

	dma_cache_inv((ulong) pp->tx_ring,
		      sizeof (struct candy_desc) * (TX_RING_SIZE + 1));
	dma_cache_inv((ulong) pp->rx_ring,
		      sizeof (struct candy_desc) * (RX_RING_SIZE + 1));

	pp->tx_ring = (void *) KSEG1ADDR(pp->tx_ring);
	pp->rx_ring = (void *) KSEG1ADDR(pp->rx_ring);

	/* allocate rx skbs */
	for (i = 0; i < RX_RING_SIZE; i++) {
		pp->rx_skb[i] = dev_alloc_skb(RX_BUF_SIZE);
		if (pp->rx_skb[i] == NULL) {
			panic("%s: failed to alloc rx skb!", dev->name);
		}
		pp->rx_skb[i]->dev = dev;
		dma_cache_inv((ulong) pp->rx_skb[i]->data, RX_BUF_SIZE);
	}

	/* set up links */
	pp->dev = dev;
	pp->msg_enable = (debug < 0 ? CANDY_DEF_MSG_ENABLE : debug);
	pp->next = candy_priv_head;
	candy_priv_head = pp;
	pp->mii_if.dev = dev;
	pp->mii_if.mdio_read = mdio_read;
	pp->mii_if.mdio_write = mdio_write;
	pp->mii_if.phy_id = pmd_addr;
	pp->mii_if.phy_id_mask = 0x1f;
	pp->mii_if.reg_num_mask = 0x1f;

	/*==============================================================
	 * hardware initialization
	 *==============================================================
	 */

	/* TODO: maybe we want to make sure the chip is there */

	/*
	 * zero out counters
	 */
	candy_out(p->rbyt, 0);
	candy_out(p->rpkt, 0);
	candy_out(p->rfcs, 0);
	candy_out(p->rmca, 0);
	candy_out(p->rbca, 0);
	candy_out(p->rxcf, 0);
	candy_out(p->rxpf, 0);
	candy_out(p->rxuo, 0);
	candy_out(p->raln, 0);
	candy_out(p->rflr, 0);
	candy_out(p->rcde, 0);
	candy_out(p->rfcr, 0);
	candy_out(p->rund, 0);
	candy_out(p->rovr, 0);
	candy_out(p->rfrg, 0);
	candy_out(p->rjbr, 0);
	candy_out(p->r64, 0);
	candy_out(p->r127, 0);
	candy_out(p->r255, 0);
	candy_out(p->r511, 0);
	candy_out(p->r1k, 0);
	candy_out(p->rmax, 0);
	candy_out(p->rvbt, 0);

	candy_out(p->tbyt, 0);
	candy_out(p->tpct, 0);
	candy_out(p->tfcs, 0);
	candy_out(p->tmca, 0);
	candy_out(p->tbca, 0);
	candy_out(p->tuca, 0);
	candy_out(p->txpf, 0);
	candy_out(p->tdfr, 0);
	candy_out(p->txdf, 0);
	candy_out(p->tscl, 0);
	candy_out(p->tmcl, 0);
	candy_out(p->tlcl, 0);
	candy_out(p->txcl, 0);
	candy_out(p->tncl, 0);
	candy_out(p->tcse, 0);
	candy_out(p->time, 0);

	candy_hw_init(dev);

	return;

      bailout:
	if (dev) {
		pp = (struct candy_private *) dev->priv;
		if (pp) {
			kfree(pp->tx_ring);
			kfree(pp->rx_ring);
		}
		kfree(pp);
	}
	kfree(dev);
	return;
}

/***********************************************************************
 * Module hookup
 ***********************************************************************
 */

/* Board-specific code must provide nec_candy_get_params() routines.
 * It is expected that different boards will provide different parames
 */
extern int nec_candy_get_boot_params(uint * irq,
				     ulong * base_addr,
				     ulong * pmd_addr, u_char * mac_addr);

static int __init
nec_candy_module_init(void)
{
	int i;
	uint irq;
	ulong base_addr;
	ulong pmd_addr;
	u_char mac_addr[6];

	printk(KERN_INFO "%s\n", version);

	for (i = 0; i < MAX_NUM_DEVS; i++) {

		if (nec_candy_get_boot_params(&irq,
					      &base_addr,
					      &pmd_addr, mac_addr)) {
			break;
		}

		if (irq == 0)
			continue;

		candy_init_one(irq, base_addr, pmd_addr, mac_addr);
	}
	return 0;
}

static void __exit
nec_candy_module_exit(void)
{
	struct candy_private *pp;
	int i;

	while (candy_priv_head != NULL) {
		pp = candy_priv_head;
		candy_priv_head = pp->next;

		kfree((void *) KSEG0ADDR(pp->rx_ring));
		kfree((void *) KSEG0ADDR(pp->tx_ring));

		for (i = 0; i < RX_RING_SIZE; i++)
			dev_kfree_skb(pp->rx_skb[i]);

		unregister_netdev(pp->dev);

		kfree(pp->dev);
		kfree(pp);
	}
}

module_init(nec_candy_module_init);
module_exit(nec_candy_module_exit);

#if defined(MODULE)
MODULE_AUTHOR("Jun Sun, jsun@mvista.com or jsun@junsun.net");
MODULE_DESCRIPTION("Ether driver for NEC Candy controller");
MODULE_LICENSE("GPL");
#endif
