/*
 * linux/drivers/net/mb86977/if_camelot.c
 *
 * network interface to mb86977 camelot chip; presets 4 ethernet ports
 * 
 * Author: Brad Parker <brad@heeltoe.com>
 *
 * 2003 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/spinlock.h>
#include <linux/init.h>
#include <linux/mii.h>

#include <asm/uaccess.h>
#include <asm/arch/irqs.h>
#include <asm/arch/hardware.h>

#include "camelot_var.h"
#include "camelot_func.h"
#include "camelot_ioctl.h"

u_char came_link[4];
u_char came_mac[6];
int came_dmz_mode;
int came_intf_up;

extern u_char came_mac_default[6];
extern int camelot_misc_init(void);
extern void camelot_misc_exit(void);

#define MIN_PACKET 64
#define MAX_PACKET 1500

/* reawaken network queue this soon after stopping; else watchdog barks */
#define TX_TIMEOUT_JIFFIES	(5*HZ)

#define	mutex_lock(x)	down(x)
#define	mutex_unlock(x)	up(x)

static int cm_change_mtu(struct net_device *net, int new_mtu)
{
	if (new_mtu <= MIN_PACKET || new_mtu > MAX_PACKET)
		return -EINVAL;

	net->mtu = new_mtu;

	return 0;
}

static struct net_device_stats *cm_get_stats (struct net_device *net)
{
	return &((struct cm_softc *) net->priv)->stats;
}

extern int camelot_put(struct cm_softc *cm, struct sk_buff *skb);

static int cm_start_xmit (struct sk_buff *skb, struct net_device *net)
{
	struct cm_softc *cm = (struct cm_softc *)net->priv;
	return camelot_put(cm, skb);
}

extern void set_inten(struct camelot_softc *sc);
extern void reset_inten(struct camelot_softc *sc);

static int cm_open(struct net_device *net)
{
	struct cm_softc *cm = (struct cm_softc *)net->priv;
	struct camelot_softc *sc = &cm->camelot;
	int retval = 0;

	mutex_lock (&cm->mutex);

	netif_start_queue (net);

	/* if first interface to come up, turn on interrupts */
	if (came_intf_up == 0) {
		set_inten(sc);
	}

	came_intf_up |= 1 << cm->unit;

	mutex_unlock (&cm->mutex);

	return retval;
}

static int cm_stop(struct net_device *net)
{
	struct cm_softc *cm = (struct cm_softc *)net->priv;
	struct camelot_softc *sc = &cm->camelot;

	mutex_lock(&cm->mutex);

	came_intf_up &= ~(1 << cm->unit);
	/* if last interface to go down, turn off interrupts */
	if (came_intf_up == 0) {
		reset_inten(sc);
	}

	netif_stop_queue(net);

	mutex_unlock(&cm->mutex);

	return 0;
}

static void cm_tx_timeout (struct net_device *net)
{
	struct cm_softc *cm = (struct cm_softc *)net->priv;
	struct camelot_softc *sc = &cm->camelot;

	printk(KERN_ERR "camelot: watchdog, cm%d\n", cm->unit);

	/* device recovery -- reset? */
	camelot_reset(sc);
}

/* provide ioctl() calls to examine the MII state */
static int cm_private_ioctl (struct net_device *dev, struct ifreq *rq, int cmd)
{
	struct cm_softc *cm = (struct cm_softc *)dev->priv;
	struct camelot_softc *sc = &cm->camelot;
	struct mii_ioctl_data *data = (struct mii_ioctl_data *) & rq->ifr_data;
	unsigned int regnum = data->reg_num;

	switch (cmd) {
	case SIOCGMIIPHY:		/* Get address of MII PHY in use. */
		/* map out unit # to the physical phy */
		switch (cm->unit) {
		case 0: /* wan 0 */
			data->phy_id = 0;
			break;
		case 1: /* cm0 */
			data->phy_id = 3;
			break;
		case 2: /* cm1 */
			data->phy_id = 2;
			break;
		case 3: /* cm2 */
			data->phy_id = 1;
			break;
		}

		/* fall through */

	case SIOCGMIIREG:		/* Read MII PHY register. */
		data->val_out = read_phy(sc, data->phy_id, regnum);
		return 0;

	case SIOCSMIIREG:		/* Write MII PHY register. */
		if (!capable (CAP_NET_ADMIN))
			return -EPERM;

		if (regnum & ~0x1f)
			return -EINVAL;

		set_phy(sc, data->phy_id & 0x1f, regnum, data->val_in);

		return 0;
	default:
		return -EOPNOTSUPP;
	}

	return -EOPNOTSUPP;
}

static int cm_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
	switch (cmd) {
	case CM_IPFNAT:
	case CM_FLT:
	case CM_DEL_UNMATCH:
	case CM_PPPOE:
	case CM_TUNNEL:
	case CM_FLUSH:
	case CM_MATCH:
	case CM_DEBUG_READ:
	case CM_SET_DMZ:
	case CM_DAT:
	case CM_QOS:
	case CM_L2:
		return camelot_ioctl(dev, rq, cmd);
	default:
		return cm_private_ioctl(dev, rq, cmd);
	}
}

/* ------------------ */

static u_int
cm_read(struct camelot_softc *sc, u_int port)
{
	unsigned long val;

        val = *(volatile unsigned long *)(sc->sc_addr + port);

	return val;
}

static void
cm_write(struct camelot_softc *sc, u_int port, u_int val)
{
	*(volatile unsigned long *)(sc->sc_addr + port) = val;
}

void camelot_intr(int irq, void *dev_id, struct pt_regs *regs);

struct cm_softc *cm_units[3];

struct cm_softc *cm_unit(int index)
{
	return cm_units[index];
}

static int __init
if_camelot_init(void)
{
	int i, err = 0;
	struct camelot_softc *sc;
	struct cm_softc *cm;
	struct net_device *net;
	
	memcpy(came_mac, came_mac_default, 6);

	/*
	 * configure 3 "cm" ethernet devices
	 *
	 * code in camelot.c maps:
	 *
	 * unit intf port w/dmz	port w/o dmz
	 * ---- ---- ----------	------------
	 * 0    wan0 wan        wan
	 * 1    cm0  lan0/lan1  lan0/lan1/dmz
	 * 2    cm1  dmz	
	 *
	 */
	for (i = 0; i < 3; i++) {
		came_link[i] = 0;
		/* set up network interface records */
		cm = (struct cm_softc *)kmalloc(sizeof(*cm), GFP_KERNEL);
		if (cm == 0)
			break;
		memset(cm, 0, sizeof *cm);

		cm_units[i] = cm;

		init_MUTEX_LOCKED(&cm->mutex);

		cm->unit = i;

		sc = &cm->camelot;
		sc->sc_read = cm_read;
		sc->sc_write = cm_write;
		sc->sc_copytobuf = camelot_copytobuf_contig;
		sc->sc_copyfrombuf = camelot_copyfrombuf_contig;

		sc->sc_addr = VA_CAMELOT_BASE;

		net = &cm->net;
		net->priv = cm;
		switch (i) {
			/* +++ these strcpy()s aren't correct */
		case 0:
			strcpy(net->name, "wan%d");
			break;
		default:
			strcpy(net->name, "cm%d");
			break;
		}
		memcpy(net->dev_addr, came_mac, sizeof(came_mac));

		ether_setup (net);

		net->change_mtu = cm_change_mtu;
		net->get_stats = cm_get_stats;
		net->hard_start_xmit = cm_start_xmit;
		net->open = cm_open;
		net->stop = cm_stop;
		net->watchdog_timeo = TX_TIMEOUT_JIFFIES;
		net->tx_timeout = cm_tx_timeout;
		net->do_ioctl = cm_ioctl;

		register_netdev (&cm->net);

		mutex_unlock (&cm->mutex);

		/* start as if the link is up */
		netif_device_attach (&cm->net);

		if (i == 0) {
			if ((err = request_irq(IRQ_EINT0, &camelot_intr, 0,
						  "camelot", cm)))
				return err;

			camelot_config(&cm->camelot);
		}
	}

	came_intf_up = 0;

	if ((err = camelot_misc_init())) {
		printk(KERN_ERR "Could not initialize camelot_misc driver\n");
	}

	return err;
}

static void __exit
if_camelot_cleanup(void)
{
	camelot_misc_exit();
}

module_init(if_camelot_init);
module_exit(if_camelot_cleanup);
