/*
 * linux/drivers/net/fujitsu-camelot/camelot_misc.c
 *
 * Fujitsu MB86977 driver
 *
 * Author: <source@mvista.com>
 *
 * Copyright (c) 2002-2003 by Fujitsu LSI Solution Ltd..  All Rights Reserved.
 *
 * 2003 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 *
 */

#include <linux/config.h>
#include <linux/module.h>

#include <linux/spinlock.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/tty.h>
#include <linux/mm.h>
#include <linux/signal.h>
#include <linux/init.h>
#include <linux/kbd_ll.h>
#include <linux/delay.h>
#include <linux/random.h>
#include <linux/poll.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/kbd_kern.h>
#include <linux/vt_kern.h>
#include <linux/smp_lock.h>
#include <linux/kd.h>
#include <linux/pm.h>

/* uncomment the following define to get debug output */
//#define CAMELOT_MISC_DEBUG 1
#ifdef CAMELOT_MISC_DEBUG
static unsigned int debug = 1;
#else
#define debug 0
#endif

/* spinlock for protecting the event queue */
static spinlock_t camelot_controller_lock = SPIN_LOCK_UNLOCKED;

/* +++what is a reasonable number for this? */
#define CAMELOT_RSTFIN_BUF_SIZE 256

struct camelot_rstfin_event {
	int event_id;   /* in case we want to keep track of events by id */
	unsigned long ip_src;    /* source IP */
	unsigned long ip_dst;    /* destination IP */
	unsigned long ip_nat;    /* NAT'd IP */
	unsigned short port_src; /* source port */
	unsigned short port_dst; /* destination port */
	unsigned short port_nat; /* NAT'd port */
};

struct camelot_rstfin_queue {
	int head;
	int tail;
	wait_queue_head_t proc_list;
	struct fasync_struct *fasync;
	/* buffer of RST/FIN event structures */
	struct camelot_rstfin_event buf[CAMELOT_RSTFIN_BUF_SIZE];
};

/* static variables used in this driver */
static struct camelot_rstfin_queue *queue;	/* RST/FIN data buffer. */
static int camelot_misc_count;  /* ref count for this misc driver */

/*
 * Put a new entry into the queue.
 * +++: does this *need* to be inlined?
 */
static inline void
put_to_queue(unsigned long ip_src,
	     unsigned long ip_dst,
	     unsigned long ip_nat,
	     unsigned short port_src,
	     unsigned short port_dst,
	     unsigned short port_nat)
{
	struct camelot_rstfin_event *event_ptr;
	unsigned long flags;
	
	if (debug)
                pr_debug("%s()\n", __FUNCTION__);
	
	if (camelot_misc_count) {
		spin_lock_irqsave(&camelot_controller_lock, flags);
		int head = queue->head;

		/* fill in the next available event */
		event_ptr = &queue->buf[head];
		event_ptr->ip_src = ip_src;
		event_ptr->ip_dst = ip_dst;
		event_ptr->ip_nat = ip_nat;
		event_ptr->port_src = port_src;
		event_ptr->port_dst = port_dst;
		event_ptr->port_nat = port_nat;

		head = (head + 1) & (CAMELOT_RSTFIN_BUF_SIZE-1);
		if (head != queue->tail) {
			queue->head = head;
			kill_fasync(&queue->fasync, SIGIO, POLL_IN);
			wake_up_interruptible(&queue->proc_list);
		}
		spin_unlock_irqrestore(&camelot_controller_lock, flags);
	}
}

/* Get the next entry from the queue. */
static struct camelot_rstfin_event *
get_from_queue(void)
{
	struct camelot_rstfin_event *event;
	unsigned long flags;

	if (debug)
	        pr_debug("%s\n", __FUNCTION__);
	
	spin_lock_irqsave(&camelot_controller_lock, flags);
	event = &queue->buf[queue->tail];
	queue->tail = (queue->tail + 1) & (CAMELOT_RSTFIN_BUF_SIZE-1);
	spin_unlock_irqrestore(&camelot_controller_lock, flags);
	return event;
}

/* Is the queue empty? */
static inline int
queue_empty(void)
{
	return queue->head == queue->tail;
}

/*
 * Callback function for processing a RST/FIN.  Places the event data
 * into the read queue.
 */
int
camelot_misc_callback_func(unsigned long ip_src,
			   unsigned long ip_dst,
			   unsigned long ip_nat,
			   unsigned short port_src,
			   unsigned short port_dst,
			   unsigned short port_nat)
{
	if (debug)
	        pr_debug("%s()\n", __FUNCTION__);

	put_to_queue(ip_src, ip_dst, ip_nat, port_src, port_dst, port_nat);

	return 0;
}

static int
fasync_camelot_misc(int fd, struct file *filp, int on)
{
	int retval;

	retval = fasync_helper(fd, filp, on, &queue->fasync);
	if (retval < 0)
		return retval;
	return 0;
}

static int
release_camelot_misc(struct inode * inode, struct file * file)
{
	lock_kernel();
	fasync_camelot_misc(-1, file, 0);
	
	if (--camelot_misc_count) {
		/* reset the ring */
		queue->head = queue->tail = 0;
		unlock_kernel();
		return 0;
	}
	
	unlock_kernel();
	/*
	 * +++similar to open, should this return an error if refcount
	 * is not zero?
	 */
	return 0;
}

/*
 * Open device.  First time device is opened, initialize the queue.
 */
static int
open_camelot_misc(struct inode * inode, struct file * file)
{
	if (debug)
	        pr_debug("%s\n", __FUNCTION__);
	if (camelot_misc_count++) {
		/*
		 * +++perhaps have an error; could return -EPERM or
		 * -EACCESS?
		 */
		return 0;
	}
	queue->head = queue->tail = 0;		/* Flush input queue */
	
	return 0;
}

/*
 * Put events from input queue to buffer.  Copy as many events as
 * allowed by the requested count into the provided buffer.  Can
 * return less than count bytes if some data is available.  Returns an
 * error if count is not a multiple of the event structure.
 */

static ssize_t
read_camelot_misc(struct file * file, char * buffer,
		  size_t count, loff_t *ppos)
{
	DECLARE_WAITQUEUE(wait, current);
	ssize_t i = count;
	struct camelot_rstfin_event *event;

	if (debug)
	        pr_debug("%s\n", __FUNCTION__);
	
	if (count % sizeof(struct camelot_rstfin_event)) {
		printk(KERN_ERR
		       "%s: Size of buffer must be multiple of %u bytes\n",
		       __FUNCTION__, sizeof(struct camelot_rstfin_event));
		return -EPERM;
	}

	if (queue_empty()) {
		if (debug)
	        	pr_debug("%s: Queue is empty\n", __FUNCTION__);
		if (file->f_flags & O_NONBLOCK)
			return -EAGAIN;
		add_wait_queue(&queue->proc_list, &wait);
repeat:
		set_current_state(TASK_INTERRUPTIBLE);
		if (queue_empty() && !signal_pending(current)) {
			schedule();
			goto repeat;
		}
		current->state = TASK_RUNNING;
		remove_wait_queue(&queue->proc_list, &wait);
	}
	while (i > 0 && !queue_empty()) {
		if (debug)
	        	pr_debug("%s: Retrieving event from queue\n",
			       __FUNCTION__);
		event = get_from_queue();

		if (copy_to_user(buffer, event,
				 sizeof(struct camelot_rstfin_event))) {
			printk(KERN_ERR "%s: copy_to_user error\n",
			       __FUNCTION__);
			return -EINVAL;
		}
		
		i -= sizeof(struct camelot_rstfin_event);
	}
	if (count-i) {
		file->f_dentry->d_inode->i_atime = CURRENT_TIME;
		return count-i;
	}
	if (signal_pending(current))
		return -ERESTARTSYS;
	return 0;
}

/* No kernel lock held - fine */
static unsigned int
camelot_misc_poll(struct file *file, poll_table * wait)
{
	poll_wait(file, &queue->proc_list, wait);
	if (!queue_empty())
		return POLLIN | POLLRDNORM;
	return 0;
}

struct file_operations camelot_misc_fops = {
	read:		read_camelot_misc,
	poll:		camelot_misc_poll,
	open:		open_camelot_misc,
	release:	release_camelot_misc,
	fasync:		fasync_camelot_misc,
};

/*
 * +++this MISC_MINOR is unused, by inspection of
 * include/linux/miscdevice.h, but should be moved there
 */
#define CAMELOT_MISC_MINOR 230
static struct miscdevice camelot_misc = {
	CAMELOT_MISC_MINOR, "camelot_misc", &camelot_misc_fops
};

extern int camelot_register_rstfin_callback(int (*func)(unsigned long,
							unsigned long,
							unsigned long,
							unsigned short,
							unsigned short,
							unsigned short));
extern int camelot_unregister_rstfin_callback(int (*func)(unsigned long,
							  unsigned long,
							  unsigned long,
							  unsigned short,
							  unsigned short,
							  unsigned short));

/*
 * Initialize driver. Create queue and register a RST/FIN callback
 * function for handling these events.
 */
int
camelot_misc_init(void)
{
	int retval;

	if (debug)
	        pr_debug("%s()\n", __FUNCTION__);

	if ((retval = misc_register(&camelot_misc))) {
		printk(KERN_ERR "Could not register camelot_misc\n");
		return retval;
	}

	queue = (struct camelot_rstfin_queue *) kmalloc(sizeof(*queue),
							GFP_KERNEL);
	if (queue == NULL) {
		printk(KERN_ERR "%s(): out of memory\n", __FUNCTION__);
		misc_deregister(&camelot_misc);
		return -ENOMEM;
	}
	memset(queue, 0, sizeof(*queue));
	queue->head = queue->tail = 0;
	init_waitqueue_head(&queue->proc_list);

	if (camelot_register_rstfin_callback(camelot_misc_callback_func)) {
		printk(KERN_ERR "%s: cannot register rstfin callback\n",
		       __FUNCTION__);
		return -ENODEV;
	}

	camelot_misc_count = 0;
	
	return 0;
}

/*
 * Prepare for exit. Cleanup queue and unregister the RST/FIN callback
 * function.
 */
void
camelot_misc_exit(void)
{
	if (debug)
	        pr_debug("%s()\n", __FUNCTION__);

	camelot_unregister_rstfin_callback(camelot_misc_callback_func);
	kfree(queue);
	/*
	 * +++these seem like good things to do; but are they
	 * necessary?
	 */
	queue = NULL;
	camelot_misc_count = 0;
	camelot_controller_lock = SPIN_LOCK_UNLOCKED;
}

#ifdef MODULE
static int __init camelot_misc_module_init(void)
{
	return (camelot_misc_init());
}

static void __exit camelot_misc_module_exit(void)
{
	camelot_misc_exit();
}

module_init(camelot_misc_init);
module_exit(camelot_misc_exit);
#endif
