/*
 * linux/drivers/net/fujitsu-camelot/camelot_ipfnat.c
 *
 * Fujitsu MB86977 IPF/NAT entry module
 *
 * Author: <source@mvista.com>
 *
 * Copyright (c) 2004 by Fujitsu LSI Solution Ltd..  All Rights Reserved.
 *
 * 2004 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 *
 */

#include <linux/config.h>
#include <linux/module.h>

#include <linux/spinlock.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/tty.h>
#include <linux/mm.h>
#include <linux/signal.h>
#include <linux/init.h>
#include <linux/kbd_ll.h>
#include <linux/delay.h>
#include <linux/random.h>
#include <linux/poll.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/kbd_kern.h>
#include <linux/vt_kern.h>
#include <linux/smp_lock.h>
#include <linux/kd.h>
#include <linux/pm.h>
#include <linux/netfilter_ipv4/ip_conntrack.h>
#include <linux/netfilter_ipv4/ip_conntrack_core.h>
#include <linux/netfilter_ipv4/ip_conntrack_protocol.h>
#include <linux/netdevice.h>
#include <net/neighbour.h>

#include <linux/netdevice.h>
#include <linux/inetdevice.h>
#include <net/route.h>

#include <linux/interrupt.h>
#include <linux/timer.h>

#include "camelot_var.h"
#include "camelot_ipfnat.h"
#include "camelot_bits.h"
#include "camelot_func.h"

extern int camelot_get_ipfnat_mode(void);
extern void camelot_set_ipfnat_mode(int);
extern struct cm_softc *cm_unit(int index);
extern struct neigh_table arp_tbl;

struct came_pppoe_session {
	unsigned long ip;
	struct neighbour *n;
	/* these two may not be used */
	unsigned long mask;
	unsigned long server_ip;
};
static struct came_pppoe_session pppoe_sessions[LEN_PPPOE];
static int num_pppoe_neighs = 0;
static struct net_device dummy_net_device[LEN_PPPOE];

#ifdef NEED_FIX
/* fix for chip errata */
#define	NATIPF_TABLE_SIZE	(ENTRY_NUM_NATIPF/2)
extern int map_natipf_index(int);
#else
/* no fix */
#define	NATIPF_TABLE_SIZE	(ENTRY_NUM_NATIPF)
#define map_natipf_index(index)		(index)
#endif

/* Change undef to define below to get debug output */
#undef CAMELOT_IPFNAT_DEBUG
#ifdef CAMELOT_IPFNAT_DEBUG
static unsigned int debug = 1;
#else
#define debug 0
#endif
/* Change undef to define below to get lower level debug output */
#undef CAMELOT_IPFNAT_DEBUG_LOW
#ifdef CAMELOT_IPFNAT_DEBUG_LOW
static unsigned int debug_low = 1;
#else
#define debug_low 0
#endif
/* Change undef to define below to get function trace */
#undef CAMELOT_IPFNAT_TRACE
#ifdef CAMELOT_IPFNAT_TRACE
static unsigned int trace = 1;
#else
#define trace 0
#endif
/* Change undef to define below to get ct queueing debug messages */
#undef CAMELOT_IPFNAT_DEBUG_CT
#ifdef CAMELOT_IPFNAT_DEBUG_CT
static unsigned int debug_ct = 1;
#else
#define debug_ct 0
#endif

/*
 * Static variables
 */
/* how frequently to poll chip match register */
static time_t poll_period = 20;
/* how long conntrack is inactive before timeout */
static time_t expire_period = 60;
static unsigned int num_periods_till_expiration = 3;
static unsigned long wan_ip;
static unsigned long wan_mask;
static unsigned long cm_ip[3];
static unsigned long cm_mask[3];

/* for queueing conntrack events for later processing */
struct ipfnat_list_entry {
	struct list_head list;
	const struct ip_conntrack *ct;
	int add;
};
struct list_head ipfnat_ct_free_list;
struct list_head ipfnat_ct_queued_list;

/*
 * Match Register Polling Helper Functions
 */
int
cm_set_poll_period(time_t period)
{
	if (period <= 0) {
		printk(KERN_ERR "camelot: ");
		printk(KERN_ERR "Invalid poll period specified (%ld)\n",
		       period);
		printk(KERN_ERR "camelot: Must be greater than zero.\n");
		return -1;
	}
	if (period != poll_period) {
		/* need to reset timer expiration? */
	}
	poll_period = period;
	return 0;
}

time_t
cm_get_poll_period(void)
{
	return poll_period;
}

int
cm_set_expire_period(time_t period)
{
	if (period <= 0) {
		printk(KERN_ERR "camelot: ");
		printk(KERN_ERR "Invalid expire period specified (%ld)\n",
		       period);
		printk(KERN_ERR "camelot: Must be greater than zero\n");
		return -1;
	}
	expire_period = period;
	return 0;
}

time_t
cm_get_expire_period(void)
{
	return expire_period;
}

int
cm_calculate_num_expiration_periods(void)
{
	/*
	 * could never timeout conntracks due to inactivity by
	 * allowing a large ratio of expire_period to
	 * poll_period--could set num_periods_till_expiration to 0 and
	 * use this to mean "never timeout due to inactivity"
	 */
	num_periods_till_expiration = (expire_period / poll_period) + 1;
	if (num_periods_till_expiration > 255) {
		printk(KERN_ERR "camelot: ");
		printk(KERN_ERR "Expiration periods too high (%d)\n",
		       num_periods_till_expiration);
		printk(KERN_ERR "camelot: The ratio of expire period to ");
		printk(KERN_ERR "poll period must be less than 255\n");
		return -1;
	}
	if (debug) pr_debug("camelot: Poll periods till expiration set to %d\n",
			  num_periods_till_expiration);
	return 0;
}

/*
 * QoS Helper Functions
 */

/*
 * table of QoS connections (necessary to specify the proper chip
 * index when adding a chip IPF/NAT entry)
 */
static QOS_TBL qos_table;

void
cm_qos_table_dump(void)
{
    struct qos_entry_s *tmp;
    int i;

    printk("Internal QOS Table:\n");
    for (i = 0; i < qos_table.num_entries; i++) {
        tmp = &qos_table.qos_entry[i];
        printk(" %d: %s %u.%u.%u.%u:%04x -> %u.%u.%u.%u:%04x %u %lu "
	       "(%d refs)\n",
	       tmp->id, tmp->ipv6 ? "IPv6" : "IPv4",
	       NIPQUAD(tmp->wan_ip[0]), htons(tmp->wan_port),
	       NIPQUAD(tmp->dmz_ip[0]), htons(tmp->dmz_port),
	       tmp->tos, tmp->flowlabel, tmp->refcount);
    }
}

static int
cm_qos_entry_match(unsigned long ip1, unsigned long ip2,
		   unsigned long p1, unsigned long p2)
{
    int i, cur_idx, match_id = 0;
    struct qos_entry_s *tmp;

    if (trace) printk("cm_qos_entry_match()\n");
    
    cur_idx = qos_table.num_entries;

    if (debug) pr_debug("Entry to check: 0x%08lx:0x%04lx 0x%08lx:0x%04lx\n",
		      ip1, p1, ip2, p2);

    if (cur_idx == 0) {
        if (debug) pr_debug("camelotd QOS table is empty\n");
        return 0;
    }
    
    /* check all entries for a match */
    for (i = 0; i < cur_idx; i++) {
        tmp = &qos_table.qos_entry[i];
        if (debug ) {
		pr_debug("Table Entry %d: 0x%08lx:0x%04x 0x%08lx:0x%04x\n",
		       i+1,
		       tmp->wan_ip[0],
		       tmp->wan_port,
		       tmp->dmz_ip[0],
		       tmp->dmz_port);
	}

        /* check only non-zero entries */
        
        /* check ip addresses */
        if (tmp->wan_ip[0] && tmp->dmz_ip[0] &&
            (tmp->wan_ip[0] != ip1 || tmp->dmz_ip[0] != ip2) &&
            (tmp->wan_ip[0] != ip2 || tmp->dmz_ip[0] != ip1)) {
            /* no match on both non-zero WAN and DMZ ip address
               entries */
            if (debug) pr_debug("Both non-zero IP addresses mismatch\n");
            continue;
        }
        else if (tmp->wan_ip[0] &&
                 (tmp->wan_ip[0] != ip1 && tmp->wan_ip[0] != ip2)) {
            /* WAN address doesn't match */
            if (debug) pr_debug("Non-zero WAN IP address mismatch\n");
            continue;
        }
        else if (tmp->dmz_ip[0] &&
                 (tmp->dmz_ip[0] != ip1 && tmp->dmz_ip[0] != ip2)) {
            /* DMZ address doesn't match */
            if (debug) pr_debug("Non-zero DMZ IP address mismatch\n");
            continue;
        }
        /* IP addresses match */

        /* check ports */
        if (tmp->wan_port && tmp->dmz_port &&
            (tmp->wan_port != p1 || tmp->dmz_port != p2) &&
            (tmp->wan_port != p2 || tmp->dmz_port != p1)) {
            /* no match on both non-zero WAN and DMZ port entries */
            if (debug) pr_debug("Both non-zero ports mismatch\n");
            continue;
        }
        else if (tmp->wan_port &&
                 (tmp->wan_port != p1 && tmp->wan_port != p2)) {
            /* WAN port doesn't match */
            if (debug) pr_debug("Non-zero WAN port mismatch\n");
            continue;
        }
        else if (tmp->dmz_port &&
                 (tmp->dmz_port != p1 && tmp->dmz_port != p2)) {
            /* DMZ port doesn't match */
            if (debug) pr_debug("Non-zero DMZ port mismatch\n");
            continue;
        }
        /* ports match */
        match_id = tmp->id;
        break;
    }

    if (match_id == 0) {
        if (debug) pr_debug("No match found in QOS table\n");
        return 0;
    }

    if (debug) pr_debug("QOS Table Entry %d matches\n", match_id);
    
    return match_id;
}

static int
cm_qos_entry_addref(int entry_id)
{
    if (trace) printk("cm_qos_entry_addref()\n");
    
    if (entry_id == 0) {
        /* this should not happen */
        if (debug) pr_debug("cm_qos_entry_addref() cannot locate QOS entry\n");
        return -1;
    }

    qos_table.qos_entry[entry_id-1].refcount++;

    if (debug) pr_debug("cm_qos_entry_addref: QOS entry id %d (%d)\n",
		      entry_id, qos_table.qos_entry[entry_id-1].refcount);
    return qos_table.qos_entry[entry_id-1].refcount;
}

static int
cm_qos_entry_delref(int entry_id)
{
    if (trace) printk("cm_qos_entry_delref()\n");
    
    if (entry_id == 0) {
        /* this should not happen */
        if (debug) pr_debug("cm_qos_entry_delref() cannot locate QOS entry\n");
        return -1;
    }

    qos_table.qos_entry[entry_id-1].refcount--;
    if (qos_table.qos_entry[entry_id-1].refcount < 0) {
        if (debug) pr_debug("QOS entry id %d: refcount < 0\n", entry_id);
        return 0;
    }

    if (debug) pr_debug("cm_qos_entry_delref: QOS entry id %d (%d)\n",
		      entry_id, qos_table.qos_entry[entry_id-1].refcount);
    return qos_table.qos_entry[entry_id-1].refcount;
}

int
cm_set_qos_table(void *data)
{
	if (copy_from_user((char *)&qos_table, data, sizeof(QOS_TBL)))
		return -EINVAL;
	if (debug) {
		pr_debug("cm_set_qos_table(): num_entries %d\n",
		       qos_table.num_entries);
		cm_qos_table_dump();
	}
	return 0;
}

static void
qos_entry_populate_fromid(int entry_id, QOS_TBL_ENTRY *qos, int wan_port,
                          unsigned long ipaddr_nat[], unsigned long port_nat)
{
    struct qos_entry_s *tmp;
    
    if (trace) printk("qos_entry_populate_fromid(id=%d)\n", entry_id);
    
    tmp = &qos_table.qos_entry[entry_id-1];

    if (wan_port) {
        /* WAN is source, DMZ is dest */
        if (ipaddr_nat[0] != tmp->dmz_ip[0]) {
            if (debug) pr_debug("Masquerading in use\n");
            qos->ipv6addr_dst[0] = ntohl(ipaddr_nat[0]);
            /* only specify nat port as destination port if port was
               specified in original QoS table entry */
            if (tmp->dmz_port)
                qos->portnum_dst = ntohs(port_nat);
        }
        else {
            if (debug) pr_debug("Masquerading is NOT in use\n");
            qos->ipv6addr_dst[0] = ntohl(tmp->dmz_ip[0]);
            qos->portnum_dst = ntohs(tmp->dmz_port);
        }
        qos->ipv6addr_src[0] = ntohl(tmp->wan_ip[0]);
        qos->portnum_src = ntohs(tmp->wan_port);
    }
    else {
        /* DMZ is source, WAN is dest */
        qos->ipv6addr_src[0] = ntohl(tmp->dmz_ip[0]);
        qos->ipv6addr_dst[0] = ntohl(tmp->wan_ip[0]);
        qos->portnum_src = ntohs(tmp->dmz_port);
        qos->portnum_dst = ntohs(tmp->wan_port);
    }

    qos->tos = ntohs(tmp->tos);
    qos->flowlabel = ntohl(tmp->flowlabel);
    qos->wan_port = wan_port;
    
    /* set control appropriately */
    qos->control = (((qos->portnum_dst == 0) & 0x1) << 7) |
                   (((qos->portnum_src == 0) & 0x1) << 6) |
                   (((qos->ipv6addr_dst[0] == 0) & 0x1) << 5) |
                   (((qos->ipv6addr_src[0] == 0) & 0x1) << 4) |
                   ((((qos->flowlabel == 0) && tmp->ipv6) & 0x1) << 3) |
                   (((qos->tos == 0) & 0x1) << 2) |
                   ((tmp->ipv6 & 0x1) << 1) |
                   (0x1 << 0);
    
    if (debug) {
	    pr_debug("qos_entry_populate: control: 0x%08x\n"
		   "sa %u.%u.%u.%u (0x%08x) da %u.%u.%u.%u (0x%08x)\n"
		   "sp 0x%04x dp 0x%04x port %s tos 0x%04x\n",
		   qos->control,
		   HIPQUAD(qos->ipv6addr_src[0]), qos->ipv6addr_src[0],
		   HIPQUAD(qos->ipv6addr_dst[0]), qos->ipv6addr_dst[0],
		   qos->portnum_src, qos->portnum_dst,
		   qos->wan_port ? "WAN" : "DMZ", qos->tos);
    }
}

extern int cm_ioctl_add_qos_entry(struct camelot_softc *, QOS_TBL_ENTRY *);

/*
 * add two corresponding Quality of Service entries, one for WAN
 * table, one for DMZ table (only IPv4 handled at this time)
 */
static int
cm_add_qos_entry(struct camelot_softc *sc, NAT_IPF_TBL *qos_nit)
{
    unsigned long ipaddr_src[4], ipaddr_dst[4], ipaddr_nat[4];
    unsigned long port_src, port_dst, port_nat;
    int ret, qos_entry_id, refcount;
    QOS_TBL_ENTRY qos_dmz, qos_wan;

    ipaddr_src[0] = htonl(qos_nit->ipv4addr_inter);
    ipaddr_dst[0] = htonl(qos_nit->ipv4addr_exter);
    ipaddr_nat[0] = htonl(qos_nit->nataddr);
    port_src = htons(qos_nit->portnum_inter);;
    port_dst = htons(qos_nit->portnum_exter);;
    port_nat = htons(qos_nit->portnum_nat);;

    /* ipv6 not supported; tos always 0 */
    if (debug) {
	    pr_debug("cm__add_qos_entry: "
		   "sa %u.%u.%u.%u (0x%04lx) da %u.%u.%u.%u (0x%04lx) "
		   "na %u.%u.%u.%u (0x%04lx)\n"
		   "sp 0x%04lx dp 0x%04lx np 0x%04lx tos 0x%04x\n",
		   NIPQUAD(ipaddr_src[0]), ipaddr_src[0],
		   NIPQUAD(ipaddr_dst[0]), ipaddr_dst[0],
		   NIPQUAD(ipaddr_nat[0]), ipaddr_nat[0],
		   port_src, port_dst, port_nat, 0);
    }

    /* don't add duplicate entries to table */
    qos_entry_id = cm_qos_entry_match(ipaddr_src[0], ipaddr_dst[0],
                                   port_src, port_dst);
    if (qos_entry_id < 0) {
        if (debug) pr_debug("cm_add_qos_entry: Can't find matching QOS entry\n");
        return -1;
    }
    refcount = cm_qos_entry_addref(qos_entry_id);
    if (refcount > 1) {
        if (debug) pr_debug("cm_add_qos_entry: entry already added to chip\n");
        return 0;
    }
    
    memset((char *)&qos_dmz, 0, sizeof(QOS_TBL_ENTRY));
    memset((char *)&qos_wan, 0, sizeof(QOS_TBL_ENTRY));

    /* only ipv4 for now */
    qos_entry_populate_fromid(qos_entry_id, &qos_dmz, 0, ipaddr_nat, port_nat);
    qos_entry_populate_fromid(qos_entry_id, &qos_wan, 1, ipaddr_nat, port_nat);
    
    ret = cm_ioctl_add_qos_entry(sc, &qos_wan);
    if (ret < 0) {
        printk(KERN_ERR "cm_add_qos_entry: CM_SET_QOS failed (wan)\n");
    }

    ret = cm_ioctl_add_qos_entry(sc, &qos_dmz);
    if (ret < 0) {
        printk(KERN_ERR "cm_add_qos_entry: CM_SET_QOS failed (dmz)\n");
    }

    return 0;
}

extern int cm_ioctl_del_qos_entry(struct camelot_softc *, QOS_TBL_ENTRY *);

/*
 * delete two corresponding Quality of Service entries, one for WAN
 * table, one for DMZ table (only IPv4 handled at this time)
 */
int
cm_del_qos_entry(struct camelot_softc *sc, NAT_IPF_TBL *qos_nit)
{
    unsigned long ipaddr_src[4], ipaddr_dst[4], ipaddr_nat[4];
    unsigned long port_src, port_dst, port_nat;
    int ret, qos_entry_id, refcount;
    QOS_TBL_ENTRY qos_wan, qos_dmz;
    
    if (trace) printk("cm_del_qos_entry()\n");

    ipaddr_src[0] = htonl(qos_nit->ipv4addr_inter);
    ipaddr_dst[0] = htonl(qos_nit->ipv4addr_exter);
    ipaddr_nat[0] = htonl(qos_nit->nataddr);
    port_src = htons(qos_nit->portnum_inter);;
    port_dst = htons(qos_nit->portnum_exter);;
    port_nat = htons(qos_nit->portnum_nat);;
    
    /* ipv6 not supported; tos always 0 */
    if (debug) {
	    pr_debug("chip_del_qos: "
		   "sa %u.%u.%u.%u (0x%04lx) da %u.%u.%u.%u (0x%04lx)\n"
		   "sp 0x%04lx dp 0x%04lx tos 0x%04x\n",
		   NIPQUAD(ipaddr_src[0]), ipaddr_src[0],
		   NIPQUAD(ipaddr_dst[0]), ipaddr_dst[0],
		   port_src, port_dst, 0);
    }

    /* delete only if refcount < 1 */
    qos_entry_id = cm_qos_entry_match(ipaddr_src[0], ipaddr_dst[0],
				      port_src, port_dst);
    if (qos_entry_id < 0) {
        if (debug) pr_debug("cm_del_qos_entry: Can't find matching QOS entry\n");
        return -1;
    }
    refcount = cm_qos_entry_delref(qos_entry_id);
    if (refcount > 0) {
        if (debug) pr_debug("cm_del_qos_entry(): entry %d still in use\n",
			  qos_entry_id);
        return 0;
    }
    
    memset((char *)&qos_wan, 0, sizeof(QOS_TBL_ENTRY));
    memset((char *)&qos_dmz, 0, sizeof(QOS_TBL_ENTRY));

    /* only ipv4 for now */
    qos_entry_populate_fromid(qos_entry_id, &qos_dmz, 0, ipaddr_nat, port_nat);
    qos_entry_populate_fromid(qos_entry_id, &qos_wan, 1, ipaddr_nat, port_nat);
    
    ret = cm_ioctl_del_qos_entry(sc, &qos_wan);
    if (ret < 0) {
        printk(KERN_ERR "cm_del_qos_entry: CM_DEL_QOS failed (wan)\n");
    }

    ret = cm_ioctl_del_qos_entry(sc, &qos_dmz);
    if (ret < 0) {
        printk(KERN_ERR "cm_del_qos_entry: CM_DEL_QOS failed (dmz)\n");
    }

    return 0;
}

/*
 * IPF/NAT Helper Functions
 */

static int
cm_is_connection_natd(const struct ip_conntrack *ct)
{
	const struct ip_conntrack_tuple *orig, *reply;

	orig = &ct->tuplehash[IP_CT_DIR_ORIGINAL].tuple;
	reply = &ct->tuplehash[IP_CT_DIR_REPLY].tuple;

	if (orig->src.ip != reply->dst.ip) {
		if (debug) {
			pr_debug("cm_is_connection_natd "
			       "orig->src different than reply->dst: NAT\n");
		}
		return 1;
	}
	
	if (orig->dst.ip != reply->src.ip) {
		if (debug) {
			pr_debug("cm_is_connection_natd "
			       "orig->dst different than reply->src: NAT\n");
		}
		return 1;
	}
	
	if (debug) pr_debug("is_connection_natd connection is NOT NAT'd\n");
	return 0;
}

static int
cm_nit_set_control_flags(NAT_IPF_TBL *nit, const struct ip_conntrack *ct,
                      int nat_connection)
{
	const struct ip_conntrack_tuple *orig;

	orig = &ct->tuplehash[IP_CT_DIR_ORIGINAL].tuple;
	
	nit->control = cntrlbit_0_valid | cntrlbit_2_ipv4;
	
	if (nat_connection) {
		nit->control |= cntrlbit_1_nat;
		
		switch (orig->dst.protonum) {
		case 6: /* tcp */
			nit->control |= cntrlbit_4_tcp;
			break;
		case 17: /* udp */
			nit->control |= cntrlbit_5_udp;
			break;
		default:
			if (debug) {
				pr_debug("cm_nit_set_control_flags() "
				       "unsupported protocol %d (ignored)\n",
				       orig->dst.protonum);
			}
			return -1;
			break;
		}
	}
	
	return 0;
}

static int
cm_ct_high_priority(const struct ip_conntrack *ct)
{
	const struct ip_conntrack_tuple *orig, *reply;
	
	if (trace) printk("cm_ct_high_priority(ct=%p)\n", ct);

	orig = &ct->tuplehash[IP_CT_DIR_ORIGINAL].tuple;
	reply = &ct->tuplehash[IP_CT_DIR_REPLY].tuple;

	if (debug) pr_debug("protocol num is %d\n", orig->dst.protonum);
    
	/* look up IP addresses to determine if they are high priority */
	if (orig->dst.protonum == 17) { /* udp */
		if (debug) pr_debug("UDP\n");
		/* only UDP is handled by chip */
		if (cm_qos_entry_match(orig->src.ip, reply->src.ip,
				       orig->src.u.udp.port,
				       reply->src.u.udp.port)) {
			if (debug) pr_debug("High Priority\n");
			return 1;
		}
	}

	/* not high priority */
	return 0;
}

static int device_matches_pppoe(int ifindex);

static int
ifindex_get_cam_intf(int ifindex, const struct ip_conntrack *ct)
{
	const struct ip_conntrack_tuple *orig, *reply;
	int high_priority = 0;

	if (trace) printk("ifindex_get_cam_intf(ifindex=%d)\n", ifindex);

	orig = &ct->tuplehash[IP_CT_DIR_ORIGINAL].tuple;
	reply = &ct->tuplehash[IP_CT_DIR_REPLY].tuple;

	if (debug) pr_debug("protocol num is %d\n", orig->dst.protonum);
    
	/* look up IP addresses to determine if they are high priority */
	if (orig->dst.protonum == 17) { /* udp */
		if (debug) pr_debug("UDP\n");
		/* only UDP is handled by chip */
		if (cm_qos_entry_match(orig->src.ip, reply->src.ip,
				       orig->src.u.udp.port,
				       reply->src.u.udp.port)) {
			if (debug) pr_debug("High Priority\n");
			high_priority = 1;
		}
	}

	if (ifindex == cm_unit(1)->net.ifindex) {
		if (debug) pr_debug("returning dest_intf_lan0 (%d)\n",
				  dest_intf_lan0);
		return (dest_intf_lan0);
	}
	if (ifindex == cm_unit(2)->net.ifindex) {
		if (debug) pr_debug("returning dest_intf_dmz_%s (%d)\n",
				  high_priority ? "hp" : "np",
				  high_priority ? dest_intf_dmz_hp :
				  dest_intf_dmz_np);
		return (high_priority ? dest_intf_dmz_hp : dest_intf_dmz_np);
	}
	/* how to get ifindex for pppoe device? */
	if ((ifindex == cm_unit(0)->net.ifindex) ||
	    device_matches_pppoe(ifindex)) {
		if (debug) pr_debug("returning dest_intf_wan_%s (%d)\n",
		       high_priority ? "hp" : "np",
		       high_priority ? dest_intf_wan_hp : dest_intf_wan_np);
		return (high_priority ? dest_intf_wan_hp : dest_intf_wan_np);
	}

	/* should not reach here */
	if (debug) pr_debug("ifindex_get_cam_intf(): "
			  "invalid ifindex specified: %d\n", ifindex);
	return -1;
}

static void
nit_determine_ports(NAT_IPF_TBL *nit, const struct ip_conntrack *ct,
                    int nat_connection, int is_src_internal)
{
	const struct ip_conntrack_tuple *orig, *reply;
	
	if (trace) printk("nit_determine_ports()\n");

	orig = &ct->tuplehash[IP_CT_DIR_ORIGINAL].tuple;
	reply = &ct->tuplehash[IP_CT_DIR_REPLY].tuple;

	if (nat_connection == 0) {
		nit->portnum_inter = 0;
		nit->portnum_nat = 0;
		nit->portnum_exter = 0;
		return;
	}

	switch (orig->dst.protonum) {
	case 6: /* tcp */
		if (is_src_internal) {
			/* picture masquerading */
			/* orig tuple:
			   internal_ip:intl_port -> external_ip:extl_port
			   reply tuple:
			   external_ip:extl_port -> nat_ip:nat_port
			*/
			nit->portnum_inter = ntohs(orig->src.u.tcp.port);
			nit->portnum_nat = ntohs(reply->dst.u.tcp.port);
			nit->portnum_exter = ntohs(reply->src.u.tcp.port);
		} else {
			/* picture port forwarding */
			/* orig tuple:
			   external_ip:external_port -> nat_ip:nat_port
			   reply tuple:
			   internal_ip:intl_port -> external_ip:extl_port
			*/
			nit->portnum_inter = ntohs(reply->src.u.tcp.port);
			nit->portnum_nat = ntohs(orig->dst.u.tcp.port);
			nit->portnum_exter = ntohs(orig->src.u.tcp.port);
		}
		break;
		
	case 17: /* udp */
		if (is_src_internal) {
			/* picture masquerading */
			/* orig tuple:
			   internal_ip:intl_port -> external_ip:extl_port
			   reply tuple:
			   external_ip:external_port -> nat_ip:nat_port
			*/
			nit->portnum_inter = ntohs(orig->src.u.udp.port);
			nit->portnum_nat = ntohs(reply->dst.u.udp.port);
			nit->portnum_exter = ntohs(reply->src.u.udp.port);
		} else {
			/* picture port forwarding */
			/* orig tuple:
			   external_ip:external_port -> nat_ip:nat_port
			   reply tuple:
			   internal_ip:intl_port -> external_ip:extl_port
			*/
			nit->portnum_inter = ntohs(reply->src.u.udp.port);
			nit->portnum_nat = ntohs(orig->dst.u.udp.port);
			nit->portnum_exter = ntohs(orig->src.u.udp.port);
		}
		break;
	}
}

extern int wan_pppoe_mode;
extern int camelot_get_pppoe_mode(void);
static int interface_get_pppoe_hdr_idx(int ifindex);

static int
nit_setup_addrs(NAT_IPF_TBL *nit, const struct ip_conntrack *ct,
                int cam_intf_src, int cam_intf_dst,
                int dev_intf_src, int dev_intf_dst,
                unsigned char *src_ether, unsigned char *dst_ether,
                int nat_connection, NAT_IPF_TBL *qos_nit)
{
	const struct ip_conntrack_tuple *orig, *reply;
	int pppoe_hdr_index = -1;
	
	if (trace) printk("nit_setup_addrs()\n");
	
	orig = &ct->tuplehash[IP_CT_DIR_ORIGINAL].tuple;
	reply = &ct->tuplehash[IP_CT_DIR_REPLY].tuple;

	/*
	 * if pppoe mode is in use and one interface is wan, turn on
	 * appropriate bits in control
	 */
	if (camelot_get_pppoe_mode()) {
		nit->control |= add_pppoe_hdr;
		if ((cam_intf_src == dest_intf_wan_hp) ||
		    (cam_intf_src == dest_intf_wan_np)) {
			pppoe_hdr_index =
				interface_get_pppoe_hdr_idx(dev_intf_src);
		} else if ((cam_intf_dst == dest_intf_wan_hp) ||
			 (cam_intf_dst == dest_intf_wan_np)) {
			pppoe_hdr_index =
				interface_get_pppoe_hdr_idx(dev_intf_dst);
		}
		if (pppoe_hdr_index == -1) {
			if (debug) pr_debug("nit_setup_addrs() "
			       "cannot determine PPPoE Header index\n");
			return -1;
		}
		nit->control |= pppoe_hdr_select(pppoe_hdr_index);
	}
	
	/* this check relies on the fact that the interfaces in
	   camelot_bits.h are arranged in ascending numerical order
	   from "most internal" to "most external" */
	if (cam_intf_src < cam_intf_dst) {
		/* the conntrack orig source is internal, orig dest is
		   external */
		nit->intf_inter = cam_intf_src;
		nit->intf_exter = cam_intf_dst;
		memcpy(nit->macaddr_inter, src_ether, 6);
		memcpy(nit->macaddr_exter, dst_ether, 6);
		nit->ipv4addr_inter = ntohl(orig->src.ip);
		nit->ipv4addr_exter = ntohl(reply->src.ip);
		/* per the camelot PDF, s6.3.1, p30:
		 *  
		 * 'Due to the hardware implementation, the
		 * destination IP address of an IPv4 packet received
		 * from an external segment is always compared with
		 * the NAT_IP field of the NAT/IP forwarding table.
		 * Therefore, in IPv4 IP forwarding mode, NAT_IP field
		 * should be set to the same value as INT_IP.'
		 */
		if (nat_connection) {
			nit->nataddr = ntohl(reply->dst.ip);
		} else {
			nit->nataddr = nit->ipv4addr_inter;
		}
		nit_determine_ports(nit, ct, nat_connection, 1);
		
		*qos_nit = *nit;
		if (!nat_connection) {
			/* calculate ports always for qos information */
			nit_determine_ports(qos_nit, ct, 1, 1);
		}
	}
	else {
		/* the conntrack orig source is external, orig dest is
		   internal */
		nit->intf_inter = cam_intf_dst;
		nit->intf_exter = cam_intf_src;
		memcpy(nit->macaddr_inter, dst_ether, 6);
		memcpy(nit->macaddr_exter, src_ether, 6);
		nit->ipv4addr_inter = ntohl(reply->src.ip);
		nit->ipv4addr_exter = ntohl(orig->src.ip);
		/* per the camelot PDF, s6.3.1, p30:
		   
		 * 'Due to the hardware implementation, the
		 * destination IP address of an IPv4 packet received
		 * from an external segment is always compared with
		 * the NAT_IP field of the NAT/IP forwarding table.
		 * Therefore, in IPv4 IP forwarding mode, NAT_IP field
		 * should be set to the same value as INT_IP.'
		 */
		if (nat_connection) {
			nit->nataddr = ntohl(orig->dst.ip);
		} else {
			nit->nataddr = nit->ipv4addr_inter;
		}
		nit_determine_ports(nit, ct, nat_connection, 0);
		
		*qos_nit = *nit;
		if (!nat_connection) {
			/* calculate ports always for qos information */
			nit_determine_ports(qos_nit, ct, 1, 0);
		}
	}
	
	return 0;
}

void
cm_create_pppoe_neigh(unsigned long ifindex, unsigned long ip,
		      unsigned long *ether)
{
	int i;
	struct neighbour *n = 0;

	/* don't exceed max */
	if (num_pppoe_neighs >= LEN_PPPOE) {
		printk(KERN_ERR "camelot: too many pppoe sessions!\n");
		return;
	}

	/* if a neighbour exists with the same ifindex, just replace
	   that entry with the information */ 
	for (i = 0; i < LEN_PPPOE; i++) {
		if (pppoe_sessions[i].n == 0)
			continue;
		if (pppoe_sessions[i].n->dev->ifindex == ifindex) {
			/* found a match, replace */
			n = pppoe_sessions[i].n;
			pppoe_sessions[i].ip = ip;
			break;
		}
	}
	/* otherwise, make a new neighbor */
	if (n == 0) {
		for (i = 0; i < LEN_PPPOE; i++) {
			if (pppoe_sessions[i].n == 0)
				break;
		}
		pppoe_sessions[i].n = (struct neighbour *)
				 kmalloc(sizeof(struct neighbour), GFP_KERNEL);
		if (pppoe_sessions[i].n == NULL) {
			printk(KERN_ERR "%s(): out of memory\n", __FUNCTION__);
			return;
		}
		n = pppoe_sessions[i].n;
		pppoe_sessions[i].ip = ip;
		num_pppoe_neighs++;
	}
	
	/* clear any variables */
	memset(n, 0, sizeof (struct neighbour));
	n->dev = &dummy_net_device[i];
	n->dev->ifindex = ifindex;
	for (i = 0; i < 6; i++) {
		n->ha[i] = (unsigned char)ether[i];
		if (debug) pr_debug("pppoe_neigh->ha[%d] = %x\n",
				  i, n->ha[i]);
	}
	n->nud_state = NUD_PERMANENT;
}

void
cm_delete_all_pppoe_neigh(void)
{
	int i;
	for (i = 0; i < LEN_PPPOE; i++) {
		if (pppoe_sessions[i].n)
			kfree(pppoe_sessions[i].n);
		pppoe_sessions[i].n = NULL;
		pppoe_sessions[i].ip = 0;
	}
	num_pppoe_neighs = 0;
}

int
cm_delete_pppoe_neigh(int ifindex)
{
	int i;
	for (i = 0; i < LEN_PPPOE; i++) {
		if (pppoe_sessions[i].n &&
		    (pppoe_sessions[i].n->dev->ifindex == ifindex)) {
			if (debug) pr_debug("Freeing pppoe neighbour index %d\n",
					  ifindex);
			kfree(pppoe_sessions[i].n);
			pppoe_sessions[i].n = NULL;
			pppoe_sessions[i].ip = 0;
			num_pppoe_neighs--;
			return 0;
		}
	}
	return 1;
}

static int
interface_get_pppoe_hdr_idx(int ifindex)
{
    int i;

    for (i = 0; i < 4; i++)
        if (pppoe_sessions[i].n &&
	    (ifindex == pppoe_sessions[i].n->dev->ifindex))
            return i;

    return -1;
}

static int
device_matches_pppoe(int ifindex)
{
	int i;
	if (camelot_get_pppoe_mode() == 0)
		return 0;

	for (i = 0; i < LEN_PPPOE; i++)
		if (pppoe_sessions[i].n &&
		    (pppoe_sessions[i].n->dev->ifindex == ifindex))
			return 1;
		
	return 0;
}

static int
ip_matches_pppoe(unsigned long ip)
{
	int i;
	if (camelot_get_pppoe_mode() == 0)
		return 0;

	for (i = 0; i < LEN_PPPOE; i++) {
		if (pppoe_sessions[i].n &&
		    (pppoe_sessions[i].ip == ip)) {
			if (debug) pr_debug("IP %u.%u.%u.%u matches pppoe %s\n",
					  NIPQUAD(ip),
					  pppoe_sessions[i].n->dev->name);
			return 1;
		}
	}
		
	return 0;
}

static struct neighbour *
cm_pppoe_neigh_lookup(int ifindex)
{
	int i;
	
	if (camelot_get_pppoe_mode() == 0)
		return NULL;

	for (i = 0; i < LEN_PPPOE; i++)
		if (pppoe_sessions[i].n &&
		    (pppoe_sessions[i].n->dev->ifindex == ifindex))
			return pppoe_sessions[i].n;
		
	return NULL;
}

static struct neighbour *
cm_get_neighbour(const struct ip_conntrack_tuple *tuple)
{
	struct neighbour *n = NULL;
	struct rtable *rt = NULL;

	if (trace) printk("cm_get_neighbour(ip=%u.%u.%u.%u)\n",
			  NIPQUAD(tuple->src.ip));
	/* try to guess the outgoing interface */
	if (ip_route_output(&rt, tuple->src.ip, 0, 0, 0)) {
		if (debug) pr_debug("No route for %u.%u.%u.%u\n",
				  NIPQUAD(tuple->src.ip));
		return 0;
	}

	if (debug) pr_debug("rt->dst %u.%u.%u.%u rt->rt_gateway %u.%u.%u.%u dev %s\n",
			  NIPQUAD(rt->rt_dst), NIPQUAD(rt->rt_gateway),
			  rt->u.dst.dev->name);

	if (camelot_get_pppoe_mode() &&
	    (strncmp(rt->u.dst.dev->name, "ppp", 3) == 0)) {
		/* using PPPoE; grap MAC addr for PPPoE server */
		if (debug) pr_debug("Using PPPoE server neighbour\n");
		n = cm_pppoe_neigh_lookup(rt->u.dst.dev->ifindex);
	}
	else if (rt->rt_dst && (rt->rt_dst == rt->rt_gateway)) {
		if (debug) {
			pr_debug("Looking up destination "
			       "%u.%u.%u.%u (%u.%u.%u.%u) dev %s\n",
			       NIPQUAD(tuple->src.ip), NIPQUAD(rt->rt_dst),
			       rt->u.dst.dev->name);
		}
		n = neigh_lookup(&arp_tbl, &(tuple->src.ip), rt->u.dst.dev);
		if (!n) {
			printk(KERN_ERR "Could not find neighbour %u.%u.%u.%u\n",
			       NIPQUAD(tuple->src.ip));
		}
	} else {
		if (debug) pr_debug("Looking up gateway %u.%u.%u.%u\n",
				  NIPQUAD(rt->rt_gateway));
		n = neigh_lookup(&arp_tbl, &(rt->rt_gateway), rt->u.dst.dev);
		if (!n) {
			printk(KERN_ERR "Could not find neighbour %u.%u.%u.%u\n",
			       NIPQUAD(rt->rt_gateway));
		}
	}
	ip_rt_put(rt);

	if (n && !neigh_is_valid(n)) {
		if (debug) pr_debug("\n\nneighbour %p is not reachable\n", n);
		if (debug) pr_debug("===========================================================================\n");
		if (debug) pr_debug("  neigh->nud_state=%x\n\n\n", n->nud_state);
		n = 0;
	}
	
	if (debug && n) {
		pr_debug("Found neighbour\n");
	}

	return n;
}
	
static int
cm_nit_fill(const struct ip_conntrack *ct, NAT_IPF_TBL *nit,
	    NAT_IPF_TBL *qos_nit, int *qos_entry_required)
{
	struct cm_softc *cm;
	struct camelot_softc *sc;
	unsigned char *p_src_ether, *p_dst_ether;
	unsigned long mac_hi, mac_low;
	int nat_connection;
	int intf_src, intf_dst, cam_intf_src, cam_intf_dst;
	const struct ip_conntrack_tuple *orig, *reply;
	struct neighbour *n_orig = 0, *n_reply = 0;
	
	if (trace) printk("cm_nit_fill(ct=%p)\n", ct);

	memset((char *)nit, 0, sizeof(NAT_IPF_TBL));

	/* determine MAC addresses */

	orig = &ct->tuplehash[IP_CT_DIR_ORIGINAL].tuple;
	reply = &ct->tuplehash[IP_CT_DIR_REPLY].tuple;

#define NIPQUAD(addr) \
	((unsigned char *)&addr)[0], \
	((unsigned char *)&addr)[1], \
	((unsigned char *)&addr)[2], \
	((unsigned char *)&addr)[3]

	n_orig = cm_get_neighbour(orig);
	n_reply = cm_get_neighbour(reply);
	if (!n_orig || !n_reply) {
		printk(KERN_WARNING "cm_nit_fill could not find neighbour\n");
		return 1;
	}

	if (debug) pr_debug("found neighbours for orig and reply tuples\n");
	p_src_ether = n_orig->ha;
	p_dst_ether = n_reply->ha;

	/* determine if NAT'ing */
	nat_connection = cm_is_connection_natd(ct);

	/* set NIT control flags appropriately */
	if (cm_nit_set_control_flags(nit, ct, nat_connection)) {
		if (debug) pr_debug("cm_nit_fill() "
				  "error setting NIT control flags\n");
		return -1;
	}

	/* set internal/external interface information appropriately */
	/* get the physical interface indeces */

	/* n_orig->dev->ifindex has interface index; notice that the
	   neighbour also has the device, if that helps */
	intf_src = n_orig->dev->ifindex;
	intf_dst = n_reply->dev->ifindex;

	if (intf_src == intf_dst) {
		if (debug) {
			pr_debug("intf_src (%d) and intf_dst (%d) are equal!\n",
			       intf_src, intf_dst);
		}
		return -1;
	}
	
	/* translate physical interface index to camelot priority
	   interface for IPF/NAT entry */
	cam_intf_src = ifindex_get_cam_intf(intf_src, ct);
	cam_intf_dst = ifindex_get_cam_intf(intf_dst, ct);
	if ((cam_intf_src < 0) || (cam_intf_dst < 0)) {
		return -1;
	}


	/* if either interface is LAN, check which camelot port */
	if (cam_intf_src == dest_intf_lan0) {
		/* need sc (to update l2 table, if necessary)  */
		cm = (struct cm_softc *)(n_orig->dev->priv);
		sc = &cm->camelot;
		mac_hi = (p_src_ether[0] << 24) | (p_src_ether[1] << 16) |
			 (p_src_ether[2] << 8) | p_src_ether[3];
		mac_low = (p_src_ether[4] << 8) | p_src_ether[5];
		if (lookup_l2_intf(sc, mac_hi, mac_low, &cam_intf_src)) {
			if (debug) {
				pr_debug("cm_nit_fill() "
				       "can't find intf for "
				       "%02x:%02x:%02x:%02x:%02x:%02x (src)\n",
				       p_src_ether[0], p_src_ether[1],
				       p_src_ether[2], p_src_ether[3],
				       p_src_ether[4], p_src_ether[5]);
			}
			return -1;
		}
	}
	if (cam_intf_dst == dest_intf_lan0) {
		cm = (struct cm_softc *)(n_reply->dev->priv);
		sc = &cm->camelot;
		mac_hi = (p_dst_ether[0] << 24) | (p_dst_ether[1] << 16) |
			 (p_dst_ether[2] << 8) | p_dst_ether[3];
		mac_low = (p_dst_ether[4] << 8) | p_dst_ether[5];
		
		if (lookup_l2_intf(sc, mac_hi, mac_low, &cam_intf_dst)) {
			if (debug) {
				pr_debug("cm_nit_fill() "
				       "can't find intf for "
				       "%02x:%02x:%02x:%02x:%02x:%02x (dst)\n",
				       p_dst_ether[0], p_dst_ether[1],
				       p_dst_ether[2], p_dst_ether[3],
				       p_dst_ether[4], p_dst_ether[5]);
			}
			return -1;
		}
	}
	
	if (debug) pr_debug("cam_intf_src: %d; cam_intf_dst: %d\n",
			  cam_intf_src, cam_intf_dst);

	/* QoS required? */
	if (((cam_intf_src == dest_intf_dmz_hp) ||
	     (cam_intf_src == dest_intf_wan_hp)) &&
	    ((cam_intf_dst == dest_intf_dmz_hp) ||
	     (cam_intf_dst == dest_intf_wan_hp))) {
		if (debug) pr_debug("QOS Entry required\n");
		*qos_entry_required = 1;
	}

	if (nit_setup_addrs(nit, ct, cam_intf_src, cam_intf_dst,
			    intf_src, intf_dst,
			    p_src_ether, p_dst_ether,
			    nat_connection, qos_nit)) {
		if (debug) pr_debug("cm_nit_fill() "
				  "error determining IP/Port information\n");
		return -1;
	}

	return 0;
}

int
cm_set_ip_addr(unsigned long index, unsigned long addr, unsigned long mask)
{
	if (trace) printk("cm_set_ip_addr(idx=%lu, addr=0x%08lx)\n",
			  index, addr);
	
	switch (index) {
	case 0:
		wan_ip = addr;
		wan_mask = mask;
		break;
	case 1:
		cm_ip[0] = addr;
		cm_mask[0] = mask;
		break;
	case 2:
		cm_ip[1] = addr;
		cm_mask[1] = mask;
		break;
	default:
		if (debug) pr_debug("cm_set_ip_addr() "
				  "Invalid index specified %lu\n",
		       index);
		return -1;
		break;
	}
	return 0;
}

/* check if the given IP address matches any of camelot's configured
   interface addresses, including loopback address */
static int
ip_addr_is_camelot(unsigned long ip)
{
    int i;

    if (trace) printk("ip_addr_is_camelot(ip=%lx,wan=%lx,cm0=%lx,cm1=%lx,cm2=%lx)\n", ip, wan_ip, cm_ip[0], cm_ip[1], cm_ip[2]);
    
    /* wan interface */
    if (wan_ip && ip == wan_ip) {
        if (debug) pr_debug("IP %u.%u.%u.%u matches wan_ip\n", NIPQUAD(ip));
        return 1;
    }

    /* pppoe interface */
    if (ip_matches_pppoe(ip)) {
        return 1;
    }

    for (i = 0; i < 3; i++) {
        /* interface */
        if (ip == cm_ip[i]) {
            if (debug) pr_debug("IP %u.%u.%u.%u matches cm%d\n", NIPQUAD(ip), i);
            return 1;
        }
    }

    /* loopback */
    if ((ntohl(ip) >> 24) == 0x7f) {
        if (debug) pr_debug("IP %u.%u.%u.%u matches loopback\n", NIPQUAD(ip));
        return 1;
    }
    if (debug) pr_debug("IP address %u.%u.%u.%u is NOT local\n", NIPQUAD(ip));

    return 0;
}

static int
ip_addr_matches_net(unsigned long ip, int index)
{
    unsigned long net;

    if (!cm_mask[index]) {
        return 0;
    }

    net = cm_ip[index] & cm_mask[index];
    if ((ip & cm_mask[index]) == net) {
        if (debug) pr_debug("ip_addr_matches_net: "
			  "interface cm%d matches %u.%u.%u.%u\n",
			  index, NIPQUAD(ip)); 
        return 1;
    }
    
    return 0;
}

/* determine if the two IP addresses match the network given by index */
static int
ip_addrs_match_net(unsigned long ip1, unsigned long ip2, int index)
{
    if (ip_addr_matches_net(ip1, index) && ip_addr_matches_net(ip2, index)) {
        if (debug) pr_debug("ip_addrs_match_net: both addrs on cm%d net\n",
			  index);
        return 1;
    }

    return 0;
    
}

/* determine if an IP address matches the WAN network */
static int
ip_addr_matches_wan(unsigned long ip)
{
    unsigned long net;

    if (trace) printk("ip_addrs_matches_wan(wan_mask=%lx, wan_ip=%lx)\n",
		      wan_mask, wan_ip);
    
    if (!wan_mask) {
        return 0;
    }

    net = wan_ip & wan_mask;
    if ((ip & wan_mask) == net) {
        if (debug) pr_debug("ip_addr_matches_wan: "
			  "interface wan0 matches %u.%u.%u.%u\n",
			  NIPQUAD(ip)); 
        return 1;
    }
    
    return 0;
}

/* determine if the two IP addresses match the WAN network */
static int
ip_addrs_match_wan_net(unsigned long ip1, unsigned long ip2)
{
	/* check if both addresses match the same LAN network */
	if (trace) printk("ip_addrs_match_wan_net(wan_mask=%lx)\n", wan_mask);
	if (!wan_mask) {
		/* WAN interface not configured */
		return 0;
	}
	
	if (ip_addr_matches_wan(ip1) && ip_addr_matches_wan(ip2)) {
		if (debug) pr_debug("ip_addrs_match_wan_net: "
				  "both addrs on wan net\n");
		return 1;
	}

	return 0;
}

/* determine if the two IP addresses match the LAN network */
static int
ip_addrs_match_lan_net(unsigned long ip1, unsigned long ip2)
{
	/* check if both addresses match the same LAN network */
	if (!cm_mask[0]) {
		/* LAN interface not configured */
		return 0;
	}
	
	return ip_addrs_match_net(ip1, ip2, 0);
}

/* determine if the two IP addresses match the DMZ network */
static int
ip_addrs_match_dmz_net(unsigned long ip1, unsigned long ip2)
{
	/* check if both addresses match the same DMZ network */
	if (!cm_mask[1]) {
		/* DMZ interface not configured */
		return 0;
	}
	
	return ip_addrs_match_net(ip1, ip2, 1);
}

static int
cm_ct_is_local(const struct ip_conntrack *ct)
{
	const struct ip_conntrack_tuple *orig, *reply;
	unsigned long orig_src;
	unsigned long orig_dst;
	unsigned long reply_src;
	unsigned long reply_dst;

	orig = &ct->tuplehash[IP_CT_DIR_ORIGINAL].tuple;
	reply = &ct->tuplehash[IP_CT_DIR_REPLY].tuple;

	orig_src = orig->src.ip;
	orig_dst = orig->dst.ip;
	reply_src = reply->src.ip;
	reply_dst = reply->dst.ip;
	
	/*
	  determine if we would add an IP forwarding rule in hardware;
	  do not do so if:
	  - either the src or dst IP addresses are any of the camelot
	    interface IP addresses
	  - if both the src AND dst IP addresses match the camelot LAN
	    interface IP network address
	*/

	/* +++: what if the DMZ is on the same "network" as the LAN
	   interfaces? do we need additional logic to handle this? */

	if (trace) printk("cm_ct_is_local (orig %u.%u.%u.%u->%u.%u.%u.%u "
			  "reply %u.%u.%u.%u->%u.%u.%u.%u)\n",
			  NIPQUAD(orig_src), NIPQUAD(orig_dst),
			  NIPQUAD(reply_src), NIPQUAD(reply_dst));
	
	/* if both the orig_src and reply_dst OR the orig_dst and
	   reply_src are camelot addresses, it is local */
	if ((ip_addr_is_camelot(orig_src) && ip_addr_is_camelot(reply_dst))
	    ||
	    (ip_addr_is_camelot(orig_dst) && ip_addr_is_camelot(reply_src))) {
		if (debug) pr_debug("cm_ct_is_local: addresses match camelot "
				  "interface addresses\n");
		return 1;
	}

	/* +++ does this check cause problems? it's not done in
	   camelotd, and i assume it's an oversight there, but just in
	   case... */
	if (ip_addrs_match_wan_net(orig_src, reply_src)) {
		if (debug) pr_debug("cm_ct_is_local: "
				  "addresses match WAN network\n");
		return 1;
	}

	if (ip_addrs_match_lan_net(orig_src, reply_src)) {
		if (debug) pr_debug("cm_ct_is_local: "
				  "addresses match LAN network\n");
		return 1;
	}
	
	if (ip_addrs_match_dmz_net(orig_src, reply_src)) {
		if (debug) pr_debug("cm_ct_is_local: "
				  "addresses match DMZ network\n");
		return 1;
	}

	return 0;
}

/*
 * Pending Conntrack List Helper Functions
 */

int
ct_list_element_queue(const struct ip_conntrack *ct, int add)
{
	struct ipfnat_list_entry *entry;
	struct ip_conntrack *tmp_ct = (struct ip_conntrack *)ct;
	struct list_head *lh;
	
	/*
	 * find an element on the free list, fill in, and move to
	 * queued list
	 */
	if (list_empty(&ipfnat_ct_free_list)) {
		if (debug) pr_debug("No free ct list elements\n");
		return -1;
	}

	atomic_inc(&tmp_ct->ct_general.use);
	
	lh = ipfnat_ct_free_list.next;
	list_del(lh);
	entry = list_entry(lh, struct ipfnat_list_entry, list);
	entry->ct = ct;
	entry->add = add;
	list_add(&entry->list, &ipfnat_ct_queued_list);
	
	return 0;
}

void
ct_list_element_free(struct ipfnat_list_entry *entry)
{
	struct ip_conntrack *tmp_ct;
	
	/* decrement ct; incremented in ct_list_element_queue() */
	tmp_ct = (struct ip_conntrack *)entry->ct;
	atomic_dec(&tmp_ct->ct_general.use);
	/* now clear entry and add to free list */
	entry->ct = 0;
	entry->add = 0;
	list_add(&entry->list, &ipfnat_ct_free_list);
}

void
ct_list_element_destroy(struct ipfnat_list_entry *entry)
{
	struct ip_conntrack *tmp_ct;
	
	tmp_ct = (struct ip_conntrack *)entry->ct;
	if (debug_ct) pr_debug("ct_list_element_destroy(ct=%p)\n", tmp_ct);
	/* incremented in ct_list_element_create() */
	atomic_dec(&tmp_ct->ct_general.use);
	if (debug_ct) pr_debug("freeing entry %p\n", entry);
	kfree(entry);
}

int
ct_matches_neigh(const struct ip_conntrack_tuple *tuple,
		 const struct neighbour *neigh)
{
	int i, match = 1, key_len;
	u8 *foo;

	if (debug_ct) pr_debug("ct_matches_neigh\n");
	
	/* +++ may have to check dst.ip too */

	if (debug_ct) pr_debug("tuple src IP is %x\n", tuple->src.ip);
	foo = (u8 *)&tuple->src.ip;
	if (debug_ct) pr_debug("accessing as bytes: %x.%x.%x.%x\n",
			     foo[0], foo[1], foo[2], foo[3]);
	
	key_len = neigh->tbl->key_len;
	for (i = 0; i < key_len; i++) {
		if (debug_ct) pr_debug("neigh->primary_key[%d] is %x\n",
				     i, neigh->primary_key[i]);
		if (foo[i] != neigh->primary_key[i]) {
			match = 0;
			break;
		}
	}
	if (debug_ct) pr_debug("match is %d\n", match);
	return match;
}

static int
cm_check_camelot_structs(struct cm_softc *cm)
{
	if ((cm == 0) || (&cm->camelot == 0)) {
		printk(KERN_ERR "cm_check_camelot_structs: ");
		printk(KERN_ERR "Invalid camelot pointer!\n");
		return -1;
	}

	return 0;
}	

extern int cm_ioctl_add_nat_entry(struct camelot_softc *, NAT_IPF_TBL *);
extern int cm_ioctl_del_nat_entry(struct camelot_softc *, NAT_IPF_TBL *);

static int
cm_ipfnat_helper(const struct ip_conntrack *ct, int add)
{
	struct cm_softc *cm = cm_unit(0);
	struct camelot_softc *sc;
	NAT_IPF_TBL nit, qos;
	struct ip_conntrack *tmp_ct = (struct ip_conntrack *)ct;
	int qos_entry_required = 0, high_priority;
	int ret = 0;
	
	if (trace) printk("cm_ipfnat_helper(ct=%p)\n", ct);

	if (cm_check_camelot_structs(cm)) {
		ret = -1;
		goto done2;
	}
	sc = &cm->camelot;

	atomic_inc(&tmp_ct->ct_general.use);

	/* check if this conntrack matches high priority/QoS */
	high_priority = cm_ct_high_priority(ct);

	/* if !high priority and !add, then remove chip entry by index
	   in conntrack->status */

	/* if high priority and !add, need to do *something* to get a
	   qos entry, but maybe it doesn't have to be all the same
	   functionality as when adding? */

	/* determine exactly what needs to be done for QoS in the hope
	   that we can have a different code path for add/remove
	   here--and only execute the necessary functions for each */
	
	/*
	 * +++: for performance purposes, can skip this step if the
	 * chip_index is reliable for deleting the HW entry, and if ct
	 * does not match a QoS entry
	 */
	/* fill in nit struct */
	ret = cm_nit_fill(ct, &nit, &qos, &qos_entry_required);
	if (ret < 0) {
		if (debug) pr_debug("cm_nit_fill failed\n");
		goto done2;
	}
	else if (ret > 0) {
		/*
		 * add conntrack to queue pending neighbour
		 * notifications from kernel
		 */
		if (ct_list_element_queue(ct, add)) {
			if(debug_ct) pr_debug("cm_nit_fill couldn't queue ct\n");
		} else {
			if (debug_ct) pr_debug("cm_nit_fill queued ct pending "
					     "neighbour notification\n");
		}
		/* ct_list_add ref'd ct; okay to release ct here */
		goto done2;
	}
	
	if (add) {
		/* add IPF/NAT chip entry */
		if (debug) pr_debug("adding chip entry\n");
		ret = cm_ioctl_add_nat_entry(sc, &nit);
		if (ret < 0) {
			printk(KERN_ERR "cm_ipfnat: Cannot add HW entry!\n");
		}
		else {
			/*
			 * if the conntrack isn't confirmed, do
			 * so--otherwise, the conntrack timer will not
			 * be operational
			 */
			if (debug) pr_debug("cm_ipfnat: Added HW entry %d\n",
					  nit.chip_index);
			/* save the index in the conntrack */
			CM_IPFNAT_IDX_SET(tmp_ct->status, nit.chip_index);
			CM_IPFNAT_INACTIVITY_SET(tmp_ct->status, 0);
			ip_ct_refresh(tmp_ct, 15*60*HZ);

			if (qos_entry_required) {
				cm_add_qos_entry(sc, &qos);
			}
		}
	} else {
		/* delete QOS entry first */
		if (qos_entry_required) {
			cm_del_qos_entry(sc, &qos);
		}

		if (CM_IPFNAT_IDX_IS_VALID(ct->status) == 0) {
			if (debug) pr_debug("No chip index associated with ct "
					  "%p; cannot remove HW rule!\n", ct);
			ret = -1;
			goto done2;
		}
		
		/* remove IPF/NAT chip entry */
		nit.chip_index = CM_IPFNAT_IDX_GET(ct->status);

		/*
		 * +++: for performance, could delete HW entry using
		 * the chip_index only
		 */
		
		if (debug) pr_debug("+++removing chip entry %d\n",
				     nit.chip_index);
		ret = cm_ioctl_del_nat_entry(sc, &nit);
		if (ret < 0) {
			printk(KERN_ERR "cm_ipfnat: Cannot delete HW entry!\n");
		}
		else {
			if (debug) pr_debug("cm_ipfnat: Deleted HW entry %d\n",
					  nit.chip_index);
		}
		/* eliminate chip index from status */
		CM_IPFNAT_IDX_SET_INVALID(tmp_ct->status);
	}

done2:
	atomic_dec(&tmp_ct->ct_general.use);

	if (trace) printk("cm_ipfnat_helper DONE\n");
	
	return ret;
}

/*
 * For each conntrack entry in the list, if waiting on the given
 * neighbour, attempt to process the conntrack entry now
 */
void
ct_list_process_neighbour(const struct neighbour *neigh)
{
	struct ipfnat_list_entry *entry;
	struct list_head *lh, *next;

	if (trace) printk("ct_list_process_neighbour\n");
	
	if (list_empty(&ipfnat_ct_queued_list))
		return;

	for (lh = ipfnat_ct_queued_list.next;
	     lh != &ipfnat_ct_queued_list;
	     lh = next) {
		const struct ip_conntrack *ct;
		const struct ip_conntrack_tuple *orig, *reply;

		entry = list_entry(lh, struct ipfnat_list_entry, list);
		ct = entry->ct;
		orig = &ct->tuplehash[IP_CT_DIR_ORIGINAL].tuple;
		reply = &ct->tuplehash[IP_CT_DIR_REPLY].tuple;

		next = lh->next;
		/* check if neigh matches either orig or reply tuple
		   in ct */
		if (ct_matches_neigh(orig, neigh) ||
		    ct_matches_neigh(reply, neigh)) {
			/* if so, remove entry from list and try
			   adding chip rule now */
			if (debug_ct) pr_debug("ct %p (%u.%u.%u.%u -> %u.%u.%u.%u) matches neighbour\n", ct, NIPQUAD(orig->src.ip), NIPQUAD(reply->src.ip));
			list_del(lh);
			cm_ipfnat_helper(ct, entry->add);
			if (debug_ct) pr_debug("calling ct_list_element_free\n");
			ct_list_element_free(entry);
		}
	}
}

/*
 * Callback Functions
 */

/*
 * callback for neighbour events--this function currently does
 * nothing
 */
int
cm_arp_callback_func(const struct neighbour *neigh)
{
	int i, key_len;
	
	if (trace) printk("cm_arp_callback_func(neigh=%p)\n", neigh);

	if (!neigh_is_valid((struct neighbour *)neigh)) {
		if (debug) pr_debug("neigh=%p is not valid\n", neigh);
		return 0;
	}

	/* grab neighbour IP address */
	key_len = neigh->tbl->key_len;
	if (debug) {
		pr_debug("neigh IP ");
		for (i = 0; i < key_len; i++) {
			printk("%02x ", neigh->primary_key[i]);
		}
		printk("\n");
	}
	if (debug) {
		pr_debug("neigh HA ");
		for (i = 0; i < 6; i++) {
			printk("%02x ", neigh->ha[i]);
		}
		printk("\n");
	}
	/* if a valid MAC address is available:
	   - determine if any pending conntracks have this IP address
	   - try to add chip entry
	*/

	if (list_empty(&ipfnat_ct_queued_list)) {
		if (debug_ct) pr_debug("cm_arp_callback_func: ct queue empty\n");
		return 0;
	}

	if (debug_ct) pr_debug("cm_arp_callback_func: checking ct list\n");
	ct_list_process_neighbour(neigh);
	
	return 0;
}

int
cm_ipfnat_callback_func(const struct ip_conntrack *ct, int add)
{
	struct ip_conntrack *tmp_ct = (struct ip_conntrack *)ct;
	int ret = 0;

	if (trace) printk("cm_ipfnat_callback_func(ct=%p)\n", ct);
	
	/* make sure conntrack exists */
	if (tmp_ct == 0) {
		if (debug) pr_debug("\n\ncm_ipfnat_callback_func() "
				  "conntrack ptr is NULL!\n");
		return -1;
	}

	/*
	 * +++ performance improvements:
	 * if add--check for an available HW slot before proceeding
	 * if del--check if QoS matches and/or if ct->status has valid
	 *         chip IDX
	 */

	atomic_inc(&tmp_ct->ct_general.use);

	if (debug) pr_debug("\n+++cm_ipfnat_callback_func(ct=%p; add=%d)\n\n",
			     ct, add);

	/* don't send local addresses to chip */
	if (cm_ct_is_local(ct))
		goto done;

	ret = cm_ipfnat_helper(ct, add);

done:
	atomic_dec(&tmp_ct->ct_general.use);

	if (trace) printk("cm_ipfnat_callback_func DONE\n");
	
	return ret;
}

static struct ip_conntrack *
cm_find_ct_by_chip_index(int index)
{
	struct ip_conntrack *found_ct = 0;
	int i, found = 0;
	
	READ_LOCK(&ip_conntrack_lock);
	for (i = 0; i < ip_conntrack_htable_size; i++) {
		const struct list_head *h = &ip_conntrack_hash[i];
		while (1) {
			struct ip_conntrack *ct;
			const struct ip_conntrack_tuple_hash *hash;

			h = h->next;
			if (h == &ip_conntrack_hash[i])
				break;

			hash = (const struct ip_conntrack_tuple_hash *)h;

			ct = hash->ctrack;
			/* check if the index stuffed in the ct matches */
			if (CM_IPFNAT_IDX_IS_VALID(ct->status) &&
			    CM_IPFNAT_IDX_GET(ct->status) == index) {
				if (debug) pr_debug("Found matching ct %p\n",
						  ct);
				found_ct = ct;
				found = 1;
				break;
			}
		}
		if (found)
			break;
	}
	READ_UNLOCK(&ip_conntrack_lock);

	if (!found && debug) {
		pr_debug("Did not find conntrack entry with index %d\n", index);
	}

	return found_ct;
}

int
cm_rstfin_callback_func(int index)
{
	struct ip_conntrack *found_ct = 0;
	
	if (trace) printk("cm_rstfin_callback_func(index=%d)\n", index);
	/* remove hw rule and timeout conntrack */
	/*
	 * first need to lookup conntrack (grrr...). this is necessary
	 * to remove any QOS entry
	 */

	if (debug) pr_debug("cm_rstfin_callback_func: index=%d\n", index);
	found_ct = cm_find_ct_by_chip_index(index);
	if (!found_ct) {
		if (debug) pr_debug("cm_rstfin_callback_func: "
				  "no conntrack associated with index %d\n",
				  index);
		return -1;
	}
	
	/* delete chip entry and timeout conntrack */
	cm_ipfnat_callback_func(found_ct, 0);
	ip_ct_refresh(found_ct, 0);
	
	return 0;
}

extern int ip_ct_register_cb_func(int (*func)(const struct ip_conntrack *,
					      int));
extern int ip_ct_unregister_cb_func(int (*func)(const struct ip_conntrack *,
						int));
extern int neighbour_register_cb_func(int (*func)(const struct neighbour *));
extern int neighbour_unregister_cb_func(int (*func)(const struct neighbour *));

extern int camelot_register_rstfin_callback_by_id(int (*func)(int));
extern int camelot_unregister_rstfin_callback_by_id(int (*func)(int));

void
cm_ipfnat_funcs_helper(int enable)
{
	if (trace) printk("cm_ipfnat_funcs_helper(enable=%d)\n", enable);
	if (enable) {
		/* register funcs */
		ip_ct_register_cb_func(cm_ipfnat_callback_func);
		neighbour_register_cb_func(cm_arp_callback_func);
		camelot_register_rstfin_callback_by_id(cm_rstfin_callback_func);
	} else {
		/* unregister funcs */
		ip_ct_unregister_cb_func(cm_ipfnat_callback_func);
		neighbour_unregister_cb_func(cm_arp_callback_func);
		camelot_unregister_rstfin_callback_by_id(cm_rstfin_callback_func);
	}
}

extern int cm_get_match_log(struct camelot_softc *, u_int32_t *);
extern int cm_ioctl_del_unmatched_entry(struct camelot_softc *,u_int32_t,int);
extern void cm_reset_match_log(int);

void cm_match_poll_tasklet_func(unsigned long);
DECLARE_TASKLET_DISABLED(cm_match_tasklet, cm_match_poll_tasklet_func, 0);

/*
 * Match Register Polling Timer
 */
void
cm_match_poll_tasklet_func(unsigned long unused)
{
	/* grab any cm_unit */
	struct cm_softc *cm = cm_unit(0);
	struct camelot_softc *sc;
	struct ip_conntrack *ct = 0;
	u_int32_t match_log[ENTRY_NUM_NATMTHLOG];
	int i, index;
	u_int32_t ctl22;

	if (trace) printk("cm_match_poll_tasklet_func()\n");
	if (cm == 0) {
		if (debug) pr_debug("Cannot retrieve camelot cm_unit 0!\n");
		return;
	}

	sc = &cm->camelot;
	if (sc == 0) {
		if (debug) pr_debug("Cannot retrieve data from cm_unit 0!\n");
		return;
	}

	/* get persistent match log */
	cm_get_match_log(sc, match_log);

	for (i = 0; i < NATIPF_TABLE_SIZE; i++) {
		/* map index into real index */
		index = map_natipf_index(i);
		if (debug_low) pr_debug("reading NATIPFTBL, index=%d\n", index);
		ctl22 = read_NATIPFTBL(sc, index, 22);
		
		/* check if entry valid */
		if ((ctl22 & 0x100000) == 0) { /* invalid */
			continue;
		}
		
		if (debug_low) pr_debug("looking for ct with index=%d\n", index);
		/* find conntrack associated with this 'i' */
		ct = cm_find_ct_by_chip_index(i);
		if (!ct) {
			if (debug) pr_debug("cm_match_poll_taslket_func: "
					  "no conntrack associated with "
					  "index %d\n", i);
			continue;
		}

		/* activity--refresh timeout & match log */
		if ((match_log[index/32] >> (index%32)) & 0x00000001) {
			if (debug) pr_debug("ACTIVE: ct=%p index=%d\n",
					  ct, index);
			ip_ct_refresh(ct, 15*60*HZ);
			CM_IPFNAT_INACTIVITY_SET(ct->status, 0);
			/* reset global match log for this entry */
			cm_reset_match_log(index);
		} else {
			if (debug) {
				pr_debug("INACTIVE: ct=%p index=%d count=%lu\n",
				       ct, index,
				       CM_IPFNAT_INACTIVITY_GET(ct->status));
			}
			/* increment inactive periods */
			CM_IPFNAT_INACTIVITY_INCR(ct->status);
			
			/* if inactive periods > expiration periods,
			   delete HW rule and timeout conntrack */
			if (CM_IPFNAT_INACTIVITY_GET(ct->status) >
			    num_periods_till_expiration) {
				/* eliminate chip index from status */
				if (debug) {
					pr_debug("TIMEOUT: ct=%p index=%d "
					       "count=%lu\n",
					       ct, index,
					       CM_IPFNAT_INACTIVITY_GET(ct->status));
				}
				CM_IPFNAT_IDX_SET_INVALID(ct->status);
				CM_IPFNAT_INACTIVITY_SET_INVALID(ct->status);
				cm_ioctl_del_unmatched_entry(sc, ctl22, i);
				ip_ct_refresh(ct, 0);
			}
		}
	}

	return;
}

struct timer_list cm_match_poll_timer;
static unsigned long reschedule_timer = 0;

void
cm_match_poll_timer_func(unsigned long unused)
{
	/* queue a tasklet to be called later */
	if (trace) printk("cm_match_poll_timer()\n");
	tasklet_schedule(&cm_match_tasklet);

	if (reschedule_timer) {
		if (debug) pr_debug("Rescheduling match poll timer\n");
		mod_timer(&cm_match_poll_timer, jiffies + poll_period * HZ);
	}
	return;
}

/*
 * Initialize module.
 */
int
camelot_ipfnat_init(void)
{
	struct ipfnat_list_entry *entry;
	int i;
	
	if (trace)
	        pr_debug("%s()\n", __FUNCTION__);

	if (camelot_get_ipfnat_mode()) {
		printk(KERN_WARNING "camelot_ipfnat_init(): IPF/NAT already enabled!\n");
		return -1;
	}

	INIT_LIST_HEAD(&ipfnat_ct_free_list);
	INIT_LIST_HEAD(&ipfnat_ct_queued_list);
	/*
	 * create entries now and put on free list--this prevents the
	 * possibility of allocating memory while in an interrupt,
	 * which caused an oops in slab.c:kmem_cache_grow()
	 */
	for (i = 0; i < NATIPF_TABLE_SIZE; i++) {
		entry = (struct ipfnat_list_entry *)
			kmalloc(sizeof(struct ipfnat_list_entry), GFP_KERNEL);
		if (!entry) {
			printk(KERN_ERR "camelot_ipfnat: no memory!\n");
			return -1;
		}
		memset(entry, 0, sizeof(struct ipfnat_list_entry));
		list_add(&entry->list, &ipfnat_ct_free_list);
	}
	
	camelot_set_ipfnat_mode(1);
	cm_ipfnat_funcs_helper(1);

	/* enable tasklet */
	if (debug) pr_debug("Enabling match poll tasklet\n");
	tasklet_enable(&cm_match_tasklet);
	
	/* start match poll timer */
	if (debug) pr_debug("Setting match poll timer\n");
	init_timer(&cm_match_poll_timer);
	reschedule_timer = 1; /* allow timer to reschedule itself */
	cm_match_poll_timer.data = 0;
	cm_match_poll_timer.function = cm_match_poll_timer_func;
	cm_match_poll_timer.expires = jiffies + poll_period * HZ;
	add_timer(&cm_match_poll_timer);
	
	return 0;
}

/*
 * Prepare for exit.
 */
void
camelot_ipfnat_exit(void)
{
	struct ipfnat_list_entry *entry;
	struct list_head *lh;
	
	if (trace)
	        pr_debug("%s()\n", __FUNCTION__);

	if (camelot_get_ipfnat_mode() == 0) {
		if (debug) pr_debug("camelot_ipfnat_exit: "
				  "IPF/NAT not enabled!\n");
		return;
	}

	if (debug) pr_debug("Disabling match poll tasklet\n");
	tasklet_disable(&cm_match_tasklet);

	if (debug) pr_debug("Deleting match poll timer\n");
	reschedule_timer = 0; /* don't allow timer to reschedule itself */
	del_timer_sync(&cm_match_poll_timer);
	
	cm_ipfnat_funcs_helper(0);
	camelot_set_ipfnat_mode(0);

	/* free all entries */
	if (!list_empty(&ipfnat_ct_queued_list)) {
		while ((lh = ipfnat_ct_queued_list.next) !=
		       &ipfnat_ct_queued_list) {
			entry = list_entry(lh, struct ipfnat_list_entry, list);
			if (debug_ct) pr_debug("calling list_del\n");
			list_del(lh);
			if (debug_ct) pr_debug("calling ct_list_element_free\n");
			ct_list_element_free(entry);
		}
	}
	if (!list_empty(&ipfnat_ct_free_list)) {
		while ((lh = ipfnat_ct_free_list.next) !=
		       &ipfnat_ct_free_list) {
			entry = list_entry(lh, struct ipfnat_list_entry, list);
			if (debug_ct) pr_debug("calling list_del\n");
			list_del(lh);
			if (debug_ct) pr_debug("freeing entry\n");
			kfree(entry);
		}
	}
	
	if (debug_ct) pr_debug("calling INIT_LIST_HEAD\n");
	INIT_LIST_HEAD(&ipfnat_ct_queued_list);
	INIT_LIST_HEAD(&ipfnat_ct_free_list);
}

#ifdef MODULE
static int __init camelot_ipfnat_module_init(void)
{
	return (camelot_ipfnat_init());
}

static void __exit camelot_ipfnat_module_exit(void)
{
	camelot_ipfnat_exit();
}

module_init(camelot_ipfnat_init);
module_exit(camelot_ipfnat_exit);
#endif
