/*
 * linux/drivers/net/mb86977/camelot_ioctl.c
 *
 * ioctl interface to camelot network driver
 * 
 * Author: Brad Parker <brad@heeltoe.com>
 *
 * 2003 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/in.h>
#include <linux/ip.h>

#include <asm/uaccess.h>

#include "camelot_var.h"
#include "camelot_func.h"
#include "camelot_ioctl.h"
#include "camelot_defs.h"
#include "camelot_reg.h"
#include "camelot_bits.h"
#include "camelot_ipfnat.h"

extern struct common_softc came_var;
extern int came_dmz_mode;
static unsigned long filter_control_value;
static int wan_pppoe_mode;

#ifdef CONFIG_MACH_FUJITSU_CAMELOT_HW_ACC
/* functions to add/delete chip IPF/NAT rules from within the driver */
extern void cm_create_pppoe_neigh(unsigned long ifindex, unsigned long ip,
				  unsigned long *ether);
extern void cm_delete_all_pppoe_neigh(void);
extern int cm_delete_pppoe_neigh(int ifindex);
extern int camelot_ipfnat_init(void);
extern void camelot_ipfnat_exit(void);
extern int cm_set_poll_period(time_t period);
extern int cm_set_expire_period(time_t period);
extern int cm_calculate_num_expiration_periods(void);
extern int cm_set_ip_addr(unsigned long index, unsigned long addr,
			  unsigned long mask);
extern int cm_set_qos_table(void *data);
#endif

static int camelot_ipfnat_mode;

int
camelot_get_ipfnat_mode(void)
{
	return camelot_ipfnat_mode;
}

void
camelot_set_ipfnat_mode(int on_off)
{
	camelot_ipfnat_mode = on_off;
}

int
camelot_get_pppoe_mode(void)
{
	return wan_pppoe_mode;
}

/*
 * NOTE:
 * if we kept shadow copies of the nat/forwarding table and the filter table
 * we could speed up the delete quite a bit.
 */

static u_int32_t global_match_log[ENTRY_NUM_NATMTHLOG];

/* reset global match log bit */
void
cm_reset_match_log(int index)
{
	int i, j;

	i = index / 32;
	j = index % 32;

	global_match_log[i] &= ~(1 << j);
}

void
cm_reset_match_log_all(void)
{
	int i;

	for (i = 0; i < ENTRY_NUM_NATMTHLOG; i++)
		global_match_log[i] = 0;
}

/* keep copy of match log in ram, because h/w version is reset-on-read */
int
cm_get_match_log(struct camelot_softc *sc, u_int32_t *match_log)
{
	int i;
	u_int32_t ml[ENTRY_NUM_NATMTHLOG];

	/*
	 * read current h/w match log
	 * (which resets the bits in h/w)
	 */
	read_NMTLOG(sc, ml);

	/* merge with our persistant copy */
	for (i = 0; i < ENTRY_NUM_NATMTHLOG; i++)
		global_match_log[i] |= ml[i];

	/* return merged version */
	for (i = 0; i < ENTRY_NUM_NATMTHLOG; i++)
		match_log[i] = global_match_log[i];

	return 0;
}

#ifdef NEED_FIX
/* fix for chip errata */
#define	NATIPF_TABLE_SIZE	(ENTRY_NUM_NATIPF/2)

/*
 * Due to the errata, two chip entries are required for each
 * NAT/IPF entry.  The paired entries map as follows:
 *
 *           NAT Table Entry
 * -------------------------------------
 * Zone 0 (0-7 are active, 15-8 are the matched pairs):
 *    ---------------------------
 *   0|                         |---------------+
 *    ---------------------------               |
 *   1|                         |----------+    |
 *    ---------------------------          |    |
 *   2|                         |-----+    |    |
 *    ---------------------------     |    |    |
 *               .                    |    |    |
 *               .                    |    |    |
 *               .                    |    |    |
 *               .                    |    |    |
 *    ---------------------------     |    |    |
 *  13|                         |-----+    |    |
 *    ---------------------------          |    |
 *  14|                         |----------+    |
 *    ---------------------------               |
 *  15|                         |---------------+
 *    ---------------------------
 * -------------------------------------
 * Zone 1 (16-23 are active, 31-24 are the matched pairs):
 *    ---------------------------
 *  16|                         |---------------+
 *    ---------------------------               |
 *  17|                         |----------+    |
 *    ---------------------------          |    |
 *  18|                         |-----+    |    |
 *    ---------------------------     |    |    |
 *               .                    |    |    |
 *               .                    |    |    |
 *               .                    |    |    |
 *               .                    |    |    |
 *    ---------------------------     |    |    |
 *  29|                         |-----+    |    |
 *    ---------------------------          |    |
 *  30|                         |----------+    |
 *    ---------------------------               |
 *  31|                         |---------------+
 *    ---------------------------
 * -------------------------------------
 * Etc.
 * 
 */

int
map_natipf_index(int index)
{
	int zone, offset;

	/* find the zone */
	zone = index / 8;
	/* find the entry */
	offset = index % 8;

	return (zone*16 + offset);
}

int
map_natipf_pair_index(int index)
{
	int zone, offset, paired_index;
	
	/* find the next highest zone */
	zone = (index / 8) + 1;
	/* find the paired entry offset */
	offset = index % 8;
	/* find the paired entry */
	paired_index = (zone*16 - 1) - offset;
	
	return paired_index;
}
#else
/* no fix */
#define	NATIPF_TABLE_SIZE	(ENTRY_NUM_NATIPF)
#define map_natipf_index(index)		(index)
#define map_natipf_pair_index(index)	(0)
#endif

/* get chip match log, marking inactive IPF entries as such */
static int
cm_get_real_time_match_log(struct camelot_softc *sc, int *match_log)
{
	u_int32_t ml[ENTRY_NUM_NATMTHLOG];
	int i, index;
	
	/*
	 * read current h/w match log
	 * (which resets the bits in h/w)
	 */
	read_NMTLOG(sc, ml);
	/* initialize log array to invalid */
	for (i = 0; i < ENTRY_NUM_NATIPF; i++) {
		match_log[i] = -1;
	}

	for (i = 0; i < NATIPF_TABLE_SIZE; i++) {
		/* map index into real index */
		index = map_natipf_index(i);

		/* read item 22, looking for non-zero entries */
		if (read_NATIPFTBL(sc, index, 22) != 0x0) {
			match_log[i] = ml[index/32] & (1 << (index%32));
		}
	}

	return 0;
}

int
cm_ioctl_add_nat_entry(struct camelot_softc *sc, NAT_IPF_TBL *nit)
{
	int i, index;

	for (i = 0; i < NATIPF_TABLE_SIZE; i++) {
		/* map index into real index */
		index = map_natipf_index(i);

		/* read item 22, looking for zero entries */
		if (read_NATIPFTBL(sc, index, 22) == 0x0)
			break;
	}

	if (i == NATIPF_TABLE_SIZE) {
		pr_debug("camelot: Nat ipf entry overflow!!\n");
		return -EBUSY;
	}

	cm_reset_match_log(i);

	set_NATIPFTBL(sc, index, nit);

#if 1
	read_dump_NTIPFTBL_entry(sc, index);
#endif

	nit->chip_index = i;

	/* handle chip errata */
	if ((index = map_natipf_pair_index(i))) {
		set_empty_NATIPFTBL(sc, index);
	}

	return 0;
}

/*
 * find an entry in the NAT/IP forwarding table and delete it
 */
int
cm_ioctl_del_nat_entry(struct camelot_softc *sc, NAT_IPF_TBL *nit)
{
	int index;
	u_int32_t int_ip, ext_ip;
	u_short int_port, ext_port;

	/* map index into real index */
	index = map_natipf_index(nit->chip_index);

	/* make sure the entry is active */
	if (read_NATIPFTBL(sc, index, 22) == 0x0) {
		printk(KERN_ERR "camelot: Invalid Nat index specified %d (%d)\n",
		       nit->chip_index, index);
		return -EINVAL;
	}

	/* read entry */
	int_ip = read_NATIPFTBL(sc, index, 0);
	ext_ip = read_NATIPFTBL(sc, index, 1);
	int_port = read_NATIPFTBL(sc, index, 12);
	ext_port = read_NATIPFTBL(sc, index, 13);

	/* sanity check */
	if (int_ip == nit->ipv4addr_inter &&
	    ext_ip == nit->ipv4addr_exter &&
	    int_port == nit->portnum_inter &&
	    ext_port == nit->portnum_exter)
	{
		setnat_root(sc, index, 22, 0x00000000);
		came_var.tbl_match[nit->chip_index] = 0;
		cm_reset_match_log(nit->chip_index);

		/* handle chip errata */
		if ((index = map_natipf_pair_index(nit->chip_index))) {
			setnat_root(sc, index, 22, 0x00000000);
		}
	}
	else {
		printk(KERN_ERR "camelot: can't find Nat ipf entry to delete!!\n");
		return -EINVAL;
	}

	return 0;
}

int
cm_ioctl_add_dat_entry(struct camelot_softc *sc, DA_TBL *dat)
{
	int index;

	index = dat_get_index(sc);

	if (index < 0) {
		printk(KERN_ERR "camelot: DAT entry overflow!!\n");
		return -EBUSY;
	}

	set_DATBL(sc, index, dat);

#if 1
	read_dump_DATBL_entry(sc, index);
#endif

	/* handle chip errata */

	return 0;
}

/*
 * find an entry in the DAT table and delete it
 */
int
cm_ioctl_del_dat_entry(struct camelot_softc *sc, DA_TBL *dat)
{
	int i, val_ver;
#if 0
	int cmd;
	int addr[2] = { 0 }, mask[2] = { 0 };
#endif
	DA_TBL tmp_dat;

	/* grab valid entry bit mask */
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, 0x00000000);
	val_ver = (*sc->sc_read)(sc, TBLDIRECT_VLDIPV_OFFSET);
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, HOST_INTEN);
	
	for (i = 0; i < ENTRY_NUM_DAT; i++) {
		/* don't bother comparing invalid entries */
		if ( (val_ver & (1 << i)) == 0 ) {
			pr_debug("Skipping invalid entry %d\n", i);
			continue;
		}

		if (read_DATBL_entry(sc, i, &tmp_dat)) {
			printk(KERN_ERR "Error reading DAT entry\n");
			continue;
		}
		
		/* compare versions */
		if (tmp_dat.ipv6 != dat->ipv6) {
			pr_debug("Skipping entry %d; "
			         "different IP versions (%d vs %d)\n",
			       i, tmp_dat.ipv6, dat->ipv6);
			continue;
		}

		if (tmp_dat.subnet_addr[0] != dat->subnet_addr[0]) {
			pr_debug("Skipping entry %d; "
			         "different addr_1 (%d vs %d)\n",
			       i, tmp_dat.subnet_addr[0], dat->subnet_addr[0]);
			continue;
		}
			
		if (tmp_dat.subnet_mask[0] != dat->subnet_mask[0]) {
			pr_debug("Skipping entry %d; "
			         "different mask_1 (%d vs %d)\n",
			       i, tmp_dat.subnet_mask[0], dat->subnet_mask[0]);
			continue;
		}

		if (dat->ipv6) {
			if (tmp_dat.subnet_addr[1] != dat->subnet_addr[1]) {
				pr_debug("Skipping entry %d; "
					 "different addr_2 (%d vs %d)\n",
				       i, tmp_dat.subnet_addr[1],
				       dat->subnet_addr[1]);
				continue;
			}

			if (tmp_dat.subnet_mask[1] != dat->subnet_mask[1]) {
				pr_debug("Skipping entry %d; "
					 "different mask_2 (%d vs %d)\n",
				       i, tmp_dat.subnet_mask[1],
				       dat->subnet_mask[1]);
				continue;
			}
		}
	}

	if (i >= ENTRY_NUM_DAT) {
		printk(KERN_ERR "camelot: "
		       "Could not find DAT entry to delete\n");
		return -1;
	}

	pr_debug("DAT DEL: found DAT entry #%d\n", i);
	read_dump_DATBL_entry(sc, i);

	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, 0x00000000);
	val_ver = (*sc->sc_read)(sc, TBLDIRECT_VLDIPV_OFFSET);

	/* clear the version and validity bits */
	val_ver = val_ver & (~ (1 << (i + 4)) | (1 << i));
	
	(*sc->sc_write)(sc, TBLDIRECT_VLDIPV_OFFSET, val_ver);
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, HOST_INTEN);
	
	/* handle chip errata */

	return 0;
}

int
cm_ioctl_add_qos_entry(struct camelot_softc *sc, QOS_TBL_ENTRY *qos)
{
	int i;

	for (i = 0; i < ENTRY_NUM_QOS; i++) {
		/*
		 * read item 11 from the appropriate table (DMZ or
		 * WAN), looking for zero entries
		*/
		if (read_QOSTBL(sc, qos->wan_port, i, 11) == 0x0)
			break;
	}

	if (i == ENTRY_NUM_QOS) {
		printk(KERN_ERR "camelot: QOS %s entry overflow!!\n",
		       (qos->wan_port == 1) ? "WAN" : "DMZ");
		return -EBUSY;
	}

	set_QOSTBL(sc, qos->wan_port, i, qos);

#if 1
	read_dump_QTALL(sc, qos->wan_port);
#endif

	/* handle chip errata */
	
	return 0;
}

/*
 * find an entry in the QOS table and delete it
 */
int
cm_ioctl_del_qos_entry(struct camelot_softc *sc, QOS_TBL_ENTRY *qos)
{
	int i, id;

	for (i = 0; i < ENTRY_NUM_QOS; i++) {
		u_int32_t src_ip[4], dst_ip[4], ports;
		u_short src_port, dst_port, flowlabel, tos;

		/* look for active entries in appropriate table */
		if (read_QOSTBL(sc, qos->wan_port, i, 11) == 0x0)
			continue;

		/* read entry */
		src_ip[0] = read_QOSTBL(sc, qos->wan_port, i, 0);
		src_ip[1] = read_QOSTBL(sc, qos->wan_port, i, 1);
		src_ip[2] = read_QOSTBL(sc, qos->wan_port, i, 2);
		src_ip[3] = read_QOSTBL(sc, qos->wan_port, i, 3);
		dst_ip[0] = read_QOSTBL(sc, qos->wan_port, i, 4);
		dst_ip[1] = read_QOSTBL(sc, qos->wan_port, i, 5);
		dst_ip[2] = read_QOSTBL(sc, qos->wan_port, i, 6);
		dst_ip[3] = read_QOSTBL(sc, qos->wan_port, i, 7);
		dst_ip[3] = read_QOSTBL(sc, qos->wan_port, i, 7);
		tos = read_QOSTBL(sc, qos->wan_port, i, 8);
		flowlabel = read_QOSTBL(sc, qos->wan_port, i, 9);
		ports = read_QOSTBL(sc, qos->wan_port, i, 10);
		src_port = ports & 0xFFFF;
		dst_port = (ports >> 16) & 0xFFFF;

		if (src_ip[3] == qos->ipv6addr_src[3] &&
		    src_ip[2] == qos->ipv6addr_src[2] &&
		    src_ip[1] == qos->ipv6addr_src[1] &&
		    src_ip[0] == qos->ipv6addr_src[0] &&
		    dst_ip[3] == qos->ipv6addr_dst[3] &&
		    dst_ip[2] == qos->ipv6addr_dst[2] &&
		    dst_ip[1] == qos->ipv6addr_dst[1] &&
		    dst_ip[0] == qos->ipv6addr_dst[0] &&
		    tos == qos->tos&&
		    flowlabel == qos->flowlabel &&
		    src_port == qos->portnum_src &&
		    dst_port == qos->portnum_dst)
		{
#if 0
			pr_debug("delete #%d\n", i);
			read_dump_QOSTBL_entry(sc, qos->wan_port, i);
#endif

			id = ((qos->wan_port & 0x1) << 3) | (i & 0x7);
			setqos_root(sc, id, 11, 0x00000000);

			/* handle chip errata */
			break;
		}
	} 

	if (i == ENTRY_NUM_QOS) {
		printk(KERN_ERR "camelot: can't find QOS entry to delete!!\n");
		return -EINVAL;
	}

	return 0;
}

int
cm_ioctl_add_l2_entry(struct camelot_softc *sc, unsigned long *v)
{
	int i;
	unsigned long data;

	/* find an unused entry */
	for (i = 0; i < L2ENTRY_NUM; i++) {
		data = (*sc->sc_read)(sc, LKUP_TBL_0o_OFFSET + (i * 8));
		if ((data & 0x1) == 0)
			break;
	}

	if (i == L2ENTRY_NUM) {
		printk(KERN_ERR "camelot: L2 table overflow!!\n");
		return -EBUSY;
	}

	set_l2_entry(sc, i, v[0], v[1]);


	/* handle chip errata */
	
	return 0;
}

int
cm_ioctl_del_l2_entry(struct camelot_softc *sc, unsigned long *v)
{
	return delete_l2_entry(sc, v[0], v[1]);
}

int
cm_ioctl_lookup_l2_intf(struct camelot_softc *sc, unsigned long *v)
{
	int intf = 0, ret;
	
	ret = lookup_l2_intf(sc, v[0], v[1], &intf);
	if (ret) {
		printk(KERN_ERR "Error looking up interface\n");
	}
	else {
		/* set the interface in the v[2] element */
		v[2] = intf;
	}
	
	return ret;
}

int
cm_ioctl_flush_ipf(struct camelot_softc *sc)
{
	int i;

	for (i = 0; i < ENTRY_NUM_NATIPF; i++) {

		came_var.tbl_match[i] = 0;
		cm_reset_match_log(i);
		
		/* look for active entries */
		if (read_NATIPFTBL(sc, i, 22) == 0x0)
			continue;
		setnat_root(sc, i, 22, 0x0);
	} 

	return 0;
}


int
cm_ioctl_add_filter_entry(struct camelot_softc *sc, int table, FLT_TBL *ft)
{
	int i;

	for (i = 0; i < ENTRY_NUM_L34; i++) {
		/* scan table front to back */
		if (!(read_L3_4_filter(sc, i, table, 10) & 0x00000001))
			break;
	}

	if (i == ENTRY_NUM_L34) {
		printk(KERN_ERR "camelot: L3/L4 filter entry overflow!!\n");
		return -EBUSY;
	}
	set_L3_4(sc, i, table, ft);

#if 0
	pr_debug("Add");
	read_dump_FLT(sc, table, i);
#endif

	return 0;
}

/*
 * find an entry in the filter table and delete it
 */
int
cm_ioctl_del_filter_entry(struct camelot_softc *sc,
			  int table,
			  struct tbl_fef *fd)
{
	int i, j;

	for (i = 0; i < ENTRY_NUM_L34; i++) {
		u_int32_t t_dst, t_src, t_port;
		u_short t_dport, t_sport;

		/* search from back to front */
		j = ENTRY_NUM_L34 - i - 1;

		/* look for active entries */
		if (read_L3_4_filter(sc, j, table, 10) == 0x0)
			continue;

		/* read entry */
		t_dst = read_L3_4_filter(sc, j, table, 4);
		t_src = read_L3_4_filter(sc, j, table, 0);

		t_port = read_L3_4_filter(sc, j, table, 8);
		t_dport = t_port >> 16;
		t_sport = t_port & 0x0000ffff;

		if (t_dst == fd->ip_nat &&
		    t_src == fd->ip_dst &&
		    t_dport == fd->port_nat &&
		    t_sport == fd->port_dst)
		{
#if 0
			pr_debug("Del");
			read_dump_FLT(sc, table, j);
#endif
			setflt_root(sc, j, 10, table, 0x0);
			break;
		}
	}


	if (i == ENTRY_NUM_L34) {
		printk(KERN_ERR 
		       "camelot: can't find filter entry to delete!!\n");
		return -EINVAL;
	}

	return 0;
}

int
cm_ioctl_flush_filters(struct camelot_softc *sc, int table)
{
	int i, tmp;

	if ((table == 0) || (table == 2)) { /* L3/L4 Filter Tables */
		for (i = 0; i < ENTRY_NUM_L34; i++) {
			/* reset the filter count to 0; read-clear */
			if (table == 2) { /* out-side */
				tmp = (*sc->sc_read)(sc,
						     (FILT_COUNT_1_0_OFFSET +
						      i*4));
			}
			else { /* in-side */
				tmp = (*sc->sc_read)(sc,
						     (FILT_COUNT_0_0_OFFSET +
						      i*4));
			}

			/* look for active entries */
			if (read_L3_4_filter(sc, i, table, 10) == 0x0)
				continue;
			/* mark entry invalid */
			setflt_root(sc, i, 10, table, 0x0);
		}
	} else { /* Protocol Type Tables */
		for (i = 0; i < ENTRY_NUM_PRT; i++) {
			/* look for active entries */
			if (read_PRTTB(sc, i, table) == 0x0)
				continue;
			/* mark entry invalid */
			setflt_root(sc, i, 12, table, 0x0);
		}

	}

	return 0;
}

int
cm_ioctl_del_unmatched_entry(struct camelot_softc *sc,
			     u_int32_t ctl22, int tindex)
{
	int index;
	u_long ip_src, ip_dst, ip_nat;
	u_short port_src, port_dst, port_nat;

	
	/* map index into real index */
	index = map_natipf_index(tindex);

	ip_src = (u_long)read_NATIPFTBL(sc, index, 0);
	ip_dst = (u_long)read_NATIPFTBL(sc, index, 1);
	ip_nat = (u_long)read_NATIPFTBL(sc, index, 2);
	port_src = (u_short)read_NATIPFTBL(sc, index, 12);
	port_dst = (u_short)read_NATIPFTBL(sc, index, 13);
	port_nat = (u_short)read_NATIPFTBL(sc, index, 14);

#if 0
	pr_debug("Del");
	read_dump_NTIPFTBL_entry(sc, index);
#endif

	setnat_root(sc, index, 22, 0x00000000);
	came_var.tbl_match[tindex] = 0;
	cm_reset_match_log(tindex);

	/* handle chip errata */
	if ((index = map_natipf_pair_index(index))) {
		setnat_root(sc, index, 22, 0x00000000);
	}

	return 0;
}

int
cm_ioctl_del_unmatched(struct camelot_softc *sc)

{
	int i, j, tindex, index;
	int thresh;
	u_int32_t ctl22;
	u_int32_t match_log[ENTRY_NUM_NATMTHLOG];

	cm_get_match_log(sc, match_log);

	for (i = 0; i < ENTRY_NUM_NATMTHLOG; i++) {
		for (j = 0; j < 32; j++){

			tindex = 32*i + j;

			/* map index into real index */
			index = map_natipf_index(tindex);
			ctl22 = read_NATIPFTBL(sc, index, 22);

			if (ctl22 & 0x100000) {
				if ((match_log[i] >> j) & 0x00000001) {
					/* matched */
					came_var.tbl_match[tindex] = 0;
				} else {
					/* unmatched */
					came_var.tbl_match[tindex]++;

					thresh = (ctl22 & 0x002000) ?
						THRESH_DEL_UNMATCH_UDP :
						THRESH_DEL_UNMATCH_TCP;

					if (came_var.tbl_match[tindex]> thresh)
					{
						cm_ioctl_del_unmatched_entry(
							sc, ctl22, tindex);
					}
				}
			}
		}
	}

	return 0;
}

static int
set_dmz_mode(struct camelot_softc *sc, int v)
{
	int data;
	
	came_dmz_mode = v;
	/* enable/disable DMZ bit in Look Up Control Register */
	data = (*sc->sc_read)(sc, LKUP_CONTRL_OFFSET);
	if (v)
		data |= DMZ_EN;
	else
		data &= ~DMZ_EN;
	(*sc->sc_write)(sc, LKUP_CONTRL_OFFSET, data);
	
	return 0;
}

static int
set_filter_control_value(struct camelot_softc *sc, int v)
{
	filter_control_value = v;
	set_FLCNT(sc, v);
}

static int
get_filter_control_value(struct camelot_softc *sc)
{
	return filter_control_value;
}


int
cm_enable_pppoe(struct camelot_softc *sc)
{
	int fc_val;

	fc_val = get_filter_control_value(sc) & ~fc_pppoe_bits;

	fc_val |=
		fc_pppoe_sess_ipv4_pass_output |
		fc_pppoe_sess_ipv4_pass_input |
		/*fc_pppoe_sess_ipv6_drop_output | */
		fc_pppoe_sess_noip_output |
		fc_pppoe_sess_ipv6_drop_input |
		/*fc_pppoe_sess_noip_input | */
		0;

	set_filter_control_value(sc, fc_val);
	wan_pppoe_mode = 1;

	return 0;
}

int
cm_disable_pppoe(struct camelot_softc *sc)
{
	int fc_val;
#ifdef CONFIG_MACH_FUJITSU_CAMELOT_HW_ACC
	if (camelot_ipfnat_mode)
		cm_delete_all_pppoe_neigh();
#endif

	fc_val = get_filter_control_value(sc) & ~fc_pppoe_bits;

	fc_val |=
		fc_pppoe_sess_ipv4_drop_output |
		fc_pppoe_sess_ipv4_drop_input |
		fc_pppoe_sess_ipv6_drop_output |
		/*fc_pppoe_sess_noip_output | */
		fc_pppoe_sess_ipv6_drop_input |
		/*fc_pppoe_sess_noip_input | */
		0; 

	set_filter_control_value(sc, fc_val);
	wan_pppoe_mode = 0;

	return 0;
}

int
cm_disable_filtering(struct camelot_softc *sc)
{
	int fc_val;

	fc_val = get_filter_control_value(sc) & ~fc_filter_bits;

	set_filter_control_value(sc, fc_val);

	return 0;
}

int
cm_set_pppoe_addr(unsigned long addr, unsigned long server, unsigned long mask,
		  unsigned long mac1, unsigned long mac2, unsigned long mac3,
		  unsigned long mac4, unsigned long mac5, unsigned long mac6,
		  unsigned long ifindex)
{
	unsigned long pppoe_server_ether[6];

	pppoe_server_ether[0] = mac1;
	pppoe_server_ether[1] = mac2;
	pppoe_server_ether[2] = mac3;
	pppoe_server_ether[3] = mac4;
	pppoe_server_ether[4] = mac5;
	pppoe_server_ether[5] = mac6;
#ifdef CONFIG_MACH_FUJITSU_CAMELOT_HW_ACC
	cm_create_pppoe_neigh(ifindex, addr, pppoe_server_ether);
#endif

	return 0;
}

int
cm_ioctl_set_pppoe(struct camelot_softc *sc)
{
	int i;

	for (i = 0; i < LEN_PPPOE; i++) {
		(*sc->sc_write)(sc, PPPOE_HEADER_0_OFFSET + 4*i,
				came_var.pppoe_info[i]);
	}

	if (cm_enable_pppoe(sc))
		return -1;

	return 0;
}

int
cm_ioctl_del_pppoe(struct camelot_softc *sc, unsigned long sid, int ifindex)
{
	int i;

	for (i = 0; i < LEN_PPPOE; i++) {
		pr_debug("pppoe_info[%d] = 0x%x; sid = 0x%lx\n",
			 i, came_var.pppoe_info[i], sid);
		if (came_var.pppoe_info[i] == sid) {
			pr_debug("Found session in slot %d\n", i);
			came_var.pppoe_info[i] = 0;
			(*sc->sc_write)(sc, PPPOE_HEADER_0_OFFSET + 4*i, 0x0);

#ifdef CONFIG_MACH_FUJITSU_CAMELOT_HW_ACC
			if (cm_delete_pppoe_neigh(ifindex)) {
				pr_debug("Could not delete pppoe neighbor "
					 "with dev %d\n", ifindex);
			}
#endif
			break;
		}
	}
	
	return 0;
}

int
cm_ioctl_flush_pppoe(struct camelot_softc *sc)
{
	int i;

	for (i = 0; i < LEN_PPPOE; i++) {
		came_var.pppoe_info[i] = 0;
		(*sc->sc_write)(sc, PPPOE_HEADER_0_OFFSET + 4*i, 0x0);
	}

	if (cm_disable_pppoe(sc))
		return -1;

	return 0;
}

int
cm_ioctl_set_tunnel(struct camelot_softc *sc)
{
	int i;

	for (i = 0; i < LEN_TUNNEL; i++){
		(*sc->sc_write)(sc, V4_HEADER_0_0_OFFSET + 4*i,
				came_var.tunnel_info[i]);
	}

	return 0;
}

int
cm_ioctl_flush_tunnels(struct camelot_softc *sc)
{
	int i;

	came_var.tunnel_info[3] = 0;
	came_var.tunnel_info[4] = 0;

	for (i = 0; i < LEN_TUNNEL; i++){
		(*sc->sc_write)(sc, V4_HEADER_0_0_OFFSET + 4*i,
				came_var.tunnel_info[i]);
	}

	return 0;
}

/* debug read routines */

static int
debug_read_ipfnat(struct camelot_softc *sc, struct camelot_debug_read *pdr)
{
	int i;
	unsigned long v[23];

	for (i = 0; i < 23; i++){
		if (i == 3 || i == 15)
			continue;

		v[i] = read_NATIPFTBL(sc, pdr->index, i);
	}

	if (copy_to_user(pdr->data, (char *)v, sizeof(v)))
	{
		return -EINVAL;
	}

	return 0;
}

static int
debug_read_qos(struct camelot_softc *sc, struct camelot_debug_read *pdr)
{
	int idx, wan_port;
	QOS_TBL_ENTRY qos;

	/* index is overloaded with wan_port in 4th bit */
	wan_port = (pdr->index >> 3) & 0x1;
	idx = pdr->index & 0x7; /* 3 bits */

	if (read_QOSTBL_entry(sc, wan_port, idx, &qos)) {
		printk(KERN_ERR "Error reading %s QOS entry %d\n",
		       wan_port ? "WAN" : "DMZ", idx);
		return -EINVAL;
	}

	if (copy_to_user(pdr->data, (char *)&qos, sizeof(QOS_TBL_ENTRY)))
	{
		return -EINVAL;
	}

	return 0;
}

static int
debug_read_dat(struct camelot_softc *sc, struct camelot_debug_read *pdr)
{
	DA_TBL dat;

	if (read_DATBL_entry(sc, pdr->index, &dat)) {
		printk(KERN_ERR "Error reading DAT entry %d\n", pdr->index);
		return -EINVAL;
	}

	if (copy_to_user(pdr->data, (char *)&dat, sizeof(DA_TBL)))
	{
		return -EINVAL;
	}

	return 0;
}

static int
debug_read_filter(struct camelot_softc *sc, struct camelot_debug_read *pdr)
{
	int i, table, id, size;
	unsigned long v[12];

	table = pdr->index >> 8;
	id = pdr->index & 0xff;

	switch (table) {
	case 0:
	case 2:
		for (i = 0; i < 12; i++) {
			v[i] = read_L3_4_filter(sc, id, table, i);
		}
		size = sizeof(v);
		break;

	case 1:
	case 3:
		v[0] = read_PRTTB(sc, id, table);
		size = sizeof(v[0]);
		break;
	default:
		return -1;
	}

	if (copy_to_user(pdr->data, (char *)v, size))
	{
		return -EINVAL;
	}

	return 0;
}

static int
debug_read_filter_modemask(struct camelot_softc *sc,
			   struct camelot_debug_read *pdr)
{
	unsigned long v[2];

	v[0] = (*sc->sc_read)(sc, FILT_CONTRL_OFFSET);
	v[1] = (*sc->sc_read)(sc, FILT_SUBNET_OFFSET);

	if (copy_to_user(pdr->data, (char *)v, sizeof(v)))
	{
		return -EINVAL;
	}

	return 0;
}

static int
debug_read_filter_count(struct camelot_softc *sc,
			struct camelot_debug_read *pdr)
{
	int i, table;
	unsigned long v[ENTRY_NUM_L34];

	table = pdr->index;

	for (i = 0; i < ENTRY_NUM_L34; i++) {
		v[i] = read_FLCNTV(sc, table, i);
	}

	if (copy_to_user(pdr->data, (char *)v, sizeof(v)))
	{
		return -EINVAL;
	}

	return 0;
}

static u_int flt_log_out[ENTRY_NUM_LOG][16];
static u_int flt_log_in[ENTRY_NUM_LOG][16];

static int
debug_read_filter_log(struct camelot_softc *sc,
		      struct camelot_debug_read *pdr)
{
	int status;

	status = 0;
	memset((char *)flt_log_in, 0, sizeof(flt_log_in));
	memset((char *)flt_log_out, 0, sizeof(flt_log_out));

	read_FLTLOG(sc, flt_log_in, flt_log_out, &status);

	/* return status */
	if (copy_to_user(pdr->data, (char *)&status, sizeof(status)))
	{
		return -EINVAL;
	}

	/* input filter log */
	pdr->data += sizeof(int);
	if (copy_to_user(pdr->data, (char *)flt_log_in, sizeof(flt_log_in))) {
		return -EINVAL;
	}

	/* output filter log */
	pdr->data += sizeof(flt_log_in);
	if (copy_to_user(pdr->data, (char *)flt_log_out, sizeof(flt_log_out)))
	{
		return -EINVAL;
	}

	return 0;
}

static int
debug_read_tunnel(struct camelot_softc *sc, struct camelot_debug_read *pdr)
{
	int i, offset;
	unsigned long v[4];

	switch (pdr->index) {
	case 0:
		offset = V4_HEADER_0_0_OFFSET;
		break;
	case 1:
		offset = V4_HEADER_1_0_OFFSET;
		break;
	case 2:
		offset = V4_HEADER_2_0_OFFSET;
		break;
	case 3:
		offset = V4_HEADER_3_0_OFFSET;
		break;
	default:
		return -1;
	}

	for (i = 0; i < 4; i++){
		v[i] = (*sc->sc_read)(sc, offset + 4*i);
	}

	if (copy_to_user(pdr->data, (char *)v, sizeof(v)))
	{
		return -EINVAL;
	}

	return 0;
}

static int
debug_read_pppoe(struct camelot_softc *sc, struct camelot_debug_read *pdr)
{
	int i;
	unsigned long v[LEN_PPPOE];

	for (i = 0; i < LEN_PPPOE; i++) {
		v[i] = (*sc->sc_read)(sc, PPPOE_HEADER_0_OFFSET + 4*i);
	}

	if (copy_to_user(pdr->data, (char *)v, sizeof(v)))
	{
		return -EINVAL;
	}

	return 0;
}

static int
debug_read_L2TBL(struct camelot_softc *sc, struct camelot_debug_read *pdr)
{
	int i;
	unsigned long v[2 * L2ENTRY_NUM + 1];

	v[0] = (*sc->sc_read)(sc, LKUP_CONTRL_OFFSET);

	for (i = 0; i < L2ENTRY_NUM; i++) {
		v[i+1] = (*sc->sc_read)(sc, LKUP_TBL_0o_OFFSET + (i * 8));
		v[i+1+L2ENTRY_NUM] =
			(*sc->sc_read)(sc, LKUP_TBL_0e_OFFSET + (i * 8));
	}

	if (copy_to_user(pdr->data, (char *)v, sizeof(v)))
	{
		return -EINVAL;
	}

	return 0;
}

/*
 * Process an ioctl request.
 */
int
camelot_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
	struct cm_softc *cm = (struct cm_softc *)dev->priv;
	struct camelot_softc *sc = &cm->camelot;
	char *data = (char *)rq->ifr_data;
	NAT_IPF_TBL nit;
	DA_TBL dat_entry;
	QOS_TBL_ENTRY qos_entry;
	FLT_TBL ft;
	struct tbl_fef fd;
	struct camelot_debug_read dr;
	struct camelot_ioctl_cmd ic;
	int error = 0;
	unsigned long v[2];
	unsigned long vals[3];
	unsigned long pppoe_vals[10];
	unsigned long set_l2_v[3];
	u_int32_t match_log[ENTRY_NUM_NATMTHLOG];
	int rt_match_log[ENTRY_NUM_NATIPF];

	switch (cmd) {
	case CM_IPFNAT:
		if (copy_from_user((char *)&ic, data, sizeof(ic)))
			return -EINVAL;
		
		switch(ic.what) {
		case CM_SET_IPFNAT:
			if (copy_from_user((char *)&v[0], ic.data,
					   sizeof(v[0])))
				return -EINVAL;

			pr_debug("camelot: set ipfnat mode %d\n", v[0]);

#ifdef CONFIG_MACH_FUJITSU_CAMELOT_HW_ACC
			if (v[0])
				camelot_ipfnat_init();
			else
				camelot_ipfnat_exit();
#endif
			break;
		case CM_SET_CT_TIMEOUTS:
			if (copy_from_user((char *)&v, ic.data, sizeof(v)))
				return -EINVAL;

			pr_debug("camelot: set timeouts: poll %d; expire %d\n",
				 v[0], v[1]);

			if (!camelot_ipfnat_mode)
				return -EACCES;
#ifdef CONFIG_MACH_FUJITSU_CAMELOT_HW_ACC
			/* set poll/expiration periods */
			if (cm_set_poll_period(v[0]))
				return -EINVAL;
			if (cm_set_expire_period(v[1]))
				return -EINVAL;
			if (cm_calculate_num_expiration_periods())
				return -EINVAL;
#endif
			break;
		case CM_SET_IPADDR:
			if (copy_from_user((char *)&vals, ic.data,
					   sizeof(vals)))
				return -EINVAL;

			pr_debug("camelot: SET_IPADDR idx %d; addr 0x%08x\n",
				 vals[0], vals[1]);

			if (!camelot_ipfnat_mode)
				return -EACCES;
#ifdef CONFIG_MACH_FUJITSU_CAMELOT_HW_ACC			
			error = cm_set_ip_addr(vals[0], vals[1], vals[2]);
#endif
			break;
		case CM_SET_PPPOEADDR:
			if (copy_from_user((char *)&pppoe_vals, ic.data,
					   sizeof(pppoe_vals)))
				return -EINVAL;

			pr_debug("camelot: SET_PPPPOEADDR addr 0x%08x, svr 0x%08x\n",
				 vals[0], vals[1]);

			if (!camelot_ipfnat_mode)
				return -EACCES;
#ifdef CONFIG_MACH_FUJITSU_CAMELOT_HW_ACC
			error = cm_set_pppoe_addr(pppoe_vals[0], pppoe_vals[1],
						  pppoe_vals[2], pppoe_vals[3],
						  pppoe_vals[4], pppoe_vals[5],
						  pppoe_vals[6], pppoe_vals[7],
						  pppoe_vals[8], pppoe_vals[9]);
#endif
			break;
		case CM_SET_NAT:
		case CM_SET_IPF:
			if (copy_from_user((char *)&nit, ic.data,
					   sizeof(NAT_IPF_TBL)))
				return -EINVAL;

			error = cm_ioctl_add_nat_entry(sc, &nit);
			if (copy_to_user((char *)ic.data, (char *)&nit,
					 sizeof(NAT_IPF_TBL))) {
				return -EINVAL;
			}
			break;
		case CM_DEL_NAT:
		case CM_DEL_IPF:
			if (copy_from_user((char *)&nit, ic.data,
					   sizeof(NAT_IPF_TBL)))
				return -EINVAL;

			error = cm_ioctl_del_nat_entry(sc, &nit);
			break;
		default:
			return -EINVAL;
		}
		break;

	case CM_FLT:
		if (copy_from_user((char *)&ic, data, sizeof(ic)))
			return -EINVAL;
		
		switch(ic.what) {
		case CM_SET_FLT_OUT:
			if (copy_from_user((char *)&ft, ic.data,
					   sizeof(FLT_TBL)))
				return -EINVAL;

			error = cm_ioctl_add_filter_entry(sc, 2, &ft);
			break;
		case CM_SET_FLT_IN:
			if (copy_from_user((char *)&ft, ic.data,
					   sizeof(FLT_TBL)))
				return -EINVAL;

			error = cm_ioctl_add_filter_entry(sc, 0, &ft);
			break;
		case CM_DEL_FLT_OUT:
			if (copy_from_user((char *)&fd, ic.data,
					   sizeof(struct tbl_fef)))
				return -EINVAL;

			error = cm_ioctl_del_filter_entry(sc, 2, &fd);
			break;
		case CM_DEL_FLT_IN:
			if (copy_from_user((char *)&fd, ic.data,
					   sizeof(struct tbl_fef)))
				return -EINVAL;

			error = cm_ioctl_del_filter_entry(sc, 0, &fd);
			break;
		case CM_SET_FLT_MODE:
			if (copy_from_user((char *)&v, ic.data, sizeof(v)))
				return -EINVAL;

			pr_debug(
			       "camelot: set filter mode %08lx, mask %08lx\n",
			       v[0], v[1]);
			
			set_filter_control_value(sc, v[0]);
			break;
		case CM_SET_FLT_MASK:
			if (copy_from_user((char *)&v, ic.data, sizeof(v)))
				return -EINVAL;

			pr_debug(
			       "camelot: set filter mask %08lx\n",
			       v[0]);
			
			set_SUBNM(sc, v[0]);
			break;
		default:
			return -EINVAL;
		}
		break;

	case CM_DEL_UNMATCH:
		error = cm_ioctl_del_unmatched(sc);
		break;

	case CM_PPPOE:
		if (copy_from_user((char *)&ic, data, sizeof(ic)))
			return -EINVAL;
		
		switch(ic.what) {
		case CM_PPPOE_SET:
			if (copy_from_user((char *)&came_var.pppoe_info[0],
					   ic.data,
					   LEN_PPPOE*sizeof(u_int32_t)))
			{
				return -EINVAL;
			}
			
			pr_debug(
				 "CM_PPPOE_SET %08x %08x %08x %08x\n",
				 came_var.pppoe_info[0],
				 came_var.pppoe_info[1],
				 came_var.pppoe_info[2],
				 came_var.pppoe_info[3]);
			
			error = cm_ioctl_set_pppoe(sc);
			break;
		case CM_PPPOE_DEL:
			if (copy_from_user((char *)&v, ic.data, sizeof(v)))
				return -EINVAL;
			
			pr_debug("CM_PPPOE_DEL %08x %d\n", v[0], v[1]);
			error = cm_ioctl_del_pppoe(sc, v[0], v[1]);
			break;
		default:
			return -EINVAL;
		}
		break;

	case CM_TUNNEL:
		if (copy_from_user((char *)&came_var.tunnel_info[3],
				   data, 2*sizeof(u_int32_t)))
		{
			return -EINVAL;
		}

		came_var.tunnel_info[0] =
			((IPVERSION << 4) & 0x000000f0) |
			(0x5 & 0x0000000f);

		came_var.tunnel_info[1] = 0x00000000;

		came_var.tunnel_info[2] =
			((IPPROTO_IPV6 << 8) & 0x0000ff00) |
			(0x1e & 0x000000ff);

		pr_debug(
		       "CM_TUNNEL %08x %08x %08x %08x %08x\n",
		       came_var.tunnel_info[0], came_var.tunnel_info[1],
		       came_var.tunnel_info[2], came_var.tunnel_info[3],
		       came_var.tunnel_info[4]);

		error = cm_ioctl_set_tunnel(sc);
		break;

	case CM_FLUSH:
		switch (rq->ifr_metric) {
		case FLUSH_FLT:
			error = cm_ioctl_flush_filters(sc, 2);
			error = cm_ioctl_flush_filters(sc, 0);
			
			error = cm_ioctl_flush_filters(sc, 3);
			error = cm_ioctl_flush_filters(sc, 1);

			/* clear filter counts */
			reset_filter_match_counts(sc);
			/* turn off filtering */
			cm_disable_filtering(sc);
			break;
		case FLUSH_FLT_COUNTS:
			reset_filter_match_counts(sc);
			break;
		case FLUSH_IPF:
		case FLUSH_NAT:
			error = cm_ioctl_flush_ipf(sc);
			break;
		case FLUSH_PPPOE:
			error = cm_ioctl_flush_pppoe(sc);
			break;
		case FLUSH_TUNNEL:
			error = cm_ioctl_flush_tunnels(sc);
			break;
		case FLUSH_DAT:
			initial_DATBL(sc);
			break;
		case FLUSH_QOS:
			initial_QOSTBL(sc);
			break;
		case FLUSH_L2_ALL:
			pr_debug(
			       "camelot: clear all L2 table entries\n");
			clear_all_l2_entries(sc);
			break;
		case FLUSH_L2_DYN:
			pr_debug(
			       "camelot: clear dynamic L2 table entries\n");
			clear_dynamic_l2_entries(sc);
			break;
		default:
			return -EINVAL;
		}
		break;

	case CM_MATCH:
		if (copy_from_user((char *)&ic, data, sizeof(ic)))
			return -EINVAL;
		
		switch(ic.what) {
		case CM_GET_MATCH:
			cm_get_match_log(sc, match_log);
			
			if (copy_to_user((char *)ic.data, (char *)&match_log,
					 sizeof(match_log)))
			{
				return -EINVAL;
			}
			break;
		
		case CM_CLEAR_MATCH:
			cm_reset_match_log_all();
			break;
		
		case CM_GET_MATCH_RT:
			cm_get_real_time_match_log(sc, rt_match_log);
			
			if (copy_to_user((char *)ic.data,
					 (char *)&rt_match_log,
					 sizeof(rt_match_log)))
			{
				return -EINVAL;
			}
			break;
		default:
			return -EINVAL;
		}
		break;

	case CM_DEBUG_READ:
		if (copy_from_user((char *)&dr, data, sizeof(dr)))
			return -EINVAL;

		switch (dr.what) {
		case DEBUG_FILTER:
			debug_read_filter(sc, &dr);
			break;
		case DEBUG_FILTER_MODEMASK:
			debug_read_filter_modemask(sc, &dr);
			break;
		case DEBUG_FILTER_COUNT:
			debug_read_filter_count(sc, &dr);
			break;
		case DEBUG_FILTER_LOG:
			debug_read_filter_log(sc, &dr);
			break;
		case DEBUG_IPFNAT:
			debug_read_ipfnat(sc, &dr);
			break;
		case DEBUG_DAT:
			debug_read_dat(sc, &dr);
			break;
		case DEBUG_QOS:
			debug_read_qos(sc, &dr);
			break;
		case DEBUG_PPPOE:
			debug_read_pppoe(sc, &dr);
			break;
		case DEBUG_TUNNEL:
			debug_read_tunnel(sc, &dr);
			break;
		case DEBUG_L2TBL:
			debug_read_L2TBL(sc, &dr);
			break;
		}

		break;

	case CM_SET_DMZ:
		if (copy_from_user((char *)&v[0], data, sizeof(v[0])))
			return -EINVAL;

		pr_debug(
		       "camelot: set dmz mode %08lx\n", v[0]);

		set_dmz_mode(sc, v[0]);
		break;

	case CM_DAT:
		if (copy_from_user((char *)&ic, data, sizeof(ic)))
			return -EINVAL;
		
		switch(ic.what) {
		case CM_SET_DAT:
			if (copy_from_user((char *)&dat_entry, ic.data,
					   sizeof(DA_TBL)))
				return -EINVAL;

			pr_debug(
			       "camelot: add DAT entry\n");
			
			error = cm_ioctl_add_dat_entry(sc, &dat_entry);
			break;
			
		case CM_DEL_DAT:
			if (copy_from_user((char *)&dat_entry, ic.data,
					   sizeof(DA_TBL)))
				return -EINVAL;

			pr_debug(
			       "camelot: delete DAT entry\n");
			
			error = cm_ioctl_del_dat_entry(sc, &dat_entry);
			break;
		case CM_CLEAR_DAT:
			pr_debug(
			       "camelot: clear DA table\n");
			initial_DATBL(sc);
			break;
		default:
			return -EINVAL;
		}
		break;

	case CM_QOS:
		if (copy_from_user((char *)&ic, data, sizeof(ic)))
			return -EINVAL;
		
		switch(ic.what) {
		case CM_SET_QOS:
			if (copy_from_user((char *)&qos_entry, ic.data,
					   sizeof(QOS_TBL_ENTRY)))
				return -EINVAL;

			pr_debug(
			       "camelot: add QOS entry\n");
			
			error = cm_ioctl_add_qos_entry(sc, &qos_entry);
			break;
		case CM_DEL_QOS:
			if (copy_from_user((char *)&qos_entry, ic.data,
					   sizeof(QOS_TBL_ENTRY)))
				return -EINVAL;

			pr_debug(
			       "camelot: delete QOS entry\n");
			
			error = cm_ioctl_del_qos_entry(sc, &qos_entry);
			break;
		case CM_CLEAR_QOS:
			pr_debug("camelot: clear QOS table\n");
			initial_QOSTBL(sc);
			break;
		case CM_SET_QOSTABLE:
			pr_debug(
			       "camelot: set QOS table\n");
			
			if (!camelot_ipfnat_mode)
				return -EACCES;
#ifdef CONFIG_MACH_FUJITSU_CAMELOT_HW_ACC				
			error = cm_set_qos_table(ic.data);
#endif
			break;
		default:
			return -EINVAL;
		}
		break;

	case CM_L2:
		if (copy_from_user((char *)&ic, data, sizeof(ic)))
			return -EINVAL;
		
		switch(ic.what) {
		case CM_SET_L2_ENTRY:
			if (copy_from_user((char *)&v, ic.data, sizeof(v)))
				return -EINVAL;

			pr_debug(
			       "camelot: add L2 entry\n");
			
			error = cm_ioctl_add_l2_entry(sc, v);
			break;
		case CM_DEL_L2_ENTRY:
			if (copy_from_user((char *)&v, ic.data, sizeof(v)))
				return -EINVAL;

			pr_debug(
			       "camelot: delete L2 entry\n");
			
			error = cm_ioctl_del_l2_entry(sc, v);
			break;
		case CM_GET_L2_INTF:
			if (copy_from_user((char *)&set_l2_v, ic.data,
					   sizeof(set_l2_v)))
				return -EINVAL;

			pr_debug(
			       "camelot: get L2 interface\n");
			error = cm_ioctl_lookup_l2_intf(sc, set_l2_v);
			if (error == 0) {
				copy_to_user((char *)ic.data, (char *)&set_l2_v,
					     sizeof(set_l2_v));
			}
			break;
		default:
			return -EINVAL;
		}
		break;

	default:
		error = -EOPNOTSUPP;
		break;
	}

	return error;
}

