/*
 * drivers/net/mb86977/camelot_func.c
 *
 * Fujitsu MB86977 driver table functions
 *
 * Author: <source@mvista.com>
 *
 * Copyright (c) 2002-2003 by Fujitsu LSI Solution Ltd..  All Rights Reserved.
 *
 * 2003 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/spinlock.h>
#include <linux/init.h>
#include <asm/uaccess.h>

#include "camelot_var.h"
#include "camelot_reg.h"
#include "camelot_defs.h"
#include "camelottbl.h"
#include "camelot_func.h"
#include "camelot_bits.h"

extern u_char came_mac[6];
extern int came_dmz_mode;

void
init_camelot(struct camelot_softc *sc)
{
	printk(KERN_NOTICE "camelot: initializing\n");

	set_L2(sc);

	/* endian setting */
	(*sc->sc_write)(sc, HOST_CONTRL_OFFSET, ENDIAN);

	/* SMI setting */
	initial_SMI(sc);

	/* Filter initialize */
	initial_FLCNT_SUBNM(sc);
	initial_L3_4(sc);
	initial_PRTTB(sc);

	/* Header initialize */
	initial_PPPoE(sc);
	initial_V4HEADER(sc);

	/* NAT/IP_F table Initialize */
	initial_NATIPFTBL(sc);

	/* Direction Analyzing Table Initialize */
	initial_DATBL(sc);

	/* QOS table Initialize */
	initial_QOSTBL(sc);

	/* Interrupt Setting */
	set_inten(sc);

	/* MAC Initialize and MAC function Start */
	initial_MAC(sc);

	/* Switch Block Set */
	(*sc->sc_write)(sc, SWENG_FLWTHSHLD_OFFSET, THSWQ);
}

void
set_table_access_priority(struct camelot_softc *sc)
{
	(*sc->sc_write)(sc, TBL_NATACPRT_OFFSET, 0x1);
}

void
reset_table_access_priority(struct camelot_softc *sc)
{
	(*sc->sc_write)(sc, TBL_NATACPRT_OFFSET, 0x0);
}

/*
 * set one complete NAT/IP Forwarding table entry
 */
int set_NATIPFTBL(struct camelot_softc *sc, int id, NAT_IPF_TBL *nit)
{
	int i1;
	int flgv4 = 0;
	int flgv6 = 0;
	int macadr_u, macadr_l, ipv6adr;

	if( id >= ENTRY_NUM_NATIPF ){
		printk(KERN_ERR
		       "camelot: Warning; NAT/IP_F Table entry > 127\n");
		return -1;
	}
	if( !(nit->control & 0x00100000)){ 
		printk(KERN_ERR 
		       "camelot: Warning; NAT/IP_F Invalid control\n");
		return -1;
	}
	else if((nit->control & 0x00010000)){
		flgv4 = 1;
	}
	else if((nit->control & 0x00020000)){
		flgv6 = 1;
	}

	/* turn off packet processing */
	/*set_table_access_priority(sc);*/

	/*   Control Bit Set   */
	setnat_root(sc, id, 22, nit->control);

	if(flgv4) {
		/*   IPv4 Internal Address Set   */
		setnat_root(sc, id, 0, nit->ipv4addr_inter);
		/*   IPv4 External Address Set   */
		setnat_root(sc, id, 1, nit->ipv4addr_exter);
		/*   IPv4 NAT Address Set   */
		setnat_root(sc, id, 2, nit->nataddr);
		/*   Internal Port Number Set   */
		setnat_root(sc, id, 12, nit->portnum_inter);
		/*   External Port Number Set   */
		setnat_root(sc, id, 13, nit->portnum_exter);
		/*   NAT Port Number Set   */
		setnat_root(sc, id, 14, nit->portnum_nat);
	}
	if(flgv6) {
		/*   IPv6 Internal Address Set   */
		for(i1=0; i1<4; i1++){  
			ipv6adr =
				(((nit->ipv6addr_inter[i1] << 24) & 0xff000000)
				 | ((nit->ipv6addr_inter[i1] << 8)  & 0xff0000)
				 | ((nit->ipv6addr_inter[i1] >> 8)  & 0xff00)
				 | ((nit->ipv6addr_inter[i1] >> 24) & 0xff));
			setnat_root(sc, id, 4+i1, ipv6adr);
		}
		/*   IPv6 External Address Set   */
		for(i1=0; i1<4; i1++){  
			ipv6adr =
				(((nit->ipv6addr_exter[i1] << 24) & 0xff000000)
				 | ((nit->ipv6addr_exter[i1] << 8)  & 0xff0000)
				 | ((nit->ipv6addr_exter[i1] >> 8)  & 0xff00)
				 | ((nit->ipv6addr_exter[i1] >> 24) & 0xff));
			setnat_root(sc, id, 8+i1, ipv6adr);
		}
	}

	/*   Internal MAC ADR Set   */
	macadr_u =  (((nit->macaddr_inter[0] << 24) & 0xff000000)
		   | ((nit->macaddr_inter[1] << 16) & 0x00ff0000)
		   | ((nit->macaddr_inter[2] << 8)  & 0x0000ff00)
		   | ((nit->macaddr_inter[3] << 0)  & 0x000000ff));
	macadr_l =  (((nit->macaddr_inter[4] << 8)  & 0x0000ff00)
		   | ((nit->macaddr_inter[5] << 0)  & 0x000000ff));
	setnat_root(sc, id, 16, macadr_u);
	setnat_root(sc, id, 17, macadr_l);

	/*   External MAC ADR Set   */
	macadr_u =  (((nit->macaddr_exter[0] << 24) & 0xff000000)
		   | ((nit->macaddr_exter[1] << 16) & 0x00ff0000)
		   | ((nit->macaddr_exter[2] << 8)  & 0x0000ff00)
		   | ((nit->macaddr_exter[3] << 0)  & 0x000000ff));
	macadr_l =  (((nit->macaddr_exter[4] << 8)  & 0x0000ff00)
		   | ((nit->macaddr_exter[5] << 0)  & 0x000000ff));
	setnat_root(sc, id, 18, macadr_u);
	setnat_root(sc, id, 19, macadr_l);

	/*   Internal Interface Number Set   */
	setnat_root(sc, id, 20, nit->intf_inter);

	/*   External Interface Number Set   */
	setnat_root(sc, id, 21, nit->intf_exter);

	/* allow packet processing */
	/*reset_table_access_priority(sc);*/

	return 0;
}

int set_empty_NATIPFTBL(struct camelot_softc *sc, int id)
{
	int i;

	if (id >= ENTRY_NUM_NATIPF) {
		printk(KERN_ERR 
		       "camelot: Warning; NAT/IP_F Table entry > 127\n");
		return -1;
	}

	/* turn off packet processing */
	/*set_table_access_priority(sc);*/

	/* Control Bit Set */
	setnat_root(sc, id, 22, cntrlbit_0_valid);

	for (i = 0; i <= 2; i++) {
		setnat_root(sc, id, i, 0);
	}

	for (i = 4; i <= 14; i++) {
		setnat_root(sc, id, i, 0);
	}

	for (i = 16; i <= 21; i++) {
		setnat_root(sc, id, i, 0);
	}

	/* enable packet processing */
	/*reset_table_access_priority(sc);*/

	return 0;
}

void initial_NATIPFTBL(struct camelot_softc *sc)
{
	int id, itm, data;
	NAT_IPF_TBL  *nit;

	/* turn off packet processing */
	/*set_table_access_priority(sc);*/

	for (id = 0; id < ENTRY_NUM_NATIPF; id++){
		for (itm = 0; itm < 23; itm++) {
			if (itm != 15) {
				(*sc->sc_write)(sc, TBL_NATDATA_OFFSET, 0x0);

				data =  ((id << 16) & 0x007f0000) |
					((itm << 8) & 0x00001f00) | 0x00000001;
				(*sc->sc_write)(sc, TBL_NATCMD_OFFSET, data);

				data = (*sc->sc_read)(sc, TBL_NATSTA_OFFSET);

				while(data) {
					data = (*sc->sc_read)(sc, TBL_NATSTA_OFFSET);
				}
			}
		}
	}

	for (id = 0;id < ENTRY_NUM_NATIPF_DFLT; id++) {
		nit = &natipf_table[id];
		set_NATIPFTBL(sc, id, nit);
	}

	/* fragment packet IP forwarding */
	(*sc->sc_write)(sc, DO_FRAG_IPF_OFFSET, DO_FLAG_IPF);

	/* enable packet processing */
	/*reset_table_access_priority(sc);*/
}

/*
 * read entry from NAT/IP forwarding table
 */
int read_NATIPFTBL(struct camelot_softc *sc, int id, int item)
{
	int data;
        int stdata;

	if (id >= ENTRY_NUM_NATIPF) {
		printk(KERN_ERR 
		       "camelot: Warning; Read id of NAT/IP_F Table > 127\n");
		return(0);
	}
	if (item >= 23) {
		printk(KERN_ERR 
		       "camelot: Warning; item of NAT/IP_F Table > 22\n");
		return(0);
	}
	else if (item == 3 || item == 15) {
		printk(KERN_ERR 
		       "camelot: Warning; item 3 or 15 is selected\n");
		return(0);
	}

	/* handle errata--turn off packet processing */
	/*set_table_access_priority(sc);*/

	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, 0x00000000);

	data = ((id << 16) & 0x007f0000) |
		((item << 8) & 0x00001f00) |
		0x00000000;

	(*sc->sc_write)(sc, TBL_NATCMD_OFFSET, data);

	stdata = (*sc->sc_read)(sc, TBL_NATSTA_OFFSET);

	while(stdata) {
		stdata = (*sc->sc_read)(sc, TBL_NATSTA_OFFSET);
	}

	data = (*sc->sc_read)(sc, TBL_NATDATA_OFFSET);

	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, HOST_INTEN);

	/* enable packet processing */
	/*reset_table_access_priority(sc);*/

	return (data);
}

/*
 * read entry from Direction Analyzing table
 */
int read_DATBL(struct camelot_softc *sc, int cmd)
{
	int data;
        int stdata;

	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, 0x00000000);

	(*sc->sc_write)(sc, TBLDIRECT_CMD_OFFSET, cmd);

	stdata = (*sc->sc_read)(sc, TBLDIRECT_STATUS_OFFSET);

	while(stdata) {
		stdata = (*sc->sc_read)(sc, TBLDIRECT_STATUS_OFFSET);
	}

	data = (*sc->sc_read)(sc, TBLDIRECT_DATA_OFFSET);

	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, HOST_INTEN);

	return (data);
}

/* read a complete DAT table entry into DA_TBL */
int read_DATBL_entry(struct camelot_softc *sc, int index, DA_TBL *dt)
{
	int val_ver, cmd;

	memset((char *)dt, 0, sizeof(DA_TBL));
	
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, 0x00000000);
	val_ver = (*sc->sc_read)(sc, TBLDIRECT_VLDIPV_OFFSET);

	dt->valid = val_ver & (1 << index);
	dt->ipv6 = val_ver & (1 << (index + 4));
	
	cmd = ((index << 13) & 0x0000E000) |  /* id */
	      ((0 << 12) &     0x00001000) |  /* address field */
	      ((0 << 8) &      0x00000100) |    /* 1st 32 bits */
	      0x00000000;                  /* read */
	dt->subnet_addr[0] = read_DATBL(sc, cmd);
	
	cmd = ((index << 13) & 0x0000E000) |  /* id */
	      ((1 << 12) &     0x00001000) |  /* mask field */
	      ((0 << 8) &      0x00000100) |    /* 1st 32 bits */
	      0x00000000;                  /* read */
	dt->subnet_mask[0] = read_DATBL(sc, cmd);

	if (dt->ipv6) {
		cmd = ((index << 13) & 0x0000E000) |  /* id */
		      ((0 << 12) &     0x00001000) |  /* address field */
		      ((1 << 8) &      0x00000100) |    /* 2nd 32 bits */
		      0x00000000;                  /* read */
		dt->subnet_addr[1] = read_DATBL(sc, cmd);
		
		cmd = ((index << 13) & 0x0000E000) |  /* id */
		      ((1 << 12) &     0x00001000) |  /* mask field */
		      ((1 << 8) &      0x00000100) |    /* 2nd 32 bits */
		      0x00000000;                  /* read */
		dt->subnet_mask[1] = read_DATBL(sc, cmd);
		
	}

	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, HOST_INTEN);

	return 0;
}

/*
 * read unmatch log
 */
void
read_NMTLOG(struct camelot_softc *sc, u_int32_t *match_log)
{
	int i, data;

	for(i = 0; i < ENTRY_NUM_NATMTHLOG; i++) {
		data = (*sc->sc_read)(sc, (NATIPF_MCHSTA_0_OFFSET + i*4));
		match_log[i] = data;
	}
}

void read_RFLOG(struct camelot_softc *sc, u_int32_t *rstfin_log)
{
	int i, data;

	for(i = 0; i < ENTRY_NUM_RSTFINLOG; i++){
		data = (*sc->sc_read)(sc, (NAT_FINRSTSTA_0_OFFSET + i*4));
		rstfin_log[i] = data;
	}
}

void read_TTL0LOG(struct camelot_softc *sc, u_int32_t *ttl0_log)
{
	int i, data;

	for(i = 0; i < ENTRY_NUM_TTL0LOG; i++){
		data = (*sc->sc_read)(sc, (TTL0_BUF_0_OFFSET + i*4));
		ttl0_log[i] = data;
	}
	data = (*sc->sc_read)(sc, TTL0_BUFFUL_OFFSET);
}

/*
 * write entry into NAP/IP Forwarding table
 */ 
void setnat_root(struct camelot_softc *sc, int id, int item, int sdt)
{
	int data;

	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, 0x00000000);
	(*sc->sc_write)(sc, TBL_NATDATA_OFFSET, sdt);

	data = ((id << 16) & 0x007f0000) |
		((item << 8) & 0x00001f00) |
		0x00000001;
	(*sc->sc_write)(sc, TBL_NATCMD_OFFSET, data);

	data = (*sc->sc_read)(sc, TBL_NATSTA_OFFSET);
	while(data) {
		data = (*sc->sc_read)(sc, TBL_NATSTA_OFFSET);
	}
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, HOST_INTEN);
}

void setdat_root(struct camelot_softc *sc, int data, int cmd)
{
	int status;

	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, 0x00000000);
	(*sc->sc_write)(sc, TBLDIRECT_DATA_OFFSET, data);
	(*sc->sc_write)(sc, TBLDIRECT_CMD_OFFSET, cmd);

	status = (*sc->sc_read)(sc, TBLDIRECT_STATUS_OFFSET);
	while(status) {
		status = (*sc->sc_read)(sc, TBLDIRECT_STATUS_OFFSET);
	}

	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, HOST_INTEN);
}

int dat_get_index(struct camelot_softc *sc)
{
	int val_ver, i;
	
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, 0x00000000);
	val_ver = (*sc->sc_read)(sc, TBLDIRECT_VLDIPV_OFFSET);
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, HOST_INTEN);

	for (i = 0; i < ENTRY_NUM_DAT; i++) {
		if ( (val_ver & (1 << i)) == 0 ) {
			return i;
		}
	}

	return -1;
}

void initial_DATBL(struct camelot_softc *sc)
{
	int val_ver;
	
	/* read/write VALID/VERSION register */
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, 0x00000000);
	val_ver = (*sc->sc_read)(sc, TBLDIRECT_VLDIPV_OFFSET);

	/* clear lower 8 bits */
	val_ver &= 0xFFFFFF00;
	
	(*sc->sc_write)(sc, TBLDIRECT_VLDIPV_OFFSET, val_ver);
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, HOST_INTEN);
}
	
int set_DATBL(struct camelot_softc *sc, int id, DA_TBL *dt)
{
	int val_ver, cmd;

	if (id >= ENTRY_NUM_DAT) {
		printk(KERN_ERR
		       "camelot: Warning: Direction Analyzer Table "
		       "entry > %d\n",
		       ENTRY_NUM_DAT);
		return -1;
	}

	if (! (dt->valid & 0x1) ) {
		printk(KERN_ERR
		       "camelot: Warning: Direction Analyzer Table Invalid\n");
		return -1;
	}

	/* write the data to the data register first */
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, 0x00000000);
	if (dt->ipv6) {
		/* write 2nd 32 bits of address */
		cmd = ((id << 13) & 0x0000E000) |
		      ((0 << 12) &  0x00001000) |
		      ((1 << 8) &   0x00000100) |
		      0x00000001;
		setdat_root(sc, dt->subnet_addr[1], cmd);
		
		/* write 2nd 32 bits of mask */
		cmd = ((id << 13) & 0x0000E000) |
		      ((1 << 12) &  0x00001000) |
		      ((1 << 8) &   0x00000100) |
		      0x00000001;
		setdat_root(sc, dt->subnet_mask[1], cmd);
	}
	
	/* always write 1st 32 bits of address/mask */
	cmd = ((id << 13) & 0x0000E000) |
	      ((0 << 12) &  0x00001000) |
	      ((0 << 8) &   0x00000100) |
	      0x00000001;
	setdat_root(sc, dt->subnet_addr[0], cmd);

	cmd = ((id << 13) & 0x0000E000) |
	      ((1 << 12) &  0x00001000) |
	      ((0 << 8) &   0x00000100) |
	      0x00000001;
	setdat_root(sc, dt->subnet_mask[0], cmd);

	/* read/write VALID/VERSION register */
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, 0x00000000);
	val_ver = (*sc->sc_read)(sc, TBLDIRECT_VLDIPV_OFFSET);
	
	val_ver = val_ver | (dt->ipv6 << (id + 4)) | (1 << id);
	
	(*sc->sc_write)(sc, TBLDIRECT_VLDIPV_OFFSET, val_ver);
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, HOST_INTEN);
	
	return 0;
}

void
read_dump_NTIPFTBL_entry(struct camelot_softc *sc, int i)
{
	int data;
#if 1
	int mac_up, mac_low;
#endif

	pr_debug("NAT Table entry #%d:\n", i);
	data = read_NATIPFTBL(sc, i, 22);
	if((data & 0x10000) && (data & 0x100000)) {
		pr_debug(
		       "   N/T v4 Entry %d : ctl=%08x la=%08x wa=%08x na=%08x\n",
		       i, read_NATIPFTBL(sc, i, 22), read_NATIPFTBL(sc, i, 0),
		       read_NATIPFTBL(sc, i, 1), read_NATIPFTBL(sc, i, 2));
		pr_debug(
		       "   li=%02x wi=%02x lp=%04x wp=%04x np=%04x ",
		       read_NATIPFTBL(sc, i, 20),
		       read_NATIPFTBL(sc, i, 21), read_NATIPFTBL(sc, i, 12),
		       read_NATIPFTBL(sc, i, 13), read_NATIPFTBL(sc, i, 14));
#if 1
		mac_up  = read_NATIPFTBL(sc, i, 16);
		mac_low = read_NATIPFTBL(sc, i, 17);
		mac_low = mac_low & 0xffff;
               	pr_debug("lma=%08x%04x ", mac_up, mac_low);
		mac_up  = read_NATIPFTBL(sc, i, 18);
		mac_low = read_NATIPFTBL(sc, i, 19);
		mac_low = mac_low & 0xffff;
               	pr_debug("wma=%08x%04x\n", mac_up, mac_low);
#else
		pr_debug("\n");
#endif
		return;
       	}
	else if((data & 0x20000) && (data & 0x100000)) {
		pr_debug(
		       "   N/T v6 Entry %d : ctl=%08x la=%08x:%08x:%08x:%08x\n",
		       i,
		       read_NATIPFTBL(sc, i, 22), read_NATIPFTBL(sc, i, 4),
		       read_NATIPFTBL(sc, i, 5), read_NATIPFTBL(sc, i, 6),
		       read_NATIPFTBL(sc, i, 7));
		pr_debug(
		       "   wa=%08x:%08x:%08x:%08x li=%02x wi=%02x\n",
		       read_NATIPFTBL(sc, i, 8), read_NATIPFTBL(sc, i, 9),
		       read_NATIPFTBL(sc, i, 10), read_NATIPFTBL(sc, i, 11),
		       read_NATIPFTBL(sc, i, 20), read_NATIPFTBL(sc, i, 21));
#if 1
		mac_up  = read_NATIPFTBL(sc, i, 16);
		mac_low = read_NATIPFTBL(sc, i, 17);
		mac_low = mac_low & 0xffff;
               	pr_debug("   lma=%08x%04x ", mac_up, mac_low);
		mac_up  = read_NATIPFTBL(sc, i, 18);
		mac_low = read_NATIPFTBL(sc, i, 19);
		mac_low = mac_low & 0xffff;
               	pr_debug("wma=%08x%04x\n", mac_up, mac_low);
#endif
		return;
	}
}

void
read_dump_DATBL_entry(struct camelot_softc *sc, int i)
{
	int val_ver;
	DA_TBL dt;

	pr_debug("DAT Table entry #%d:\n", i);

	if (i >= ENTRY_NUM_DAT) {
		printk(KERN_ERR
		       "camelot: Warning: Direction Analyzer Table "
		       "entry > %d\n",
		       ENTRY_NUM_DAT);
		return;
	}

	/* don't bother reading invalid entries */
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, 0x00000000);
	val_ver = (*sc->sc_read)(sc, TBLDIRECT_VLDIPV_OFFSET);
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, HOST_INTEN);

	if ( !(val_ver & (1 << i)) ) {
		return;
	}

	/* read entire (valid) entry */
	if (read_DATBL_entry(sc, i, &dt)) {
		printk(KERN_ERR
		       "camelot: Error reading DAT entry %d\n", i);
		return;
	}

#if 0
	cmd = ((i << 13) & 0x0000E000) |  /* id */
	      ((0 << 12) & 0x000010000) |  /* address field */
	      ((0 << 8) & 0x00000100) |    /* 1st 32 bits */
	      0x00000000;                  /* read */
	addr[0] = read_DATBL(sc, cmd);
	
	cmd = ((i << 13) & 0x0000E000) |  /* id */
	      ((1 << 12) & 0x000010000) |  /* mask field */
	      ((0 << 8) & 0x00000100) |    /* 1st 32 bits */
	      0x00000000;                  /* read */
	mask[0] = read_DATBL(sc, cmd);
	
	if (data & (1 << (i + 4))) {
		/* ipv6 entry; need to read both 32 bit chunks */
		cmd = ((i << 13) & 0x0000E000) |  /* id */
		      ((0 << 12) & 0x000010000) |  /* address field */
		      ((1 << 8) & 0x00000100) |    /* 2nd 32 bits */
		      0x00000000;                  /* read */
		addr[1] = read_DATBL(sc, cmd);

		cmd = ((i << 13) & 0x0000E000) |  /* id */
		      ((1 << 12) & 0x000010000) |  /* mask field */
		      ((1 << 8) & 0x00000100) |    /* 2nd 32 bits */
		      0x00000000;                  /* read */
		mask[1] = read_DATBL(sc, cmd);
	}
#endif

	pr_debug(
	       "  Direction Analyzing Table Entry %d: (%s, %s)\n"
	       "  addr1 = 0x%04x, addr2 = 0x%04x, mask1 = 0x%04x, "
	       "mask2 = 0x%04x\n",
	       i,
	       (val_ver & (1 << i)) ? "Valid" : "Invalid",
	       dt.ipv6 ? "IPv6" : "IPv4",
	       dt.subnet_addr[0], dt.subnet_addr[1],
	       dt.subnet_mask[0], dt.subnet_mask[1]);

	return;
}

void
read_dump_NTALL(struct camelot_softc *sc)
{
	int i, data;
	int mac_up, mac_low;

	pr_debug("NAT Table dump:\n");
	for (i = 0; i < ENTRY_NUM_NATIPF; i++) {
		data = read_NATIPFTBL(sc, i, 22);
		if ((data & 0x10000) && (data & 0x100000)) {
			pr_debug(
			       "   N/T v4 Entry %d : "
			       "ctl=%08x la=%08x wa=%08x na=%08x\n",
			       i,
			       read_NATIPFTBL(sc, i, 22),
			       read_NATIPFTBL(sc, i, 0),
			       read_NATIPFTBL(sc, i, 1),
			       read_NATIPFTBL(sc, i, 2));

			pr_debug(
			       "   li=%02x wi=%02x lp=%04x wp=%04x np=%04x ",
			       read_NATIPFTBL(sc, i, 20),
			       read_NATIPFTBL(sc, i, 21),
			       read_NATIPFTBL(sc, i, 12),
			       read_NATIPFTBL(sc, i, 13),
			       read_NATIPFTBL(sc, i, 14));

			mac_up  = read_NATIPFTBL(sc, i, 16);
			mac_low = read_NATIPFTBL(sc, i, 17);
			mac_low = mac_low & 0xffff;
                	pr_debug("lma=%08x%04x ", mac_up, mac_low);
			mac_up  = read_NATIPFTBL(sc, i, 18);
			mac_low = read_NATIPFTBL(sc, i, 19);
			mac_low = mac_low & 0xffff;
                	pr_debug("wma=%08x%04x\n", mac_up, mac_low);
		}
		else if ((data & 0x20000) && (data & 0x100000)) {
			pr_debug(
			       "   N/T v6 Entry %d: "
			       "ctl=%08x la=%08x:%08x:%08x:%08x\n",
			       i,
			       read_NATIPFTBL(sc, i, 22),
			       read_NATIPFTBL(sc, i, 7),
			       read_NATIPFTBL(sc, i, 6),
			       read_NATIPFTBL(sc, i, 5), read_NATIPFTBL(sc, i, 4));
			pr_debug(
			       "   wa=%08x:%08x:%08x:%08x li=%02x wi=%02x\n",
			       read_NATIPFTBL(sc, i, 8),
			       read_NATIPFTBL(sc, i, 9),
			       read_NATIPFTBL(sc, i, 10),
			       read_NATIPFTBL(sc, i, 11),
			       read_NATIPFTBL(sc, i, 20),
			       read_NATIPFTBL(sc, i, 21));

			mac_up  = read_NATIPFTBL(sc, i, 16);
			mac_low = read_NATIPFTBL(sc, i, 17);
			mac_low = mac_low & 0xffff;
       	        	pr_debug("   lma=%08x%04x ", mac_up, mac_low);
			mac_up  = read_NATIPFTBL(sc, i, 18);
			mac_low = read_NATIPFTBL(sc, i, 19);
			mac_low = mac_low & 0xffff;
       	        	pr_debug("wma=%08x%04x\n", mac_up, mac_low);
		}
	}
}

void set_L2(struct camelot_softc *sc)
{
	int data;

	setl2_root(sc, 0, came_mac, CONTROL_L2_CAME);

	/* Look Up Control Setting */
	if (came_dmz_mode)
		data = DMZ_EN | NO_MATCH_WAN | NO_MATCH_DMZ | NO_MATCH_LAN;
	else
		data = NO_MATCH_WAN | NO_MATCH_DMZ | NO_MATCH_LAN;

	(*sc->sc_write)(sc, LKUP_CONTRL_OFFSET, data);

	/* Look Up Table Timer Setting */
	(*sc->sc_write)(sc, LKUP_TIMER_OFFSET, LKP_TIMER);
}

void read_dump_L2ALL(struct camelot_softc *sc)
{
	int i;
	int data;
	u_int32_t mac_hi, mac_low;

	pr_debug("L2 Table:\n");
     	for (i = 0; i< L2ENTRY_NUM; i++) {
   		data = (*sc->sc_read)(sc, (LKUP_TBL_0o_OFFSET + (i * 8)));
	        if(data & 0x1){
			mac_hi  = (*sc->sc_read)(sc, (LKUP_TBL_0e_OFFSET +
						      (i * 8)));
			mac_low = (*sc->sc_read)(sc, (LKUP_TBL_0o_OFFSET +
						      (i * 8)));
			mac_low = (mac_low >> 16) & 0x0000ffff;
			data = (*sc->sc_read)(sc, (LKUP_TBL_0o_OFFSET +
						   (i * 8)));
		        data = data & 0xffff;
			pr_debug(
			       "   L2/T Entry %d: ctl=%04x ma=%08x%04x\n",
			       i, data, mac_hi, mac_low);
		}
	}
}

struct l2_table_s {
	unsigned long mac_hi;
	unsigned long mac_low_cntl;
};
static struct l2_table_s l2_table[L2ENTRY_NUM];
static unsigned long last_lkup_jiffies = 0;

static int
populate_l2_tbl(struct camelot_softc *sc)
{
	int i;
	int data;
	
	pr_debug("populate_l2_tbl()\n");
     	for (i = 0; i< L2ENTRY_NUM; i++) {
   		data = (*sc->sc_read)(sc, (LKUP_TBL_0o_OFFSET + (i * 8)));
	        if(data & 0x1){
			/* valid entry */
			l2_table[i].mac_low_cntl = data;
			l2_table[i].mac_hi
				= (*sc->sc_read)(sc,
						 (LKUP_TBL_0e_OFFSET + (i * 8)));
		}
	}

	last_lkup_jiffies = jiffies;

	return 0;
}

static int
find_l2_addr(unsigned long mac_hi, unsigned long mac_low, int *intf)
{
	int i;
	
	*intf = 0;
     	for (i = 0; i< L2ENTRY_NUM; i++) {
   		if (l2_table[i].mac_low_cntl & L2_ENTRY_VALID) {
			pr_debug("camelot: "
			       "comparing entry %d: %08lx%04lx: ctrl %04lx\n",
			       i+1,
			       l2_table[i].mac_hi,
			       (l2_table[i].mac_low_cntl >> 16) & 0xFFFF,
			       l2_table[i].mac_low_cntl & 0xFFFF);
			/* valid entry */
			if ((((l2_table[i].mac_low_cntl >> 16) & 0xFFFF) ==
			     mac_low) &&
			    (l2_table[i].mac_hi == mac_hi)) {
				/* match */
				*intf = (l2_table[i].mac_low_cntl &
					L2_ENTRY_INTF_MASK) >> 4;
				pr_debug("camelot: "
				       "Found interface %d\n", *intf);
				break;
			}
		}
	}

	return *intf;
}
	

/*
 * if L2 shadow table is current, lookup address and return
 * interface. if table is out-of-date or if first lookup fails to find
 * mac address, re-read table and search. note that this function
 * repopulates the table if 10 seconds have elapsed since the last
 * lookup
 */
int
lookup_l2_intf(struct camelot_softc *sc, unsigned long mac_hi,
	       unsigned long mac_low, int *intf)
{
	pr_debug("camelot: lookup_l2_intf(%08lx%04lx) "
	       "jiffies = %lu, last_jiffies = %lu\n",
	       mac_hi, mac_low, jiffies, last_lkup_jiffies);

	*intf = 0;
	if ((last_lkup_jiffies == 0) ||
	    (((jiffies - last_lkup_jiffies) / HZ) > 10) ||
	    (find_l2_addr(mac_hi, mac_low, intf) == 0)) {
		/* repopulate l2 table */
		pr_debug("camelot: Repopulating L2 table\n");
		populate_l2_tbl(sc);
	}

	/* only look up intf if not found above */
	if ((*intf == 0) &&
	    find_l2_addr(mac_hi, mac_low, intf) == 0) {
		pr_debug("camelot: "
		       "Could not find intf for L2 mac %08lx%04lx\n",
		       mac_hi, mac_low);
		return -1;
	}

	pr_debug("camelot: Found interface %d for L2 mac %08lx%04lx\n",
	       *intf, mac_hi, mac_low);

	return 0;
}

/*
 * set MAC address in root of L2 Address Lookup Table
 * (SOHO_MAC_ADR)
 */
void setl2_root(struct camelot_softc *sc, int id, unsigned char *cp, int cntl)
{
	u32  mac_hi, mac_low;

	mac_hi = (cp[0] << 24) | (cp[1] << 16) | (cp[2] << 8) | cp[3];

	(*sc->sc_write)(sc, (LKUP_TBL_0e_OFFSET + (id * 8)), mac_hi);

	mac_low = (cp[4] << 8) | cp[5];

	(*sc->sc_write)(sc, (LKUP_TBL_0o_OFFSET + (id * 8)),
			((mac_low << 16) | cntl));
}

void set_l2_entry(struct camelot_softc *sc, int id, unsigned long mac_hi,
		 unsigned long mac_low_cntl)
{
	(*sc->sc_write)(sc, (LKUP_TBL_0e_OFFSET + (id * 8)), mac_hi);

	(*sc->sc_write)(sc, (LKUP_TBL_0o_OFFSET + (id * 8)), mac_low_cntl);
}

void clear_l2_entry(struct camelot_softc *sc, int id)
{
	(*sc->sc_write)(sc, (LKUP_TBL_0e_OFFSET + (id * 8)), 0);

	(*sc->sc_write)(sc, (LKUP_TBL_0o_OFFSET + (id * 8)), 0);
}

int delete_l2_entry(struct camelot_softc *sc, unsigned long mac_hi,
		     unsigned long mac_low)
{
	int i;
	unsigned long data;
	
	/* find this entry */
	for (i = 0; i < L2ENTRY_NUM; i++) {
		data = (*sc->sc_read)(sc, LKUP_TBL_0o_OFFSET + (i * 8));
		if ((data & L2_ENTRY_VALID) == 0)
			continue;
		/* valid entry, check hi and low portions of mac */
		if ((data >> 16) == (mac_low >> 16)) {
			data = (*sc->sc_read)(sc, LKUP_TBL_0e_OFFSET + (i * 8));
			if (data == mac_hi) {
				clear_l2_entry(sc, i);
				break;
			}
		}
	}

	if (i == L2ENTRY_NUM) {
		printk(KERN_ERR "camelot: couldn't find L2 entry to delete.\n");
		return -EBUSY;
	}

	return 0;
}

void
clear_dynamic_l2_entries(struct camelot_softc *sc)
{
	int i;
	unsigned long data;

	/* find this entry */
	for (i = 0; i < L2ENTRY_NUM; i++) {
		data = (*sc->sc_read)(sc, LKUP_TBL_0o_OFFSET + (i * 8));
		if ((data & L2_ENTRY_VALID) == 0)
			continue;
		if (data & L2_ENTRY_DYNAMIC) {
			/* dynamic entry, clear it */
			clear_l2_entry(sc, i);
		}
	}
}	

/*
 * clears all entries except the first and second (SOHO_MAC_ADDR and
 * BROAD_CAST_ADDR)
 */
void
clear_all_l2_entries(struct camelot_softc *sc)
{
	int i;
	unsigned long data;

	/* find this entry */
	for (i = 2; i < L2ENTRY_NUM; i++) {
		data = (*sc->sc_read)(sc, LKUP_TBL_0o_OFFSET + (i * 8));
		if ((data & L2_ENTRY_VALID) == 0)
			continue;
		
		clear_l2_entry(sc, i);
	}
}	

void set_phy(struct camelot_softc *sc, int dev_adr, int reg_adr, int reg_data)
{
	int  data, tmp;

	(*sc->sc_write)(sc, SMI_CMDDATA_OFFSET, reg_data);

	tmp = (*sc->sc_read)(sc, SMI_POLINTVL_OFFSET);
	(*sc->sc_write)(sc, SMI_POLINTVL_OFFSET, 0x0);

	data = (*sc->sc_read)(sc, SMI_CMDST_OFFSET);
	while (data & 0x01) {
		data = (*sc->sc_read)(sc, SMI_CMDST_OFFSET);
	}

	data = 0x00008000 | ((reg_adr << 8) & 0x00001f00) |
		(dev_adr & 0x0000001f);
	(*sc->sc_write)(sc, SMI_COMMAND_OFFSET, data);

	data = (*sc->sc_read)(sc, SMI_CMDST_OFFSET);
	while (data & 0x01) {
		data = (*sc->sc_read)(sc, SMI_CMDST_OFFSET);
		if(data & 0x02) {
			printk(KERN_ERR "camelot: PHY Register Read Error\n");
			return;
		}
	}
	(*sc->sc_write)(sc, SMI_POLINTVL_OFFSET, tmp);
}

int read_phy(struct camelot_softc *sc, int dev_adr, int reg_adr)
{
	int tmp, data;

	tmp = (*sc->sc_read)(sc, SMI_POLINTVL_OFFSET);
	(*sc->sc_write)(sc, SMI_POLINTVL_OFFSET, 0x0);

	data = (*sc->sc_read)(sc, SMI_CMDST_OFFSET);
	while(data & 0x01){
		data = (*sc->sc_read)(sc, SMI_CMDST_OFFSET);
	}

	data = ((reg_adr << 8) & 0x00001f00) | (dev_adr & 0x0000001f);
	(*sc->sc_write)(sc, SMI_COMMAND_OFFSET, data);

	data = (*sc->sc_read)(sc, SMI_CMDST_OFFSET);
	while(data & 0x01){
		data = (*sc->sc_read)(sc, SMI_CMDST_OFFSET);
		if(data & 0x02){
			printk(KERN_ERR "camelot: PHY Register Read Error\n");
			return(0);
		}
	}

	(*sc->sc_write)(sc, SMI_POLINTVL_OFFSET, tmp);
	data = (*sc->sc_read)(sc, SMI_CMDDATA_OFFSET);

	return(data);
}

void initial_SMI(struct camelot_softc *sc)
{
	int data;

#if 0
	/*printk("SMI PHY Address Set\n");*/
	data = (DEV4PHTADD << 24) |
		(DEV3PHTADD << 16) |
		(DEV2PHTADD << 8) |
		DEV1PHTADD;
	(*sc->sc_write)(sc, SMI_PHYADR_OFFSET, data);
#endif

	(*sc->sc_write)(sc, SMI_MDCDIV_OFFSET, MDC_DIV);
	(*sc->sc_write)(sc, SMI_POLINTVL_OFFSET, POLLING_INT);
	/*printk("SMI Control Reg Set\n");*/

	/* enable mii on all ports */
	data = 0xffff;
	(*sc->sc_write)(sc, SMI_CONTRL_OFFSET, data);

#if 0
	{
		int i;
		u32 data;

		set_phy(sc, 0x0, 0, 0x9200);
		for (i = 0; i < 6; i++) {
			data = read_phy(sc, 0x0, i);
			printk("phy 0, reg %d %08x\n", i, data);
		}
	}
#endif
}

int read_SMIST(struct camelot_softc *sc)
{
	int data;

	data = (*sc->sc_read)(sc, SMI_STATUS_OFFSET);

	return(data);
}

void disable_MAC(struct camelot_softc *sc)
{
	int data;

	/* disnable all ports */
	data = 0x2222;
	(*sc->sc_write)(sc, MAC_CONTRL_OFFSET, data);
}

void initial_MAC(struct camelot_softc *sc)
{
	int data;

	/* IPG Timer Set */
	(*sc->sc_write)(sc, MAC_IPGTMR_OFFSET, IPG_TMR);

	/* Frame Size Set */
	(*sc->sc_write)(sc, MAC_MINLENGTH_OFFSET, MAC_MIN_LEN);
	(*sc->sc_write)(sc, MAC_MAXLENGTH_OFFSET, MAC_MAX_LEN);

	/* Flow Contorol Pause Timer */
	(*sc->sc_write)(sc, MAC_PAUSETMR_OFFSET, PAUSE_TMR);

	/* MAC Contorol Reg Set */

	/* enable all ports, w/o flow control, w/o partition */
	data = 0x00000000;
	(*sc->sc_write)(sc, MAC_CONTRL_OFFSET, data);
}

int read_MACST(struct camelot_softc *sc, int port)
{
	int data;

	switch (port) {
	case 0:
		data = (*sc->sc_read)(sc, MAC_0_STATUS_OFFSET);
		break;
	case 1:
		data = (*sc->sc_read)(sc, MAC_1_STATUS_OFFSET);
		break;
	case 2:
		data = (*sc->sc_read)(sc, MAC_2_STATUS_OFFSET);
		break;
	case 3:
		data = (*sc->sc_read)(sc, MAC_3_STATUS_OFFSET);
		break;
	default:
		printk(KERN_ERR "camelot: Status Read Error, Port Number>4\n");
		data = 0;
	}

	return(data);
}

void set_inten(struct camelot_softc *sc)
{
	int data;
	int tmp;

	/* enable interrupt on link status change for all ports */
	data = SMI_INT_LINKST3 | SMI_INT_LINKST2 | 
		SMI_INT_LINKST1 | SMI_INT_LINKST0;
	(*sc->sc_write)(sc, SMI_INTEN_OFFSET, data);

	(*sc->sc_write)(sc, FILT_CNTINTEN_0_OFFSET, FL_CNT_IN_INTEN_31_0);
	(*sc->sc_write)(sc, FILT_CNTINTEN_1_OFFSET, FL_CNT_IN_INTEN_63_32);
	(*sc->sc_write)(sc, FILT_CNTINTEN_2_OFFSET, FL_CNT_IN_INTEN_31_0);
	(*sc->sc_write)(sc, FILT_CNTINTEN_3_OFFSET, FL_CNT_IN_INTEN_63_32);

	(*sc->sc_write)(sc, MAC_0_INTEN_OFFSET, MAC_INTEN_0);
	(*sc->sc_write)(sc, MAC_1_INTEN_OFFSET, MAC_INTEN_1);
	(*sc->sc_write)(sc, MAC_2_INTEN_OFFSET, MAC_INTEN_2);
	(*sc->sc_write)(sc, MAC_3_INTEN_OFFSET, MAC_INTEN_3);

	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, HOST_INTEN);

	tmp = (*sc->sc_read)(sc, MAC_0_INTEN_OFFSET);
	/*printk("Host Int Enable = %x %x\n\n", HOST_INTEN, tmp);*/
}

void reset_inten(struct camelot_softc *sc)
{
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, 0);
}

/*
 * filter_match_counts[0][] is in side of filter
 * filter_match_counts[1][] is out side of filter
 */
enum {
	FLT_CNT_IN_SIDE = 0,
	FLT_CNT_OUT_SIDE = 1
};
static unsigned long filter_match_counts[2][ENTRY_NUM_L34];

void
reset_filter_match_counts(struct camelot_softc *sc)
{
	int i, j, tmp;
	for (j = 0; j < ENTRY_NUM_L34; j++) {
		/* clear hardware counts */
		tmp = (*sc->sc_read)(sc,
				     (FILT_COUNT_1_0_OFFSET + j*4));
		tmp = (*sc->sc_read)(sc,
				     (FILT_COUNT_0_0_OFFSET + j*4));
		/* clear sofware counts */
		for (i = 0; i < 2; i++) {
			filter_match_counts[i][j] = 0;
		}
	}
}

void
decode_FLT_TBL(char *buf, int id, int table, FLT_TBL *ft)
{
	buf += sprintf(buf, "id %d, table %d\n", id, table);

#define append(s) buf += sprintf(buf, s)

	buf += sprintf(buf,
		       "v4: src %08x, dst %08x, "
		       "sport %04x-%04x, dport %04x-%04x\n",
		       ft->ipv4addr_src, ft->ipv4addr_dst,
		       ft->lowport_src, ft->upport_src,
		       ft->lowport_dst, ft->upport_dst);

	/* control_10 */
	switch (ft->control_10 & L4PTYPESEL_MASK) {
	case L4PTYPESEL_ICMP: append("ICMP,"); break;
	case L4PTYPESEL_TCP_OR_UDP: append("TCP/UDP,"); break;
	case L4PTYPESEL_UDP: append("UDP,"); break;
	case L4PTYPESEL_TCP: append("TCP,"); break;
	case L4PTYPESEL_DONT_CARE: append("DONT_CARE,"); break;
	}

	if (ft->control_10 & L4_dst_range) append("L4_dst_range,");
	if (ft->control_10 & L4_src_range) append("L4_src_range,");
	if (ft->control_10 & L4_dont_care_dst) append("L4_dont_care_dst,");
	if (ft->control_10 & L4_dont_care_src) append("L4_dont_care_src,");
	if (ft->control_10 & v6_dont_care_dst_TLA) append("v6_dont_care_dst_TLA,");
	if (ft->control_10 & v6_dont_care_dst_NLA) append("v6_dont_care_dst_NLA,");
	if (ft->control_10 & v6_dont_cast_dst_SLA) append("v6_dont_cast_dst_SLA,");
	if (ft->control_10 & v6_dont_cast_dst_host) append("v6_dont_cast_dst_host,");
	if (ft->control_10 & v6_dont_care_src_TLA) append("v6_dont_care_src_TLA,");
	if (ft->control_10 & v6_dont_care_src_NLA) append("v6_dont_care_src_NLA,");
	if (ft->control_10 & v6_dont_cast_src_SLA) append("v6_dont_cast_src_SLA,");
	if (ft->control_10 & v6_dont_cast_src_host) append("v6_dont_cast_src_host,");
	if (ft->control_10 & v4_mask_use_dst) append("v4_mask_use_dst,");
	if (ft->control_10 & v4_mask_use_src) append("v4_mask_use_src,");
	if (ft->control_10 & v4_dont_care_dst) append("v4_dont_care_dst,");
	if (ft->control_10 & v4_dont_care_src) append("v4_dont_care_src,");
	if (ft->control_10 & icmp_dont_care) append("icmp_dont_care,");

	if (ft->control_10 & ip_choice_ipv4) append("ipv4,");
	else append("ipv6,");

	if (ft->control_10 & entry_valid) append("entry_valid");
	else append("not_valid");
	append("\n");

	/* control_11 */
	if (ft->control_11 & do_log) append("log,");
	else append("dont_log,");
	if (ft->control_11 & pass_drop) append("drop,");
	else append("pass,");
	if (ft->control_11 & ack_fg_en_1) append("ack_fg_en_1,");
	if (ft->control_11 & ack_fg_en_0) append("ack_fg_en_0,");
	switch (ft->control_11 & 0x7) { /* 3 bits */
	case 4: append("lan,"); break;
	case 3: append("wan/dmz,"); break;
	case 2: append("dmz,"); break;
	case 1: append("wan,"); break;
	case 0: append("wan/dmz/lan,"); break;
	}
#undef append
}

void
dump_FLT_TBL(int id, int table, FLT_TBL *ft)
{
	char buffer[512];

	decode_FLT_TBL(buffer, id, table, ft);
	pr_debug("%s", buffer);
}


void set_L3_4(struct camelot_softc *sc, int id, int table, FLT_TBL *ft)
{
	int i, data, flgv4, flgv6;
	int port;

	dump_FLT_TBL(id,table,ft);

	flgv4 = 0;
	flgv6 = 0;
	if( id >= ENTRY_NUM_L34 )  {
		printk(KERN_ERR
		       "camelot: Warning - entry # %d > 63 in L3/L4 set\n",
		       id);
		return;
	}
	if( table != 0 && table != 2 )  {
		printk(KERN_ERR
		       "camelot: Warning - table # %d wrong in L3/L4 set\n",
		       table);
		return;
	}
	if(table == 0) {	 /* in side */
		data = ft->control_10;
		if(!(data & 0x1)) {
			printk(KERN_ERR
			       "Filter Entry(%d) Invalid\n", id);
			return;
		}
		else if((data & 0x2)) {		/*   IPv4   */
			setflt_root(sc, id, 0, table, ft->ipv4addr_src);
			setflt_root(sc, id, 4, table, ft->ipv4addr_dst);
		}
		else if(!(data & 0x2)) {	/*   IPv6   */
			for(i = 0; i < 4; i++){
				setflt_root(sc, id, i, table, ft->ipv6addr_src[i]);
			}
			for(i = 4; i < 8; i++){
				setflt_root(sc, id, i, table, ft->ipv6addr_dst[i-4]);
			}
		}
		port = ((ft->lowport_dst << 16) & 0xffff0000) | (ft->lowport_src & 0x0000ffff);
		setflt_root(sc, id, 8, table, port);
		port = ((ft->upport_dst << 16) & 0xffff0000) | (ft->upport_src & 0x0000ffff);
		setflt_root(sc, id, 9, table, port);
		setflt_root(sc, id, 10, table, ft->control_10);
		setflt_root(sc, id, 11, table, ft->control_11);
		/* reset counter */
		filter_match_counts[FLT_CNT_IN_SIDE][id] = 0;
	}
	else if(table == 2) {  /* out side */
		data = ft->control_10;
		if(!(data & 0x1)) {
			printk(KERN_ERR "Filter Entry(%d) Invalid\n", id);
			return;
		}
		else if((data & 0x2)) {		/*   IPv4   */
			setflt_root(sc, id, 0, table, ft->ipv4addr_src);
			setflt_root(sc, id, 4, table, ft->ipv4addr_dst);
		}
		else if(!(data & 0x2)) {	/*   IPv6   */
			for(i = 0; i < 4; i++) {
				setflt_root(sc, id, i, table, ft->ipv6addr_src[i]);
			}
			for(i = 4; i < 8; i++) {
				setflt_root(sc, id, i, table, ft->ipv6addr_dst[i-4]);
			}
		}

		port = ((ft->lowport_dst << 16) & 0xffff0000) |
			(ft->lowport_src & 0x0000ffff);
		setflt_root(sc, id, 8, table, port);

		port = ((ft->upport_dst << 16) & 0xffff0000) |
			(ft->upport_src & 0x0000ffff);
		setflt_root(sc, id, 9, table, port);
		setflt_root(sc, id, 10, table, ft->control_10);
		setflt_root(sc, id, 11, table, ft->control_11);
		/* reset counter */
		filter_match_counts[FLT_CNT_OUT_SIDE][id] = 0;
	}
}

int
read_L3_4_filter(struct camelot_softc *sc, int id, int table, int item)
{
	int data;

	if (id >= ENTRY_NUM_L34) {
		printk(KERN_ERR
		       "camelot: Error; Entry # %d > 63 in L3/L4 read\n",
		       id);
		return(0);
	}

	if (table != 0 && table != 2) {
		printk(KERN_ERR
		       "camelot: Error; Table # %d wrong in L3/L4 read\n",
		       table);
		return(0);
	}

	if (item >= 12) {
		printk(KERN_ERR
		       "camelot: Error; Item # %d bad in L3/L4 read\n", item);
		return(0);
	}

	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, 0x00000000);

	data = (0x0003f000 & (id << 12)) |
		(0x00000f00 & (item << 8)) |
		(0x00000030 & (table << 4)) | 0x00000000;

	(*sc->sc_write)(sc, TBL_FILTCMD_OFFSET, data);

	data = (*sc->sc_read)(sc, TBL_FILTSTA_OFFSET);

	while(data) { 
		data = (*sc->sc_read)(sc, TBL_FILTSTA_OFFSET);
	}

	data = (*sc->sc_read)(sc, TBL_FILTDATA_OFFSET);
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, HOST_INTEN);

	return (data);
}


void
initial_L3_4(struct camelot_softc *sc)
{
	int i, id, table, data;
	FLT_TBL *ft;

	/* in side */
	table = 0;
	for (id = 0; id < ENTRY_NUM_L34; id++) {
		for (i = 0; i < 12; i++) {
			/* zero entry */
			(*sc->sc_write)(sc, TBL_FILTDATA_OFFSET, 0x0);

			data = (0x0003f000 & (id << 12)) |
				(0x00000f00 & (i << 8)) |
				(0x00000030 & (table << 4)) | 0x00000001;
			(*sc->sc_write)(sc, TBL_FILTCMD_OFFSET, data);

			data = (*sc->sc_read)(sc, TBL_FILTSTA_OFFSET);
			while(data) {
				data = (*sc->sc_read)(sc, TBL_FILTSTA_OFFSET);
			}
		}
	}

#if 1 /* this is just for debugging; but what about the tcp entry? */
	for (id = 0; id < ENTRY_NUM_L34_IN_DFLT; id++) {
		/*  input filter  */
		ft = &flt_table_in[id];
		set_L3_4(sc, (ENTRY_NUM_L34 - 1 - id), table, ft);
	}
#endif

	/* out side */
	table = 2;
	for (id = 0; id < ENTRY_NUM_L34; id++) {
		for (i = 0; i < 12; i++) {
			/* zero entry */
			(*sc->sc_write)(sc, TBL_FILTDATA_OFFSET, 0x0);

			data = (0x0003f000 & (id << 12)) |
				(0x00000f00 & (i << 8)) |
				(0x00000030 & (table << 4)) | 0x00000001;
			(*sc->sc_write)(sc, TBL_FILTCMD_OFFSET, data);

			data = (*sc->sc_read)(sc, TBL_FILTSTA_OFFSET);
			while(data) {
				data = (*sc->sc_read)(sc, TBL_FILTSTA_OFFSET);
			}
		}
	}

#if 0
	/* this is just for debugging */
	for (id = 0; id < ENTRY_NUM_L34_OUT_DFLT; id++) {
		/*  output filter */
		ft = &flt_table_out[id];
		set_L3_4(sc, (ENTRY_NUM_L34 - 1 - id), table, ft);
	}
#endif
}

#if 0
/* this is just for debugging */
void set_PRTTB(struct camelot_softc *sc, int id, int table)
{
	int data;

	/*printk("Camelot Protocol Type Table Setting Start\n");*/

	if ( id >= ENTRY_NUM_PRT ) {
		printk(KERN_ERR
		       "camelot: Warning - id # %d > 7 in set_PRTTB\n", id);
		return;
	}
	if ( table != 1 && table != 3 ) {
		printk(KERN_ERR
		       "camelot: Warning - table # %d bad in set_PRTTB\n",
		       table);
		return;
	}
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, 0x00000000);
	if (table == 1) {	 /* in side */
		(*sc->sc_write)(sc, TBL_FILTDATA_OFFSET, prt_table_in[id]);
		data = (0x0003f000 & (id << 12)) |
			(0x00000f00 & (12 << 8)) |
			(0x00000030 & (table << 4)) | 0x00000001;
		(*sc->sc_write)(sc, TBL_FILTCMD_OFFSET, data);
		data = (*sc->sc_read)(sc, TBL_FILTSTA_OFFSET);
		while(data){
			data = (*sc->sc_read)(sc, TBL_FILTSTA_OFFSET);
		}
	}
	else if(table == 3) {  /* out side */
		(*sc->sc_write)(sc, TBL_FILTDATA_OFFSET, prt_table_out[id]);
		data = (0x0003f000 & (id << 12)) |
			(0x00000f00 & (12 << 8)) |
			(0x00000030 & (table << 4)) | 0x00000001;
		(*sc->sc_write)(sc, TBL_FILTCMD_OFFSET, data);
		data = (*sc->sc_read)(sc, TBL_FILTSTA_OFFSET);
		while(data) {
			data = (*sc->sc_read)(sc, TBL_FILTSTA_OFFSET);
		}
	}
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, HOST_INTEN);
}
#endif

int read_PRTTB(struct camelot_softc *sc, int id, int table)
{
	int data;

	/*printk("Camelot Protocol Type Table Setting Start\n");*/
	if( id >= ENTRY_NUM_PRT )  {
		printk(KERN_ERR
		       "camelot: Warning - id # %d > 7 in read_PRTTB\n", id);
		return(0);
	}
	if ( table != 1 && table != 3 )  {
		printk(KERN_ERR
		       "camelot: Warning - Table # %d bad in read_PRTTB\n",
		       table);
		return(0);
	}

	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, 0x00000000);

	data = (0x0003f000 & (id << 12)) |
		(0x00000f00 & (12 << 8)) |
		(0x00000030 & (table << 4)) |
		0x00000000;

	(*sc->sc_write)(sc,  TBL_FILTCMD_OFFSET, data);

	data = (*sc->sc_read)(sc, TBL_FILTSTA_OFFSET);

	while(data){
		data = (*sc->sc_read)(sc, TBL_FILTSTA_OFFSET);
	}

	data = (*sc->sc_read)(sc, TBL_FILTDATA_OFFSET);
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, HOST_INTEN);

	return(data);
}

void initial_PRTTB(struct camelot_softc *sc)
{
	int id, table, data;

	table = 1;	 /* in side */
	for (id = 0; id < ENTRY_NUM_PRT; id++) {
		(*sc->sc_write)(sc, TBL_FILTDATA_OFFSET, 0x0);

		data  = (0x0003f000 & (id << 12)) | (0x00000f00 & (12 << 8))
		      | (0x00000030 & (table << 4)) | 0x00000001;
		(*sc->sc_write)(sc, TBL_FILTCMD_OFFSET, data);
		data = (*sc->sc_read)(sc, TBL_FILTSTA_OFFSET);

		while(data) {
			data = (*sc->sc_read)(sc, TBL_FILTSTA_OFFSET);
		}
	}

	table = 3; /* out side */
	for (id = 0; id < ENTRY_NUM_PRT; id++) {
		(*sc->sc_write)(sc, TBL_FILTDATA_OFFSET, 0x0);

		data = (0x0003f000 & (id << 12)) | (0x00000f00 & (12 << 8))
		     | (0x00000030 & (table << 4)) | 0x00000001;
		(*sc->sc_write)(sc, TBL_FILTCMD_OFFSET, data);

		data = (*sc->sc_read)(sc, TBL_FILTSTA_OFFSET);
		while(data) {
			data = (*sc->sc_read)(sc, TBL_FILTSTA_OFFSET);
		}
	}
}

void read_FLTLOG(struct camelot_softc *sc,
		 u_int32_t flt_log_in[][16],
		 u_int32_t flt_log_out[][16],
		 int *pstatus)
{
	int i, status, offset;
 
	(*sc->sc_write)(sc, FILT_LOGBUFLOCK_OFFSET, 0x1);   /* Lock ON */

	status = (*sc->sc_read)(sc, FILT_LOGSTATUS_OFFSET);

	*pstatus = status;

	if (status & 0x01) {	/* log @ buf_0  */
		offset = FILT_LOG_0_0_0_OFFSET;
		for (i = 0; i < 16; i++) {
			flt_log_in[0][i] = (*sc->sc_read)(sc, offset);
			offset += 4;
		}
	}

	if (status & 0x02) {	/* log @ buf_1  */
		offset = FILT_LOG_0_1_0_OFFSET;
		for (i = 0; i < 16; i++) {
			flt_log_in[1][i] = (*sc->sc_read)(sc, offset);
			offset += 4;
		}
	}

	if (status & 0x04) {	/* log @ buf_2  */
		offset = FILT_LOG_0_2_0_OFFSET;
		for (i = 0; i < 16; i++) {
			flt_log_in[2][i] = (*sc->sc_read)(sc, offset);
			offset += 4;
		}
	}

	if (status & 0x08) {	/* log @ buf_3  */
		offset = FILT_LOG_0_3_0_OFFSET;
		for (i = 0; i < 16; i++) {
			flt_log_in[3][i] = (*sc->sc_read)(sc, offset);
			offset += 4;
		}
	}

	if (status & 0x010) {	/* log @ buf_4  */
		offset = FILT_LOG_1_0_0_OFFSET;
		for (i = 0; i < 16; i++) {
			flt_log_out[0][i] = (*sc->sc_read)(sc, offset);
			offset += 4;
		}
	}

	if (status & 0x020) {	/* log @ buf_5  */
		offset = FILT_LOG_1_1_0_OFFSET;
		for (i = 0; i < 16; i++) {
			flt_log_out[1][i] = (*sc->sc_read)(sc, offset);
			offset += 4;
		}
	}

	if (status & 0x040) {	/* log @ buf_6  */
		offset = FILT_LOG_1_2_0_OFFSET;
		for (i = 0; i < 16; i++) {
			flt_log_out[2][i] = (*sc->sc_read)(sc, offset);
			offset += 4;
		}
	}

	if (status & 0x080) {	/* log @ buf_7  */
		offset = FILT_LOG_1_3_0_OFFSET;
		for (i = 0; i < 16; i++) {
			flt_log_out[3][i] = (*sc->sc_read)(sc, offset);
			offset += 4;
		}
	}

	(*sc->sc_write)(sc, FILT_LOGBUFLOCK_OFFSET, 0x0);   /* Lock OFF */
}

void initial_FLCNT_SUBNM(struct camelot_softc *sc)
{
	int data;

	/* turn off all filters */
	data = 0x00000000;
	(*sc->sc_write)(sc, FILT_CONTRL_OFFSET, data);

	/* set subnet mask to 255.255.0.0 */
	(*sc->sc_write)(sc, FILT_SUBNET_OFFSET, SUBNET_MASK);

	/* set log control */
	data = LOG_PPPOE | LOG_FLAGMENT | LOG_UNKOWN;
	(*sc->sc_write)(sc, FILT_LOGCONT_OFFSET, data);
}

int
set_FLCNT(struct camelot_softc *sc, unsigned long data)
{
	(*sc->sc_write)(sc, FILT_CONTRL_OFFSET, data);
	return 0;
}

int
set_SUBNM(struct camelot_softc *sc, unsigned long mask)
{
	(*sc->sc_write)(sc, FILT_SUBNET_OFFSET, mask);
	return 0;
}

int read_FLCNTST(struct camelot_softc *sc, int io, int part)
{
	int data = 0;

	if (io == 0 && part == 1) {
		data = (*sc->sc_read)(sc, FILT_CNTSTA_0_OFFSET);
	}
	else if (io == 0 && part == 2) {
		data = (*sc->sc_read)(sc, FILT_CNTSTA_1_OFFSET);
	}
	else if (io == 1 && part == 1) {
		data = (*sc->sc_read)(sc, FILT_CNTSTA_2_OFFSET);
	}
	else if (io == 1 && part == 2) {
		data = (*sc->sc_read)(sc, FILT_CNTSTA_3_OFFSET);
	}
	else {
		printk(KERN_ERR
		       "camelot: Error; Filter Count Status Reg Read\n");
	}
	return(data);
}

void incr_FLCNTV(struct camelot_softc *sc, int io, int id)
{
	if(io > 2) {
		return;
	}
	if(id > 64) {
		return;
	}
	if(io) {  /* out-side */
		filter_match_counts[FLT_CNT_OUT_SIDE][id] += FILT_OVERFLOW_COUNT;
	}
	else   {  /* in-side  */
		filter_match_counts[FLT_CNT_IN_SIDE][id] += FILT_OVERFLOW_COUNT;
	}
}

int read_FLCNTV(struct camelot_softc *sc, int io, int id)
{
	int data;

	if(io > 2) {
		return(0);
	}
	if(id > 64) {
		return(0);
	}
	if(io) {  /* out-side */
		data = (*sc->sc_read)(sc, (FILT_COUNT_1_0_OFFSET + id*4));
		filter_match_counts[FLT_CNT_OUT_SIDE][id] += data;
		data = filter_match_counts[FLT_CNT_OUT_SIDE][id];

	}
	else   {  /* in-side  */
		data = (*sc->sc_read)(sc, (FILT_COUNT_0_0_OFFSET + id*4));
		filter_match_counts[FLT_CNT_IN_SIDE][id] += data;
		data = filter_match_counts[FLT_CNT_IN_SIDE][id];
	}
	return(data);
}

/*
 * set filter table entry
 */
void setflt_root(struct camelot_softc *sc, int id, int item, int tb, int stdt)
{
	int data;

	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, 0x00000000);
	(*sc->sc_write)(sc, TBL_FILTDATA_OFFSET, stdt);

	data = (0x0003f000 & (id << 12)) |
		(0x00000f00 & (item << 8)) |
		(0x00000030 & (tb << 4)) |
		0x00000001;
	(*sc->sc_write)(sc, TBL_FILTCMD_OFFSET, data);
	data = (*sc->sc_read)(sc, TBL_FILTSTA_OFFSET);

	while(data) {
		data = (*sc->sc_read)(sc, TBL_FILTSTA_OFFSET);
	}
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, HOST_INTEN);
}

/*
 * read & dump L3/L4 filter table entry
 */
void
read_dump_FLT(struct camelot_softc *sc, int table, int i)
{
	int data;

	data = read_L3_4_filter(sc, i, table, 10);
	if (data & 0x3) {	/* IPv4 */
		if (table == 0){
               		pr_debug(
			       "   In F/T Entry %d : ctl1=%08x ctl2=%08x\n",
			       i,
			       read_L3_4_filter(sc, i, table, 10),
			       read_L3_4_filter(sc, i, table, 11));
		} else {
               		pr_debug(
			       "   Out F/T Entry %d : ctl1=%08x ctl2=%08x\n",
			       i,
			       read_L3_4_filter(sc, i, table, 10),
			       read_L3_4_filter(sc, i, table, 11));
		}

               	pr_debug("   sa=%08x da=%08x lp=%08x up=%08x\n",
		       read_L3_4_filter(sc, i, table, 0),
		       read_L3_4_filter(sc, i, table, 4),
		       read_L3_4_filter(sc, i, table, 8),
		       read_L3_4_filter(sc, i, table, 9));
	}
#if 0
	else if ((data & 0x1)) { /* IPv6 */
               	printk(" Sorry, IPv6 Under Construction \n");
		return;
	}
#endif
}

void read_dump_FLTALL(struct camelot_softc *sc, int table)
{
	int i;
	int data;

	pr_debug("L3/L4 filter table(sc=%p)\n", sc);

	pr_debug(
	       "%s  # ctl1     ctl2     sa       da       lp       up\n",
	       table ? "Out" : "In ");

	for (i = 0; i < ENTRY_NUM_L34; i++) {
		data = read_L3_4_filter(sc, i, table, 10);
		if (data & 0x3) {	/* IPv4 */
			pr_debug(
			       "    %2d %08x %08x %08x %08x %08x %08x\n",
			       i,
			       read_L3_4_filter(sc, i, table, 10),
			       read_L3_4_filter(sc, i, table, 11),
			       read_L3_4_filter(sc, i, table, 0),
			       read_L3_4_filter(sc, i, table, 4),
			       read_L3_4_filter(sc, i, table, 8),
			       read_L3_4_filter(sc, i, table, 9));
		}
#if 0
		else if((data & 0x1)) { /* IPv6 */
                	printk(" Sorry, IPv6 Under Construction \n");
			return;
		}
#endif
	}
}

/*
 * Routines for accessing the transmit and receive buffers.
 * The various CPU and adapter configurations supported by this
 * driver require three different access methods for buffers
 * and descriptors:
 *      (1) contig (contiguous data; no padding),
 *      (2) gap2 (two bytes of data followed by two bytes of padding), 
 *      (3) gap16 (16 bytes of data followed by 16 bytes of padding).
 */     
                
/*      
 * contig: contiguous data with no padding.
 *      
 * Buffers may have any alignment.
 */     
        
void            
camelot_copytobuf_contig(struct camelot_softc *sc,
			 void *from, int to, int wlen)
{               
	if (wlen <= 0 || wlen > 1516/4) {
		panic("camelot: bad copyto length");
	}

	memcpy((char *)(sc->sc_addr + to), from, wlen*4);
}       
        
void    
camelot_copyfrombuf_contig(sc, to, from, wlen)
	struct camelot_softc *sc;
	void *to;
	int from;
	int wlen;
{
	if (wlen <= 0 || wlen > 1516/4) {
		panic("camelot: bad copyfrom length");
	}

	memcpy(to, (char *)sc->sc_addr + from, wlen*4);
}

void            
camelot_copytobuf_contig_bytes(struct camelot_softc *sc,
			       void *from, int to, int len)
{               
	if (len <= 0 || len > 1516) {
		panic("camelot: bad copytobuf length");
	}

	memcpy((char *)(sc->sc_addr + to), from, len);
}       
        
void    
camelot_copyfrombuf_contig_bytes(struct camelot_softc *sc, void *to, 
				 int from, int len)
{
	if (len <= 0 || len > 1516) {
		panic("camelot: bad copyfrombuf length");
	}

	memcpy(to, (char *)(sc->sc_addr + from), len);
}

void initial_QOSTBL(struct camelot_softc *sc)
{
	int id, item, data;

	for (id = 0; id < ENTRY_NUM_QOS*2; id++) {
		for (item = 0; item < 12; item++) {
			(*sc->sc_write)(sc, TBL_QOSDATA_OFFSET, 0x0 );

			data =  ((id << 12) & 0x0f000) |
				((item << 8) & 0x0f00) | 0x01;

			(*sc->sc_write)(sc, TBL_QOSCMD_OFFSET, data);

			data = (*sc->sc_read)(sc, TBL_QOSSTA_OFFSET);

			while(data) {
				data = (*sc->sc_read)(sc, TBL_QOSSTA_OFFSET);
			}
		}
	}

#if 0
	/* for debug only */
	for (id = 0; id< ENTRY_NUM_QOS_D_DFLT; id++) {
		qt = &qos_dmz_table[id];
		set_QOSTBL(sc, 0, id, qt);
	}

	for (id = 0/*ENTRY_NUM_QOS_D_DFLT*/; id < ENTRY_NUM_QOS; id++) {
		for (item = 0; item < 12; item++) {
			(*sc->sc_write)(sc, TBL_QOSDATA_OFFSET, 0x0 );

			data =  ((id << 12) & 0x0f000) |
				((item << 8) & 0x0f00) | 0x01;
			(*sc->sc_write)(sc, TBL_QOSCMD_OFFSET, data);

			data = (*sc->sc_read)(sc, TBL_QOSSTA_OFFSET);
			while(data) {
				data = (*sc->sc_read)(sc, TBL_QOSSTA_OFFSET);
			}
		}
	}

	for (id = 0; id< ENTRY_NUM_QOS_W_DFLT; id++) {
		qt = &qos_wan_table[id];
		set_QOSTBL(sc, 1, id, qt);
	}

	for (id = ENTRY_NUM_QOS+ENTRY_NUM_QOS_W_DFLT;
	     id < ENTRY_NUM_QOS*2;
	     id++)
	{
		for (item = 0; item < 12; item++) {
			(*sc->sc_write)(sc, TBL_QOSDATA_OFFSET, 0x0 );

			data =  ((id << 12) & 0x0f000) |
				((item << 8) & 0x0f00) | 0x01;
			(*sc->sc_write)(sc, TBL_QOSCMD_OFFSET, data);

			data = (*sc->sc_read)(sc, TBL_QOSSTA_OFFSET);
			while(data) {
				data = (*sc->sc_read)(sc, TBL_QOSSTA_OFFSET);
			}
		}
	}
#endif
}
 
void set_QOSTBL(struct camelot_softc *sc, 
		int port,	/* WAN :1   DMZ :0 */
		int entry,	/* entry number */
		QOS_TBL_ENTRY *qt)
{
	int id, ports;

	if (port != 0 && port != 1) {
		printk(KERN_ERR
		       "camelot: Warning - QOS Port incorrect\n");
		return;
	}
	if (entry >= ENTRY_NUM_QOS)  {
		printk(KERN_ERR 
		       "camelot: Warning -Entry # of QOS Table > 7\n");
		return;
	}

	id = ((port << 3) & 0x8) | (entry & 0x7);
	setqos_root(sc, id, 11, qt->control);
	ports = ((qt->portnum_dst << 16) & 0xFFFF0000) |
		(qt->portnum_src & 0xFFFF);
	setqos_root(sc, id, 10, ports);
	setqos_root(sc, id,  9, qt->flowlabel);
	setqos_root(sc, id,  8, qt->tos);
	setqos_root(sc, id,  7, qt->ipv6addr_dst[3]);
	setqos_root(sc, id,  6, qt->ipv6addr_dst[2]);
	setqos_root(sc, id,  5, qt->ipv6addr_dst[1]);
	setqos_root(sc, id,  4, qt->ipv6addr_dst[0]);
	setqos_root(sc, id,  3, qt->ipv6addr_src[3]);
	setqos_root(sc, id,  2, qt->ipv6addr_src[2]);
	setqos_root(sc, id,  1, qt->ipv6addr_src[1]);
	setqos_root(sc, id,  0, qt->ipv6addr_src[0]);
}

void setqos_root(struct camelot_softc *sc, int id, int item, int sdt)
{
	int data;

	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, 0x00000000);
	(*sc->sc_write)(sc, TBL_QOSDATA_OFFSET, sdt);

	data =  ((id << 12) & 0x0f000) | ((item << 8) & 0x0f00) | 0x01;
	(*sc->sc_write)(sc, TBL_QOSCMD_OFFSET, data);

	data = (*sc->sc_read)(sc, TBL_QOSSTA_OFFSET);

	while(data) {
		data = (*sc->sc_read)(sc, TBL_QOSSTA_OFFSET);
	}

	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, HOST_INTEN);
}

int read_QOSTBL(struct camelot_softc *sc,
		int port,	/* WAN :1   DMZ :0 */
		int entry,	/* entry number */
		int item)	/* select item  */
{
	int data, id;
	
	if (port != 0 && port != 1) {
		printk(KERN_ERR "camelot: Warning; Port Select bad\n");
		return(0);
	}
	if ( entry >= ENTRY_NUM_QOS )  {
		printk(KERN_ERR 
		       "camelot: Warning; Entry Number of QOS Table > 7\n");
		return(0);
	}
	if ( item >= 12 )  {
		printk(KERN_ERR "camelot: Warning; item of QOS Table > 11\n");
		return(0);
	}
	
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, 0x00000000);
	id = ((port << 3) & 0x8) | (entry & 0x7);
	data = ((id << 12) & 0x0f000) | ((item << 8) & 0x0f00) | 0x00;
	(*sc->sc_write)(sc, TBL_QOSCMD_OFFSET, data);
	data = (*sc->sc_read)(sc, TBL_QOSSTA_OFFSET);
	while(data) {
		data = (*sc->sc_read)(sc, TBL_QOSSTA_OFFSET);
	}
	data = (*sc->sc_read)(sc, TBL_QOSDATA_OFFSET);
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, HOST_INTEN);
	return(data);
}

int read_QOSTBL_entry(struct camelot_softc *sc, int wan_port, int entry,
		      QOS_TBL_ENTRY *qos)
{
	int ports;
	
	memset((char *)qos, 0, sizeof(QOS_TBL_ENTRY));

	if (wan_port != 0 && wan_port != 1) {
		printk(KERN_ERR
		       "camelot: Warning - QOS Port incorrect\n");
		return -1;
	}

	if (entry >= ENTRY_NUM_QOS) {
		printk(KERN_ERR
		       "camelot: Warning - Entry # of QOS Table > 7\n");
		return -1;
	}

	qos->wan_port = wan_port;
	qos->control = read_QOSTBL(sc, wan_port, entry, 11);
	ports = read_QOSTBL(sc, wan_port, entry, 10);
	qos->portnum_dst = (ports >> 16) & 0x0000FFFF;
	qos->portnum_src = ports & 0x0000FFFF;
	qos->flowlabel = read_QOSTBL(sc, wan_port, entry, 9) &
			 0x000FFFFF; /* 20 bits */
	qos->tos = read_QOSTBL(sc, wan_port, entry, 8) & 0xFF; /* 8 bits */
	qos->ipv6addr_dst[3] = read_QOSTBL(sc, wan_port, entry, 7);
	qos->ipv6addr_dst[2] = read_QOSTBL(sc, wan_port, entry, 6);
	qos->ipv6addr_dst[1] = read_QOSTBL(sc, wan_port, entry, 5);
	qos->ipv6addr_dst[0] = read_QOSTBL(sc, wan_port, entry, 4);
	qos->ipv6addr_src[3] = read_QOSTBL(sc, wan_port, entry, 3);
	qos->ipv6addr_src[2] = read_QOSTBL(sc, wan_port, entry, 2);
	qos->ipv6addr_src[1] = read_QOSTBL(sc, wan_port, entry, 1);
	qos->ipv6addr_src[0] = read_QOSTBL(sc, wan_port, entry, 0);
	
	return 0;
}

void read_dump_QOSTBL_entry(struct camelot_softc *sc, int port, int idx)
{
	QOS_TBL_ENTRY qos;

	if (read_QOSTBL_entry(sc, port, idx, &qos)) {
		return;
	}

	/* if valid, dump info */
	if (qos.control & 0x1) {
		pr_debug(
		       "   %2d %08x %08x %08x %08x %08x %04x %04x\n",
		       idx, qos.control, qos.tos,
		       qos.ipv6addr_src[0], qos.ipv6addr_dst[0],
		       qos.flowlabel, qos.portnum_src, qos.portnum_dst);
	}
}

void read_dump_QTALL(struct camelot_softc *sc, int port)
{
	int i;

	pr_debug(
	       "   %s #  ctl      tos      sa      da       fl     sp    dp\n",
	       port ? "Wan" : "Dmz");

	for (i = 0; i < ENTRY_NUM_QOS; i++) {
		read_dump_QOSTBL_entry(sc, port, i);
	}
}

void initial_PPPoE(sc)
	struct camelot_softc *sc;
{
	int  data;

	data = 0;
	(*sc->sc_write)(sc, PPPOE_HEADER_0_OFFSET, data);
	(*sc->sc_write)(sc, PPPOE_HEADER_1_OFFSET, data);
	(*sc->sc_write)(sc, PPPOE_HEADER_2_OFFSET, data);
	(*sc->sc_write)(sc, PPPOE_HEADER_3_OFFSET, data);
 
#if 0
	data = (NO0_SESSION_ID << 16) & 0xffff0000 | (NO0_CODE << 8) & 0xff00 | NO0_VERTYPE & 0xff;
	(*sc->sc_write)(sc, PPPOE_HEADER_0_OFFSET , data);
	data = (NO1_SESSION_ID << 16) & 0xffff0000 | (NO1_CODE << 8) & 0xff00 | NO1_VERTYPE & 0xff;
	(*sc->sc_write)(sc, PPPOE_HEADER_1_OFFSET , data);
	data = (NO2_SESSION_ID << 16) & 0xffff0000 | (NO2_CODE << 8) & 0xff00 | NO2_VERTYPE & 0xff;
	(*sc->sc_write)(sc, PPPOE_HEADER_2_OFFSET , data);
	data = (NO3_SESSION_ID << 16) & 0xffff0000 | (NO3_CODE << 8) & 0xff00 | NO3_VERTYPE & 0xff;
	(*sc->sc_write)(sc, PPPOE_HEADER_3_OFFSET , data);
#endif
}

void initial_V4HEADER(sc)
	struct camelot_softc *sc;
{
	int i, j, data, offset;

	data = 0;
	for (i = 0; i < 4; i++) {
		switch (i) {
		default:
		case 0:
			offset = V4_HEADER_0_0_OFFSET;
			break;
		case 1:
			offset = V4_HEADER_1_0_OFFSET;
			break;
		case 2:
			offset = V4_HEADER_2_0_OFFSET;
			break;
		case 3:
			offset = V4_HEADER_3_0_OFFSET;
			break;
		}

		for (j = 0; j < 5; j++) {
			(*sc->sc_write)(sc, offset+j*4, data);
		}
	}
 
#if 0
	data = (NO0_LENGTH << 16) & 0xffff0000 | (NO0_TOS << 8) & 0xff00 | (NO0_VERSION << 4) & 0xf0 | NO0_IHL;
	(*sc->sc_write)(sc, V4_HEADER_0_0_OFFSET , data);
	data = (NO0_FRAG_FLAG << 16) & 0xffff0000 | NO0_ID;
	(*sc->sc_write)(sc, V4_HEADER_0_1_OFFSET , data);
	data = (NO0_CHECKSUM << 16) & 0xffff0000 | (NO0_PROTOCOL << 8) & 0xff00 |  NO0_TTL;
	(*sc->sc_write)(sc, V4_HEADER_0_2_OFFSET , data);
	(*sc->sc_write)(sc, V4_HEADER_0_3_OFFSET , NO0_SRCADD);
	(*sc->sc_write)(sc, V4_HEADER_0_4_OFFSET , NO0_DSTADD);

	data = (NO1_LENGTH << 16) & 0xffff0000 | (NO1_TOS << 8) & 0xff00 | (NO1_VERSION << 4) & 0xf0 | NO1_IHL;
	(*sc->sc_write)(sc, V4_HEADER_1_0_OFFSET , data);
	data = (NO1_FRAG_FLAG << 16) & 0xffff0000 | NO1_ID;
	(*sc->sc_write)(sc, V4_HEADER_1_1_OFFSET , data);
	data = (NO1_CHECKSUM << 16) & 0xffff0000 | (NO1_PROTOCOL << 8) & 0xff00 |  NO1_TTL;
	(*sc->sc_write)(sc, V4_HEADER_1_2_OFFSET , data);
	(*sc->sc_write)(sc, V4_HEADER_1_3_OFFSET , NO1_SRCADD);
	(*sc->sc_write)(sc, V4_HEADER_1_4_OFFSET , NO1_DSTADD);

	data = (NO2_LENGTH << 16) & 0xffff0000 | (NO2_TOS << 8) & 0xff00 | (NO2_VERSION << 4) & 0xf0 | NO2_IHL;
	(*sc->sc_write)(sc, V4_HEADER_2_0_OFFSET , data);
	data = (NO2_FRAG_FLAG << 16) & 0xffff0000 | NO2_ID;
	(*sc->sc_write)(sc, V4_HEADER_2_1_OFFSET , data);
	data = (NO2_CHECKSUM << 16) & 0xffff0000 | (NO2_PROTOCOL << 8) & 0xff00 |  NO2_TTL;
	(*sc->sc_write)(sc, V4_HEADER_2_2_OFFSET , data);
	(*sc->sc_write)(sc, V4_HEADER_2_3_OFFSET , NO2_SRCADD);
	(*sc->sc_write)(sc, V4_HEADER_2_4_OFFSET , NO2_DSTADD);

	data = (NO3_LENGTH << 16) & 0xffff0000 | (NO3_TOS << 8) & 0xff00 | (NO3_VERSION << 4) & 0xf0 | NO3_IHL;
	(*sc->sc_write)(sc, V4_HEADER_3_0_OFFSET , data);
	data = (NO3_FRAG_FLAG << 16) & 0xffff0000 | NO3_ID;
	(*sc->sc_write)(sc, V4_HEADER_3_1_OFFSET , data);
	data = (NO3_CHECKSUM << 16) & 0xffff0000 | (NO3_PROTOCOL << 8) & 0xff00 |  NO3_TTL;
	(*sc->sc_write)(sc, V4_HEADER_3_2_OFFSET , data);
	(*sc->sc_write)(sc, V4_HEADER_3_3_OFFSET , NO3_SRCADD);
	(*sc->sc_write)(sc, V4_HEADER_3_4_OFFSET , NO3_DSTADD);
#endif
}
