/*
 * drivers/net/mb86977/camelot.c
 *
 * Fujitsu MB86977 driver
 *
 * Author: <source@mvista.com>
 *
 * Copyright (c) 2002-2003 by Fujitsu LSI Solution Ltd..  All Rights Reserved.
 *
 * 2003 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 *
 */

//#define SHOW_CONFIG_AT_INIT
//#define CAMELOT_DEBUG 1
#ifdef	CAMELOT_DEBUG
#define	DEBUG
#endif

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/spinlock.h>
#include <linux/init.h>
#include <asm/uaccess.h>

#include "camelot_var.h"
#include "camelot_reg.h"
#include "camelot_func.h"

extern struct common_softc came_var;
extern NAT_IPF_TBL  natipf_table[1];
static int txbfull;

extern struct cm_softc *cm_unit(int index);
extern u_char came_mac[6];
extern int came_dmz_mode;
extern u_char came_link[4];

#ifdef CAMELOT_DEBUG
static unsigned int camelot_debug = 1;
#else
#define camelot_debug 0
#endif

#ifdef NEED_FIX
/* fix for chip errata */
#define	NATIPF_TABLE_SIZE	(ENTRY_NUM_NATIPF/2)
extern int map_natipf_index(int);
extern void disable_MAC(struct camelot_softc *sc);
#else
/* no fix */
#define	NATIPF_TABLE_SIZE	(ENTRY_NUM_NATIPF)
#define map_natipf_index(index)		(index)
#define map_natipf_pair_index(index)	(0)
#endif

void
camelot_reset(sc)
	struct camelot_softc *sc;
{
	unsigned long flags;

	pr_debug("%s()\n", __FUNCTION__);

	local_irq_save(flags);
	disable_MAC(sc);

	init_camelot(sc);

	local_irq_restore(flags);
}


/*
 * camelot_read -- pull packet off interface and forward to appropriate
 * protocol handler
 */
void
camelot_read(struct cm_softc *cm, int stp, int blen, int mark)
{
	struct camelot_softc *sc = &cm->camelot;
	int addr;
	int wlen, rwlen; 
	int rdp;
	struct sk_buff *skb;
	int status;
	u_char *sdata;

#define ETHER_CRC_LEN	4
#define ETHER_HDR_LEN	14
	/* +++: shouldn't this be 1500? */
#define ETHERMTU	1536
#define ETHERMIN	64

	/* get input data length */
	blen  = blen - ETHER_CRC_LEN;
	if (blen <= ETHER_HDR_LEN ||
	    blen > ETHERMTU + ETHER_HDR_LEN)
	{
		printk(KERN_ERR "cm%d: bad packet length received, %d bytes\n",
		       cm->unit, blen);
		cm->stats.rx_errors++;
		return;
	}

	if (blen % 4)
		wlen = blen/4 + 1;
	else
		wlen = blen/4;

	/* get an skb for packet */
	skb = alloc_skb(blen+4/*2*/, GFP_ATOMIC);
	if (skb == 0) {
		cm->stats.rx_dropped++;
		return;
	}

	skb_reserve(skb, 2);

	sdata = skb_put(skb, blen);

	/* copy packet */
	rdp  = (stp + 1) % RXBUFSIZE;
	addr = (int)(CAME_BANK1 + (rdp * 4));
	if (rdp + wlen > RXBUFSIZE){
		rwlen = rdp + wlen - RXBUFSIZE;
		(*sc->sc_copyfrombuf)(sc, sdata/*skb->data*/, addr, (wlen - rwlen));
		addr = (int)(CAME_BANK1 + 0);
		(*sc->sc_copyfrombuf)(sc, (sdata/*skb->data*/ + 4*(wlen - rwlen)),
				      addr, rwlen);
	}
	else{
		(*sc->sc_copyfrombuf)(sc, sdata/*skb->data*/, addr, wlen);
	}

	/* set up skb */
	skb->len = blen;
	skb->dev = &cm->net;
	skb->protocol = eth_type_trans(skb, &cm->net);

	cm->stats.rx_packets++;
	cm->stats.rx_bytes += blen;

	/* pass the packet up */
	status = netif_rx(skb);

	/* don't log congestion messages */
	if (status != NET_RX_SUCCESS)
		if ((status == NET_RX_DROP) || (status == NET_RX_BAD))
			printk(KERN_ERR "camelot: bad rx status 0x%x\n",
			       status);
}

int camelot_put(struct cm_softc *cm, struct sk_buff *skb)
{
	struct camelot_softc *sc = &cm->camelot;
	int retval = NET_XMIT_SUCCESS;
	int length = skb->len;
	u_int strp, clrp;
	int wemp = 0;
	int wlen, rwlen;
	u_int desc_data = 0;
	u_int ds_add;

#if 0
	/* first check the tramsmited packet total length */
	if (length > ETHERMTU + ETHER_HDR_LEN) {
		printk(KERN_ERR
		       "camelot: transmit packet too large (>1514 bytes)\n");
	}

	if (length < (ETHERMIN + ETHER_HDR_LEN)) {
		/*printk("Send packet runt %d\n", pblen);*/
		length = 60;
	}  
#else
	/* first check the tramsmited packet total length */
	if (length > ETHERMTU) {
		printk(KERN_ERR
		       "camelot: transmit packet too large (>1514 bytes)\n");
	}

	if (length < (ETHERMIN - ETHER_CRC_LEN)) {
		/*printk("Send packet runt %d\n", pblen);*/
		length = ETHERMIN - ETHER_CRC_LEN;
	}  
#endif

	if (length % 4)
		wlen = (length / 4) + 1;
	else
		wlen = length / 4;

	/* Check transmiter buffer capacity */
	strp = (*sc->sc_read)(sc, HOST_TXCPUWRSTP_OFFSET) & 0x03ff;
	clrp = (*sc->sc_read)(sc, HOST_TXCMBCLR_OFFSET) & 0x03ff;

	if (strp == clrp) {
		if (txbfull)
			wemp = 0;
		else
			wemp = TXBUFSIZE;
	} else
		if (strp < clrp)
			wemp = clrp - strp;
		else
			if (strp > clrp)
				wemp = TXBUFSIZE - (strp - clrp);

	if ((wlen + 1) > wemp) {
		printk(KERN_ERR "camelot: waiting TX, not enough space!\n");
		cm->stats.tx_dropped++;
		dev_kfree_skb_any(skb);
		return NET_XMIT_DROP;
	}

	/* fill up the Currrent Tranmsmit Descriptor and buffer */
	switch (cm->unit) {
	case 0:
		desc_data = 0x03000000 | (INTF_WAN << 16);
		break;
	case 1:
		desc_data = 0x02000000;
		if (came_dmz_mode)
			desc_data |= ((INTF_LAN0 | INTF_LAN1) << 16);
		else
			desc_data |= ((INTF_LAN0 | INTF_LAN1 |INTF_DMZ) << 16);
		break;

	case 2:
        	desc_data = 0x03000000 | (INTF_DMZ << 16);
		break;
	default:
		desc_data = 0;
		printk(KERN_ERR
		       "camelot: TX error; unknown ethernet unit %d\n",
		       cm->unit);
		dev_kfree_skb_any(skb);
		return -ENODEV;
	}

	desc_data |= length & 0x0000ffff;

	(*sc->sc_write)(sc, CAME_BANK0 + (strp * 4), desc_data);
	strp = (strp + 1) % TXBUFSIZE;

	memcpy(&skb->data[6], came_mac, 6);

	ds_add = (u_int32_t)(CAME_BANK0 + (strp * 4));
	if (strp + wlen > TXBUFSIZE){
		rwlen = strp + wlen - TXBUFSIZE;
		(*sc->sc_copytobuf)(sc, skb->data, ds_add, (wlen - rwlen));
		ds_add = (int)(CAME_BANK0 + 0);
		(*sc->sc_copytobuf)(sc, (skb->data + 4*(wlen - rwlen)), ds_add, rwlen);
	} else {
		(*sc->sc_copytobuf)(sc, skb->data, ds_add, wlen);
	}

	dev_kfree_skb_any(skb);

	strp = (strp + wlen) % TXBUFSIZE;
	if ((wlen + 1)== wemp)
		txbfull = 1;

	(*sc->sc_write)(sc, HOST_TXCPUWRSTP_OFFSET, strp);

	/* kick transmitter */
	(*sc->sc_write)(sc, HOST_TXSTEN_OFFSET, 0x1);

	cm->stats.tx_packets++;
	cm->stats.tx_bytes += length;

	return retval;
}

/*
 * Receive interrupt routine
 */
void
camelot_rint(struct camelot_softc *sc)
{
	int rxpcknum, reread;
	int rendp, stp;
	int blen;
	u_int rx_stat, rx_port;

	rxpcknum = (*sc->sc_read)(sc, HOST_RXPACKNUM_OFFSET) & 0x3f;

	/* errata - read until zero */
	reread = (*sc->sc_read)(sc, HOST_RXPACKNUM_OFFSET) & 0x3f;
	while (reread != 0) {
		rxpcknum += reread;
		reread = (*sc->sc_read)(sc, HOST_RXPACKNUM_OFFSET) & 0x3f;
	}

	if (rxpcknum == 0) {
		printk(KERN_ERR "camelot: receive packet number = %d\n",
		       rxpcknum);
		return;
	}

	while (rxpcknum) {
		rendp = (*sc->sc_read)(sc, HOST_RXCPURDEND_OFFSET) & 0x03ff;
		stp   = (rendp + 1) % RXBUFSIZE;

		rx_stat =
			(*sc->sc_read)(sc, CAME_BANK1+(stp*4)) & 0xffffffff;

		blen = rx_stat & 0x000007ff;
		rx_port = (rx_stat >> 16) & 0x00ff;

		if (rx_port & INTF_LAN0) {
			camelot_read(cm_unit(1), stp, blen, 0); 
		} else if (rx_port & INTF_LAN1) {
			camelot_read(cm_unit(1), stp, blen, 1); 
		} else if (rx_port & (INTF_DMZ | INTF_DMZ_H)) {
			if (came_dmz_mode) {
				camelot_read(cm_unit(2), stp, blen, 2);
			} else {
				camelot_read(cm_unit(1), stp, blen, 2);
			}
		} else if (rx_port & (INTF_WAN | INTF_WAN_H)){
			camelot_read(cm_unit(0), stp, blen, 3); 
		} else {
			printk(KERN_ERR "camelot: RX error, bad port %08x\n",
			       rx_port); 
#if 0
			print_rcv_pkt(sc, stp, blen);
#endif
			return;
		}

		if (blen % 4)
			stp = stp + blen/4 + 1 ;
		else
			stp = stp + blen/4 ;

		stp = stp % RXBUFSIZE;

		(*sc->sc_write)(sc, HOST_RXCPURDEND_OFFSET, stp);
		rxpcknum--;
	}
}

void
camelot_tint(struct camelot_softc *sc)
{
	(*sc->sc_write)(sc, HOST_TXSTEN_OFFSET, 0x0);
}

static u_int ttl0_log[ENTRY_NUM_TTL0LOG];
static u_int flt_log_out[ENTRY_NUM_LOG][16];
static u_int flt_log_in[ENTRY_NUM_LOG][16];
static u_int rstfin_log[ENTRY_NUM_RSTFINLOG];

#include <linux/ip.h>
#include <linux/icmp.h>
#include <asm/checksum.h>
#include <linux/inetdevice.h>

static void
create_eth_reply_header(struct ethhdr *new, struct ethhdr *orig)
{
	memset(new, 0, sizeof(struct ethhdr));
	new->h_proto = orig->h_proto;
	memcpy(new->h_dest, orig->h_source, 6);
	memcpy(new->h_source, orig->h_dest, 6);
}

static void
create_ip_reply_header(struct iphdr *new, struct iphdr *orig,
		       unsigned int src_addr)
{
	memset(new, 0, sizeof(struct iphdr));
	new->version = orig->version;
	new->ihl = 0x45;
	new->tos = orig->tos;
	new->tot_len = htons(16 + sizeof(struct icmphdr) +
			     2 * sizeof(struct iphdr));
	new->id = htons(0x1A1B);
	new->ttl = 0xFF;
	new->protocol = 1; /* ICMP */
	/* use the appropriate interface IP */
	new->saddr = src_addr;
	new->daddr = orig->saddr;
	new->check = ip_compute_csum((unsigned char *)new,
				     sizeof(struct iphdr));
}

static void
create_icmp_ttl_header(struct icmphdr *icmp, struct iphdr *orig_iphdr,
		       unsigned char *icmp_data_ptr)
{
	unsigned char tmp_buf[sizeof(struct icmphdr) +
			      sizeof(struct iphdr) + 16];
	unsigned char *ptr = tmp_buf;
	
	memset(icmp, 0, sizeof(struct icmphdr));
	
	icmp->type = ICMP_TIME_EXCEEDED;
	icmp->code = ICMP_EXC_TTL;
	
	memcpy(ptr, icmp, sizeof(struct icmphdr));
	ptr += sizeof(struct icmphdr);
	memcpy(ptr, orig_iphdr, sizeof(struct iphdr));
	ptr += sizeof(struct iphdr);
	memcpy(ptr, icmp_data_ptr, 16);
	ptr += 16;
	
	icmp->checksum = ip_compute_csum(tmp_buf, (ptr-tmp_buf));
}

static void
copy_pkt_to_skb(u_char *sdata, struct ethhdr *eth, struct iphdr *ip,
		struct icmphdr *icmp, struct iphdr *orig_ip,
		unsigned char *icmp_data)
{

	int size;
	
	memcpy(sdata, eth, sizeof(struct ethhdr));
	size = sizeof(struct ethhdr);
	memcpy(sdata+size, ip, sizeof(struct iphdr));
	size += sizeof(struct iphdr);
	memcpy(sdata+size, icmp, sizeof(struct icmphdr));
	size += sizeof(struct icmphdr);
	memcpy(sdata+size, orig_ip, sizeof(struct iphdr));
	size += sizeof(struct iphdr);
	memcpy(sdata+size, icmp_data, 16);
}

#define NIPQUAD(addr) \
	((unsigned char *)&addr)[0], \
	((unsigned char *)&addr)[1], \
	((unsigned char *)&addr)[2], \
	((unsigned char *)&addr)[3]

static void
dump_ip_hdr(struct iphdr *ip_hdr)
{
	pr_debug("IP Header\n"
	       "  version:     %x\n"
	       "  ihl          %x\n"
	       "  tos          %x\n"
	       "  len          %02x\n"
	       "  id           %02x\n"
	       "  frag_off     %02x\n"
	       "  ttl          %x\n"
	       "  proto        %x\n"
	       "  checksum     %02x\n"
	       "  saddr        %u.%u.%u.%u\n"
	       "  daddr        %u.%u.%u.%u\n",
	       ip_hdr->version,
	       ip_hdr->ihl,
	       ip_hdr->tos,
	       ntohs(ip_hdr->tot_len),
	       ntohs(ip_hdr->id),
	       ip_hdr->frag_off,
	       ip_hdr->ttl,
	       ip_hdr->protocol,
	       ntohs(ip_hdr->check),
	       NIPQUAD(ip_hdr->saddr),
	       NIPQUAD(ip_hdr->daddr));
}

static void
dump_eth_hdr(struct ethhdr *eth_hdr)
{
	pr_debug("Eth Hdr: dst %02x:%02x:%02x:%02x:%02x:%02x  "
	       "src %02x:%02x:%02x:%02x:%02x:%02x pro %04x\n",
	       eth_hdr->h_dest[0], eth_hdr->h_dest[1],
	       eth_hdr->h_dest[2], eth_hdr->h_dest[3],
	       eth_hdr->h_dest[4], eth_hdr->h_dest[5],
	       eth_hdr->h_source[0], eth_hdr->h_source[1],
	       eth_hdr->h_source[2], eth_hdr->h_source[3],
	       eth_hdr->h_source[4], eth_hdr->h_source[5],
	       ntohs(eth_hdr->h_proto));
}	

static void
camelot_intr_ttl0(struct cm_softc *cm)
{
	struct cm_softc *cm_ptr;
	struct camelot_softc *sc = &cm->camelot;
	struct net_device *net_dev;
	struct in_device *in_dev;
	struct in_ifaddr **ifap = NULL;
	struct in_ifaddr *ifa = NULL;
	struct sk_buff *skb;
	struct ethhdr *eth_hdr, eth;
	struct iphdr *ip_hdr, ip;
	struct icmphdr icmp;
	int i, length, ret, intf;
	unsigned long intf_ip = 0;
	u_char *sdata;
	unsigned char *ip_data;
	char *dev_name;

	pr_debug("%s: TTL0 error in FEF\n", __FUNCTION__);

	read_TTL0LOG(sc, ttl0_log);
	for (i = 0; i < ENTRY_NUM_TTL0LOG; i++) {
		pr_debug("ttl0_log[%x] = 0x%08x\n", i, ttl0_log[i]);
	}

	intf = ttl0_log[0] >> 16 & 0xFF;

	eth_hdr = (struct ethhdr *)&ttl0_log[1];

	/* make sure it's IP */
	if (ntohs(eth_hdr->h_proto) != 0x0800) {
		pr_debug("Unhandled TTL = 0 packet (Ether type %04x)\n",
			 ntohs(eth_hdr->h_proto));
		return;
	}

	create_eth_reply_header(&eth, eth_hdr);

	/* grab pointers to IP header and data */
	ip_hdr = (struct iphdr *)
		 ((unsigned char *)eth_hdr + sizeof(struct ethhdr));
	ip_data = (unsigned char *)ip_hdr + sizeof(struct iphdr);

	/* get the send device; default to LAN */
	cm_ptr = cm_unit(1);
	dev_name = "cm0";
	if (((intf & INTF_DMZ) || (intf & INTF_DMZ_H)) && came_dmz_mode) {
		cm_ptr = cm_unit(2);
		dev_name = "cm1";
	}
	else if ((intf & INTF_WAN) || (intf & INTF_WAN_H)) {
		cm_ptr = cm_unit(0);
		dev_name = "wan0";
	}

	net_dev = &cm_ptr->net;
	in_dev = __in_dev_get(net_dev);

	/* find ip address */
	for (ifap=&in_dev->ifa_list; (ifa=*ifap) != NULL; ifap=&ifa->ifa_next) {
		if (strcmp(dev_name, ifa->ifa_label) == 0) {
			intf_ip = ifa->ifa_address;
			break; /* found */
		}
	}

	if (intf_ip == 0) {
		/* uh oh...*/
		pr_debug("No IP address found from which to send ICMP "
			 "packet\n");
		return;
	}

	create_ip_reply_header(&ip, ip_hdr, intf_ip);
	create_icmp_ttl_header(&icmp, ip_hdr, ip_data);
	
	/* get an skb for packet */
	length = 16 + sizeof(icmp) + 2 * sizeof(ip) + sizeof(eth);
	skb = alloc_skb(length, GFP_ATOMIC);
	if (skb == 0) {
		printk(KERN_WARNING "SKB is null\n");
		return;
	}

	skb_reserve(skb, 2);
	sdata = skb_put(skb, length);
	copy_pkt_to_skb(sdata, &eth, &ip, &icmp, ip_hdr, ip_data);

	pr_debug("Sending ICMP (cm->unit is %d, intf is %d)\n",
		 cm->unit, intf);
	ret = camelot_put(cm_ptr, skb);
	if (ret != NET_XMIT_SUCCESS) {
		pr_debug("Error sending ICMP\n");
	}
	
	cm->stats.rx_errors++;
	/*camelot_reset(sc);*/
}

static void
camelot_intr_log_oflo(struct cm_softc *cm)
{
	struct camelot_softc *sc = &cm->camelot;
	int i, j, data, status;

	printk(KERN_ERR "camelot: filter log overflow\n");

	read_FLTLOG(sc, flt_log_in, flt_log_out, &status);

	pr_debug(" Status %08x\n", status);

	pr_debug(" Out-Side Filter  \n");
	for(i=0; i < ENTRY_NUM_LOG ; i++){
		pr_debug(" ENTRY NUMBER = %x \n", i);
		for(j=0; j < 16 ; j++){
			data = flt_log_out[i][j] ;
			pr_debug(" ID = %x   %x \n", j, data);
		}
	}
	pr_debug(" In-Side Filter  \n");
	for(i=0; i < ENTRY_NUM_LOG ; i++){
		pr_debug(" ENTRY NUMBER = %x \n", i);
		for(j=0; j < 16 ; j++){
			data = flt_log_in[i][j] ;
			pr_debug(" ID = %x   %x \n", j, data);
		}
	}
}

/* callback function pointer for handling RST/FIN events */
static int (*rstfin_callback_func)(unsigned long srci,
				   unsigned long dsti,
				   unsigned long nati,
				   unsigned short srcp,
				   unsigned short dstp,
				   unsigned short natp) = NULL;

/*
 * Register a RST/FIN callback function pointer.
 * When a callback function pointer is registered, it is invoked when
 * a RST/FIN interrupt occurs.
 */
int
camelot_register_rstfin_callback(int (*func)(unsigned long srci,
					     unsigned long dsti,
					     unsigned long nati,
					     unsigned short srcp,
					     unsigned short dstp,
					     unsigned short natp))
{
	if (camelot_debug)
                pr_debug("%s\n", __FUNCTION__);
	if (rstfin_callback_func) {
		return -1;
	}
	rstfin_callback_func = func;
	return 0;
}

/* Unregister a RST/FIN callback function pointer */
int
camelot_unregister_rstfin_callback(int (*func)(unsigned long srci,
					       unsigned long dsti,
					       unsigned long nati,
					       unsigned short srcp,
					       unsigned short dstp,
					       unsigned short natp))
{
	if (camelot_debug)
                pr_debug("%s\n", __FUNCTION__);
	if (func != rstfin_callback_func){
		return -1;
	}
	rstfin_callback_func = NULL;
	return 0;
}
	
/* callback function pointer for handling RST/FIN events within driver */
static int (*rstfin_callback_func_by_id)(int id) = NULL;

/*
 * Register a RST/FIN callback function pointer.
 * When a callback function pointer is registered, it is invoked when
 * a RST/FIN interrupt occurs.
 */
int
camelot_register_rstfin_callback_by_id(int (*func)(int id))
{
	if (camelot_debug)
                pr_debug("%s\n", __FUNCTION__);
	if (rstfin_callback_func_by_id) {
		return -1;
	}
	rstfin_callback_func_by_id = func;
	return 0;
}

/* Unregister a RST/FIN callback function pointer */
int
camelot_unregister_rstfin_callback_by_id(int (*func)(int id))
{
	if (camelot_debug)
                pr_debug("%s\n", __FUNCTION__);
	if (func != rstfin_callback_func_by_id){
		return -1;
	}
	rstfin_callback_func_by_id = NULL;
	return 0;
}
	
/*
 * Handle a RST/FIN interrupt.
 * Log the RST/FIN event and invoke the RST/FIN callback function, if
 * one is registered.
 */ 
static void
camelot_intr_rst_fin(struct cm_softc *cm)
{
	struct camelot_softc *sc = &cm->camelot;
	u_long ip_src, ip_dst, ip_nat;
	u_short port_src, port_dst, port_nat;
	int i, data, id, index;
#if 0
	int i, j, data, id, index;
#endif

	read_RFLOG(sc, rstfin_log);

	for (i = 0; i < NATIPF_TABLE_SIZE ; i++) {
		/* map index into real index */
		index = map_natipf_index(i);

		data = rstfin_log[index/32];
		if (data == 0x0)
			continue;
			
		if ((data & (1 << (index%32))) == 0)
			continue;

		id = index;
		ip_src = (u_long)read_NATIPFTBL(sc, id, 0);
		ip_dst = (u_long)read_NATIPFTBL(sc, id, 1);
		ip_nat = (u_long)read_NATIPFTBL(sc, id, 2);
		port_src = (u_short)read_NATIPFTBL(sc, id, 12);
		port_dst = (u_short)read_NATIPFTBL(sc, id, 13);
		port_nat = (u_short)read_NATIPFTBL(sc, id, 14);
		if (camelot_debug) {
			pr_debug("Processing RST/FIN:\n"
				 "  sIP 0x%4lx dIP 0x%4lx "
				 "nIP 0x%4lx\n"
				 "   sPt 0x%2x dPt 0x%2x "
				 "nPt 0x%2x\n",
				 ip_src, ip_dst, ip_nat,
				 port_src, port_dst, port_nat);
		}
		if (rstfin_callback_func_by_id) {
			rstfin_callback_func_by_id(i);
		}
		if (rstfin_callback_func) {
			rstfin_callback_func(ip_src, ip_dst,
					     ip_nat,
					     port_src,
					     port_dst,
					     port_nat);
		}
	}
#if 0
	for (i = 0; i < ENTRY_NUM_RSTFINLOG ; i++) {
		for (i = 0; i < ENTRY_NUM_RSTFINLOG ; i++) {
			
			data = rstfin_log[i];
			if (data == 0x0)
		                continue;
			
			for (j = 0; j < 32; j++) {
				if ((data & (1 << j)) == 0)
			                continue;

				id = j + 32*i;
				ip_src = (u_long)read_NATIPFTBL(sc, id, 0);
				ip_dst = (u_long)read_NATIPFTBL(sc, id, 1);
				ip_nat = (u_long)read_NATIPFTBL(sc, id, 2);
				port_src = (u_short)read_NATIPFTBL(sc, id, 12);
				port_dst = (u_short)read_NATIPFTBL(sc, id, 13);
				port_nat = (u_short)read_NATIPFTBL(sc, id, 14);
				if (camelot_debug) {
					pr_debug("Processing RST/FIN:\n"
					       "  sIP 0x%4lx dIP 0x%4lx "
					       "nIP 0x%4lx\n"
					       "   sPt 0x%2x dPt 0x%2x "
					       "nPt 0x%2x\n",
					       ip_src, ip_dst, ip_nat,
					       port_src, port_dst, port_nat);
				}
				if (rstfin_callback_func_by_id) {
					rstfin_callback_func_by_id(id);
				}
				if (rstfin_callback_func) {
					rstfin_callback_func(ip_src, ip_dst,
							     ip_nat,
							     port_src,
							     port_dst,
							     port_nat);
				}
			}
		}
	}
#endif
}

void
camelot_intr(int irq, void *dev_id, struct pt_regs *regs)
{
	struct cm_softc *cm = (struct cm_softc *)dev_id;
	struct camelot_softc *sc = &cm->camelot;
	int  status[4];
	int  i, data;
	int  mask, hstatus;

	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, 0x00000000);
	set_table_access_priority(sc);

	hstatus = (*sc->sc_read)(sc, HOST_STATUS_OFFSET);
 
	/* RX/TX */
	if (hstatus & 0x2) {
		camelot_rint(sc); 
	}

	if (hstatus & 0x1) {
		camelot_tint(sc); 
	}

	/* TTL0 Error */
	if (hstatus & 0x10000) {
		camelot_intr_ttl0(cm);
	}

	/* SMI Interrupt Status */
	if (hstatus & 0x8000) {
		data = read_SMIST(sc);

		if (data & 0x00080000) {
			/* Link Status Change WAN */
			if (data & 0x00008000) {
				came_link[3] = 1;
				if (cm_unit(0))
					netif_carrier_on(&cm_unit(0)->net);
			} else {
				came_link[3] = 0;
				if (cm_unit(0))
					netif_carrier_off(&cm_unit(0)->net);
			}
		}
		if (data & 0x00040000) {
			/* Link Status Change DMZ */
			if (data & 0x00000800) {
				came_link[2] = 1;
				if (cm_unit(2))
					netif_carrier_on(&cm_unit(2)->net);
			} else {
				came_link[2] = 0;
				if (cm_unit(2))
					netif_carrier_off(&cm_unit(2)->net);
			}
		}
		if (data & 0x00020000) {
			/* Link Status Change LAN1 */
			if (data & 0x00000080) {
				came_link[1] = 1;
				if ((came_link[0] == 0) && cm_unit(1))
					netif_carrier_on(&cm_unit(1)->net);
			} else {
				came_link[1] = 0;
				if ((came_link[0] == 0) && cm_unit(1))
					netif_carrier_off(&cm_unit(1)->net);
			}
		}
		if (data & 0x00010000) {
			/* Link Status Change LAN0 */
			if (data & 0x00000008) {
				came_link[0] = 1;
				if ((came_link[1] == 0) && cm_unit(1))
					netif_carrier_on(&cm_unit(1)->net);
			} else {
				came_link[0] = 0;
				if ((came_link[1] == 0) && cm_unit(1))
					netif_carrier_off(&cm_unit(1)->net);
			}
		}
		if (data & 0x00008888){
		}
	}

	/* Out-Side Filter Counter Over Flow Interrupt Status */
	if (hstatus & 0x4000) {
		printk(KERN_ERR "camelot: Out Filter Counter = 255 :");
		status[0] = read_FLCNTST(sc, 1, 1);
		status[1] = read_FLCNTST(sc, 1, 2);
               
		/* Part 1 Read */
		mask = 0x01;
		for(i=0; i < 32; i++){
			if (status[0] & mask) {
				incr_FLCNTV(sc, 1, i);
				data = read_FLCNTV(sc, 1, i);
				pr_debug("Entry %d %x \n", i, data);
			}
			mask <<= 1;
		}
               
		/* Part 2 Read */
		mask = 0x01;
		for(i=0; i < 32; i++){
			if (status[1] & mask) {
				incr_FLCNTV(sc, 1, (i+32));
				data = read_FLCNTV(sc, 1, (i+32));
				pr_debug("Entry %d %x \n",
				       (i+32), data);
			}
			mask <<= 1;
		}
	}

	/* In-Side Filter Counter Over Flow Interrupt Status */
	if (hstatus & 0x2000) {
		printk(KERN_ERR "camelot: In Filter Counter = 255:");
		status[0] = read_FLCNTST(sc, 0, 1);
		status[1] = read_FLCNTST(sc, 0, 2);
               
		/* Part 1 Read */
		mask = 0x01 ;
		for(i=0; i < 32; i++){
			if (status[0] & mask) {
				incr_FLCNTV(sc, 0, i);
				data = read_FLCNTV(sc, 0, i);
				pr_debug("Entry %d %x \n", i, data);
			}
			mask <<= 1;
		}
               
		/* Part 2 Read */
		mask = 0x01;
		for(i=0; i < 32; i++){
			if (status[1] & mask) {
				incr_FLCNTV(sc, 0, (i+32));
				data = read_FLCNTV(sc, 0, (i+32));
				pr_debug("Entry %d %x \n",
				       (i+32), data);
			}
			mask <<= 1;
		}
	}

	/* MAC Port Status   */
	if (hstatus & 0x1000) {
#ifndef	DEBUG
		/* REVISIT: Driver failure with "bad rx status"
		 * side-effect if we do not do read_MACST() here.
		 */
		read_MACST(sc, 3);
#endif
		pr_debug("camelot: MAC Port 3 Interrupt Status %x\n",
		       read_MACST(sc, 3));
		/*printk( " SMI STatus %x \n", read_SMIST(sc));*/
	}

	if (hstatus & 0x800) {
#ifndef	DEBUG
		/* REVISIT: Driver failure with "bad rx status"
		 * side-effect if we do not do read_MACST() here.
		 */
		read_MACST(sc, 2);
#endif
		pr_debug("camelot: MAC Port 2 Interrupt Status %x\n",
		       read_MACST(sc, 2));
		/*printk( " SMI STatus %x \n", read_SMIST(sc));*/
	}

	if (hstatus & 0x400) {
#ifndef	DEBUG
		/* REVISIT: Driver failure with "bad rx status"
		 * side-effect if we do not do read_MACST() here.
		 */
		read_MACST(sc, 1);
#endif
		pr_debug("camelot: MAC Port 1 Interrupt Status %x\n",
		       read_MACST(sc, 1));
		/*printk( " SMI STatus %x \n", read_SMIST(sc));*/
	}

	if (hstatus & 0x200) {
#ifndef	DEBUG
		/* REVISIT: Driver failure with "bad rx status"
		 * side-effect if we do not do read_MACST() here.
		 */
		read_MACST(sc, 0);
#endif
		pr_debug("camelot: MAC Port 0 Interrupt Status %x\n",
		       read_MACST(sc, 0));
		/*printk( " SMI STatus %x \n", read_SMIST(sc));*/
	}

	/* Filter LOG Over Flow   */
	if (hstatus & 0x100) {
		camelot_intr_log_oflo(cm);
	}

	/*  FIN/RST Catch  */
	if (hstatus & 0x80) {
		camelot_intr_rst_fin(cm);
	}

	/* Switch Block Flow Control   */
	if (hstatus & 0x40) {
		printk(KERN_ERR
		       "camelot: Switch Block Flow Control interrupt\n");
	}

	/* Switch Block PRAM Full   */
	if (hstatus & 0x20) {
		printk(KERN_ERR "camelot: Switch Block PRAM Full Interrupt\n");
	}

	/* Host Deadlock   */
	if (hstatus & 0x10) {
		printk(KERN_ERR "camelot: Host Deadlock Interrupt!!\n");
		cm->stats.rx_errors++;
		camelot_reset(sc);
	}

	/* TX/RX Overrun */
	if (hstatus & 0x8) {
		printk(KERN_ERR "camelot: TX overrun!!\n");
		cm->stats.rx_errors++;
		sc->sc_havecarrier = 0;
		camelot_reset(sc);
	}

	if (hstatus & 0x4) {
		printk(KERN_ERR "camelot: RX Underrun!!\n");
		cm->stats.rx_errors++;
		camelot_reset(sc);
	}

	reset_table_access_priority(sc);
	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, HOST_INTEN);
}

void
camelot_config(struct camelot_softc *sc)
{
	init_camelot(sc);

#ifdef SHOW_CONFIG_AT_INIT
	read_dump_L2ALL(sc);
	read_dump_NTALL(sc);
	read_dump_FLTALL(sc, 0);
	read_dump_FLTALL(sc, 2);
	read_dump_QTALL(sc, 0);
	read_dump_QTALL(sc, 1);
#endif
}

