/*
************************************************************************************
korina.c

Copyright 2003 Integrated Device Technology, Inc.
Author: Integrated Device Technology, Inc.

Copyright 2002 MontaVista Software Inc.
Author: MontaVista Software, Inc.
	stevel@mvista.com or source@mvista.com
  
A driver for the IDT RC32434 on-chip ethernet controller.
************************************************************************************
B Maruthanayagam: 	Ported from sonic.c 
************************************************************************************
cgg: 			Modified code. 
************************************************************************************
P Sadik: 		Applied filters.
************************************************************************************
Haofeng Kou			6/7/03

Using the IOD COD and DMA chaining mode to handle the Transmit. 
In order to prevent the DMA Race Condition, make software access NDPTR only for tansmit.
Using the IOF COF and NDPTR of DMA chaining to handle the Receive.
In order to prevent the DMA Race Condition, use NDPTR 
 trigger DMA chaining once overwrite happen for receive.
Modify korina_restart() to make it recover Bridge mode setting after reset.
Adjust the dev_kfree_skb_any() and dev_kfree_skb_irq() usage.
Adjust the netif_wake_queue() and netif_start_queue() usage.
Add in code to handle the case once there is no memory in system(only for special situations).
************************************************************************************
Haofeng Kou			6/16/03
Add in code to handle the TX Underflow.
************************************************************************************
Kiran Rao 
Ported to Korina.
************************************************************************************
Haofeng Kou			10/09/03
Trigger the waiting queue in the TX ISR
Finish the Korina porting.
*************************************************************************************
P. Sadik  Oct 10, 2003
'Management Clock Prescaler Divisor' is no longer determined at compile time, but
is caculated at run time, based on idt_cpu_freq.
************************************************************************************
Haofeng Kou			10/27/03
Change the ISR to use Tasklet for the TX and RX.
Add in the DMA Abort code and use DMA Abort instead of the DMA halt. (DMA abort finished 
faster than DMA halt, so replace the halt code by abort)
Optimize the interrupt disable/enable actions to reduce the number of interrupt.
Improve the CPU usage.
*************************************************************************************
Haofeng Kou			11/18/03
Update all the debug features(ASSERT, DBG, ERR, INFO and WARN).
Add the MII-PHY LXT972 duplex mode auto-negotiation support.(Not enabled at this time)
*************************************************************************************
*/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/ctype.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/interrupt.h>
#include <linux/ptrace.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <linux/proc_fs.h>
#include <linux/in.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/delay.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/errno.h>
#include <asm/bootinfo.h>
#include <asm/system.h>
#include <asm/bitops.h>
#include <asm/pgtable.h>
#include <asm/segment.h>
#include <asm/io.h>
#include <asm/dma.h>

#include "korina.h"
#include  <asm/rc32434/rc32434.h>

#define STATION_ADDRESS_HIGH(dev) (((dev)->dev_addr[0] << 8) | \
			           ((dev)->dev_addr[1]))
#define STATION_ADDRESS_LOW(dev)  (((dev)->dev_addr[2] << 24) | \
				   ((dev)->dev_addr[3] << 16) | \
				   ((dev)->dev_addr[4] << 8)  | \
				   ((dev)->dev_addr[5]))

#define MII_CLOCK 1250000	/* no more than 2.5MHz */
static char mac[18] = "08:00:06:05:40:01";

MODULE_PARM(mac, "c18");
MODULE_PARM(mac1, "c18");
MODULE_PARM_DESC(mac, "MAC address for RC32434 ethernet");
MODULE_PARM_DESC(mac1, "MAC address for RC32434 ethernet1");

static struct korina_if_t {
	struct net_device *dev;
	char *mac_str;
	u32 iobase;
	int rx_dma_irq;
	int tx_dma_irq;
	int rx_ovr_irq;
	int tx_und_irq;
} korina_iflist[] = {
	{
	NULL, mac, ETH_PhysicalAddress,
		    ETH_DMA_RX_IRQ, ETH_DMA_TX_IRQ, ETH_RX_OVR_IRQ,
		    ETH_TX_UND_IRQ}, {
	NULL, 0, 0}
};

static int
parse_mac_addr(struct net_device *dev, char *macstr)
{
	int i, j;
	unsigned char result, value;

	for (i = 0; i < 6; i++) {
		result = 0;
		if (i != 5 && *(macstr + 2) != ':') {
			pr_debug(__FILE__ "invalid mac address format: %d %c\n",
				 i, *(macstr + 2));
			return -EINVAL;
		}

		for (j = 0; j < 2; j++) {
			if (isxdigit(*macstr) &&
			    (value = isdigit(*macstr) ? *macstr - '0' :
			     toupper(*macstr) - 'A' + 10) < 16) {
				result = result * 16 + value;
				macstr++;
			} else {
				pr_debug(__FILE__ "invalid mac address "
					 "character: %c\n", *macstr);
				return -EINVAL;
			}
		}

		macstr++;
		dev->dev_addr[i] = result;
	}

	return 0;
}

static inline void
korina_abort_tx(struct net_device *dev)
{
	struct korina_local *lp = (struct korina_local *) dev->priv;

	rc32434_abort_dma(dev, lp->tx_dma_regs);

}
static inline void
korina_abort_rx(struct net_device *dev)
{
	struct korina_local *lp = (struct korina_local *) dev->priv;

	rc32434_abort_dma(dev, lp->rx_dma_regs);

}
static inline void
korina_halt_tx(struct net_device *dev)
{
	struct korina_local *lp = (struct korina_local *) dev->priv;
	if (rc32434_halt_dma(lp->tx_dma_regs))
		pr_debug("%s: timeout!\n", __FUNCTION__);
}
static inline void
korina_halt_rx(struct net_device *dev)
{
	struct korina_local *lp = (struct korina_local *) dev->priv;
	if (rc32434_halt_dma(lp->rx_dma_regs))
		pr_debug("%s: timeout!\n", __FUNCTION__);
}

static inline void
korina_start_tx(struct korina_local *lp, volatile DMAD_t td)
{
	rc32434_start_dma(lp->tx_dma_regs, PHYSADDR(td));
}
static inline void
korina_start_rx(struct korina_local *lp, volatile DMAD_t rd)
{
	rc32434_start_dma(lp->rx_dma_regs, PHYSADDR(rd));
}

static inline void
korina_chain_tx(struct korina_local *lp, volatile DMAD_t td)
{
	rc32434_chain_dma(lp->tx_dma_regs, PHYSADDR(td));
}
static inline void
korina_chain_rx(struct korina_local *lp, volatile DMAD_t rd)
{
	rc32434_chain_dma(lp->rx_dma_regs, PHYSADDR(rd));
}

#ifdef KORINA_PROC_DEBUG
static int
korina_read_proc(char *buf, char **start, off_t fpos,
		 int length, int *eof, void *data)
{
	struct net_device *dev = (struct net_device *) data;
	struct korina_local *lp = (struct korina_local *) dev->priv;
	int len = 0;

	/* print out header */
	len += sprintf(buf + len, "\n\tRC32434 Ethernet Debug\n\n");

	len += sprintf(buf + len,
		       "DMA halt count      = %10d, total pkt cnt = %10d\n",
		       lp->dma_halt_cnt, lp->halt_tx_count);
	len += sprintf(buf + len,
		       "DMA run count       = %10d, total pkt cnt = %10d\n",
		       lp->dma_run_cnt, lp->run_tx_count);
	len += sprintf(buf + len,
		       "DMA race count      = %10d, total pkt cnt = %10d\n",
		       lp->dma_race_cnt, lp->race_tx_count);
	len += sprintf(buf + len,
		       "DMA collision count = %10d, total pkt cnt = %10d\n",
		       lp->dma_collide_cnt, lp->collide_tx_count);

	if (fpos >= len) {
		*start = buf;
		*eof = 1;
		return 0;
	}
	*start = buf + fpos;
	if ((len -= fpos) > length)
		return length;
	*eof = 1;
	return len;

}
#endif

/*
 * Restart the KORINA ethernet controller. 
 */
static int
korina_restart(struct net_device *dev)
{
	struct korina_local *lp = (struct korina_local *) dev->priv;

	/*
	 * Disable interrupts
	 */
	disable_irq(lp->rx_irq);
	disable_irq(lp->tx_irq);
	disable_irq(lp->und_irq);

	/* Mask F E bit in Tx DMA */
	local_writel(local_readl(&lp->tx_dma_regs->dmasm) | DMASM_f_m |
		     DMASM_e_m, &lp->tx_dma_regs->dmasm);
	/* Mask D H E bit in Rx DMA */
	local_writel(local_readl(&lp->rx_dma_regs->dmasm) | DMASM_d_m |
		     DMASM_h_m | DMASM_e_m, &lp->rx_dma_regs->dmasm);

	korina_init(dev);
	korina_multicast_list(dev);

	enable_irq(lp->und_irq);
	enable_irq(lp->tx_irq);
	enable_irq(lp->rx_irq);

	return 0;
}

int
korina_init_module(void)
{
	int i, retval = 0;

	for (i = 0; korina_iflist[i].iobase; i++) {
		retval |= korina_probe(i);
	}

	return retval;
}

static int
korina_probe(int port_num)
{
	struct korina_local *lp = NULL;
	struct korina_if_t *bif = NULL;
	struct net_device *dev = NULL;
	int i, retval;
	bif = &korina_iflist[port_num];

	request_region(bif->iobase, 0x24C, "KORINA");

	/* Allocate a new 'dev' if needed */
	dev = init_etherdev(0, sizeof (struct korina_local));
	bif->dev = dev;

	INFO("RC32434 ethernet found at 0x%08x\n", bif->iobase);

	/* Fill in the 'dev' fields. */
	dev->base_addr = bif->iobase;
	/* just use the rx dma irq */
	dev->irq = bif->rx_dma_irq;

	if ((retval = parse_mac_addr(dev, bif->mac_str))) {
		pr_debug("%s: MAC address parse failed\n", __FUNCTION__);
		retval = -EINVAL;
		goto probe_err_out;
	}

	INFO("HW Address ");
	for (i = 0; i < 6; i++) {
		printk("%2.2x", dev->dev_addr[i]);
		if (i < 5)
			printk(":");
	}
	printk("\n");

	INFO("Rx IRQ %d, Tx IRQ %d\n", bif->rx_dma_irq, bif->tx_dma_irq);

	/* Initialize the device structure. */
	if (dev->priv == NULL) {
		lp = (struct korina_local *) kmalloc(sizeof (*lp), GFP_KERNEL);
		memset(lp, 0, sizeof (struct korina_local));
	} else {
		lp = (struct korina_local *) dev->priv;
	}

	dev->priv = lp;

	lp->rx_irq = bif->rx_dma_irq;
	lp->tx_irq = bif->tx_dma_irq;
	lp->ovr_irq = bif->rx_ovr_irq;
	lp->und_irq = bif->tx_und_irq;

	lp->eth_regs = ioremap_nocache(bif->iobase, sizeof (*lp->eth_regs));

	if (!lp->eth_regs) {
		pr_debug("Can't remap eth registers\n");
		retval = -ENXIO;
		goto probe_err_out;
	}

	lp->rx_dma_regs =
	    ioremap_nocache(DMA0_PhysicalAddress + 0 * DMA_CHAN_OFFSET,
			    sizeof (struct DMA_Chan_s));

	if (!lp->rx_dma_regs) {
		pr_debug("Can't remap Rx DMA registers\n");
		retval = -ENXIO;
		goto probe_err_out;
	}
	lp->tx_dma_regs =
	    ioremap_nocache(DMA0_PhysicalAddress + 1 * DMA_CHAN_OFFSET,
			    sizeof (struct DMA_Chan_s));

	if (!lp->tx_dma_regs) {
		pr_debug("Can't remap Tx DMA registers\n");
		retval = -ENXIO;
		goto probe_err_out;
	}

	lp->td_ring = (DMAD_t) kmalloc(TD_RING_SIZE + RD_RING_SIZE, GFP_KERNEL);
	if (!lp->td_ring) {
		pr_debug("Can't allocate descriptors\n");
		retval = -ENOMEM;
		goto probe_err_out;
	}

	dma_cache_inv((unsigned long) (lp->td_ring),
		      TD_RING_SIZE + RD_RING_SIZE);

	/* now convert TD_RING pointer to KSEG1 */
	lp->td_ring = (DMAD_t) KSEG1ADDR(lp->td_ring);
	lp->rd_ring = &lp->td_ring[KORINA_NUM_TDS];

	spin_lock_init(&lp->lock);

	dev->open = korina_open;
	dev->stop = korina_close;
	dev->hard_start_xmit = korina_send_packet;
	dev->get_stats = korina_get_stats;
	dev->set_multicast_list = &korina_multicast_list;
	dev->tx_timeout = korina_tx_timeout;
	dev->watchdog_timeo = KORINA_TX_TIMEOUT;

	lp->rx_tasklet = kmalloc(sizeof (struct tasklet_struct), GFP_KERNEL);
	tasklet_init(lp->rx_tasklet, korina_rx_tasklet, (unsigned long) dev);
	lp->tx_tasklet = kmalloc(sizeof (struct tasklet_struct), GFP_KERNEL);
	tasklet_init(lp->tx_tasklet, korina_tx_tasklet, (unsigned long) dev);

#ifdef KORINA_PROC_DEBUG
	lp->ps = create_proc_read_entry("net/rc32434", 0, NULL,
					korina_read_proc, dev);
#endif

	/* Fill in the fields of the device structure with ethernet values. */
	ether_setup(dev);
	return 0;

      probe_err_out:
	korina_cleanup_module();
	pr_debug("%s failed.  Returns %d\n", __FUNCTION__, retval);
	return retval;
}

/*
 * Open/initialize the KORINA controller.
 *
 * This routine should set everything up anew at each open, even
 *  registers that "should" only need to be set once at boot, so that
 *  there is non-reboot way to recover if something goes wrong.
 */

static int
korina_open(struct net_device *dev)
{
	struct korina_local *lp = (struct korina_local *) dev->priv;

	MOD_INC_USE_COUNT;

	/*
	 * Initialize
	 */
	if (korina_init(dev)) {
		pr_debug("Erroe: cannot open the Ethernet device\n");
		return -EAGAIN;
	}

	/*
	 * Install the interrupt handler that handles the dma Done and
	 * Finished Events.
	 */
	if (request_irq(lp->rx_irq, &korina_rx_dma_interrupt,
			SA_SHIRQ | SA_INTERRUPT, "korina ethernet Rx", dev)) {
		pr_debug("%s: unable to get Rx DMA IRQ %d\n", __FUNCTION__,
			 lp->rx_irq);
		MOD_DEC_USE_COUNT;
		return -EAGAIN;
	}
	if (request_irq(lp->tx_irq, &korina_tx_dma_interrupt,
			SA_SHIRQ | SA_INTERRUPT, "korina ethernet Tx", dev)) {
		pr_debug("%s: unable to get Tx DMA IRQ %d\n", __FUNCTION__,
			 lp->tx_irq);
		free_irq(lp->rx_irq, dev);
		MOD_DEC_USE_COUNT;
		return -EAGAIN;
	}

	/* Install handler for underflow error. */
	if (request_irq(lp->und_irq, &korina_und_interrupt,
			SA_SHIRQ | SA_INTERRUPT, "Ethernet Underflow", dev)) {
		pr_debug("%s: unable to get UND IRQ %d\n", __FUNCTION__,
			 lp->und_irq);
		free_irq(lp->rx_irq, dev);
		free_irq(lp->tx_irq, dev);
		MOD_DEC_USE_COUNT;
		return -EAGAIN;
	}

	return 0;
}

/*
 * Close the KORINA device
 */
static int
korina_close(struct net_device *dev)
{
	struct korina_local *lp = (struct korina_local *) dev->priv;
	u32 tmp;

	/*
	 * Disable interrupts
	 */
	disable_irq(lp->rx_irq);
	disable_irq(lp->tx_irq);
	disable_irq(lp->und_irq);

	tmp = local_readl(&lp->tx_dma_regs->dmasm);
	tmp = tmp | DMASM_f_m | DMASM_e_m;
	local_writel(tmp, &lp->tx_dma_regs->dmasm);

	tmp = local_readl(&lp->rx_dma_regs->dmasm);
	tmp = tmp | DMASM_d_m | DMASM_h_m | DMASM_e_m;
	local_writel(tmp, &lp->rx_dma_regs->dmasm);

	free_irq(lp->rx_irq, dev);
	free_irq(lp->tx_irq, dev);
	free_irq(lp->und_irq, dev);

	/*Not enabled this feature at this time. */
	/*del_timer(&lp->mii_phy_timer); */

	MOD_DEC_USE_COUNT;
	return 0;
}

/* transmit packet */
static int
korina_send_packet(struct sk_buff *skb, struct net_device *dev)
{
	struct korina_local *lp = (struct korina_local *) dev->priv;
	unsigned long flags;
	u32 length;
	DMAD_t td;
	spin_lock_irqsave(&lp->lock, flags);

	td = &lp->td_ring[lp->tx_chain_tail];

	/* stop queue when full, drop pkts if queue already full */
	if (lp->tx_count >= (KORINA_NUM_TDS - 2)) {

		lp->tx_full = 1;

		if (lp->tx_count == (KORINA_NUM_TDS - 2)) {
			/* this pkt is about to fill the queue */
			pr_debug("Tx Ring now full, queue stopped.\n");
			netif_stop_queue(dev);
		} else {
			/* this pkt cannot be added to the full queue */
			pr_debug("Tx ring full, packet dropped\n");
			lp->stats.tx_dropped++;
			dev_kfree_skb_any(skb);
			spin_unlock_irqrestore(&lp->lock, flags);
			return 1;
		}
	}
	lp->tx_count++;

	if (lp->tx_skb[lp->tx_chain_tail] != NULL)
		dev_kfree_skb_any(lp->tx_skb[lp->tx_chain_tail]);

	lp->tx_skb[lp->tx_chain_tail] = skb;

	length = skb->len;

	/*Setup the transmit descriptor. */
	td->ca = PHYSADDR(skb->data);

	/* Using the NDPTR to handle the DMA Race Condition */
	if (local_readl(&(lp->tx_dma_regs->dmandptr)) == 0) {
		if (lp->tx_chain_status == empty) {
			td->control =
			    DMA_COUNT(length) | DMAD_cof_m | DMAD_iof_m;
			lp->tx_chain_tail =
			    (lp->tx_chain_tail + 1) & KORINA_TDS_MASK;
			local_writel(PHYSADDR(&lp->td_ring[lp->tx_chain_head]),
				     &(lp->tx_dma_regs->dmandptr));
			lp->tx_chain_head = lp->tx_chain_tail;
		} else {
			lp->td_ring[(lp->tx_chain_tail -
				     1) & KORINA_TDS_MASK].control &=
			    ~(DMAD_cof_m);
			lp->td_ring[(lp->tx_chain_tail -
				     1) & KORINA_TDS_MASK].link = PHYSADDR(td);
			td->control =
			    DMA_COUNT(length) | DMAD_cof_m | DMAD_iof_m;
			lp->tx_chain_tail =
			    (lp->tx_chain_tail + 1) & KORINA_TDS_MASK;
			lp->tx_chain_status = empty;
			local_writel(PHYSADDR(&lp->td_ring[lp->tx_chain_head]),
				     &(lp->tx_dma_regs->dmandptr));

			lp->tx_chain_head = lp->tx_chain_tail;
		}
	} else {
		if (lp->tx_chain_status == empty) {
			td->control =
			    DMA_COUNT(length) | DMAD_cof_m | DMAD_iof_m;
			lp->td_ring[(lp->tx_chain_tail -
				     1) & KORINA_TDS_MASK].link = PHYSADDR(td);
			lp->tx_chain_tail =
			    (lp->tx_chain_tail + 1) & KORINA_TDS_MASK;
			lp->tx_chain_status = filled;
		} else {
			lp->td_ring[(lp->tx_chain_tail -
				     1) & KORINA_TDS_MASK].control &=
			    ~(DMAD_cof_m);
			lp->td_ring[(lp->tx_chain_tail -
				     1) & KORINA_TDS_MASK].link = PHYSADDR(td);
			td->control =
			    DMA_COUNT(length) | DMAD_cof_m | DMAD_iof_m;
			lp->tx_chain_tail =
			    (lp->tx_chain_tail + 1) & KORINA_TDS_MASK;
		}
	}

	dev->trans_start = jiffies;

	spin_unlock_irqrestore(&lp->lock, flags);

	return 0;
}

/* Ethernet Tx Underflow interrupt */
static void
korina_und_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct net_device *dev = (struct net_device *) dev_id;
	struct korina_local *lp;
	unsigned int i;

	ASSERT(dev != NULL);

	netif_stop_queue(dev);
	lp = (struct korina_local *) dev->priv;

	pr_debug("Tx underflow - i/f reset\n");
	spin_lock(&lp->lock);

	i = local_readl(&lp->eth_regs->ethintfc);
	i &= ~ETHINTFC_und_m;
	local_writel(i, &lp->eth_regs->ethintfc);

	/* Restart interface */
	korina_restart(dev);

	spin_unlock(&lp->lock);

}

/* Ethernet Rx DMA interrupt */
static void
korina_rx_dma_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct net_device *dev = (struct net_device *) dev_id;
	struct korina_local *lp;
	u32 dmas;

	ASSERT(dev != NULL);

	lp = (struct korina_local *) dev->priv;

	spin_lock(&lp->lock);

	/* Mask D H E bit in Rx DMA */
	local_writel(local_readl(&lp->rx_dma_regs->dmasm) |
		     (DMASM_d_m | DMASM_h_m | DMASM_e_m),
		     &lp->rx_dma_regs->dmasm);
	dmas = local_readl(&lp->rx_dma_regs->dmas);

	if (dmas & (DMAS_d_m | DMAS_h_m | DMAS_e_m)) {
		tasklet_hi_schedule(lp->rx_tasklet);
	}

	spin_unlock(&lp->lock);
}

/*
 * We have a good packet(s), get it/them out of the buffers.
 *
 * hkou: this driver works by creating (once) a circular list of receiver
 *       DMA descriptors that will be used serially by the Korina.
 *       Because the descriptors are never unlinked from the list _they
 *       are always live_.  We are counting on Linux (and the chosen number
 *	 of buffers) to keep ahead of the hardware otherwise the same
 *	 descriptor might be used for more than one reception.
 */

static void
korina_rx_tasklet(unsigned long rx_data_dev)
{
	struct net_device *dev = (struct net_device *) rx_data_dev;
	struct korina_local *lp = (struct korina_local *) dev->priv;
	volatile DMAD_t rd = &lp->rd_ring[lp->rx_next_done];
	struct sk_buff *skb, *skb_new;
	u8 *pkt_buf;
	u32 devcs, count, pkt_len, pktuncrc_len;
	unsigned long flags;
	volatile u32 dmas;

	spin_lock_irqsave(&lp->lock, flags);

	/* keep going while we have received into more descriptors */
	while (IS_DMA_DONE(rd->control)) {

		/* init the var. used for the later operations within the while loop */
		skb_new = NULL;
		devcs = rd->devcs;
		pkt_len = RCVPKT_LENGTH(devcs);
		pktuncrc_len = pkt_len - 4;
		skb = lp->rx_skb[lp->rx_next_done];

		count = KORINA_RBSIZE - (u32) DMA_COUNT(rd->control);

		if (count != pkt_len) {
			/*
			 * Due to a bug in korina processor, the packet length
			 * given by devcs field and count field sometimes differ.
			 * If that is the case, report Error.
			 */
			printk("Packet size mismatch\n");
			lp->stats.rx_errors++;
			lp->stats.rx_dropped++;
		} else if (count < 64) {
			lp->stats.rx_errors++;
			lp->stats.rx_dropped++;
		} else if ((devcs & (ETHRX_ld_m)) != ETHRX_ld_m) {
			/* check that this is a whole packet */
			/* WARNING: DMA_FD bit incorrectly set in Korina (errata ref #077) */
			lp->stats.rx_errors++;
			lp->stats.rx_dropped++;
		} else if (devcs & ETHRX_rok_m) {
			/* must be the (first and) last descriptor then */
			pkt_buf = (u8 *) lp->rx_skb[lp->rx_next_done]->data;

			/* invalidate the cache before copying the buffer */
			dma_cache_inv((unsigned long) pkt_buf, pktuncrc_len);

			/* Malloc up new buffer. */
			skb_new = dev_alloc_skb(KORINA_RBSIZE + 2);

			if (skb_new != NULL) {
				/* Make room */
				skb_put(skb, pktuncrc_len);

				skb->protocol = eth_type_trans(skb, dev);

				/* pass the packet to upper layers */
				netif_rx(skb);

				dev->last_rx = jiffies;
				lp->stats.rx_packets++;
				lp->stats.rx_bytes += pktuncrc_len;

				if (IS_RCV_MP(devcs))
					lp->stats.multicast++;

				/* 16 bit align */
				skb_reserve(skb_new, 2);

				skb_new->dev = dev;
				lp->rx_skb[lp->rx_next_done] = skb_new;

				/* reset descriptor's curr_addr */
				rd->ca = PHYSADDR(skb_new->data);

			} else {
				pr_debug("no memory, dropping rx packet.\n");
				lp->stats.rx_errors++;
				lp->stats.rx_dropped++;
			}

		} else {
			/* This should only happen if we enable accepting broken packets */
			lp->stats.rx_errors++;
			lp->stats.rx_dropped++;

			/* add statistics counters */
			if (IS_RCV_CRC_ERR(devcs)) {
				pr_debug("RX CRC error\n");
				lp->stats.rx_crc_errors++;
			} else if (IS_RCV_LOR_ERR(devcs)) {
				pr_debug("RX LOR error\n");
				lp->stats.rx_length_errors++;
			} else if (IS_RCV_LE_ERR(devcs)) {
				pr_debug("RX LE error\n");
				lp->stats.rx_length_errors++;
			} else if (IS_RCV_OVR_ERR(devcs)) {
				/*
				 * The overflow errors are handled through
				 * an interrupt handler.
				 */
				lp->stats.rx_over_errors++;
			} else if (IS_RCV_CV_ERR(devcs)) {
				/* code violation */
				pr_debug("RX CV error\n");
				lp->stats.rx_frame_errors++;
			} else if (IS_RCV_CES_ERR(devcs)) {
				pr_debug("RX Preamble error\n");
			}
		}

		rd->control =
		    DMA_COUNT(KORINA_RBSIZE) | DMAD_cod_m | DMAD_iod_m;
		lp->rd_ring[(lp->rx_next_done - 1) & KORINA_RDS_MASK].control &=
		    ~(DMAD_cod_m);
		rd->devcs = 0;

		/* restore descriptor's curr_addr */
		if (skb_new == NULL) {
			rd->ca = PHYSADDR(skb->data);
		}

		lp->rx_next_done = (lp->rx_next_done + 1) & KORINA_RDS_MASK;
		rd = &lp->rd_ring[lp->rx_next_done];

		/*
		 * we'll deal with all possible interrupts up to the last
		 * used descriptor - so cancel any interrupts that may have
		 * arrisen while we've been processing.
		 */
	}

	dmas = local_readl(&lp->rx_dma_regs->dmas);
	if (dmas & DMAS_h_m) {
		korina_start_rx(lp, rd);
		lp->dma_halt_cnt++;
	} else if (dmas & DMAS_e_m) {
		pr_debug("%s: DMAS_e_m\n", __FUNCTION__);
		lp->stats.rx_errors++;
	}
	local_writel(~dmas, &lp->rx_dma_regs->dmas);

	/* Enable D E H bit in Rx DMA */
	local_writel(local_readl(&lp->rx_dma_regs->dmasm) &
		     ~(DMASM_d_m | DMASM_h_m | DMASM_e_m),
		     &lp->rx_dma_regs->dmasm);
	spin_unlock_irqrestore(&lp->lock, flags);

}

/* Ethernet Tx DMA interrupt */
static void
korina_tx_dma_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct net_device *dev = (struct net_device *) dev_id;
	struct korina_local *lp;
	u32 dmas;

	ASSERT(dev != NULL);

	lp = (struct korina_local *) dev->priv;

	spin_lock(&lp->lock);

	/* Mask F E bit in Tx DMA */
	local_writel(local_readl(&lp->tx_dma_regs->dmasm) |
		     (DMASM_f_m | DMASM_e_m), &lp->tx_dma_regs->dmasm);
	dmas = local_readl(&lp->tx_dma_regs->dmas);
	if (dmas & (DMAS_f_m | DMAS_e_m)) {
		tasklet_hi_schedule(lp->tx_tasklet);

		if (dmas & DMAS_e_m)
			pr_debug("%s: DMA error\n", __FUNCTION__);

	}
	local_writel(~dmas, &lp->tx_dma_regs->dmas);

	if (lp->tx_chain_status == filled
	    && (local_readl(&(lp->tx_dma_regs->dmandptr)) == 0)) {
		local_writel(PHYSADDR(&lp->td_ring[lp->tx_chain_head]),
			     &(lp->tx_dma_regs->dmandptr));
		lp->tx_chain_status = empty;
		lp->tx_chain_head = lp->tx_chain_tail;
	}
	spin_unlock(&lp->lock);
}

static void
korina_tx_tasklet(unsigned long tx_data_dev)
{
	struct net_device *dev = (struct net_device *) tx_data_dev;
	struct korina_local *lp = (struct korina_local *) dev->priv;
	volatile DMAD_t td = &lp->td_ring[lp->tx_next_done];
	u32 devcs;
	unsigned long flags;

	spin_lock_irqsave(&lp->lock, flags);

	/* process all desc that are done */
	while (IS_DMA_FINISHED(td->control)) {

		if (lp->tx_full == 1) {
			netif_wake_queue(dev);
			lp->tx_full = 0;
		}

		devcs = lp->td_ring[lp->tx_next_done].devcs;
		if ((devcs & (ETHTX_fd_m | ETHTX_ld_m)) !=
		    (ETHTX_fd_m | ETHTX_ld_m)) {
			lp->stats.tx_errors++;
			lp->stats.tx_dropped++;

			/* should never happen */
			pr_debug("%s: split tx ignored\n", __FUNCTION__);
		} else if (IS_TX_TOK(devcs)) {
			/* transmit OK */
			lp->stats.tx_packets++;
		} else {
			lp->stats.tx_errors++;
			lp->stats.tx_dropped++;

			/* underflow */
			if (IS_TX_UND_ERR(devcs))
				lp->stats.tx_fifo_errors++;

			/* oversized frame */
			if (IS_TX_OF_ERR(devcs))
				lp->stats.tx_aborted_errors++;

			/* excessive deferrals */
			if (IS_TX_ED_ERR(devcs))
				lp->stats.tx_carrier_errors++;

			/* collisions: medium busy */
			if (IS_TX_EC_ERR(devcs))
				lp->stats.collisions++;

			/* late collision */
			if (IS_TX_LC_ERR(devcs))
				lp->stats.tx_window_errors++;

		}

		/* We must always free the original skb */
		if (lp->tx_skb[lp->tx_next_done] != NULL) {
			dev_kfree_skb_any(lp->tx_skb[lp->tx_next_done]);
			lp->tx_skb[lp->tx_next_done] = NULL;
		}

		lp->td_ring[lp->tx_next_done].control = DMAD_iof_m;
		lp->td_ring[lp->tx_next_done].devcs = ETHTX_fd_m | ETHTX_ld_m;
		lp->td_ring[lp->tx_next_done].link = 0;
		lp->td_ring[lp->tx_next_done].ca = 0;
		lp->tx_count--;

		/* go on to next transmission */
		lp->tx_next_done = (lp->tx_next_done + 1) & KORINA_TDS_MASK;
		td = &lp->td_ring[lp->tx_next_done];

	}

	/* Enable F E bit in Tx DMA */
	local_writel(local_readl(&lp->tx_dma_regs->dmasm) &
		     ~(DMASM_f_m | DMASM_e_m), &lp->tx_dma_regs->dmasm);
	spin_unlock_irqrestore(&lp->lock, flags);

}

/*
 * Get the current statistics.
 * This may be called with the device open or closed.
 */
static struct net_device_stats *
korina_get_stats(struct net_device *dev)
{
	struct korina_local *lp = (struct korina_local *) dev->priv;
	return &lp->stats;
}

/*
 * Set or clear the multicast filter for this adaptor.
 */
static void
korina_multicast_list(struct net_device *dev)
{
	/* listen to broadcasts always and to treat     */
	/*       IFF bits independantly */
	struct korina_local *lp = (struct korina_local *) dev->priv;
	unsigned long flags;
	u32 recognise = ETHARC_ab_m;	/* always accept broadcasts */

	if (dev->flags & IFF_PROMISC)	/* set promiscuous mode */
		recognise |= ETHARC_pro_m;

	if ((dev->flags & IFF_ALLMULTI) || (dev->mc_count > 15))
		recognise |= ETHARC_am_m;	/* all multicast & bcast */

	else if (dev->mc_count > 0) {
		pr_debug("%s: mc_count %d\n", __FUNCTION__, dev->mc_count);

		recognise |= ETHARC_am_m;	/* for the time being */
	}

	spin_lock_irqsave(&lp->lock, flags);
	local_writel(recognise, &lp->eth_regs->etharc);
	spin_unlock_irqrestore(&lp->lock, flags);
}

static void
korina_tx_timeout(struct net_device *dev)
{
	struct korina_local *lp = (struct korina_local *) dev->priv;
	unsigned long flags;

	spin_lock_irqsave(&lp->lock, flags);
	korina_restart(dev);
	spin_unlock_irqrestore(&lp->lock, flags);

}

/*
 * Initialize the KORINA ethernet controller.
 */
static int
korina_init(struct net_device *dev)
{
	struct korina_local *lp = (struct korina_local *) dev->priv;
	int i, j;

	/* Disable DMA */
	korina_abort_tx(dev);
	korina_abort_rx(dev);

	/* reset ethernet logic */
	local_writel(0, &lp->eth_regs->ethintfc);
	while ((local_readl(&lp->eth_regs->ethintfc) & ETHINTFC_rip_m))
		dev->trans_start = jiffies;

	/* Enable Ethernet Interface */
	local_writel(ETHINTFC_en_m, &lp->eth_regs->ethintfc);

	tasklet_disable(lp->rx_tasklet);
	tasklet_disable(lp->tx_tasklet);

	/* Initialize the transmit Descriptors */
	for (i = 0; i < KORINA_NUM_TDS; i++) {
		lp->td_ring[i].control = DMAD_iof_m;
		lp->td_ring[i].devcs = ETHTX_fd_m | ETHTX_ld_m;
		lp->td_ring[i].ca = 0;
		lp->td_ring[i].link = 0;
		if (lp->tx_skb[i] != NULL) {
			/* free dangling skb */
			dev_kfree_skb_any(lp->tx_skb[i]);
			lp->tx_skb[i] = NULL;
		}
	}
	lp->tx_next_done = lp->tx_chain_head = lp->tx_chain_tail =
	    lp->tx_count = lp->tx_full = 0;

	lp->tx_chain_status = empty;

	/*
	 * Initialize the receive descriptors so that they
	 * become a circular linked list, ie. let the last
	 * descriptor point to the first again.
	 */
	for (i = 0; i < KORINA_NUM_RDS; i++) {
		struct sk_buff *skb = lp->rx_skb[i];

		if (lp->rx_skb[i] == NULL) {
			skb = dev_alloc_skb(KORINA_RBSIZE + 2);
			if (skb == NULL) {
				pr_debug("No memory in the system\n");
				for (j = 0; j < KORINA_NUM_RDS; j++)
					if (lp->rx_skb[j] != NULL)
						dev_kfree_skb_any(lp->
								  rx_skb[j]);

				return 1;
			} else {
				skb->dev = dev;
				skb_reserve(skb, 2);
				/*skb->ip_summed = CHECKSUM_HW; */
				lp->rx_skb[i] = skb;
				lp->rd_ring[i].ca = PHYSADDR(skb->data);

			}
		}
		lp->rd_ring[i].control = DMAD_iod_m | DMA_COUNT(KORINA_RBSIZE);
		lp->rd_ring[i].devcs = 0;
		lp->rd_ring[i].ca = PHYSADDR(skb->data);
		lp->rd_ring[i].link = PHYSADDR(&lp->rd_ring[i + 1]);

	}
	/* loop back */
	lp->rd_ring[KORINA_NUM_RDS - 1].link = PHYSADDR(&lp->rd_ring[0]);
	lp->rx_next_done = 0;

	lp->rd_ring[KORINA_NUM_RDS - 1].control |= DMAD_cod_m;
	lp->rx_chain_head = 0;
	lp->rx_chain_tail = 0;
	lp->rx_chain_status = empty;

	local_writel(0, &lp->rx_dma_regs->dmas);
	/* Start Rx DMA */
	korina_start_rx(lp, &lp->rd_ring[0]);

	/* Enable F E bit in Tx DMA */
	local_writel(local_readl(&lp->tx_dma_regs->dmasm) &
		     ~(DMASM_f_m | DMASM_e_m), &lp->tx_dma_regs->dmasm);
	/* Enable D H E bit in Rx DMA */
	local_writel(local_readl(&lp->rx_dma_regs->dmasm) &
		     ~(DMASM_d_m | DMASM_h_m | DMASM_e_m),
		     &lp->rx_dma_regs->dmasm);

	/* Accept only packets destined for this Ethernet device address */
	local_writel(ETHARC_ab_m, &lp->eth_regs->etharc);

	/* Set all Ether station address registers to their initial values */
	local_writel(STATION_ADDRESS_LOW(dev), &lp->eth_regs->ethsal0);
	local_writel(STATION_ADDRESS_HIGH(dev), &lp->eth_regs->ethsah0);

	local_writel(STATION_ADDRESS_LOW(dev), &lp->eth_regs->ethsal1);
	local_writel(STATION_ADDRESS_HIGH(dev), &lp->eth_regs->ethsah1);

	local_writel(STATION_ADDRESS_LOW(dev), &lp->eth_regs->ethsal2);
	local_writel(STATION_ADDRESS_HIGH(dev), &lp->eth_regs->ethsah2);

	local_writel(STATION_ADDRESS_LOW(dev), &lp->eth_regs->ethsal3);
	local_writel(STATION_ADDRESS_HIGH(dev), &lp->eth_regs->ethsah3);

	/* Frame Length Checking, Pad Enable, CRC Enable, Full Duplex set */
	local_writel(ETHMAC2_pe_m | ETHMAC2_cen_m | ETHMAC2_fd_m,
		     &lp->eth_regs->ethmac2);

	/* Back to back inter-packet-gap */
	local_writel(0x15, &lp->eth_regs->ethipgt);
	/* Non - Back to back inter-packet-gap */
	local_writel(0x12, &lp->eth_regs->ethipgr);

	/* Management Clock Prescaler Divisor */
	/* Clock independent setting */
	local_writel(((idt_cpu_freq) / MII_CLOCK + 1) & ~1,
		     &lp->eth_regs->ethmcp);

	/* don't transmit until fifo contains 48b */
	local_writel(48, &lp->eth_regs->ethfifott);

	local_writel(ETHMAC1_re_m, &lp->eth_regs->ethmac1);

	tasklet_enable(lp->rx_tasklet);
	tasklet_enable(lp->tx_tasklet);

	netif_start_queue(dev);

	return 0;

}

static void
korina_cleanup_module(void)
{
	int i;

	for (i = 0; korina_iflist[i].iobase; i++) {
		struct korina_if_t *bif = &korina_iflist[i];
		if (bif->dev != NULL) {
			struct korina_local *lp =
			    (struct korina_local *) bif->dev->priv;
			if (lp != NULL) {
				if (lp->eth_regs)
					iounmap((void *) lp->eth_regs);
				if (lp->rx_dma_regs)
					iounmap((void *) lp->rx_dma_regs);
				if (lp->tx_dma_regs)
					iounmap((void *) lp->tx_dma_regs);
				if (lp->td_ring)
					kfree((void *) KSEG0ADDR(lp->td_ring));

#ifdef KORINA_PROC_DEBUG
				if (lp->ps)
					remove_proc_entry("net/rc32434", NULL);
#endif
				kfree(lp);
			}

			unregister_netdev(bif->dev);
			kfree(bif->dev);
			release_region(bif->iobase, 0x24C);
		}
	}
}

#ifndef MODULE

static int __init
korina_setup(char *options)
{
	/* no options yet */
	return 1;
}

static int __init
korina_setup_ethaddr(char *options)
{
	memcpy(mac, options, 17);
	mac[17] = '\0';
	return 1;
}

__setup("rc32434eth=", korina_setup);
__setup("ethaddr=", korina_setup_ethaddr);

#endif				/* !MODULE */

module_init(korina_init_module);
module_exit(korina_cleanup_module);
