/*
 * ibm_ocp_rgmii.c
 *
 * Copyright 2003 MontaVista Softare Inc.
 *
 * This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 */

#include <linux/config.h>
#include <linux/netdevice.h>
#include <asm/io.h>
#include <asm/pgtable.h>
#include <asm/ocp.h>
#include "ocp_rgmii.h"
#include "ocp_zmii.h"
#include "ibm_ocp_enet.h"

static unsigned int rgmii_enable[][4] = {
	{0, 0, 0, 0},
	{0, 0, 0, 0},
	{RGMII_RTBI2, RGMII_RGMII2, RGMII_TBI2,
	 ~(ZMII_MDI0 | ZMII_MDI1 | ZMII_MDI3)},
	{RGMII_RTBI3, RGMII_RGMII3, RGMII_TBI3,
	 ~(ZMII_MDI0 | ZMII_MDI1 | ZMII_MDI2)}};

static unsigned int rgmii_speed_mask[] = {0,
					  0,
					  RGMII_MII2_SPDMASK,
					  RGMII_MII3_SPDMASK};

static unsigned int rgmii_speed100[] = {0,
					0,
					RGMII_MII2_100MB,
					RGMII_MII3_100MB};

static unsigned int rgmii_speed1000[] = {0,
					 0,
					 RGMII_MII2_1000MB,
					 RGMII_MII3_1000MB};

void
rgmii_enable_port(struct net_device *dev)
{
	struct ocp_enet_private *fep = dev->priv;
	rgmii_t *rgmiip = fep->rgmii_base;
	unsigned int mask;

	/* set FER reg in RGMII for port */	
	mask = in_be32(&rgmiip->fer);
	mask |= rgmii_enable[fep->emac_num][fep->rgmii_mode];
	out_be32(&rgmiip->fer, mask);
}

void
rgmii_set_port_speed(int speed, struct net_device *dev)
{
	struct ocp_enet_private *fep = dev->priv;
	rgmii_t *rgmiip = fep->rgmii_base;
	unsigned int rgmii_speed;

	rgmii_speed = in_be32(&rgmiip->ssr);

	rgmii_speed &= ~rgmii_speed_mask[fep->emac_num];

	if (speed == 100)
		rgmii_speed |= rgmii_speed100[fep->emac_num];
	else if (speed == 1000)
		rgmii_speed |= rgmii_speed1000[fep->emac_num];

	out_be32(&rgmiip->ssr, rgmii_speed);
}

int
rgmii_init(int mode, struct net_device *dev)
{
	struct ocp_enet_private *fep = dev->priv;
	struct rgmii_regs *rgmiip;
	struct ocp_dev *rgmii_dev;
	char *mode_name[] = { "RGMII", "TBI", "GMII", "RTBI" };
	unsigned int fer;

	rgmii_dev = ocp_get_dev(RGMIIB, 0);
	if (rgmii_dev == NULL) {
		if (!(rgmii_dev = ocp_alloc_dev(0)))
			return -ENOMEM;
		rgmii_dev->type = RGMIIB;
		if (ocp_register(rgmii_dev) == -ENXIO) {
			ocp_free_dev(rgmii_dev);
			return -ENXIO;
		}
	}

	rgmiip = (struct rgmii_regs *)
	    __ioremap(rgmii_dev->paddr, sizeof(*rgmiip), _PAGE_NO_CACHE);
	fep->rgmii_base = rgmiip;
	fep->rgmii_mode = mode;
	
	/* Assume Group 4 */
	if (mode == RGMII_AUTO) {
		switch(fep->emac_num){
		case 2:
			fer = rgmiip->fer & 0x00000007;
			if (fer == RGMII_RGMII2)
				fep->rgmii_mode = RGMII;
			else
			if (fer == RGMII_RTBI2)
				fep->rgmii_mode = RTBI;
			else
			if (fer == RGMII_TBI2)
				fep->rgmii_mode = TBI;
			else
			if (fer == RGMII_GMII2)
				fep->rgmii_mode = GMII;
			else
				printk("Invalid FER for 2 value= %x\n",fer);
			break;
		case 3:
			fer = rgmiip->fer & 0x00000070;
			if (fer == RGMII_RGMII3)
				fep->rgmii_mode = RGMII;
			else
			if (fer == RGMII_RTBI3)
				fep->rgmii_mode = RTBI;
			else
			if (fer == RGMII_TBI3)
				fep->rgmii_mode = TBI;
			else
				printk("Invalid FER for 3 value= %x\n",fer);
			break;
		default:
			printk(" Invalid prot # %d\n", fep->emac_num);
			break;
		}
			
		
		/* Failsafe: ZMII_AUTO is invalid index into the arrays,
		   so force SMII if all else fails. */

	}

	printk(KERN_NOTICE "IBM RGMII: EMAC %d: %s mode\n",
			fep->emac_num,
			mode_name[fep->rgmii_mode]);

	return (fep->rgmii_mode);
}
