
/*
 * linux/drivers/net/cirrus.c
 *
 * Author: Abraham van der Merwe <abraham@2d3d.co.za>
 *
 * A Cirrus Logic CS8900A driver for Linux
 * based on the cs89x0 driver written by Russell Nelson,
 * Donald Becker, and others.
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 */

/*
 * At the moment the driver does not support memory mode operation.
 * It is trivial to implement this, but not worth the effort.
 */

/*
 * TODO:
 *
 *   1. If !ready in send_start(), queue buffer and send it in interrupt handler
 *      when we receive a BufEvent with Rdy4Tx, send it again. dangerous!
 *   2. how do we prevent interrupt handler destroying integrity of get_stats()?
 *   3. Change reset code to check status.
 *   4. Implement set_mac_address and remove fake mac address
 *   5. Link status detection stuff
 *   6. Write utility to write EEPROM, do self testing, etc.
 *   7. Implement DMA routines (I need a board w/ DMA support for that)
 *   8. Power management
 *   9. Add support for multiple ethernet chips
 *  10. Add support for other cs89xx chips (need hardware for that)
 */

#include <linux/config.h>
#include <linux/version.h>
#include <linux/module.h>

#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/ioport.h>
#include <linux/init.h>
#include <linux/delay.h>

#include <asm/irq.h>
#include <asm/hardware.h>
#include <asm/io.h>

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/spinlock.h>

#include "cirrus.h"

/* #define DEBUG */
/* #define FULL_DUPLEX */

#ifdef CONFIG_SA1100_FRODO
#	define CIRRUS_DEFAULT_IO FRODO_ETH_IO + 0x300
#	define CIRRUS_DEFAULT_IRQ FRODO_ETH_IRQ
#elif CONFIG_SA1100_CERF
#	define CIRRUS_DEFAULT_IO CERF_ETH_IO + 0x300
#	define CIRRUS_DEFAULT_IRQ CERF_ETH_IRQ
#elif CONFIG_ARCH_CDB89712
#	define CIRRUS_DEFAULT_IO ETHER_BASE + 0x300
#	define CIRRUS_DEFAULT_IRQ IRQ_EINT3
#elif CONFIG_ARCH_PNX0105 	/* AAG */

#include <linux/compiler.h>

#define CIRRUS_DEFAULT_BASE	0x28200000	/* = Physical address 0x48200000 */
#define CIRRUS_DEFAULT_IRQ	IRQ_EVENTROUTER1/* Event inputs bank 1 - ID 35/bit 3 */
#define EVENT_ROUTER_RANGE	0x1C04		/* = 7 kB */
#define PADMUX1_RANGE		0x32		/* = 7 registers + 1 reserved */
#define RESET_GPIO_PIN		0x1		/* bit 1 = DAI1_WS */
#define IRQ_GPIO_PIN		0x0		/* bit 0 = DAI1_BCK */

static unsigned char eth0mac[6] 	    = {0,0,0,0,0,0} ;
static unsigned long cirrus_gpio_addr = IOCONF_BASE + 0x00000080L ; /* IOCONF_PADMUX1 */
static void         *cirrus_gpio_cookie    = NULL ;
static unsigned long event_router_addr = EROUTER_BASE ;   		 /* 0x13000000 */
static void         *event_router_cookie    = NULL ;

static void waitmsec( unsigned int ms ) ;
static unsigned char get_random_byte( void ) ;

static void initialize_ebi( void ) ;

static int  map_cirrus_gpio( struct net_device *dev ) ;
static void unmap_cirrus_gpio( void ) ;
static void cirrus_gpio_enable_bit( unsigned long bit, unsigned long offset ) ;
static void cirrus_gpio_disable_bit( unsigned long bit, unsigned long offset ) ;

static int  map_event_router( struct net_device *dev ) ;
static void unmap_event_router( void ) ;
static void event_router_enable_pin( unsigned long pin, unsigned long offset ) ;
static void event_router_disable_pin( unsigned long pin, unsigned long offset ) ;
static unsigned int event_router_pin_enabled( unsigned long pin, unsigned long offset ) ;

static void reset_cirrus( void ) ;
static void enable_cirrus_irq( void ) ;
static unsigned int is_cirrus_irq( void ) ;
static void clear_cirrus_irq( void ) ;

#else
#	define CIRRUS_DEFAULT_IO	0
#	define CIRRUS_DEFAULT_IRQ	0
#endif


typedef struct {
	struct net_device_stats stats;
	u16 txlen;
	u16 txafter;	/* Default is After5 (0) */
} cirrus_t;

typedef struct {
	u16 io_base;		/* I/O Base Address			*/
	u16 irq;			/* Interrupt Number			*/
	u16 dma;			/* DMA Channel Numbers		*/
	u32 mem_base;		/* Memory Base Address		*/
	u32 rom_base;		/* Boot PROM Base Address	*/
	u32 rom_mask;		/* Boot PROM Address Mask	*/
	u8 mac[6];			/* Individual Address		*/
} cirrus_eeprom_t;

typedef struct {
	struct sk_buff 		*skb;
	struct net_device	*dev;
} cirrus_tx_data_t;

static cirrus_tx_data_t tx_data;
static void cirrus_tx_tasklet(unsigned long data);
DECLARE_TASKLET(cirrus_tasklet, cirrus_tx_tasklet, 0);

/*
 * I/O routines
 */

static inline u16 cirrus_read (struct net_device *dev,u16 reg)
{
	outw (reg,dev->base_addr + PP_Address);
	return (inw (dev->base_addr + PP_Data));
}

static inline void cirrus_write (struct net_device *dev,u16 reg,u16 value)
{
	outw (reg,dev->base_addr + PP_Address);
	outw (value,dev->base_addr + PP_Data);
}

static inline void cirrus_set (struct net_device *dev,u16 reg,u16 value)
{
	cirrus_write (dev,reg,cirrus_read (dev,reg) | value);
}

static inline void cirrus_clear (struct net_device *dev,u16 reg,u16 value)
{
	cirrus_write (dev,reg,cirrus_read (dev,reg) & ~value);
}

static inline void cirrus_frame_read (struct net_device *dev,struct sk_buff *skb,u16 length)
{
	insw (dev->base_addr,skb_put (skb,length),(length + 1) / 2);
}

static inline void cirrus_frame_write (struct net_device *dev,struct sk_buff *skb)
{
	outsw (dev->base_addr,skb->data,(skb->len + 1) / 2);
}

/*
 * Debugging functions
 */

#ifdef DEBUG
static inline int printable (int c)
{
	return ((c >= 32 && c <= 126) ||
			(c >= 174 && c <= 223) ||
			(c >= 242 && c <= 243) ||
			(c >= 252 && c <= 253));
}

static void dump16 (struct net_device *dev,const u8 *s,size_t len)
{
	int i;
	char str[128];

	if (!len) return;

	*str = '\0';

	for (i = 0; i < len; i++) {
		if (i && !(i % 4)) strcat (str," ");
		sprintf (str,"%s%.2x ",str,s[i]);
	}

	for ( ; i < 16; i++) {
		if (i && !(i % 4)) strcat (str," ");
		strcat (str,"   ");
	}

	strcat (str," ");
	for (i = 0; i < len; i++) sprintf (str,"%s%c",str,printable (s[i]) ? s[i] : '.');

	printk (KERN_DEBUG "%s:     %s\n",dev->name,str);
}

static void hexdump (struct net_device *dev,const void *ptr,size_t size)
{
	const u8 *s = (u8 *) ptr;
	int i;
	for (i = 0; i < size / 16; i++, s += 16) dump16 (dev,s,16);
	dump16 (dev,s,size % 16);
}

static void dump_packet (struct net_device *dev,struct sk_buff *skb,const char *type)
{
	printk (KERN_INFO "%s: %s %d byte frame %.2x:%.2x:%.2x:%.2x:%.2x:%.2x to %.2x:%.2x:%.2x:%.2x:%.2x:%.2x type %.4x\n",
			dev->name,
			type,
			skb->len,
			skb->data[0],skb->data[1],skb->data[2],skb->data[3],skb->data[4],skb->data[5],
			skb->data[6],skb->data[7],skb->data[8],skb->data[9],skb->data[10],skb->data[11],
			(skb->data[12] << 8) | skb->data[13]);
	if (skb->len < 0x100) hexdump (dev,skb->data,skb->len);
}
#endif	/* #ifdef DEBUG */

/*
 * Driver functions
 */

static void cirrus_receive (struct net_device *dev)
{
	cirrus_t *priv = (cirrus_t *) dev->priv;
	struct sk_buff *skb;
	u16 status,length;

	status = cirrus_read (dev,PP_RxStatus);
	length = cirrus_read (dev,PP_RxLength);

	if (!(status & RxOK)) {
		priv->stats.rx_errors++;
		if ((status & (Runt | Extradata))) priv->stats.rx_length_errors++;
		if ((status & CRCerror)) priv->stats.rx_crc_errors++;
		return;
	}

	if ((skb = dev_alloc_skb (length + 4)) == NULL) {
		priv->stats.rx_dropped++;
		return;
	}

	skb->dev = dev;
	skb_reserve (skb,2);

	cirrus_frame_read (dev,skb,length);

#ifdef DEBUG
	dump_packet (dev,skb,"recv");
#endif	/* #ifdef DEBUG */

	skb->protocol = eth_type_trans (skb,dev);

	netif_rx (skb);
	dev->last_rx = jiffies;

	priv->stats.rx_packets++;
	priv->stats.rx_bytes += length;
}

static int cirrus_send_start (struct sk_buff *skb,struct net_device *dev)
{
	cirrus_t *priv = (cirrus_t *) dev->priv;
	u16 status;

	/* Tx start must be done with irq disabled
	 * else status can be wrong */
	disable_irq (dev->irq);

	netif_stop_queue (dev);
	cirrus_write (dev,PP_TxCMD,TxStart (priv->txafter));
	cirrus_write (dev,PP_TxLength,skb->len);

	status = cirrus_read (dev,PP_BusST);

	enable_irq (dev->irq);

	if (unlikely (status & TxBidErr)) {
		printk (KERN_WARNING "%s: Invalid frame size %d!\n",dev->name,skb->len);
		priv->stats.tx_errors++;
		priv->stats.tx_aborted_errors++;
		priv->txlen = 0;
		return (1);
	}

	if (!(status & Rdy4TxNOW)) {
#if 0
		printk (KERN_WARNING "%s: Transmit buffer not free!\n",dev->name);
		priv->stats.tx_errors++;
		priv->txlen = 0;
		/* FIXME: store skb and send it in interrupt handler */
		return (1);
#endif
		/* Queue SKB for transmission when tx buffer becomes 
		 * available.
		 */
		tx_data.dev = dev;
		tx_data.skb = skb;
		return 0;
	}

	cirrus_frame_write (dev,skb);

#ifdef DEBUG
	dump_packet (dev,skb,"send");
#endif	/* #ifdef DEBUG */

	dev->trans_start = jiffies;

	dev_kfree_skb (skb);

	priv->txlen = skb->len;
	return 0;
}

static void cirrus_tx_tasklet (unsigned long data)
{
	struct net_device *dev = tx_data.dev;
	struct sk_buff *skb = tx_data.skb;
	cirrus_t *priv = (cirrus_t *) dev->priv;

	tx_data.dev = NULL;
	tx_data.skb = NULL;
	cirrus_frame_write (dev, skb);
#ifdef DEBUG
	dump_packet (dev,skb,"send");
#endif

	dev->trans_start = jiffies;
	dev_kfree_skb (skb);
	priv->txlen = skb->len;
	netif_wake_queue (dev);
}

static void cirrus_interrupt (int irq,void *id,struct pt_regs *regs)
{
	struct net_device *dev = (struct net_device *) id;
	cirrus_t *priv;
	u16 status;

	if (dev->priv == NULL) {
		printk (KERN_WARNING "%s: irq %d for unknown device.\n",dev->name,irq);
		return;
	}

	priv = (cirrus_t *) dev->priv;

#ifdef CONFIG_ARCH_PNX0105
	if ( 0 == is_cirrus_irq() )
	{
	 printk( KERN_DEBUG "%s: irq %d was not generated by CS9800a (= ID 35/bit 3).\n",
                dev->name, irq );
	 return ;
	}

#endif

	while ((status = cirrus_read (dev,PP_ISQ))) {
		switch (RegNum (status)) {
		case RxEvent:
			cirrus_receive (dev);
			break;

		case TxEvent:
			priv->stats.collisions += ColCount (cirrus_read (dev,PP_TxCOL));
			if (!(RegContent (status) & TxOK)) {
				priv->stats.tx_errors++;
				if ((RegContent (status) & Out_of_window)) priv->stats.tx_window_errors++;
				if ((RegContent (status) & Jabber)) priv->stats.tx_aborted_errors++;
				break;
			} else if (priv->txlen) {
				priv->stats.tx_packets++;
				priv->stats.tx_bytes += priv->txlen;
			}
			priv->txlen = 0;
			if (tx_data.skb == NULL)
			netif_wake_queue (dev);
			break;

		case BufEvent:
			if ((RegContent (status) & RxMiss)) {
				u16 missed = MissCount (cirrus_read (dev,PP_RxMISS));
				priv->stats.rx_errors += missed;
				priv->stats.rx_missed_errors += missed;
			}
			if ((RegContent (status) & Rdy4Tx)) {
				tasklet_schedule(&cirrus_tasklet);
			}
			if ((RegContent (status) & TxUnderrun)) {
				priv->stats.tx_errors++;
				/* Shift start tx, if underruns come too often */
				switch (++priv->stats.tx_fifo_errors >> 2) {
					case 1: priv->txafter = After381; break;
					case 2: priv->txafter = After1021; break;
					default: priv->txafter = AfterAll; break;
				}
			}
			if (RegContent (status) & TxUnderrun) {
			priv->txlen = 0;
				if (tx_data.skb == NULL)
			netif_wake_queue (dev);
                        }
			break;

		case RxMISS:
                       printk ("cirrus_interrupt(): RxMISS\n");
			status = MissCount (cirrus_read (dev,PP_RxMISS));
			priv->stats.rx_errors += status;
			priv->stats.rx_missed_errors += status;
			break;

		case TxCOL:
                       printk ("cirrus_interrupt(): TxCol\n");
			priv->stats.collisions += ColCount (cirrus_read (dev,PP_TxCOL));
			break;
		}
	}

#ifdef CONFIG_ARCH_PNX0105
 	clear_cirrus_irq() ;
#endif
}

static void cirrus_transmit_timeout (struct net_device *dev)
{
	cirrus_t *priv = (cirrus_t *) dev->priv;
	priv->stats.tx_errors++;
	priv->stats.tx_heartbeat_errors++;
	priv->txlen = 0;
	if (tx_data.skb == NULL)
	netif_wake_queue (dev);
}

#ifdef CONFIG_ARCH_PNX0105
static int __init mac_fn( char* new_mac )
{
	char *ptr = new_mac;
	int i, j;

	if (!ptr)
		return -EINVAL;

	for (i = 0; i < 6; i++) {
		unsigned long tmp = simple_strtoul(ptr, &ptr, 16);

		if (tmp > 0xff)
			break;

		eth0mac[i] = tmp;

		if (*ptr != ':' && *ptr != '-')
			break;
		ptr++;
	}

	if (i < 5) {
		for (j = 0; j <= i; j++)
			eth0mac[i] = 0;
		return -EINVAL;
	}

	return 0;

		
}
__setup("MAC=", mac_fn );
#endif

static int cirrus_start (struct net_device *dev)
{
	int result;

	/* valid ethernet address? */
	if (!is_valid_ether_addr(dev->dev_addr)) {
		printk(KERN_ERR "%s: invalid ethernet MAC address\n",dev->name);
		return (-EINVAL);
	}

	/* install interrupt handler */
	if ((result = request_irq (dev->irq,&cirrus_interrupt,0,dev->name,dev)) < 0) {
		printk (KERN_ERR "%s: could not register interrupt %d\n",dev->name,dev->irq);
		return (result);
	}

	/* enable the ethernet controller */
	cirrus_set (dev,PP_RxCFG,RxOKiE | BufferCRC | CRCerroriE | RuntiE | ExtradataiE);
	cirrus_set (dev,PP_RxCTL,RxOKA | IndividualA | BroadcastA);
	cirrus_set (dev,PP_TxCFG,TxOKiE | Out_of_windowiE | JabberiE);
	cirrus_set (dev,PP_BufCFG,Rdy4TxiE | RxMissiE | TxUnderruniE | TxColOvfiE | MissOvfloiE);
	cirrus_set (dev,PP_LineCTL,SerRxON | SerTxON);
	cirrus_set (dev,PP_BusCTL,EnableRQ);

#ifdef FULL_DUPLEX
	cirrus_set (dev,PP_TestCTL,FDX);
#endif	/* #ifdef FULL_DUPLEX */

	/* start the queue */
	netif_start_queue (dev);

	MOD_INC_USE_COUNT;

	return (0);
}

static int cirrus_stop (struct net_device *dev)
{
	/* disable ethernet controller */
	cirrus_write (dev,PP_BusCTL,0);
	cirrus_write (dev,PP_TestCTL,0);
	cirrus_write (dev,PP_SelfCTL,0);
	cirrus_write (dev,PP_LineCTL,0);
	cirrus_write (dev,PP_BufCFG,0);
	cirrus_write (dev,PP_TxCFG,0);
	cirrus_write (dev,PP_RxCTL,0);
	cirrus_write (dev,PP_RxCFG,0);

	/* uninstall interrupt handler */
	free_irq (dev->irq,dev);

	/* stop the queue */
	netif_stop_queue (dev);

	MOD_DEC_USE_COUNT;

	return (0);
}

static int cirrus_set_mac_address (struct net_device *dev, void *p)
{
	struct sockaddr *addr = (struct sockaddr *)p;
	int i;

	if (netif_running(dev))
		return -EBUSY;

	memcpy(dev->dev_addr, addr->sa_data, dev->addr_len);

	/* configure MAC address */
	for (i = 0; i < ETH_ALEN; i += 2)
		cirrus_write (dev,PP_IA + i,dev->dev_addr[i] | (dev->dev_addr[i + 1] << 8));

	return 0;
}

static struct net_device_stats *cirrus_get_stats (struct net_device *dev)
{
	cirrus_t *priv = (cirrus_t *) dev->priv;
	return (&priv->stats);
}

static void cirrus_set_receive_mode (struct net_device *dev)
{
	if ((dev->flags & IFF_PROMISC))
		cirrus_set (dev,PP_RxCTL,PromiscuousA);
	else
		cirrus_clear (dev,PP_RxCTL,PromiscuousA);

	if ((dev->flags & IFF_ALLMULTI) && dev->mc_list)
		cirrus_set (dev,PP_RxCTL,MulticastA);
	else
		cirrus_clear (dev,PP_RxCTL,MulticastA);
}

static int cirrus_eeprom_wait (struct net_device *dev)
{
	int i;

	for (i = 0; i < 200; i++) {
		if (!(cirrus_read (dev,PP_SelfST) & SIBUSY))
			return (0);
		udelay (1);
	}

	return (-1);
}

static int cirrus_eeprom_read (struct net_device *dev,u16 *value,u16 offset)
{
	if (cirrus_eeprom_wait (dev) < 0)
		return (-1);

	cirrus_write (dev,PP_EEPROMCommand,offset | EEReadRegister);

	if (cirrus_eeprom_wait (dev) < 0)
		return (-1);

	*value = cirrus_read (dev,PP_EEPROMData);

	return (0);
}

static int cirrus_eeprom (struct net_device *dev,cirrus_eeprom_t *eeprom)
{
	u16 offset,buf[16],*word;
	u8 checksum = 0,*byte;

	if (cirrus_eeprom_read (dev,buf,0) < 0) {
		read_timed_out:
		printk (KERN_DEBUG "%s: EEPROM read timed out\n",dev->name);
		return (-ETIMEDOUT);
	}

	if ((buf[0] >> 8) != 0xa1) {
		printk (KERN_DEBUG "%s: No EEPROM present\n",dev->name);
		return (-ENODEV);
	}

	if ((buf[0] & 0xff) < sizeof (buf)) {
		eeprom_too_small:
		printk (KERN_DEBUG "%s: EEPROM too small\n",dev->name);
		return (-ENODEV);
	}

	for (offset = 1; offset < (buf[0] & 0xff); offset++) {
		if (cirrus_eeprom_read (dev,buf + offset,offset) < 0)
			goto read_timed_out;

		if (buf[offset] == 0xffff)
			goto eeprom_too_small;
	}

	if (buf[1] != 0x2020) {
		printk (KERN_DEBUG "%s: Group Header #1 mismatch\n",dev->name);
		return (-EIO);
	}

	if (buf[5] != 0x502c) {
		printk (KERN_DEBUG "%s: Group Header #2 mismatch\n",dev->name);
		return (-EIO);
	}

	if (buf[12] != 0x2158) {
		printk (KERN_DEBUG "%s: Group Header #3 mismatch\n",dev->name);
		return (-EIO);
	}

	eeprom->io_base = buf[2];
	eeprom->irq = buf[3];
	eeprom->dma = buf[4];
	eeprom->mem_base = (buf[7] << 16) | buf[6];
	eeprom->rom_base = (buf[9] << 16) | buf[8];
	eeprom->rom_mask = (buf[11] << 16) | buf[10];

	word = (u16 *) eeprom->mac;
	for (offset = 0; offset < 3; offset++) word[offset] = buf[13 + offset];

	byte = (u8 *) buf;
	for (offset = 0; offset < sizeof (buf); offset++) checksum += byte[offset];

	if (cirrus_eeprom_read (dev,&offset,0x10) < 0)
		goto read_timed_out;

	if ((offset >> 8) != (u8) (0x100 - checksum)) {
		printk (KERN_DEBUG "%s: Checksum mismatch (expected 0x%.2x, got 0x%.2x instead\n",
				dev->name,
				(u8) (0x100 - checksum),
				offset >> 8);
		return (-EIO);
	}

	return (0);
}

/*
 * Architecture dependant code
 */

#ifdef CONFIG_SA1100_FRODO
static void frodo_reset (struct net_device *dev)
{
	int i;
	volatile u16 value;

	/* reset ethernet controller */
	FRODO_CPLD_ETHERNET |= FRODO_ETH_RESET;
	mdelay (50);
	FRODO_CPLD_ETHERNET &= ~FRODO_ETH_RESET;
	mdelay (50);

	/* we tied SBHE to CHIPSEL, so each memory access ensure the chip is in 16-bit mode */
	for (i = 0; i < 3; i++) value = cirrus_read (dev,0);

	/* FIXME: poll status bit */
}
#endif	/* #ifdef CONFIG_SA1100_FRODO */

/*
 * Driver initialization routines
 */

#ifndef CONFIG_ARCH_PNX0105
static int io = 0;
static int irq = 0;
#endif

int __init cirrus_probe (struct net_device *dev)
{
	static cirrus_t priv;
	int i,result;
	u16 value;
	cirrus_eeprom_t eeprom;

#ifdef CONFIG_ARCH_PNX0105
	printk ("Cirrus Logic CS8900A driver for Linux (V0.02.pnx0105)\n");
#else
	printk ("Cirrus Logic CS8900A driver for Linux (V0.02)\n");
#endif

	memset (&priv,0,sizeof (cirrus_t));

	ether_setup (dev);

	dev->open               = cirrus_start;
	dev->stop               = cirrus_stop;
	dev->hard_start_xmit    = cirrus_send_start;
	dev->get_stats          = cirrus_get_stats;
	dev->set_multicast_list = cirrus_set_receive_mode;
	dev->set_mac_address	= cirrus_set_mac_address;
	dev->tx_timeout         = cirrus_transmit_timeout;
	dev->watchdog_timeo     = HZ;

	dev->dev_addr[0] = 0x00;
	dev->dev_addr[1] = 0x00;
	dev->dev_addr[2] = 0x00;
	dev->dev_addr[3] = 0x00;
	dev->dev_addr[4] = 0x00;
	dev->dev_addr[5] = 0x00;

	dev->if_port   = IF_PORT_10BASET;
	dev->priv      = (void *) &priv;

	SET_MODULE_OWNER (dev);

#ifdef CONFIG_ARCH_PNX0105

	dev->irq = CIRRUS_DEFAULT_IRQ;
	
	if ( !request_region (CIRRUS_DEFAULT_BASE, 16, dev->name) )
        {
		printk ("CS8900 registers physical memory (0x%08x) already claimed.\n", CIRRUS_DEFAULT_BASE);
		return -ENODEV;
	}
	
	dev->base_addr = (unsigned long) ioremap_nocache (CIRRUS_DEFAULT_BASE, 16);
	if (dev->base_addr == 0) {
		printk ("Failed to ioremap CS8900 registers at phys 0x%08x.\n", CIRRUS_DEFAULT_BASE);
		return -ENODEV;
	}
	
	if (((dev->dev_addr[0] = eth0mac[0]) |
	     (dev->dev_addr[1] = eth0mac[1]) |
	     (dev->dev_addr[2] = eth0mac[2]) |
	     (dev->dev_addr[3] = eth0mac[3]) |
	     (dev->dev_addr[4] = eth0mac[4]) |
	     (dev->dev_addr[5] = eth0mac[5])) == 0)
	{
                printk ("CS8900A: No MAC-address defined ... fallback to default MAC address.\n");

		dev->dev_addr[0] = 0x00;
		dev->dev_addr[1] = 0x99;
		dev->dev_addr[2] = 0x88;
		dev->dev_addr[3] = 0x77;
		dev->dev_addr[4] = 0x66;
		dev->dev_addr[5] = 0x55;
	}
		
#ifdef CONFIG_ARCH_PNX0105
	
	/* Print our MAC-address */
	printk ("CS8900A: (%s) The MAC address is ", dev->name ) ;
	for ( i=0 ; i<6 ; i++ )
	 printk( "%02x-", dev->dev_addr[i] ) ;
	printk( "\b.\n" ) ;
	
	/* Initialise the EBI */
	initialize_ebi() ;
	
	/* Map GPIO registers for the pins connected to the CS8900a. */
	if ( map_cirrus_gpio(dev) < 0 )
	 return (-ENODEV) ;
	
	/* Reset the chip. */
	reset_cirrus() ;
	
	/* Map event-router registers. */
	if ( map_event_router(dev) < 0 )
	 return (-ENODEV) ;
	
	/* Initialise our pin on the event-router. */
	enable_cirrus_irq() ;
	
	/* For now we did all we had to do with the GPIO registers. */
	unmap_cirrus_gpio() ;
	
#endif

	/* 
	   We want the chip in 16-bit mode. Therefore SBHE (actually nSBHE) needs to
	   be toggled: It is active-low. Because we tied SBHE to CHIPSEL, and CHIPSEL
	   is active-low too, giving a few dummy reads ensures the chip is in 16-bit
	   mode. See also chapter 2.0, page 13, of the CS9800a datasheet.
        */
	for ( i=0 ; i<3 ; i++ ) cirrus_read( dev, 0 ) ;

#else

	dev->base_addr = CIRRUS_DEFAULT_IO;
	dev->irq = CIRRUS_DEFAULT_IRQ;

	/* module parameters override everything */
	if (io > 0) dev->base_addr = io;
	if (irq > 0) dev->irq = irq;

	if (!dev->base_addr) {
		printk (KERN_ERR
				"%s: No default I/O base address defined. Use io=... or\n"
				"%s: define CIRRUS_DEFAULT_IO for your platform\n",
				dev->name,dev->name);
		return (-EINVAL);
	}

	if (!dev->irq) {
		printk (KERN_ERR
				"%s: No default IRQ number defined. Use irq=... or\n"
				"%s: define CIRRUS_DEFAULT_IRQ for your platform\n",
				dev->name,dev->name);
		return (-EINVAL);
	}

	if ((result = check_region (dev->base_addr,16))) {
		printk (KERN_ERR "%s: can't get I/O port address 0x%lx\n",dev->name,dev->base_addr);
		return (result);
	}

	if (!request_region (dev->base_addr,16,dev->name))
		return -EBUSY;

#ifdef CONFIG_SA1100_FRODO
	frodo_reset (dev);
#endif


	/* if an EEPROM is present, use it's MAC address */
	if (!cirrus_eeprom (dev,&eeprom))
		for (i = 0; i < 6; i++)
			dev->dev_addr[i] = eeprom.mac[i];

#endif	/* CONFIG_ARCH_PNX0105 */

#ifdef CONFIG_ARCH_PNX0105
	
	value = cirrus_read( dev, PP_SelfST ) ;
	if ( 0 == (value & 0x80) )
	{
	 printk( KERN_ERR "Warning: CS9800A status-register indicates initialisation didn't complete yet.\n" ) ;
	}
	
#endif

	/* verify EISA registration number for Cirrus Logic */
	if ((value = cirrus_read (dev,PP_ProductID)) != EISA_REG_CODE) {
		printk (KERN_ERR "%s: incorrect signature 0x%.4x\n",dev->name,value);
		return (-ENXIO);
	}

	/* verify chip version */
	value = cirrus_read (dev,PP_ProductID + 2);
	if (VERSION (value) != CS8900A) {
		printk (KERN_ERR "%s: unknown chip version 0x%.8x\n",dev->name,VERSION (value));
		return (-ENXIO);
	}
	printk (KERN_INFO "%s: CS8900A rev %c detected\n",dev->name,'B' + REVISION (value) - REV_B);

	/* setup interrupt number */
	cirrus_write (dev,PP_IntNum,0);

	/* configure MAC address */
	for (i = 0; i < ETH_ALEN; i += 2)
		cirrus_write (dev,PP_IA + i,dev->dev_addr[i] | (dev->dev_addr[i + 1] << 8));

	return (0);
}


#ifdef CONFIG_ARCH_PNX0105

static void waitmsec( unsigned int ms )
{
	unsigned int i = 0 ;
 
	for ( i=0 ; i<ms ; i++ )
		udelay( 1000 ) ;
}


static unsigned char get_random_byte( void )
{
	unsigned char *ptr ;
	struct timeval tv ;
 
	do_gettimeofday( &tv ) ;
 
	ptr = (unsigned char *)( &(tv.tv_usec) );
 
	return (*ptr) ;
}


static void initialize_ebi( void )
{
	/* Is done in the PNX0105 bootloader ... */
}


static int map_event_router( struct net_device *dev )
{
	/* Prepare our memory-map */
	if ( !request_mem_region( event_router_addr, EVENT_ROUTER_RANGE, "EventRouter") ) {
		printk(KERN_ERR "%s: Unable to reserve the event-router memory region.\n", dev->name);
		return -1;
	}
 
	/* Map the physical address to a virtual address packed in a cookie */
	event_router_cookie = ioremap_nocache(event_router_addr, EVENT_ROUTER_RANGE);
	if (!event_router_cookie) {
		printk( KERN_ERR "%s: Unable to map the event-router memory region.\n", dev->name ) ;
		return -1 ;
	}

	return 0;
}


static int map_cirrus_gpio( struct net_device *dev )
{
	/* Prepare our memory-map */
	if (!request_mem_region(cirrus_gpio_addr, PADMUX1_RANGE, "PADMUX1")) {
		printk( KERN_ERR "%s: Unable to reserve the CS8900a GPIO memory region.\n", dev->name);
	return -1 ;
	}
 
	/* Map the physical address to a virtual address packed in a cookie */
	cirrus_gpio_cookie = ioremap_nocache(cirrus_gpio_addr, PADMUX1_RANGE);
	if (!cirrus_gpio_cookie) {
		printk( KERN_ERR "%s: Unable to map the CS8900a GPIO memory region.\n", dev->name ) ;
		return -1 ;
	}

	return 0 ;
}


static void unmap_event_router( void )
{
	/* Cleanup the mapping */
	iounmap(event_router_cookie);
	release_mem_region(event_router_addr, EVENT_ROUTER_RANGE);
	event_router_cookie = NULL ;
}


static void unmap_cirrus_gpio( void )
{
	/* Cleanup the mapping */
	iounmap(cirrus_gpio_cookie) ;
	release_mem_region(cirrus_gpio_addr, PADMUX1_RANGE);
	cirrus_gpio_cookie = NULL ;
}


static void event_router_enable_pin( unsigned long pin, unsigned long offset )
{
	unsigned long ulValue = 0 ;
	unsigned long ulPin = (1 << pin) ;
 
	ulValue = readl(event_router_cookie + offset); 	/* read */
	ulValue = (ulValue | ulPin ) ;   		/* activate pin */
	writel(ulValue, event_router_cookie + offset); 	/* write */
}


static void event_router_disable_pin( unsigned long pin, unsigned long offset )
{
	unsigned long ulValue = 0 ;
	unsigned long ulPin = (1 << pin) ;
 
	ulPin = ( ~ulPin ) ;

	ulValue = readl(event_router_cookie + offset); 	/* read */
	ulValue = (ulValue & ulPin ) ;   		/* de-activate pin */
	writel(ulValue, event_router_cookie + offset); 	/* write */
}


static unsigned int event_router_pin_enabled( unsigned long pin, unsigned long offset )
{
	unsigned long ulValue = 0;
	unsigned long ulPin = (1 << pin);
 
	ulValue = readl(event_router_cookie + offset); 	/* read */
	ulValue = (ulValue & ulPin) ;   		/* check pin value */
 
	return ulValue ;
}


static void cirrus_gpio_enable_bit( unsigned long bit, unsigned long offset )
{
	unsigned long ulValue = 0 ;
	unsigned long ulPin = (1 << bit) ;
 
	ulValue = readl(cirrus_gpio_cookie + offset);  	/* read */
	ulValue = (ulValue | ulPin ) ;   		/* activate pin */
	writel(ulValue, cirrus_gpio_cookie + offset);  	/* write */
}


static void cirrus_gpio_disable_bit( unsigned long bit, unsigned long offset )
{
	unsigned long ulValue = 0 ;
	unsigned long ulPin   = (1 << bit) ;
 
	ulPin = ( ~ulPin ) ;
 
	ulValue = readl(cirrus_gpio_cookie + offset);  	/* read */
	ulValue = (ulValue & ulPin); 	  		/* de-activate pin */
	writel(ulValue, cirrus_gpio_cookie + offset); 	/* write */
}


/* Routine to reset the chip.
   The RESET line is connected to the DAI1_WS pin.
   So we need to do the following:
   - set it to GPIO mode
   - set it to GPIO-output mode
   - set the pin to active-high for at least 400ns
 */
static void reset_cirrus( void )
{
	/* Set it the GPIO output mode to low:
	   { mode0, mode1 } = {0,1} = low
	*/
	cirrus_gpio_disable_bit(RESET_GPIO_PIN, 0x10);	/* Modify mode0 = 0 */
	cirrus_gpio_enable_bit(RESET_GPIO_PIN, 0x20);	/* Modify mode1 = 1 */

	/* Wait a milisecond. (Just to be sure.) */
	waitmsec( 1 ) ;
 
	/*
	  Now reset the chip by setting the output to high.
	  { mode0, mode1 } = {1,1} = high
	 */
	cirrus_gpio_enable_bit(RESET_GPIO_PIN, 0x10);	/* Modify mode0 = 1 */
 
	/* Wait a milisecond. */
	waitmsec( 1 ) ;
 
	/* Now lower the output again:
	   { mode0, mode1 } = {0,1} = low
	 */
	cirrus_gpio_disable_bit(RESET_GPIO_PIN, 0x10);	/* Modify mode0 = 0 */
 
	/* Wait to allow the reset operation to complete: This takes typically
	   10 ms according to the datasheet. Since I don't care about 
           'typically', but instead want to wait for the maximum period it can
           take, I double it to be sure.
	 */
	waitmsec( 20 ) ;
}


/* Routine to enbable the IRQ-line of the chip.
   The IRQ line is connected to the DAI1_BCK pin.
   So we need to do the following:
   - set it to GPIO mode
   - set it to GPIO input-mode
   - set active-high polarity and non-latch event properties
   - enable the event in the event-router.
   - route the event output to interrupt 1.
 */
static void enable_cirrus_irq( void )
{
	/* 1) Set the IRQ-line to GPIO-mode.
	   2) Set the GPIO-pin to input-mode.
    
	   This is easy, since on PNX0105 the input-mode is always active
	   and for this pin doesn't require changing any multiplex settings.
	   (We just cannot use the I2S-in1 connector together with the CS8900a.)
	 */
 
	/* 3) Set the properties of the event. */
	event_router_enable_pin(3, 0x0CC4);   /* Set apr[1] pin 4 to active-high */
	event_router_disable_pin(3, 0x0CE4);  /* Set atr[1] pin 4 to non-latch */
 
	/* 4) Enable the event in the event-router. */
	event_router_disable_pin(3, 0x0C24);  /* Set int_clr[1] pin 4 to cleared */
	event_router_enable_pin(3, 0x0C64);   /* Set mask[1] pin 4 to mask on */
 
	/* 5) Route the event output to interrupt 1. */
	event_router_enable_pin(3, 0x1424 );  /* Set intoutMask[1][1] to route on */
	event_router_disable_pin(3, 0x1404 ); /* Set intoutMask[0][1] to route off */
	event_router_disable_pin(3, 0x1434 ); /* Set intoutMask[2][1] to route off */
	event_router_disable_pin(3, 0x1444 ); /* Set intoutMask[3][1] to route off */
	event_router_disable_pin(3, 0x1454 ); /* Set intoutMask[4][1] to route off */
}


/* Routine to check if the IRQ is from the CS8900a chip.
   The IRQ-line is connected to an event-router. Multiple
   events are connected to one IRQ.
   So we need to check if the IRQ is caused by our event.
 */
static unsigned int is_cirrus_irq( void )
{
	/* Our IRQ is connected to ID 35/bit 3 of the 'Event inputs bank 1'.
	   See page 32 of the Programming Guide.
	 */
 
	/* Question: Use raw, pending or intout-pending event-status? */
	/* Answer: Test-code uses intout-pending, so let's use that too ... */
	return event_router_pin_enabled(3, 0x1024);	/* intout-pending */
}


/* Because our IRQ-line is connected to an event-router, which in case
   of an event (=interrupt) on our IRQ-line generates an interrupt on
   the PNX0105 interrupt controller, clearing the interrupt in the
   controller, doesn't clear the event on the router.
   So we need to clear the event in the event-router.
 */
static void clear_cirrus_irq( void )
{
	/* Info: If we use non-latched events, this routine can be empty.
	   In that case, reading in the interrupt-handler the ISQ
	   register of the CS8900a, will lower the IRQ-line and
	   therefore automaticaly also de-asserts the event.
	 */
}

#endif /* CONFIG_ARCH_PNX0105 */



EXPORT_NO_SYMBOLS;

static struct net_device dev;

static int __init cirrus_init (void)
{
	memset (&dev,0,sizeof (struct net_device));
	dev.init = cirrus_probe;
	return (register_netdev (&dev));
}

static void __exit cirrus_cleanup (void)
{
#ifdef CONFIG_ARCH_PNX0105
	unmap_event_router() ;
#endif
	release_region (dev.base_addr, 16);
	unregister_netdev (&dev);
}

MODULE_AUTHOR ("Abraham van der Merwe <abraham@2d3d.co.za>");
MODULE_DESCRIPTION ("Cirrus Logic CS8900A driver for Linux (V0.02)");
MODULE_LICENSE ("GPL");
MODULE_PARM_DESC (io,"I/O Base Address");
MODULE_PARM (io,"i");
MODULE_PARM_DESC (irq,"IRQ Number");
MODULE_PARM (irq,"i");

module_init (cirrus_init);
module_exit (cirrus_cleanup);

