/*
 *  drivers/mtd/stw_nand.c
 *
 *  Copyright (C) 2000 Steven J. Hill (sjhill@cotw.com)
 *  Copyright (C) 2002-2003  ATI Technologies Inc.
 *  Copyright (C) 2003, Metro Link, Inc., All rights reserved
 *
 * $Id: stw_nand.c,v 1.4.2.2 2004/11/15 12:47:45 msyrchin Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, 
 * MA  02111-1307, USA.
 */

/*
 * This driver is dramatically changed to fit ATI's implementation of
 * the NAND flash controller which is based on a "pflash" state machine
 * instead of the GPIO pin direct-access mode.
 *
 * This is a stand-alone nand driver for ATI specific NAND controller 
 * hardware architecture.
 *
 * Chin Zhou (czhou@ati.com)
 *
 */

 /*
  *  Descriptions:
  *
  *  An initialization module for the MTD NAND flash device which is used 
  *  on the ATI STW systems.  This driver is based on ATI's specific
  *  NAND/PFLASH controller state machine, and provide basic interface 
  *  between the device specific layer and standard MTD-NAND layer. 
  *
  */

/*
 * Support for NAND flash modules on ATI Xilleon Set Top Wonder
 *
 */

#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/interrupt.h>

#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/partitions.h>

#ifdef CONFIG_MTD_NAND_ECC
#include <linux/mtd/nand_ecc.h>
#endif

#include <asm/io.h>
#include <asm/ati/xilleon.h>
#include <asm/ati/stw_flash.h>


/* Strategies for NAND invalid block handling */

/*
  Block Erase:  
    - Never erase invalid blocks.
    - Update invalid block table (IBT) if a new invalid block is detected.

  Read/Write:
    - Update IBT if any new invalid blocks are found.
    - Logical-physical block mapping for invalid block replacement.
    - ECC correction for 1-bit error.
*/

/*
   Logical-physical block mapping table scheme:
   - Create logical block table which links to valid good blocks,
     skip invalid blocks;
   - Make sure to have contineous logical partitions;
   - adjust the offset of logical blocks to the offset of the valid physical blocks.

   Invalid Block Tables (IBT) and special purpose blocks in NAND
   - First two good blocks are used for invalid block tables (IBT)
   - Third good block is used for MMON Linux boot lines for dual-boot
     Linux kernel images.
   - Each special purpose block has its own magic number.
*/

/* Error in WR/RD operations according to Samsung specifications */
/*
   - Erase failure:
       Status read after Erase, do block replacement, update IBT.
   - Program failure:
       Status read after Program, do block replacement, using L-P table.
   - Verify after Program (read back):
       ECC correction or block replacement.
   - Read error:
       ECC correction for single bit error.
*/


//#define NAND_FLASH_FS_TEST

extern void dump_mtd_info(struct mtd_info *mtd);
extern void dump_nand_chip_info(struct nand_chip *this);
extern int pflashReadInvalidBlockTable(void);
extern void stw_flashinfo (int *flashsize, int *default_num);

/* STW NAND flash related */

#ifdef CONFIG_STW5X226_NAN
#define NUM_PARTITIONS           3
#else
#define NUM_PARTITIONS           5
#endif

#define NAND_ROOT_FS_OFFSET      4*1024*1024
#define NAND_ROOT_FS_SIZE        8*1024*1024

#define NAND_STW_OFFSET          0
#define NAND_STW_SIZE            2*16*1024*1024


/* MTD structure for STW */
static struct mtd_info *stw_nand_mtd = NULL;

/*
 * Define partitions for flash device, need to be block-aligned
 */
const static struct mtd_partition partition_info[] = {
#if defined (NAND_FLASH_FS_TEST)
        { name: "MTD NAND partition 0",
	  offset: 2*16*1024,
	  size: (2*1024*1024-2*16*1024)},
	{ name: "MTD NAND partition 1 - root FS1",
	  offset: 2*1024*1024,
	  size: 4*1024*1024},
	{ name: "MTD NAND partition 2 - root FS2",
	  offset: 6*1024*1024,
	  size: 4*1024*1024 },
	{ name: "MTD NAND partition 3",
	  offset: 29*1024*1024,
	  size: (2*1024*1024 - (16*1024))},
	{ name: "MTD NAND partition 4",
	  offset: (31*1024*1024 - (16*1024)),
	  size: 16*1024 }
#else
#ifdef CONFIG_STW5X226_NAN
        { name: "MTD NAND partition 0",
          offset: (16*1024*1024+2*16*1024),
          size: (2*1024*1024-2*16*1024)},
        { name: "MTD NAND partition 1",
	  offset: (16*1024*1024+2*1024*1024),
	  size: 2*1024*1024},
	{ name: "MTD NAND partition 2 - root FS",
	  offset: (16*1024*1024+NAND_ROOT_FS_OFFSET),
	  size: NAND_ROOT_FS_SIZE }
#else
	{ name: "MTD NAND partition 0",
	  offset: 2*16*1024,
	  size: (2*1024*1024-2*16*1024)},
	{ name: "MTD NAND partition 1",
	  offset: 2*1024*1024,
	  size: 2*1024*1024},
	{ name: "MTD NAND partition 2 - root FS",
	  offset: NAND_ROOT_FS_OFFSET,
	  size: NAND_ROOT_FS_SIZE },
	{ name: "MTD NAND partition 3",
	  offset: 29*1024*1024,
	  size: (2*1024*1024 - (16*1024))},
	{ name: "MTD NAND partition 4",
	  offset: (31*1024*1024 - (16*1024)),
	  size: 16*1024 }
#endif
#endif
};

#ifdef CONFIG_NAND_BAD_BLOCK_CHECK
/* start block 0 */
uint32_t nand_logic_block[2*STW_MAX_NUM_BLOCKS];
uint8_t  nand_phy_block[2*STW_MAX_NUM_BLOCKS];
#endif

#ifdef CONFIG_NAND_BAD_BLOCK_CHECK
extern uint32_t nand_logic_block[2*STW_MAX_NUM_BLOCKS];
extern uint8_t  nand_phy_block[2*STW_MAX_NUM_BLOCKS];
#endif


uint32_t  adj_offset, lb_num;
uint32_t  poffset;

/* START OPTIMIZED PAGE BUFFER WITH DWORD NAND READ ACCESS */

unsigned char fast_pflash_pagebuf[512];
unsigned int  fast_pflash_pagenum = 0xFFFFFFFF;

#define FAST_PFLASH_WRITE_8( x, y ) { \
  fast_pflash_pagenum = 0xFFFFFFFF; /* invalidate cache */ \
  PFLASH_WRITE_8( (x), (y) ); \
}

unsigned char FAST_PFLASH_READ_8( unsigned int offset )
{
  unsigned int pagenum = offset >> 9;

  if( fast_pflash_pagenum != pagenum ) {
    /* request is outside of current cached page */
    unsigned int idx;
    unsigned int pmem = (offset & 0xFFFFFE00);

    for( idx = 0; idx < 512; idx += 4, pmem += 4 ) {
      *((unsigned int*)(fast_pflash_pagebuf + idx)) =
	                            PFLASH_READ_32( pmem );
    }
    fast_pflash_pagenum = pagenum;
  }

  /* fulfill request from current cached page */
  return fast_pflash_pagebuf[offset & 0x1FF];
}

/* END OPTIMIZED PAGE BUFFER WITH DWORD NAND READ ACCESS */


/*
 * NAND low-level MTD interface functions
 */

extern int pflashClearStateMachine(void);
extern int pflash_oob_enable(uint32_t timeout);

static int nand_read (struct mtd_info *mtd, 
		      loff_t from, 
		      size_t len,
		      size_t *retlen,
		      u_char *buf);

static int nand_read_ecc (struct mtd_info *mtd,
			  loff_t from,
			  size_t len,
			  size_t *retlen,
			  u_char *buf,
			  u_char *oob_buf,
			  struct nand_oobinfo *oobsel);

static int nand_read_oob (struct mtd_info *mtd,
			  loff_t from,
			  size_t len,
			  size_t *retlen,
			  u_char *buf);

static int nand_write (struct mtd_info *mtd,
		       loff_t to, size_t len,
		       size_t *retlen, const u_char *buf);

static int nand_write_ecc (struct mtd_info *mtd,
			   loff_t to,
			   size_t len,
			   size_t *retlen,
			   const u_char *buf,
			  u_char *oob_buf,
			  struct nand_oobinfo *oobsel);

static int nand_write_oob (struct mtd_info *mtd,
			   loff_t to,
			   size_t len,
			   size_t *retlen,
			   const u_char *buf);

static int nand_erase (struct mtd_info *mtd,
		       struct erase_info *instr);

static void nand_sync (struct mtd_info *mtd);



#ifdef CONFIG_NAND_BAD_BLOCK_CHECK

static uint32_t stw_nand_block_map(uint32_t *offset);

static int getAdjOffset(int x)
{
    poffset = x;
    return stw_nand_block_map(&poffset);
}
#else
#define getAdjOffset(x) (x)
#endif

/*******************************************************************/

/* 
   Supported command for the STW NAND device are:
       NAND_CMD_READ0
       NAND_CMD_READ1
       NAND_CMD_READOOB    0x50
       NAND_CMD_SEQIN      0x80
       NAND_CMD_PAGEPROG   0x10

       NAND_CMD_ERASE1     0x60
       NAND_CMD_STATUS     0x70
       NAND_CMD_READID     0x90
       NAND_CMD_RESET      0xff


   Comments on the ATI's pflash state-machine implementation:
   Need to clear read_cycle and write_cycle bits for the
   read/write transition; transitions between the main and
   OOB areas, to set/reset SPARE_EN of the PFLASH_CNTL.

   Register accesses always need 32-bit Dword, while PCU NAND 
   flash access can be in Byte/word/dword.

   READID:
   STATUS:
    - set PFLASH_BE_ADDR  because of 2 device CS;
    - writeREG32 (command, PFLASH_CNTL);

    - readREG32 (PFLASH_ID_STAT_DATA);   (for STATUS and READID);

   RESET:
   ERASE1:

    - set PFLASH_BE_ADDR  because of 2 device CS;
    - writeREG32 (command, PFLASH_CNTL);

   READ0/READ1:
    - readPCU8/16/32(PCU_NAND_MAIN_ADDR);

   READOOB:
    - writeREG32 ( cmmand, PFLASH_CNTL )  (set the SPARE_EN);
    - readPCU8/16/32 (PCU_NAND_OOD_ADDR);

   WRITEOOB:
    - writeREG32 ( cmmand, PFLASH_CNTL )  (set the SPARE_EN);
    - writePCU8/16/32 (PCU_NAND_OOD_ADDR, byte/word/dword0;

   PAGEPROG:
    - writePCU8/16/32 (PCU_NAND_MAIN_ADDR, byte/word/dword);

    - ATI hardware implementation is based on automatic page-program:
      NAND controller issue 80h command first, then issue 10h command
      either when clear_state_machine or address reach the end of the page.
*/

/**************************************************************************/

/*
 * Send command to NAND device, ATI's hardware implementation.
 */
static void nand_command (struct mtd_info *mtd,
			  unsigned command,
			  int column,
			  int page_addr,
			  CHIP_SEL chipsel,
			  uint32_t timeout)
{

  uint32_t blk_address;
  uint32_t regVal;
  uint32_t status;

  switch (command) {

  case NAND_CMD_RESET:
	  pflashClearStateMachine();
	  blk_address = pflashGetChipStartOffset(chipsel);
	  regVal = blk_address >> PFLASH_BE_ADDR__BLOCK_ERASE_ADDR__SHIFT;
	  SETFLD_REGMM32(PFLASH_BE_ADDR, BLOCK_ERASE_ADDR, regVal); 
	  SETFLD_REGMM32(PFLASH_CNTL, CMD_TRIG, PFLASH_CMD_RESET); 
	  status = pflashWaitCMDTriggerDone();
	  break;

  case NAND_CMD_READID:
	  pflashClearStateMachine();
	  blk_address = pflashGetChipStartOffset(chipsel);
	  regVal = blk_address >> PFLASH_BE_ADDR__BLOCK_ERASE_ADDR__SHIFT;
	  SETFLD_REGMM32(PFLASH_BE_ADDR, BLOCK_ERASE_ADDR, regVal); 
	  SETFLD_REGMM32(PFLASH_CNTL, CMD_TRIG, PFLASH_CMD_ID_READ); 
	  status = pflashWaitCMDTriggerDone();
	  if(status == PFLASH_OK)
	    {
	      while(GETFLD_REGMM32(PFLASH_STATUS, ID_READ_DONE) == 0)
		{
		  udelay(1000);
		  if(--timeout == 0)
		    {
		      printk("nand_command: timeout waiting for ID_READ.\n");
		      return;
		    }
		}
	
	    }
	  break;

  case NAND_CMD_STATUS:
	  /*
	  pflashClearStateMachine();
	  blk_address = pflash_get_chip_start_address(chipsel);
	  regVal = blk_address >> PFLASH_BE_ADDR__BLOCK_ERASE_ADDR__SHIFT;
	  SETFLD_REGMM32(PFLASH_BE_ADDR, BLOCK_ERASE_ADDR, regVal); 
	  SETFLD_REGMM32(PFLASH_CNTL, CMD_TRIG, PFLASH_CMD_READ_STATUS); 
	  status = pflashWaitCMDTriggerDone();
	  */
	  break;

  case NAND_CMD_ERASE1:
	  /*
	  pflashClearStateMachine();
	  blk_address = pflash_get_chip_start_address(chipsel);
	  regVal = blk_address >> PFLASH_BE_ADDR__BLOCK_ERASE_ADDR__SHIFT;
	  SETFLD_REGMM32(PFLASH_BE_ADDR, BLOCK_ERASE_ADDR, regVal); 
	  SETFLD_REGMM32(PFLASH_CNTL, CMD_TRIG, PFLASH_CMD_BLOCK_ERASE); 
	  status = pflashWaitCMDTriggerDone();
	  */
	  break;
	
  default:
	  printk(KERN_INFO "Unknown NAND command is issued.\n");
	  break;

  }
  udelay (10);
}

/*
 * NAND read
 */
static int nand_read (struct mtd_info *mtd,
		      loff_t from,
		      size_t len,
		      size_t *retlen,
		      u_char *buf)
{
  return nand_read_ecc (mtd, from, len, retlen, buf, NULL, NULL);
}

/*
 * NAND read with ECC
 */
static int nand_read_ecc (struct mtd_info *mtd,
			  loff_t from,
			  size_t len,
			  size_t *retlen,
			  u_char *buf,
			  u_char *oob_buf,
			  struct nand_oobinfo *oobsel)
{
  int j, col, page, state;
  int erase_state = 0;
  struct nand_chip *this = mtd->priv;
  DECLARE_WAITQUEUE(wait, current);
#ifdef CONFIG_MTD_NAND_ECC
  u_char *data_poi;
  int ecc_result;
  size_t blockoffset;
  u_char ecc_calc[6];
  u_char ecc_code[6];
#endif

  DEBUG ( MTD_DEBUG_LEVEL2,
	  "nand_read_ecc: from = 0x%08x, len = %i\n",
	  (unsigned int) from, (int) len );

  /* Do not allow reads past end of device */
  if ((from + len) > (mtd->size) ) {
    DEBUG ( MTD_DEBUG_LEVEL0,
	    "nand_read_ecc: Attempt read beyond end of device\n" );
    *retlen = 0;
    return -EINVAL;
  }

  /* Grab the lock and see if the device is available */
 retry:
  spin_lock_bh (&this->chip_lock);
  switch (this->state) {
  case FL_READY:
    this->state = FL_READING;
    spin_unlock_bh (&this->chip_lock);
    break;

  case FL_ERASING:
    this->state = FL_READING;
    erase_state = 1;
    spin_unlock_bh (&this->chip_lock);
    break;

  default:
    set_current_state (TASK_UNINTERRUPTIBLE);
    add_wait_queue (&this->wq, &wait);
    spin_unlock_bh (&this->chip_lock);
    schedule();
    remove_wait_queue (&this->wq, &wait);
    goto retry;
  };


#ifdef CONFIG_PCU_ARBITER
  pcuSetNandAccess();
#endif

  /* First we calculate the starting page */
  page = from >> this->page_shift;

  /* Get raw starting column */
  col = from & (mtd->oobblock - 1);

  /* State machine for devices having pages larger than 256 bytes */
  state = (col < mtd->eccsize) ? 0 : 1;

  /* Calculate column address within ECC block context */
  col = (col >= mtd->eccsize) ? (col - mtd->eccsize) : col;

  /* Initialize return value */
  *retlen = 0;

  pflashClearStateMachine();

  /* Loop until all data read */
  while (*retlen < len) {

#ifdef CONFIG_MTD_NAND_ECC
    int need_copy;

    /* 
     * If the read is half-page aligned and takes up a full half-page,
     * we read into return buffer directly; otherwise, we have to read
     * into data buffer due and copy later.  We also modify 'blockoffset' so
     * that adding blockoffset to 'from' will align the beginning of the
     * read.
     */
    if (!col && (len - *retlen) >= mtd->eccsize) {
      data_poi = &buf[*retlen];
      blockoffset = *retlen;
      need_copy = 0;
    }
    else  {
      data_poi = this->data_buf;
      blockoffset = *retlen - col;
      need_copy = 1;
    }

    DEBUG ( MTD_DEBUG_LEVEL3, "blockoffset=0x%x, *retlen=0x%x\n",
	    blockoffset, *retlen );

    /* Read in a block big enough for ECC */
    pflashClearStateMachine();
    for (j=0 ; j < mtd->eccsize; j++) {
      adj_offset = getAdjOffset(from+blockoffset+j);
      data_poi[j]= FAST_PFLASH_READ_8(adj_offset);
    }

    /* Calculate the ECC and verify/correct 1 bit */
    if (!state) {
      for (j=0 ; j<3 ; j++){
	pflash_oob_enable(10);
	adj_offset = getAdjOffset(from+blockoffset+j);
	ecc_code[j] = PFLASH_READ_8(adj_offset);
      }

      nand_calculate_ecc (mtd, data_poi, &ecc_calc[0]);
      ecc_result = nand_correct_data (mtd, data_poi,
				      &ecc_code[0], &ecc_calc[0]);

      DEBUG ( MTD_DEBUG_LEVEL3,
	      "nand_read_ecc: ecc0=0x%x ecc1=0x%x ecc2=0x%x\n",
	      ecc_code[0], ecc_code[1], ecc_code[2]);
      DEBUG ( MTD_DEBUG_LEVEL3,
	      "nand_read_ecc: cecc0=0x%x cecc1=0x%x cecc2=0x%x\n",
	      ecc_calc[0], ecc_calc[1], ecc_calc[2]);
    }
    else {
      pflash_oob_enable(10);
      adj_offset = getAdjOffset(from+blockoffset-mtd->eccsize+SPARE_AREA_U256_ECC0_OFFSET);
      ecc_code[3]= PFLASH_READ_8(adj_offset);

      pflash_oob_enable(10);
      adj_offset = getAdjOffset(from+blockoffset-mtd->eccsize+SPARE_AREA_U256_ECC1_OFFSET);
      ecc_code[4] = PFLASH_READ_8(adj_offset);

      pflash_oob_enable(10);
      adj_offset = getAdjOffset(from+blockoffset-mtd->eccsize+SPARE_AREA_U256_ECC2_OFFSET);
      ecc_code[5] = PFLASH_READ_8(adj_offset);
      pflashClearStateMachine();

      nand_calculate_ecc (mtd, data_poi, &ecc_calc[3]);
      ecc_result = nand_correct_data (mtd, data_poi,
				      &ecc_code[3], &ecc_calc[3]);

      DEBUG ( MTD_DEBUG_LEVEL3,
	      "nand_read_ecc: ecc3=0x%x ecc4=0x%x ecc5=0x%x\n",
	      ecc_code[3], ecc_code[4], ecc_code[5]);
      DEBUG ( MTD_DEBUG_LEVEL3,
	      "nand_read_ecc: cecc3=0x%x cecc4=0x%x cecc5=0x%x\n",
	      ecc_calc[3], ecc_calc[4], ecc_calc[5]);
    }

    if (ecc_result == -1) {
      DEBUG (MTD_DEBUG_LEVEL2,
	     "nand_read_ecc: " \
	     "Failed ECC read, page 0x%08x\n", page);

#ifdef CONFIG_PCU_ARBITER
      pcuSetFlexbusAccess();
#endif

      spin_lock_bh (&this->chip_lock);
      if (erase_state)
	this->state = FL_ERASING;
      else
	this->state = FL_READY;
      wake_up (&this->wq);
      spin_unlock_bh (&this->chip_lock);
      return -EIO;
    }

    /* Read the data from ECC data buffer into return buffer */
    if (need_copy) {
      for (j=col; (j < mtd->eccsize) && (*retlen < len); j++)
	buf[(*retlen)++] = data_poi[j];
    } else {
      *retlen += mtd->eccsize;
    }
#else  /* CONFIG_MTD_NAND_ECC */

    /* Read the data directly into the return buffer */
    if ((*retlen + (mtd->eccsize - col)) >= len) {
      while (*retlen < len){

#ifdef CONFIG_NAND_BAD_BLOCK_CHECK
	poffset =   from+(*retlen);
	adj_offset = stw_nand_block_map(&poffset);
#else
	adj_offset = from+(*retlen);
#endif
	buf[(*retlen)++] = FAST_PFLASH_READ_8(adj_offset);
      }
      /* We're done */
      continue;
    }
    else
      for (j=col ; j < mtd->eccsize; j++) {

#ifdef CONFIG_NAND_BAD_BLOCK_CHECK
	poffset =   from+(*retlen);
	adj_offset = stw_nand_block_map(&poffset);
#else
	adj_offset = from+(*retlen);
#endif
	buf[(*retlen)++] = FAST_PFLASH_READ_8(adj_offset);
      }

#endif  /* CONFIG_MTD_NAND_ECC */

    /*
     * If the amount of data to be read is greater than
     * (256 - col), then all subsequent reads will take
     * place on page or half-page (in the case of 512 byte
     * page devices) aligned boundaries and the column
     * address will be zero. Setting the column address to
     * to zero after the first read allows us to simplify
     * the reading of data and the if/else statements above.
     */
    if (col)
      col = 0x00;

    /* Increment page address */
    if ( (mtd->oobblock == 256) || state )
      page++;

    /* Toggle state machine */
    if (mtd->oobblock == 512)
      state = state ? 0 : 1;
  }

#ifdef CONFIG_PCU_ARBITER
  pcuSetFlexbusAccess();
#endif

  /* Wake up anyone waiting on the device */
  spin_lock_bh (&this->chip_lock);
  if (erase_state)
    this->state = FL_ERASING;
  else
    this->state = FL_READY;
  wake_up (&this->wq);
  spin_unlock_bh (&this->chip_lock);

  DEBUG ( MTD_DEBUG_LEVEL2,
	  "nand_read_ecc(end): from = 0x%08x, len = %d\n",
	  (unsigned int) from, (int) *retlen );

  /* Return OK */
  return 0;

}

/*
 * NAND read out-of-band
 */
static int nand_read_oob (struct mtd_info *mtd,
			  loff_t from,
			  size_t len,
			  size_t *retlen,
			  u_char *buf)
{
  int j, col, i,offset;
  
  int erase_state = 0;
  struct nand_chip *this = mtd->priv;
  DECLARE_WAITQUEUE(wait, current);

    
  DEBUG ( MTD_DEBUG_LEVEL2,
	  "nand_read_oob: from = 0x%08x, len = %i\n",
	  (unsigned int) from, (int) len );

  /* Do not allow reads past end of device */
  if ((from + len) > (mtd->size) ) {
    DEBUG ( MTD_DEBUG_LEVEL0,
	    "nand_read_oob: Attempt read beyond end of device\n" );
    *retlen = 0;
    return -EINVAL;
  }

  /* Grab the lock and see if the device is available */
retry:
  spin_lock_bh (&this->chip_lock);
  switch (this->state) {
  case FL_READY:
    this->state = FL_READING;
    spin_unlock_bh (&this->chip_lock);
    break;

  case FL_ERASING:
    this->state = FL_READING;
    erase_state = 1;
    spin_unlock_bh (&this->chip_lock);
    break;

  default:
    set_current_state (TASK_UNINTERRUPTIBLE);
    add_wait_queue (&this->wq, &wait);
    spin_unlock_bh (&this->chip_lock);
    schedule();
    remove_wait_queue (&this->wq, &wait);
    goto retry;
  };


  /* Mask to get column */
  col = from & 0x0f;

  /* Assuming no ECC on oob data. Not used in nand.c -- ahj */
  /* Loop until all data read */
  i = 0;
  offset = 0;
  while (i < len) {
    int thislen = (mtd->oobsize - col) & (mtd->oobsize - 1);
    if (!thislen)
      thislen = mtd->oobsize;
    thislen = min_t(int, thislen, len);
    j = 0;
    while (j < thislen) {
      pflash_oob_enable(10);
      adj_offset = getAdjOffset(from+offset+j);
      buf[i + j++] = PFLASH_READ_8(adj_offset);
    }
    i += thislen;
    col += thislen;
    offset += mtd->oobblock;
    /* Delay between pages */
    udelay (this->chip_delay);
  }
  pflashClearStateMachine();

  /* Wake up anyone waiting on the device */
  spin_lock_bh (&this->chip_lock);
  if (erase_state)
    this->state = FL_ERASING;
  else
    this->state = FL_READY;
  wake_up (&this->wq);
  spin_unlock_bh (&this->chip_lock);

  /* Return OK */
  *retlen = i;
  return 0;

}

/*
 * NAND write
 */
static int nand_write (struct mtd_info *mtd,
		       loff_t to,
		       size_t len,
		       size_t *retlen,
		       const u_char *buf)
{
  return nand_write_ecc (mtd, to, len, retlen, buf, NULL, NULL);
}


/*
 * NAND write with ECC
 */
static int nand_write_ecc (struct mtd_info *mtd,
			   loff_t to,
			   size_t len,
			   size_t *retlen,
			   const u_char *buf,
			   u_char *oob_buf,
			   struct nand_oobinfo *oobsel)
{
#define NOTALIGNED(x) (x & (mtd->oobblock-1)) != 0
  int i,j, page, col, cnt;
  struct nand_chip *this = mtd->priv;
  DECLARE_WAITQUEUE(wait, current);

#ifdef CONFIG_MTD_NAND_ECC
  int ecc_bytes = (mtd->oobblock == 512) ? 6 : 3;
  u_char ecc_code[6];
#endif

  DEBUG (MTD_DEBUG_LEVEL2,
	 "nand_write_ecc: to = 0x%08x, len = 0x%x\n",
	 (unsigned int) to, (int) len);

  /* Do not allow write past end of page */
  if ((to + len) > mtd->size) {
    DEBUG (MTD_DEBUG_LEVEL0,
	   "nand_write_ecc: Attempted write past end of device\n");
    return -EINVAL;
  }

  /* reject writes, which are not page aligned */	
  if (NOTALIGNED (to)) {
	  printk (KERN_NOTICE "nand_write_ecc: Attempt to write not page aligned data\n");
	  return -EINVAL;
  }


 retry:
  /* Grab the lock and see if the device is available */
  spin_lock_bh (&this->chip_lock);
  switch (this->state) {
  case FL_READY:
    this->state = FL_WRITING;
    spin_unlock_bh (&this->chip_lock);
    break;

  default:
    set_current_state (TASK_UNINTERRUPTIBLE);
    add_wait_queue (&this->wq, &wait);
    spin_unlock_bh (&this->chip_lock);
    schedule();

    remove_wait_queue (&this->wq, &wait);
    goto retry;
  };


#ifdef CONFIG_PCU_ARBITER
  pcuSetNandAccess();
#endif

  /* Shift to get page */
  page = ((int) to) >> this->page_shift;

  /* Get the starting column */
  /* This will always be zero if we are writting on a page boundary. */
  col = to & (mtd->oobblock - 1);

  /* Initialize return length value */
  *retlen = 0;

  DEBUG( MTD_DEBUG_LEVEL3, "nand_write_ecc col: %x page: %x\n",
	 col, page );

  /* Loop until all data is written */
  while (*retlen < len) {
    /* Write data into buffer */
    if ((col + len) >= mtd->oobblock)
      for(i=col, cnt=0 ; i < mtd->oobblock ; i++, cnt++)
	this->data_buf[i] = buf[(*retlen + cnt)];
    else
      for(i=col, cnt=0 ; cnt < (len - *retlen) ; i++, cnt++)
	this->data_buf[i] = buf[(*retlen + cnt)];

    /* Write post-padding bytes into buffer.  Can happen only if length
     * isn't a multiple of page size.  Must happen before we calculate
     * the ecc values.  -- assumption is that memory following the
     * written space is all 0xff.  If not, we would need to read in that
     * memory here.   For now, we'll assume we only write to a page once
     * w/o erasing.  */
    if ((col + (len - *retlen)) < mtd->oobblock) {
      for(i=(col + cnt) ; i < mtd->oobblock ; i++)
	this->data_buf[i] = 0xff;
    }

#ifdef CONFIG_MTD_NAND_ECC
    /* Zero out the ECC array */
    for (i=0 ; i < 6 ; i++)
      ecc_code[i] = 0x00;

    /* Calculate and write the first ECC. */
    if ((col < mtd->eccsize) &&
	((col + (len - *retlen)) >= mtd->eccsize)) {

      DEBUG( MTD_DEBUG_LEVEL3, "NAND_ECC write_ecc - col=%x\n", col);

      /*  This loop should never happen on a page boundary. */
      for (i=0 ; i < col ; i++) {

#ifdef CONFIG_NAND_BAD_BLOCK_CHECK
	poffset   = to-col+i;
	adj_offset = stw_nand_block_map(&poffset);
#else
	adj_offset = to-col+i;
#endif
	this->data_buf[i] = FAST_PFLASH_READ_8(adj_offset);

      }
      nand_calculate_ecc (mtd, &this->data_buf[0], &ecc_code[0]);
      DEBUG( MTD_DEBUG_LEVEL3, "ecc_code L256: 0x%x 0x%x 0x%x\n",
	     ecc_code[0], ecc_code[1], ecc_code[2]);

      /* pay attention to the OOB offset for the ECC data */
      for (i=0 ; i<3 ; i++)
	this->data_buf[(mtd->oobblock + i)] = ecc_code[i];
    }

    /* Calculate and write the second ECC if we have enough data */
    /* In the original code, 'enough data' meant a full page.  I believe
     * the correct meaning would be any data on the second half of the
     * page would require ecc values. */
    if ((mtd->oobblock == 512) && ((col + (len - *retlen)) > mtd->eccsize)) {
      nand_calculate_ecc (mtd, &this->data_buf[256], &ecc_code[3]);

	    /* pay attention to the OOB offset for the ECC data */
      this->data_buf[(mtd->oobblock + SPARE_AREA_U256_ECC0_OFFSET)]
	= ecc_code[3];
      this->data_buf[(mtd->oobblock + SPARE_AREA_U256_ECC1_OFFSET)]
	= ecc_code[4];
      this->data_buf[(mtd->oobblock + SPARE_AREA_U256_ECC2_OFFSET)]
	= ecc_code[5];

      DEBUG( MTD_DEBUG_LEVEL3, "ecc_code U256: 0x%x 0x%x 0x%x\n",
	     ecc_code[3], ecc_code[4], ecc_code[5]);
      DEBUG( MTD_DEBUG_LEVEL3, "bad block marker: 0x%x\n",
	     this->data_buf[mtd->oobblock+SPARE_AREA_VALID_BLOCK_OFFSET]);

    }
		
    /* Write ones */
    this->data_buf[516] = 0xff;
    this->data_buf[mtd->oobblock+SPARE_AREA_VALID_BLOCK_OFFSET] = 0xff;
    for (i=(ecc_bytes+2) ; i < mtd->oobsize ; i++)
      this->data_buf[(mtd->oobblock + i)] = 0xff;
#else
    /* Write ones */
    for (i=mtd->oobblock ; i < (mtd->oobblock + mtd->oobsize) ; i++)
      this->data_buf[i] = 0xff;
    /*  Fake ECC values for jffs2 */
    this->data_buf[mtd->oobblock + SPARE_AREA_L256_ECC0_OFFSET] = 0x54;
    this->data_buf[mtd->oobblock + SPARE_AREA_U256_ECC0_OFFSET] = 0x42;
#endif

    /* Write pre-padding bytes into buffer -- should never happen
     * as long as we write only on page boundaries.  */
    for (i=0 ; i < col ; i++)
      this->data_buf[i] = 0xff;

    /* Write out complete page of data */
    pflashClearStateMachine(); 
    for (i=0 ; i < (mtd->oobblock + mtd->oobsize) ; i++) {
      if(i>=mtd->oobblock) {
	pflashClearStateMachine(); 
	j = i-mtd->oobblock;
        adj_offset = getAdjOffset(to-col+j+(*retlen));
	pflash_oob_enable(10);
      }
      else {
        adj_offset = getAdjOffset(to-col+i+(*retlen));
      }
      FAST_PFLASH_WRITE_8( (adj_offset), this->data_buf[i] );
    }

#ifdef CONFIG_MTD_NAND_VERIFY_WRITE
    /*
     * The NAND device assumes that it is always writing to
     * a cleanly erased page. Hence, it performs its internal
     * write verification only on bits that transitioned from
     * 1 to 0. The device does NOT verify the whole page on a
     * byte by byte basis. It is possible that the page was
     * not completely erased or the page is becoming unusable
     * due to wear. The read with ECC would catch the error
     * later when the ECC page check fails, but we would rather
     * catch it early in the page write stage. Better to write
     * no data than invalid data.
     */

    pflashClearStateMachine();		

    DEBUG ( MTD_DEBUG_LEVEL3,
	    "NAND_VERIFY_WRITE - col=0x%x, cnt=0x%x, *retlen=0x%x\n",
	    col, cnt, *retlen );

    /* Loop through and verify the data */
    for (i=col ; i < cnt ; i++) {

#ifdef CONFIG_NAND_BAD_BLOCK_CHECK
      poffset =   to-col+i+(*retlen);
      adj_offset = stw_nand_block_map(&poffset);
#else
      adj_offset = to-col+i+(*retlen);
#endif
      if (this->data_buf[i] != FAST_PFLASH_READ_8(adj_offset)) {
	DEBUG (MTD_DEBUG_LEVEL0,
	       "nand_write_ecc: " \
	       "Failed write verify, page 0x%08x, " \
	       "%6i bytes were succesful\n",
	       page, *retlen);

#ifdef CONFIG_PCU_ARBITER
	pcuSetFlexbusAccess();
#endif

	spin_lock_bh (&this->chip_lock);
	this->state = FL_READY;
	wake_up (&this->wq);
	spin_unlock_bh (&this->chip_lock);
	return -EIO;
      }
    }

#ifdef CONFIG_MTD_NAND_ECC
      /*
       * We also want to check that the ECC bytes wrote
       * correctly for the same reasons stated above.
       */
      /* READ OOB */
      pflash_oob_enable(10);
      for (i=0 ; i < (ecc_bytes-2) ; i++) {
	pflash_oob_enable(10);
#ifdef CONFIG_NAND_BAD_BLOCK_CHECK
	poffset =   to+i;
	adj_offset = stw_nand_block_map(&poffset);
#else
	adj_offset = to+i;
#endif
	if ((PFLASH_READ_8(adj_offset) != ecc_code[i]) && ecc_code[i]) {
	  DEBUG (MTD_DEBUG_LEVEL0,
		 "nand_write_ecc: Failed ECC write " \
		 "verify, page 0x%08x, " \
		 "%6i bytes were succesful, ecc_code[%x]=0x%0x\n",
		 page, i, i, ecc_code[i]);

#ifdef CONFIG_PCU_ARBITER
	pcuSetFlexbusAccess();
#endif

	  spin_lock_bh (&this->chip_lock);
	  this->state = FL_READY;
	  wake_up (&this->wq);
	  spin_unlock_bh (&this->chip_lock);
	  return -EIO;
	}
      }

      pflash_oob_enable(10);

#ifdef CONFIG_NAND_BAD_BLOCK_CHECK
      poffset   = to + SPARE_AREA_U256_ECC1_OFFSET;
      adj_offset = stw_nand_block_map(&poffset);
#else
      adj_offset = to + SPARE_AREA_U256_ECC1_OFFSET;
#endif
	if ((PFLASH_READ_8(adj_offset) != ecc_code[4]) && ecc_code[4]) {
	  DEBUG (MTD_DEBUG_LEVEL0,
		 "nand_write_ecc: Failed ECC write " \
		 "verify, page 0x%08x, " \
		 "%6i bytes were succesful, ecc_code[%x]=0x%0x\n",
		 page, 4, 4, ecc_code[4]);
#ifdef CONFIG_PCU_ARBITER
	pcuSetFlexbusAccess();
#endif
	  spin_lock_bh (&this->chip_lock);
	  this->state = FL_READY;
	  wake_up (&this->wq);
	  spin_unlock_bh (&this->chip_lock);
	  return -EIO;
	}

	pflash_oob_enable(10);

#ifdef CONFIG_NAND_BAD_BLOCK_CHECK
	poffset   = to + SPARE_AREA_U256_ECC2_OFFSET;
	adj_offset = stw_nand_block_map(&poffset);
#else
	adj_offset = to + SPARE_AREA_U256_ECC2_OFFSET;
#endif
	if ((PFLASH_READ_8(adj_offset) != ecc_code[5]) && ecc_code[5]) {
	  DEBUG (MTD_DEBUG_LEVEL0,
		 "nand_write_ecc: Failed ECC write " \
		 "verify, page 0x%08x, " \
		 "%6i bytes were succesful, ecc_code[%x]=0x%0x\n",
		 page, 5, 5, ecc_code[5]);
#ifdef CONFIG_PCU_ARBITER
	pcuSetFlexbusAccess();
#endif
	  spin_lock_bh (&this->chip_lock);
	  this->state = FL_READY;
	  wake_up (&this->wq);
	  spin_unlock_bh (&this->chip_lock);
	  return -EIO;
	}

	pflashClearStateMachine();
#endif
#endif
	
	/*
	 * If we are writing a large amount of data and/or it
	 * crosses page or half-page boundaries, we set the
	 * the column to zero. It simplifies the program logic.
	 */
	if (col)
	  col = 0x00;

	/* Update written bytes count */
	*retlen += cnt;

	/* Increment page address */
	page++;

	pflashClearStateMachine();

  }  /* while len loop */

#ifdef CONFIG_PCU_ARBITER
	pcuSetFlexbusAccess();
#endif

  /* Wake up anyone waiting on the device */
  spin_lock_bh (&this->chip_lock);
  this->state = FL_READY;
  wake_up (&this->wq);
  spin_unlock_bh (&this->chip_lock);

  /* Return OK */
  *retlen = len;
  return 0;

#undef NOTALIGNED
}


/* This code was lifted from writev.c in the fs/jffs2 directory.
 * Quick and dirty way to do the write.  At some point we may replace it to
 * be a little more efficient.  */
static int mtd_fake_writev_ecc(struct mtd_info *mtd,
			       const struct iovec *vecs, unsigned long count,
			       loff_t to, size_t * retlen, u_char *eccbuf,
			       struct nand_oobinfo *oobsel)
{
  size_t totlen = 0, thislen;
  unsigned char outbuf[512];
  unsigned char *bufp;
  int ret = 0;
  int vecndx = 0;
  int writelen;


  /* Loop until all iovecs' data has been written */
  while (count) {
  DEBUG (MTD_DEBUG_LEVEL2,
	 "mtd_fake_writev_ecc: to = 0x%08x, len = 0x%x\n",
	 (unsigned int) to, (int) vecs->iov_len);

    /*  Just skip any vectors whose size is zero */
    if (vecs->iov_len == 0 || vecs->iov_base == NULL) {
      count--;
      vecs++;
      continue;
    }

    /*
     *  Check, if the tuple gives us enough data for a full page write or if
     *  this is the last vector.  If so, we can use the iov direct, else we
     *  have to copy into data_buf.
     */
    if (vecs->iov_len >= mtd->oobblock || count == 1) {
      bufp = (u_char *) &vecs->iov_base[vecndx];
      writelen = min_t(int, mtd->oobblock, vecs->iov_len - vecndx) ;
      vecndx += writelen;

      /* Check, if we have to switch to the next tuple */
      if (vecndx >= (int) vecs->iov_len) {
	vecs++;
	vecndx = 0;
	count--;
      }
    } else {
      int cnt = 0;
      /*
       * Read data out of each tuple until we have a full page
       * to write or we've read all the tuples.
       */
      DEBUG (MTD_DEBUG_LEVEL2,
	     "mtd_fake_writev_ecc(1): to = 0x%08x, len = 0x%x ",
	     (unsigned int) to+totlen, (int) vecs->iov_len);
      DEBUG (MTD_DEBUG_LEVEL2,
	     "count = %d, ndx = 0x%x\n", count, vecndx);
      while ((cnt < mtd->oobblock) && count > 0) {
	if (vecs->iov_base != NULL && vecs->iov_len) {
	  outbuf[cnt++] = ((u_char *) vecs->iov_base)[vecndx++];
	  /* Check, if we have to switch to the next tuple */
	  if (vecndx >= (int) vecs->iov_len) {
	    vecs++;
	    vecndx = 0;
	    count--;
	    DEBUG (MTD_DEBUG_LEVEL2,
		   "mtd_fake_writev_ecc(2): to = 0x%08x, len = 0x%x ",
		   (unsigned int) to+totlen, (int) vecs->iov_len);
	    DEBUG (MTD_DEBUG_LEVEL2,
		   "count = %d, cnt = 0x%x\n", count, cnt);
	  }
	} else {
	  vecs++;
	  count--;
	  DEBUG (MTD_DEBUG_LEVEL2,
		 "mtd_fake_writev_ecc(3): to = 0x%08x, len = 0x%x ",
		 (unsigned int) to+totlen, (int) vecs->iov_len);
	  DEBUG (MTD_DEBUG_LEVEL2,
		 "count = %d, cnt = 0x%x\n", count, cnt);
	}
      }
      writelen = cnt;
      bufp = outbuf;
    }

    /* We use the same function for write and writev !) */
    ret = mtd->write_ecc(mtd, to+totlen, writelen, &thislen, bufp, eccbuf, oobsel);

    if (ret != 0) {
      DEBUG (MTD_DEBUG_LEVEL2,
	 "mtd_fake_writev_ecc **** FAILED: to = 0x%08x, len = 0x%x\n",
	 (unsigned int) to+totlen, (int) writelen);
      return ret;
    }

    /* Update written bytes count */
    totlen += writelen;;

  }

  if (retlen)
    *retlen = totlen;

  DEBUG (MTD_DEBUG_LEVEL2,
     "mtd_fake_writev_ecc **** SUCCEEDED: to = 0x%08x, len = %d\n",
     	(unsigned int)to, (int)totlen);
  return 0;
}

/*
 * NAND write with iovec
 */
static int nand_writev (struct mtd_info *mtd, const struct iovec *vecs, unsigned long count,
		loff_t to, size_t * retlen)
{
	return (mtd_fake_writev_ecc (mtd, vecs, count, to, retlen, NULL, 0));
}

/*
 * NAND write out-of-band
 */
static int nand_write_oob (struct mtd_info *mtd,
			   loff_t to,
			   size_t len,
			   size_t *retlen,
			   const u_char *buf)
{
  int column, page, i;
  struct nand_chip *this = mtd->priv;
  DECLARE_WAITQUEUE(wait, current);

  DEBUG (MTD_DEBUG_LEVEL2, "nand_write_oob: to = 0x%08x, len = %i\n", (unsigned int) to, (int) len);

  /* Shift to get page */
  page = ((int) to) >> this->page_shift;

  /* Mask to get column */
  column = to & 0x1f;

  /* Initialize return length value */
  *retlen = 0;

  /* Do not allow write past end of page */
  if ((column + len) > mtd->oobsize) {
	  DEBUG (MTD_DEBUG_LEVEL0, "nand_write_oob: Attempt to write past end of page\n");
	  return -EINVAL;
  }

  /* Do not allow write past end of page */
  if ((to + len) > mtd->size) {
    DEBUG (MTD_DEBUG_LEVEL0,
	   "nand_write_oob: Attempted write past end of device\n");
    return -EINVAL;
  }

retry:
  /* Grab the lock and see if the device is available */
  spin_lock_bh (&this->chip_lock);
  switch (this->state) {
  case FL_READY:
    this->state = FL_WRITING;
    spin_unlock_bh (&this->chip_lock);
    break;

  default:
    set_current_state (TASK_UNINTERRUPTIBLE);
    add_wait_queue (&this->wq, &wait);
    spin_unlock_bh (&this->chip_lock);
    schedule();

    remove_wait_queue (&this->wq, &wait);
    goto retry;
  };


  DEBUG( MTD_DEBUG_LEVEL3, "nand_write_oob col: %x page: %x\n",
	 column, page );

#if 0
  /* Prepad with 0xff */
  for (i=0; i < column; i++) {
    pflash_oob_enable(10);
    adj_offset = getAdjOffset(to-column+i);
    PFLASH_WRITE_8(adj_offset, 0xff);
    /* printk(KERN_INFO "oobwrite: 0x%x -> 0x%x\n", 
              0xff, adj_offset); 
    */
  }
#endif

  /* Write data into buffer */
  for (i=0; i < len; i++) {
    pflash_oob_enable(10);
    adj_offset = getAdjOffset(to+i);
    PFLASH_WRITE_8(adj_offset, buf[i]);
    pflashClearStateMachine();
    /* printk(KERN_INFO "oobwrite: 0x%x -> 0x%x\n", 
              buf[i], adj_offset); 
    */
  }

#if 0
  /* Postpad with 0xff */
  for (i=0; i < mtd->oobsize - (len + column); i++) {
    pflash_oob_enable(10);
    adj_offset = getAdjOffset(to+column+i);
    PFLASH_WRITE_8(adj_offset, 0xff);
    /* printk(KERN_INFO "oobwrite: 0x%x -> 0x%x\n", 
       0xff, adj_offset); 
    */
  }
#endif

  pflashClearStateMachine();

#ifdef CONFIG_MTD_NAND_VERIFY_WRITE
  /*
   * The NAND device assumes that it is always writing to
   * a cleanly erased page. Hence, it performs its internal
   * write verification only on bits that transitioned from
   * 1 to 0. The device does NOT verify the whole page on a
   * byte by byte basis. It is possible that the page was
   * not completely erased or the page is becoming unusable
   * due to wear. The read with ECC would catch the error
   * later when the ECC page check fails, but we would rather
   * catch it early in the page write stage. Better to write
   * no data than invalid data.
   */

  DEBUG ( MTD_DEBUG_LEVEL3,
	  "NAND_VERIFY_WRITE - col=0x%x, len=0x%x\n",
	  column, len );
  /* Loop through and verify the data. */
  for (i=0 ; i < len ; i++) {
    unsigned char read;
    pflash_oob_enable(10);
    adj_offset = to+i;
    read = PFLASH_READ_8(adj_offset);
    if (buf[i] != read) {
      DEBUG (MTD_DEBUG_LEVEL0,
	     "nand_write_oob: "
	     "Failed write verify, page 0x%08x, "
	     "Wrote %x, got %x, "
	     "%6i bytes were succesful\n",
	     page, buf[i], read, i);

      spin_lock_bh (&this->chip_lock);
      this->state = FL_READY;
      wake_up (&this->wq);
      spin_unlock_bh (&this->chip_lock);
      *retlen = i;
      return -EIO;
    }
  }
  pflashClearStateMachine();

#endif


  /* Wake up anyone waiting on the device */
  spin_lock_bh (&this->chip_lock);
  this->state = FL_READY;
  wake_up (&this->wq);
  spin_unlock_bh (&this->chip_lock);

  /* Return OK */
  *retlen = len;
  return 0;

}

/*
 * NAND erase a block, do not erase special and invalid blocks which marked as
 * non-zero values in IBTs.
 */
static int nand_erase (struct mtd_info *mtd,
		       struct erase_info *instr)
{

  int page, len, status, pages_per_block,ret;
  struct nand_chip *this = mtd->priv;
  CHIP_SEL   erase_cs;
  uint32_t   start_addr, pflash_be_num, pflash_be_addr;
  uint32_t   trial_num = 10;
#ifndef CONFIG_NAND_BAD_BLOCK_CHECK
  u_char marker;
#endif

  DECLARE_WAITQUEUE(wait, current);

  DEBUG (MTD_DEBUG_LEVEL2,
	 "nand_erase: start = 0x%08x, len = %i\n",
	 (unsigned int) instr->addr, (unsigned int) instr->len);

  /* Start address must align on block boundary */
  if (instr->addr & (mtd->erasesize - 1)) {
    DEBUG (MTD_DEBUG_LEVEL0,
	   "nand_erase: Unaligned address\n");
    return -EINVAL;
  }

  /* Length must align on block boundary */
  if (instr->len & (mtd->erasesize - 1)) {
    DEBUG (MTD_DEBUG_LEVEL0,
	   "nand_erase: Length not block aligned\n");
    return -EINVAL;
  }

  /* Do not allow erase past end of device */
  if ((instr->len + instr->addr) > mtd->size) {
    DEBUG (MTD_DEBUG_LEVEL0,
	   "nand_erase: Erase past end of device\n");
    return -EINVAL;
  }

retry:
  /* Grab the lock and see if the device is available */
  spin_lock_bh (&this->chip_lock);

  switch (this->state) {
  case FL_READY:
    this->state = FL_ERASING;
    break;

  default:
    set_current_state (TASK_UNINTERRUPTIBLE);
    add_wait_queue (&this->wq, &wait);
    spin_unlock_bh (&this->chip_lock);
    schedule();

    remove_wait_queue (&this->wq, &wait);
    goto retry;
  };

#ifdef CONFIG_PCU_ARBITER
  pcuSetNandAccess();
#endif

  /* Shift to get first page */
  page = (int) (instr->addr >> this->page_shift);

  /* Calculate pages in each block */
  pages_per_block = mtd->erasesize / mtd->oobblock;


  /* Loop through the pages */
  len = instr->len;
  start_addr = instr->addr;
  pflashClearStateMachine();

  instr->state = MTD_ERASING;
  while (len) {
    if ( start_addr < PFLASH_NAND_CS1_OFFSET )
      erase_cs = CS0;
    else
      erase_cs = CS1;

#ifdef CONFIG_STW5X226_NAN
    erase_cs = CS1;
#endif

    /* erase the blocks, ATI's state-machine mode */
    pflash_be_addr = start_addr >>
      PFLASH_BE_ADDR__BLOCK_ERASE_ADDR__SHIFT;
    pflash_be_num = pflash_be_addr >>
      STW_NAND_BLOCK_NUM_SHIFT;

    DEBUG (MTD_DEBUG_LEVEL3,
	   "pflash_be_addr: %x pflash_be_num: %x erase_cs: %x\n",
	   pflash_be_addr, pflash_be_num, erase_cs);

    /* Add bad block checking here, do not erase it
       if a special and invalid block detected.
    */
#ifndef CONFIG_NAND_BAD_BLOCK_CHECK
    pflash_oob_enable(10);
    marker = PFLASH_READ_8(start_addr+SPARE_AREA_VALID_BLOCK_OFFSET);
    pflashClearStateMachine();
    if (marker == 0xff)
#else
    if (pflashIsBlockInvalid(erase_cs, pflash_be_num, 0) == 0 )
#endif
    {
      /* set the address */
      SETFLD_REGMM32(PFLASH_BE_ADDR,BLOCK_ERASE_ADDR, pflash_be_addr);
      /* trigger the block erase */
      SETFLD_REGMM32(PFLASH_CNTL, CMD_TRIG, PFLASH_CMD_BLOCK_ERASE); 
      status = pflashWaitCMDTriggerDone();
      
      /* 
	 check erase failure
      */
      if(status == PFLASH_OK) {
	while(GETFLD_REGMM32(PFLASH_STATUS, BLOCK_ERASE_DONE) == 0) {
	  udelay(1000);
	  if (--trial_num == 0) {
	    printk(KERN_WARNING "timeout waiting for BLOCK_ERASE_DONE\n");
#ifndef CONFIG_NAND_BAD_BLOCK_CHECK
	    /*
	       marking this block invalid in IBTs for block relpacement
	    */
	    if (pflashSetBlockInvalid(erase_cs, pflash_be_num)) {
	      printk(KERN_ERR "set_block_invalid failed.\n");
	    }
#endif
#ifdef CONFIG_PCU_ARBITER
	    pcuSetFlexbusAccess();
#endif
	    return(PFLASH_BLOCK_ERASE_NOT_DONE);
	  }
	}
	status = pflashGetStatus(erase_cs);
	SETFLD_REGMM32(PFLASH_STATUS, BLOCK_ERASE_DONE, 0x0);
	pflashClearStateMachine();
      }
    }
    else{
      /* report invalid blocks for display purpose, do NOT erase it */
      printk(KERN_INFO "...IB at: 0x%x(block: 0x%x)...\n",
	     start_addr, pflash_be_num);
    }

    /* Increment page address and decrement length */
    len -= mtd->erasesize;
    /* increment to next block */
    start_addr += mtd->erasesize;
    page += pages_per_block;

    /* Release the spin lock */
    spin_unlock_bh (&this->chip_lock);

  erase_retry:
    /* Check the state and sleep if it changed */
    spin_lock_bh (&this->chip_lock);
    if (this->state == FL_ERASING) {
      continue;
    }
    else {
      set_current_state (TASK_UNINTERRUPTIBLE);
      add_wait_queue (&this->wq, &wait);
      spin_unlock_bh (&this->chip_lock);
      schedule();

      remove_wait_queue (&this->wq, &wait);
      goto erase_retry;
    }
  }
  instr->state = MTD_ERASE_DONE;

#ifdef CONFIG_PCU_ARBITER
  pcuSetFlexbusAccess();
#endif
  spin_unlock_bh (&this->chip_lock);

  ret = instr->state == MTD_ERASE_DONE ? 0 : -EIO;;
  /* Do call back function */
  if (!ret && instr->callback)
    instr->callback (instr);

	/* The device is ready */
  spin_lock_bh (&this->chip_lock);
  this->state = FL_READY;
  spin_unlock_bh (&this->chip_lock);

  /* Return OK */
  return ret;
}

/*
 * NAND sync
 */
static void nand_sync (struct mtd_info *mtd)
{
  struct nand_chip *this = mtd->priv;
  DECLARE_WAITQUEUE(wait, current);

  DEBUG (MTD_DEBUG_LEVEL3, "nand_sync: called\n");

retry:
  /* Grab the spinlock */
  spin_lock_bh(&this->chip_lock);

  /* See what's going on */
  switch(this->state) {
  case FL_READY:
  case FL_SYNCING:
    this->state = FL_SYNCING;
    spin_unlock_bh (&this->chip_lock);
    break;

  default:
    /* Not an idle state */
    add_wait_queue (&this->wq, &wait);
    spin_unlock_bh (&this->chip_lock);
    schedule ();

    remove_wait_queue (&this->wq, &wait);
    goto retry;
  }

  /* Lock the device */
  spin_lock_bh (&this->chip_lock);

  /* Set the device to be ready again */
  if (this->state == FL_SYNCING) {
    this->state = FL_READY;
    wake_up (&this->wq);
  }

  /* Unlock the device */
  spin_unlock_bh (&this->chip_lock);

}


/*
 * Scan for the NAND devices using ATI's state machine implementation.
 */
int nand_scan (struct mtd_info *mtd, int maxchips)
{
  int i, id, nand_maf_id, nand_dev_id;
  struct nand_chip *this = mtd->priv;


  /* Send the command for reading device ID */
#ifdef CONFIG_STW5X226
#ifdef CONFIG_STW5X226_NAN
  nand_command (mtd, NAND_CMD_READID, 0x00, -1, CS1, 0);
#else
  nand_command (mtd, NAND_CMD_READID, 0x00, -1, CS0, 0);
#endif
#else
  nand_command (mtd, NAND_CMD_READID, 0x00, -1, CS0, 0);
#endif

  id = GETFLD_REGMM32(PFLASH_ID_STAT_DATA, PF_ID_DATA);
  SETFLD_REGMM32(PFLASH_STATUS, ID_READ_DONE, 0x0);
  pflashClearStateMachine();
  nand_maf_id = (id & 0xff);
  nand_dev_id = (id >> 8);

  /* Read manufacturer and device IDs */
  /*
    printk("NAND ChipID: %x  %x  %x\n", id, nand_maf_id, nand_dev_id);
    printk("NAND ChipId: %x ChipId String: %s\n",
    pflash_get_chipID(0,10), pflash_get_chipID_string(0));
    id = pflash_get_chipID(0, 10);
    nand_maf_id = (id & 0xff);
    nand_dev_id = (id >> 8);
  */

	/* Print and store flash device information */

  for (i = 0; nand_flash_ids[i].name != NULL; i++) {
    if (nand_dev_id == nand_flash_ids[i].id && !mtd->size) {
      mtd->name = nand_flash_ids[i].name;
      mtd->erasesize = nand_flash_ids[i].erasesize;
      mtd->size = 2 * (1 << nand_flash_ids[i].chipshift);
      mtd->eccsize = 256;
      this->chipshift = nand_flash_ids[i].chipshift;
      if (nand_flash_ids[i].page256) {
	mtd->oobblock = 256;
	mtd->oobsize = 8;
	this->page_shift = 8;
      } else {
	mtd->oobblock = 512;
	mtd->oobsize = 16;
	this->page_shift = 9;
      }
      /* Try to identify manufacturer */
      for (i = 0; nand_manuf_ids[i].id != 0x0; i++) {
	if (nand_manuf_ids[i].id == nand_maf_id)
	  break;
      }
      printk (KERN_INFO "NAND device: Manufacturer ID:"
      " 0x%02x, Chip ID: 0x%02x (%s %s)\n", nand_maf_id, nand_dev_id,
      nand_manuf_ids[i].name , mtd->name);
      break;
    }
  }


  /* Initialize state and spinlock */
  this->state = FL_READY;
  init_waitqueue_head(&this->wq);
  spin_lock_init(&this->chip_lock);

  /* Print warning message for no device */
  if (!mtd->size) {
    printk (KERN_WARNING "No NAND device found!!!\n");
    return 1;
  }

  /* Fill in remaining MTD driver data */
  mtd->type = MTD_NANDFLASH;
  mtd->flags = MTD_CAP_NANDFLASH | MTD_ECC;
  mtd->ecctype = MTD_ECC_SW;
  mtd->erase = nand_erase;
  mtd->point = NULL;
  mtd->unpoint = NULL;
  mtd->read = nand_read;
  mtd->write = nand_write;
  mtd->read_ecc = nand_read_ecc;
  mtd->write_ecc = nand_write_ecc;
  mtd->read_oob = nand_read_oob;
  mtd->write_oob = nand_write_oob;
  mtd->sync = nand_sync;
  mtd->readv = NULL;
  mtd->writev = nand_writev;
  mtd->writev_ecc = mtd_fake_writev_ecc;
  mtd->lock = NULL;
  mtd->unlock = NULL;
  mtd->suspend = NULL;
  mtd->resume = NULL;

  return 0;
}

#ifdef CONFIG_NAND_BAD_BLOCK_CHECK
static uint32_t stw_nand_block_map(uint32_t *offset)
{
  uint32_t  loc_adj_off;

  /* check if offset adjust is needed */
  lb_num = (*offset) >> BLOCK_NUM_TOTAL_SHIFT;
  if ( nand_logic_block[lb_num] < lb_num ) {
    printk(KERN_WARNING "block mapping failed!\n");
    return -EINVAL;
  }
  else {
    loc_adj_off = (*offset) +
      ( ( nand_logic_block[lb_num]-lb_num ) << BLOCK_NUM_TOTAL_SHIFT );
  }

  /* for debugging purpose */
  /*
  printk(KERN_INFO "offset: 0x%x adj_off: 0x%x lblock: 0x%x pblock: 0x%x\n",
	 *offset, loc_adj_off, lb_num, nand_logic_block[lb_num] );
	 */

  return loc_adj_off;

}
#endif


/*
 * Main initialization routine
 */
int __init stw_nand_init (void)
{

  struct nand_chip *this;
  uint32_t stw_bb=0, bi, lcount, gcount, nb;
  uint32_t pf_block_start, pf_block_end;
  uint8_t  np;

  u32 flashsize, defnum;

  printk(KERN_INFO "STW NAND Flash initializing...\n");

  /* Allocate memory for MTD device structure and private data */
  stw_nand_mtd = kmalloc (sizeof(struct mtd_info) 
			 + sizeof (struct nand_chip), GFP_KERNEL);
  if (!stw_nand_mtd) {
    printk ("Unable to allocate memory for NAND MTD.\n");
    return -ENOMEM;
  }

  /* Get pointer to private data */
  this = (struct nand_chip *) (&stw_nand_mtd[1]);

  /* Initialize structures */
  memset((char *) stw_nand_mtd, 0, sizeof(struct mtd_info));
  memset((char *) this, 0, sizeof(struct nand_chip));

  /* Link the private data with the MTD structure */
  stw_nand_mtd->priv = this;

#ifdef CONFIG_STW5X226
  stw_flashinfo(&flashsize, &defnum);
  pflashInit();
#endif

#ifdef CONFIG_STW4X225
  pflashInit();
#endif

  /* Scan to find existance of the device */
  if (nand_scan (stw_nand_mtd, 2)) {
    kfree (stw_nand_mtd);
    return -ENXIO;
  }

  /* for debug purpose */
  dump_mtd_info(stw_nand_mtd);
  dump_nand_chip_info(this);

  /* Allocate memory for internal data buffer */
  this->data_buf = kmalloc (sizeof(u_char) * 
			    (stw_nand_mtd->oobblock + stw_nand_mtd->oobsize), 
			    GFP_KERNEL);
  if (!this->data_buf) {
    printk ("Unable to allocate NAND data buffer for STW.\n");
    kfree (stw_nand_mtd);
    return -ENOMEM;
  }


#ifdef CONFIG_NAND_BAD_BLOCK_CHECK
  /* 
     Create logical and physical blocks mapping table to make sure 
     the logical blocks used by the root file system are always good.
     
     NOTES: IBTs are created, check the block magic number, and IBTs
     before erasing any blocks. Make sure do not erase invalid and 
     special blocks/

  */

  /* Read IBTs, the tables should be created by MMON or EJTAG program */
  if (pflashReadInvalidBlockTable()){
    printk(KERN_WARNING "stw_nand_init - IBT not found.\n"); 
  }
  
  /* detect special and bad blocks, using IBTs */
  printk(KERN_INFO "Creating logical block table...\n");
  for (bi=0; bi<STW_MAX_NUM_BLOCKS; bi++) {
    /* check attributes of the block */
    if ( pflashIsBlockInvalid(CS0, bi, 0) == 0 ) {
      nand_phy_block[bi] = PFLASH_VALID_BLOCK_VALUE;
    }
    else {
      /* printk(KERN_INFO "Special or invalid block detected - cs: %x block: 0x%x.\n",
	 CS0, bi); */
      nand_phy_block[bi] = PFLASH_INVALID_BLOCK_VALUE;
    }

    if ( pflashIsBlockInvalid(CS1, bi, 0) == 0 ) {
      nand_phy_block[bi+STW_MAX_NUM_BLOCKS] = PFLASH_VALID_BLOCK_VALUE;
    }
    else {
      /* printk(KERN_INFO "Special or invalid block detected - cs: %x block: 0x%x.\n",
	 CS1, bi);*/
      nand_phy_block[bi+STW_MAX_NUM_BLOCKS] = PFLASH_INVALID_BLOCK_VALUE;
    }
  }
  
  /* then, create the logical - valid physical block mapping table */
  /*
  fs_block_start = (NAND_ROOT_FS_OFFSET >> BLOCK_NUM_TOTAL_SHIFT);
  fs_block_end = fs_block_start + 
                   (NAND_ROOT_FS_SIZE >> BLOCK_NUM_TOTAL_SHIFT);
  */

  pf_block_start = (NAND_STW_OFFSET >> BLOCK_NUM_TOTAL_SHIFT);
  pf_block_end = pf_block_start + 
                   (NAND_STW_SIZE >> BLOCK_NUM_TOTAL_SHIFT);

  /* printk(KERN_INFO "Block_start: 0x%x block end: 0x%x\n", 
     pf_block_start, pf_block_end); */

  for ( lcount = 0; lcount < 2*STW_MAX_NUM_BLOCKS; lcount++ ){
    nand_logic_block[lcount]=lcount;
  }

  stw_bb = 0;
  gcount = 0;
  for ( lcount = pf_block_start; lcount < pf_block_end; lcount++ ) {
    if ( (lcount+stw_bb) >= 2*STW_MAX_NUM_BLOCKS ) {
      printk(KERN_INFO "Reach the capacity of valid blocks (total: 0x%3x).\n", lcount);
      break;
    }
    if ( nand_phy_block[lcount] == PFLASH_VALID_BLOCK_VALUE ) {
      nand_logic_block[gcount] = lcount;
      gcount++;
    }
    else {
      stw_bb++;
      /* printk(KERN_INFO "Special or invalid physical block: 0x%3x, total: 0x%x\n", 
	 lcount, stw_bb); */
    }
  }

  for (np=0; np<NUM_PARTITIONS; np++) {
    nb = (partition_info[np].offset >> BLOCK_NUM_TOTAL_SHIFT);
    printk(KERN_INFO "NAND partition %x logical block: 0x%3x physical block: 0x%3x\n", 
	   np, nb, nand_logic_block[nb]);
  }

  printk(KERN_INFO "Creating logical block table done.\n");
#endif


  /* Register the partitions */
  add_mtd_partitions(stw_nand_mtd, 
		     (struct mtd_partition*) partition_info, 
		     (int)NUM_PARTITIONS);

  printk(KERN_INFO "STW NAND Flash devices initialization done.\n");
	
  /* Return OK */
  return 0;

}
module_init(stw_nand_init);

/*
 * Clean up routine
 */
#ifdef MODULE
static void __exit stw_nand_cleanup (void)
{

  struct nand_chip *this = (struct nand_chip *) &stw_nand_mtd[1];

  /* Unregister the device */
  del_mtd_device (stw_nand_mtd);
  del_mtd_partitions(stw_nand_mtd);

  /* Free internal data buffer */
  kfree (this->data_buf);

  /* Free the MTD device structure */
  kfree (stw_nand_mtd);
}
module_exit(stw_nand_cleanup);
#endif

EXPORT_SYMBOL (nand_scan);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("czhou@ati.com");
MODULE_DESCRIPTION("STW NAND flash device interface");







