/******************************************************************************

	drivers/mtd/nand/mx2_nand.c
	driver for i.MX21 on-chip NAND Flash controller.
	tested with NAND Flash devices that come with MX21ADS board:
	SAMSUNG K9F1208Q0A (8bit i/o)
	SAMSUNG K9F5616Q0C (16Bit i/o)

	Copyright (c) 2004 MontaVista Software, Inc. <source@mvista.com>

	Based on mx2 nand support code from Motorola's MX21 BSP:

	Copyright (C) 2003 Motorola Inc. Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
	

********************************************************************************/

#include <linux/slab.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/partitions.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <asm/io.h>
#include <asm/arch/gpio.h>

#define NAND_FLASH_CONFIG2_INT 0x8000
#define NAND_FLASH_CONFIG2_FCMD 0x1
#define NAND_FLASH_CONFIG2_FADD 0x2
#define NAND_FLASH_CONFIG2_FDI  0x4
#define NAND_FLASH_CONFIG2_PAGE_OUT  0x8
#define NAND_FLASH_CONFIG2_ID_OUT  0x10
#define NAND_FLASH_CONFIG2_STATUS_OUT  0x20

#define NAND_FLASH_PAGE_PER_BLOCK           32
#define NAND_FLASH_PAGE_MAIN_SIZE           512
#define	NAND_FLASH_PAGE_SPARE_SIZE          16

#define READCMD                             1
#define READBUF                             2

#ifdef  CONFIG_MX2_16_BIT_NAND

#define MX2NAND_16BIT_DECLARE(name) u32 name

#define MX2NAND_TURNON_16BIT(x) do { \
	save_flags(x); \
	cli(); \
	SYS_FMCR |= (1 << 4); \
} while (0)

#define MX2NAND_TURNOFF_16BIT(x) do { \
	SYS_FMCR &= ~(1 << 4); \
	restore_flags(x); \
} while (0)

#else

#define MX2NAND_16BIT_DECLARE(name)
#define MX2NAND_TURNON_16BIT(x)
#define MX2NAND_TURNOFF_16BIT(x)

#endif

#ifdef CONFIG_MTD_CMDLINE_PARTS
const char *mx2nand_probes[] = { "cmdlinepart", NULL };
#endif

static u32 *mx2nand_mab;
static u32 mx2nand_mab_counter;
static u32 mx2nand_cmd;
static u32 mx2nand_ic = 0;
static u32 mx2nand_page;
static u32 mx2nand_status = 0;
static struct mtd_info *mx2nand_mtd = NULL;

static u8 mx2nand_region_init;
static u8 mx2nand_gpio_init;
static void __init mx2nand_cleanup(void);

#define NAND_TIMEOUT HZ*5
#ifdef  CONFIG_MX2_16_BIT_NAND
#define ADDRCYCLE  3
#else
#define ADDRCYCLE  4
#endif

static int
mx2nand_prog_wait(struct mtd_info *mtd, struct nand_chip *this, int state)
{

	unsigned long timeo = jiffies;
	int status;

	if (state == FL_ERASING)
		timeo += (HZ * 400) / 1000;
	else
		timeo += (HZ * 20) / 1000;

	spin_lock_bh(&this->chip_lock);

	do {
		this->cmdfunc(mtd, NAND_CMD_STATUS, -1, -1);
		status = (int) this->read_byte(mtd);
		if (status & 0x40)
			break;

		/* Check, if we were interrupted */
		if (this->state != state) {
			spin_unlock_bh(&this->chip_lock);
			return 0;
		}

		if (this->dev_ready) {
			if (this->dev_ready(mtd))
				break;
		}
	} while (time_before(jiffies, timeo));

	spin_unlock_bh(&this->chip_lock);

	return status;
}

static int
mx2nand_correct_data(struct mtd_info *mtd, u_char * dat,
		     u_char * read_ecc, u_char * calc_ecc)
{
/*ecc is handled entirely by the MX2 NFC*/
/*report ECC Uncorrectable error - always from main area*/
	if (NFC_ECC_STAT_RES & 0x2)
		return -1;
	else
		return 0;
}

static void
mx2nand_enable_hwecc(struct mtd_info *mtd, int mode)
{
/*ecc is handled entirely by the MX2 NFC*/
}

static void
mx2nand_calculate_ecc(struct mtd_info *mtd, const u_char * dat,
		      u_char * ecc_code)
{
/*ecc is handled entirely by the MX2 NFC*/
}

static void
mx2nand_wait(void)
{
	unsigned long timeout;
	timeout = jiffies + NAND_TIMEOUT;
	while (!(NFC_NF_CONFIG2 & NAND_FLASH_CONFIG2_INT)) {
		if (time_after(jiffies, timeout)) {
			printk(KERN_ERR
			       "MX2_NAND: timeout waiting for Nand command complete");
			return;
		}
	}
}

static u32
mx2nand_send_command(u16 command)
{

	MX2NAND_16BIT_DECLARE(flags);

	NFC_NAND_FLASH_CMD = command;
	MX2NAND_TURNON_16BIT(flags);
	NFC_NF_CONFIG2 = NAND_FLASH_CONFIG2_FCMD;
	mx2nand_wait();
	MX2NAND_TURNOFF_16BIT(flags);
	if (!(NFC_NF_CONFIG2 & NAND_FLASH_CONFIG2_INT))
		return -ETIMEDOUT;

	return 0;
}

static inline u32
mx2nand_send_address(u32 addr, int cycle)
{
	int i = 0;
	MX2NAND_16BIT_DECLARE(flags);

	for (i = 0; i < cycle; i++) {

		if (cycle == ADDRCYCLE && i > 0) {
			NFC_NAND_FLASH_ADDR = (addr >> (i * 8 + 1)) & 0xFF;
		} else {
			NFC_NAND_FLASH_ADDR = (addr >> i * 8) & 0xFF;
		}

		MX2NAND_TURNON_16BIT(flags);
		NFC_NF_CONFIG2 = NAND_FLASH_CONFIG2_FADD;

		mx2nand_wait();
		MX2NAND_TURNOFF_16BIT(flags);
		if (!(NFC_NF_CONFIG2 & NAND_FLASH_CONFIG2_INT))
			return -ETIMEDOUT;
	}
	return 0;
}

static u_char
mx2nand_read_byte(struct mtd_info *mtd)
{
	char b;

	b = (*mx2nand_mab) >> mx2nand_ic * 8;
	mx2nand_ic++;
#ifdef  CONFIG_MX2_16_BIT_NAND
	if (mx2nand_cmd == NAND_CMD_READID || mx2nand_cmd == NAND_CMD_STATUS)
		mx2nand_ic++;
#endif

	if (mx2nand_ic == 4) {
		mx2nand_ic = 0;
		mx2nand_mab++;
	}
	return b;
}

static void
mx2nand_command(struct mtd_info *mtd, unsigned command, int column,
		int page_addr)
{
	int num = 0;
	int i;
	char sc;
	int address;
	MX2NAND_16BIT_DECLARE(flags);

	mx2nand_cmd = command;
	mx2nand_page = page_addr;

	NFC_CONFIGURATION = 0x0002;
	NFC_ULOCK_START_BLK = 0;
	NFC_ULOCK_END_BLK = 0xffff;	/*unlock all */
	NFC_NF_WR_PROT = 0x0004;
	NFC_NF_CONFIG1 = 0x000A;	/*enable ECC, disable Spare Area */
	NFC_RAM_BUF_ADDR = 0x3;

	if (command == NAND_CMD_PAGEPROG) {
		MX2NAND_TURNON_16BIT(flags);
		NFC_NF_CONFIG2 = NAND_FLASH_CONFIG2_FDI;
		mx2nand_wait();
		MX2NAND_TURNOFF_16BIT(flags);
		if (!(NFC_NF_CONFIG2 & NAND_FLASH_CONFIG2_INT)) {
			return;
		}
	}

	if (command == NAND_CMD_READOOB) {
		mx2nand_send_command(NAND_CMD_READ1);
	} else {
		mx2nand_send_command(command);
	}

	switch (command) {
	case NAND_CMD_READ1:
	case NAND_CMD_READ0:
	case NAND_CMD_SEQIN:
	case NAND_CMD_READOOB:
#ifdef CONFIG_MX2_16_BIT_NAND
		num = 3;
#else
		num = 4;
#endif
		address = page_addr * NAND_FLASH_PAGE_MAIN_SIZE;
		break;
	case NAND_CMD_READID:
		num = 1;
		address = 0;
		break;
	case NAND_CMD_ERASE1:
#ifdef CONFIG_MX2_16_BIT_NAND
		num = 2;
#else
		num = 3;
#endif
		address = page_addr;
		break;
	case NAND_CMD_STATUS:
	case NAND_CMD_PAGEPROG:
	case NAND_CMD_ERASE2:
		num = 0;
		break;
	}
	if (num > 0) {
		mx2nand_send_address(address, num);

	}

	switch (command) {
	case NAND_CMD_READOOB:
		MX2NAND_TURNON_16BIT(flags);
		NFC_NF_CONFIG2 = NAND_FLASH_CONFIG2_PAGE_OUT;
		mx2nand_wait();
		MX2NAND_TURNOFF_16BIT(flags);
		if (!(NFC_NF_CONFIG2 & NAND_FLASH_CONFIG2_INT))
			return;

		break;
	case NAND_CMD_READID:
		MX2NAND_TURNON_16BIT(flags);
		NFC_NF_CONFIG2 = NAND_FLASH_CONFIG2_ID_OUT;
		mx2nand_wait();
		MX2NAND_TURNOFF_16BIT(flags);
		if (!(NFC_NF_CONFIG2 & NAND_FLASH_CONFIG2_INT))
			return;

		break;

	case NAND_CMD_STATUS:
		MX2NAND_TURNON_16BIT(flags);
		NFC_NF_CONFIG2 = NAND_FLASH_CONFIG2_STATUS_OUT;
		mx2nand_wait();
		MX2NAND_TURNOFF_16BIT(flags);
		if (!(NFC_NF_CONFIG2 & NAND_FLASH_CONFIG2_INT))
			return;

		break;
	}

	mx2nand_mab = (u32 *) (NFC_MAB_BASE(3));
	mx2nand_mab_counter = 0;
	mx2nand_ic = 0;

	if (command == NAND_CMD_READOOB) {
		mx2nand_mab = (u32 *) (NFC_SAB_BASE(3));
		for (i = 1; i < column; i++) {
			sc = mx2nand_read_byte(mtd);
		}

	}

}

static void
mx2nand_read_buf(struct mtd_info *mtd, u_char * buf, int len)
{
	MX2NAND_16BIT_DECLARE(flags);
	int i;
	u32 *cp = (u32 *) buf;

	if (mx2nand_cmd == NAND_CMD_READ0) {
		switch (mx2nand_status) {
		case READBUF:
			mx2nand_command(mtd, mx2nand_cmd, 0, ++mx2nand_page);
			break;
		case READCMD:
			mx2nand_status = READBUF;
			break;
		}
	}
	mx2nand_mab = (u32 *) (NFC_MAB_BASE(3));
	mx2nand_mab_counter = 0;

	MX2NAND_TURNON_16BIT(flags);
	NFC_NF_CONFIG2 = NAND_FLASH_CONFIG2_PAGE_OUT;
	mx2nand_wait();
	MX2NAND_TURNOFF_16BIT(flags);
	if (!(NFC_NF_CONFIG2 & NAND_FLASH_CONFIG2_INT)) {
		return;
	}

	for (i = 0; i < len / sizeof (u32); i++) {
		*cp++ = *mx2nand_mab++;
	}

}

static void
mx2nand_write_buf(struct mtd_info *mtd, const u_char * buf, int len)
{
	int i;

	u32 *cp = (u32 *) buf;
	u32 *bp = (u32 *) (NFC_MAB_BASE(3));

	if (len == 16) {
		bp = (u32 *) (NFC_SAB_BASE(3));
	}

	for (i = 0; i < len / sizeof (u32); i++) {
		*bp++ = *cp++;
	}

}

static int
mx2nand_verify_buf(struct mtd_info *mtd, const u_char * buf, int len)
{
	MX2NAND_16BIT_DECLARE(flags);
	int i;
	u32 *cp = (u32 *) buf;
	u32 *bp = (u32 *) (NFC_MAB_BASE(3));

	MX2NAND_TURNON_16BIT(flags);
	NFC_NF_CONFIG2 = NAND_FLASH_CONFIG2_PAGE_OUT;
	mx2nand_wait();
	MX2NAND_TURNOFF_16BIT(flags);
	if (!(NFC_NF_CONFIG2 & NAND_FLASH_CONFIG2_INT)) {
		return -ETIMEDOUT;
	}

	for (i = 0; i < len / sizeof (u32); i++) {
		if (*cp++ != *bp++) {
			return i * 4;
		}
	}
	return 0;
}

static void
mx2nand_hwcontrol(struct mtd_info *mtd, int cmd)
{
	mx2nand_status = cmd;
	if (cmd == NAND_CTL_SETNCE)
		mx2nand_status = READCMD;
	else if (cmd == NAND_CTL_CLRNCE)
		mx2nand_status = 0;

}

const static struct mtd_partition mx2nand_partition_info[] = {

      {name:"ipl",
	      offset:0,
      size:0x04000},

      {name:"spl",
	      offset:MTDPART_OFS_APPEND,
      size:0x100000},

      {name:"kernel",
	      offset:MTDPART_OFS_APPEND,
      size:0x200000},

      {name:"RootDisk",
	      offset:MTDPART_OFS_APPEND,
      size:MTDPART_SIZ_FULL}

};

#define MX2_NAND_PARTITIONS_NUM 4

/*NAND pinout:
NFCE_B   T15  PF1
NFCLE    U15  PF3
NFALE    U14  PF4
NFWE_B   T13  PF6
NFRE_B   W13  PF5
NFWP_B   L12  PF2
NFRB     M12  PF0
NFIO0..7  W12, U13, L11, T12, U12, T11, V13, M11  PF7..PF14
A13..A15, A21..A25  F3, F1, G3, G1, H3, L2, L1, M2

Note: PF0..PF14 port mux mode is PRIMARY
*/

#define MX2_NAND_GPIO_PORTF_MASK 0x7fff

/**
 * Module initialization routine
 */

int __init
mx2nand_init(void)
{
	struct nand_chip *this;
	int mtd_parts_nb = 0;
	int tmp;

#ifdef CONFIG_MX2_16_BIT_NAND
	printk(KERN_INFO "MX2 NAND: 16-bit i/o mode\n");
#else
	printk(KERN_INFO "MX2 NAND: 8-bit i/o mode\n");
#endif

	if (!(request_region(NFC_BASE, NFC_IO_SIZE, "mx2nand"))) {
		printk(KERN_ERR "MX2 NAND: NFC region is already in use \n");
		mx2nand_cleanup();
		return -1;
	}
	mx2nand_region_init = 1;

	tmp = mx2_register_gpios(PORT_F, MX2_NAND_GPIO_PORTF_MASK, PRIMARY);
	if (tmp < 0) {
		printk(KERN_ERR
		       "MX2 NAND: Unable to request GPIO pins: PORT F, mask 0x%x\n",
		       MX2_NAND_GPIO_PORTF_MASK);
		mx2nand_cleanup();
		return tmp;
	}
	mx2nand_gpio_init = 1;

	/* Allocate memory for MTD device structure and private data */
	mx2nand_mtd =
	    kmalloc(sizeof (struct mtd_info) + sizeof (struct nand_chip),
		    GFP_KERNEL);
	if (!mx2nand_mtd) {
		printk(KERN_ERR
		       "MX2 NAND: Not enough memory for a MTD device structure\n");
		mx2nand_cleanup();
		return -ENOMEM;
	}

	/* Get pointer to private data */
	this = (struct nand_chip *) (&mx2nand_mtd[1]);

	/* Initialize structures */
	memset((char *) mx2nand_mtd, 0, sizeof (struct mtd_info));
	memset((char *) this, 0, sizeof (struct nand_chip));
	mx2nand_mtd->oobinfo.useecc = 1;

	/* Link the private data with the MTD structure */
	mx2nand_mtd->priv = this;

	this->cmdfunc = mx2nand_command;
	this->hwcontrol = mx2nand_hwcontrol;
	this->read_byte = mx2nand_read_byte;
	this->read_buf = mx2nand_read_buf;
	this->write_buf = mx2nand_write_buf;
	this->verify_buf = mx2nand_verify_buf;
	this->select_chip = NULL;
	this->enable_hwecc = mx2nand_enable_hwecc;
	this->calculate_ecc = mx2nand_calculate_ecc;
	this->eccmode = NAND_ECC_HW3_512;
	this->correct_data = mx2nand_correct_data;
	this->waitfunc = mx2nand_prog_wait;

	CRM_PCDR = (CRM_PCDR & 0xFFFF0FFF) | 0x7000;	/*clock control */

	/* Scan to find existence of the device */
	tmp = nand_scan(mx2nand_mtd, 1);
	if (tmp || (mx2nand_mtd->size == 1)) {	/*nand_scan is buggy */
		printk(KERN_ERR "MX2 NAND: device not found\n");
		mx2nand_cleanup();
		return -ENXIO;
	}

	mx2nand_mtd->read_ecc = NULL;	/*forces use of master mtd oobinfo.useecc */
	mx2nand_mtd->write_ecc = NULL;	/*forces use of master mtd oobinfo.useecc */
	mx2nand_mtd->writev_ecc = NULL;	/*forces use of master mtd oobinfo.useecc */

	/* Allocate memory for internal data buffer */
	this->data_buf =
	    kmalloc(sizeof (u_char) *
		    (mx2nand_mtd->oobblock + mx2nand_mtd->oobsize), GFP_KERNEL);
	if (!this->data_buf) {
		printk(KERN_ERR
		       "MX2 NAND: Unable to allocate NAND data buffer for Mx2.\n");
		mx2nand_cleanup();
		return -ENOMEM;
	}

	/* Register the partitions */
#ifdef CONFIG_MTD_PARTITIONS
#ifdef CONFIG_MTD_CMDLINE_PARTS
	{
		struct mtd_partition *mtd_parts;
		mx2nand_mtd->name = "mx2-nand";
		mtd_parts_nb =
		    parse_mtd_partitions(mx2nand_mtd, mx2nand_probes,
					 &mtd_parts, 0);
		if (mtd_parts_nb > 0)
			tmp =
			    add_mtd_partitions(mx2nand_mtd, mtd_parts,
					       mtd_parts_nb);
	}
#endif				/* end of CONFIG_MTD_CMDLINE_PARTS */
	if (mtd_parts_nb <= 0) {
		tmp = add_mtd_partitions(mx2nand_mtd, mx2nand_partition_info,
					 MX2_NAND_PARTITIONS_NUM);
	}
	if (tmp < 0) {
		printk(KERN_ERR "MX2 NAND: Error registering MTD partitions\n");
		mx2nand_cleanup();
		return tmp;
	}
#else				/* if !CONFIG_MTD_PARTITIONS */
	tmp = add_mtd_device(mx2nand_mtd);
	if (tmp < 0) {
		printk(KERN_ERR "MX2 NAND: Error registering MTD device\n");
		mx2nand_cleanup();
		return tmp;
	}
#endif				/* end of CONFIG_MTD_PARTITIONS */

	/* Return happy */
	return 0;
}

/**
 * Module clean up routine
 */
static void __init
mx2nand_cleanup(void)
{
	struct nand_chip *this = (struct nand_chip *) &mx2nand_mtd[1];

#ifdef CONFIG_MTD_PARTITIONS
	/* Clean up partitions */
	del_mtd_partitions(mx2nand_mtd);
#else
	/* Unregister the device */
	del_mtd_device(mx2nand_mtd);
#endif

	/* Free internal data buffer */
	kfree(this->data_buf);

	/* Free the MTD device structure */
	kfree(mx2nand_mtd);

	if (mx2nand_gpio_init) {
		mx2_unregister_gpios(PORT_F, MX2_NAND_GPIO_PORTF_MASK);
	}

	if (mx2nand_region_init) {
		release_region(NFC_BASE, NFC_IO_SIZE);
	}

}

module_init(mx2nand_init);
module_exit(mx2nand_cleanup);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("MontaVista Software, Inc");
MODULE_DESCRIPTION("MTD Driver for MX2 Nand Flash Controller");
#define NFC_IO_SIZE			0x1000
