/*======================================================================

    drivers/mtd/maps/versatile-flash.c: ARM Flash Layout/Partitioning
  
    Copyright (C) 2002-2003 ARM Limited
  
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
  
   This is access code for flashes using ARM's flash partitioning 
   standards.

======================================================================*/

#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/ioport.h>
#include <linux/init.h>

#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>

#include <asm/hardware.h>
#include <asm/io.h>
#include <asm/system.h>

// board specific stuff - sorry, it should be in arch/arm/mach-*.
#ifdef CONFIG_ARCH_VERSATILE_PB

#define FLASH_BASE		VERSATILE_FLASH_BASE
//#define FLASH_BASE		VERSATILE_BOOT_ROM_BASE
#define FLASH_PART_SIZE 	0x400000
#define VERSATILE_FLASHCTRL	(IO_ADDRESS(VERSATILE_SYS_BASE) + VERSATILE_SYS_FLASH_OFFSET)

static unsigned int versatile_flash_size;

static struct mtd_partition *parsed_parts;
static int dynamic_parts = 0;

static const char *part_probes[] = { "cmdlinepart", "RedBoot", "afs", NULL };

static struct mtd_partition versatile_partitions[] = {
	{
		.name		= "BootLoader",
		.size		= 0xc0000,
		.offset		= 0,
		.mask_flags	= MTD_WRITEABLE,  /* force read-only */
	}, 
	{
		.name		= "Flash0 FileSys",
		.size		= VERSATILE_FLASH_SIZE - 0xc0000,
		.offset		= MTDPART_OFS_APPEND,
	}, 
};

/*
 * Initialise the flash access systems:
 */
static void armflash_flash_init(void)
{
	unsigned int tmp;

	tmp = __raw_readl(VERSATILE_FLASHCTRL);
	tmp |= VERSATILE_FLASHPROG_FLVPPEN;
	__raw_writel(tmp, VERSATILE_FLASHCTRL);
}

/*
 * Shutdown the flash access systems:
 *  - Disable VPP
 *  - Assert WP
 */
static void armflash_flash_exit(void)
{
	unsigned int tmp;

	tmp = __raw_readl(VERSATILE_FLASHCTRL);
	tmp &= ~(VERSATILE_FLASHPROG_FLVPPEN);
	__raw_writel(tmp, VERSATILE_FLASHCTRL);
}

static void armflash_flash_wp(int on)
{
}

static void armflash_set_vpp(struct map_info *map, int on)
{
	unsigned int tmp;

	tmp = __raw_readl(VERSATILE_FLASHCTRL);

	if (on)
	    tmp |= VERSATILE_FLASHPROG_FLVPPEN;
	else
    	    tmp &= ~VERSATILE_FLASHPROG_FLVPPEN;

	__raw_writel(tmp, VERSATILE_FLASHCTRL);
}
#endif


static __u8 armflash_read8(struct map_info *map, unsigned long ofs)
{
	return __raw_readb(ofs + map->map_priv_2);
}

static __u16 armflash_read16(struct map_info *map, unsigned long ofs)
{
	return __raw_readw(ofs + map->map_priv_2);
}

static __u32 armflash_read32(struct map_info *map, unsigned long ofs)
{
	return __raw_readl(ofs + map->map_priv_2);
}

static void armflash_copy_from(struct map_info *map, void *to, unsigned long from, ssize_t len)
{
	memcpy(to, (void *) (from + map->map_priv_2), len);
}

static void armflash_write8(struct map_info *map, __u8 d, unsigned long adr)
{
	__raw_writeb(d, adr + map->map_priv_2);
}

static void armflash_write16(struct map_info *map, __u16 d, unsigned long adr)
{
	__raw_writew(d, adr + map->map_priv_2);
}

static void armflash_write32(struct map_info *map, __u32 d, unsigned long adr)
{
	__raw_writel(d, adr + map->map_priv_2);
}

static void armflash_copy_to(struct map_info *map, unsigned long to, const void *from, ssize_t len)
{
	memcpy((void *) (to + map->map_priv_2), from, len);
}

static struct map_info armflash_map =
{
	.name =		"Versatile",
	.read8 =		armflash_read8,
	.read16 =		armflash_read16,
	.read32 =		armflash_read32,
	.copy_from =	armflash_copy_from,
	.write8 =		armflash_write8,
	.write16 =	armflash_write16,
	.write32 =	armflash_write32,
	.copy_to =	armflash_copy_to,
	.set_vpp =	armflash_set_vpp,
};

static int __init
versatile_static_partitions(struct mtd_partition **parts)
{
	int nb_parts = 0;

	*parts       = versatile_partitions;
	nb_parts     = ARRAY_SIZE(versatile_partitions);

	return nb_parts;
}

static struct mtd_info *mtd;

static int __init armflash_cfi_init(void *base, u_int size)
{
	int nb_parts = 0;
	int ret;

	armflash_flash_init();
	armflash_flash_wp(1);

	/*
	 * look for CFI based flash parts fitted to this board
	 */
	armflash_map.size       = size;
	armflash_map.buswidth   = 4;
	armflash_map.map_priv_2 = (unsigned long) base;
	armflash_map.virt = (unsigned long) base;

	/*
	 * Also, the CFI layer automatically works out what size
	 * of chips we have, and does the necessary identification
	 * for us automatically.
	 */
	mtd = do_map_probe("cfi_probe", &armflash_map);
	if (!mtd)
		return -ENXIO;

	mtd->owner = THIS_MODULE;

        do {
#ifdef CONFIG_MTD_PARTITIONS
		nb_parts = parse_mtd_partitions(mtd, part_probes, &parsed_parts, 0);
		if (nb_parts > 0)
		{
			dynamic_parts = 1;
			break;
		}
#endif
		nb_parts = versatile_static_partitions(&parsed_parts);
	} while (0);

	if (nb_parts < 0) {
		printk(KERN_NOTICE "Versatile flash: no partition info available,"
				   "registering whole flash at once\n");
		add_mtd_device(mtd);
	} else {
		printk(KERN_NOTICE "Using cmdline partition definition\n");
		ret = add_mtd_partitions(mtd, parsed_parts, nb_parts);
		if (ret)
		    printk(KERN_ERR "mtd partition registration failed: %d\n", ret);
	}

	/*
	 * If we got an error, free all resources.
	 */
	if (ret < 0) {
		del_mtd_partitions(mtd);
		map_destroy(mtd);
	}

	return ret;
}

static void armflash_cfi_exit(void)
{
	if (mtd) {
		del_mtd_partitions(mtd);
		map_destroy(mtd);
	}
	if (parsed_parts && dynamic_parts)
		kfree(parsed_parts);
}

static int __init armflash_init(void)
{
	int err = -EBUSY;
	void *base;

	versatile_flash_size = VERSATILE_FLASH_SIZE;

	if (request_mem_region(FLASH_BASE, versatile_flash_size, "flash") == NULL)
		goto out;

	base = ioremap(FLASH_BASE, versatile_flash_size);
	err = -ENOMEM;
	if (base == NULL)
		goto release;

	err = armflash_cfi_init(base, versatile_flash_size);
	if (err) {
		iounmap(base);
release:
		release_mem_region(FLASH_BASE, versatile_flash_size);
	}
out:
	return err;
}

static void __exit armflash_exit(void)
{
	armflash_cfi_exit();
	iounmap((void *)armflash_map.map_priv_2);
	release_mem_region(FLASH_BASE, versatile_flash_size);
	armflash_flash_exit();
}

module_init(armflash_init);
module_exit(armflash_exit);

MODULE_AUTHOR("ARM Ltd");
MODULE_DESCRIPTION("ARM Versatile CFI map driver");
MODULE_LICENSE("GPL");
